import warnings

import numpy as np

from generic import generic
from issmsettings import issmsettings
from meshintersect3d import meshintersect3d
from miscellaneous import miscellaneous
from pairoptions import pairoptions
from private import private
from TwoDToThreeD import TwoDToThreeD

class sealevelmodel(object):
    '''
    SEALEVELMODEL class definition

        Usage:
            slm = sealevelmodel(*args)

            where args is a variable list of options

        Example:
            slm = sealevelmodel(
                'icecap', md_greenland,
                'icecap', md_antarctica,
                'earth', md_earth
            )
    '''

    def __init__(self, *args): #{{{
        self.icecaps        = [] # list of land/ice models; name should be changed later
        self.earth          = 0 # model for the whole earth
        self.basins         = [] # list of basins, matching icecaps, where shapefile info is held
        self.cluster        = 0
        self.miscellaneous  = 0
        self.settings       = 0
        self.private        = 0
        self.mergedcaps     = 0
        self.transitions    = []
        self.eltransitions  = []
        self.planet         = ''

        # Create a default object
        self.setdefaultparameters()

        if len(args):
            # Use provided options to set fields
            options = pairoptions(*args)

            # Recover all the icecap models
            self.icecaps = options.getfieldvalue('ice_cap', [])

            # Recover the earth models
            self.earth = options.getfieldvalue('earth', 0)

            # Set planet type
            self.planet = options.getfieldvalue('planet', 'earth')
    #}}}

    def __repr__(self): # {{{
        s = '%s\n' % fielddisplay(self, 'icecaps', 'ice caps')
        s += '%s\n' % fielddisplay(self, 'earth', 'earth')
        s += '%s\n' % fielddisplay(self, 'settings', 'settings properties')
        s += '%s\n' % fielddisplay(self, 'cluster', 'cluster parameters (number of cpus...')
        s += '%s\n' % fielddisplay(self, 'miscellaneous', 'miscellaneous fields')
    #}}}

    def setdefaultparameters(self): # {{{
        self.icecaps        = []
        self.earth          = []
        self.cluster        = generic()
        self.miscellaneous  = miscellaneous()
        self.settings       = issmsettings()
        self.private        = private()
        self.transitions    = []
        self.eltransitions  = []
        self.planet         = 'earth'

        return self
    #}}}

    @staticmethod
    def checkconsistency(slm, solutiontype): # {{{
        # Is the coupler turned on?
        for i in range(len(slm.icecaps)):
            if slm.icecaps[i].transient.iscoupler == 0:
                warnings.warn('sealevelmodel checkconsistency error: icecap model %s should have the transient coupler option turned on!' % slm.icecaps[i].miscellaneous.name)

        if slm.earth.transient.iscoupler == 0:
            warnings.warn('sealevelmodel checkconsistency error: earth model should have the transient coupler option turned on!')

        # Check that the transition vectors have the right size
        for i in range(len(slm.icecaps)):
            if slm.icecaps[i].mesh.numberofvertices != len(slm.earth.slr.transitions[i]):
                raise RuntimeError('sealevelmodel checkconsistency issue with size of transition vector for ice cap: %i name: %s' % (i, slm.icecaps[i].miscellaneous.name))

        # Check that run frequency is the same everywhere
        for i in range(len(slm.icecaps)):
            if slm.icecaps[i].slr.geodetic_run_frequency != slm.earth.geodetic_run_frequency:
                raise RuntimeError('sealevelmodel checkconsistency error: icecap model %s should have the same run frequency as earth!' % slm.icecaps[i].miscellaneous.name)

        # Make sure steric_rate is the same everywhere
        for i in range(len(slm.icecaps)):
            md = slm.icecaps[i]
            if np.nonzero(md.slr.steric_rate - slm.earth.slr.steric_rate[slm.earth.slr.transitions[i]]) != []:
                raise RuntimeError('steric rate on ice cap %s is not the same as for the earth' % md.miscellaneous.name)
    #}}}

    def mergeresults(self): # {{{
        champs = fieldnames(self.icecaps[i].results.TransientSolution)
        for i in range(len(self.mergedcaps / 2)):
            md = self.mergedcaps[2 * i]
            trans = self.mergedcaps[2 * i + 1]
            for j in range(len(self.icecaps[0].results.TransientSolution)):
                for k in range(len(champs)):
                    if isinstance(getattr(icecaps[0].results.TransientSolution[j], champs[k]), float):
                        # Vertex or element?
                        if len(getattr(icecaps[0].results.TransientSolution[j], champs[k]) == icecaps[0].mesh.numberofvertices):
                            setattr(md.results.TransientSolution[j], champs[k], np.zeros(md.mesh.numberofvertices))
                            for l in range(len(trans)):
                                resultcap = getattr(icecaps[l].results.TransientSolution[j], champs[k])
                                setattr(getattr(md.results.TransientSolution[j], champs[k]), trans[l], resultcap)
                        else:
                            if champs[k] == 'IceVolume' or champs[k] == 'IceVolumeAboveFlotation':
                                setattr(md.results.TransientSolution, champs[k], 0)
                                for l in range(len(trans)):
                                    resultcap = getattr(icecaps[l].results.TransientSolution[j], champs[k])
                                    setattr(md.results.TransientSolution[j], champs[k], getattr(md.results.TransientSolution[j], champs[k]) + resultcap)
                            elif champs[k] == 'time':
                                setattr(md.results.TransientSolution[j], champs[k], getattr(icecaps[0].results.TransientSolution[j], champs[k]))
                            else:
                                continue
                    else:
                        continue
            self.mergedcaps[2 * i] = md
    #}}}
    
    def listcaps(self): # {{{
        for i in range(len(self.icecaps)):
            print('%i: %s' % (i, self.icecaps[i].miscellaneous.name))
    #}}}

    def continents(self): # {{{
        list = []
        for i in range(len(self.basins)):
            list.append = self.basins[i].continent
        return np.unique(list)
    #}}}

    def basinsfromcontinent(self, continent): # {{{
        list = []
        for i in range(len(self.icecaps)):
            if self.basins[i].continent == continent:
                list.append = self.basins[i].name
        return np.unique(list)
    #}}}

    def addbasin(self, bas): # {{{
        if bas.__class__.__name__ != 'basin':
            raise RuntimeError('addbasin method only takes a \'basin\' class object as input')
        self.basins.append(bas)
    #}}}

    def intersections(self, *args): #{{{
        options = pairoptions(*args)
        force = options.getfieldvalue('force', 0)

        # Initialize, to avoid issues of having more transitions than meshes
        self.transitions = []
        self.eltransitions = []

        # For elements
        onesmatrix = np.array([[1], [1], [1]])
        xe = self.earth.mesh.x[self.earth.mesh.elements] * onesmatrix / 3
        ye = self.earth.mesh.y[self.earth.mesh.elements] * onesmatrix / 3
        ze = self.earth.mesh.z[self.earth.mesh.elements] * onesmatrix / 3

        for i in range(len(self.icecaps)):
            mdi = self.icecaps[i]
            mdi = TwoDToThreeD(mdi, self.planet)

            # For elements
            zei = mdi.mesh.x[mdi.mesh.elements] * onesmatrix / 3
            yei = mdi.mesh.y[mdi.mesh.elements] * onesmatrix / 3
            zei = mdi.mesh.z[mdi.mesh.elements] * onesmatrix / 3

            print('Computing vertex intersections for basin %s' % self.basins[i].name)

            self.transitions.append(meshintersect3d(self.earth.mesh.x, self.earth.mesh.y, self.earth.mesh.z, mdi.mesh.x, mdi.mesh.y, mdi.mesh.z, 'force', force))
            self.eltransitions.append(meshintersect3d(xe, ye, ze, xei, yei, zei, 'force', force))
    #}}}
