/*!\file:  diagnostic.cpp
 * \brief: diagnostic solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "diagnostic"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif


int main(int argc,char* *argv){
	
	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   qmu_analysis=0;
	int   control_analysis=0;
	char* control_type=NULL;

	/*Model: */
	Model* model=NULL;

	/*Results: */
	DataSet* results=NULL;
	DataSet* processed_results=NULL;
	Result*  result=NULL;
	
	ParameterInputs* inputs=NULL;
	int waitonlock=0;
	int numberofnodes;
	
	double* u_g_initial=NULL;
	double* u_g_obs=NULL;
	Param*  param=NULL;
	int      count;
	DataSet* parameters=NULL;

	/*time*/
	double   start, finish;
	double   start_core, finish_core;
	double   start_init, finish_init;

	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	/*Initialize Petsc and get start time*/
	PetscInitialize(&argc,&argv,(char *)0,"");  
	MPI_Barrier(MPI_COMM_WORLD); start=MPI_Wtime();

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover , input file name and output file name:\n");
	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Initialize model structure: */
	MPI_Barrier(MPI_COMM_WORLD); start_init=MPI_Wtime();
	model=new Model();

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	_printf_("read and create finite element model:\n");
	_printf_("\n   reading diagnostic horiz model data:\n");
	model->AddFormulation(fid,DiagnosticAnalysisEnum(),HorizAnalysisEnum());

	_printf_("\n   reading diagnostic vert model data:\n");
	model->AddFormulation(fid,DiagnosticAnalysisEnum(),VertAnalysisEnum());
	
	_printf_("\n   reading diagnostic stokes model data:\n");
	model->AddFormulation(fid,DiagnosticAnalysisEnum(),StokesAnalysisEnum());
	
	_printf_("\n   reading diagnostic hutter model data:\n");
	model->AddFormulation(fid,DiagnosticAnalysisEnum(),HutterAnalysisEnum());
	
	_printf_("\n   reading surface and bed slope computation model data:\n");
	model->AddFormulation(fid,SlopeComputeAnalysisEnum());

	/*get parameters: */
	model->FindParam(&qmu_analysis,"qmu_analysis");
	model->FindParam(&control_analysis,"control_analysis");
	model->FindParam(&waitonlock,"waitonlock");

	_printf_("initialize inputs:\n");
	model->FindParam(&u_g_initial,NULL,NULL,"u_g",DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	model->FindParam(&numberofnodes,"numberofnodes");
	if(control_analysis){
		model->FindParam(&u_g_obs,NULL,NULL,"u_g_obs",DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	}

	inputs=new ParameterInputs;

	inputs->Add("velocity",u_g_initial,3,numberofnodes);
	if(control_analysis){
		inputs->Add("velocity_obs",u_g_obs,2,numberofnodes);
	}
	
	_printf_("initialize results:\n");
	results=new DataSet(ResultsEnum());
	MPI_Barrier(MPI_COMM_WORLD); finish_init=MPI_Wtime();

	/*are we running the solution sequence, or a qmu wrapper around it? : */
	if(!qmu_analysis){
		if(!control_analysis){
			/*run diagnostic analysis: */
			_printf_("call computational core:\n");
			MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
			diagnostic_core(results,model,inputs);
			MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );

			/*Add analysis_type to results: */
			result=new Result(results->Size()+1,0,1,"analysis_type","diagnostic");
			results->AddObject(result);
			
			_printf_("process results:\n");
			ProcessResults(&processed_results,results,model,DiagnosticAnalysisEnum());
		}
		else{
			/*run control analysis: */
			_printf_("call computational core:\n");
			MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
			control_core(results,model,inputs);
			MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );

			/*Add analysis_type and control_type to results: */
			result=new Result(results->Size()+1,0,1,"analysis_type","diagnostic");
			results->AddObject(result);
			
			model->FindParam(&control_type,"control_type");
			result=new Result(results->Size()+1,0,1,"control_type",control_type);
			results->AddObject(result);

			_printf_("process results:\n");
			ProcessResults(&processed_results,results,model,ControlAnalysisEnum());
		}

		_printf_("write results to disk:\n");
		OutputResults(processed_results,outputfilename);
	}
	else{
		/*run qmu analysis: */
		_printf_("calling qmu analysis on diagnostic core:\n");

		#ifdef _HAVE_DAKOTA_ 
		MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
		Qmux(model,inputs,DiagnosticAnalysisEnum(),NoneAnalysisEnum());
		MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );
	 	#else
		throw ErrorException(__FUNCT__," Dakota not present, cannot do qmu!");
		#endif
	}

	_printf_("write lock file:\n");
	if (waitonlock){
		WriteLockFile(lockname);
	}

	/*Free ressources */
	xfree((void**)&u_g_initial);
	xfree((void**)&u_g_obs);
	delete model;
	delete inputs;
	delete results;
	delete processed_results;

	/*Get finish time and close*/
	MPI_Barrier(MPI_COMM_WORLD); finish = MPI_Wtime( );
	_printf_("\n   %-34s %f seconds  \n","Model initialization elapsed time:",finish_init-start_init);
	_printf_("   %-34s %f seconds  \n","Core solution elapsed time:",finish_core-start_core);
	_printf_("   %-34s %f seconds\n\n","Total elapsed time:",finish-start);
	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 
	
	/*end module: */
	MODULEEND();

	return 0; //unix success return;
}
