%BOUNDARY class definition
%
%   Usage:
%      boundary=boundary();

classdef boundary
	properties (SetAccess=public) 
		shppath           = '';
		shpfilename       = '';
		orientation       = 'normal';  %other value is 'reverse'
		epsg              = 4326; %EPSG number, default value is WGS 84 Lat,Long reference frame.
	end
	methods (Static)
		function self = loadobj(self) % {{{
			% This function is directly called by matlab when a model object is
			% loaded. Update old properties here
		end% }}}
	end
	methods
		function self = boundary(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					self=setdefaultparameters(self);
					options=pairoptions(varargin{:}); 
			
					%recover field values: 
					self.shppath=getfieldvalue(options,'shppath','');
					self.shpfilename=getfieldvalue(options,'shpfilename','');
					self.orientation=getfieldvalue(options,'orientation','normal');
					self.epsg=getfieldvalue(options,'epsg',4326);
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{
		self.shppath='';
		self.shpfilename='';
		self.orientation='normal';
		self.epsg=4326;
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   boundary parameters:'));

			fielddisplay(self,'shppath','path to filename for this boundary');
			fielddisplay(self,'shpfilename','shape file name');
			fielddisplay(self,'orientation','orientation (default is ''normal'', can be ''reverse'')');
			fielddisplay(self,'epsg','EPSG number defining projection for the shape file');

		end % }}}
		function output=name(self) % {{{
			output=self.shpfilename;
		end % }}}
		function output=edges(self) % {{{
		
		%read domain:
		[path,name,ext]=fileparts(self.shpfilename);
		if ~strcmpi(ext,'shp'),
			ext='shp';
		end
		output=shpread([self.shppath '/' name '.' ext]);

		%do we reverse? 
		if strcmpi(self.orientation,'reverse'),
			output.x=flipud(output.x);
			output.y=flipud(output.y);
		end
		end % }}}
		function plot(self,varargin) % {{{
			%recover options
		
			options=pairoptions(varargin{:});

			%parse input:
			figurenumber=getfieldvalue(options,'figure',1);
			color=getfieldvalue(options,'color','r');
			linewidth=getfieldvalue(options,'linewidth',1);
			markersize=getfieldvalue(options,'markersize',1);
			unitmultiplier=getfieldvalue(options,'unit',1);
			epsg=getfieldvalue(options,'epsg',4326);
			radius=getfieldvalue(options,'radius',6371012);
			aboveground=getfieldvalue(options,'aboveground',1000);
			offset=getfieldvalue(options,'offset',.1);
			fontsize=getfieldvalue(options,'fontsize',10);
			label=getfieldvalue(options,'label','on');

			%read domain:
			[path,name,ext]=fileparts(self.shpfilename);
			if ~strcmpi(ext,'shp'),
				ext='shp';
			end
			domain=shpread([self.shppath '/' name '.' ext]);

			%convert boundary to another reference frame:  {{{

			for i=1:length(domain),
				try, 
					[x,y] = gdaltransform(domain(i).x,domain(i).y,sprintf('EPSG:%i',self.epsg),sprintf('EPSG:%i',epsg));
				catch me
					disp(me.message());
					self.disp();
				end
				domain(i).x=x; domain(i).y=y;
			end

			for i=1:length(domain),
				hold on;
				x=domain(i).x*unitmultiplier;
				y=domain(i).y*unitmultiplier;
				if length(x)==1,
					p=plot(x,y,'k*'); 
					set(p,'MarkerSize',markersize,'Color',color);
					if strcmpi(label,'on'),
						t=text(x,y,self.shpfilename,'FontSize',fontsize);
					end
				else
					p=plot(x,y,'k-'); 
					set(p,'MarkerSize',markersize,'Color',color);
					if strcmpi(label,'on'),
						text(sum(x)/length(x),sum(y)/length(y),self.shpfilename,'FontSize',fontsize);
					end
				end
				set(p,'Color',color);
				set(p,'LineWidth',linewidth);
			end
			%}}}
		end % }}}
		function plot3d(self,varargin) % {{{
			%recover options
		
			options=pairoptions(varargin{:});

			%parse input:
			figurenumber=getfieldvalue(options,'figure',1);
			color=getfieldvalue(options,'color','r');
			linewidth=getfieldvalue(options,'linewidth',1);
			markersize=getfieldvalue(options,'markersize',1);
			unitmultiplier=getfieldvalue(options,'unit',1);
			epsg=getfieldvalue(options,'epsg',4326);
			radius=getfieldvalue(options,'radius',6371012);
			aboveground=getfieldvalue(options,'aboveground',1000);
			offset=getfieldvalue(options,'offset',.1);
			fontsize=getfieldvalue(options,'fontsize',10);

			%read domain:
			[path,name,ext]=fileparts(self.shpfilename);
			if ~strcmpi(ext,'shp'),
				ext='shp';
			end
			domain=shpread([self.shppath '/' name '.' ext]);

			if epsg==4326,
				%convert boundary to lat,long: {{{

				for i=1:length(domain),
					try, 
						[lat,long] = gdaltransform(domain(i).x,domain(i).y,sprintf('EPSG:%i',self.epsg),'EPSG:4326');
					catch me
						disp(me.message());
						self.disp();
					end
					domain(i).x=long; domain(i).y=lat;
				end

				for i=1:length(domain),

					%make sure lat,long are what they are supposed to be: 
					%if any(domain(i).x>90 | domain(i).x<-90), 
					%	long=domain(i).x; lat=domain(i).y;
					%else
					%	long=domain(i).y; lat=domain(i).x;
					%end

					%project on x,y,z reference frame.
					[x,y,z]=AboveGround(domain(i).x,domain(i).y,radius,aboveground);
					[xt,yt,zt]=AboveGround(domain(i).x+offset,domain(i).y+offset,radius,300000);
					hold on;
					if length(x)==1,
						p=plot3(x,y,z,'k*'); 
						set(p,'MarkerSize',markersize);
						t=text(xt,yt,zt,self.shpfilename,'FontSize',fontsize);
					else
						p=plot3(x,y,z,'k-'); 
						mid=floor(length(xt)/2);
						text(xt(mid),yt(mid),zt(mid),self.shpfilename,'FontSize',fontsize);
					end
					set(p,'Color',color);
					set(p,'LineWidth',linewidth);

				end
				%}}}
			else
				%convert boundary to another reference frame:  {{{

				for i=1:length(domain),
					try, 
						[x,y] = gdaltransform(domain(i).x,domain(i).y,sprintf('EPSG:%i',self.epsg),sprintf('EPSG:%i',epsg));
					catch me
						disp(me.message());
						self.disp();
					end
					domain(i).x=x; domain(i).y=y;
				end

				for i=1:length(domain),
					hold on;
					if length(x)==1,
						p=plot(x,y,'k*'); 
						set(p,'MarkerSize',markersize);
						t=text(x,y,self.shpfilename,'FontSize',fontsize);
					else
						p=plot(x,y,'k-'); 
						text(sum(x)/length(x),sum(y)/length(y),self.shpfilename,'FontSize',fontsize);
					end
					set(p,'Color',color);
					set(p,'LineWidth',linewidth);
				end
				%}}}
			end
		end % }}}
	end
end
