import numpy as np


def love_numbers(value, * varargin):
    '''LOVE_NUMBERS: provide love numbers (value 'h', 'k', 'l', 'gamma' and 'lambda'
             retrieved from: http://www.srosat.com/iag-jsg/loveNb.php
    Usage:   series = love_numbers(value)
           series = love_numbers(value, reference_frame)

           where value is one of 'h', 'k', 'l', 'gamma' and 'lambda'.
        reference_frame = one of 'CM' (default) and 'CF'.

    Example:
        love_k = love_numbers('k')
        love_k = love_numbers('k', 'CF')

    '''

    # some checks:
    if len(varargin) == 0:
        frame = 'CM'
        print('Info: computation is done in Center of Mass (CM) reference frame by default')
    elif len(varargin) == 1:
        reference_frame = varargin[0]
        if (reference_frame in ['CF', 'CM']):
            frame = reference_frame
        else:
            raise RuntimeError('reference_frame should be one of ''CM'' or ''CF''')
    else:
        raise RuntimeError('love_numbers error message: bad usage')

    if value not in ['h', 'k', 'l', 'gamma', 'lambda']:
        raise RuntimeError('value should be one of ''h'', ''k'', ''l'', ''gamma'' and ''lambda''')

    if len(varargin) > 1:
        raise RuntimeError('love_numbers error message: wrong usage')

    love_numbers = np.array([[0, 0, 0, 0, 0, 0, 0],
                             [- 1.28740059, -1.00000000, -0.89858519, 1.28740059, 0.42519882, 0.89858519, 0.00000000],
                             [- 1.00025365, -0.30922675, 0.02060926, 1.69102690, 0.46358648, 0.67016399, 0.61829668],
                             [- 1.06243501, -0.19927948, 0.06801636, 1.86315553, 0.55741597, 0.73270416, 0.56270589],
                             [- 1.06779588, -0.13649834, 0.05667027, 1.93129754, 0.63672498, 0.80683140, 0.51132745],
                             [- 1.10365923, -0.10736896, 0.04401221, 1.99629027, 0.68737906, 0.84861883, 0.48642259],
                             [- 1.16440348, -0.09295485, 0.03638747, 2.07144863, 0.72031283, 0.87065768, 0.47898268],
                             [- 1.23634156, -0.08469861, 0.03202759, 2.15164295, 0.74355796, 0.88327380, 0.47955214],
                             [- 1.31140380, -0.07921412, 0.02937593, 2.23218968, 0.76126493, 0.89140995, 0.48323250],
                             [- 1.38582399, -0.07513541, 0.02762338, 2.31068858, 0.77552290, 0.89724121, 0.48795424],
                             [- 1.45807465, -0.07187005, 0.02638627, 2.38620460, 0.78744212, 0.90174369, 0.49291061],
                             [- 1.52763314, -0.06913154, 0.02547640, 2.45850160, 0.79766475, 0.90539206, 0.49779422],
                             [- 1.59437866, -0.06676258, 0.02479080, 2.52761607, 0.80659635, 0.90844662, 0.50248477],
                             [- 1.65833071, -0.06466619, 0.02426511, 2.59366452, 0.81451271, 0.91106870, 0.50693175],
                             [- 1.71954820, -0.06277732, 0.02385464, 2.65677088, 0.82161167, 0.91336804, 0.51111243],
                             [- 1.77809640, -0.06105001, 0.02352654, 2.71704639, 0.82804049, 0.91542346, 0.51501712],
                             [- 1.83403970, -0.05945081, 0.02325609, 2.77458889, 0.83391153, 0.91729309, 0.51864363],
                             [- 1.88744242, -0.05795502, 0.02302469, 2.82948740, 0.83931209, 0.91902029, 0.52199490],
                             [- 1.93837115, -0.05654418, 0.02281843, 2.88182697, 0.84431095, 0.92063739, 0.52507761],
                             [- 1.98689666, -0.05520447, 0.02262706, 2.93169219, 0.84896295, 0.92216847, 0.52790108],
                             [- 2.03309477, -0.05392545, 0.02244322, 2.97916932, 0.85331225, 0.92363132, 0.53047654],
                             [- 2.07704643, -0.05269926, 0.02226173, 3.02434717, 0.85739480, 0.92503902, 0.53281639],
                             [- 2.11883714, -0.05151988, 0.02207909, 3.06731726, 0.86124014, 0.92640103, 0.53493369],
                             [- 2.15855611, -0.05038274, 0.02189307, 3.10817337, 0.86487276, 0.92772419, 0.53684176],
                             [- 2.19629514, -0.04928430, 0.02170238, 3.14701084, 0.86831322, 0.92901331, 0.53855386],
                             [- 2.23214747, -0.04822179, 0.02150643, 3.18392568, 0.87157886, 0.93027178, 0.54008294],
                             [- 2.26620674, -0.04719301, 0.02130509, 3.21901373, 0.87468453, 0.93150190, 0.54144148],
                             [- 2.29856595, -0.04619619, 0.02109858, 3.25236976, 0.87764301, 0.93270523, 0.54264140],
                             [- 2.32931659, -0.04522983, 0.02088735, 3.28408675, 0.88046543, 0.93388282, 0.54369397],
                             [- 2.35854794, -0.04429270, 0.02067197, 3.31425524, 0.88316156, 0.93503533, 0.54460979],
                             [- 2.38634650, -0.04338368, 0.02045310, 3.34296281, 0.88574004, 0.93616321, 0.54539877],
                             [- 2.41279547, -0.04250179, 0.02023142, 3.37029367, 0.88820859, 0.93726678, 0.54607015],
                             [- 2.43797451, -0.04164613, 0.02000761, 3.39632839, 0.89057416, 0.93834626, 0.54663248],
                             [- 2.46195951, -0.04081583, 0.01978231, 3.42114367, 0.89284301, 0.93940185, 0.54709369],
                             [- 2.48482241, -0.04001011, 0.01955614, 3.44481230, 0.89502085, 0.94043375, 0.54746112],
                             [- 2.50663126, -0.03922817, 0.01932966, 3.46740309, 0.89711291, 0.94144217, 0.54774153],
                             [- 2.52745016, -0.03846928, 0.01910337, 3.48898088, 0.89912397, 0.94242735, 0.54794114],
                             [- 2.54733938, -0.03773269, 0.01887774, 3.50960670, 0.90105847, 0.94338957, 0.54806571],
                             [- 2.56635547, -0.03701769, 0.01865317, 3.52933779, 0.90292050, 0.94432915, 0.54812051],
                             [- 2.58455138, -0.03632358, 0.01843000, 3.54822780, 0.90471386, 0.94524642, 0.54811044],
                             [- 2.60197665, -0.03564968, 0.01820854, 3.56632697, 0.90644209, 0.94614178, 0.54803997],
                             [- 2.61867756, -0.03499532, 0.01798905, 3.58368224, 0.90810850, 0.94701563, 0.54791326],
                             [- 2.63469733, -0.03435985, 0.01777176, 3.60033748, 0.90971616, 0.94786840, 0.54773413],
                             [- 2.65007629, -0.03374263, 0.01755683, 3.61633367, 0.91126798, 0.94870054, 0.54750610],
                             [- 2.66485208, -0.03314303, 0.01734443, 3.63170905, 0.91276665, 0.94951253, 0.54723245],
                             [- 2.67905981, -0.03256047, 0.01713468, 3.64649934, 0.91421471, 0.95030485, 0.54691620],
                             [- 2.69273222, -0.03199435, 0.01692767, 3.66073787, 0.91561457, 0.95107798, 0.54656015],
                             [- 2.70589990, -0.03144411, 0.01672347, 3.67445580, 0.91696845, 0.95183242, 0.54616691],
                             [- 2.71859139, -0.03090919, 0.01652215, 3.68768220, 0.91827849, 0.95256866, 0.54573889],
                             [- 2.73083334, -0.03038907, 0.01632374, 3.70044427, 0.91954667, 0.95328719, 0.54527835],
                             [- 2.74265068, -0.02988323, 0.01612826, 3.71276745, 0.92077487, 0.95398851, 0.54478739],
                             [- 2.75406669, -0.02939118, 0.01593573, 3.72467551, 0.92196486, 0.95467309, 0.54426797],
                             [- 2.76510320, -0.02891245, 0.01574615, 3.73619076, 0.92311833, 0.95534141, 0.54372191],
                             [- 2.77578063, -0.02844656, 0.01555950, 3.74733406, 0.92423685, 0.95599393, 0.54315095],
                             [- 2.78611812, -0.02799309, 0.01537578, 3.75812503, 0.92532192, 0.95663113, 0.54255669],
                             [- 2.79613364, -0.02755161, 0.01519496, 3.76858203, 0.92637496, 0.95725343, 0.54194065],
                             [- 2.80584405, -0.02712170, 0.01501701, 3.77872235, 0.92739730, 0.95786128, 0.54130424],
                             [- 2.81526521, -0.02670298, 0.01484191, 3.78856223, 0.92839022, 0.95845511, 0.54064880],
                             [- 2.82441204, -0.02629506, 0.01466961, 3.79811697, 0.92935491, 0.95903532, 0.53997561],
                             [- 2.83329857, -0.02589759, 0.01450009, 3.80740098, 0.93029251, 0.95960232, 0.53928586],
                             [- 2.84193804, -0.02551021, 0.01433329, 3.81642782, 0.93120412, 0.96015649, 0.53858067],
                             [- 2.85034293, -0.02513260, 0.01416919, 3.82521033, 0.93209074, 0.96069821, 0.53786112],
                             [- 2.85852503, -0.02476443, 0.01400773, 3.83376061, 0.93295337, 0.96122784, 0.53712821],
                             [- 2.86649548, -0.02440538, 0.01384888, 3.84209010, 0.93379291, 0.96174574, 0.53638291],
                             [- 2.87426481, -0.02405518, 0.01369258, 3.85020963, 0.93461026, 0.96225224, 0.53562612],
                             [- 2.88184299, -0.02371352, 0.01353880, 3.85812947, 0.93540625, 0.96274768, 0.53485873],
                             [- 2.88923945, -0.02338014, 0.01338749, 3.86585931, 0.93618168, 0.96323236, 0.53408154],
                             [- 2.89646316, -0.02305478, 0.01323861, 3.87340838, 0.93693730, 0.96370661, 0.53329534],
                             [- 2.90352261, -0.02273718, 0.01309211, 3.88078542, 0.93767383, 0.96417071, 0.53250089],
                             [- 2.91042585, -0.02242710, 0.01294795, 3.88799874, 0.93839197, 0.96462494, 0.53169888],
                             [- 2.91718054, -0.02212431, 0.01280609, 3.89505623, 0.93909236, 0.96506960, 0.53089002],
                             [- 2.92379397, -0.02182859, 0.01266648, 3.90196538, 0.93977564, 0.96550493, 0.53007493],
                             [- 2.93027306, -0.02153971, 0.01252908, 3.90873334, 0.94044240, 0.96593120, 0.52925424],
                             [- 2.93662439, -0.02125748, 0.01239386, 3.91536691, 0.94109322, 0.96634866, 0.52842854],
                             [- 2.94285425, -0.02098169, 0.01226077, 3.92187256, 0.94172863, 0.96675754, 0.52759839],
                             [- 2.94896860, -0.02071215, 0.01212977, 3.92825645, 0.94234915, 0.96715808, 0.52676434],
                             [- 2.95497314, -0.02044868, 0.01200082, 3.93452446, 0.94295529, 0.96755050, 0.52592690],
                             [- 2.96087331, -0.02019110, 0.01187388, 3.94068220, 0.94354752, 0.96793501, 0.52508656],
                             [- 2.96667427, -0.01993924, 0.01174893, 3.94673503, 0.94412630, 0.96831183, 0.52424380],
                             [- 2.97238097, -0.01969293, 0.01162591, 3.95268804, 0.94469206, 0.96868116, 0.52339906],
                             [- 2.97799813, -0.01945201, 0.01150481, 3.95854612, 0.94524521, 0.96904318, 0.52255277],
                             [- 2.98353025, -0.01921634, 0.01138557, 3.96431391, 0.94578617, 0.96939809, 0.52170535],
                             [- 2.98898162, -0.01898576, 0.01126817, 3.96999586, 0.94631531, 0.96974607, 0.52085719],
                             [- 2.99435636, -0.01876014, 0.01115257, 3.97559622, 0.94683300, 0.97008729, 0.52000868],
                             [- 2.99965838, -0.01853932, 0.01103875, 3.98111905, 0.94733959, 0.97042193, 0.51916016],
                             [- 3.00489143, -0.01832319, 0.01092666, 3.98656824, 0.94783543, 0.97075015, 0.51831198],
                             [- 3.01005909, -0.01811161, 0.01081628, 3.99194748, 0.94832084, 0.97107211, 0.51746448],
                             [- 3.01516479, -0.01790446, 0.01070757, 3.99726033, 0.94879613, 0.97138796, 0.51661796],
                             [- 3.02021180, -0.01770162, 0.01060052, 4.00251017, 0.94926160, 0.97169786, 0.51577273],
                             [- 3.02520323, -0.01750298, 0.01049508, 4.00770025, 0.94971755, 0.97200194, 0.51492908],
                             [- 3.03014209, -0.01730842, 0.01039123, 4.01283367, 0.95016424, 0.97230035, 0.51408727],
                             [- 3.03503122, -0.01711783, 0.01028894, 4.01791339, 0.95060195, 0.97259323, 0.51324758],
                             [- 3.03987336, -0.01693111, 0.01018819, 4.02294225, 0.95103094, 0.97288070, 0.51241024],
                             [- 3.04467112, -0.01674816, 0.01008894, 4.02792295, 0.95145145, 0.97316290, 0.51157550],
                             [- 3.04942699, -0.01656889, 0.00999117, 4.03285810, 0.95186373, 0.97343995, 0.51074358],
                             [- 3.05414335, -0.01639319, 0.00989485, 4.03775017, 0.95226799, 0.97371196, 0.50991471],
                             [- 3.05882250, -0.01622097, 0.00979997, 4.04260153, 0.95266447, 0.97397906, 0.50908908],
                             [- 3.06346660, -0.01605215, 0.00970649, 4.04741445, 0.95305338, 0.97424136, 0.50826689],
                             [- 3.06807773, -0.01588664, 0.00961439, 4.05219109, 0.95343492, 0.97449897, 0.50744832],
                             [- 3.07265789, -0.01572436, 0.00952364, 4.05693353, 0.95380929, 0.97475200, 0.50663356],
                             [- 3.07720897, -0.01556522, 0.00943423, 4.06164375, 0.95417670, 0.97500055, 0.50582277],
                             [- 3.08173279, -0.01540916, 0.00934613, 4.06632364, 0.95453731, 0.97524472, 0.50501611],
                             [- 3.08623109, -0.01525608, 0.00925931, 4.07097501, 0.95489131, 0.97548461, 0.50421372],
                             [- 3.09070551, -0.01510592, 0.00917376, 4.07559959, 0.95523888, 0.97572032, 0.50341576],
                             [- 3.09515765, -0.01495861, 0.00908946, 4.08019904, 0.95558018, 0.97595193, 0.50262236],
                             [- 3.09958899, -0.01481408, 0.00900637, 4.08477492, 0.95591537, 0.97617955, 0.50183364],
                             [- 3.10400100, -0.01467225, 0.00892449, 4.08932875, 0.95624461, 0.97640325, 0.50104973],
                             [- 3.10839504, -0.01453308, 0.00884379, 4.09386196, 0.95656806, 0.97662313, 0.50027073],
                             [- 3.11277241, -0.01439648, 0.00876425, 4.09837593, 0.95688585, 0.97683927, 0.49949676],
                             [- 3.11713438, -0.01426240, 0.00868586, 4.10287198, 0.95719812, 0.97705174, 0.49872791],
                             [- 3.12148213, -0.01413079, 0.00860858, 4.10735134, 0.95750503, 0.97726063, 0.49796429],
                             [- 3.12581680, -0.01400157, 0.00853241, 4.11181522, 0.95780669, 0.97746601, 0.49720597],
                             [- 3.13013947, -0.01387471, 0.00845733, 4.11626476, 0.95810324, 0.97766796, 0.49645304],
                             [- 3.13445117, -0.01375013, 0.00838331, 4.12070104, 0.95839480, 0.97786656, 0.49570558],
                             [- 3.13875289, -0.01362779, 0.00831034, 4.12512510, 0.95868150, 0.97806186, 0.49496366],
                             [- 3.14304556, -0.01350764, 0.00823841, 4.12953792, 0.95896344, 0.97825395, 0.49422734],
                             [- 3.14733008, -0.01338963, 0.00816748, 4.13394045, 0.95924075, 0.97844289, 0.49349669],
                             [- 3.15160728, -0.01327370, 0.00809756, 4.13833358, 0.95951352, 0.97862874, 0.49277177],
                             [- 3.15587797, -0.01315981, 0.00802862, 4.14271816, 0.95978188, 0.97881157, 0.49205262],
                             [- 3.16014293, -0.01304792, 0.00796064, 4.14709501, 0.96004592, 0.97899144, 0.49133930],
                             [- 3.16440288, -0.01293797, 0.00789361, 4.15146491, 0.96030574, 0.97916842, 0.49063185],
                             [- 3.16865852, -0.01282993, 0.00782751, 4.15582858, 0.96056144, 0.97934256, 0.48993030],
                             [- 3.17291049, -0.01272375, 0.00776233, 4.16018673, 0.96081312, 0.97951392, 0.48923471],
                             [- 3.17715942, -0.01261940, 0.00769805, 4.16454003, 0.96106086, 0.97968255, 0.48854509],
                             [- 3.18140591, -0.01251682, 0.00763466, 4.16888910, 0.96130476, 0.97984852, 0.48786148],
                             [- 3.18565052, -0.01241598, 0.00757215, 4.17323454, 0.96154490, 0.98001187, 0.48718390],
                             [- 3.18989378, -0.01231685, 0.00751049, 4.17757693, 0.96178137, 0.98017266, 0.48651237],
                             [- 3.19413619, -0.01221938, 0.00744968, 4.18191681, 0.96201424, 0.98033094, 0.48584692],
                             [- 3.19837823, -0.01212354, 0.00738970, 4.18625469, 0.96224360, 0.98048676, 0.48518756],
                             [- 3.20262035, -0.01202930, 0.00733053, 4.19059105, 0.96246952, 0.98064017, 0.48453431],
                             [- 3.20686298, -0.01193661, 0.00727217, 4.19492637, 0.96269208, 0.98079121, 0.48388717],
                             [- 3.21110653, -0.01184546, 0.00721461, 4.19926107, 0.96291135, 0.98093994, 0.48324615],
                             [- 3.21535137, -0.01175579, 0.00715782, 4.20359557, 0.96312741, 0.98108639, 0.48261126],
                             [- 3.21959786, -0.01166759, 0.00710179, 4.20793027, 0.96334031, 0.98123062, 0.48198250],
                             [- 3.22384634, -0.01158082, 0.00704652, 4.21226552, 0.96355014, 0.98137266, 0.48135988],
                             [- 3.22809714, -0.01149545, 0.00699199, 4.21660169, 0.96375694, 0.98151256, 0.48074338],
                             [- 3.23235055, -0.01141146, 0.00693819, 4.22093909, 0.96396080, 0.98165035, 0.48013301],
                             [- 3.23660685, -0.01132880, 0.00688511, 4.22527805, 0.96416176, 0.98178609, 0.47952876],
                             [- 3.24086631, -0.01124746, 0.00683273, 4.22961885, 0.96435989, 0.98191980, 0.47893063],
                             [- 3.24512918, -0.01116741, 0.00678105, 4.23396177, 0.96455525, 0.98205153, 0.47833860],
                             [- 3.24939569, -0.01108862, 0.00673005, 4.23830707, 0.96474789, 0.98218132, 0.47775267],
                             [- 3.25366606, -0.01101107, 0.00667973, 4.24265499, 0.96493787, 0.98230920, 0.47717282],
                             [- 3.25794050, -0.01093473, 0.00663007, 4.24700577, 0.96512525, 0.98243520, 0.47659903],
                             [- 3.26221918, -0.01085957, 0.00658106, 4.25135961, 0.96531007, 0.98255937, 0.47603130],
                             [- 3.26650230, -0.01078557, 0.00653269, 4.25571672, 0.96549239, 0.98268174, 0.47546960],
                             [- 3.27079000, -0.01071272, 0.00648495, 4.26007729, 0.96567225, 0.98280233, 0.47491391],
                             [- 3.27508246, -0.01064097, 0.00643784, 4.26444149, 0.96584971, 0.98292119, 0.47436422],
                             [- 3.27937980, -0.01057032, 0.00639134, 4.26880948, 0.96602482, 0.98303834, 0.47382051],
                             [- 3.28368216, -0.01050074, 0.00634544, 4.27318141, 0.96619761, 0.98315382, 0.47328275],
                             [- 3.28798965, -0.01043222, 0.00630013, 4.27755743, 0.96636814, 0.98326765, 0.47275091],
                             [- 3.29230239, -0.01036472, 0.00625541, 4.28193767, 0.96653645, 0.98337988, 0.47222499],
                             [- 3.29662047, -0.01029823, 0.00621126, 4.28632224, 0.96670258, 0.98349051, 0.47170494],
                             [- 3.30094399, -0.01023273, 0.00616768, 4.29071126, 0.96686657, 0.98359960, 0.47119074],
                             [- 3.30527303, -0.01016819, 0.00612465, 4.29510483, 0.96702847, 0.98370715, 0.47068237],
                             [- 3.30960766, -0.01010461, 0.00608218, 4.29950304, 0.96718831, 0.98381321, 0.47017979],
                             [- 3.31394795, -0.01004197, 0.00604024, 4.30390598, 0.96734614, 0.98391779, 0.46968299],
                             [- 3.31829395, -0.00998024, 0.00599883, 4.30831372, 0.96750198, 0.98402093, 0.46919192],
                             [- 3.32264573, -0.00991940, 0.00595795, 4.31272633, 0.96765588, 0.98412265, 0.46870656],
                             [- 3.32700331, -0.00985945, 0.00591759, 4.31714387, 0.96780788, 0.98422297, 0.46822687],
                             [- 3.33136675, -0.00980035, 0.00587773, 4.32156640, 0.96795801, 0.98432191, 0.46775284],
                             [- 3.33573607, -0.00974211, 0.00583838, 4.32599396, 0.96810630, 0.98441951, 0.46728441],
                             [- 3.34011130, -0.00968470, 0.00579951, 4.33042660, 0.96825278, 0.98451579, 0.46682157],
                             [- 3.34449246, -0.00962810, 0.00576113, 4.33486436, 0.96839750, 0.98461077, 0.46636427],
                             [- 3.34887956, -0.00957230, 0.00572323, 4.33930726, 0.96854048, 0.98470447, 0.46591248],
                             [- 3.35327261, -0.00951729, 0.00568581, 4.34375533, 0.96868175, 0.98479691, 0.46546617],
                             [- 3.35767163, -0.00946304, 0.00564884, 4.34820858, 0.96882135, 0.98488812, 0.46502531],
                             [- 3.36207660, -0.00940956, 0.00561233, 4.35266704, 0.96895930, 0.98497811, 0.46458986],
                             [- 3.36648753, -0.00935681, 0.00557627, 4.35713071, 0.96909563, 0.98506691, 0.46415977],
                             [- 3.37090440, -0.00930480, 0.00554066, 4.36159960, 0.96923037, 0.98515454, 0.46373503],
                             [- 3.37532721, -0.00925350, 0.00550548, 4.36607371, 0.96936355, 0.98524102, 0.46331559],
                             [- 3.37975593, -0.00920290, 0.00547073, 4.37055303, 0.96949520, 0.98532636, 0.46290141],
                             [- 3.38419056, -0.00915300, 0.00543641, 4.37503756, 0.96962535, 0.98541059, 0.46249246],
                             [- 3.38863105, -0.00910377, 0.00540251, 4.37952729, 0.96975401, 0.98549373, 0.46208870],
                             [- 3.39307740, -0.00905520, 0.00536901, 4.38402220, 0.96988122, 0.98557578, 0.46169009],
                             [- 3.39752956, -0.00900729, 0.00533593, 4.38852227, 0.97000699, 0.98565678, 0.46129660],
                             [- 3.40198751, -0.00896002, 0.00530324, 4.39302749, 0.97013137, 0.98573674, 0.46090819],
                             [- 3.40645121, -0.00891338, 0.00527095, 4.39753783, 0.97025435, 0.98581567, 0.46052482],
                             [- 3.41092063, -0.00886736, 0.00523904, 4.40205326, 0.97037598, 0.98589360, 0.46014645],
                             [- 3.41539571, -0.00882195, 0.00520752, 4.40657376, 0.97049628, 0.98597053, 0.45977305],
                             [- 3.41987643, -0.00877713, 0.00517637, 4.41109929, 0.97061526, 0.98604649, 0.45940458],
                             [- 3.42436272, -0.00873290, 0.00514560, 4.41562982, 0.97073295, 0.98612149, 0.45904100],
                             [- 3.42885456, -0.00868925, 0.00511520, 4.42016531, 0.97084936, 0.98619555, 0.45868227],
                             [- 3.43335188, -0.00864617, 0.00508515, 4.42470571, 0.97096453, 0.98626868, 0.45832835],
                             [- 3.43785464, -0.00860364, 0.00505546, 4.42925100, 0.97107847, 0.98634090, 0.45797921],
                             [- 3.44236278, -0.00856166, 0.00502613, 4.43380112, 0.97119120, 0.98641222, 0.45763480],
                             [- 3.44687625, -0.00852021, 0.00499714, 4.43835604, 0.97130274, 0.98648265, 0.45729509],
                             [- 3.45139500, -0.00847930, 0.00496849, 4.44291570, 0.97141311, 0.98655221, 0.45696005],
                             [- 3.45591895, -0.00843890, 0.00494017, 4.44748005, 0.97152233, 0.98662092, 0.45662962],
                             [- 3.46044807, -0.00839902, 0.00491219, 4.45204905, 0.97163042, 0.98668879, 0.45630378],
                             [- 3.46498227, -0.00835964, 0.00488454, 4.45662264, 0.97173739, 0.98675583, 0.45598249],
                             [- 3.46952151, -0.00832075, 0.00485721, 4.46120077, 0.97184326, 0.98682205, 0.45566570],
                             [- 3.47406572, -0.00828234, 0.00483019, 4.46578338, 0.97194805, 0.98688746, 0.45535338],
                             [- 3.47861484, -0.00824442, 0.00480349, 4.47037042, 0.97205179, 0.98695209, 0.45504550],
                             [- 3.48316880, -0.00820696, 0.00477710, 4.47496184, 0.97215447, 0.98701594, 0.45474201],
                             [- 3.48772753, -0.00816996, 0.00475102, 4.47955756, 0.97225612, 0.98707902, 0.45444287],
                             [- 3.49229097, -0.00813342, 0.00472523, 4.48415755, 0.97235676, 0.98714134, 0.45414806],
                             [- 3.49685904, -0.00809733, 0.00469975, 4.48876172, 0.97245640, 0.98720293, 0.45385753],
                             [- 3.50143169, -0.00806167, 0.00467455, 4.49337002, 0.97255506, 0.98726378, 0.45357123],
                             [- 3.50600884, -0.00802644, 0.00464964, 4.49798240, 0.97265275, 0.98732391, 0.45328915],
                             [- 3.51059042, -0.00799164, 0.00462502, 4.50259878, 0.97274949, 0.98738333, 0.45301123],
                             [- 3.51517637, -0.00795726, 0.00460068, 4.50721911, 0.97284528, 0.98744206, 0.45273745],
                             [- 3.51976660, -0.00792329, 0.00457662, 4.51184331, 0.97294015, 0.98750009, 0.45246776],
                             [- 3.52436105, -0.00788972, 0.00455283, 4.51647133, 0.97303411, 0.98755745, 0.45220214],
                             [- 3.52895964, -0.00785655, 0.00452930, 4.52110309, 0.97312718, 0.98761414, 0.45194053],
                             [- 3.53356231, -0.00782377, 0.00450605, 4.52573854, 0.97321936, 0.98767018, 0.45168291],
                             [- 3.53816898, -0.00779138, 0.00448306, 4.53037760, 0.97331067, 0.98772556, 0.45142923],
                             [- 3.54277957, -0.00775937, 0.00446032, 4.53502021, 0.97340111, 0.98778031, 0.45117947],
                             [- 3.54739402, -0.00772773, 0.00443784, 4.53966629, 0.97349072, 0.98783443, 0.45093359],
                             [- 3.55201224, -0.00769645, 0.00441562, 4.54431579, 0.97357949, 0.98788793, 0.45069155],
                             [- 3.55663417, -0.00766554, 0.00439364, 4.54896864, 0.97366744, 0.98794082, 0.45045331],
                             [- 3.56125973, -0.00763498, 0.00437190, 4.55362475, 0.97375458, 0.98799311, 0.45021885],
                             [- 3.56588885, -0.00760478, 0.00435041, 4.55828407, 0.97384092, 0.98804481, 0.44998812],
                             [- 3.57052145, -0.00757491, 0.00432916, 4.56294653, 0.97392648, 0.98809593, 0.44976109],
                             [- 3.57515745, -0.00754539, 0.00430814, 4.56761206, 0.97401126, 0.98814646, 0.44953772],
                             [- 3.57979678, -0.00751620, 0.00428736, 4.57228058, 0.97409528, 0.98819644, 0.44931799],
                             [- 3.58443937, -0.00748734, 0.00426681, 4.57695203, 0.97417854, 0.98824585, 0.44910185],
                             [- 3.58908514, -0.00745880, 0.00424648, 4.58162633, 0.97426107, 0.98829472, 0.44888928],
                             [- 3.59373401, -0.00743059, 0.00422637, 4.58630343, 0.97434286, 0.98834304, 0.44868023],
                             [- 3.59838592, -0.00740268, 0.00420649, 4.59098323, 0.97442393, 0.98839083, 0.44847468],
                             [- 3.60304078, -0.00737509, 0.00418682, 4.59566569, 0.97450428, 0.98843809, 0.44827259],
                             [- 3.60769852, -0.00734780, 0.00416737, 4.60035072, 0.97458394, 0.98848483, 0.44807392],
                             [- 3.61235907, -0.00732081, 0.00414813, 4.60503826, 0.97466290, 0.98853106, 0.44787865],
                             [- 3.61702235, -0.00729411, 0.00412910, 4.60972823, 0.97474118, 0.98857678, 0.44768674],
                             [- 3.62168828, -0.00726771, 0.00411028, 4.61442057, 0.97481879, 0.98862201, 0.44749816],
                             [- 3.62635680, -0.00724159, 0.00409166, 4.61911521, 0.97489573, 0.98866675, 0.44731288],
                             [- 3.63102782, -0.00721575, 0.00407325, 4.62381207, 0.97497202, 0.98871100, 0.44713086],
                             [- 3.63570128, -0.00719020, 0.00405503, 4.62851108, 0.97504766, 0.98875478, 0.44695207],
                             [- 3.64037709, -0.00716491, 0.00403700, 4.63321218, 0.97512267, 0.98879808, 0.44677649],
                             [- 3.64505519, -0.00713990, 0.00401918, 4.63791530, 0.97519704, 0.98884093, 0.44660407],
                             [- 3.64973550, -0.00711515, 0.00400154, 4.64262036, 0.97527080, 0.98888331, 0.44643478],
                             [- 3.65441795, -0.00709066, 0.00398409, 4.64732729, 0.97534394, 0.98892525, 0.44626861],
                             [- 3.65910247, -0.00706643, 0.00396683, 4.65203604, 0.97541648, 0.98896674, 0.44610551],
                             [- 3.66378898, -0.00704246, 0.00394975, 4.65674652, 0.97548842, 0.98900779, 0.44594545],
                             [- 3.66847740, -0.00701873, 0.00393286, 4.66145867, 0.97555978, 0.98904841, 0.44578841],
                             [- 3.67316767, -0.00699526, 0.00391614, 4.66617242, 0.97563055, 0.98908860, 0.44563435],
                             [- 3.67785972, -0.00697202, 0.00389960, 4.67088770, 0.97570076, 0.98912838, 0.44548324],
                             [- 3.68255347, -0.00694903, 0.00388324, 4.67560444, 0.97577039, 0.98916773, 0.44533506],
                             [- 3.68724885, -0.00692627, 0.00386705, 4.68032258, 0.97583947, 0.98920668, 0.44518977],
                             [- 3.69194579, -0.00690374, 0.00385103, 4.68504204, 0.97590800, 0.98924523, 0.44504735],
                             [- 3.69664421, -0.00688145, 0.00383518, 4.68976277, 0.97597598, 0.98928338, 0.44490776],
                             [- 3.70134406, -0.00685938, 0.00381949, 4.69448468, 0.97604342, 0.98932113, 0.44477099],
                             [- 3.70604525, -0.00683753, 0.00380397, 4.69920772, 0.97611034, 0.98935850, 0.44463698],
                             [- 3.71074772, -0.00681590, 0.00378861, 4.70393182, 0.97617673, 0.98939548, 0.44450573],
                             [- 3.71545140, -0.00679449, 0.00377342, 4.70865691, 0.97624261, 0.98943209, 0.44437720],
                             [- 3.72015622, -0.00677330, 0.00375838, 4.71338292, 0.97630797, 0.98946833, 0.44425137],
                             [- 3.72486211, -0.00675231, 0.00374349, 4.71810980, 0.97637283, 0.98950420, 0.44412820],
                             [- 3.72956899, -0.00673153, 0.00372877, 4.72283746, 0.97643720, 0.98953970, 0.44400767],
                             [- 3.73427682, -0.00671096, 0.00371419, 4.72756585, 0.97650107, 0.98957485, 0.44388975],
                             [- 3.73898550, -0.00669059, 0.00369976, 4.73229491, 0.97656446, 0.98960965, 0.44377441],
                             [- 3.74369498, -0.00667042, 0.00368549, 4.73702457, 0.97662737, 0.98964409, 0.44366163],
                             [- 3.74840519, -0.00665044, 0.00367136, 4.74175475, 0.97668980, 0.98967820, 0.44355139],
                             [- 3.75311607, -0.00663066, 0.00365738, 4.74648541, 0.97675177, 0.98971196, 0.44344364],
                             [- 3.75782754, -0.00661107, 0.00364354, 4.75121648, 0.97681327, 0.98974540, 0.44333838],
                             [- 3.76253955, -0.00659167, 0.00362984, 4.75594788, 0.97687432, 0.98977850, 0.44323557],
                             [- 3.76725202, -0.00657245, 0.00361628, 4.76067957, 0.97693492, 0.98981127, 0.44313518],
                             [- 3.77196489, -0.00655341, 0.00360286, 4.76541147, 0.97699508, 0.98984372, 0.44303720],
                             [- 3.77667809, -0.00653456, 0.00358958, 4.77014353, 0.97705479, 0.98987586, 0.44294159],
                             [- 3.78139156, -0.00651589, 0.00357643, 4.77487568, 0.97711407, 0.98990768, 0.44284833],
                             [- 3.78610525, -0.00649739, 0.00356342, 4.77960786, 0.97717292, 0.98993920, 0.44275740],
                             [- 3.79081907, -0.00647906, 0.00355053, 4.78434001, 0.97723134, 0.98997040, 0.44266877],
                             [- 3.79553298, -0.00646091, 0.00353778, 4.78907207, 0.97728935, 0.99000131, 0.44258241],
                             [- 3.80024690, -0.00644292, 0.00352516, 4.79380398, 0.97734694, 0.99003192, 0.44249831],
                             [- 3.80496078, -0.00642510, 0.00351266, 4.79853567, 0.97740413, 0.99006223, 0.44241644],
                             [- 3.80967455, -0.00640745, 0.00350029, 4.80326710, 0.97746090, 0.99009226, 0.44233677],
                             [- 3.81438815, -0.00638996, 0.00348804, 4.80799819, 0.97751728, 0.99012200, 0.44225928],
                             [- 3.81910152, -0.00637262, 0.00347592, 4.81272889, 0.97757326, 0.99015145, 0.44218395],
                             [- 3.82381460, -0.00635545, 0.00346392, 4.81745915, 0.97762886, 0.99018063, 0.44211076],
                             [- 3.82852732, -0.00633843, 0.00345204, 4.82218889, 0.97768406, 0.99020953, 0.44203968],
                             [- 3.83323964, -0.00632157, 0.00344027, 4.82691808, 0.97773889, 0.99023816, 0.44197068],
                             [- 3.83795149, -0.00630485, 0.00342863, 4.83164664, 0.97779333, 0.99026652, 0.44190376],
                             [- 3.84266280, -0.00628829, 0.00341709, 4.83637452, 0.97784741, 0.99029462, 0.44183887],
                             [- 3.84737353, -0.00627187, 0.00340568, 4.84110166, 0.97790111, 0.99032245, 0.44177601],
                             [- 3.85208361, -0.00625560, 0.00339437, 4.84582801, 0.97795446, 0.99035003, 0.44171515],
                             [- 3.85679299, -0.00623948, 0.00338318, 4.85055351, 0.97800744, 0.99037735, 0.44165627],
                             [- 3.86150160, -0.00622349, 0.00337210, 4.85527811, 0.97806006, 0.99040441, 0.44159934],
                             [- 3.86620939, -0.00620765, 0.00336112, 4.86000175, 0.97811233, 0.99043123, 0.44154435],
                             [- 3.87091631, -0.00619194, 0.00335026, 4.86472437, 0.97816426, 0.99045780, 0.44149127],
                             [- 3.87562229, -0.00617637, 0.00333950, 4.86944592, 0.97821584, 0.99048413, 0.44144009],
                             [- 3.88032729, -0.00616094, 0.00332885, 4.87416635, 0.97826708, 0.99051022, 0.44139078],
                             [- 3.88503124, -0.00614564, 0.00331830, 4.87888561, 0.97831798, 0.99053607, 0.44134332],
                             [- 3.88973410, -0.00613047, 0.00330785, 4.88360363, 0.97836855, 0.99056168, 0.44129769],
                             [- 3.89443580, -0.00611543, 0.00329750, 4.88832037, 0.97841879, 0.99058707, 0.44125387],
                             [- 3.89913629, -0.00610051, 0.00328726, 4.89303577, 0.97846870, 0.99061223, 0.44121185],
                             [- 3.90383552, -0.00608573, 0.00327711, 4.89774979, 0.97851829, 0.99063716, 0.44117159],
                             [- 3.90853343, -0.00607107, 0.00326707, 4.90246236, 0.97856756, 0.99066187, 0.44113309],
                             [- 3.91322998, -0.00605653, 0.00325712, 4.90717345, 0.97861652, 0.99068635, 0.44109632],
                             [- 3.91792511, -0.00604212, 0.00324726, 4.91188299, 0.97866516, 0.99071062, 0.44106126],
                             [- 3.92261876, -0.00602782, 0.00323750, 4.91659094, 0.97871350, 0.99073468, 0.44102790],
                             [- 3.92731089, -0.00601364, 0.00322784, 4.92129724, 0.97876153, 0.99075852, 0.44099621],
                             [- 3.93200144, -0.00599958, 0.00321826, 4.92600186, 0.97880926, 0.99078215, 0.44096618],
                             [- 3.93669036, -0.00598564, 0.00320878, 4.93070472, 0.97885669, 0.99080558, 0.44093779],
                             [- 3.94137761, -0.00597181, 0.00319939, 4.93540580, 0.97890383, 0.99082880, 0.44091101],
                             [- 3.94606313, -0.00595809, 0.00319009, 4.94010504, 0.97895067, 0.99085182, 0.44088584],
                             [- 3.95074687, -0.00594449, 0.00318088, 4.94480238, 0.97899722, 0.99087463, 0.44086225],
                             [- 3.95542878, -0.00593099, 0.00317175, 4.94949779, 0.97904349, 0.99089725, 0.44084022],
                             [- 3.96010882, -0.00591761, 0.00316271, 4.95419121, 0.97908947, 0.99091968, 0.44081975],
                             [- 3.96478693, -0.00590433, 0.00315376, 4.95888260, 0.97913517, 0.99094191, 0.44080080],
                             [- 3.96946306, -0.00589116, 0.00314489, 4.96357191, 0.97918060, 0.99096395, 0.44078336],
                             [- 3.97413718, -0.00587809, 0.00313611, 4.96825909, 0.97922575, 0.99098581, 0.44076742],
                             [- 3.97880922, -0.00586512, 0.00312740, 4.97294410, 0.97927063, 0.99100747, 0.44075296],
                             [- 3.98347915, -0.00585226, 0.00311878, 4.97762689, 0.97931524, 0.99102895, 0.44073996],
                             [- 3.98814692, -0.00583950, 0.00311024, 4.98230742, 0.97935959, 0.99105026, 0.44072841],
                             [- 3.99281247, -0.00582684, 0.00310178, 4.98698564, 0.97940367, 0.99107138, 0.44071828],
                             [- 3.99747577, -0.00581428, 0.00309340, 4.99166150, 0.97944749, 0.99109232, 0.44070956],
                             [- 4.00213677, -0.00580181, 0.00308510, 4.99633496, 0.97949105, 0.99111309, 0.44070224],
                             [- 4.00679542, -0.00578944, 0.00307688, 5.00100598, 0.97953436, 0.99113368, 0.44069630],
                             [- 4.01145168, -0.00577717, 0.00306873, 5.00567451, 0.97957741, 0.99115410, 0.44069173],
                             [- 4.01610551, -0.00576499, 0.00306065, 5.01034052, 0.97962021, 0.99117436, 0.44068850],
                             [- 4.02075685, -0.00575290, 0.00305266, 5.01500395, 0.97966277, 0.99119444, 0.44068660],
                             [- 4.02540567, -0.00574091, 0.00304473, 5.01966476, 0.97970508, 0.99121436, 0.44068602],
                             [- 4.03005191, -0.00572900, 0.00303688, 5.02432291, 0.97974715, 0.99123412, 0.44068674],
                             [- 4.03469555, -0.00571719, 0.00302910, 5.02897837, 0.97978897, 0.99125371, 0.44068875],
                             [- 4.03933654, -0.00570546, 0.00302139, 5.03363108, 0.97983056, 0.99127315, 0.44069203],
                             [- 4.04397482, -0.00569382, 0.00301375, 5.03828100, 0.97987192, 0.99129242, 0.44069657],
                             [- 4.04861037, -0.00568227, 0.00300619, 5.04292810, 0.97991304, 0.99131154, 0.44070234],
                             [- 4.05324314, -0.00567080, 0.00299869, 5.04757234, 0.97995393, 0.99133051, 0.44070935],
                             [- 4.05787308, -0.00565942, 0.00299126, 5.05221367, 0.97999459, 0.99134932, 0.44071756],
                             [- 4.06250017, -0.00564812, 0.00298390, 5.05685205, 0.98003502, 0.99136799, 0.44072698],
                             [- 4.06712435, -0.00563690, 0.00297660, 5.06148744, 0.98007523, 0.99138650, 0.44073757],
                             [- 4.07174558, -0.00562577, 0.00296937, 5.06611981, 0.98011522, 0.99140486, 0.44074934],
                             [- 4.07636383, -0.00561471, 0.00296221, 5.07074912, 0.98015498, 0.99142308, 0.44076227],
                             [- 4.08097906, -0.00560374, 0.00295511, 5.07537532, 0.98019453, 0.99144116, 0.44077633],
                             [- 4.08559122, -0.00559284, 0.00294807, 5.07999838, 0.98023386, 0.99145909, 0.44079153],
                             [- 4.09020028, -0.00558202, 0.00294110, 5.08461826, 0.98027298, 0.99147688, 0.44080784],
                             [- 4.09480620, -0.00557128, 0.00293419, 5.08923492, 0.98031189, 0.99149453, 0.44082525],
                             [- 4.09940894, -0.00556061, 0.00292734, 5.09384833, 0.98035059, 0.99151204, 0.44084375],
                             [- 4.10400846, -0.00555002, 0.00292056, 5.09845844, 0.98038908, 0.99152942, 0.44086333],
                             [- 4.10860473, -0.00553950, 0.00291383, 5.10306522, 0.98042736, 0.99154666, 0.44088396],
                             [- 4.11319770, -0.00552906, 0.00290717, 5.10766864, 0.98046544, 0.99156377, 0.44090565],
                             [- 4.11778734, -0.00551869, 0.00290056, 5.11226865, 0.98050332, 0.99158075, 0.44092838],
                             [- 4.12237362, -0.00550839, 0.00289401, 5.11686523, 0.98054100, 0.99159760, 0.44095213],
                             [- 4.12695649, -0.00549816, 0.00288752, 5.12145833, 0.98057848, 0.99161431, 0.44097689],
                             [- 4.13153592, -0.00548801, 0.00288109, 5.12604792, 0.98061577, 0.99163090, 0.44100265],
                             [- 4.13611188, -0.00547792, 0.00287471, 5.13063396, 0.98065286, 0.99164737, 0.44102940],
                             [- 4.14068433, -0.00546790, 0.00286839, 5.13521643, 0.98068975, 0.99166371, 0.44105712],
                             [- 4.14525323, -0.00545795, 0.00286213, 5.13979528, 0.98072646, 0.99167992, 0.44108581],
                             [- 4.14981854, -0.00544806, 0.00285592, 5.14437048, 0.98076298, 0.99169602, 0.44111544],
                             [- 4.15438025, -0.00543824, 0.00284976, 5.14894200, 0.98079931, 0.99171199, 0.44114602],
                             [- 4.15893830, -0.00542849, 0.00284366, 5.15350981, 0.98083545, 0.99172785, 0.44117753],
                             [- 4.16349267, -0.00541880, 0.00283761, 5.15807386, 0.98087141, 0.99174358, 0.44120995],
                             [- 4.16804332, -0.00540918, 0.00283162, 5.16263414, 0.98090719, 0.99175920, 0.44124328],
                             [- 4.17259021, -0.00539962, 0.00282567, 5.16719060, 0.98094278, 0.99177471, 0.44127750],
                             [- 4.17713333, -0.00539012, 0.00281978, 5.17174321, 0.98097820, 0.99179010, 0.44131260],
                             [- 4.18167262, -0.00538069, 0.00281394, 5.17629194, 0.98101344, 0.99180537, 0.44134857],
                             [- 4.18620807, -0.00537131, 0.00280815, 5.18083676, 0.98104851, 0.99182054, 0.44138541],
                             [- 4.19073963, -0.00536200, 0.00280241, 5.18537763, 0.98108340, 0.99183560, 0.44142309],
                             [- 4.19526728, -0.00535274, 0.00279671, 5.18991453, 0.98111811, 0.99185054, 0.44146162],
                             [- 4.19979098, -0.00534355, 0.00279107, 5.19444743, 0.98115266, 0.99186538, 0.44150097],
                             [- 4.20431070, -0.00533441, 0.00278548, 5.19897629, 0.98118704, 0.99188011, 0.44154114],
                             [- 4.20882641, -0.00532534, 0.00277993, 5.20350108, 0.98122125, 0.99189474, 0.44158211],
                             [- 4.21333809, -0.00531632, 0.00277443, 5.20802177, 0.98125529, 0.99190926, 0.44162389],
                             [- 4.21784569, -0.00530735, 0.00276897, 5.21253834, 0.98128916, 0.99192367, 0.44166645],
                             [- 4.22234919, -0.00529845, 0.00276357, 5.21705075, 0.98132288, 0.99193799, 0.44170979],
                             [- 4.22684856, -0.00528959, 0.00275820, 5.22155897, 0.98135643, 0.99195220, 0.44175389],
                             [- 4.23134377, -0.00528080, 0.00275289, 5.22606297, 0.98138982, 0.99196631, 0.44179875],
                             [- 4.23583479, -0.00527206, 0.00274762, 5.23056273, 0.98142305, 0.99198033, 0.44184436],
                             [- 4.24032159, -0.00526337, 0.00274239, 5.23505822, 0.98145612, 0.99199424, 0.44189070],
                             [- 4.24480413, -0.00525473, 0.00273720, 5.23954940, 0.98148904, 0.99200806, 0.44193777],
                             [- 4.24928241, -0.00524615, 0.00273206, 5.24403626, 0.98152180, 0.99202179, 0.44198557],
                             [- 4.25375637, -0.00523762, 0.00272697, 5.24851875, 0.98155440, 0.99203541, 0.44203406],
                             [- 4.25822600, -0.00522914, 0.00272191, 5.25299686, 0.98158685, 0.99204895, 0.44208326],
                             [- 4.26269127, -0.00522071, 0.00271690, 5.25747055, 0.98161916, 0.99206239, 0.44213315],
                             [- 4.26715214, -0.00521233, 0.00271193, 5.26193981, 0.98165131, 0.99207574, 0.44218372],
                             [- 4.27160860, -0.00520401, 0.00270700, 5.26640459, 0.98168331, 0.99208900, 0.44223496],
                             [- 4.27606061, -0.00519573, 0.00270211, 5.27086489, 0.98171516, 0.99210217, 0.44228686],
                             [- 4.28050816, -0.00518750, 0.00269726, 5.27532066, 0.98174687, 0.99211524, 0.44233942],
                             [- 4.28495120, -0.00517932, 0.00269245, 5.27977188, 0.98177844, 0.99212824, 0.44239262],
                             [- 4.28938971, -0.00517119, 0.00268767, 5.28421853, 0.98180986, 0.99214114, 0.44244646],
                             [- 4.29382368, -0.00516310, 0.00268294, 5.28866058, 0.98184113, 0.99215396, 0.44250093],
                             [- 4.29825306, -0.00515506, 0.00267825, 5.29309800, 0.98187227, 0.99216669, 0.44255601],
                             [- 4.30267785, -0.00514707, 0.00267359, 5.29753078, 0.98190326, 0.99217934, 0.44261171],
                             [- 4.30709800, -0.00513912, 0.00266898, 5.30195888, 0.98193412, 0.99219190, 0.44266801],
                             [- 4.31151350, -0.00513122, 0.00266440, 5.30638227, 0.98196483, 0.99220438, 0.44272490],
                             [- 4.31592431, -0.00512337, 0.00265985, 5.31080095, 0.98199542, 0.99221678, 0.44278238],
                             [- 4.32033043, -0.00511555, 0.00265535, 5.31521487, 0.98202586, 0.99222910, 0.44284044],
                             [- 4.32473181, -0.00510779, 0.00265088, 5.31962403, 0.98205617, 0.99224134, 0.44289907],
                             [- 4.32912844, -0.00510006, 0.00264644, 5.32402838, 0.98208635, 0.99225350, 0.44295825],
                             [- 4.33352030, -0.00509238, 0.00264204, 5.32842792, 0.98211639, 0.99226558, 0.44301800],
                             [- 4.33790735, -0.00508474, 0.00263768, 5.33282261, 0.98214630, 0.99227758, 0.44307829],
                             [- 4.34228957, -0.00507715, 0.00263335, 5.33721243, 0.98217609, 0.99228950, 0.44313911],
                             [- 4.34666695, -0.00506959, 0.00262906, 5.34159736, 0.98220574, 0.99230135, 0.44320047],
                             [- 4.35103946, -0.00506208, 0.00262479, 5.34597738, 0.98223526, 0.99231313, 0.44326235],
                             [- 4.35540706, -0.00505461, 0.00262057, 5.35035246, 0.98226466, 0.99232483, 0.44332475],
                             [- 4.35976976, -0.00504718, 0.00261637, 5.35472258, 0.98229393, 0.99233645, 0.44338766],
                             [- 4.36412751, -0.00503978, 0.00261221, 5.35908772, 0.98232308, 0.99234800, 0.44345107],
                             [- 4.36848029, -0.00503243, 0.00260808, 5.36344786, 0.98235210, 0.99235949, 0.44351497],
                             [- 4.37282810, -0.00502512, 0.00260399, 5.36780298, 0.98238100, 0.99237089, 0.44357936],
                             [- 4.37717089, -0.00501785, 0.00259992, 5.37215304, 0.98240977, 0.99238223, 0.44364422],
                             [- 4.38150866, -0.00501061, 0.00259589, 5.37649804, 0.98243843, 0.99239350, 0.44370956],
                             [- 4.38584137, -0.00500341, 0.00259189, 5.38083796, 0.98246696, 0.99240470, 0.44377537],
                             [- 4.39016901, -0.00499626, 0.00258792, 5.38517276, 0.98249538, 0.99241583, 0.44384163],
                             [- 4.39449156, -0.00498913, 0.00258397, 5.38950243, 0.98252368, 0.99242689, 0.44390835],
                             [- 4.39880900, -0.00498205, 0.00258006, 5.39382695, 0.98255186, 0.99243789, 0.44397551],
                             [- 4.40312130, -0.00497500, 0.00257618, 5.39814630, 0.98257992, 0.99244881, 0.44404311],
                             [- 4.40742845, -0.00496799, 0.00257233, 5.40246046, 0.98260787, 0.99245968, 0.44411114],
                             [- 4.41173042, -0.00496101, 0.00256851, 5.40676940, 0.98263570, 0.99247047, 0.44417960],
                             [- 4.41602719, -0.00495407, 0.00256472, 5.41107312, 0.98266342, 0.99248121, 0.44424847],
                             [- 4.42031875, -0.00494717, 0.00256096, 5.41537158, 0.98269102, 0.99249187, 0.44431776],
                             [- 4.42460508, -0.00494030, 0.00255722, 5.41966478, 0.98271852, 0.99250248, 0.44438745],
                             [- 4.42888615, -0.00493346, 0.00255351, 5.42395268, 0.98274590, 0.99251302, 0.44445755],
                             [- 4.43316194, -0.00492666, 0.00254984, 5.42823528, 0.98277317, 0.99252350, 0.44452803],
                             [- 4.43743244, -0.00491989, 0.00254618, 5.43251255, 0.98280033, 0.99253392, 0.44459891],
                             [- 4.44169763, -0.00491316, 0.00254256, 5.43678447, 0.98282738, 0.99254428, 0.44467016],
                             [- 4.44595749, -0.00490646, 0.00253896, 5.44105103, 0.98285433, 0.99255458, 0.44474179],
                             [- 4.45021200, -0.00489979, 0.00253539, 5.44531221, 0.98288117, 0.99256482, 0.44481379],
                             [- 4.45446115, -0.00489316, 0.00253185, 5.44956799, 0.98290790, 0.99257500, 0.44488616],
                             [- 4.45870490, -0.00488655, 0.00252833, 5.45381835, 0.98293452, 0.99258512, 0.44495888],
                             [- 4.46294326, -0.00487998, 0.00252484, 5.45806327, 0.98296105, 0.99259518, 0.44503195],
                             [- 4.46717619, -0.00487344, 0.00252137, 5.46230275, 0.98298746, 0.99260519, 0.44510537],
                             [- 4.47140368, -0.00486693, 0.00251793, 5.46653675, 0.98301378, 0.99261514, 0.44517912],
                             [- 4.47562571, -0.00486046, 0.00251451, 5.47076526, 0.98303999, 0.99262503, 0.44525322],
                             [- 4.47984227, -0.00485401, 0.00251112, 5.47498827, 0.98306610, 0.99263487, 0.44532764],
                             [- 4.48405334, -0.00484759, 0.00250775, 5.47920575, 0.98309211, 0.99264465, 0.44540238],
                             [- 4.48825891, -0.00484121, 0.00250441, 5.48341770, 0.98311802, 0.99265438, 0.44547744],
                             [- 4.49245894, -0.00483485, 0.00250109, 5.48762409, 0.98314383, 0.99266406, 0.44555282],
                             [- 4.49665344, -0.00482852, 0.00249780, 5.49182492, 0.98316954, 0.99267368, 0.44562850],
                             [- 4.50084238, -0.00482223, 0.00249453, 5.49602015, 0.98319515, 0.99268325, 0.44570449],
                             [- 4.50502575, -0.00481596, 0.00249128, 5.50020979, 0.98322067, 0.99269277, 0.44578077],
                             [- 4.50920352, -0.00480972, 0.00248805, 5.50439380, 0.98324609, 0.99270223, 0.44585734],
                             [- 4.51337569, -0.00480350, 0.00248485, 5.50857219, 0.98327141, 0.99271164, 0.44593420],
                             [- 4.51754224, -0.00479732, 0.00248167, 5.51274492, 0.98329664, 0.99272101, 0.44601134],
                             [- 4.52170315, -0.00479117, 0.00247851, 5.51691199, 0.98332177, 0.99273032, 0.44608876],
                             [- 4.52585842, -0.00478504, 0.00247538, 5.52107338, 0.98334681, 0.99273958, 0.44616645],
                             [- 4.53000801, -0.00477894, 0.00247227, 5.52522907, 0.98337176, 0.99274880, 0.44624440],
                             [- 4.53415192, -0.00477286, 0.00246917, 5.52937906, 0.98339661, 0.99275796, 0.44632262],
                             [- 4.53829014, -0.00476682, 0.00246610, 5.53352332, 0.98342137, 0.99276708, 0.44640109],
                             [- 4.54242264, -0.00476080, 0.00246306, 5.53766184, 0.98344605, 0.99277615, 0.44647982],
                             [- 4.54654942, -0.00475480, 0.00246003, 5.54179461, 0.98347063, 0.99278517, 0.44655879],
                             [- 4.55067046, -0.00474884, 0.00245702, 5.54592162, 0.98349512, 0.99279414, 0.44663801],
                             [- 4.55478574, -0.00474290, 0.00245404, 5.55004285, 0.98351952, 0.99280307, 0.44671746],
                             [- 4.55889526, -0.00473698, 0.00245107, 5.55415828, 0.98354383, 0.99281195, 0.44679715],
                             [- 4.56299899, -0.00473109, 0.00244812, 5.55826790, 0.98356806, 0.99282079, 0.44687706],
                             [- 4.56709693, -0.00472522, 0.00244520, 5.56237170, 0.98359219, 0.99282958, 0.44695720],
                             [- 4.57118906, -0.00471938, 0.00244229, 5.56646967, 0.98361625, 0.99283832, 0.44703756],
                             [- 4.57527536, -0.00471357, 0.00243940, 5.57056179, 0.98364021, 0.99284703, 0.44711814],
                             [- 4.57935583, -0.00470778, 0.00243654, 5.57464806, 0.98366409, 0.99285569, 0.44719893],
                             [- 4.58343045, -0.00470201, 0.00243369, 5.57872844, 0.98368788, 0.99286430, 0.44727992],
                             [- 4.58749921, -0.00469627, 0.00243086, 5.58280295, 0.98371159, 0.99287287, 0.44736112],
                             [- 4.59156210, -0.00469055, 0.00242805, 5.58687155, 0.98373522, 0.99288140, 0.44744252],
                             [- 4.59561910, -0.00468485, 0.00242526, 5.59093424, 0.98375876, 0.99288989, 0.44752411],
                             [- 4.59967020, -0.00467918, 0.00242248, 5.59499102, 0.98378222, 0.99289833, 0.44760589],
                             [- 4.60371538, -0.00467353, 0.00241973, 5.59904185, 0.98380560, 0.99290674, 0.44768785],
                             [- 4.60775465, -0.00466791, 0.00241699, 5.60308674, 0.98382890, 0.99291510, 0.44777000],
                             [- 4.61178797, -0.00466230, 0.00241427, 5.60712567, 0.98385211, 0.99292343, 0.44785233],
                             [- 4.61581536, -0.00465672, 0.00241157, 5.61115863, 0.98387525, 0.99293171, 0.44793483],
                             [- 4.61983678, -0.00465116, 0.00240889, 5.61518561, 0.98389830, 0.99293995, 0.44801750],
                             [- 4.62385223, -0.00464563, 0.00240622, 5.61920660, 0.98392128, 0.99294815, 0.44810034],
                             [- 4.62786170, -0.00464011, 0.00240357, 5.62322158, 0.98394418, 0.99295632, 0.44818334],
                             [- 4.63186517, -0.00463462, 0.00240093, 5.62723055, 0.98396700, 0.99296444, 0.44826650],
                             [- 4.63586264, -0.00462915, 0.00239832, 5.63123349, 0.98398974, 0.99297253, 0.44834982],
                             [- 4.63985410, -0.00462370, 0.00239572, 5.63523040, 0.98401240, 0.99298058, 0.44843328],
                             [- 4.64383953, -0.00461827, 0.00239313, 5.63922126, 0.98403499, 0.99298859, 0.44851690],
                             [- 4.64781892, -0.00461286, 0.00239057, 5.64320606, 0.98405750, 0.99299657, 0.44860066],
                             [- 4.65179227, -0.00460748, 0.00238802, 5.64718479, 0.98407993, 0.99300451, 0.44868455],
                             [- 4.65575956, -0.00460211, 0.00238548, 5.65115744, 0.98410229, 0.99301241, 0.44876859],
                             [- 4.65972078, -0.00459677, 0.00238296, 5.65512401, 0.98412458, 0.99302027, 0.44885276],
                             [- 4.66367592, -0.00459144, 0.00238046, 5.65908448, 0.98414679, 0.99302810, 0.44893706],
                             [- 4.66762497, -0.00458614, 0.00237797, 5.66303884, 0.98416893, 0.99303590, 0.44902148],
                             [- 4.67156793, -0.00458085, 0.00237549, 5.66698708, 0.98419099, 0.99304366, 0.44910603],
                             [- 4.67550478, -0.00457558, 0.00237303, 5.67092920, 0.98421298, 0.99305138, 0.44919070],
                             [- 4.67943552, -0.00457034, 0.00237059, 5.67486518, 0.98423490, 0.99305907, 0.44927549],
                             [- 4.68336012, -0.00456511, 0.00236816, 5.67879501, 0.98425675, 0.99306673, 0.44936038],
                             [- 4.68727860, -0.00455990, 0.00236575, 5.68271869, 0.98427852, 0.99307435, 0.44944539],
                             [- 4.69119092, -0.00455472, 0.00236335, 5.68663621, 0.98430023, 0.99308194, 0.44953051],
                             [- 4.69509710, -0.00454955, 0.00236096, 5.69054755, 0.98432186, 0.99308949, 0.44961572],
                             [- 4.69899711, -0.00454440, 0.00235859, 5.69445271, 0.98434343, 0.99309701, 0.44970104],
                             [- 4.70289095, -0.00453926, 0.00235623, 5.69835168, 0.98436492, 0.99310450, 0.44978646],
                             [- 4.70677861, -0.00453415, 0.00235389, 5.70224446, 0.98438635, 0.99311196, 0.44987197],
                             [- 4.71066008, -0.00452905, 0.00235156, 5.70613103, 0.98440771, 0.99311939, 0.44995757],
                             [- 4.71453535, -0.00452398, 0.00234924, 5.71001138, 0.98442899, 0.99312678, 0.45004326],
                             [- 4.71840442, -0.00451892, 0.00234694, 5.71388551, 0.98445021, 0.99313414, 0.45012903],
                             [- 4.72226728, -0.00451387, 0.00234465, 5.71775341, 0.98447137, 0.99314147, 0.45021488],
                             [- 4.72612392, -0.00450885, 0.00234237, 5.72161507, 0.98449245, 0.99314877, 0.45030082],
                             [- 4.72997433, -0.00450384, 0.00234011, 5.72547048, 0.98451347, 0.99315604, 0.45038683],
                             [- 4.73381850, -0.00449885, 0.00233786, 5.72931964, 0.98453443, 0.99316328, 0.45047291],
                             [- 4.73765643, -0.00449388, 0.00233562, 5.73316254, 0.98455532, 0.99317049, 0.45055907],
                             [- 4.74148810, -0.00448893, 0.00233340, 5.73699917, 0.98457614, 0.99317767, 0.45064529],
                             [- 4.74531352, -0.00448399, 0.00233119, 5.74082953, 0.98459690, 0.99318483, 0.45073158],
                             [- 4.74913267, -0.00447907, 0.00232899, 5.74465360, 0.98461759, 0.99319195, 0.45081792],
                             [- 4.75294555, -0.00447416, 0.00232680, 5.74847138, 0.98463822, 0.99319904, 0.45090433],
                             [- 4.75675214, -0.00446927, 0.00232462, 5.75228287, 0.98465878, 0.99320610, 0.45099080],
                             [- 4.76055246, -0.00446440, 0.00232246, 5.75608805, 0.98467929, 0.99321314, 0.45107732],
                             [- 4.76434647, -0.00445955, 0.00232031, 5.75988693, 0.98469973, 0.99322015, 0.45116389],
                             [- 4.76813419, -0.00445471, 0.00231816, 5.76367948, 0.98472010, 0.99322713, 0.45125051],
                             [- 4.77191560, -0.00444988, 0.00231604, 5.76746572, 0.98474042, 0.99323408, 0.45133717],
                             [- 4.77569070, -0.00444507, 0.00231392, 5.77124562, 0.98476067, 0.99324101, 0.45142388],
                             [- 4.77945947, -0.00444028, 0.00231181, 5.77501919, 0.98478086, 0.99324791, 0.45151063],
                             [- 4.78322192, -0.00443550, 0.00230972, 5.77878642, 0.98480099, 0.99325478, 0.45159742],
                             [- 4.78697804, -0.00443074, 0.00230763, 5.78254730, 0.98482106, 0.99326162, 0.45168425],
                             [- 4.79072783, -0.00442600, 0.00230556, 5.78630183, 0.98484107, 0.99326844, 0.45177111],
                             [- 4.79447127, -0.00442127, 0.00230350, 5.79005000, 0.98486102, 0.99327523, 0.45185800],
                             [- 4.79820836, -0.00441655, 0.00230145, 5.79379181, 0.98488091, 0.99328200, 0.45194492],
                             [- 4.80193909, -0.00441185, 0.00229941, 5.79752724, 0.98490074, 0.99328874, 0.45203187],
                             [- 4.80566347, -0.00440716, 0.00229738, 5.80125630, 0.98492051, 0.99329546, 0.45211884],
                             [- 4.80938148, -0.00440249, 0.00229536, 5.80497899, 0.98494022, 0.99330215, 0.45220583],
                             [- 4.81309312, -0.00439783, 0.00229335, 5.80869528, 0.98495988, 0.99330881, 0.45229285],
                             [- 4.81679838, -0.00439319, 0.00229135, 5.81240519, 0.98497947, 0.99331546, 0.45237988],
                             [- 4.82049726, -0.00438856, 0.00228937, 5.81610870, 0.98499901, 0.99332207, 0.45246692],
                             [- 4.82418976, -0.00438395, 0.00228739, 5.81980581, 0.98501850, 0.99332866, 0.45255398],
                             [- 4.82787587, -0.00437935, 0.00228542, 5.82349652, 0.98503792, 0.99333523, 0.45264105],
                             [- 4.83155558, -0.00437476, 0.00228346, 5.82718082, 0.98505729, 0.99334178, 0.45272813],
                             [- 4.83522889, -0.00437019, 0.00228151, 5.83085870, 0.98507660, 0.99334830, 0.45281521],
                             [- 4.83889580, -0.00436563, 0.00227957, 5.83453017, 0.98509586, 0.99335480, 0.45290231],
                             [- 4.84255630, -0.00436109, 0.00227764, 5.83819521, 0.98511506, 0.99336127, 0.45298940],
                             [- 4.84621038, -0.00435656, 0.00227572, 5.84185383, 0.98513421, 0.99336772, 0.45307649],
                             [- 4.84985805, -0.00435204, 0.00227381, 5.84550601, 0.98515330, 0.99337415, 0.45316358],
                             [- 4.85349930, -0.00434753, 0.00227191, 5.84915177, 0.98517233, 0.99338056, 0.45325067],
                             [- 4.85713412, -0.00434304, 0.00227002, 5.85279108, 0.98519132, 0.99338694, 0.45333775],
                             [- 4.86076252, -0.00433856, 0.00226813, 5.85642396, 0.98521024, 0.99339330, 0.45342482],
                             [- 4.86438448, -0.00433410, 0.00226626, 5.86005038, 0.98522912, 0.99339964, 0.45351189],
                             [- 4.86800001, -0.00432965, 0.00226439, 5.86367036, 0.98524794, 0.99340596, 0.45359894],
                             [- 4.87160909, -0.00432521, 0.00226253, 5.86728389, 0.98526671, 0.99341226, 0.45368598],
                             [- 4.87521174, -0.00432078, 0.00226069, 5.87089096, 0.98528542, 0.99341853, 0.45377301],
                             [- 4.87880793, -0.00431637, 0.00225885, 5.87449157, 0.98530409, 0.99342479, 0.45386001],
                             [- 4.88239768, -0.00431196, 0.00225701, 5.87808572, 0.98532270, 0.99343102, 0.45394700],
                             [- 4.88598098, -0.00430758, 0.00225519, 5.88167340, 0.98534126, 0.99343723, 0.45403397],
                             [- 4.88955781, -0.00430320, 0.00225338, 5.88525462, 0.98535976, 0.99344342, 0.45412092],
                             [- 4.89312819, -0.00429883, 0.00225157, 5.88882936, 0.98537822, 0.99344960, 0.45420784],
                             [- 4.89669211, -0.00429448, 0.00224977, 5.89239763, 0.98539662, 0.99345575, 0.45429473],
                             [- 4.90024957, -0.00429014, 0.00224798, 5.89595942, 0.98541498, 0.99346188, 0.45438160],
                             [- 4.90380055, -0.00428581, 0.00224620, 5.89951474, 0.98543328, 0.99346799, 0.45446844],
                             [- 4.90734507, -0.00428150, 0.00224442, 5.90306357, 0.98545154, 0.99347408, 0.45455524],
                             [- 4.91088312, -0.00427719, 0.00224266, 5.90660592, 0.98546974, 0.99348015, 0.45464201],
                             [- 4.91441469, -0.00427290, 0.00224090, 5.91014179, 0.98548790, 0.99348620, 0.45472875],
                             [- 4.91793978, -0.00426862, 0.00223915, 5.91367116, 0.98550600, 0.99349224, 0.45481546],
                             [- 4.92145840, -0.00426435, 0.00223740, 5.91719405, 0.98552406, 0.99349825, 0.45490212],
                             [- 4.92497053, -0.00426009, 0.00223566, 5.92071044, 0.98554206, 0.99350425, 0.45498875],
                             [- 4.92847618, -0.00425584, 0.00223394, 5.92422034, 0.98556002, 0.99351022, 0.45507533],
                             [- 4.93197535, -0.00425160, 0.00223221, 5.92772375, 0.98557793, 0.99351618, 0.45516187],
                             [- 4.93546803, -0.00424738, 0.00223050, 5.93122065, 0.98559579, 0.99352212, 0.45524837],
                             [- 4.93895423, -0.00424317, 0.00222879, 5.93471106, 0.98561361, 0.99352804, 0.45533482],
                             [- 4.94243393, -0.00423896, 0.00222709, 5.93819497, 0.98563138, 0.99353395, 0.45542123],
                             [- 4.94590714, -0.00423477, 0.00222540, 5.94167237, 0.98564910, 0.99353983, 0.45550758],
                             [- 4.94937386, -0.00423059, 0.00222371, 5.94514327, 0.98566677, 0.99354570, 0.45559389],
                             [- 4.95283409, -0.00422642, 0.00222203, 5.94860767, 0.98568439, 0.99355155, 0.45568014],
                             [- 4.95628782, -0.00422226, 0.00222036, 5.95206556, 0.98570197, 0.99355738, 0.45576634],
                             [- 4.95973505, -0.00421811, 0.00221869, 5.95551694, 0.98571951, 0.99356320, 0.45585249],
                             [- 4.96317579, -0.00421397, 0.00221703, 5.95896181, 0.98573699, 0.99356900, 0.45593858],
                             [- 4.96661002, -0.00420984, 0.00221538, 5.96240018, 0.98575444, 0.99357478, 0.45602462],
                             [- 4.97003776, -0.00420573, 0.00221373, 5.96583204, 0.98577183, 0.99358054, 0.45611059],
                             [- 4.97345900, -0.00420162, 0.00221209, 5.96925738, 0.98578918, 0.99358629, 0.45619651],
                             [- 4.97687374, -0.00419752, 0.00221046, 5.97267622, 0.98580649, 0.99359202, 0.45628236],
                             [- 4.98028197, -0.00419344, 0.00220883, 5.97608854, 0.98582375, 0.99359774, 0.45636816],
                             [- 4.98368371, -0.00418936, 0.00220721, 5.97949435, 0.98584096, 0.99360343, 0.45645388],
                             [- 4.98707894, -0.00418529, 0.00220559, 5.98289365, 0.98585814, 0.99360912, 0.45653955],
                             [- 4.99046767, -0.00418123, 0.00220398, 5.98628643, 0.98587526, 0.99361478, 0.45662514],
                             [- 4.99384989, -0.00417719, 0.00220238, 5.98967270, 0.98589235, 0.99362043, 0.45671067],
                             [- 4.99722561, -0.00417315, 0.00220078, 5.99305246, 0.98590939, 0.99362607, 0.45679613],
                             [- 5.00059483, -0.00416912, 0.00219919, 5.99642571, 0.98592638, 0.99363169, 0.45688152],
                             [- 5.00395754, -0.00416511, 0.00219761, 5.99979244, 0.98594334, 0.99363729, 0.45696684],
                             [- 5.00731375, -0.00416110, 0.00219603, 6.00315266, 0.98596025, 0.99364288, 0.45705209],
                             [- 5.01066346, -0.00415710, 0.00219445, 6.00650636, 0.98597712, 0.99364845, 0.45713726],
                             [- 5.01400667, -0.00415311, 0.00219288, 6.00985356, 0.98599394, 0.99365401, 0.45722236],
                             [- 5.01734337, -0.00414913, 0.00219132, 6.01319424, 0.98601072, 0.99365955, 0.45730738],
                             [- 5.02067356, -0.00414516, 0.00218976, 6.01652841, 0.98602746, 0.99366508, 0.45739233],
                             [- 5.02399726, -0.00414120, 0.00218821, 6.01985606, 0.98604416, 0.99367059, 0.45747719],
                             [- 5.02731445, -0.00413724, 0.00218666, 6.02317721, 0.98606082, 0.99367609, 0.45756198],
                             [- 5.03062515, -0.00413330, 0.00218512, 6.02649184, 0.98607744, 0.99368157, 0.45764669],
                             [- 5.03392934, -0.00412937, 0.00218359, 6.02979997, 0.98609401, 0.99368704, 0.45773131],
                             [- 5.03722703, -0.00412544, 0.00218206, 6.03310159, 0.98611054, 0.99369250, 0.45781585],
                             [- 5.04051822, -0.00412153, 0.00218053, 6.03639670, 0.98612704, 0.99369794, 0.45790031],
                             [- 5.04380292, -0.00411762, 0.00217901, 6.03968530, 0.98614349, 0.99370337, 0.45798469],
                             [- 5.04708112, -0.00411372, 0.00217750, 6.04296739, 0.98615990, 0.99370878, 0.45806897],
                             [- 5.05035282, -0.00410983, 0.00217599, 6.04624299, 0.98617627, 0.99371418, 0.45815318],
                             [- 5.05361802, -0.00410595, 0.00217448, 6.04951207, 0.98619260, 0.99371957, 0.45823729],
                             [- 5.05687674, -0.00410208, 0.00217298, 6.05277466, 0.98620889, 0.99372494, 0.45832131],
                             [- 5.06012896, -0.00409821, 0.00217148, 6.05603074, 0.98622514, 0.99373030, 0.45840525],
                             [- 5.06337469, -0.00409436, 0.00216999, 6.05928033, 0.98624135, 0.99373565, 0.45848909],
                             [- 5.06661393, -0.00409051, 0.00216851, 6.06252341, 0.98625753, 0.99374098, 0.45857285],
                             [- 5.06984668, -0.00408668, 0.00216703, 6.06576000, 0.98627366, 0.99374630, 0.45865651],
                             [- 5.07307294, -0.00408285, 0.00216555, 6.06899010, 0.98628975, 0.99375161, 0.45874007],
                             [- 5.07629272, -0.00407902, 0.00216408, 6.07221370, 0.98630581, 0.99375690, 0.45882355],
                             [- 5.07950602, -0.00407521, 0.00216261, 6.07543081, 0.98632182, 0.99376218, 0.45890693],
                             [- 5.08271283, -0.00407141, 0.00216115, 6.07864143, 0.98633780, 0.99376745, 0.45899021],
                             [- 5.08591317, -0.00406761, 0.00215969, 6.08184556, 0.98635374, 0.99377270, 0.45907339],
                             [- 5.08910703, -0.00406382, 0.00215823, 6.08504321, 0.98636965, 0.99377795, 0.45915648],
                             [- 5.09229441, -0.00406004, 0.00215678, 6.08823437, 0.98638551, 0.99378318, 0.45923947],
                             [- 5.09547532, -0.00405627, 0.00215534, 6.09141905, 0.98640134, 0.99378840, 0.45932235],
                             [- 5.09864975, -0.00405250, 0.00215390, 6.09459725, 0.98641713, 0.99379360, 0.45940514],
                             [- 5.10181772, -0.00404874, 0.00215246, 6.09776897, 0.98643288, 0.99379880, 0.45948783],
                             [- 5.10497922, -0.00404500, 0.00215102, 6.10093422, 0.98644859, 0.99380398, 0.45957041],
                             [- 5.10813425, -0.00404125, 0.00214960, 6.10409300, 0.98646427, 0.99380915, 0.45965289],
                             [- 5.11128282, -0.00403752, 0.00214817, 6.10724530, 0.98647991, 0.99381431, 0.45973527],
                             [- 5.11442494, -0.00403379, 0.00214675, 6.11039114, 0.98649552, 0.99381946, 0.45981755],
                             [- 5.11756059, -0.00403008, 0.00214533, 6.11353051, 0.98651108, 0.99382459, 0.45989972],
                             [- 5.12068979, -0.00402637, 0.00214392, 6.11666343, 0.98652662, 0.99382971, 0.45998178],
                             [- 5.12381254, -0.00402266, 0.00214251, 6.11978988, 0.98654211, 0.99383483, 0.46006373],
                             [- 5.12692884, -0.00401897, 0.00214110, 6.12290987, 0.98655757, 0.99383993, 0.46014558],
                             [- 5.13003869, -0.00401528, 0.00213970, 6.12602341, 0.98657300, 0.99384502, 0.46022732],
                             [- 5.13314210, -0.00401160, 0.00213831, 6.12913050, 0.98658838, 0.99385010, 0.46030896],
                             [- 5.13623906, -0.00400792, 0.00213691, 6.13223114, 0.98660374, 0.99385516, 0.46039048],
                             [- 5.13932959, -0.00400426, 0.00213552, 6.13532533, 0.98661906, 0.99386022, 0.46047189],
                             [- 5.14241368, -0.00400060, 0.00213413, 6.13841308, 0.98663434, 0.99386527, 0.46055319],
                             [- 5.14549135, -0.00399695, 0.00213275, 6.14149440, 0.98664959, 0.99387030, 0.46063438],
                             [- 5.14856258, -0.00399330, 0.00213137, 6.14456928, 0.98666480, 0.99387532, 0.46071546],
                             [- 5.15162739, -0.00398967, 0.00213000, 6.14763772, 0.98667998, 0.99388034, 0.46079643],
                             [- 5.15468577, -0.00398604, 0.00212862, 6.15069974, 0.98669512, 0.99388534, 0.46087728],
                             [- 5.15773774, -0.00398241, 0.00212725, 6.15375533, 0.98671023, 0.99389033, 0.46095802],
                             [- 5.16078329, -0.00397880, 0.00212589, 6.15680449, 0.98672531, 0.99389532, 0.46103864],
                             [- 5.16382243, -0.00397519, 0.00212453, 6.15984724, 0.98674035, 0.99390029, 0.46111915],
                             [- 5.16685516, -0.00397159, 0.00212317, 6.16288358, 0.98675535, 0.99390525, 0.46119954],
                             [- 5.16988149, -0.00396799, 0.00212181, 6.16591350, 0.98677033, 0.99391020, 0.46127982],
                             [- 5.17290141, -0.00396440, 0.00212046, 6.16893701, 0.98678527, 0.99391514, 0.46135997],
                             [- 5.17591494, -0.00396082, 0.00211911, 6.17195412, 0.98680017, 0.99392007, 0.46144001],
                             [- 5.17892208, -0.00395724, 0.00211776, 6.17496483, 0.98681505, 0.99392499, 0.46151994],
                             [- 5.18192282, -0.00395368, 0.00211642, 6.17796914, 0.98682989, 0.99392990, 0.46159974],
                             [- 5.18491718, -0.00395011, 0.00211508, 6.18096706, 0.98684470, 0.99393481, 0.46167943],
                             [- 5.18790516, -0.00394656, 0.00211374, 6.18395860, 0.98685947, 0.99393970, 0.46175899],
                             [- 5.19088675, -0.00394301, 0.00211241, 6.18694374, 0.98687421, 0.99394458, 0.46183843],
                             [- 5.19386198, -0.00393947, 0.00211108, 6.18992251, 0.98688892, 0.99394945, 0.46191776],
                             [- 5.19683083, -0.00393593, 0.00210975, 6.19289490, 0.98690360, 0.99395432, 0.46199696],
                             [- 5.19979332, -0.00393241, 0.00210843, 6.19586092, 0.98691824, 0.99395917, 0.46207604],
                             [- 5.20274945, -0.00392888, 0.00210710, 6.19882057, 0.98693285, 0.99396401, 0.46215500],
                             [- 5.20569922, -0.00392537, 0.00210578, 6.20177385, 0.98694743, 0.99396885, 0.46223383],
                             [- 5.20864264, -0.00392186, 0.00210447, 6.20472078, 0.98696198, 0.99397367, 0.46231254],
                             [- 5.21157971, -0.00391835, 0.00210315, 6.20766135, 0.98697650, 0.99397849, 0.46239113],
                             [- 5.21451043, -0.00391486, 0.00210184, 6.21059558, 0.98699098, 0.99398330, 0.46246959],
                             [- 5.21743482, -0.00391137, 0.00210054, 6.21352345, 0.98700544, 0.99398810, 0.46254793],
                             [- 5.22035287, -0.00390788, 0.00209923, 6.21644499, 0.98701986, 0.99399289, 0.46262614],
                             [- 5.22326459, -0.00390440, 0.00209793, 6.21936019, 0.98703425, 0.99399767, 0.46270423],
                             [- 5.22616999, -0.00390093, 0.00209663, 6.22226905, 0.98704861, 0.99400244, 0.46278219],
                             [- 5.22906906, -0.00389747, 0.00209533, 6.22517159, 0.98706294, 0.99400720, 0.46286003],
                             [- 5.23196182, -0.00389401, 0.00209404, 6.22806781, 0.98707724, 0.99401196, 0.46293774],
                             [- 5.23484826, -0.00389055, 0.00209274, 6.23095771, 0.98709151, 0.99401670, 0.46301532],
                             [- 5.23772840, -0.00388711, 0.00209145, 6.23384129, 0.98710574, 0.99402144, 0.46309277],
                             [- 5.24060224, -0.00388366, 0.00209017, 6.23671857, 0.98711995, 0.99402617, 0.46317009],
                             [- 5.24346978, -0.00388023, 0.00208888, 6.23958955, 0.98713413, 0.99403089, 0.46324729],
                             [- 5.24633103, -0.00387680, 0.00208760, 6.24245423, 0.98714827, 0.99403560, 0.46332436],
                             [- 5.24918599, -0.00387338, 0.00208632, 6.24531261, 0.98716239, 0.99404030, 0.46340129],
                             [- 5.25203467, -0.00386996, 0.00208504, 6.24816471, 0.98717648, 0.99404500, 0.46347810],
                             [- 5.25487707, -0.00386655, 0.00208377, 6.25101053, 0.98719053, 0.99404969, 0.46355478],
                             [- 5.25771320, -0.00386314, 0.00208250, 6.25385007, 0.98720456, 0.99405436, 0.46363133],
                             [- 5.26054307, -0.00385974, 0.00208123, 6.25668333, 0.98721856, 0.99405904, 0.46370775],
                             [- 5.26336668, -0.00385634, 0.00207996, 6.25951033, 0.98723253, 0.99406370, 0.46378403],
                             [- 5.26618402, -0.00385295, 0.00207869, 6.26233107, 0.98724646, 0.99406835, 0.46386019],
                             [- 5.26899512, -0.00384957, 0.00207743, 6.26514555, 0.98726037, 0.99407300, 0.46393621],
                             [- 5.27179998, -0.00384619, 0.00207617, 6.26795379, 0.98727425, 0.99407764, 0.46401210],
                             [- 5.27459860, -0.00384282, 0.00207491, 6.27075577, 0.98728810, 0.99408227, 0.46408786],
                             [- 5.27739098, -0.00383945, 0.00207365, 6.27355152, 0.98730193, 0.99408689, 0.46416348],
                             [- 5.28017713, -0.00383609, 0.00207240, 6.27634104, 0.98731572, 0.99409151, 0.46423898],
                             [- 5.28295706, -0.00383274, 0.00207115, 6.27912432, 0.98732949, 0.99409612, 0.46431433],
                             [- 5.28573078, -0.00382939, 0.00206989, 6.28190139, 0.98734322, 0.99410072, 0.46438956],
                             [- 5.28849828, -0.00382604, 0.00206865, 6.28467224, 0.98735693, 0.99410531, 0.46446465],
                             [- 5.29125958, -0.00382271, 0.00206740, 6.28743687, 0.98737061, 0.99410989, 0.46453961],
                             [- 5.29401468, -0.00381937, 0.00206616, 6.29019530, 0.98738426, 0.99411447, 0.46461443],
                             [- 5.29676358, -0.00381604, 0.00206491, 6.29294754, 0.98739789, 0.99411904, 0.46468912],
                             [- 5.29950630, -0.00381272, 0.00206367, 6.29569358, 0.98741148, 0.99412361, 0.46476368],
                             [- 5.30224283, -0.00380940, 0.00206243, 6.29843343, 0.98742505, 0.99412816, 0.46483809],
                             [- 5.30497319, -0.00380609, 0.00206120, 6.30116710, 0.98743859, 0.99413271, 0.46491238],
                             [- 5.30769738, -0.00380279, 0.00205996, 6.30389459, 0.98745210, 0.99413725, 0.46498652],
                             [- 5.31041540, -0.00379948, 0.00205873, 6.30661592, 0.98746559, 0.99414179, 0.46506054],
                             [- 5.31312727, -0.00379619, 0.00205750, 6.30933108, 0.98747905, 0.99414631, 0.46513441],
                             [- 5.31583299, -0.00379290, 0.00205627, 6.31204009, 0.98749248, 0.99415083, 0.46520815],
                             [- 5.31853255, -0.00378961, 0.00205504, 6.31474294, 0.98750588, 0.99415535, 0.46528175],
                             [- 5.32122598, -0.00378633, 0.00205382, 6.31743965, 0.98751926, 0.99415985, 0.46535522],
                             [- 5.32391328, -0.00378306, 0.00205259, 6.32013023, 0.98753261, 0.99416435, 0.46542855],
                             [- 5.32659445, -0.00377979, 0.00205137, 6.32281466, 0.98754593, 0.99416884, 0.46550174],
                             [- 5.32926950, -0.00377652, 0.00205015, 6.32549298, 0.98755923, 0.99417333, 0.46557479],
                             [- 5.33193843, -0.00377326, 0.00204893, 6.32816517, 0.98757249, 0.99417781, 0.46564771],
                             [- 5.33460126, -0.00377000, 0.00204772, 6.33083125, 0.98758574, 0.99418228, 0.46572049],
                             [- 5.33725798, -0.00376675, 0.00204650, 6.33349123, 0.98759895, 0.99418674, 0.46579313],
                             [- 5.33990861, -0.00376351, 0.00204529, 6.33614511, 0.98761214, 0.99419120, 0.46586563],
                             [- 5.34255316, -0.00376027, 0.00204408, 6.33879289, 0.98762531, 0.99419565, 0.46593799],
                             [- 5.34519162, -0.00375703, 0.00204287, 6.34143458, 0.98763844, 0.99420010, 0.46601022],
                             [- 5.34782400, -0.00375380, 0.00204166, 6.34407020, 0.98765155, 0.99420454, 0.46608230],
                             [- 5.35045032, -0.00375058, 0.00204045, 6.34669974, 0.98766464, 0.99420897, 0.46615425],
                             [- 5.35307057, -0.00374736, 0.00203925, 6.34932321, 0.98767770, 0.99421340, 0.46622606],
                             [- 5.35568477, -0.00374414, 0.00203804, 6.35194063, 0.98769073, 0.99421781, 0.46629773],
                             [- 5.35829292, -0.00374093, 0.00203684, 6.35455199, 0.98770374, 0.99422223, 0.46636926],
                             [- 5.36089503, -0.00373773, 0.00203564, 6.35715730, 0.98771672, 0.99422663, 0.46644065],
                             [- 5.36349110, -0.00373453, 0.00203444, 6.35975657, 0.98772968, 0.99423103, 0.46651190],
                             [- 5.36608114, -0.00373133, 0.00203324, 6.36234981, 0.98774261, 0.99423543, 0.46658301],
                             [- 5.36866517, -0.00372814, 0.00203205, 6.36493703, 0.98775552, 0.99423982, 0.46665398],
                             [- 5.37124318, -0.00372495, 0.00203085, 6.36751823, 0.98776840, 0.99424420, 0.46672482],
                             [- 5.37381518, -0.00372177, 0.00202966, 6.37009341, 0.98778125, 0.99424857, 0.46679551],
                             [- 5.37638118, -0.00371859, 0.00202847, 6.37266259, 0.98779408, 0.99425294, 0.46686606],
                             [- 5.37894119, -0.00371542, 0.00202728, 6.37522577, 0.98780689, 0.99425730, 0.46693647],
                             [- 5.38149521, -0.00371225, 0.00202609, 6.37778296, 0.98781967, 0.99426166, 0.46700674],
                             [- 5.38404325, -0.00370909, 0.00202490, 6.38033416, 0.98783243, 0.99426601, 0.46707687],
                             [- 5.38658532, -0.00370593, 0.00202371, 6.38287939, 0.98784516, 0.99427036, 0.46714686],
                             [- 5.38912142, -0.00370278, 0.00202253, 6.38541865, 0.98785786, 0.99427470, 0.46721671],
                             [- 5.39165157, -0.00369963, 0.00202134, 6.38795194, 0.98787055, 0.99427903, 0.46728642],
                             [- 5.39417576, -0.00369648, 0.00202016, 6.39047928, 0.98788320, 0.99428336, 0.46735598],
                             [- 5.39669401, -0.00369334, 0.00201898, 6.39300067, 0.98789584, 0.99428768, 0.46742541],
                             [- 5.39920633, -0.00369021, 0.00201780, 6.39551612, 0.98790845, 0.99429200, 0.46749470],
                             [- 5.40171272, -0.00368707, 0.00201662, 6.39802564, 0.98792103, 0.99429631, 0.46756384],
                             [- 5.40421318, -0.00368395, 0.00201544, 6.40052923, 0.98793359, 0.99430061, 0.46763284],
                             [- 5.40670773, -0.00368082, 0.00201427, 6.40302690, 0.98794613, 0.99430491, 0.46770170],
                             [- 5.40919637, -0.00367771, 0.00201309, 6.40551867, 0.98795864, 0.99430920, 0.46777042],
                             [- 5.41167912, -0.00367459, 0.00201192, 6.40800452, 0.98797113, 0.99431349, 0.46783900],
                             [- 5.41415597, -0.00367148, 0.00201075, 6.41048448, 0.98798360, 0.99431777, 0.46790744],
                             [- 5.41662693, -0.00366838, 0.00200957, 6.41295855, 0.98799604, 0.99432205, 0.46797574],
                             [- 5.41909202, -0.00366528, 0.00200840, 6.41542674, 0.98800846, 0.99432632, 0.46804389],
                             [- 5.42155124, -0.00366218, 0.00200724, 6.41788906, 0.98802085, 0.99433058, 0.46811190],
                             [- 5.42400460, -0.00365909, 0.00200607, 6.42034551, 0.98803323, 0.99433484, 0.46817978],
                             [- 5.42645210, -0.00365600, 0.00200490, 6.42279610, 0.98804557, 0.99433910, 0.46824751],
                             [- 5.42889376, -0.00365292, 0.00200374, 6.42524084, 0.98805790, 0.99434334, 0.46831509],
                             [- 5.43132958, -0.00364984, 0.00200257, 6.42767973, 0.98807020, 0.99434759, 0.46838254],
                             [- 5.43375956, -0.00364677, 0.00200141, 6.43011279, 0.98808248, 0.99435183, 0.46844985],
                             [- 5.43618372, -0.00364370, 0.00200024, 6.43254002, 0.98809474, 0.99435606, 0.46851701],
                             [- 5.43860207, -0.00364063, 0.00199908, 6.43496143, 0.98810697, 0.99436029, 0.46858403],
                             [- 5.44101460, -0.00363757, 0.00199792, 6.43737703, 0.98811918, 0.99436451, 0.46865091],
                             [- 5.44342134, -0.00363451, 0.00199676, 6.43978683, 0.98813137, 0.99436872, 0.46871765],
                             [- 5.44582228, -0.00363146, 0.00199561, 6.44219082, 0.98814353, 0.99437294, 0.46878424],
                             [- 5.44821744, -0.00362841, 0.00199445, 6.44458903, 0.98815567, 0.99437714, 0.46885070],
                             [- 5.45060682, -0.00362536, 0.00199329, 6.44698145, 0.98816779, 0.99438134, 0.46891701],
                             [- 5.45299043, -0.00362232, 0.00199214, 6.44936811, 0.98817989, 0.99438554, 0.46898318],
                             [- 5.45536828, -0.00361929, 0.00199098, 6.45174899, 0.98819196, 0.99438973, 0.46904921],
                             [- 5.45774037, -0.00361625, 0.00198983, 6.45412412, 0.98820402, 0.99439392, 0.46911510],
                             [- 5.46010672, -0.00361323, 0.00198868, 6.45649350, 0.98821605, 0.99439810, 0.46918084],
                             [- 5.46246734, -0.00361020, 0.00198753, 6.45885714, 0.98822805, 0.99440227, 0.46924645],
                             [- 5.46482222, -0.00360718, 0.00198638, 6.46121504, 0.98824004, 0.99440644, 0.46931191],
                             [- 5.46717138, -0.00360417, 0.00198523, 6.46356722, 0.98825200, 0.99441061, 0.46937723],
                             [- 5.46951483, -0.00360115, 0.00198408, 6.46591367, 0.98826395, 0.99441477, 0.46944241],
                             [- 5.47185257, -0.00359814, 0.00198293, 6.46825442, 0.98827587, 0.99441892, 0.46950744],
                             [- 5.47418461, -0.00359514, 0.00198178, 6.47058947, 0.98828776, 0.99442308, 0.46957234],
                             [- 5.47651097, -0.00359214, 0.00198064, 6.47291883, 0.98829964, 0.99442722, 0.46963709],
                             [- 5.47883164, -0.00358914, 0.00197949, 6.47524250, 0.98831150, 0.99443136, 0.46970170],
                             [- 5.48114664, -0.00358615, 0.00197835, 6.47756049, 0.98832333, 0.99443550, 0.46976617],
                             [- 5.48345598, -0.00358316, 0.00197721, 6.47987281, 0.98833514, 0.99443963, 0.46983050],
                             [- 5.48575966, -0.00358018, 0.00197606, 6.48217948, 0.98834693, 0.99444376, 0.46989469],
                             [- 5.48805769, -0.00357720, 0.00197492, 6.48448049, 0.98835870, 0.99444788, 0.46995873],
                             [- 5.49035007, -0.00357422, 0.00197378, 6.48677585, 0.98837045, 0.99445200, 0.47002264],
                             [- 5.49263683, -0.00357125, 0.00197264, 6.48906558, 0.98838217, 0.99445611, 0.47008640],
                             [- 5.49491797, -0.00356828, 0.00197150, 6.49134968, 0.98839388, 0.99446022, 0.47015002],
                             [- 5.49719348, -0.00356532, 0.00197036, 6.49362817, 0.98840556, 0.99446432, 0.47021350],
                             [- 5.49946339, -0.00356236, 0.00196923, 6.49590104, 0.98841723, 0.99446842, 0.47027684],
                             [- 5.50172771, -0.00355940, 0.00196809, 6.49816831, 0.98842887, 0.99447251, 0.47034004],
                             [- 5.50398643, -0.00355645, 0.00196695, 6.50042998, 0.98844049, 0.99447660, 0.47040310],
                             [- 5.50623957, -0.00355350, 0.00196582, 6.50268607, 0.98845209, 0.99448068, 0.47046602],
                             [- 5.50848713, -0.00355055, 0.00196468, 6.50493658, 0.98846367, 0.99448476, 0.47052879],
                             [- 5.51072913, -0.00354761, 0.00196355, 6.50718152, 0.98847523, 0.99448884, 0.47059143],
                             [- 5.51296557, -0.00354467, 0.00196242, 6.50942090, 0.98848677, 0.99449291, 0.47065392],
                             [- 5.51519647, -0.00354174, 0.00196129, 6.51165473, 0.98849828, 0.99449698, 0.47071627],
                             [- 5.51742182, -0.00353881, 0.00196016, 6.51388301, 0.98850978, 0.99450104, 0.47077849],
                             [- 5.51964164, -0.00353588, 0.00195902, 6.51610576, 0.98852126, 0.99450510, 0.47084056],
                             [- 5.52185594, -0.00353296, 0.00195789, 6.51832298, 0.98853271, 0.99450915, 0.47090249],
                             [- 5.52406473, -0.00353004, 0.00195677, 6.52053469, 0.98854415, 0.99451320, 0.47096428],
                             [- 5.52626800, -0.00352712, 0.00195564, 6.52274088, 0.98855556, 0.99451724, 0.47102593],
                             [- 5.52846578, -0.00352421, 0.00195451, 6.52494157, 0.98856696, 0.99452128, 0.47108744],
                             [- 5.53065808, -0.00352130, 0.00195338, 6.52713677, 0.98857834, 0.99452532, 0.47114881],
                             [- 5.53284489, -0.00351840, 0.00195226, 6.52932649, 0.98858969, 0.99452935, 0.47121004],
                             [- 5.53502623, -0.00351550, 0.00195113, 6.53151073, 0.98860103, 0.99453337, 0.47127113],
                             [- 5.53720210, -0.00351260, 0.00195001, 6.53368950, 0.98861234, 0.99453740, 0.47133208],
                             [- 5.53937252, -0.00350971, 0.00194888, 6.53586282, 0.98862364, 0.99454141, 0.47139289],
                             [- 5.54153750, -0.00350682, 0.00194776, 6.53803068, 0.98863491, 0.99454543, 0.47145356],
                             [- 5.54369704, -0.00350393, 0.00194663, 6.54019311, 0.98864617, 0.99454944, 0.47151409],
                             [- 5.54585115, -0.00350105, 0.00194551, 6.54235010, 0.98865740, 0.99455344, 0.47157449],
                             [- 5.54799984, -0.00349817, 0.00194439, 6.54450167, 0.98866862, 0.99455744, 0.47163474],
                             [- 5.55014311, -0.00349529, 0.00194327, 6.54664782, 0.98867982, 0.99456144, 0.47169485],
                             [- 5.55228099, -0.00349242, 0.00194215, 6.54878857, 0.98869099, 0.99456543, 0.47175483],
                             [- 5.55441347, -0.00348955, 0.00194103, 6.55092392, 0.98870215, 0.99456942, 0.47181466],
                             [- 5.55654057, -0.00348669, 0.00193991, 6.55305388, 0.98871329, 0.99457340, 0.47187436],
                             [- 5.55866229, -0.00348383, 0.00193879, 6.55517846, 0.98872441, 0.99457738, 0.47193391],
                             [- 5.56077864, -0.00348097, 0.00193767, 6.55729767, 0.98873551, 0.99458136, 0.47199333],
                             [- 5.56288964, -0.00347811, 0.00193655, 6.55941152, 0.98874659, 0.99458533, 0.47205261],
                             [- 5.56499528, -0.00347526, 0.00193544, 6.56152002, 0.98875765, 0.99458930, 0.47211176],
                             [- 5.56709558, -0.00347242, 0.00193432, 6.56362317, 0.98876869, 0.99459326, 0.47217076],
                             [- 5.56919055, -0.00346957, 0.00193321, 6.56572098, 0.98877972, 0.99459722, 0.47222962],
                             [- 5.57128020, -0.00346673, 0.00193209, 6.56781347, 0.98879072, 0.99460118, 0.47228835],
                             [- 5.57336453, -0.00346390, 0.00193098, 6.56990064, 0.98880171, 0.99460513, 0.47234694],
                             [- 5.57544356, -0.00346106, 0.00192986, 6.57198250, 0.98881267, 0.99460907, 0.47240539],
                             [- 5.57751729, -0.00345823, 0.00192875, 6.57405906, 0.98882362, 0.99461302, 0.47246371],
                             [- 5.57958573, -0.00345541, 0.00192764, 6.57613032, 0.98883455, 0.99461696, 0.47252188],
                             [- 5.58164889, -0.00345259, 0.00192652, 6.57819631, 0.98884546, 0.99462089, 0.47257992],
                             [- 5.58370679, -0.00344977, 0.00192541, 6.58025702, 0.98885635, 0.99462482, 0.47263782],
                             [- 5.58575942, -0.00344695, 0.00192430, 6.58231247, 0.98886722, 0.99462875, 0.47269559],
                             [- 5.58780679, -0.00344414, 0.00192319, 6.58436266, 0.98887808, 0.99463267, 0.47275321],
                             [- 5.58984893, -0.00344133, 0.00192208, 6.58640760, 0.98888891, 0.99463659, 0.47281070],
                             [- 5.59188583, -0.00343852, 0.00192097, 6.58844731, 0.98889973, 0.99464051, 0.47286806],
                             [- 5.59391750, -0.00343572, 0.00191986, 6.59048178, 0.98891053, 0.99464442, 0.47292528],
                             [- 5.59594396, -0.00343292, 0.00191875, 6.59251104, 0.98892131, 0.99464833, 0.47298236],
                             [- 5.59796521, -0.00343013, 0.00191764, 6.59453508, 0.98893207, 0.99465223, 0.47303930],
                             [- 5.59998126, -0.00342733, 0.00191653, 6.59655393, 0.98894281, 0.99465613, 0.47309611],
                             [- 5.60199212, -0.00342455, 0.00191543, 6.59856758, 0.98895354, 0.99466003, 0.47315278],
                             [- 5.60399781, -0.00342176, 0.00191432, 6.60057605, 0.98896425, 0.99466392, 0.47320932],
                             [- 5.60599831, -0.00341898, 0.00191321, 6.60257934, 0.98897493, 0.99466781, 0.47326572],
                             [- 5.60799366, -0.00341620, 0.00191211, 6.60457746, 0.98898561, 0.99467169, 0.47332198],
                             [- 5.60998385, -0.00341342, 0.00191100, 6.60657043, 0.98899626, 0.99467557, 0.47337811],
                             [- 5.61196890, -0.00341065, 0.00190990, 6.60855825, 0.98900689, 0.99467945, 0.47343411],
                             [- 5.61394881, -0.00340788, 0.00190879, 6.61054093, 0.98901751, 0.99468332, 0.47348997],
                             [- 5.61592360, -0.00340512, 0.00190769, 6.61251848, 0.98902811, 0.99468719, 0.47354569],
                             [- 5.61789327, -0.00340235, 0.00190659, 6.61449091, 0.98903869, 0.99469106, 0.47360128],
                             [- 5.61985783, -0.00339960, 0.00190548, 6.61645823, 0.98904926, 0.99469492, 0.47365674],
                             [- 5.62181729, -0.00339684, 0.00190438, 6.61842045, 0.98905980, 0.99469878, 0.47371206],
                             [- 5.62377166, -0.00339409, 0.00190328, 6.62037757, 0.98907033, 0.99470263, 0.47376724],
                             [- 5.62572095, -0.00339134, 0.00190218, 6.62232961, 0.98908084, 0.99470648, 0.47382230],
                             [- 5.62766517, -0.00338859, 0.00190108, 6.62427658, 0.98909133, 0.99471033, 0.47387721],
                             [- 5.62960432, -0.00338585, 0.00189998, 6.62621847, 0.98910181, 0.99471417, 0.47393200],
                             [- 5.63153842, -0.00338311, 0.00189888, 6.62815531, 0.98911227, 0.99471801, 0.47398665],
                             [- 5.63346748, -0.00338037, 0.00189778, 6.63008711, 0.98912271, 0.99472185, 0.47404117],
                             [- 5.63539150, -0.00337764, 0.00189668, 6.63201386, 0.98913313, 0.99472568, 0.47409555],
                             [- 5.63731050, -0.00337491, 0.00189558, 6.63393558, 0.98914354, 0.99472951, 0.47414980],
                             [- 5.63922447, -0.00337218, 0.00189448, 6.63585229, 0.98915393, 0.99473333, 0.47420392],
                             [- 5.64113344, -0.00336946, 0.00189338, 6.63776398, 0.98916430, 0.99473716, 0.47425791],
                             [- 5.64303741, -0.00336674, 0.00189229, 6.63967067, 0.98917466, 0.99474097, 0.47431176],
                             [- 5.64493639, -0.00336402, 0.00189119, 6.64157237, 0.98918499, 0.99474479, 0.47436548],
                             [- 5.64683039, -0.00336131, 0.00189009, 6.64346908, 0.98919531, 0.99474860, 0.47441907],
                             [- 5.64871942, -0.00335860, 0.00188899, 6.64536082, 0.98920562, 0.99475241, 0.47447252],
                             [- 5.65060348, -0.00335589, 0.00188790, 6.64724759, 0.98921590, 0.99475621, 0.47452585],
                             [- 5.65248259, -0.00335319, 0.00188680, 6.64912941, 0.98922617, 0.99476001, 0.47457904],
                             [- 5.65435676, -0.00335049, 0.00188571, 6.65100628, 0.98923642, 0.99476381, 0.47463210],
                             [- 5.65622600, -0.00334779, 0.00188461, 6.65287821, 0.98924666, 0.99476760, 0.47468503],
                             [- 5.65809030, -0.00334509, 0.00188352, 6.65474521, 0.98925688, 0.99477139, 0.47473783],
                             [- 5.65994970, -0.00334240, 0.00188243, 6.65660729, 0.98926708, 0.99477517, 0.47479050],
                             [- 5.66180418, -0.00333971, 0.00188133, 6.65846447, 0.98927727, 0.99477896, 0.47484303],
                             [- 5.66365377, -0.00333703, 0.00188024, 6.66031674, 0.98928744, 0.99478273, 0.47489544],
                             [- 5.66549846, -0.00333434, 0.00187915, 6.66216412, 0.98929759, 0.99478651, 0.47494772],
                             [- 5.66733828, -0.00333166, 0.00187805, 6.66400662, 0.98930772, 0.99479028, 0.47499986],
                             [- 5.66917323, -0.00332899, 0.00187696, 6.66584424, 0.98931784, 0.99479405, 0.47505187],
                             [- 5.67100331, -0.00332631, 0.00187587, 6.66767700, 0.98932794, 0.99479781, 0.47510376],
                             [- 5.67282855, -0.00332364, 0.00187478, 6.66950490, 0.98933803, 0.99480158, 0.47515552],
                             [- 5.67464894, -0.00332098, 0.00187369, 6.67132796, 0.98934810, 0.99480533, 0.47520714],
                             [- 5.67646450, -0.00331831, 0.00187260, 6.67314618, 0.98935815, 0.99480909, 0.47525864],
                             [- 5.67827523, -0.00331565, 0.00187151, 6.67495958, 0.98936819, 0.99481284, 0.47531001],
                             [- 5.68008115, -0.00331299, 0.00187042, 6.67676816, 0.98937821, 0.99481659, 0.47536125],
                             [- 5.68188226, -0.00331034, 0.00186933, 6.67857192, 0.98938821, 0.99482033, 0.47541236],
                             [- 5.68367858, -0.00330769, 0.00186824, 6.68037089, 0.98939820, 0.99482407, 0.47546334],
                             [- 5.68547011, -0.00330504, 0.00186715, 6.68216507, 0.98940817, 0.99482781, 0.47551419],
                             [- 5.68725686, -0.00330239, 0.00186606, 6.68395447, 0.98941813, 0.99483154, 0.47556492],
                             [- 5.68903884, -0.00329975, 0.00186498, 6.68573909, 0.98942807, 0.99483527, 0.47561551],
                             [- 5.69081606, -0.00329711, 0.00186389, 6.68751895, 0.98943799, 0.99483900, 0.47566598],
                             [- 5.69258853, -0.00329447, 0.00186280, 6.68929406, 0.98944790, 0.99484272, 0.47571632],
                             [- 5.69435627, -0.00329184, 0.00186172, 6.69106443, 0.98945779, 0.99484644, 0.47576654],
                             [- 5.69611926, -0.00328921, 0.00186063, 6.69283005, 0.98946767, 0.99485016, 0.47581663],
                             [- 5.69787754, -0.00328658, 0.00185954, 6.69459096, 0.98947753, 0.99485388, 0.47586659],
                             [- 5.69963110, -0.00328396, 0.00185846, 6.69634715, 0.98948737, 0.99485759, 0.47591642],
                             [- 5.70137996, -0.00328133, 0.00185737, 6.69809863, 0.98949720, 0.99486129, 0.47596613],
                             [- 5.70312413, -0.00327872, 0.00185629, 6.69984541, 0.98950701, 0.99486500, 0.47601571],
                             [- 5.70486360, -0.00327610, 0.00185520, 6.70158750, 0.98951681, 0.99486870, 0.47606516],
                             [- 5.70659840, -0.00327349, 0.00185412, 6.70332492, 0.98952659, 0.99487239, 0.47611449],
                             [- 5.70832854, -0.00327088, 0.00185304, 6.70505766, 0.98953636, 0.99487609, 0.47616370],
                             [- 5.71005402, -0.00326827, 0.00185195, 6.70678575, 0.98954611, 0.99487978, 0.47621277],
                             [- 5.71177484, -0.00326567, 0.00185087, 6.70850918, 0.98955584, 0.99488347, 0.47626172],
                             [- 5.71349103, -0.00326306, 0.00184979, 6.71022797, 0.98956556, 0.99488715, 0.47631055],
                             [- 5.71520259, -0.00326047, 0.00184870, 6.71194212, 0.98957526, 0.99489083, 0.47635925],
                             [- 5.71690953, -0.00325787, 0.00184762, 6.71365166, 0.98958495, 0.99489451, 0.47640783],
                             [- 5.71861185, -0.00325528, 0.00184654, 6.71535658, 0.98959462, 0.99489818, 0.47645628],
                             [- 5.72030958, -0.00325269, 0.00184546, 6.71705689, 0.98960428, 0.99490185, 0.47650461],
                             [- 5.72200271, -0.00325010, 0.00184438, 6.71875261, 0.98961392, 0.99490552, 0.47655282],
                             [- 5.72369126, -0.00324752, 0.00184330, 6.72044374, 0.98962355, 0.99490918, 0.47660090],
                             [- 5.72537523, -0.00324494, 0.00184222, 6.72213029, 0.98963316, 0.99491284, 0.47664886],
                             [- 5.72705463, -0.00324236, 0.00184114, 6.72381227, 0.98964276, 0.99491650, 0.47669669],
                             [- 5.72872948, -0.00323979, 0.00184006, 6.72548970, 0.98965234, 0.99492016, 0.47674440],
                             [- 5.73039979, -0.00323721, 0.00183898, 6.72716257, 0.98966190, 0.99492381, 0.47679198],
                             [- 5.73206555, -0.00323464, 0.00183790, 6.72883091, 0.98967145, 0.99492746, 0.47683945],
                             [- 5.73372679, -0.00323208, 0.00183682, 6.73049471, 0.98968099, 0.99493110, 0.47688679],
                             [- 5.73538351, -0.00322952, 0.00183574, 6.73215399, 0.98969051, 0.99493474, 0.47693401],
                             [- 5.73703572, -0.00322695, 0.00183466, 6.73380876, 0.98970002, 0.99493838, 0.47698111],
                             [- 5.73868343, -0.00322440, 0.00183359, 6.73545903, 0.98970951, 0.99494202, 0.47702808],
                             [- 5.74032664, -0.00322184, 0.00183251, 6.73710480, 0.98971898, 0.99494565, 0.47707494],
                             [- 5.74196538, -0.00321929, 0.00183143, 6.73874609, 0.98972844, 0.99494928, 0.47712167],
                             [- 5.74359964, -0.00321674, 0.00183036, 6.74038290, 0.98973789, 0.99495290, 0.47716828],
                             [- 5.74522943, -0.00321419, 0.00182928, 6.74201524, 0.98974732, 0.99495653, 0.47721477],
                             [- 5.74685478, -0.00321165, 0.00182820, 6.74364312, 0.98975674, 0.99496014, 0.47726113],
                             [- 5.74847567, -0.00320911, 0.00182713, 6.74526656, 0.98976614, 0.99496376, 0.47730738],
                             [- 5.75009213, -0.00320657, 0.00182605, 6.74688556, 0.98977552, 0.99496737, 0.47735351],
                             [- 5.75170417, -0.00320404, 0.00182498, 6.74850013, 0.98978490, 0.99497098, 0.47739952],
                             [- 5.75331179, -0.00320151, 0.00182390, 6.75011028, 0.98979425, 0.99497459, 0.47744540],
                             [- 5.75491499, -0.00319898, 0.00182283, 6.75171602, 0.98980360, 0.99497819, 0.47749117],
                             [- 5.75651380, -0.00319645, 0.00182176, 6.75331735, 0.98981293, 0.99498179, 0.47753682],
                             [- 5.75810822, -0.00319393, 0.00182068, 6.75491429, 0.98982224, 0.99498539, 0.47758234],
                             [- 5.75969826, -0.00319141, 0.00181961, 6.75650685, 0.98983154, 0.99498899, 0.47762775],
                             [- 5.76128392, -0.00318889, 0.00181854, 6.75809504, 0.98984082, 0.99499258, 0.47767304],
                             [- 5.76286523, -0.00318637, 0.00181746, 6.75967886, 0.98985009, 0.99499616, 0.47771821],
                             [- 5.76444218, -0.00318386, 0.00181639, 6.76125832, 0.98985935, 0.99499975, 0.47776327],
                             [- 5.76601479, -0.00318135, 0.00181532, 6.76283343, 0.98986859, 0.99500333, 0.47780820],
                             [- 5.76758306, -0.00317884, 0.00181425, 6.76440421, 0.98987782, 0.99500691, 0.47785302],
                             [- 5.76914700, -0.00317634, 0.00181318, 6.76597066, 0.98988703, 0.99501049, 0.47789772],
                             [- 5.77070663, -0.00317384, 0.00181210, 6.76753279, 0.98989623, 0.99501406, 0.47794230],
                             [- 5.77226195, -0.00317134, 0.00181103, 6.76909061, 0.98990542, 0.99501763, 0.47798676],
                             [- 5.77381298, -0.00316884, 0.00180996, 6.77064413, 0.98991459, 0.99502119, 0.47803110],
                             [- 5.77535971, -0.00316635, 0.00180889, 6.77219336, 0.98992374, 0.99502476, 0.47807533],
                             [- 5.77690216, -0.00316386, 0.00180782, 6.77373830, 0.98993289, 0.99502832, 0.47811945],
                             [- 5.77844035, -0.00316137, 0.00180675, 6.77527898, 0.98994201, 0.99503187, 0.47816344],
                             [- 5.77997427, -0.00315889, 0.00180569, 6.77681538, 0.98995113, 0.99503543, 0.47820732],
                             [- 5.78150394, -0.00315640, 0.00180462, 6.77834753, 0.98996023, 0.99503898, 0.47825108],
                             [- 5.78302936, -0.00315393, 0.00180355, 6.77987544, 0.98996931, 0.99504253, 0.47829473],
                             [- 5.78455056, -0.00315145, 0.00180248, 6.78139911, 0.98997839, 0.99504607, 0.47833826],
                             [- 5.78606752, -0.00314897, 0.00180141, 6.78291855, 0.98998744, 0.99504961, 0.47838168],
                             [- 5.78758027, -0.00314650, 0.00180035, 6.78443377, 0.98999649, 0.99505315, 0.47842498],
                             [- 5.78908882, -0.00314403, 0.00179928, 6.78594478, 0.99000552, 0.99505669, 0.47846816],
                             [- 5.79059316, -0.00314157, 0.00179821, 6.78745159, 0.99001453, 0.99506022, 0.47851123],
                             [- 5.79209332, -0.00313911, 0.00179715, 6.78895421, 0.99002354, 0.99506375, 0.47855419],
                             [- 5.79358930, -0.00313665, 0.00179608, 6.79045265, 0.99003253, 0.99506727, 0.47859703],
                             [- 5.79508110, -0.00313419, 0.00179501, 6.79194692, 0.99004150, 0.99507080, 0.47863976],
                             [- 5.79656875, -0.00313173, 0.00179395, 6.79343702, 0.99005046, 0.99507432, 0.47868238],
                             [- 5.79805224, -0.00312928, 0.00179288, 6.79492296, 0.99005941, 0.99507784, 0.47872487],
                             [- 5.79953159, -0.00312683, 0.00179182, 6.79640476, 0.99006834, 0.99508135, 0.47876726],
                             [- 5.80100681, -0.00312438, 0.00179075, 6.79788243, 0.99007726, 0.99508486, 0.47880954],
                             [- 5.80247790, -0.00312194, 0.00178969, 6.79935596, 0.99008617, 0.99508837, 0.47885170],
                             [- 5.80394487, -0.00311950, 0.00178863, 6.80082538, 0.99009506, 0.99509187, 0.47889374],
                             [- 5.80540774, -0.00311706, 0.00178756, 6.80229068, 0.99010394, 0.99509538, 0.47893568],
                             [- 5.80686651, -0.00311462, 0.00178650, 6.80375189, 0.99011281, 0.99509888, 0.47897750],
                             [- 5.80832119, -0.00311219, 0.00178544, 6.80520900, 0.99012166, 0.99510237, 0.47901921],
                             [- 5.80977179, -0.00310976, 0.00178438, 6.80666204, 0.99013050, 0.99510587, 0.47906081],
                             [- 5.81121832, -0.00310733, 0.00178331, 6.80811099, 0.99013933, 0.99510936, 0.47910230],
                             [- 5.81266079, -0.00310490, 0.00178225, 6.80955589, 0.99014814, 0.99511284, 0.47914367],
                             [- 5.81409921, -0.00310248, 0.00178119, 6.81099673, 0.99015694, 0.99511633, 0.47918494],
                             [- 5.81553358, -0.00310006, 0.00178013, 6.81243352, 0.99016573, 0.99511981, 0.47922609],
                             [- 5.81696392, -0.00309764, 0.00177907, 6.81386627, 0.99017450, 0.99512329, 0.47926714],
                             [- 5.81839023, -0.00309523, 0.00177801, 6.81529500, 0.99018326, 0.99512676, 0.47930807],
                             [- 5.81981252, -0.00309282, 0.00177695, 6.81671970, 0.99019200, 0.99513024, 0.47934889],
                             [- 5.82123081, -0.00309041, 0.00177589, 6.81814040, 0.99020074, 0.99513370, 0.47938960],
                             [- 5.82264509, -0.00308800, 0.00177483, 6.81955709, 0.99020946, 0.99513717, 0.47943020],
                             [- 5.82405539, -0.00308560, 0.00177377, 6.82096980, 0.99021816, 0.99514063, 0.47947070],
                             [- 5.82546171, -0.00308319, 0.00177271, 6.82237851, 0.99022686, 0.99514410, 0.47951108],
                             [- 5.82686405, -0.00308079, 0.00177165, 6.82378326, 0.99023554, 0.99514755, 0.47955135],
                             [- 5.82826243, -0.00307840, 0.00177059, 6.82518403, 0.99024421, 0.99515101, 0.47959152],
                             [- 5.82965686, -0.00307600, 0.00176954, 6.82658086, 0.99025286, 0.99515446, 0.47963157],
                             [- 5.83104734, -0.00307361, 0.00176848, 6.82797373, 0.99026150, 0.99515791, 0.47967152],
                             [- 5.83243389, -0.00307122, 0.00176742, 6.82936266, 0.99027013, 0.99516135, 0.47971136],
                             [- 5.83381650, -0.00306884, 0.00176637, 6.83074767, 0.99027875, 0.99516480, 0.47975109],
                             [- 5.83519520, -0.00306645, 0.00176531, 6.83212875, 0.99028735, 0.99516824, 0.47979072],
                             [- 5.83656999, -0.00306407, 0.00176425, 6.83350592, 0.99029594, 0.99517167, 0.47983023],
                             [- 5.83794088, -0.00306169, 0.00176320, 6.83487918, 0.99030451, 0.99517511, 0.47986964],
                             [- 5.83930788, -0.00305932, 0.00176214, 6.83624856, 0.99031308, 0.99517854, 0.47990894],
                             [- 5.84067099, -0.00305695, 0.00176109, 6.83761404, 0.99032163, 0.99518197, 0.47994814],
                             [- 5.84203023, -0.00305458, 0.00176003, 6.83897565, 0.99033017, 0.99518539, 0.47998722],
                             [- 5.84338560, -0.00305221, 0.00175898, 6.84033340, 0.99033869, 0.99518881, 0.48002620],
                             [- 5.84473712, -0.00304984, 0.00175793, 6.84168728, 0.99034721, 0.99519223, 0.48006508],
                             [- 5.84608479, -0.00304748, 0.00175687, 6.84303731, 0.99035571, 0.99519565, 0.48010385],
                             [- 5.84742862, -0.00304512, 0.00175582, 6.84438350, 0.99036419, 0.99519906, 0.48014251],
                             [- 5.84876862, -0.00304276, 0.00175477, 6.84572586, 0.99037267, 0.99520247, 0.48018107],
                             [- 5.85010480, -0.00304041, 0.00175371, 6.84706439, 0.99038113, 0.99520588, 0.48021952],
                             [- 5.85143717, -0.00303805, 0.00175266, 6.84839911, 0.99038958, 0.99520929, 0.48025787],
                             [- 5.85276573, -0.00303570, 0.00175161, 6.84973003, 0.99039802, 0.99521269, 0.48029611],
                             [- 5.85409050, -0.00303336, 0.00175056, 6.85105714, 0.99040644, 0.99521609, 0.48033425],
                             [- 5.85541148, -0.00303101, 0.00174951, 6.85238047, 0.99041485, 0.99521948, 0.48037228],
                             [- 5.85672868, -0.00302867, 0.00174846, 6.85370002, 0.99042325, 0.99522288, 0.48041021],
                             [- 5.85804212, -0.00302633, 0.00174741, 6.85501579, 0.99043164, 0.99522627, 0.48044803],
                             [- 5.85935180, -0.00302399, 0.00174636, 6.85632781, 0.99044002, 0.99522965, 0.48048575],
                             [- 5.86065772, -0.00302166, 0.00174531, 6.85763607, 0.99044838, 0.99523304, 0.48052337],
                             [- 5.86195991, -0.00301932, 0.00174426, 6.85894058, 0.99045673, 0.99523642, 0.48056088],
                             [- 5.86325835, -0.00301699, 0.00174321, 6.86024136, 0.99046507, 0.99523980, 0.48059829],
                             [- 5.86455308, -0.00301467, 0.00174216, 6.86153841, 0.99047339, 0.99524317, 0.48063560],
                             [- 5.86584409, -0.00301234, 0.00174111, 6.86283174, 0.99048171, 0.99524655, 0.48067281],
                             [- 5.86713139, -0.00301002, 0.00174006, 6.86412137, 0.99049001, 0.99524992, 0.48070991],
                             [- 5.86841499, -0.00300770, 0.00173902, 6.86540729, 0.99049829, 0.99525328, 0.48074691],
                             [- 5.86969490, -0.00300538, 0.00173797, 6.86668952, 0.99050657, 0.99525665, 0.48078381],
                             [- 5.87097113, -0.00300307, 0.00173692, 6.86796806, 0.99051483, 0.99526001, 0.48082060],
                             [- 5.87224368, -0.00300076, 0.00173588, 6.86924293, 0.99052309, 0.99526337, 0.48085730],
                             [- 5.87351257, -0.00299845, 0.00173483, 6.87051413, 0.99053133, 0.99526672, 0.48089389],
                             [- 5.87477781, -0.00299614, 0.00173378, 6.87178167, 0.99053955, 0.99527008, 0.48093038],
                             [- 5.87603940, -0.00299383, 0.00173274, 6.87304557, 0.99054777, 0.99527343, 0.48096678],
                             [- 5.87729735, -0.00299153, 0.00173169, 6.87430582, 0.99055597, 0.99527677, 0.48100307],
                             [- 5.87855167, -0.00298923, 0.00173065, 6.87556244, 0.99056416, 0.99528012, 0.48103926],
                             [- 5.87980237, -0.00298694, 0.00172961, 6.87681543, 0.99057234, 0.99528346, 0.48107535],
                             [- 5.88104946, -0.00298464, 0.00172856, 6.87806482, 0.99058051, 0.99528680, 0.48111134],
                             [- 5.88229294, -0.00298235, 0.00172752, 6.87931059, 0.99058867, 0.99529013, 0.48114723],
                             [- 5.88353283, -0.00298006, 0.00172648, 6.88055277, 0.99059681, 0.99529347, 0.48118302],
                             [- 5.88476913, -0.00297777, 0.00172543, 6.88179135, 0.99060494, 0.99529680, 0.48121871],
                             [- 5.88600185, -0.00297549, 0.00172439, 6.88302636, 0.99061306, 0.99530012, 0.48125430],
                             [- 5.88723100, -0.00297320, 0.00172335, 6.88425779, 0.99062117, 0.99530345, 0.48128980],
                             [- 5.88845659, -0.00297093, 0.00172231, 6.88548566, 0.99062927, 0.99530677, 0.48132519],
                             [- 5.88967862, -0.00296865, 0.00172127, 6.88670998, 0.99063735, 0.99531009, 0.48136049],
                             [- 5.89089712, -0.00296637, 0.00172022, 6.88793074, 0.99064542, 0.99531340, 0.48139569],
                             [- 5.89211207, -0.00296410, 0.00171918, 6.88914797, 0.99065348, 0.99531671, 0.48143079],
                             [- 5.89332350, -0.00296183, 0.00171814, 6.89036167, 0.99066153, 0.99532002, 0.48146579],
                             [- 5.89453141, -0.00295956, 0.00171710, 6.89157185, 0.99066957, 0.99532333, 0.48150069],
                             [- 5.89573581, -0.00295730, 0.00171607, 6.89277851, 0.99067760, 0.99532664, 0.48153550],
                             [- 5.89693670, -0.00295504, 0.00171503, 6.89398167, 0.99068561, 0.99532994, 0.48157021],
                             [- 5.89813411, -0.00295278, 0.00171399, 6.89518133, 0.99069361, 0.99533324, 0.48160482],
                             [- 5.89932802, -0.00295052, 0.00171295, 6.89637751, 0.99070160, 0.99533653, 0.48163934],
                             [- 5.90051846, -0.00294826, 0.00171191, 6.89757020, 0.99070958, 0.99533983, 0.48167376],
                             [- 5.90170544, -0.00294601, 0.00171088, 6.89875943, 0.99071755, 0.99534312, 0.48170809],
                             [- 5.90288895, -0.00294376, 0.00170984, 6.89994519, 0.99072550, 0.99534640, 0.48174231],
                             [- 5.90406901, -0.00294151, 0.00170880, 6.90112750, 0.99073345, 0.99534969, 0.48177645],
                             [- 5.90524562, -0.00293927, 0.00170777, 6.90230636, 0.99074138, 0.99535297, 0.48181048],
                             [- 5.90641881, -0.00293702, 0.00170673, 6.90348178, 0.99074930, 0.99535625, 0.48184443],
                             [- 5.90758856, -0.00293478, 0.00170569, 6.90465378, 0.99075721, 0.99535952, 0.48187827],
                             [- 5.90875490, -0.00293254, 0.00170466, 6.90582235, 0.99076511, 0.99536280, 0.48191202],
                             [- 5.90991782, -0.00293031, 0.00170363, 6.90698751, 0.99077300, 0.99536607, 0.48194568],
                             [- 5.91107735, -0.00292807, 0.00170259, 6.90814927, 0.99078087, 0.99536933, 0.48197924],
                             [- 5.91223348, -0.00292584, 0.00170156, 6.90930763, 0.99078874, 0.99537260, 0.48201271],
                             [- 5.91338622, -0.00292362, 0.00170052, 6.91046261, 0.99079659, 0.99537586, 0.48204608],
                             [- 5.91453559, -0.00292139, 0.00169949, 6.91161420, 0.99080443, 0.99537912, 0.48207936],
                             [- 5.91568159, -0.00291917, 0.00169846, 6.91276243, 0.99081226, 0.99538238, 0.48211255],
                             [- 5.91682423, -0.00291694, 0.00169743, 6.91390729, 0.99082008, 0.99538563, 0.48214564],
                             [- 5.91796352, -0.00291472, 0.00169639, 6.91504879, 0.99082789, 0.99538888, 0.48217864],
                             [- 5.91909946, -0.00291251, 0.00169536, 6.91618695, 0.99083569, 0.99539213, 0.48221155],
                             [- 5.92023207, -0.00291029, 0.00169433, 6.91732177, 0.99084347, 0.99539537, 0.48224436],
                             [- 5.92136135, -0.00290808, 0.00169330, 6.91845327, 0.99085124, 0.99539862, 0.48227709],
                             [- 5.92248731, -0.00290587, 0.00169227, 6.91958144, 0.99085901, 0.99540186, 0.48230971],
                             [- 5.92360996, -0.00290367, 0.00169124, 6.92070629, 0.99086676, 0.99540509, 0.48234225],
                             [- 5.92472930, -0.00290146, 0.00169021, 6.92182784, 0.99087450, 0.99540833, 0.48237470],
                             [- 5.92584536, -0.00289926, 0.00168918, 6.92294610, 0.99088223, 0.99541156, 0.48240705],
                             [- 5.92695812, -0.00289706, 0.00168816, 6.92406106, 0.99088995, 0.99541479, 0.48243931],
                             [- 5.92806761, -0.00289486, 0.00168713, 6.92517274, 0.99089766, 0.99541801, 0.48247148],
                             [- 5.92917382, -0.00289267, 0.00168610, 6.92628115, 0.99090535, 0.99542123, 0.48250356],
                             [- 5.93027677, -0.00289047, 0.00168507, 6.92738630, 0.99091304, 0.99542445, 0.48253555],
                             [- 5.93137647, -0.00288828, 0.00168405, 6.92848819, 0.99092071, 0.99542767, 0.48256745],
                             [- 5.93247293, -0.00288610, 0.00168302, 6.92958683, 0.99092837, 0.99543089, 0.48259926],
                             [- 5.93356614, -0.00288391, 0.00168199, 6.93068223, 0.99093603, 0.99543410, 0.48263098],
                             [- 5.93465613, -0.00288173, 0.00168097, 6.93177440, 0.99094367, 0.99543731, 0.48266260],
                             [- 5.93574289, -0.00287955, 0.00167994, 6.93286334, 0.99095130, 0.99544051, 0.48269414],
                             [- 5.93682644, -0.00287737, 0.00167892, 6.93394907, 0.99095892, 0.99544371, 0.48272559],
                             [- 5.93790678, -0.00287519, 0.00167789, 6.93503159, 0.99096653, 0.99544692, 0.48275695],
                             [- 5.93898392, -0.00287302, 0.00167687, 6.93611091, 0.99097412, 0.99545011, 0.48278822],
                             [- 5.94005788, -0.00287085, 0.00167585, 6.93718703, 0.99098171, 0.99545331, 0.48281940],
                             [- 5.94112865, -0.00286868, 0.00167482, 6.93825997, 0.99098929, 0.99545650, 0.48285049],
                             [- 5.94219625, -0.00286651, 0.00167380, 6.93932974, 0.99099685, 0.99545969, 0.48288150],
                             [- 5.94326068, -0.00286435, 0.00167278, 6.94039634, 0.99100441, 0.99546288, 0.48291241],
                             [- 5.94432196, -0.00286218, 0.00167176, 6.94145977, 0.99101195, 0.99546606, 0.48294324],
                             [- 5.94538008, -0.00286002, 0.00167074, 6.94252006, 0.99101949, 0.99546924, 0.48297398],
                             [- 5.94643506, -0.00285787, 0.00166971, 6.94357720, 0.99102701, 0.99547242, 0.48300463],
                             [- 5.94748691, -0.00285571, 0.00166869, 6.94463120, 0.99103452, 0.99547559, 0.48303520],
                             [- 5.94853563, -0.00285356, 0.00166767, 6.94568207, 0.99104202, 0.99547877, 0.48306568],
                             [- 5.94958124, -0.00285141, 0.00166666, 6.94672983, 0.99104951, 0.99548194, 0.48309607],
                             [- 5.95062373, -0.00284926, 0.00166564, 6.94777447, 0.99105699, 0.99548510, 0.48312637],
                             [- 5.95166312, -0.00284711, 0.00166462, 6.94881600, 0.99106446, 0.99548827, 0.48315659],
                             [- 5.95269941, -0.00284497, 0.00166360, 6.94985444, 0.99107192, 0.99549143, 0.48318672],
                             [- 5.95373262, -0.00284283, 0.00166258, 6.95088979, 0.99107937, 0.99549459, 0.48321676],
                             [- 5.95476275, -0.00284069, 0.00166156, 6.95192206, 0.99108681, 0.99549775, 0.48324672],
                             [- 5.95578980, -0.00283855, 0.00166055, 6.95295125, 0.99109423, 0.99550090, 0.48327660],
                             [- 5.95681380, -0.00283642, 0.00165953, 6.95397738, 0.99110165, 0.99550405, 0.48330638],
                             [- 5.95783474, -0.00283429, 0.00165852, 6.95500045, 0.99110906, 0.99550720, 0.48333609],
                             [- 5.95885262, -0.00283216, 0.00165750, 6.95602047, 0.99111645, 0.99551034, 0.48336571],
                             [- 5.95986747, -0.00283003, 0.00165648, 6.95703744, 0.99112384, 0.99551348, 0.48339524],
                             [- 5.96087929, -0.00282791, 0.00165547, 6.95805138, 0.99113121, 0.99551662, 0.48342469],
                             [- 5.96188808, -0.00282578, 0.00165446, 6.95906230, 0.99113858, 0.99551976, 0.48345405],
                             [- 5.96289385, -0.00282366, 0.00165344, 6.96007019, 0.99114593, 0.99552290, 0.48348333],
                             [- 5.96389662, -0.00282154, 0.00165243, 6.96107507, 0.99115327, 0.99552603, 0.48351252],
                             [- 5.96489638, -0.00281943, 0.00165142, 6.96207695, 0.99116061, 0.99552916, 0.48354163],
                             [- 5.96589315, -0.00281732, 0.00165040, 6.96307584, 0.99116793, 0.99553228, 0.48357066],
                             [- 5.96688693, -0.00281520, 0.00164939, 6.96407173, 0.99117524, 0.99553540, 0.48359961],
                             [- 5.96787774, -0.00281310, 0.00164838, 6.96506464, 0.99118255, 0.99553852, 0.48362846],
                             [- 5.96886557, -0.00281099, 0.00164737, 6.96605458, 0.99118984, 0.99554164, 0.48365724],
                             [- 5.96985044, -0.00280888, 0.00164636, 6.96704156, 0.99119712, 0.99554476, 0.48368594],
                             [- 5.97083235, -0.00280678, 0.00164535, 6.96802557, 0.99120439, 0.99554787, 0.48371455],
                             [- 5.97181132, -0.00280468, 0.00164434, 6.96900664, 0.99121165, 0.99555098, 0.48374308],
                             [- 5.97278734, -0.00280258, 0.00164333, 6.96998476, 0.99121891, 0.99555409, 0.48377153],
                             [- 5.97376044, -0.00280049, 0.00164232, 6.97095995, 0.99122615, 0.99555719, 0.48379989],
                             [- 5.97473060, -0.00279840, 0.00164131, 6.97193221, 0.99123338, 0.99556029, 0.48382818],
                             [- 5.97569785, -0.00279631, 0.00164031, 6.97290155, 0.99124060, 0.99556339, 0.48385638],
                             [- 5.97666219, -0.00279422, 0.00163930, 6.97386797, 0.99124781, 0.99556648, 0.48388450],
                             [- 5.97762362, -0.00279213, 0.00163829, 6.97483149, 0.99125501, 0.99556958, 0.48391254],
                             [- 5.97858216, -0.00279005, 0.00163729, 6.97579212, 0.99126220, 0.99557267, 0.48394050],
                             [- 5.97953782, -0.00278796, 0.00163628, 6.97674985, 0.99126938, 0.99557576, 0.48396837],
                             [- 5.98049059, -0.00278589, 0.00163527, 6.97770470, 0.99127655, 0.99557884, 0.48399617],
                             [- 5.98144049, -0.00278381, 0.00163427, 6.97865668, 0.99128371, 0.99558192, 0.48402389],
                             [- 5.98238752, -0.00278173, 0.00163326, 6.97960579, 0.99129087, 0.99558500, 0.48405152],
                             [- 5.98333170, -0.00277966, 0.00163226, 6.98055204, 0.99129801, 0.99558808, 0.48407908],
                             [- 5.98427302, -0.00277759, 0.00163126, 6.98149543, 0.99130514, 0.99559115, 0.48410655],
                             [- 5.98521150, -0.00277552, 0.00163025, 6.98243598, 0.99131226, 0.99559422, 0.48413395],
                             [- 5.98614715, -0.00277346, 0.00162925, 6.98337369, 0.99131937, 0.99559729, 0.48416127],
                             [- 5.98707997, -0.00277139, 0.00162825, 6.98430858, 0.99132647, 0.99560036, 0.48418851],
                             [- 5.98800996, -0.00276933, 0.00162725, 6.98524063, 0.99133356, 0.99560342, 0.48421567],
                             [- 5.98893715, -0.00276727, 0.00162625, 6.98616988, 0.99134064, 0.99560648, 0.48424275],
                             [- 5.98986153, -0.00276521, 0.00162525, 6.98709631, 0.99134771, 0.99560954, 0.48426975],
                             [- 5.99078310, -0.00276316, 0.00162425, 6.98801995, 0.99135477, 0.99561259, 0.48429667],
                             [- 5.99170189, -0.00276111, 0.00162325, 6.98894079, 0.99136183, 0.99561565, 0.48432352],
                             [- 5.99261789, -0.00275906, 0.00162225, 6.98985884, 0.99136887, 0.99561870, 0.48435028],
                             [- 5.99353112, -0.00275701, 0.00162125, 6.99077411, 0.99137590, 0.99562174, 0.48437697],
                             [- 5.99444158, -0.00275496, 0.00162025, 6.99168662, 0.99138292, 0.99562479, 0.48440358],
                             [- 5.99534927, -0.00275292, 0.00161925, 6.99259635, 0.99138994, 0.99562783, 0.48443012],
                             [- 5.99625421, -0.00275088, 0.00161826, 6.99350334, 0.99139694, 0.99563087, 0.48445657],
                             [- 5.99715640, -0.00274884, 0.00161726, 6.99440757, 0.99140393, 0.99563390, 0.48448295],
                             [- 5.99805585, -0.00274680, 0.00161626, 6.99530906, 0.99141092, 0.99563694, 0.48450926],
                             [- 5.99895257, -0.00274476, 0.00161527, 6.99620781, 0.99141789, 0.99563997, 0.48453548],
                             [- 5.99984656, -0.00274273, 0.00161427, 6.99710383, 0.99142485, 0.99564299, 0.48456163],
                             [- 6.00073784, -0.00274070, 0.00161328, 6.99799714, 0.99143181, 0.99564602, 0.48458771],
                             [- 6.00162640, -0.00273867, 0.00161228, 6.99888773, 0.99143875, 0.99564904, 0.48461370],
                             [- 6.00251225, -0.00273665, 0.00161129, 6.99977561, 0.99144569, 0.99565206, 0.48463962],
                             [- 6.00339541, -0.00273462, 0.00161030, 7.00066079, 0.99145261, 0.99565508, 0.48466547],
                             [- 6.00427588, -0.00273260, 0.00160931, 7.00154328, 0.99145953, 0.99565809, 0.48469124],
                             [- 6.00515367, -0.00273058, 0.00160831, 7.00242309, 0.99146644, 0.99566111, 0.48471693],
                             [- 6.00602878, -0.00272856, 0.00160732, 7.00330021, 0.99147334, 0.99566411, 0.48474255],
                             [- 6.00690122, -0.00272655, 0.00160633, 7.00417467, 0.99148022, 0.99566712, 0.48476810],
                             [- 6.00777100, -0.00272454, 0.00160534, 7.00504646, 0.99148710, 0.99567012, 0.48479357],
                             [- 6.00863812, -0.00272252, 0.00160435, 7.00591560, 0.99149397, 0.99567313, 0.48481896],
                             [- 6.00950260, -0.00272052, 0.00160336, 7.00678208, 0.99150083, 0.99567612, 0.48484429],
                             [- 6.01036444, -0.00271851, 0.00160237, 7.00764593, 0.99150768, 0.99567912, 0.48486953],
                             [- 6.01122364, -0.00271651, 0.00160138, 7.00850713, 0.99151452, 0.99568211, 0.48489471],
                             [- 6.01208022, -0.00271450, 0.00160039, 7.00936571, 0.99152135, 0.99568510, 0.48491980],
                             [- 6.01293417, -0.00271250, 0.00159941, 7.01022167, 0.99152817, 0.99568809, 0.48494483],
                             [- 6.01378552, -0.00271051, 0.00159842, 7.01107501, 0.99153498, 0.99569107, 0.48496978],
                             [- 6.01463426, -0.00270851, 0.00159743, 7.01192575, 0.99154179, 0.99569406, 0.48499466],
                             [- 6.01548040, -0.00270652, 0.00159645, 7.01277388, 0.99154858, 0.99569704, 0.48501947],
                             [- 6.01632394, -0.00270453, 0.00159546, 7.01361942, 0.99155537, 0.99570001, 0.48504420],
                             [- 6.01716491, -0.00270254, 0.00159448, 7.01446237, 0.99156214, 0.99570299, 0.48506886],
                             [- 6.01800329, -0.00270055, 0.00159349, 7.01530274, 0.99156891, 0.99570596, 0.48509345],
                             [- 6.01883911, -0.00269856, 0.00159251, 7.01614055, 0.99157566, 0.99570893, 0.48511797],
                             [- 6.01967236, -0.00269658, 0.00159153, 7.01697578, 0.99158241, 0.99571189, 0.48514241],
                             [- 6.02050306, -0.00269460, 0.00159054, 7.01780845, 0.99158915, 0.99571486, 0.48516678],
                             [- 6.02133120, -0.00269262, 0.00158956, 7.01863858, 0.99159588, 0.99571782, 0.48519108],
                             [- 6.02215680, -0.00269065, 0.00158858, 7.01946615, 0.99160260, 0.99572077, 0.48521531],
                             [- 6.02297987, -0.00268867, 0.00158760, 7.02029119, 0.99160931, 0.99572373, 0.48523947],
                             [- 6.02380040, -0.00268670, 0.00158662, 7.02111370, 0.99161601, 0.99572668, 0.48526355],
                             [- 6.02461841, -0.00268473, 0.00158564, 7.02193368, 0.99162270, 0.99572963, 0.48528757],
                             [- 6.02543391, -0.00268276, 0.00158466, 7.02275115, 0.99162938, 0.99573258, 0.48531151],
                             [- 6.02624690, -0.00268080, 0.00158368, 7.02356610, 0.99163606, 0.99573552, 0.48533539],
                             [- 6.02705738, -0.00267883, 0.00158270, 7.02437855, 0.99164272, 0.99573847, 0.48535919],
                             [- 6.02786537, -0.00267687, 0.00158172, 7.02518850, 0.99164938, 0.99574141, 0.48538292],
                             [- 6.02867087, -0.00267491, 0.00158074, 7.02599596, 0.99165602, 0.99574434, 0.48540659],
                             [- 6.02947389, -0.00267296, 0.00157977, 7.02680093, 0.99166266, 0.99574728, 0.48543018],
                             [- 6.03027443, -0.00267100, 0.00157879, 7.02760343, 0.99166929, 0.99575021, 0.48545370],
                             [- 6.03107251, -0.00266905, 0.00157781, 7.02840346, 0.99167591, 0.99575314, 0.48547716],
                             [- 6.03186812, -0.00266710, 0.00157684, 7.02920102, 0.99168252, 0.99575606, 0.48550054],
                             [- 6.03266127, -0.00266515, 0.00157586, 7.02999612, 0.99168912, 0.99575899, 0.48552386],
                             [- 6.03345198, -0.00266320, 0.00157489, 7.03078878, 0.99169571, 0.99576191, 0.48554710],
                             [- 6.03424025, -0.00266126, 0.00157391, 7.03157899, 0.99170230, 0.99576483, 0.48557028],
                             [- 6.03502608, -0.00265932, 0.00157294, 7.03236676, 0.99170887, 0.99576774, 0.48559339],
                             [- 6.03580948, -0.00265738, 0.00157197, 7.03315210, 0.99171544, 0.99577065, 0.48561643],
                             [- 6.03659046, -0.00265544, 0.00157100, 7.03393502, 0.99172199, 0.99577356, 0.48563940],
                             [- 6.03736903, -0.00265350, 0.00157002, 7.03471552, 0.99172854, 0.99577647, 0.48566231],
                             [- 6.03814518, -0.00265157, 0.00156905, 7.03549361, 0.99173508, 0.99577938, 0.48568514],
                             [- 6.03891894, -0.00264964, 0.00156808, 7.03626930, 0.99174161, 0.99578228, 0.48570791],
                             [- 6.03969029, -0.00264771, 0.00156711, 7.03704258, 0.99174813, 0.99578518, 0.48573061],
                             [- 6.04045926, -0.00264578, 0.00156614, 7.03781348, 0.99175464, 0.99578807, 0.48575324],
                             [- 6.04122585, -0.00264386, 0.00156517, 7.03858199, 0.99176115, 0.99579097, 0.48577581],
                             [- 6.04199006, -0.00264193, 0.00156421, 7.03934813, 0.99176764, 0.99579386, 0.48579831],
                             [- 6.04275190, -0.00264001, 0.00156324, 7.04011189, 0.99177413, 0.99579675, 0.48582075],
                             [- 6.04351138, -0.00263809, 0.00156227, 7.04087328, 0.99178061, 0.99579964, 0.48584311],
                             [- 6.04426850, -0.00263618, 0.00156130, 7.04163232, 0.99178707, 0.99580252, 0.48586541],
                             [- 6.04502327, -0.00263426, 0.00156034, 7.04238901, 0.99179353, 0.99580540, 0.48588765],
                             [- 6.04577570, -0.00263235, 0.00155937, 7.04314335, 0.99179998, 0.99580828, 0.48590981],
                             [- 6.04652579, -0.00263044, 0.00155841, 7.04389535, 0.99180643, 0.99581115, 0.48593191],
                             [- 6.04727354, -0.00262853, 0.00155744, 7.04464502, 0.99181286, 0.99581403, 0.48595395],
                             [- 6.04801898, -0.00262662, 0.00155648, 7.04539236, 0.99181929, 0.99581690, 0.48597592],
                             [- 6.04876210, -0.00262472, 0.00155552, 7.04613738, 0.99182570, 0.99581977, 0.48599782],
                             [- 6.04950290, -0.00262282, 0.00155455, 7.04688009, 0.99183211, 0.99582263, 0.48601966],
                             [- 6.05024140, -0.00262092, 0.00155359, 7.04762049, 0.99183851, 0.99582549, 0.48604144],
                             [- 6.05097760, -0.00261902, 0.00155263, 7.04835858, 0.99184490, 0.99582835, 0.48606315],
                             [- 6.05171151, -0.00261712, 0.00155167, 7.04909439, 0.99185128, 0.99583121, 0.48608479],
                             [- 6.05244313, -0.00261523, 0.00155071, 7.04982791, 0.99185765, 0.99583407, 0.48610637],
                             [- 6.05317248, -0.00261334, 0.00154975, 7.05055914, 0.99186402, 0.99583692, 0.48612789],
                             [- 6.05389955, -0.00261145, 0.00154879, 7.05128810, 0.99187037, 0.99583977, 0.48614934],
                             [- 6.05462435, -0.00260956, 0.00154783, 7.05201479, 0.99187672, 0.99584261, 0.48617073],
                             [- 6.05534689, -0.00260767, 0.00154687, 7.05273922, 0.99188306, 0.99584546, 0.48619205],
                             [- 6.05606718, -0.00260579, 0.00154591, 7.05346139, 0.99188939, 0.99584830, 0.48621331],
                             [- 6.05678521, -0.00260391, 0.00154495, 7.05418131, 0.99189571, 0.99585114, 0.48623450],
                             [- 6.05750101, -0.00260203, 0.00154400, 7.05489898, 0.99190203, 0.99585398, 0.48625564],
                             [- 6.05821457, -0.00260015, 0.00154304, 7.05561442, 0.99190833, 0.99585681, 0.48627671],
                             [- 6.05892590, -0.00259827, 0.00154209, 7.05632763, 0.99191463, 0.99585964, 0.48629772],
                             [- 6.05963501, -0.00259640, 0.00154113, 7.05703861, 0.99192092, 0.99586247, 0.48631866],
                             [- 6.06034190, -0.00259453, 0.00154018, 7.05774737, 0.99192720, 0.99586529, 0.48633954],
                             [- 6.06104657, -0.00259266, 0.00153922, 7.05845392, 0.99193347, 0.99586812, 0.48636036],
                             [- 6.06174905, -0.00259079, 0.00153827, 7.05915826, 0.99193973, 0.99587094, 0.48638112],
                             [- 6.06244932, -0.00258893, 0.00153732, 7.05986040, 0.99194598, 0.99587376, 0.48640181],
                             [- 6.06314740, -0.00258706, 0.00153637, 7.06056034, 0.99195223, 0.99587657, 0.48642244],
                             [- 6.06384330, -0.00258520, 0.00153541, 7.06125810, 0.99195847, 0.99587938, 0.48644301],
                             [- 6.06453701, -0.00258334, 0.00153446, 7.06195367, 0.99196470, 0.99588220, 0.48646352],
                             [- 6.06522855, -0.00258148, 0.00153351, 7.06264707, 0.99197092, 0.99588500, 0.48648397],
                             [- 6.06591793, -0.00257963, 0.00153256, 7.06333830, 0.99197713, 0.99588781, 0.48650436],
                             [- 6.06660514, -0.00257778, 0.00153161, 7.06402736, 0.99198334, 0.99589061, 0.48652468],
                             [- 6.06729019, -0.00257592, 0.00153067, 7.06471427, 0.99198953, 0.99589341, 0.48654494],
                             [- 6.06797309, -0.00257408, 0.00152972, 7.06539902, 0.99199572, 0.99589621, 0.48656515],
                             [- 6.06865385, -0.00257223, 0.00152877, 7.06608163, 0.99200190, 0.99589900, 0.48658529],
                             [- 6.06933248, -0.00257038, 0.00152782, 7.06676209, 0.99200807, 0.99590179, 0.48660537],
                             [- 6.07000896, -0.00256854, 0.00152688, 7.06744043, 0.99201424, 0.99590458, 0.48662539],
                             [- 6.07068333, -0.00256670, 0.00152593, 7.06811663, 0.99202039, 0.99590737, 0.48664535],
                             [- 6.07135557, -0.00256486, 0.00152499, 7.06879071, 0.99202654, 0.99591015, 0.48666525],
                             [- 6.07202570, -0.00256302, 0.00152404, 7.06946268, 0.99203268, 0.99591294, 0.48668509],
                             [- 6.07269372, -0.00256119, 0.00152310, 7.07013254, 0.99203881, 0.99591572, 0.48670487],
                             [- 6.07335964, -0.00255935, 0.00152215, 7.07080029, 0.99204493, 0.99591849, 0.48672460],
                             [- 6.07402346, -0.00255752, 0.00152121, 7.07146594, 0.99205104, 0.99592127, 0.48674426],
                             [- 6.07468520, -0.00255569, 0.00152027, 7.07212950, 0.99205715, 0.99592404, 0.48676386],
                             [- 6.07534484, -0.00255387, 0.00151933, 7.07279098, 0.99206325, 0.99592681, 0.48678341],
                             [- 6.07600241, -0.00255204, 0.00151839, 7.07345037, 0.99206934, 0.99592957, 0.48680289],
                             [- 6.07665791, -0.00255022, 0.00151745, 7.07410769, 0.99207542, 0.99593234, 0.48682232],
                             [- 6.07731134, -0.00254840, 0.00151651, 7.07476294, 0.99208149, 0.99593510, 0.48684169],
                             [- 6.07796271, -0.00254658, 0.00151557, 7.07541613, 0.99208756, 0.99593786, 0.48686100],
                             [- 6.07861202, -0.00254476, 0.00151463, 7.07606726, 0.99209362, 0.99594061, 0.48688025],
                             [- 6.07925928, -0.00254294, 0.00151369, 7.07671634, 0.99209967, 0.99594336, 0.48689944],
                             [- 6.07990450, -0.00254113, 0.00151275, 7.07736337, 0.99210571, 0.99594612, 0.48691858],
                             [- 6.08054769, -0.00253932, 0.00151182, 7.07800837, 0.99211174, 0.99594886, 0.48693766],
                             [- 6.08118884, -0.00253751, 0.00151088, 7.07865133, 0.99211777, 0.99595161, 0.48695668],
                             [- 6.08182796, -0.00253570, 0.00150994, 7.07929226, 0.99212379, 0.99595435, 0.48697564],
                             [- 6.08246507, -0.00253390, 0.00150901, 7.07993117, 0.99212980, 0.99595709, 0.48699454],
                             [- 6.08310016, -0.00253210, 0.00150807, 7.08056806, 0.99213580, 0.99595983, 0.48701339],
                             [- 6.08373324, -0.00253029, 0.00150714, 7.08120294, 0.99214179, 0.99596257, 0.48703218],
                             [- 6.08436431, -0.00252849, 0.00150621, 7.08183582, 0.99214778, 0.99596530, 0.48705092],
                             [- 6.08499339, -0.00252670, 0.00150527, 7.08246669, 0.99215376, 0.99596803, 0.48706960],
                             [- 6.08562048, -0.00252490, 0.00150434, 7.08309558, 0.99215973, 0.99597076, 0.48708821],
                             [- 6.08624558, -0.00252311, 0.00150341, 7.08372247, 0.99216569, 0.99597348, 0.48710678],
                             [- 6.08686870, -0.00252132, 0.00150248, 7.08434738, 0.99217164, 0.99597620, 0.48712529],
                             [- 6.08748985, -0.00251953, 0.00150155, 7.08497032, 0.99217759, 0.99597892, 0.48714374],
                             [- 6.08810902, -0.00251774, 0.00150062, 7.08559128, 0.99218353, 0.99598164, 0.48716214],
                             [- 6.08872624, -0.00251595, 0.00149969, 7.08621028, 0.99218946, 0.99598436, 0.48718048],
                             [- 6.08934149, -0.00251417, 0.00149876, 7.08682732, 0.99219538, 0.99598707, 0.48719876],
                             [- 6.08995479, -0.00251239, 0.00149784, 7.08744241, 0.99220130, 0.99598978, 0.48721699],
                             [- 6.09056615, -0.00251061, 0.00149691, 7.08805554, 0.99220721, 0.99599248, 0.48723517],
                             [- 6.09117556, -0.00250883, 0.00149598, 7.08866673, 0.99221310, 0.99599519, 0.48725329],
                             [- 6.09178304, -0.00250705, 0.00149506, 7.08927599, 0.99221900, 0.99599789, 0.48727135],
                             [- 6.09238859, -0.00250528, 0.00149413, 7.08988331, 0.99222488, 0.99600059, 0.48728936],
                             [- 6.09299221, -0.00250351, 0.00149320, 7.09048871, 0.99223076, 0.99600329, 0.48730732],
                             [- 6.09359392, -0.00250174, 0.00149228, 7.09109218, 0.99223663, 0.99600598, 0.48732522],
                             [- 6.09419371, -0.00249997, 0.00149136, 7.09169374, 0.99224249, 0.99600867, 0.48734306],
                             [- 6.09479159, -0.00249820, 0.00149043, 7.09229339, 0.99224834, 0.99601136, 0.48736085],
                             [- 6.09538757, -0.00249644, 0.00148951, 7.09289113, 0.99225419, 0.99601405, 0.48737859],
                             [- 6.09598165, -0.00249468, 0.00148859, 7.09348698, 0.99226003, 0.99601673, 0.48739628],
                             [- 6.09657385, -0.00249292, 0.00148767, 7.09408093, 0.99226586, 0.99601942, 0.48741390],
                             [- 6.09716415, -0.00249116, 0.00148675, 7.09467299, 0.99227168, 0.99602210, 0.48743148],
                             [- 6.09775257, -0.00248940, 0.00148583, 7.09526317, 0.99227749, 0.99602477, 0.48744900],
                             [- 6.09833912, -0.00248765, 0.00148491, 7.09585148, 0.99228330, 0.99602745, 0.48746647],
                             [- 6.09892380, -0.00248589, 0.00148399, 7.09643791, 0.99228910, 0.99603012, 0.48748389],
                             [- 6.09950662, -0.00248414, 0.00148307, 7.09702248, 0.99229489, 0.99603279, 0.48750125],
                             [- 6.10008757, -0.00248239, 0.00148215, 7.09760518, 0.99230068, 0.99603545, 0.48751856],
                             [- 6.10066667, -0.00248065, 0.00148124, 7.09818603, 0.99230646, 0.99603812, 0.48753582],
                             [- 6.10124393, -0.00247890, 0.00148032, 7.09876503, 0.99231223, 0.99604078, 0.48755302],
                             [- 6.10181934, -0.00247716, 0.00147941, 7.09934218, 0.99231799, 0.99604344, 0.48757018],
                             [- 6.10239291, -0.00247542, 0.00147849, 7.09991749, 0.99232374, 0.99604609, 0.48758728],
                             [- 6.10296465, -0.00247368, 0.00147758, 7.10049097, 0.99232949, 0.99604875, 0.48760432],
                             [- 6.10353456, -0.00247194, 0.00147666, 7.10106262, 0.99233523, 0.99605140, 0.48762132],
                             [- 6.10410265, -0.00247020, 0.00147575, 7.10163245, 0.99234096, 0.99605405, 0.48763826],
                             [- 6.10466892, -0.00246847, 0.00147484, 7.10220045, 0.99234669, 0.99605669, 0.48765516],
                             [- 6.10523338, -0.00246674, 0.00147393, 7.10276665, 0.99235240, 0.99605934, 0.48767200],
                             [- 6.10579604, -0.00246501, 0.00147301, 7.10333103, 0.99235811, 0.99606198, 0.48768879],
                             [- 6.10635689, -0.00246328, 0.00147210, 7.10389362, 0.99236382, 0.99606462, 0.48770552],
                             [- 6.10691595, -0.00246155, 0.00147119, 7.10445440, 0.99236951, 0.99606725, 0.48772221],
                             [- 6.10747322, -0.00245983, 0.00147028, 7.10501339, 0.99237520, 0.99606989, 0.48773885],
                             [- 6.10802871, -0.00245811, 0.00146937, 7.10557060, 0.99238088, 0.99607252, 0.48775543],
                             [- 6.10858241, -0.00245639, 0.00146847, 7.10612603, 0.99238655, 0.99607515, 0.48777197],
                             [- 6.10913434, -0.00245467, 0.00146756, 7.10667967, 0.99239222, 0.99607777, 0.48778845],
                             [- 6.10968450, -0.00245295, 0.00146665, 7.10723155, 0.99239787, 0.99608040, 0.48780488],
                             [- 6.11023289, -0.00245123, 0.00146575, 7.10778166, 0.99240353, 0.99608302, 0.48782127],
                             [- 6.11077953, -0.00244952, 0.00146484, 7.10833001, 0.99240917, 0.99608564, 0.48783760],
                             [- 6.11132441, -0.00244781, 0.00146393, 7.10887660, 0.99241480, 0.99608825, 0.48785388],
                             [- 6.11186754, -0.00244610, 0.00146303, 7.10942144, 0.99242043, 0.99609087, 0.48787012],
                             [- 6.11240893, -0.00244439, 0.00146213, 7.10996454, 0.99242605, 0.99609348, 0.48788630],
                             [- 6.11294858, -0.00244269, 0.00146122, 7.11050589, 0.99243167, 0.99609609, 0.48790243],
                             [- 6.11348649, -0.00244098, 0.00146032, 7.11104551, 0.99243728, 0.99609869, 0.48791852],
                             [- 6.11402268, -0.00243928, 0.00145942, 7.11158340, 0.99244288, 0.99610130, 0.48793455],
                             [- 6.11455714, -0.00243758, 0.00145852, 7.11211956, 0.99244847, 0.99610390, 0.48795053],
                             [- 6.11508989, -0.00243588, 0.00145762, 7.11265400, 0.99245405, 0.99610650, 0.48796647],
                             [- 6.11562092, -0.00243419, 0.00145672, 7.11318673, 0.99245963, 0.99610909, 0.48798236],
                             [- 6.11615024, -0.00243249, 0.00145582, 7.11371775, 0.99246520, 0.99611169, 0.48799820],
                             [- 6.11667786, -0.00243080, 0.00145492, 7.11424706, 0.99247077, 0.99611428, 0.48801399],
                             [- 6.11720378, -0.00242911, 0.00145402, 7.11477467, 0.99247632, 0.99611687, 0.48802973],
                             [- 6.11772800, -0.00242742, 0.00145312, 7.11530058, 0.99248187, 0.99611945, 0.48804543],
                             [- 6.11825054, -0.00242574, 0.00145223, 7.11582481, 0.99248741, 0.99612204, 0.48806107],
                             [- 6.11877140, -0.00242405, 0.00145133, 7.11634735, 0.99249295, 0.99612462, 0.48807667],
                             [- 6.11929057, -0.00242237, 0.00145043, 7.11686820, 0.99249847, 0.99612720, 0.48809222],
                             [- 6.11980807, -0.00242069, 0.00144954, 7.11738739, 0.99250400, 0.99612978, 0.48810772],
                             [- 6.12032390, -0.00241901, 0.00144864, 7.11790490, 0.99250951, 0.99613235, 0.48812317],
                             [- 6.12083807, -0.00241733, 0.00144775, 7.11842074, 0.99251501, 0.99613492, 0.48813858],
                             [- 6.12135058, -0.00241565, 0.00144686, 7.11893493, 0.99252051, 0.99613749, 0.48815394],
                             [- 6.12186144, -0.00241398, 0.00144597, 7.11944746, 0.99252601, 0.99614006, 0.48816925],
                             [- 6.12237064, -0.00241231, 0.00144507, 7.11995834, 0.99253149, 0.99614262, 0.48818451],
                             [- 6.12287820, -0.00241064, 0.00144418, 7.12046757, 0.99253697, 0.99614518, 0.48819973],
                             [- 6.12338412, -0.00240897, 0.00144329, 7.12097516, 0.99254244, 0.99614774, 0.48821490],
                             [- 6.12388841, -0.00240730, 0.00144240, 7.12148111, 0.99254790, 0.99615030, 0.48823002],
                             [- 6.12439107, -0.00240563, 0.00144151, 7.12198543, 0.99255336, 0.99615285, 0.48824510],
                             [- 6.12489210, -0.00240397, 0.00144062, 7.12248813, 0.99255881, 0.99615540, 0.48826013],
                             [- 6.12539151, -0.00240231, 0.00143974, 7.12298920, 0.99256425, 0.99615795, 0.48827512],
                             [- 6.12588931, -0.00240065, 0.00143885, 7.12348866, 0.99256969, 0.99616050, 0.48829005],
                             [- 6.12638549, -0.00239899, 0.00143796, 7.12398650, 0.99257512, 0.99616304, 0.48830495],
                             [- 6.12688007, -0.00239734, 0.00143708, 7.12448273, 0.99258054, 0.99616559, 0.48831979],
                             [- 6.12737305, -0.00239568, 0.00143619, 7.12497736, 0.99258596, 0.99616813, 0.48833460],
                             [- 6.12786443, -0.00239403, 0.00143531, 7.12547040, 0.99259136, 0.99617066, 0.48834935],
                             [- 6.12835422, -0.00239238, 0.00143442, 7.12596184, 0.99259676, 0.99617320, 0.48836406],
                             [- 6.12884242, -0.00239073, 0.00143354, 7.12645169, 0.99260216, 0.99617573, 0.48837872],
                             [- 6.12932904, -0.00238908, 0.00143266, 7.12693996, 0.99260755, 0.99617826, 0.48839334],
                             [- 6.12981408, -0.00238744, 0.00143177, 7.12742664, 0.99261293, 0.99618079, 0.48840792],
                             [- 6.13029755, -0.00238580, 0.00143089, 7.12791176, 0.99261830, 0.99618331, 0.48842244],
                             [- 6.13077945, -0.00238416, 0.00143001, 7.12839530, 0.99262367, 0.99618583, 0.48843693],
                             [- 6.13125979, -0.00238252, 0.00142913, 7.12887728, 0.99262903, 0.99618835, 0.48845136],
                             [- 6.13173857, -0.00238088, 0.00142825, 7.12935769, 0.99263438, 0.99619087, 0.48846576],
                             [- 6.13221579, -0.00237924, 0.00142737, 7.12983655, 0.99263972, 0.99619339, 0.48848011],
                             [- 6.13269147, -0.00237761, 0.00142649, 7.13031386, 0.99264506, 0.99619590, 0.48849441],
                             [- 6.13316560, -0.00237598, 0.00142562, 7.13078962, 0.99265040, 0.99619841, 0.48850867],
                             [- 6.13363819, -0.00237434, 0.00142474, 7.13126384, 0.99265572, 0.99620092, 0.48852289],
                             [- 6.13410924, -0.00237272, 0.00142386, 7.13173652, 0.99266104, 0.99620342, 0.48853706],
                             [- 6.13457876, -0.00237109, 0.00142299, 7.13220767, 0.99266635, 0.99620593, 0.48855119],
                             [- 6.13504676, -0.00236946, 0.00142211, 7.13267729, 0.99267166, 0.99620843, 0.48856528],
                             [- 6.13551323, -0.00236784, 0.00142124, 7.13314539, 0.99267696, 0.99621092, 0.48857932],
                             [- 6.13597818, -0.00236622, 0.00142036, 7.13361197, 0.99268225, 0.99621342, 0.48859332],
                             [- 6.13644163, -0.00236460, 0.00141949, 7.13407703, 0.99268753, 0.99621591, 0.48860727],
                             [- 6.13690356, -0.00236298, 0.00141862, 7.13454058, 0.99269281, 0.99621840, 0.48862118],
                             [- 6.13736399, -0.00236136, 0.00141774, 7.13500263, 0.99269808, 0.99622089, 0.48863505],
                             [- 6.13782292, -0.00235975, 0.00141687, 7.13546317, 0.99270335, 0.99622338, 0.48864888],
                             [- 6.13828035, -0.00235814, 0.00141600, 7.13592222, 0.99270861, 0.99622586, 0.48866266],
                             [- 6.13873630, -0.00235652, 0.00141513, 7.13637977, 0.99271386, 0.99622834, 0.48867640],
                             [- 6.13919076, -0.00235492, 0.00141426, 7.13683584, 0.99271910, 0.99623082, 0.48869010],
                             [- 6.13964373, -0.00235331, 0.00141339, 7.13729042, 0.99272434, 0.99623330, 0.48870375],
                             [- 6.14009523, -0.00235170, 0.00141253, 7.13774353, 0.99272957, 0.99623577, 0.48871736],
                             [- 6.14054526, -0.00235010, 0.00141166, 7.13819516, 0.99273480, 0.99623824, 0.48873093],
                             [- 6.14099381, -0.00234850, 0.00141079, 7.13864532, 0.99274001, 0.99624071, 0.48874446],
                             [- 6.14144091, -0.00234689, 0.00140993, 7.13909401, 0.99274523, 0.99624318, 0.48875794],
                             [- 6.14188654, -0.00234530, 0.00140906, 7.13954125, 0.99275043, 0.99624564, 0.48877139],
                             [- 6.14233072, -0.00234370, 0.00140820, 7.13998702, 0.99275563, 0.99624811, 0.48878479],
                             [- 6.14277345, -0.00234210, 0.00140733, 7.14043134, 0.99276082, 0.99625057, 0.48879815],
                             [- 6.14321473, -0.00234051, 0.00140647, 7.14087422, 0.99276601, 0.99625302, 0.48881147],
                             [- 6.14365457, -0.00233892, 0.00140561, 7.14131565, 0.99277118, 0.99625548, 0.48882475],
                             [- 6.14409297, -0.00233733, 0.00140474, 7.14175564, 0.99277636, 0.99625793, 0.48883798],
                             [- 6.14452994, -0.00233574, 0.00140388, 7.14219420, 0.99278152, 0.99626038, 0.48885118],
                             [- 6.14496548, -0.00233415, 0.00140302, 7.14263132, 0.99278668, 0.99626283, 0.48886433],
                             [- 6.14539959, -0.00233257, 0.00140216, 7.14306702, 0.99279183, 0.99626527, 0.48887745],
                             [- 6.14583228, -0.00233098, 0.00140130, 7.14350130, 0.99279698, 0.99626772, 0.48889052],
                             [- 6.14626356, -0.00232940, 0.00140044, 7.14393416, 0.99280212, 0.99627016, 0.48890355],
                             [- 6.14669342, -0.00232782, 0.00139958, 7.14436560, 0.99280725, 0.99627259, 0.48891654],
                             [- 6.14712188, -0.00232625, 0.00139873, 7.14479564, 0.99281238, 0.99627503, 0.48892949],
                             [- 6.14754894, -0.00232467, 0.00139787, 7.14522427, 0.99281750, 0.99627746, 0.48894240],
                             [- 6.14797459, -0.00232309, 0.00139701, 7.14565150, 0.99282261, 0.99627989, 0.48895527],
                             [- 6.14839885, -0.00232152, 0.00139616, 7.14607733, 0.99282772, 0.99628232, 0.48896810],
                             [- 6.14882172, -0.00231995, 0.00139530, 7.14650177, 0.99283282, 0.99628475, 0.48898089],
                             [- 6.14924320, -0.00231838, 0.00139445, 7.14692482, 0.99283791, 0.99628717, 0.48899364],
                             [- 6.14966330, -0.00231681, 0.00139359, 7.14734649, 0.99284300, 0.99628959, 0.48900634],
                             [- 6.15008202, -0.00231525, 0.00139274, 7.14776677, 0.99284808, 0.99629201, 0.48901902],
                             [- 6.15049937, -0.00231368, 0.00139189, 7.14818569, 0.99285316, 0.99629443, 0.48903165],
                             [- 6.15091535, -0.00231212, 0.00139104, 7.14860323, 0.99285823, 0.99629684, 0.48904424],
                             [- 6.15132996, -0.00231056, 0.00139018, 7.14901940, 0.99286329, 0.99629926, 0.48905679],
                             [- 6.15174321, -0.00230900, 0.00138933, 7.14943421, 0.99286834, 0.99630166, 0.48906930],
                             [- 6.15215510, -0.00230744, 0.00138848, 7.14984766, 0.99287339, 0.99630407, 0.48908178],
                             [- 6.15256564, -0.00230589, 0.00138763, 7.15025975, 0.99287844, 0.99630648, 0.48909421],
                             [- 6.15297483, -0.00230433, 0.00138679, 7.15067050, 0.99288347, 0.99630888, 0.48910661],
                             [- 6.15338268, -0.00230278, 0.00138594, 7.15107989, 0.99288850, 0.99631128, 0.48911897],
                             [- 6.15378918, -0.00230123, 0.00138509, 7.15148795, 0.99289353, 0.99631368, 0.48913128],
                             [- 6.15419435, -0.00229968, 0.00138424, 7.15189467, 0.99289855, 0.99631607, 0.48914356],
                             [- 6.15459818, -0.00229814, 0.00138340, 7.15230005, 0.99290356, 0.99631847, 0.48915581],
                             [- 6.15500069, -0.00229659, 0.00138255, 7.15270410, 0.99290856, 0.99632086, 0.48916801],
                             [- 6.15540187, -0.00229505, 0.00138171, 7.15310682, 0.99291356, 0.99632324, 0.48918017],
                             [- 6.15580173, -0.00229350, 0.00138087, 7.15350822, 0.99291855, 0.99632563, 0.48919230],
                             [- 6.15620027, -0.00229196, 0.00138002, 7.15390831, 0.99292354, 0.99632801, 0.48920439],
                             [- 6.15659750, -0.00229042, 0.00137918, 7.15430708, 0.99292852, 0.99633040, 0.48921644],
                             [- 6.15699342, -0.00228889, 0.00137834, 7.15470454, 0.99293349, 0.99633278, 0.48922846],
                             [- 6.15738804, -0.00228735, 0.00137750, 7.15510069, 0.99293846, 0.99633515, 0.48924043],
                             [- 6.15778136, -0.00228582, 0.00137666, 7.15549554, 0.99294342, 0.99633753, 0.48925237],
                             [- 6.15817338, -0.00228429, 0.00137582, 7.15588909, 0.99294838, 0.99633990, 0.48926427],
                             [- 6.15856410, -0.00228276, 0.00137498, 7.15628135, 0.99295333, 0.99634227, 0.48927613],
                             [- 6.15895354, -0.00228123, 0.00137414, 7.15667231, 0.99295827, 0.99634464, 0.48928796],
                             [- 6.15934170, -0.00227970, 0.00137330, 7.15706199, 0.99296321, 0.99634700, 0.48929975],
                             [- 6.15972857, -0.00227818, 0.00137246, 7.15745039, 0.99296814, 0.99634936, 0.48931151],
                             [- 6.16011416, -0.00227665, 0.00137162, 7.15783751, 0.99297306, 0.99635172, 0.48932322],
                             [- 6.16049848, -0.00227513, 0.00137079, 7.15822335, 0.99297798, 0.99635408, 0.48933490],
                             [- 6.16088154, -0.00227361, 0.00136995, 7.15860793, 0.99298289, 0.99635644, 0.48934654],
                             [- 6.16126332, -0.00227209, 0.00136912, 7.15899123, 0.99298780, 0.99635879, 0.48935815],
                             [- 6.16164385, -0.00227057, 0.00136828, 7.15937328, 0.99299270, 0.99636114, 0.48936972],
                             [- 6.16202312, -0.00226906, 0.00136745, 7.15975406, 0.99299759, 0.99636349, 0.48938125],
                             [- 6.16240113, -0.00226754, 0.00136662, 7.16013359, 0.99300248, 0.99636584, 0.48939275],
                             [- 6.16277790, -0.00226603, 0.00136579, 7.16051186, 0.99300736, 0.99636818, 0.48940421],
                             [- 6.16315341, -0.00226452, 0.00136495, 7.16088889, 0.99301224, 0.99637052, 0.48941563],
                             [- 6.16352769, -0.00226301, 0.00136412, 7.16126468, 0.99301711, 0.99637286, 0.48942702],
                             [- 6.16390073, -0.00226151, 0.00136329, 7.16163922, 0.99302197, 0.99637520, 0.48943837],
                             [- 6.16427253, -0.00226000, 0.00136246, 7.16201253, 0.99302683, 0.99637753, 0.48944969],
                             [- 6.16464310, -0.00225850, 0.00136164, 7.16238461, 0.99303168, 0.99637987, 0.48946098],
                             [- 6.16501245, -0.00225700, 0.00136081, 7.16275545, 0.99303653, 0.99638220, 0.48947222],
                             [- 6.16538057, -0.00225549, 0.00135998, 7.16312507, 0.99304136, 0.99638453, 0.48948343],
                             [- 6.16574747, -0.00225400, 0.00135915, 7.16349348, 0.99304620, 0.99638685, 0.48949461],
                             [- 6.16611316, -0.00225250, 0.00135833, 7.16386066, 0.99305103, 0.99638918, 0.48950575],
                             [- 6.16647763, -0.00225100, 0.00135750, 7.16422663, 0.99305585, 0.99639150, 0.48951685],
                             [- 6.16684090, -0.00224951, 0.00135668, 7.16459139, 0.99306066, 0.99639382, 0.48952792],
                             [- 6.16720296, -0.00224802, 0.00135585, 7.16495494, 0.99306547, 0.99639613, 0.48953896],
                             [- 6.16756382, -0.00224653, 0.00135503, 7.16531729, 0.99307027, 0.99639845, 0.48954996],
                             [- 6.16792348, -0.00224504, 0.00135421, 7.16567845, 0.99307507, 0.99640076, 0.48956093],
                             [- 6.16828195, -0.00224355, 0.00135338, 7.16603840, 0.99307986, 0.99640307, 0.48957186],
                             [- 6.16863923, -0.00224206, 0.00135256, 7.16639717, 0.99308465, 0.99640538, 0.48958276],
                             [- 6.16899533, -0.00224058, 0.00135174, 7.16675475, 0.99308943, 0.99640768, 0.48959363],
                             [- 6.16935024, -0.00223910, 0.00135092, 7.16711114, 0.99309420, 0.99640998, 0.48960445],
                             [- 6.16970397, -0.00223762, 0.00135010, 7.16746636, 0.99309897, 0.99641228, 0.48961525],
                             [- 6.17005653, -0.00223614, 0.00134928, 7.16782039, 0.99310373, 0.99641458, 0.48962601],
                             [- 6.17040791, -0.00223466, 0.00134846, 7.16817326, 0.99310849, 0.99641688, 0.48963673],
                             [- 6.17075813, -0.00223318, 0.00134765, 7.16852495, 0.99311324, 0.99641917, 0.48964743],
                             [- 6.17110719, -0.00223171, 0.00134683, 7.16887548, 0.99311798, 0.99642146, 0.48965809],
                             [- 6.17145508, -0.00223024, 0.00134601, 7.16922484, 0.99312272, 0.99642375, 0.48966871],
                             [- 6.17180182, -0.00222876, 0.00134520, 7.16957305, 0.99312745, 0.99642604, 0.48967931],
                             [- 6.17214740, -0.00222729, 0.00134438, 7.16992010, 0.99313218, 0.99642832, 0.48968987],
                             [- 6.17249183, -0.00222583, 0.00134357, 7.17026601, 0.99313690, 0.99643061, 0.48970039],
                             [- 6.17283512, -0.00222436, 0.00134275, 7.17061076, 0.99314162, 0.99643289, 0.48971088],
                             [- 6.17317726, -0.00222289, 0.00134194, 7.17095437, 0.99314633, 0.99643517, 0.48972134],
                             [- 6.17351827, -0.00222143, 0.00134113, 7.17129683, 0.99315103, 0.99643744, 0.48973177],
                             [- 6.17385813, -0.00221997, 0.00134032, 7.17163816, 0.99315573, 0.99643971, 0.48974217],
                             [- 6.17419687, -0.00221851, 0.00133950, 7.17197836, 0.99316042, 0.99644199, 0.48975252],
                             [- 6.17453448, -0.00221705, 0.00133869, 7.17231743, 0.99316511, 0.99644425, 0.48976285],
                             [- 6.17487096, -0.00221559, 0.00133788, 7.17265537, 0.99316979, 0.99644652, 0.48977314],
                             [- 6.17520632, -0.00221414, 0.00133708, 7.17299218, 0.99317446, 0.99644879, 0.48978341],
                             [- 6.17554056, -0.00221269, 0.00133627, 7.17332788, 0.99317913, 0.99645105, 0.48979364],
                             [- 6.17587369, -0.00221123, 0.00133546, 7.17366246, 0.99318379, 0.99645331, 0.48980384],
                             [- 6.17620571, -0.00220978, 0.00133465, 7.17399593, 0.99318845, 0.99645557, 0.48981400],
                             [- 6.17653662, -0.00220833, 0.00133385, 7.17432829, 0.99319310, 0.99645782, 0.48982414],
                             [- 6.17686642, -0.00220689, 0.00133304, 7.17465954, 0.99319775, 0.99646007, 0.48983424],
                             [- 6.17719513, -0.00220544, 0.00133223, 7.17498969, 0.99320239, 0.99646233, 0.48984431],
                             [- 6.17752273, -0.00220400, 0.00133143, 7.17531874, 0.99320702, 0.99646457, 0.48985434],
                             [- 6.17784925, -0.00220255, 0.00133063, 7.17564669, 0.99321165, 0.99646682, 0.48986435],
                             [- 6.17817467, -0.00220111, 0.00132982, 7.17597356, 0.99321628, 0.99646906, 0.48987432],
                             [- 6.17849900, -0.00219967, 0.00132902, 7.17629933, 0.99322089, 0.99647131, 0.48988427],
                             [- 6.17882225, -0.00219823, 0.00132822, 7.17662402, 0.99322551, 0.99647355, 0.48989418],
                             [- 6.17914442, -0.00219680, 0.00132742, 7.17694762, 0.99323011, 0.99647578, 0.48990406],
                             [- 6.17946552, -0.00219536, 0.00132662, 7.17727015, 0.99323471, 0.99647802, 0.48991391],
                             [- 6.17978554, -0.00219393, 0.00132582, 7.17759161, 0.99323931, 0.99648025, 0.48992373],
                             [- 6.18010448, -0.00219250, 0.00132502, 7.17791199, 0.99324390, 0.99648248, 0.48993351],
                             [- 6.18042237, -0.00219107, 0.00132422, 7.17823130, 0.99324848, 0.99648471, 0.48994327],
                             [- 6.18073919, -0.00218964, 0.00132342, 7.17854955, 0.99325306, 0.99648694, 0.48995299],
                             [- 6.18105494, -0.00218821, 0.00132262, 7.17886673, 0.99325763, 0.99648916, 0.48996269],
                             [- 6.18136964, -0.00218679, 0.00132183, 7.17918286, 0.99326220, 0.99649139, 0.48997235],
                             [- 6.18168329, -0.00218536, 0.00132103, 7.17949793, 0.99326676, 0.99649361, 0.48998198],
                             [- 6.18199589, -0.00218394, 0.00132024, 7.17981195, 0.99327132, 0.99649582, 0.48999158],
                             [- 6.18230744, -0.00218252, 0.00131944, 7.18012492, 0.99327587, 0.99649804, 0.49000115],
                             [- 6.18261795, -0.00218110, 0.00131865, 7.18043684, 0.99328041, 0.99650025, 0.49001070],
                             [- 6.18292741, -0.00217968, 0.00131785, 7.18074773, 0.99328495, 0.99650246, 0.49002020],
                             [- 6.18323584, -0.00217827, 0.00131706, 7.18105757, 0.99328949, 0.99650467, 0.49002969],
                             [- 6.18354323, -0.00217685, 0.00131627, 7.18136638, 0.99329401, 0.99650688, 0.49003913],
                             [- 6.18384960, -0.00217544, 0.00131548, 7.18167416, 0.99329854, 0.99650908, 0.49004855],
                             [- 6.18415493, -0.00217403, 0.00131469, 7.18198091, 0.99330305, 0.99651129, 0.49005794],
                             [- 6.18445924, -0.00217262, 0.00131390, 7.18228663, 0.99330757, 0.99651349, 0.49006730],
                             [- 6.18476253, -0.00217121, 0.00131311, 7.18259133, 0.99331207, 0.99651568, 0.49007664],
                             [- 6.18506481, -0.00216980, 0.00131232, 7.18289501, 0.99331657, 0.99651788, 0.49008594],
                             [- 6.18536606, -0.00216840, 0.00131153, 7.18319767, 0.99332107, 0.99652007, 0.49009521],
                             [- 6.18566631, -0.00216699, 0.00131074, 7.18349932, 0.99332556, 0.99652226, 0.49010445],
                             [- 6.18596555, -0.00216559, 0.00130996, 7.18379996, 0.99333004, 0.99652445, 0.49011367],
                             [- 6.18626378, -0.00216419, 0.00130917, 7.18409959, 0.99333452, 0.99652664, 0.49012285],
                             [- 6.18656101, -0.00216279, 0.00130839, 7.18439822, 0.99333900, 0.99652882, 0.49013200],
                             [- 6.18685724, -0.00216139, 0.00130760, 7.18469585, 0.99334346, 0.99653101, 0.49014113],
                             [- 6.18715248, -0.00216000, 0.00130682, 7.18499249, 0.99334793, 0.99653319, 0.49015022],
                             [- 6.18744673, -0.00215860, 0.00130603, 7.18528812, 0.99335238, 0.99653537, 0.49015929],
                             [- 6.18773998, -0.00215721, 0.00130525, 7.18558277, 0.99335684, 0.99653754, 0.49016833],
                             [- 6.18803225, -0.00215582, 0.00130447, 7.18587643, 0.99336128, 0.99653972, 0.49017734],
                             [- 6.18832353, -0.00215443, 0.00130369, 7.18616911, 0.99336572, 0.99654189, 0.49018632],
                             [- 6.18861384, -0.00215304, 0.00130291, 7.18646080, 0.99337016, 0.99654406, 0.49019527],
                             [- 6.18890316, -0.00215165, 0.00130213, 7.18675152, 0.99337459, 0.99654622, 0.49020420],
                             [- 6.18919152, -0.00215026, 0.00130135, 7.18704125, 0.99337901, 0.99654839, 0.49021309],
                             [- 6.18947890, -0.00214888, 0.00130057, 7.18733002, 0.99338343, 0.99655055, 0.49022196],
                             [- 6.18976532, -0.00214750, 0.00129979, 7.18761782, 0.99338785, 0.99655271, 0.49023080],
                             [- 6.19005077, -0.00214612, 0.00129901, 7.18790465, 0.99339226, 0.99655487, 0.49023961],
                             [- 6.19033526, -0.00214474, 0.00129823, 7.18819052, 0.99339666, 0.99655703, 0.49024839],
                             [- 6.19061879, -0.00214336, 0.00129746, 7.18847543, 0.99340106, 0.99655918, 0.49025714],
                             [- 6.19090136, -0.00214198, 0.00129668, 7.18875938, 0.99340545, 0.99656134, 0.49026588],
                             [- 6.19118298, -0.00214061, 0.00129591, 7.18904238, 0.99340984, 0.99656349, 0.49027457],
                             [- 6.19146366, -0.00213923, 0.00129513, 7.18932442, 0.99341422, 0.99656563, 0.49028324],
                             [- 6.19174338, -0.00213786, 0.00129436, 7.18960552, 0.99341860, 0.99656778, 0.49029189],
                             [- 6.19202217, -0.00213649, 0.00129359, 7.18988568, 0.99342297, 0.99656992, 0.49030051],
                             [- 6.19230001, -0.00213512, 0.00129281, 7.19016489, 0.99342734, 0.99657207, 0.49030909],
                             [- 6.19257691, -0.00213375, 0.00129204, 7.19044316, 0.99343170, 0.99657421, 0.49031766],
                             [- 6.19285288, -0.00213239, 0.00129127, 7.19072050, 0.99343606, 0.99657634, 0.49032619],
                             [- 6.19312792, -0.00213102, 0.00129050, 7.19099690, 0.99344041, 0.99657848, 0.49033470],
                             [- 6.19340203, -0.00212966, 0.00128973, 7.19127237, 0.99344475, 0.99658061, 0.49034317],
                             [- 6.19367521, -0.00212830, 0.00128896, 7.19154692, 0.99344909, 0.99658274, 0.49035163],
                             [- 6.19394748, -0.00212694, 0.00128819, 7.19182054, 0.99345343, 0.99658487, 0.49036006],
                             [- 6.19421882, -0.00212558, 0.00128743, 7.19209324, 0.99345776, 0.99658700, 0.49036845],
                             [- 6.19448924, -0.00212422, 0.00128666, 7.19236502, 0.99346208, 0.99658912, 0.49037683],
                             [- 6.19475875, -0.00212286, 0.00128589, 7.19263589, 0.99346640, 0.99659124, 0.49038517],
                             [- 6.19502735, -0.00212151, 0.00128513, 7.19290584, 0.99347072, 0.99659336, 0.49039349],
                             [- 6.19529504, -0.00212016, 0.00128436, 7.19317489, 0.99347503, 0.99659548, 0.49040178],
                             [- 6.19556183, -0.00211880, 0.00128360, 7.19344302, 0.99347933, 0.99659760, 0.49041005],
                             [- 6.19582771, -0.00211745, 0.00128283, 7.19371026, 0.99348363, 0.99659971, 0.49041829],
                             [- 6.19609270, -0.00211611, 0.00128207, 7.19397659, 0.99348792, 0.99660182, 0.49042650],
                             [- 6.19635678, -0.00211476, 0.00128131, 7.19424203, 0.99349221, 0.99660393, 0.49043469],
                             [- 6.19661998, -0.00211341, 0.00128054, 7.19450656, 0.99349650, 0.99660604, 0.49044285],
                             [- 6.19688228, -0.00211207, 0.00127978, 7.19477021, 0.99350077, 0.99660815, 0.49045099],
                             [- 6.19714369, -0.00211073, 0.00127902, 7.19503297, 0.99350505, 0.99661025, 0.49045909],
                             [- 6.19740422, -0.00210938, 0.00127826, 7.19529484, 0.99350932, 0.99661235, 0.49046717],
                             [- 6.19766387, -0.00210804, 0.00127750, 7.19555583, 0.99351358, 0.99661445, 0.49047523],
                             [- 6.19792264, -0.00210671, 0.00127674, 7.19581593, 0.99351784, 0.99661655, 0.49048326],
                             [- 6.19818053, -0.00210537, 0.00127599, 7.19607516, 0.99352209, 0.99661864, 0.49049127],
                             [- 6.19843754, -0.00210403, 0.00127523, 7.19633351, 0.99352634, 0.99662074, 0.49049925],
                             [- 6.19869369, -0.00210270, 0.00127447, 7.19659099, 0.99353058, 0.99662283, 0.49050720],
                             [- 6.19894896, -0.00210137, 0.00127372, 7.19684760, 0.99353482, 0.99662492, 0.49051513],
                             [- 6.19920337, -0.00210004, 0.00127296, 7.19710334, 0.99353905, 0.99662700, 0.49052303],
                             [- 6.19945692, -0.00209871, 0.00127221, 7.19735822, 0.99354328, 0.99662909, 0.49053091],
                             [- 6.19970961, -0.00209738, 0.00127145, 7.19761223, 0.99354750, 0.99663117, 0.49053876],
                             [- 6.19996144, -0.00209605, 0.00127070, 7.19786539, 0.99355172, 0.99663325, 0.49054659],
                             [- 6.20021242, -0.00209473, 0.00126995, 7.19811769, 0.99355593, 0.99663533, 0.49055439],
                             [- 6.20046254, -0.00209340, 0.00126919, 7.19836914, 0.99356014, 0.99663741, 0.49056217],
                             [- 6.20071181, -0.00209208, 0.00126844, 7.19861973, 0.99356434, 0.99663948, 0.49056992],
                             [- 6.20096024, -0.00209076, 0.00126769, 7.19886948, 0.99356854, 0.99664155, 0.49057765],
                             [- 6.20120782, -0.00208944, 0.00126694, 7.19911839, 0.99357273, 0.99664362, 0.49058536],
                             [- 6.20145457, -0.00208812, 0.00126619, 7.19936645, 0.99357692, 0.99664569, 0.49059304],
                             [- 6.20170047, -0.00208680, 0.00126544, 7.19961367, 0.99358110, 0.99664776, 0.49060069],
                             [- 6.20194554, -0.00208549, 0.00126469, 7.19986005, 0.99358528, 0.99664982, 0.49060832],
                             [- 6.20218978, -0.00208417, 0.00126395, 7.20010560, 0.99358945, 0.99665188, 0.49061593],
                             [- 6.20243318, -0.00208286, 0.00126320, 7.20035032, 0.99359362, 0.99665394, 0.49062351],
                             [- 6.20267576, -0.00208155, 0.00126245, 7.20059421, 0.99359778, 0.99665600, 0.49063106],
                             [- 6.20291751, -0.00208024, 0.00126171, 7.20083727, 0.99360194, 0.99665805, 0.49063859],
                             [- 6.20315844, -0.00207893, 0.00126096, 7.20107951, 0.99360609, 0.99666011, 0.49064611],
                             [- 6.20339855, -0.00207762, 0.00126022, 7.20132093, 0.99361024, 0.99666216, 0.49065358],
                             [- 6.20363785, -0.00207632, 0.00125947, 7.20156153, 0.99361438, 0.99666421, 0.49066104],
                             [- 6.20387633, -0.00207501, 0.00125873, 7.20180131, 0.99361852, 0.99666626, 0.49066848],
                             [- 6.20411399, -0.00207371, 0.00125799, 7.20204028, 0.99362265, 0.99666830, 0.49067589],
                             [- 6.20435085, -0.00207241, 0.00125725, 7.20227844, 0.99362678, 0.99667034, 0.49068328],
                             [- 6.20458690, -0.00207111, 0.00125650, 7.20251579, 0.99363090, 0.99667238, 0.49069064],
                             [- 6.20482215, -0.00206981, 0.00125576, 7.20275234, 0.99363502, 0.99667442, 0.49069799],
                             [- 6.20505660, -0.00206852, 0.00125502, 7.20298808, 0.99363914, 0.99667646, 0.49070530],
                             [- 6.20529024, -0.00206722, 0.00125428, 7.20322302, 0.99364325, 0.99667850, 0.49071260],
                             [- 6.20552310, -0.00206593, 0.00125355, 7.20345717, 0.99364735, 0.99668053, 0.49071987],
                             [- 6.20575515, -0.00206463, 0.00125281, 7.20369052, 0.99365145, 0.99668256, 0.49072711],
                             [- 6.20598642, -0.00206334, 0.00125207, 7.20392308, 0.99365554, 0.99668459, 0.49073434],
                             [- 6.20621690, -0.00206205, 0.00125133, 7.20415485, 0.99365963, 0.99668662, 0.49074154],
                             [- 6.20644659, -0.00206076, 0.00125060, 7.20438583, 0.99366372, 0.99668864, 0.49074872],
                             [- 6.20667550, -0.00205947, 0.00124986, 7.20461602, 0.99366780, 0.99669066, 0.49075587],
                             [- 6.20690362, -0.00205819, 0.00124913, 7.20484543, 0.99367187, 0.99669268, 0.49076300],
                             [- 6.20713097, -0.00205690, 0.00124839, 7.20507407, 0.99367594, 0.99669470, 0.49077011],
                             [- 6.20735754, -0.00205562, 0.00124766, 7.20530192, 0.99368001, 0.99669672, 0.49077720],
                             [- 6.20758334, -0.00205434, 0.00124693, 7.20552900, 0.99368407, 0.99669873, 0.49078426],
                             [- 6.20780837, -0.00205306, 0.00124619, 7.20575531, 0.99368812, 0.99670075, 0.49079130],
                             [- 6.20803263, -0.00205178, 0.00124546, 7.20598085, 0.99369217, 0.99670276, 0.49079832],
                             [- 6.20825612, -0.00205050, 0.00124473, 7.20620562, 0.99369622, 0.99670477, 0.49080531],
                             [- 6.20847885, -0.00204923, 0.00124400, 7.20642963, 0.99370026, 0.99670677, 0.49081228],
                             [- 6.20870082, -0.00204795, 0.00124327, 7.20665287, 0.99370430, 0.99670878, 0.49081924],
                             [- 6.20892203, -0.00204668, 0.00124254, 7.20687536, 0.99370833, 0.99671078, 0.49082617],
                             [- 6.20914249, -0.00204540, 0.00124181, 7.20709708, 0.99371236, 0.99671278, 0.49083307],
                             [- 6.20936219, -0.00204413, 0.00124109, 7.20731805, 0.99371638, 0.99671478, 0.49083996],
                             [- 6.20958114, -0.00204286, 0.00124036, 7.20753827, 0.99372040, 0.99671678, 0.49084681],
                             [- 6.20979934, -0.00204160, 0.00123963, 7.20775774, 0.99372441, 0.99671877, 0.49085365],
                             [- 6.21001679, -0.00204033, 0.00123891, 7.20797646, 0.99372842, 0.99672076, 0.49086047],
                             [- 6.21023350, -0.00203906, 0.00123818, 7.20819444, 0.99373242, 0.99672275, 0.49086727],
                             [- 6.21044947, -0.00203780, 0.00123746, 7.20841167, 0.99373642, 0.99672474, 0.49087404],
                             [- 6.21066470, -0.00203654, 0.00123673, 7.20862816, 0.99374042, 0.99672673, 0.49088079],
                             [- 6.21087920, -0.00203528, 0.00123601, 7.20884392, 0.99374441, 0.99672871, 0.49088752],
                             [- 6.21109295, -0.00203402, 0.00123529, 7.20905894, 0.99374839, 0.99673070, 0.49089423],
                             [- 6.21130598, -0.00203276, 0.00123456, 7.20927322, 0.99375237, 0.99673268, 0.49090092],
                             [- 6.21151828, -0.00203150, 0.00123384, 7.20948678, 0.99375635, 0.99673466, 0.49090759],
                             [- 6.21172985, -0.00203024, 0.00123312, 7.20969960, 0.99376032, 0.99673663, 0.49091423],
                             [- 6.21194069, -0.00202899, 0.00123240, 7.20991170, 0.99376428, 0.99673861, 0.49092085],
                             [- 6.21215082, -0.00202774, 0.00123168, 7.21012308, 0.99376825, 0.99674058, 0.49092745],
                             [- 6.21236022, -0.00202648, 0.00123096, 7.21033373, 0.99377220, 0.99674255, 0.49093403],
                             [- 6.21256890, -0.00202523, 0.00123025, 7.21054367, 0.99377616, 0.99674452, 0.49094059],
                             [- 6.21277687, -0.00202398, 0.00122953, 7.21075289, 0.99378010, 0.99674649, 0.49094712],
                             [- 6.21298413, -0.00202274, 0.00122881, 7.21096139, 0.99378405, 0.99674845, 0.49095364],
                             [- 6.21319067, -0.00202149, 0.00122809, 7.21116918, 0.99378798, 0.99675042, 0.49096013],
                             [- 6.21339651, -0.00202025, 0.00122738, 7.21137627, 0.99379192, 0.99675238, 0.49096661],
                             [- 6.21360164, -0.00201900, 0.00122666, 7.21158264, 0.99379585, 0.99675434, 0.49097306],
                             [- 6.21380607, -0.00201776, 0.00122595, 7.21178831, 0.99379977, 0.99675629, 0.49097950],
                             [- 6.21400979, -0.00201652, 0.00122523, 7.21199328, 0.99380369, 0.99675825, 0.49098591],
                             [- 6.21421282, -0.00201528, 0.00122452, 7.21219754, 0.99380761, 0.99676020, 0.49099230],
                             [- 6.21441515, -0.00201404, 0.00122381, 7.21240111, 0.99381152, 0.99676215, 0.49099867],
                             [- 6.21461678, -0.00201280, 0.00122310, 7.21260398, 0.99381543, 0.99676410, 0.49100502],
                             [- 6.21481773, -0.00201157, 0.00122239, 7.21280616, 0.99381933, 0.99676605, 0.49101135],
                             [- 6.21501798, -0.00201033, 0.00122167, 7.21300765, 0.99382323, 0.99676799, 0.49101767],
                             [- 6.21521754, -0.00200910, 0.00122096, 7.21320844, 0.99382712, 0.99676994, 0.49102395],
                             [- 6.21541642, -0.00200787, 0.00122026, 7.21340855, 0.99383101, 0.99677188, 0.49103022],
                             [- 6.21561461, -0.00200664, 0.00121955, 7.21360798, 0.99383489, 0.99677382, 0.49103647],
                             [- 6.21581213, -0.00200541, 0.00121884, 7.21380672, 0.99383877, 0.99677575, 0.49104269],
                             [- 6.21600896, -0.00200418, 0.00121813, 7.21400478, 0.99384265, 0.99677769, 0.49104891],
                             [- 6.21620512, -0.00200295, 0.00121742, 7.21420217, 0.99384652, 0.99677962, 0.49105509],
                             [- 6.21640060, -0.00200173, 0.00121672, 7.21439887, 0.99385038, 0.99678156, 0.49106126],
                             [- 6.21659541, -0.00200050, 0.00121601, 7.21459491, 0.99385424, 0.99678349, 0.49106741],
                             [- 6.21678955, -0.00199928, 0.00121531, 7.21479027, 0.99385810, 0.99678541, 0.49107353],
                             [- 6.21698302, -0.00199806, 0.00121460, 7.21498497, 0.99386195, 0.99678734, 0.49107964],
                             [- 6.21717583, -0.00199684, 0.00121390, 7.21517899, 0.99386580, 0.99678926, 0.49108574],
                             [- 6.21736797, -0.00199562, 0.00121320, 7.21537235, 0.99386965, 0.99679119, 0.49109180],
                             [- 6.21755945, -0.00199440, 0.00121249, 7.21556505, 0.99387349, 0.99679311, 0.49109786],
                             [- 6.21775028, -0.00199319, 0.00121179, 7.21575709, 0.99387732, 0.99679502, 0.49110388],
                             [- 6.21794044, -0.00199197, 0.00121109, 7.21594847, 0.99388115, 0.99679694, 0.49110989],
                             [- 6.21812995, -0.00199076, 0.00121039, 7.21613919, 0.99388498, 0.99679885, 0.49111588],
                             [- 6.21831881, -0.00198955, 0.00120969, 7.21632926, 0.99388880, 0.99680077, 0.49112186],
                             [- 6.21850702, -0.00198833, 0.00120899, 7.21651868, 0.99389261, 0.99680268, 0.49112781],
                             [- 6.21869457, -0.00198712, 0.00120829, 7.21670745, 0.99389643, 0.99680459, 0.49113374],
                             [- 6.21888149, -0.00198592, 0.00120759, 7.21689557, 0.99390024, 0.99680649, 0.49113965],
                             [- 6.21906775, -0.00198471, 0.00120689, 7.21708304, 0.99390404, 0.99680840, 0.49114555],
                             [- 6.21925338, -0.00198350, 0.00120620, 7.21726987, 0.99390784, 0.99681030, 0.49115142],
                             [- 6.21943836, -0.00198230, 0.00120550, 7.21745606, 0.99391163, 0.99681220, 0.49115728],
                             [- 6.21962271, -0.00198110, 0.00120481, 7.21764161, 0.99391542, 0.99681410, 0.49116312],
                             [- 6.21980642, -0.00197989, 0.00120411, 7.21782653, 0.99391921, 0.99681600, 0.49116895],
                             [- 6.21998950, -0.00197869, 0.00120342, 7.21801081, 0.99392299, 0.99681789, 0.49117474],
                             [- 6.22017194, -0.00197749, 0.00120272, 7.21819445, 0.99392677, 0.99681979, 0.49118052],
                             [- 6.22035376, -0.00197630, 0.00120203, 7.21837746, 0.99393054, 0.99682168, 0.49118629],
                             [- 6.22053495, -0.00197510, 0.00120134, 7.21855985, 0.99393431, 0.99682357, 0.49119203],
                             [- 6.22071551, -0.00197390, 0.00120064, 7.21874161, 0.99393808, 0.99682545, 0.49119776],
                             [- 6.22089545, -0.00197271, 0.00119995, 7.21892274, 0.99394184, 0.99682734, 0.49120347],
                             [- 6.22107477, -0.00197152, 0.00119926, 7.21910325, 0.99394559, 0.99682922, 0.49120916],
                             [- 6.22125347, -0.00197032, 0.00119857, 7.21928314, 0.99394935, 0.99683111, 0.49121482],
                             [- 6.22143155, -0.00196913, 0.00119788, 7.21946242, 0.99395309, 0.99683299, 0.49122047],
                             [- 6.22160902, -0.00196794, 0.00119719, 7.21964107, 0.99395684, 0.99683486, 0.49122611],
                             [- 6.22178587, -0.00196676, 0.00119650, 7.21981911, 0.99396058, 0.99683674, 0.49123173],
                             [- 6.22196211, -0.00196557, 0.00119582, 7.21999654, 0.99396431, 0.99683861, 0.49123732],
                             [- 6.22213774, -0.00196438, 0.00119513, 7.22017336, 0.99396804, 0.99684049, 0.49124290],
                             [- 6.22231277, -0.00196320, 0.00119444, 7.22034957, 0.99397177, 0.99684236, 0.49124846],
                             [- 6.22248719, -0.00196202, 0.00119376, 7.22052517, 0.99397549, 0.99684423, 0.49125400],
                             [- 6.22266101, -0.00196083, 0.00119307, 7.22070017, 0.99397920, 0.99684609, 0.49125953],
                             [- 6.22283422, -0.00195965, 0.00119239, 7.22087457, 0.99398292, 0.99684796, 0.49126504],
                             [- 6.22300684, -0.00195848, 0.00119170, 7.22104836, 0.99398663, 0.99684982, 0.49127053],
                             [- 6.22317886, -0.00195730, 0.00119102, 7.22122156, 0.99399033, 0.99685168, 0.49127600],
                             [- 6.22335028, -0.00195612, 0.00119034, 7.22139416, 0.99399403, 0.99685354, 0.49128145],
                             [- 6.22352111, -0.00195495, 0.00118965, 7.22156617, 0.99399773, 0.99685540, 0.49128689],
                             [- 6.22369135, -0.00195377, 0.00118897, 7.22173758, 0.99400142, 0.99685726, 0.49129231],
                             [- 6.22386100, -0.00195260, 0.00118829, 7.22190841, 0.99400511, 0.99685911, 0.49129771],
                             [- 6.22403007, -0.00195143, 0.00118761, 7.22207864, 0.99400879, 0.99686096, 0.49130310],
                             [- 6.22419854, -0.00195026, 0.00118693, 7.22224829, 0.99401247, 0.99686281, 0.49130846],
                             [- 6.22436644, -0.00194909, 0.00118625, 7.22241735, 0.99401615, 0.99686466, 0.49131381],
                             [- 6.22453375, -0.00194792, 0.00118557, 7.22258583, 0.99401982, 0.99686651, 0.49131914],
                             [- 6.22470048, -0.00194675, 0.00118490, 7.22275373, 0.99402348, 0.99686835, 0.49132446],
                             [- 6.22486664, -0.00194559, 0.00118422, 7.22292105, 0.99402715, 0.99687019, 0.49132975],
                             [- 6.22503222, -0.00194442, 0.00118354, 7.22308780, 0.99403081, 0.99687204, 0.49133503],
                             [- 6.22519722, -0.00194326, 0.00118287, 7.22325396, 0.99403446, 0.99687387, 0.49134030],
                             [- 6.22536166, -0.00194210, 0.00118219, 7.22341956, 0.99403811, 0.99687571, 0.49134554],
                             [- 6.22552552, -0.00194094, 0.00118152, 7.22358458, 0.99404176, 0.99687755, 0.49135077],
                             [- 6.22568881, -0.00193978, 0.00118084, 7.22374904, 0.99404540, 0.99687938, 0.49135598],
                             [- 6.22585154, -0.00193862, 0.00118017, 7.22391292, 0.99404904, 0.99688121, 0.49136117],
                             [- 6.22601370, -0.00193746, 0.00117950, 7.22407624, 0.99405267, 0.99688304, 0.49136635],
                             [- 6.22617531, -0.00193631, 0.00117882, 7.22423900, 0.99405630, 0.99688487, 0.49137152],
                             [- 6.22633635, -0.00193515, 0.00117815, 7.22440119, 0.99405992, 0.99688670, 0.49137666],
                             [- 6.22649683, -0.00193400, 0.00117748, 7.22456283, 0.99406354, 0.99688852, 0.49138179],
                             [- 6.22665675, -0.00193285, 0.00117681, 7.22472390, 0.99406716, 0.99689034, 0.49138690],
                             [- 6.22681612, -0.00193170, 0.00117614, 7.22488442, 0.99407077, 0.99689216, 0.49139199],
                             [- 6.22697494, -0.00193055, 0.00117547, 7.22504439, 0.99407438, 0.99689398, 0.49139707],
                             [- 6.22713320, -0.00192940, 0.00117480, 7.22520380, 0.99407799, 0.99689580, 0.49140214],
                             [- 6.22729092, -0.00192825, 0.00117413, 7.22536266, 0.99408159, 0.99689762, 0.49140718],
                             [- 6.22744808, -0.00192711, 0.00117346, 7.22552098, 0.99408519, 0.99689943, 0.49141221],
                             [- 6.22760470, -0.00192596, 0.00117280, 7.22567874, 0.99408878, 0.99690124, 0.49141723],
                             [- 6.22776078, -0.00192482, 0.00117213, 7.22583596, 0.99409237, 0.99690305, 0.49142222],
                             [- 6.22791631, -0.00192368, 0.00117147, 7.22599264, 0.99409595, 0.99690486, 0.49142719],
                             [- 6.22807131, -0.00192253, 0.00117080, 7.22614877, 0.99409953, 0.99690667, 0.49143216],
                             [- 6.22822576, -0.00192139, 0.00117014, 7.22630437, 0.99410311, 0.99690847, 0.49143711],
                             [- 6.22837968, -0.00192026, 0.00116947, 7.22645942, 0.99410668, 0.99691027, 0.49144204],
                             [- 6.22853306, -0.00191912, 0.00116881, 7.22661394, 0.99411025, 0.99691207, 0.49144696],
                             [- 6.22868591, -0.00191798, 0.00116815, 7.22676793, 0.99411381, 0.99691387, 0.49145186],
                             [- 6.22883823, -0.00191685, 0.00116748, 7.22692138, 0.99411737, 0.99691567, 0.49145674],
                             [- 6.22899002, -0.00191571, 0.00116682, 7.22707430, 0.99412093, 0.99691747, 0.49146161],
                             [- 6.22914128, -0.00191458, 0.00116616, 7.22722670, 0.99412448, 0.99691926, 0.49146647],
                             [- 6.22929201, -0.00191345, 0.00116550, 7.22737856, 0.99412803, 0.99692105, 0.49147130],
                             [- 6.22944222, -0.00191232, 0.00116484, 7.22752990, 0.99413157, 0.99692284, 0.49147613],
                             [- 6.22959190, -0.00191119, 0.00116418, 7.22768071, 0.99413511, 0.99692463, 0.49148093],
                             [- 6.22974107, -0.00191006, 0.00116352, 7.22783101, 0.99413865, 0.99692642, 0.49148571],
                             [- 6.22988971, -0.00190893, 0.00116286, 7.22798078, 0.99414218, 0.99692820, 0.49149049],
                             [- 6.23003784, -0.00190781, 0.00116221, 7.22813003, 0.99414571, 0.99692998, 0.49149525],
                             [- 6.23018545, -0.00190669, 0.00116155, 7.22827877, 0.99414924, 0.99693176, 0.49149999],
                             [- 6.23033255, -0.00190556, 0.00116089, 7.22842699, 0.99415276, 0.99693354, 0.49150472],
                             [- 6.23047913, -0.00190444, 0.00116024, 7.22857469, 0.99415627, 0.99693532, 0.49150944],
                             [- 6.23062521, -0.00190332, 0.00115958, 7.22872189, 0.99415978, 0.99693710, 0.49151413],
                             [- 6.23077077, -0.00190220, 0.00115893, 7.22886857, 0.99416329, 0.99693887, 0.49151881],
                             [- 6.23091583, -0.00190108, 0.00115828, 7.22901475, 0.99416680, 0.99694064, 0.49152348],
                             [- 6.23106038, -0.00189996, 0.00115762, 7.22916042, 0.99417030, 0.99694241, 0.49152813],
                             [- 6.23120443, -0.00189885, 0.00115697, 7.22930558, 0.99417379, 0.99694418, 0.49153277],
                             [- 6.23134798, -0.00189773, 0.00115632, 7.22945024, 0.99417729, 0.99694595, 0.49153739],
                             [- 6.23149102, -0.00189662, 0.00115567, 7.22959440, 0.99418078, 0.99694772, 0.49154200],
                             [- 6.23163357, -0.00189551, 0.00115501, 7.22973806, 0.99418426, 0.99694948, 0.49154659],
                             [- 6.23177562, -0.00189439, 0.00115436, 7.22988122, 0.99418774, 0.99695124, 0.49155116],
                             [- 6.23191717, -0.00189328, 0.00115371, 7.23002389, 0.99419122, 0.99695300, 0.49155572],
                             [- 6.23205823, -0.00189218, 0.00115307, 7.23016606, 0.99419469, 0.99695476, 0.49156028],
                             [- 6.23219880, -0.00189107, 0.00115242, 7.23030773, 0.99419816, 0.99695652, 0.49156481],
                             [- 6.23233888, -0.00188996, 0.00115177, 7.23044891, 0.99420163, 0.99695827, 0.49156932],
                             [- 6.23247846, -0.00188886, 0.00115112, 7.23058961, 0.99420509, 0.99696002, 0.49157383],
                             [- 6.23261756, -0.00188775, 0.00115047, 7.23072981, 0.99420855, 0.99696177, 0.49157832],
                             [- 6.23275618, -0.00188665, 0.00114983, 7.23086953, 0.99421200, 0.99696352, 0.49158279],
                             [- 6.23289431, -0.00188555, 0.00114918, 7.23100876, 0.99421545, 0.99696527, 0.49158725],
                             [- 6.23303196, -0.00188444, 0.00114854, 7.23114751, 0.99421890, 0.99696702, 0.49159170],
                             [- 6.23316912, -0.00188334, 0.00114789, 7.23128578, 0.99422234, 0.99696876, 0.49159612],
                             [- 6.23330581, -0.00188225, 0.00114725, 7.23142357, 0.99422578, 0.99697050, 0.49160053],
                             [- 6.23344202, -0.00188115, 0.00114661, 7.23156087, 0.99422921, 0.99697225, 0.49160494],
                             [- 6.23357776, -0.00188005, 0.00114596, 7.23169771, 0.99423264, 0.99697398, 0.49160932],
                             [- 6.23371302, -0.00187896, 0.00114532, 7.23183406, 0.99423607, 0.99697572, 0.49161370],
                             [- 6.23384781, -0.00187786, 0.00114468, 7.23196994, 0.99423949, 0.99697746, 0.49161806],
                             [- 6.23398212, -0.00187677, 0.00114404, 7.23210535, 0.99424291, 0.99697919, 0.49162241],
                             [- 6.23411597, -0.00187568, 0.00114340, 7.23224029, 0.99424633, 0.99698092, 0.49162674],
                             [- 6.23424935, -0.00187459, 0.00114276, 7.23237476, 0.99424974, 0.99698265, 0.49163105],
                             [- 6.23438226, -0.00187350, 0.00114212, 7.23250876, 0.99425315, 0.99698438, 0.49163535],
                             [- 6.23451471, -0.00187241, 0.00114148, 7.23264230, 0.99425655, 0.99698611, 0.49163964],
                             [- 6.23464669, -0.00187132, 0.00114084, 7.23277537, 0.99425995, 0.99698783, 0.49164392],
                             [- 6.23477821, -0.00187024, 0.00114021, 7.23290797, 0.99426335, 0.99698956, 0.49164818],
                             [- 6.23490927, -0.00186915, 0.00113957, 7.23304012, 0.99426674, 0.99699128, 0.49165242],
                             [- 6.23503988, -0.00186807, 0.00113893, 7.23317181, 0.99427013, 0.99699300, 0.49165666],
                             [- 6.23517002, -0.00186698, 0.00113830, 7.23330304, 0.99427352, 0.99699472, 0.49166087],
                             [- 6.23529971, -0.00186590, 0.00113766, 7.23343381, 0.99427690, 0.99699644, 0.49166508],
                             [- 6.23542895, -0.00186482, 0.00113703, 7.23356413, 0.99428028, 0.99699815, 0.49166927],
                             [- 6.23555773, -0.00186374, 0.00113639, 7.23369399, 0.99428365, 0.99699986, 0.49167345],
                             [- 6.23568606, -0.00186267, 0.00113576, 7.23382340, 0.99428702, 0.99700157, 0.49167762],
                             [- 6.23581395, -0.00186159, 0.00113513, 7.23395236, 0.99429039, 0.99700328, 0.49168176],
                             [- 6.23594138, -0.00186051, 0.00113449, 7.23408087, 0.99429375, 0.99700499, 0.49168591],
                             [- 6.23606837, -0.00185944, 0.00113386, 7.23420893, 0.99429711, 0.99700670, 0.49169003],
                             [- 6.23619491, -0.00185836, 0.00113323, 7.23433655, 0.99430047, 0.99700840, 0.49169415],
                             [- 6.23632102, -0.00185729, 0.00113260, 7.23446372, 0.99430382, 0.99701011, 0.49169823],
                             [- 6.23644667, -0.00185622, 0.00113197, 7.23459045, 0.99430717, 0.99701181, 0.49170232],
                             [- 6.23657189, -0.00185515, 0.00113134, 7.23471674, 0.99431051, 0.99701351, 0.49170640],
                             [- 6.23669667, -0.00185408, 0.00113071, 7.23484259, 0.99431385, 0.99701521, 0.49171044],
                             [- 6.23682101, -0.00185301, 0.00113009, 7.23496800, 0.99431719, 0.99701690, 0.49171449],
                             [- 6.23694492, -0.00185195, 0.00112946, 7.23509297, 0.99432052, 0.99701860, 0.49171853],
                             [- 6.23706839, -0.00185088, 0.00112883, 7.23521751, 0.99432385, 0.99702029, 0.49172254],
                             [- 6.23719143, -0.00184981, 0.00112820, 7.23534161, 0.99432718, 0.99702198, 0.49172655],
                             [- 6.23731403, -0.00184875, 0.00112758, 7.23546528, 0.99433050, 0.99702367, 0.49173054],
                             [- 6.23743621, -0.00184769, 0.00112695, 7.23558852, 0.99433382, 0.99702536, 0.49173452],
                             [- 6.23755795, -0.00184663, 0.00112633, 7.23571133, 0.99433713, 0.99702705, 0.49173848],
                             [- 6.23767927, -0.00184557, 0.00112570, 7.23583371, 0.99434044, 0.99702873, 0.49174243],
                             [- 6.23780017, -0.00184451, 0.00112508, 7.23595566, 0.99434375, 0.99703041, 0.49174639],
                             [- 6.23792064, -0.00184345, 0.00112446, 7.23607719, 0.99434706, 0.99703209, 0.49175031],
                             [- 6.23804068, -0.00184239, 0.00112383, 7.23619829, 0.99435036, 0.99703377, 0.49175422],
                             [- 6.23816031, -0.00184134, 0.00112321, 7.23631897, 0.99435365, 0.99703545, 0.49175813],
                             [- 6.23827952, -0.00184028, 0.00112259, 7.23643923, 0.99435695, 0.99703713, 0.49176201],
                             [- 6.23839830, -0.00183923, 0.00112197, 7.23655907, 0.99436024, 0.99703880, 0.49176589],
                             [- 6.23851667, -0.00183817, 0.00112135, 7.23667850, 0.99436352, 0.99704048, 0.49176975],
                             [- 6.23863462, -0.00183712, 0.00112073, 7.23679750, 0.99436680, 0.99704215, 0.49177361],
                             [- 6.23875216, -0.00183607, 0.00112011, 7.23691609, 0.99437008, 0.99704382, 0.49177744],
                             [- 6.23886929, -0.00183502, 0.00111949, 7.23703427, 0.99437336, 0.99704548, 0.49178127],
                             [- 6.23898600, -0.00183398, 0.00111887, 7.23715203, 0.99437663, 0.99704715, 0.49178508],
                             [- 6.23910231, -0.00183293, 0.00111826, 7.23726938, 0.99437990, 0.99704881, 0.49178889],
                             [- 6.23921820, -0.00183188, 0.00111764, 7.23738632, 0.99438316, 0.99705048, 0.49179268],
                             [- 6.23933369, -0.00183084, 0.00111702, 7.23750286, 0.99438642, 0.99705214, 0.49179645],
                             [- 6.23944877, -0.00182979, 0.00111641, 7.23761898, 0.99438968, 0.99705380, 0.49180022],
                             [- 6.23956345, -0.00182875, 0.00111579, 7.23773470, 0.99439293, 0.99705546, 0.49180397],
                             [- 6.23967773, -0.00182771, 0.00111518, 7.23785002, 0.99439618, 0.99705711, 0.49180771],
                             [- 6.23979160, -0.00182667, 0.00111456, 7.23796493, 0.99439943, 0.99705877, 0.49181144],
                             [- 6.23990507, -0.00182563, 0.00111395, 7.23807944, 0.99440267, 0.99706042, 0.49181516],
                             [- 6.24001814, -0.00182459, 0.00111334, 7.23819355, 0.99440591, 0.99706207, 0.49181886],
                             [- 6.24013082, -0.00182355, 0.00111273, 7.23830726, 0.99440915, 0.99706372, 0.49182255],
                             [- 6.24024309, -0.00182251, 0.00111211, 7.23842058, 0.99441238, 0.99706537, 0.49182623],
                             [- 6.24035498, -0.00182148, 0.00111150, 7.23853350, 0.99441561, 0.99706702, 0.49182990],
                             [- 6.24046647, -0.00182045, 0.00111089, 7.23864602, 0.99441884, 0.99706866, 0.49183355],
                             [- 6.24057756, -0.00181941, 0.00111028, 7.23875815, 0.99442206, 0.99707031, 0.49183719],
                             [- 6.24068827, -0.00181838, 0.00110967, 7.23886989, 0.99442528, 0.99707195, 0.49184083],
                             [- 6.24079858, -0.00181735, 0.00110906, 7.23898123, 0.99442849, 0.99707359, 0.49184446],
                             [- 6.24090851, -0.00181632, 0.00110846, 7.23909219, 0.99443170, 0.99707523, 0.49184806],
                             [- 6.24101805, -0.00181529, 0.00110785, 7.23920276, 0.99443491, 0.99707686, 0.49185165],
                             [- 6.24112720, -0.00181426, 0.00110724, 7.23931294, 0.99443812, 0.99707850, 0.49185524],
                             [- 6.24123597, -0.00181324, 0.00110663, 7.23942274, 0.99444132, 0.99708013, 0.49185881],
                             [- 6.24134436, -0.00181221, 0.00110603, 7.23953215, 0.99444451, 0.99708176, 0.49186237],
                             [- 6.24145237, -0.00181118, 0.00110542, 7.23964118, 0.99444771, 0.99708339, 0.49186593],
                             [- 6.24155999, -0.00181016, 0.00110482, 7.23974983, 0.99445090, 0.99708502, 0.49186947],
                             [- 6.24166723, -0.00180914, 0.00110421, 7.23985810, 0.99445408, 0.99708665, 0.49187299],
                             [- 6.24177410, -0.00180812, 0.00110361, 7.23996598, 0.99445727, 0.99708828, 0.49187651],
                             [- 6.24188059, -0.00180710, 0.00110300, 7.24007349, 0.99446045, 0.99708990, 0.49188001],
                             [- 6.24198671, -0.00180608, 0.00110240, 7.24018063, 0.99446363, 0.99709152, 0.49188351],
                             [- 6.24209245, -0.00180506, 0.00110180, 7.24028739, 0.99446680, 0.99709314, 0.49188699],
                             [- 6.24219781, -0.00180404, 0.00110120, 7.24039377, 0.99446997, 0.99709476, 0.49189046],
                             [- 6.24230281, -0.00180302, 0.00110060, 7.24049978, 0.99447313, 0.99709638, 0.49189392],
                             [- 6.24240743, -0.00180201, 0.00110000, 7.24060543, 0.99447630, 0.99709800, 0.49189737],
                             [- 6.24251169, -0.00180100, 0.00109940, 7.24071070, 0.99447946, 0.99709961, 0.49190081],
                             [- 6.24261558, -0.00179998, 0.00109880, 7.24081560, 0.99448261, 0.99710122, 0.49190424],
                             [- 6.24271910, -0.00179897, 0.00109820, 7.24092013, 0.99448577, 0.99710283, 0.49190765],
                             [- 6.24282226, -0.00179796, 0.00109760, 7.24102430, 0.99448891, 0.99710444, 0.49191105],
                             [- 6.24292505, -0.00179695, 0.00109700, 7.24112810, 0.99449206, 0.99710605, 0.49191444],
                             [- 6.24302748, -0.00179594, 0.00109640, 7.24123154, 0.99449520, 0.99710766, 0.49191782],
                             [- 6.24312955, -0.00179493, 0.00109581, 7.24133462, 0.99449834, 0.99710926, 0.49192119],
                             [- 6.24323126, -0.00179392, 0.00109521, 7.24143733, 0.99450148, 0.99711087, 0.49192455],
                             [- 6.24333261, -0.00179292, 0.00109461, 7.24153969, 0.99450461, 0.99711247, 0.49192789],
                             [- 6.24343360, -0.00179191, 0.00109402, 7.24164168, 0.99450774, 0.99711407, 0.49193123],
                             [- 6.24353423, -0.00179091, 0.00109342, 7.24174332, 0.99451086, 0.99711567, 0.49193456],
                             [- 6.24363451, -0.00178991, 0.00109283, 7.24184460, 0.99451399, 0.99711726, 0.49193787],
                             [- 6.24373444, -0.00178891, 0.00109224, 7.24194553, 0.99451711, 0.99711886, 0.49194119],
                             [- 6.24383401, -0.00178790, 0.00109164, 7.24204610, 0.99452022, 0.99712045, 0.49194448],
                             [- 6.24393323, -0.00178690, 0.00109105, 7.24214632, 0.99452333, 0.99712204, 0.49194776],
                             [- 6.24403210, -0.00178591, 0.00109046, 7.24224619, 0.99452644, 0.99712364, 0.49195103],
                             [- 6.24413062, -0.00178491, 0.00108987, 7.24234571, 0.99452955, 0.99712522, 0.49195429],
                             [- 6.24422879, -0.00178391, 0.00108928, 7.24244488, 0.99453265, 0.99712681, 0.49195755],
                             [- 6.24432661, -0.00178292, 0.00108869, 7.24254370, 0.99453575, 0.99712840, 0.49196079],
                             [- 6.24442409, -0.00178192, 0.00108810, 7.24264217, 0.99453884, 0.99712998, 0.49196402],
                             [- 6.24452123, -0.00178093, 0.00108751, 7.24274030, 0.99454194, 0.99713157, 0.49196723],
                             [- 6.24461802, -0.00177993, 0.00108692, 7.24283808, 0.99454502, 0.99713315, 0.49197045],
                             [- 6.24471446, -0.00177894, 0.00108633, 7.24293552, 0.99454811, 0.99713473, 0.49197365],
                             [- 6.24481057, -0.00177795, 0.00108574, 7.24303262, 0.99455119, 0.99713630, 0.49197684],
                             [- 6.24490634, -0.00177696, 0.00108516, 7.24312938, 0.99455427, 0.99713788, 0.49198002],
                             [- 6.24500177, -0.00177597, 0.00108457, 7.24322579, 0.99455735, 0.99713946, 0.49198318],
                             [- 6.24509686, -0.00177499, 0.00108398, 7.24332187, 0.99456042, 0.99714103, 0.49198634],
                             [- 6.24519161, -0.00177400, 0.00108340, 7.24341761, 0.99456349, 0.99714260, 0.49198949],
                             [- 6.24528603, -0.00177301, 0.00108281, 7.24351302, 0.99456655, 0.99714417, 0.49199263],
                             [- 6.24538012, -0.00177203, 0.00108223, 7.24360809, 0.99456961, 0.99714574, 0.49199575],
                             [- 6.24547387, -0.00177105, 0.00108164, 7.24370283, 0.99457267, 0.99714731, 0.49199887],
                             [- 6.24556729, -0.00177006, 0.00108106, 7.24379723, 0.99457573, 0.99714887, 0.49200198],
                             [- 6.24566038, -0.00176908, 0.00108048, 7.24389130, 0.99457878, 0.99715044, 0.49200507],
                             [- 6.24575314, -0.00176810, 0.00107990, 7.24398504, 0.99458183, 0.99715200, 0.49200817],
                             [- 6.24584557, -0.00176712, 0.00107931, 7.24407845, 0.99458488, 0.99715356, 0.49201123],
                             [- 6.24593768, -0.00176614, 0.00107873, 7.24417153, 0.99458792, 0.99715512, 0.49201431],
                             [- 6.24602946, -0.00176517, 0.00107815, 7.24426429, 0.99459096, 0.99715668, 0.49201737],
                             [- 6.24612091, -0.00176419, 0.00107757, 7.24435672, 0.99459400, 0.99715824, 0.49202041],
                             [- 6.24621204, -0.00176321, 0.00107699, 7.24444883, 0.99459703, 0.99715979, 0.49202344],
                             [- 6.24630285, -0.00176224, 0.00107641, 7.24454061, 0.99460006, 0.99716135, 0.49202647],
                             [- 6.24639333, -0.00176127, 0.00107584, 7.24463206, 0.99460309, 0.99716290, 0.49202949],
                             [- 6.24648350, -0.00176029, 0.00107526, 7.24472320, 0.99460611, 0.99716445, 0.49203250],
                             [- 6.24657334, -0.00175932, 0.00107468, 7.24481402, 0.99460913, 0.99716600, 0.49203550],
                             [- 6.24666287, -0.00175835, 0.00107410, 7.24490451, 0.99461215, 0.99716755, 0.49203848],
                             [- 6.24675208, -0.00175738, 0.00107353, 7.24499469, 0.99461516, 0.99716909, 0.49204146],
                             [- 6.24684097, -0.00175641, 0.00107295, 7.24508455, 0.99461817, 0.99717064, 0.49204443],
                             [- 6.24692955, -0.00175545, 0.00107238, 7.24517410, 0.99462118, 0.99717218, 0.49204739],
                             [- 6.24701781, -0.00175448, 0.00107180, 7.24526333, 0.99462418, 0.99717372, 0.49205033],
                             [- 6.24710576, -0.00175351, 0.00107123, 7.24535225, 0.99462718, 0.99717526, 0.49205328],
                             [- 6.24719340, -0.00175255, 0.00107065, 7.24544085, 0.99463018, 0.99717680, 0.49205621],
                             [- 6.24728073, -0.00175158, 0.00107008, 7.24552914, 0.99463317, 0.99717834, 0.49205913],
                             [- 6.24736775, -0.00175062, 0.00106951, 7.24561712, 0.99463617, 0.99717987, 0.49206204],
                             [- 6.24745446, -0.00174966, 0.00106893, 7.24570479, 0.99463915, 0.99718141, 0.49206494],
                             [- 6.24754086, -0.00174870, 0.00106836, 7.24579216, 0.99464214, 0.99718294, 0.49206783],
                             [- 6.24762695, -0.00174774, 0.00106779, 7.24587921, 0.99464512, 0.99718447, 0.49207072],
                             [- 6.24771274, -0.00174678, 0.00106722, 7.24596596, 0.99464810, 0.99718600, 0.49207359],
                             [- 6.24779823, -0.00174582, 0.00106665, 7.24605241, 0.99465108, 0.99718753, 0.49207645],
                             [- 6.24788341, -0.00174487, 0.00106608, 7.24613854, 0.99465405, 0.99718905, 0.49207931],
                             [- 6.24796829, -0.00174391, 0.00106551, 7.24622438, 0.99465702, 0.99719058, 0.49208216],
                             [- 6.24805287, -0.00174296, 0.00106494, 7.24630991, 0.99465998, 0.99719210, 0.49208499],
                             [- 6.24813715, -0.00174200, 0.00106438, 7.24639515, 0.99466295, 0.99719362, 0.49208783],
                             [- 6.24822113, -0.00174105, 0.00106381, 7.24648008, 0.99466591, 0.99719514, 0.49209064],
                             [- 6.24830481, -0.00174010, 0.00106324, 7.24656471, 0.99466886, 0.99719666, 0.49209345],
                             [- 6.24838819, -0.00173915, 0.00106267, 7.24664905, 0.99467182, 0.99719818, 0.49209625],
                             [- 6.24847128, -0.00173819, 0.00106211, 7.24673309, 0.99467477, 0.99719970, 0.49209904],
                             [- 6.24855407, -0.00173725, 0.00106154, 7.24681683, 0.99467771, 0.99720121, 0.49210182],
                             [- 6.24863657, -0.00173630, 0.00106098, 7.24690027, 0.99468066, 0.99720272, 0.49210459],
                             [- 6.24871878, -0.00173535, 0.00106041, 7.24698343, 0.99468360, 0.99720424, 0.49210735],
                             [- 6.24880069, -0.00173440, 0.00105985, 7.24706629, 0.99468654, 0.99720575, 0.49211010],
                             [- 6.24888231, -0.00173346, 0.00105929, 7.24714886, 0.99468947, 0.99720725, 0.49211286],
                             [- 6.24896365, -0.00173251, 0.00105872, 7.24723113, 0.99469240, 0.99720876, 0.49211559],
                             [- 6.24904469, -0.00173157, 0.00105816, 7.24731312, 0.99469533, 0.99721027, 0.49211832],
                             [- 6.24912545, -0.00173063, 0.00105760, 7.24739482, 0.99469826, 0.99721177, 0.49212104],
                             [- 6.24920591, -0.00172969, 0.00105704, 7.24747623, 0.99470118, 0.99721327, 0.49212375],
                             [- 6.24928610, -0.00172875, 0.00105648, 7.24755735, 0.99470410, 0.99721478, 0.49212645],
                             [- 6.24936599, -0.00172781, 0.00105592, 7.24763819, 0.99470702, 0.99721628, 0.49212915],
                             [- 6.24944561, -0.00172687, 0.00105536, 7.24771874, 0.99470993, 0.99721777, 0.49213183],
                             [- 6.24952494, -0.00172593, 0.00105480, 7.24779901, 0.99471284, 0.99721927, 0.49213451],
                             [- 6.24960399, -0.00172499, 0.00105424, 7.24787899, 0.99471575, 0.99722077, 0.49213717],
                             [- 6.24968275, -0.00172406, 0.00105368, 7.24795870, 0.99471865, 0.99722226, 0.49213983],
                             [- 6.24976124, -0.00172312, 0.00105312, 7.24803812, 0.99472155, 0.99722375, 0.49214248],
                             [- 6.24983945, -0.00172219, 0.00105257, 7.24811726, 0.99472445, 0.99722524, 0.49214512],
                             [- 6.24991738, -0.00172126, 0.00105201, 7.24819612, 0.99472735, 0.99722673, 0.49214776],
                             [- 6.24999503, -0.00172032, 0.00105145, 7.24827471, 0.99473024, 0.99722822, 0.49215038],
                             [- 6.25007241, -0.00171939, 0.00105090, 7.24835302, 0.99473313, 0.99722971, 0.49215299],
                             [- 6.25014951, -0.00171846, 0.00105034, 7.24843105, 0.99473601, 0.99723119, 0.49215561],
                             [- 6.25022634, -0.00171753, 0.00104979, 7.24850880, 0.99473890, 0.99723268, 0.49215820],
                             [- 6.25030289, -0.00171660, 0.00104923, 7.24858628, 0.99474178, 0.99723416, 0.49216079],
                             [- 6.25037917, -0.00171568, 0.00104868, 7.24866349, 0.99474465, 0.99723564, 0.49216338],
                             [- 6.25045518, -0.00171475, 0.00104813, 7.24874043, 0.99474753, 0.99723712, 0.49216595],
                             [- 6.25053092, -0.00171382, 0.00104758, 7.24881709, 0.99475040, 0.99723860, 0.49216851],
                             [- 6.25060639, -0.00171290, 0.00104702, 7.24889349, 0.99475327, 0.99724008, 0.49217106],
                             [- 6.25068159, -0.00171198, 0.00104647, 7.24896961, 0.99475613, 0.99724155, 0.49217362],
                             [- 6.25075652, -0.00171105, 0.00104592, 7.24904547, 0.99475899, 0.99724303, 0.49217616],
                             [- 6.25083119, -0.00171013, 0.00104537, 7.24912106, 0.99476185, 0.99724450, 0.49217869],
                             [- 6.25090559, -0.00170921, 0.00104482, 7.24919638, 0.99476471, 0.99724597, 0.49218121],
                             [- 6.25097972, -0.00170829, 0.00104427, 7.24927143, 0.99476756, 0.99724744, 0.49218373],
                             [- 6.25105359, -0.00170737, 0.00104372, 7.24934622, 0.99477041, 0.99724891, 0.49218624],
                             [- 6.25112720, -0.00170645, 0.00104317, 7.24942075, 0.99477326, 0.99725037, 0.49218873],
                             [- 6.25120055, -0.00170554, 0.00104262, 7.24949501, 0.99477610, 0.99725184, 0.49219122],
                             [- 6.25127363, -0.00170462, 0.00104208, 7.24956901, 0.99477894, 0.99725330, 0.49219371],
                             [- 6.25134646, -0.00170370, 0.00104153, 7.24964275, 0.99478178, 0.99725477, 0.49219618],
                             [- 6.25141902, -0.00170279, 0.00104098, 7.24971623, 0.99478462, 0.99725623, 0.49219865],
                             [- 6.25149133, -0.00170188, 0.00104044, 7.24978945, 0.99478745, 0.99725769, 0.49220112],
                             [- 6.25156338, -0.00170096, 0.00103989, 7.24986241, 0.99479028, 0.99725915, 0.49220356],
                             [- 6.25163517, -0.00170005, 0.00103935, 7.24993512, 0.99479311, 0.99726060, 0.49220600],
                             [- 6.25170670, -0.00169914, 0.00103880, 7.25000756, 0.99479593, 0.99726206, 0.49220844],
                             [- 6.25177798, -0.00169823, 0.00103826, 7.25007975, 0.99479875, 0.99726351, 0.49221087],
                             [- 6.25184901, -0.00169732, 0.00103771, 7.25015169, 0.99480157, 0.99726496, 0.49221329],
                             [- 6.25191978, -0.00169641, 0.00103717, 7.25022337, 0.99480438, 0.99726642, 0.49221570],
                             [- 6.25199031, -0.00169551, 0.00103663, 7.25029480, 0.99480719, 0.99726787, 0.49221811],
                             [- 6.25206057, -0.00169460, 0.00103609, 7.25036598, 0.99481000, 0.99726931, 0.49222050],
                             [- 6.25213059, -0.00169369, 0.00103555, 7.25043690, 0.99481281, 0.99727076, 0.49222289],
                             [- 6.25220036, -0.00169279, 0.00103500, 7.25050757, 0.99481561, 0.99727221, 0.49222527],
                             [- 6.25226988, -0.00169189, 0.00103446, 7.25057800, 0.99481841, 0.99727365, 0.49222765],
                             [- 6.25233916, -0.00169098, 0.00103392, 7.25064817, 0.99482121, 0.99727509, 0.49223001],
                             [- 6.25240818, -0.00169008, 0.00103338, 7.25071810, 0.99482400, 0.99727653, 0.49223237],
                             [- 6.25247696, -0.00168918, 0.00103285, 7.25078778, 0.99482680, 0.99727797, 0.49223472],
                             [- 6.25254550, -0.00168828, 0.00103231, 7.25085722, 0.99482958, 0.99727941, 0.49223706],
                             [- 6.25261379, -0.00168738, 0.00103177, 7.25092640, 0.99483237, 0.99728085, 0.49223939],
                             [- 6.25268183, -0.00168648, 0.00103123, 7.25099535, 0.99483515, 0.99728229, 0.49224172],
                             [- 6.25274963, -0.00168559, 0.00103069, 7.25106405, 0.99483793, 0.99728372, 0.49224404],
                             [- 6.25281720, -0.00168469, 0.00103016, 7.25113251, 0.99484071, 0.99728515, 0.49224636],
                             [- 6.25288452, -0.00168379, 0.00102962, 7.25120072, 0.99484348, 0.99728659, 0.49224866],
                             [- 6.25295160, -0.00168290, 0.00102909, 7.25126870, 0.99484626, 0.99728802, 0.49225097],
                             [- 6.25301844, -0.00168200, 0.00102855, 7.25133643, 0.99484902, 0.99728945, 0.49225325],
                             [- 6.25308504, -0.00168111, 0.00102802, 7.25140393, 0.99485179, 0.99729087, 0.49225553],
                             [- 6.25315140, -0.00168022, 0.00102748, 7.25147118, 0.99485455, 0.99729230, 0.49225780],
                             [- 6.25321753, -0.00167933, 0.00102695, 7.25153820, 0.99485731, 0.99729372, 0.49226008],
                             [- 6.25328342, -0.00167844, 0.00102641, 7.25160498, 0.99486007, 0.99729515, 0.49226233],
                             [- 6.25334907, -0.00167755, 0.00102588, 7.25167153, 0.99486282, 0.99729657, 0.49226459],
                             [- 6.25341450, -0.00167666, 0.00102535, 7.25173784, 0.99486558, 0.99729799, 0.49226684],
                             [- 6.25347968, -0.00167577, 0.00102482, 7.25180391, 0.99486832, 0.99729941, 0.49226907],
                             [- 6.25354464, -0.00167489, 0.00102429, 7.25186975, 0.99487107, 0.99730083, 0.49227131],
                             [- 6.25360936, -0.00167400, 0.00102376, 7.25193536, 0.99487381, 0.99730224, 0.49227354],
                             [- 6.25367385, -0.00167312, 0.00102323, 7.25200074, 0.99487655, 0.99730366, 0.49227575],
                             [- 6.25373811, -0.00167223, 0.00102270, 7.25206588, 0.99487929, 0.99730507, 0.49227796],
                             [- 6.25380215, -0.00167135, 0.00102217, 7.25213080, 0.99488202, 0.99730649, 0.49228016],
                             [- 6.25386595, -0.00167047, 0.00102164, 7.25219548, 0.99488476, 0.99730790, 0.49228236],
                             [- 6.25392952, -0.00166958, 0.00102111, 7.25225994, 0.99488749, 0.99730931, 0.49228455],
                             [- 6.25399287, -0.00166870, 0.00102058, 7.25232417, 0.99489021, 0.99731072, 0.49228673],
                             [- 6.25405599, -0.00166782, 0.00102005, 7.25238817, 0.99489294, 0.99731212, 0.49228892],
                             [- 6.25411889, -0.00166694, 0.00101953, 7.25245194, 0.99489566, 0.99731353, 0.49229108],
                             [- 6.25418156, -0.00166607, 0.00101900, 7.25251549, 0.99489837, 0.99731493, 0.49229324],
                             [- 6.25424401, -0.00166519, 0.00101847, 7.25257882, 0.99490109, 0.99731634, 0.49229539],
                             [- 6.25430623, -0.00166431, 0.00101795, 7.25264192, 0.99490380, 0.99731774, 0.49229754],
                             [- 6.25436823, -0.00166344, 0.00101742, 7.25270479, 0.99490651, 0.99731914, 0.49229968],
                             [- 6.25443001, -0.00166256, 0.00101690, 7.25276745, 0.99490922, 0.99732054, 0.49230182],
                             [- 6.25449157, -0.00166169, 0.00101638, 7.25282988, 0.99491192, 0.99732194, 0.49230394],
                             [- 6.25455291, -0.00166082, 0.00101585, 7.25289209, 0.99491462, 0.99732333, 0.49230606],
                             [- 6.25461403, -0.00165994, 0.00101533, 7.25295408, 0.99491732, 0.99732473, 0.49230817],
                             [- 6.25467493, -0.00165907, 0.00101481, 7.25301586, 0.99492002, 0.99732612, 0.49231027],
                             [- 6.25473561, -0.00165820, 0.00101428, 7.25307741, 0.99492271, 0.99732751, 0.49231238],
                             [- 6.25479608, -0.00165733, 0.00101376, 7.25313874, 0.99492540, 0.99732890, 0.49231446],
                             [- 6.25485633, -0.00165647, 0.00101324, 7.25319986, 0.99492809, 0.99733029, 0.49231656],
                             [- 6.25491636, -0.00165560, 0.00101272, 7.25326076, 0.99493077, 0.99733168, 0.49231863],
                             [- 6.25497618, -0.00165473, 0.00101220, 7.25332145, 0.99493345, 0.99733307, 0.49232070],
                             [- 6.25503579, -0.00165386, 0.00101168, 7.25338192, 0.99493613, 0.99733445, 0.49232277],
                             [- 6.25509518, -0.00165300, 0.00101116, 7.25344218, 0.99493881, 0.99733584, 0.49232483],
                             [- 6.25515436, -0.00165214, 0.00101064, 7.25350222, 0.99494148, 0.99733722, 0.49232688],
                             [- 6.25521333, -0.00165127, 0.00101012, 7.25356206, 0.99494415, 0.99733860, 0.49232894],
                             [- 6.25527209, -0.00165041, 0.00100961, 7.25362168, 0.99494682, 0.99733998, 0.49233097],
                             [- 6.25533063, -0.00164955, 0.00100909, 7.25368108, 0.99494949, 0.99734136, 0.49233301],
                             [- 6.25538897, -0.00164869, 0.00100857, 7.25374028, 0.99495215, 0.99734274, 0.49233504],
                             [- 6.25544710, -0.00164783, 0.00100806, 7.25379927, 0.99495481, 0.99734412, 0.49233704],
                             [- 6.25550502, -0.00164697, 0.00100754, 7.25385805, 0.99495747, 0.99734549, 0.49233906],
                             [- 6.25556273, -0.00164611, 0.00100702, 7.25391662, 0.99496012, 0.99734687, 0.49234107],
                             [- 6.25562024, -0.00164525, 0.00100651, 7.25397499, 0.99496278, 0.99734824, 0.49234308],
                             [- 6.25567754, -0.00164440, 0.00100599, 7.25403314, 0.99496543, 0.99734961, 0.49234506],
                             [- 6.25573464, -0.00164354, 0.00100548, 7.25409110, 0.99496807, 0.99735098, 0.49234706],
                             [- 6.25579153, -0.00164268, 0.00100497, 7.25414884, 0.99497072, 0.99735235, 0.49234904],
                             [- 6.25584821, -0.00164183, 0.00100445, 7.25420638, 0.99497336, 0.99735372, 0.49235101],
                             [- 6.25590470, -0.00164098, 0.00100394, 7.25426372, 0.99497600, 0.99735508, 0.49235297],
                             [- 6.25596098, -0.00164012, 0.00100343, 7.25432086, 0.99497863, 0.99735645, 0.49235495],
                             [- 6.25601706, -0.00163927, 0.00100292, 7.25437779, 0.99498127, 0.99735781, 0.49235690],
                             [- 6.25607294, -0.00163842, 0.00100241, 7.25443452, 0.99498390, 0.99735917, 0.49235884],
                             [- 6.25612862, -0.00163757, 0.00100189, 7.25449105, 0.99498652, 0.99736053, 0.49236079],
                             [- 6.25618410, -0.00163672, 0.00100138, 7.25454738, 0.99498915, 0.99736189, 0.49236272],
                             [- 6.25623938, -0.00163587, 0.00100087, 7.25460351, 0.99499177, 0.99736325, 0.49236466],
                             [- 6.25629447, -0.00163503, 0.00100036, 7.25465944, 0.99499439, 0.99736461, 0.49236658],
                             [- 6.25634936, -0.00163418, 0.00099986, 7.25471518, 0.99499701, 0.99736596, 0.49236850],
                             [- 6.25640405, -0.00163333, 0.00099935, 7.25477071, 0.99499963, 0.99736732, 0.49237042],
                             [- 6.25645854, -0.00163249, 0.00099884, 7.25482605, 0.99500224, 0.99736867, 0.49237232],
                             [- 6.25651284, -0.00163164, 0.00099833, 7.25488119, 0.99500485, 0.99737002, 0.49237422],
                             [- 6.25656694, -0.00163080, 0.00099782, 7.25493614, 0.99500745, 0.99737137, 0.49237612],
                             [- 6.25662085, -0.00162996, 0.00099732, 7.25499089, 0.99501006, 0.99737272, 0.49237800],
                             [- 6.25667457, -0.00162912, 0.00099681, 7.25504545, 0.99501266, 0.99737407, 0.49237989],
                             [- 6.25672809, -0.00162828, 0.00099631, 7.25509982, 0.99501526, 0.99737542, 0.49238176],
                             [- 6.25678143, -0.00162744, 0.00099580, 7.25515399, 0.99501786, 0.99737676, 0.49238364],
                             [- 6.25683457, -0.00162660, 0.00099530, 7.25520797, 0.99502045, 0.99737811, 0.49238550],
                             [- 6.25688752, -0.00162576, 0.00099479, 7.25526176, 0.99502304, 0.99737945, 0.49238736],
                             [- 6.25694028, -0.00162492, 0.00099429, 7.25531536, 0.99502563, 0.99738079, 0.49238921],
                             [- 6.25699285, -0.00162408, 0.00099378, 7.25536877, 0.99502822, 0.99738213, 0.49239105],
                             [- 6.25704524, -0.00162325, 0.00099328, 7.25542199, 0.99503080, 0.99738347, 0.49239289],
                             [- 6.25709743, -0.00162241, 0.00099278, 7.25547502, 0.99503338, 0.99738481, 0.49239473],
                             [- 6.25714944, -0.00162158, 0.00099228, 7.25552786, 0.99503596, 0.99738615, 0.49239656],
                             [- 6.25720127, -0.00162074, 0.00099177, 7.25558052, 0.99503853, 0.99738748, 0.49239838],
                             [- 6.25725290, -0.00161991, 0.00099127, 7.25563299, 0.99504111, 0.99738881, 0.49240020],
                             [- 6.25730435, -0.00161908, 0.00099077, 7.25568527, 0.99504368, 0.99739015, 0.49240201],
                             [- 6.25735562, -0.00161825, 0.00099027, 7.25573737, 0.99504625, 0.99739148, 0.49240381],
                             [- 6.25740670, -0.00161742, 0.00098977, 7.25578929, 0.99504881, 0.99739281, 0.49240561],
                             [- 6.25745761, -0.00161659, 0.00098927, 7.25584102, 0.99505137, 0.99739414, 0.49240741],
                             [- 6.25750832, -0.00161576, 0.00098877, 7.25589256, 0.99505393, 0.99739547, 0.49240919],
                             [- 6.25755886, -0.00161493, 0.00098828, 7.25594393, 0.99505649, 0.99739679, 0.49241098],
                             [- 6.25760921, -0.00161411, 0.00098778, 7.25599511, 0.99505905, 0.99739812, 0.49241276],
                             [- 6.25765939, -0.00161328, 0.00098728, 7.25604611, 0.99506160, 0.99739944, 0.49241453],
                             [- 6.25770938, -0.00161245, 0.00098678, 7.25609693, 0.99506415, 0.99740076, 0.49241630],
                             [- 6.25775920, -0.00161163, 0.00098629, 7.25614757, 0.99506670, 0.99740209, 0.49241806],
                             [- 6.25780883, -0.00161080, 0.00098579, 7.25619803, 0.99506924, 0.99740341, 0.49241981],
                             [- 6.25785829, -0.00160998, 0.00098529, 7.25624831, 0.99507179, 0.99740472, 0.49242156],
                             [- 6.25790757, -0.00160916, 0.00098480, 7.25629841, 0.99507433, 0.99740604, 0.49242330],
                             [- 6.25795667, -0.00160834, 0.00098430, 7.25634834, 0.99507686, 0.99740736, 0.49242504],
                             [- 6.25800560, -0.00160752, 0.00098381, 7.25639808, 0.99507940, 0.99740867, 0.49242678],
                             [- 6.25805435, -0.00160670, 0.00098332, 7.25644766, 0.99508193, 0.99740999, 0.49242849],
                             [- 6.25810293, -0.00160588, 0.00098282, 7.25649705, 0.99508446, 0.99741130, 0.49243022],
                             [- 6.25815133, -0.00160506, 0.00098233, 7.25654627, 0.99508699, 0.99741261, 0.49243193],
                             [- 6.25819956, -0.00160424, 0.00098184, 7.25659532, 0.99508951, 0.99741392, 0.49243365],
                             [- 6.25824762, -0.00160342, 0.00098134, 7.25664419, 0.99509203, 0.99741523, 0.49243535],
                             [- 6.25829550, -0.00160261, 0.00098085, 7.25669289, 0.99509455, 0.99741654, 0.49243704],
                             [- 6.25834321, -0.00160179, 0.00098036, 7.25674142, 0.99509707, 0.99741784, 0.49243875],
                             [- 6.25839075, -0.00160098, 0.00097987, 7.25678978, 0.99509959, 0.99741915, 0.49244043],
                             [- 6.25843812, -0.00160017, 0.00097938, 7.25683796, 0.99510210, 0.99742045, 0.49244213],
                             [- 6.25848532, -0.00159935, 0.00097889, 7.25688597, 0.99510461, 0.99742176, 0.49244379],
                             [- 6.25853235, -0.00159854, 0.00097840, 7.25693381, 0.99510712, 0.99742306, 0.49244547],
                             [- 6.25857922, -0.00159773, 0.00097791, 7.25698149, 0.99510962, 0.99742436, 0.49244714],
                             [- 6.25862591, -0.00159692, 0.00097742, 7.25702899, 0.99511212, 0.99742566, 0.49244880],
                             [- 6.25867244, -0.00159611, 0.00097693, 7.25707633, 0.99511462, 0.99742696, 0.49245046],
                             [- 6.25871879, -0.00159530, 0.00097645, 7.25712350, 0.99511712, 0.99742825, 0.49245210],
                             [- 6.25876499, -0.00159449, 0.00097596, 7.25717050, 0.99511962, 0.99742955, 0.49245376],
                             [- 6.25881101, -0.00159368, 0.00097547, 7.25721733, 0.99512211, 0.99743084, 0.49245539],
                             [- 6.25885688, -0.00159288, 0.00097499, 7.25726400, 0.99512460, 0.99743214, 0.49245702],
                             [- 6.25890257, -0.00159207, 0.00097450, 7.25731050, 0.99512709, 0.99743343, 0.49245866],
                             [- 6.25894810, -0.00159127, 0.00097402, 7.25735684, 0.99512957, 0.99743472, 0.49246028],
                             [- 6.25899347, -0.00159046, 0.00097353, 7.25740301, 0.99513205, 0.99743601, 0.49246190],
                             [- 6.25903868, -0.00158966, 0.00097305, 7.25744902, 0.99513453, 0.99743730, 0.49246352],
                             [- 6.25908372, -0.00158886, 0.00097256, 7.25749487, 0.99513701, 0.99743858, 0.49246513],
                             [- 6.25912861, -0.00158805, 0.00097208, 7.25754055, 0.99513949, 0.99743987, 0.49246673],
                             [- 6.25917333, -0.00158725, 0.00097159, 7.25758607, 0.99514196, 0.99744115, 0.49246833],
                             [- 6.25921789, -0.00158645, 0.00097111, 7.25763144, 0.99514443, 0.99744244, 0.49246993],
                             [- 6.25926229, -0.00158565, 0.00097063, 7.25767664, 0.99514690, 0.99744372, 0.49247152],
                             [- 6.25930653, -0.00158485, 0.00097015, 7.25772168, 0.99514936, 0.99744500, 0.49247309],
                             [- 6.25935061, -0.00158406, 0.00096966, 7.25776656, 0.99515183, 0.99744628, 0.49247468],
                             [- 6.25939453, -0.00158326, 0.00096918, 7.25781128, 0.99515429, 0.99744756, 0.49247625],
                             [- 6.25943830, -0.00158246, 0.00096870, 7.25785584, 0.99515675, 0.99744884, 0.49247782],
                             [- 6.25948191, -0.00158167, 0.00096822, 7.25790024, 0.99515920, 0.99745011, 0.49247939],
                             [- 6.25952536, -0.00158087, 0.00096774, 7.25794449, 0.99516165, 0.99745139, 0.49248095],
                             [- 6.25956866, -0.00158008, 0.00096726, 7.25798858, 0.99516411, 0.99745266, 0.49248251],
                             [- 6.25961180, -0.00157928, 0.00096678, 7.25803252, 0.99516655, 0.99745393, 0.49248405],
                             [- 6.25965478, -0.00157849, 0.00096631, 7.25807629, 0.99516900, 0.99745521, 0.49248560],
                             [- 6.25969761, -0.00157770, 0.00096583, 7.25811992, 0.99517144, 0.99745648, 0.49248715],
                             [- 6.25974029, -0.00157691, 0.00096535, 7.25816339, 0.99517389, 0.99745774, 0.49248867],
                             [- 6.25978282, -0.00157612, 0.00096487, 7.25820670, 0.99517633, 0.99745901, 0.49249021],
                             [- 6.25982519, -0.00157533, 0.00096440, 7.25824986, 0.99517876, 0.99746028, 0.49249174],
                             [- 6.25986741, -0.00157454, 0.00096392, 7.25829287, 0.99518120, 0.99746154, 0.49249325],
                             [- 6.25990947, -0.00157375, 0.00096344, 7.25833572, 0.99518363, 0.99746281, 0.49249477],
                             [- 6.25995139, -0.00157296, 0.00096297, 7.25837843, 0.99518606, 0.99746407, 0.49249628],
                             [- 6.25999315, -0.00157217, 0.00096249, 7.25842098, 0.99518848, 0.99746533, 0.49249779],
                             [- 6.26003477, -0.00157139, 0.00096202, 7.25846338, 0.99519091, 0.99746659, 0.49249929],
                             [- 6.26007624, -0.00157060, 0.00096154, 7.25850563, 0.99519333, 0.99746785, 0.49250079],
                             [- 6.26011755, -0.00156982, 0.00096107, 7.25854773, 0.99519575, 0.99746911, 0.49250228],
                             [- 6.26015872, -0.00156903, 0.00096060, 7.25858969, 0.99519817, 0.99747037, 0.49250377],
                             [- 6.26019974, -0.00156825, 0.00096012, 7.25863149, 0.99520058, 0.99747163, 0.49250524],
                             [- 6.26024061, -0.00156747, 0.00095965, 7.25867315, 0.99520300, 0.99747288, 0.49250672],
                             [- 6.26028134, -0.00156669, 0.00095918, 7.25871465, 0.99520541, 0.99747414, 0.49250819],
                             [- 6.26032192, -0.00156591, 0.00095871, 7.25875601, 0.99520782, 0.99747539, 0.49250967],
                             [- 6.26036235, -0.00156513, 0.00095823, 7.25879723, 0.99521022, 0.99747664, 0.49251114],
                             [- 6.26040264, -0.00156435, 0.00095776, 7.25883830, 0.99521263, 0.99747789, 0.49251259],
                             [- 6.26044279, -0.00156357, 0.00095729, 7.25887922, 0.99521503, 0.99747914, 0.49251404],
                             [- 6.26048279, -0.00156279, 0.00095682, 7.25892000, 0.99521743, 0.99748039, 0.49251549],
                             [- 6.26052264, -0.00156201, 0.00095635, 7.25896063, 0.99521982, 0.99748163, 0.49251694],
                             [- 6.26056235, -0.00156124, 0.00095588, 7.25900112, 0.99522222, 0.99748288, 0.49251838],
                             [- 6.26060192, -0.00156046, 0.00095542, 7.25904146, 0.99522461, 0.99748412, 0.49251982],
                             [- 6.26064135, -0.00155969, 0.00095495, 7.25908167, 0.99522700, 0.99748537, 0.49252125],
                             [- 6.26068064, -0.00155891, 0.00095448, 7.25912173, 0.99522939, 0.99748661, 0.49252269],
                             [- 6.26071978, -0.00155814, 0.00095401, 7.25916164, 0.99523177, 0.99748785, 0.49252410],
                             [- 6.26075879, -0.00155737, 0.00095354, 7.25920142, 0.99523415, 0.99748909, 0.49252552],
                             [- 6.26079765, -0.00155659, 0.00095308, 7.25924106, 0.99523653, 0.99749033, 0.49252694],
                             [- 6.26083637, -0.00155582, 0.00095261, 7.25928055, 0.99523891, 0.99749157, 0.49252834],
                             [- 6.26087496, -0.00155505, 0.00095215, 7.25931991, 0.99524129, 0.99749280, 0.49252975],
                             [- 6.26091341, -0.00155428, 0.00095168, 7.25935912, 0.99524366, 0.99749404, 0.49253116],
                             [- 6.26095171, -0.00155351, 0.00095122, 7.25939820, 0.99524603, 0.99749527, 0.49253255],
                             [- 6.26098988, -0.00155274, 0.00095075, 7.25943714, 0.99524840, 0.99749651, 0.49253394],
                             [- 6.26102792, -0.00155198, 0.00095029, 7.25947594, 0.99525077, 0.99749774, 0.49253533],
                             [- 6.26106581, -0.00155121, 0.00094982, 7.25951460, 0.99525313, 0.99749897, 0.49253672],
                             [- 6.26110357, -0.00155044, 0.00094936, 7.25955313, 0.99525549, 0.99750020, 0.49253810],
                             [- 6.26114120, -0.00154968, 0.00094890, 7.25959152, 0.99525785, 0.99750143, 0.49253947],
                             [- 6.26117869, -0.00154891, 0.00094843, 7.25962977, 0.99526021, 0.99750265, 0.49254085],
                             [- 6.26121604, -0.00154815, 0.00094797, 7.25966789, 0.99526256, 0.99750388, 0.49254220],
                             [- 6.26125326, -0.00154739, 0.00094751, 7.25970587, 0.99526492, 0.99750510, 0.49254357],
                             [- 6.26129035, -0.00154662, 0.00094705, 7.25974372, 0.99526727, 0.99750633, 0.49254493],
                             [- 6.26132730, -0.00154586, 0.00094659, 7.25978144, 0.99526962, 0.99750755, 0.49254628],
                             [- 6.26136412, -0.00154510, 0.00094613, 7.25981902, 0.99527196, 0.99750877, 0.49254763],
                             [- 6.26140081, -0.00154434, 0.00094567, 7.25985647, 0.99527431, 0.99750999, 0.49254897],
                             [- 6.26143736, -0.00154358, 0.00094521, 7.25989378, 0.99527665, 0.99751121, 0.49255032],
                             [- 6.26147379, -0.00154282, 0.00094475, 7.25993097, 0.99527899, 0.99751243, 0.49255166],
                             [- 6.26151008, -0.00154206, 0.00094429, 7.25996802, 0.99528132, 0.99751365, 0.49255299],
                             [- 6.26154624, -0.00154131, 0.00094383, 7.26000494, 0.99528366, 0.99751486, 0.49255432],
                             [- 6.26158228, -0.00154055, 0.00094337, 7.26004173, 0.99528599, 0.99751608, 0.49255564],
                             [- 6.26161818, -0.00153979, 0.00094292, 7.26007839, 0.99528832, 0.99751729, 0.49255696],
                             [- 6.26165395, -0.00153904, 0.00094246, 7.26011492, 0.99529065, 0.99751851, 0.49255828],
                             [- 6.26168960, -0.00153828, 0.00094200, 7.26015132, 0.99529297, 0.99751972, 0.49255958],
                             [- 6.26172512, -0.00153753, 0.00094155, 7.26018759, 0.99529530, 0.99752093, 0.49256090],
                             [- 6.26176051, -0.00153677, 0.00094109, 7.26022373, 0.99529762, 0.99752214, 0.49256219],
                             [- 6.26179577, -0.00153602, 0.00094063, 7.26025975, 0.99529994, 0.99752334, 0.49256349],
                             [- 6.26183090, -0.00153527, 0.00094018, 7.26029563, 0.99530226, 0.99752455, 0.49256478],
                             [- 6.26186591, -0.00153452, 0.00093972, 7.26033139, 0.99530457, 0.99752576, 0.49256609],
                             [- 6.26190080, -0.00153377, 0.00093927, 7.26036703, 0.99530688, 0.99752696, 0.49256738],
                             [- 6.26193555, -0.00153302, 0.00093882, 7.26040254, 0.99530919, 0.99752817, 0.49256866],
                             [- 6.26197019, -0.00153227, 0.00093836, 7.26043792, 0.99531150, 0.99752937, 0.49256993],
                             [- 6.26200470, -0.00153152, 0.00093791, 7.26047317, 0.99531381, 0.99753057, 0.49257121],
                             [- 6.26203908, -0.00153077, 0.00093746, 7.26050831, 0.99531611, 0.99753177, 0.49257249],
                             [- 6.26207334, -0.00153003, 0.00093700, 7.26054331, 0.99531841, 0.99753297, 0.49257375],
                             [- 6.26210748, -0.00152928, 0.00093655, 7.26057820, 0.99532071, 0.99753417, 0.49257501],
                             [- 6.26214149, -0.00152853, 0.00093610, 7.26061296, 0.99532301, 0.99753537, 0.49257629],
                             [- 6.26217539, -0.00152779, 0.00093565, 7.26064760, 0.99532530, 0.99753656, 0.49257754],
                             [- 6.26220916, -0.00152705, 0.00093520, 7.26068211, 0.99532759, 0.99753776, 0.49257879],
                             [- 6.26224280, -0.00152630, 0.00093475, 7.26071650, 0.99532988, 0.99753895, 0.49258003],
                             [- 6.26227633, -0.00152556, 0.00093430, 7.26075077, 0.99533217, 0.99754014, 0.49258128],
                             [- 6.26230974, -0.00152482, 0.00093385, 7.26078492, 0.99533446, 0.99754133, 0.49258252],
                             [- 6.26234303, -0.00152408, 0.00093340, 7.26081895, 0.99533674, 0.99754253, 0.49258376],
                             [- 6.26237620, -0.00152333, 0.00093295, 7.26085286, 0.99533902, 0.99754372, 0.49258498],
                             [- 6.26240924, -0.00152259, 0.00093250, 7.26088665, 0.99534130, 0.99754490, 0.49258622],
                             [- 6.26244217, -0.00152186, 0.00093205, 7.26092032, 0.99534358, 0.99754609, 0.49258745],
                             [- 6.26247499, -0.00152112, 0.00093161, 7.26095387, 0.99534586, 0.99754728, 0.49258868],
                             [- 6.26250768, -0.00152038, 0.00093116, 7.26098730, 0.99534813, 0.99754846, 0.49258989],
                             [- 6.26254025, -0.00151964, 0.00093071, 7.26102061, 0.99535040, 0.99754965, 0.49259110],
                             [- 6.26257271, -0.00151890, 0.00093027, 7.26105381, 0.99535267, 0.99755083, 0.49259230],
                             [- 6.26260505, -0.00151817, 0.00092982, 7.26108689, 0.99535493, 0.99755201, 0.49259351],
                             [- 6.26263728, -0.00151743, 0.00092937, 7.26111985, 0.99535720, 0.99755319, 0.49259471],
                             [- 6.26266939, -0.00151670, 0.00092893, 7.26115269, 0.99535946, 0.99755437, 0.49259592],
                             [- 6.26270138, -0.00151596, 0.00092848, 7.26118542, 0.99536172, 0.99755555, 0.49259711],
                             [- 6.26273326, -0.00151523, 0.00092804, 7.26121803, 0.99536398, 0.99755673, 0.49259829],
                             [- 6.26276503, -0.00151450, 0.00092760, 7.26125053, 0.99536623, 0.99755790, 0.49259950],
                             [- 6.26279668, -0.00151377, 0.00092715, 7.26128291, 0.99536849, 0.99755908, 0.49260067],
                             [- 6.26282822, -0.00151304, 0.00092671, 7.26131518, 0.99537074, 0.99756025, 0.49260186],
                             [- 6.26285964, -0.00151231, 0.00092627, 7.26134733, 0.99537299, 0.99756143, 0.49260303],
                             [- 6.26289095, -0.00151158, 0.00092582, 7.26137937, 0.99537524, 0.99756260, 0.49260421],
                             [- 6.26292215, -0.00151085, 0.00092538, 7.26141130, 0.99537748, 0.99756377, 0.49260538],
                             [- 6.26295323, -0.00151012, 0.00092494, 7.26144311, 0.99537972, 0.99756494, 0.49260653],
                             [- 6.26298421, -0.00150939, 0.00092450, 7.26147481, 0.99538196, 0.99756611, 0.49260770],
                             [- 6.26301507, -0.00150866, 0.00092406, 7.26150640, 0.99538420, 0.99756728, 0.49260887],
                             [- 6.26304582, -0.00150794, 0.00092362, 7.26153788, 0.99538644, 0.99756844, 0.49261002],
                             [- 6.26307646, -0.00150721, 0.00092318, 7.26156925, 0.99538867, 0.99756961, 0.49261117],
                             [- 6.26310699, -0.00150649, 0.00092274, 7.26160050, 0.99539091, 0.99757078, 0.49261231],
                             [- 6.26313741, -0.00150576, 0.00092230, 7.26163165, 0.99539314, 0.99757194, 0.49261347],
                             [- 6.26316772, -0.00150504, 0.00092186, 7.26166269, 0.99539536, 0.99757310, 0.49261461],
                             [- 6.26319793, -0.00150431, 0.00092142, 7.26169361, 0.99539759, 0.99757426, 0.49261575],
                             [- 6.26322802, -0.00150359, 0.00092098, 7.26172443, 0.99539981, 0.99757542, 0.49261688],
                             [- 6.26325801, -0.00150287, 0.00092055, 7.26175513, 0.99540204, 0.99757658, 0.49261801],
                             [- 6.26328788, -0.00150215, 0.00092011, 7.26178573, 0.99540426, 0.99757774, 0.49261913],
                             [- 6.26331765, -0.00150143, 0.00091967, 7.26181623, 0.99540647, 0.99757890, 0.49262025],
                             [- 6.26334732, -0.00150071, 0.00091924, 7.26184661, 0.99540869, 0.99758006, 0.49262138],
                             [- 6.26337687, -0.00149999, 0.00091880, 7.26187688, 0.99541090, 0.99758121, 0.49262250],
                             [- 6.26340633, -0.00149927, 0.00091836, 7.26190705, 0.99541311, 0.99758236, 0.49262361],
                             [- 6.26343567, -0.00149855, 0.00091793, 7.26193712, 0.99541532, 0.99758352, 0.49262472],
                             [- 6.26346491, -0.00149784, 0.00091749, 7.26196707, 0.99541753, 0.99758467, 0.49262583],
                             [- 6.26349404, -0.00149712, 0.00091706, 7.26199692, 0.99541974, 0.99758582, 0.49262693],
                             [- 6.26352307, -0.00149640, 0.00091662, 7.26202667, 0.99542194, 0.99758697, 0.49262803],
                             [- 6.26355200, -0.00149569, 0.00091619, 7.26205631, 0.99542414, 0.99758812, 0.49262913],
                             [- 6.26358082, -0.00149498, 0.00091576, 7.26208585, 0.99542634, 0.99758927, 0.49263021],
                             [- 6.26360954, -0.00149426, 0.00091532, 7.26211528, 0.99542854, 0.99759042, 0.49263131],
                             [- 6.26363815, -0.00149355, 0.00091489, 7.26214460, 0.99543073, 0.99759156, 0.49263239],
                             [- 6.26366666, -0.00149284, 0.00091446, 7.26217383, 0.99543292, 0.99759271, 0.49263348],
                             [- 6.26369507, -0.00149212, 0.00091403, 7.26220295, 0.99543511, 0.99759385, 0.49263456],
                             [- 6.26372338, -0.00149141, 0.00091359, 7.26223197, 0.99543730, 0.99759499, 0.49263563],
                             [- 6.26375159, -0.00149070, 0.00091316, 7.26226088, 0.99543949, 0.99759613, 0.49263671],
                             [- 6.26377969, -0.00148999, 0.00091273, 7.26228970, 0.99544167, 0.99759728, 0.49263778],
                             [- 6.26380769, -0.00148928, 0.00091230, 7.26231841, 0.99544386, 0.99759841, 0.49263884],
                             [- 6.26383560, -0.00148857, 0.00091187, 7.26234702, 0.99544604, 0.99759955, 0.49263991],
                             [- 6.26386340, -0.00148787, 0.00091144, 7.26237553, 0.99544821, 0.99760069, 0.49264097],
                             [- 6.26389110, -0.00148716, 0.00091101, 7.26240394, 0.99545039, 0.99760183, 0.49264203],
                             [- 6.26391870, -0.00148645, 0.00091058, 7.26243225, 0.99545257, 0.99760296, 0.49264308],
                             [- 6.26394621, -0.00148575, 0.00091016, 7.26246046, 0.99545474, 0.99760410, 0.49264413],
                             [- 6.26397361, -0.00148504, 0.00090973, 7.26248857, 0.99545691, 0.99760523, 0.49264518],
                             [- 6.26400092, -0.00148434, 0.00090930, 7.26251658, 0.99545908, 0.99760636, 0.49264622],
                             [- 6.26402813, -0.00148363, 0.00090887, 7.26254449, 0.99546124, 0.99760750, 0.49264726],
                             [- 6.26405524, -0.00148293, 0.00090844, 7.26257231, 0.99546341, 0.99760863, 0.49264830],
                             [- 6.26408225, -0.00148223, 0.00090802, 7.26260002, 0.99546557, 0.99760976, 0.49264934],
                             [- 6.26410916, -0.00148152, 0.00090759, 7.26262764, 0.99546773, 0.99761088, 0.49265036],
                             [- 6.26413598, -0.00148082, 0.00090717, 7.26265516, 0.99546989, 0.99761201, 0.49265140],
                             [- 6.26416271, -0.00148012, 0.00090674, 7.26268258, 0.99547204, 0.99761314, 0.49265242],
                             [- 6.26418933, -0.00147942, 0.00090631, 7.26270991, 0.99547420, 0.99761426, 0.49265345],
                             [- 6.26421586, -0.00147872, 0.00090589, 7.26273714, 0.99547635, 0.99761539, 0.49265447],
                             [- 6.26424230, -0.00147802, 0.00090547, 7.26276428, 0.99547850, 0.99761651, 0.49265547],
                             [- 6.26426864, -0.00147733, 0.00090504, 7.26279132, 0.99548065, 0.99761763, 0.49265649],
                             [- 6.26429489, -0.00147663, 0.00090462, 7.26281826, 0.99548279, 0.99761875, 0.49265751],
                             [- 6.26432104, -0.00147593, 0.00090419, 7.26284511, 0.99548494, 0.99761987, 0.49265851],
                             [- 6.26434710, -0.00147523, 0.00090377, 7.26287186, 0.99548708, 0.99762099, 0.49265951],
                             [- 6.26437306, -0.00147454, 0.00090335, 7.26289852, 0.99548922, 0.99762211, 0.49266051],
                             [- 6.26439893, -0.00147384, 0.00090293, 7.26292509, 0.99549136, 0.99762323, 0.49266151],
                             [- 6.26442471, -0.00147315, 0.00090250, 7.26295156, 0.99549349, 0.99762435, 0.49266249],
                             [- 6.26445040, -0.00147246, 0.00090208, 7.26297794, 0.99549563, 0.99762546, 0.49266350],
                             [- 6.26447599, -0.00147176, 0.00090166, 7.26300423, 0.99549776, 0.99762658, 0.49266448],
                             [- 6.26450150, -0.00147107, 0.00090124, 7.26303043, 0.99549989, 0.99762769, 0.49266547],
                             [- 6.26452691, -0.00147038, 0.00090082, 7.26305653, 0.99550202, 0.99762880, 0.49266646],
                             [- 6.26455222, -0.00146969, 0.00090040, 7.26308254, 0.99550415, 0.99762991, 0.49266744],
                             [- 6.26457745, -0.00146900, 0.00089998, 7.26310846, 0.99550627, 0.99763102, 0.49266841],
                             [- 6.26460259, -0.00146831, 0.00089956, 7.26313428, 0.99550839, 0.99763213, 0.49266939],
                             [- 6.26462764, -0.00146762, 0.00089914, 7.26316002, 0.99551051, 0.99763324, 0.49267035],
                             [- 6.26465259, -0.00146693, 0.00089872, 7.26318567, 0.99551263, 0.99763435, 0.49267132],
                             [- 6.26467746, -0.00146624, 0.00089831, 7.26321122, 0.99551475, 0.99763545, 0.49267230],
                             [- 6.26470224, -0.00146555, 0.00089789, 7.26323669, 0.99551686, 0.99763656, 0.49267325],
                             [- 6.26472693, -0.00146486, 0.00089747, 7.26326207, 0.99551898, 0.99763766, 0.49267421],
                             [- 6.26475153, -0.00146418, 0.00089705, 7.26328735, 0.99552109, 0.99763877, 0.49267517],
                             [- 6.26477604, -0.00146349, 0.00089664, 7.26331255, 0.99552319, 0.99763987, 0.49267611],
                             [- 6.26480047, -0.00146281, 0.00089622, 7.26333766, 0.99552530, 0.99764097, 0.49267707],
                             [- 6.26482481, -0.00146212, 0.00089580, 7.26336268, 0.99552741, 0.99764207, 0.49267802],
                             [- 6.26484906, -0.00146144, 0.00089539, 7.26338762, 0.99552951, 0.99764317, 0.49267897],
                             [- 6.26487322, -0.00146076, 0.00089497, 7.26341246, 0.99553161, 0.99764427, 0.49267991],
                             [- 6.26489730, -0.00146007, 0.00089456, 7.26343722, 0.99553371, 0.99764537, 0.49268085],
                             [- 6.26492129, -0.00145939, 0.00089414, 7.26346189, 0.99553581, 0.99764646, 0.49268179],
                             [- 6.26494519, -0.00145871, 0.00089373, 7.26348648, 0.99553790, 0.99764756, 0.49268272],
                             [- 6.26496901, -0.00145803, 0.00089332, 7.26351098, 0.99553999, 0.99764865, 0.49268365],
                             [- 6.26499274, -0.00145735, 0.00089290, 7.26353539, 0.99554209, 0.99764975, 0.49268458],
                             [- 6.26501639, -0.00145667, 0.00089249, 7.26355972, 0.99554417, 0.99765084, 0.49268549],
                             [- 6.26503995, -0.00145599, 0.00089208, 7.26358396, 0.99554626, 0.99765193, 0.49268643],
                             [- 6.26506343, -0.00145531, 0.00089166, 7.26360811, 0.99554835, 0.99765302, 0.49268735],
                             [- 6.26508682, -0.00145463, 0.00089125, 7.26363219, 0.99555043, 0.99765411, 0.49268826],
                             [- 6.26511013, -0.00145396, 0.00089084, 7.26365617, 0.99555251, 0.99765520, 0.49268918],
                             [- 6.26513336, -0.00145328, 0.00089043, 7.26368008, 0.99555459, 0.99765629, 0.49269008],
                             [- 6.26515650, -0.00145261, 0.00089002, 7.26370389, 0.99555667, 0.99765738, 0.49269099],
                             [- 6.26517956, -0.00145193, 0.00088961, 7.26372763, 0.99555875, 0.99765846, 0.49269190],
                             [- 6.26520254, -0.00145126, 0.00088920, 7.26375128, 0.99556082, 0.99765955, 0.49269281],
                             [- 6.26522543, -0.00145058, 0.00088879, 7.26377485, 0.99556289, 0.99766063, 0.49269370],
                             [- 6.26524824, -0.00144991, 0.00088838, 7.26379834, 0.99556496, 0.99766171, 0.49269460],
                             [- 6.26527098, -0.00144924, 0.00088797, 7.26382174, 0.99556703, 0.99766280, 0.49269550],
                             [- 6.26529363, -0.00144856, 0.00088756, 7.26384506, 0.99556910, 0.99766388, 0.49269640],
                             [- 6.26531619, -0.00144789, 0.00088715, 7.26386830, 0.99557116, 0.99766496, 0.49269728],
                             [- 6.26533868, -0.00144722, 0.00088674, 7.26389146, 0.99557323, 0.99766604, 0.49269817],
                             [- 6.26536109, -0.00144655, 0.00088634, 7.26391454, 0.99557529, 0.99766711, 0.49269905],
                             [- 6.26538341, -0.00144588, 0.00088593, 7.26393753, 0.99557735, 0.99766819, 0.49269994],
                             [- 6.26540566, -0.00144521, 0.00088552, 7.26396045, 0.99557940, 0.99766927, 0.49270083],
                             [- 6.26542783, -0.00144454, 0.00088511, 7.26398329, 0.99558146, 0.99767034, 0.49270168],
                             [- 6.26544992, -0.00144387, 0.00088471, 7.26400604, 0.99558351, 0.99767142, 0.49270256],
                             [- 6.26547192, -0.00144321, 0.00088430, 7.26402872, 0.99558556, 0.99767249, 0.49270344],
                             [- 6.26549385, -0.00144254, 0.00088390, 7.26405131, 0.99558761, 0.99767356, 0.49270431],
                             [- 6.26551570, -0.00144187, 0.00088349, 7.26407383, 0.99558966, 0.99767463, 0.49270518],
                             [- 6.26553748, -0.00144121, 0.00088309, 7.26409627, 0.99559171, 0.99767571, 0.49270604],
                             [- 6.26555917, -0.00144054, 0.00088268, 7.26411863, 0.99559375, 0.99767678, 0.49270690],
                             [- 6.26558079, -0.00143988, 0.00088228, 7.26414091, 0.99559579, 0.99767784, 0.49270776],
                             [- 6.26560233, -0.00143922, 0.00088187, 7.26416311, 0.99559783, 0.99767891, 0.49270862],
                             [- 6.26562379, -0.00143855, 0.00088147, 7.26418524, 0.99559987, 0.99767998, 0.49270947],
                             [- 6.26564518, -0.00143789, 0.00088107, 7.26420729, 0.99560191, 0.99768104, 0.49271032],
                             [- 6.26566648, -0.00143723, 0.00088066, 7.26422926, 0.99560394, 0.99768211, 0.49271118],
                             [- 6.26568772, -0.00143657, 0.00088026, 7.26425115, 0.99560598, 0.99768317, 0.49271202],
                             [- 6.26570887, -0.00143590, 0.00087986, 7.26427297, 0.99560801, 0.99768424, 0.49271286],
                             [- 6.26572995, -0.00143524, 0.00087946, 7.26429471, 0.99561004, 0.99768530, 0.49271371],
                             [- 6.26575096, -0.00143458, 0.00087906, 7.26431637, 0.99561206, 0.99768636, 0.49271455],
                             [- 6.26577189, -0.00143393, 0.00087865, 7.26433796, 0.99561409, 0.99768742, 0.49271537],
                             [- 6.26579274, -0.00143327, 0.00087825, 7.26435948, 0.99561611, 0.99768848, 0.49271621],
                             [- 6.26581352, -0.00143261, 0.00087785, 7.26438092, 0.99561813, 0.99768954, 0.49271704],
                             [- 6.26583423, -0.00143195, 0.00087745, 7.26440228, 0.99562015, 0.99769059, 0.49271787],
                             [- 6.26585486, -0.00143130, 0.00087705, 7.26442357, 0.99562217, 0.99769165, 0.49271871],
                             [- 6.26587542, -0.00143064, 0.00087665, 7.26444478, 0.99562419, 0.99769271, 0.49271954],
                             [- 6.26589591, -0.00142998, 0.00087626, 7.26446592, 0.99562620, 0.99769376, 0.49272035],
                             [- 6.26591632, -0.00142933, 0.00087586, 7.26448699, 0.99562822, 0.99769482, 0.49272117],
                             [- 6.26593666, -0.00142867, 0.00087546, 7.26450798, 0.99563023, 0.99769587, 0.49272198],
                             [- 6.26595692, -0.00142802, 0.00087506, 7.26452890, 0.99563224, 0.99769692, 0.49272279],
                             [- 6.26597711, -0.00142737, 0.00087466, 7.26454975, 0.99563424, 0.99769797, 0.49272362],
                             [- 6.26599724, -0.00142671, 0.00087426, 7.26457052, 0.99563625, 0.99769902, 0.49272441],
                             [- 6.26601728, -0.00142606, 0.00087387, 7.26459122, 0.99563825, 0.99770007, 0.49272522],
                             [- 6.26603726, -0.00142541, 0.00087347, 7.26461185, 0.99564025, 0.99770112, 0.49272604],
                             [- 6.26605717, -0.00142476, 0.00087307, 7.26463241, 0.99564225, 0.99770217, 0.49272683],
                             [- 6.26607700, -0.00142411, 0.00087268, 7.26465289, 0.99564425, 0.99770321, 0.49272764],
                             [- 6.26609676, -0.00142346, 0.00087228, 7.26467331, 0.99564625, 0.99770426, 0.49272843],
                             [- 6.26611646, -0.00142281, 0.00087189, 7.26469365, 0.99564824, 0.99770530, 0.49272921],
                             [- 6.26613608, -0.00142216, 0.00087149, 7.26471392, 0.99565024, 0.99770635, 0.49273003],
                             [- 6.26615563, -0.00142151, 0.00087110, 7.26473412, 0.99565223, 0.99770739, 0.49273081],
                             [- 6.26617512, -0.00142087, 0.00087070, 7.26475425, 0.99565422, 0.99770843, 0.49273160],
                             [- 6.26619453, -0.00142022, 0.00087031, 7.26477431, 0.99565620, 0.99770947, 0.49273238],
                             [- 6.26621387, -0.00141957, 0.00086992, 7.26479430, 0.99565819, 0.99771051, 0.49273317],
                             [- 6.26623315, -0.00141893, 0.00086952, 7.26481422, 0.99566017, 0.99771155, 0.49273396],
                             [- 6.26625235, -0.00141828, 0.00086913, 7.26483407, 0.99566216, 0.99771259, 0.49273473],
                             [- 6.26627149, -0.00141764, 0.00086874, 7.26485385, 0.99566414, 0.99771362, 0.49273551],
                             [- 6.26629056, -0.00141699, 0.00086835, 7.26487356, 0.99566612, 0.99771466, 0.49273628],
                             [- 6.26630956, -0.00141635, 0.00086795, 7.26489321, 0.99566809, 0.99771570, 0.49273706],
                             [- 6.26632849, -0.00141571, 0.00086756, 7.26491278, 0.99567007, 0.99771673, 0.49273783],
                             [- 6.26634735, -0.00141506, 0.00086717, 7.26493229, 0.99567204, 0.99771776, 0.49273859],
                             [- 6.26636615, -0.00141442, 0.00086678, 7.26495173, 0.99567401, 0.99771880, 0.49273935],
                             [- 6.26638488, -0.00141378, 0.00086639, 7.26497110, 0.99567598, 0.99771983, 0.49274013],
                             [- 6.26640355, -0.00141314, 0.00086600, 7.26499041, 0.99567795, 0.99772086, 0.49274089],
                             [- 6.26642214, -0.00141250, 0.00086561, 7.26500964, 0.99567992, 0.99772189, 0.49274164],
                             [- 6.26644067, -0.00141186, 0.00086522, 7.26502881, 0.99568188, 0.99772292, 0.49274241],
                             [- 6.26645914, -0.00141122, 0.00086483, 7.26504792, 0.99568384, 0.99772395, 0.49274316],
                             [- 6.26647754, -0.00141058, 0.00086444, 7.26506695, 0.99568580, 0.99772497, 0.49274392],
                             [- 6.26649587, -0.00140995, 0.00086405, 7.26508592, 0.99568776, 0.99772600, 0.49274466],
                             [- 6.26651414, -0.00140931, 0.00086366, 7.26510483, 0.99568972, 0.99772703, 0.49274542],
                             [- 6.26653234, -0.00140867, 0.00086328, 7.26512367, 0.99569168, 0.99772805, 0.49274618],
                             [- 6.26655048, -0.00140804, 0.00086289, 7.26514244, 0.99569363, 0.99772908, 0.49274690],
                             [- 6.26656855, -0.00140740, 0.00086250, 7.26516115, 0.99569558, 0.99773010, 0.49274765],
                             [- 6.26658656, -0.00140677, 0.00086211, 7.26517979, 0.99569753, 0.99773112, 0.49274839],
                             [- 6.26660450, -0.00140613, 0.00086173, 7.26519837, 0.99569948, 0.99773214, 0.49274912],
                             [- 6.26662238, -0.00140550, 0.00086134, 7.26521688, 0.99570143, 0.99773316, 0.49274986],
                             [- 6.26664020, -0.00140486, 0.00086096, 7.26523533, 0.99570337, 0.99773418, 0.49275059],
                             [- 6.26665795, -0.00140423, 0.00086057, 7.26525372, 0.99570532, 0.99773520, 0.49275132],
                             [- 6.26667564, -0.00140360, 0.00086018, 7.26527204, 0.99570726, 0.99773622, 0.49275206],
                             [- 6.26669326, -0.00140297, 0.00085980, 7.26529030, 0.99570920, 0.99773723, 0.49275276],
                             [- 6.26671083, -0.00140234, 0.00085942, 7.26530849, 0.99571114, 0.99773825, 0.49275350],
                             [- 6.26672833, -0.00140171, 0.00085903, 7.26532662, 0.99571307, 0.99773926, 0.49275423],
                             [- 6.26674577, -0.00140108, 0.00085865, 7.26534469, 0.99571501, 0.99774028, 0.49275495],
                             [- 6.26676314, -0.00140045, 0.00085826, 7.26536270, 0.99571694, 0.99774129, 0.49275567],
                             [- 6.26678046, -0.00139982, 0.00085788, 7.26538064, 0.99571887, 0.99774230, 0.49275639],
                             [- 6.26679771, -0.00139919, 0.00085750, 7.26539852, 0.99572080, 0.99774331, 0.49275711],
                             [- 6.26681490, -0.00139856, 0.00085711, 7.26541634, 0.99572273, 0.99774433, 0.49275782],
                             [- 6.26683203, -0.00139793, 0.00085673, 7.26543410, 0.99572466, 0.99774533, 0.49275852],
                             [- 6.26684910, -0.00139731, 0.00085635, 7.26545179, 0.99572658, 0.99774634, 0.49275924],
                             [- 6.26686611, -0.00139668, 0.00085597, 7.26546943, 0.99572851, 0.99774735, 0.49275995],
                             [- 6.26688306, -0.00139605, 0.00085559, 7.26548700, 0.99573043, 0.99774836, 0.49276065],
                             [- 6.26689994, -0.00139543, 0.00085521, 7.26550451, 0.99573235, 0.99774936, 0.49276134],
                             [- 6.26691677, -0.00139481, 0.00085482, 7.26552196, 0.99573426, 0.99775037, 0.49276204],
                             [- 6.26693354, -0.00139418, 0.00085444, 7.26553936, 0.99573618, 0.99775137, 0.49276274],
                             [- 6.26695024, -0.00139356, 0.00085406, 7.26555669, 0.99573809, 0.99775238, 0.49276344],
                             [- 6.26696689, -0.00139293, 0.00085368, 7.26557396, 0.99574001, 0.99775338, 0.49276413],
                             [- 6.26698348, -0.00139231, 0.00085331, 7.26559117, 0.99574192, 0.99775438, 0.49276484],
                             [- 6.26700001, -0.00139169, 0.00085293, 7.26560832, 0.99574383, 0.99775538, 0.49276553],
                             [- 6.26701648, -0.00139107, 0.00085255, 7.26562541, 0.99574574, 0.99775638, 0.49276621],
                             [- 6.26703289, -0.00139045, 0.00085217, 7.26564245, 0.99574764, 0.99775738, 0.49276691],
                             [- 6.26704925, -0.00138983, 0.00085179, 7.26565942, 0.99574955, 0.99775838, 0.49276759],
                             [- 6.26706554, -0.00138921, 0.00085141, 7.26567633, 0.99575145, 0.99775938, 0.49276826],
                             [- 6.26708178, -0.00138859, 0.00085104, 7.26569319, 0.99575335, 0.99776038, 0.49276895],
                             [- 6.26709796, -0.00138797, 0.00085066, 7.26570999, 0.99575525, 0.99776137, 0.49276963],
                             [- 6.26711408, -0.00138735, 0.00085028, 7.26572673, 0.99575715, 0.99776237, 0.49277030],
                             [- 6.26713015, -0.00138674, 0.00084990, 7.26574341, 0.99575904, 0.99776336, 0.49277097],
                             [- 6.26714616, -0.00138612, 0.00084953, 7.26576004, 0.99576094, 0.99776435, 0.49277165],
                             [- 6.26716211, -0.00138550, 0.00084915, 7.26577661, 0.99576283, 0.99776535, 0.49277232],
                             [- 6.26717800, -0.00138489, 0.00084878, 7.26579312, 0.99576472, 0.99776634, 0.49277298],
                             [- 6.26719384, -0.00138427, 0.00084840, 7.26580957, 0.99576661, 0.99776733, 0.49277366],
                             [- 6.26720962, -0.00138366, 0.00084803, 7.26582597, 0.99576850, 0.99776832, 0.49277432],
                             [- 6.26722535, -0.00138304, 0.00084765, 7.26584231, 0.99577038, 0.99776931, 0.49277499],
                             [- 6.26724102, -0.00138243, 0.00084728, 7.26585859, 0.99577227, 0.99777029, 0.49277565],
                             [- 6.26725663, -0.00138181, 0.00084690, 7.26587482, 0.99577415, 0.99777128, 0.49277630],
                             [- 6.26727219, -0.00138120, 0.00084653, 7.26589099, 0.99577603, 0.99777227, 0.49277698],
                             [- 6.26728769, -0.00138059, 0.00084616, 7.26590710, 0.99577791, 0.99777325, 0.49277763],
                             [- 6.26730314, -0.00137998, 0.00084578, 7.26592316, 0.99577979, 0.99777424, 0.49277828],
                             [- 6.26731854, -0.00137937, 0.00084541, 7.26593917, 0.99578166, 0.99777522, 0.49277893],
                             [- 6.26733388, -0.00137876, 0.00084504, 7.26595512, 0.99578354, 0.99777621, 0.49277959],
                             [- 6.26734916, -0.00137815, 0.00084467, 7.26597101, 0.99578541, 0.99777719, 0.49278024],
                             [- 6.26736439, -0.00137754, 0.00084429, 7.26598685, 0.99578728, 0.99777817, 0.49278088],
                             [- 6.26737957, -0.00137693, 0.00084392, 7.26600264, 0.99578915, 0.99777915, 0.49278153],
                             [- 6.26739469, -0.00137632, 0.00084355, 7.26601837, 0.99579102, 0.99778013, 0.49278219],
                             [- 6.26740976, -0.00137571, 0.00084318, 7.26603405, 0.99579289, 0.99778111, 0.49278281],
                             [- 6.26742477, -0.00137510, 0.00084281, 7.26604967, 0.99579475, 0.99778209, 0.49278347],
                             [- 6.26743974, -0.00137450, 0.00084244, 7.26606524, 0.99579661, 0.99778306, 0.49278410],
                             [- 6.26745464, -0.00137389, 0.00084207, 7.26608075, 0.99579847, 0.99778404, 0.49278472],
                             [- 6.26746950, -0.00137328, 0.00084170, 7.26609622, 0.99580033, 0.99778502, 0.49278536],
                             [- 6.26748430, -0.00137268, 0.00084133, 7.26611162, 0.99580219, 0.99778599, 0.49278600],
                             [- 6.26749906, -0.00137207, 0.00084096, 7.26612698, 0.99580405, 0.99778696, 0.49278663],
                             [- 6.26751375, -0.00137147, 0.00084059, 7.26614228, 0.99580590, 0.99778794, 0.49278724],
                             [- 6.26752840, -0.00137087, 0.00084022, 7.26615753, 0.99580776, 0.99778891, 0.49278789],
                             [- 6.26754300, -0.00137026, 0.00083986, 7.26617273, 0.99580961, 0.99778988, 0.49278852],
                             [- 6.26755754, -0.00136966, 0.00083949, 7.26618788, 0.99581146, 0.99779085, 0.49278914],
                             [- 6.26757203, -0.00136906, 0.00083912, 7.26620297, 0.99581331, 0.99779182, 0.49278975],
                             [- 6.26758647, -0.00136846, 0.00083875, 7.26621801, 0.99581515, 0.99779279, 0.49279037],
                             [- 6.26760086, -0.00136786, 0.00083839, 7.26623300, 0.99581700, 0.99779376, 0.49279099],
                             [- 6.26761520, -0.00136725, 0.00083802, 7.26624794, 0.99581884, 0.99779472, 0.49279161],
                             [- 6.26762949, -0.00136665, 0.00083765, 7.26626283, 0.99582068, 0.99779569, 0.49279223],
                             [- 6.26764372, -0.00136605, 0.00083729, 7.26627767, 0.99582252, 0.99779666, 0.49279284],
                             [- 6.26765791, -0.00136546, 0.00083692, 7.26629245, 0.99582436, 0.99779762, 0.49279345],
                             [- 6.26767205, -0.00136486, 0.00083656, 7.26630719, 0.99582620, 0.99779859, 0.49279407],
                             [- 6.26768613, -0.00136426, 0.00083619, 7.26632187, 0.99582804, 0.99779955, 0.49279468],
                             [- 6.26770017, -0.00136366, 0.00083583, 7.26633651, 0.99582987, 0.99780051, 0.49279528],
                             [- 6.26771416, -0.00136306, 0.00083546, 7.26635109, 0.99583170, 0.99780147, 0.49279588],
                             [- 6.26772809, -0.00136247, 0.00083510, 7.26636562, 0.99583353, 0.99780243, 0.49279649],
                             [- 6.26774198, -0.00136187, 0.00083474, 7.26638011, 0.99583536, 0.99780339, 0.49279710],
                             [- 6.26775582, -0.00136128, 0.00083437, 7.26639454, 0.99583719, 0.99780435, 0.49279770],
                             [- 6.26776961, -0.00136068, 0.00083401, 7.26640893, 0.99583902, 0.99780531, 0.49279829],
                             [- 6.26778335, -0.00136009, 0.00083365, 7.26642326, 0.99584084, 0.99780627, 0.49279888],
                             [- 6.26779704, -0.00135949, 0.00083328, 7.26643755, 0.99584266, 0.99780722, 0.49279946],
                             [- 6.26781069, -0.00135890, 0.00083292, 7.26645179, 0.99584449, 0.99780818, 0.49280008],
                             [- 6.26782428, -0.00135831, 0.00083256, 7.26646598, 0.99584631, 0.99780913, 0.49280067],
                             [- 6.26783783, -0.00135771, 0.00083220, 7.26648012, 0.99584812, 0.99781009, 0.49280125],
                             [- 6.26785133, -0.00135712, 0.00083184, 7.26649421, 0.99584994, 0.99781104, 0.49280185],
                             [- 6.26786479, -0.00135653, 0.00083148, 7.26650826, 0.99585175, 0.99781199, 0.49280244],
                             [- 6.26787819, -0.00135594, 0.00083111, 7.26652225, 0.99585357, 0.99781295, 0.49280301],
                             [- 6.26789155, -0.00135535, 0.00083075, 7.26653620, 0.99585538, 0.99781390, 0.49280360],
                             [- 6.26790486, -0.00135476, 0.00083039, 7.26655010, 0.99585719, 0.99781485, 0.49280419],
                             [- 6.26791812, -0.00135417, 0.00083003, 7.26656395, 0.99585900, 0.99781580, 0.49280476],
                             [- 6.26793134, -0.00135358, 0.00082967, 7.26657776, 0.99586081, 0.99781675, 0.49280534],
                             [- 6.26794451, -0.00135299, 0.00082931, 7.26659152, 0.99586261, 0.99781769, 0.49280594],
                             [- 6.26795764, -0.00135240, 0.00082896, 7.26660523, 0.99586442, 0.99781864, 0.49280650],
                             [- 6.26797071, -0.00135182, 0.00082860, 7.26661890, 0.99586622, 0.99781959, 0.49280708],
                             [- 6.26798374, -0.00135123, 0.00082824, 7.26663252, 0.99586802, 0.99782053, 0.49280764],
                             [- 6.26799673, -0.00135064, 0.00082788, 7.26664609, 0.99586982, 0.99782148, 0.49280822],
                             [- 6.26800967, -0.00135006, 0.00082752, 7.26665961, 0.99587162, 0.99782242, 0.49280878],
                             [- 6.26802256, -0.00134947, 0.00082717, 7.26667309, 0.99587341, 0.99782336, 0.49280935],
                             [- 6.26803541, -0.00134889, 0.00082681, 7.26668653, 0.99587521, 0.99782431, 0.49280992],
                             [- 6.26804822, -0.00134830, 0.00082645, 7.26669991, 0.99587700, 0.99782525, 0.49281048],
                             [- 6.26806098, -0.00134772, 0.00082609, 7.26671326, 0.99587879, 0.99782619, 0.49281107],
                             [- 6.26807369, -0.00134713, 0.00082574, 7.26672655, 0.99588058, 0.99782713, 0.49281162],
                             [- 6.26808636, -0.00134655, 0.00082538, 7.26673981, 0.99588237, 0.99782807, 0.49281218],
                             [- 6.26809898, -0.00134597, 0.00082503, 7.26675301, 0.99588416, 0.99782900, 0.49281274],
                             [- 6.26811156, -0.00134539, 0.00082467, 7.26676617, 0.99588595, 0.99782994, 0.49281329],
                             [- 6.26812410, -0.00134481, 0.00082432, 7.26677929, 0.99588773, 0.99783088, 0.49281384],
                             [- 6.26813659, -0.00134422, 0.00082396, 7.26679236, 0.99588951, 0.99783181, 0.49281439],
                             [- 6.26814904, -0.00134364, 0.00082361, 7.26680539, 0.99589129, 0.99783275, 0.49281496],
                             [- 6.26816144, -0.00134306, 0.00082325, 7.26681838, 0.99589307, 0.99783368, 0.49281552],
                             [- 6.26817380, -0.00134248, 0.00082290, 7.26683132, 0.99589485, 0.99783462, 0.49281605],
                             [- 6.26818612, -0.00134191, 0.00082255, 7.26684421, 0.99589663, 0.99783555, 0.49281661],
                             [- 6.26819839, -0.00134133, 0.00082219, 7.26685706, 0.99589840, 0.99783648, 0.49281716],
                             [- 6.26821062, -0.00134075, 0.00082184, 7.26686987, 0.99590018, 0.99783741, 0.49281769],
                             [- 6.26822281, -0.00134017, 0.00082149, 7.26688264, 0.99590195, 0.99783834, 0.49281824],
                             [- 6.26823495, -0.00133959, 0.00082113, 7.26689536, 0.99590372, 0.99783927, 0.49281878],
                             [- 6.26824705, -0.00133902, 0.00082078, 7.26690804, 0.99590549, 0.99784020, 0.49281933],
                             [- 6.26825911, -0.00133844, 0.00082043, 7.26692067, 0.99590725, 0.99784113, 0.49281987],
                             [- 6.26827113, -0.00133787, 0.00082008, 7.26693326, 0.99590902, 0.99784206, 0.49282040],
                             [- 6.26828310, -0.00133729, 0.00081973, 7.26694581, 0.99591078, 0.99784298, 0.49282094],
                             [- 6.26829504, -0.00133672, 0.00081938, 7.26695832, 0.99591255, 0.99784391, 0.49282147],
                             [- 6.26830693, -0.00133614, 0.00081902, 7.26697079, 0.99591431, 0.99784483, 0.49282201],
                             [- 6.26831878, -0.00133557, 0.00081867, 7.26698321, 0.99591607, 0.99784576, 0.49282254],
                             [- 6.26833058, -0.00133499, 0.00081832, 7.26699559, 0.99591783, 0.99784668, 0.49282308],
                             [- 6.26834235, -0.00133442, 0.00081797, 7.26700793, 0.99591958, 0.99784760, 0.49282360],
                             [- 6.26835407, -0.00133385, 0.00081762, 7.26702023, 0.99592134, 0.99784853, 0.49282414],
                             [- 6.26836576, -0.00133328, 0.00081728, 7.26703248, 0.99592309, 0.99784945, 0.49282465],
                             [- 6.26837740, -0.00133271, 0.00081693, 7.26704469, 0.99592485, 0.99785037, 0.49282520],
                             [- 6.26838900, -0.00133214, 0.00081658, 7.26705687, 0.99592660, 0.99785129, 0.49282572],
                             [- 6.26840056, -0.00133157, 0.00081623, 7.26706900, 0.99592835, 0.99785221, 0.49282624],
                             [- 6.26841209, -0.00133100, 0.00081588, 7.26708109, 0.99593009, 0.99785312, 0.49282676],
                             [- 6.26842357, -0.00133043, 0.00081553, 7.26709314, 0.99593184, 0.99785404, 0.49282728],
                             [- 6.26843501, -0.00132986, 0.00081519, 7.26710515, 0.99593359, 0.99785496, 0.49282780],
                             [- 6.26844641, -0.00132929, 0.00081484, 7.26711712, 0.99593533, 0.99785587, 0.49282829],
                             [- 6.26845777, -0.00132872, 0.00081449, 7.26712905, 0.99593707, 0.99785679, 0.49282883],
                             [- 6.26846909, -0.00132815, 0.00081414, 7.26714093, 0.99593881, 0.99785770, 0.49282934],
                             [- 6.26848037, -0.00132759, 0.00081380, 7.26715278, 0.99594055, 0.99785862, 0.49282986],
                             [- 6.26849161, -0.00132702, 0.00081345, 7.26716459, 0.99594229, 0.99785953, 0.49283037],
                             [- 6.26850281, -0.00132645, 0.00081311, 7.26717636, 0.99594402, 0.99786044, 0.49283087],
                             [- 6.26851397, -0.00132589, 0.00081276, 7.26718809, 0.99594576, 0.99786135, 0.49283138],
                             [- 6.26852510, -0.00132532, 0.00081242, 7.26719977, 0.99594749, 0.99786226, 0.49283189],
                             [- 6.26853618, -0.00132476, 0.00081207, 7.26721142, 0.99594922, 0.99786317, 0.49283239],
                             [- 6.26854723, -0.00132419, 0.00081173, 7.26722303, 0.99595095, 0.99786408, 0.49283291],
                             [- 6.26855824, -0.00132363, 0.00081138, 7.26723461, 0.99595268, 0.99786499, 0.49283341],
                             [- 6.26856921, -0.00132307, 0.00081104, 7.26724614, 0.99595441, 0.99786590, 0.49283391],
                             [- 6.26858014, -0.00132251, 0.00081069, 7.26725763, 0.99595614, 0.99786680, 0.49283441],
                             [- 6.26859103, -0.00132194, 0.00081035, 7.26726909, 0.99595786, 0.99786771, 0.49283490],
                             [- 6.26860188, -0.00132138, 0.00081001, 7.26728050, 0.99595958, 0.99786861, 0.49283543],
                             [- 6.26861270, -0.00132082, 0.00080966, 7.26729188, 0.99596131, 0.99786952, 0.49283590],
                             [- 6.26862348, -0.00132026, 0.00080932, 7.26730322, 0.99596303, 0.99787042, 0.49283641],
                             [- 6.26863422, -0.00131970, 0.00080898, 7.26731452, 0.99596474, 0.99787132, 0.49283690],
                             [- 6.26864493, -0.00131914, 0.00080864, 7.26732579, 0.99596646, 0.99787223, 0.49283740],
                             [- 6.26865559, -0.00131858, 0.00080829, 7.26733701, 0.99596818, 0.99787313, 0.49283789],
                             [- 6.26866622, -0.00131802, 0.00080795, 7.26734820, 0.99596989, 0.99787403, 0.49283838],
                             [- 6.26867681, -0.00131746, 0.00080761, 7.26735935, 0.99597160, 0.99787493, 0.49283886],
                             [- 6.26868737, -0.00131690, 0.00080727, 7.26737047, 0.99597332, 0.99787583, 0.49283935],
                             [- 6.26869789, -0.00131635, 0.00080693, 7.26738154, 0.99597503, 0.99787673, 0.49283983],
                             [- 6.26870837, -0.00131579, 0.00080659, 7.26739258, 0.99597673, 0.99787762, 0.49284032],
                             [- 6.26871882, -0.00131523, 0.00080625, 7.26740358, 0.99597844, 0.99787852, 0.49284083],
                             [- 6.26872923, -0.00131468, 0.00080591, 7.26741455, 0.99598015, 0.99787942, 0.49284129],
                             [- 6.26873960, -0.00131412, 0.00080557, 7.26742548, 0.99598185, 0.99788031, 0.49284178],
                             [- 6.26874993, -0.00131357, 0.00080523, 7.26743637, 0.99598355, 0.99788121, 0.49284226],
                             [- 6.26876024, -0.00131301, 0.00080489, 7.26744723, 0.99598526, 0.99788210, 0.49284273],
                             [- 6.26877050, -0.00131246, 0.00080455, 7.26745804, 0.99598696, 0.99788299, 0.49284320],
                             [- 6.26878073, -0.00131190, 0.00080421, 7.26746883, 0.99598865, 0.99788389, 0.49284369],
                             [- 6.26879092, -0.00131135, 0.00080387, 7.26747958, 0.99599035, 0.99788478, 0.49284417],
                             [- 6.26880108, -0.00131080, 0.00080354, 7.26749029, 0.99599205, 0.99788567, 0.49284465],
                             [- 6.26881120, -0.00131024, 0.00080320, 7.26750096, 0.99599374, 0.99788656, 0.49284510],
                             [- 6.26882129, -0.00130969, 0.00080286, 7.26751160, 0.99599543, 0.99788745, 0.49284559],
                             [- 6.26883135, -0.00130914, 0.00080252, 7.26752221, 0.99599713, 0.99788834, 0.49284606],
                             [- 6.26884136, -0.00130859, 0.00080219, 7.26753278, 0.99599882, 0.99788923, 0.49284653],
                             [- 6.26885135, -0.00130804, 0.00080185, 7.26754331, 0.99600050, 0.99789011, 0.49284699],
                             [- 6.26886129, -0.00130749, 0.00080151, 7.26755381, 0.99600219, 0.99789100, 0.49284746],
                             [- 6.26887121, -0.00130694, 0.00080118, 7.26756427, 0.99600388, 0.99789189, 0.49284793],
                             [- 6.26888109, -0.00130639, 0.00080084, 7.26757470, 0.99600556, 0.99789277, 0.49284838],
                             [- 6.26889093, -0.00130584, 0.00080051, 7.26758509, 0.99600724, 0.99789365, 0.49284885],
                             [- 6.26890074, -0.00130529, 0.00080017, 7.26759545, 0.99600893, 0.99789454, 0.49284932],
                             [- 6.26891052, -0.00130474, 0.00079984, 7.26760578, 0.99601061, 0.99789542, 0.49284977],
                             [- 6.26892026, -0.00130420, 0.00079950, 7.26761607, 0.99601228, 0.99789630, 0.49285024],
                             [- 6.26892997, -0.00130365, 0.00079917, 7.26762633, 0.99601396, 0.99789719, 0.49285069],
                             [- 6.26893965, -0.00130310, 0.00079883, 7.26763655, 0.99601564, 0.99789807, 0.49285114],
                             [- 6.26894929, -0.00130256, 0.00079850, 7.26764674, 0.99601731, 0.99789895, 0.49285162],
                             [- 6.26895890, -0.00130201, 0.00079816, 7.26765689, 0.99601899, 0.99789983, 0.49285206],
                             [- 6.26896848, -0.00130146, 0.00079783, 7.26766701, 0.99602066, 0.99790070, 0.49285251],
                             [- 6.26897802, -0.00130092, 0.00079750, 7.26767710, 0.99602233, 0.99790158, 0.49285297],
                             [- 6.26898753, -0.00130038, 0.00079717, 7.26768715, 0.99602400, 0.99790246, 0.49285344],
                             [- 6.26899700, -0.00129983, 0.00079683, 7.26769717, 0.99602566, 0.99790334, 0.49285388],
                             [- 6.26900645, -0.00129929, 0.00079650, 7.26770716, 0.99602733, 0.99790421, 0.49285432],
                             [- 6.26901586, -0.00129875, 0.00079617, 7.26771711, 0.99602900, 0.99790509, 0.49285477],
                             [- 6.26902524, -0.00129820, 0.00079584, 7.26772703, 0.99603066, 0.99790596, 0.49285523],
                             [- 6.26903458, -0.00129766, 0.00079550, 7.26773692, 0.99603232, 0.99790683, 0.49285566],
                             [- 6.26904390, -0.00129712, 0.00079517, 7.26774678, 0.99603398, 0.99790771, 0.49285611],
                             [- 6.26905318, -0.00129658, 0.00079484, 7.26775660, 0.99603564, 0.99790858, 0.49285655],
                             [- 6.26906243, -0.00129604, 0.00079451, 7.26776639, 0.99603730, 0.99790945, 0.49285701],
                             [- 6.26907164, -0.00129550, 0.00079418, 7.26777615, 0.99603896, 0.99791032, 0.49285743],
                             [- 6.26908083, -0.00129496, 0.00079385, 7.26778587, 0.99604061, 0.99791119, 0.49285787],
                             [- 6.26908998, -0.00129442, 0.00079352, 7.26779557, 0.99604227, 0.99791206, 0.49285833],
                             [- 6.26909911, -0.00129388, 0.00079319, 7.26780523, 0.99604392, 0.99791293, 0.49285876],
                             [- 6.26910820, -0.00129334, 0.00079286, 7.26781486, 0.99604557, 0.99791380, 0.49285921],
                             [- 6.26911726, -0.00129280, 0.00079253, 7.26782446, 0.99604722, 0.99791467, 0.49285963],
                             [- 6.26912629, -0.00129226, 0.00079220, 7.26783402, 0.99604887, 0.99791553, 0.49286007],
                             [- 6.26913528, -0.00129173, 0.00079188, 7.26784356, 0.99605051, 0.99791640, 0.49286049],
                             [- 6.26914425, -0.00129119, 0.00079155, 7.26785306, 0.99605216, 0.99791726, 0.49286093],
                             [- 6.26915318, -0.00129065, 0.00079122, 7.26786253, 0.99605380, 0.99791813, 0.49286138],
                             [- 6.26916209, -0.00129012, 0.00079089, 7.26787197, 0.99605545, 0.99791899, 0.49286178],
                             [- 6.26917096, -0.00128958, 0.00079056, 7.26788138, 0.99605709, 0.99791985, 0.49286223],
                             [- 6.26917981, -0.00128905, 0.00079024, 7.26789076, 0.99605873, 0.99792072, 0.49286266],
                             [- 6.26918862, -0.00128851, 0.00078991, 7.26790011, 0.99606037, 0.99792158, 0.49286308],
                             [- 6.26919740, -0.00128798, 0.00078958, 7.26790942, 0.99606201, 0.99792244, 0.49286350],
                             [- 6.26920615, -0.00128744, 0.00078926, 7.26791871, 0.99606364, 0.99792330, 0.49286392],
                             [- 6.26921488, -0.00128691, 0.00078893, 7.26792797, 0.99606528, 0.99792416, 0.49286436],
                             [- 6.26922357, -0.00128638, 0.00078860, 7.26793719, 0.99606691, 0.99792502, 0.49286478],
                             [- 6.26923223, -0.00128585, 0.00078828, 7.26794639, 0.99606854, 0.99792588, 0.49286521],
                             [- 6.26924086, -0.00128531, 0.00078795, 7.26795555, 0.99607017, 0.99792673, 0.49286564],
                             [- 6.26924947, -0.00128478, 0.00078763, 7.26796468, 0.99607180, 0.99792759, 0.49286605],
                             [- 6.26925804, -0.00128425, 0.00078730, 7.26797379, 0.99607343, 0.99792845, 0.49286646],
                             [- 6.26926658, -0.00128372, 0.00078698, 7.26798286, 0.99607506, 0.99792930, 0.49286689],
                             [- 6.26927510, -0.00128319, 0.00078665, 7.26799191, 0.99607668, 0.99793016, 0.49286730],
                             [- 6.26928358, -0.00128266, 0.00078633, 7.26800092, 0.99607831, 0.99793101, 0.49286770],
                             [- 6.26929204, -0.00128213, 0.00078601, 7.26800991, 0.99607993, 0.99793186, 0.49286812],
                             [- 6.26930047, -0.00128160, 0.00078568, 7.26801887, 0.99608155, 0.99793272, 0.49286854],
                             [- 6.26930887, -0.00128107, 0.00078536, 7.26802779, 0.99608317, 0.99793357, 0.49286895],
                             [- 6.26931724, -0.00128054, 0.00078504, 7.26803669, 0.99608479, 0.99793442, 0.49286937],
                             [- 6.26932558, -0.00128002, 0.00078471, 7.26804556, 0.99608641, 0.99793527, 0.49286977],
                             [- 6.26933389, -0.00127949, 0.00078439, 7.26805440, 0.99608802, 0.99793612, 0.49287019],
                             [- 6.26934217, -0.00127896, 0.00078407, 7.26806321, 0.99608964, 0.99793697, 0.49287058],
                             [- 6.26935043, -0.00127844, 0.00078375, 7.26807199, 0.99609125, 0.99793782, 0.49287100],
                             [- 6.26935866, -0.00127791, 0.00078342, 7.26808075, 0.99609287, 0.99793867, 0.49287142],
                             [- 6.26936686, -0.00127738, 0.00078310, 7.26808947, 0.99609448, 0.99793951, 0.49287182],
                             [- 6.26937503, -0.00127686, 0.00078278, 7.26809817, 0.99609609, 0.99794036, 0.49287224],
                             [- 6.26938317, -0.00127634, 0.00078246, 7.26810683, 0.99609769, 0.99794120, 0.49287263],
                             [- 6.26939129, -0.00127581, 0.00078214, 7.26811547, 0.99609930, 0.99794205, 0.49287302],
                             [- 6.26939937, -0.00127529, 0.00078182, 7.26812409, 0.99610091, 0.99794289, 0.49287345],
                             [- 6.26940743, -0.00127476, 0.00078150, 7.26813267, 0.99610251, 0.99794374, 0.49287384],
                             [- 6.26941547, -0.00127424, 0.00078118, 7.26814123, 0.99610411, 0.99794458, 0.49287423],
                             [- 6.26942347, -0.00127372, 0.00078086, 7.26814975, 0.99610572, 0.99794542, 0.49287464],
                             [- 6.26943145, -0.00127320, 0.00078054, 7.26815825, 0.99610732, 0.99794626, 0.49287504],
                             [- 6.26943940, -0.00127267, 0.00078022, 7.26816673, 0.99610892, 0.99794711, 0.49287543],
                             [- 6.26944732, -0.00127215, 0.00077990, 7.26817517, 0.99611051, 0.99794795, 0.49287581],
                             [- 6.26945522, -0.00127163, 0.00077958, 7.26818359, 0.99611211, 0.99794879, 0.49287621],
                             [- 6.26946309, -0.00127111, 0.00077926, 7.26819198, 0.99611370, 0.99794962, 0.49287661],
                             [- 6.26947093, -0.00127059, 0.00077895, 7.26820034, 0.99611530, 0.99795046, 0.49287701],
                             [- 6.26947875, -0.00127007, 0.00077863, 7.26820868, 0.99611689, 0.99795130, 0.49287739],
                             [- 6.26948654, -0.00126955, 0.00077831, 7.26821698, 0.99611848, 0.99795214, 0.49287779],
                             [- 6.26949430, -0.00126903, 0.00077799, 7.26822527, 0.99612007, 0.99795297, 0.49287817],
                             [- 6.26950204, -0.00126852, 0.00077768, 7.26823352, 0.99612166, 0.99795381, 0.49287857],
                             [- 6.26950975, -0.00126800, 0.00077736, 7.26824175, 0.99612325, 0.99795464, 0.49287896],
                             [- 6.26951743, -0.00126748, 0.00077704, 7.26824995, 0.99612484, 0.99795548, 0.49287934],
                             [- 6.26952509, -0.00126696, 0.00077673, 7.26825813, 0.99612642, 0.99795631, 0.49287973],
                             [- 6.26953272, -0.00126645, 0.00077641, 7.26826627, 0.99612800, 0.99795714, 0.49288012],
                             [- 6.26954033, -0.00126593, 0.00077609, 7.26827440, 0.99612959, 0.99795798, 0.49288050],
                             [- 6.26954791, -0.00126541, 0.00077578, 7.26828249, 0.99613117, 0.99795881, 0.49288087],
                             [- 6.26955546, -0.00126490, 0.00077546, 7.26829056, 0.99613275, 0.99795964, 0.49288126],
                             [- 6.26956299, -0.00126438, 0.00077515, 7.26829860, 0.99613433, 0.99796047, 0.49288164],
                             [- 6.26957049, -0.00126387, 0.00077483, 7.26830662, 0.99613590, 0.99796130, 0.49288203],
                             [- 6.26957797, -0.00126335, 0.00077452, 7.26831461, 0.99613748, 0.99796213, 0.49288240],
                             [- 6.26958542, -0.00126284, 0.00077420, 7.26832258, 0.99613905, 0.99796296, 0.49288278],
                             [- 6.26959285, -0.00126233, 0.00077389, 7.26833052, 0.99614063, 0.99796378, 0.49288317],
                             [- 6.26960025, -0.00126181, 0.00077357, 7.26833844, 0.99614220, 0.99796461, 0.49288354],
                             [- 6.26960763, -0.00126130, 0.00077326, 7.26834632, 0.99614377, 0.99796544, 0.49288392],
                             [- 6.26961498, -0.00126079, 0.00077295, 7.26835419, 0.99614534, 0.99796626, 0.49288432],
                             [- 6.26962230, -0.00126028, 0.00077263, 7.26836203, 0.99614691, 0.99796709, 0.49288467],
                             [- 6.26962961, -0.00125977, 0.00077232, 7.26836984, 0.99614847, 0.99796791, 0.49288504],
                             [- 6.26963688, -0.00125926, 0.00077201, 7.26837763, 0.99615004, 0.99796874, 0.49288540],
                             [- 6.26964414, -0.00125874, 0.00077170, 7.26838539, 0.99615160, 0.99796956, 0.49288580],
                             [- 6.26965137, -0.00125823, 0.00077138, 7.26839313, 0.99615317, 0.99797038, 0.49288618],
                             [- 6.26965857, -0.00125772, 0.00077107, 7.26840084, 0.99615473, 0.99797120, 0.49288651],
                             [- 6.26966575, -0.00125722, 0.00077076, 7.26840853, 0.99615629, 0.99797202, 0.49288691],
                             [- 6.26967290, -0.00125671, 0.00077045, 7.26841620, 0.99615785, 0.99797284, 0.49288729],
                             [- 6.26968003, -0.00125620, 0.00077014, 7.26842384, 0.99615941, 0.99797366, 0.49288765],
                             [- 6.26968714, -0.00125569, 0.00076983, 7.26843145, 0.99616097, 0.99797448, 0.49288801],
                             [- 6.26969422, -0.00125518, 0.00076952, 7.26843904, 0.99616252, 0.99797530, 0.49288835],
                             [- 6.26970128, -0.00125468, 0.00076920, 7.26844661, 0.99616408, 0.99797612, 0.49288872],
                             [- 6.26970832, -0.00125417, 0.00076889, 7.26845415, 0.99616563, 0.99797694, 0.49288910],
                             [- 6.26971533, -0.00125366, 0.00076858, 7.26846167, 0.99616718, 0.99797775, 0.49288946],
                             [- 6.26972232, -0.00125316, 0.00076827, 7.26846916, 0.99616873, 0.99797857, 0.49288983],
                             [- 6.26972928, -0.00125265, 0.00076797, 7.26847663, 0.99617028, 0.99797938, 0.49289019],
                             [- 6.26973622, -0.00125214, 0.00076766, 7.26848408, 0.99617183, 0.99798020, 0.49289056],
                             [- 6.26974314, -0.00125164, 0.00076735, 7.26849150, 0.99617338, 0.99798101, 0.49289092],
                             [- 6.26975003, -0.00125114, 0.00076704, 7.26849890, 0.99617492, 0.99798183, 0.49289126],
                             [- 6.26975691, -0.00125063, 0.00076673, 7.26850627, 0.99617647, 0.99798264, 0.49289162],
                             [- 6.26976375, -0.00125013, 0.00076642, 7.26851363, 0.99617801, 0.99798345, 0.49289199],
                             [- 6.26977058, -0.00124962, 0.00076611, 7.26852095, 0.99617955, 0.99798426, 0.49289234],
                             [- 6.26977738, -0.00124912, 0.00076581, 7.26852826, 0.99618109, 0.99798507, 0.49289269],
                             [- 6.26978416, -0.00124862, 0.00076550, 7.26853554, 0.99618263, 0.99798588, 0.49289304],
                             [- 6.26979092, -0.00124812, 0.00076519, 7.26854280, 0.99618417, 0.99798669, 0.49289341],
                             [- 6.26979765, -0.00124762, 0.00076488, 7.26855003, 0.99618571, 0.99798750, 0.49289376],
                             [- 6.26980436, -0.00124711, 0.00076458, 7.26855725, 0.99618724, 0.99798831, 0.49289409],
                             [- 6.26981105, -0.00124661, 0.00076427, 7.26856444, 0.99618878, 0.99798912, 0.49289444],
                             [- 6.26981772, -0.00124611, 0.00076396, 7.26857160, 0.99619031, 0.99798992, 0.49289482],
                             [- 6.26982436, -0.00124561, 0.00076366, 7.26857875, 0.99619184, 0.99799073, 0.49289516],
                             [- 6.26983098, -0.00124511, 0.00076335, 7.26858587, 0.99619337, 0.99799154, 0.49289554],
                             [- 6.26983758, -0.00124461, 0.00076305, 7.26859297, 0.99619490, 0.99799234, 0.49289586],
                             [- 6.26984416, -0.00124412, 0.00076274, 7.26860004, 0.99619643, 0.99799314, 0.49289620],
                             [- 6.26985071, -0.00124362, 0.00076244, 7.26860710, 0.99619796, 0.99799395, 0.49289657],
                             [- 6.26985725, -0.00124312, 0.00076213, 7.26861413, 0.99619949, 0.99799475, 0.49289690],
                             [- 6.26986376, -0.00124262, 0.00076183, 7.26862114, 0.99620101, 0.99799555, 0.49289724],
                             [- 6.26987025, -0.00124212, 0.00076152, 7.26862812, 0.99620253, 0.99799636, 0.49289757],
                             [- 6.26987672, -0.00124163, 0.00076122, 7.26863509, 0.99620406, 0.99799716, 0.49289792],
                             [- 6.26988316, -0.00124113, 0.00076091, 7.26864203, 0.99620558, 0.99799796, 0.49289828],
                             [- 6.26988959, -0.00124063, 0.00076061, 7.26864895, 0.99620710, 0.99799876, 0.49289862],
                             [- 6.26989599, -0.00124014, 0.00076031, 7.26865585, 0.99620862, 0.99799956, 0.49289895],
                             [- 6.26990237, -0.00123964, 0.00076000, 7.26866273, 0.99621013, 0.99800035, 0.49289931],
                             [- 6.26990873, -0.00123915, 0.00075970, 7.26866958, 0.99621165, 0.99800115, 0.49289965],
                             [- 6.26991507, -0.00123865, 0.00075940, 7.26867642, 0.99621317, 0.99800195, 0.49289998],
                             [- 6.26992139, -0.00123816, 0.00075909, 7.26868323, 0.99621468, 0.99800275, 0.49290032],
                             [- 6.26992769, -0.00123767, 0.00075879, 7.26869002, 0.99621619, 0.99800354, 0.49290066],
                             [- 6.26993396, -0.00123717, 0.00075849, 7.26869679, 0.99621770, 0.99800434, 0.49290100],
                             [- 6.26994022, -0.00123668, 0.00075819, 7.26870354, 0.99621921, 0.99800513, 0.49290132],
                             [- 6.26994645, -0.00123619, 0.00075789, 7.26871027, 0.99622072, 0.99800593, 0.49290166],
                             [- 6.26995267, -0.00123569, 0.00075758, 7.26871697, 0.99622223, 0.99800672, 0.49290199],
                             [- 6.26995886, -0.00123520, 0.00075728, 7.26872366, 0.99622374, 0.99800751, 0.49290231],
                             [- 6.26996503, -0.00123471, 0.00075698, 7.26873032, 0.99622524, 0.99800831, 0.49290268],
                             [- 6.26997119, -0.00123422, 0.00075668, 7.26873697, 0.99622675, 0.99800910, 0.49290298],
                             [- 6.26997732, -0.00123373, 0.00075638, 7.26874359, 0.99622825, 0.99800989, 0.49290333],
                             [- 6.26998343, -0.00123324, 0.00075608, 7.26875019, 0.99622975, 0.99801068, 0.49290366],
                             [- 6.26998952, -0.00123275, 0.00075578, 7.26875677, 0.99623126, 0.99801147, 0.49290397],
                             [- 6.26999559, -0.00123226, 0.00075548, 7.26876333, 0.99623275, 0.99801226, 0.49290432],
                             [- 6.27000164, -0.00123177, 0.00075518, 7.26876987, 0.99623425, 0.99801305, 0.49290464],
                             [- 6.27000767, -0.00123128, 0.00075488, 7.26877639, 0.99623575, 0.99801384, 0.49290499],
                             [- 6.27001368, -0.00123079, 0.00075458, 7.26878289, 0.99623725, 0.99801462, 0.49290530],
                             [- 6.27001968, -0.00123031, 0.00075428, 7.26878937, 0.99623874, 0.99801541, 0.49290562],
                             [- 6.27002565, -0.00122982, 0.00075399, 7.26879583, 0.99624024, 0.99801620, 0.49290596],
                             [- 6.27003160, -0.00122933, 0.00075369, 7.26880227, 0.99624173, 0.99801698, 0.49290628],
                             [- 6.27003753, -0.00122884, 0.00075339, 7.26880869, 0.99624322, 0.99801777, 0.49290660],
                             [- 6.27004344, -0.00122836, 0.00075309, 7.26881508, 0.99624471, 0.99801855, 0.49290693],
                             [- 6.27004933, -0.00122787, 0.00075279, 7.26882146, 0.99624620, 0.99801934, 0.49290725],
                             [- 6.27005521, -0.00122739, 0.00075250, 7.26882782, 0.99624769, 0.99802012, 0.49290756],
                             [- 6.27006106, -0.00122690, 0.00075220, 7.26883416, 0.99624917, 0.99802090, 0.49290790],
                             [- 6.27006690, -0.00122642, 0.00075190, 7.26884048, 0.99625066, 0.99802168, 0.49290821],
                             [- 6.27007271, -0.00122593, 0.00075160, 7.26884678, 0.99625214, 0.99802246, 0.49290851],
                             [- 6.27007851, -0.00122545, 0.00075131, 7.26885306, 0.99625363, 0.99802324, 0.49290885],
                             [- 6.27008428, -0.00122496, 0.00075101, 7.26885932, 0.99625511, 0.99802402, 0.49290916],
                             [- 6.27009004, -0.00122448, 0.00075072, 7.26886556, 0.99625659, 0.99802480, 0.49290948],
                             [- 6.27009578, -0.00122400, 0.00075042, 7.26887178, 0.99625807, 0.99802558, 0.49290980],
                             [- 6.27010150, -0.00122351, 0.00075012, 7.26887799, 0.99625955, 0.99802636, 0.49291012],
                             [- 6.27010720, -0.00122303, 0.00074983, 7.26888417, 0.99626102, 0.99802714, 0.49291045],
                             [- 6.27011288, -0.00122255, 0.00074953, 7.26889033, 0.99626250, 0.99802792, 0.49291075],
                             [- 6.27011855, -0.00122207, 0.00074924, 7.26889648, 0.99626398, 0.99802869, 0.49291108],
                             [- 6.27012419, -0.00122159, 0.00074894, 7.26890260, 0.99626545, 0.99802947, 0.49291137],
                             [- 6.27012982, -0.00122111, 0.00074865, 7.26890871, 0.99626692, 0.99803024, 0.49291169],
                             [- 6.27013543, -0.00122063, 0.00074836, 7.26891480, 0.99626839, 0.99803102, 0.49291199],
                             [- 6.27014102, -0.00122015, 0.00074806, 7.26892087, 0.99626986, 0.99803179, 0.49291232],
                             [- 6.27014659, -0.00121967, 0.00074777, 7.26892692, 0.99627133, 0.99803257, 0.49291262],
                             [- 6.27015214, -0.00121919, 0.00074747, 7.26893295, 0.99627280, 0.99803334, 0.49291295],
                             [- 6.27015767, -0.00121871, 0.00074718, 7.26893897, 0.99627427, 0.99803411, 0.49291325],
                             [- 6.27016319, -0.00121823, 0.00074689, 7.26894496, 0.99627574, 0.99803488, 0.49291356],
                             [- 6.27016869, -0.00121775, 0.00074659, 7.26895094, 0.99627720, 0.99803565, 0.49291386],
                             [- 6.27017417, -0.00121727, 0.00074630, 7.26895690, 0.99627866, 0.99803642, 0.49291418],
                             [- 6.27017963, -0.00121680, 0.00074601, 7.26896284, 0.99628013, 0.99803719, 0.49291448],
                             [- 6.27018508, -0.00121632, 0.00074572, 7.26896876, 0.99628159, 0.99803796, 0.49291478],
                             [- 6.27019050, -0.00121584, 0.00074543, 7.26897466, 0.99628305, 0.99803873, 0.49291509],
                             [- 6.27019591, -0.00121537, 0.00074513, 7.26898055, 0.99628451, 0.99803950, 0.49291539],
                             [- 6.27020130, -0.00121489, 0.00074484, 7.26898641, 0.99628596, 0.99804027, 0.49291571],
                             [- 6.27020668, -0.00121441, 0.00074455, 7.26899226, 0.99628742, 0.99804103, 0.49291599],
                             [- 6.27021203, -0.00121394, 0.00074426, 7.26899810, 0.99628888, 0.99804180, 0.49291629],
                             [- 6.27021737, -0.00121346, 0.00074397, 7.26900391, 0.99629033, 0.99804257, 0.49291660],
                             [- 6.27022270, -0.00121299, 0.00074368, 7.26900971, 0.99629178, 0.99804333, 0.49291688],
                             [- 6.27022800, -0.00121252, 0.00074339, 7.26901548, 0.99629324, 0.99804410, 0.49291722],
                             [- 6.27023329, -0.00121204, 0.00074310, 7.26902124, 0.99629469, 0.99804486, 0.49291750],
                             [- 6.27023856, -0.00121157, 0.00074281, 7.26902699, 0.99629614, 0.99804562, 0.49291781],
                             [- 6.27024381, -0.00121110, 0.00074252, 7.26903271, 0.99629759, 0.99804639, 0.49291812],
                             [- 6.27024904, -0.00121062, 0.00074223, 7.26903842, 0.99629903, 0.99804715, 0.49291840],
                             [- 6.27025426, -0.00121015, 0.00074194, 7.26904411, 0.99630048, 0.99804791, 0.49291870],
                             [- 6.27025946, -0.00120968, 0.00074165, 7.26904979, 0.99630193, 0.99804867, 0.49291902],
                             [- 6.27026465, -0.00120921, 0.00074136, 7.26905544, 0.99630337, 0.99804943, 0.49291932],
                             [- 6.27026982, -0.00120874, 0.00074107, 7.26906108, 0.99630481, 0.99805019, 0.49291960],
                             [- 6.27027497, -0.00120827, 0.00074078, 7.26906670, 0.99630626, 0.99805095, 0.49291988],
                             [- 6.27028010, -0.00120780, 0.00074050, 7.26907231, 0.99630770, 0.99805171, 0.49292018],
                             [- 6.27028522, -0.00120732, 0.00074021, 7.26907789, 0.99630914, 0.99805247, 0.49292047],
                             [- 6.27029032, -0.00120686, 0.00073992, 7.26908347, 0.99631057, 0.99805322, 0.49292077],
                             [- 6.27029541, -0.00120639, 0.00073963, 7.26908902, 0.99631201, 0.99805398, 0.49292106],
                             [- 6.27030047, -0.00120592, 0.00073934, 7.26909456, 0.99631345, 0.99805474, 0.49292137],
                             [- 6.27030552, -0.00120545, 0.00073906, 7.26910008, 0.99631488, 0.99805549, 0.49292164],
                             [- 6.27031056, -0.00120498, 0.00073877, 7.26910558, 0.99631632, 0.99805625, 0.49292193],
                             [- 6.27031558, -0.00120451, 0.00073848, 7.26911107, 0.99631775, 0.99805700, 0.49292223],
                             [- 6.27032058, -0.00120404, 0.00073820, 7.26911654, 0.99631918, 0.99805776, 0.49292252],
                             [- 6.27032557, -0.00120358, 0.00073791, 7.26912199, 0.99632061, 0.99805851, 0.49292278],
                             [- 6.27033054, -0.00120311, 0.00073763, 7.26912743, 0.99632204, 0.99805926, 0.49292312],
                             [- 6.27033549, -0.00120264, 0.00073734, 7.26913285, 0.99632347, 0.99806002, 0.49292338],
                             [- 6.27034043, -0.00120218, 0.00073705, 7.26913825, 0.99632490, 0.99806077, 0.49292367],
                             [- 6.27034536, -0.00120171, 0.00073677, 7.26914364, 0.99632633, 0.99806152, 0.49292394],
                             [- 6.27035026, -0.00120125, 0.00073648, 7.26914902, 0.99632775, 0.99806227, 0.49292425],
                             [- 6.27035515, -0.00120078, 0.00073620, 7.26915437, 0.99632918, 0.99806302, 0.49292452],
                             [- 6.27036003, -0.00120032, 0.00073591, 7.26915971, 0.99633060, 0.99806377, 0.49292480],
                             [- 6.27036489, -0.00119985, 0.00073563, 7.26916504, 0.99633202, 0.99806452, 0.49292511],
                             [- 6.27036973, -0.00119939, 0.00073535, 7.26917034, 0.99633344, 0.99806527, 0.49292536],
                             [- 6.27037456, -0.00119892, 0.00073506, 7.26917564, 0.99633486, 0.99806601, 0.49292567],
                             [- 6.27037937, -0.00119846, 0.00073478, 7.26918091, 0.99633628, 0.99806676, 0.49292595],
                             [- 6.27038417, -0.00119800, 0.00073449, 7.26918617, 0.99633770, 0.99806751, 0.49292622],
                             [- 6.27038895, -0.00119754, 0.00073421, 7.26919142, 0.99633912, 0.99806825, 0.49292651],
                             [- 6.27039372, -0.00119707, 0.00073393, 7.26919664, 0.99634053, 0.99806900, 0.49292678],
                             [- 6.27039847, -0.00119661, 0.00073364, 7.26920186, 0.99634195, 0.99806974, 0.49292707],
                             [- 6.27040321, -0.00119615, 0.00073336, 7.26920706, 0.99634336, 0.99807049, 0.49292736],
                             [- 6.27040793, -0.00119569, 0.00073308, 7.26921224, 0.99634477, 0.99807123, 0.49292762],
                             [- 6.27041263, -0.00119523, 0.00073280, 7.26921740, 0.99634618, 0.99807198, 0.49292790],
                             [- 6.27041732, -0.00119477, 0.00073251, 7.26922256, 0.99634759, 0.99807272, 0.49292816],
                             [- 6.27042200, -0.00119431, 0.00073223, 7.26922769, 0.99634900, 0.99807346, 0.49292845],
                             [- 6.27042666, -0.00119385, 0.00073195, 7.26923281, 0.99635041, 0.99807420, 0.49292873],
                             [- 6.27043131, -0.00119339, 0.00073167, 7.26923792, 0.99635182, 0.99807494, 0.49292902],
                             [- 6.27043594, -0.00119293, 0.00073139, 7.26924301, 0.99635322, 0.99807568, 0.49292929],
                             [- 6.27044055, -0.00119247, 0.00073111, 7.26924808, 0.99635463, 0.99807642, 0.49292955],
                             [- 6.27044516, -0.00119201, 0.00073083, 7.26925314, 0.99635603, 0.99807716, 0.49292983],
                             [- 6.27044974, -0.00119155, 0.00073054, 7.26925819, 0.99635743, 0.99807790, 0.49293011],
                             [- 6.27045432, -0.00119110, 0.00073026, 7.26926322, 0.99635884, 0.99807864, 0.49293037],
                             [- 6.27045887, -0.00119064, 0.00072998, 7.26926823, 0.99636024, 0.99807938, 0.49293066],
                             [- 6.27046342, -0.00119018, 0.00072970, 7.26927323, 0.99636164, 0.99808011, 0.49293095],
                             [- 6.27046795, -0.00118973, 0.00072942, 7.26927822, 0.99636303, 0.99808085, 0.49293118],
                             [- 6.27047246, -0.00118927, 0.00072914, 7.26928319, 0.99636443, 0.99808158, 0.49293146],
                             [- 6.27047696, -0.00118881, 0.00072887, 7.26928815, 0.99636583, 0.99808232, 0.49293175],
                             [- 6.27048145, -0.00118836, 0.00072859, 7.26929309, 0.99636722, 0.99808305, 0.49293201],
                             [- 6.27048592, -0.00118790, 0.00072831, 7.26929802, 0.99636862, 0.99808379, 0.49293228],
                             [- 6.27049038, -0.00118745, 0.00072803, 7.26930293, 0.99637001, 0.99808452, 0.49293255],
                             [- 6.27049482, -0.00118699, 0.00072775, 7.26930783, 0.99637140, 0.99808526, 0.49293281],
                             [- 6.27049925, -0.00118654, 0.00072747, 7.26931271, 0.99637279, 0.99808599, 0.49293309],
                             [- 6.27050366, -0.00118609, 0.00072719, 7.26931758, 0.99637418, 0.99808672, 0.49293334],
                             [- 6.27050807, -0.00118563, 0.00072692, 7.26932243, 0.99637557, 0.99808745, 0.49293360],
                             [- 6.27051245, -0.00118518, 0.00072664, 7.26932727, 0.99637696, 0.99808818, 0.49293390],
                             [- 6.27051683, -0.00118473, 0.00072636, 7.26933210, 0.99637835, 0.99808891, 0.49293415],
                             [- 6.27052119, -0.00118427, 0.00072608, 7.26933691, 0.99637973, 0.99808964, 0.49293444],
                             [- 6.27052553, -0.00118382, 0.00072581, 7.26934171, 0.99638112, 0.99809037, 0.49293467],
                             [- 6.27052986, -0.00118337, 0.00072553, 7.26934649, 0.99638250, 0.99809110, 0.49293494],
                             [- 6.27053418, -0.00118292, 0.00072525, 7.26935126, 0.99638388, 0.99809183, 0.49293521],
                             [- 6.27053849, -0.00118247, 0.00072498, 7.26935602, 0.99638526, 0.99809256, 0.49293547],
                             [- 6.27054278, -0.00118202, 0.00072470, 7.26936076, 0.99638664, 0.99809328, 0.49293575],
                             [- 6.27054706, -0.00118157, 0.00072442, 7.26936549, 0.99638802, 0.99809401, 0.49293600],
                             [- 6.27055132, -0.00118112, 0.00072415, 7.26937020, 0.99638940, 0.99809474, 0.49293625],
                             [- 6.27055557, -0.00118067, 0.00072387, 7.26937491, 0.99639078, 0.99809546, 0.49293651],
                             [- 6.27055981, -0.00118022, 0.00072360, 7.26937959, 0.99639215, 0.99809619, 0.49293677],
                             [- 6.27056403, -0.00117977, 0.00072332, 7.26938427, 0.99639353, 0.99809691, 0.49293703],
                             [- 6.27056825, -0.00117932, 0.00072305, 7.26938893, 0.99639490, 0.99809763, 0.49293729],
                             [- 6.27057244, -0.00117887, 0.00072277, 7.26939357, 0.99639628, 0.99809836, 0.49293754],
                             [- 6.27057663, -0.00117842, 0.00072250, 7.26939820, 0.99639765, 0.99809908, 0.49293783],
                             [- 6.27058080, -0.00117798, 0.00072222, 7.26940282, 0.99639902, 0.99809980, 0.49293807],
                             [- 6.27058496, -0.00117753, 0.00072195, 7.26940743, 0.99640039, 0.99810052, 0.49293833],
                             [- 6.27058910, -0.00117708, 0.00072167, 7.26941202, 0.99640176, 0.99810124, 0.49293858],
                             [- 6.27059324, -0.00117663, 0.00072140, 7.26941660, 0.99640312, 0.99810196, 0.49293885],
                             [- 6.27059736, -0.00117619, 0.00072113, 7.26942117, 0.99640449, 0.99810268, 0.49293910],
                             [- 6.27060146, -0.00117574, 0.00072085, 7.26942572, 0.99640586, 0.99810340, 0.49293936],
                             [- 6.27060556, -0.00117530, 0.00072058, 7.26943026, 0.99640722, 0.99810412, 0.49293961],
                             [- 6.27060964, -0.00117485, 0.00072031, 7.26943479, 0.99640859, 0.99810484, 0.49293988],
                             [- 6.27061371, -0.00117441, 0.00072004, 7.26943930, 0.99640995, 0.99810556, 0.49294014],
                             [- 6.27061776, -0.00117396, 0.00071976, 7.26944380, 0.99641131, 0.99810628, 0.49294037],
                             [- 6.27062180, -0.00117352, 0.00071949, 7.26944829, 0.99641267, 0.99810699, 0.49294061],
                             [- 6.27062583, -0.00117307, 0.00071922, 7.26945276, 0.99641403, 0.99810771, 0.49294091],
                             [- 6.27062985, -0.00117263, 0.00071895, 7.26945722, 0.99641539, 0.99810842, 0.49294113],
                             [- 6.27063386, -0.00117219, 0.00071867, 7.26946167, 0.99641675, 0.99810914, 0.49294137],
                             [- 6.27063785, -0.00117174, 0.00071840, 7.26946611, 0.99641810, 0.99810985, 0.49294161],
                             [- 6.27064183, -0.00117130, 0.00071813, 7.26947053, 0.99641946, 0.99811057, 0.49294187],
                             [- 6.27064580, -0.00117086, 0.00071786, 7.26947494, 0.99642081, 0.99811128, 0.49294216],
                             [- 6.27064976, -0.00117042, 0.00071759, 7.26947934, 0.99642217, 0.99811199, 0.49294239],
                             [- 6.27065370, -0.00116997, 0.00071732, 7.26948372, 0.99642352, 0.99811271, 0.49294263],
                             [- 6.27065763, -0.00116953, 0.00071705, 7.26948810, 0.99642487, 0.99811342, 0.49294289],
                             [- 6.27066155, -0.00116909, 0.00071678, 7.26949246, 0.99642622, 0.99811413, 0.49294314],
                             [- 6.27066546, -0.00116865, 0.00071651, 7.26949680, 0.99642757, 0.99811484, 0.49294338],
                             [- 6.27066935, -0.00116821, 0.00071624, 7.26950114, 0.99642892, 0.99811555, 0.49294362],
                             [- 6.27067323, -0.00116777, 0.00071597, 7.26950546, 0.99643026, 0.99811626, 0.49294387],
                             [- 6.27067711, -0.00116733, 0.00071570, 7.26950977, 0.99643161, 0.99811697, 0.49294414],
                             [- 6.27068096, -0.00116689, 0.00071543, 7.26951407, 0.99643296, 0.99811768, 0.49294435],
                             [- 6.27068481, -0.00116645, 0.00071516, 7.26951836, 0.99643430, 0.99811839, 0.49294462],
                             [- 6.27068865, -0.00116601, 0.00071489, 7.26952263, 0.99643564, 0.99811909, 0.49294486],
                             [- 6.27069247, -0.00116558, 0.00071462, 7.26952689, 0.99643699, 0.99811980, 0.49294512],
                             [- 6.27069628, -0.00116514, 0.00071435, 7.26953114, 0.99643833, 0.99812051, 0.49294533],
                             [- 6.27070008, -0.00116470, 0.00071409, 7.26953538, 0.99643967, 0.99812121, 0.49294559],
                             [- 6.27070387, -0.00116426, 0.00071382, 7.26953961, 0.99644101, 0.99812192, 0.49294582],
                             [- 6.27070765, -0.00116383, 0.00071355, 7.26954382, 0.99644235, 0.99812262, 0.49294607],
                             [- 6.27071141, -0.00116339, 0.00071328, 7.26954802, 0.99644368, 0.99812333, 0.49294633],
                             [- 6.27071516, -0.00116295, 0.00071301, 7.26955221, 0.99644502, 0.99812403, 0.49294657],
                             [- 6.27071891, -0.00116252, 0.00071275, 7.26955639, 0.99644635, 0.99812474, 0.49294678],
                             [- 6.27072264, -0.00116208, 0.00071248, 7.26956056, 0.99644769, 0.99812544, 0.49294701],
                             [- 6.27072636, -0.00116165, 0.00071221, 7.26956471, 0.99644902, 0.99812614, 0.49294726],
                             [- 6.27073006, -0.00116121, 0.00071195, 7.26956885, 0.99645035, 0.99812684, 0.49294753],
                             [- 6.27073376, -0.00116078, 0.00071168, 7.26957298, 0.99645169, 0.99812755, 0.49294774],
                             [- 6.27073744, -0.00116034, 0.00071141, 7.26957710, 0.99645302, 0.99812825, 0.49294799],
                             [- 6.27074112, -0.00115991, 0.00071115, 7.26958121, 0.99645434, 0.99812895, 0.49294824],
                             [- 6.27074478, -0.00115947, 0.00071088, 7.26958531, 0.99645567, 0.99812965, 0.49294849],
                             [- 6.27074843, -0.00115904, 0.00071062, 7.26958939, 0.99645700, 0.99813035, 0.49294867],
                             [- 6.27075207, -0.00115861, 0.00071035, 7.26959347, 0.99645833, 0.99813104, 0.49294894],
                             [- 6.27075570, -0.00115817, 0.00071008, 7.26959753, 0.99645965, 0.99813174, 0.49294918],
                             [- 6.27075932, -0.00115774, 0.00070982, 7.26960158, 0.99646098, 0.99813244, 0.49294940],
                             [- 6.27076293, -0.00115731, 0.00070955, 7.26960562, 0.99646230, 0.99813314, 0.49294966],
                             [- 6.27076652, -0.00115688, 0.00070929, 7.26960965, 0.99646362, 0.99813383, 0.49294988],
                             [- 6.27077011, -0.00115644, 0.00070902, 7.26961366, 0.99646494, 0.99813453, 0.49295014],
                             [- 6.27077368, -0.00115601, 0.00070876, 7.26961767, 0.99646626, 0.99813523, 0.49295035],
                             [- 6.27077724, -0.00115558, 0.00070850, 7.26962166, 0.99646758, 0.99813592, 0.49295058],
                             [- 6.27078080, -0.00115515, 0.00070823, 7.26962565, 0.99646890, 0.99813662, 0.49295081],
                             [- 6.27078434, -0.00115472, 0.00070797, 7.26962962, 0.99647022, 0.99813731, 0.49295107],
                             [- 6.27078787, -0.00115429, 0.00070770, 7.26963358, 0.99647154, 0.99813800, 0.49295129],
                             [- 6.27079139, -0.00115386, 0.00070744, 7.26963753, 0.99647285, 0.99813870, 0.49295151],
                             [- 6.27079490, -0.00115343, 0.00070718, 7.26964147, 0.99647417, 0.99813939, 0.49295175],
                             [- 6.27079840, -0.00115300, 0.00070692, 7.26964540, 0.99647548, 0.99814008, 0.49295200],
                             [- 6.27080189, -0.00115257, 0.00070665, 7.26964931, 0.99647679, 0.99814077, 0.49295221],
                             [- 6.27080537, -0.00115215, 0.00070639, 7.26965322, 0.99647810, 0.99814146, 0.49295246],
                             [- 6.27080883, -0.00115172, 0.00070613, 7.26965711, 0.99647941, 0.99814215, 0.49295267],
                             [- 6.27081229, -0.00115129, 0.00070586, 7.26966100, 0.99648072, 0.99814284, 0.49295289],
                             [- 6.27081574, -0.00115086, 0.00070560, 7.26966487, 0.99648203, 0.99814353, 0.49295314],
                             [- 6.27081917, -0.00115044, 0.00070534, 7.26966874, 0.99648334, 0.99814422, 0.49295337],
                             [- 6.27082260, -0.00115001, 0.00070508, 7.26967259, 0.99648465, 0.99814491, 0.49295357],
                             [- 6.27082601, -0.00114958, 0.00070482, 7.26967643, 0.99648595, 0.99814560, 0.49295381],
                             [- 6.27082942, -0.00114916, 0.00070456, 7.26968026, 0.99648726, 0.99814629, 0.49295404],
                             [- 6.27083281, -0.00114873, 0.00070430, 7.26968408, 0.99648856, 0.99814697, 0.49295428],
                             [- 6.27083620, -0.00114830, 0.00070403, 7.26968789, 0.99648986, 0.99814766, 0.49295451],
                             [- 6.27083957, -0.00114788, 0.00070377, 7.26969169, 0.99649117, 0.99814835, 0.49295473],
                             [- 6.27084294, -0.00114745, 0.00070351, 7.26969548, 0.99649247, 0.99814903, 0.49295497],
                             [- 6.27084629, -0.00114703, 0.00070325, 7.26969926, 0.99649377, 0.99814972, 0.49295518],
                             [- 6.27084964, -0.00114661, 0.00070299, 7.26970303, 0.99649507, 0.99815040, 0.49295540],
                             [- 6.27085297, -0.00114618, 0.00070273, 7.26970679, 0.99649636, 0.99815109, 0.49295563],
                             [- 6.27085630, -0.00114576, 0.00070247, 7.26971054, 0.99649766, 0.99815177, 0.49295584],
                             [- 6.27085961, -0.00114533, 0.00070221, 7.26971428, 0.99649896, 0.99815245, 0.49295609],
                             [- 6.27086292, -0.00114491, 0.00070195, 7.26971800, 0.99650025, 0.99815313, 0.49295629],
                             [- 6.27086621, -0.00114449, 0.00070169, 7.26972172, 0.99650155, 0.99815382, 0.49295653],
                             [- 6.27086950, -0.00114407, 0.00070144, 7.26972543, 0.99650284, 0.99815450, 0.49295673],
                             [- 6.27087277, -0.00114364, 0.00070118, 7.26972913, 0.99650413, 0.99815518, 0.49295696],
                             [- 6.27087604, -0.00114322, 0.00070092, 7.26973281, 0.99650542, 0.99815586, 0.49295720],
                             [- 6.27087929, -0.00114280, 0.00070066, 7.26973649, 0.99650671, 0.99815654, 0.49295739],
                             [- 6.27088254, -0.00114238, 0.00070040, 7.26974016, 0.99650800, 0.99815722, 0.49295764],
                             [- 6.27088577, -0.00114196, 0.00070014, 7.26974382, 0.99650929, 0.99815790, 0.49295785],
                             [- 6.27088900, -0.00114154, 0.00069989, 7.26974746, 0.99651058, 0.99815858, 0.49295811],
                             [- 6.27089222, -0.00114112, 0.00069963, 7.26975110, 0.99651187, 0.99815925, 0.49295829],
                             [- 6.27089543, -0.00114070, 0.00069937, 7.26975473, 0.99651315, 0.99815993, 0.49295850],
                             [- 6.27089862, -0.00114028, 0.00069911, 7.26975835, 0.99651444, 0.99816061, 0.49295875],
                             [- 6.27090181, -0.00113986, 0.00069886, 7.26976195, 0.99651572, 0.99816129, 0.49295895],
                             [- 6.27090499, -0.00113944, 0.00069860, 7.26976555, 0.99651700, 0.99816196, 0.49295918],
                             [- 6.27090816, -0.00113902, 0.00069834, 7.26976914, 0.99651829, 0.99816264, 0.49295938],
                             [- 6.27091132, -0.00113860, 0.00069809, 7.26977272, 0.99651957, 0.99816331, 0.49295963],
                             [- 6.27091447, -0.00113818, 0.00069783, 7.26977629, 0.99652085, 0.99816399, 0.49295982],
                             [- 6.27091762, -0.00113777, 0.00069757, 7.26977985, 0.99652213, 0.99816466, 0.49296003],
                             [- 6.27092075, -0.00113735, 0.00069732, 7.26978340, 0.99652340, 0.99816533, 0.49296028],
                             [- 6.27092387, -0.00113693, 0.00069706, 7.26978694, 0.99652468, 0.99816601, 0.49296049],
                             [- 6.27092699, -0.00113651, 0.00069681, 7.26979047, 0.99652596, 0.99816668, 0.49296071],
                             [- 6.27093009, -0.00113610, 0.00069655, 7.26979399, 0.99652723, 0.99816735, 0.49296091],
                             [- 6.27093319, -0.00113568, 0.00069629, 7.26979751, 0.99652851, 0.99816802, 0.49296112],
                             [- 6.27093627, -0.00113527, 0.00069604, 7.26980101, 0.99652978, 0.99816869, 0.49296137],
                             [- 6.27093935, -0.00113485, 0.00069578, 7.26980450, 0.99653105, 0.99816937, 0.49296155],
                             [- 6.27094242, -0.00113443, 0.00069553, 7.26980799, 0.99653232, 0.99817004, 0.49296176],
                             [- 6.27094548, -0.00113402, 0.00069528, 7.26981146, 0.99653360, 0.99817070, 0.49296198],
                             [- 6.27094853, -0.00113360, 0.00069502, 7.26981493, 0.99653487, 0.99817137, 0.49296221],
                             [- 6.27095157, -0.00113319, 0.00069477, 7.26981838, 0.99653613, 0.99817204, 0.49296241],
                             [- 6.27095461, -0.00113278, 0.00069451, 7.26982183, 0.99653740, 0.99817271, 0.49296261],
                             [- 6.27095763, -0.00113236, 0.00069426, 7.26982527, 0.99653867, 0.99817338, 0.49296284],
                             [- 6.27096065, -0.00113195, 0.00069401, 7.26982870, 0.99653994, 0.99817405, 0.49296306],
                             [- 6.27096365, -0.00113154, 0.00069375, 7.26983212, 0.99654120, 0.99817471, 0.49296324],
                             [- 6.27096665, -0.00113112, 0.00069350, 7.26983553, 0.99654246, 0.99817538, 0.49296349],
                             [- 6.27096964, -0.00113071, 0.00069325, 7.26983893, 0.99654373, 0.99817604, 0.49296367],
                             [- 6.27097262, -0.00113030, 0.00069299, 7.26984232, 0.99654499, 0.99817671, 0.49296390],
                             [- 6.27097559, -0.00112989, 0.00069274, 7.26984571, 0.99654625, 0.99817737, 0.49296411],
                             [- 6.27097856, -0.00112947, 0.00069249, 7.26984908, 0.99654751, 0.99817804, 0.49296432],
                             [- 6.27098151, -0.00112906, 0.00069224, 7.26985245, 0.99654877, 0.99817870, 0.49296451],
                             [- 6.27098446, -0.00112865, 0.00069198, 7.26985581, 0.99655003, 0.99817937, 0.49296475],
                             [- 6.27098739, -0.00112824, 0.00069173, 7.26985915, 0.99655129, 0.99818003, 0.49296493],
                             [- 6.27099032, -0.00112783, 0.00069148, 7.26986249, 0.99655255, 0.99818069, 0.49296516],
                             [- 6.27099324, -0.00112742, 0.00069123, 7.26986583, 0.99655380, 0.99818135, 0.49296535],
                             [- 6.27099616, -0.00112701, 0.00069098, 7.26986915, 0.99655506, 0.99818202, 0.49296556],
                             [- 6.27099906, -0.00112660, 0.00069072, 7.26987246, 0.99655631, 0.99818268, 0.49296577],
                             [- 6.27100196, -0.00112619, 0.00069047, 7.26987577, 0.99655756, 0.99818334, 0.49296597],
                             [- 6.27100484, -0.00112578, 0.00069022, 7.26987906, 0.99655882, 0.99818400, 0.49296619],
                             [- 6.27100772, -0.00112537, 0.00068997, 7.26988235, 0.99656007, 0.99818466, 0.49296639],
                             [- 6.27101059, -0.00112496, 0.00068972, 7.26988563, 0.99656132, 0.99818532, 0.49296663],
                             [- 6.27101345, -0.00112455, 0.00068947, 7.26988890, 0.99656257, 0.99818597, 0.49296681],
                             [- 6.27101631, -0.00112415, 0.00068922, 7.26989216, 0.99656382, 0.99818663, 0.49296699],
                             [- 6.27101915, -0.00112374, 0.00068897, 7.26989541, 0.99656506, 0.99818729, 0.49296722],
                             [- 6.27102199, -0.00112333, 0.00068872, 7.26989866, 0.99656631, 0.99818795, 0.49296743],
                             [- 6.27102482, -0.00112292, 0.00068847, 7.26990189, 0.99656756, 0.99818860, 0.49296764],
                             [- 6.27102764, -0.00112252, 0.00068822, 7.26990512, 0.99656880, 0.99818926, 0.49296783],
                             [- 6.27103045, -0.00112211, 0.00068797, 7.26990834, 0.99657005, 0.99818992, 0.49296802],
                             [- 6.27103326, -0.00112171, 0.00068772, 7.26991155, 0.99657129, 0.99819057, 0.49296822],
                             [- 6.27103606, -0.00112130, 0.00068747, 7.26991476, 0.99657253, 0.99819123, 0.49296843],
                             [- 6.27103885, -0.00112089, 0.00068723, 7.26991795, 0.99657377, 0.99819188, 0.49296863],
                             [- 6.27104163, -0.00112049, 0.00068698, 7.26992114, 0.99657501, 0.99819253, 0.49296882],
                             [- 6.27104440, -0.00112008, 0.00068673, 7.26992432, 0.99657625, 0.99819319, 0.49296903],
                             [- 6.27104717, -0.00111968, 0.00068648, 7.26992749, 0.99657749, 0.99819384, 0.49296925],
                             [- 6.27104992, -0.00111927, 0.00068623, 7.26993065, 0.99657873, 0.99819449, 0.49296948],
                             [- 6.27105267, -0.00111887, 0.00068599, 7.26993380, 0.99657997, 0.99819514, 0.49296966],
                             [- 6.27105541, -0.00111847, 0.00068574, 7.26993695, 0.99658120, 0.99819580, 0.49296986],
                             [- 6.27105815, -0.00111806, 0.00068549, 7.26994009, 0.99658244, 0.99819645, 0.49297006],
                             [- 6.27106088, -0.00111766, 0.00068524, 7.26994322, 0.99658367, 0.99819710, 0.49297025],
                             [- 6.27106359, -0.00111726, 0.00068500, 7.26994634, 0.99658491, 0.99819775, 0.49297043],
                             [- 6.27106630, -0.00111685, 0.00068475, 7.26994945, 0.99658614, 0.99819840, 0.49297064],
                             [- 6.27106901, -0.00111645, 0.00068450, 7.26995256, 0.99658737, 0.99819905, 0.49297085],
                             [- 6.27107170, -0.00111605, 0.00068426, 7.26995565, 0.99658860, 0.99819970, 0.49297106],
                             [- 6.27107439, -0.00111565, 0.00068401, 7.26995874, 0.99658983, 0.99820034, 0.49297123],
                             [- 6.27107707, -0.00111525, 0.00068376, 7.26996183, 0.99659106, 0.99820099, 0.49297145],
                             [- 6.27107974, -0.00111484, 0.00068352, 7.26996490, 0.99659229, 0.99820164, 0.49297165],
                             [- 6.27108241, -0.00111444, 0.00068327, 7.26996797, 0.99659352, 0.99820229, 0.49297182],
                             [- 6.27108507, -0.00111404, 0.00068302, 7.26997103, 0.99659474, 0.99820293, 0.49297202],
                             [- 6.27108772, -0.00111364, 0.00068278, 7.26997408, 0.99659597, 0.99820358, 0.49297222],
                             [- 6.27109036, -0.00111324, 0.00068253, 7.26997712, 0.99659719, 0.99820422, 0.49297241],
                             [- 6.27109300, -0.00111284, 0.00068229, 7.26998015, 0.99659842, 0.99820487, 0.49297262],
                             [- 6.27109562, -0.00111244, 0.00068204, 7.26998318, 0.99659964, 0.99820551, 0.49297281],
                             [- 6.27109825, -0.00111204, 0.00068180, 7.26998620, 0.99660086, 0.99820616, 0.49297301],
                             [- 6.27110086, -0.00111164, 0.00068155, 7.26998921, 0.99660208, 0.99820680, 0.49297320],
                             [- 6.27110347, -0.00111125, 0.00068131, 7.26999222, 0.99660330, 0.99820744, 0.49297339],
                             [- 6.27110606, -0.00111085, 0.00068106, 7.26999522, 0.99660452, 0.99820809, 0.49297359],
                             [- 6.27110866, -0.00111045, 0.00068082, 7.26999821, 0.99660574, 0.99820873, 0.49297380],
                             [- 6.27111124, -0.00111005, 0.00068058, 7.27000119, 0.99660696, 0.99820937, 0.49297397],
                             [- 6.27111382, -0.00110965, 0.00068033, 7.27000416, 0.99660818, 0.99821001, 0.49297417],
                             [- 6.27111639, -0.00110926, 0.00068009, 7.27000713, 0.99660939, 0.99821065, 0.49297437],
                             [- 6.27111895, -0.00110886, 0.00067985, 7.27001009, 0.99661061, 0.99821129, 0.49297456],
                             [- 6.27112151, -0.00110846, 0.00067960, 7.27001304, 0.99661182, 0.99821193, 0.49297477],
                             [- 6.27112405, -0.00110807, 0.00067936, 7.27001599, 0.99661303, 0.99821257, 0.49297493],
                             [- 6.27112660, -0.00110767, 0.00067912, 7.27001892, 0.99661425, 0.99821321, 0.49297516],
                             [- 6.27112913, -0.00110728, 0.00067887, 7.27002185, 0.99661546, 0.99821385, 0.49297533],
                             [- 6.27113166, -0.00110688, 0.00067863, 7.27002478, 0.99661667, 0.99821449, 0.49297550],
                             [- 6.27113418, -0.00110648, 0.00067839, 7.27002769, 0.99661788, 0.99821513, 0.49297572],
                             [- 6.27113669, -0.00110609, 0.00067815, 7.27003060, 0.99661909, 0.99821576, 0.49297589],
                             [- 6.27113920, -0.00110569, 0.00067790, 7.27003350, 0.99662030, 0.99821640, 0.49297607],
                             [- 6.27114170, -0.00110530, 0.00067766, 7.27003640, 0.99662150, 0.99821704, 0.49297628],
                             [- 6.27114419, -0.00110491, 0.00067742, 7.27003928, 0.99662271, 0.99821767, 0.49297646],
                             [- 6.27114668, -0.00110451, 0.00067718, 7.27004216, 0.99662392, 0.99821831, 0.49297664],
                             [- 6.27114915, -0.00110412, 0.00067694, 7.27004504, 0.99662512, 0.99821894, 0.49297684],
                             [- 6.27115163, -0.00110372, 0.00067670, 7.27004790, 0.99662632, 0.99821958, 0.49297702],
                             [- 6.27115409, -0.00110333, 0.00067646, 7.27005076, 0.99662753, 0.99822021, 0.49297723],
                             [- 6.27115655, -0.00110294, 0.00067621, 7.27005361, 0.99662873, 0.99822085, 0.49297742],
                             [- 6.27115900, -0.00110255, 0.00067597, 7.27005646, 0.99662993, 0.99822148, 0.49297759],
                             [- 6.27116145, -0.00110215, 0.00067573, 7.27005929, 0.99663113, 0.99822211, 0.49297781],
                             [- 6.27116389, -0.00110176, 0.00067549, 7.27006212, 0.99663233, 0.99822274, 0.49297799],
                             [- 6.27116632, -0.00110137, 0.00067525, 7.27006495, 0.99663353, 0.99822338, 0.49297818],
                             [- 6.27116874, -0.00110098, 0.00067501, 7.27006776, 0.99663473, 0.99822401, 0.49297837],
                             [- 6.27117116, -0.00110059, 0.00067477, 7.27007057, 0.99663592, 0.99822464, 0.49297853],
                             [- 6.27117357, -0.00110020, 0.00067453, 7.27007338, 0.99663712, 0.99822527, 0.49297873],
                             [- 6.27117598, -0.00109981, 0.00067429, 7.27007617, 0.99663832, 0.99822590, 0.49297890],
                             [- 6.27117838, -0.00109942, 0.00067405, 7.27007896, 0.99663951, 0.99822653, 0.49297910],
                             [- 6.27118077, -0.00109903, 0.00067381, 7.27008174, 0.99664070, 0.99822716, 0.49297929],
                             [- 6.27118315, -0.00109864, 0.00067358, 7.27008452, 0.99664190, 0.99822779, 0.49297945],
                             [- 6.27118553, -0.00109825, 0.00067334, 7.27008729, 0.99664309, 0.99822842, 0.49297965],
                             [- 6.27118791, -0.00109786, 0.00067310, 7.27009005, 0.99664428, 0.99822904, 0.49297987],
                             [- 6.27119027, -0.00109747, 0.00067286, 7.27009280, 0.99664547, 0.99822967, 0.49298005],
                             [- 6.27119263, -0.00109708, 0.00067262, 7.27009555, 0.99664666, 0.99823030, 0.49298023],
                             [- 6.27119499, -0.00109669, 0.00067238, 7.27009829, 0.99664785, 0.99823092, 0.49298041],
                             [- 6.27119733, -0.00109630, 0.00067214, 7.27010103, 0.99664904, 0.99823155, 0.49298056],
                             [- 6.27119968, -0.00109592, 0.00067191, 7.27010376, 0.99665022, 0.99823218, 0.49298074],
                             [- 6.27120201, -0.00109553, 0.00067167, 7.27010648, 0.99665141, 0.99823280, 0.49298095],
                             [- 6.27120434, -0.00109514, 0.00067143, 7.27010920, 0.99665259, 0.99823343, 0.49298115],
                             [- 6.27120666, -0.00109475, 0.00067119, 7.27011190, 0.99665378, 0.99823405, 0.49298131],
                             [- 6.27120898, -0.00109437, 0.00067096, 7.27011461, 0.99665496, 0.99823467, 0.49298149],
                             [- 6.27121129, -0.00109398, 0.00067072, 7.27011730, 0.99665614, 0.99823530, 0.49298166],
                             [- 6.27121359, -0.00109360, 0.00067048, 7.27011999, 0.99665733, 0.99823592, 0.49298185],
                             [- 6.27121589, -0.00109321, 0.00067025, 7.27012268, 0.99665851, 0.99823654, 0.49298202],
                             [- 6.27121818, -0.00109282, 0.00067001, 7.27012535, 0.99665969, 0.99823717, 0.49298224],
                             [- 6.27122046, -0.00109244, 0.00066977, 7.27012802, 0.99666087, 0.99823779, 0.49298236],
                             [- 6.27122274, -0.00109205, 0.00066954, 7.27013069, 0.99666205, 0.99823841, 0.49298260],
                             [- 6.27122501, -0.00109167, 0.00066930, 7.27013334, 0.99666322, 0.99823903, 0.49298275],
                             [- 6.27122728, -0.00109128, 0.00066907, 7.27013600, 0.99666440, 0.99823965, 0.49298294],
                             [- 6.27122954, -0.00109090, 0.00066883, 7.27013864, 0.99666558, 0.99824027, 0.49298313],
                             [- 6.27123179, -0.00109052, 0.00066859, 7.27014128, 0.99666675, 0.99824089, 0.49298330],
                             [- 6.27123404, -0.00109013, 0.00066836, 7.27014391, 0.99666793, 0.99824151, 0.49298351],
                             [- 6.27123629, -0.00108975, 0.00066812, 7.27014654, 0.99666910, 0.99824213, 0.49298363],
                             [- 6.27123852, -0.00108937, 0.00066789, 7.27014916, 0.99667027, 0.99824275, 0.49298382],
                             [- 6.27124075, -0.00108898, 0.00066765, 7.27015177, 0.99667144, 0.99824336, 0.49298401],
                             [- 6.27124298, -0.00108860, 0.00066742, 7.27015438, 0.99667261, 0.99824398, 0.49298420],
                             [- 6.27124520, -0.00108822, 0.00066718, 7.27015698, 0.99667379, 0.99824460, 0.49298438],
                             [- 6.27124741, -0.00108784, 0.00066695, 7.27015957, 0.99667495, 0.99824521, 0.49298453],
                             [- 6.27124962, -0.00108745, 0.00066672, 7.27016216, 0.99667612, 0.99824583, 0.49298471],
                             [- 6.27125182, -0.00108707, 0.00066648, 7.27016475, 0.99667729, 0.99824645, 0.49298491],
                             [- 6.27125401, -0.00108669, 0.00066625, 7.27016732, 0.99667846, 0.99824706, 0.49298510],
                             [- 6.27125620, -0.00108631, 0.00066601, 7.27016989, 0.99667962, 0.99824768, 0.49298524],
                             [- 6.27125839, -0.00108593, 0.00066578, 7.27017246, 0.99668079, 0.99824829, 0.49298544],
                             [- 6.27126057, -0.00108555, 0.00066555, 7.27017502, 0.99668195, 0.99824890, 0.49298562],
                             [- 6.27126274, -0.00108517, 0.00066531, 7.27017757, 0.99668312, 0.99824952, 0.49298575],
                             [- 6.27126490, -0.00108479, 0.00066508, 7.27018012, 0.99668428, 0.99825013, 0.49298596],
                             [- 6.27126707, -0.00108441, 0.00066485, 7.27018266, 0.99668544, 0.99825074, 0.49298614],
                             [- 6.27126922, -0.00108403, 0.00066461, 7.27018519, 0.99668660, 0.99825136, 0.49298629],
                             [- 6.27127137, -0.00108365, 0.00066438, 7.27018772, 0.99668776, 0.99825197, 0.49298647],
                             [- 6.27127351, -0.00108327, 0.00066415, 7.27019024, 0.99668892, 0.99825258, 0.49298665],
                             [- 6.27127565, -0.00108289, 0.00066392, 7.27019276, 0.99669008, 0.99825319, 0.49298685],
                             [- 6.27127779, -0.00108251, 0.00066369, 7.27019527, 0.99669124, 0.99825380, 0.49298701],
                             [- 6.27127991, -0.00108214, 0.00066345, 7.27019778, 0.99669240, 0.99825441, 0.49298719],
                             [- 6.27128204, -0.00108176, 0.00066322, 7.27020028, 0.99669356, 0.99825502, 0.49298735],
                             [- 6.27128415, -0.00108138, 0.00066299, 7.27020277, 0.99669471, 0.99825563, 0.49298754],
                             [- 6.27128626, -0.00108100, 0.00066276, 7.27020526, 0.99669587, 0.99825624, 0.49298772],
                             [- 6.27128837, -0.00108063, 0.00066253, 7.27020774, 0.99669702, 0.99825685, 0.49298787],
                             [- 6.27129047, -0.00108025, 0.00066230, 7.27021022, 0.99669817, 0.99825746, 0.49298806],
                             [- 6.27129256, -0.00107987, 0.00066206, 7.27021269, 0.99669933, 0.99825806, 0.49298822],
                             [- 6.27129465, -0.00107950, 0.00066183, 7.27021516, 0.99670048, 0.99825867, 0.49298838],
                             [- 6.27129674, -0.00107912, 0.00066160, 7.27021762, 0.99670163, 0.99825928, 0.49298857],
                             [- 6.27129881, -0.00107874, 0.00066137, 7.27022007, 0.99670278, 0.99825988, 0.49298872],
                             [- 6.27130089, -0.00107837, 0.00066114, 7.27022252, 0.99670393, 0.99826049, 0.49298890],
                             [- 6.27130295, -0.00107799, 0.00066091, 7.27022496, 0.99670508, 0.99826109, 0.49298906],
                             [- 6.27130502, -0.00107762, 0.00066068, 7.27022740, 0.99670622, 0.99826170, 0.49298921],
                             [- 6.27130707, -0.00107724, 0.00066045, 7.27022983, 0.99670737, 0.99826230, 0.49298942],
                             [- 6.27130913, -0.00107687, 0.00066022, 7.27023226, 0.99670852, 0.99826291, 0.49298958],
                             [- 6.27131117, -0.00107649, 0.00065999, 7.27023468, 0.99670966, 0.99826351, 0.49298977],
                             [- 6.27131321, -0.00107612, 0.00065976, 7.27023709, 0.99671081, 0.99826412, 0.49298992],
                             [- 6.27131525, -0.00107575, 0.00065953, 7.27023950, 0.99671195, 0.99826472, 0.49299011],
                             [- 6.27131728, -0.00107537, 0.00065931, 7.27024191, 0.99671309, 0.99826532, 0.49299028],
                             [- 6.27131930, -0.00107500, 0.00065908, 7.27024430, 0.99671423, 0.99826592, 0.49299042],
                             [- 6.27132133, -0.00107463, 0.00065885, 7.27024670, 0.99671538, 0.99826653, 0.49299058],
                             [- 6.27132334, -0.00107425, 0.00065862, 7.27024909, 0.99671652, 0.99826713, 0.49299077],
                             [- 6.27132535, -0.00107388, 0.00065839, 7.27025147, 0.99671766, 0.99826773, 0.49299096],
                             [- 6.27132736, -0.00107351, 0.00065816, 7.27025385, 0.99671879, 0.99826833, 0.49299111],
                             [- 6.27132936, -0.00107314, 0.00065793, 7.27025622, 0.99671993, 0.99826893, 0.49299126],
                             [- 6.27133135, -0.00107277, 0.00065771, 7.27025858, 0.99672107, 0.99826953, 0.49299144],
                             [- 6.27133334, -0.00107239, 0.00065748, 7.27026094, 0.99672221, 0.99827013, 0.49299160],
                             [- 6.27133532, -0.00107202, 0.00065725, 7.27026330, 0.99672334, 0.99827073, 0.49299180],
                             [- 6.27133730, -0.00107165, 0.00065702, 7.27026565, 0.99672448, 0.99827132, 0.49299193],
                             [- 6.27133928, -0.00107128, 0.00065680, 7.27026800, 0.99672561, 0.99827192, 0.49299212],
                             [- 6.27134125, -0.00107091, 0.00065657, 7.27027034, 0.99672674, 0.99827252, 0.49299227],
                             [- 6.27134321, -0.00107054, 0.00065634, 7.27027267, 0.99672788, 0.99827312, 0.49299243],
                             [- 6.27134517, -0.00107017, 0.00065611, 7.27027500, 0.99672901, 0.99827371, 0.49299259],
                             [- 6.27134713, -0.00106980, 0.00065589, 7.27027733, 0.99673014, 0.99827431, 0.49299280],
                             [- 6.27134908, -0.00106943, 0.00065566, 7.27027964, 0.99673127, 0.99827491, 0.49299296],
                             [- 6.27135102, -0.00106906, 0.00065543, 7.27028196, 0.99673240, 0.99827550, 0.49299309],
                             [- 6.27135296, -0.00106870, 0.00065521, 7.27028427, 0.99673353, 0.99827610, 0.49299327],
                             [- 6.27135490, -0.00106833, 0.00065498, 7.27028657, 0.99673466, 0.99827669, 0.49299344],
                             [- 6.27135683, -0.00106796, 0.00065476, 7.27028887, 0.99673578, 0.99827729, 0.49299361],
                             [- 6.27135875, -0.00106759, 0.00065453, 7.27029116, 0.99673691, 0.99827788, 0.49299378],
                             [- 6.27136067, -0.00106722, 0.00065430, 7.27029345, 0.99673804, 0.99827847, 0.49299395],
                             [- 6.27136259, -0.00106685, 0.00065408, 7.27029574, 0.99673916, 0.99827907, 0.49299413],
                             [- 6.27136450, -0.00106649, 0.00065385, 7.27029801, 0.99674028, 0.99827966, 0.49299427],
                             [- 6.27136641, -0.00106612, 0.00065363, 7.27030029, 0.99674141, 0.99828025, 0.49299442],
                             [- 6.27136831, -0.00106575, 0.00065340, 7.27030256, 0.99674253, 0.99828084, 0.49299460],
                             [- 6.27137021, -0.00106539, 0.00065318, 7.27030482, 0.99674365, 0.99828143, 0.49299474],
                             [- 6.27137210, -0.00106502, 0.00065295, 7.27030708, 0.99674477, 0.99828203, 0.49299490],
                             [- 6.27137399, -0.00106465, 0.00065273, 7.27030933, 0.99674589, 0.99828262, 0.49299509],
                             [- 6.27137587, -0.00106429, 0.00065250, 7.27031158, 0.99674701, 0.99828321, 0.49299525],
                             [- 6.27137775, -0.00106392, 0.00065228, 7.27031382, 0.99674813, 0.99828380, 0.49299541],
                             [- 6.27137962, -0.00106356, 0.00065206, 7.27031606, 0.99674925, 0.99828439, 0.49299561],
                             [- 6.27138149, -0.00106319, 0.00065183, 7.27031830, 0.99675037, 0.99828497, 0.49299573],
                             [- 6.27138335, -0.00106283, 0.00065161, 7.27032053, 0.99675148, 0.99828556, 0.49299589],
                             [- 6.27138521, -0.00106246, 0.00065139, 7.27032275, 0.99675260, 0.99828615, 0.49299603],
                             [- 6.27138707, -0.00106210, 0.00065116, 7.27032497, 0.99675371, 0.99828674, 0.49299621],
                             [- 6.27138892, -0.00106173, 0.00065094, 7.27032719, 0.99675483, 0.99828733, 0.49299635],
                             [- 6.27139077, -0.00106137, 0.00065072, 7.27032940, 0.99675594, 0.99828791, 0.49299653],
                             [- 6.27139261, -0.00106101, 0.00065049, 7.27033160, 0.99675705, 0.99828850, 0.49299670],
                             [- 6.27139445, -0.00106064, 0.00065027, 7.27033380, 0.99675816, 0.99828909, 0.49299687],
                             [- 6.27139628, -0.00106028, 0.00065005, 7.27033600, 0.99675928, 0.99828967, 0.49299705],
                             [- 6.27139811, -0.00105992, 0.00064982, 7.27033819, 0.99676039, 0.99829026, 0.49299718],
                             [- 6.27139993, -0.00105956, 0.00064960, 7.27034037, 0.99676150, 0.99829084, 0.49299736],
                             [- 6.27140175, -0.00105919, 0.00064938, 7.27034256, 0.99676260, 0.99829143, 0.49299751],
                             [- 6.27140356, -0.00105883, 0.00064916, 7.27034473, 0.99676371, 0.99829201, 0.49299765],
                             [- 6.27140537, -0.00105847, 0.00064893, 7.27034690, 0.99676482, 0.99829260, 0.49299782],
                             [- 6.27140718, -0.00105811, 0.00064871, 7.27034907, 0.99676593, 0.99829318, 0.49299800],
                             [- 6.27140898, -0.00105775, 0.00064849, 7.27035123, 0.99676703, 0.99829376, 0.49299816],
                             [- 6.27141078, -0.00105739, 0.00064827, 7.27035339, 0.99676814, 0.99829434, 0.49299833],
                             [- 6.27141257, -0.00105702, 0.00064805, 7.27035555, 0.99676924, 0.99829493, 0.49299845],
                             [- 6.27141436, -0.00105666, 0.00064783, 7.27035770, 0.99677034, 0.99829551, 0.49299862],
                             [- 6.27141614, -0.00105630, 0.00064761, 7.27035984, 0.99677145, 0.99829609, 0.49299878],
                             [- 6.27141792, -0.00105594, 0.00064739, 7.27036198, 0.99677255, 0.99829667, 0.49299891],
                             [- 6.27141970, -0.00105558, 0.00064716, 7.27036412, 0.99677365, 0.99829725, 0.49299912],
                             [- 6.27142147, -0.00105522, 0.00064694, 7.27036625, 0.99677475, 0.99829783, 0.49299926],
                             [- 6.27142324, -0.00105486, 0.00064672, 7.27036837, 0.99677585, 0.99829841, 0.49299943],
                             [- 6.27142500, -0.00105451, 0.00064650, 7.27037050, 0.99677695, 0.99829899, 0.49299955],
                             [- 6.27142676, -0.00105415, 0.00064628, 7.27037261, 0.99677805, 0.99829957, 0.49299971],
                             [- 6.27142851, -0.00105379, 0.00064606, 7.27037473, 0.99677914, 0.99830015, 0.49299984],
                             [- 6.27143026, -0.00105343, 0.00064584, 7.27037683, 0.99678024, 0.99830073, 0.49300001],
                             [- 6.27143201, -0.00105307, 0.00064562, 7.27037894, 0.99678134, 0.99830131, 0.49300017],
                             [- 6.27143375, -0.00105271, 0.00064540, 7.27038104, 0.99678243, 0.99830188, 0.49300033],
                             [- 6.27143549, -0.00105236, 0.00064518, 7.27038313, 0.99678353, 0.99830246, 0.49300050],
                             [- 6.27143722, -0.00105200, 0.00064496, 7.27038522, 0.99678462, 0.99830304, 0.49300065],
                             [- 6.27143895, -0.00105164, 0.00064475, 7.27038731, 0.99678571, 0.99830361, 0.49300081],
                             [- 6.27144068, -0.00105128, 0.00064453, 7.27038939, 0.99678681, 0.99830419, 0.49300096],
                             [- 6.27144240, -0.00105093, 0.00064431, 7.27039147, 0.99678790, 0.99830477, 0.49300112],
                             [- 6.27144412, -0.00105057, 0.00064409, 7.27039355, 0.99678899, 0.99830534, 0.49300126],
                             [- 6.27144583, -0.00105021, 0.00064387, 7.27039561, 0.99679008, 0.99830592, 0.49300141],
                             [- 6.27144754, -0.00104986, 0.00064365, 7.27039768, 0.99679117, 0.99830649, 0.49300158],
                             [- 6.27144924, -0.00104950, 0.00064343, 7.27039974, 0.99679226, 0.99830706, 0.49300174],
                             [- 6.27145094, -0.00104915, 0.00064322, 7.27040180, 0.99679334, 0.99830764, 0.49300191],
                             [- 6.27145264, -0.00104879, 0.00064300, 7.27040385, 0.99679443, 0.99830821, 0.49300203],
                             [- 6.27145433, -0.00104844, 0.00064278, 7.27040590, 0.99679552, 0.99830878, 0.49300220],
                             [- 6.27145602, -0.00104808, 0.00064256, 7.27040794, 0.99679660, 0.99830936, 0.49300234],
                             [- 6.27145771, -0.00104773, 0.00064234, 7.27040998, 0.99679769, 0.99830993, 0.49300248],
                             [- 6.27145939, -0.00104737, 0.00064213, 7.27041202, 0.99679877, 0.99831050, 0.49300264],
                             [- 6.27146107, -0.00104702, 0.00064191, 7.27041405, 0.99679986, 0.99831107, 0.49300280],
                             [- 6.27146274, -0.00104667, 0.00064169, 7.27041607, 0.99680094, 0.99831164, 0.49300294],
                             [- 6.27146441, -0.00104631, 0.00064148, 7.27041810, 0.99680202, 0.99831221, 0.49300310],
                             [- 6.27146607, -0.00104596, 0.00064126, 7.27042012, 0.99680310, 0.99831278, 0.49300325],
                             [- 6.27146774, -0.00104561, 0.00064104, 7.27042213, 0.99680418, 0.99831335, 0.49300340],
                             [- 6.27146939, -0.00104525, 0.00064083, 7.27042414, 0.99680526, 0.99831392, 0.49300357],
                             [- 6.27147105, -0.00104490, 0.00064061, 7.27042615, 0.99680634, 0.99831449, 0.49300370],
                             [- 6.27147270, -0.00104455, 0.00064039, 7.27042815, 0.99680742, 0.99831506, 0.49300390],
                             [- 6.27147434, -0.00104420, 0.00064018, 7.27043015, 0.99680850, 0.99831563, 0.49300402],
                             [- 6.27147599, -0.00104384, 0.00063996, 7.27043214, 0.99680957, 0.99831620, 0.49300417],
                             [- 6.27147762, -0.00104349, 0.00063975, 7.27043413, 0.99681065, 0.99831676, 0.49300431],
                             [- 6.27147926, -0.00104314, 0.00063953, 7.27043612, 0.99681172, 0.99831733, 0.49300449],
                             [- 6.27148089, -0.00104279, 0.00063931, 7.27043810, 0.99681280, 0.99831790, 0.49300460],
                             [- 6.27148252, -0.00104244, 0.00063910, 7.27044008, 0.99681387, 0.99831846, 0.49300478],
                             [- 6.27148414, -0.00104209, 0.00063888, 7.27044205, 0.99681495, 0.99831903, 0.49300494],
                             [- 6.27148576, -0.00104174, 0.00063867, 7.27044402, 0.99681602, 0.99831959, 0.49300507],
                             [- 6.27148738, -0.00104139, 0.00063845, 7.27044599, 0.99681709, 0.99832016, 0.49300523],
                             [- 6.27148899, -0.00104104, 0.00063824, 7.27044795, 0.99681816, 0.99832072, 0.49300537],
                             [- 6.27149060, -0.00104069, 0.00063802, 7.27044991, 0.99681923, 0.99832129, 0.49300552],
                             [- 6.27149220, -0.00104034, 0.00063781, 7.27045186, 0.99682030, 0.99832185, 0.49300567],
                             [- 6.27149380, -0.00103999, 0.00063760, 7.27045382, 0.99682137, 0.99832242, 0.49300583],
                             [- 6.27149540, -0.00103964, 0.00063738, 7.27045576, 0.99682244, 0.99832298, 0.49300595],
                             [- 6.27149699, -0.00103929, 0.00063717, 7.27045771, 0.99682351, 0.99832354, 0.49300609],
                             [- 6.27149859, -0.00103894, 0.00063695, 7.27045964, 0.99682457, 0.99832411, 0.49300627],
                             [- 6.27150017, -0.00103859, 0.00063674, 7.27046158, 0.99682564, 0.99832467, 0.49300642],
                             [- 6.27150175, -0.00103824, 0.00063653, 7.27046351, 0.99682670, 0.99832523, 0.49300653],
                             [- 6.27150333, -0.00103790, 0.00063631, 7.27046544, 0.99682777, 0.99832579, 0.49300671],
                             [- 6.27150491, -0.00103755, 0.00063610, 7.27046736, 0.99682883, 0.99832635, 0.49300686],
                             [- 6.27150648, -0.00103720, 0.00063589, 7.27046928, 0.99682990, 0.99832691, 0.49300700],
                             [- 6.27150805, -0.00103685, 0.00063567, 7.27047120, 0.99683096, 0.99832747, 0.49300717],
                             [- 6.27150962, -0.00103651, 0.00063546, 7.27047311, 0.99683202, 0.99832803, 0.49300731],
                             [- 6.27151118, -0.00103616, 0.00063525, 7.27047502, 0.99683308, 0.99832859, 0.49300742],
                             [- 6.27151274, -0.00103581, 0.00063503, 7.27047692, 0.99683414, 0.99832915, 0.49300759],
                             [- 6.27151429, -0.00103547, 0.00063482, 7.27047882, 0.99683520, 0.99832971, 0.49300775],
                             [- 6.27151584, -0.00103512, 0.00063461, 7.27048072, 0.99683626, 0.99833027, 0.49300787],
                             [- 6.27151739, -0.00103478, 0.00063440, 7.27048262, 0.99683732, 0.99833083, 0.49300803],
                             [- 6.27151893, -0.00103443, 0.00063419, 7.27048451, 0.99683838, 0.99833138, 0.49300818],
                             [- 6.27152048, -0.00103408, 0.00063397, 7.27048639, 0.99683943, 0.99833194, 0.49300831],
                             [- 6.27152201, -0.00103374, 0.00063376, 7.27048827, 0.99684049, 0.99833250, 0.49300846],
                             [- 6.27152355, -0.00103339, 0.00063355, 7.27049015, 0.99684154, 0.99833306, 0.49300863],
                             [- 6.27152508, -0.00103305, 0.00063334, 7.27049203, 0.99684260, 0.99833361, 0.49300877],
                             [- 6.27152660, -0.00103270, 0.00063313, 7.27049390, 0.99684365, 0.99833417, 0.49300891],
                             [- 6.27152813, -0.00103236, 0.00063292, 7.27049577, 0.99684471, 0.99833472, 0.49300907],
                             [- 6.27152965, -0.00103202, 0.00063271, 7.27049763, 0.99684576, 0.99833528, 0.49300922],
                             [- 6.27153117, -0.00103167, 0.00063249, 7.27049949, 0.99684681, 0.99833583, 0.49300935],
                             [- 6.27153268, -0.00103133, 0.00063228, 7.27050135, 0.99684786, 0.99833639, 0.49300949],
                             [- 6.27153419, -0.00103099, 0.00063207, 7.27050320, 0.99684891, 0.99833694, 0.49300966],
                             [- 6.27153570, -0.00103064, 0.00063186, 7.27050505, 0.99684996, 0.99833749, 0.49300980],
                             [- 6.27153720, -0.00103030, 0.00063165, 7.27050690, 0.99685101, 0.99833805, 0.49300992],
                             [- 6.27153870, -0.00102996, 0.00063144, 7.27050874, 0.99685206, 0.99833860, 0.49301003],
                             [- 6.27154020, -0.00102961, 0.00063123, 7.27051058, 0.99685311, 0.99833915, 0.49301024],
                             [- 6.27154169, -0.00102927, 0.00063102, 7.27051242, 0.99685415, 0.99833971, 0.49301036],
                             [- 6.27154318, -0.00102893, 0.00063081, 7.27051425, 0.99685520, 0.99834026, 0.49301051],
                             [- 6.27154467, -0.00102859, 0.00063060, 7.27051608, 0.99685625, 0.99834081, 0.49301064],
                             [- 6.27154615, -0.00102825, 0.00063039, 7.27051791, 0.99685729, 0.99834136, 0.49301076],
                             [- 6.27154763, -0.00102791, 0.00063018, 7.27051973, 0.99685834, 0.99834191, 0.49301089],
                             [- 6.27154911, -0.00102756, 0.00062997, 7.27052155, 0.99685938, 0.99834246, 0.49301105],
                             [- 6.27155059, -0.00102722, 0.00062976, 7.27052336, 0.99686042, 0.99834301, 0.49301119],
                             [- 6.27155206, -0.00102688, 0.00062956, 7.27052517, 0.99686146, 0.99834356, 0.49301134],
                             [- 6.27155352, -0.00102654, 0.00062935, 7.27052698, 0.99686251, 0.99834411, 0.49301148],
                             [- 6.27155499, -0.00102620, 0.00062914, 7.27052879, 0.99686355, 0.99834466, 0.49301163],
                             [- 6.27155645, -0.00102586, 0.00062893, 7.27053059, 0.99686459, 0.99834521, 0.49301176],
                             [- 6.27155791, -0.00102552, 0.00062872, 7.27053239, 0.99686563, 0.99834576, 0.49301188],
                             [- 6.27155936, -0.00102518, 0.00062851, 7.27053418, 0.99686666, 0.99834630, 0.49301206],
                             [- 6.27156082, -0.00102484, 0.00062830, 7.27053597, 0.99686770, 0.99834685, 0.49301220],
                             [- 6.27156226, -0.00102451, 0.00062810, 7.27053776, 0.99686874, 0.99834740, 0.49301233],
                             [- 6.27156371, -0.00102417, 0.00062789, 7.27053954, 0.99686978, 0.99834794, 0.49301246],
                             [- 6.27156515, -0.00102383, 0.00062768, 7.27054132, 0.99687081, 0.99834849, 0.49301262],
                             [- 6.27156659, -0.00102349, 0.00062747, 7.27054310, 0.99687185, 0.99834904, 0.49301280],
                             [- 6.27156803, -0.00102315, 0.00062727, 7.27054488, 0.99687288, 0.99834958, 0.49301294],
                             [- 6.27156946, -0.00102281, 0.00062706, 7.27054665, 0.99687392, 0.99835013, 0.49301309],
                             [- 6.27157089, -0.00102248, 0.00062685, 7.27054842, 0.99687495, 0.99835067, 0.49301319],
                             [- 6.27157232, -0.00102214, 0.00062664, 7.27055018, 0.99687598, 0.99835122, 0.49301335],
                             [- 6.27157374, -0.00102180, 0.00062644, 7.27055194, 0.99687701, 0.99835176, 0.49301345],
                             [- 6.27157517, -0.00102146, 0.00062623, 7.27055370, 0.99687805, 0.99835231, 0.49301362],
                             [- 6.27157658, -0.00102113, 0.00062602, 7.27055546, 0.99687908, 0.99835285, 0.49301372],
                             [- 6.27157800, -0.00102079, 0.00062582, 7.27055721, 0.99688011, 0.99835339, 0.49301383],
                             [- 6.27157941, -0.00102045, 0.00062561, 7.27055896, 0.99688113, 0.99835393, 0.49301405],
                             [- 6.27158082, -0.00102012, 0.00062540, 7.27056070, 0.99688216, 0.99835448, 0.49301418],
                             [- 6.27158223, -0.00101978, 0.00062520, 7.27056244, 0.99688319, 0.99835502, 0.49301433],
                             [- 6.27158363, -0.00101945, 0.00062499, 7.27056418, 0.99688422, 0.99835556, 0.49301444],
                             [- 6.27158503, -0.00101911, 0.00062479, 7.27056592, 0.99688524, 0.99835610, 0.49301454],
                             [- 6.27158643, -0.00101878, 0.00062458, 7.27056765, 0.99688627, 0.99835664, 0.49301468],
                             [- 6.27158782, -0.00101844, 0.00062438, 7.27056938, 0.99688730, 0.99835718, 0.49301483],
                             [- 6.27158921, -0.00101811, 0.00062417, 7.27057111, 0.99688832, 0.99835772, 0.49301499],
                             [- 6.27159060, -0.00101777, 0.00062396, 7.27057283, 0.99688934, 0.99835826, 0.49301513],
                             [- 6.27159199, -0.00101744, 0.00062376, 7.27057455, 0.99689037, 0.99835880, 0.49301528],
                             [- 6.27159337, -0.00101710, 0.00062355, 7.27057627, 0.99689139, 0.99835934, 0.49301541],
                             [- 6.27159475, -0.00101677, 0.00062335, 7.27057798, 0.99689241, 0.99835988, 0.49301553],
                             [- 6.27159613, -0.00101644, 0.00062314, 7.27057969, 0.99689343, 0.99836042, 0.49301569],
                             [- 6.27159750, -0.00101610, 0.00062294, 7.27058140, 0.99689445, 0.99836096, 0.49301584],
                             [- 6.27159887, -0.00101577, 0.00062274, 7.27058310, 0.99689547, 0.99836150, 0.49301593],
                             [- 6.27160024, -0.00101544, 0.00062253, 7.27058480, 0.99689649, 0.99836203, 0.49301608],
                             [- 6.27160160, -0.00101510, 0.00062233, 7.27058650, 0.99689751, 0.99836257, 0.49301621],
                             [- 6.27160297, -0.00101477, 0.00062212, 7.27058820, 0.99689853, 0.99836311, 0.49301634],
                             [- 6.27160433, -0.00101444, 0.00062192, 7.27058989, 0.99689954, 0.99836364, 0.49301647],
                             [- 6.27160568, -0.00101410, 0.00062171, 7.27059158, 0.99690056, 0.99836418, 0.49301663],
                             [- 6.27160704, -0.00101377, 0.00062151, 7.27059327, 0.99690158, 0.99836472, 0.49301679],
                             [- 6.27160839, -0.00101344, 0.00062131, 7.27059495, 0.99690259, 0.99836525, 0.49301693],
                             [- 6.27160974, -0.00101311, 0.00062110, 7.27059663, 0.99690361, 0.99836579, 0.49301706],
                             [- 6.27161108, -0.00101278, 0.00062090, 7.27059831, 0.99690462, 0.99836632, 0.49301722],
                             [- 6.27161243, -0.00101245, 0.00062070, 7.27059998, 0.99690563, 0.99836686, 0.49301730],
                             [- 6.27161377, -0.00101212, 0.00062049, 7.27060165, 0.99690664, 0.99836739, 0.49301748],
                             [- 6.27161511, -0.00101179, 0.00062029, 7.27060332, 0.99690766, 0.99836792, 0.49301757],
                             [- 6.27161644, -0.00101146, 0.00062009, 7.27060499, 0.99690867, 0.99836846, 0.49301772],
                             [- 6.27161777, -0.00101112, 0.00061989, 7.27060665, 0.99690968, 0.99836899, 0.49301787],
                             [- 6.27161910, -0.00101079, 0.00061968, 7.27060831, 0.99691069, 0.99836952, 0.49301798],
                             [- 6.27162043, -0.00101046, 0.00061948, 7.27060996, 0.99691170, 0.99837005, 0.49301811],
                             [- 6.27162175, -0.00101014, 0.00061928, 7.27061162, 0.99691271, 0.99837059, 0.49301825],
                             [- 6.27162307, -0.00100981, 0.00061908, 7.27061327, 0.99691371, 0.99837112, 0.49301840],
                             [- 6.27162439, -0.00100948, 0.00061887, 7.27061492, 0.99691472, 0.99837165, 0.49301855],
                             [- 6.27162571, -0.00100915, 0.00061867, 7.27061656, 0.99691573, 0.99837218, 0.49301866],
                             [- 6.27162702, -0.00100882, 0.00061847, 7.27061820, 0.99691673, 0.99837271, 0.49301881],
                             [- 6.27162833, -0.00100849, 0.00061827, 7.27061984, 0.99691774, 0.99837324, 0.49301892],
                             [- 6.27162964, -0.00100816, 0.00061807, 7.27062148, 0.99691874, 0.99837377, 0.49301906],
                             [- 6.27163095, -0.00100783, 0.00061787, 7.27062311, 0.99691975, 0.99837430, 0.49301923],
                             [- 6.27163225, -0.00100751, 0.00061767, 7.27062474, 0.99692075, 0.99837483, 0.49301935],
                             [- 6.27163355, -0.00100718, 0.00061746, 7.27062637, 0.99692175, 0.99837536, 0.49301944],
                             [- 6.27163485, -0.00100685, 0.00061726, 7.27062800, 0.99692275, 0.99837589, 0.49301961],
                             [- 6.27163614, -0.00100652, 0.00061706, 7.27062962, 0.99692375, 0.99837641, 0.49301974],
                             [- 6.27163743, -0.00100620, 0.00061686, 7.27063124, 0.99692476, 0.99837694, 0.49301990],
                             [- 6.27163872, -0.00100587, 0.00061666, 7.27063285, 0.99692576, 0.99837747, 0.49301996],
                             [- 6.27164001, -0.00100554, 0.00061646, 7.27063447, 0.99692675, 0.99837800, 0.49302013],
                             [- 6.27164130, -0.00100522, 0.00061626, 7.27063608, 0.99692775, 0.99837852, 0.49302024],
                             [- 6.27164258, -0.00100489, 0.00061606, 7.27063769, 0.99692875, 0.99837905, 0.49302039],
                             [- 6.27164386, -0.00100456, 0.00061586, 7.27063929, 0.99692975, 0.99837957, 0.49302050],
                             [- 6.27164513, -0.00100424, 0.00061566, 7.27064090, 0.99693075, 0.99838010, 0.49302064],
                             [- 6.27164641, -0.00100391, 0.00061546, 7.27064250, 0.99693174, 0.99838063, 0.49302080],
                             [- 6.27164768, -0.00100359, 0.00061526, 7.27064409, 0.99693274, 0.99838115, 0.49302092],
                             [- 6.27164895, -0.00100326, 0.00061506, 7.27064569, 0.99693373, 0.99838168, 0.49302106],
                             [- 6.27165022, -0.00100294, 0.00061486, 7.27064728, 0.99693473, 0.99838220, 0.49302117],
                             [- 6.27165148, -0.00100261, 0.00061466, 7.27064887, 0.99693572, 0.99838272, 0.49302135],
                             [- 6.27165274, -0.00100229, 0.00061446, 7.27065045, 0.99693671, 0.99838325, 0.49302147],
                             [- 6.27165400, -0.00100196, 0.00061427, 7.27065204, 0.99693770, 0.99838377, 0.49302161],
                             [- 6.27165526, -0.00100164, 0.00061407, 7.27065362, 0.99693870, 0.99838429, 0.49302174],
                             [- 6.27165651, -0.00100132, 0.00061387, 7.27065520, 0.99693969, 0.99838482, 0.49302182],
                             [- 6.27165777, -0.00100099, 0.00061367, 7.27065677, 0.99694068, 0.99838534, 0.49302200],
                             [- 6.27165902, -0.00100067, 0.00061347, 7.27065835, 0.99694167, 0.99838586, 0.49302214],
                             [- 6.27166026, -0.00100035, 0.00061327, 7.27065992, 0.99694266, 0.99838638, 0.49302227],
                             [- 6.27166151, -0.00100002, 0.00061307, 7.27066149, 0.99694364, 0.99838690, 0.49302237],
                             [- 6.27166275, -0.00099970, 0.00061288, 7.27066305, 0.99694463, 0.99838742, 0.49302253],
                             [- 6.27166399, -0.00099938, 0.00061268, 7.27066461, 0.99694562, 0.99838795, 0.49302264],
                             [- 6.27166523, -0.00099905, 0.00061248, 7.27066617, 0.99694660, 0.99838847, 0.49302277],
                             [- 6.27166646, -0.00099873, 0.00061228, 7.27066773, 0.99694759, 0.99838899, 0.49302290],
                             [- 6.27166770, -0.00099841, 0.00061208, 7.27066928, 0.99694858, 0.99838951, 0.49302301],
                             [- 6.27166893, -0.00099809, 0.00061189, 7.27067084, 0.99694956, 0.99839002, 0.49302315],
                             [- 6.27167015, -0.00099777, 0.00061169, 7.27067239, 0.99695054, 0.99839054, 0.49302331],
                             [- 6.27167138, -0.00099745, 0.00061149, 7.27067393, 0.99695153, 0.99839106, 0.49302343],
                             [- 6.27167260, -0.00099712, 0.00061130, 7.27067548, 0.99695251, 0.99839158, 0.49302354],
                             [- 6.27167382, -0.00099680, 0.00061110, 7.27067702, 0.99695349, 0.99839210, 0.49302367],
                             [- 6.27167504, -0.00099648, 0.00061090, 7.27067856, 0.99695447, 0.99839262, 0.49302378],
                             [- 6.27167626, -0.00099616, 0.00061070, 7.27068010, 0.99695545, 0.99839313, 0.49302391],
                             [- 6.27167747, -0.00099584, 0.00061051, 7.27068163, 0.99695643, 0.99839365, 0.49302405],
                             [- 6.27167868, -0.00099552, 0.00061031, 7.27068316, 0.99695741, 0.99839417, 0.49302418],
                             [- 6.27167989, -0.00099520, 0.00061012, 7.27068469, 0.99695839, 0.99839468, 0.49302428],
                             [- 6.27168110, -0.00099488, 0.00060992, 7.27068622, 0.99695937, 0.99839520, 0.49302442],
                             [- 6.27168231, -0.00099456, 0.00060972, 7.27068774, 0.99696035, 0.99839571, 0.49302457],
                             [- 6.27168351, -0.00099424, 0.00060953, 7.27068927, 0.99696133, 0.99839623, 0.49302469],
                             [- 6.27168471, -0.00099392, 0.00060933, 7.27069078, 0.99696230, 0.99839674, 0.49302483],
                             [- 6.27168591, -0.00099360, 0.00060914, 7.27069230, 0.99696328, 0.99839726, 0.49302493],
                             [- 6.27168710, -0.00099329, 0.00060894, 7.27069382, 0.99696425, 0.99839777, 0.49302510],
                             [- 6.27168829, -0.00099297, 0.00060874, 7.27069533, 0.99696523, 0.99839829, 0.49302519],
                             [- 6.27168949, -0.00099265, 0.00060855, 7.27069684, 0.99696620, 0.99839880, 0.49302531],
                             [- 6.27169067, -0.00099233, 0.00060835, 7.27069834, 0.99696717, 0.99839931, 0.49302546],
                             [- 6.27169186, -0.00099201, 0.00060816, 7.27069985, 0.99696815, 0.99839983, 0.49302559],
                             [- 6.27169305, -0.00099170, 0.00060796, 7.27070135, 0.99696912, 0.99840034, 0.49302572],
                             [- 6.27169423, -0.00099138, 0.00060777, 7.27070285, 0.99697009, 0.99840085, 0.49302581],
                             [- 6.27169541, -0.00099106, 0.00060757, 7.27070435, 0.99697106, 0.99840137, 0.49302594],
                             [- 6.27169659, -0.00099074, 0.00060738, 7.27070584, 0.99697203, 0.99840188, 0.49302606],
                             [- 6.27169776, -0.00099043, 0.00060719, 7.27070734, 0.99697300, 0.99840239, 0.49302620],
                             [- 6.27169893, -0.00099011, 0.00060699, 7.27070883, 0.99697397, 0.99840290, 0.49302635],
                             [- 6.27170011, -0.00098979, 0.00060680, 7.27071031, 0.99697494, 0.99840341, 0.49302644],
                             [- 6.27170128, -0.00098948, 0.00060660, 7.27071180, 0.99697590, 0.99840392, 0.49302662],
                             [- 6.27170244, -0.00098916, 0.00060641, 7.27071328, 0.99697687, 0.99840443, 0.49302676],
                             [- 6.27170361, -0.00098884, 0.00060622, 7.27071476, 0.99697784, 0.99840494, 0.49302682],
                             [- 6.27170477, -0.00098853, 0.00060602, 7.27071624, 0.99697880, 0.99840545, 0.49302696],
                             [- 6.27170593, -0.00098821, 0.00060583, 7.27071772, 0.99697977, 0.99840596, 0.49302711],
                             [- 6.27170709, -0.00098790, 0.00060563, 7.27071919, 0.99698073, 0.99840647, 0.49302721],
                             [- 6.27170825, -0.00098758, 0.00060544, 7.27072066, 0.99698170, 0.99840698, 0.49302733],
                             [- 6.27170940, -0.00098727, 0.00060525, 7.27072213, 0.99698266, 0.99840748, 0.49302748],
                             [- 6.27171055, -0.00098695, 0.00060505, 7.27072360, 0.99698362, 0.99840799, 0.49302765],
                             [- 6.27171170, -0.00098664, 0.00060486, 7.27072506, 0.99698459, 0.99840850, 0.49302773],
                             [- 6.27171285, -0.00098633, 0.00060467, 7.27072652, 0.99698555, 0.99840901, 0.49302786],
                             [- 6.27171399, -0.00098601, 0.00060448, 7.27072798, 0.99698651, 0.99840951, 0.49302797],
                             [- 6.27171514, -0.00098570, 0.00060428, 7.27072944, 0.99698747, 0.99841002, 0.49302811],
                             [- 6.27171628, -0.00098538, 0.00060409, 7.27073090, 0.99698843, 0.99841053, 0.49302824],
                             [- 6.27171742, -0.00098507, 0.00060390, 7.27073235, 0.99698939, 0.99841103, 0.49302835],
                             [- 6.27171856, -0.00098476, 0.00060371, 7.27073380, 0.99699035, 0.99841154, 0.49302848],
                             [- 6.27171969, -0.00098444, 0.00060351, 7.27073525, 0.99699130, 0.99841204, 0.49302858],
                             [- 6.27172082, -0.00098413, 0.00060332, 7.27073669, 0.99699226, 0.99841255, 0.49302871],
                             [- 6.27172196, -0.00098382, 0.00060313, 7.27073814, 0.99699322, 0.99841305, 0.49302887],
                             [- 6.27172309, -0.00098351, 0.00060294, 7.27073958, 0.99699417, 0.99841356, 0.49302895],
                             [- 6.27172421, -0.00098319, 0.00060275, 7.27074102, 0.99699513, 0.99841406, 0.49302912],
                             [- 6.27172534, -0.00098288, 0.00060256, 7.27074246, 0.99699608, 0.99841456, 0.49302925],
                             [- 6.27172646, -0.00098257, 0.00060236, 7.27074389, 0.99699704, 0.99841507, 0.49302935],
                             [- 6.27172758, -0.00098226, 0.00060217, 7.27074532, 0.99699799, 0.99841557, 0.49302945],
                             [- 6.27172870, -0.00098195, 0.00060198, 7.27074676, 0.99699894, 0.99841607, 0.49302959],
                             [- 6.27172982, -0.00098163, 0.00060179, 7.27074818, 0.99699990, 0.99841657, 0.49302970],
                             [- 6.27173093, -0.00098132, 0.00060160, 7.27074961, 0.99700085, 0.99841708, 0.49302981],
                             [- 6.27173205, -0.00098101, 0.00060141, 7.27075103, 0.99700180, 0.99841758, 0.49302992],
                             [- 6.27173316, -0.00098070, 0.00060122, 7.27075246, 0.99700275, 0.99841808, 0.49303009],
                             [- 6.27173427, -0.00098039, 0.00060103, 7.27075388, 0.99700370, 0.99841858, 0.49303019],
                             [- 6.27173537, -0.00098008, 0.00060084, 7.27075529, 0.99700465, 0.99841908, 0.49303031],
                             [- 6.27173648, -0.00097977, 0.00060065, 7.27075671, 0.99700560, 0.99841958, 0.49303044],
                             [- 6.27173758, -0.00097946, 0.00060046, 7.27075812, 0.99700655, 0.99842008, 0.49303056],
                             [- 6.27173868, -0.00097915, 0.00060027, 7.27075953, 0.99700749, 0.99842058, 0.49303069],
                             [- 6.27173978, -0.00097884, 0.00060008, 7.27076094, 0.99700844, 0.99842108, 0.49303080],
                             [- 6.27174088, -0.00097853, 0.00059989, 7.27076235, 0.99700939, 0.99842158, 0.49303094],
                             [- 6.27174198, -0.00097822, 0.00059970, 7.27076375, 0.99701033, 0.99842208, 0.49303107],
                             [- 6.27174307, -0.00097791, 0.00059951, 7.27076516, 0.99701128, 0.99842258, 0.49303120],
                             [- 6.27174416, -0.00097761, 0.00059932, 7.27076656, 0.99701222, 0.99842308, 0.49303132],
                             [- 6.27174525, -0.00097730, 0.00059913, 7.27076795, 0.99701317, 0.99842357, 0.49303142],
                             [- 6.27174634, -0.00097699, 0.00059894, 7.27076935, 0.99701411, 0.99842407, 0.49303154],
                             [- 6.27174743, -0.00097668, 0.00059875, 7.27077075, 0.99701505, 0.99842457, 0.49303168],
                             [- 6.27174851, -0.00097637, 0.00059856, 7.27077214, 0.99701599, 0.99842507, 0.49303176],
                             [- 6.27174959, -0.00097606, 0.00059837, 7.27077353, 0.99701693, 0.99842556, 0.49303192],
                             [- 6.27175067, -0.00097576, 0.00059818, 7.27077491, 0.99701788, 0.99842606, 0.49303205],
                             [- 6.27175175, -0.00097545, 0.00059800, 7.27077630, 0.99701882, 0.99842656, 0.49303212],
                             [- 6.27175283, -0.00097514, 0.00059781, 7.27077769, 0.99701976, 0.99842705, 0.49303223],
                             [- 6.27175390, -0.00097484, 0.00059762, 7.27077907, 0.99702070, 0.99842755, 0.49303239],
                             [- 6.27175497, -0.00097453, 0.00059743, 7.27078045, 0.99702163, 0.99842804, 0.49303247],
                             [- 6.27175605, -0.00097422, 0.00059724, 7.27078182, 0.99702257, 0.99842854, 0.49303263],
                             [- 6.27175712, -0.00097392, 0.00059705, 7.27078320, 0.99702351, 0.99842903, 0.49303277],
                             [- 6.27175818, -0.00097361, 0.00059687, 7.27078457, 0.99702445, 0.99842952, 0.49303287],
                             [- 6.27175925, -0.00097330, 0.00059668, 7.27078595, 0.99702538, 0.99843002, 0.49303298],
                             [- 6.27176031, -0.00097300, 0.00059649, 7.27078731, 0.99702632, 0.99843051, 0.49303309],
                             [- 6.27176137, -0.00097269, 0.00059630, 7.27078868, 0.99702725, 0.99843101, 0.49303324],
                             [- 6.27176243, -0.00097239, 0.00059612, 7.27079005, 0.99702819, 0.99843150, 0.49303337],
                             [- 6.27176349, -0.00097208, 0.00059593, 7.27079141, 0.99702912, 0.99843199, 0.49303342],
                             [- 6.27176455, -0.00097178, 0.00059574, 7.27079277, 0.99703005, 0.99843248, 0.49303355],
                             [- 6.27176560, -0.00097147, 0.00059555, 7.27079413, 0.99703099, 0.99843298, 0.49303367],
                             [- 6.27176666, -0.00097117, 0.00059537, 7.27079549, 0.99703192, 0.99843347, 0.49303384],
                             [- 6.27176771, -0.00097086, 0.00059518, 7.27079685, 0.99703285, 0.99843396, 0.49303391],
                             [- 6.27176876, -0.00097056, 0.00059499, 7.27079820, 0.99703378, 0.99843445, 0.49303404],
                             [- 6.27176980, -0.00097025, 0.00059481, 7.27079955, 0.99703471, 0.99843494, 0.49303417],
                             [- 6.27177085, -0.00096995, 0.00059462, 7.27080090, 0.99703564, 0.99843543, 0.49303426],
                             [- 6.27177189, -0.00096965, 0.00059443, 7.27080225, 0.99703657, 0.99843592, 0.49303442],
                             [- 6.27177294, -0.00096934, 0.00059425, 7.27080359, 0.99703750, 0.99843641, 0.49303454],
                             [- 6.27177398, -0.00096904, 0.00059406, 7.27080494, 0.99703843, 0.99843690, 0.49303464],
                             [- 6.27177502, -0.00096874, 0.00059388, 7.27080628, 0.99703936, 0.99843739, 0.49303476],
                             [- 6.27177605, -0.00096843, 0.00059369, 7.27080762, 0.99704028, 0.99843788, 0.49303487],
                             [- 6.27177709, -0.00096813, 0.00059350, 7.27080896, 0.99704121, 0.99843837, 0.49303501],
                             [- 6.27177812, -0.00096783, 0.00059332, 7.27081030, 0.99704213, 0.99843886, 0.49303512],
                             [- 6.27177915, -0.00096752, 0.00059313, 7.27081163, 0.99704306, 0.99843934, 0.49303521],
                             [- 6.27178018, -0.00096722, 0.00059295, 7.27081296, 0.99704398, 0.99843983, 0.49303537],
                             [- 6.27178121, -0.00096692, 0.00059276, 7.27081429, 0.99704491, 0.99844032, 0.49303546],
                             [- 6.27178224, -0.00096662, 0.00059258, 7.27081562, 0.99704583, 0.99844081, 0.49303558],
                             [- 6.27178326, -0.00096632, 0.00059239, 7.27081695, 0.99704675, 0.99844129, 0.49303575],
                             [- 6.27178429, -0.00096602, 0.00059221, 7.27081827, 0.99704768, 0.99844178, 0.49303581],
                             [- 6.27178531, -0.00096571, 0.00059202, 7.27081960, 0.99704860, 0.99844227, 0.49303591],
                             [- 6.27178633, -0.00096541, 0.00059184, 7.27082092, 0.99704952, 0.99844275, 0.49303609],
                             [- 6.27178735, -0.00096511, 0.00059165, 7.27082224, 0.99705044, 0.99844324, 0.49303618],
                             [- 6.27178836, -0.00096481, 0.00059147, 7.27082355, 0.99705136, 0.99844372, 0.49303628],
                             [- 6.27178938, -0.00096451, 0.00059128, 7.27082487, 0.99705228, 0.99844421, 0.49303641],
                             [- 6.27179039, -0.00096421, 0.00059110, 7.27082618, 0.99705320, 0.99844469, 0.49303652],
                             [- 6.27179140, -0.00096391, 0.00059091, 7.27082749, 0.99705412, 0.99844518, 0.49303666],
                             [- 6.27179242, -0.00096361, 0.00059073, 7.27082881, 0.99705503, 0.99844566, 0.49303673],
                             [- 6.27179342, -0.00096331, 0.00059055, 7.27083011, 0.99705595, 0.99844614, 0.49303690],
                             [- 6.27179443, -0.00096301, 0.00059036, 7.27083142, 0.99705687, 0.99844663, 0.49303700],
                             [- 6.27179544, -0.00096271, 0.00059018, 7.27083272, 0.99705778, 0.99844711, 0.49303712],
                             [- 6.27179644, -0.00096241, 0.00058999, 7.27083403, 0.99705870, 0.99844759, 0.49303722],
                             [- 6.27179744, -0.00096211, 0.00058981, 7.27083533, 0.99705961, 0.99844808, 0.49303732],
                             [- 6.27179844, -0.00096181, 0.00058963, 7.27083663, 0.99706053, 0.99844856, 0.49303745],
                             [- 6.27179944, -0.00096152, 0.00058944, 7.27083792, 0.99706144, 0.99844904, 0.49303757],
                             [- 6.27180044, -0.00096122, 0.00058926, 7.27083922, 0.99706235, 0.99844952, 0.49303771],
                             [- 6.27180143, -0.00096092, 0.00058908, 7.27084051, 0.99706327, 0.99845000, 0.49303777],
                             [- 6.27180243, -0.00096062, 0.00058890, 7.27084181, 0.99706418, 0.99845048, 0.49303795],
                             [- 6.27180342, -0.00096032, 0.00058871, 7.27084310, 0.99706509, 0.99845097, 0.49303802],
                             [- 6.27180441, -0.00096002, 0.00058853, 7.27084438, 0.99706600, 0.99845145, 0.49303815],
                             [- 6.27180540, -0.00095973, 0.00058835, 7.27084567, 0.99706691, 0.99845193, 0.49303827],
                             [- 6.27180639, -0.00095943, 0.00058816, 7.27084696, 0.99706782, 0.99845241, 0.49303836],
                             [- 6.27180737, -0.00095913, 0.00058798, 7.27084824, 0.99706873, 0.99845289, 0.49303844],
                             [- 6.27180836, -0.00095884, 0.00058780, 7.27084952, 0.99706964, 0.99845336, 0.49303861],
                             [- 6.27180934, -0.00095854, 0.00058762, 7.27085080, 0.99707055, 0.99845384, 0.49303868],
                             [- 6.27181032, -0.00095824, 0.00058744, 7.27085208, 0.99707145, 0.99845432, 0.49303884],
                             [- 6.27181130, -0.00095795, 0.00058725, 7.27085335, 0.99707236, 0.99845480, 0.49303895],
                             [- 6.27181228, -0.00095765, 0.00058707, 7.27085463, 0.99707327, 0.99845528, 0.49303908],
                             [- 6.27181325, -0.00095735, 0.00058689, 7.27085590, 0.99707417, 0.99845576, 0.49303915],
                             [- 6.27181423, -0.00095706, 0.00058671, 7.27085717, 0.99707508, 0.99845623, 0.49303928],
                             [- 6.27181520, -0.00095676, 0.00058653, 7.27085844, 0.99707598, 0.99845671, 0.49303940],
                             [- 6.27181618, -0.00095647, 0.00058635, 7.27085971, 0.99707689, 0.99845719, 0.49303950],
                             [- 6.27181715, -0.00095617, 0.00058616, 7.27086098, 0.99707779, 0.99845766, 0.49303965],
                             [- 6.27181811, -0.00095588, 0.00058598, 7.27086224, 0.99707869, 0.99845814, 0.49303979],
                             [- 6.27181908, -0.00095558, 0.00058580, 7.27086350, 0.99707960, 0.99845862, 0.49303989],
                             [- 6.27182005, -0.00095529, 0.00058562, 7.27086476, 0.99708050, 0.99845909, 0.49303994],
                             [- 6.27182101, -0.00095499, 0.00058544, 7.27086602, 0.99708140, 0.99845957, 0.49304006],
                             [- 6.27182198, -0.00095470, 0.00058526, 7.27086728, 0.99708230, 0.99846004, 0.49304023],
                             [- 6.27182294, -0.00095440, 0.00058508, 7.27086854, 0.99708320, 0.99846052, 0.49304031],
                             [- 6.27182390, -0.00095411, 0.00058490, 7.27086979, 0.99708410, 0.99846099, 0.49304040],
                             [- 6.27182486, -0.00095381, 0.00058472, 7.27087104, 0.99708500, 0.99846147, 0.49304052],
                             [- 6.27182581, -0.00095352, 0.00058454, 7.27087229, 0.99708590, 0.99846194, 0.49304066],
                             [- 6.27182677, -0.00095323, 0.00058436, 7.27087354, 0.99708680, 0.99846242, 0.49304078],
                             [- 6.27182772, -0.00095293, 0.00058418, 7.27087479, 0.99708769, 0.99846289, 0.49304090],
                             [- 6.27182867, -0.00095264, 0.00058400, 7.27087603, 0.99708859, 0.99846336, 0.49304102],
                             [- 6.27182963, -0.00095235, 0.00058382, 7.27087728, 0.99708949, 0.99846383, 0.49304112],
                             [- 6.27183058, -0.00095205, 0.00058364, 7.27087852, 0.99709038, 0.99846431, 0.49304120],
                             [- 6.27183152, -0.00095176, 0.00058346, 7.27087976, 0.99709128, 0.99846478, 0.49304129],
                             [- 6.27183247, -0.00095147, 0.00058328, 7.27088100, 0.99709217, 0.99846525, 0.49304145],
                             [- 6.27183342, -0.00095118, 0.00058310, 7.27088224, 0.99709307, 0.99846572, 0.49304157],
                             [- 6.27183436, -0.00095088, 0.00058292, 7.27088348, 0.99709396, 0.99846619, 0.49304165],
                             [- 6.27183530, -0.00095059, 0.00058274, 7.27088471, 0.99709485, 0.99846667, 0.49304175],
                             [- 6.27183624, -0.00095030, 0.00058256, 7.27088594, 0.99709575, 0.99846714, 0.49304190],
                             [- 6.27183718, -0.00095001, 0.00058238, 7.27088717, 0.99709664, 0.99846761, 0.49304204],
                             [- 6.27183812, -0.00094972, 0.00058221, 7.27088840, 0.99709753, 0.99846808, 0.49304212],
                             [- 6.27183906, -0.00094943, 0.00058203, 7.27088963, 0.99709842, 0.99846855, 0.49304222],
                             [- 6.27183999, -0.00094914, 0.00058185, 7.27089086, 0.99709931, 0.99846902, 0.49304233],
                             [- 6.27184093, -0.00094884, 0.00058167, 7.27089208, 0.99710020, 0.99846949, 0.49304245],
                             [- 6.27184186, -0.00094855, 0.00058149, 7.27089331, 0.99710109, 0.99846996, 0.49304255],
                             [- 6.27184279, -0.00094826, 0.00058131, 7.27089453, 0.99710198, 0.99847042, 0.49304263],
                             [- 6.27184372, -0.00094797, 0.00058113, 7.27089575, 0.99710287, 0.99847089, 0.49304277],
                             [- 6.27184465, -0.00094768, 0.00058096, 7.27089697, 0.99710375, 0.99847136, 0.49304288],
                             [- 6.27184558, -0.00094739, 0.00058078, 7.27089818, 0.99710464, 0.99847183, 0.49304302],
                             [- 6.27184650, -0.00094710, 0.00058060, 7.27089940, 0.99710553, 0.99847230, 0.49304309],
                             [- 6.27184743, -0.00094681, 0.00058042, 7.27090061, 0.99710641, 0.99847276, 0.49304322],
                             [- 6.27184835, -0.00094652, 0.00058025, 7.27090182, 0.99710730, 0.99847323, 0.49304332],
                             [- 6.27184927, -0.00094623, 0.00058007, 7.27090304, 0.99710818, 0.99847370, 0.49304348],
                             [- 6.27185019, -0.00094595, 0.00057989, 7.27090424, 0.99710907, 0.99847416, 0.49304351],
                             [- 6.27185111, -0.00094566, 0.00057971, 7.27090545, 0.99710995, 0.99847463, 0.49304363],
                             [- 6.27185203, -0.00094537, 0.00057954, 7.27090666, 0.99711083, 0.99847510, 0.49304378],
                             [- 6.27185294, -0.00094508, 0.00057936, 7.27090786, 0.99711172, 0.99847556, 0.49304390],
                             [- 6.27185386, -0.00094479, 0.00057918, 7.27090907, 0.99711260, 0.99847603, 0.49304399],
                             [- 6.27185477, -0.00094450, 0.00057901, 7.27091027, 0.99711348, 0.99847649, 0.49304412],
                             [- 6.27185568, -0.00094421, 0.00057883, 7.27091147, 0.99711436, 0.99847696, 0.49304419],
                             [- 6.27185659, -0.00094393, 0.00057865, 7.27091267, 0.99711524, 0.99847742, 0.49304433],
                             [- 6.27185750, -0.00094364, 0.00057848, 7.27091386, 0.99711612, 0.99847789, 0.49304445],
                             [- 6.27185841, -0.00094335, 0.00057830, 7.27091506, 0.99711700, 0.99847835, 0.49304453],
                             [- 6.27185932, -0.00094306, 0.00057812, 7.27091625, 0.99711788, 0.99847881, 0.49304463],
                             [- 6.27186022, -0.00094278, 0.00057795, 7.27091745, 0.99711876, 0.99847928, 0.49304471],
                             [- 6.27186113, -0.00094249, 0.00057777, 7.27091864, 0.99711964, 0.99847974, 0.49304488],
                             [- 6.27186203, -0.00094220, 0.00057759, 7.27091983, 0.99712051, 0.99848020, 0.49304496],
                             [- 6.27186293, -0.00094192, 0.00057742, 7.27092101, 0.99712139, 0.99848067, 0.49304509],
                             [- 6.27186383, -0.00094163, 0.00057724, 7.27092220, 0.99712227, 0.99848113, 0.49304521],
                             [- 6.27186473, -0.00094134, 0.00057707, 7.27092339, 0.99712314, 0.99848159, 0.49304528],
                             [- 6.27186563, -0.00094106, 0.00057689, 7.27092457, 0.99712402, 0.99848205, 0.49304542],
                             [- 6.27186652, -0.00094077, 0.00057672, 7.27092575, 0.99712489, 0.99848251, 0.49304552],
                             [- 6.27186742, -0.00094049, 0.00057654, 7.27092693, 0.99712577, 0.99848297, 0.49304562],
                             [- 6.27186831, -0.00094020, 0.00057636, 7.27092811, 0.99712664, 0.99848344, 0.49304571],
                             [- 6.27186920, -0.00093991, 0.00057619, 7.27092929, 0.99712751, 0.99848390, 0.49304585],
                             [- 6.27187010, -0.00093963, 0.00057601, 7.27093047, 0.99712839, 0.99848436, 0.49304595],
                             [- 6.27187098, -0.00093934, 0.00057584, 7.27093164, 0.99712926, 0.99848482, 0.49304606],
                             [- 6.27187187, -0.00093906, 0.00057567, 7.27093281, 0.99713013, 0.99848528, 0.49304619],
                             [- 6.27187276, -0.00093877, 0.00057549, 7.27093399, 0.99713100, 0.99848574, 0.49304625],
                             [- 6.27187365, -0.00093849, 0.00057532, 7.27093516, 0.99713187, 0.99848619, 0.49304638],
                             [- 6.27187453, -0.00093821, 0.00057514, 7.27093633, 0.99713274, 0.99848665, 0.49304650],
                             [- 6.27187541, -0.00093792, 0.00057497, 7.27093749, 0.99713361, 0.99848711, 0.49304662],
                             [- 6.27187630, -0.00093764, 0.00057479, 7.27093866, 0.99713448, 0.99848757, 0.49304675],
                             [- 6.27187718, -0.00093735, 0.00057462, 7.27093983, 0.99713535, 0.99848803, 0.49304680],
                             [- 6.27187806, -0.00093707, 0.00057444, 7.27094099, 0.99713622, 0.99848849, 0.49304696],
                             [- 6.27187894, -0.00093679, 0.00057427, 7.27094215, 0.99713708, 0.99848894, 0.49304701],
                             [- 6.27187981, -0.00093650, 0.00057410, 7.27094331, 0.99713795, 0.99848940, 0.49304717],
                             [- 6.27188069, -0.00093622, 0.00057392, 7.27094447, 0.99713882, 0.99848986, 0.49304726],
                             [- 6.27188156, -0.00093594, 0.00057375, 7.27094563, 0.99713968, 0.99849031, 0.49304734],
                             [- 6.27188244, -0.00093565, 0.00057358, 7.27094678, 0.99714055, 0.99849077, 0.49304749],
                             [- 6.27188331, -0.00093537, 0.00057340, 7.27094794, 0.99714141, 0.99849123, 0.49304756],
                             [- 6.27188418, -0.00093509, 0.00057323, 7.27094909, 0.99714228, 0.99849168, 0.49304769],
                             [- 6.27188505, -0.00093481, 0.00057306, 7.27095025, 0.99714314, 0.99849214, 0.49304781],
                             [- 6.27188592, -0.00093452, 0.00057288, 7.27095140, 0.99714400, 0.99849259, 0.49304789],
                             [- 6.27188679, -0.00093424, 0.00057271, 7.27095255, 0.99714486, 0.99849305, 0.49304797],
                             [- 6.27188765, -0.00093396, 0.00057254, 7.27095369, 0.99714573, 0.99849350, 0.49304811],
                             [- 6.27188852, -0.00093368, 0.00057236, 7.27095484, 0.99714659, 0.99849396, 0.49304819],
                             [- 6.27188938, -0.00093340, 0.00057219, 7.27095599, 0.99714745, 0.99849441, 0.49304832],
                             [- 6.27189025, -0.00093312, 0.00057202, 7.27095713, 0.99714831, 0.99849487, 0.49304841],
                             [- 6.27189111, -0.00093283, 0.00057185, 7.27095827, 0.99714917, 0.99849532, 0.49304852],
                             [- 6.27189197, -0.00093255, 0.00057167, 7.27095942, 0.99715003, 0.99849577, 0.49304861],
                             [- 6.27189283, -0.00093227, 0.00057150, 7.27096056, 0.99715089, 0.99849623, 0.49304875],
                             [- 6.27189369, -0.00093199, 0.00057133, 7.27096169, 0.99715175, 0.99849668, 0.49304884],
                             [- 6.27189454, -0.00093171, 0.00057116, 7.27096283, 0.99715260, 0.99849713, 0.49304894],
                             [- 6.27189540, -0.00093143, 0.00057098, 7.27096397, 0.99715346, 0.99849758, 0.49304909],
                             [- 6.27189625, -0.00093115, 0.00057081, 7.27096510, 0.99715432, 0.99849804, 0.49304918],
                             [- 6.27189711, -0.00093087, 0.00057064, 7.27096624, 0.99715517, 0.99849849, 0.49304926],
                             [- 6.27189796, -0.00093059, 0.00057047, 7.27096737, 0.99715603, 0.99849894, 0.49304941],
                             [- 6.27189881, -0.00093031, 0.00057030, 7.27096850, 0.99715689, 0.99849939, 0.49304950],
                             [- 6.27189966, -0.00093003, 0.00057013, 7.27096963, 0.99715774, 0.99849984, 0.49304958],
                             [- 6.27190051, -0.00092975, 0.00056995, 7.27097076, 0.99715860, 0.99850029, 0.49304964],
                             [- 6.27190136, -0.00092947, 0.00056978, 7.27097188, 0.99715945, 0.99850074, 0.49304982],
                             [- 6.27190220, -0.00092919, 0.00056961, 7.27097301, 0.99716030, 0.99850119, 0.49304989],
                             [- 6.27190305, -0.00092892, 0.00056944, 7.27097413, 0.99716115, 0.99850164, 0.49305004],
                             [- 6.27190389, -0.00092864, 0.00056927, 7.27097525, 0.99716201, 0.99850209, 0.49305011],
                             [- 6.27190474, -0.00092836, 0.00056910, 7.27097638, 0.99716286, 0.99850254, 0.49305025],
                             [- 6.27190558, -0.00092808, 0.00056893, 7.27097750, 0.99716371, 0.99850299, 0.49305033],
                             [- 6.27190642, -0.00092780, 0.00056876, 7.27097862, 0.99716456, 0.99850344, 0.49305045],
                             [- 6.27190726, -0.00092752, 0.00056859, 7.27097973, 0.99716541, 0.99850389, 0.49305056],
                             [- 6.27190810, -0.00092725, 0.00056842, 7.27098085, 0.99716626, 0.99850434, 0.49305065],
                             [- 6.27190893, -0.00092697, 0.00056825, 7.27098197, 0.99716711, 0.99850478, 0.49305076],
                             [- 6.27190977, -0.00092669, 0.00056808, 7.27098308, 0.99716796, 0.99850523, 0.49305085],
                             [- 6.27191061, -0.00092641, 0.00056791, 7.27098419, 0.99716881, 0.99850568, 0.49305094],
                             [- 6.27191144, -0.00092614, 0.00056774, 7.27098530, 0.99716965, 0.99850613, 0.49305108],
                             [- 6.27191227, -0.00092586, 0.00056757, 7.27098641, 0.99717050, 0.99850657, 0.49305115],
                             [- 6.27191311, -0.00092558, 0.00056740, 7.27098752, 0.99717135, 0.99850702, 0.49305125],
                             [- 6.27191394, -0.00092531, 0.00056723, 7.27098863, 0.99717219, 0.99850747, 0.49305139],
                             [- 6.27191477, -0.00092503, 0.00056706, 7.27098974, 0.99717304, 0.99850791, 0.49305148],
                             [- 6.27191559, -0.00092475, 0.00056689, 7.27099084, 0.99717389, 0.99850836, 0.49305158],
                             [- 6.27191642, -0.00092448, 0.00056672, 7.27099194, 0.99717473, 0.99850880, 0.49305167],
                             [- 6.27191725, -0.00092420, 0.00056655, 7.27099305, 0.99717557, 0.99850925, 0.49305181],
                             [- 6.27191807, -0.00092393, 0.00056638, 7.27099415, 0.99717642, 0.99850969, 0.49305190],
                             [- 6.27191890, -0.00092365, 0.00056621, 7.27099525, 0.99717726, 0.99851014, 0.49305201],
                             [- 6.27191972, -0.00092338, 0.00056604, 7.27099635, 0.99717810, 0.99851058, 0.49305208],
                             [- 6.27192054, -0.00092310, 0.00056587, 7.27099744, 0.99717895, 0.99851103, 0.49305223],
                             [- 6.27192136, -0.00092282, 0.00056570, 7.27099854, 0.99717979, 0.99851147, 0.49305232],
                             [- 6.27192218, -0.00092255, 0.00056553, 7.27099964, 0.99718063, 0.99851192, 0.49305243],
                             [- 6.27192300, -0.00092227, 0.00056537, 7.27100073, 0.99718147, 0.99851236, 0.49305250],
                             [- 6.27192382, -0.00092200, 0.00056520, 7.27100182, 0.99718231, 0.99851280, 0.49305258],
                             [- 6.27192464, -0.00092173, 0.00056503, 7.27100291, 0.99718315, 0.99851325, 0.49305275],
                             [- 6.27192545, -0.00092145, 0.00056486, 7.27100400, 0.99718399, 0.99851369, 0.49305278],
                             [- 6.27192627, -0.00092118, 0.00056469, 7.27100509, 0.99718483, 0.99851413, 0.49305295],
                             [- 6.27192708, -0.00092090, 0.00056452, 7.27100618, 0.99718566, 0.99851457, 0.49305299],
                             [- 6.27192790, -0.00092063, 0.00056436, 7.27100727, 0.99718650, 0.99851502, 0.49305314],
                             [- 6.27192871, -0.00092036, 0.00056419, 7.27100835, 0.99718734, 0.99851546, 0.49305323],
                             [- 6.27192952, -0.00092008, 0.00056402, 7.27100944, 0.99718818, 0.99851590, 0.49305333],
                             [- 6.27193033, -0.00091981, 0.00056385, 7.27101052, 0.99718901, 0.99851634, 0.49305345],
                             [- 6.27193114, -0.00091954, 0.00056368, 7.27101160, 0.99718985, 0.99851678, 0.49305349],
                             [- 6.27193194, -0.00091926, 0.00056352, 7.27101268, 0.99719068, 0.99851722, 0.49305365],
                             [- 6.27193275, -0.00091899, 0.00056335, 7.27101376, 0.99719152, 0.99851766, 0.49305371],
                             [- 6.27193356, -0.00091872, 0.00056318, 7.27101484, 0.99719235, 0.99851810, 0.49305382],
                             [- 6.27193436, -0.00091844, 0.00056301, 7.27101592, 0.99719318, 0.99851854, 0.49305392],
                             [- 6.27193516, -0.00091817, 0.00056285, 7.27101699, 0.99719402, 0.99851898, 0.49305400],
                             [- 6.27193597, -0.00091790, 0.00056268, 7.27101807, 0.99719485, 0.99851942, 0.49305415],
                             [- 6.27193677, -0.00091763, 0.00056251, 7.27101914, 0.99719568, 0.99851986, 0.49305427],
                             [- 6.27193757, -0.00091736, 0.00056235, 7.27102021, 0.99719651, 0.99852030, 0.49305437],
                             [- 6.27193837, -0.00091708, 0.00056218, 7.27102128, 0.99719735, 0.99852074, 0.49305443],
                             [- 6.27193917, -0.00091681, 0.00056201, 7.27102235, 0.99719818, 0.99852117, 0.49305458],
                             [- 6.27193996, -0.00091654, 0.00056185, 7.27102342, 0.99719901, 0.99852161, 0.49305464],
                             [- 6.27194076, -0.00091627, 0.00056168, 7.27102449, 0.99719984, 0.99852205, 0.49305472],
                             [- 6.27194155, -0.00091600, 0.00056151, 7.27102556, 0.99720066, 0.99852249, 0.49305485],
                             [- 6.27194235, -0.00091573, 0.00056135, 7.27102662, 0.99720149, 0.99852292, 0.49305488],
                             [- 6.27194314, -0.00091546, 0.00056118, 7.27102769, 0.99720232, 0.99852336, 0.49305501],
                             [- 6.27194394, -0.00091519, 0.00056102, 7.27102875, 0.99720315, 0.99852380, 0.49305520],
                             [- 6.27194473, -0.00091492, 0.00056085, 7.27102981, 0.99720398, 0.99852423, 0.49305527],
                             [- 6.27194552, -0.00091465, 0.00056068, 7.27103087, 0.99720480, 0.99852467, 0.49305537],
                             [- 6.27194631, -0.00091438, 0.00056052, 7.27103193, 0.99720563, 0.99852511, 0.49305541],
                             [- 6.27194710, -0.00091411, 0.00056035, 7.27103299, 0.99720646, 0.99852554, 0.49305558],
                             [- 6.27194788, -0.00091384, 0.00056019, 7.27103405, 0.99720728, 0.99852598, 0.49305568],
                             [- 6.27194867, -0.00091357, 0.00056002, 7.27103510, 0.99720811, 0.99852641, 0.49305582],
                             [- 6.27194946, -0.00091330, 0.00055986, 7.27103616, 0.99720893, 0.99852685, 0.49305585],
                             [- 6.27195024, -0.00091303, 0.00055969, 7.27103721, 0.99720975, 0.99852728, 0.49305599],
                             [- 6.27195102, -0.00091276, 0.00055953, 7.27103827, 0.99721058, 0.99852772, 0.49305606],
                             [- 6.27195181, -0.00091249, 0.00055936, 7.27103932, 0.99721140, 0.99852815, 0.49305614],
                             [- 6.27195259, -0.00091222, 0.00055920, 7.27104037, 0.99721222, 0.99852858, 0.49305629],
                             [- 6.27195337, -0.00091195, 0.00055903, 7.27104142, 0.99721304, 0.99852902, 0.49305636],
                             [- 6.27195415, -0.00091168, 0.00055887, 7.27104247, 0.99721387, 0.99852945, 0.49305643],
                             [- 6.27195493, -0.00091141, 0.00055870, 7.27104351, 0.99721469, 0.99852988, 0.49305655],
                             [- 6.27195571, -0.00091115, 0.00055854, 7.27104456, 0.99721551, 0.99853032, 0.49305663],
                             [- 6.27195648, -0.00091088, 0.00055837, 7.27104561, 0.99721633, 0.99853075, 0.49305679],
                             [- 6.27195726, -0.00091061, 0.00055821, 7.27104665, 0.99721715, 0.99853118, 0.49305682],
                             [- 6.27195803, -0.00091034, 0.00055804, 7.27104769, 0.99721796, 0.99853161, 0.49305696],
                             [- 6.27195881, -0.00091007, 0.00055788, 7.27104873, 0.99721878, 0.99853205, 0.49305711],
                             [- 6.27195958, -0.00090981, 0.00055771, 7.27104978, 0.99721960, 0.99853248, 0.49305719],
                             [- 6.27196035, -0.00090954, 0.00055755, 7.27105081, 0.99722042, 0.99853291, 0.49305721],
                             [- 6.27196113, -0.00090927, 0.00055739, 7.27105185, 0.99722124, 0.99853334, 0.49305736],
                             [- 6.27196190, -0.00090901, 0.00055722, 7.27105289, 0.99722205, 0.99853377, 0.49305741],
                             [- 6.27196267, -0.00090874, 0.00055706, 7.27105393, 0.99722287, 0.99853420, 0.49305756],
                             [- 6.27196344, -0.00090847, 0.00055690, 7.27105496, 0.99722368, 0.99853463, 0.49305766],
                             [- 6.27196420, -0.00090821, 0.00055673, 7.27105600, 0.99722450, 0.99853506, 0.49305779],
                             [- 6.27196497, -0.00090794, 0.00055657, 7.27105703, 0.99722531, 0.99853549, 0.49305788],
                             [- 6.27196574, -0.00090767, 0.00055641, 7.27105806, 0.99722613, 0.99853592, 0.49305797],
                             [- 6.27196650, -0.00090741, 0.00055624, 7.27105909, 0.99722694, 0.99853635, 0.49305806],
                             [- 6.27196726, -0.00090714, 0.00055608, 7.27106012, 0.99722776, 0.99853678, 0.49305817],
                             [- 6.27196803, -0.00090688, 0.00055592, 7.27106115, 0.99722857, 0.99853721, 0.49305825],
                             [- 6.27196879, -0.00090661, 0.00055575, 7.27106218, 0.99722938, 0.99853764, 0.49305840],
                             [- 6.27196955, -0.00090634, 0.00055559, 7.27106321, 0.99723019, 0.99853807, 0.49305847],
                             [- 6.27197031, -0.00090608, 0.00055543, 7.27106423, 0.99723100, 0.99853849, 0.49305859],
                             [- 6.27197107, -0.00090581, 0.00055526, 7.27106526, 0.99723181, 0.99853892, 0.49305865],
                             [- 6.27197183, -0.00090555, 0.00055510, 7.27106628, 0.99723262, 0.99853935, 0.49305875],
                             [- 6.27197259, -0.00090528, 0.00055494, 7.27106730, 0.99723343, 0.99853978, 0.49305889],
                             [- 6.27197335, -0.00090502, 0.00055478, 7.27106833, 0.99723424, 0.99854020, 0.49305896],
                             [- 6.27197410, -0.00090476, 0.00055461, 7.27106935, 0.99723505, 0.99854063, 0.49305906],
                             [- 6.27197486, -0.00090449, 0.00055445, 7.27107037, 0.99723586, 0.99854106, 0.49305911],
                             [- 6.27197561, -0.00090423, 0.00055429, 7.27107138, 0.99723667, 0.99854148, 0.49305929],
                             [- 6.27197637, -0.00090396, 0.00055413, 7.27107240, 0.99723748, 0.99854191, 0.49305933],
                             [- 6.27197712, -0.00090370, 0.00055397, 7.27107342, 0.99723828, 0.99854233, 0.49305943],
                             [- 6.27197787, -0.00090344, 0.00055380, 7.27107443, 0.99723909, 0.99854276, 0.49305954],
                             [- 6.27197862, -0.00090317, 0.00055364, 7.27107545, 0.99723990, 0.99854319, 0.49305965],
                             [- 6.27197937, -0.00090291, 0.00055348, 7.27107646, 0.99724070, 0.99854361, 0.49305972],
                             [- 6.27198012, -0.00090264, 0.00055332, 7.27107748, 0.99724151, 0.99854404, 0.49305986],
                             [- 6.27198087, -0.00090238, 0.00055316, 7.27107849, 0.99724231, 0.99854446, 0.49305996],
                             [- 6.27198162, -0.00090212, 0.00055300, 7.27107950, 0.99724312, 0.99854488, 0.49306003],
                             [- 6.27198236, -0.00090186, 0.00055284, 7.27108051, 0.99724392, 0.99854531, 0.49306014],
                             [- 6.27198311, -0.00090159, 0.00055267, 7.27108151, 0.99724472, 0.99854573, 0.49306027],
                             [- 6.27198385, -0.00090133, 0.00055251, 7.27108252, 0.99724553, 0.99854616, 0.49306036],
                             [- 6.27198460, -0.00090107, 0.00055235, 7.27108353, 0.99724633, 0.99854658, 0.49306043],
                             [- 6.27198534, -0.00090081, 0.00055219, 7.27108453, 0.99724713, 0.99854700, 0.49306047],
                             [- 6.27198608, -0.00090054, 0.00055203, 7.27108554, 0.99724793, 0.99854742, 0.49306066],
                             [- 6.27198682, -0.00090028, 0.00055187, 7.27108654, 0.99724873, 0.99854785, 0.49306072],
                             [- 6.27198756, -0.00090002, 0.00055171, 7.27108754, 0.99724953, 0.99854827, 0.49306079],
                             [- 6.27198830, -0.00089976, 0.00055155, 7.27108855, 0.99725033, 0.99854869, 0.49306090],
                             [- 6.27198904, -0.00089950, 0.00055139, 7.27108955, 0.99725113, 0.99854911, 0.49306100],
                             [- 6.27198978, -0.00089924, 0.00055123, 7.27109054, 0.99725193, 0.99854954, 0.49306107],
                             [- 6.27199052, -0.00089898, 0.00055107, 7.27109154, 0.99725273, 0.99854996, 0.49306122],
                             [- 6.27199126, -0.00089871, 0.00055091, 7.27109254, 0.99725353, 0.99855038, 0.49306129],
                             [- 6.27199199, -0.00089845, 0.00055075, 7.27109354, 0.99725433, 0.99855080, 0.49306135],
                             [- 6.27199273, -0.00089819, 0.00055059, 7.27109453, 0.99725512, 0.99855122, 0.49306146],
                             [- 6.27199346, -0.00089793, 0.00055043, 7.27109553, 0.99725592, 0.99855164, 0.49306163],
                             [- 6.27199419, -0.00089767, 0.00055027, 7.27109652, 0.99725672, 0.99855206, 0.49306163],
                             [- 6.27199493, -0.00089741, 0.00055011, 7.27109751, 0.99725751, 0.99855248, 0.49306178],
                             [- 6.27199566, -0.00089715, 0.00054995, 7.27109851, 0.99725831, 0.99855290, 0.49306188],
                             [- 6.27199639, -0.00089689, 0.00054979, 7.27109950, 0.99725910, 0.99855332, 0.49306196],
                             [- 6.27199712, -0.00089663, 0.00054963, 7.27110049, 0.99725990, 0.99855374, 0.49306206],
                             [- 6.27199785, -0.00089637, 0.00054947, 7.27110147, 0.99726069, 0.99855416, 0.49306218],
                             [- 6.27199857, -0.00089611, 0.00054931, 7.27110246, 0.99726149, 0.99855458, 0.49306227],
                             [- 6.27199930, -0.00089585, 0.00054915, 7.27110345, 0.99726228, 0.99855499, 0.49306238],
                             [- 6.27200003, -0.00089559, 0.00054899, 7.27110443, 0.99726307, 0.99855541, 0.49306244],
                             [- 6.27200076, -0.00089534, 0.00054883, 7.27110542, 0.99726386, 0.99855583, 0.49306254],
                             [- 6.27200148, -0.00089508, 0.00054868, 7.27110640, 0.99726465, 0.99855625, 0.49306260],
                             [- 6.27200220, -0.00089482, 0.00054852, 7.27110739, 0.99726545, 0.99855667, 0.49306270],
                             [- 6.27200293, -0.00089456, 0.00054836, 7.27110837, 0.99726624, 0.99855708, 0.49306284],
                             [- 6.27200365, -0.00089430, 0.00054820, 7.27110935, 0.99726703, 0.99855750, 0.49306289],
                             [- 6.27200437, -0.00089404, 0.00054804, 7.27111033, 0.99726782, 0.99855792, 0.49306299],
                             [- 6.27200509, -0.00089378, 0.00054788, 7.27111131, 0.99726861, 0.99855833, 0.49306315],
                             [- 6.27200581, -0.00089353, 0.00054772, 7.27111229, 0.99726940, 0.99855875, 0.49306325],
                             [- 6.27200653, -0.00089327, 0.00054757, 7.27111326, 0.99727018, 0.99855916, 0.49306334],
                             [- 6.27200725, -0.00089301, 0.00054741, 7.27111424, 0.99727097, 0.99855958, 0.49306342],
                             [- 6.27200797, -0.00089275, 0.00054725, 7.27111522, 0.99727176, 0.99856000, 0.49306348],
                             [- 6.27200869, -0.00089250, 0.00054709, 7.27111619, 0.99727255, 0.99856041, 0.49306360],
                             [- 6.27200940, -0.00089224, 0.00054693, 7.27111717, 0.99727333, 0.99856083, 0.49306368],
                             [- 6.27201012, -0.00089198, 0.00054678, 7.27111814, 0.99727412, 0.99856124, 0.49306382],
                             [- 6.27201083, -0.00089172, 0.00054662, 7.27111911, 0.99727491, 0.99856166, 0.49306389],
                             [- 6.27201155, -0.00089147, 0.00054646, 7.27112008, 0.99727569, 0.99856207, 0.49306395],
                             [- 6.27201226, -0.00089121, 0.00054630, 7.27112105, 0.99727648, 0.99856248, 0.49306405],
                             [- 6.27201297, -0.00089095, 0.00054615, 7.27112202, 0.99727726, 0.99856290, 0.49306415],
                             [- 6.27201369, -0.00089070, 0.00054599, 7.27112299, 0.99727804, 0.99856331, 0.49306425],
                             [- 6.27201440, -0.00089044, 0.00054583, 7.27112395, 0.99727883, 0.99856373, 0.49306438],
                             [- 6.27201511, -0.00089019, 0.00054568, 7.27112492, 0.99727961, 0.99856414, 0.49306445],
                             [- 6.27201582, -0.00088993, 0.00054552, 7.27112589, 0.99728039, 0.99856455, 0.49306454],
                             [- 6.27201653, -0.00088967, 0.00054536, 7.27112685, 0.99728118, 0.99856496, 0.49306460],
                             [- 6.27201723, -0.00088942, 0.00054520, 7.27112782, 0.99728196, 0.99856538, 0.49306473],
                             [- 6.27201794, -0.00088916, 0.00054505, 7.27112878, 0.99728274, 0.99856579, 0.49306481],
                             [- 6.27201865, -0.00088891, 0.00054489, 7.27112974, 0.99728352, 0.99856620, 0.49306490],
                             [- 6.27201935, -0.00088865, 0.00054473, 7.27113070, 0.99728430, 0.99856661, 0.49306499],
                             [- 6.27202006, -0.00088840, 0.00054458, 7.27113166, 0.99728508, 0.99856702, 0.49306507],
                             [- 6.27202076, -0.00088814, 0.00054442, 7.27113262, 0.99728586, 0.99856744, 0.49306519],
                             [- 6.27202147, -0.00088789, 0.00054427, 7.27113358, 0.99728664, 0.99856785, 0.49306522],
                             [- 6.27202217, -0.00088763, 0.00054411, 7.27113454, 0.99728742, 0.99856826, 0.49306541],
                             [- 6.27202287, -0.00088738, 0.00054395, 7.27113549, 0.99728820, 0.99856867, 0.49306548],
                             [- 6.27202357, -0.00088713, 0.00054380, 7.27113645, 0.99728897, 0.99856908, 0.49306558],
                             [- 6.27202427, -0.00088687, 0.00054364, 7.27113740, 0.99728975, 0.99856949, 0.49306563],
                             [- 6.27202497, -0.00088662, 0.00054349, 7.27113836, 0.99729053, 0.99856990, 0.49306575],
                             [- 6.27202567, -0.00088636, 0.00054333, 7.27113931, 0.99729130, 0.99857031, 0.49306582],
                             [- 6.27202637, -0.00088611, 0.00054317, 7.27114026, 0.99729208, 0.99857072, 0.49306593],
                             [- 6.27202707, -0.00088586, 0.00054302, 7.27114121, 0.99729285, 0.99857113, 0.49306602],
                             [- 6.27202777, -0.00088560, 0.00054286, 7.27114216, 0.99729363, 0.99857153, 0.49306615],
                             [- 6.27202846, -0.00088535, 0.00054271, 7.27114311, 0.99729440, 0.99857194, 0.49306620],
                             [- 6.27202916, -0.00088510, 0.00054255, 7.27114406, 0.99729518, 0.99857235, 0.49306631],
                             [- 6.27202985, -0.00088484, 0.00054240, 7.27114501, 0.99729595, 0.99857276, 0.49306638],
                             [- 6.27203055, -0.00088459, 0.00054224, 7.27114596, 0.99729673, 0.99857317, 0.49306650],
                             [- 6.27203124, -0.00088434, 0.00054209, 7.27114690, 0.99729750, 0.99857358, 0.49306661],
                             [- 6.27203193, -0.00088409, 0.00054193, 7.27114785, 0.99729827, 0.99857398, 0.49306664],
                             [- 6.27203262, -0.00088383, 0.00054178, 7.27114879, 0.99729904, 0.99857439, 0.49306674],
                             [- 6.27203332, -0.00088358, 0.00054162, 7.27114973, 0.99729981, 0.99857480, 0.49306687],
                             [- 6.27203401, -0.00088333, 0.00054147, 7.27115068, 0.99730059, 0.99857520, 0.49306699],
                             [- 6.27203470, -0.00088308, 0.00054131, 7.27115162, 0.99730136, 0.99857561, 0.49306701],
                             [- 6.27203539, -0.00088282, 0.00054116, 7.27115256, 0.99730213, 0.99857602, 0.49306713],
                             [- 6.27203607, -0.00088257, 0.00054100, 7.27115350, 0.99730290, 0.99857642, 0.49306721],
                             [- 6.27203676, -0.00088232, 0.00054085, 7.27115444, 0.99730366, 0.99857683, 0.49306728],
                             [- 6.27203745, -0.00088207, 0.00054069, 7.27115538, 0.99730443, 0.99857724, 0.49306742],
                             [- 6.27203813, -0.00088182, 0.00054054, 7.27115632, 0.99730520, 0.99857764, 0.49306753],
                             [- 6.27203882, -0.00088157, 0.00054039, 7.27115725, 0.99730597, 0.99857805, 0.49306761],
                             [- 6.27203950, -0.00088132, 0.00054023, 7.27115819, 0.99730674, 0.99857845, 0.49306769],
                             [- 6.27204019, -0.00088107, 0.00054008, 7.27115912, 0.99730750, 0.99857886, 0.49306776],
                             [- 6.27204087, -0.00088082, 0.00053992, 7.27116006, 0.99730827, 0.99857926, 0.49306785],
                             [- 6.27204156, -0.00088056, 0.00053977, 7.27116099, 0.99730904, 0.99857966, 0.49306801],
                             [- 6.27204224, -0.00088031, 0.00053962, 7.27116192, 0.99730980, 0.99858007, 0.49306813],
                             [- 6.27204292, -0.00088006, 0.00053946, 7.27116286, 0.99731057, 0.99858047, 0.49306816],
                             [- 6.27204360, -0.00087981, 0.00053931, 7.27116379, 0.99731133, 0.99858088, 0.49306827],
                             [- 6.27204428, -0.00087956, 0.00053916, 7.27116472, 0.99731210, 0.99858128, 0.49306833],
                             [- 6.27204496, -0.00087931, 0.00053900, 7.27116565, 0.99731286, 0.99858168, 0.49306842],
                             [- 6.27204564, -0.00087906, 0.00053885, 7.27116657, 0.99731363, 0.99858209, 0.49306852],
                             [- 6.27204632, -0.00087881, 0.00053870, 7.27116750, 0.99731439, 0.99858249, 0.49306865],
                             [- 6.27204699, -0.00087857, 0.00053854, 7.27116843, 0.99731515, 0.99858289, 0.49306872],
                             [- 6.27204767, -0.00087832, 0.00053839, 7.27116935, 0.99731591, 0.99858329, 0.49306875],
                             [- 6.27204835, -0.00087807, 0.00053824, 7.27117028, 0.99731668, 0.99858369, 0.49306889],
                             [- 6.27204902, -0.00087782, 0.00053809, 7.27117120, 0.99731744, 0.99858410, 0.49306899],
                             [- 6.27204970, -0.00087757, 0.00053793, 7.27117213, 0.99731820, 0.99858450, 0.49306909],
                             [- 6.27205037, -0.00087732, 0.00053778, 7.27117305, 0.99731896, 0.99858490, 0.49306915],
                             [- 6.27205104, -0.00087707, 0.00053763, 7.27117397, 0.99731972, 0.99858530, 0.49306931],
                             [- 6.27205172, -0.00087682, 0.00053748, 7.27117489, 0.99732048, 0.99858570, 0.49306931],
                             [- 6.27205239, -0.00087658, 0.00053732, 7.27117581, 0.99732124, 0.99858610, 0.49306947],
                             [- 6.27205306, -0.00087633, 0.00053717, 7.27117673, 0.99732200, 0.99858650, 0.49306949],
                             [- 6.27205373, -0.00087608, 0.00053702, 7.27117765, 0.99732276, 0.99858690, 0.49306961],
                             [- 6.27205440, -0.00087583, 0.00053687, 7.27117857, 0.99732351, 0.99858730, 0.49306974],
                             [- 6.27205507, -0.00087558, 0.00053671, 7.27117949, 0.99732427, 0.99858770, 0.49306982],
                             [- 6.27205574, -0.00087534, 0.00053656, 7.27118040, 0.99732503, 0.99858810, 0.49306986],
                             [- 6.27205641, -0.00087509, 0.00053641, 7.27118132, 0.99732579, 0.99858850, 0.49306997],
                             [- 6.27205707, -0.00087484, 0.00053626, 7.27118223, 0.99732654, 0.99858890, 0.49307002],
                             [- 6.27205774, -0.00087459, 0.00053611, 7.27118315, 0.99732730, 0.99858930, 0.49307020],
                             [- 6.27205841, -0.00087435, 0.00053596, 7.27118406, 0.99732805, 0.99858970, 0.49307021],
                             [- 6.27205907, -0.00087410, 0.00053580, 7.27118497, 0.99732881, 0.99859010, 0.49307035],
                             [- 6.27205974, -0.00087385, 0.00053565, 7.27118588, 0.99732956, 0.99859049, 0.49307047],
                             [- 6.27206040, -0.00087361, 0.00053550, 7.27118679, 0.99733032, 0.99859089, 0.49307051],
                             [- 6.27206106, -0.00087336, 0.00053535, 7.27118770, 0.99733107, 0.99859129, 0.49307062],
                             [- 6.27206173, -0.00087311, 0.00053520, 7.27118861, 0.99733182, 0.99859169, 0.49307069],
                             [- 6.27206239, -0.00087287, 0.00053505, 7.27118952, 0.99733258, 0.99859208, 0.49307080],
                             [- 6.27206305, -0.00087262, 0.00053490, 7.27119043, 0.99733333, 0.99859248, 0.49307089],
                             [- 6.27206371, -0.00087238, 0.00053475, 7.27119134, 0.99733408, 0.99859288, 0.49307096],
                             [- 6.27206437, -0.00087213, 0.00053460, 7.27119224, 0.99733483, 0.99859327, 0.49307103],
                             [- 6.27206503, -0.00087188, 0.00053445, 7.27119315, 0.99733558, 0.99859367, 0.49307121],
                             [- 6.27206569, -0.00087164, 0.00053429, 7.27119405, 0.99733633, 0.99859407, 0.49307121],
                             [- 6.27206635, -0.00087139, 0.00053414, 7.27119496, 0.99733709, 0.99859446, 0.49307131],
                             [- 6.27206701, -0.00087115, 0.00053399, 7.27119586, 0.99733783, 0.99859486, 0.49307143],
                             [- 6.27206767, -0.00087090, 0.00053384, 7.27119676, 0.99733858, 0.99859525, 0.49307152],
                             [- 6.27206832, -0.00087066, 0.00053369, 7.27119766, 0.99733933, 0.99859565, 0.49307163],
                             [- 6.27206898, -0.00087041, 0.00053354, 7.27119856, 0.99734008, 0.99859604, 0.49307170],
                             [- 6.27206963, -0.00087017, 0.00053339, 7.27119946, 0.99734083, 0.99859644, 0.49307178],
                             [- 6.27207029, -0.00086992, 0.00053324, 7.27120036, 0.99734158, 0.99859683, 0.49307182],
                             [- 6.27207094, -0.00086968, 0.00053309, 7.27120126, 0.99734233, 0.99859723, 0.49307196],
                             [- 6.27207160, -0.00086944, 0.00053294, 7.27120216, 0.99734307, 0.99859762, 0.49307197],
                             [- 6.27207225, -0.00086919, 0.00053279, 7.27120306, 0.99734382, 0.99859802, 0.49307220],
                             [- 6.27207290, -0.00086895, 0.00053264, 7.27120395, 0.99734457, 0.99859841, 0.49307226],
                             [- 6.27207355, -0.00086870, 0.00053249, 7.27120485, 0.99734531, 0.99859880, 0.49307233],
                             [- 6.27207420, -0.00086846, 0.00053234, 7.27120574, 0.99734606, 0.99859920, 0.49307241],
                             [- 6.27207485, -0.00086822, 0.00053219, 7.27120664, 0.99734680, 0.99859959, 0.49307252],
                             [- 6.27207550, -0.00086797, 0.00053205, 7.27120753, 0.99734755, 0.99859998, 0.49307258],
                             [- 6.27207615, -0.00086773, 0.00053190, 7.27120842, 0.99734829, 0.99860037, 0.49307271],
                             [- 6.27207680, -0.00086749, 0.00053175, 7.27120931, 0.99734903, 0.99860077, 0.49307274],
                             [- 6.27207745, -0.00086724, 0.00053160, 7.27121021, 0.99734978, 0.99860116, 0.49307287],
                             [- 6.27207810, -0.00086700, 0.00053145, 7.27121110, 0.99735052, 0.99860155, 0.49307293],
                             [- 6.27207874, -0.00086676, 0.00053130, 7.27121199, 0.99735126, 0.99860194, 0.49307303],
                             [- 6.27207939, -0.00086652, 0.00053115, 7.27121287, 0.99735200, 0.99860233, 0.49307313],
                             [- 6.27208004, -0.00086627, 0.00053100, 7.27121376, 0.99735275, 0.99860273, 0.49307325],
                             [- 6.27208068, -0.00086603, 0.00053085, 7.27121465, 0.99735349, 0.99860312, 0.49307329],
                             [- 6.27208132, -0.00086579, 0.00053070, 7.27121554, 0.99735423, 0.99860351, 0.49307333],
                             [- 6.27208197, -0.00086555, 0.00053056, 7.27121642, 0.99735497, 0.99860390, 0.49307349],
                             [- 6.27208261, -0.00086530, 0.00053041, 7.27121731, 0.99735571, 0.99860429, 0.49307355],
                             [- 6.27208325, -0.00086506, 0.00053026, 7.27121819, 0.99735645, 0.99860468, 0.49307366],
                             [- 6.27208390, -0.00086482, 0.00053011, 7.27121907, 0.99735719, 0.99860507, 0.49307376],
                             [- 6.27208454, -0.00086458, 0.00052996, 7.27121996, 0.99735792, 0.99860546, 0.49307384],
                             [- 6.27208518, -0.00086434, 0.00052981, 7.27122084, 0.99735866, 0.99860585, 0.49307389],
                             [- 6.27208582, -0.00086410, 0.00052967, 7.27122172, 0.99735940, 0.99860624, 0.49307398],
                             [- 6.27208646, -0.00086386, 0.00052952, 7.27122260, 0.99736014, 0.99860663, 0.49307410],
                             [- 6.27208710, -0.00086361, 0.00052937, 7.27122348, 0.99736088, 0.99860701, 0.49307418],
                             [- 6.27208774, -0.00086337, 0.00052922, 7.27122436, 0.99736161, 0.99860740, 0.49307431],
                             [- 6.27208837, -0.00086313, 0.00052908, 7.27122524, 0.99736235, 0.99860779, 0.49307430],
                             [- 6.27208901, -0.00086289, 0.00052893, 7.27122612, 0.99736308, 0.99860818, 0.49307450],
                             [- 6.27208965, -0.00086265, 0.00052878, 7.27122700, 0.99736382, 0.99860857, 0.49307452],
                             [- 6.27209028, -0.00086241, 0.00052863, 7.27122787, 0.99736455, 0.99860896, 0.49307461],
                             [- 6.27209092, -0.00086217, 0.00052849, 7.27122875, 0.99736529, 0.99860934, 0.49307473],
                             [- 6.27209155, -0.00086193, 0.00052834, 7.27122962, 0.99736602, 0.99860973, 0.49307476],
                             [- 6.27209219, -0.00086169, 0.00052819, 7.27123050, 0.99736676, 0.99861012, 0.49307493],
                             [- 6.27209282, -0.00086145, 0.00052804, 7.27123137, 0.99736749, 0.99861050, 0.49307491],
                             [- 6.27209346, -0.00086121, 0.00052790, 7.27123224, 0.99736822, 0.99861089, 0.49307506],
                             [- 6.27209409, -0.00086097, 0.00052775, 7.27123312, 0.99736896, 0.99861128, 0.49307515],
                             [- 6.27209472, -0.00086073, 0.00052760, 7.27123399, 0.99736969, 0.99861166, 0.49307527],
                             [- 6.27209535, -0.00086049, 0.00052746, 7.27123486, 0.99737042, 0.99861205, 0.49307528],
                             [- 6.27209598, -0.00086025, 0.00052731, 7.27123573, 0.99737115, 0.99861244, 0.49307544],
                             [- 6.27209661, -0.00086002, 0.00052716, 7.27123660, 0.99737188, 0.99861282, 0.49307551],
                             [- 6.27209724, -0.00085978, 0.00052702, 7.27123747, 0.99737261, 0.99861321, 0.49307555],
                             [- 6.27209787, -0.00085954, 0.00052687, 7.27123833, 0.99737334, 0.99861359, 0.49307566],
                             [- 6.27209850, -0.00085930, 0.00052672, 7.27123920, 0.99737407, 0.99861398, 0.49307570],
                             [- 6.27209913, -0.00085906, 0.00052658, 7.27124007, 0.99737480, 0.99861436, 0.49307581],
                             [- 6.27209976, -0.00085882, 0.00052643, 7.27124094, 0.99737553, 0.99861475, 0.49307599],
                             [- 6.27210038, -0.00085858, 0.00052628, 7.27124180, 0.99737626, 0.99861513, 0.49307598],
                             [- 6.27210101, -0.00085835, 0.00052614, 7.27124266, 0.99737699, 0.99861551, 0.49307614],
                             [- 6.27210164, -0.00085811, 0.00052599, 7.27124353, 0.99737771, 0.99861590, 0.49307614],
                             [- 6.27210226, -0.00085787, 0.00052585, 7.27124439, 0.99737844, 0.99861628, 0.49307625],
                             [- 6.27210289, -0.00085763, 0.00052570, 7.27124525, 0.99737917, 0.99861667, 0.49307637],
                             [- 6.27210351, -0.00085740, 0.00052555, 7.27124612, 0.99737989, 0.99861705, 0.49307645],
                             [- 6.27210414, -0.00085716, 0.00052541, 7.27124698, 0.99738062, 0.99861743, 0.49307653],
                             [- 6.27210476, -0.00085692, 0.00052526, 7.27124784, 0.99738135, 0.99861782, 0.49307663],
                             [- 6.27210538, -0.00085668, 0.00052512, 7.27124870, 0.99738207, 0.99861820, 0.49307669],
                             [- 6.27210600, -0.00085645, 0.00052497, 7.27124956, 0.99738280, 0.99861858, 0.49307677],
                             [- 6.27210663, -0.00085621, 0.00052483, 7.27125042, 0.99738352, 0.99861896, 0.49307688],
                             [- 6.27210725, -0.00085597, 0.00052468, 7.27125127, 0.99738424, 0.99861934, 0.49307699],
                             [- 6.27210787, -0.00085574, 0.00052454, 7.27125213, 0.99738497, 0.99861973, 0.49307706],
                             [- 6.27210849, -0.00085550, 0.00052439, 7.27125299, 0.99738569, 0.99862011, 0.49307704],
                             [- 6.27210911, -0.00085526, 0.00052425, 7.27125384, 0.99738641, 0.99862049, 0.49307723],
                             [- 6.27210973, -0.00085503, 0.00052410, 7.27125470, 0.99738714, 0.99862087, 0.49307731],
                             [- 6.27211034, -0.00085479, 0.00052396, 7.27125555, 0.99738786, 0.99862125, 0.49307742],
                             [- 6.27211096, -0.00085456, 0.00052381, 7.27125641, 0.99738858, 0.99862163, 0.49307753],
                             [- 6.27211158, -0.00085432, 0.00052367, 7.27125726, 0.99738930, 0.99862201, 0.49307753],
                             [- 6.27211220, -0.00085408, 0.00052352, 7.27125811, 0.99739002, 0.99862239, 0.49307764],
                             [- 6.27211281, -0.00085385, 0.00052338, 7.27125896, 0.99739074, 0.99862277, 0.49307770],
                             [- 6.27211343, -0.00085361, 0.00052323, 7.27125981, 0.99739146, 0.99862315, 0.49307780],
                             [- 6.27211404, -0.00085338, 0.00052309, 7.27126067, 0.99739218, 0.99862353, 0.49307792],
                             [- 6.27211466, -0.00085314, 0.00052295, 7.27126152, 0.99739290, 0.99862391, 0.49307794],
                             [- 6.27211527, -0.00085291, 0.00052280, 7.27126236, 0.99739362, 0.99862429, 0.49307803],
                             [- 6.27211589, -0.00085267, 0.00052266, 7.27126321, 0.99739434, 0.99862467, 0.49307820],
                             [- 6.27211650, -0.00085244, 0.00052251, 7.27126406, 0.99739506, 0.99862505, 0.49307827],
                             [- 6.27211711, -0.00085220, 0.00052237, 7.27126491, 0.99739577, 0.99862543, 0.49307830],
                             [- 6.27211772, -0.00085197, 0.00052222, 7.27126575, 0.99739649, 0.99862581, 0.49307839],
                             [- 6.27211833, -0.00085173, 0.00052208, 7.27126660, 0.99739721, 0.99862618, 0.49307851],
                             [- 6.27211895, -0.00085150, 0.00052194, 7.27126745, 0.99739792, 0.99862656, 0.49307861],
                             [- 6.27211956, -0.00085127, 0.00052179, 7.27126829, 0.99739864, 0.99862694, 0.49307868],
                             [- 6.27212017, -0.00085103, 0.00052165, 7.27126913, 0.99739936, 0.99862732, 0.49307878],
                             [- 6.27212078, -0.00085080, 0.00052151, 7.27126998, 0.99740007, 0.99862770, 0.49307884],
                             [- 6.27212138, -0.00085056, 0.00052136, 7.27127082, 0.99740079, 0.99862807, 0.49307888],
                             [- 6.27212199, -0.00085033, 0.00052122, 7.27127166, 0.99740150, 0.99862845, 0.49307902],
                             [- 6.27212260, -0.00085010, 0.00052108, 7.27127250, 0.99740221, 0.99862883, 0.49307911],
                             [- 6.27212321, -0.00084986, 0.00052093, 7.27127334, 0.99740293, 0.99862920, 0.49307920],
                             [- 6.27212381, -0.00084963, 0.00052079, 7.27127418, 0.99740364, 0.99862958, 0.49307926],
                             [- 6.27212442, -0.00084940, 0.00052065, 7.27127502, 0.99740435, 0.99862996, 0.49307935],
                             [- 6.27212503, -0.00084916, 0.00052050, 7.27127586, 0.99740507, 0.99863033, 0.49307944],
                             [- 6.27212563, -0.00084893, 0.00052036, 7.27127670, 0.99740578, 0.99863071, 0.49307944],
                             [- 6.27212624, -0.00084870, 0.00052022, 7.27127754, 0.99740649, 0.99863108, 0.49307954],
                             [- 6.27212684, -0.00084847, 0.00052008, 7.27127837, 0.99740720, 0.99863146, 0.49307961],
                             [- 6.27212744, -0.00084823, 0.00051993, 7.27127921, 0.99740791, 0.99863183, 0.49307978],
                             [- 6.27212805, -0.00084800, 0.00051979, 7.27128005, 0.99740862, 0.99863221, 0.49307983],
                             [- 6.27212865, -0.00084777, 0.00051965, 7.27128088, 0.99740933, 0.99863258, 0.49307996],
                             [- 6.27212925, -0.00084754, 0.00051951, 7.27128172, 0.99741004, 0.99863296, 0.49308005],
                             [- 6.27212986, -0.00084730, 0.00051936, 7.27128255, 0.99741075, 0.99863333, 0.49308014],
                             [- 6.27213046, -0.00084707, 0.00051922, 7.27128338, 0.99741146, 0.99863371, 0.49308018],
                             [- 6.27213106, -0.00084684, 0.00051908, 7.27128422, 0.99741217, 0.99863408, 0.49308030],
                             [- 6.27213166, -0.00084661, 0.00051894, 7.27128505, 0.99741288, 0.99863445, 0.49308040],
                             [- 6.27213226, -0.00084638, 0.00051879, 7.27128588, 0.99741359, 0.99863483, 0.49308040],
                             [- 6.27213286, -0.00084615, 0.00051865, 7.27128671, 0.99741429, 0.99863520, 0.49308050],
                             [- 6.27213345, -0.00084592, 0.00051851, 7.27128754, 0.99741500, 0.99863557, 0.49308061],
                             [- 6.27213405, -0.00084568, 0.00051837, 7.27128837, 0.99741571, 0.99863595, 0.49308064],
                             [- 6.27213465, -0.00084545, 0.00051823, 7.27128920, 0.99741641, 0.99863632, 0.49308083],
                             [- 6.27213525, -0.00084522, 0.00051809, 7.27129003, 0.99741712, 0.99863669, 0.49308089],
                             [- 6.27213585, -0.00084499, 0.00051794, 7.27129085, 0.99741783, 0.99863706, 0.49308103],
                             [- 6.27213644, -0.00084476, 0.00051780, 7.27129168, 0.99741853, 0.99863744, 0.49308104],
                             [- 6.27213704, -0.00084453, 0.00051766, 7.27129251, 0.99741924, 0.99863781, 0.49308114],
                             [- 6.27213763, -0.00084430, 0.00051752, 7.27129333, 0.99741994, 0.99863818, 0.49308122],
                             [- 6.27213823, -0.00084407, 0.00051738, 7.27129416, 0.99742064, 0.99863855, 0.49308125],
                             [- 6.27213882, -0.00084384, 0.00051724, 7.27129498, 0.99742135, 0.99863892, 0.49308128],
                             [- 6.27213942, -0.00084361, 0.00051710, 7.27129581, 0.99742205, 0.99863929, 0.49308149],
                             [- 6.27214001, -0.00084338, 0.00051696, 7.27129663, 0.99742275, 0.99863966, 0.49308153],
                             [- 6.27214060, -0.00084315, 0.00051681, 7.27129745, 0.99742346, 0.99864003, 0.49308160],
                             [- 6.27214120, -0.00084292, 0.00051667, 7.27129827, 0.99742416, 0.99864041, 0.49308174],
                             [- 6.27214179, -0.00084269, 0.00051653, 7.27129910, 0.99742486, 0.99864078, 0.49308171],
                             [- 6.27214238, -0.00084246, 0.00051639, 7.27129992, 0.99742556, 0.99864115, 0.49308189],
                             [- 6.27214297, -0.00084223, 0.00051625, 7.27130074, 0.99742626, 0.99864152, 0.49308189],
                             [- 6.27214356, -0.00084200, 0.00051611, 7.27130156, 0.99742696, 0.99864188, 0.49308208],
                             [- 6.27214415, -0.00084178, 0.00051597, 7.27130237, 0.99742766, 0.99864225, 0.49308218],
                             [- 6.27214474, -0.00084155, 0.00051583, 7.27130319, 0.99742836, 0.99864262, 0.49308217],
                             [- 6.27214533, -0.00084132, 0.00051569, 7.27130401, 0.99742906, 0.99864299, 0.49308225],
                             [- 6.27214592, -0.00084109, 0.00051555, 7.27130483, 0.99742976, 0.99864336, 0.49308239],
                             [- 6.27214651, -0.00084086, 0.00051541, 7.27130564, 0.99743046, 0.99864373, 0.49308247],
                             [- 6.27214709, -0.00084063, 0.00051527, 7.27130646, 0.99743116, 0.99864410, 0.49308255],
                             [- 6.27214768, -0.00084040, 0.00051513, 7.27130728, 0.99743186, 0.99864447, 0.49308258],
                             [- 6.27214827, -0.00084018, 0.00051499, 7.27130809, 0.99743255, 0.99864483, 0.49308266],
                             [- 6.27214885, -0.00083995, 0.00051485, 7.27130891, 0.99743325, 0.99864520, 0.49308274],
                             [- 6.27214944, -0.00083972, 0.00051471, 7.27130972, 0.99743395, 0.99864557, 0.49308290],
                             [- 6.27215002, -0.00083949, 0.00051457, 7.27131053, 0.99743464, 0.99864594, 0.49308290],
                             [- 6.27215061, -0.00083926, 0.00051443, 7.27131134, 0.99743534, 0.99864631, 0.49308300],
                             [- 6.27215119, -0.00083904, 0.00051429, 7.27131216, 0.99743604, 0.99864667, 0.49308315],
                             [- 6.27215178, -0.00083881, 0.00051415, 7.27131297, 0.99743673, 0.99864704, 0.49308314],
                             [- 6.27215236, -0.00083858, 0.00051401, 7.27131378, 0.99743743, 0.99864741, 0.49308324],
                             [- 6.27215294, -0.00083836, 0.00051387, 7.27131459, 0.99743812, 0.99864777, 0.49308338],
                             [- 6.27215353, -0.00083813, 0.00051373, 7.27131540, 0.99743881, 0.99864814, 0.49308338],
                             [- 6.27215411, -0.00083790, 0.00051359, 7.27131621, 0.99743951, 0.99864850, 0.49308350],
                             [- 6.27215469, -0.00083768, 0.00051345, 7.27131702, 0.99744020, 0.99864887, 0.49308357],
                             [- 6.27215527, -0.00083745, 0.00051332, 7.27131782, 0.99744089, 0.99864924, 0.49308374],
                             [- 6.27215585, -0.00083722, 0.00051318, 7.27131863, 0.99744159, 0.99864960, 0.49308379],
                             [- 6.27215643, -0.00083700, 0.00051304, 7.27131944, 0.99744228, 0.99864997, 0.49308384],
                             [- 6.27215701, -0.00083677, 0.00051290, 7.27132024, 0.99744297, 0.99865033, 0.49308393],
                             [- 6.27215759, -0.00083654, 0.00051276, 7.27132105, 0.99744366, 0.99865070, 0.49308399],
                             [- 6.27215817, -0.00083632, 0.00051262, 7.27132185, 0.99744435, 0.99865106, 0.49308412],
                             [- 6.27215875, -0.00083609, 0.00051248, 7.27132266, 0.99744504, 0.99865143, 0.49308421],
                             [- 6.27215933, -0.00083587, 0.00051234, 7.27132346, 0.99744574, 0.99865179, 0.49308425],
                             [- 6.27215990, -0.00083564, 0.00051221, 7.27132426, 0.99744643, 0.99865215, 0.49308434],
                             [- 6.27216048, -0.00083541, 0.00051207, 7.27132507, 0.99744711, 0.99865252, 0.49308442],
                             [- 6.27216106, -0.00083519, 0.00051193, 7.27132587, 0.99744780, 0.99865288, 0.49308450],
                             [- 6.27216164, -0.00083496, 0.00051179, 7.27132667, 0.99744849, 0.99865324, 0.49308461],
                             [- 6.27216221, -0.00083474, 0.00051165, 7.27132747, 0.99744918, 0.99865361, 0.49308467],
                             [- 6.27216279, -0.00083451, 0.00051151, 7.27132827, 0.99744987, 0.99865397, 0.49308473],
                             [- 6.27216336, -0.00083429, 0.00051138, 7.27132907, 0.99745056, 0.99865433, 0.49308483],
                             [- 6.27216394, -0.00083406, 0.00051124, 7.27132987, 0.99745124, 0.99865470, 0.49308489],
                             [- 6.27216451, -0.00083384, 0.00051110, 7.27133067, 0.99745193, 0.99865506, 0.49308501],
                             [- 6.27216508, -0.00083361, 0.00051096, 7.27133147, 0.99745262, 0.99865542, 0.49308509],
                             [- 6.27216566, -0.00083339, 0.00051083, 7.27133227, 0.99745330, 0.99865578, 0.49308519],
                             [- 6.27216623, -0.00083317, 0.00051069, 7.27133306, 0.99745399, 0.99865615, 0.49308520],
                             [- 6.27216680, -0.00083294, 0.00051055, 7.27133386, 0.99745468, 0.99865651, 0.49308532],
                             [- 6.27216737, -0.00083272, 0.00051041, 7.27133465, 0.99745536, 0.99865687, 0.49308537],
                             [- 6.27216794, -0.00083249, 0.00051028, 7.27133545, 0.99745605, 0.99865723, 0.49308548],
                             [- 6.27216851, -0.00083227, 0.00051014, 7.27133624, 0.99745673, 0.99865759, 0.49308555],
                             [- 6.27216909, -0.00083205, 0.00051000, 7.27133704, 0.99745741, 0.99865795, 0.49308568],
                             [- 6.27216966, -0.00083182, 0.00050986, 7.27133783, 0.99745810, 0.99865831, 0.49308575],
                             [- 6.27217022, -0.00083160, 0.00050973, 7.27133863, 0.99745878, 0.99865867, 0.49308580],
                             [- 6.27217079, -0.00083138, 0.00050959, 7.27133942, 0.99745946, 0.99865903, 0.49308589],
                             [- 6.27217136, -0.00083115, 0.00050945, 7.27134021, 0.99746015, 0.99865939, 0.49308593],
                             [- 6.27217193, -0.00083093, 0.00050932, 7.27134100, 0.99746083, 0.99865975, 0.49308603],
                             [- 6.27217250, -0.00083071, 0.00050918, 7.27134179, 0.99746151, 0.99866011, 0.49308603],
                             [- 6.27217307, -0.00083048, 0.00050904, 7.27134258, 0.99746219, 0.99866047, 0.49308626],
                             [- 6.27217363, -0.00083026, 0.00050891, 7.27134337, 0.99746287, 0.99866083, 0.49308628],
                             [- 6.27217420, -0.00083004, 0.00050877, 7.27134416, 0.99746355, 0.99866119, 0.49308633],
                             [- 6.27217477, -0.00082982, 0.00050863, 7.27134495, 0.99746424, 0.99866155, 0.49308648],
                             [- 6.27217533, -0.00082959, 0.00050850, 7.27134574, 0.99746492, 0.99866191, 0.49308650],
                             [- 6.27217590, -0.00082937, 0.00050836, 7.27134653, 0.99746559, 0.99866227, 0.49308659],
                             [- 6.27217646, -0.00082915, 0.00050822, 7.27134731, 0.99746627, 0.99866263, 0.49308664],
                             [- 6.27217703, -0.00082893, 0.00050809, 7.27134810, 0.99746695, 0.99866299, 0.49308674],
                             [- 6.27217759, -0.00082871, 0.00050795, 7.27134888, 0.99746763, 0.99866334, 0.49308688],
                             [- 6.27217815, -0.00082848, 0.00050782, 7.27134967, 0.99746831, 0.99866370, 0.49308691],
                             [- 6.27217872, -0.00082826, 0.00050768, 7.27135045, 0.99746899, 0.99866406, 0.49308700],
                             [- 6.27217928, -0.00082804, 0.00050754, 7.27135124, 0.99746967, 0.99866442, 0.49308706],
                             [- 6.27217984, -0.00082782, 0.00050741, 7.27135202, 0.99747034, 0.99866477, 0.49308711],
                             [- 6.27218040, -0.00082760, 0.00050727, 7.27135281, 0.99747102, 0.99866513, 0.49308731],
                             [- 6.27218097, -0.00082738, 0.00050714, 7.27135359, 0.99747170, 0.99866549, 0.49308733],
                             [- 6.27218153, -0.00082716, 0.00050700, 7.27135437, 0.99747237, 0.99866584, 0.49308746],
                             [- 6.27218209, -0.00082693, 0.00050686, 7.27135515, 0.99747305, 0.99866620, 0.49308753],
                             [- 6.27218265, -0.00082671, 0.00050673, 7.27135593, 0.99747372, 0.99866656, 0.49308759],
                             [- 6.27218321, -0.00082649, 0.00050659, 7.27135671, 0.99747440, 0.99866691, 0.49308757],
                             [- 6.27218377, -0.00082627, 0.00050646, 7.27135749, 0.99747507, 0.99866727, 0.49308770],
                             [- 6.27218433, -0.00082605, 0.00050632, 7.27135827, 0.99747575, 0.99866763, 0.49308772],
                             [- 6.27218488, -0.00082583, 0.00050619, 7.27135905, 0.99747642, 0.99866798, 0.49308784],
                             [- 6.27218544, -0.00082561, 0.00050605, 7.27135983, 0.99747709, 0.99866834, 0.49308794],
                             [- 6.27218600, -0.00082539, 0.00050592, 7.27136061, 0.99747777, 0.99866869, 0.49308810],
                             [- 6.27218656, -0.00082517, 0.00050578, 7.27136139, 0.99747844, 0.99866905, 0.49308812],
                             [- 6.27218711, -0.00082495, 0.00050565, 7.27136216, 0.99747911, 0.99866940, 0.49308821],
                             [- 6.27218767, -0.00082473, 0.00050551, 7.27136294, 0.99747978, 0.99866976, 0.49308825],
                             [- 6.27218823, -0.00082451, 0.00050538, 7.27136371, 0.99748046, 0.99867011, 0.49308833],
                             [- 6.27218878, -0.00082429, 0.00050524, 7.27136449, 0.99748113, 0.99867046, 0.49308847],
                             [- 6.27218934, -0.00082407, 0.00050511, 7.27136526, 0.99748180, 0.99867082, 0.49308860],
                             [- 6.27218989, -0.00082385, 0.00050497, 7.27136604, 0.99748247, 0.99867117, 0.49308859],
                             [- 6.27219045, -0.00082363, 0.00050484, 7.27136681, 0.99748314, 0.99867153, 0.49308869],
                             [- 6.27219100, -0.00082341, 0.00050471, 7.27136759, 0.99748381, 0.99867188, 0.49308873],
                             [- 6.27219155, -0.00082320, 0.00050457, 7.27136836, 0.99748448, 0.99867223, 0.49308878],
                             [- 6.27219211, -0.00082298, 0.00050444, 7.27136913, 0.99748515, 0.99867259, 0.49308888],
                             [- 6.27219266, -0.00082276, 0.00050430, 7.27136990, 0.99748582, 0.99867294, 0.49308897],
                             [- 6.27219321, -0.00082254, 0.00050417, 7.27137067, 0.99748649, 0.99867329, 0.49308901],
                             [- 6.27219376, -0.00082232, 0.00050403, 7.27137144, 0.99748715, 0.99867364, 0.49308913],
                             [- 6.27219432, -0.00082210, 0.00050390, 7.27137221, 0.99748782, 0.99867400, 0.49308916],
                             [- 6.27219487, -0.00082188, 0.00050377, 7.27137298, 0.99748849, 0.99867435, 0.49308935],
                             [- 6.27219542, -0.00082167, 0.00050363, 7.27137375, 0.99748916, 0.99867470, 0.49308938],
                             [- 6.27219597, -0.00082145, 0.00050350, 7.27137452, 0.99748982, 0.99867505, 0.49308944],
                             [- 6.27219652, -0.00082123, 0.00050337, 7.27137529, 0.99749049, 0.99867540, 0.49308959],
                             [- 6.27219707, -0.00082101, 0.00050323, 7.27137606, 0.99749116, 0.99867576, 0.49308961],
                             [- 6.27219762, -0.00082080, 0.00050310, 7.27137682, 0.99749182, 0.99867611, 0.49308966],
                             [- 6.27219817, -0.00082058, 0.00050296, 7.27137759, 0.99749249, 0.99867646, 0.49308979],
                             [- 6.27219872, -0.00082036, 0.00050283, 7.27137836, 0.99749315, 0.99867681, 0.49308985],
                             [- 6.27219926, -0.00082014, 0.00050270, 7.27137912, 0.99749382, 0.99867716, 0.49308989],
                             [- 6.27219981, -0.00081993, 0.00050256, 7.27137989, 0.99749448, 0.99867751, 0.49309002],
                             [- 6.27220036, -0.00081971, 0.00050243, 7.27138065, 0.99749514, 0.99867786, 0.49309006],
                             [- 6.27220091, -0.00081949, 0.00050230, 7.27138141, 0.99749581, 0.99867821, 0.49309016],
                             [- 6.27220145, -0.00081927, 0.00050216, 7.27138218, 0.99749647, 0.99867856, 0.49309022],
                             [- 6.27220200, -0.00081906, 0.00050203, 7.27138294, 0.99749713, 0.99867891, 0.49309028],
                             [- 6.27220254, -0.00081884, 0.00050190, 7.27138370, 0.99749780, 0.99867926, 0.49309037],
                             [- 6.27220309, -0.00081862, 0.00050177, 7.27138446, 0.99749846, 0.99867961, 0.49309053],
                             [- 6.27220363, -0.00081841, 0.00050163, 7.27138523, 0.99749912, 0.99867996, 0.49309055],
                             [- 6.27220418, -0.00081819, 0.00050150, 7.27138599, 0.99749978, 0.99868031, 0.49309069],
                             [- 6.27220472, -0.00081798, 0.00050137, 7.27138675, 0.99750044, 0.99868066, 0.49309070],
                             [- 6.27220527, -0.00081776, 0.00050124, 7.27138751, 0.99750110, 0.99868100, 0.49309083],
                             [- 6.27220581, -0.00081754, 0.00050110, 7.27138827, 0.99750176, 0.99868135, 0.49309090],
                             [- 6.27220635, -0.00081733, 0.00050097, 7.27138902, 0.99750242, 0.99868170, 0.49309098],
                             [- 6.27220690, -0.00081711, 0.00050084, 7.27138978, 0.99750308, 0.99868205, 0.49309101],
                             [- 6.27220744, -0.00081690, 0.00050071, 7.27139054, 0.99750374, 0.99868240, 0.49309113],
                             [- 6.27220798, -0.00081668, 0.00050057, 7.27139130, 0.99750440, 0.99868275, 0.49309116],
                             [- 6.27220852, -0.00081647, 0.00050044, 7.27139205, 0.99750506, 0.99868309, 0.49309123],
                             [- 6.27220906, -0.00081625, 0.00050031, 7.27139281, 0.99750572, 0.99868344, 0.49309142],
                             [- 6.27220960, -0.00081604, 0.00050018, 7.27139357, 0.99750638, 0.99868379, 0.49309142],
                             [- 6.27221014, -0.00081582, 0.00050005, 7.27139432, 0.99750703, 0.99868413, 0.49309146],
                             [- 6.27221068, -0.00081561, 0.00049991, 7.27139508, 0.99750769, 0.99868448, 0.49309158],
                             [- 6.27221122, -0.00081539, 0.00049978, 7.27139583, 0.99750835, 0.99868483, 0.49309166],
                             [- 6.27221176, -0.00081518, 0.00049965, 7.27139659, 0.99750901, 0.99868517, 0.49309171],
                             [- 6.27221230, -0.00081496, 0.00049952, 7.27139734, 0.99750966, 0.99868552, 0.49309183],
                             [- 6.27221284, -0.00081475, 0.00049939, 7.27139809, 0.99751032, 0.99868587, 0.49309186],
                             [- 6.27221338, -0.00081453, 0.00049926, 7.27139884, 0.99751097, 0.99868621, 0.49309196],
                             [- 6.27221391, -0.00081432, 0.00049912, 7.27139960, 0.99751163, 0.99868656, 0.49309197],
                             [- 6.27221445, -0.00081410, 0.00049899, 7.27140035, 0.99751228, 0.99868690, 0.49309212],
                             [- 6.27221499, -0.00081389, 0.00049886, 7.27140110, 0.99751294, 0.99868725, 0.49309226],
                             [- 6.27221552, -0.00081368, 0.00049873, 7.27140185, 0.99751359, 0.99868759, 0.49309233],
                             [- 6.27221606, -0.00081346, 0.00049860, 7.27140260, 0.99751425, 0.99868794, 0.49309240],
                             [- 6.27221660, -0.00081325, 0.00049847, 7.27140335, 0.99751490, 0.99868828, 0.49309241],
                             [- 6.27221713, -0.00081303, 0.00049834, 7.27140410, 0.99751555, 0.99868863, 0.49309250],
                             [- 6.27221767, -0.00081282, 0.00049821, 7.27140485, 0.99751621, 0.99868897, 0.49309252],
                             [- 6.27221820, -0.00081261, 0.00049807, 7.27140559, 0.99751686, 0.99868932, 0.49309262],
                             [- 6.27221874, -0.00081239, 0.00049794, 7.27140634, 0.99751751, 0.99868966, 0.49309274],
                             [- 6.27221927, -0.00081218, 0.00049781, 7.27140709, 0.99751816, 0.99869001, 0.49309277],
                             [- 6.27221980, -0.00081197, 0.00049768, 7.27140784, 0.99751881, 0.99869035, 0.49309287],
                             [- 6.27222034, -0.00081176, 0.00049755, 7.27140858, 0.99751946, 0.99869069, 0.49309295],
                             [- 6.27222087, -0.00081154, 0.00049742, 7.27140933, 0.99752012, 0.99869104, 0.49309305],
                             [- 6.27222140, -0.00081133, 0.00049729, 7.27141007, 0.99752077, 0.99869138, 0.49309311],
                             [- 6.27222193, -0.00081112, 0.00049716, 7.27141082, 0.99752142, 0.99869172, 0.49309317],
                             [- 6.27222247, -0.00081090, 0.00049703, 7.27141156, 0.99752207, 0.99869207, 0.49309328],
                             [- 6.27222300, -0.00081069, 0.00049690, 7.27141230, 0.99752271, 0.99869241, 0.49309332],
                             [- 6.27222353, -0.00081048, 0.00049677, 7.27141305, 0.99752336, 0.99869275, 0.49309344],
                             [- 6.27222406, -0.00081027, 0.00049664, 7.27141379, 0.99752401, 0.99869309, 0.49309341],
                             [- 6.27222459, -0.00081006, 0.00049651, 7.27141453, 0.99752466, 0.99869344, 0.49309354],
                             [- 6.27222512, -0.00080984, 0.00049638, 7.27141527, 0.99752531, 0.99869378, 0.49309361],
                             [- 6.27222565, -0.00080963, 0.00049625, 7.27141602, 0.99752596, 0.99869412, 0.49309375],
                             [- 6.27222618, -0.00080942, 0.00049612, 7.27141676, 0.99752660, 0.99869446, 0.49309380],
                             [- 6.27222671, -0.00080921, 0.00049599, 7.27141750, 0.99752725, 0.99869480, 0.49309382],
                             [- 6.27222723, -0.00080900, 0.00049586, 7.27141824, 0.99752790, 0.99869514, 0.49309395],
                             [- 6.27222776, -0.00080879, 0.00049573, 7.27141898, 0.99752854, 0.99869548, 0.49309403],
                             [- 6.27222829, -0.00080857, 0.00049560, 7.27141972, 0.99752919, 0.99869582, 0.49309409],
                             [- 6.27222882, -0.00080836, 0.00049547, 7.27142045, 0.99752984, 0.99869617, 0.49309410],
                             [- 6.27222934, -0.00080815, 0.00049534, 7.27142119, 0.99753048, 0.99869651, 0.49309425],
                             [- 6.27222987, -0.00080794, 0.00049521, 7.27142193, 0.99753113, 0.99869685, 0.49309434],
                             [- 6.27223040, -0.00080773, 0.00049508, 7.27142267, 0.99753177, 0.99869719, 0.49309443],
                             [- 6.27223092, -0.00080752, 0.00049495, 7.27142340, 0.99753241, 0.99869753, 0.49309447],
                             [- 6.27223145, -0.00080731, 0.00049482, 7.27142414, 0.99753306, 0.99869787, 0.49309457],
                             [- 6.27223197, -0.00080710, 0.00049469, 7.27142488, 0.99753370, 0.99869821, 0.49309459],
                             [- 6.27223250, -0.00080689, 0.00049457, 7.27142561, 0.99753435, 0.99869855, 0.49309463],
                             [- 6.27223302, -0.00080668, 0.00049444, 7.27142635, 0.99753499, 0.99869888, 0.49309473],
                             [- 6.27223355, -0.00080647, 0.00049431, 7.27142708, 0.99753563, 0.99869922, 0.49309486],
                             [- 6.27223407, -0.00080626, 0.00049418, 7.27142781, 0.99753627, 0.99869956, 0.49309496],
                             [- 6.27223460, -0.00080605, 0.00049405, 7.27142855, 0.99753692, 0.99869990, 0.49309501],
                             [- 6.27223512, -0.00080584, 0.00049392, 7.27142928, 0.99753756, 0.99870024, 0.49309506],
                             [- 6.27223564, -0.00080563, 0.00049379, 7.27143001, 0.99753820, 0.99870058, 0.49309510],
                             [- 6.27223616, -0.00080542, 0.00049366, 7.27143074, 0.99753884, 0.99870092, 0.49309524],
                             [- 6.27223669, -0.00080521, 0.00049354, 7.27143148, 0.99753948, 0.99870125, 0.49309538],
                             [- 6.27223721, -0.00080500, 0.00049341, 7.27143221, 0.99754012, 0.99870159, 0.49309541],
                             [- 6.27223773, -0.00080479, 0.00049328, 7.27143294, 0.99754076, 0.99870193, 0.49309547],
                             [- 6.27223825, -0.00080458, 0.00049315, 7.27143367, 0.99754140, 0.99870227, 0.49309552],
                             [- 6.27223877, -0.00080437, 0.00049302, 7.27143440, 0.99754204, 0.99870260, 0.49309562],
                             [- 6.27223929, -0.00080416, 0.00049289, 7.27143513, 0.99754268, 0.99870294, 0.49309564],
                             [- 6.27223981, -0.00080395, 0.00049277, 7.27143586, 0.99754332, 0.99870328, 0.49309575],
                             [- 6.27224033, -0.00080375, 0.00049264, 7.27143658, 0.99754395, 0.99870362, 0.49309585],
                             [- 6.27224085, -0.00080354, 0.00049251, 7.27143731, 0.99754459, 0.99870395, 0.49309591],
                             [- 6.27224137, -0.00080333, 0.00049238, 7.27143804, 0.99754523, 0.99870429, 0.49309595],
                             [- 6.27224189, -0.00080312, 0.00049225, 7.27143877, 0.99754587, 0.99870463, 0.49309610],
                             [- 6.27224241, -0.00080291, 0.00049213, 7.27143949, 0.99754650, 0.99870496, 0.49309609],
                             [- 6.27224292, -0.00080270, 0.00049200, 7.27144022, 0.99754714, 0.99870530, 0.49309625],
                             [- 6.27224344, -0.00080250, 0.00049187, 7.27144094, 0.99754778, 0.99870563, 0.49309625],
                             [- 6.27224396, -0.00080229, 0.00049174, 7.27144167, 0.99754841, 0.99870597, 0.49309632],
                             [- 6.27224447, -0.00080208, 0.00049162, 7.27144239, 0.99754905, 0.99870630, 0.49309639],
                             [- 6.27224499, -0.00080187, 0.00049149, 7.27144312, 0.99754968, 0.99870664, 0.49309648],
                             [- 6.27224551, -0.00080166, 0.00049136, 7.27144384, 0.99755032, 0.99870697, 0.49309655],
                             [- 6.27224602, -0.00080146, 0.00049123, 7.27144457, 0.99755095, 0.99870731, 0.49309667],
                             [- 6.27224654, -0.00080125, 0.00049111, 7.27144529, 0.99755159, 0.99870764, 0.49309666],
                             [- 6.27224705, -0.00080104, 0.00049098, 7.27144601, 0.99755222, 0.99870798, 0.49309679],
                             [- 6.27224757, -0.00080084, 0.00049085, 7.27144673, 0.99755286, 0.99870831, 0.49309687],
                             [- 6.27224808, -0.00080063, 0.00049072, 7.27144746, 0.99755349, 0.99870865, 0.49309695],
                             [- 6.27224860, -0.00080042, 0.00049060, 7.27144818, 0.99755412, 0.99870898, 0.49309704],
                             [- 6.27224911, -0.00080021, 0.00049047, 7.27144890, 0.99755476, 0.99870932, 0.49309706],
                             [- 6.27224962, -0.00080001, 0.00049034, 7.27144962, 0.99755539, 0.99870965, 0.49309713],
                             [- 6.27225014, -0.00079980, 0.00049022, 7.27145034, 0.99755602, 0.99870998, 0.49309722],
                             [- 6.27225065, -0.00079959, 0.00049009, 7.27145106, 0.99755665, 0.99871032, 0.49309728],
                             [- 6.27225116, -0.00079939, 0.00048996, 7.27145178, 0.99755728, 0.99871065, 0.49309738],
                             [- 6.27225168, -0.00079918, 0.00048984, 7.27145249, 0.99755791, 0.99871098, 0.49309747],
                             [- 6.27225219, -0.00079897, 0.00048971, 7.27145321, 0.99755854, 0.99871131, 0.49309753],
                             [- 6.27225270, -0.00079877, 0.00048958, 7.27145393, 0.99755918, 0.99871165, 0.49309771],
                             [- 6.27225321, -0.00079856, 0.00048946, 7.27145465, 0.99755981, 0.99871198, 0.49309765],
                             [- 6.27225372, -0.00079836, 0.00048933, 7.27145536, 0.99756044, 0.99871231, 0.49309782],
                             [- 6.27225423, -0.00079815, 0.00048921, 7.27145608, 0.99756106, 0.99871264, 0.49309784],
                             [- 6.27225474, -0.00079795, 0.00048908, 7.27145680, 0.99756169, 0.99871298, 0.49309793],
                             [- 6.27225525, -0.00079774, 0.00048895, 7.27145751, 0.99756232, 0.99871331, 0.49309799],
                             [- 6.27225576, -0.00079753, 0.00048883, 7.27145823, 0.99756295, 0.99871364, 0.49309812],
                             [- 6.27225627, -0.00079733, 0.00048870, 7.27145894, 0.99756358, 0.99871397, 0.49309805],
                             [- 6.27225678, -0.00079712, 0.00048857, 7.27145966, 0.99756421, 0.99871430, 0.49309821],
                             [- 6.27225729, -0.00079692, 0.00048845, 7.27146037, 0.99756483, 0.99871463, 0.49309831],
                             [- 6.27225780, -0.00079671, 0.00048832, 7.27146108, 0.99756546, 0.99871496, 0.49309836],
                             [- 6.27225830, -0.00079651, 0.00048820, 7.27146180, 0.99756609, 0.99871529, 0.49309842],
                             [- 6.27225881, -0.00079630, 0.00048807, 7.27146251, 0.99756671, 0.99871563, 0.49309856],
                             [- 6.27225932, -0.00079610, 0.00048795, 7.27146322, 0.99756734, 0.99871596, 0.49309861],
                             [- 6.27225983, -0.00079589, 0.00048782, 7.27146393, 0.99756797, 0.99871629, 0.49309861],
                             [- 6.27226033, -0.00079569, 0.00048769, 7.27146464, 0.99756859, 0.99871662, 0.49309873],
                             [- 6.27226084, -0.00079548, 0.00048757, 7.27146535, 0.99756922, 0.99871695, 0.49309877],
                             [- 6.27226134, -0.00079528, 0.00048744, 7.27146606, 0.99756984, 0.99871728, 0.49309886],
                             [- 6.27226185, -0.00079508, 0.00048732, 7.27146677, 0.99757047, 0.99871761, 0.49309893],
                             [- 6.27226235, -0.00079487, 0.00048719, 7.27146748, 0.99757109, 0.99871793, 0.49309902],
                             [- 6.27226286, -0.00079467, 0.00048707, 7.27146819, 0.99757172, 0.99871826, 0.49309912],
                             [- 6.27226336, -0.00079446, 0.00048694, 7.27146890, 0.99757234, 0.99871859, 0.49309921],
                             [- 6.27226387, -0.00079426, 0.00048682, 7.27146961, 0.99757296, 0.99871892, 0.49309921],
                             [- 6.27226437, -0.00079406, 0.00048669, 7.27147032, 0.99757359, 0.99871925, 0.49309928],
                             [- 6.27226488, -0.00079385, 0.00048657, 7.27147102, 0.99757421, 0.99871958, 0.49309939],
                             [- 6.27226538, -0.00079365, 0.00048644, 7.27147173, 0.99757483, 0.99871991, 0.49309945],
                             [- 6.27226588, -0.00079345, 0.00048632, 7.27147244, 0.99757545, 0.99872024, 0.49309958],
                             [- 6.27226639, -0.00079324, 0.00048619, 7.27147314, 0.99757607, 0.99872056, 0.49309958],
                             [- 6.27226689, -0.00079304, 0.00048607, 7.27147385, 0.99757670, 0.99872089, 0.49309961],
                             [- 6.27226739, -0.00079284, 0.00048594, 7.27147455, 0.99757732, 0.99872122, 0.49309967],
                             [- 6.27226789, -0.00079263, 0.00048582, 7.27147526, 0.99757794, 0.99872155, 0.49309986],
                             [- 6.27226839, -0.00079243, 0.00048570, 7.27147596, 0.99757856, 0.99872187, 0.49309996],
                             [- 6.27226889, -0.00079223, 0.00048557, 7.27147667, 0.99757918, 0.99872220, 0.49309999],
                             [- 6.27226939, -0.00079202, 0.00048545, 7.27147737, 0.99757980, 0.99872253, 0.49310010],
                             [- 6.27226990, -0.00079182, 0.00048532, 7.27147807, 0.99758042, 0.99872286, 0.49310015],
                             [- 6.27227040, -0.00079162, 0.00048520, 7.27147878, 0.99758104, 0.99872318, 0.49310022],
                             [- 6.27227090, -0.00079142, 0.00048507, 7.27147948, 0.99758166, 0.99872351, 0.49310023],
                             [- 6.27227139, -0.00079121, 0.00048495, 7.27148018, 0.99758227, 0.99872384, 0.49310029],
                             [- 6.27227189, -0.00079101, 0.00048483, 7.27148088, 0.99758289, 0.99872416, 0.49310038],
                             [- 6.27227239, -0.00079081, 0.00048470, 7.27148158, 0.99758351, 0.99872449, 0.49310046],
                             [- 6.27227289, -0.00079061, 0.00048458, 7.27148228, 0.99758413, 0.99872481, 0.49310052],
                             [- 6.27227339, -0.00079041, 0.00048445, 7.27148298, 0.99758474, 0.99872514, 0.49310058],
                             [- 6.27227389, -0.00079020, 0.00048433, 7.27148368, 0.99758536, 0.99872546, 0.49310064],
                             [- 6.27227439, -0.00079000, 0.00048421, 7.27148438, 0.99758598, 0.99872579, 0.49310077],
                             [- 6.27227488, -0.00078980, 0.00048408, 7.27148508, 0.99758659, 0.99872612, 0.49310085],
                             [- 6.27227538, -0.00078960, 0.00048396, 7.27148578, 0.99758721, 0.99872644, 0.49310092],
                             [- 6.27227588, -0.00078940, 0.00048384, 7.27148648, 0.99758783, 0.99872677, 0.49310100],
                             [- 6.27227637, -0.00078920, 0.00048371, 7.27148717, 0.99758844, 0.99872709, 0.49310112],
                             [- 6.27227687, -0.00078900, 0.00048359, 7.27148787, 0.99758906, 0.99872741, 0.49310113],
                             [- 6.27227736, -0.00078880, 0.00048347, 7.27148857, 0.99758967, 0.99872774, 0.49310122],
                             [- 6.27227786, -0.00078859, 0.00048334, 7.27148927, 0.99759029, 0.99872806, 0.49310127],
                             [- 6.27227836, -0.00078839, 0.00048322, 7.27148996, 0.99759090, 0.99872839, 0.49310130],
                             [- 6.27227885, -0.00078819, 0.00048310, 7.27149066, 0.99759151, 0.99872871, 0.49310141],
                             [- 6.27227934, -0.00078799, 0.00048297, 7.27149135, 0.99759213, 0.99872903, 0.49310152],
                             [- 6.27227984, -0.00078779, 0.00048285, 7.27149205, 0.99759274, 0.99872936, 0.49310152],
                             [- 6.27228033, -0.00078759, 0.00048273, 7.27149274, 0.99759335, 0.99872968, 0.49310163],
                             [- 6.27228083, -0.00078739, 0.00048260, 7.27149344, 0.99759397, 0.99873001, 0.49310168],
                             [- 6.27228132, -0.00078719, 0.00048248, 7.27149413, 0.99759458, 0.99873033, 0.49310178],
                             [- 6.27228181, -0.00078699, 0.00048236, 7.27149482, 0.99759519, 0.99873065, 0.49310189],
                             [- 6.27228231, -0.00078679, 0.00048224, 7.27149552, 0.99759580, 0.99873097, 0.49310191],
                             [- 6.27228280, -0.00078659, 0.00048211, 7.27149621, 0.99759641, 0.99873130, 0.49310203],
                             [- 6.27228329, -0.00078639, 0.00048199, 7.27149690, 0.99759702, 0.99873162, 0.49310202],
                             [- 6.27228378, -0.00078619, 0.00048187, 7.27149759, 0.99759763, 0.99873194, 0.49310216],
                             [- 6.27228427, -0.00078599, 0.00048175, 7.27149828, 0.99759825, 0.99873226, 0.49310220],
                             [- 6.27228476, -0.00078579, 0.00048162, 7.27149897, 0.99759886, 0.99873259, 0.49310228],
                             [- 6.27228526, -0.00078559, 0.00048150, 7.27149966, 0.99759946, 0.99873291, 0.49310231],
                             [- 6.27228575, -0.00078539, 0.00048138, 7.27150035, 0.99760007, 0.99873323, 0.49310245],
                             [- 6.27228624, -0.00078519, 0.00048126, 7.27150104, 0.99760068, 0.99873355, 0.49310250],
                             [- 6.27228673, -0.00078500, 0.00048113, 7.27150173, 0.99760129, 0.99873387, 0.49310256],
                             [- 6.27228722, -0.00078480, 0.00048101, 7.27150242, 0.99760190, 0.99873419, 0.49310269],
                             [- 6.27228771, -0.00078460, 0.00048089, 7.27150311, 0.99760251, 0.99873451, 0.49310278],
                             [- 6.27228819, -0.00078440, 0.00048077, 7.27150380, 0.99760312, 0.99873483, 0.49310277],
                             [- 6.27228868, -0.00078420, 0.00048065, 7.27150448, 0.99760372, 0.99873515, 0.49310291],
                             [- 6.27228917, -0.00078400, 0.00048052, 7.27150517, 0.99760433, 0.99873548, 0.49310291],
                             [- 6.27228966, -0.00078380, 0.00048040, 7.27150586, 0.99760494, 0.99873580, 0.49310301],
                             [- 6.27229015, -0.00078360, 0.00048028, 7.27150654, 0.99760555, 0.99873612, 0.49310307],
                             [- 6.27229064, -0.00078341, 0.00048016, 7.27150723, 0.99760615, 0.99873644, 0.49310321],
                             [- 6.27229112, -0.00078321, 0.00048004, 7.27150792, 0.99760676, 0.99873676, 0.49310322],
                             [- 6.27229161, -0.00078301, 0.00047992, 7.27150860, 0.99760736, 0.99873707, 0.49310330],
                             [- 6.27229210, -0.00078281, 0.00047979, 7.27150929, 0.99760797, 0.99873739, 0.49310330],
                             [- 6.27229258, -0.00078261, 0.00047967, 7.27150997, 0.99760857, 0.99873771, 0.49310340],
                             [- 6.27229307, -0.00078242, 0.00047955, 7.27151065, 0.99760918, 0.99873803, 0.49310344],
                             [- 6.27229356, -0.00078222, 0.00047943, 7.27151134, 0.99760978, 0.99873835, 0.49310362],
                             [- 6.27229404, -0.00078202, 0.00047931, 7.27151202, 0.99761039, 0.99873867, 0.49310362],
                             [- 6.27229453, -0.00078182, 0.00047919, 7.27151270, 0.99761099, 0.99873899, 0.49310369],
                             [- 6.27229501, -0.00078163, 0.00047907, 7.27151339, 0.99761160, 0.99873931, 0.49310382],
                             [- 6.27229550, -0.00078143, 0.00047895, 7.27151407, 0.99761220, 0.99873963, 0.49310378],
                             [- 6.27229598, -0.00078123, 0.00047882, 7.27151475, 0.99761280, 0.99873994, 0.49310395],
                             [- 6.27229646, -0.00078103, 0.00047870, 7.27151543, 0.99761340, 0.99874026, 0.49310398],
                             [- 6.27229695, -0.00078084, 0.00047858, 7.27151611, 0.99761401, 0.99874058, 0.49310409],
                             [- 6.27229743, -0.00078064, 0.00047846, 7.27151679, 0.99761461, 0.99874090, 0.49310410],
                             [- 6.27229791, -0.00078044, 0.00047834, 7.27151747, 0.99761521, 0.99874122, 0.49310419],
                             [- 6.27229840, -0.00078025, 0.00047822, 7.27151815, 0.99761581, 0.99874153, 0.49310426],
                             [- 6.27229888, -0.00078005, 0.00047810, 7.27151883, 0.99761641, 0.99874185, 0.49310431],
                             [- 6.27229936, -0.00077985, 0.00047798, 7.27151951, 0.99761701, 0.99874217, 0.49310439],
                             [- 6.27229984, -0.00077966, 0.00047786, 7.27152019, 0.99761762, 0.99874248, 0.49310456],
                             [- 6.27230033, -0.00077946, 0.00047774, 7.27152087, 0.99761822, 0.99874280, 0.49310453],
                             [- 6.27230081, -0.00077926, 0.00047762, 7.27152154, 0.99761882, 0.99874312, 0.49310456],
                             [- 6.27230129, -0.00077907, 0.00047750, 7.27152222, 0.99761942, 0.99874343, 0.49310472],
                             [- 6.27230177, -0.00077887, 0.00047738, 7.27152290, 0.99762001, 0.99874375, 0.49310478],
                             [- 6.27230225, -0.00077868, 0.00047726, 7.27152357, 0.99762061, 0.99874407, 0.49310482],
                             [- 6.27230273, -0.00077848, 0.00047714, 7.27152425, 0.99762121, 0.99874438, 0.49310485],
                             [- 6.27230321, -0.00077828, 0.00047702, 7.27152493, 0.99762181, 0.99874470, 0.49310496],
                             [- 6.27230369, -0.00077809, 0.00047690, 7.27152560, 0.99762241, 0.99874501, 0.49310508],
                             [- 6.27230417, -0.00077789, 0.00047678, 7.27152628, 0.99762301, 0.99874533, 0.49310514],
                             [- 6.27230465, -0.00077770, 0.00047666, 7.27152695, 0.99762360, 0.99874564, 0.49310512],
                             [- 6.27230513, -0.00077750, 0.00047654, 7.27152763, 0.99762420, 0.99874596, 0.49310522],
                             [- 6.27230561, -0.00077731, 0.00047642, 7.27152830, 0.99762480, 0.99874627, 0.49310530],
                             [- 6.27230609, -0.00077711, 0.00047630, 7.27152897, 0.99762540, 0.99874659, 0.49310543],
                             [- 6.27230656, -0.00077692, 0.00047618, 7.27152965, 0.99762599, 0.99874690, 0.49310543],
                             [- 6.27230704, -0.00077672, 0.00047606, 7.27153032, 0.99762659, 0.99874722, 0.49310556],
                             [- 6.27230752, -0.00077653, 0.00047594, 7.27153099, 0.99762718, 0.99874753, 0.49310555],
                             [- 6.27230800, -0.00077633, 0.00047582, 7.27153166, 0.99762778, 0.99874785, 0.49310570],
                             [- 6.27230847, -0.00077614, 0.00047570, 7.27153233, 0.99762837, 0.99874816, 0.49310570],
                             [- 6.27230895, -0.00077594, 0.00047558, 7.27153301, 0.99762897, 0.99874848, 0.49310582],
                             [- 6.27230943, -0.00077575, 0.00047546, 7.27153368, 0.99762956, 0.99874879, 0.49310585],
                             [- 6.27230990, -0.00077556, 0.00047534, 7.27153435, 0.99763016, 0.99874910, 0.49310599],
                             [- 6.27231038, -0.00077536, 0.00047522, 7.27153502, 0.99763075, 0.99874942, 0.49310605],
                             [- 6.27231085, -0.00077517, 0.00047510, 7.27153569, 0.99763135, 0.99874973, 0.49310608],
                             [- 6.27231133, -0.00077497, 0.00047498, 7.27153636, 0.99763194, 0.99875004, 0.49310612],
                             [- 6.27231180, -0.00077478, 0.00047487, 7.27153702, 0.99763253, 0.99875036, 0.49310624],
                             [- 6.27231228, -0.00077458, 0.00047475, 7.27153769, 0.99763312, 0.99875067, 0.49310622],
                             [- 6.27231275, -0.00077439, 0.00047463, 7.27153836, 0.99763372, 0.99875098, 0.49310639],
                             [- 6.27231323, -0.00077420, 0.00047451, 7.27153903, 0.99763431, 0.99875129, 0.49310643],
                             [- 6.27231370, -0.00077400, 0.00047439, 7.27153970, 0.99763490, 0.99875161, 0.49310644],
                             [- 6.27231417, -0.00077381, 0.00047427, 7.27154036, 0.99763549, 0.99875192, 0.49310654],
                             [- 6.27231465, -0.00077362, 0.00047415, 7.27154103, 0.99763608, 0.99875223, 0.49310665],
                             [- 6.27231512, -0.00077342, 0.00047403, 7.27154170, 0.99763668, 0.99875254, 0.49310674],
                             [- 6.27231559, -0.00077323, 0.00047392, 7.27154236, 0.99763727, 0.99875285, 0.49310670],
                             [- 6.27231607, -0.00077304, 0.00047380, 7.27154303, 0.99763786, 0.99875317, 0.49310684],
                             [- 6.27231654, -0.00077284, 0.00047368, 7.27154369, 0.99763845, 0.99875348, 0.49310692],
                             [- 6.27231701, -0.00077265, 0.00047356, 7.27154436, 0.99763904, 0.99875379, 0.49310699],
                             [- 6.27231748, -0.00077246, 0.00047344, 7.27154502, 0.99763963, 0.99875410, 0.49310700],
                             [- 6.27231795, -0.00077227, 0.00047332, 7.27154569, 0.99764021, 0.99875441, 0.49310703],
                             [- 6.27231842, -0.00077207, 0.00047321, 7.27154635, 0.99764080, 0.99875472, 0.49310722],
                             [- 6.27231889, -0.00077188, 0.00047309, 7.27154701, 0.99764139, 0.99875503, 0.49310725],
                             [- 6.27231936, -0.00077169, 0.00047297, 7.27154768, 0.99764198, 0.99875534, 0.49310729],
                             [- 6.27231984, -0.00077150, 0.00047285, 7.27154834, 0.99764257, 0.99875565, 0.49310745],
                             [- 6.27232031, -0.00077130, 0.00047273, 7.27154900, 0.99764316, 0.99875596, 0.49310751],
                             [- 6.27232077, -0.00077111, 0.00047262, 7.27154966, 0.99764374, 0.99875627, 0.49310749],
                             [- 6.27232124, -0.00077092, 0.00047250, 7.27155032, 0.99764433, 0.99875658, 0.49310765],
                             [- 6.27232171, -0.00077073, 0.00047238, 7.27155099, 0.99764492, 0.99875689, 0.49310767],
                             [- 6.27232218, -0.00077054, 0.00047226, 7.27155165, 0.99764550, 0.99875720, 0.49310775],
                             [- 6.27232265, -0.00077034, 0.00047215, 7.27155231, 0.99764609, 0.99875751, 0.49310781],
                             [- 6.27232312, -0.00077015, 0.00047203, 7.27155297, 0.99764668, 0.99875782, 0.49310787],
                             [- 6.27232359, -0.00076996, 0.00047191, 7.27155363, 0.99764726, 0.99875813, 0.49310796],
                             [- 6.27232406, -0.00076977, 0.00047179, 7.27155429, 0.99764785, 0.99875844, 0.49310787],
                             [- 6.27232452, -0.00076958, 0.00047168, 7.27155494, 0.99764843, 0.99875875, 0.49310807],
                             [- 6.27232499, -0.00076939, 0.00047156, 7.27155560, 0.99764902, 0.99875906, 0.49310819],
                             [- 6.27232546, -0.00076920, 0.00047144, 7.27155626, 0.99764960, 0.99875936, 0.49310824],
                             [- 6.27232592, -0.00076900, 0.00047132, 7.27155692, 0.99765019, 0.99875967, 0.49310831],
                             [- 6.27232639, -0.00076881, 0.00047121, 7.27155758, 0.99765077, 0.99875998, 0.49310826],
                             [- 6.27232686, -0.00076862, 0.00047109, 7.27155823, 0.99765135, 0.99876029, 0.49310841],
                             [- 6.27232732, -0.00076843, 0.00047097, 7.27155889, 0.99765194, 0.99876060, 0.49310851],
                             [- 6.27232779, -0.00076824, 0.00047086, 7.27155955, 0.99765252, 0.99876090, 0.49310855],
                             [- 6.27232825, -0.00076805, 0.00047074, 7.27156020, 0.99765310, 0.99876121, 0.49310862],
                             [- 6.27232872, -0.00076786, 0.00047062, 7.27156086, 0.99765369, 0.99876152, 0.49310868],
                             [- 6.27232918, -0.00076767, 0.00047050, 7.27156151, 0.99765427, 0.99876183, 0.49310870],
                             [- 6.27232965, -0.00076748, 0.00047039, 7.27156217, 0.99765485, 0.99876213, 0.49310885],
                             [- 6.27233011, -0.00076729, 0.00047027, 7.27156282, 0.99765543, 0.99876244, 0.49310881],
                             [- 6.27233058, -0.00076710, 0.00047015, 7.27156348, 0.99765601, 0.99876275, 0.49310901],
                             [- 6.27233104, -0.00076691, 0.00047004, 7.27156413, 0.99765660, 0.99876305, 0.49310905],
                             [- 6.27233150, -0.00076672, 0.00046992, 7.27156478, 0.99765718, 0.99876336, 0.49310906],
                             [- 6.27233197, -0.00076653, 0.00046981, 7.27156544, 0.99765776, 0.99876367, 0.49310915],
                             [- 6.27233243, -0.00076634, 0.00046969, 7.27156609, 0.99765834, 0.99876397, 0.49310926],
                             [- 6.27233289, -0.00076615, 0.00046957, 7.27156674, 0.99765892, 0.99876428, 0.49310922],
                             [- 6.27233336, -0.00076596, 0.00046946, 7.27156740, 0.99765950, 0.99876458, 0.49310945],
                             [- 6.27233382, -0.00076577, 0.00046934, 7.27156805, 0.99766008, 0.99876489, 0.49310949],
                             [- 6.27233428, -0.00076558, 0.00046922, 7.27156870, 0.99766065, 0.99876520, 0.49310955],
                             [- 6.27233474, -0.00076539, 0.00046911, 7.27156935, 0.99766123, 0.99876550, 0.49310960],
                             [- 6.27233520, -0.00076520, 0.00046899, 7.27157000, 0.99766181, 0.99876581, 0.49310966],
                             [- 6.27233566, -0.00076501, 0.00046888, 7.27157065, 0.99766239, 0.99876611, 0.49310976],
                             [- 6.27233612, -0.00076482, 0.00046876, 7.27157130, 0.99766297, 0.99876642, 0.49310982],
                             [- 6.27233658, -0.00076464, 0.00046864, 7.27157195, 0.99766355, 0.99876672, 0.49310988],
                             [- 6.27233705, -0.00076445, 0.00046853, 7.27157260, 0.99766412, 0.99876703, 0.49310995],
                             [- 6.27233751, -0.00076426, 0.00046841, 7.27157325, 0.99766470, 0.99876733, 0.49310998],
                             [- 6.27233797, -0.00076407, 0.00046830, 7.27157390, 0.99766528, 0.99876763, 0.49311004],
                             [- 6.27233843, -0.00076388, 0.00046818, 7.27157454, 0.99766585, 0.99876794, 0.49311017],
                             [- 6.27233888, -0.00076369, 0.00046806, 7.27157519, 0.99766643, 0.99876824, 0.49311019],
                             [- 6.27233934, -0.00076350, 0.00046795, 7.27157584, 0.99766701, 0.99876855, 0.49311023],
                             [- 6.27233980, -0.00076332, 0.00046783, 7.27157649, 0.99766758, 0.99876885, 0.49311028],
                             [- 6.27234026, -0.00076313, 0.00046772, 7.27157713, 0.99766816, 0.99876915, 0.49311042],
                             [- 6.27234072, -0.00076294, 0.00046760, 7.27157778, 0.99766873, 0.99876946, 0.49311047],
                             [- 6.27234118, -0.00076275, 0.00046749, 7.27157843, 0.99766931, 0.99876976, 0.49311045],
                             [- 6.27234163, -0.00076256, 0.00046737, 7.27157907, 0.99766988, 0.99877006, 0.49311061],
                             [- 6.27234209, -0.00076238, 0.00046726, 7.27157972, 0.99767046, 0.99877037, 0.49311059],
                             [- 6.27234255, -0.00076219, 0.00046714, 7.27158036, 0.99767103, 0.99877067, 0.49311072],
                             [- 6.27234301, -0.00076200, 0.00046703, 7.27158101, 0.99767160, 0.99877097, 0.49311081],
                             [- 6.27234346, -0.00076181, 0.00046691, 7.27158165, 0.99767218, 0.99877128, 0.49311088],
                             [- 6.27234392, -0.00076163, 0.00046680, 7.27158229, 0.99767275, 0.99877158, 0.49311099],
                             [- 6.27234438, -0.00076144, 0.00046668, 7.27158294, 0.99767332, 0.99877188, 0.49311100],
                             [- 6.27234483, -0.00076125, 0.00046657, 7.27158358, 0.99767389, 0.99877218, 0.49311108],
                             [- 6.27234529, -0.00076106, 0.00046645, 7.27158422, 0.99767447, 0.99877248, 0.49311114],
                             [- 6.27234574, -0.00076088, 0.00046634, 7.27158487, 0.99767504, 0.99877279, 0.49311126],
                             [- 6.27234620, -0.00076069, 0.00046622, 7.27158551, 0.99767561, 0.99877309, 0.49311118],
                             [- 6.27234665, -0.00076050, 0.00046611, 7.27158615, 0.99767618, 0.99877339, 0.49311132],
                             [- 6.27234711, -0.00076032, 0.00046599, 7.27158679, 0.99767675, 0.99877369, 0.49311134],
                             [- 6.27234756, -0.00076013, 0.00046588, 7.27158743, 0.99767732, 0.99877399, 0.49311138],
                             [- 6.27234802, -0.00075994, 0.00046576, 7.27158807, 0.99767789, 0.99877429, 0.49311158],
                             [- 6.27234847, -0.00075976, 0.00046565, 7.27158872, 0.99767846, 0.99877459, 0.49311152],
                             [- 6.27234892, -0.00075957, 0.00046554, 7.27158935, 0.99767903, 0.99877489, 0.49311170],
                             [- 6.27234938, -0.00075938, 0.00046542, 7.27158999, 0.99767960, 0.99877519, 0.49311176],
                             [- 6.27234983, -0.00075920, 0.00046531, 7.27159063, 0.99768017, 0.99877550, 0.49311184],
                             [- 6.27235028, -0.00075901, 0.00046519, 7.27159127, 0.99768074, 0.99877580, 0.49311180],
                             [- 6.27235074, -0.00075882, 0.00046508, 7.27159191, 0.99768131, 0.99877610, 0.49311187],
                             [- 6.27235119, -0.00075864, 0.00046496, 7.27159255, 0.99768188, 0.99877640, 0.49311206],
                             [- 6.27235164, -0.00075845, 0.00046485, 7.27159319, 0.99768245, 0.99877670, 0.49311212],
                             [- 6.27235209, -0.00075827, 0.00046474, 7.27159383, 0.99768302, 0.99877700, 0.49311212],
                             [- 6.27235255, -0.00075808, 0.00046462, 7.27159446, 0.99768358, 0.99877730, 0.49311214],
                             [- 6.27235300, -0.00075790, 0.00046451, 7.27159510, 0.99768415, 0.99877759, 0.49311223],
                             [- 6.27235345, -0.00075771, 0.00046440, 7.27159574, 0.99768472, 0.99877789, 0.49311231],
                             [- 6.27235390, -0.00075753, 0.00046428, 7.27159637, 0.99768529, 0.99877819, 0.49311242],
                             [- 6.27235435, -0.00075734, 0.00046417, 7.27159701, 0.99768585, 0.99877849, 0.49311242],
                             [- 6.27235480, -0.00075715, 0.00046405, 7.27159765, 0.99768642, 0.99877879, 0.49311257],
                             [- 6.27235525, -0.00075697, 0.00046394, 7.27159828, 0.99768699, 0.99877909, 0.49311263],
                             [- 6.27235570, -0.00075678, 0.00046383, 7.27159892, 0.99768755, 0.99877939, 0.49311265],
                             [- 6.27235615, -0.00075660, 0.00046371, 7.27159955, 0.99768812, 0.99877969, 0.49311271],
                             [- 6.27235660, -0.00075641, 0.00046360, 7.27160019, 0.99768868, 0.99877999, 0.49311273],
                             [- 6.27235705, -0.00075623, 0.00046349, 7.27160082, 0.99768925, 0.99878028, 0.49311292],
                             [- 6.27235750, -0.00075605, 0.00046337, 7.27160145, 0.99768981, 0.99878058, 0.49311290],
                             [- 6.27235795, -0.00075586, 0.00046326, 7.27160209, 0.99769038, 0.99878088, 0.49311294],
                             [- 6.27235840, -0.00075568, 0.00046315, 7.27160272, 0.99769094, 0.99878118, 0.49311305],
                             [- 6.27235884, -0.00075549, 0.00046303, 7.27160335, 0.99769150, 0.99878147, 0.49311309],
                             [- 6.27235929, -0.00075531, 0.00046292, 7.27160398, 0.99769207, 0.99878177, 0.49311327],
                             [- 6.27235974, -0.00075512, 0.00046281, 7.27160462, 0.99769263, 0.99878207, 0.49311324],
                             [- 6.27236019, -0.00075494, 0.00046269, 7.27160525, 0.99769319, 0.99878237, 0.49311327],
                             [- 6.27236063, -0.00075475, 0.00046258, 7.27160588, 0.99769376, 0.99878266, 0.49311333],
                             [- 6.27236108, -0.00075457, 0.00046247, 7.27160651, 0.99769432, 0.99878296, 0.49311348],
                             [- 6.27236153, -0.00075439, 0.00046236, 7.27160714, 0.99769488, 0.99878326, 0.49311361],
                             [- 6.27236197, -0.00075420, 0.00046224, 7.27160777, 0.99769544, 0.99878355, 0.49311357],
                             [- 6.27236242, -0.00075402, 0.00046213, 7.27160840, 0.99769601, 0.99878385, 0.49311362],
                             [- 6.27236287, -0.00075384, 0.00046202, 7.27160903, 0.99769657, 0.99878415, 0.49311367],
                             [- 6.27236331, -0.00075365, 0.00046191, 7.27160966, 0.99769713, 0.99878444, 0.49311366],
                             [- 6.27236376, -0.00075347, 0.00046179, 7.27161029, 0.99769769, 0.99878474, 0.49311384],
                             [- 6.27236420, -0.00075329, 0.00046168, 7.27161092, 0.99769825, 0.99878503, 0.49311393],
                             [- 6.27236465, -0.00075310, 0.00046157, 7.27161155, 0.99769881, 0.99878533, 0.49311405],
                             [- 6.27236509, -0.00075292, 0.00046146, 7.27161217, 0.99769937, 0.99878563, 0.49311410],
                             [- 6.27236554, -0.00075274, 0.00046134, 7.27161280, 0.99769993, 0.99878592, 0.49311401],
                             [- 6.27236598, -0.00075255, 0.00046123, 7.27161343, 0.99770049, 0.99878622, 0.49311424],
                             [- 6.27236643, -0.00075237, 0.00046112, 7.27161406, 0.99770105, 0.99878651, 0.49311425],
                             [- 6.27236687, -0.00075219, 0.00046101, 7.27161468, 0.99770161, 0.99878681, 0.49311423],
                             [- 6.27236731, -0.00075200, 0.00046089, 7.27161531, 0.99770217, 0.99878710, 0.49311442],
                             [- 6.27236776, -0.00075182, 0.00046078, 7.27161594, 0.99770272, 0.99878740, 0.49311431],
                             [- 6.27236820, -0.00075164, 0.00046067, 7.27161656, 0.99770328, 0.99878769, 0.49311449],
                             [- 6.27236864, -0.00075146, 0.00046056, 7.27161719, 0.99770384, 0.99878798, 0.49311451],
                             [- 6.27236909, -0.00075127, 0.00046045, 7.27161781, 0.99770440, 0.99878828, 0.49311462],
                             [- 6.27236953, -0.00075109, 0.00046034, 7.27161844, 0.99770496, 0.99878857, 0.49311465],
                             [- 6.27236997, -0.00075091, 0.00046022, 7.27161906, 0.99770551, 0.99878887, 0.49311479],
                             [- 6.27237041, -0.00075073, 0.00046011, 7.27161968, 0.99770607, 0.99878916, 0.49311488],
                             [- 6.27237085, -0.00075055, 0.00046000, 7.27162031, 0.99770663, 0.99878945, 0.49311489],
                             [- 6.27237130, -0.00075036, 0.00045989, 7.27162093, 0.99770718, 0.99878975, 0.49311496],
                             [- 6.27237174, -0.00075018, 0.00045978, 7.27162155, 0.99770774, 0.99879004, 0.49311494],
                             [- 6.27237218, -0.00075000, 0.00045967, 7.27162218, 0.99770829, 0.99879033, 0.49311510],
                             [- 6.27237262, -0.00074982, 0.00045955, 7.27162280, 0.99770885, 0.99879063, 0.49311504],
                             [- 6.27237306, -0.00074964, 0.00045944, 7.27162342, 0.99770941, 0.99879092, 0.49311523],
                             [- 6.27237350, -0.00074946, 0.00045933, 7.27162404, 0.99770996, 0.99879121, 0.49311526],
                             [- 6.27237394, -0.00074927, 0.00045922, 7.27162466, 0.99771051, 0.99879151, 0.49311531],
                             [- 6.27237438, -0.00074909, 0.00045911, 7.27162529, 0.99771107, 0.99879180, 0.49311549],
                             [- 6.27237482, -0.00074891, 0.00045900, 7.27162591, 0.99771162, 0.99879209, 0.49311546],
                             [- 6.27237526, -0.00074873, 0.00045889, 7.27162653, 0.99771218, 0.99879238, 0.49311554],
                             [- 6.27237570, -0.00074855, 0.00045878, 7.27162715, 0.99771273, 0.99879268, 0.49311557],
                             [- 6.27237614, -0.00074837, 0.00045866, 7.27162777, 0.99771328, 0.99879297, 0.49311573],
                             [- 6.27237657, -0.00074819, 0.00045855, 7.27162839, 0.99771384, 0.99879326, 0.49311571],
                             [- 6.27237701, -0.00074801, 0.00045844, 7.27162901, 0.99771439, 0.99879355, 0.49311580],
                             [- 6.27237745, -0.00074783, 0.00045833, 7.27162962, 0.99771494, 0.99879384, 0.49311585],
                             [- 6.27237789, -0.00074765, 0.00045822, 7.27163024, 0.99771549, 0.99879413, 0.49311592],
                             [- 6.27237833, -0.00074747, 0.00045811, 7.27163086, 0.99771605, 0.99879442, 0.49311607],
                             [- 6.27237876, -0.00074728, 0.00045800, 7.27163148, 0.99771660, 0.99879472, 0.49311608],
                             [- 6.27237920, -0.00074710, 0.00045789, 7.27163210, 0.99771715, 0.99879501, 0.49311612],
                             [- 6.27237964, -0.00074692, 0.00045778, 7.27163271, 0.99771770, 0.99879530, 0.49311622],
                             [- 6.27238007, -0.00074674, 0.00045767, 7.27163333, 0.99771825, 0.99879559, 0.49311623],
                             [- 6.27238051, -0.00074656, 0.00045756, 7.27163395, 0.99771880, 0.99879588, 0.49311629],
                             [- 6.27238095, -0.00074638, 0.00045745, 7.27163456, 0.99771935, 0.99879617, 0.49311629],
                             [- 6.27238138, -0.00074620, 0.00045734, 7.27163518, 0.99771990, 0.99879646, 0.49311644],
                             [- 6.27238182, -0.00074602, 0.00045723, 7.27163580, 0.99772045, 0.99879675, 0.49311641],
                             [- 6.27238225, -0.00074584, 0.00045712, 7.27163641, 0.99772100, 0.99879704, 0.49311653],
                             [- 6.27238269, -0.00074566, 0.00045701, 7.27163703, 0.99772155, 0.99879733, 0.49311671],
                             [- 6.27238312, -0.00074548, 0.00045690, 7.27163764, 0.99772210, 0.99879762, 0.49311673],
                             [- 6.27238356, -0.00074531, 0.00045679, 7.27163825, 0.99772265, 0.99879791, 0.49311683],
                             [- 6.27238399, -0.00074513, 0.00045668, 7.27163887, 0.99772320, 0.99879820, 0.49311676],
                             [- 6.27238443, -0.00074495, 0.00045657, 7.27163948, 0.99772375, 0.99879849, 0.49311684],
                             [- 6.27238486, -0.00074477, 0.00045646, 7.27164010, 0.99772429, 0.99879878, 0.49311690],
                             [- 6.27238530, -0.00074459, 0.00045635, 7.27164071, 0.99772484, 0.99879907, 0.49311703],
                             [- 6.27238573, -0.00074441, 0.00045624, 7.27164132, 0.99772539, 0.99879936, 0.49311710],
                             [- 6.27238616, -0.00074423, 0.00045613, 7.27164193, 0.99772594, 0.99879964, 0.49311708],
                             [- 6.27238660, -0.00074405, 0.00045602, 7.27164255, 0.99772648, 0.99879993, 0.49311725],
                             [- 6.27238703, -0.00074387, 0.00045591, 7.27164316, 0.99772703, 0.99880022, 0.49311732],
                             [- 6.27238746, -0.00074369, 0.00045580, 7.27164377, 0.99772758, 0.99880051, 0.49311738],
                             [- 6.27238790, -0.00074352, 0.00045569, 7.27164438, 0.99772812, 0.99880080, 0.49311739],
                             [- 6.27238833, -0.00074334, 0.00045558, 7.27164499, 0.99772867, 0.99880109, 0.49311746],
                             [- 6.27238876, -0.00074316, 0.00045547, 7.27164560, 0.99772921, 0.99880137, 0.49311752],
                             [- 6.27238919, -0.00074298, 0.00045536, 7.27164621, 0.99772976, 0.99880166, 0.49311758],
                             [- 6.27238962, -0.00074280, 0.00045525, 7.27164682, 0.99773030, 0.99880195, 0.49311766],
                             [- 6.27239006, -0.00074262, 0.00045514, 7.27164743, 0.99773085, 0.99880224, 0.49311772],
                             [- 6.27239049, -0.00074245, 0.00045503, 7.27164804, 0.99773139, 0.99880252, 0.49311777],
                             [- 6.27239092, -0.00074227, 0.00045492, 7.27164865, 0.99773194, 0.99880281, 0.49311785],
                             [- 6.27239135, -0.00074209, 0.00045481, 7.27164926, 0.99773248, 0.99880310, 0.49311791],
                             [- 6.27239178, -0.00074191, 0.00045470, 7.27164987, 0.99773303, 0.99880339, 0.49311797],
                             [- 6.27239221, -0.00074173, 0.00045459, 7.27165048, 0.99773357, 0.99880367, 0.49311806],
                             [- 6.27239264, -0.00074156, 0.00045449, 7.27165108, 0.99773411, 0.99880396, 0.49311809],
                             [- 6.27239307, -0.00074138, 0.00045438, 7.27165169, 0.99773466, 0.99880425, 0.49311816],
                             [- 6.27239350, -0.00074120, 0.00045427, 7.27165230, 0.99773520, 0.99880453, 0.49311824],
                             [- 6.27239393, -0.00074102, 0.00045416, 7.27165291, 0.99773574, 0.99880482, 0.49311829],
                             [- 6.27239436, -0.00074085, 0.00045405, 7.27165351, 0.99773628, 0.99880510, 0.49311835],
                             [- 6.27239479, -0.00074067, 0.00045394, 7.27165412, 0.99773683, 0.99880539, 0.49311839],
                             [- 6.27239522, -0.00074049, 0.00045383, 7.27165472, 0.99773737, 0.99880568, 0.49311849],
                             [- 6.27239564, -0.00074031, 0.00045372, 7.27165533, 0.99773791, 0.99880596, 0.49311853],
                             [- 6.27239607, -0.00074014, 0.00045362, 7.27165593, 0.99773845, 0.99880625, 0.49311862],
                             [- 6.27239650, -0.00073996, 0.00045351, 7.27165654, 0.99773899, 0.99880653, 0.49311867],
                             [- 6.27239693, -0.00073978, 0.00045340, 7.27165714, 0.99773953, 0.99880682, 0.49311873],
                             [- 6.27239736, -0.00073961, 0.00045329, 7.27165775, 0.99774007, 0.99880710, 0.49311878],
                             [- 6.27239778, -0.00073943, 0.00045318, 7.27165835, 0.99774061, 0.99880739, 0.49311885],
                             [- 6.27239821, -0.00073925, 0.00045307, 7.27165896, 0.99774115, 0.99880767, 0.49311891],
                             [- 6.27239864, -0.00073908, 0.00045296, 7.27165956, 0.99774169, 0.99880796, 0.49311897],
                             [- 6.27239906, -0.00073890, 0.00045286, 7.27166016, 0.99774223, 0.99880824, 0.49311905],
                             [- 6.27239949, -0.00073872, 0.00045275, 7.27166077, 0.99774277, 0.99880853, 0.49311912],
                             [- 6.27239992, -0.00073855, 0.00045264, 7.27166137, 0.99774331, 0.99880881, 0.49311918],
                             [- 6.27240034, -0.00073837, 0.00045253, 7.27166197, 0.99774385, 0.99880910, 0.49311924],
                             [- 6.27240077, -0.00073820, 0.00045242, 7.27166257, 0.99774439, 0.99880938, 0.49311929],
                             [- 6.27240119, -0.00073802, 0.00045232, 7.27166317, 0.99774492, 0.99880966, 0.49311935],
                             [- 6.27240162, -0.00073784, 0.00045221, 7.27166378, 0.99774546, 0.99880995, 0.49311941],
                             [- 6.27240205, -0.00073767, 0.00045210, 7.27166438, 0.99774600, 0.99881023, 0.49311949],
                             [- 6.27240247, -0.00073749, 0.00045199, 7.27166498, 0.99774654, 0.99881051, 0.49311956],
                             [- 6.27240289, -0.00073732, 0.00045188, 7.27166558, 0.99774707, 0.99881080, 0.49311961],
                             [- 6.27240332, -0.00073714, 0.00045178, 7.27166618, 0.99774761, 0.99881108, 0.49311968],
                             [- 6.27240374, -0.00073697, 0.00045167, 7.27166678, 0.99774815, 0.99881136, 0.49311975],
                             [- 6.27240417, -0.00073679, 0.00045156, 7.27166738, 0.99774868, 0.99881165, 0.49311980],
                             [- 6.27240459, -0.00073662, 0.00045145, 7.27166798, 0.99774922, 0.99881193, 0.49311987],
                             [- 6.27240502, -0.00073644, 0.00045135, 7.27166858, 0.99774976, 0.99881221, 0.49311992],
                             [- 6.27240544, -0.00073626, 0.00045124, 7.27166917, 0.99775029, 0.99881250, 0.49312000],
                             [- 6.27240586, -0.00073609, 0.00045113, 7.27166977, 0.99775083, 0.99881278, 0.49312006],
                             [- 6.27240628, -0.00073591, 0.00045102, 7.27167037, 0.99775136, 0.99881306, 0.49312011],
                             [- 6.27240671, -0.00073574, 0.00045092, 7.27167097, 0.99775190, 0.99881334, 0.49312017],
                             [- 6.27240713, -0.00073556, 0.00045081, 7.27167157, 0.99775243, 0.99881362, 0.49312022],
                             [- 6.27240755, -0.00073539, 0.00045070, 7.27167216, 0.99775297, 0.99881391, 0.49312031],
                             [- 6.27240797, -0.00073522, 0.00045060, 7.27167276, 0.99775350, 0.99881419, 0.49312033],
                             [- 6.27240840, -0.00073504, 0.00045049, 7.27167336, 0.99775403, 0.99881447, 0.49312043],
                             [- 6.27240882, -0.00073487, 0.00045038, 7.27167395, 0.99775457, 0.99881475, 0.49312049],
                             [- 6.27240924, -0.00073469, 0.00045027, 7.27167455, 0.99775510, 0.99881503, 0.49312054],
                             [- 6.27240966, -0.00073452, 0.00045017, 7.27167514, 0.99775563, 0.99881531, 0.49312061],
                             [- 6.27241008, -0.00073434, 0.00045006, 7.27167574, 0.99775617, 0.99881560, 0.49312068],
                             [- 6.27241050, -0.00073417, 0.00044995, 7.27167633, 0.99775670, 0.99881588, 0.49312074],
                             [- 6.27241092, -0.00073400, 0.00044985, 7.27167693, 0.99775723, 0.99881616, 0.49312080],
                             [- 6.27241134, -0.00073382, 0.00044974, 7.27167752, 0.99775776, 0.99881644, 0.49312087],
                             [- 6.27241176, -0.00073365, 0.00044963, 7.27167812, 0.99775829, 0.99881672, 0.49312093],
                             [- 6.27241218, -0.00073347, 0.00044953, 7.27167871, 0.99775883, 0.99881700, 0.49312099],
                             [- 6.27241260, -0.00073330, 0.00044942, 7.27167930, 0.99775936, 0.99881728, 0.49312106],
                             [- 6.27241302, -0.00073313, 0.00044931, 7.27167990, 0.99775989, 0.99881756, 0.49312110],
                             [- 6.27241344, -0.00073295, 0.00044921, 7.27168049, 0.99776042, 0.99881784, 0.49312117],
                             [- 6.27241386, -0.00073278, 0.00044910, 7.27168108, 0.99776095, 0.99881812, 0.49312124],
                             [- 6.27241428, -0.00073261, 0.00044899, 7.27168167, 0.99776148, 0.99881840, 0.49312129],
                             [- 6.27241470, -0.00073243, 0.00044889, 7.27168227, 0.99776201, 0.99881868, 0.49312137],
                             [- 6.27241512, -0.00073226, 0.00044878, 7.27168286, 0.99776254, 0.99881896, 0.49312141],
                             [- 6.27241554, -0.00073209, 0.00044868, 7.27168345, 0.99776307, 0.99881924, 0.49312147],
                             [- 6.27241595, -0.00073191, 0.00044857, 7.27168404, 0.99776360, 0.99881952, 0.49312156],
                             [- 6.27241637, -0.00073174, 0.00044846, 7.27168463, 0.99776413, 0.99881980, 0.49312161],
                             [- 6.27241679, -0.00073157, 0.00044836, 7.27168522, 0.99776466, 0.99882008, 0.49312166],
                             [- 6.27241721, -0.00073139, 0.00044825, 7.27168581, 0.99776518, 0.99882035, 0.49312172],
                             [- 6.27241762, -0.00073122, 0.00044815, 7.27168640, 0.99776571, 0.99882063, 0.49312178],
                             [- 6.27241804, -0.00073105, 0.00044804, 7.27168699, 0.99776624, 0.99882091, 0.49312185],
                             [- 6.27241846, -0.00073088, 0.00044793, 7.27168758, 0.99776677, 0.99882119, 0.49312193],
                             [- 6.27241887, -0.00073070, 0.00044783, 7.27168817, 0.99776730, 0.99882147, 0.49312197],
                             [- 6.27241929, -0.00073053, 0.00044772, 7.27168876, 0.99776782, 0.99882175, 0.49312203],
                             [- 6.27241971, -0.00073036, 0.00044762, 7.27168935, 0.99776835, 0.99882202, 0.49312211],
                             [- 6.27242012, -0.00073019, 0.00044751, 7.27168994, 0.99776888, 0.99882230, 0.49312215],
                             [- 6.27242054, -0.00073001, 0.00044741, 7.27169052, 0.99776940, 0.99882258, 0.49312222],
                             [- 6.27242095, -0.00072984, 0.00044730, 7.27169111, 0.99776993, 0.99882286, 0.49312230],
                             [- 6.27242137, -0.00072967, 0.00044719, 7.27169170, 0.99777045, 0.99882314, 0.49312233],
                             [- 6.27242178, -0.00072950, 0.00044709, 7.27169229, 0.99777098, 0.99882341, 0.49312241],
                             [- 6.27242220, -0.00072933, 0.00044698, 7.27169287, 0.99777151, 0.99882369, 0.49312246],
                             [- 6.27242261, -0.00072915, 0.00044688, 7.27169346, 0.99777203, 0.99882397, 0.49312253],
                             [- 6.27242303, -0.00072898, 0.00044677, 7.27169405, 0.99777256, 0.99882424, 0.49312260],
                             [- 6.27242344, -0.00072881, 0.00044667, 7.27169463, 0.99777308, 0.99882452, 0.49312265],
                             [- 6.27242386, -0.00072864, 0.00044656, 7.27169522, 0.99777361, 0.99882480, 0.49312272],
                             [- 6.27242427, -0.00072847, 0.00044646, 7.27169580, 0.99777413, 0.99882508, 0.49312276],
                             [- 6.27242468, -0.00072830, 0.00044635, 7.27169639, 0.99777465, 0.99882535, 0.49312285],
                             [- 6.27242510, -0.00072813, 0.00044625, 7.27169697, 0.99777518, 0.99882563, 0.49312291],
                             [- 6.27242551, -0.00072795, 0.00044614, 7.27169756, 0.99777570, 0.99882590, 0.49312295],
                             [- 6.27242592, -0.00072778, 0.00044604, 7.27169814, 0.99777622, 0.99882618, 0.49312303],
                             [- 6.27242634, -0.00072761, 0.00044593, 7.27169872, 0.99777675, 0.99882646, 0.49312307],
                             [- 6.27242675, -0.00072744, 0.00044583, 7.27169931, 0.99777727, 0.99882673, 0.49312312],
                             [- 6.27242716, -0.00072727, 0.00044572, 7.27169989, 0.99777779, 0.99882701, 0.49312319],
                             [- 6.27242757, -0.00072710, 0.00044562, 7.27170047, 0.99777831, 0.99882728, 0.49312326],
                             [- 6.27242798, -0.00072693, 0.00044551, 7.27170106, 0.99777884, 0.99882756, 0.49312335],
                             [- 6.27242840, -0.00072676, 0.00044541, 7.27170164, 0.99777936, 0.99882783, 0.49312338],
                             [- 6.27242881, -0.00072659, 0.00044530, 7.27170222, 0.99777988, 0.99882811, 0.49312345],
                             [- 6.27242922, -0.00072642, 0.00044520, 7.27170280, 0.99778040, 0.99882838, 0.49312349],
                             [- 6.27242963, -0.00072625, 0.00044509, 7.27170338, 0.99778092, 0.99882866, 0.49312358],
                             [- 6.27243004, -0.00072608, 0.00044499, 7.27170397, 0.99778144, 0.99882893, 0.49312364],
                             [- 6.27243045, -0.00072591, 0.00044488, 7.27170455, 0.99778196, 0.99882921, 0.49312368],
                             [- 6.27243086, -0.00072574, 0.00044478, 7.27170513, 0.99778248, 0.99882948, 0.49312376],
                             [- 6.27243127, -0.00072557, 0.00044468, 7.27170571, 0.99778300, 0.99882976, 0.49312381],
                             [- 6.27243168, -0.00072540, 0.00044457, 7.27170629, 0.99778352, 0.99883003, 0.49312388],
                             [- 6.27243209, -0.00072523, 0.00044447, 7.27170687, 0.99778404, 0.99883031, 0.49312394],
                             [- 6.27243250, -0.00072506, 0.00044436, 7.27170745, 0.99778456, 0.99883058, 0.49312397],
                             [- 6.27243291, -0.00072489, 0.00044426, 7.27170803, 0.99778508, 0.99883086, 0.49312405],
                             [- 6.27243332, -0.00072472, 0.00044416, 7.27170860, 0.99778560, 0.99883113, 0.49312410],
                             [- 6.27243373, -0.00072455, 0.00044405, 7.27170918, 0.99778612, 0.99883140, 0.49312419],
                             [- 6.27243414, -0.00072438, 0.00044395, 7.27170976, 0.99778664, 0.99883168, 0.49312424],
                             [- 6.27243455, -0.00072421, 0.00044384, 7.27171034, 0.99778716, 0.99883195, 0.49312429],
                             [- 6.27243496, -0.00072404, 0.00044374, 7.27171092, 0.99778767, 0.99883222, 0.49312436],
                             [- 6.27243536, -0.00072387, 0.00044364, 7.27171150, 0.99778819, 0.99883250, 0.49312441],
                             [- 6.27243577, -0.00072370, 0.00044353, 7.27171207, 0.99778871, 0.99883277, 0.49312447],
                             [- 6.27243618, -0.00072353, 0.00044343, 7.27171265, 0.99778923, 0.99883304, 0.49312454],
                             [- 6.27243659, -0.00072336, 0.00044332, 7.27171323, 0.99778974, 0.99883331, 0.49312460],
                             [- 6.27243700, -0.00072319, 0.00044322, 7.27171380, 0.99779026, 0.99883359, 0.49312464],
                             [- 6.27243740, -0.00072302, 0.00044312, 7.27171438, 0.99779078, 0.99883386, 0.49312471],
                             [- 6.27243781, -0.00072285, 0.00044301, 7.27171495, 0.99779129, 0.99883413, 0.49312476],
                             [- 6.27243822, -0.00072269, 0.00044291, 7.27171553, 0.99779181, 0.99883440, 0.49312483],
                             [- 6.27243862, -0.00072252, 0.00044281, 7.27171611, 0.99779232, 0.99883468, 0.49312489],
                             [- 6.27243903, -0.00072235, 0.00044270, 7.27171668, 0.99779284, 0.99883495, 0.49312495],
                             [- 6.27243944, -0.00072218, 0.00044260, 7.27171726, 0.99779336, 0.99883522, 0.49312500],
                             [- 6.27243984, -0.00072201, 0.00044250, 7.27171783, 0.99779387, 0.99883549, 0.49312508],
                             [- 6.27244025, -0.00072184, 0.00044239, 7.27171840, 0.99779438, 0.99883576, 0.49312514],
                             [- 6.27244065, -0.00072167, 0.00044229, 7.27171898, 0.99779490, 0.99883604, 0.49312518],
                             [- 6.27244106, -0.00072151, 0.00044219, 7.27171955, 0.99779541, 0.99883631, 0.49312525],
                             [- 6.27244146, -0.00072134, 0.00044208, 7.27172012, 0.99779593, 0.99883658, 0.49312531],
                             [- 6.27244187, -0.00072117, 0.00044198, 7.27172070, 0.99779644, 0.99883685, 0.49312537],
                             [- 6.27244227, -0.00072100, 0.00044188, 7.27172127, 0.99779696, 0.99883712, 0.49312544],
                             [- 6.27244268, -0.00072083, 0.00044177, 7.27172184, 0.99779747, 0.99883739, 0.49312550],
                             [- 6.27244308, -0.00072067, 0.00044167, 7.27172242, 0.99779798, 0.99883766, 0.49312555],
                             [- 6.27244349, -0.00072050, 0.00044157, 7.27172299, 0.99779849, 0.99883793, 0.49312563],
                             [- 6.27244389, -0.00072033, 0.00044147, 7.27172356, 0.99779901, 0.99883820, 0.49312567],
                             [- 6.27244429, -0.00072016, 0.00044136, 7.27172413, 0.99779952, 0.99883847, 0.49312573],
                             [- 6.27244470, -0.00072000, 0.00044126, 7.27172470, 0.99780003, 0.99883874, 0.49312578],
                             [- 6.27244510, -0.00071983, 0.00044116, 7.27172527, 0.99780054, 0.99883901, 0.49312585],
                             [- 6.27244550, -0.00071966, 0.00044105, 7.27172584, 0.99780105, 0.99883928, 0.49312592],
                             [- 6.27244591, -0.00071949, 0.00044095, 7.27172641, 0.99780157, 0.99883955, 0.49312598],
                             [- 6.27244631, -0.00071933, 0.00044085, 7.27172698, 0.99780208, 0.99883982, 0.49312604],
                             [- 6.27244671, -0.00071916, 0.00044075, 7.27172755, 0.99780259, 0.99884009, 0.49312608],
                             [- 6.27244712, -0.00071899, 0.00044064, 7.27172812, 0.99780310, 0.99884036, 0.49312614],
                             [- 6.27244752, -0.00071883, 0.00044054, 7.27172869, 0.99780361, 0.99884063, 0.49312622],
                             [- 6.27244792, -0.00071866, 0.00044044, 7.27172926, 0.99780412, 0.99884090, 0.49312626],
                             [- 6.27244832, -0.00071849, 0.00044034, 7.27172983, 0.99780463, 0.99884117, 0.49312632],
                             [- 6.27244872, -0.00071833, 0.00044024, 7.27173040, 0.99780514, 0.99884144, 0.49312638],
                             [- 6.27244912, -0.00071816, 0.00044013, 7.27173097, 0.99780565, 0.99884171, 0.49312645],
                             [- 6.27244953, -0.00071799, 0.00044003, 7.27173153, 0.99780616, 0.99884198, 0.49312649],
                             [- 6.27244993, -0.00071783, 0.00043993, 7.27173210, 0.99780667, 0.99884225, 0.49312657],
                             [- 6.27245033, -0.00071766, 0.00043983, 7.27173267, 0.99780718, 0.99884251, 0.49312664],
                             [- 6.27245073, -0.00071749, 0.00043972, 7.27173323, 0.99780768, 0.99884278, 0.49312669],
                             [- 6.27245113, -0.00071733, 0.00043962, 7.27173380, 0.99780819, 0.99884305, 0.49312674],
                             [- 6.27245153, -0.00071716, 0.00043952, 7.27173437, 0.99780870, 0.99884332, 0.49312680],
                             [- 6.27245193, -0.00071699, 0.00043942, 7.27173493, 0.99780921, 0.99884359, 0.49312685],
                             [- 6.27245233, -0.00071683, 0.00043932, 7.27173550, 0.99780972, 0.99884385, 0.49312691],
                             [- 6.27245273, -0.00071666, 0.00043922, 7.27173607, 0.99781022, 0.99884412, 0.49312701],
                             [- 6.27245313, -0.00071650, 0.00043911, 7.27173663, 0.99781073, 0.99884439, 0.49312703],
                             [- 6.27245353, -0.00071633, 0.00043901, 7.27173720, 0.99781124, 0.99884466, 0.49312710],
                             [- 6.27245393, -0.00071617, 0.00043891, 7.27173776, 0.99781174, 0.99884492, 0.49312716],
                             [- 6.27245432, -0.00071600, 0.00043881, 7.27173833, 0.99781225, 0.99884519, 0.49312722],
                             [- 6.27245472, -0.00071583, 0.00043871, 7.27173889, 0.99781276, 0.99884546, 0.49312728],
                             [- 6.27245512, -0.00071567, 0.00043861, 7.27173945, 0.99781326, 0.99884573, 0.49312736],
                             [- 6.27245552, -0.00071550, 0.00043850, 7.27174002, 0.99781377, 0.99884599, 0.49312741],
                             [- 6.27245592, -0.00071534, 0.00043840, 7.27174058, 0.99781427, 0.99884626, 0.49312745],
                             [- 6.27245632, -0.00071517, 0.00043830, 7.27174114, 0.99781478, 0.99884653, 0.49312752],
                             [- 6.27245671, -0.00071501, 0.00043820, 7.27174171, 0.99781528, 0.99884679, 0.49312756],
                             [- 6.27245711, -0.00071484, 0.00043810, 7.27174227, 0.99781579, 0.99884706, 0.49312763],
                             [- 6.27245751, -0.00071468, 0.00043800, 7.27174283, 0.99781629, 0.99884733, 0.49312770],
                             [- 6.27245791, -0.00071451, 0.00043790, 7.27174339, 0.99781680, 0.99884759, 0.49312773],
                             [- 6.27245830, -0.00071435, 0.00043779, 7.27174396, 0.99781730, 0.99884786, 0.49312781],
                             [- 6.27245870, -0.00071418, 0.00043769, 7.27174452, 0.99781781, 0.99884812, 0.49312788],
                             [- 6.27245910, -0.00071402, 0.00043759, 7.27174508, 0.99781831, 0.99884839, 0.49312794],
                             [- 6.27245949, -0.00071385, 0.00043749, 7.27174564, 0.99781881, 0.99884866, 0.49312798],
                             [- 6.27245989, -0.00071369, 0.00043739, 7.27174620, 0.99781932, 0.99884892, 0.49312802],
                             [- 6.27246028, -0.00071352, 0.00043729, 7.27174676, 0.99781982, 0.99884919, 0.49312810],
                             [- 6.27246068, -0.00071336, 0.00043719, 7.27174732, 0.99782032, 0.99884945, 0.49312816],
                             [- 6.27246108, -0.00071320, 0.00043709, 7.27174788, 0.99782083, 0.99884972, 0.49312820],
                             [- 6.27246147, -0.00071303, 0.00043699, 7.27174844, 0.99782133, 0.99884998, 0.49312828],
                             [- 6.27246187, -0.00071287, 0.00043689, 7.27174900, 0.99782183, 0.99885025, 0.49312834],
                             [- 6.27246226, -0.00071270, 0.00043679, 7.27174956, 0.99782233, 0.99885051, 0.49312839],
                             [- 6.27246266, -0.00071254, 0.00043669, 7.27175012, 0.99782283, 0.99885078, 0.49312847],
                             [- 6.27246305, -0.00071237, 0.00043658, 7.27175068, 0.99782333, 0.99885104, 0.49312849],
                             [- 6.27246345, -0.00071221, 0.00043648, 7.27175123, 0.99782384, 0.99885131, 0.49312857],
                             [- 6.27246384, -0.00071205, 0.00043638, 7.27175179, 0.99782434, 0.99885157, 0.49312862],
                             [- 6.27246423, -0.00071188, 0.00043628, 7.27175235, 0.99782484, 0.99885183, 0.49312870],
                             [- 6.27246463, -0.00071172, 0.00043618, 7.27175291, 0.99782534, 0.99885210, 0.49312875],
                             [- 6.27246502, -0.00071156, 0.00043608, 7.27175347, 0.99782584, 0.99885236, 0.49312878],
                             [- 6.27246541, -0.00071139, 0.00043598, 7.27175402, 0.99782634, 0.99885263, 0.49312886],
                             [- 6.27246581, -0.00071123, 0.00043588, 7.27175458, 0.99782684, 0.99885289, 0.49312892],
                             [- 6.27246620, -0.00071107, 0.00043578, 7.27175514, 0.99782734, 0.99885315, 0.49312899],
                             [- 6.27246659, -0.00071090, 0.00043568, 7.27175569, 0.99782784, 0.99885342, 0.49312906],
                             [- 6.27246699, -0.00071074, 0.00043558, 7.27175625, 0.99782833, 0.99885368, 0.49312908],
                             [- 6.27246738, -0.00071058, 0.00043548, 7.27175680, 0.99782883, 0.99885394, 0.49312914],
                             [- 6.27246777, -0.00071041, 0.00043538, 7.27175736, 0.99782933, 0.99885421, 0.49312922],
                             [- 6.27246816, -0.00071025, 0.00043528, 7.27175791, 0.99782983, 0.99885447, 0.49312926],
                             [- 6.27246856, -0.00071009, 0.00043518, 7.27175847, 0.99783033, 0.99885473, 0.49312934],
                             [- 6.27246895, -0.00070992, 0.00043508, 7.27175902, 0.99783083, 0.99885499, 0.49312938],
                             [- 6.27246934, -0.00070976, 0.00043498, 7.27175958, 0.99783132, 0.99885526, 0.49312944],
                             [- 6.27246973, -0.00070960, 0.00043488, 7.27176013, 0.99783182, 0.99885552, 0.49312948],
                             [- 6.27247012, -0.00070944, 0.00043478, 7.27176069, 0.99783232, 0.99885578, 0.49312957],
                             [- 6.27247051, -0.00070927, 0.00043468, 7.27176124, 0.99783282, 0.99885604, 0.49312962],
                             [- 6.27247090, -0.00070911, 0.00043458, 7.27176179, 0.99783331, 0.99885631, 0.49312968],
                             [- 6.27247129, -0.00070895, 0.00043448, 7.27176235, 0.99783381, 0.99885657, 0.49312972],
                             [- 6.27247168, -0.00070879, 0.00043438, 7.27176290, 0.99783431, 0.99885683, 0.49312980],
                             [- 6.27247207, -0.00070862, 0.00043428, 7.27176345, 0.99783480, 0.99885709, 0.49312985],
                             [- 6.27247246, -0.00070846, 0.00043418, 7.27176400, 0.99783530, 0.99885735, 0.49312992],
                             [- 6.27247285, -0.00070830, 0.00043409, 7.27176456, 0.99783579, 0.99885762, 0.49312999],
                             [- 6.27247324, -0.00070814, 0.00043399, 7.27176511, 0.99783629, 0.99885788, 0.49313003],
                             [- 6.27247363, -0.00070798, 0.00043389, 7.27176566, 0.99783678, 0.99885814, 0.49313009],
                             [- 6.27247402, -0.00070781, 0.00043379, 7.27176621, 0.99783728, 0.99885840, 0.49313014],
                             [- 6.27247441, -0.00070765, 0.00043369, 7.27176676, 0.99783777, 0.99885866, 0.49313021],
                             [- 6.27247480, -0.00070749, 0.00043359, 7.27176731, 0.99783827, 0.99885892, 0.49313025],
                             [- 6.27247519, -0.00070733, 0.00043349, 7.27176786, 0.99783876, 0.99885918, 0.49313031],
                             [- 6.27247558, -0.00070717, 0.00043339, 7.27176841, 0.99783926, 0.99885944, 0.49313036],
                             [- 6.27247597, -0.00070701, 0.00043329, 7.27176896, 0.99783975, 0.99885970, 0.49313043],
                             [- 6.27247636, -0.00070684, 0.00043319, 7.27176951, 0.99784024, 0.99885996, 0.49313047],
                             [- 6.27247674, -0.00070668, 0.00043309, 7.27177006, 0.99784074, 0.99886022, 0.49313056],
                             [- 6.27247713, -0.00070652, 0.00043299, 7.27177061, 0.99784123, 0.99886048, 0.49313062],
                             [- 6.27247752, -0.00070636, 0.00043290, 7.27177116, 0.99784172, 0.99886074, 0.49313067],
                             [- 6.27247791, -0.00070620, 0.00043280, 7.27177171, 0.99784222, 0.99886100, 0.49313072],
                             [- 6.27247829, -0.00070604, 0.00043270, 7.27177226, 0.99784271, 0.99886126, 0.49313077],
                             [- 6.27247868, -0.00070588, 0.00043260, 7.27177280, 0.99784320, 0.99886152, 0.49313084],
                             [- 6.27247907, -0.00070572, 0.00043250, 7.27177335, 0.99784369, 0.99886178, 0.49313089],
                             [- 6.27247945, -0.00070556, 0.00043240, 7.27177390, 0.99784418, 0.99886204, 0.49313094],
                             [- 6.27247984, -0.00070539, 0.00043230, 7.27177445, 0.99784468, 0.99886230, 0.49313101],
                             [- 6.27248023, -0.00070523, 0.00043220, 7.27177499, 0.99784517, 0.99886256, 0.49313105],
                             [- 6.27248061, -0.00070507, 0.00043211, 7.27177554, 0.99784566, 0.99886282, 0.49313111],
                             [- 6.27248100, -0.00070491, 0.00043201, 7.27177609, 0.99784615, 0.99886308, 0.49313117],
                             [- 6.27248139, -0.00070475, 0.00043191, 7.27177663, 0.99784664, 0.99886334, 0.49313123],
                             [- 6.27248177, -0.00070459, 0.00043181, 7.27177718, 0.99784713, 0.99886360, 0.49313129],
                             [- 6.27248216, -0.00070443, 0.00043171, 7.27177773, 0.99784762, 0.99886386, 0.49313133],
                             [- 6.27248254, -0.00070427, 0.00043161, 7.27177827, 0.99784811, 0.99886411, 0.49313142],
                             [- 6.27248293, -0.00070411, 0.00043152, 7.27177882, 0.99784860, 0.99886437, 0.49313147],
                             [- 6.27248331, -0.00070395, 0.00043142, 7.27177936, 0.99784909, 0.99886463, 0.49313152],
                             [- 6.27248370, -0.00070379, 0.00043132, 7.27177991, 0.99784958, 0.99886489, 0.49313157],
                             [- 6.27248408, -0.00070363, 0.00043122, 7.27178045, 0.99785007, 0.99886515, 0.49313161],
                             [- 6.27248447, -0.00070347, 0.00043112, 7.27178099, 0.99785056, 0.99886541, 0.49313168],
                             [- 6.27248485, -0.00070331, 0.00043103, 7.27178154, 0.99785104, 0.99886566, 0.49313174],
                             [- 6.27248523, -0.00070315, 0.00043093, 7.27178208, 0.99785153, 0.99886592, 0.49313183],
                             [- 6.27248562, -0.00070299, 0.00043083, 7.27178263, 0.99785202, 0.99886618, 0.49313186],
                             [- 6.27248600, -0.00070283, 0.00043073, 7.27178317, 0.99785251, 0.99886644, 0.49313191],
                             [- 6.27248638, -0.00070267, 0.00043063, 7.27178371, 0.99785300, 0.99886669, 0.49313198],
                             [- 6.27248677, -0.00070251, 0.00043054, 7.27178425, 0.99785348, 0.99886695, 0.49313204],
                             [- 6.27248715, -0.00070235, 0.00043044, 7.27178480, 0.99785397, 0.99886721, 0.49313208],
                             [- 6.27248753, -0.00070219, 0.00043034, 7.27178534, 0.99785446, 0.99886747, 0.49313214],
                             [- 6.27248792, -0.00070203, 0.00043024, 7.27178588, 0.99785495, 0.99886772, 0.49313220],
                             [- 6.27248830, -0.00070188, 0.00043014, 7.27178642, 0.99785543, 0.99886798, 0.49313224],
                             [- 6.27248868, -0.00070172, 0.00043005, 7.27178696, 0.99785592, 0.99886824, 0.49313233],
                             [- 6.27248906, -0.00070156, 0.00042995, 7.27178751, 0.99785641, 0.99886849, 0.49313235],
                             [- 6.27248945, -0.00070140, 0.00042985, 7.27178805, 0.99785689, 0.99886875, 0.49313243],
                             [- 6.27248983, -0.00070124, 0.00042975, 7.27178859, 0.99785738, 0.99886901, 0.49313250],
                             [- 6.27249021, -0.00070108, 0.00042966, 7.27178913, 0.99785786, 0.99886926, 0.49313254],
                             [- 6.27249059, -0.00070092, 0.00042956, 7.27178967, 0.99785835, 0.99886952, 0.49313259],
                             [- 6.27249097, -0.00070076, 0.00042946, 7.27179021, 0.99785883, 0.99886977, 0.49313265],
                             [- 6.27249135, -0.00070060, 0.00042937, 7.27179075, 0.99785932, 0.99887003, 0.49313270],
                             [- 6.27249173, -0.00070045, 0.00042927, 7.27179129, 0.99785980, 0.99887029, 0.49313277],
                             [- 6.27249211, -0.00070029, 0.00042917, 7.27179183, 0.99786029, 0.99887054, 0.49313282],
                             [- 6.27249250, -0.00070013, 0.00042907, 7.27179237, 0.99786077, 0.99887080, 0.49313289],
                             [- 6.27249288, -0.00069997, 0.00042898, 7.27179290, 0.99786126, 0.99887105, 0.49313291],
                             [- 6.27249326, -0.00069981, 0.00042888, 7.27179344, 0.99786174, 0.99887131, 0.49313301],
                             [- 6.27249364, -0.00069965, 0.00042878, 7.27179398, 0.99786222, 0.99887156, 0.49313305],
                             [- 6.27249402, -0.00069950, 0.00042869, 7.27179452, 0.99786271, 0.99887182, 0.49313311],
                             [- 6.27249440, -0.00069934, 0.00042859, 7.27179506, 0.99786319, 0.99887207, 0.49313316],
                             [- 6.27249477, -0.00069918, 0.00042849, 7.27179559, 0.99786367, 0.99887233, 0.49313322],
                             [- 6.27249515, -0.00069902, 0.00042840, 7.27179613, 0.99786415, 0.99887258, 0.49313329],
                             [- 6.27249553, -0.00069887, 0.00042830, 7.27179667, 0.99786464, 0.99887284, 0.49313333],
                             [- 6.27249591, -0.00069871, 0.00042820, 7.27179720, 0.99786512, 0.99887309, 0.49313340],
                             [- 6.27249629, -0.00069855, 0.00042810, 7.27179774, 0.99786560, 0.99887335, 0.49313343],
                             [- 6.27249667, -0.00069839, 0.00042801, 7.27179828, 0.99786608, 0.99887360, 0.49313349],
                             [- 6.27249705, -0.00069823, 0.00042791, 7.27179881, 0.99786656, 0.99887385, 0.49313355],
                             [- 6.27249743, -0.00069808, 0.00042781, 7.27179935, 0.99786705, 0.99887411, 0.49313361],
                             [- 6.27249780, -0.00069792, 0.00042772, 7.27179988, 0.99786753, 0.99887436, 0.49313365],
                             [- 6.27249818, -0.00069776, 0.00042762, 7.27180042, 0.99786801, 0.99887462, 0.49313373],
                             [- 6.27249856, -0.00069761, 0.00042753, 7.27180095, 0.99786849, 0.99887487, 0.49313378],
                             [- 6.27249894, -0.00069745, 0.00042743, 7.27180149, 0.99786897, 0.99887512, 0.49313382],
                             [- 6.27249931, -0.00069729, 0.00042733, 7.27180202, 0.99786945, 0.99887538, 0.49313387],
                             [- 6.27249969, -0.00069713, 0.00042724, 7.27180256, 0.99786993, 0.99887563, 0.49313395],
                             [- 6.27250007, -0.00069698, 0.00042714, 7.27180309, 0.99787041, 0.99887588, 0.49313401],
                             [- 6.27250045, -0.00069682, 0.00042704, 7.27180363, 0.99787089, 0.99887614, 0.49313405],
                             [- 6.27250082, -0.00069666, 0.00042695, 7.27180416, 0.99787137, 0.99887639, 0.49313411],
                             [- 6.27250120, -0.00069651, 0.00042685, 7.27180469, 0.99787185, 0.99887664, 0.49313417],
                             [- 6.27250158, -0.00069635, 0.00042676, 7.27180523, 0.99787233, 0.99887689, 0.49313425],
                             [- 6.27250195, -0.00069619, 0.00042666, 7.27180576, 0.99787281, 0.99887715, 0.49313428],
                             [- 6.27250233, -0.00069604, 0.00042656, 7.27180629, 0.99787328, 0.99887740, 0.49313433],
                             [- 6.27250270, -0.00069588, 0.00042647, 7.27180682, 0.99787376, 0.99887765, 0.49313436],
                             [- 6.27250308, -0.00069572, 0.00042637, 7.27180736, 0.99787424, 0.99887790, 0.49313447],
                             [- 6.27250345, -0.00069557, 0.00042628, 7.27180789, 0.99787472, 0.99887816, 0.49313453],
                             [- 6.27250383, -0.00069541, 0.00042618, 7.27180842, 0.99787520, 0.99887841, 0.49313456],
                             [- 6.27250420, -0.00069525, 0.00042608, 7.27180895, 0.99787567, 0.99887866, 0.49313463],
                             [- 6.27250458, -0.00069510, 0.00042599, 7.27180948, 0.99787615, 0.99887891, 0.49313466],
                             [- 6.27250495, -0.00069494, 0.00042589, 7.27181001, 0.99787663, 0.99887916, 0.49313471],
                             [- 6.27250533, -0.00069479, 0.00042580, 7.27181054, 0.99787711, 0.99887942, 0.49313478],
                             [- 6.27250570, -0.00069463, 0.00042570, 7.27181107, 0.99787758, 0.99887967, 0.49313484],
                             [- 6.27250608, -0.00069447, 0.00042561, 7.27181160, 0.99787806, 0.99887992, 0.49313489],
                             [- 6.27250645, -0.00069432, 0.00042551, 7.27181213, 0.99787854, 0.99888017, 0.49313499],
                             [- 6.27250683, -0.00069416, 0.00042541, 7.27181266, 0.99787901, 0.99888042, 0.49313500],
                             [- 6.27250720, -0.00069401, 0.00042532, 7.27181319, 0.99787949, 0.99888067, 0.49313506],
                             [- 6.27250757, -0.00069385, 0.00042522, 7.27181372, 0.99787996, 0.99888092, 0.49313510],
                             [- 6.27250795, -0.00069370, 0.00042513, 7.27181425, 0.99788044, 0.99888117, 0.49313518],
                             [- 6.27250832, -0.00069354, 0.00042503, 7.27181478, 0.99788091, 0.99888143, 0.49313523],
                             [- 6.27250869, -0.00069339, 0.00042494, 7.27181531, 0.99788139, 0.99888168, 0.49313528],
                             [- 6.27250906, -0.00069323, 0.00042484, 7.27181583, 0.99788186, 0.99888193, 0.49313532],
                             [- 6.27250944, -0.00069308, 0.00042475, 7.27181636, 0.99788234, 0.99888218, 0.49313537],
                             [- 6.27250981, -0.00069292, 0.00042465, 7.27181689, 0.99788281, 0.99888243, 0.49313545],
                             [- 6.27251018, -0.00069277, 0.00042456, 7.27181742, 0.99788329, 0.99888268, 0.49313551],
                             [- 6.27251055, -0.00069261, 0.00042446, 7.27181794, 0.99788376, 0.99888293, 0.49313557],
                             [- 6.27251093, -0.00069246, 0.00042437, 7.27181847, 0.99788423, 0.99888318, 0.49313560],
                             [- 6.27251130, -0.00069230, 0.00042427, 7.27181900, 0.99788471, 0.99888343, 0.49313566],
                             [- 6.27251167, -0.00069215, 0.00042418, 7.27181952, 0.99788518, 0.99888368, 0.49313574],
                             [- 6.27251204, -0.00069199, 0.00042408, 7.27182005, 0.99788565, 0.99888393, 0.49313579],
                             [- 6.27251241, -0.00069184, 0.00042399, 7.27182058, 0.99788613, 0.99888418, 0.49313583],
                             [- 6.27251278, -0.00069168, 0.00042389, 7.27182110, 0.99788660, 0.99888443, 0.49313587],
                             [- 6.27251315, -0.00069153, 0.00042380, 7.27182163, 0.99788707, 0.99888468, 0.49313593],
                             [- 6.27251353, -0.00069137, 0.00042370, 7.27182215, 0.99788754, 0.99888492, 0.49313600],
                             [- 6.27251390, -0.00069122, 0.00042361, 7.27182268, 0.99788801, 0.99888517, 0.49313604],
                             [- 6.27251427, -0.00069106, 0.00042351, 7.27182320, 0.99788849, 0.99888542, 0.49313613],
                             [- 6.27251464, -0.00069091, 0.00042342, 7.27182373, 0.99788896, 0.99888567, 0.49313616],
                             [- 6.27251501, -0.00069076, 0.00042332, 7.27182425, 0.99788943, 0.99888592, 0.49313621],
                             [- 6.27251538, -0.00069060, 0.00042323, 7.27182478, 0.99788990, 0.99888617, 0.49313628],
                             [- 6.27251575, -0.00069045, 0.00042314, 7.27182530, 0.99789037, 0.99888642, 0.49313634],
                             [- 6.27251612, -0.00069029, 0.00042304, 7.27182582, 0.99789084, 0.99888667, 0.49313637],
                             [- 6.27251649, -0.00069014, 0.00042295, 7.27182635, 0.99789131, 0.99888691, 0.49313644],
                             [- 6.27251685, -0.00068999, 0.00042285, 7.27182687, 0.99789178, 0.99888716, 0.49313651],
                             [- 6.27251722, -0.00068983, 0.00042276, 7.27182739, 0.99789225, 0.99888741, 0.49313656],
                             [- 6.27251759, -0.00068968, 0.00042266, 7.27182791, 0.99789272, 0.99888766, 0.49313664],
                             [- 6.27251796, -0.00068952, 0.00042257, 7.27182844, 0.99789319, 0.99888791, 0.49313666],
                             [- 6.27251833, -0.00068937, 0.00042248, 7.27182896, 0.99789366, 0.99888815, 0.49313670],
                             [- 6.27251870, -0.00068922, 0.00042238, 7.27182948, 0.99789413, 0.99888840, 0.49313677],
                             [- 6.27251907, -0.00068906, 0.00042229, 7.27183000, 0.99789460, 0.99888865, 0.49313683],
                             [- 6.27251943, -0.00068891, 0.00042219, 7.27183052, 0.99789507, 0.99888890, 0.49313687],
                             [- 6.27251980, -0.00068876, 0.00042210, 7.27183104, 0.99789554, 0.99888914, 0.49313691],
                             [- 6.27252017, -0.00068860, 0.00042200, 7.27183157, 0.99789601, 0.99888939, 0.49313700],
                             [- 6.27252054, -0.00068845, 0.00042191, 7.27183209, 0.99789647, 0.99888964, 0.49313706],
                             [- 6.27252091, -0.00068830, 0.00042182, 7.27183261, 0.99789694, 0.99888988, 0.49313707],
                             [- 6.27252127, -0.00068815, 0.00042172, 7.27183313, 0.99789741, 0.99889013, 0.49313711],
                             [- 6.27252164, -0.00068799, 0.00042163, 7.27183365, 0.99789788, 0.99889038, 0.49313720],
                             [- 6.27252201, -0.00068784, 0.00042154, 7.27183417, 0.99789835, 0.99889062, 0.49313727],
                             [- 6.27252237, -0.00068769, 0.00042144, 7.27183469, 0.99789881, 0.99889087, 0.49313733],
                             [- 6.27252274, -0.00068753, 0.00042135, 7.27183521, 0.99789928, 0.99889112, 0.49313737],
                             [- 6.27252311, -0.00068738, 0.00042125, 7.27183572, 0.99789975, 0.99889136, 0.49313744],
                             [- 6.27252347, -0.00068723, 0.00042116, 7.27183624, 0.99790021, 0.99889161, 0.49313749],
                             [- 6.27252384, -0.00068708, 0.00042107, 7.27183676, 0.99790068, 0.99889186, 0.49313754],
                             [- 6.27252420, -0.00068692, 0.00042097, 7.27183728, 0.99790115, 0.99889210, 0.49313760],
                             [- 6.27252457, -0.00068677, 0.00042088, 7.27183780, 0.99790161, 0.99889235, 0.49313767],
                             [- 6.27252494, -0.00068662, 0.00042079, 7.27183832, 0.99790208, 0.99889259, 0.49313771],
                             [- 6.27252530, -0.00068647, 0.00042069, 7.27183883, 0.99790254, 0.99889284, 0.49313775],
                             [- 6.27252567, -0.00068631, 0.00042060, 7.27183935, 0.99790301, 0.99889309, 0.49313782],
                             [- 6.27252603, -0.00068616, 0.00042051, 7.27183987, 0.99790347, 0.99889333, 0.49313786],
                             [- 6.27252640, -0.00068601, 0.00042041, 7.27184039, 0.99790394, 0.99889358, 0.49313791],
                             [- 6.27252676, -0.00068586, 0.00042032, 7.27184090, 0.99790440, 0.99889382, 0.49313797],
                             [- 6.27252712, -0.00068571, 0.00042023, 7.27184142, 0.99790487, 0.99889407, 0.49313801],
                             [- 6.27252749, -0.00068555, 0.00042013, 7.27184193, 0.99790533, 0.99889431, 0.49313810],
                             [- 6.27252785, -0.00068540, 0.00042004, 7.27184245, 0.99790579, 0.99889456, 0.49313815],
                             [- 6.27252822, -0.00068525, 0.00041995, 7.27184297, 0.99790626, 0.99889480, 0.49313818],
                             [- 6.27252858, -0.00068510, 0.00041985, 7.27184348, 0.99790672, 0.99889505, 0.49313824],
                             [- 6.27252895, -0.00068495, 0.00041976, 7.27184400, 0.99790719, 0.99889529, 0.49313828],
                             [- 6.27252931, -0.00068480, 0.00041967, 7.27184451, 0.99790765, 0.99889553, 0.49313835],
                             [- 6.27252967, -0.00068464, 0.00041958, 7.27184503, 0.99790811, 0.99889578, 0.49313842],
                             [- 6.27253004, -0.00068449, 0.00041948, 7.27184554, 0.99790857, 0.99889602, 0.49313845],
                             [- 6.27253040, -0.00068434, 0.00041939, 7.27184606, 0.99790904, 0.99889627, 0.49313853],
                             [- 6.27253076, -0.00068419, 0.00041930, 7.27184657, 0.99790950, 0.99889651, 0.49313854],
                             [- 6.27253112, -0.00068404, 0.00041920, 7.27184708, 0.99790996, 0.99889676, 0.49313861],
                             [- 6.27253149, -0.00068389, 0.00041911, 7.27184760, 0.99791042, 0.99889700, 0.49313867],
                             [- 6.27253185, -0.00068374, 0.00041902, 7.27184811, 0.99791089, 0.99889724, 0.49313874],
                             [- 6.27253221, -0.00068359, 0.00041893, 7.27184862, 0.99791135, 0.99889749, 0.49313876],
                             [- 6.27253257, -0.00068344, 0.00041883, 7.27184914, 0.99791181, 0.99889773, 0.49313881],
                             [- 6.27253293, -0.00068328, 0.00041874, 7.27184965, 0.99791227, 0.99889797, 0.49313886],
                             [- 6.27253330, -0.00068313, 0.00041865, 7.27185016, 0.99791273, 0.99889822, 0.49313893],
                             [- 6.27253366, -0.00068298, 0.00041856, 7.27185068, 0.99791319, 0.99889846, 0.49313897],
                             [- 6.27253402, -0.00068283, 0.00041846, 7.27185119, 0.99791365, 0.99889870, 0.49313904],
                             [- 6.27253438, -0.00068268, 0.00041837, 7.27185170, 0.99791411, 0.99889895, 0.49313909],
                             [- 6.27253474, -0.00068253, 0.00041828, 7.27185221, 0.99791457, 0.99889919, 0.49313915],
                             [- 6.27253510, -0.00068238, 0.00041819, 7.27185272, 0.99791503, 0.99889943, 0.49313920],
                             [- 6.27253546, -0.00068223, 0.00041810, 7.27185323, 0.99791549, 0.99889967, 0.49313924],
                             [- 6.27253582, -0.00068208, 0.00041800, 7.27185374, 0.99791595, 0.99889992, 0.49313932],
                             [- 6.27253618, -0.00068193, 0.00041791, 7.27185425, 0.99791641, 0.99890016, 0.49313937],
                             [- 6.27253654, -0.00068178, 0.00041782, 7.27185477, 0.99791687, 0.99890040, 0.49313944],
                             [- 6.27253690, -0.00068163, 0.00041773, 7.27185528, 0.99791733, 0.99890064, 0.49313947],
                             [- 6.27253726, -0.00068148, 0.00041763, 7.27185579, 0.99791779, 0.99890089, 0.49313953],
                             [- 6.27253762, -0.00068133, 0.00041754, 7.27185630, 0.99791825, 0.99890113, 0.49313960],
                             [- 6.27253798, -0.00068118, 0.00041745, 7.27185680, 0.99791871, 0.99890137, 0.49313964],
                             [- 6.27253834, -0.00068103, 0.00041736, 7.27185731, 0.99791916, 0.99890161, 0.49313969],
                             [- 6.27253870, -0.00068088, 0.00041727, 7.27185782, 0.99791962, 0.99890185, 0.49313973],
                             [- 6.27253906, -0.00068073, 0.00041718, 7.27185833, 0.99792008, 0.99890209, 0.49313978],
                             [- 6.27253942, -0.00068058, 0.00041708, 7.27185884, 0.99792054, 0.99890234, 0.49313983],
                             [- 6.27253978, -0.00068043, 0.00041699, 7.27185935, 0.99792099, 0.99890258, 0.49313991],
                             [- 6.27254014, -0.00068028, 0.00041690, 7.27185986, 0.99792145, 0.99890282, 0.49313998],
                             [- 6.27254050, -0.00068013, 0.00041681, 7.27186036, 0.99792191, 0.99890306, 0.49314002],
                             [- 6.27254086, -0.00067998, 0.00041672, 7.27186087, 0.99792237, 0.99890330, 0.49314008],
                             [- 6.27254121, -0.00067983, 0.00041662, 7.27186138, 0.99792282, 0.99890354, 0.49314012],
                             [- 6.27254157, -0.00067968, 0.00041653, 7.27186189, 0.99792328, 0.99890378, 0.49314018],
                             [- 6.27254193, -0.00067953, 0.00041644, 7.27186239, 0.99792373, 0.99890402, 0.49314019],
                             [- 6.27254229, -0.00067939, 0.00041635, 7.27186290, 0.99792419, 0.99890426, 0.49314025],
                             [- 6.27254264, -0.00067924, 0.00041626, 7.27186341, 0.99792465, 0.99890450, 0.49314034],
                             [- 6.27254300, -0.00067909, 0.00041617, 7.27186391, 0.99792510, 0.99890474, 0.49314040],
                             [- 6.27254336, -0.00067894, 0.00041608, 7.27186442, 0.99792556, 0.99890499, 0.49314044],
                             [- 6.27254372, -0.00067879, 0.00041598, 7.27186493, 0.99792601, 0.99890523, 0.49314049],
                             [- 6.27254407, -0.00067864, 0.00041589, 7.27186543, 0.99792647, 0.99890547, 0.49314054],
                             [- 6.27254443, -0.00067849, 0.00041580, 7.27186594, 0.99792692, 0.99890571, 0.49314058],
                             [- 6.27254479, -0.00067834, 0.00041571, 7.27186644, 0.99792738, 0.99890595, 0.49314062],
                             [- 6.27254514, -0.00067819, 0.00041562, 7.27186695, 0.99792783, 0.99890618, 0.49314070],
                             [- 6.27254550, -0.00067805, 0.00041553, 7.27186745, 0.99792828, 0.99890642, 0.49314075],
                             [- 6.27254585, -0.00067790, 0.00041544, 7.27186796, 0.99792874, 0.99890666, 0.49314079],
                             [- 6.27254621, -0.00067775, 0.00041535, 7.27186846, 0.99792919, 0.99890690, 0.49314085],
                             [- 6.27254657, -0.00067760, 0.00041526, 7.27186897, 0.99792965, 0.99890714, 0.49314090],
                             [- 6.27254692, -0.00067745, 0.00041516, 7.27186947, 0.99793010, 0.99890738, 0.49314094],
                             [- 6.27254728, -0.00067730, 0.00041507, 7.27186997, 0.99793055, 0.99890762, 0.49314100],
                             [- 6.27254763, -0.00067716, 0.00041498, 7.27187048, 0.99793101, 0.99890786, 0.49314106],
                             [- 6.27254799, -0.00067701, 0.00041489, 7.27187098, 0.99793146, 0.99890810, 0.49314111],
                             [- 6.27254834, -0.00067686, 0.00041480, 7.27187148, 0.99793191, 0.99890834, 0.49314118],
                             [- 6.27254870, -0.00067671, 0.00041471, 7.27187199, 0.99793236, 0.99890858, 0.49314123],
                             [- 6.27254905, -0.00067656, 0.00041462, 7.27187249, 0.99793282, 0.99890882, 0.49314127],
                             [- 6.27254941, -0.00067642, 0.00041453, 7.27187299, 0.99793327, 0.99890905, 0.49314132],
                             [- 6.27254976, -0.00067627, 0.00041444, 7.27187349, 0.99793372, 0.99890929, 0.49314138],
                             [- 6.27255012, -0.00067612, 0.00041435, 7.27187399, 0.99793417, 0.99890953, 0.49314144],
                             [- 6.27255047, -0.00067597, 0.00041426, 7.27187450, 0.99793462, 0.99890977, 0.49314149],
                             [- 6.27255082, -0.00067583, 0.00041417, 7.27187500, 0.99793507, 0.99891001, 0.49314155],
                             [- 6.27255118, -0.00067568, 0.00041408, 7.27187550, 0.99793552, 0.99891025, 0.49314159],
                             [- 6.27255153, -0.00067553, 0.00041399, 7.27187600, 0.99793597, 0.99891048, 0.49314164],
                             [- 6.27255188, -0.00067538, 0.00041390, 7.27187650, 0.99793643, 0.99891072, 0.49314169],
                             [- 6.27255224, -0.00067524, 0.00041381, 7.27187700, 0.99793688, 0.99891096, 0.49314176],
                             [- 6.27255259, -0.00067509, 0.00041372, 7.27187750, 0.99793733, 0.99891120, 0.49314181],
                             [- 6.27255294, -0.00067494, 0.00041362, 7.27187800, 0.99793778, 0.99891143, 0.49314185],
                             [- 6.27255330, -0.00067479, 0.00041353, 7.27187850, 0.99793823, 0.99891167, 0.49314190],
                             [- 6.27255365, -0.00067465, 0.00041344, 7.27187900, 0.99793868, 0.99891191, 0.49314197],
                             [- 6.27255400, -0.00067450, 0.00041335, 7.27187950, 0.99793912, 0.99891215, 0.49314200],
                             [- 6.27255435, -0.00067435, 0.00041326, 7.27188000, 0.99793957, 0.99891238, 0.49314208],
                             [- 6.27255470, -0.00067421, 0.00041317, 7.27188050, 0.99794002, 0.99891262, 0.49314213],
                             [- 6.27255506, -0.00067406, 0.00041308, 7.27188100, 0.99794047, 0.99891286, 0.49314217],
                             [- 6.27255541, -0.00067391, 0.00041299, 7.27188150, 0.99794092, 0.99891309, 0.49314220],
                             [- 6.27255576, -0.00067377, 0.00041290, 7.27188199, 0.99794137, 0.99891333, 0.49314225],
                             [- 6.27255611, -0.00067362, 0.00041281, 7.27188249, 0.99794182, 0.99891357, 0.49314232],
                             [- 6.27255646, -0.00067347, 0.00041272, 7.27188299, 0.99794227, 0.99891380, 0.49314238],
                             [- 6.27255681, -0.00067333, 0.00041263, 7.27188349, 0.99794271, 0.99891404, 0.49314241],
                             [- 6.27255717, -0.00067318, 0.00041254, 7.27188398, 0.99794316, 0.99891428, 0.49314247],
                             [- 6.27255752, -0.00067303, 0.00041245, 7.27188448, 0.99794361, 0.99891451, 0.49314253],
                             [- 6.27255787, -0.00067289, 0.00041236, 7.27188498, 0.99794406, 0.99891475, 0.49314258],
                             [- 6.27255822, -0.00067274, 0.00041228, 7.27188548, 0.99794450, 0.99891498, 0.49314263],
                             [- 6.27255857, -0.00067260, 0.00041219, 7.27188597, 0.99794495, 0.99891522, 0.49314268],
                             [- 6.27255892, -0.00067245, 0.00041210, 7.27188647, 0.99794540, 0.99891545, 0.49314274],
                             [- 6.27255927, -0.00067230, 0.00041201, 7.27188696, 0.99794584, 0.99891569, 0.49314280],
                             [- 6.27255962, -0.00067216, 0.00041192, 7.27188746, 0.99794629, 0.99891593, 0.49314284],
                             [- 6.27255997, -0.00067201, 0.00041183, 7.27188796, 0.99794673, 0.99891616, 0.49314287],
                             [- 6.27256032, -0.00067187, 0.00041174, 7.27188845, 0.99794718, 0.99891640, 0.49314297],
                             [- 6.27256067, -0.00067172, 0.00041165, 7.27188895, 0.99794763, 0.99891663, 0.49314303],
                             [- 6.27256102, -0.00067157, 0.00041156, 7.27188944, 0.99794807, 0.99891687, 0.49314307],
                             [- 6.27256137, -0.00067143, 0.00041147, 7.27188994, 0.99794852, 0.99891710, 0.49314312],
                             [- 6.27256171, -0.00067128, 0.00041138, 7.27189043, 0.99794896, 0.99891734, 0.49314315],
                             [- 6.27256206, -0.00067114, 0.00041129, 7.27189093, 0.99794941, 0.99891757, 0.49314322],
                             [- 6.27256241, -0.00067099, 0.00041120, 7.27189142, 0.99794985, 0.99891781, 0.49314325],
                             [- 6.27256276, -0.00067085, 0.00041111, 7.27189191, 0.99795030, 0.99891804, 0.49314333],
                             [- 6.27256311, -0.00067070, 0.00041102, 7.27189241, 0.99795074, 0.99891828, 0.49314337],
                             [- 6.27256346, -0.00067056, 0.00041093, 7.27189290, 0.99795119, 0.99891851, 0.49314340],
                             [- 6.27256381, -0.00067041, 0.00041085, 7.27189339, 0.99795163, 0.99891874, 0.49314348],
                             [- 6.27256415, -0.00067027, 0.00041076, 7.27189389, 0.99795207, 0.99891898, 0.49314353],
                             [- 6.27256450, -0.00067012, 0.00041067, 7.27189438, 0.99795252, 0.99891921, 0.49314358],
                             [- 6.27256485, -0.00066998, 0.00041058, 7.27189487, 0.99795296, 0.99891945, 0.49314365],
                             [- 6.27256520, -0.00066983, 0.00041049, 7.27189537, 0.99795340, 0.99891968, 0.49314368],
                             [- 6.27256554, -0.00066969, 0.00041040, 7.27189586, 0.99795385, 0.99891991, 0.49314376],
                             [- 6.27256589, -0.00066954, 0.00041031, 7.27189635, 0.99795429, 0.99892015, 0.49314379],
                             [- 6.27256624, -0.00066940, 0.00041022, 7.27189684, 0.99795473, 0.99892038, 0.49314383],
                             [- 6.27256658, -0.00066925, 0.00041013, 7.27189733, 0.99795517, 0.99892061, 0.49314389],
                             [- 6.27256693, -0.00066911, 0.00041005, 7.27189782, 0.99795562, 0.99892085, 0.49314395],
                             [- 6.27256728, -0.00066896, 0.00040996, 7.27189832, 0.99795606, 0.99892108, 0.49314398],
                             [- 6.27256762, -0.00066882, 0.00040987, 7.27189881, 0.99795650, 0.99892131, 0.49314403],
                             [- 6.27256797, -0.00066867, 0.00040978, 7.27189930, 0.99795694, 0.99892155, 0.49314409],
                             [- 6.27256832, -0.00066853, 0.00040969, 7.27189979, 0.99795738, 0.99892178, 0.49314415],
                             [- 6.27256866, -0.00066838, 0.00040960, 7.27190028, 0.99795782, 0.99892201, 0.49314419],
                             [- 6.27256901, -0.00066824, 0.00040951, 7.27190077, 0.99795827, 0.99892225, 0.49314425],
                             [- 6.27256935, -0.00066810, 0.00040943, 7.27190126, 0.99795871, 0.99892248, 0.49314428],
                             [- 6.27256970, -0.00066795, 0.00040934, 7.27190175, 0.99795915, 0.99892271, 0.49314436],
                             [- 6.27257004, -0.00066781, 0.00040925, 7.27190224, 0.99795959, 0.99892294, 0.49314439],
                             [- 6.27257039, -0.00066766, 0.00040916, 7.27190273, 0.99796003, 0.99892318, 0.49314446],
                             [- 6.27257073, -0.00066752, 0.00040907, 7.27190322, 0.99796047, 0.99892341, 0.49314452],
                             [- 6.27257108, -0.00066738, 0.00040898, 7.27190370, 0.99796091, 0.99892364, 0.49314453],
                             [- 6.27257142, -0.00066723, 0.00040890, 7.27190419, 0.99796135, 0.99892387, 0.49314460],
                             [- 6.27257177, -0.00066709, 0.00040881, 7.27190468, 0.99796179, 0.99892410, 0.49314466],
                             [- 6.27257211, -0.00066694, 0.00040872, 7.27190517, 0.99796223, 0.99892434, 0.49314470],
                             [- 6.27257246, -0.00066680, 0.00040863, 7.27190566, 0.99796267, 0.99892457, 0.49314476],
                             [- 6.27257280, -0.00066666, 0.00040854, 7.27190615, 0.99796311, 0.99892480, 0.49314482],
                             [- 6.27257315, -0.00066651, 0.00040846, 7.27190663, 0.99796354, 0.99892503, 0.49314486],
                             [- 6.27257349, -0.00066637, 0.00040837, 7.27190712, 0.99796398, 0.99892526, 0.49314490],
                             [- 6.27257383, -0.00066623, 0.00040828, 7.27190761, 0.99796442, 0.99892549, 0.49314498],
                             [- 6.27257418, -0.00066608, 0.00040819, 7.27190809, 0.99796486, 0.99892573, 0.49314501],
                             [- 6.27257452, -0.00066594, 0.00040810, 7.27190858, 0.99796530, 0.99892596, 0.49314505],
                             [- 6.27257486, -0.00066580, 0.00040802, 7.27190907, 0.99796574, 0.99892619, 0.49314512],
                             [- 6.27257521, -0.00066565, 0.00040793, 7.27190955, 0.99796617, 0.99892642, 0.49314515],
                             [- 6.27257555, -0.00066551, 0.00040784, 7.27191004, 0.99796661, 0.99892665, 0.49314521],
                             [- 6.27257589, -0.00066537, 0.00040775, 7.27191053, 0.99796705, 0.99892688, 0.49314528],
                             [- 6.27257623, -0.00066522, 0.00040766, 7.27191101, 0.99796748, 0.99892711, 0.49314532],
                             [- 6.27257658, -0.00066508, 0.00040758, 7.27191150, 0.99796792, 0.99892734, 0.49314537],
                             [- 6.27257692, -0.00066494, 0.00040749, 7.27191198, 0.99796836, 0.99892757, 0.49314541],
                             [- 6.27257726, -0.00066480, 0.00040740, 7.27191247, 0.99796880, 0.99892780, 0.49314547],
                             [- 6.27257760, -0.00066465, 0.00040731, 7.27191295, 0.99796923, 0.99892803, 0.49314555],
                             [- 6.27257795, -0.00066451, 0.00040723, 7.27191344, 0.99796967, 0.99892826, 0.49314558],
                             [- 6.27257829, -0.00066437, 0.00040714, 7.27191392, 0.99797010, 0.99892849, 0.49314564],
                             [- 6.27257863, -0.00066422, 0.00040705, 7.27191440, 0.99797054, 0.99892872, 0.49314568],
                             [- 6.27257897, -0.00066408, 0.00040696, 7.27191489, 0.99797098, 0.99892895, 0.49314573],
                             [- 6.27257931, -0.00066394, 0.00040688, 7.27191537, 0.99797141, 0.99892918, 0.49314579],
                             [- 6.27257965, -0.00066380, 0.00040679, 7.27191586, 0.99797185, 0.99892941, 0.49314582],
                             [- 6.27257999, -0.00066366, 0.00040670, 7.27191634, 0.99797228, 0.99892964, 0.49314588],
                             [- 6.27258033, -0.00066351, 0.00040662, 7.27191682, 0.99797272, 0.99892987, 0.49314593],
                             [- 6.27258068, -0.00066337, 0.00040653, 7.27191730, 0.99797315, 0.99893010, 0.49314598],
                             [- 6.27258102, -0.00066323, 0.00040644, 7.27191779, 0.99797358, 0.99893033, 0.49314603],
                             [- 6.27258136, -0.00066309, 0.00040635, 7.27191827, 0.99797402, 0.99893056, 0.49314608],
                             [- 6.27258170, -0.00066294, 0.00040627, 7.27191875, 0.99797445, 0.99893079, 0.49314611],
                             [- 6.27258204, -0.00066280, 0.00040618, 7.27191923, 0.99797489, 0.99893102, 0.49314621],
                             [- 6.27258238, -0.00066266, 0.00040609, 7.27191972, 0.99797532, 0.99893125, 0.49314623],
                             [- 6.27258272, -0.00066252, 0.00040601, 7.27192020, 0.99797575, 0.99893148, 0.49314626],
                             [- 6.27258306, -0.00066238, 0.00040592, 7.27192068, 0.99797619, 0.99893170, 0.49314635],
                             [- 6.27258340, -0.00066224, 0.00040583, 7.27192116, 0.99797662, 0.99893193, 0.49314639],
                             [- 6.27258373, -0.00066209, 0.00040574, 7.27192164, 0.99797705, 0.99893216, 0.49314644],
                             [- 6.27258407, -0.00066195, 0.00040566, 7.27192212, 0.99797749, 0.99893239, 0.49314649],
                             [- 6.27258441, -0.00066181, 0.00040557, 7.27192260, 0.99797792, 0.99893262, 0.49314651],
                             [- 6.27258475, -0.00066167, 0.00040548, 7.27192308, 0.99797835, 0.99893285, 0.49314660],
                             [- 6.27258509, -0.00066153, 0.00040540, 7.27192356, 0.99797878, 0.99893307, 0.49314663],
                             [- 6.27258543, -0.00066139, 0.00040531, 7.27192404, 0.99797922, 0.99893330, 0.49314669],
                             [- 6.27258577, -0.00066125, 0.00040522, 7.27192452, 0.99797965, 0.99893353, 0.49314673],
                             [- 6.27258611, -0.00066110, 0.00040514, 7.27192500, 0.99798008, 0.99893376, 0.49314679],
                             [- 6.27258644, -0.00066096, 0.00040505, 7.27192548, 0.99798051, 0.99893399, 0.49314686],
                             [- 6.27258678, -0.00066082, 0.00040496, 7.27192596, 0.99798094, 0.99893421, 0.49314689],
                             [- 6.27258712, -0.00066068, 0.00040488, 7.27192644, 0.99798137, 0.99893444, 0.49314694],
                             [- 6.27258746, -0.00066054, 0.00040479, 7.27192692, 0.99798181, 0.99893467, 0.49314700],
                             [- 6.27258780, -0.00066040, 0.00040471, 7.27192740, 0.99798224, 0.99893490, 0.49314705],
                             [- 6.27258813, -0.00066026, 0.00040462, 7.27192788, 0.99798267, 0.99893512, 0.49314708],
                             [- 6.27258847, -0.00066012, 0.00040453, 7.27192835, 0.99798310, 0.99893535, 0.49314712],
                             [- 6.27258881, -0.00065998, 0.00040445, 7.27192883, 0.99798353, 0.99893558, 0.49314719],
                             [- 6.27258914, -0.00065984, 0.00040436, 7.27192931, 0.99798396, 0.99893580, 0.49314727],
                             [- 6.27258948, -0.00065970, 0.00040427, 7.27192979, 0.99798439, 0.99893603, 0.49314727],
                             [- 6.27258982, -0.00065955, 0.00040419, 7.27193026, 0.99798482, 0.99893626, 0.49314731],
                             [- 6.27259015, -0.00065941, 0.00040410, 7.27193074, 0.99798525, 0.99893648, 0.49314740],
                             [- 6.27259049, -0.00065927, 0.00040401, 7.27193122, 0.99798568, 0.99893671, 0.49314745],
                             [- 6.27259083, -0.00065913, 0.00040393, 7.27193169, 0.99798610, 0.99893694, 0.49314746],
                             [- 6.27259116, -0.00065899, 0.00040384, 7.27193217, 0.99798653, 0.99893716, 0.49314754],
                             [- 6.27259150, -0.00065885, 0.00040376, 7.27193265, 0.99798696, 0.99893739, 0.49314761],
                             [- 6.27259184, -0.00065871, 0.00040367, 7.27193312, 0.99798739, 0.99893762, 0.49314766],
                             [- 6.27259217, -0.00065857, 0.00040358, 7.27193360, 0.99798782, 0.99893784, 0.49314769],
                             [- 6.27259251, -0.00065843, 0.00040350, 7.27193407, 0.99798825, 0.99893807, 0.49314775],
                             [- 6.27259284, -0.00065829, 0.00040341, 7.27193455, 0.99798868, 0.99893829, 0.49314778],
                             [- 6.27259318, -0.00065815, 0.00040333, 7.27193502, 0.99798910, 0.99893852, 0.49314784],
                             [- 6.27259351, -0.00065801, 0.00040324, 7.27193550, 0.99798953, 0.99893875, 0.49314788],
                             [- 6.27259385, -0.00065787, 0.00040316, 7.27193597, 0.99798996, 0.99893897, 0.49314794],
                             [- 6.27259418, -0.00065773, 0.00040307, 7.27193645, 0.99799039, 0.99893920, 0.49314799],
                             [- 6.27259452, -0.00065759, 0.00040298, 7.27193692, 0.99799081, 0.99893942, 0.49314805],
                             [- 6.27259485, -0.00065745, 0.00040290, 7.27193740, 0.99799124, 0.99893965, 0.49314807],
                             [- 6.27259519, -0.00065731, 0.00040281, 7.27193787, 0.99799167, 0.99893987, 0.49314813],
                             [- 6.27259552, -0.00065717, 0.00040273, 7.27193835, 0.99799209, 0.99894010, 0.49314822],
                             [- 6.27259585, -0.00065703, 0.00040264, 7.27193882, 0.99799252, 0.99894032, 0.49314825],
                             [- 6.27259619, -0.00065690, 0.00040256, 7.27193929, 0.99799295, 0.99894055, 0.49314830],
                             [- 6.27259652, -0.00065676, 0.00040247, 7.27193977, 0.99799337, 0.99894077, 0.49314836],
                             [- 6.27259686, -0.00065662, 0.00040239, 7.27194024, 0.99799380, 0.99894100, 0.49314839],
                             [- 6.27259719, -0.00065648, 0.00040230, 7.27194071, 0.99799422, 0.99894122, 0.49314844],
                             [- 6.27259752, -0.00065634, 0.00040221, 7.27194118, 0.99799465, 0.99894145, 0.49314848],
                             [- 6.27259786, -0.00065620, 0.00040213, 7.27194166, 0.99799507, 0.99894167, 0.49314852],
                             [- 6.27259819, -0.00065606, 0.00040204, 7.27194213, 0.99799550, 0.99894190, 0.49314858],
                             [- 6.27259852, -0.00065592, 0.00040196, 7.27194260, 0.99799593, 0.99894212, 0.49314864],
                             [- 6.27259885, -0.00065578, 0.00040187, 7.27194307, 0.99799635, 0.99894234, 0.49314867],
                             [- 6.27259919, -0.00065564, 0.00040179, 7.27194354, 0.99799677, 0.99894257, 0.49314872],
                             [- 6.27259952, -0.00065550, 0.00040170, 7.27194402, 0.99799720, 0.99894279, 0.49314880],
                             [- 6.27259985, -0.00065537, 0.00040162, 7.27194449, 0.99799762, 0.99894302, 0.49314880],
                             [- 6.27260018, -0.00065523, 0.00040153, 7.27194496, 0.99799805, 0.99894324, 0.49314888],
                             [- 6.27260052, -0.00065509, 0.00040145, 7.27194543, 0.99799847, 0.99894346, 0.49314897],
                             [- 6.27260085, -0.00065495, 0.00040136, 7.27194590, 0.99799889, 0.99894369, 0.49314899],
                             [- 6.27260118, -0.00065481, 0.00040128, 7.27194637, 0.99799932, 0.99894391, 0.49314902],
                             [- 6.27260151, -0.00065467, 0.00040119, 7.27194684, 0.99799974, 0.99894413, 0.49314908],
                             [- 6.27260184, -0.00065453, 0.00040111, 7.27194731, 0.99800016, 0.99894436, 0.49314911],
                             [- 6.27260217, -0.00065440, 0.00040102, 7.27194778, 0.99800059, 0.99894458, 0.49314920],
                             [- 6.27260251, -0.00065426, 0.00040094, 7.27194825, 0.99800101, 0.99894480, 0.49314922],
                             [- 6.27260284, -0.00065412, 0.00040085, 7.27194872, 0.99800143, 0.99894503, 0.49314928],
                             [- 6.27260317, -0.00065398, 0.00040077, 7.27194919, 0.99800186, 0.99894525, 0.49314931],
                             [- 6.27260350, -0.00065384, 0.00040068, 7.27194966, 0.99800228, 0.99894547, 0.49314937],
                             [- 6.27260383, -0.00065370, 0.00040060, 7.27195013, 0.99800270, 0.99894570, 0.49314942],
                             [- 6.27260416, -0.00065357, 0.00040051, 7.27195059, 0.99800312, 0.99894592, 0.49314947],
                             [- 6.27260449, -0.00065343, 0.00040043, 7.27195106, 0.99800354, 0.99894614, 0.49314952],
                             [- 6.27260482, -0.00065329, 0.00040035, 7.27195153, 0.99800397, 0.99894636, 0.49314960],
                             [- 6.27260515, -0.00065315, 0.00040026, 7.27195200, 0.99800439, 0.99894659, 0.49314963],
                             [- 6.27260548, -0.00065302, 0.00040018, 7.27195247, 0.99800481, 0.99894681, 0.49314965],
                             [- 6.27260581, -0.00065288, 0.00040009, 7.27195293, 0.99800523, 0.99894703, 0.49314972],
                             [- 6.27260614, -0.00065274, 0.00040001, 7.27195340, 0.99800565, 0.99894725, 0.49314975],
                             [- 6.27260647, -0.00065260, 0.00039992, 7.27195387, 0.99800607, 0.99894747, 0.49314983],
                             [- 6.27260680, -0.00065246, 0.00039984, 7.27195434, 0.99800649, 0.99894770, 0.49314987],
                             [- 6.27260713, -0.00065233, 0.00039975, 7.27195480, 0.99800691, 0.99894792, 0.49314994],
                             [- 6.27260746, -0.00065219, 0.00039967, 7.27195527, 0.99800733, 0.99894814, 0.49314996],
                             [- 6.27260779, -0.00065205, 0.00039959, 7.27195573, 0.99800775, 0.99894836, 0.49314998],
                             [- 6.27260812, -0.00065192, 0.00039950, 7.27195620, 0.99800817, 0.99894858, 0.49315004],
                             [- 6.27260844, -0.00065178, 0.00039942, 7.27195667, 0.99800859, 0.99894880, 0.49315011],
                             [- 6.27260877, -0.00065164, 0.00039933, 7.27195713, 0.99800901, 0.99894903, 0.49315016],
                             [- 6.27260910, -0.00065150, 0.00039925, 7.27195760, 0.99800943, 0.99894925, 0.49315020],
                             [- 6.27260943, -0.00065137, 0.00039917, 7.27195806, 0.99800985, 0.99894947, 0.49315028],
                             [- 6.27260976, -0.00065123, 0.00039908, 7.27195853, 0.99801027, 0.99894969, 0.49315027],
                             [- 6.27261009, -0.00065109, 0.00039900, 7.27195899, 0.99801069, 0.99894991, 0.49315035],
                             [- 6.27261041, -0.00065096, 0.00039891, 7.27195946, 0.99801110, 0.99895013, 0.49315041],
                             [- 6.27261074, -0.00065082, 0.00039883, 7.27195992, 0.99801152, 0.99895035, 0.49315047],
                             [- 6.27261107, -0.00065068, 0.00039875, 7.27196039, 0.99801194, 0.99895057, 0.49315053],
                             [- 6.27261140, -0.00065055, 0.00039866, 7.27196085, 0.99801236, 0.99895079, 0.49315055],
                             [- 6.27261172, -0.00065041, 0.00039858, 7.27196132, 0.99801278, 0.99895101, 0.49315059],
                             [- 6.27261205, -0.00065027, 0.00039849, 7.27196178, 0.99801320, 0.99895123, 0.49315063],
                             [- 6.27261238, -0.00065014, 0.00039841, 7.27196224, 0.99801361, 0.99895145, 0.49315070],
                             [- 6.27261271, -0.00065000, 0.00039833, 7.27196271, 0.99801403, 0.99895167, 0.49315074],
                             [- 6.27261303, -0.00064986, 0.00039824, 7.27196317, 0.99801445, 0.99895189, 0.49315081],
                             [- 6.27261336, -0.00064973, 0.00039816, 7.27196363, 0.99801486, 0.99895212, 0.49315083],
                             [- 6.27261369, -0.00064959, 0.00039808, 7.27196410, 0.99801528, 0.99895233, 0.49315088],
                             [- 6.27261401, -0.00064945, 0.00039799, 7.27196456, 0.99801570, 0.99895255, 0.49315095],
                             [- 6.27261434, -0.00064932, 0.00039791, 7.27196502, 0.99801611, 0.99895277, 0.49315102],
                             [- 6.27261466, -0.00064918, 0.00039782, 7.27196548, 0.99801653, 0.99895299, 0.49315101],
                             [- 6.27261499, -0.00064904, 0.00039774, 7.27196595, 0.99801695, 0.99895321, 0.49315109],
                             [- 6.27261532, -0.00064891, 0.00039766, 7.27196641, 0.99801736, 0.99895343, 0.49315112],
                             [- 6.27261564, -0.00064877, 0.00039757, 7.27196687, 0.99801778, 0.99895365, 0.49315120],
                             [- 6.27261597, -0.00064864, 0.00039749, 7.27196733, 0.99801819, 0.99895387, 0.49315122],
                             [- 6.27261629, -0.00064850, 0.00039741, 7.27196779, 0.99801861, 0.99895409, 0.49315129],
                             [- 6.27261662, -0.00064836, 0.00039732, 7.27196825, 0.99801902, 0.99895431, 0.49315134],
                             [- 6.27261694, -0.00064823, 0.00039724, 7.27196871, 0.99801944, 0.99895453, 0.49315136],
                             [- 6.27261727, -0.00064809, 0.00039716, 7.27196917, 0.99801985, 0.99895475, 0.49315141],
                             [- 6.27261759, -0.00064796, 0.00039707, 7.27196963, 0.99802027, 0.99895497, 0.49315149],
                             [- 6.27261792, -0.00064782, 0.00039699, 7.27197009, 0.99802068, 0.99895519, 0.49315153],
                             [- 6.27261824, -0.00064769, 0.00039691, 7.27197056, 0.99802110, 0.99895540, 0.49315156],
                             [- 6.27261857, -0.00064755, 0.00039683, 7.27197101, 0.99802151, 0.99895562, 0.49315162],
                             [- 6.27261889, -0.00064742, 0.00039674, 7.27197147, 0.99802193, 0.99895584, 0.49315167],
                             [- 6.27261921, -0.00064728, 0.00039666, 7.27197193, 0.99802234, 0.99895606, 0.49315173],
                             [- 6.27261954, -0.00064715, 0.00039658, 7.27197239, 0.99802275, 0.99895628, 0.49315176],
                             [- 6.27261986, -0.00064701, 0.00039649, 7.27197285, 0.99802317, 0.99895650, 0.49315180],
                             [- 6.27262019, -0.00064687, 0.00039641, 7.27197331, 0.99802358, 0.99895671, 0.49315186],
                             [- 6.27262051, -0.00064674, 0.00039633, 7.27197377, 0.99802399, 0.99895693, 0.49315191],
                             [- 6.27262083, -0.00064660, 0.00039624, 7.27197423, 0.99802441, 0.99895715, 0.49315198],
                             [- 6.27262116, -0.00064647, 0.00039616, 7.27197469, 0.99802482, 0.99895737, 0.49315200],
                             [- 6.27262148, -0.00064633, 0.00039608, 7.27197515, 0.99802523, 0.99895759, 0.49315204],
                             [- 6.27262180, -0.00064620, 0.00039600, 7.27197560, 0.99802564, 0.99895780, 0.49315211],
                             [- 6.27262213, -0.00064606, 0.00039591, 7.27197606, 0.99802606, 0.99895802, 0.49315216],
                             [- 6.27262245, -0.00064593, 0.00039583, 7.27197652, 0.99802647, 0.99895824, 0.49315219],
                             [- 6.27262277, -0.00064579, 0.00039575, 7.27197698, 0.99802688, 0.99895846, 0.49315225],
                             [- 6.27262309, -0.00064566, 0.00039567, 7.27197743, 0.99802729, 0.99895867, 0.49315226],
                             [- 6.27262342, -0.00064553, 0.00039558, 7.27197789, 0.99802770, 0.99895889, 0.49315233],
                             [- 6.27262374, -0.00064539, 0.00039550, 7.27197835, 0.99802812, 0.99895911, 0.49315238],
                             [- 6.27262406, -0.00064526, 0.00039542, 7.27197880, 0.99802853, 0.99895933, 0.49315244],
                             [- 6.27262438, -0.00064512, 0.00039534, 7.27197926, 0.99802894, 0.99895954, 0.49315248],
                             [- 6.27262470, -0.00064499, 0.00039525, 7.27197972, 0.99802935, 0.99895976, 0.49315254],
                             [- 6.27262503, -0.00064485, 0.00039517, 7.27198017, 0.99802976, 0.99895998, 0.49315255],
                             [- 6.27262535, -0.00064472, 0.00039509, 7.27198063, 0.99803017, 0.99896019, 0.49315260],
                             [- 6.27262567, -0.00064458, 0.00039501, 7.27198108, 0.99803058, 0.99896041, 0.49315266],
                             [- 6.27262599, -0.00064445, 0.00039492, 7.27198154, 0.99803099, 0.99896063, 0.49315272],
                             [- 6.27262631, -0.00064432, 0.00039484, 7.27198199, 0.99803140, 0.99896084, 0.49315276],
                             [- 6.27262663, -0.00064418, 0.00039476, 7.27198245, 0.99803181, 0.99896106, 0.49315283],
                             [- 6.27262695, -0.00064405, 0.00039468, 7.27198290, 0.99803222, 0.99896127, 0.49315287],
                             [- 6.27262727, -0.00064391, 0.00039460, 7.27198336, 0.99803263, 0.99896149, 0.49315291],
                             [- 6.27262759, -0.00064378, 0.00039451, 7.27198381, 0.99803304, 0.99896171, 0.49315298],
                             [- 6.27262792, -0.00064365, 0.00039443, 7.27198427, 0.99803345, 0.99896192, 0.49315301],
                             [- 6.27262824, -0.00064351, 0.00039435, 7.27198472, 0.99803386, 0.99896214, 0.49315305],
                             [- 6.27262856, -0.00064338, 0.00039427, 7.27198518, 0.99803427, 0.99896235, 0.49315310],
                             [- 6.27262888, -0.00064325, 0.00039418, 7.27198563, 0.99803468, 0.99896257, 0.49315315],
                             [- 6.27262920, -0.00064311, 0.00039410, 7.27198608, 0.99803508, 0.99896279, 0.49315320],
                             [- 6.27262952, -0.00064298, 0.00039402, 7.27198654, 0.99803549, 0.99896300, 0.49315324],
                             [- 6.27262984, -0.00064284, 0.00039394, 7.27198699, 0.99803590, 0.99896322, 0.49315329],
                             [- 6.27263015, -0.00064271, 0.00039386, 7.27198744, 0.99803631, 0.99896343, 0.49315335],
                             [- 6.27263047, -0.00064258, 0.00039378, 7.27198790, 0.99803672, 0.99896365, 0.49315341],
                             [- 6.27263079, -0.00064244, 0.00039369, 7.27198835, 0.99803712, 0.99896386, 0.49315341],
                             [- 6.27263111, -0.00064231, 0.00039361, 7.27198880, 0.99803753, 0.99896408, 0.49315348],
                             [- 6.27263143, -0.00064218, 0.00039353, 7.27198925, 0.99803794, 0.99896429, 0.49315355],
                             [- 6.27263175, -0.00064204, 0.00039345, 7.27198971, 0.99803835, 0.99896451, 0.49315360],
                             [- 6.27263207, -0.00064191, 0.00039337, 7.27199016, 0.99803875, 0.99896472, 0.49315362],
                             [- 6.27263239, -0.00064178, 0.00039329, 7.27199061, 0.99803916, 0.99896494, 0.49315366],
                             [- 6.27263271, -0.00064165, 0.00039320, 7.27199106, 0.99803957, 0.99896515, 0.49315370],
                             [- 6.27263302, -0.00064151, 0.00039312, 7.27199151, 0.99803997, 0.99896537, 0.49315378],
                             [- 6.27263334, -0.00064138, 0.00039304, 7.27199196, 0.99804038, 0.99896558, 0.49315382],
                             [- 6.27263366, -0.00064125, 0.00039296, 7.27199241, 0.99804079, 0.99896579, 0.49315384],
                             [- 6.27263398, -0.00064111, 0.00039288, 7.27199287, 0.99804119, 0.99896601, 0.49315390],
                             [- 6.27263430, -0.00064098, 0.00039280, 7.27199332, 0.99804160, 0.99896622, 0.49315398],
                             [- 6.27263461, -0.00064085, 0.00039271, 7.27199377, 0.99804200, 0.99896644, 0.49315398],
                             [- 6.27263493, -0.00064072, 0.00039263, 7.27199422, 0.99804241, 0.99896665, 0.49315402],
                             [- 6.27263525, -0.00064058, 0.00039255, 7.27199467, 0.99804281, 0.99896686, 0.49315410],
                             [- 6.27263557, -0.00064045, 0.00039247, 7.27199512, 0.99804322, 0.99896708, 0.49315415],
                             [- 6.27263588, -0.00064032, 0.00039239, 7.27199557, 0.99804362, 0.99896729, 0.49315420],
                             [- 6.27263620, -0.00064019, 0.00039231, 7.27199602, 0.99804403, 0.99896751, 0.49315423],
                             [- 6.27263652, -0.00064005, 0.00039223, 7.27199646, 0.99804443, 0.99896772, 0.49315429],
                             [- 6.27263683, -0.00063992, 0.00039215, 7.27199691, 0.99804484, 0.99896793, 0.49315433],
                             [- 6.27263715, -0.00063979, 0.00039206, 7.27199736, 0.99804524, 0.99896815, 0.49315438],
                             [- 6.27263747, -0.00063966, 0.00039198, 7.27199781, 0.99804565, 0.99896836, 0.49315441],
                             [- 6.27263778, -0.00063952, 0.00039190, 7.27199826, 0.99804605, 0.99896857, 0.49315445],
                             [- 6.27263810, -0.00063939, 0.00039182, 7.27199871, 0.99804645, 0.99896879, 0.49315452],
                             [- 6.27263842, -0.00063926, 0.00039174, 7.27199916, 0.99804686, 0.99896900, 0.49315453],
                             [- 6.27263873, -0.00063913, 0.00039166, 7.27199960, 0.99804726, 0.99896921, 0.49315461],
                             [- 6.27263905, -0.00063900, 0.00039158, 7.27200005, 0.99804766, 0.99896942, 0.49315465],
                             [- 6.27263936, -0.00063886, 0.00039150, 7.27200050, 0.99804807, 0.99896964, 0.49315471],
                             [- 6.27263968, -0.00063873, 0.00039142, 7.27200095, 0.99804847, 0.99896985, 0.49315476],
                             [- 6.27263999, -0.00063860, 0.00039134, 7.27200139, 0.99804887, 0.99897006, 0.49315480],
                             [- 6.27264031, -0.00063847, 0.00039126, 7.27200184, 0.99804928, 0.99897028, 0.49315486],
                             [- 6.27264063, -0.00063834, 0.00039117, 7.27200229, 0.99804968, 0.99897049, 0.49315489],
                             [- 6.27264094, -0.00063821, 0.00039109, 7.27200273, 0.99805008, 0.99897070, 0.49315491],
                             [- 6.27264126, -0.00063807, 0.00039101, 7.27200318, 0.99805048, 0.99897091, 0.49315503],
                             [- 6.27264157, -0.00063794, 0.00039093, 7.27200363, 0.99805088, 0.99897112, 0.49315504],
                             [- 6.27264188, -0.00063781, 0.00039085, 7.27200407, 0.99805129, 0.99897134, 0.49315507],
                             [- 6.27264220, -0.00063768, 0.00039077, 7.27200452, 0.99805169, 0.99897155, 0.49315512],
                             [- 6.27264251, -0.00063755, 0.00039069, 7.27200496, 0.99805209, 0.99897176, 0.49315517],
                             [- 6.27264283, -0.00063742, 0.00039061, 7.27200541, 0.99805249, 0.99897197, 0.49315521],
                             [- 6.27264314, -0.00063729, 0.00039053, 7.27200586, 0.99805289, 0.99897218, 0.49315529],
                             [- 6.27264346, -0.00063716, 0.00039045, 7.27200630, 0.99805329, 0.99897240, 0.49315533],
                             [- 6.27264377, -0.00063702, 0.00039037, 7.27200675, 0.99805369, 0.99897261, 0.49315538],
                             [- 6.27264408, -0.00063689, 0.00039029, 7.27200719, 0.99805409, 0.99897282, 0.49315540],
                             [- 6.27264440, -0.00063676, 0.00039021, 7.27200764, 0.99805450, 0.99897303, 0.49315545],
                             [- 6.27264471, -0.00063663, 0.00039013, 7.27200808, 0.99805490, 0.99897324, 0.49315550],
                             [- 6.27264502, -0.00063650, 0.00039005, 7.27200852, 0.99805530, 0.99897345, 0.49315556],
                             [- 6.27264534, -0.00063637, 0.00038997, 7.27200897, 0.99805570, 0.99897366, 0.49315563],
                             [- 6.27264565, -0.00063624, 0.00038989, 7.27200941, 0.99805610, 0.99897387, 0.49315563],
                             [- 6.27264596, -0.00063611, 0.00038981, 7.27200986, 0.99805649, 0.99897409, 0.49315568],
                             [- 6.27264628, -0.00063598, 0.00038973, 7.27201030, 0.99805689, 0.99897430, 0.49315572],
                             [- 6.27264659, -0.00063585, 0.00038965, 7.27201074, 0.99805729, 0.99897451, 0.49315579],
                             [- 6.27264690, -0.00063572, 0.00038957, 7.27201119, 0.99805769, 0.99897472, 0.49315582],
                             [- 6.27264721, -0.00063559, 0.00038949, 7.27201163, 0.99805809, 0.99897493, 0.49315589],
                             [- 6.27264753, -0.00063545, 0.00038941, 7.27201207, 0.99805849, 0.99897514, 0.49315595],
                             [- 6.27264784, -0.00063532, 0.00038933, 7.27201251, 0.99805889, 0.99897535, 0.49315600],
                             [- 6.27264815, -0.00063519, 0.00038925, 7.27201296, 0.99805929, 0.99897556, 0.49315603],
                             [- 6.27264846, -0.00063506, 0.00038917, 7.27201340, 0.99805969, 0.99897577, 0.49315606],
                             [- 6.27264877, -0.00063493, 0.00038909, 7.27201384, 0.99806008, 0.99897598, 0.49315609],
                             [- 6.27264909, -0.00063480, 0.00038901, 7.27201428, 0.99806048, 0.99897619, 0.49315614],
                             [- 6.27264940, -0.00063467, 0.00038893, 7.27201472, 0.99806088, 0.99897640, 0.49315621],
                             [- 6.27264971, -0.00063454, 0.00038885, 7.27201517, 0.99806128, 0.99897661, 0.49315622],
                             [- 6.27265002, -0.00063441, 0.00038877, 7.27201561, 0.99806168, 0.99897682, 0.49315632],
                             [- 6.27265033, -0.00063428, 0.00038869, 7.27201605, 0.99806207, 0.99897703, 0.49315636],
                             [- 6.27265064, -0.00063415, 0.00038861, 7.27201649, 0.99806247, 0.99897724, 0.49315639],
                             [- 6.27265095, -0.00063402, 0.00038853, 7.27201693, 0.99806287, 0.99897745, 0.49315644],
                             [- 6.27265126, -0.00063389, 0.00038845, 7.27201737, 0.99806326, 0.99897766, 0.49315648],
                             [- 6.27265157, -0.00063376, 0.00038837, 7.27201781, 0.99806366, 0.99897787, 0.49315654],
                             [- 6.27265189, -0.00063363, 0.00038829, 7.27201825, 0.99806406, 0.99897808, 0.49315656],
                             [- 6.27265220, -0.00063350, 0.00038821, 7.27201869, 0.99806445, 0.99897829, 0.49315663],
                             [- 6.27265251, -0.00063337, 0.00038813, 7.27201913, 0.99806485, 0.99897849, 0.49315664],
                             [- 6.27265282, -0.00063324, 0.00038805, 7.27201957, 0.99806525, 0.99897870, 0.49315672],
                             [- 6.27265313, -0.00063312, 0.00038797, 7.27202001, 0.99806564, 0.99897891, 0.49315678],
                             [- 6.27265344, -0.00063299, 0.00038789, 7.27202045, 0.99806604, 0.99897912, 0.49315679],
                             [- 6.27265375, -0.00063286, 0.00038781, 7.27202089, 0.99806643, 0.99897933, 0.49315682],
                             [- 6.27265406, -0.00063273, 0.00038773, 7.27202133, 0.99806683, 0.99897954, 0.49315688],
                             [- 6.27265436, -0.00063260, 0.00038765, 7.27202177, 0.99806722, 0.99897975, 0.49315697],
                             [- 6.27265467, -0.00063247, 0.00038758, 7.27202221, 0.99806762, 0.99897996, 0.49315698],
                             [- 6.27265498, -0.00063234, 0.00038750, 7.27202264, 0.99806801, 0.99898016, 0.49315705],
                             [- 6.27265529, -0.00063221, 0.00038742, 7.27202308, 0.99806841, 0.99898037, 0.49315710],
                             [- 6.27265560, -0.00063208, 0.00038734, 7.27202352, 0.99806880, 0.99898058, 0.49315714],
                             [- 6.27265591, -0.00063195, 0.00038726, 7.27202396, 0.99806920, 0.99898079, 0.49315716],
                             [- 6.27265622, -0.00063182, 0.00038718, 7.27202440, 0.99806959, 0.99898100, 0.49315722],
                             [- 6.27265653, -0.00063169, 0.00038710, 7.27202483, 0.99806999, 0.99898121, 0.49315726],
                             [- 6.27265684, -0.00063157, 0.00038702, 7.27202527, 0.99807038, 0.99898141, 0.49315731],
                             [- 6.27265714, -0.00063144, 0.00038694, 7.27202571, 0.99807077, 0.99898162, 0.49315737],
                             [- 6.27265745, -0.00063131, 0.00038686, 7.27202614, 0.99807117, 0.99898183, 0.49315740],
                             [- 6.27265776, -0.00063118, 0.00038678, 7.27202658, 0.99807156, 0.99898204, 0.49315744],
                             [- 6.27265807, -0.00063105, 0.00038671, 7.27202702, 0.99807195, 0.99898224, 0.49315747],
                             [- 6.27265838, -0.00063092, 0.00038663, 7.27202745, 0.99807235, 0.99898245, 0.49315751],
                             [- 6.27265868, -0.00063079, 0.00038655, 7.27202789, 0.99807274, 0.99898266, 0.49315758],
                             [- 6.27265899, -0.00063066, 0.00038647, 7.27202833, 0.99807313, 0.99898287, 0.49315760],
                             [- 6.27265930, -0.00063054, 0.00038639, 7.27202876, 0.99807353, 0.99898307, 0.49315770],
                             [- 6.27265961, -0.00063041, 0.00038631, 7.27202920, 0.99807392, 0.99898328, 0.49315771],
                             [- 6.27265991, -0.00063028, 0.00038623, 7.27202963, 0.99807431, 0.99898349, 0.49315780],
                             [- 6.27266022, -0.00063015, 0.00038615, 7.27203007, 0.99807470, 0.99898369, 0.49315780],
                             [- 6.27266053, -0.00063002, 0.00038608, 7.27203051, 0.99807510, 0.99898390, 0.49315786],
                             [- 6.27266083, -0.00062989, 0.00038600, 7.27203094, 0.99807549, 0.99898411, 0.49315792],
                             [- 6.27266114, -0.00062977, 0.00038592, 7.27203138, 0.99807588, 0.99898432, 0.49315794],
                             [- 6.27266145, -0.00062964, 0.00038584, 7.27203181, 0.99807627, 0.99898452, 0.49315802],
                             [- 6.27266175, -0.00062951, 0.00038576, 7.27203224, 0.99807666, 0.99898473, 0.49315805],
                             [- 6.27266206, -0.00062938, 0.00038568, 7.27203268, 0.99807705, 0.99898493, 0.49315808],
                             [- 6.27266237, -0.00062925, 0.00038560, 7.27203311, 0.99807744, 0.99898514, 0.49315812],
                             [- 6.27266267, -0.00062913, 0.00038553, 7.27203355, 0.99807784, 0.99898535, 0.49315818],
                             [- 6.27266298, -0.00062900, 0.00038545, 7.27203398, 0.99807823, 0.99898555, 0.49315821],
                             [- 6.27266329, -0.00062887, 0.00038537, 7.27203441, 0.99807862, 0.99898576, 0.49315828],
                             [- 6.27266359, -0.00062874, 0.00038529, 7.27203485, 0.99807901, 0.99898597, 0.49315830],
                             [- 6.27266390, -0.00062862, 0.00038521, 7.27203528, 0.99807940, 0.99898617, 0.49315836],
                             [- 6.27266420, -0.00062849, 0.00038513, 7.27203571, 0.99807979, 0.99898638, 0.49315841],
                             [- 6.27266451, -0.00062836, 0.00038506, 7.27203615, 0.99808018, 0.99898658, 0.49315847],
                             [- 6.27266481, -0.00062823, 0.00038498, 7.27203658, 0.99808057, 0.99898679, 0.49315850],
                             [- 6.27266512, -0.00062811, 0.00038490, 7.27203701, 0.99808096, 0.99898700, 0.49315856],
                             [- 6.27266542, -0.00062798, 0.00038482, 7.27203744, 0.99808135, 0.99898720, 0.49315859],
                             [- 6.27266573, -0.00062785, 0.00038474, 7.27203788, 0.99808174, 0.99898741, 0.49315862],
                             [- 6.27266603, -0.00062772, 0.00038467, 7.27203831, 0.99808213, 0.99898761, 0.49315867],
                             [- 6.27266634, -0.00062760, 0.00038459, 7.27203874, 0.99808251, 0.99898782, 0.49315875],
                             [- 6.27266664, -0.00062747, 0.00038451, 7.27203917, 0.99808290, 0.99898802, 0.49315880],
                             [- 6.27266695, -0.00062734, 0.00038443, 7.27203960, 0.99808329, 0.99898823, 0.49315883],
                             [- 6.27266725, -0.00062721, 0.00038435, 7.27204004, 0.99808368, 0.99898843, 0.49315887],
                             [- 6.27266755, -0.00062709, 0.00038428, 7.27204047, 0.99808407, 0.99898864, 0.49315892],
                             [- 6.27266786, -0.00062696, 0.00038420, 7.27204090, 0.99808446, 0.99898884, 0.49315895],
                             [- 6.27266816, -0.00062683, 0.00038412, 7.27204133, 0.99808485, 0.99898905, 0.49315899],
                             [- 6.27266847, -0.00062671, 0.00038404, 7.27204176, 0.99808523, 0.99898925, 0.49315906],
                             [- 6.27266877, -0.00062658, 0.00038396, 7.27204219, 0.99808562, 0.99898946, 0.49315910],
                             [- 6.27266907, -0.00062645, 0.00038389, 7.27204262, 0.99808601, 0.99898966, 0.49315911],
                             [- 6.27266938, -0.00062633, 0.00038381, 7.27204305, 0.99808640, 0.99898987, 0.49315917],
                             [- 6.27266968, -0.00062620, 0.00038373, 7.27204348, 0.99808678, 0.99899007, 0.49315922],
                             [- 6.27266998, -0.00062607, 0.00038365, 7.27204391, 0.99808717, 0.99899027, 0.49315926],
                             [- 6.27267028, -0.00062595, 0.00038358, 7.27204434, 0.99808756, 0.99899048, 0.49315930],
                             [- 6.27267059, -0.00062582, 0.00038350, 7.27204477, 0.99808794, 0.99899068, 0.49315934],
                             [- 6.27267089, -0.00062569, 0.00038342, 7.27204520, 0.99808833, 0.99899089, 0.49315940],
                             [- 6.27267119, -0.00062557, 0.00038334, 7.27204563, 0.99808872, 0.99899109, 0.49315945],
                             [- 6.27267150, -0.00062544, 0.00038327, 7.27204606, 0.99808910, 0.99899129, 0.49315950],
                             [- 6.27267180, -0.00062531, 0.00038319, 7.27204648, 0.99808949, 0.99899150, 0.49315953],
                             [- 6.27267210, -0.00062519, 0.00038311, 7.27204691, 0.99808988, 0.99899170, 0.49315956],
                             [- 6.27267240, -0.00062506, 0.00038303, 7.27204734, 0.99809026, 0.99899191, 0.49315961],
                             [- 6.27267270, -0.00062494, 0.00038296, 7.27204777, 0.99809065, 0.99899211, 0.49315968],
                             [- 6.27267301, -0.00062481, 0.00038288, 7.27204820, 0.99809103, 0.99899231, 0.49315973],
                             [- 6.27267331, -0.00062468, 0.00038280, 7.27204863, 0.99809142, 0.99899252, 0.49315977],
                             [- 6.27267361, -0.00062456, 0.00038272, 7.27204905, 0.99809180, 0.99899272, 0.49315980],
                             [- 6.27267391, -0.00062443, 0.00038265, 7.27204948, 0.99809219, 0.99899292, 0.49315984],
                             [- 6.27267421, -0.00062431, 0.00038257, 7.27204991, 0.99809257, 0.99899313, 0.49315989],
                             [- 6.27267451, -0.00062418, 0.00038249, 7.27205033, 0.99809296, 0.99899333, 0.49315993],
                             [- 6.27267481, -0.00062405, 0.00038241, 7.27205076, 0.99809334, 0.99899353, 0.49315999],
                             [- 6.27267512, -0.00062393, 0.00038234, 7.27205119, 0.99809373, 0.99899373, 0.49316004],
                             [- 6.27267542, -0.00062380, 0.00038226, 7.27205162, 0.99809411, 0.99899394, 0.49316007],
                             [- 6.27267572, -0.00062368, 0.00038218, 7.27205204, 0.99809450, 0.99899414, 0.49316011],
                             [- 6.27267602, -0.00062355, 0.00038211, 7.27205247, 0.99809488, 0.99899434, 0.49316016],
                             [- 6.27267632, -0.00062342, 0.00038203, 7.27205289, 0.99809527, 0.99899455, 0.49316020],
                             [- 6.27267662, -0.00062330, 0.00038195, 7.27205332, 0.99809565, 0.99899475, 0.49316023],
                             [- 6.27267692, -0.00062317, 0.00038188, 7.27205375, 0.99809603, 0.99899495, 0.49316033],
                             [- 6.27267722, -0.00062305, 0.00038180, 7.27205417, 0.99809642, 0.99899515, 0.49316032],
                             [- 6.27267752, -0.00062292, 0.00038172, 7.27205460, 0.99809680, 0.99899536, 0.49316037],
                             [- 6.27267782, -0.00062280, 0.00038164, 7.27205502, 0.99809718, 0.99899556, 0.49316043],
                             [- 6.27267812, -0.00062267, 0.00038157, 7.27205545, 0.99809757, 0.99899576, 0.49316049],
                             [- 6.27267842, -0.00062255, 0.00038149, 7.27205587, 0.99809795, 0.99899596, 0.49316053],
                             [- 6.27267872, -0.00062242, 0.00038141, 7.27205630, 0.99809833, 0.99899616, 0.49316060],
                             [- 6.27267902, -0.00062230, 0.00038134, 7.27205672, 0.99809871, 0.99899637, 0.49316061],
                             [- 6.27267932, -0.00062217, 0.00038126, 7.27205715, 0.99809910, 0.99899657, 0.49316068],
                             [- 6.27267962, -0.00062205, 0.00038118, 7.27205757, 0.99809948, 0.99899677, 0.49316072],
                             [- 6.27267992, -0.00062192, 0.00038111, 7.27205799, 0.99809986, 0.99899697, 0.49316074],
                             [- 6.27268021, -0.00062180, 0.00038103, 7.27205842, 0.99810024, 0.99899717, 0.49316079],
                             [- 6.27268051, -0.00062167, 0.00038095, 7.27205884, 0.99810062, 0.99899737, 0.49316083],
                             [- 6.27268081, -0.00062155, 0.00038088, 7.27205927, 0.99810100, 0.99899757, 0.49316088],
                             [- 6.27268111, -0.00062142, 0.00038080, 7.27205969, 0.99810139, 0.99899778, 0.49316090],
                             [- 6.27268141, -0.00062130, 0.00038072, 7.27206011, 0.99810177, 0.99899798, 0.49316099],
                             [- 6.27268171, -0.00062117, 0.00038065, 7.27206053, 0.99810215, 0.99899818, 0.49316102],
                             [- 6.27268201, -0.00062105, 0.00038057, 7.27206096, 0.99810253, 0.99899838, 0.49316107],
                             [- 6.27268230, -0.00062092, 0.00038050, 7.27206138, 0.99810291, 0.99899858, 0.49316107],
                             [- 6.27268260, -0.00062080, 0.00038042, 7.27206180, 0.99810329, 0.99899878, 0.49316112],
                             [- 6.27268290, -0.00062067, 0.00038034, 7.27206223, 0.99810367, 0.99899898, 0.49316121],
                             [- 6.27268320, -0.00062055, 0.00038027, 7.27206265, 0.99810405, 0.99899918, 0.49316122],
                             [- 6.27268350, -0.00062043, 0.00038019, 7.27206307, 0.99810443, 0.99899938, 0.49316127],
                             [- 6.27268379, -0.00062030, 0.00038011, 7.27206349, 0.99810481, 0.99899958, 0.49316133],
                             [- 6.27268409, -0.00062018, 0.00038004, 7.27206391, 0.99810519, 0.99899978, 0.49316137],
                             [- 6.27268439, -0.00062005, 0.00037996, 7.27206433, 0.99810557, 0.99899999, 0.49316141],
                             [- 6.27268468, -0.00061993, 0.00037989, 7.27206476, 0.99810595, 0.99900019, 0.49316145],
                             [- 6.27268498, -0.00061980, 0.00037981, 7.27206518, 0.99810633, 0.99900039, 0.49316150],
                             [- 6.27268528, -0.00061968, 0.00037973, 7.27206560, 0.99810671, 0.99900059, 0.49316152],
                             [- 6.27268558, -0.00061956, 0.00037966, 7.27206602, 0.99810709, 0.99900079, 0.49316158],
                             [- 6.27268587, -0.00061943, 0.00037958, 7.27206644, 0.99810747, 0.99900099, 0.49316160],
                             [- 6.27268617, -0.00061931, 0.00037951, 7.27206686, 0.99810785, 0.99900119, 0.49316174],
                             [- 6.27268647, -0.00061919, 0.00037943, 7.27206728, 0.99810823, 0.99900139, 0.49316174],
                             [- 6.27268676, -0.00061906, 0.00037935, 7.27206770, 0.99810860, 0.99900159, 0.49316173],
                             [- 6.27268706, -0.00061894, 0.00037928, 7.27206812, 0.99810898, 0.99900179, 0.49316182],
                             [- 6.27268735, -0.00061881, 0.00037920, 7.27206854, 0.99810936, 0.99900198, 0.49316185],
                             [- 6.27268765, -0.00061869, 0.00037913, 7.27206896, 0.99810974, 0.99900218, 0.49316190],
                             [- 6.27268795, -0.00061857, 0.00037905, 7.27206938, 0.99811012, 0.99900238, 0.49316194],
                             [- 6.27268824, -0.00061844, 0.00037897, 7.27206980, 0.99811049, 0.99900258, 0.49316199],
                             [- 6.27268854, -0.00061832, 0.00037890, 7.27207022, 0.99811087, 0.99900278, 0.49316201],
                             [- 6.27268883, -0.00061820, 0.00037882, 7.27207064, 0.99811125, 0.99900298, 0.49316207],
                             [- 6.27268913, -0.00061807, 0.00037875, 7.27207106, 0.99811163, 0.99900318, 0.49316215],
                             [- 6.27268942, -0.00061795, 0.00037867, 7.27207147, 0.99811200, 0.99900338, 0.49316215],
                             [- 6.27268972, -0.00061783, 0.00037860, 7.27207189, 0.99811238, 0.99900358, 0.49316222],
                             [- 6.27269001, -0.00061770, 0.00037852, 7.27207231, 0.99811276, 0.99900378, 0.49316225],
                             [- 6.27269031, -0.00061758, 0.00037844, 7.27207273, 0.99811313, 0.99900398, 0.49316226],
                             [- 6.27269060, -0.00061746, 0.00037837, 7.27207315, 0.99811351, 0.99900418, 0.49316232],
                             [- 6.27269090, -0.00061733, 0.00037829, 7.27207356, 0.99811389, 0.99900437, 0.49316241],
                             [- 6.27269119, -0.00061721, 0.00037822, 7.27207398, 0.99811426, 0.99900457, 0.49316240],
                             [- 6.27269149, -0.00061709, 0.00037814, 7.27207440, 0.99811464, 0.99900477, 0.49316244],
                             [- 6.27269178, -0.00061696, 0.00037807, 7.27207482, 0.99811502, 0.99900497, 0.49316250],
                             [- 6.27269208, -0.00061684, 0.00037799, 7.27207523, 0.99811539, 0.99900517, 0.49316255],
                             [- 6.27269237, -0.00061672, 0.00037792, 7.27207565, 0.99811577, 0.99900537, 0.49316260],
                             [- 6.27269266, -0.00061660, 0.00037784, 7.27207607, 0.99811614, 0.99900556, 0.49316268],
                             [- 6.27269296, -0.00061647, 0.00037777, 7.27207648, 0.99811652, 0.99900576, 0.49316264],
                             [- 6.27269325, -0.00061635, 0.00037769, 7.27207690, 0.99811689, 0.99900596, 0.49316274],
                             [- 6.27269354, -0.00061623, 0.00037762, 7.27207732, 0.99811727, 0.99900616, 0.49316275],
                             [- 6.27269384, -0.00061610, 0.00037754, 7.27207773, 0.99811764, 0.99900636, 0.49316280],
                             [- 6.27269413, -0.00061598, 0.00037746, 7.27207815, 0.99811802, 0.99900655, 0.49316280],
                             [- 6.27269442, -0.00061586, 0.00037739, 7.27207857, 0.99811839, 0.99900675, 0.49316288],
                             [- 6.27269472, -0.00061574, 0.00037731, 7.27207898, 0.99811877, 0.99900695, 0.49316293],
                             [- 6.27269501, -0.00061561, 0.00037724, 7.27207940, 0.99811914, 0.99900715, 0.49316299],
                             [- 6.27269530, -0.00061549, 0.00037716, 7.27207981, 0.99811952, 0.99900734, 0.49316305],
                             [- 6.27269560, -0.00061537, 0.00037709, 7.27208023, 0.99811989, 0.99900754, 0.49316305],
                             [- 6.27269589, -0.00061525, 0.00037701, 7.27208064, 0.99812026, 0.99900774, 0.49316309],
                             [- 6.27269618, -0.00061512, 0.00037694, 7.27208106, 0.99812064, 0.99900794, 0.49316319],
                             [- 6.27269647, -0.00061500, 0.00037686, 7.27208147, 0.99812101, 0.99900813, 0.49316320],
                             [- 6.27269677, -0.00061488, 0.00037679, 7.27208189, 0.99812138, 0.99900833, 0.49316326],
                             [- 6.27269706, -0.00061476, 0.00037671, 7.27208230, 0.99812176, 0.99900853, 0.49316331],
                             [- 6.27269735, -0.00061464, 0.00037664, 7.27208271, 0.99812213, 0.99900872, 0.49316334],
                             [- 6.27269764, -0.00061451, 0.00037656, 7.27208313, 0.99812250, 0.99900892, 0.49316337],
                             [- 6.27269793, -0.00061439, 0.00037649, 7.27208354, 0.99812288, 0.99900912, 0.49316343],
                             [- 6.27269823, -0.00061427, 0.00037642, 7.27208396, 0.99812325, 0.99900931, 0.49316350],
                             [- 6.27269852, -0.00061415, 0.00037634, 7.27208437, 0.99812362, 0.99900951, 0.49316348],
                             [- 6.27269881, -0.00061403, 0.00037627, 7.27208478, 0.99812399, 0.99900971, 0.49316356],
                             [- 6.27269910, -0.00061390, 0.00037619, 7.27208520, 0.99812437, 0.99900990, 0.49316359],
                             [- 6.27269939, -0.00061378, 0.00037612, 7.27208561, 0.99812474, 0.99901010, 0.49316364],
                             [- 6.27269968, -0.00061366, 0.00037604, 7.27208602, 0.99812511, 0.99901030, 0.49316368],
                             [- 6.27269997, -0.00061354, 0.00037597, 7.27208643, 0.99812548, 0.99901049, 0.49316372],
                             [- 6.27270027, -0.00061342, 0.00037589, 7.27208685, 0.99812585, 0.99901069, 0.49316379],
                             [- 6.27270056, -0.00061330, 0.00037582, 7.27208726, 0.99812623, 0.99901089, 0.49316380],
                             [- 6.27270085, -0.00061318, 0.00037574, 7.27208767, 0.99812660, 0.99901108, 0.49316383],
                             [- 6.27270114, -0.00061305, 0.00037567, 7.27208808, 0.99812697, 0.99901128, 0.49316392],
                             [- 6.27270143, -0.00061293, 0.00037559, 7.27208850, 0.99812734, 0.99901147, 0.49316393],
                             [- 6.27270172, -0.00061281, 0.00037552, 7.27208891, 0.99812771, 0.99901167, 0.49316398],
                             [- 6.27270201, -0.00061269, 0.00037545, 7.27208932, 0.99812808, 0.99901186, 0.49316403],
                             [- 6.27270230, -0.00061257, 0.00037537, 7.27208973, 0.99812845, 0.99901206, 0.49316407],
                             [- 6.27270259, -0.00061245, 0.00037530, 7.27209014, 0.99812882, 0.99901226, 0.49316411],
                             [- 6.27270288, -0.00061233, 0.00037522, 7.27209055, 0.99812919, 0.99901245, 0.49316419],
                             [- 6.27270317, -0.00061221, 0.00037515, 7.27209096, 0.99812956, 0.99901265, 0.49316420],
                             [- 6.27270346, -0.00061208, 0.00037507, 7.27209137, 0.99812993, 0.99901284, 0.49316424],
                             [- 6.27270375, -0.00061196, 0.00037500, 7.27209178, 0.99813030, 0.99901304, 0.49316431],
                             [- 6.27270404, -0.00061184, 0.00037493, 7.27209219, 0.99813067, 0.99901323, 0.49316432],
                             [- 6.27270433, -0.00061172, 0.00037485, 7.27209260, 0.99813104, 0.99901343, 0.49316435],
                             [- 6.27270462, -0.00061160, 0.00037478, 7.27209301, 0.99813141, 0.99901362, 0.49316437],
                             [- 6.27270490, -0.00061148, 0.00037470, 7.27209342, 0.99813178, 0.99901382, 0.49316447],
                             [- 6.27270519, -0.00061136, 0.00037463, 7.27209383, 0.99813215, 0.99901401, 0.49316452],
                             [- 6.27270548, -0.00061124, 0.00037456, 7.27209424, 0.99813252, 0.99901421, 0.49316453],
                             [- 6.27270577, -0.00061112, 0.00037448, 7.27209465, 0.99813289, 0.99901440, 0.49316460],
                             [- 6.27270606, -0.00061100, 0.00037441, 7.27209506, 0.99813326, 0.99901460, 0.49316462],
                             [- 6.27270635, -0.00061088, 0.00037433, 7.27209547, 0.99813362, 0.99901479, 0.49316468],
                             [- 6.27270664, -0.00061076, 0.00037426, 7.27209588, 0.99813399, 0.99901498, 0.49316472],
                             [- 6.27270692, -0.00061064, 0.00037419, 7.27209629, 0.99813436, 0.99901518, 0.49316476],
                             [- 6.27270721, -0.00061051, 0.00037411, 7.27209670, 0.99813473, 0.99901537, 0.49316483],
                             [- 6.27270750, -0.00061039, 0.00037404, 7.27209711, 0.99813510, 0.99901557, 0.49316482],
                             [- 6.27270779, -0.00061027, 0.00037396, 7.27209751, 0.99813547, 0.99901576, 0.49316493],
                             [- 6.27270808, -0.00061015, 0.00037389, 7.27209792, 0.99813583, 0.99901596, 0.49316495],
                             [- 6.27270836, -0.00061003, 0.00037382, 7.27209833, 0.99813620, 0.99901615, 0.49316498],
                             [- 6.27270865, -0.00060991, 0.00037374, 7.27209874, 0.99813657, 0.99901634, 0.49316501],
                             [- 6.27270894, -0.00060979, 0.00037367, 7.27209915, 0.99813694, 0.99901654, 0.49316504],
                             [- 6.27270923, -0.00060967, 0.00037360, 7.27209955, 0.99813730, 0.99901673, 0.49316509],
                             [- 6.27270951, -0.00060955, 0.00037352, 7.27209996, 0.99813767, 0.99901692, 0.49316516],
                             [- 6.27270980, -0.00060943, 0.00037345, 7.27210037, 0.99813804, 0.99901712, 0.49316521],
                             [- 6.27271009, -0.00060931, 0.00037338, 7.27210078, 0.99813840, 0.99901731, 0.49316522],
                             [- 6.27271038, -0.00060919, 0.00037330, 7.27210118, 0.99813877, 0.99901751, 0.49316529],
                             [- 6.27271066, -0.00060907, 0.00037323, 7.27210159, 0.99813914, 0.99901770, 0.49316533],
                             [- 6.27271095, -0.00060895, 0.00037315, 7.27210200, 0.99813950, 0.99901789, 0.49316538],
                             [- 6.27271124, -0.00060883, 0.00037308, 7.27210240, 0.99813987, 0.99901808, 0.49316545],
                             [- 6.27271152, -0.00060871, 0.00037301, 7.27210281, 0.99814023, 0.99901828, 0.49316550],
                             [- 6.27271181, -0.00060859, 0.00037293, 7.27210321, 0.99814060, 0.99901847, 0.49316550],
                             [- 6.27271209, -0.00060847, 0.00037286, 7.27210362, 0.99814097, 0.99901866, 0.49316554],
                             [- 6.27271238, -0.00060836, 0.00037279, 7.27210403, 0.99814133, 0.99901886, 0.49316555],
                             [- 6.27271267, -0.00060824, 0.00037271, 7.27210443, 0.99814170, 0.99901905, 0.49316564],
                             [- 6.27271295, -0.00060812, 0.00037264, 7.27210484, 0.99814206, 0.99901924, 0.49316569],
                             [- 6.27271324, -0.00060800, 0.00037257, 7.27210524, 0.99814243, 0.99901944, 0.49316569],
                             [- 6.27271352, -0.00060788, 0.00037249, 7.27210565, 0.99814279, 0.99901963, 0.49316572],
                             [- 6.27271381, -0.00060776, 0.00037242, 7.27210605, 0.99814316, 0.99901982, 0.49316582],
                             [- 6.27271410, -0.00060764, 0.00037235, 7.27210646, 0.99814352, 0.99902001, 0.49316585],
                             [- 6.27271438, -0.00060752, 0.00037228, 7.27210686, 0.99814389, 0.99902021, 0.49316587],
                             [- 6.27271467, -0.00060740, 0.00037220, 7.27210727, 0.99814425, 0.99902040, 0.49316592],
                             [- 6.27271495, -0.00060728, 0.00037213, 7.27210767, 0.99814461, 0.99902059, 0.49316598],
                             [- 6.27271524, -0.00060716, 0.00037206, 7.27210807, 0.99814498, 0.99902078, 0.49316599],
                             [- 6.27271552, -0.00060704, 0.00037198, 7.27210848, 0.99814534, 0.99902097, 0.49316604],
                             [- 6.27271581, -0.00060692, 0.00037191, 7.27210888, 0.99814571, 0.99902117, 0.49316608],
                             [- 6.27271609, -0.00060680, 0.00037184, 7.27210929, 0.99814607, 0.99902136, 0.49316611],
                             [- 6.27271637, -0.00060669, 0.00037176, 7.27210969, 0.99814643, 0.99902155, 0.49316616],
                             [- 6.27271666, -0.00060657, 0.00037169, 7.27211009, 0.99814680, 0.99902174, 0.49316618],
                             [- 6.27271694, -0.00060645, 0.00037162, 7.27211050, 0.99814716, 0.99902193, 0.49316624],
                             [- 6.27271723, -0.00060633, 0.00037155, 7.27211090, 0.99814752, 0.99902213, 0.49316630],
                             [- 6.27271751, -0.00060621, 0.00037147, 7.27211130, 0.99814789, 0.99902232, 0.49316633],
                             [- 6.27271780, -0.00060609, 0.00037140, 7.27211170, 0.99814825, 0.99902251, 0.49316640],
                             [- 6.27271808, -0.00060597, 0.00037133, 7.27211211, 0.99814861, 0.99902270, 0.49316642],
                             [- 6.27271836, -0.00060585, 0.00037125, 7.27211251, 0.99814897, 0.99902289, 0.49316649],
                             [- 6.27271865, -0.00060574, 0.00037118, 7.27211291, 0.99814934, 0.99902308, 0.49316653],
                             [- 6.27271893, -0.00060562, 0.00037111, 7.27211331, 0.99814970, 0.99902327, 0.49316657],
                             [- 6.27271921, -0.00060550, 0.00037104, 7.27211371, 0.99815006, 0.99902346, 0.49316659],
                             [- 6.27271950, -0.00060538, 0.00037096, 7.27211412, 0.99815042, 0.99902366, 0.49316664],
                             [- 6.27271978, -0.00060526, 0.00037089, 7.27211452, 0.99815079, 0.99902385, 0.49316670],
                             [- 6.27272006, -0.00060514, 0.00037082, 7.27211492, 0.99815115, 0.99902404, 0.49316670],
                             [- 6.27272035, -0.00060503, 0.00037075, 7.27211532, 0.99815151, 0.99902423, 0.49316673],
                             [- 6.27272063, -0.00060491, 0.00037067, 7.27211572, 0.99815187, 0.99902442, 0.49316679],
                             [- 6.27272091, -0.00060479, 0.00037060, 7.27211612, 0.99815223, 0.99902461, 0.49316685],
                             [- 6.27272120, -0.00060467, 0.00037053, 7.27211652, 0.99815259, 0.99902480, 0.49316688],
                             [- 6.27272148, -0.00060455, 0.00037046, 7.27211692, 0.99815295, 0.99902499, 0.49316692],
                             [- 6.27272176, -0.00060444, 0.00037038, 7.27211732, 0.99815331, 0.99902518, 0.49316699],
                             [- 6.27272204, -0.00060432, 0.00037031, 7.27211773, 0.99815367, 0.99902537, 0.49316698],
                             [- 6.27272232, -0.00060420, 0.00037024, 7.27211813, 0.99815404, 0.99902556, 0.49316707],
                             [- 6.27272261, -0.00060408, 0.00037017, 7.27211853, 0.99815440, 0.99902575, 0.49316710],
                             [- 6.27272289, -0.00060396, 0.00037009, 7.27211893, 0.99815476, 0.99902594, 0.49316715],
                             [- 6.27272317, -0.00060385, 0.00037002, 7.27211933, 0.99815512, 0.99902613, 0.49316718],
                             [- 6.27272345, -0.00060373, 0.00036995, 7.27211972, 0.99815548, 0.99902632, 0.49316722],
                             [- 6.27272373, -0.00060361, 0.00036988, 7.27212012, 0.99815584, 0.99902651, 0.49316726],
                             [- 6.27272402, -0.00060349, 0.00036981, 7.27212052, 0.99815620, 0.99902670, 0.49316729],
                             [- 6.27272430, -0.00060337, 0.00036973, 7.27212092, 0.99815656, 0.99902689, 0.49316735],
                             [- 6.27272458, -0.00060326, 0.00036966, 7.27212132, 0.99815691, 0.99902708, 0.49316741],
                             [- 6.27272486, -0.00060314, 0.00036959, 7.27212172, 0.99815727, 0.99902727, 0.49316741],
                             [- 6.27272514, -0.00060302, 0.00036952, 7.27212212, 0.99815763, 0.99902746, 0.49316748],
                             [- 6.27272542, -0.00060290, 0.00036945, 7.27212252, 0.99815799, 0.99902765, 0.49316753],
                             [- 6.27272570, -0.00060279, 0.00036937, 7.27212292, 0.99815835, 0.99902784, 0.49316756],
                             [- 6.27272598, -0.00060267, 0.00036930, 7.27212331, 0.99815871, 0.99902803, 0.49316761],
                             [- 6.27272626, -0.00060255, 0.00036923, 7.27212371, 0.99815907, 0.99902822, 0.49316767],
                             [- 6.27272654, -0.00060244, 0.00036916, 7.27212411, 0.99815943, 0.99902841, 0.49316768],
                             [- 6.27272683, -0.00060232, 0.00036909, 7.27212451, 0.99815978, 0.99902860, 0.49316772],
                             [- 6.27272711, -0.00060220, 0.00036901, 7.27212490, 0.99816014, 0.99902879, 0.49316778],
                             [- 6.27272739, -0.00060208, 0.00036894, 7.27212530, 0.99816050, 0.99902897, 0.49316778],
                             [- 6.27272767, -0.00060197, 0.00036887, 7.27212570, 0.99816086, 0.99902916, 0.49316783],
                             [- 6.27272795, -0.00060185, 0.00036880, 7.27212610, 0.99816122, 0.99902935, 0.49316788],
                             [- 6.27272823, -0.00060173, 0.00036873, 7.27212649, 0.99816157, 0.99902954, 0.49316787],
                             [- 6.27272851, -0.00060162, 0.00036865, 7.27212689, 0.99816193, 0.99902973, 0.49316799],
                             [- 6.27272879, -0.00060150, 0.00036858, 7.27212729, 0.99816229, 0.99902992, 0.49316801],
                             [- 6.27272907, -0.00060138, 0.00036851, 7.27212768, 0.99816265, 0.99903011, 0.49316805],
                             [- 6.27272934, -0.00060127, 0.00036844, 7.27212808, 0.99816300, 0.99903029, 0.49316810],
                             [- 6.27272962, -0.00060115, 0.00036837, 7.27212848, 0.99816336, 0.99903048, 0.49316812],
                             [- 6.27272990, -0.00060103, 0.00036830, 7.27212887, 0.99816372, 0.99903067, 0.49316815],
                             [- 6.27273018, -0.00060092, 0.00036823, 7.27212927, 0.99816407, 0.99903086, 0.49316824],
                             [- 6.27273046, -0.00060080, 0.00036815, 7.27212966, 0.99816443, 0.99903105, 0.49316825],
                             [- 6.27273074, -0.00060068, 0.00036808, 7.27213006, 0.99816479, 0.99903124, 0.49316832],
                             [- 6.27273102, -0.00060057, 0.00036801, 7.27213045, 0.99816514, 0.99903142, 0.49316835],
                             [- 6.27273130, -0.00060045, 0.00036794, 7.27213085, 0.99816550, 0.99903161, 0.49316839],
                             [- 6.27273158, -0.00060033, 0.00036787, 7.27213124, 0.99816585, 0.99903180, 0.49316844],
                             [- 6.27273186, -0.00060022, 0.00036780, 7.27213164, 0.99816621, 0.99903199, 0.49316848],
                             [- 6.27273213, -0.00060010, 0.00036773, 7.27213203, 0.99816657, 0.99903218, 0.49316850],
                             [- 6.27273241, -0.00059998, 0.00036765, 7.27213243, 0.99816692, 0.99903236, 0.49316856],
                             [- 6.27273269, -0.00059987, 0.00036758, 7.27213282, 0.99816728, 0.99903255, 0.49316861],
                             [- 6.27273297, -0.00059975, 0.00036751, 7.27213322, 0.99816763, 0.99903274, 0.49316864],
                             [- 6.27273325, -0.00059963, 0.00036744, 7.27213361, 0.99816799, 0.99903293, 0.49316869],
                             [- 6.27273352, -0.00059952, 0.00036737, 7.27213401, 0.99816834, 0.99903311, 0.49316872],
                             [- 6.27273380, -0.00059940, 0.00036730, 7.27213440, 0.99816870, 0.99903330, 0.49316876],
                             [- 6.27273408, -0.00059929, 0.00036723, 7.27213479, 0.99816905, 0.99903349, 0.49316877],
                             [- 6.27273436, -0.00059917, 0.00036716, 7.27213519, 0.99816941, 0.99903367, 0.49316887],
                             [- 6.27273463, -0.00059905, 0.00036708, 7.27213558, 0.99816976, 0.99903386, 0.49316888],
                             [- 6.27273491, -0.00059894, 0.00036701, 7.27213597, 0.99817012, 0.99903405, 0.49316895],
                             [- 6.27273519, -0.00059882, 0.00036694, 7.27213637, 0.99817047, 0.99903424, 0.49316897],
                             [- 6.27273547, -0.00059871, 0.00036687, 7.27213676, 0.99817082, 0.99903442, 0.49316901],
                             [- 6.27273574, -0.00059859, 0.00036680, 7.27213715, 0.99817118, 0.99903461, 0.49316906],
                             [- 6.27273602, -0.00059848, 0.00036673, 7.27213754, 0.99817153, 0.99903480, 0.49316911],
                             [- 6.27273630, -0.00059836, 0.00036666, 7.27213794, 0.99817189, 0.99903498, 0.49316913],
                             [- 6.27273657, -0.00059824, 0.00036659, 7.27213833, 0.99817224, 0.99903517, 0.49316914],
                             [- 6.27273685, -0.00059813, 0.00036652, 7.27213872, 0.99817259, 0.99903535, 0.49316924],
                             [- 6.27273713, -0.00059801, 0.00036645, 7.27213911, 0.99817295, 0.99903554, 0.49316926],
                             [- 6.27273740, -0.00059790, 0.00036637, 7.27213951, 0.99817330, 0.99903573, 0.49316930],
                             [- 6.27273768, -0.00059778, 0.00036630, 7.27213990, 0.99817365, 0.99903591, 0.49316938],
                             [- 6.27273796, -0.00059767, 0.00036623, 7.27214029, 0.99817400, 0.99903610, 0.49316938],
                             [- 6.27273823, -0.00059755, 0.00036616, 7.27214068, 0.99817436, 0.99903629, 0.49316938],
                             [- 6.27273851, -0.00059744, 0.00036609, 7.27214107, 0.99817471, 0.99903647, 0.49316946],
                             [- 6.27273878, -0.00059732, 0.00036602, 7.27214146, 0.99817506, 0.99903666, 0.49316948],
                             [- 6.27273906, -0.00059721, 0.00036595, 7.27214185, 0.99817541, 0.99903684, 0.49316954],
                             [- 6.27273933, -0.00059709, 0.00036588, 7.27214224, 0.99817577, 0.99903703, 0.49316961],
                             [- 6.27273961, -0.00059698, 0.00036581, 7.27214263, 0.99817612, 0.99903722, 0.49316964],
                             [- 6.27273989, -0.00059686, 0.00036574, 7.27214303, 0.99817647, 0.99903740, 0.49316966],
                             [- 6.27274016, -0.00059674, 0.00036567, 7.27214342, 0.99817682, 0.99903759, 0.49316973],
                             [- 6.27274044, -0.00059663, 0.00036560, 7.27214381, 0.99817717, 0.99903777, 0.49316976],
                             [- 6.27274071, -0.00059651, 0.00036553, 7.27214420, 0.99817752, 0.99903796, 0.49316978],
                             [- 6.27274099, -0.00059640, 0.00036546, 7.27214459, 0.99817788, 0.99903814, 0.49316985],
                             [- 6.27274126, -0.00059629, 0.00036539, 7.27214498, 0.99817823, 0.99903833, 0.49316985],
                             [- 6.27274154, -0.00059617, 0.00036532, 7.27214537, 0.99817858, 0.99903851, 0.49316990],
                             [- 6.27274181, -0.00059606, 0.00036525, 7.27214575, 0.99817893, 0.99903870, 0.49316996],
                             [- 6.27274208, -0.00059594, 0.00036517, 7.27214614, 0.99817928, 0.99903888, 0.49317001],
                             [- 6.27274236, -0.00059583, 0.00036510, 7.27214653, 0.99817963, 0.99903907, 0.49317000],
                             [- 6.27274263, -0.00059571, 0.00036503, 7.27214692, 0.99817998, 0.99903925, 0.49317004],
                             [- 6.27274291, -0.00059560, 0.00036496, 7.27214731, 0.99818033, 0.99903944, 0.49317008],
                             [- 6.27274318, -0.00059548, 0.00036489, 7.27214770, 0.99818068, 0.99903962, 0.49317014],
                             [- 6.27274346, -0.00059537, 0.00036482, 7.27214809, 0.99818103, 0.99903981, 0.49317022],
                             [- 6.27274373, -0.00059525, 0.00036475, 7.27214848, 0.99818138, 0.99903999, 0.49317024],
                             [- 6.27274400, -0.00059514, 0.00036468, 7.27214886, 0.99818173, 0.99904018, 0.49317029],
                             [- 6.27274428, -0.00059502, 0.00036461, 7.27214925, 0.99818208, 0.99904036, 0.49317032],
                             [- 6.27274455, -0.00059491, 0.00036454, 7.27214964, 0.99818243, 0.99904055, 0.49317037],
                             [- 6.27274482, -0.00059480, 0.00036447, 7.27215003, 0.99818278, 0.99904073, 0.49317042],
                             [- 6.27274510, -0.00059468, 0.00036440, 7.27215042, 0.99818313, 0.99904092, 0.49317048],
                             [- 6.27274537, -0.00059457, 0.00036433, 7.27215080, 0.99818348, 0.99904110, 0.49317047],
                             [- 6.27274564, -0.00059445, 0.00036426, 7.27215119, 0.99818383, 0.99904128, 0.49317056],
                             [- 6.27274592, -0.00059434, 0.00036419, 7.27215158, 0.99818418, 0.99904147, 0.49317055],
                             [- 6.27274619, -0.00059422, 0.00036412, 7.27215197, 0.99818452, 0.99904165, 0.49317062],
                             [- 6.27274646, -0.00059411, 0.00036405, 7.27215235, 0.99818487, 0.99904184, 0.49317064],
                             [- 6.27274674, -0.00059400, 0.00036398, 7.27215274, 0.99818522, 0.99904202, 0.49317066],
                             [- 6.27274701, -0.00059388, 0.00036391, 7.27215313, 0.99818557, 0.99904220, 0.49317073],
                             [- 6.27274728, -0.00059377, 0.00036384, 7.27215351, 0.99818592, 0.99904239, 0.49317076],
                             [- 6.27274755, -0.00059365, 0.00036377, 7.27215390, 0.99818627, 0.99904257, 0.49317082],
                             [- 6.27274783, -0.00059354, 0.00036370, 7.27215428, 0.99818661, 0.99904276, 0.49317086],
                             [- 6.27274810, -0.00059343, 0.00036363, 7.27215467, 0.99818696, 0.99904294, 0.49317091],
                             [- 6.27274837, -0.00059331, 0.00036356, 7.27215506, 0.99818731, 0.99904312, 0.49317096],
                             [- 6.27274864, -0.00059320, 0.00036349, 7.27215544, 0.99818766, 0.99904331, 0.49317092],
                             [- 6.27274891, -0.00059309, 0.00036342, 7.27215583, 0.99818800, 0.99904349, 0.49317097],
                             [- 6.27274919, -0.00059297, 0.00036335, 7.27215621, 0.99818835, 0.99904367, 0.49317108],
                             [- 6.27274946, -0.00059286, 0.00036329, 7.27215660, 0.99818870, 0.99904386, 0.49317108],
                             [- 6.27274973, -0.00059275, 0.00036322, 7.27215698, 0.99818905, 0.99904404, 0.49317113],
                             [- 6.27275000, -0.00059263, 0.00036315, 7.27215737, 0.99818939, 0.99904422, 0.49317119],
                             [- 6.27275027, -0.00059252, 0.00036308, 7.27215775, 0.99818974, 0.99904440, 0.49317125],
                             [- 6.27275054, -0.00059241, 0.00036301, 7.27215814, 0.99819009, 0.99904459, 0.49317124],
                             [- 6.27275081, -0.00059229, 0.00036294, 7.27215852, 0.99819043, 0.99904477, 0.49317130],
                             [- 6.27275109, -0.00059218, 0.00036287, 7.27215891, 0.99819078, 0.99904495, 0.49317131],
                             [- 6.27275136, -0.00059207, 0.00036280, 7.27215929, 0.99819113, 0.99904514, 0.49317136],
                             [- 6.27275163, -0.00059195, 0.00036273, 7.27215968, 0.99819147, 0.99904532, 0.49317137],
                             [- 6.27275190, -0.00059184, 0.00036266, 7.27216006, 0.99819182, 0.99904550, 0.49317148],
                             [- 6.27275217, -0.00059173, 0.00036259, 7.27216044, 0.99819216, 0.99904568, 0.49317152],
                             [- 6.27275244, -0.00059161, 0.00036252, 7.27216083, 0.99819251, 0.99904587, 0.49317157],
                             [- 6.27275271, -0.00059150, 0.00036245, 7.27216121, 0.99819285, 0.99904605, 0.49317157],
                             [- 6.27275298, -0.00059139, 0.00036238, 7.27216159, 0.99819320, 0.99904623, 0.49317163],
                             [- 6.27275325, -0.00059127, 0.00036231, 7.27216198, 0.99819355, 0.99904641, 0.49317168],
                             [- 6.27275352, -0.00059116, 0.00036224, 7.27216236, 0.99819389, 0.99904660, 0.49317166],
                             [- 6.27275379, -0.00059105, 0.00036217, 7.27216274, 0.99819424, 0.99904678, 0.49317177],
                             [- 6.27275406, -0.00059094, 0.00036211, 7.27216313, 0.99819458, 0.99904696, 0.49317179],
                             [- 6.27275433, -0.00059082, 0.00036204, 7.27216351, 0.99819493, 0.99904714, 0.49317185],
                             [- 6.27275460, -0.00059071, 0.00036197, 7.27216389, 0.99819527, 0.99904732, 0.49317189],
                             [- 6.27275487, -0.00059060, 0.00036190, 7.27216427, 0.99819561, 0.99904751, 0.49317188],
                             [- 6.27275514, -0.00059048, 0.00036183, 7.27216466, 0.99819596, 0.99904769, 0.49317197],
                             [- 6.27275541, -0.00059037, 0.00036176, 7.27216504, 0.99819630, 0.99904787, 0.49317199],
                             [- 6.27275568, -0.00059026, 0.00036169, 7.27216542, 0.99819665, 0.99904805, 0.49317206],
                             [- 6.27275595, -0.00059015, 0.00036162, 7.27216580, 0.99819699, 0.99904823, 0.49317206],
                             [- 6.27275622, -0.00059003, 0.00036155, 7.27216618, 0.99819733, 0.99904841, 0.49317210],
                             [- 6.27275649, -0.00058992, 0.00036148, 7.27216656, 0.99819768, 0.99904859, 0.49317215],
                             [- 6.27275675, -0.00058981, 0.00036141, 7.27216695, 0.99819802, 0.99904878, 0.49317218],
                             [- 6.27275702, -0.00058970, 0.00036135, 7.27216733, 0.99819837, 0.99904896, 0.49317220],
                             [- 6.27275729, -0.00058958, 0.00036128, 7.27216771, 0.99819871, 0.99904914, 0.49317224],
                             [- 6.27275756, -0.00058947, 0.00036121, 7.27216809, 0.99819905, 0.99904932, 0.49317231],
                             [- 6.27275783, -0.00058936, 0.00036114, 7.27216847, 0.99819939, 0.99904950, 0.49317228],
                             [- 6.27275810, -0.00058925, 0.00036107, 7.27216885, 0.99819974, 0.99904968, 0.49317238],
                             [- 6.27275837, -0.00058914, 0.00036100, 7.27216923, 0.99820008, 0.99904986, 0.49317241],
                             [- 6.27275863, -0.00058902, 0.00036093, 7.27216961, 0.99820042, 0.99905004, 0.49317245],
                             [- 6.27275890, -0.00058891, 0.00036086, 7.27216999, 0.99820077, 0.99905022, 0.49317253],
                             [- 6.27275917, -0.00058880, 0.00036080, 7.27217037, 0.99820111, 0.99905040, 0.49317256],
                             [- 6.27275944, -0.00058869, 0.00036073, 7.27217075, 0.99820145, 0.99905059, 0.49317260],
                             [- 6.27275971, -0.00058858, 0.00036066, 7.27217113, 0.99820179, 0.99905077, 0.49317263],
                             [- 6.27275997, -0.00058846, 0.00036059, 7.27217151, 0.99820213, 0.99905095, 0.49317270],
                             [- 6.27276024, -0.00058835, 0.00036052, 7.27217189, 0.99820248, 0.99905113, 0.49317272],
                             [- 6.27276051, -0.00058824, 0.00036045, 7.27217227, 0.99820282, 0.99905131, 0.49317275],
                             [- 6.27276078, -0.00058813, 0.00036038, 7.27217265, 0.99820316, 0.99905149, 0.49317277],
                             [- 6.27276104, -0.00058802, 0.00036032, 7.27217303, 0.99820350, 0.99905167, 0.49317281],
                             [- 6.27276131, -0.00058790, 0.00036025, 7.27217341, 0.99820384, 0.99905185, 0.49317287],
                             [- 6.27276158, -0.00058779, 0.00036018, 7.27217378, 0.99820418, 0.99905203, 0.49317289],
                             [- 6.27276184, -0.00058768, 0.00036011, 7.27217416, 0.99820453, 0.99905221, 0.49317299],
                             [- 6.27276211, -0.00058757, 0.00036004, 7.27217454, 0.99820487, 0.99905239, 0.49317296],
                             [- 6.27276238, -0.00058746, 0.00035997, 7.27217492, 0.99820521, 0.99905257, 0.49317302],
                             [- 6.27276265, -0.00058735, 0.00035991, 7.27217530, 0.99820555, 0.99905275, 0.49317307],
                             [- 6.27276291, -0.00058724, 0.00035984, 7.27217568, 0.99820589, 0.99905293, 0.49317310],
                             [- 6.27276318, -0.00058712, 0.00035977, 7.27217605, 0.99820623, 0.99905311, 0.49317316],
                             [- 6.27276344, -0.00058701, 0.00035970, 7.27217643, 0.99820657, 0.99905329, 0.49317319],
                             [- 6.27276371, -0.00058690, 0.00035963, 7.27217681, 0.99820691, 0.99905347, 0.49317321],
                             [- 6.27276398, -0.00058679, 0.00035956, 7.27217719, 0.99820725, 0.99905365, 0.49317328],
                             [- 6.27276424, -0.00058668, 0.00035950, 7.27217756, 0.99820759, 0.99905383, 0.49317329],
                             [- 6.27276451, -0.00058657, 0.00035943, 7.27217794, 0.99820793, 0.99905400, 0.49317335],
                             [- 6.27276478, -0.00058646, 0.00035936, 7.27217832, 0.99820827, 0.99905418, 0.49317337],
                             [- 6.27276504, -0.00058635, 0.00035929, 7.27217870, 0.99820861, 0.99905436, 0.49317339],
                             [- 6.27276531, -0.00058623, 0.00035922, 7.27217907, 0.99820895, 0.99905454, 0.49317346],
                             [- 6.27276557, -0.00058612, 0.00035916, 7.27217945, 0.99820929, 0.99905472, 0.49317350],
                             [- 6.27276584, -0.00058601, 0.00035909, 7.27217982, 0.99820963, 0.99905490, 0.49317354],
                             [- 6.27276610, -0.00058590, 0.00035902, 7.27218020, 0.99820997, 0.99905508, 0.49317358],
                             [- 6.27276637, -0.00058579, 0.00035895, 7.27218058, 0.99821030, 0.99905526, 0.49317361],
                             [- 6.27276663, -0.00058568, 0.00035888, 7.27218095, 0.99821064, 0.99905544, 0.49317361],
                             [- 6.27276690, -0.00058557, 0.00035882, 7.27218133, 0.99821098, 0.99905562, 0.49317370],
                             [- 6.27276716, -0.00058546, 0.00035875, 7.27218170, 0.99821132, 0.99905579, 0.49317372],
                             [- 6.27276743, -0.00058535, 0.00035868, 7.27218208, 0.99821166, 0.99905597, 0.49317378],
                             [- 6.27276769, -0.00058524, 0.00035861, 7.27218246, 0.99821200, 0.99905615, 0.49317376],
                             [- 6.27276796, -0.00058513, 0.00035854, 7.27218283, 0.99821233, 0.99905633, 0.49317387],
                             [- 6.27276822, -0.00058502, 0.00035848, 7.27218321, 0.99821267, 0.99905651, 0.49317387],
                             [- 6.27276849, -0.00058491, 0.00035841, 7.27218358, 0.99821301, 0.99905669, 0.49317394],
                             [- 6.27276875, -0.00058480, 0.00035834, 7.27218396, 0.99821335, 0.99905686, 0.49317397],
                             [- 6.27276902, -0.00058468, 0.00035827, 7.27218433, 0.99821369, 0.99905704, 0.49317398],
                             [- 6.27276928, -0.00058457, 0.00035820, 7.27218471, 0.99821402, 0.99905722, 0.49317402],
                             [- 6.27276954, -0.00058446, 0.00035814, 7.27218508, 0.99821436, 0.99905740, 0.49317408],
                             [- 6.27276981, -0.00058435, 0.00035807, 7.27218545, 0.99821470, 0.99905758, 0.49317410],
                             [- 6.27277007, -0.00058424, 0.00035800, 7.27218583, 0.99821504, 0.99905775, 0.49317417],
                             [- 6.27277034, -0.00058413, 0.00035793, 7.27218620, 0.99821537, 0.99905793, 0.49317422],
                             [- 6.27277060, -0.00058402, 0.00035787, 7.27218658, 0.99821571, 0.99905811, 0.49317425],
                             [- 6.27277086, -0.00058391, 0.00035780, 7.27218695, 0.99821605, 0.99905829, 0.49317429],
                             [- 6.27277113, -0.00058380, 0.00035773, 7.27218732, 0.99821638, 0.99905847, 0.49317428],
                             [- 6.27277139, -0.00058369, 0.00035766, 7.27218770, 0.99821672, 0.99905864, 0.49317437],
                             [- 6.27277165, -0.00058358, 0.00035760, 7.27218807, 0.99821706, 0.99905882, 0.49317446],
                             [- 6.27277192, -0.00058347, 0.00035753, 7.27218844, 0.99821739, 0.99905900, 0.49317445],
                             [- 6.27277218, -0.00058336, 0.00035746, 7.27218882, 0.99821773, 0.99905918, 0.49317449],
                             [- 6.27277244, -0.00058325, 0.00035739, 7.27218919, 0.99821806, 0.99905935, 0.49317453],
                             [- 6.27277270, -0.00058314, 0.00035733, 7.27218956, 0.99821840, 0.99905953, 0.49317457],
                             [- 6.27277297, -0.00058303, 0.00035726, 7.27218993, 0.99821874, 0.99905971, 0.49317458],
                             [- 6.27277323, -0.00058292, 0.00035719, 7.27219031, 0.99821907, 0.99905988, 0.49317467],
                             [- 6.27277349, -0.00058281, 0.00035712, 7.27219068, 0.99821941, 0.99906006, 0.49317464],
                             [- 6.27277376, -0.00058270, 0.00035706, 7.27219105, 0.99821974, 0.99906024, 0.49317472],
                             [- 6.27277402, -0.00058259, 0.00035699, 7.27219142, 0.99822008, 0.99906042, 0.49317476],
                             [- 6.27277428, -0.00058248, 0.00035692, 7.27219180, 0.99822041, 0.99906059, 0.49317478],
                             [- 6.27277454, -0.00058237, 0.00035686, 7.27219217, 0.99822075, 0.99906077, 0.49317481],
                             [- 6.27277480, -0.00058227, 0.00035679, 7.27219254, 0.99822108, 0.99906095, 0.49317486],
                             [- 6.27277507, -0.00058216, 0.00035672, 7.27219291, 0.99822142, 0.99906112, 0.49317492],
                             [- 6.27277533, -0.00058205, 0.00035665, 7.27219328, 0.99822175, 0.99906130, 0.49317497],
                             [- 6.27277559, -0.00058194, 0.00035659, 7.27219365, 0.99822209, 0.99906148, 0.49317500],
                             [- 6.27277585, -0.00058183, 0.00035652, 7.27219402, 0.99822242, 0.99906165, 0.49317504],
                             [- 6.27277611, -0.00058172, 0.00035645, 7.27219440, 0.99822275, 0.99906183, 0.49317502],
                             [- 6.27277637, -0.00058161, 0.00035639, 7.27219477, 0.99822309, 0.99906200, 0.49317510],
                             [- 6.27277664, -0.00058150, 0.00035632, 7.27219514, 0.99822342, 0.99906218, 0.49317515],
                             [- 6.27277690, -0.00058139, 0.00035625, 7.27219551, 0.99822376, 0.99906236, 0.49317515],
                             [- 6.27277716, -0.00058128, 0.00035619, 7.27219588, 0.99822409, 0.99906253, 0.49317527],
                             [- 6.27277742, -0.00058117, 0.00035612, 7.27219625, 0.99822442, 0.99906271, 0.49317521],
                             [- 6.27277768, -0.00058106, 0.00035605, 7.27219662, 0.99822476, 0.99906289, 0.49317533],
                             [- 6.27277794, -0.00058095, 0.00035598, 7.27219699, 0.99822509, 0.99906306, 0.49317536],
                             [- 6.27277820, -0.00058084, 0.00035592, 7.27219736, 0.99822542, 0.99906324, 0.49317532],
                             [- 6.27277846, -0.00058074, 0.00035585, 7.27219773, 0.99822576, 0.99906341, 0.49317542],
                             [- 6.27277872, -0.00058063, 0.00035578, 7.27219810, 0.99822609, 0.99906359, 0.49317544],
                             [- 6.27277898, -0.00058052, 0.00035572, 7.27219847, 0.99822642, 0.99906376, 0.49317549],
                             [- 6.27277924, -0.00058041, 0.00035565, 7.27219884, 0.99822675, 0.99906394, 0.49317552],
                             [- 6.27277950, -0.00058030, 0.00035558, 7.27219920, 0.99822709, 0.99906412, 0.49317558],
                             [- 6.27277977, -0.00058019, 0.00035552, 7.27219957, 0.99822742, 0.99906429, 0.49317562],
                             [- 6.27278003, -0.00058008, 0.00035545, 7.27219994, 0.99822775, 0.99906447, 0.49317566],
                             [- 6.27278029, -0.00057997, 0.00035538, 7.27220031, 0.99822808, 0.99906464, 0.49317568],
                             [- 6.27278055, -0.00057987, 0.00035532, 7.27220068, 0.99822842, 0.99906482, 0.49317573],
                             [- 6.27278080, -0.00057976, 0.00035525, 7.27220105, 0.99822875, 0.99906499, 0.49317574],
                             [- 6.27278106, -0.00057965, 0.00035518, 7.27220142, 0.99822908, 0.99906517, 0.49317581],
                             [- 6.27278132, -0.00057954, 0.00035512, 7.27220178, 0.99822941, 0.99906534, 0.49317584],
                             [- 6.27278158, -0.00057943, 0.00035505, 7.27220215, 0.99822974, 0.99906552, 0.49317591],
                             [- 6.27278184, -0.00057932, 0.00035498, 7.27220252, 0.99823008, 0.99906569, 0.49317591],
                             [- 6.27278210, -0.00057921, 0.00035492, 7.27220289, 0.99823041, 0.99906587, 0.49317595],
                             [- 6.27278236, -0.00057911, 0.00035485, 7.27220326, 0.99823074, 0.99906604, 0.49317597],
                             [- 6.27278262, -0.00057900, 0.00035479, 7.27220362, 0.99823107, 0.99906622, 0.49317601],
                             [- 6.27278288, -0.00057889, 0.00035472, 7.27220399, 0.99823140, 0.99906639, 0.49317609],
                             [- 6.27278314, -0.00057878, 0.00035465, 7.27220436, 0.99823173, 0.99906657, 0.49317610],
                             [- 6.27278340, -0.00057867, 0.00035459, 7.27220472, 0.99823206, 0.99906674, 0.49317615],
                             [- 6.27278366, -0.00057857, 0.00035452, 7.27220509, 0.99823239, 0.99906691, 0.49317621],
                             [- 6.27278392, -0.00057846, 0.00035445, 7.27220546, 0.99823272, 0.99906709, 0.49317622],
                             [- 6.27278417, -0.00057835, 0.00035439, 7.27220582, 0.99823305, 0.99906726, 0.49317632],
                             [- 6.27278443, -0.00057824, 0.00035432, 7.27220619, 0.99823338, 0.99906744, 0.49317629],
                             [- 6.27278469, -0.00057813, 0.00035425, 7.27220656, 0.99823371, 0.99906761, 0.49317636],
                             [- 6.27278495, -0.00057802, 0.00035419, 7.27220692, 0.99823404, 0.99906779, 0.49317638],
                             [- 6.27278521, -0.00057792, 0.00035412, 7.27220729, 0.99823437, 0.99906796, 0.49317641],
                             [- 6.27278547, -0.00057781, 0.00035406, 7.27220766, 0.99823470, 0.99906813, 0.49317644],
                             [- 6.27278572, -0.00057770, 0.00035399, 7.27220802, 0.99823503, 0.99906831, 0.49317646],
                             [- 6.27278598, -0.00057759, 0.00035392, 7.27220839, 0.99823536, 0.99906848, 0.49317651],
                             [- 6.27278624, -0.00057749, 0.00035386, 7.27220875, 0.99823569, 0.99906866, 0.49317654],
                             [- 6.27278650, -0.00057738, 0.00035379, 7.27220912, 0.99823602, 0.99906883, 0.49317660],
                             [- 6.27278675, -0.00057727, 0.00035373, 7.27220948, 0.99823635, 0.99906900, 0.49317666],
                             [- 6.27278701, -0.00057716, 0.00035366, 7.27220985, 0.99823668, 0.99906918, 0.49317668],
                             [- 6.27278727, -0.00057706, 0.00035359, 7.27221021, 0.99823701, 0.99906935, 0.49317676],
                             [- 6.27278753, -0.00057695, 0.00035353, 7.27221058, 0.99823734, 0.99906952, 0.49317676],
                             [- 6.27278778, -0.00057684, 0.00035346, 7.27221094, 0.99823766, 0.99906970, 0.49317678],
                             [- 6.27278804, -0.00057673, 0.00035340, 7.27221131, 0.99823799, 0.99906987, 0.49317683],
                             [- 6.27278830, -0.00057663, 0.00035333, 7.27221167, 0.99823832, 0.99907004, 0.49317687],
                             [- 6.27278855, -0.00057652, 0.00035326, 7.27221204, 0.99823865, 0.99907022, 0.49317691],
                             [- 6.27278881, -0.00057641, 0.00035320, 7.27221240, 0.99823898, 0.99907039, 0.49317697],
                             [- 6.27278907, -0.00057630, 0.00035313, 7.27221277, 0.99823931, 0.99907056, 0.49317699],
                             [- 6.27278933, -0.00057620, 0.00035307, 7.27221313, 0.99823963, 0.99907074, 0.49317702],
                             [- 6.27278958, -0.00057609, 0.00035300, 7.27221349, 0.99823996, 0.99907091, 0.49317708],
                             [- 6.27278984, -0.00057598, 0.00035294, 7.27221386, 0.99824029, 0.99907108, 0.49317710],
                             [- 6.27279009, -0.00057587, 0.00035287, 7.27221422, 0.99824062, 0.99907126, 0.49317717],
                             [- 6.27279035, -0.00057577, 0.00035280, 7.27221458, 0.99824094, 0.99907143, 0.49317720],
                             [- 6.27279061, -0.00057566, 0.00035274, 7.27221495, 0.99824127, 0.99907160, 0.49317720],
                             [- 6.27279086, -0.00057555, 0.00035267, 7.27221531, 0.99824160, 0.99907177, 0.49317727],
                             [- 6.27279112, -0.00057545, 0.00035261, 7.27221567, 0.99824193, 0.99907195, 0.49317735],
                             [- 6.27279137, -0.00057534, 0.00035254, 7.27221604, 0.99824225, 0.99907212, 0.49317732],
                             [- 6.27279163, -0.00057523, 0.00035248, 7.27221640, 0.99824258, 0.99907229, 0.49317739],
                             [- 6.27279189, -0.00057513, 0.00035241, 7.27221676, 0.99824291, 0.99907246, 0.49317738],
                             [- 6.27279214, -0.00057502, 0.00035235, 7.27221712, 0.99824323, 0.99907264, 0.49317745],
                             [- 6.27279240, -0.00057491, 0.00035228, 7.27221749, 0.99824356, 0.99907281, 0.49317748],
                             [- 6.27279265, -0.00057480, 0.00035221, 7.27221785, 0.99824389, 0.99907298, 0.49317752],
                             [- 6.27279291, -0.00057470, 0.00035215, 7.27221821, 0.99824421, 0.99907315, 0.49317757],
                             [- 6.27279316, -0.00057459, 0.00035208, 7.27221857, 0.99824454, 0.99907332, 0.49317760],
                             [- 6.27279342, -0.00057448, 0.00035202, 7.27221893, 0.99824486, 0.99907350, 0.49317764],
                             [- 6.27279367, -0.00057438, 0.00035195, 7.27221930, 0.99824519, 0.99907367, 0.49317766],
                             [- 6.27279393, -0.00057427, 0.00035189, 7.27221966, 0.99824551, 0.99907384, 0.49317770],
                             [- 6.27279418, -0.00057417, 0.00035182, 7.27222002, 0.99824584, 0.99907401, 0.49317777],
                             [- 6.27279444, -0.00057406, 0.00035176, 7.27222038, 0.99824617, 0.99907418, 0.49317774],
                             [- 6.27279469, -0.00057395, 0.00035169, 7.27222074, 0.99824649, 0.99907436, 0.49317781],
                             [- 6.27279495, -0.00057385, 0.00035163, 7.27222110, 0.99824682, 0.99907453, 0.49317792],
                             [- 6.27279520, -0.00057374, 0.00035156, 7.27222146, 0.99824714, 0.99907470, 0.49317788],
                             [- 6.27279546, -0.00057363, 0.00035150, 7.27222182, 0.99824747, 0.99907487, 0.49317793],
                             [- 6.27279571, -0.00057353, 0.00035143, 7.27222218, 0.99824779, 0.99907504, 0.49317800],
                             [- 6.27279596, -0.00057342, 0.00035137, 7.27222254, 0.99824812, 0.99907521, 0.49317798],
                             [- 6.27279622, -0.00057331, 0.00035130, 7.27222290, 0.99824844, 0.99907538, 0.49317810],
                             [- 6.27279647, -0.00057321, 0.00035124, 7.27222326, 0.99824877, 0.99907556, 0.49317811],
                             [- 6.27279673, -0.00057310, 0.00035117, 7.27222362, 0.99824909, 0.99907573, 0.49317808],
                             [- 6.27279698, -0.00057300, 0.00035111, 7.27222398, 0.99824941, 0.99907590, 0.49317818],
                             [- 6.27279723, -0.00057289, 0.00035104, 7.27222434, 0.99824974, 0.99907607, 0.49317820],
                             [- 6.27279749, -0.00057278, 0.00035098, 7.27222470, 0.99825006, 0.99907624, 0.49317824],
                             [- 6.27279774, -0.00057268, 0.00035091, 7.27222506, 0.99825039, 0.99907641, 0.49317828],
                             [- 6.27279799, -0.00057257, 0.00035085, 7.27222542, 0.99825071, 0.99907658, 0.49317828],
                             [- 6.27279825, -0.00057247, 0.00035078, 7.27222578, 0.99825103, 0.99907675, 0.49317836],
                             [- 6.27279850, -0.00057236, 0.00035072, 7.27222614, 0.99825136, 0.99907692, 0.49317841],
                             [- 6.27279875, -0.00057225, 0.00035065, 7.27222650, 0.99825168, 0.99907709, 0.49317844],
                             [- 6.27279901, -0.00057215, 0.00035059, 7.27222686, 0.99825200, 0.99907727, 0.49317846],
                             [- 6.27279926, -0.00057204, 0.00035052, 7.27222722, 0.99825233, 0.99907744, 0.49317853],
                             [- 6.27279951, -0.00057194, 0.00035046, 7.27222758, 0.99825265, 0.99907761, 0.49317855],
                             [- 6.27279977, -0.00057183, 0.00035039, 7.27222793, 0.99825297, 0.99907778, 0.49317862],
                             [- 6.27280002, -0.00057173, 0.00035033, 7.27222829, 0.99825330, 0.99907795, 0.49317862],
                             [- 6.27280027, -0.00057162, 0.00035026, 7.27222865, 0.99825362, 0.99907812, 0.49317865],
                             [- 6.27280052, -0.00057152, 0.00035020, 7.27222901, 0.99825394, 0.99907829, 0.49317872],
                             [- 6.27280078, -0.00057141, 0.00035013, 7.27222937, 0.99825426, 0.99907846, 0.49317874],
                             [- 6.27280103, -0.00057130, 0.00035007, 7.27222972, 0.99825459, 0.99907863, 0.49317875],
                             [- 6.27280128, -0.00057120, 0.00035000, 7.27223008, 0.99825491, 0.99907880, 0.49317879],
                             [- 6.27280153, -0.00057109, 0.00034994, 7.27223044, 0.99825523, 0.99907897, 0.49317885],
                             [- 6.27280178, -0.00057099, 0.00034987, 7.27223080, 0.99825555, 0.99907914, 0.49317886],
                             [- 6.27280204, -0.00057088, 0.00034981, 7.27223115, 0.99825587, 0.99907931, 0.49317891],
                             [- 6.27280229, -0.00057078, 0.00034974, 7.27223151, 0.99825620, 0.99907948, 0.49317896],
                             [- 6.27280254, -0.00057067, 0.00034968, 7.27223187, 0.99825652, 0.99907965, 0.49317901],
                             [- 6.27280279, -0.00057057, 0.00034962, 7.27223222, 0.99825684, 0.99907982, 0.49317904],
                             [- 6.27280304, -0.00057046, 0.00034955, 7.27223258, 0.99825716, 0.99907999, 0.49317904],
                             [- 6.27280329, -0.00057036, 0.00034949, 7.27223294, 0.99825748, 0.99908016, 0.49317910],
                             [- 6.27280355, -0.00057025, 0.00034942, 7.27223329, 0.99825780, 0.99908033, 0.49317919],
                             [- 6.27280380, -0.00057015, 0.00034936, 7.27223365, 0.99825812, 0.99908050, 0.49317919],
                             [- 6.27280405, -0.00057004, 0.00034929, 7.27223401, 0.99825844, 0.99908066, 0.49317919],
                             [- 6.27280430, -0.00056994, 0.00034923, 7.27223436, 0.99825877, 0.99908083, 0.49317929],
                             [- 6.27280455, -0.00056983, 0.00034916, 7.27223472, 0.99825909, 0.99908100, 0.49317932],
                             [- 6.27280480, -0.00056973, 0.00034910, 7.27223507, 0.99825941, 0.99908117, 0.49317933],
                             [- 6.27280505, -0.00056962, 0.00034904, 7.27223543, 0.99825973, 0.99908134, 0.49317936],
                             [- 6.27280530, -0.00056952, 0.00034897, 7.27223579, 0.99826005, 0.99908151, 0.49317942],
                             [- 6.27280555, -0.00056941, 0.00034891, 7.27223614, 0.99826037, 0.99908168, 0.49317941],
                             [- 6.27280580, -0.00056931, 0.00034884, 7.27223650, 0.99826069, 0.99908185, 0.49317944],
                             [- 6.27280606, -0.00056920, 0.00034878, 7.27223685, 0.99826101, 0.99908202, 0.49317951],
                             [- 6.27280631, -0.00056910, 0.00034872, 7.27223721, 0.99826133, 0.99908219, 0.49317958],
                             [- 6.27280656, -0.00056899, 0.00034865, 7.27223756, 0.99826165, 0.99908235, 0.49317962],
                             [- 6.27280681, -0.00056889, 0.00034859, 7.27223792, 0.99826197, 0.99908252, 0.49317961],
                             [- 6.27280706, -0.00056878, 0.00034852, 7.27223827, 0.99826229, 0.99908269, 0.49317963],
                             [- 6.27280731, -0.00056868, 0.00034846, 7.27223863, 0.99826261, 0.99908286, 0.49317970],
                             [- 6.27280756, -0.00056858, 0.00034839, 7.27223898, 0.99826293, 0.99908303, 0.49317974],
                             [- 6.27280781, -0.00056847, 0.00034833, 7.27223933, 0.99826324, 0.99908320, 0.49317978],
                             [- 6.27280806, -0.00056837, 0.00034827, 7.27223969, 0.99826356, 0.99908337, 0.49317977],
                             [- 6.27280830, -0.00056826, 0.00034820, 7.27224004, 0.99826388, 0.99908353, 0.49317984],
                             [- 6.27280855, -0.00056816, 0.00034814, 7.27224040, 0.99826420, 0.99908370, 0.49317991],
                             [- 6.27280880, -0.00056805, 0.00034807, 7.27224075, 0.99826452, 0.99908387, 0.49317993],
                             [- 6.27280905, -0.00056795, 0.00034801, 7.27224110, 0.99826484, 0.99908404, 0.49317997],
                             [- 6.27280930, -0.00056785, 0.00034795, 7.27224146, 0.99826516, 0.99908421, 0.49317999],
                             [- 6.27280955, -0.00056774, 0.00034788, 7.27224181, 0.99826547, 0.99908438, 0.49318004],
                             [- 6.27280980, -0.00056764, 0.00034782, 7.27224216, 0.99826579, 0.99908454, 0.49318009],
                             [- 6.27281005, -0.00056753, 0.00034776, 7.27224252, 0.99826611, 0.99908471, 0.49318013],
                             [- 6.27281030, -0.00056743, 0.00034769, 7.27224287, 0.99826643, 0.99908488, 0.49318014],
                             [- 6.27281055, -0.00056733, 0.00034763, 7.27224322, 0.99826675, 0.99908505, 0.49318020],
                             [- 6.27281080, -0.00056722, 0.00034756, 7.27224357, 0.99826706, 0.99908521, 0.49318023],
                             [- 6.27281104, -0.00056712, 0.00034750, 7.27224393, 0.99826738, 0.99908538, 0.49318021],
                             [- 6.27281129, -0.00056701, 0.00034744, 7.27224428, 0.99826770, 0.99908555, 0.49318033],
                             [- 6.27281154, -0.00056691, 0.00034737, 7.27224463, 0.99826802, 0.99908572, 0.49318033],
                             [- 6.27281179, -0.00056681, 0.00034731, 7.27224498, 0.99826833, 0.99908588, 0.49318036],
                             [- 6.27281204, -0.00056670, 0.00034725, 7.27224534, 0.99826865, 0.99908605, 0.49318042],
                             [- 6.27281229, -0.00056660, 0.00034718, 7.27224569, 0.99826897, 0.99908622, 0.49318046],
                             [- 6.27281253, -0.00056649, 0.00034712, 7.27224604, 0.99826929, 0.99908639, 0.49318048],
                             [- 6.27281278, -0.00056639, 0.00034706, 7.27224639, 0.99826960, 0.99908655, 0.49318053],
                             [- 6.27281303, -0.00056629, 0.00034699, 7.27224674, 0.99826992, 0.99908672, 0.49318056],
                             [- 6.27281328, -0.00056618, 0.00034693, 7.27224709, 0.99827024, 0.99908689, 0.49318059],
                             [- 6.27281353, -0.00056608, 0.00034686, 7.27224745, 0.99827055, 0.99908705, 0.49318066],
                             [- 6.27281377, -0.00056598, 0.00034680, 7.27224780, 0.99827087, 0.99908722, 0.49318067],
                             [- 6.27281402, -0.00056587, 0.00034674, 7.27224815, 0.99827119, 0.99908739, 0.49318073],
                             [- 6.27281427, -0.00056577, 0.00034667, 7.27224850, 0.99827150, 0.99908756, 0.49318072],
                             [- 6.27281452, -0.00056567, 0.00034661, 7.27224885, 0.99827182, 0.99908772, 0.49318077],
                             [- 6.27281476, -0.00056556, 0.00034655, 7.27224920, 0.99827213, 0.99908789, 0.49318079],
                             [- 6.27281501, -0.00056546, 0.00034648, 7.27224955, 0.99827245, 0.99908806, 0.49318086],
                             [- 6.27281526, -0.00056536, 0.00034642, 7.27224990, 0.99827276, 0.99908822, 0.49318086],
                             [- 6.27281550, -0.00056525, 0.00034636, 7.27225025, 0.99827308, 0.99908839, 0.49318091],
                             [- 6.27281575, -0.00056515, 0.00034629, 7.27225060, 0.99827340, 0.99908856, 0.49318093],
                             [- 6.27281600, -0.00056505, 0.00034623, 7.27225095, 0.99827371, 0.99908872, 0.49318097],
                             [- 6.27281624, -0.00056494, 0.00034617, 7.27225130, 0.99827403, 0.99908889, 0.49318102],
                             [- 6.27281649, -0.00056484, 0.00034610, 7.27225165, 0.99827434, 0.99908905, 0.49318109],
                             [- 6.27281674, -0.00056474, 0.00034604, 7.27225200, 0.99827466, 0.99908922, 0.49318111],
                             [- 6.27281698, -0.00056463, 0.00034598, 7.27225235, 0.99827497, 0.99908939, 0.49318114],
                             [- 6.27281723, -0.00056453, 0.00034592, 7.27225270, 0.99827529, 0.99908955, 0.49318118],
                             [- 6.27281748, -0.00056443, 0.00034585, 7.27225305, 0.99827560, 0.99908972, 0.49318122],
                             [- 6.27281772, -0.00056433, 0.00034579, 7.27225340, 0.99827591, 0.99908988, 0.49318124],
                             [- 6.27281797, -0.00056422, 0.00034573, 7.27225375, 0.99827623, 0.99909005, 0.49318128],
                             [- 6.27281822, -0.00056412, 0.00034566, 7.27225409, 0.99827654, 0.99909022, 0.49318131],
                             [- 6.27281846, -0.00056402, 0.00034560, 7.27225444, 0.99827686, 0.99909038, 0.49318135],
                             [- 6.27281871, -0.00056391, 0.00034554, 7.27225479, 0.99827717, 0.99909055, 0.49318140],
                             [- 6.27281895, -0.00056381, 0.00034547, 7.27225514, 0.99827749, 0.99909071, 0.49318139],
                             [- 6.27281920, -0.00056371, 0.00034541, 7.27225549, 0.99827780, 0.99909088, 0.49318148],
                             [- 6.27281944, -0.00056361, 0.00034535, 7.27225584, 0.99827811, 0.99909105, 0.49318152],
                             [- 6.27281969, -0.00056350, 0.00034528, 7.27225619, 0.99827843, 0.99909121, 0.49318156],
                             [- 6.27281993, -0.00056340, 0.00034522, 7.27225653, 0.99827874, 0.99909138, 0.49318159],
                             [- 6.27282018, -0.00056330, 0.00034516, 7.27225688, 0.99827905, 0.99909154, 0.49318163],
                             [- 6.27282043, -0.00056320, 0.00034510, 7.27225723, 0.99827937, 0.99909171, 0.49318166],
                             [- 6.27282067, -0.00056309, 0.00034503, 7.27225758, 0.99827968, 0.99909187, 0.49318168],
                             [- 6.27282092, -0.00056299, 0.00034497, 7.27225792, 0.99827999, 0.99909204, 0.49318179],
                             [- 6.27282116, -0.00056289, 0.00034491, 7.27225827, 0.99828031, 0.99909220, 0.49318171],
                             [- 6.27282141, -0.00056279, 0.00034485, 7.27225862, 0.99828062, 0.99909237, 0.49318177],
                             [- 6.27282165, -0.00056268, 0.00034478, 7.27225897, 0.99828093, 0.99909253, 0.49318188],
                             [- 6.27282189, -0.00056258, 0.00034472, 7.27225931, 0.99828124, 0.99909270, 0.49318185],
                             [- 6.27282214, -0.00056248, 0.00034466, 7.27225966, 0.99828156, 0.99909286, 0.49318190],
                             [- 6.27282238, -0.00056238, 0.00034459, 7.27226001, 0.99828187, 0.99909303, 0.49318196],
                             [- 6.27282263, -0.00056228, 0.00034453, 7.27226035, 0.99828218, 0.99909319, 0.49318197],
                             [- 6.27282287, -0.00056217, 0.00034447, 7.27226070, 0.99828249, 0.99909336, 0.49318203],
                             [- 6.27282312, -0.00056207, 0.00034441, 7.27226104, 0.99828281, 0.99909352, 0.49318204],
                             [- 6.27282336, -0.00056197, 0.00034434, 7.27226139, 0.99828312, 0.99909369, 0.49318210],
                             [- 6.27282360, -0.00056187, 0.00034428, 7.27226174, 0.99828343, 0.99909385, 0.49318213],
                             [- 6.27282385, -0.00056177, 0.00034422, 7.27226208, 0.99828374, 0.99909402, 0.49318217],
                             [- 6.27282409, -0.00056166, 0.00034416, 7.27226243, 0.99828405, 0.99909418, 0.49318219],
                             [- 6.27282434, -0.00056156, 0.00034409, 7.27226277, 0.99828436, 0.99909434, 0.49318217],
                             [- 6.27282458, -0.00056146, 0.00034403, 7.27226312, 0.99828467, 0.99909451, 0.49318229],
                             [- 6.27282482, -0.00056136, 0.00034397, 7.27226347, 0.99828499, 0.99909467, 0.49318235],
                             [- 6.27282507, -0.00056126, 0.00034391, 7.27226381, 0.99828530, 0.99909484, 0.49318234],
                             [- 6.27282531, -0.00056115, 0.00034384, 7.27226416, 0.99828561, 0.99909500, 0.49318238],
                             [- 6.27282555, -0.00056105, 0.00034378, 7.27226450, 0.99828592, 0.99909516, 0.49318244],
                             [- 6.27282580, -0.00056095, 0.00034372, 7.27226485, 0.99828623, 0.99909533, 0.49318245],
                             [- 6.27282604, -0.00056085, 0.00034366, 7.27226519, 0.99828654, 0.99909549, 0.49318251],
                             [- 6.27282628, -0.00056075, 0.00034360, 7.27226554, 0.99828685, 0.99909566, 0.49318249],
                             [- 6.27282653, -0.00056065, 0.00034353, 7.27226588, 0.99828716, 0.99909582, 0.49318260],
                             [- 6.27282677, -0.00056055, 0.00034347, 7.27226622, 0.99828747, 0.99909598, 0.49318256],
                             [- 6.27282701, -0.00056044, 0.00034341, 7.27226657, 0.99828778, 0.99909615, 0.49318268],
                             [- 6.27282725, -0.00056034, 0.00034335, 7.27226691, 0.99828809, 0.99909631, 0.49318269],
                             [- 6.27282750, -0.00056024, 0.00034328, 7.27226726, 0.99828840, 0.99909648, 0.49318273],
                             [- 6.27282774, -0.00056014, 0.00034322, 7.27226760, 0.99828871, 0.99909664, 0.49318276],
                             [- 6.27282798, -0.00056004, 0.00034316, 7.27226794, 0.99828902, 0.99909680, 0.49318278],
                             [- 6.27282822, -0.00055994, 0.00034310, 7.27226829, 0.99828933, 0.99909697, 0.49318281],
                             [- 6.27282847, -0.00055984, 0.00034304, 7.27226863, 0.99828964, 0.99909713, 0.49318285],
                             [- 6.27282871, -0.00055973, 0.00034297, 7.27226898, 0.99828995, 0.99909729, 0.49318289],
                             [- 6.27282895, -0.00055963, 0.00034291, 7.27226932, 0.99829026, 0.99909746, 0.49318294],
                             [- 6.27282919, -0.00055953, 0.00034285, 7.27226966, 0.99829057, 0.99909762, 0.49318296],
                             [- 6.27282944, -0.00055943, 0.00034279, 7.27227000, 0.99829088, 0.99909778, 0.49318295],
                             [- 6.27282968, -0.00055933, 0.00034273, 7.27227035, 0.99829119, 0.99909794, 0.49318302],
                             [- 6.27282992, -0.00055923, 0.00034266, 7.27227069, 0.99829150, 0.99909811, 0.49318310],
                             [- 6.27283016, -0.00055913, 0.00034260, 7.27227103, 0.99829180, 0.99909827, 0.49318311],
                             [- 6.27283040, -0.00055903, 0.00034254, 7.27227138, 0.99829211, 0.99909843, 0.49318314],
                             [- 6.27283064, -0.00055893, 0.00034248, 7.27227172, 0.99829242, 0.99909860, 0.49318318],
                             [- 6.27283089, -0.00055882, 0.00034242, 7.27227206, 0.99829273, 0.99909876, 0.49318320],
                             [- 6.27283113, -0.00055872, 0.00034235, 7.27227240, 0.99829304, 0.99909892, 0.49318320],
                             [- 6.27283137, -0.00055862, 0.00034229, 7.27227274, 0.99829335, 0.99909908, 0.49318327],
                             [- 6.27283161, -0.00055852, 0.00034223, 7.27227309, 0.99829365, 0.99909925, 0.49318333],
                             [- 6.27283185, -0.00055842, 0.00034217, 7.27227343, 0.99829396, 0.99909941, 0.49318333],
                             [- 6.27283209, -0.00055832, 0.00034211, 7.27227377, 0.99829427, 0.99909957, 0.49318336],
                             [- 6.27283233, -0.00055822, 0.00034204, 7.27227411, 0.99829458, 0.99909973, 0.49318343],
                             [- 6.27283257, -0.00055812, 0.00034198, 7.27227445, 0.99829489, 0.99909990, 0.49318344],
                             [- 6.27283281, -0.00055802, 0.00034192, 7.27227479, 0.99829519, 0.99910006, 0.49318351],
                             [- 6.27283305, -0.00055792, 0.00034186, 7.27227514, 0.99829550, 0.99910022, 0.49318353],
                             [- 6.27283329, -0.00055782, 0.00034180, 7.27227548, 0.99829581, 0.99910038, 0.49318358],
                             [- 6.27283353, -0.00055772, 0.00034174, 7.27227582, 0.99829611, 0.99910055, 0.49318362],
                             [- 6.27283378, -0.00055762, 0.00034168, 7.27227616, 0.99829642, 0.99910071, 0.49318360],
                             [- 6.27283402, -0.00055752, 0.00034161, 7.27227650, 0.99829673, 0.99910087, 0.49318367],
                             [- 6.27283426, -0.00055742, 0.00034155, 7.27227684, 0.99829704, 0.99910103, 0.49318371],
                             [- 6.27283450, -0.00055732, 0.00034149, 7.27227718, 0.99829734, 0.99910119, 0.49318374],
                             [- 6.27283474, -0.00055722, 0.00034143, 7.27227752, 0.99829765, 0.99910136, 0.49318380],
                             [- 6.27283498, -0.00055712, 0.00034137, 7.27227786, 0.99829796, 0.99910152, 0.49318376],
                             [- 6.27283522, -0.00055701, 0.00034131, 7.27227820, 0.99829826, 0.99910168, 0.49318386],
                             [- 6.27283546, -0.00055691, 0.00034124, 7.27227854, 0.99829857, 0.99910184, 0.49318389],
                             [- 6.27283570, -0.00055681, 0.00034118, 7.27227888, 0.99829887, 0.99910200, 0.49318390],
                             [- 6.27283593, -0.00055671, 0.00034112, 7.27227922, 0.99829918, 0.99910216, 0.49318396],
                             [- 6.27283617, -0.00055661, 0.00034106, 7.27227956, 0.99829949, 0.99910233, 0.49318401],
                             [- 6.27283641, -0.00055651, 0.00034100, 7.27227990, 0.99829979, 0.99910249, 0.49318410],
                             [- 6.27283665, -0.00055641, 0.00034094, 7.27228024, 0.99830010, 0.99910265, 0.49318407],
                             [- 6.27283689, -0.00055631, 0.00034088, 7.27228058, 0.99830040, 0.99910281, 0.49318407],
                             [- 6.27283713, -0.00055621, 0.00034081, 7.27228092, 0.99830071, 0.99910297, 0.49318414],
                             [- 6.27283737, -0.00055611, 0.00034075, 7.27228126, 0.99830101, 0.99910313, 0.49318415],
                             [- 6.27283761, -0.00055601, 0.00034069, 7.27228159, 0.99830132, 0.99910329, 0.49318423],
                             [- 6.27283785, -0.00055591, 0.00034063, 7.27228193, 0.99830163, 0.99910345, 0.49318423],
                             [- 6.27283809, -0.00055581, 0.00034057, 7.27228227, 0.99830193, 0.99910362, 0.49318427],
                             [- 6.27283833, -0.00055571, 0.00034051, 7.27228261, 0.99830224, 0.99910378, 0.49318428],
                             [- 6.27283856, -0.00055562, 0.00034045, 7.27228295, 0.99830254, 0.99910394, 0.49318433],
                             [- 6.27283880, -0.00055552, 0.00034039, 7.27228329, 0.99830285, 0.99910410, 0.49318443],
                             [- 6.27283904, -0.00055542, 0.00034033, 7.27228363, 0.99830315, 0.99910426, 0.49318443],
                             [- 6.27283928, -0.00055532, 0.00034026, 7.27228396, 0.99830345, 0.99910442, 0.49318449],
                             [- 6.27283952, -0.00055522, 0.00034020, 7.27228430, 0.99830376, 0.99910458, 0.49318449],
                             [- 6.27283976, -0.00055512, 0.00034014, 7.27228464, 0.99830406, 0.99910474, 0.49318454],
                             [- 6.27283999, -0.00055502, 0.00034008, 7.27228498, 0.99830437, 0.99910490, 0.49318457],
                             [- 6.27284023, -0.00055492, 0.00034002, 7.27228531, 0.99830467, 0.99910506, 0.49318459],
                             [- 6.27284047, -0.00055482, 0.00033996, 7.27228565, 0.99830498, 0.99910522, 0.49318460],
                             [- 6.27284071, -0.00055472, 0.00033990, 7.27228599, 0.99830528, 0.99910538, 0.49318465],
                             [- 6.27284095, -0.00055462, 0.00033984, 7.27228633, 0.99830558, 0.99910554, 0.49318472],
                             [- 6.27284118, -0.00055452, 0.00033978, 7.27228666, 0.99830589, 0.99910570, 0.49318477],
                             [- 6.27284142, -0.00055442, 0.00033972, 7.27228700, 0.99830619, 0.99910586, 0.49318479],
                             [- 6.27284166, -0.00055432, 0.00033965, 7.27228734, 0.99830649, 0.99910602, 0.49318483],
                             [- 6.27284190, -0.00055422, 0.00033959, 7.27228767, 0.99830680, 0.99910618, 0.49318487],
                             [- 6.27284213, -0.00055412, 0.00033953, 7.27228801, 0.99830710, 0.99910634, 0.49318489],
                             [- 6.27284237, -0.00055402, 0.00033947, 7.27228835, 0.99830740, 0.99910650, 0.49318490],
                             [- 6.27284261, -0.00055393, 0.00033941, 7.27228868, 0.99830771, 0.99910666, 0.49318496],
                             [- 6.27284285, -0.00055383, 0.00033935, 7.27228902, 0.99830801, 0.99910682, 0.49318499],
                             [- 6.27284308, -0.00055373, 0.00033929, 7.27228936, 0.99830831, 0.99910698, 0.49318499],
                             [- 6.27284332, -0.00055363, 0.00033923, 7.27228969, 0.99830861, 0.99910714, 0.49318509],
                             [- 6.27284356, -0.00055353, 0.00033917, 7.27229003, 0.99830892, 0.99910730, 0.49318511],
                             [- 6.27284379, -0.00055343, 0.00033911, 7.27229036, 0.99830922, 0.99910746, 0.49318515],
                             [- 6.27284403, -0.00055333, 0.00033905, 7.27229070, 0.99830952, 0.99910762, 0.49318516],
                             [- 6.27284427, -0.00055323, 0.00033899, 7.27229103, 0.99830982, 0.99910778, 0.49318524],
                             [- 6.27284450, -0.00055313, 0.00033893, 7.27229137, 0.99831013, 0.99910794, 0.49318524],
                             [- 6.27284474, -0.00055303, 0.00033887, 7.27229170, 0.99831043, 0.99910810, 0.49318528],
                             [- 6.27284498, -0.00055294, 0.00033880, 7.27229204, 0.99831073, 0.99910826, 0.49318529],
                             [- 6.27284521, -0.00055284, 0.00033874, 7.27229237, 0.99831103, 0.99910842, 0.49318533],
                             [- 6.27284545, -0.00055274, 0.00033868, 7.27229271, 0.99831133, 0.99910858, 0.49318539],
                             [- 6.27284568, -0.00055264, 0.00033862, 7.27229304, 0.99831164, 0.99910874, 0.49318542],
                             [- 6.27284592, -0.00055254, 0.00033856, 7.27229338, 0.99831194, 0.99910890, 0.49318543],
                             [- 6.27284616, -0.00055244, 0.00033850, 7.27229371, 0.99831224, 0.99910906, 0.49318547],
                             [- 6.27284639, -0.00055234, 0.00033844, 7.27229405, 0.99831254, 0.99910921, 0.49318555],
                             [- 6.27284663, -0.00055225, 0.00033838, 7.27229438, 0.99831284, 0.99910937, 0.49318557],
                             [- 6.27284686, -0.00055215, 0.00033832, 7.27229472, 0.99831314, 0.99910953, 0.49318561],
                             [- 6.27284710, -0.00055205, 0.00033826, 7.27229505, 0.99831344, 0.99910969, 0.49318565],
                             [- 6.27284733, -0.00055195, 0.00033820, 7.27229538, 0.99831374, 0.99910985, 0.49318570],
                             [- 6.27284757, -0.00055185, 0.00033814, 7.27229572, 0.99831404, 0.99911001, 0.49318569],
                             [- 6.27284781, -0.00055175, 0.00033808, 7.27229605, 0.99831435, 0.99911017, 0.49318574],
                             [- 6.27284804, -0.00055165, 0.00033802, 7.27229639, 0.99831465, 0.99911033, 0.49318575],
                             [- 6.27284828, -0.00055156, 0.00033796, 7.27229672, 0.99831495, 0.99911048, 0.49318581],
                             [- 6.27284851, -0.00055146, 0.00033790, 7.27229705, 0.99831525, 0.99911064, 0.49318582],
                             [- 6.27284875, -0.00055136, 0.00033784, 7.27229739, 0.99831555, 0.99911080, 0.49318586],
                             [- 6.27284898, -0.00055126, 0.00033778, 7.27229772, 0.99831585, 0.99911096, 0.49318588],
                             [- 6.27284922, -0.00055116, 0.00033772, 7.27229805, 0.99831615, 0.99911112, 0.49318595],
                             [- 6.27284945, -0.00055107, 0.00033766, 7.27229838, 0.99831645, 0.99911128, 0.49318599],
                             [- 6.27284968, -0.00055097, 0.00033760, 7.27229872, 0.99831675, 0.99911143, 0.49318604],
                             [- 6.27284992, -0.00055087, 0.00033754, 7.27229905, 0.99831705, 0.99911159, 0.49318604],
                             [- 6.27285015, -0.00055077, 0.00033748, 7.27229938, 0.99831735, 0.99911175, 0.49318612],
                             [- 6.27285039, -0.00055067, 0.00033742, 7.27229971, 0.99831765, 0.99911191, 0.49318613],
                             [- 6.27285062, -0.00055058, 0.00033736, 7.27230005, 0.99831794, 0.99911207, 0.49318622],
                             [- 6.27285086, -0.00055048, 0.00033730, 7.27230038, 0.99831824, 0.99911223, 0.49318623],
                             [- 6.27285109, -0.00055038, 0.00033724, 7.27230071, 0.99831854, 0.99911238, 0.49318626],
                             [- 6.27285133, -0.00055028, 0.00033718, 7.27230104, 0.99831884, 0.99911254, 0.49318626],
                             [- 6.27285156, -0.00055018, 0.00033712, 7.27230138, 0.99831914, 0.99911270, 0.49318628],
                             [- 6.27285179, -0.00055009, 0.00033706, 7.27230171, 0.99831944, 0.99911286, 0.49318633],
                             [- 6.27285203, -0.00054999, 0.00033700, 7.27230204, 0.99831974, 0.99911301, 0.49318637],
                             [- 6.27285226, -0.00054989, 0.00033694, 7.27230237, 0.99832004, 0.99911317, 0.49318637],
                             [- 6.27285249, -0.00054979, 0.00033688, 7.27230270, 0.99832034, 0.99911333, 0.49318644],
                             [- 6.27285273, -0.00054970, 0.00033682, 7.27230303, 0.99832063, 0.99911349, 0.49318645],
                             [- 6.27285296, -0.00054960, 0.00033676, 7.27230336, 0.99832093, 0.99911364, 0.49318646],
                             [- 6.27285319, -0.00054950, 0.00033670, 7.27230369, 0.99832123, 0.99911380, 0.49318651],
                             [- 6.27285343, -0.00054940, 0.00033664, 7.27230403, 0.99832153, 0.99911396, 0.49318650],
                             [- 6.27285366, -0.00054931, 0.00033658, 7.27230436, 0.99832183, 0.99911412, 0.49318659],
                             [- 6.27285389, -0.00054921, 0.00033652, 7.27230469, 0.99832212, 0.99911427, 0.49318666],
                             [- 6.27285413, -0.00054911, 0.00033646, 7.27230502, 0.99832242, 0.99911443, 0.49318665],
                             [- 6.27285436, -0.00054901, 0.00033640, 7.27230535, 0.99832272, 0.99911459, 0.49318670],
                             [- 6.27285459, -0.00054892, 0.00033634, 7.27230568, 0.99832302, 0.99911474, 0.49318673],
                             [- 6.27285483, -0.00054882, 0.00033628, 7.27230601, 0.99832332, 0.99911490, 0.49318677],
                             [- 6.27285506, -0.00054872, 0.00033622, 7.27230634, 0.99832361, 0.99911506, 0.49318685],
                             [- 6.27285529, -0.00054862, 0.00033616, 7.27230667, 0.99832391, 0.99911522, 0.49318683],
                             [- 6.27285552, -0.00054853, 0.00033610, 7.27230700, 0.99832421, 0.99911537, 0.49318692],
                             [- 6.27285576, -0.00054843, 0.00033604, 7.27230733, 0.99832450, 0.99911553, 0.49318690],
                             [- 6.27285599, -0.00054833, 0.00033598, 7.27230766, 0.99832480, 0.99911569, 0.49318694],
                             [- 6.27285622, -0.00054824, 0.00033592, 7.27230799, 0.99832510, 0.99911584, 0.49318700],
                             [- 6.27285645, -0.00054814, 0.00033586, 7.27230832, 0.99832539, 0.99911600, 0.49318701],
                             [- 6.27285669, -0.00054804, 0.00033580, 7.27230865, 0.99832569, 0.99911616, 0.49318705],
                             [- 6.27285692, -0.00054794, 0.00033574, 7.27230897, 0.99832599, 0.99911631, 0.49318711],
                             [- 6.27285715, -0.00054785, 0.00033568, 7.27230930, 0.99832628, 0.99911647, 0.49318708],
                             [- 6.27285738, -0.00054775, 0.00033563, 7.27230963, 0.99832658, 0.99911662, 0.49318719],
                             [- 6.27285761, -0.00054765, 0.00033557, 7.27230996, 0.99832688, 0.99911678, 0.49318717],
                             [- 6.27285785, -0.00054756, 0.00033551, 7.27231029, 0.99832717, 0.99911694, 0.49318724],
                             [- 6.27285808, -0.00054746, 0.00033545, 7.27231062, 0.99832747, 0.99911709, 0.49318723],
                             [- 6.27285831, -0.00054736, 0.00033539, 7.27231095, 0.99832776, 0.99911725, 0.49318729],
                             [- 6.27285854, -0.00054727, 0.00033533, 7.27231127, 0.99832806, 0.99911741, 0.49318737],
                             [- 6.27285877, -0.00054717, 0.00033527, 7.27231160, 0.99832836, 0.99911756, 0.49318736],
                             [- 6.27285900, -0.00054707, 0.00033521, 7.27231193, 0.99832865, 0.99911772, 0.49318737],
                             [- 6.27285923, -0.00054698, 0.00033515, 7.27231226, 0.99832895, 0.99911787, 0.49318744],
                             [- 6.27285947, -0.00054688, 0.00033509, 7.27231259, 0.99832924, 0.99911803, 0.49318747],
                             [- 6.27285970, -0.00054678, 0.00033503, 7.27231291, 0.99832954, 0.99911819, 0.49318750],
                             [- 6.27285993, -0.00054669, 0.00033497, 7.27231324, 0.99832983, 0.99911834, 0.49318752],
                             [- 6.27286016, -0.00054659, 0.00033491, 7.27231357, 0.99833013, 0.99911850, 0.49318756],
                             [- 6.27286039, -0.00054649, 0.00033485, 7.27231390, 0.99833042, 0.99911865, 0.49318760],
                             [- 6.27286062, -0.00054640, 0.00033480, 7.27231422, 0.99833072, 0.99911881, 0.49318765],
                             [- 6.27286085, -0.00054630, 0.00033474, 7.27231455, 0.99833101, 0.99911896, 0.49318766],
                             [- 6.27286108, -0.00054620, 0.00033468, 7.27231488, 0.99833131, 0.99911912, 0.49318767],
                             [- 6.27286131, -0.00054611, 0.00033462, 7.27231521, 0.99833160, 0.99911927, 0.49318769],
                             [- 6.27286154, -0.00054601, 0.00033456, 7.27231553, 0.99833190, 0.99911943, 0.49318777],
                             [- 6.27286177, -0.00054591, 0.00033450, 7.27231586, 0.99833219, 0.99911959, 0.49318778],
                             [- 6.27286200, -0.00054582, 0.00033444, 7.27231619, 0.99833248, 0.99911974, 0.49318783],
                             [- 6.27286223, -0.00054572, 0.00033438, 7.27231651, 0.99833278, 0.99911990, 0.49318785],
                             [- 6.27286246, -0.00054563, 0.00033432, 7.27231684, 0.99833307, 0.99912005, 0.49318790],
                             [- 6.27286269, -0.00054553, 0.00033426, 7.27231716, 0.99833337, 0.99912021, 0.49318796],
                             [- 6.27286292, -0.00054543, 0.00033420, 7.27231749, 0.99833366, 0.99912036, 0.49318798],
                             [- 6.27286315, -0.00054534, 0.00033415, 7.27231782, 0.99833395, 0.99912052, 0.49318802],
                             [- 6.27286338, -0.00054524, 0.00033409, 7.27231814, 0.99833425, 0.99912067, 0.49318801],
                             [- 6.27286361, -0.00054515, 0.00033403, 7.27231847, 0.99833454, 0.99912083, 0.49318810],
                             [- 6.27286384, -0.00054505, 0.00033397, 7.27231879, 0.99833483, 0.99912098, 0.49318815],
                             [- 6.27286407, -0.00054495, 0.00033391, 7.27231912, 0.99833513, 0.99912114, 0.49318815],
                             [- 6.27286430, -0.00054486, 0.00033385, 7.27231944, 0.99833542, 0.99912129, 0.49318821],
                             [- 6.27286453, -0.00054476, 0.00033379, 7.27231977, 0.99833571, 0.99912145, 0.49318820],
                             [- 6.27286476, -0.00054467, 0.00033373, 7.27232010, 0.99833601, 0.99912160, 0.49318828],
                             [- 6.27286499, -0.00054457, 0.00033368, 7.27232042, 0.99833630, 0.99912175, 0.49318827],
                             [- 6.27286522, -0.00054447, 0.00033362, 7.27232075, 0.99833659, 0.99912191, 0.49318830],
                             [- 6.27286545, -0.00054438, 0.00033356, 7.27232107, 0.99833689, 0.99912206, 0.49318834],
                             [- 6.27286568, -0.00054428, 0.00033350, 7.27232139, 0.99833718, 0.99912222, 0.49318840],
                             [- 6.27286591, -0.00054419, 0.00033344, 7.27232172, 0.99833747, 0.99912237, 0.49318849],
                             [- 6.27286614, -0.00054409, 0.00033338, 7.27232204, 0.99833776, 0.99912253, 0.49318845],
                             [- 6.27286636, -0.00054400, 0.00033332, 7.27232237, 0.99833805, 0.99912268, 0.49318852],
                             [- 6.27286659, -0.00054390, 0.00033326, 7.27232269, 0.99833835, 0.99912283, 0.49318851],
                             [- 6.27286682, -0.00054380, 0.00033321, 7.27232302, 0.99833864, 0.99912299, 0.49318855],
                             [- 6.27286705, -0.00054371, 0.00033315, 7.27232334, 0.99833893, 0.99912314, 0.49318857],
                             [- 6.27286728, -0.00054361, 0.00033309, 7.27232366, 0.99833922, 0.99912330, 0.49318863],
                             [- 6.27286751, -0.00054352, 0.00033303, 7.27232399, 0.99833951, 0.99912345, 0.49318865],
                             [- 6.27286774, -0.00054342, 0.00033297, 7.27232431, 0.99833981, 0.99912360, 0.49318871],
                             [- 6.27286796, -0.00054333, 0.00033291, 7.27232464, 0.99834010, 0.99912376, 0.49318871],
                             [- 6.27286819, -0.00054323, 0.00033286, 7.27232496, 0.99834039, 0.99912391, 0.49318882],
                             [- 6.27286842, -0.00054314, 0.00033280, 7.27232528, 0.99834068, 0.99912407, 0.49318885],
                             [- 6.27286865, -0.00054304, 0.00033274, 7.27232561, 0.99834097, 0.99912422, 0.49318882],
                             [- 6.27286888, -0.00054295, 0.00033268, 7.27232593, 0.99834126, 0.99912437, 0.49318888],
                             [- 6.27286910, -0.00054285, 0.00033262, 7.27232625, 0.99834155, 0.99912453, 0.49318890],
                             [- 6.27286933, -0.00054276, 0.00033256, 7.27232657, 0.99834184, 0.99912468, 0.49318892],
                             [- 6.27286956, -0.00054266, 0.00033250, 7.27232690, 0.99834214, 0.99912483, 0.49318896],
                             [- 6.27286979, -0.00054257, 0.00033245, 7.27232722, 0.99834243, 0.99912499, 0.49318900],
                             [- 6.27287001, -0.00054247, 0.00033239, 7.27232754, 0.99834272, 0.99912514, 0.49318899],
                             [- 6.27287024, -0.00054238, 0.00033233, 7.27232787, 0.99834301, 0.99912529, 0.49318908],
                             [- 6.27287047, -0.00054228, 0.00033227, 7.27232819, 0.99834330, 0.99912545, 0.49318910],
                             [- 6.27287070, -0.00054219, 0.00033221, 7.27232851, 0.99834359, 0.99912560, 0.49318913],
                             [- 6.27287092, -0.00054209, 0.00033216, 7.27232883, 0.99834388, 0.99912575, 0.49318913],
                             [- 6.27287115, -0.00054200, 0.00033210, 7.27232915, 0.99834417, 0.99912591, 0.49318919],
                             [- 6.27287138, -0.00054190, 0.00033204, 7.27232948, 0.99834446, 0.99912606, 0.49318918],
                             [- 6.27287160, -0.00054181, 0.00033198, 7.27232980, 0.99834475, 0.99912621, 0.49318924],
                             [- 6.27287183, -0.00054171, 0.00033192, 7.27233012, 0.99834504, 0.99912637, 0.49318928],
                             [- 6.27287206, -0.00054162, 0.00033186, 7.27233044, 0.99834533, 0.99912652, 0.49318937],
                             [- 6.27287228, -0.00054152, 0.00033181, 7.27233076, 0.99834562, 0.99912667, 0.49318939],
                             [- 6.27287251, -0.00054143, 0.00033175, 7.27233108, 0.99834591, 0.99912682, 0.49318942],
                             [- 6.27287274, -0.00054133, 0.00033169, 7.27233140, 0.99834620, 0.99912698, 0.49318944],
                             [- 6.27287296, -0.00054124, 0.00033163, 7.27233173, 0.99834649, 0.99912713, 0.49318945],
                             [- 6.27287319, -0.00054114, 0.00033157, 7.27233205, 0.99834677, 0.99912728, 0.49318948],
                             [- 6.27287342, -0.00054105, 0.00033152, 7.27233237, 0.99834706, 0.99912744, 0.49318948],
                             [- 6.27287364, -0.00054095, 0.00033146, 7.27233269, 0.99834735, 0.99912759, 0.49318960],
                             [- 6.27287387, -0.00054086, 0.00033140, 7.27233301, 0.99834764, 0.99912774, 0.49318965],
                             [- 6.27287409, -0.00054076, 0.00033134, 7.27233333, 0.99834793, 0.99912789, 0.49318963],
                             [- 6.27287432, -0.00054067, 0.00033128, 7.27233365, 0.99834822, 0.99912805, 0.49318967],
                             [- 6.27287455, -0.00054058, 0.00033123, 7.27233397, 0.99834851, 0.99912820, 0.49318974],
                             [- 6.27287477, -0.00054048, 0.00033117, 7.27233429, 0.99834880, 0.99912835, 0.49318975],
                             [- 6.27287500, -0.00054039, 0.00033111, 7.27233461, 0.99834908, 0.99912850, 0.49318982],
                             [- 6.27287522, -0.00054029, 0.00033105, 7.27233493, 0.99834937, 0.99912865, 0.49318976],
                             [- 6.27287545, -0.00054020, 0.00033100, 7.27233525, 0.99834966, 0.99912881, 0.49318982],
                             [- 6.27287568, -0.00054010, 0.00033094, 7.27233557, 0.99834995, 0.99912896, 0.49318990],
                             [- 6.27287590, -0.00054001, 0.00033088, 7.27233589, 0.99835024, 0.99912911, 0.49318991],
                             [- 6.27287613, -0.00053992, 0.00033082, 7.27233621, 0.99835052, 0.99912926, 0.49318988],
                             [- 6.27287635, -0.00053982, 0.00033076, 7.27233653, 0.99835081, 0.99912941, 0.49319001],
                             [- 6.27287658, -0.00053973, 0.00033071, 7.27233685, 0.99835110, 0.99912957, 0.49318998],
                             [- 6.27287680, -0.00053963, 0.00033065, 7.27233717, 0.99835139, 0.99912972, 0.49319007],
                             [- 6.27287703, -0.00053954, 0.00033059, 7.27233749, 0.99835168, 0.99912987, 0.49319009],
                             [- 6.27287725, -0.00053945, 0.00033053, 7.27233781, 0.99835196, 0.99913002, 0.49319013],
                             [- 6.27287748, -0.00053935, 0.00033048, 7.27233812, 0.99835225, 0.99913017, 0.49319012],
                             [- 6.27287770, -0.00053926, 0.00033042, 7.27233844, 0.99835254, 0.99913032, 0.49319022],
                             [- 6.27287793, -0.00053916, 0.00033036, 7.27233876, 0.99835282, 0.99913048, 0.49319019],
                             [- 6.27287815, -0.00053907, 0.00033030, 7.27233908, 0.99835311, 0.99913063, 0.49319021],
                             [- 6.27287838, -0.00053898, 0.00033025, 7.27233940, 0.99835340, 0.99913078, 0.49319029],
                             [- 6.27287860, -0.00053888, 0.00033019, 7.27233972, 0.99835368, 0.99913093, 0.49319029],
                             [- 6.27287882, -0.00053879, 0.00033013, 7.27234004, 0.99835397, 0.99913108, 0.49319033],
                             [- 6.27287905, -0.00053869, 0.00033007, 7.27234035, 0.99835426, 0.99913123, 0.49319038],
                             [- 6.27287927, -0.00053860, 0.00033002, 7.27234067, 0.99835454, 0.99913138, 0.49319044],
                             [- 6.27287950, -0.00053851, 0.00032996, 7.27234099, 0.99835483, 0.99913153, 0.49319044],
                             [- 6.27287972, -0.00053841, 0.00032990, 7.27234131, 0.99835512, 0.99913169, 0.49319052],
                             [- 6.27287994, -0.00053832, 0.00032984, 7.27234162, 0.99835540, 0.99913184, 0.49319053],
                             [- 6.27288017, -0.00053823, 0.00032979, 7.27234194, 0.99835569, 0.99913199, 0.49319055],
                             [- 6.27288039, -0.00053813, 0.00032973, 7.27234226, 0.99835598, 0.99913214, 0.49319059],
                             [- 6.27288062, -0.00053804, 0.00032967, 7.27234258, 0.99835626, 0.99913229, 0.49319064],
                             [- 6.27288084, -0.00053795, 0.00032961, 7.27234289, 0.99835655, 0.99913244, 0.49319065],
                             [- 6.27288106, -0.00053785, 0.00032956, 7.27234321, 0.99835683, 0.99913259, 0.49319063],
                             [- 6.27288129, -0.00053776, 0.00032950, 7.27234353, 0.99835712, 0.99913274, 0.49319073],
                             [- 6.27288151, -0.00053767, 0.00032944, 7.27234385, 0.99835740, 0.99913289, 0.49319071],
                             [- 6.27288173, -0.00053757, 0.00032938, 7.27234416, 0.99835769, 0.99913304, 0.49319080],
                             [- 6.27288196, -0.00053748, 0.00032933, 7.27234448, 0.99835797, 0.99913319, 0.49319079],
                             [- 6.27288218, -0.00053739, 0.00032927, 7.27234480, 0.99835826, 0.99913334, 0.49319084],
                             [- 6.27288240, -0.00053729, 0.00032921, 7.27234511, 0.99835854, 0.99913349, 0.49319094],
                             [- 6.27288263, -0.00053720, 0.00032916, 7.27234543, 0.99835883, 0.99913365, 0.49319090],
                             [- 6.27288285, -0.00053711, 0.00032910, 7.27234574, 0.99835911, 0.99913380, 0.49319093],
                             [- 6.27288307, -0.00053701, 0.00032904, 7.27234606, 0.99835940, 0.99913395, 0.49319095],
                             [- 6.27288330, -0.00053692, 0.00032898, 7.27234638, 0.99835968, 0.99913410, 0.49319102],
                             [- 6.27288352, -0.00053683, 0.00032893, 7.27234669, 0.99835997, 0.99913425, 0.49319110],
                             [- 6.27288374, -0.00053673, 0.00032887, 7.27234701, 0.99836025, 0.99913440, 0.49319107],
                             [- 6.27288396, -0.00053664, 0.00032881, 7.27234732, 0.99836054, 0.99913455, 0.49319115],
                             [- 6.27288419, -0.00053655, 0.00032876, 7.27234764, 0.99836082, 0.99913470, 0.49319113],
                             [- 6.27288441, -0.00053645, 0.00032870, 7.27234795, 0.99836111, 0.99913485, 0.49319119],
                             [- 6.27288463, -0.00053636, 0.00032864, 7.27234827, 0.99836139, 0.99913500, 0.49319119],
                             [- 6.27288485, -0.00053627, 0.00032859, 7.27234859, 0.99836167, 0.99913515, 0.49319124],
                             [- 6.27288508, -0.00053618, 0.00032853, 7.27234890, 0.99836196, 0.99913530, 0.49319131],
                             [- 6.27288530, -0.00053608, 0.00032847, 7.27234922, 0.99836224, 0.99913545, 0.49319132],
                             [- 6.27288552, -0.00053599, 0.00032841, 7.27234953, 0.99836253, 0.99913560, 0.49319134],
                             [- 6.27288574, -0.00053590, 0.00032836, 7.27234985, 0.99836281, 0.99913575, 0.49319141],
                             [- 6.27288596, -0.00053580, 0.00032830, 7.27235016, 0.99836309, 0.99913590, 0.49319142],
                             [- 6.27288619, -0.00053571, 0.00032824, 7.27235047, 0.99836338, 0.99913604, 0.49319138],
                             [- 6.27288641, -0.00053562, 0.00032819, 7.27235079, 0.99836366, 0.99913619, 0.49319145],
                             [- 6.27288663, -0.00053553, 0.00032813, 7.27235110, 0.99836394, 0.99913634, 0.49319149],
                             [- 6.27288685, -0.00053543, 0.00032807, 7.27235142, 0.99836423, 0.99913649, 0.49319155],
                             [- 6.27288707, -0.00053534, 0.00032802, 7.27235173, 0.99836451, 0.99913664, 0.49319156],
                             [- 6.27288729, -0.00053525, 0.00032796, 7.27235205, 0.99836479, 0.99913679, 0.49319156],
                             [- 6.27288752, -0.00053516, 0.00032790, 7.27235236, 0.99836507, 0.99913694, 0.49319162],
                             [- 6.27288774, -0.00053506, 0.00032785, 7.27235267, 0.99836536, 0.99913709, 0.49319169],
                             [- 6.27288796, -0.00053497, 0.00032779, 7.27235299, 0.99836564, 0.99913724, 0.49319169],
                             [- 6.27288818, -0.00053488, 0.00032773, 7.27235330, 0.99836592, 0.99913739, 0.49319176],
                             [- 6.27288840, -0.00053479, 0.00032768, 7.27235361, 0.99836620, 0.99913754, 0.49319177],
                             [- 6.27288862, -0.00053469, 0.00032762, 7.27235393, 0.99836649, 0.99913769, 0.49319182],
                             [- 6.27288884, -0.00053460, 0.00032756, 7.27235424, 0.99836677, 0.99913784, 0.49319183],
                             [- 6.27288906, -0.00053451, 0.00032751, 7.27235455, 0.99836705, 0.99913798, 0.49319186],
                             [- 6.27288928, -0.00053442, 0.00032745, 7.27235487, 0.99836733, 0.99913813, 0.49319195],
                             [- 6.27288950, -0.00053432, 0.00032739, 7.27235518, 0.99836762, 0.99913828, 0.49319191],
                             [- 6.27288973, -0.00053423, 0.00032734, 7.27235549, 0.99836790, 0.99913843, 0.49319195],
                             [- 6.27288995, -0.00053414, 0.00032728, 7.27235581, 0.99836818, 0.99913858, 0.49319198],
                             [- 6.27289017, -0.00053405, 0.00032722, 7.27235612, 0.99836846, 0.99913873, 0.49319204],
                             [- 6.27289039, -0.00053396, 0.00032717, 7.27235643, 0.99836874, 0.99913888, 0.49319208],
                             [- 6.27289061, -0.00053386, 0.00032711, 7.27235674, 0.99836902, 0.99913903, 0.49319213],
                             [- 6.27289083, -0.00053377, 0.00032705, 7.27235706, 0.99836931, 0.99913917, 0.49319212],
                             [- 6.27289105, -0.00053368, 0.00032700, 7.27235737, 0.99836959, 0.99913932, 0.49319218],
                             [- 6.27289127, -0.00053359, 0.00032694, 7.27235768, 0.99836987, 0.99913947, 0.49319222],
                             [- 6.27289149, -0.00053350, 0.00032689, 7.27235799, 0.99837015, 0.99913962, 0.49319223],
                             [- 6.27289171, -0.00053340, 0.00032683, 7.27235830, 0.99837043, 0.99913977, 0.49319229],
                             [- 6.27289193, -0.00053331, 0.00032677, 7.27235862, 0.99837071, 0.99913992, 0.49319228],
                             [- 6.27289215, -0.00053322, 0.00032672, 7.27235893, 0.99837099, 0.99914006, 0.49319230],
                             [- 6.27289237, -0.00053313, 0.00032666, 7.27235924, 0.99837127, 0.99914021, 0.49319236],
                             [- 6.27289259, -0.00053304, 0.00032660, 7.27235955, 0.99837155, 0.99914036, 0.49319240],
                             [- 6.27289281, -0.00053294, 0.00032655, 7.27235986, 0.99837183, 0.99914051, 0.49319244],
                             [- 6.27289303, -0.00053285, 0.00032649, 7.27236017, 0.99837211, 0.99914066, 0.49319243],
                             [- 6.27289325, -0.00053276, 0.00032643, 7.27236048, 0.99837239, 0.99914080, 0.49319249],
                             [- 6.27289346, -0.00053267, 0.00032638, 7.27236080, 0.99837267, 0.99914095, 0.49319257],
                             [- 6.27289368, -0.00053258, 0.00032632, 7.27236111, 0.99837295, 0.99914110, 0.49319253],
                             [- 6.27289390, -0.00053249, 0.00032627, 7.27236142, 0.99837323, 0.99914125, 0.49319261],
                             [- 6.27289412, -0.00053239, 0.00032621, 7.27236173, 0.99837351, 0.99914140, 0.49319263],
                             [- 6.27289434, -0.00053230, 0.00032615, 7.27236204, 0.99837379, 0.99914154, 0.49319267],
                             [- 6.27289456, -0.00053221, 0.00032610, 7.27236235, 0.99837407, 0.99914169, 0.49319271],
                             [- 6.27289478, -0.00053212, 0.00032604, 7.27236266, 0.99837435, 0.99914184, 0.49319271],
                             [- 6.27289500, -0.00053203, 0.00032599, 7.27236297, 0.99837463, 0.99914199, 0.49319270],
                             [- 6.27289522, -0.00053194, 0.00032593, 7.27236328, 0.99837491, 0.99914213, 0.49319278],
                             [- 6.27289544, -0.00053185, 0.00032587, 7.27236359, 0.99837519, 0.99914228, 0.49319281],
                             [- 6.27289565, -0.00053175, 0.00032582, 7.27236390, 0.99837547, 0.99914243, 0.49319284],
                             [- 6.27289587, -0.00053166, 0.00032576, 7.27236421, 0.99837575, 0.99914258, 0.49319286],
                             [- 6.27289609, -0.00053157, 0.00032571, 7.27236452, 0.99837603, 0.99914272, 0.49319287],
                             [- 6.27289631, -0.00053148, 0.00032565, 7.27236483, 0.99837631, 0.99914287, 0.49319297],
                             [- 6.27289653, -0.00053139, 0.00032559, 7.27236514, 0.99837659, 0.99914302, 0.49319300],
                             [- 6.27289675, -0.00053130, 0.00032554, 7.27236545, 0.99837687, 0.99914316, 0.49319302],
                             [- 6.27289696, -0.00053121, 0.00032548, 7.27236576, 0.99837714, 0.99914331, 0.49319305],
                             [- 6.27289718, -0.00053112, 0.00032543, 7.27236607, 0.99837742, 0.99914346, 0.49319306],
                             [- 6.27289740, -0.00053102, 0.00032537, 7.27236638, 0.99837770, 0.99914361, 0.49319318],
                             [- 6.27289762, -0.00053093, 0.00032531, 7.27236668, 0.99837798, 0.99914375, 0.49319313],
                             [- 6.27289784, -0.00053084, 0.00032526, 7.27236699, 0.99837826, 0.99914390, 0.49319318],
                             [- 6.27289805, -0.00053075, 0.00032520, 7.27236730, 0.99837854, 0.99914405, 0.49319320],
                             [- 6.27289827, -0.00053066, 0.00032515, 7.27236761, 0.99837881, 0.99914419, 0.49319315],
                             [- 6.27289849, -0.00053057, 0.00032509, 7.27236792, 0.99837909, 0.99914434, 0.49319329],
                             [- 6.27289871, -0.00053048, 0.00032504, 7.27236823, 0.99837937, 0.99914449, 0.49319329],
                             [- 6.27289892, -0.00053039, 0.00032498, 7.27236854, 0.99837965, 0.99914463, 0.49319336],
                             [- 6.27289914, -0.00053030, 0.00032492, 7.27236884, 0.99837993, 0.99914478, 0.49319338],
                             [- 6.27289936, -0.00053021, 0.00032487, 7.27236915, 0.99838020, 0.99914493, 0.49319340],
                             [- 6.27289958, -0.00053012, 0.00032481, 7.27236946, 0.99838048, 0.99914507, 0.49319339],
                             [- 6.27289979, -0.00053002, 0.00032476, 7.27236977, 0.99838076, 0.99914522, 0.49319350],
                             [- 6.27290001, -0.00052993, 0.00032470, 7.27237008, 0.99838104, 0.99914536, 0.49319352],
                             [- 6.27290023, -0.00052984, 0.00032465, 7.27237038, 0.99838131, 0.99914551, 0.49319359],
                             [- 6.27290044, -0.00052975, 0.00032459, 7.27237069, 0.99838159, 0.99914566, 0.49319354],
                             [- 6.27290066, -0.00052966, 0.00032453, 7.27237100, 0.99838187, 0.99914580, 0.49319355],
                             [- 6.27290088, -0.00052957, 0.00032448, 7.27237131, 0.99838214, 0.99914595, 0.49319365],
                             [- 6.27290109, -0.00052948, 0.00032442, 7.27237161, 0.99838242, 0.99914610, 0.49319372],
                             [- 6.27290131, -0.00052939, 0.00032437, 7.27237192, 0.99838270, 0.99914624, 0.49319371],
                             [- 6.27290153, -0.00052930, 0.00032431, 7.27237223, 0.99838297, 0.99914639, 0.49319369],
                             [- 6.27290174, -0.00052921, 0.00032426, 7.27237254, 0.99838325, 0.99914653, 0.49319379],
                             [- 6.27290196, -0.00052912, 0.00032420, 7.27237284, 0.99838353, 0.99914668, 0.49319379],
                             [- 6.27290218, -0.00052903, 0.00032415, 7.27237315, 0.99838380, 0.99914683, 0.49319387],
                             [- 6.27290239, -0.00052894, 0.00032409, 7.27237346, 0.99838408, 0.99914697, 0.49319384],
                             [- 6.27290261, -0.00052885, 0.00032404, 7.27237376, 0.99838436, 0.99914712, 0.49319390],
                             [- 6.27290283, -0.00052876, 0.00032398, 7.27237407, 0.99838463, 0.99914726, 0.49319391],
                             [- 6.27290304, -0.00052867, 0.00032392, 7.27237438, 0.99838491, 0.99914741, 0.49319397],
                             [- 6.27290326, -0.00052858, 0.00032387, 7.27237468, 0.99838518, 0.99914755, 0.49319399],
                             [- 6.27290347, -0.00052849, 0.00032381, 7.27237499, 0.99838546, 0.99914770, 0.49319399],
                             [- 6.27290369, -0.00052840, 0.00032376, 7.27237529, 0.99838574, 0.99914785, 0.49319402],
                             [- 6.27290391, -0.00052831, 0.00032370, 7.27237560, 0.99838601, 0.99914799, 0.49319410],
                             [- 6.27290412, -0.00052822, 0.00032365, 7.27237591, 0.99838629, 0.99914814, 0.49319411],
                             [- 6.27290434, -0.00052813, 0.00032359, 7.27237621, 0.99838656, 0.99914828, 0.49319417],
                             [- 6.27290455, -0.00052804, 0.00032354, 7.27237652, 0.99838684, 0.99914843, 0.49319421],
                             [- 6.27290477, -0.00052795, 0.00032348, 7.27237682, 0.99838711, 0.99914857, 0.49319423],
                             [- 6.27290498, -0.00052786, 0.00032343, 7.27237713, 0.99838739, 0.99914872, 0.49319424],
                             [- 6.27290520, -0.00052777, 0.00032337, 7.27237743, 0.99838766, 0.99914886, 0.49319429],
                             [- 6.27290541, -0.00052768, 0.00032332, 7.27237774, 0.99838794, 0.99914901, 0.49319426],
                             [- 6.27290563, -0.00052759, 0.00032326, 7.27237804, 0.99838821, 0.99914915, 0.49319434],
                             [- 6.27290584, -0.00052750, 0.00032321, 7.27237835, 0.99838849, 0.99914930, 0.49319436],
                             [- 6.27290606, -0.00052741, 0.00032315, 7.27237865, 0.99838876, 0.99914944, 0.49319441],
                             [- 6.27290627, -0.00052732, 0.00032310, 7.27237896, 0.99838904, 0.99914959, 0.49319443],
                             [- 6.27290649, -0.00052723, 0.00032304, 7.27237926, 0.99838931, 0.99914973, 0.49319450],
                             [- 6.27290670, -0.00052714, 0.00032299, 7.27237957, 0.99838959, 0.99914988, 0.49319447],
                             [- 6.27290692, -0.00052705, 0.00032293, 7.27237987, 0.99838986, 0.99915002, 0.49319454],
                             [- 6.27290713, -0.00052696, 0.00032288, 7.27238018, 0.99839013, 0.99915017, 0.49319453],
                             [- 6.27290735, -0.00052687, 0.00032282, 7.27238048, 0.99839041, 0.99915031, 0.49319457],
                             [- 6.27290756, -0.00052678, 0.00032277, 7.27238078, 0.99839068, 0.99915046, 0.49319463],
                             [- 6.27290778, -0.00052669, 0.00032271, 7.27238109, 0.99839096, 0.99915060, 0.49319468],
                             [- 6.27290799, -0.00052660, 0.00032266, 7.27238139, 0.99839123, 0.99915075, 0.49319471],
                             [- 6.27290821, -0.00052651, 0.00032260, 7.27238170, 0.99839150, 0.99915089, 0.49319472],
                             [- 6.27290842, -0.00052642, 0.00032255, 7.27238200, 0.99839178, 0.99915103, 0.49319477],
                             [- 6.27290863, -0.00052633, 0.00032249, 7.27238230, 0.99839205, 0.99915118, 0.49319479],
                             [- 6.27290885, -0.00052624, 0.00032244, 7.27238261, 0.99839232, 0.99915132, 0.49319479],
                             [- 6.27290906, -0.00052615, 0.00032238, 7.27238291, 0.99839260, 0.99915147, 0.49319486],
                             [- 6.27290928, -0.00052606, 0.00032233, 7.27238321, 0.99839287, 0.99915161, 0.49319496],
                             [- 6.27290949, -0.00052597, 0.00032227, 7.27238352, 0.99839314, 0.99915176, 0.49319494],
                             [- 6.27290970, -0.00052588, 0.00032222, 7.27238382, 0.99839342, 0.99915190, 0.49319490],
                             [- 6.27290992, -0.00052579, 0.00032216, 7.27238412, 0.99839369, 0.99915204, 0.49319495],
                             [- 6.27291013, -0.00052570, 0.00032211, 7.27238443, 0.99839396, 0.99915219, 0.49319503],
                             [- 6.27291034, -0.00052561, 0.00032205, 7.27238473, 0.99839424, 0.99915233, 0.49319499],
                             [- 6.27291056, -0.00052553, 0.00032200, 7.27238503, 0.99839451, 0.99915248, 0.49319506],
                             [- 6.27291077, -0.00052544, 0.00032194, 7.27238534, 0.99839478, 0.99915262, 0.49319516],
                             [- 6.27291098, -0.00052535, 0.00032189, 7.27238564, 0.99839505, 0.99915276, 0.49319515],
                             [- 6.27291120, -0.00052526, 0.00032183, 7.27238594, 0.99839533, 0.99915291, 0.49319519],
                             [- 6.27291141, -0.00052517, 0.00032178, 7.27238624, 0.99839560, 0.99915305, 0.49319519],
                             [- 6.27291162, -0.00052508, 0.00032173, 7.27238654, 0.99839587, 0.99915320, 0.49319522],
                             [- 6.27291184, -0.00052499, 0.00032167, 7.27238685, 0.99839614, 0.99915334, 0.49319528],
                             [- 6.27291205, -0.00052490, 0.00032162, 7.27238715, 0.99839642, 0.99915348, 0.49319533],
                             [- 6.27291226, -0.00052481, 0.00032156, 7.27238745, 0.99839669, 0.99915363, 0.49319536],
                             [- 6.27291248, -0.00052472, 0.00032151, 7.27238775, 0.99839696, 0.99915377, 0.49319537],
                             [- 6.27291269, -0.00052463, 0.00032145, 7.27238805, 0.99839723, 0.99915391, 0.49319544],
                             [- 6.27291290, -0.00052455, 0.00032140, 7.27238836, 0.99839750, 0.99915406, 0.49319545],
                             [- 6.27291311, -0.00052446, 0.00032134, 7.27238866, 0.99839777, 0.99915420, 0.49319547],
                             [- 6.27291333, -0.00052437, 0.00032129, 7.27238896, 0.99839805, 0.99915434, 0.49319547],
                             [- 6.27291354, -0.00052428, 0.00032123, 7.27238926, 0.99839832, 0.99915449, 0.49319552],
                             [- 6.27291375, -0.00052419, 0.00032118, 7.27238956, 0.99839859, 0.99915463, 0.49319555],
                             [- 6.27291396, -0.00052410, 0.00032113, 7.27238986, 0.99839886, 0.99915477, 0.49319555],
                             [- 6.27291418, -0.00052401, 0.00032107, 7.27239016, 0.99839913, 0.99915492, 0.49319565],
                             [- 6.27291439, -0.00052392, 0.00032102, 7.27239046, 0.99839940, 0.99915506, 0.49319563],
                             [- 6.27291460, -0.00052384, 0.00032096, 7.27239077, 0.99839967, 0.99915520, 0.49319565],
                             [- 6.27291481, -0.00052375, 0.00032091, 7.27239107, 0.99839994, 0.99915534, 0.49319571],
                             [- 6.27291503, -0.00052366, 0.00032085, 7.27239137, 0.99840021, 0.99915549, 0.49319576],
                             [- 6.27291524, -0.00052357, 0.00032080, 7.27239167, 0.99840049, 0.99915563, 0.49319577],
                             [- 6.27291545, -0.00052348, 0.00032075, 7.27239197, 0.99840076, 0.99915577, 0.49319579],
                             [- 6.27291566, -0.00052339, 0.00032069, 7.27239227, 0.99840103, 0.99915592, 0.49319582],
                             [- 6.27291587, -0.00052330, 0.00032064, 7.27239257, 0.99840130, 0.99915606, 0.49319587],
                             [- 6.27291608, -0.00052322, 0.00032058, 7.27239287, 0.99840157, 0.99915620, 0.49319592],
                             [- 6.27291630, -0.00052313, 0.00032053, 7.27239317, 0.99840184, 0.99915634, 0.49319593],
                             [- 6.27291651, -0.00052304, 0.00032047, 7.27239347, 0.99840211, 0.99915649, 0.49319594],
                             [- 6.27291672, -0.00052295, 0.00032042, 7.27239377, 0.99840238, 0.99915663, 0.49319596],
                             [- 6.27291693, -0.00052286, 0.00032037, 7.27239407, 0.99840265, 0.99915677, 0.49319602],
                             [- 6.27291714, -0.00052277, 0.00032031, 7.27239437, 0.99840292, 0.99915691, 0.49319607],
                             [- 6.27291735, -0.00052269, 0.00032026, 7.27239467, 0.99840319, 0.99915706, 0.49319609],
                             [- 6.27291756, -0.00052260, 0.00032020, 7.27239497, 0.99840346, 0.99915720, 0.49319609],
                             [- 6.27291777, -0.00052251, 0.00032015, 7.27239526, 0.99840373, 0.99915734, 0.49319619],
                             [- 6.27291799, -0.00052242, 0.00032010, 7.27239556, 0.99840400, 0.99915748, 0.49319624],
                             [- 6.27291820, -0.00052233, 0.00032004, 7.27239586, 0.99840427, 0.99915763, 0.49319616],
                             [- 6.27291841, -0.00052225, 0.00031999, 7.27239616, 0.99840453, 0.99915777, 0.49319625],
                             [- 6.27291862, -0.00052216, 0.00031993, 7.27239646, 0.99840480, 0.99915791, 0.49319632],
                             [- 6.27291883, -0.00052207, 0.00031988, 7.27239676, 0.99840507, 0.99915805, 0.49319625],
                             [- 6.27291904, -0.00052198, 0.00031983, 7.27239706, 0.99840534, 0.99915819, 0.49319634],
                             [- 6.27291925, -0.00052189, 0.00031977, 7.27239736, 0.99840561, 0.99915834, 0.49319638],
                             [- 6.27291946, -0.00052180, 0.00031972, 7.27239766, 0.99840588, 0.99915848, 0.49319640],
                             [- 6.27291967, -0.00052172, 0.00031966, 7.27239795, 0.99840615, 0.99915862, 0.49319643],
                             [- 6.27291988, -0.00052163, 0.00031961, 7.27239825, 0.99840642, 0.99915876, 0.49319646],
                             [- 6.27292009, -0.00052154, 0.00031956, 7.27239855, 0.99840669, 0.99915890, 0.49319643],
                             [- 6.27292030, -0.00052145, 0.00031950, 7.27239885, 0.99840695, 0.99915904, 0.49319651],
                             [- 6.27292051, -0.00052137, 0.00031945, 7.27239915, 0.99840722, 0.99915919, 0.49319652],
                             [- 6.27292072, -0.00052128, 0.00031939, 7.27239944, 0.99840749, 0.99915933, 0.49319657],
                             [- 6.27292093, -0.00052119, 0.00031934, 7.27239974, 0.99840776, 0.99915947, 0.49319663],
                             [- 6.27292114, -0.00052110, 0.00031929, 7.27240004, 0.99840803, 0.99915961, 0.49319665],
                             [- 6.27292135, -0.00052101, 0.00031923, 7.27240034, 0.99840830, 0.99915975, 0.49319671],
                             [- 6.27292156, -0.00052093, 0.00031918, 7.27240063, 0.99840856, 0.99915989, 0.49319676],
                             [- 6.27292177, -0.00052084, 0.00031913, 7.27240093, 0.99840883, 0.99916004, 0.49319677],
                             [- 6.27292198, -0.00052075, 0.00031907, 7.27240123, 0.99840910, 0.99916018, 0.49319672],
                             [- 6.27292219, -0.00052066, 0.00031902, 7.27240153, 0.99840937, 0.99916032, 0.49319676],
                             [- 6.27292240, -0.00052058, 0.00031896, 7.27240182, 0.99840964, 0.99916046, 0.49319684],
                             [- 6.27292261, -0.00052049, 0.00031891, 7.27240212, 0.99840990, 0.99916060, 0.49319683],
                             [- 6.27292282, -0.00052040, 0.00031886, 7.27240242, 0.99841017, 0.99916074, 0.49319688],
                             [- 6.27292303, -0.00052031, 0.00031880, 7.27240271, 0.99841044, 0.99916088, 0.49319696],
                             [- 6.27292324, -0.00052023, 0.00031875, 7.27240301, 0.99841070, 0.99916102, 0.49319696],
                             [- 6.27292345, -0.00052014, 0.00031870, 7.27240331, 0.99841097, 0.99916117, 0.49319698],
                             [- 6.27292366, -0.00052005, 0.00031864, 7.27240360, 0.99841124, 0.99916131, 0.49319702],
                             [- 6.27292386, -0.00051996, 0.00031859, 7.27240390, 0.99841151, 0.99916145, 0.49319705],
                             [- 6.27292407, -0.00051988, 0.00031854, 7.27240420, 0.99841177, 0.99916159, 0.49319708],
                             [- 6.27292428, -0.00051979, 0.00031848, 7.27240449, 0.99841204, 0.99916173, 0.49319705],
                             [- 6.27292449, -0.00051970, 0.00031843, 7.27240479, 0.99841231, 0.99916187, 0.49319715],
                             [- 6.27292470, -0.00051962, 0.00031837, 7.27240508, 0.99841257, 0.99916201, 0.49319716],
                             [- 6.27292491, -0.00051953, 0.00031832, 7.27240538, 0.99841284, 0.99916215, 0.49319722],
                             [- 6.27292512, -0.00051944, 0.00031827, 7.27240568, 0.99841311, 0.99916229, 0.49319725],
                             [- 6.27292533, -0.00051935, 0.00031821, 7.27240597, 0.99841337, 0.99916243, 0.49319725],
                             [- 6.27292553, -0.00051927, 0.00031816, 7.27240627, 0.99841364, 0.99916257, 0.49319734],
                             [- 6.27292574, -0.00051918, 0.00031811, 7.27240656, 0.99841390, 0.99916271, 0.49319744],
                             [- 6.27292595, -0.00051909, 0.00031805, 7.27240686, 0.99841417, 0.99916285, 0.49319739],
                             [- 6.27292616, -0.00051901, 0.00031800, 7.27240715, 0.99841444, 0.99916299, 0.49319736],
                             [- 6.27292637, -0.00051892, 0.00031795, 7.27240745, 0.99841470, 0.99916313, 0.49319748],
                             [- 6.27292658, -0.00051883, 0.00031789, 7.27240774, 0.99841497, 0.99916327, 0.49319747],
                             [- 6.27292678, -0.00051874, 0.00031784, 7.27240804, 0.99841523, 0.99916341, 0.49319748],
                             [- 6.27292699, -0.00051866, 0.00031779, 7.27240833, 0.99841550, 0.99916355, 0.49319753],
                             [- 6.27292720, -0.00051857, 0.00031773, 7.27240863, 0.99841577, 0.99916369, 0.49319759],
                             [- 6.27292741, -0.00051848, 0.00031768, 7.27240892, 0.99841603, 0.99916384, 0.49319759],
                             [- 6.27292762, -0.00051840, 0.00031763, 7.27240922, 0.99841630, 0.99916398, 0.49319759],
                             [- 6.27292782, -0.00051831, 0.00031757, 7.27240951, 0.99841656, 0.99916412, 0.49319764],
                             [- 6.27292803, -0.00051822, 0.00031752, 7.27240981, 0.99841683, 0.99916426, 0.49319769],
                             [- 6.27292824, -0.00051814, 0.00031747, 7.27241010, 0.99841709, 0.99916440, 0.49319768],
                             [- 6.27292845, -0.00051805, 0.00031742, 7.27241040, 0.99841736, 0.99916453, 0.49319773],
                             [- 6.27292865, -0.00051796, 0.00031736, 7.27241069, 0.99841762, 0.99916467, 0.49319777],
                             [- 6.27292886, -0.00051788, 0.00031731, 7.27241098, 0.99841789, 0.99916481, 0.49319789],
                             [- 6.27292907, -0.00051779, 0.00031726, 7.27241128, 0.99841815, 0.99916495, 0.49319781],
                             [- 6.27292927, -0.00051770, 0.00031720, 7.27241157, 0.99841842, 0.99916509, 0.49319785],
                             [- 6.27292948, -0.00051762, 0.00031715, 7.27241186, 0.99841868, 0.99916523, 0.49319784],
                             [- 6.27292969, -0.00051753, 0.00031710, 7.27241216, 0.99841895, 0.99916537, 0.49319795],
                             [- 6.27292990, -0.00051744, 0.00031704, 7.27241245, 0.99841921, 0.99916551, 0.49319794],
                             [- 6.27293010, -0.00051736, 0.00031699, 7.27241275, 0.99841947, 0.99916565, 0.49319797],
                             [- 6.27293031, -0.00051727, 0.00031694, 7.27241304, 0.99841974, 0.99916579, 0.49319807],
                             [- 6.27293052, -0.00051718, 0.00031688, 7.27241333, 0.99842000, 0.99916593, 0.49319800],
                             [- 6.27293072, -0.00051710, 0.00031683, 7.27241363, 0.99842027, 0.99916607, 0.49319809],
                             [- 6.27293093, -0.00051701, 0.00031678, 7.27241392, 0.99842053, 0.99916621, 0.49319811],
                             [- 6.27293114, -0.00051693, 0.00031673, 7.27241421, 0.99842079, 0.99916635, 0.49319811],
                             [- 6.27293134, -0.00051684, 0.00031667, 7.27241450, 0.99842106, 0.99916649, 0.49319817],
                             [- 6.27293155, -0.00051675, 0.00031662, 7.27241480, 0.99842132, 0.99916663, 0.49319820],
                             [- 6.27293176, -0.00051667, 0.00031657, 7.27241509, 0.99842159, 0.99916677, 0.49319821],
                             [- 6.27293196, -0.00051658, 0.00031651, 7.27241538, 0.99842185, 0.99916691, 0.49319833],
                             [- 6.27293217, -0.00051649, 0.00031646, 7.27241567, 0.99842211, 0.99916704, 0.49319829],
                             [- 6.27293237, -0.00051641, 0.00031641, 7.27241597, 0.99842238, 0.99916718, 0.49319831],
                             [- 6.27293258, -0.00051632, 0.00031636, 7.27241626, 0.99842264, 0.99916732, 0.49319838],
                             [- 6.27293279, -0.00051624, 0.00031630, 7.27241655, 0.99842290, 0.99916746, 0.49319843],
                             [- 6.27293299, -0.00051615, 0.00031625, 7.27241684, 0.99842316, 0.99916760, 0.49319840],
                             [- 6.27293320, -0.00051606, 0.00031620, 7.27241713, 0.99842343, 0.99916774, 0.49319841],
                             [- 6.27293340, -0.00051598, 0.00031614, 7.27241743, 0.99842369, 0.99916788, 0.49319849],
                             [- 6.27293361, -0.00051589, 0.00031609, 7.27241772, 0.99842395, 0.99916802, 0.49319850],
                             [- 6.27293382, -0.00051581, 0.00031604, 7.27241801, 0.99842422, 0.99916816, 0.49319856],
                             [- 6.27293402, -0.00051572, 0.00031599, 7.27241830, 0.99842448, 0.99916829, 0.49319860],
                             [- 6.27293423, -0.00051563, 0.00031593, 7.27241859, 0.99842474, 0.99916843, 0.49319855],
                             [- 6.27293443, -0.00051555, 0.00031588, 7.27241888, 0.99842500, 0.99916857, 0.49319861],
                             [- 6.27293464, -0.00051546, 0.00031583, 7.27241918, 0.99842527, 0.99916871, 0.49319870],
                             [- 6.27293484, -0.00051538, 0.00031578, 7.27241947, 0.99842553, 0.99916885, 0.49319866],
                             [- 6.27293505, -0.00051529, 0.00031572, 7.27241976, 0.99842579, 0.99916899, 0.49319873],
                             [- 6.27293525, -0.00051520, 0.00031567, 7.27242005, 0.99842605, 0.99916912, 0.49319872],
                             [- 6.27293546, -0.00051512, 0.00031562, 7.27242034, 0.99842632, 0.99916926, 0.49319875],
                             [- 6.27293566, -0.00051503, 0.00031557, 7.27242063, 0.99842658, 0.99916940, 0.49319885],
                             [- 6.27293587, -0.00051495, 0.00031551, 7.27242092, 0.99842684, 0.99916954, 0.49319881],
                             [- 6.27293607, -0.00051486, 0.00031546, 7.27242121, 0.99842710, 0.99916968, 0.49319885],
                             [- 6.27293628, -0.00051478, 0.00031541, 7.27242150, 0.99842736, 0.99916982, 0.49319897],
                             [- 6.27293648, -0.00051469, 0.00031536, 7.27242179, 0.99842762, 0.99916995, 0.49319898],
                             [- 6.27293669, -0.00051461, 0.00031530, 7.27242208, 0.99842789, 0.99917009, 0.49319895],
                             [- 6.27293689, -0.00051452, 0.00031525, 7.27242237, 0.99842815, 0.99917023, 0.49319902],
                             [- 6.27293710, -0.00051443, 0.00031520, 7.27242266, 0.99842841, 0.99917037, 0.49319899],
                             [- 6.27293730, -0.00051435, 0.00031515, 7.27242295, 0.99842867, 0.99917051, 0.49319907],
                             [- 6.27293751, -0.00051426, 0.00031509, 7.27242324, 0.99842893, 0.99917064, 0.49319910],
                             [- 6.27293771, -0.00051418, 0.00031504, 7.27242353, 0.99842919, 0.99917078, 0.49319911],
                             [- 6.27293792, -0.00051409, 0.00031499, 7.27242382, 0.99842945, 0.99917092, 0.49319911],
                             [- 6.27293812, -0.00051401, 0.00031494, 7.27242411, 0.99842971, 0.99917106, 0.49319918],
                             [- 6.27293832, -0.00051392, 0.00031488, 7.27242440, 0.99842997, 0.99917119, 0.49319923],
                             [- 6.27293853, -0.00051384, 0.00031483, 7.27242469, 0.99843024, 0.99917133, 0.49319919],
                             [- 6.27293873, -0.00051375, 0.00031478, 7.27242498, 0.99843050, 0.99917147, 0.49319924],
                             [- 6.27293894, -0.00051367, 0.00031473, 7.27242527, 0.99843076, 0.99917161, 0.49319925],
                             [- 6.27293914, -0.00051358, 0.00031467, 7.27242556, 0.99843102, 0.99917174, 0.49319934],
                             [- 6.27293934, -0.00051350, 0.00031462, 7.27242585, 0.99843128, 0.99917188, 0.49319932],
                             [- 6.27293955, -0.00051341, 0.00031457, 7.27242614, 0.99843154, 0.99917202, 0.49319938],
                             [- 6.27293975, -0.00051333, 0.00031452, 7.27242643, 0.99843180, 0.99917216, 0.49319942],
                             [- 6.27293995, -0.00051324, 0.00031447, 7.27242671, 0.99843206, 0.99917229, 0.49319947],
                             [- 6.27294016, -0.00051315, 0.00031441, 7.27242700, 0.99843232, 0.99917243, 0.49319948],
                             [- 6.27294036, -0.00051307, 0.00031436, 7.27242729, 0.99843258, 0.99917257, 0.49319954],
                             [- 6.27294057, -0.00051298, 0.00031431, 7.27242758, 0.99843284, 0.99917271, 0.49319956],
                             [- 6.27294077, -0.00051290, 0.00031426, 7.27242787, 0.99843310, 0.99917284, 0.49319957],
                             [- 6.27294097, -0.00051281, 0.00031421, 7.27242816, 0.99843336, 0.99917298, 0.49319962],
                             [- 6.27294118, -0.00051273, 0.00031415, 7.27242845, 0.99843362, 0.99917312, 0.49319969],
                             [- 6.27294138, -0.00051265, 0.00031410, 7.27242873, 0.99843388, 0.99917325, 0.49319972],
                             [- 6.27294158, -0.00051256, 0.00031405, 7.27242902, 0.99843414, 0.99917339, 0.49319967],
                             [- 6.27294178, -0.00051248, 0.00031400, 7.27242931, 0.99843439, 0.99917353, 0.49319971],
                             [- 6.27294199, -0.00051239, 0.00031395, 7.27242960, 0.99843465, 0.99917366, 0.49319975],
                             [- 6.27294219, -0.00051231, 0.00031389, 7.27242988, 0.99843491, 0.99917380, 0.49319976],
                             [- 6.27294239, -0.00051222, 0.00031384, 7.27243017, 0.99843517, 0.99917394, 0.49319985],
                             [- 6.27294260, -0.00051214, 0.00031379, 7.27243046, 0.99843543, 0.99917407, 0.49319980],
                             [- 6.27294280, -0.00051205, 0.00031374, 7.27243075, 0.99843569, 0.99917421, 0.49319987],
                             [- 6.27294300, -0.00051197, 0.00031369, 7.27243103, 0.99843595, 0.99917435, 0.49319991],
                             [- 6.27294320, -0.00051188, 0.00031363, 7.27243132, 0.99843621, 0.99917448, 0.49319996],
                             [- 6.27294341, -0.00051180, 0.00031358, 7.27243161, 0.99843647, 0.99917462, 0.49320001],
                             [- 6.27294361, -0.00051171, 0.00031353, 7.27243190, 0.99843673, 0.99917476, 0.49319998],
                             [- 6.27294381, -0.00051163, 0.00031348, 7.27243218, 0.99843698, 0.99917489, 0.49320000],
                             [- 6.27294401, -0.00051154, 0.00031343, 7.27243247, 0.99843724, 0.99917503, 0.49320003],
                             [- 6.27294422, -0.00051146, 0.00031337, 7.27243276, 0.99843750, 0.99917517, 0.49320007],
                             [- 6.27294442, -0.00051137, 0.00031332, 7.27243304, 0.99843776, 0.99917530, 0.49320006],
                             [- 6.27294462, -0.00051129, 0.00031327, 7.27243333, 0.99843802, 0.99917544, 0.49320012],
                             [- 6.27294482, -0.00051121, 0.00031322, 7.27243362, 0.99843827, 0.99917558, 0.49320018],
                             [- 6.27294502, -0.00051112, 0.00031317, 7.27243390, 0.99843853, 0.99917571, 0.49320018],
                             [- 6.27294523, -0.00051104, 0.00031312, 7.27243419, 0.99843879, 0.99917585, 0.49320017],
                             [- 6.27294543, -0.00051095, 0.00031306, 7.27243448, 0.99843905, 0.99917598, 0.49320022],
                             [- 6.27294563, -0.00051087, 0.00031301, 7.27243476, 0.99843931, 0.99917612, 0.49320033],
                             [- 6.27294583, -0.00051078, 0.00031296, 7.27243505, 0.99843956, 0.99917626, 0.49320035],
                             [- 6.27294603, -0.00051070, 0.00031291, 7.27243533, 0.99843982, 0.99917639, 0.49320035],
                             [- 6.27294623, -0.00051062, 0.00031286, 7.27243562, 0.99844008, 0.99917653, 0.49320037],
                             [- 6.27294644, -0.00051053, 0.00031281, 7.27243590, 0.99844034, 0.99917666, 0.49320040],
                             [- 6.27294664, -0.00051045, 0.00031275, 7.27243619, 0.99844059, 0.99917680, 0.49320046],
                             [- 6.27294684, -0.00051036, 0.00031270, 7.27243648, 0.99844085, 0.99917693, 0.49320050],
                             [- 6.27294704, -0.00051028, 0.00031265, 7.27243676, 0.99844111, 0.99917707, 0.49320049],
                             [- 6.27294724, -0.00051019, 0.00031260, 7.27243705, 0.99844136, 0.99917721, 0.49320054],
                             [- 6.27294744, -0.00051011, 0.00031255, 7.27243733, 0.99844162, 0.99917734, 0.49320055],
                             [- 6.27294764, -0.00051003, 0.00031250, 7.27243762, 0.99844188, 0.99917748, 0.49320064],
                             [- 6.27294784, -0.00050994, 0.00031244, 7.27243790, 0.99844214, 0.99917761, 0.49320065],
                             [- 6.27294805, -0.00050986, 0.00031239, 7.27243819, 0.99844239, 0.99917775, 0.49320067],
                             [- 6.27294825, -0.00050977, 0.00031234, 7.27243847, 0.99844265, 0.99917788, 0.49320073],
                             [- 6.27294845, -0.00050969, 0.00031229, 7.27243876, 0.99844291, 0.99917802, 0.49320072],
                             [- 6.27294865, -0.00050961, 0.00031224, 7.27243904, 0.99844316, 0.99917815, 0.49320076],
                             [- 6.27294885, -0.00050952, 0.00031219, 7.27243933, 0.99844342, 0.99917829, 0.49320077],
                             [- 6.27294905, -0.00050944, 0.00031214, 7.27243961, 0.99844367, 0.99917843, 0.49320076],
                             [- 6.27294925, -0.00050936, 0.00031208, 7.27243989, 0.99844393, 0.99917856, 0.49320084],
                             [- 6.27294945, -0.00050927, 0.00031203, 7.27244018, 0.99844419, 0.99917870, 0.49320087],
                             [- 6.27294965, -0.00050919, 0.00031198, 7.27244046, 0.99844444, 0.99917883, 0.49320088],
                             [- 6.27294985, -0.00050910, 0.00031193, 7.27244075, 0.99844470, 0.99917897, 0.49320097],
                             [- 6.27295005, -0.00050902, 0.00031188, 7.27244103, 0.99844495, 0.99917910, 0.49320091],
                             [- 6.27295025, -0.00050894, 0.00031183, 7.27244132, 0.99844521, 0.99917924, 0.49320097],
                             [- 6.27295045, -0.00050885, 0.00031178, 7.27244160, 0.99844547, 0.99917937, 0.49320101],
                             [- 6.27295065, -0.00050877, 0.00031173, 7.27244188, 0.99844572, 0.99917951, 0.49320102],
                             [- 6.27295085, -0.00050869, 0.00031167, 7.27244217, 0.99844598, 0.99917964, 0.49320101],
                             [- 6.27295105, -0.00050860, 0.00031162, 7.27244245, 0.99844623, 0.99917978, 0.49320112],
                             [- 6.27295125, -0.00050852, 0.00031157, 7.27244273, 0.99844649, 0.99917991, 0.49320117],
                             [- 6.27295145, -0.00050844, 0.00031152, 7.27244302, 0.99844674, 0.99918004, 0.49320117],
                             [- 6.27295165, -0.00050835, 0.00031147, 7.27244330, 0.99844700, 0.99918018, 0.49320119],
                             [- 6.27295185, -0.00050827, 0.00031142, 7.27244358, 0.99844725, 0.99918031, 0.49320122],
                             [- 6.27295205, -0.00050818, 0.00031137, 7.27244387, 0.99844751, 0.99918045, 0.49320128],
                             [- 6.27295225, -0.00050810, 0.00031132, 7.27244415, 0.99844776, 0.99918058, 0.49320130],
                             [- 6.27295245, -0.00050802, 0.00031126, 7.27244443, 0.99844802, 0.99918072, 0.49320132],
                             [- 6.27295265, -0.00050793, 0.00031121, 7.27244472, 0.99844827, 0.99918085, 0.49320130],
                             [- 6.27295285, -0.00050785, 0.00031116, 7.27244500, 0.99844853, 0.99918099, 0.49320137],
                             [- 6.27295305, -0.00050777, 0.00031111, 7.27244528, 0.99844878, 0.99918112, 0.49320141],
                             [- 6.27295325, -0.00050768, 0.00031106, 7.27244556, 0.99844904, 0.99918126, 0.49320145],
                             [- 6.27295345, -0.00050760, 0.00031101, 7.27244585, 0.99844929, 0.99918139, 0.49320144],
                             [- 6.27295365, -0.00050752, 0.00031096, 7.27244613, 0.99844955, 0.99918152, 0.49320149],
                             [- 6.27295385, -0.00050743, 0.00031091, 7.27244641, 0.99844980, 0.99918166, 0.49320154],
                             [- 6.27295404, -0.00050735, 0.00031086, 7.27244669, 0.99845005, 0.99918179, 0.49320156],
                             [- 6.27295424, -0.00050727, 0.00031081, 7.27244697, 0.99845031, 0.99918193, 0.49320161],
                             [- 6.27295444, -0.00050719, 0.00031075, 7.27244726, 0.99845056, 0.99918206, 0.49320163],
                             [- 6.27295464, -0.00050710, 0.00031070, 7.27244754, 0.99845082, 0.99918219, 0.49320168],
                             [- 6.27295484, -0.00050702, 0.00031065, 7.27244782, 0.99845107, 0.99918233, 0.49320163],
                             [- 6.27295504, -0.00050694, 0.00031060, 7.27244810, 0.99845132, 0.99918246, 0.49320171],
                             [- 6.27295524, -0.00050685, 0.00031055, 7.27244838, 0.99845158, 0.99918260, 0.49320174],
                             [- 6.27295544, -0.00050677, 0.00031050, 7.27244867, 0.99845183, 0.99918273, 0.49320177],
                             [- 6.27295563, -0.00050669, 0.00031045, 7.27244895, 0.99845208, 0.99918286, 0.49320176],
                             [- 6.27295583, -0.00050660, 0.00031040, 7.27244923, 0.99845234, 0.99918300, 0.49320182],
                             [- 6.27295603, -0.00050652, 0.00031035, 7.27244951, 0.99845259, 0.99918313, 0.49320186],
                             [- 6.27295623, -0.00050644, 0.00031030, 7.27244979, 0.99845284, 0.99918327, 0.49320185],
                             [- 6.27295643, -0.00050636, 0.00031025, 7.27245007, 0.99845310, 0.99918340, 0.49320186],
                             [- 6.27295663, -0.00050627, 0.00031019, 7.27245035, 0.99845335, 0.99918353, 0.49320197],
                             [- 6.27295682, -0.00050619, 0.00031014, 7.27245063, 0.99845360, 0.99918367, 0.49320197],
                             [- 6.27295702, -0.00050611, 0.00031009, 7.27245091, 0.99845386, 0.99918380, 0.49320194],
                             [- 6.27295722, -0.00050602, 0.00031004, 7.27245120, 0.99845411, 0.99918393, 0.49320202],
                             [- 6.27295742, -0.00050594, 0.00030999, 7.27245148, 0.99845436, 0.99918407, 0.49320205],
                             [- 6.27295762, -0.00050586, 0.00030994, 7.27245176, 0.99845462, 0.99918420, 0.49320207],
                             [- 6.27295781, -0.00050578, 0.00030989, 7.27245204, 0.99845487, 0.99918433, 0.49320207],
                             [- 6.27295801, -0.00050569, 0.00030984, 7.27245232, 0.99845512, 0.99918447, 0.49320217],
                             [- 6.27295821, -0.00050561, 0.00030979, 7.27245260, 0.99845537, 0.99918460, 0.49320221],
                             [- 6.27295841, -0.00050553, 0.00030974, 7.27245288, 0.99845563, 0.99918473, 0.49320219],
                             [- 6.27295860, -0.00050545, 0.00030969, 7.27245316, 0.99845588, 0.99918487, 0.49320222],
                             [- 6.27295880, -0.00050536, 0.00030964, 7.27245344, 0.99845613, 0.99918500, 0.49320228],
                             [- 6.27295900, -0.00050528, 0.00030959, 7.27245372, 0.99845638, 0.99918513, 0.49320230],
                             [- 6.27295920, -0.00050520, 0.00030954, 7.27245400, 0.99845663, 0.99918526, 0.49320235],
                             [- 6.27295939, -0.00050512, 0.00030949, 7.27245428, 0.99845689, 0.99918540, 0.49320231],
                             [- 6.27295959, -0.00050503, 0.00030944, 7.27245456, 0.99845714, 0.99918553, 0.49320239],
                             [- 6.27295979, -0.00050495, 0.00030938, 7.27245484, 0.99845739, 0.99918566, 0.49320239],
                             [- 6.27295998, -0.00050487, 0.00030933, 7.27245512, 0.99845764, 0.99918580, 0.49320243],
                             [- 6.27296018, -0.00050479, 0.00030928, 7.27245539, 0.99845789, 0.99918593, 0.49320240],
                             [- 6.27296038, -0.00050470, 0.00030923, 7.27245567, 0.99845814, 0.99918606, 0.49320248],
                             [- 6.27296058, -0.00050462, 0.00030918, 7.27245595, 0.99845840, 0.99918619, 0.49320254],
                             [- 6.27296077, -0.00050454, 0.00030913, 7.27245623, 0.99845865, 0.99918633, 0.49320254],
                             [- 6.27296097, -0.00050446, 0.00030908, 7.27245651, 0.99845890, 0.99918646, 0.49320257],
                             [- 6.27296117, -0.00050438, 0.00030903, 7.27245679, 0.99845915, 0.99918659, 0.49320260],
                             [- 6.27296136, -0.00050429, 0.00030898, 7.27245707, 0.99845940, 0.99918673, 0.49320260],
                             [- 6.27296156, -0.00050421, 0.00030893, 7.27245735, 0.99845965, 0.99918686, 0.49320264],
                             [- 6.27296176, -0.00050413, 0.00030888, 7.27245763, 0.99845990, 0.99918699, 0.49320270],
                             [- 6.27296195, -0.00050405, 0.00030883, 7.27245790, 0.99846015, 0.99918712, 0.49320271],
                             [- 6.27296215, -0.00050397, 0.00030878, 7.27245818, 0.99846040, 0.99918726, 0.49320271],
                             [- 6.27296234, -0.00050388, 0.00030873, 7.27245846, 0.99846066, 0.99918739, 0.49320279],
                             [- 6.27296254, -0.00050380, 0.00030868, 7.27245874, 0.99846091, 0.99918752, 0.49320285],
                             [- 6.27296274, -0.00050372, 0.00030863, 7.27245902, 0.99846116, 0.99918765, 0.49320278],
                             [- 6.27296293, -0.00050364, 0.00030858, 7.27245930, 0.99846141, 0.99918778, 0.49320286],
                             [- 6.27296313, -0.00050356, 0.00030853, 7.27245957, 0.99846166, 0.99918792, 0.49320284],
                             [- 6.27296332, -0.00050347, 0.00030848, 7.27245985, 0.99846191, 0.99918805, 0.49320294],
                             [- 6.27296352, -0.00050339, 0.00030843, 7.27246013, 0.99846216, 0.99918818, 0.49320297],
                             [- 6.27296372, -0.00050331, 0.00030838, 7.27246041, 0.99846241, 0.99918831, 0.49320295],
                             [- 6.27296391, -0.00050323, 0.00030833, 7.27246068, 0.99846266, 0.99918844, 0.49320301],
                             [- 6.27296411, -0.00050315, 0.00030828, 7.27246096, 0.99846291, 0.99918858, 0.49320303],
                             [- 6.27296430, -0.00050306, 0.00030823, 7.27246124, 0.99846316, 0.99918871, 0.49320305],
                             [- 6.27296450, -0.00050298, 0.00030818, 7.27246152, 0.99846341, 0.99918884, 0.49320310],
                             [- 6.27296469, -0.00050290, 0.00030813, 7.27246179, 0.99846366, 0.99918897, 0.49320314],
                             [- 6.27296489, -0.00050282, 0.00030808, 7.27246207, 0.99846391, 0.99918910, 0.49320318],
                             [- 6.27296509, -0.00050274, 0.00030803, 7.27246235, 0.99846416, 0.99918924, 0.49320319],
                             [- 6.27296528, -0.00050266, 0.00030798, 7.27246263, 0.99846441, 0.99918937, 0.49320326],
                             [- 6.27296548, -0.00050257, 0.00030793, 7.27246290, 0.99846466, 0.99918950, 0.49320325],
                             [- 6.27296567, -0.00050249, 0.00030788, 7.27246318, 0.99846490, 0.99918963, 0.49320329],
                             [- 6.27296587, -0.00050241, 0.00030783, 7.27246346, 0.99846515, 0.99918976, 0.49320330],
                             [- 6.27296606, -0.00050233, 0.00030778, 7.27246373, 0.99846540, 0.99918989, 0.49320333],
                             [- 6.27296626, -0.00050225, 0.00030773, 7.27246401, 0.99846565, 0.99919002, 0.49320339],
                             [- 6.27296645, -0.00050217, 0.00030768, 7.27246429, 0.99846590, 0.99919016, 0.49320339],
                             [- 6.27296665, -0.00050209, 0.00030763, 7.27246456, 0.99846615, 0.99919029, 0.49320338],
                             [- 6.27296684, -0.00050200, 0.00030758, 7.27246484, 0.99846640, 0.99919042, 0.49320344],
                             [- 6.27296704, -0.00050192, 0.00030753, 7.27246511, 0.99846665, 0.99919055, 0.49320353],
                             [- 6.27296723, -0.00050184, 0.00030748, 7.27246539, 0.99846690, 0.99919068, 0.49320349],
                             [- 6.27296743, -0.00050176, 0.00030743, 7.27246567, 0.99846715, 0.99919081, 0.49320357],
                             [- 6.27296762, -0.00050168, 0.00030738, 7.27246594, 0.99846739, 0.99919094, 0.49320354],
                             [- 6.27296781, -0.00050160, 0.00030733, 7.27246622, 0.99846764, 0.99919108, 0.49320360],
                             [- 6.27296801, -0.00050152, 0.00030728, 7.27246649, 0.99846789, 0.99919121, 0.49320366],
                             [- 6.27296820, -0.00050143, 0.00030723, 7.27246677, 0.99846814, 0.99919134, 0.49320363],
                             [- 6.27296840, -0.00050135, 0.00030718, 7.27246704, 0.99846839, 0.99919147, 0.49320369],
                             [- 6.27296859, -0.00050127, 0.00030713, 7.27246732, 0.99846864, 0.99919160, 0.49320376],
                             [- 6.27296879, -0.00050119, 0.00030708, 7.27246760, 0.99846888, 0.99919173, 0.49320377],
                             [- 6.27296898, -0.00050111, 0.00030703, 7.27246787, 0.99846913, 0.99919186, 0.49320381],
                             [- 6.27296917, -0.00050103, 0.00030698, 7.27246815, 0.99846938, 0.99919199, 0.49320377],
                             [- 6.27296937, -0.00050095, 0.00030693, 7.27246842, 0.99846963, 0.99919212, 0.49320384],
                             [- 6.27296956, -0.00050087, 0.00030688, 7.27246870, 0.99846987, 0.99919225, 0.49320380],
                             [- 6.27296976, -0.00050079, 0.00030683, 7.27246897, 0.99847012, 0.99919238, 0.49320387],
                             [- 6.27296995, -0.00050070, 0.00030678, 7.27246925, 0.99847037, 0.99919251, 0.49320387],
                             [- 6.27297014, -0.00050062, 0.00030673, 7.27246952, 0.99847062, 0.99919265, 0.49320397],
                             [- 6.27297034, -0.00050054, 0.00030668, 7.27246980, 0.99847086, 0.99919278, 0.49320397],
                             [- 6.27297053, -0.00050046, 0.00030663, 7.27247007, 0.99847111, 0.99919291, 0.49320399],
                             [- 6.27297073, -0.00050038, 0.00030658, 7.27247034, 0.99847136, 0.99919304, 0.49320405],
                             [- 6.27297092, -0.00050030, 0.00030653, 7.27247062, 0.99847161, 0.99919317, 0.49320409],
                             [- 6.27297111, -0.00050022, 0.00030648, 7.27247089, 0.99847185, 0.99919330, 0.49320412],
                             [- 6.27297131, -0.00050014, 0.00030643, 7.27247117, 0.99847210, 0.99919343, 0.49320414],
                             [- 6.27297150, -0.00050006, 0.00030638, 7.27247144, 0.99847235, 0.99919356, 0.49320414],
                             [- 6.27297169, -0.00049998, 0.00030633, 7.27247172, 0.99847259, 0.99919369, 0.49320417],
                             [- 6.27297189, -0.00049990, 0.00030629, 7.27247199, 0.99847284, 0.99919382, 0.49320418],
                             [- 6.27297208, -0.00049982, 0.00030624, 7.27247226, 0.99847309, 0.99919395, 0.49320420],
                             [- 6.27297227, -0.00049973, 0.00030619, 7.27247254, 0.99847333, 0.99919408, 0.49320426],
                             [- 6.27297246, -0.00049965, 0.00030614, 7.27247281, 0.99847358, 0.99919421, 0.49320420],
                             [- 6.27297266, -0.00049957, 0.00030609, 7.27247308, 0.99847383, 0.99919434, 0.49320435],
                             [- 6.27297285, -0.00049949, 0.00030604, 7.27247336, 0.99847407, 0.99919447, 0.49320431],
                             [- 6.27297304, -0.00049941, 0.00030599, 7.27247363, 0.99847432, 0.99919460, 0.49320437],
                             [- 6.27297324, -0.00049933, 0.00030594, 7.27247390, 0.99847457, 0.99919473, 0.49320437],
                             [- 6.27297343, -0.00049925, 0.00030589, 7.27247418, 0.99847481, 0.99919486, 0.49320443],
                             [- 6.27297362, -0.00049917, 0.00030584, 7.27247445, 0.99847506, 0.99919499, 0.49320448],
                             [- 6.27297381, -0.00049909, 0.00030579, 7.27247472, 0.99847530, 0.99919512, 0.49320448],
                             [- 6.27297401, -0.00049901, 0.00030574, 7.27247500, 0.99847555, 0.99919525, 0.49320449],
                             [- 6.27297420, -0.00049893, 0.00030569, 7.27247527, 0.99847580, 0.99919538, 0.49320451],
                             [- 6.27297439, -0.00049885, 0.00030564, 7.27247554, 0.99847604, 0.99919551, 0.49320453],
                             [- 6.27297458, -0.00049877, 0.00030559, 7.27247582, 0.99847629, 0.99919564, 0.49320461],
                             [- 6.27297478, -0.00049869, 0.00030554, 7.27247609, 0.99847653, 0.99919577, 0.49320461],
                             [- 6.27297497, -0.00049861, 0.00030550, 7.27247636, 0.99847678, 0.99919590, 0.49320465],
                             [- 6.27297516, -0.00049853, 0.00030545, 7.27247663, 0.99847702, 0.99919603, 0.49320475],
                             [- 6.27297535, -0.00049845, 0.00030540, 7.27247691, 0.99847727, 0.99919616, 0.49320473],
                             [- 6.27297554, -0.00049837, 0.00030535, 7.27247718, 0.99847751, 0.99919629, 0.49320474],
                             [- 6.27297574, -0.00049829, 0.00030530, 7.27247745, 0.99847776, 0.99919641, 0.49320475],
                             [- 6.27297593, -0.00049821, 0.00030525, 7.27247772, 0.99847800, 0.99919654, 0.49320486],
                             [- 6.27297612, -0.00049813, 0.00030520, 7.27247799, 0.99847825, 0.99919667, 0.49320483],
                             [- 6.27297631, -0.00049805, 0.00030515, 7.27247827, 0.99847849, 0.99919680, 0.49320494],
                             [- 6.27297650, -0.00049797, 0.00030510, 7.27247854, 0.99847874, 0.99919693, 0.49320494],
                             [- 6.27297670, -0.00049789, 0.00030505, 7.27247881, 0.99847898, 0.99919706, 0.49320488],
                             [- 6.27297689, -0.00049781, 0.00030500, 7.27247908, 0.99847923, 0.99919719, 0.49320497],
                             [- 6.27297708, -0.00049773, 0.00030495, 7.27247935, 0.99847947, 0.99919732, 0.49320498],
                             [- 6.27297727, -0.00049765, 0.00030491, 7.27247962, 0.99847972, 0.99919745, 0.49320504],
                             [- 6.27297746, -0.00049757, 0.00030486, 7.27247990, 0.99847996, 0.99919758, 0.49320498],
                             [- 6.27297765, -0.00049749, 0.00030481, 7.27248017, 0.99848021, 0.99919771, 0.49320508],
                             [- 6.27297784, -0.00049741, 0.00030476, 7.27248044, 0.99848045, 0.99919784, 0.49320506],
                             [- 6.27297804, -0.00049733, 0.00030471, 7.27248071, 0.99848070, 0.99919796, 0.49320512],
                             [- 6.27297823, -0.00049725, 0.00030466, 7.27248098, 0.99848094, 0.99919809, 0.49320520],
                             [- 6.27297842, -0.00049717, 0.00030461, 7.27248125, 0.99848118, 0.99919822, 0.49320518],
                             [- 6.27297861, -0.00049709, 0.00030456, 7.27248152, 0.99848143, 0.99919835, 0.49320519],
                             [- 6.27297880, -0.00049701, 0.00030451, 7.27248179, 0.99848167, 0.99919848, 0.49320524],
                             [- 6.27297899, -0.00049693, 0.00030446, 7.27248206, 0.99848192, 0.99919861, 0.49320527],
                             [- 6.27297918, -0.00049685, 0.00030442, 7.27248233, 0.99848216, 0.99919874, 0.49320529],
                             [- 6.27297937, -0.00049677, 0.00030437, 7.27248260, 0.99848240, 0.99919887, 0.49320530],
                             [- 6.27297956, -0.00049669, 0.00030432, 7.27248288, 0.99848265, 0.99919899, 0.49320533],
                             [- 6.27297975, -0.00049661, 0.00030427, 7.27248315, 0.99848289, 0.99919912, 0.49320533],
                             [- 6.27297994, -0.00049653, 0.00030422, 7.27248342, 0.99848313, 0.99919925, 0.49320545],
                             [- 6.27298013, -0.00049645, 0.00030417, 7.27248369, 0.99848338, 0.99919938, 0.49320550],
                             [- 6.27298033, -0.00049637, 0.00030412, 7.27248396, 0.99848362, 0.99919951, 0.49320544],
                             [- 6.27298052, -0.00049629, 0.00030407, 7.27248423, 0.99848386, 0.99919964, 0.49320542],
                             [- 6.27298071, -0.00049621, 0.00030403, 7.27248450, 0.99848411, 0.99919976, 0.49320549],
                             [- 6.27298090, -0.00049613, 0.00030398, 7.27248477, 0.99848435, 0.99919989, 0.49320553],
                             [- 6.27298109, -0.00049605, 0.00030393, 7.27248504, 0.99848459, 0.99920002, 0.49320549],
                             [- 6.27298128, -0.00049597, 0.00030388, 7.27248531, 0.99848484, 0.99920015, 0.49320559],
                             [- 6.27298147, -0.00049589, 0.00030383, 7.27248557, 0.99848508, 0.99920028, 0.49320560],
                             [- 6.27298166, -0.00049581, 0.00030378, 7.27248584, 0.99848532, 0.99920041, 0.49320562],
                             [- 6.27298185, -0.00049573, 0.00030373, 7.27248611, 0.99848556, 0.99920053, 0.49320572],
                             [- 6.27298204, -0.00049565, 0.00030368, 7.27248638, 0.99848581, 0.99920066, 0.49320570],
                             [- 6.27298223, -0.00049557, 0.00030364, 7.27248665, 0.99848605, 0.99920079, 0.49320571],
                             [- 6.27298242, -0.00049550, 0.00030359, 7.27248692, 0.99848629, 0.99920092, 0.49320573],
                             [- 6.27298261, -0.00049542, 0.00030354, 7.27248719, 0.99848653, 0.99920105, 0.49320576],
                             [- 6.27298280, -0.00049534, 0.00030349, 7.27248746, 0.99848678, 0.99920117, 0.49320585],
                             [- 6.27298299, -0.00049526, 0.00030344, 7.27248773, 0.99848702, 0.99920130, 0.49320584],
                             [- 6.27298318, -0.00049518, 0.00030339, 7.27248800, 0.99848726, 0.99920143, 0.49320590],
                             [- 6.27298336, -0.00049510, 0.00030334, 7.27248827, 0.99848750, 0.99920156, 0.49320592],
                             [- 6.27298355, -0.00049502, 0.00030330, 7.27248853, 0.99848774, 0.99920168, 0.49320594],
                             [- 6.27298374, -0.00049494, 0.00030325, 7.27248880, 0.99848799, 0.99920181, 0.49320595],
                             [- 6.27298393, -0.00049486, 0.00030320, 7.27248907, 0.99848823, 0.99920194, 0.49320597],
                             [- 6.27298412, -0.00049478, 0.00030315, 7.27248934, 0.99848847, 0.99920207, 0.49320596],
                             [- 6.27298431, -0.00049470, 0.00030310, 7.27248961, 0.99848871, 0.99920220, 0.49320603],
                             [- 6.27298450, -0.00049462, 0.00030305, 7.27248988, 0.99848895, 0.99920232, 0.49320613],
                             [- 6.27298469, -0.00049455, 0.00030300, 7.27249014, 0.99848919, 0.99920245, 0.49320607],
                             [- 6.27298488, -0.00049447, 0.00030296, 7.27249041, 0.99848944, 0.99920258, 0.49320616],
                             [- 6.27298507, -0.00049439, 0.00030291, 7.27249068, 0.99848968, 0.99920270, 0.49320618],
                             [- 6.27298526, -0.00049431, 0.00030286, 7.27249095, 0.99848992, 0.99920283, 0.49320613],
                             [- 6.27298545, -0.00049423, 0.00030281, 7.27249122, 0.99849016, 0.99920296, 0.49320620],
                             [- 6.27298563, -0.00049415, 0.00030276, 7.27249148, 0.99849040, 0.99920309, 0.49320622],
                             [- 6.27298582, -0.00049407, 0.00030271, 7.27249175, 0.99849064, 0.99920321, 0.49320626],
                             [- 6.27298601, -0.00049399, 0.00030267, 7.27249202, 0.99849088, 0.99920334, 0.49320625],
                             [- 6.27298620, -0.00049391, 0.00030262, 7.27249229, 0.99849112, 0.99920347, 0.49320640],
                             [- 6.27298639, -0.00049384, 0.00030257, 7.27249255, 0.99849136, 0.99920360, 0.49320640],
                             [- 6.27298658, -0.00049376, 0.00030252, 7.27249282, 0.99849161, 0.99920372, 0.49320636],
                             [- 6.27298677, -0.00049368, 0.00030247, 7.27249309, 0.99849185, 0.99920385, 0.49320641],
                             [- 6.27298695, -0.00049360, 0.00030242, 7.27249336, 0.99849209, 0.99920398, 0.49320644],
                             [- 6.27298714, -0.00049352, 0.00030238, 7.27249362, 0.99849233, 0.99920410, 0.49320643],
                             [- 6.27298733, -0.00049344, 0.00030233, 7.27249389, 0.99849257, 0.99920423, 0.49320647],
                             [- 6.27298752, -0.00049336, 0.00030228, 7.27249416, 0.99849281, 0.99920436, 0.49320649],
                             [- 6.27298771, -0.00049328, 0.00030223, 7.27249442, 0.99849305, 0.99920448, 0.49320659],
                             [- 6.27298790, -0.00049321, 0.00030218, 7.27249469, 0.99849329, 0.99920461, 0.49320657],
                             [- 6.27298808, -0.00049313, 0.00030214, 7.27249496, 0.99849353, 0.99920474, 0.49320661],
                             [- 6.27298827, -0.00049305, 0.00030209, 7.27249522, 0.99849377, 0.99920486, 0.49320660],
                             [- 6.27298846, -0.00049297, 0.00030204, 7.27249549, 0.99849401, 0.99920499, 0.49320663],
                             [- 6.27298865, -0.00049289, 0.00030199, 7.27249576, 0.99849425, 0.99920512, 0.49320673],
                             [- 6.27298883, -0.00049281, 0.00030194, 7.27249602, 0.99849449, 0.99920524, 0.49320673],
                             [- 6.27298902, -0.00049273, 0.00030189, 7.27249629, 0.99849473, 0.99920537, 0.49320669],
                             [- 6.27298921, -0.00049266, 0.00030185, 7.27249655, 0.99849497, 0.99920550, 0.49320677],
                             [- 6.27298940, -0.00049258, 0.00030180, 7.27249682, 0.99849521, 0.99920562, 0.49320680],
                             [- 6.27298959, -0.00049250, 0.00030175, 7.27249709, 0.99849545, 0.99920575, 0.49320687],
                             [- 6.27298977, -0.00049242, 0.00030170, 7.27249735, 0.99849569, 0.99920588, 0.49320693],
                             [- 6.27298996, -0.00049234, 0.00030165, 7.27249762, 0.99849593, 0.99920600, 0.49320681],
                             [- 6.27299015, -0.00049226, 0.00030161, 7.27249788, 0.99849616, 0.99920613, 0.49320699],
                             [- 6.27299033, -0.00049219, 0.00030156, 7.27249815, 0.99849640, 0.99920626, 0.49320687],
                             [- 6.27299052, -0.00049211, 0.00030151, 7.27249841, 0.99849664, 0.99920638, 0.49320699],
                             [- 6.27299071, -0.00049203, 0.00030146, 7.27249868, 0.99849688, 0.99920651, 0.49320705],
                             [- 6.27299090, -0.00049195, 0.00030141, 7.27249894, 0.99849712, 0.99920663, 0.49320709],
                             [- 6.27299108, -0.00049187, 0.00030137, 7.27249921, 0.99849736, 0.99920676, 0.49320704],
                             [- 6.27299127, -0.00049180, 0.00030132, 7.27249948, 0.99849760, 0.99920689, 0.49320700],
                             [- 6.27299146, -0.00049172, 0.00030127, 7.27249974, 0.99849784, 0.99920701, 0.49320709],
                             [- 6.27299164, -0.00049164, 0.00030122, 7.27250001, 0.99849808, 0.99920714, 0.49320722],
                             [- 6.27299183, -0.00049156, 0.00030118, 7.27250027, 0.99849831, 0.99920726, 0.49320713],
                             [- 6.27299202, -0.00049148, 0.00030113, 7.27250053, 0.99849855, 0.99920739, 0.49320717],
                             [- 6.27299220, -0.00049140, 0.00030108, 7.27250080, 0.99849879, 0.99920752, 0.49320720],
                             [- 6.27299239, -0.00049133, 0.00030103, 7.27250106, 0.99849903, 0.99920764, 0.49320724],
                             [- 6.27299258, -0.00049125, 0.00030098, 7.27250133, 0.99849927, 0.99920777, 0.49320731],
                             [- 6.27299276, -0.00049117, 0.00030094, 7.27250159, 0.99849951, 0.99920789, 0.49320733],
                             [- 6.27299295, -0.00049109, 0.00030089, 7.27250186, 0.99849974, 0.99920802, 0.49320736],
                             [- 6.27299314, -0.00049102, 0.00030084, 7.27250212, 0.99849998, 0.99920814, 0.49320742],
                             [- 6.27299332, -0.00049094, 0.00030079, 7.27250239, 0.99850022, 0.99920827, 0.49320737],
                             [- 6.27299351, -0.00049086, 0.00030075, 7.27250265, 0.99850046, 0.99920840, 0.49320741],
                             [- 6.27299370, -0.00049078, 0.00030070, 7.27250291, 0.99850070, 0.99920852, 0.49320747],
                             [- 6.27299388, -0.00049070, 0.00030065, 7.27250318, 0.99850093, 0.99920865, 0.49320746],
                             [- 6.27299407, -0.00049063, 0.00030060, 7.27250344, 0.99850117, 0.99920877, 0.49320747],
                             [- 6.27299425, -0.00049055, 0.00030055, 7.27250371, 0.99850141, 0.99920890, 0.49320753],
                             [- 6.27299444, -0.00049047, 0.00030051, 7.27250397, 0.99850165, 0.99920902, 0.49320754],
                             [- 6.27299463, -0.00049039, 0.00030046, 7.27250423, 0.99850188, 0.99920915, 0.49320761],
                             [- 6.27299481, -0.00049032, 0.00030041, 7.27250450, 0.99850212, 0.99920927, 0.49320761],
                             [- 6.27299500, -0.00049024, 0.00030036, 7.27250476, 0.99850236, 0.99920940, 0.49320763],
                             [- 6.27299518, -0.00049016, 0.00030032, 7.27250502, 0.99850260, 0.99920952, 0.49320763],
                             [- 6.27299537, -0.00049008, 0.00030027, 7.27250529, 0.99850283, 0.99920965, 0.49320764],
                             [- 6.27299556, -0.00049000, 0.00030022, 7.27250555, 0.99850307, 0.99920977, 0.49320768],
                             [- 6.27299574, -0.00048993, 0.00030017, 7.27250581, 0.99850331, 0.99920990, 0.49320774],
                             [- 6.27299593, -0.00048985, 0.00030013, 7.27250608, 0.99850354, 0.99921002, 0.49320770],
                             [- 6.27299611, -0.00048977, 0.00030008, 7.27250634, 0.99850378, 0.99921015, 0.49320782],
                             [- 6.27299630, -0.00048969, 0.00030003, 7.27250660, 0.99850402, 0.99921027, 0.49320781],
                             [- 6.27299648, -0.00048962, 0.00029998, 7.27250687, 0.99850426, 0.99921040, 0.49320784],
                             [- 6.27299667, -0.00048954, 0.00029994, 7.27250713, 0.99850449, 0.99921052, 0.49320787],
                             [- 6.27299685, -0.00048946, 0.00029989, 7.27250739, 0.99850473, 0.99921065, 0.49320791],
                             [- 6.27299704, -0.00048938, 0.00029984, 7.27250765, 0.99850496, 0.99921077, 0.49320794],
                             [- 6.27299722, -0.00048931, 0.00029979, 7.27250792, 0.99850520, 0.99921090, 0.49320795],
                             [- 6.27299741, -0.00048923, 0.00029975, 7.27250818, 0.99850544, 0.99921102, 0.49320801],
                             [- 6.27299759, -0.00048915, 0.00029970, 7.27250844, 0.99850567, 0.99921115, 0.49320802],
                             [- 6.27299778, -0.00048908, 0.00029965, 7.27250870, 0.99850591, 0.99921127, 0.49320805],
                             [- 6.27299796, -0.00048900, 0.00029960, 7.27250896, 0.99850615, 0.99921140, 0.49320810],
                             [- 6.27299815, -0.00048892, 0.00029956, 7.27250923, 0.99850638, 0.99921152, 0.49320806],
                             [- 6.27299833, -0.00048884, 0.00029951, 7.27250949, 0.99850662, 0.99921165, 0.49320815],
                             [- 6.27299852, -0.00048877, 0.00029946, 7.27250975, 0.99850685, 0.99921177, 0.49320816],
                             [- 6.27299870, -0.00048869, 0.00029941, 7.27251001, 0.99850709, 0.99921190, 0.49320817],
                             [- 6.27299889, -0.00048861, 0.00029937, 7.27251027, 0.99850733, 0.99921202, 0.49320820],
                             [- 6.27299907, -0.00048854, 0.00029932, 7.27251054, 0.99850756, 0.99921214, 0.49320826],
                             [- 6.27299926, -0.00048846, 0.00029927, 7.27251080, 0.99850780, 0.99921227, 0.49320827],
                             [- 6.27299944, -0.00048838, 0.00029923, 7.27251106, 0.99850803, 0.99921239, 0.49320823],
                             [- 6.27299962, -0.00048830, 0.00029918, 7.27251132, 0.99850827, 0.99921252, 0.49320833],
                             [- 6.27299981, -0.00048823, 0.00029913, 7.27251158, 0.99850850, 0.99921264, 0.49320833],
                             [- 6.27299999, -0.00048815, 0.00029908, 7.27251184, 0.99850874, 0.99921277, 0.49320839],
                             [- 6.27300018, -0.00048807, 0.00029904, 7.27251210, 0.99850897, 0.99921289, 0.49320841],
                             [- 6.27300036, -0.00048800, 0.00029899, 7.27251236, 0.99850921, 0.99921301, 0.49320850],
                             [- 6.27300055, -0.00048792, 0.00029894, 7.27251263, 0.99850944, 0.99921314, 0.49320846],
                             [- 6.27300073, -0.00048784, 0.00029890, 7.27251289, 0.99850968, 0.99921326, 0.49320848],
                             [- 6.27300091, -0.00048777, 0.00029885, 7.27251315, 0.99850991, 0.99921339, 0.49320847],
                             [- 6.27300110, -0.00048769, 0.00029880, 7.27251341, 0.99851015, 0.99921351, 0.49320858],
                             [- 6.27300128, -0.00048761, 0.00029875, 7.27251367, 0.99851038, 0.99921363, 0.49320864],
                             [- 6.27300146, -0.00048754, 0.00029871, 7.27251393, 0.99851062, 0.99921376, 0.49320855],
                             [- 6.27300165, -0.00048746, 0.00029866, 7.27251419, 0.99851085, 0.99921388, 0.49320856],
                             [- 6.27300183, -0.00048738, 0.00029861, 7.27251445, 0.99851109, 0.99921401, 0.49320863],
                             [- 6.27300202, -0.00048730, 0.00029857, 7.27251471, 0.99851132, 0.99921413, 0.49320867],
                             [- 6.27300220, -0.00048723, 0.00029852, 7.27251497, 0.99851156, 0.99921425, 0.49320861],
                             [- 6.27300238, -0.00048715, 0.00029847, 7.27251523, 0.99851179, 0.99921438, 0.49320874],
                             [- 6.27300257, -0.00048707, 0.00029842, 7.27251549, 0.99851202, 0.99921450, 0.49320874],
                             [- 6.27300275, -0.00048700, 0.00029838, 7.27251575, 0.99851226, 0.99921462, 0.49320881],
                             [- 6.27300293, -0.00048692, 0.00029833, 7.27251601, 0.99851249, 0.99921475, 0.49320880],
                             [- 6.27300312, -0.00048685, 0.00029828, 7.27251627, 0.99851273, 0.99921487, 0.49320887],
                             [- 6.27300330, -0.00048677, 0.00029824, 7.27251653, 0.99851296, 0.99921499, 0.49320886],
                             [- 6.27300348, -0.00048669, 0.00029819, 7.27251679, 0.99851320, 0.99921512, 0.49320893],
                             [- 6.27300367, -0.00048662, 0.00029814, 7.27251705, 0.99851343, 0.99921524, 0.49320893],
                             [- 6.27300385, -0.00048654, 0.00029810, 7.27251731, 0.99851366, 0.99921536, 0.49320895],
                             [- 6.27300403, -0.00048646, 0.00029805, 7.27251757, 0.99851390, 0.99921549, 0.49320892],
                             [- 6.27300421, -0.00048639, 0.00029800, 7.27251783, 0.99851413, 0.99921561, 0.49320898],
                             [- 6.27300440, -0.00048631, 0.00029796, 7.27251809, 0.99851436, 0.99921573, 0.49320901],
                             [- 6.27300458, -0.00048623, 0.00029791, 7.27251835, 0.99851460, 0.99921586, 0.49320901],
                             [- 6.27300476, -0.00048616, 0.00029786, 7.27251861, 0.99851483, 0.99921598, 0.49320912],
                             [- 6.27300495, -0.00048608, 0.00029782, 7.27251886, 0.99851506, 0.99921610, 0.49320910],
                             [- 6.27300513, -0.00048600, 0.00029777, 7.27251912, 0.99851530, 0.99921623, 0.49320913],
                             [- 6.27300531, -0.00048593, 0.00029772, 7.27251938, 0.99851553, 0.99921635, 0.49320917],
                             [- 6.27300549, -0.00048585, 0.00029768, 7.27251964, 0.99851576, 0.99921647, 0.49320916],
                             [- 6.27300568, -0.00048578, 0.00029763, 7.27251990, 0.99851600, 0.99921660, 0.49320920],
                             [- 6.27300586, -0.00048570, 0.00029758, 7.27252016, 0.99851623, 0.99921672, 0.49320924],
                             [- 6.27300604, -0.00048562, 0.00029753, 7.27252042, 0.99851646, 0.99921684, 0.49320926],
                             [- 6.27300622, -0.00048555, 0.00029749, 7.27252068, 0.99851669, 0.99921696, 0.49320930],
                             [- 6.27300640, -0.00048547, 0.00029744, 7.27252093, 0.99851693, 0.99921709, 0.49320931],
                             [- 6.27300659, -0.00048539, 0.00029739, 7.27252119, 0.99851716, 0.99921721, 0.49320938],
                             [- 6.27300677, -0.00048532, 0.00029735, 7.27252145, 0.99851739, 0.99921733, 0.49320939],
                             [- 6.27300695, -0.00048524, 0.00029730, 7.27252171, 0.99851763, 0.99921746, 0.49320948],
                             [- 6.27300713, -0.00048517, 0.00029725, 7.27252197, 0.99851786, 0.99921758, 0.49320954],
                             [- 6.27300732, -0.00048509, 0.00029721, 7.27252222, 0.99851809, 0.99921770, 0.49320950],
                             [- 6.27300750, -0.00048501, 0.00029716, 7.27252248, 0.99851832, 0.99921782, 0.49320952],
                             [- 6.27300768, -0.00048494, 0.00029712, 7.27252274, 0.99851855, 0.99921795, 0.49320954],
                             [- 6.27300786, -0.00048486, 0.00029707, 7.27252300, 0.99851879, 0.99921807, 0.49320960],
                             [- 6.27300804, -0.00048479, 0.00029702, 7.27252326, 0.99851902, 0.99921819, 0.49320959],
                             [- 6.27300822, -0.00048471, 0.00029698, 7.27252351, 0.99851925, 0.99921831, 0.49320958],
                             [- 6.27300841, -0.00048463, 0.00029693, 7.27252377, 0.99851948, 0.99921844, 0.49320965],
                             [- 6.27300859, -0.00048456, 0.00029688, 7.27252403, 0.99851971, 0.99921856, 0.49320967],
                             [- 6.27300877, -0.00048448, 0.00029684, 7.27252429, 0.99851995, 0.99921868, 0.49320972],
                             [- 6.27300895, -0.00048441, 0.00029679, 7.27252454, 0.99852018, 0.99921880, 0.49320973],
                             [- 6.27300913, -0.00048433, 0.00029674, 7.27252480, 0.99852041, 0.99921893, 0.49320972],
                             [- 6.27300931, -0.00048426, 0.00029670, 7.27252506, 0.99852064, 0.99921905, 0.49320980],
                             [- 6.27300949, -0.00048418, 0.00029665, 7.27252531, 0.99852087, 0.99921917, 0.49320981],
                             [- 6.27300968, -0.00048410, 0.00029660, 7.27252557, 0.99852110, 0.99921929, 0.49320974],
                             [- 6.27300986, -0.00048403, 0.00029656, 7.27252583, 0.99852134, 0.99921941, 0.49320985],
                             [- 6.27301004, -0.00048395, 0.00029651, 7.27252608, 0.99852157, 0.99921954, 0.49320992],
                             [- 6.27301022, -0.00048388, 0.00029646, 7.27252634, 0.99852180, 0.99921966, 0.49320992],
                             [- 6.27301040, -0.00048380, 0.00029642, 7.27252660, 0.99852203, 0.99921978, 0.49320993],
                             [- 6.27301058, -0.00048373, 0.00029637, 7.27252685, 0.99852226, 0.99921990, 0.49320993],
                             [- 6.27301076, -0.00048365, 0.00029633, 7.27252711, 0.99852249, 0.99922002, 0.49321001],
                             [- 6.27301094, -0.00048357, 0.00029628, 7.27252737, 0.99852272, 0.99922015, 0.49321003],
                             [- 6.27301112, -0.00048350, 0.00029623, 7.27252762, 0.99852295, 0.99922027, 0.49321005],
                             [- 6.27301130, -0.00048342, 0.00029619, 7.27252788, 0.99852318, 0.99922039, 0.49321008],
                             [- 6.27301148, -0.00048335, 0.00029614, 7.27252814, 0.99852341, 0.99922051, 0.49321008],
                             [- 6.27301166, -0.00048327, 0.00029609, 7.27252839, 0.99852364, 0.99922063, 0.49321010],
                             [- 6.27301185, -0.00048320, 0.00029605, 7.27252865, 0.99852388, 0.99922075, 0.49321017],
                             [- 6.27301203, -0.00048312, 0.00029600, 7.27252890, 0.99852411, 0.99922088, 0.49321023],
                             [- 6.27301221, -0.00048305, 0.00029596, 7.27252916, 0.99852434, 0.99922100, 0.49321020],
                             [- 6.27301239, -0.00048297, 0.00029591, 7.27252942, 0.99852457, 0.99922112, 0.49321023],
                             [- 6.27301257, -0.00048290, 0.00029586, 7.27252967, 0.99852480, 0.99922124, 0.49321024],
                             [- 6.27301275, -0.00048282, 0.00029582, 7.27252993, 0.99852503, 0.99922136, 0.49321030],
                             [- 6.27301293, -0.00048275, 0.00029577, 7.27253018, 0.99852526, 0.99922148, 0.49321031],
                             [- 6.27301311, -0.00048267, 0.00029572, 7.27253044, 0.99852549, 0.99922161, 0.49321030],
                             [- 6.27301329, -0.00048259, 0.00029568, 7.27253069, 0.99852572, 0.99922173, 0.49321030],
                             [- 6.27301347, -0.00048252, 0.00029563, 7.27253095, 0.99852595, 0.99922185, 0.49321039],
                             [- 6.27301365, -0.00048244, 0.00029559, 7.27253120, 0.99852618, 0.99922197, 0.49321037],
                             [- 6.27301383, -0.00048237, 0.00029554, 7.27253146, 0.99852641, 0.99922209, 0.49321046],
                             [- 6.27301401, -0.00048229, 0.00029549, 7.27253171, 0.99852664, 0.99922221, 0.49321048],
                             [- 6.27301419, -0.00048222, 0.00029545, 7.27253197, 0.99852687, 0.99922233, 0.49321049],
                             [- 6.27301437, -0.00048214, 0.00029540, 7.27253222, 0.99852710, 0.99922245, 0.49321046],
                             [- 6.27301455, -0.00048207, 0.00029536, 7.27253248, 0.99852732, 0.99922258, 0.49321053],
                             [- 6.27301473, -0.00048199, 0.00029531, 7.27253273, 0.99852755, 0.99922270, 0.49321060],
                             [- 6.27301491, -0.00048192, 0.00029526, 7.27253299, 0.99852778, 0.99922282, 0.49321066],
                             [- 6.27301509, -0.00048184, 0.00029522, 7.27253324, 0.99852801, 0.99922294, 0.49321062],
                             [- 6.27301526, -0.00048177, 0.00029517, 7.27253350, 0.99852824, 0.99922306, 0.49321065],
                             [- 6.27301544, -0.00048169, 0.00029513, 7.27253375, 0.99852847, 0.99922318, 0.49321069],
                             [- 6.27301562, -0.00048162, 0.00029508, 7.27253400, 0.99852870, 0.99922330, 0.49321066],
                             [- 6.27301580, -0.00048154, 0.00029503, 7.27253426, 0.99852893, 0.99922342, 0.49321075],
                             [- 6.27301598, -0.00048147, 0.00029499, 7.27253451, 0.99852916, 0.99922354, 0.49321074],
                             [- 6.27301616, -0.00048139, 0.00029494, 7.27253477, 0.99852939, 0.99922366, 0.49321082],
                             [- 6.27301634, -0.00048132, 0.00029490, 7.27253502, 0.99852962, 0.99922378, 0.49321081],
                             [- 6.27301652, -0.00048124, 0.00029485, 7.27253528, 0.99852984, 0.99922391, 0.49321083],
                             [- 6.27301670, -0.00048117, 0.00029480, 7.27253553, 0.99853007, 0.99922403, 0.49321094],
                             [- 6.27301688, -0.00048109, 0.00029476, 7.27253578, 0.99853030, 0.99922415, 0.49321096],
                             [- 6.27301706, -0.00048102, 0.00029471, 7.27253604, 0.99853053, 0.99922427, 0.49321093],
                             [- 6.27301724, -0.00048095, 0.00029467, 7.27253629, 0.99853076, 0.99922439, 0.49321094],
                             [- 6.27301741, -0.00048087, 0.00029462, 7.27253654, 0.99853099, 0.99922451, 0.49321098],
                             [- 6.27301759, -0.00048080, 0.00029458, 7.27253680, 0.99853121, 0.99922463, 0.49321102],
                             [- 6.27301777, -0.00048072, 0.00029453, 7.27253705, 0.99853144, 0.99922475, 0.49321101],
                             [- 6.27301795, -0.00048065, 0.00029448, 7.27253730, 0.99853167, 0.99922487, 0.49321109],
                             [- 6.27301813, -0.00048057, 0.00029444, 7.27253756, 0.99853190, 0.99922499, 0.49321107],
                             [- 6.27301831, -0.00048050, 0.00029439, 7.27253781, 0.99853213, 0.99922511, 0.49321107],
                             [- 6.27301849, -0.00048042, 0.00029435, 7.27253806, 0.99853235, 0.99922523, 0.49321110],
                             [- 6.27301866, -0.00048035, 0.00029430, 7.27253832, 0.99853258, 0.99922535, 0.49321118],
                             [- 6.27301884, -0.00048027, 0.00029426, 7.27253857, 0.99853281, 0.99922547, 0.49321119],
                             [- 6.27301902, -0.00048020, 0.00029421, 7.27253882, 0.99853304, 0.99922559, 0.49321116],
                             [- 6.27301920, -0.00048012, 0.00029416, 7.27253907, 0.99853327, 0.99922571, 0.49321127],
                             [- 6.27301938, -0.00048005, 0.00029412, 7.27253933, 0.99853349, 0.99922583, 0.49321123],
                             [- 6.27301956, -0.00047998, 0.00029407, 7.27253958, 0.99853372, 0.99922595, 0.49321129],
                             [- 6.27301973, -0.00047990, 0.00029403, 7.27253983, 0.99853395, 0.99922607, 0.49321135],
                             [- 6.27301991, -0.00047983, 0.00029398, 7.27254008, 0.99853418, 0.99922619, 0.49321135],
                             [- 6.27302009, -0.00047975, 0.00029394, 7.27254034, 0.99853440, 0.99922631, 0.49321139],
                             [- 6.27302027, -0.00047968, 0.00029389, 7.27254059, 0.99853463, 0.99922643, 0.49321146],
                             [- 6.27302044, -0.00047960, 0.00029384, 7.27254084, 0.99853486, 0.99922655, 0.49321142],
                             [- 6.27302062, -0.00047953, 0.00029380, 7.27254109, 0.99853508, 0.99922667, 0.49321144],
                             [- 6.27302080, -0.00047946, 0.00029375, 7.27254135, 0.99853531, 0.99922679, 0.49321144],
                             [- 6.27302098, -0.00047938, 0.00029371, 7.27254160, 0.99853554, 0.99922691, 0.49321157],
                             [- 6.27302116, -0.00047931, 0.00029366, 7.27254185, 0.99853577, 0.99922703, 0.49321152],
                             [- 6.27302133, -0.00047923, 0.00029362, 7.27254210, 0.99853599, 0.99922715, 0.49321153],
                             [- 6.27302151, -0.00047916, 0.00029357, 7.27254235, 0.99853622, 0.99922727, 0.49321157],
                             [- 6.27302169, -0.00047908, 0.00029353, 7.27254260, 0.99853645, 0.99922739, 0.49321160],
                             [- 6.27302187, -0.00047901, 0.00029348, 7.27254286, 0.99853667, 0.99922751, 0.49321170],
                             [- 6.27302204, -0.00047894, 0.00029344, 7.27254311, 0.99853690, 0.99922763, 0.49321172],
                             [- 6.27302222, -0.00047886, 0.00029339, 7.27254336, 0.99853712, 0.99922775, 0.49321163],
                             [- 6.27302240, -0.00047879, 0.00029334, 7.27254361, 0.99853735, 0.99922787, 0.49321166],
                             [- 6.27302257, -0.00047871, 0.00029330, 7.27254386, 0.99853758, 0.99922799, 0.49321175],
                             [- 6.27302275, -0.00047864, 0.00029325, 7.27254411, 0.99853780, 0.99922811, 0.49321177],
                             [- 6.27302293, -0.00047857, 0.00029321, 7.27254436, 0.99853803, 0.99922823, 0.49321185],
                             [- 6.27302311, -0.00047849, 0.00029316, 7.27254461, 0.99853826, 0.99922834, 0.49321182],
                             [- 6.27302328, -0.00047842, 0.00029312, 7.27254486, 0.99853848, 0.99922846, 0.49321190],
                             [- 6.27302346, -0.00047834, 0.00029307, 7.27254512, 0.99853871, 0.99922858, 0.49321191],
                             [- 6.27302364, -0.00047827, 0.00029303, 7.27254537, 0.99853893, 0.99922870, 0.49321187],
                             [- 6.27302381, -0.00047820, 0.00029298, 7.27254562, 0.99853916, 0.99922882, 0.49321186],
                             [- 6.27302399, -0.00047812, 0.00029294, 7.27254587, 0.99853939, 0.99922894, 0.49321193],
                             [- 6.27302417, -0.00047805, 0.00029289, 7.27254612, 0.99853961, 0.99922906, 0.49321201],
                             [- 6.27302434, -0.00047797, 0.00029285, 7.27254637, 0.99853984, 0.99922918, 0.49321205],
                             [- 6.27302452, -0.00047790, 0.00029280, 7.27254662, 0.99854006, 0.99922930, 0.49321210],
                             [- 6.27302470, -0.00047783, 0.00029276, 7.27254687, 0.99854029, 0.99922942, 0.49321211],
                             [- 6.27302487, -0.00047775, 0.00029271, 7.27254712, 0.99854051, 0.99922954, 0.49321206],
                             [- 6.27302505, -0.00047768, 0.00029267, 7.27254737, 0.99854074, 0.99922965, 0.49321212],
                             [- 6.27302523, -0.00047761, 0.00029262, 7.27254762, 0.99854096, 0.99922977, 0.49321212],
                             [- 6.27302540, -0.00047753, 0.00029257, 7.27254787, 0.99854119, 0.99922989, 0.49321221],
                             [- 6.27302558, -0.00047746, 0.00029253, 7.27254812, 0.99854141, 0.99923001, 0.49321218],
                             [- 6.27302575, -0.00047739, 0.00029248, 7.27254837, 0.99854164, 0.99923013, 0.49321223],
                             [- 6.27302593, -0.00047731, 0.00029244, 7.27254862, 0.99854186, 0.99923025, 0.49321227],
                             [- 6.27302611, -0.00047724, 0.00029239, 7.27254887, 0.99854209, 0.99923037, 0.49321227],
                             [- 6.27302628, -0.00047716, 0.00029235, 7.27254912, 0.99854231, 0.99923049, 0.49321226],
                             [- 6.27302646, -0.00047709, 0.00029230, 7.27254937, 0.99854254, 0.99923060, 0.49321229],
                             [- 6.27302664, -0.00047702, 0.00029226, 7.27254962, 0.99854276, 0.99923072, 0.49321232],
                             [- 6.27302681, -0.00047694, 0.00029221, 7.27254987, 0.99854299, 0.99923084, 0.49321239],
                             [- 6.27302699, -0.00047687, 0.00029217, 7.27255012, 0.99854321, 0.99923096, 0.49321240],
                             [- 6.27302716, -0.00047680, 0.00029212, 7.27255037, 0.99854344, 0.99923108, 0.49321246],
                             [- 6.27302734, -0.00047672, 0.00029208, 7.27255061, 0.99854366, 0.99923120, 0.49321244],
                             [- 6.27302751, -0.00047665, 0.00029203, 7.27255086, 0.99854388, 0.99923132, 0.49321242],
                             [- 6.27302769, -0.00047658, 0.00029199, 7.27255111, 0.99854411, 0.99923143, 0.49321242],
                             [- 6.27302787, -0.00047650, 0.00029194, 7.27255136, 0.99854433, 0.99923155, 0.49321258],
                             [- 6.27302804, -0.00047643, 0.00029190, 7.27255161, 0.99854456, 0.99923167, 0.49321251],
                             [- 6.27302822, -0.00047636, 0.00029185, 7.27255186, 0.99854478, 0.99923179, 0.49321261],
                             [- 6.27302839, -0.00047628, 0.00029181, 7.27255211, 0.99854501, 0.99923191, 0.49321264],
                             [- 6.27302857, -0.00047621, 0.00029176, 7.27255236, 0.99854523, 0.99923203, 0.49321265],
                             [- 6.27302874, -0.00047614, 0.00029172, 7.27255261, 0.99854545, 0.99923214, 0.49321268],
                             [- 6.27302892, -0.00047606, 0.00029167, 7.27255285, 0.99854568, 0.99923226, 0.49321269],
                             [- 6.27302909, -0.00047599, 0.00029163, 7.27255310, 0.99854590, 0.99923238, 0.49321277],
                             [- 6.27302927, -0.00047592, 0.00029158, 7.27255335, 0.99854612, 0.99923250, 0.49321272],
                             [- 6.27302944, -0.00047584, 0.00029154, 7.27255360, 0.99854635, 0.99923262, 0.49321278],
                             [- 6.27302962, -0.00047577, 0.00029150, 7.27255385, 0.99854657, 0.99923273, 0.49321287],
                             [- 6.27302979, -0.00047570, 0.00029145, 7.27255410, 0.99854679, 0.99923285, 0.49321280],
                             [- 6.27302997, -0.00047562, 0.00029141, 7.27255434, 0.99854702, 0.99923297, 0.49321291],
                             [- 6.27303014, -0.00047555, 0.00029136, 7.27255459, 0.99854724, 0.99923309, 0.49321279],
                             [- 6.27303032, -0.00047548, 0.00029132, 7.27255484, 0.99854746, 0.99923321, 0.49321284],
                             [- 6.27303049, -0.00047541, 0.00029127, 7.27255509, 0.99854769, 0.99923332, 0.49321288],
                             [- 6.27303067, -0.00047533, 0.00029123, 7.27255533, 0.99854791, 0.99923344, 0.49321294],
                             [- 6.27303084, -0.00047526, 0.00029118, 7.27255558, 0.99854813, 0.99923356, 0.49321292],
                             [- 6.27303102, -0.00047519, 0.00029114, 7.27255583, 0.99854836, 0.99923368, 0.49321300],
                             [- 6.27303119, -0.00047511, 0.00029109, 7.27255608, 0.99854858, 0.99923379, 0.49321302],
                             [- 6.27303137, -0.00047504, 0.00029105, 7.27255632, 0.99854880, 0.99923391, 0.49321303],
                             [- 6.27303154, -0.00047497, 0.00029100, 7.27255657, 0.99854903, 0.99923403, 0.49321311],
                             [- 6.27303171, -0.00047490, 0.00029096, 7.27255682, 0.99854925, 0.99923415, 0.49321311],
                             [- 6.27303189, -0.00047482, 0.00029091, 7.27255707, 0.99854947, 0.99923426, 0.49321315],
                             [- 6.27303206, -0.00047475, 0.00029087, 7.27255731, 0.99854969, 0.99923438, 0.49321319],
                             [- 6.27303224, -0.00047468, 0.00029082, 7.27255756, 0.99854992, 0.99923450, 0.49321317],
                             [- 6.27303241, -0.00047460, 0.00029078, 7.27255781, 0.99855014, 0.99923462, 0.49321321],
                             [- 6.27303259, -0.00047453, 0.00029073, 7.27255805, 0.99855036, 0.99923473, 0.49321326],
                             [- 6.27303276, -0.00047446, 0.00029069, 7.27255830, 0.99855058, 0.99923485, 0.49321318],
                             [- 6.27303293, -0.00047439, 0.00029065, 7.27255855, 0.99855081, 0.99923497, 0.49321326],
                             [- 6.27303311, -0.00047431, 0.00029060, 7.27255879, 0.99855103, 0.99923509, 0.49321333],
                             [- 6.27303328, -0.00047424, 0.00029056, 7.27255904, 0.99855125, 0.99923520, 0.49321333],
                             [- 6.27303345, -0.00047417, 0.00029051, 7.27255929, 0.99855147, 0.99923532, 0.49321344],
                             [- 6.27303363, -0.00047410, 0.00029047, 7.27255953, 0.99855169, 0.99923544, 0.49321337],
                             [- 6.27303380, -0.00047402, 0.00029042, 7.27255978, 0.99855192, 0.99923555, 0.49321345],
                             [- 6.27303398, -0.00047395, 0.00029038, 7.27256003, 0.99855214, 0.99923567, 0.49321337],
                             [- 6.27303415, -0.00047388, 0.00029033, 7.27256027, 0.99855236, 0.99923579, 0.49321344],
                             [- 6.27303432, -0.00047380, 0.00029029, 7.27256052, 0.99855258, 0.99923591, 0.49321346],
                             [- 6.27303450, -0.00047373, 0.00029025, 7.27256076, 0.99855280, 0.99923602, 0.49321354],
                             [- 6.27303467, -0.00047366, 0.00029020, 7.27256101, 0.99855302, 0.99923614, 0.49321353],
                             [- 6.27303484, -0.00047359, 0.00029016, 7.27256126, 0.99855325, 0.99923626, 0.49321361],
                             [- 6.27303502, -0.00047351, 0.00029011, 7.27256150, 0.99855347, 0.99923637, 0.49321358],
                             [- 6.27303519, -0.00047344, 0.00029007, 7.27256175, 0.99855369, 0.99923649, 0.49321363],
                             [- 6.27303536, -0.00047337, 0.00029002, 7.27256199, 0.99855391, 0.99923661, 0.49321367],
                             [- 6.27303554, -0.00047330, 0.00028998, 7.27256224, 0.99855413, 0.99923672, 0.49321364],
                             [- 6.27303571, -0.00047323, 0.00028993, 7.27256248, 0.99855435, 0.99923684, 0.49321377],
                             [- 6.27303588, -0.00047315, 0.00028989, 7.27256273, 0.99855457, 0.99923696, 0.49321375],
                             [- 6.27303606, -0.00047308, 0.00028985, 7.27256298, 0.99855479, 0.99923707, 0.49321379],
                             [- 6.27303623, -0.00047301, 0.00028980, 7.27256322, 0.99855501, 0.99923719, 0.49321377],
                             [- 6.27303640, -0.00047294, 0.00028976, 7.27256347, 0.99855524, 0.99923731, 0.49321372],
                             [- 6.27303658, -0.00047286, 0.00028971, 7.27256371, 0.99855546, 0.99923742, 0.49321383],
                             [- 6.27303675, -0.00047279, 0.00028967, 7.27256396, 0.99855568, 0.99923754, 0.49321384],
                             [- 6.27303692, -0.00047272, 0.00028962, 7.27256420, 0.99855590, 0.99923766, 0.49321394],
                             [- 6.27303709, -0.00047265, 0.00028958, 7.27256445, 0.99855612, 0.99923777, 0.49321386],
                             [- 6.27303727, -0.00047258, 0.00028954, 7.27256469, 0.99855634, 0.99923789, 0.49321390],
                             [- 6.27303744, -0.00047250, 0.00028949, 7.27256494, 0.99855656, 0.99923801, 0.49321401],
                             [- 6.27303761, -0.00047243, 0.00028945, 7.27256518, 0.99855678, 0.99923812, 0.49321394],
                             [- 6.27303778, -0.00047236, 0.00028940, 7.27256543, 0.99855700, 0.99923824, 0.49321395],
                             [- 6.27303796, -0.00047229, 0.00028936, 7.27256567, 0.99855722, 0.99923835, 0.49321400],
                             [- 6.27303813, -0.00047221, 0.00028931, 7.27256591, 0.99855744, 0.99923847, 0.49321392],
                             [- 6.27303830, -0.00047214, 0.00028927, 7.27256616, 0.99855766, 0.99923859, 0.49321409],
                             [- 6.27303847, -0.00047207, 0.00028923, 7.27256640, 0.99855788, 0.99923870, 0.49321404],
                             [- 6.27303865, -0.00047200, 0.00028918, 7.27256665, 0.99855810, 0.99923882, 0.49321416],
                             [- 6.27303882, -0.00047193, 0.00028914, 7.27256689, 0.99855832, 0.99923893, 0.49321409],
                             [- 6.27303899, -0.00047185, 0.00028909, 7.27256714, 0.99855854, 0.99923905, 0.49321418],
                             [- 6.27303916, -0.00047178, 0.00028905, 7.27256738, 0.99855876, 0.99923917, 0.49321418],
                             [- 6.27303933, -0.00047171, 0.00028901, 7.27256762, 0.99855898, 0.99923928, 0.49321424],
                             [- 6.27303951, -0.00047164, 0.00028896, 7.27256787, 0.99855920, 0.99923940, 0.49321434],
                             [- 6.27303968, -0.00047157, 0.00028892, 7.27256811, 0.99855942, 0.99923951, 0.49321431],
                             [- 6.27303985, -0.00047150, 0.00028887, 7.27256835, 0.99855964, 0.99923963, 0.49321429],
                             [- 6.27304002, -0.00047142, 0.00028883, 7.27256860, 0.99855986, 0.99923975, 0.49321428],
                             [- 6.27304019, -0.00047135, 0.00028879, 7.27256884, 0.99856008, 0.99923986, 0.49321434],
                             [- 6.27304037, -0.00047128, 0.00028874, 7.27256909, 0.99856030, 0.99923998, 0.49321435],
                             [- 6.27304054, -0.00047121, 0.00028870, 7.27256933, 0.99856052, 0.99924009, 0.49321437],
                             [- 6.27304071, -0.00047114, 0.00028865, 7.27256957, 0.99856073, 0.99924021, 0.49321446],
                             [- 6.27304088, -0.00047106, 0.00028861, 7.27256982, 0.99856095, 0.99924032, 0.49321442],
                             [- 6.27304105, -0.00047099, 0.00028857, 7.27257006, 0.99856117, 0.99924044, 0.49321442],
                             [- 6.27304122, -0.00047092, 0.00028852, 7.27257030, 0.99856139, 0.99924056, 0.49321451],
                             [- 6.27304139, -0.00047085, 0.00028848, 7.27257054, 0.99856161, 0.99924067, 0.49321455],
                             [- 6.27304157, -0.00047078, 0.00028843, 7.27257079, 0.99856183, 0.99924079, 0.49321455],
                             [- 6.27304174, -0.00047071, 0.00028839, 7.27257103, 0.99856205, 0.99924090, 0.49321459],
                             [- 6.27304191, -0.00047064, 0.00028835, 7.27257127, 0.99856227, 0.99924102, 0.49321464],
                             [- 6.27304208, -0.00047056, 0.00028830, 7.27257152, 0.99856249, 0.99924113, 0.49321460],
                             [- 6.27304225, -0.00047049, 0.00028826, 7.27257176, 0.99856270, 0.99924125, 0.49321463],
                             [- 6.27304242, -0.00047042, 0.00028822, 7.27257200, 0.99856292, 0.99924136, 0.49321476],
                             [- 6.27304259, -0.00047035, 0.00028817, 7.27257224, 0.99856314, 0.99924148, 0.49321475],
                             [- 6.27304276, -0.00047028, 0.00028813, 7.27257249, 0.99856336, 0.99924159, 0.49321471],
                             [- 6.27304294, -0.00047021, 0.00028808, 7.27257273, 0.99856358, 0.99924171, 0.49321475],
                             [- 6.27304311, -0.00047013, 0.00028804, 7.27257297, 0.99856380, 0.99924183, 0.49321476],
                             [- 6.27304328, -0.00047006, 0.00028800, 7.27257321, 0.99856401, 0.99924194, 0.49321484],
                             [- 6.27304345, -0.00046999, 0.00028795, 7.27257346, 0.99856423, 0.99924206, 0.49321477],
                             [- 6.27304362, -0.00046992, 0.00028791, 7.27257370, 0.99856445, 0.99924217, 0.49321483],
                             [- 6.27304379, -0.00046985, 0.00028787, 7.27257394, 0.99856467, 0.99924229, 0.49321495],
                             [- 6.27304396, -0.00046978, 0.00028782, 7.27257418, 0.99856489, 0.99924240, 0.49321491],
                             [- 6.27304413, -0.00046971, 0.00028778, 7.27257442, 0.99856510, 0.99924252, 0.49321493],
                             [- 6.27304430, -0.00046964, 0.00028773, 7.27257467, 0.99856532, 0.99924263, 0.49321498],
                             [- 6.27304447, -0.00046956, 0.00028769, 7.27257491, 0.99856554, 0.99924275, 0.49321496],
                             [- 6.27304464, -0.00046949, 0.00028765, 7.27257515, 0.99856576, 0.99924286, 0.49321502],
                             [- 6.27304481, -0.00046942, 0.00028760, 7.27257539, 0.99856597, 0.99924298, 0.49321503],
                             [- 6.27304498, -0.00046935, 0.00028756, 7.27257563, 0.99856619, 0.99924309, 0.49321507],
                             [- 6.27304515, -0.00046928, 0.00028752, 7.27257587, 0.99856641, 0.99924320, 0.49321515],
                             [- 6.27304532, -0.00046921, 0.00028747, 7.27257611, 0.99856663, 0.99924332, 0.49321508],
                             [- 6.27304549, -0.00046914, 0.00028743, 7.27257636, 0.99856684, 0.99924343, 0.49321514],
                             [- 6.27304566, -0.00046907, 0.00028738, 7.27257660, 0.99856706, 0.99924355, 0.49321526],
                             [- 6.27304583, -0.00046900, 0.00028734, 7.27257684, 0.99856728, 0.99924366, 0.49321521],
                             [- 6.27304600, -0.00046892, 0.00028730, 7.27257708, 0.99856750, 0.99924378, 0.49321521],
                             [- 6.27304617, -0.00046885, 0.00028725, 7.27257732, 0.99856771, 0.99924389, 0.49321523],
                             [- 6.27304634, -0.00046878, 0.00028721, 7.27257756, 0.99856793, 0.99924401, 0.49321526],
                             [- 6.27304651, -0.00046871, 0.00028717, 7.27257780, 0.99856815, 0.99924412, 0.49321530],
                             [- 6.27304668, -0.00046864, 0.00028712, 7.27257804, 0.99856836, 0.99924424, 0.49321535],
                             [- 6.27304685, -0.00046857, 0.00028708, 7.27257828, 0.99856858, 0.99924435, 0.49321541],
                             [- 6.27304702, -0.00046850, 0.00028704, 7.27257852, 0.99856880, 0.99924446, 0.49321529],
                             [- 6.27304719, -0.00046843, 0.00028699, 7.27257876, 0.99856901, 0.99924458, 0.49321541],
                             [- 6.27304736, -0.00046836, 0.00028695, 7.27257901, 0.99856923, 0.99924469, 0.49321535],
                             [- 6.27304753, -0.00046829, 0.00028691, 7.27257925, 0.99856945, 0.99924481, 0.49321543],
                             [- 6.27304770, -0.00046821, 0.00028686, 7.27257949, 0.99856966, 0.99924492, 0.49321548],
                             [- 6.27304787, -0.00046814, 0.00028682, 7.27257973, 0.99856988, 0.99924504, 0.49321551],
                             [- 6.27304804, -0.00046807, 0.00028678, 7.27257997, 0.99857010, 0.99924515, 0.49321545],
                             [- 6.27304821, -0.00046800, 0.00028673, 7.27258021, 0.99857031, 0.99924526, 0.49321554],
                             [- 6.27304838, -0.00046793, 0.00028669, 7.27258045, 0.99857053, 0.99924538, 0.49321557],
                             [- 6.27304855, -0.00046786, 0.00028665, 7.27258069, 0.99857074, 0.99924549, 0.49321561],
                             [- 6.27304872, -0.00046779, 0.00028660, 7.27258093, 0.99857096, 0.99924561, 0.49321567],
                             [- 6.27304889, -0.00046772, 0.00028656, 7.27258117, 0.99857118, 0.99924572, 0.49321567],
                             [- 6.27304906, -0.00046765, 0.00028652, 7.27258141, 0.99857139, 0.99924583, 0.49321571],
                             [- 6.27304922, -0.00046758, 0.00028647, 7.27258165, 0.99857161, 0.99924595, 0.49321566],
                             [- 6.27304939, -0.00046751, 0.00028643, 7.27258189, 0.99857182, 0.99924606, 0.49321576],
                             [- 6.27304956, -0.00046744, 0.00028639, 7.27258213, 0.99857204, 0.99924618, 0.49321567],
                             [- 6.27304973, -0.00046737, 0.00028634, 7.27258236, 0.99857226, 0.99924629, 0.49321585],
                             [- 6.27304990, -0.00046730, 0.00028630, 7.27258260, 0.99857247, 0.99924640, 0.49321576],
                             [- 6.27305007, -0.00046723, 0.00028626, 7.27258284, 0.99857269, 0.99924652, 0.49321581],
                             [- 6.27305024, -0.00046715, 0.00028621, 7.27258308, 0.99857290, 0.99924663, 0.49321580],
                             [- 6.27305041, -0.00046708, 0.00028617, 7.27258332, 0.99857312, 0.99924675, 0.49321590],
                             [- 6.27305057, -0.00046701, 0.00028613, 7.27258356, 0.99857333, 0.99924686, 0.49321582],
                             [- 6.27305074, -0.00046694, 0.00028608, 7.27258380, 0.99857355, 0.99924697, 0.49321587],
                             [- 6.27305091, -0.00046687, 0.00028604, 7.27258404, 0.99857376, 0.99924709, 0.49321595],
                             [- 6.27305108, -0.00046680, 0.00028600, 7.27258428, 0.99857398, 0.99924720, 0.49321595],
                             [- 6.27305125, -0.00046673, 0.00028595, 7.27258452, 0.99857419, 0.99924731, 0.49321600],
                             [- 6.27305142, -0.00046666, 0.00028591, 7.27258476, 0.99857441, 0.99924743, 0.49321598],
                             [- 6.27305159, -0.00046659, 0.00028587, 7.27258499, 0.99857462, 0.99924754, 0.49321613],
                             [- 6.27305175, -0.00046652, 0.00028582, 7.27258523, 0.99857484, 0.99924765, 0.49321611],
                             [- 6.27305192, -0.00046645, 0.00028578, 7.27258547, 0.99857505, 0.99924777, 0.49321600],
                             [- 6.27305209, -0.00046638, 0.00028574, 7.27258571, 0.99857527, 0.99924788, 0.49321621],
                             [- 6.27305226, -0.00046631, 0.00028570, 7.27258595, 0.99857548, 0.99924799, 0.49321615],
                             [- 6.27305243, -0.00046624, 0.00028565, 7.27258619, 0.99857570, 0.99924811, 0.49321620],
                             [- 6.27305259, -0.00046617, 0.00028561, 7.27258642, 0.99857591, 0.99924822, 0.49321625],
                             [- 6.27305276, -0.00046610, 0.00028557, 7.27258666, 0.99857613, 0.99924833, 0.49321621],
                             [- 6.27305293, -0.00046603, 0.00028552, 7.27258690, 0.99857634, 0.99924845, 0.49321630],
                             [- 6.27305310, -0.00046596, 0.00028548, 7.27258714, 0.99857656, 0.99924856, 0.49321623],
                             [- 6.27305327, -0.00046589, 0.00028544, 7.27258738, 0.99857677, 0.99924867, 0.49321633],
                             [- 6.27305343, -0.00046582, 0.00028539, 7.27258761, 0.99857698, 0.99924879, 0.49321637],
                             [- 6.27305360, -0.00046575, 0.00028535, 7.27258785, 0.99857720, 0.99924890, 0.49321640],
                             [- 6.27305377, -0.00046568, 0.00028531, 7.27258809, 0.99857741, 0.99924901, 0.49321641],
                             [- 6.27305394, -0.00046561, 0.00028527, 7.27258833, 0.99857763, 0.99924913, 0.49321644],
                             [- 6.27305410, -0.00046554, 0.00028522, 7.27258857, 0.99857784, 0.99924924, 0.49321640],
                             [- 6.27305427, -0.00046547, 0.00028518, 7.27258880, 0.99857805, 0.99924935, 0.49321644],
                             [- 6.27305444, -0.00046540, 0.00028514, 7.27258904, 0.99857827, 0.99924946, 0.49321647],
                             [- 6.27305461, -0.00046533, 0.00028509, 7.27258928, 0.99857848, 0.99924958, 0.49321655],
                             [- 6.27305477, -0.00046526, 0.00028505, 7.27258952, 0.99857870, 0.99924969, 0.49321652],
                             [- 6.27305494, -0.00046519, 0.00028501, 7.27258975, 0.99857891, 0.99924980, 0.49321651],
                             [- 6.27305511, -0.00046512, 0.00028497, 7.27258999, 0.99857912, 0.99924992, 0.49321656],
                             [- 6.27305528, -0.00046505, 0.00028492, 7.27259023, 0.99857934, 0.99925003, 0.49321656],
                             [- 6.27305544, -0.00046498, 0.00028488, 7.27259046, 0.99857955, 0.99925014, 0.49321665],
                             [- 6.27305561, -0.00046491, 0.00028484, 7.27259070, 0.99857976, 0.99925025, 0.49321668],
                             [- 6.27305578, -0.00046484, 0.00028479, 7.27259094, 0.99857998, 0.99925037, 0.49321665],
                             [- 6.27305594, -0.00046477, 0.00028475, 7.27259117, 0.99858019, 0.99925048, 0.49321672],
                             [- 6.27305611, -0.00046470, 0.00028471, 7.27259141, 0.99858040, 0.99925059, 0.49321675],
                             [- 6.27305628, -0.00046463, 0.00028467, 7.27259165, 0.99858062, 0.99925070, 0.49321677],
                             [- 6.27305645, -0.00046456, 0.00028462, 7.27259188, 0.99858083, 0.99925082, 0.49321675],
                             [- 6.27305661, -0.00046449, 0.00028458, 7.27259212, 0.99858104, 0.99925093, 0.49321677],
                             [- 6.27305678, -0.00046442, 0.00028454, 7.27259236, 0.99858126, 0.99925104, 0.49321687],
                             [- 6.27305695, -0.00046435, 0.00028449, 7.27259259, 0.99858147, 0.99925115, 0.49321692],
                             [- 6.27305711, -0.00046428, 0.00028445, 7.27259283, 0.99858168, 0.99925127, 0.49321690],
                             [- 6.27305728, -0.00046421, 0.00028441, 7.27259307, 0.99858189, 0.99925138, 0.49321690],
                             [- 6.27305745, -0.00046414, 0.00028437, 7.27259330, 0.99858211, 0.99925149, 0.49321703],
                             [- 6.27305761, -0.00046407, 0.00028432, 7.27259354, 0.99858232, 0.99925160, 0.49321695],
                             [- 6.27305778, -0.00046400, 0.00028428, 7.27259377, 0.99858253, 0.99925171, 0.49321702],
                             [- 6.27305795, -0.00046393, 0.00028424, 7.27259401, 0.99858275, 0.99925183, 0.49321702],
                             [- 6.27305811, -0.00046386, 0.00028420, 7.27259425, 0.99858296, 0.99925194, 0.49321707],
                             [- 6.27305828, -0.00046380, 0.00028415, 7.27259448, 0.99858317, 0.99925205, 0.49321711],
                             [- 6.27305844, -0.00046373, 0.00028411, 7.27259472, 0.99858338, 0.99925216, 0.49321702],
                             [- 6.27305861, -0.00046366, 0.00028407, 7.27259495, 0.99858359, 0.99925227, 0.49321714],
                             [- 6.27305878, -0.00046359, 0.00028403, 7.27259519, 0.99858381, 0.99925239, 0.49321714],
                             [- 6.27305894, -0.00046352, 0.00028398, 7.27259543, 0.99858402, 0.99925250, 0.49321712],
                             [- 6.27305911, -0.00046345, 0.00028394, 7.27259566, 0.99858423, 0.99925261, 0.49321724],
                             [- 6.27305927, -0.00046338, 0.00028390, 7.27259590, 0.99858444, 0.99925272, 0.49321723],
                             [- 6.27305944, -0.00046331, 0.00028386, 7.27259613, 0.99858465, 0.99925283, 0.49321718],
                             [- 6.27305961, -0.00046324, 0.00028381, 7.27259637, 0.99858487, 0.99925295, 0.49321725],
                             [- 6.27305977, -0.00046317, 0.00028377, 7.27259660, 0.99858508, 0.99925306, 0.49321729],
                             [- 6.27305994, -0.00046310, 0.00028373, 7.27259684, 0.99858529, 0.99925317, 0.49321728],
                             [- 6.27306010, -0.00046303, 0.00028369, 7.27259707, 0.99858550, 0.99925328, 0.49321728],
                             [- 6.27306027, -0.00046296, 0.00028364, 7.27259731, 0.99858571, 0.99925339, 0.49321738],
                             [- 6.27306044, -0.00046289, 0.00028360, 7.27259754, 0.99858593, 0.99925351, 0.49321730],
                             [- 6.27306060, -0.00046282, 0.00028356, 7.27259778, 0.99858614, 0.99925362, 0.49321736],
                             [- 6.27306077, -0.00046276, 0.00028352, 7.27259801, 0.99858635, 0.99925373, 0.49321743],
                             [- 6.27306093, -0.00046269, 0.00028347, 7.27259825, 0.99858656, 0.99925384, 0.49321740],
                             [- 6.27306110, -0.00046262, 0.00028343, 7.27259848, 0.99858677, 0.99925395, 0.49321757],
                             [- 6.27306126, -0.00046255, 0.00028339, 7.27259872, 0.99858698, 0.99925406, 0.49321754],
                             [- 6.27306143, -0.00046248, 0.00028335, 7.27259895, 0.99858719, 0.99925417, 0.49321760],
                             [- 6.27306159, -0.00046241, 0.00028330, 7.27259918, 0.99858740, 0.99925429, 0.49321757],
                             [- 6.27306176, -0.00046234, 0.00028326, 7.27259942, 0.99858762, 0.99925440, 0.49321761],
                             [- 6.27306192, -0.00046227, 0.00028322, 7.27259965, 0.99858783, 0.99925451, 0.49321757],
                             [- 6.27306209, -0.00046220, 0.00028318, 7.27259989, 0.99858804, 0.99925462, 0.49321764],
                             [- 6.27306225, -0.00046213, 0.00028314, 7.27260012, 0.99858825, 0.99925473, 0.49321769],
                             [- 6.27306242, -0.00046206, 0.00028309, 7.27260036, 0.99858846, 0.99925484, 0.49321767],
                             [- 6.27306259, -0.00046200, 0.00028305, 7.27260059, 0.99858867, 0.99925495, 0.49321773],
                             [- 6.27306275, -0.00046193, 0.00028301, 7.27260082, 0.99858888, 0.99925506, 0.49321772],
                             [- 6.27306292, -0.00046186, 0.00028297, 7.27260106, 0.99858909, 0.99925518, 0.49321771],
                             [- 6.27306308, -0.00046179, 0.00028292, 7.27260129, 0.99858930, 0.99925529, 0.49321779],
                             [- 6.27306324, -0.00046172, 0.00028288, 7.27260152, 0.99858951, 0.99925540, 0.49321779],
                             [- 6.27306341, -0.00046165, 0.00028284, 7.27260176, 0.99858972, 0.99925551, 0.49321780],
                             [- 6.27306357, -0.00046158, 0.00028280, 7.27260199, 0.99858993, 0.99925562, 0.49321791],
                             [- 6.27306374, -0.00046151, 0.00028275, 7.27260223, 0.99859014, 0.99925573, 0.49321791],
                             [- 6.27306390, -0.00046144, 0.00028271, 7.27260246, 0.99859035, 0.99925584, 0.49321788],
                             [- 6.27306407, -0.00046138, 0.00028267, 7.27260269, 0.99859056, 0.99925595, 0.49321791],
                             [- 6.27306423, -0.00046131, 0.00028263, 7.27260293, 0.99859077, 0.99925606, 0.49321799],
                             [- 6.27306440, -0.00046124, 0.00028259, 7.27260316, 0.99859098, 0.99925618, 0.49321795],
                             [- 6.27306456, -0.00046117, 0.00028254, 7.27260339, 0.99859119, 0.99925629, 0.49321797],
                             [- 6.27306473, -0.00046110, 0.00028250, 7.27260363, 0.99859140, 0.99925640, 0.49321800],
                             [- 6.27306489, -0.00046103, 0.00028246, 7.27260386, 0.99859161, 0.99925651, 0.49321804],
                             [- 6.27306505, -0.00046096, 0.00028242, 7.27260409, 0.99859182, 0.99925662, 0.49321800],
                             [- 6.27306522, -0.00046090, 0.00028238, 7.27260432, 0.99859203, 0.99925673, 0.49321804],
                             [- 6.27306538, -0.00046083, 0.00028233, 7.27260456, 0.99859224, 0.99925684, 0.49321806],
                             [- 6.27306555, -0.00046076, 0.00028229, 7.27260479, 0.99859245, 0.99925695, 0.49321814],
                             [- 6.27306571, -0.00046069, 0.00028225, 7.27260502, 0.99859266, 0.99925706, 0.49321818],
                             [- 6.27306588, -0.00046062, 0.00028221, 7.27260525, 0.99859287, 0.99925717, 0.49321811],
                             [- 6.27306604, -0.00046055, 0.00028217, 7.27260549, 0.99859308, 0.99925728, 0.49321821],
                             [- 6.27306620, -0.00046048, 0.00028212, 7.27260572, 0.99859329, 0.99925739, 0.49321828],
                             [- 6.27306637, -0.00046042, 0.00028208, 7.27260595, 0.99859350, 0.99925750, 0.49321829],
                             [- 6.27306653, -0.00046035, 0.00028204, 7.27260618, 0.99859371, 0.99925761, 0.49321826],
                             [- 6.27306670, -0.00046028, 0.00028200, 7.27260642, 0.99859392, 0.99925772, 0.49321833],
                             [- 6.27306686, -0.00046021, 0.00028196, 7.27260665, 0.99859413, 0.99925783, 0.49321833],
                             [- 6.27306702, -0.00046014, 0.00028191, 7.27260688, 0.99859434, 0.99925794, 0.49321830],
                             [- 6.27306719, -0.00046007, 0.00028187, 7.27260711, 0.99859455, 0.99925805, 0.49321842],
                             [- 6.27306735, -0.00046000, 0.00028183, 7.27260735, 0.99859475, 0.99925817, 0.49321838],
                             [- 6.27306751, -0.00045994, 0.00028179, 7.27260758, 0.99859496, 0.99925828, 0.49321849],
                             [- 6.27306768, -0.00045987, 0.00028175, 7.27260781, 0.99859517, 0.99925839, 0.49321848],
                             [- 6.27306784, -0.00045980, 0.00028170, 7.27260804, 0.99859538, 0.99925850, 0.49321848],
                             [- 6.27306800, -0.00045973, 0.00028166, 7.27260827, 0.99859559, 0.99925861, 0.49321852],
                             [- 6.27306817, -0.00045966, 0.00028162, 7.27260850, 0.99859580, 0.99925872, 0.49321855],
                             [- 6.27306833, -0.00045959, 0.00028158, 7.27260874, 0.99859601, 0.99925883, 0.49321849],
                             [- 6.27306849, -0.00045953, 0.00028154, 7.27260897, 0.99859622, 0.99925894, 0.49321861],
                             [- 6.27306866, -0.00045946, 0.00028150, 7.27260920, 0.99859642, 0.99925905, 0.49321858],
                             [- 6.27306882, -0.00045939, 0.00028145, 7.27260943, 0.99859663, 0.99925916, 0.49321869],
                             [- 6.27306898, -0.00045932, 0.00028141, 7.27260966, 0.99859684, 0.99925927, 0.49321873],
                             [- 6.27306915, -0.00045925, 0.00028137, 7.27260989, 0.99859705, 0.99925938, 0.49321868],
                             [- 6.27306931, -0.00045919, 0.00028133, 7.27261012, 0.99859726, 0.99925949, 0.49321872],
                             [- 6.27306947, -0.00045912, 0.00028129, 7.27261035, 0.99859746, 0.99925960, 0.49321879],
                             [- 6.27306964, -0.00045905, 0.00028124, 7.27261059, 0.99859767, 0.99925971, 0.49321873],
                             [- 6.27306980, -0.00045898, 0.00028120, 7.27261082, 0.99859788, 0.99925982, 0.49321881],
                             [- 6.27306996, -0.00045891, 0.00028116, 7.27261105, 0.99859809, 0.99925993, 0.49321869],
                             [- 6.27307012, -0.00045885, 0.00028112, 7.27261128, 0.99859830, 0.99926003, 0.49321889],
                             [- 6.27307029, -0.00045878, 0.00028108, 7.27261151, 0.99859850, 0.99926014, 0.49321883],
                             [- 6.27307045, -0.00045871, 0.00028104, 7.27261174, 0.99859871, 0.99926025, 0.49321889],
                             [- 6.27307061, -0.00045864, 0.00028099, 7.27261197, 0.99859892, 0.99926036, 0.49321886],
                             [- 6.27307078, -0.00045857, 0.00028095, 7.27261220, 0.99859913, 0.99926047, 0.49321894],
                             [- 6.27307094, -0.00045851, 0.00028091, 7.27261243, 0.99859934, 0.99926058, 0.49321894],
                             [- 6.27307110, -0.00045844, 0.00028087, 7.27261266, 0.99859954, 0.99926069, 0.49321891],
                             [- 6.27307126, -0.00045837, 0.00028083, 7.27261289, 0.99859975, 0.99926080, 0.49321907],
                             [- 6.27307142, -0.00045830, 0.00028079, 7.27261312, 0.99859996, 0.99926091, 0.49321892],
                             [- 6.27307159, -0.00045823, 0.00028074, 7.27261335, 0.99860016, 0.99926102, 0.49321904],
                             [- 6.27307175, -0.00045817, 0.00028070, 7.27261358, 0.99860037, 0.99926113, 0.49321907],
                             [- 6.27307191, -0.00045810, 0.00028066, 7.27261381, 0.99860058, 0.99926124, 0.49321903],
                             [- 6.27307207, -0.00045803, 0.00028062, 7.27261404, 0.99860079, 0.99926135, 0.49321912],
                             [- 6.27307224, -0.00045796, 0.00028058, 7.27261427, 0.99860099, 0.99926146, 0.49321912],
                             [- 6.27307240, -0.00045790, 0.00028054, 7.27261450, 0.99860120, 0.99926157, 0.49321912],
                             [- 6.27307256, -0.00045783, 0.00028050, 7.27261473, 0.99860141, 0.99926168, 0.49321924],
                             [- 6.27307272, -0.00045776, 0.00028045, 7.27261496, 0.99860161, 0.99926179, 0.49321924],
                             [- 6.27307288, -0.00045769, 0.00028041, 7.27261519, 0.99860182, 0.99926190, 0.49321921],
                             [- 6.27307305, -0.00045762, 0.00028037, 7.27261542, 0.99860203, 0.99926200, 0.49321927],
                             [- 6.27307321, -0.00045756, 0.00028033, 7.27261565, 0.99860223, 0.99926211, 0.49321925],
                             [- 6.27307337, -0.00045749, 0.00028029, 7.27261588, 0.99860244, 0.99926222, 0.49321926],
                             [- 6.27307353, -0.00045742, 0.00028025, 7.27261611, 0.99860265, 0.99926233, 0.49321932],
                             [- 6.27307369, -0.00045735, 0.00028021, 7.27261634, 0.99860285, 0.99926244, 0.49321935],
                             [- 6.27307386, -0.00045729, 0.00028016, 7.27261657, 0.99860306, 0.99926255, 0.49321931],
                             [- 6.27307402, -0.00045722, 0.00028012, 7.27261680, 0.99860327, 0.99926266, 0.49321940],
                             [- 6.27307418, -0.00045715, 0.00028008, 7.27261703, 0.99860347, 0.99926277, 0.49321940],
                             [- 6.27307434, -0.00045708, 0.00028004, 7.27261726, 0.99860368, 0.99926288, 0.49321938],
                             [- 6.27307450, -0.00045702, 0.00028000, 7.27261749, 0.99860389, 0.99926299, 0.49321955],
                             [- 6.27307466, -0.00045695, 0.00027996, 7.27261771, 0.99860409, 0.99926309, 0.49321946],
                             [- 6.27307482, -0.00045688, 0.00027992, 7.27261794, 0.99860430, 0.99926320, 0.49321952],
                             [- 6.27307499, -0.00045681, 0.00027987, 7.27261817, 0.99860450, 0.99926331, 0.49321956],
                             [- 6.27307515, -0.00045675, 0.00027983, 7.27261840, 0.99860471, 0.99926342, 0.49321950],
                             [- 6.27307531, -0.00045668, 0.00027979, 7.27261863, 0.99860492, 0.99926353, 0.49321959],
                             [- 6.27307547, -0.00045661, 0.00027975, 7.27261886, 0.99860512, 0.99926364, 0.49321954],
                             [- 6.27307563, -0.00045654, 0.00027971, 7.27261909, 0.99860533, 0.99926375, 0.49321964],
                             [- 6.27307579, -0.00045648, 0.00027967, 7.27261931, 0.99860553, 0.99926385, 0.49321960],
                             [- 6.27307595, -0.00045641, 0.00027963, 7.27261954, 0.99860574, 0.99926396, 0.49321972],
                             [- 6.27307611, -0.00045634, 0.00027959, 7.27261977, 0.99860595, 0.99926407, 0.49321973],
                             [- 6.27307628, -0.00045628, 0.00027954, 7.27262000, 0.99860615, 0.99926418, 0.49321968],
                             [- 6.27307644, -0.00045621, 0.00027950, 7.27262023, 0.99860636, 0.99926429, 0.49321967],
                             [- 6.27307660, -0.00045614, 0.00027946, 7.27262046, 0.99860656, 0.99926440, 0.49321972],
                             [- 6.27307676, -0.00045607, 0.00027942, 7.27262068, 0.99860677, 0.99926451, 0.49321981],
                             [- 6.27307692, -0.00045601, 0.00027938, 7.27262091, 0.99860697, 0.99926461, 0.49321979],
                             [- 6.27307708, -0.00045594, 0.00027934, 7.27262114, 0.99860718, 0.99926472, 0.49321986],
                             [- 6.27307724, -0.00045587, 0.00027930, 7.27262137, 0.99860738, 0.99926483, 0.49321988],
                             [- 6.27307740, -0.00045581, 0.00027926, 7.27262160, 0.99860759, 0.99926494, 0.49321990],
                             [- 6.27307756, -0.00045574, 0.00027921, 7.27262182, 0.99860779, 0.99926505, 0.49321998],
                             [- 6.27307772, -0.00045567, 0.00027917, 7.27262205, 0.99860800, 0.99926516, 0.49322003],
                             [- 6.27307788, -0.00045560, 0.00027913, 7.27262228, 0.99860820, 0.99926526, 0.49321994],
                             [- 6.27307804, -0.00045554, 0.00027909, 7.27262251, 0.99860841, 0.99926537, 0.49322002],
                             [- 6.27307820, -0.00045547, 0.00027905, 7.27262273, 0.99860861, 0.99926548, 0.49321999],
                             [- 6.27307836, -0.00045540, 0.00027901, 7.27262296, 0.99860882, 0.99926559, 0.49322005],
                             [- 6.27307852, -0.00045534, 0.00027897, 7.27262319, 0.99860902, 0.99926570, 0.49322007],
                             [- 6.27307868, -0.00045527, 0.00027893, 7.27262342, 0.99860923, 0.99926580, 0.49322002],
                             [- 6.27307884, -0.00045520, 0.00027889, 7.27262364, 0.99860943, 0.99926591, 0.49322013],
                             [- 6.27307900, -0.00045514, 0.00027884, 7.27262387, 0.99860964, 0.99926602, 0.49322009],
                             [- 6.27307916, -0.00045507, 0.00027880, 7.27262410, 0.99860984, 0.99926613, 0.49322010],
                             [- 6.27307932, -0.00045500, 0.00027876, 7.27262432, 0.99861004, 0.99926624, 0.49322020],
                             [- 6.27307949, -0.00045493, 0.00027872, 7.27262455, 0.99861025, 0.99926634, 0.49322020],
                             [- 6.27307964, -0.00045487, 0.00027868, 7.27262478, 0.99861045, 0.99926645, 0.49322021],
                             [- 6.27307980, -0.00045480, 0.00027864, 7.27262500, 0.99861066, 0.99926656, 0.49322031],
                             [- 6.27307996, -0.00045473, 0.00027860, 7.27262523, 0.99861086, 0.99926667, 0.49322031],
                             [- 6.27308012, -0.00045467, 0.00027856, 7.27262546, 0.99861107, 0.99926677, 0.49322033],
                             [- 6.27308028, -0.00045460, 0.00027852, 7.27262568, 0.99861127, 0.99926688, 0.49322029],
                             [- 6.27308044, -0.00045453, 0.00027848, 7.27262591, 0.99861147, 0.99926699, 0.49322034],
                             [- 6.27308060, -0.00045447, 0.00027844, 7.27262614, 0.99861168, 0.99926710, 0.49322033],
                             [- 6.27308076, -0.00045440, 0.00027839, 7.27262636, 0.99861188, 0.99926721, 0.49322036],
                             [- 6.27308092, -0.00045433, 0.00027835, 7.27262659, 0.99861209, 0.99926731, 0.49322047],
                             [- 6.27308108, -0.00045427, 0.00027831, 7.27262682, 0.99861229, 0.99926742, 0.49322045],
                             [- 6.27308124, -0.00045420, 0.00027827, 7.27262704, 0.99861249, 0.99926753, 0.49322049],
                             [- 6.27308140, -0.00045413, 0.00027823, 7.27262727, 0.99861270, 0.99926764, 0.49322055],
                             [- 6.27308156, -0.00045407, 0.00027819, 7.27262749, 0.99861290, 0.99926774, 0.49322054],
                             [- 6.27308172, -0.00045400, 0.00027815, 7.27262772, 0.99861310, 0.99926785, 0.49322056],
                             [- 6.27308188, -0.00045393, 0.00027811, 7.27262795, 0.99861331, 0.99926796, 0.49322059],
                             [- 6.27308204, -0.00045387, 0.00027807, 7.27262817, 0.99861351, 0.99926807, 0.49322050],
                             [- 6.27308220, -0.00045380, 0.00027803, 7.27262840, 0.99861371, 0.99926817, 0.49322063],
                             [- 6.27308236, -0.00045373, 0.00027799, 7.27262862, 0.99861392, 0.99926828, 0.49322066],
                             [- 6.27308252, -0.00045367, 0.00027795, 7.27262885, 0.99861412, 0.99926839, 0.49322063],
                             [- 6.27308268, -0.00045360, 0.00027790, 7.27262907, 0.99861432, 0.99926849, 0.49322070],
                             [- 6.27308284, -0.00045353, 0.00027786, 7.27262930, 0.99861453, 0.99926860, 0.49322066],
                             [- 6.27308299, -0.00045347, 0.00027782, 7.27262953, 0.99861473, 0.99926871, 0.49322073],
                             [- 6.27308315, -0.00045340, 0.00027778, 7.27262975, 0.99861493, 0.99926882, 0.49322074],
                             [- 6.27308331, -0.00045334, 0.00027774, 7.27262998, 0.99861514, 0.99926892, 0.49322073],
                             [- 6.27308347, -0.00045327, 0.00027770, 7.27263020, 0.99861534, 0.99926903, 0.49322080],
                             [- 6.27308363, -0.00045320, 0.00027766, 7.27263043, 0.99861554, 0.99926914, 0.49322087],
                             [- 6.27308379, -0.00045314, 0.00027762, 7.27263065, 0.99861574, 0.99926924, 0.49322079],
                             [- 6.27308395, -0.00045307, 0.00027758, 7.27263088, 0.99861595, 0.99926935, 0.49322088],
                             [- 6.27308411, -0.00045300, 0.00027754, 7.27263110, 0.99861615, 0.99926946, 0.49322089],
                             [- 6.27308426, -0.00045294, 0.00027750, 7.27263133, 0.99861635, 0.99926956, 0.49322091],
                             [- 6.27308442, -0.00045287, 0.00027746, 7.27263155, 0.99861655, 0.99926967, 0.49322099],
                             [- 6.27308458, -0.00045281, 0.00027742, 7.27263178, 0.99861676, 0.99926978, 0.49322093],
                             [- 6.27308474, -0.00045274, 0.00027738, 7.27263200, 0.99861696, 0.99926989, 0.49322094],
                             [- 6.27308490, -0.00045267, 0.00027734, 7.27263223, 0.99861716, 0.99926999, 0.49322104],
                             [- 6.27308506, -0.00045261, 0.00027729, 7.27263245, 0.99861736, 0.99927010, 0.49322101],
                             [- 6.27308522, -0.00045254, 0.00027725, 7.27263268, 0.99861757, 0.99927021, 0.49322100],
                             [- 6.27308537, -0.00045247, 0.00027721, 7.27263290, 0.99861777, 0.99927031, 0.49322109],
                             [- 6.27308553, -0.00045241, 0.00027717, 7.27263312, 0.99861797, 0.99927042, 0.49322112],
                             [- 6.27308569, -0.00045234, 0.00027713, 7.27263335, 0.99861817, 0.99927053, 0.49322118],
                             [- 6.27308585, -0.00045228, 0.00027709, 7.27263357, 0.99861837, 0.99927063, 0.49322108],
                             [- 6.27308601, -0.00045221, 0.00027705, 7.27263380, 0.99861858, 0.99927074, 0.49322119],
                             [- 6.27308616, -0.00045214, 0.00027701, 7.27263402, 0.99861878, 0.99927085, 0.49322124],
                             [- 6.27308632, -0.00045208, 0.00027697, 7.27263424, 0.99861898, 0.99927095, 0.49322120],
                             [- 6.27308648, -0.00045201, 0.00027693, 7.27263447, 0.99861918, 0.99927106, 0.49322125],
                             [- 6.27308664, -0.00045195, 0.00027689, 7.27263469, 0.99861938, 0.99927116, 0.49322131],
                             [- 6.27308680, -0.00045188, 0.00027685, 7.27263492, 0.99861959, 0.99927127, 0.49322139],
                             [- 6.27308695, -0.00045181, 0.00027681, 7.27263514, 0.99861979, 0.99927138, 0.49322128],
                             [- 6.27308711, -0.00045175, 0.00027677, 7.27263536, 0.99861999, 0.99927148, 0.49322133],
                             [- 6.27308727, -0.00045168, 0.00027673, 7.27263559, 0.99862019, 0.99927159, 0.49322134],
                             [- 6.27308743, -0.00045162, 0.00027669, 7.27263581, 0.99862039, 0.99927170, 0.49322141],
                             [- 6.27308759, -0.00045155, 0.00027665, 7.27263604, 0.99862059, 0.99927180, 0.49322137],
                             [- 6.27308774, -0.00045148, 0.00027661, 7.27263626, 0.99862079, 0.99927191, 0.49322148],
                             [- 6.27308790, -0.00045142, 0.00027657, 7.27263648, 0.99862099, 0.99927202, 0.49322145],
                             [- 6.27308806, -0.00045135, 0.00027653, 7.27263671, 0.99862120, 0.99927212, 0.49322138],
                             [- 6.27308822, -0.00045129, 0.00027649, 7.27263693, 0.99862140, 0.99927223, 0.49322155],
                             [- 6.27308837, -0.00045122, 0.00027645, 7.27263715, 0.99862160, 0.99927233, 0.49322147],
                             [- 6.27308853, -0.00045116, 0.00027641, 7.27263737, 0.99862180, 0.99927244, 0.49322164],
                             [- 6.27308869, -0.00045109, 0.00027636, 7.27263760, 0.99862200, 0.99927255, 0.49322155],
                             [- 6.27308884, -0.00045102, 0.00027632, 7.27263782, 0.99862220, 0.99927265, 0.49322156],
                             [- 6.27308900, -0.00045096, 0.00027628, 7.27263804, 0.99862240, 0.99927276, 0.49322154],
                             [- 6.27308916, -0.00045089, 0.00027624, 7.27263827, 0.99862260, 0.99927286, 0.49322167],
                             [- 6.27308932, -0.00045083, 0.00027620, 7.27263849, 0.99862280, 0.99927297, 0.49322165],
                             [- 6.27308947, -0.00045076, 0.00027616, 7.27263871, 0.99862300, 0.99927308, 0.49322177],
                             [- 6.27308963, -0.00045070, 0.00027612, 7.27263894, 0.99862320, 0.99927318, 0.49322172],
                             [- 6.27308979, -0.00045063, 0.00027608, 7.27263916, 0.99862340, 0.99927329, 0.49322177],
                             [- 6.27308994, -0.00045056, 0.00027604, 7.27263938, 0.99862360, 0.99927339, 0.49322171],
                             [- 6.27309010, -0.00045050, 0.00027600, 7.27263960, 0.99862381, 0.99927350, 0.49322172],
                             [- 6.27309026, -0.00045043, 0.00027596, 7.27263983, 0.99862401, 0.99927360, 0.49322182],
                             [- 6.27309042, -0.00045037, 0.00027592, 7.27264005, 0.99862421, 0.99927371, 0.49322183],
                             [- 6.27309057, -0.00045030, 0.00027588, 7.27264027, 0.99862441, 0.99927382, 0.49322187],
                             [- 6.27309073, -0.00045024, 0.00027584, 7.27264049, 0.99862461, 0.99927392, 0.49322184],
                             [- 6.27309089, -0.00045017, 0.00027580, 7.27264071, 0.99862481, 0.99927403, 0.49322189],
                             [- 6.27309104, -0.00045011, 0.00027576, 7.27264094, 0.99862501, 0.99927413, 0.49322190],
                             [- 6.27309120, -0.00045004, 0.00027572, 7.27264116, 0.99862521, 0.99927424, 0.49322188],
                             [- 6.27309136, -0.00044997, 0.00027568, 7.27264138, 0.99862541, 0.99927434, 0.49322197],
                             [- 6.27309151, -0.00044991, 0.00027564, 7.27264160, 0.99862561, 0.99927445, 0.49322195],
                             [- 6.27309167, -0.00044984, 0.00027560, 7.27264182, 0.99862581, 0.99927455, 0.49322198],
                             [- 6.27309183, -0.00044978, 0.00027556, 7.27264205, 0.99862601, 0.99927466, 0.49322195],
                             [- 6.27309198, -0.00044971, 0.00027552, 7.27264227, 0.99862621, 0.99927477, 0.49322211],
                             [- 6.27309214, -0.00044965, 0.00027548, 7.27264249, 0.99862641, 0.99927487, 0.49322215],
                             [- 6.27309229, -0.00044958, 0.00027544, 7.27264271, 0.99862660, 0.99927498, 0.49322209],
                             [- 6.27309245, -0.00044952, 0.00027540, 7.27264293, 0.99862680, 0.99927508, 0.49322211],
                             [- 6.27309261, -0.00044945, 0.00027536, 7.27264315, 0.99862700, 0.99927519, 0.49322213],
                             [- 6.27309276, -0.00044939, 0.00027532, 7.27264338, 0.99862720, 0.99927529, 0.49322220],
                             [- 6.27309292, -0.00044932, 0.00027528, 7.27264360, 0.99862740, 0.99927540, 0.49322206],
                             [- 6.27309307, -0.00044926, 0.00027524, 7.27264382, 0.99862760, 0.99927550, 0.49322222],
                             [- 6.27309323, -0.00044919, 0.00027520, 7.27264404, 0.99862780, 0.99927561, 0.49322230],
                             [- 6.27309339, -0.00044913, 0.00027516, 7.27264426, 0.99862800, 0.99927571, 0.49322233],
                             [- 6.27309354, -0.00044906, 0.00027512, 7.27264448, 0.99862820, 0.99927582, 0.49322226],
                             [- 6.27309370, -0.00044900, 0.00027508, 7.27264470, 0.99862840, 0.99927592, 0.49322237],
                             [- 6.27309385, -0.00044893, 0.00027504, 7.27264492, 0.99862860, 0.99927603, 0.49322238],
                             [- 6.27309401, -0.00044887, 0.00027500, 7.27264514, 0.99862880, 0.99927613, 0.49322229],
                             [- 6.27309417, -0.00044880, 0.00027496, 7.27264537, 0.99862899, 0.99927624, 0.49322245],
                             [- 6.27309432, -0.00044874, 0.00027492, 7.27264559, 0.99862919, 0.99927634, 0.49322245],
                             [- 6.27309448, -0.00044867, 0.00027488, 7.27264581, 0.99862939, 0.99927645, 0.49322246],
                             [- 6.27309463, -0.00044861, 0.00027484, 7.27264603, 0.99862959, 0.99927655, 0.49322247],
                             [- 6.27309479, -0.00044854, 0.00027480, 7.27264625, 0.99862979, 0.99927666, 0.49322254],
                             [- 6.27309494, -0.00044848, 0.00027476, 7.27264647, 0.99862999, 0.99927676, 0.49322253],
                             [- 6.27309510, -0.00044841, 0.00027472, 7.27264669, 0.99863019, 0.99927687, 0.49322258],
                             [- 6.27309526, -0.00044835, 0.00027468, 7.27264691, 0.99863039, 0.99927697, 0.49322250],
                             [- 6.27309541, -0.00044828, 0.00027464, 7.27264713, 0.99863058, 0.99927708, 0.49322253],
                             [- 6.27309557, -0.00044822, 0.00027460, 7.27264735, 0.99863078, 0.99927718, 0.49322260],
                             [- 6.27309572, -0.00044815, 0.00027456, 7.27264757, 0.99863098, 0.99927729, 0.49322265],
                             [- 6.27309588, -0.00044809, 0.00027452, 7.27264779, 0.99863118, 0.99927739, 0.49322261],
                             [- 6.27309603, -0.00044802, 0.00027448, 7.27264801, 0.99863138, 0.99927750, 0.49322265],
                             [- 6.27309619, -0.00044796, 0.00027444, 7.27264823, 0.99863158, 0.99927760, 0.49322274],
                             [- 6.27309634, -0.00044789, 0.00027440, 7.27264845, 0.99863177, 0.99927770, 0.49322271],
                             [- 6.27309650, -0.00044783, 0.00027436, 7.27264867, 0.99863197, 0.99927781, 0.49322271],
                             [- 6.27309665, -0.00044776, 0.00027432, 7.27264889, 0.99863217, 0.99927791, 0.49322277],
                             [- 6.27309681, -0.00044770, 0.00027429, 7.27264911, 0.99863237, 0.99927802, 0.49322282],
                             [- 6.27309696, -0.00044763, 0.00027425, 7.27264933, 0.99863256, 0.99927812, 0.49322277],
                             [- 6.27309712, -0.00044757, 0.00027421, 7.27264955, 0.99863276, 0.99927823, 0.49322279],
                             [- 6.27309727, -0.00044750, 0.00027417, 7.27264977, 0.99863296, 0.99927833, 0.49322290],
                             [- 6.27309743, -0.00044744, 0.00027413, 7.27264999, 0.99863316, 0.99927844, 0.49322286],
                             [- 6.27309758, -0.00044737, 0.00027409, 7.27265021, 0.99863336, 0.99927854, 0.49322289],
                             [- 6.27309774, -0.00044731, 0.00027405, 7.27265043, 0.99863355, 0.99927864, 0.49322292],
                             [- 6.27309789, -0.00044724, 0.00027401, 7.27265065, 0.99863375, 0.99927875, 0.49322303],
                             [- 6.27309805, -0.00044718, 0.00027397, 7.27265087, 0.99863395, 0.99927885, 0.49322297],
                             [- 6.27309820, -0.00044712, 0.00027393, 7.27265108, 0.99863415, 0.99927896, 0.49322299],
                             [- 6.27309835, -0.00044705, 0.00027389, 7.27265130, 0.99863434, 0.99927906, 0.49322300],
                             [- 6.27309851, -0.00044699, 0.00027385, 7.27265152, 0.99863454, 0.99927916, 0.49322302],
                             [- 6.27309866, -0.00044692, 0.00027381, 7.27265174, 0.99863474, 0.99927927, 0.49322302],
                             [- 6.27309882, -0.00044686, 0.00027377, 7.27265196, 0.99863493, 0.99927937, 0.49322309],
                             [- 6.27309897, -0.00044679, 0.00027373, 7.27265218, 0.99863513, 0.99927948, 0.49322304],
                             [- 6.27309913, -0.00044673, 0.00027369, 7.27265240, 0.99863533, 0.99927958, 0.49322317],
                             [- 6.27309928, -0.00044666, 0.00027365, 7.27265262, 0.99863553, 0.99927968, 0.49322317],
                             [- 6.27309944, -0.00044660, 0.00027361, 7.27265284, 0.99863572, 0.99927979, 0.49322320],
                             [- 6.27309959, -0.00044653, 0.00027357, 7.27265305, 0.99863592, 0.99927989, 0.49322325],
                             [- 6.27309974, -0.00044647, 0.00027353, 7.27265327, 0.99863612, 0.99928000, 0.49322322],
                             [- 6.27309990, -0.00044641, 0.00027349, 7.27265349, 0.99863631, 0.99928010, 0.49322331],
                             [- 6.27310005, -0.00044634, 0.00027345, 7.27265371, 0.99863651, 0.99928020, 0.49322327],
                             [- 6.27310021, -0.00044628, 0.00027341, 7.27265393, 0.99863671, 0.99928031, 0.49322319],
                             [- 6.27310036, -0.00044621, 0.00027338, 7.27265415, 0.99863690, 0.99928041, 0.49322336],
                             [- 6.27310051, -0.00044615, 0.00027334, 7.27265436, 0.99863710, 0.99928052, 0.49322341],
                             [- 6.27310067, -0.00044608, 0.00027330, 7.27265458, 0.99863730, 0.99928062, 0.49322335],
                             [- 6.27310082, -0.00044602, 0.00027326, 7.27265480, 0.99863749, 0.99928072, 0.49322336],
                             [- 6.27310097, -0.00044596, 0.00027322, 7.27265502, 0.99863769, 0.99928083, 0.49322343],
                             [- 6.27310113, -0.00044589, 0.00027318, 7.27265524, 0.99863789, 0.99928093, 0.49322345],
                             [- 6.27310128, -0.00044583, 0.00027314, 7.27265546, 0.99863808, 0.99928103, 0.49322341],
                             [- 6.27310144, -0.00044576, 0.00027310, 7.27265567, 0.99863828, 0.99928114, 0.49322343],
                             [- 6.27310159, -0.00044570, 0.00027306, 7.27265589, 0.99863847, 0.99928124, 0.49322353],
                             [- 6.27310174, -0.00044563, 0.00027302, 7.27265611, 0.99863867, 0.99928134, 0.49322355],
                             [- 6.27310190, -0.00044557, 0.00027298, 7.27265633, 0.99863887, 0.99928145, 0.49322359],
                             [- 6.27310205, -0.00044551, 0.00027294, 7.27265654, 0.99863906, 0.99928155, 0.49322357],
                             [- 6.27310220, -0.00044544, 0.00027290, 7.27265676, 0.99863926, 0.99928165, 0.49322365],
                             [- 6.27310236, -0.00044538, 0.00027286, 7.27265698, 0.99863945, 0.99928176, 0.49322365],
                             [- 6.27310251, -0.00044531, 0.00027282, 7.27265720, 0.99863965, 0.99928186, 0.49322366],
                             [- 6.27310266, -0.00044525, 0.00027279, 7.27265741, 0.99863984, 0.99928196, 0.49322350],
                             [- 6.27310282, -0.00044519, 0.00027275, 7.27265763, 0.99864004, 0.99928207, 0.49322363],
                             [- 6.27310297, -0.00044512, 0.00027271, 7.27265785, 0.99864024, 0.99928217, 0.49322376],
                             [- 6.27310312, -0.00044506, 0.00027267, 7.27265807, 0.99864043, 0.99928227, 0.49322376],
                             [- 6.27310328, -0.00044499, 0.00027263, 7.27265828, 0.99864063, 0.99928238, 0.49322386],
                             [- 6.27310343, -0.00044493, 0.00027259, 7.27265850, 0.99864082, 0.99928248, 0.49322387],
                             [- 6.27310358, -0.00044487, 0.00027255, 7.27265872, 0.99864102, 0.99928258, 0.49322377],
                             [- 6.27310374, -0.00044480, 0.00027251, 7.27265893, 0.99864121, 0.99928269, 0.49322371],
                             [- 6.27310389, -0.00044474, 0.00027247, 7.27265915, 0.99864141, 0.99928279, 0.49322385],
                             [- 6.27310404, -0.00044467, 0.00027243, 7.27265937, 0.99864160, 0.99928289, 0.49322391],
                             [- 6.27310419, -0.00044461, 0.00027239, 7.27265958, 0.99864180, 0.99928300, 0.49322388],
                             [- 6.27310435, -0.00044455, 0.00027235, 7.27265980, 0.99864199, 0.99928310, 0.49322387],
                             [- 6.27310450, -0.00044448, 0.00027231, 7.27266002, 0.99864219, 0.99928320, 0.49322394],
                             [- 6.27310465, -0.00044442, 0.00027228, 7.27266023, 0.99864238, 0.99928330, 0.49322403],
                             [- 6.27310481, -0.00044436, 0.00027224, 7.27266045, 0.99864258, 0.99928341, 0.49322393],
                             [- 6.27310496, -0.00044429, 0.00027220, 7.27266067, 0.99864277, 0.99928351, 0.49322396],
                             [- 6.27310511, -0.00044423, 0.00027216, 7.27266088, 0.99864297, 0.99928361, 0.49322397],
                             [- 6.27310526, -0.00044416, 0.00027212, 7.27266110, 0.99864316, 0.99928372, 0.49322408],
                             [- 6.27310542, -0.00044410, 0.00027208, 7.27266132, 0.99864336, 0.99928382, 0.49322399],
                             [- 6.27310557, -0.00044404, 0.00027204, 7.27266153, 0.99864355, 0.99928392, 0.49322405],
                             [- 6.27310572, -0.00044397, 0.00027200, 7.27266175, 0.99864375, 0.99928402, 0.49322411],
                             [- 6.27310587, -0.00044391, 0.00027196, 7.27266196, 0.99864394, 0.99928413, 0.49322415],
                             [- 6.27310603, -0.00044385, 0.00027192, 7.27266218, 0.99864414, 0.99928423, 0.49322414],
                             [- 6.27310618, -0.00044378, 0.00027189, 7.27266240, 0.99864433, 0.99928433, 0.49322429],
                             [- 6.27310633, -0.00044372, 0.00027185, 7.27266261, 0.99864453, 0.99928444, 0.49322422],
                             [- 6.27310648, -0.00044365, 0.00027181, 7.27266283, 0.99864472, 0.99928454, 0.49322422],
                             [- 6.27310663, -0.00044359, 0.00027177, 7.27266304, 0.99864491, 0.99928464, 0.49322418],
                             [- 6.27310679, -0.00044353, 0.00027173, 7.27266326, 0.99864511, 0.99928474, 0.49322426],
                             [- 6.27310694, -0.00044346, 0.00027169, 7.27266347, 0.99864530, 0.99928485, 0.49322432],
                             [- 6.27310709, -0.00044340, 0.00027165, 7.27266369, 0.99864550, 0.99928495, 0.49322429],
                             [- 6.27310724, -0.00044334, 0.00027161, 7.27266391, 0.99864569, 0.99928505, 0.49322440],
                             [- 6.27310739, -0.00044327, 0.00027157, 7.27266412, 0.99864589, 0.99928515, 0.49322431],
                             [- 6.27310755, -0.00044321, 0.00027153, 7.27266434, 0.99864608, 0.99928526, 0.49322435],
                             [- 6.27310770, -0.00044315, 0.00027150, 7.27266455, 0.99864627, 0.99928536, 0.49322446],
                             [- 6.27310785, -0.00044308, 0.00027146, 7.27266477, 0.99864647, 0.99928546, 0.49322440],
                             [- 6.27310800, -0.00044302, 0.00027142, 7.27266498, 0.99864666, 0.99928556, 0.49322441],
                             [- 6.27310815, -0.00044296, 0.00027138, 7.27266520, 0.99864685, 0.99928566, 0.49322447],
                             [- 6.27310831, -0.00044289, 0.00027134, 7.27266541, 0.99864705, 0.99928577, 0.49322449],
                             [- 6.27310846, -0.00044283, 0.00027130, 7.27266563, 0.99864724, 0.99928587, 0.49322455],
                             [- 6.27310861, -0.00044277, 0.00027126, 7.27266584, 0.99864744, 0.99928597, 0.49322455],
                             [- 6.27310876, -0.00044270, 0.00027122, 7.27266606, 0.99864763, 0.99928607, 0.49322448],
                             [- 6.27310891, -0.00044264, 0.00027119, 7.27266627, 0.99864782, 0.99928618, 0.49322460],
                             [- 6.27310906, -0.00044258, 0.00027115, 7.27266649, 0.99864802, 0.99928628, 0.49322463],
                             [- 6.27310921, -0.00044251, 0.00027111, 7.27266670, 0.99864821, 0.99928638, 0.49322466],
                             [- 6.27310937, -0.00044245, 0.00027107, 7.27266692, 0.99864840, 0.99928648, 0.49322476],
                             [- 6.27310952, -0.00044239, 0.00027103, 7.27266713, 0.99864860, 0.99928658, 0.49322465],
                             [- 6.27310967, -0.00044232, 0.00027099, 7.27266735, 0.99864879, 0.99928669, 0.49322471],
                             [- 6.27310982, -0.00044226, 0.00027095, 7.27266756, 0.99864898, 0.99928679, 0.49322475],
                             [- 6.27310997, -0.00044220, 0.00027091, 7.27266777, 0.99864917, 0.99928689, 0.49322474],
                             [- 6.27311012, -0.00044213, 0.00027088, 7.27266799, 0.99864937, 0.99928699, 0.49322474],
                             [- 6.27311027, -0.00044207, 0.00027084, 7.27266820, 0.99864956, 0.99928709, 0.49322480],
                             [- 6.27311042, -0.00044201, 0.00027080, 7.27266842, 0.99864975, 0.99928719, 0.49322485],
                             [- 6.27311058, -0.00044194, 0.00027076, 7.27266863, 0.99864995, 0.99928730, 0.49322481],
                             [- 6.27311073, -0.00044188, 0.00027072, 7.27266884, 0.99865014, 0.99928740, 0.49322488],
                             [- 6.27311088, -0.00044182, 0.00027068, 7.27266906, 0.99865033, 0.99928750, 0.49322485],
                             [- 6.27311103, -0.00044176, 0.00027064, 7.27266927, 0.99865052, 0.99928760, 0.49322488],
                             [- 6.27311118, -0.00044169, 0.00027060, 7.27266949, 0.99865072, 0.99928770, 0.49322494],
                             [- 6.27311133, -0.00044163, 0.00027057, 7.27266970, 0.99865091, 0.99928780, 0.49322492],
                             [- 6.27311148, -0.00044157, 0.00027053, 7.27266991, 0.99865110, 0.99928791, 0.49322499],
                             [- 6.27311163, -0.00044150, 0.00027049, 7.27267013, 0.99865129, 0.99928801, 0.49322490],
                             [- 6.27311178, -0.00044144, 0.00027045, 7.27267034, 0.99865149, 0.99928811, 0.49322494],
                             [- 6.27311193, -0.00044138, 0.00027041, 7.27267056, 0.99865168, 0.99928821, 0.49322503],
                             [- 6.27311208, -0.00044131, 0.00027037, 7.27267077, 0.99865187, 0.99928831, 0.49322507],
                             [- 6.27311223, -0.00044125, 0.00027033, 7.27267098, 0.99865206, 0.99928841, 0.49322506],
                             [- 6.27311238, -0.00044119, 0.00027030, 7.27267120, 0.99865226, 0.99928852, 0.49322507],
                             [- 6.27311253, -0.00044113, 0.00027026, 7.27267141, 0.99865245, 0.99928862, 0.49322511],
                             [- 6.27311269, -0.00044106, 0.00027022, 7.27267162, 0.99865264, 0.99928872, 0.49322515],
                             [- 6.27311284, -0.00044100, 0.00027018, 7.27267184, 0.99865283, 0.99928882, 0.49322525],
                             [- 6.27311299, -0.00044094, 0.00027014, 7.27267205, 0.99865302, 0.99928892, 0.49322511],
                             [- 6.27311314, -0.00044087, 0.00027010, 7.27267226, 0.99865322, 0.99928902, 0.49322518],
                             [- 6.27311329, -0.00044081, 0.00027006, 7.27267247, 0.99865341, 0.99928912, 0.49322526],
                             [- 6.27311344, -0.00044075, 0.00027003, 7.27267269, 0.99865360, 0.99928922, 0.49322521],
                             [- 6.27311359, -0.00044069, 0.00026999, 7.27267290, 0.99865379, 0.99928933, 0.49322528],
                             [- 6.27311374, -0.00044062, 0.00026995, 7.27267311, 0.99865398, 0.99928943, 0.49322529],
                             [- 6.27311389, -0.00044056, 0.00026991, 7.27267333, 0.99865418, 0.99928953, 0.49322527],
                             [- 6.27311404, -0.00044050, 0.00026987, 7.27267354, 0.99865437, 0.99928963, 0.49322539],
                             [- 6.27311419, -0.00044044, 0.00026983, 7.27267375, 0.99865456, 0.99928973, 0.49322538],
                             [- 6.27311434, -0.00044037, 0.00026980, 7.27267396, 0.99865475, 0.99928983, 0.49322547],
                             [- 6.27311449, -0.00044031, 0.00026976, 7.27267418, 0.99865494, 0.99928993, 0.49322544],
                             [- 6.27311464, -0.00044025, 0.00026972, 7.27267439, 0.99865513, 0.99929003, 0.49322545],
                             [- 6.27311479, -0.00044018, 0.00026968, 7.27267460, 0.99865532, 0.99929013, 0.49322545],
                             [- 6.27311494, -0.00044012, 0.00026964, 7.27267481, 0.99865551, 0.99929024, 0.49322537],
                             [- 6.27311509, -0.00044006, 0.00026960, 7.27267503, 0.99865571, 0.99929034, 0.49322549],
                             [- 6.27311524, -0.00044000, 0.00026957, 7.27267524, 0.99865590, 0.99929044, 0.49322552],
                             [- 6.27311539, -0.00043993, 0.00026953, 7.27267545, 0.99865609, 0.99929054, 0.49322561],
                             [- 6.27311554, -0.00043987, 0.00026949, 7.27267566, 0.99865628, 0.99929064, 0.49322551],
                             [- 6.27311569, -0.00043981, 0.00026945, 7.27267588, 0.99865647, 0.99929074, 0.49322559],
                             [- 6.27311583, -0.00043975, 0.00026941, 7.27267609, 0.99865666, 0.99929084, 0.49322564],
                             [- 6.27311598, -0.00043968, 0.00026937, 7.27267630, 0.99865685, 0.99929094, 0.49322558],
                             [- 6.27311613, -0.00043962, 0.00026934, 7.27267651, 0.99865704, 0.99929104, 0.49322567],
                             [- 6.27311628, -0.00043956, 0.00026930, 7.27267672, 0.99865723, 0.99929114, 0.49322564],
                             [- 6.27311643, -0.00043950, 0.00026926, 7.27267693, 0.99865742, 0.99929124, 0.49322573],
                             [- 6.27311658, -0.00043944, 0.00026922, 7.27267715, 0.99865762, 0.99929134, 0.49322571],
                             [- 6.27311673, -0.00043937, 0.00026918, 7.27267736, 0.99865781, 0.99929144, 0.49322569],
                             [- 6.27311688, -0.00043931, 0.00026914, 7.27267757, 0.99865800, 0.99929155, 0.49322578],
                             [- 6.27311703, -0.00043925, 0.00026911, 7.27267778, 0.99865819, 0.99929165, 0.49322578],
                             [- 6.27311718, -0.00043919, 0.00026907, 7.27267799, 0.99865838, 0.99929175, 0.49322584],
                             [- 6.27311733, -0.00043912, 0.00026903, 7.27267820, 0.99865857, 0.99929185, 0.49322586],
                             [- 6.27311748, -0.00043906, 0.00026899, 7.27267842, 0.99865876, 0.99929195, 0.49322590],
                             [- 6.27311763, -0.00043900, 0.00026895, 7.27267863, 0.99865895, 0.99929205, 0.49322588],
                             [- 6.27311777, -0.00043894, 0.00026892, 7.27267884, 0.99865914, 0.99929215, 0.49322594],
                             [- 6.27311792, -0.00043887, 0.00026888, 7.27267905, 0.99865933, 0.99929225, 0.49322584],
                             [- 6.27311807, -0.00043881, 0.00026884, 7.27267926, 0.99865952, 0.99929235, 0.49322593],
                             [- 6.27311822, -0.00043875, 0.00026880, 7.27267947, 0.99865971, 0.99929245, 0.49322593],
                             [- 6.27311837, -0.00043869, 0.00026876, 7.27267968, 0.99865990, 0.99929255, 0.49322599],
                             [- 6.27311852, -0.00043863, 0.00026872, 7.27267989, 0.99866009, 0.99929265, 0.49322601],
                             [- 6.27311867, -0.00043856, 0.00026869, 7.27268010, 0.99866028, 0.99929275, 0.49322609],
                             [- 6.27311882, -0.00043850, 0.00026865, 7.27268032, 0.99866047, 0.99929285, 0.49322605],
                             [- 6.27311897, -0.00043844, 0.00026861, 7.27268053, 0.99866066, 0.99929295, 0.49322609],
                             [- 6.27311911, -0.00043838, 0.00026857, 7.27268074, 0.99866085, 0.99929305, 0.49322609],
                             [- 6.27311926, -0.00043832, 0.00026853, 7.27268095, 0.99866104, 0.99929315, 0.49322617],
                             [- 6.27311941, -0.00043825, 0.00026850, 7.27268116, 0.99866123, 0.99929325, 0.49322622],
                             [- 6.27311956, -0.00043819, 0.00026846, 7.27268137, 0.99866142, 0.99929335, 0.49322614],
                             [- 6.27311971, -0.00043813, 0.00026842, 7.27268158, 0.99866161, 0.99929345, 0.49322631],
                             [- 6.27311986, -0.00043807, 0.00026838, 7.27268179, 0.99866180, 0.99929355, 0.49322620],
                             [- 6.27312000, -0.00043800, 0.00026834, 7.27268200, 0.99866199, 0.99929365, 0.49322616],
                             [- 6.27312015, -0.00043794, 0.00026831, 7.27268221, 0.99866217, 0.99929375, 0.49322624],
                             [- 6.27312030, -0.00043788, 0.00026827, 7.27268242, 0.99866236, 0.99929385, 0.49322623],
                             [- 6.27312045, -0.00043782, 0.00026823, 7.27268263, 0.99866255, 0.99929395, 0.49322635],
                             [- 6.27312060, -0.00043776, 0.00026819, 7.27268284, 0.99866274, 0.99929405, 0.49322632],
                             [- 6.27312075, -0.00043770, 0.00026815, 7.27268305, 0.99866293, 0.99929415, 0.49322635],
                             [- 6.27312089, -0.00043763, 0.00026812, 7.27268326, 0.99866312, 0.99929425, 0.49322639],
                             [- 6.27312104, -0.00043757, 0.00026808, 7.27268347, 0.99866331, 0.99929435, 0.49322639],
                             [- 6.27312119, -0.00043751, 0.00026804, 7.27268368, 0.99866350, 0.99929445, 0.49322638],
                             [- 6.27312134, -0.00043745, 0.00026800, 7.27268389, 0.99866369, 0.99929455, 0.49322647],
                             [- 6.27312149, -0.00043739, 0.00026796, 7.27268410, 0.99866388, 0.99929465, 0.49322642],
                             [- 6.27312163, -0.00043732, 0.00026793, 7.27268431, 0.99866407, 0.99929475, 0.49322644],
                             [- 6.27312178, -0.00043726, 0.00026789, 7.27268452, 0.99866425, 0.99929485, 0.49322646],
                             [- 6.27312193, -0.00043720, 0.00026785, 7.27268473, 0.99866444, 0.99929495, 0.49322653],
                             [- 6.27312208, -0.00043714, 0.00026781, 7.27268494, 0.99866463, 0.99929505, 0.49322663],
                             [- 6.27312223, -0.00043708, 0.00026778, 7.27268515, 0.99866482, 0.99929515, 0.49322655],
                             [- 6.27312237, -0.00043702, 0.00026774, 7.27268536, 0.99866501, 0.99929525, 0.49322658],
                             [- 6.27312252, -0.00043695, 0.00026770, 7.27268557, 0.99866520, 0.99929535, 0.49322660],
                             [- 6.27312267, -0.00043689, 0.00026766, 7.27268578, 0.99866539, 0.99929545, 0.49322666],
                             [- 6.27312282, -0.00043683, 0.00026762, 7.27268599, 0.99866557, 0.99929555, 0.49322663],
                             [- 6.27312296, -0.00043677, 0.00026759, 7.27268619, 0.99866576, 0.99929564, 0.49322672],
                             [- 6.27312311, -0.00043671, 0.00026755, 7.27268640, 0.99866595, 0.99929574, 0.49322669],
                             [- 6.27312326, -0.00043665, 0.00026751, 7.27268661, 0.99866614, 0.99929584, 0.49322667],
                             [- 6.27312341, -0.00043658, 0.00026747, 7.27268682, 0.99866633, 0.99929594, 0.49322665],
                             [- 6.27312355, -0.00043652, 0.00026744, 7.27268703, 0.99866652, 0.99929604, 0.49322678],
                             [- 6.27312370, -0.00043646, 0.00026740, 7.27268724, 0.99866670, 0.99929614, 0.49322680],
                             [- 6.27312385, -0.00043640, 0.00026736, 7.27268745, 0.99866689, 0.99929624, 0.49322677],
                             [- 6.27312399, -0.00043634, 0.00026732, 7.27268766, 0.99866708, 0.99929634, 0.49322687],
                             [- 6.27312414, -0.00043628, 0.00026728, 7.27268787, 0.99866727, 0.99929644, 0.49322695],
                             [- 6.27312429, -0.00043622, 0.00026725, 7.27268807, 0.99866746, 0.99929654, 0.49322684],
                             [- 6.27312444, -0.00043615, 0.00026721, 7.27268828, 0.99866764, 0.99929664, 0.49322695],
                             [- 6.27312458, -0.00043609, 0.00026717, 7.27268849, 0.99866783, 0.99929674, 0.49322694],
                             [- 6.27312473, -0.00043603, 0.00026713, 7.27268870, 0.99866802, 0.99929684, 0.49322694],
                             [- 6.27312488, -0.00043597, 0.00026710, 7.27268891, 0.99866821, 0.99929693, 0.49322692],
                             [- 6.27312502, -0.00043591, 0.00026706, 7.27268912, 0.99866839, 0.99929703, 0.49322694],
                             [- 6.27312517, -0.00043585, 0.00026702, 7.27268932, 0.99866858, 0.99929713, 0.49322699],
                             [- 6.27312532, -0.00043579, 0.00026698, 7.27268953, 0.99866877, 0.99929723, 0.49322697],
                             [- 6.27312546, -0.00043572, 0.00026695, 7.27268974, 0.99866896, 0.99929733, 0.49322709],
                             [- 6.27312561, -0.00043566, 0.00026691, 7.27268995, 0.99866914, 0.99929743, 0.49322698],
                             [- 6.27312576, -0.00043560, 0.00026687, 7.27269016, 0.99866933, 0.99929753, 0.49322713],
                             [- 6.27312590, -0.00043554, 0.00026683, 7.27269037, 0.99866952, 0.99929763, 0.49322708],
                             [- 6.27312605, -0.00043548, 0.00026680, 7.27269057, 0.99866971, 0.99929773, 0.49322723],
                             [- 6.27312620, -0.00043542, 0.00026676, 7.27269078, 0.99866989, 0.99929782, 0.49322717],
                             [- 6.27312634, -0.00043536, 0.00026672, 7.27269099, 0.99867008, 0.99929792, 0.49322722],
                             [- 6.27312649, -0.00043529, 0.00026668, 7.27269120, 0.99867027, 0.99929802, 0.49322714],
                             [- 6.27312664, -0.00043523, 0.00026665, 7.27269140, 0.99867045, 0.99929812, 0.49322725],
                             [- 6.27312678, -0.00043517, 0.00026661, 7.27269161, 0.99867064, 0.99929822, 0.49322739],
                             [- 6.27312693, -0.00043511, 0.00026657, 7.27269182, 0.99867083, 0.99929832, 0.49322721],
                             [- 6.27312708, -0.00043505, 0.00026653, 7.27269203, 0.99867101, 0.99929842, 0.49322728],
                             [- 6.27312722, -0.00043499, 0.00026650, 7.27269223, 0.99867120, 0.99929852, 0.49322731],
                             [- 6.27312737, -0.00043493, 0.00026646, 7.27269244, 0.99867139, 0.99929861, 0.49322735],
                             [- 6.27312752, -0.00043487, 0.00026642, 7.27269265, 0.99867158, 0.99929871, 0.49322737],
                             [- 6.27312766, -0.00043481, 0.00026638, 7.27269286, 0.99867176, 0.99929881, 0.49322735],
                             [- 6.27312781, -0.00043474, 0.00026635, 7.27269306, 0.99867195, 0.99929891, 0.49322744],
                             [- 6.27312795, -0.00043468, 0.00026631, 7.27269327, 0.99867214, 0.99929901, 0.49322745],
                             [- 6.27312810, -0.00043462, 0.00026627, 7.27269348, 0.99867232, 0.99929911, 0.49322747],
                             [- 6.27312825, -0.00043456, 0.00026623, 7.27269368, 0.99867251, 0.99929920, 0.49322750],
                             [- 6.27312839, -0.00043450, 0.00026620, 7.27269389, 0.99867269, 0.99929930, 0.49322745],
                             [- 6.27312854, -0.00043444, 0.00026616, 7.27269410, 0.99867288, 0.99929940, 0.49322761],
                             [- 6.27312868, -0.00043438, 0.00026612, 7.27269431, 0.99867307, 0.99929950, 0.49322746],
                             [- 6.27312883, -0.00043432, 0.00026608, 7.27269451, 0.99867325, 0.99929960, 0.49322757],
                             [- 6.27312898, -0.00043426, 0.00026605, 7.27269472, 0.99867344, 0.99929970, 0.49322765],
                             [- 6.27312912, -0.00043420, 0.00026601, 7.27269493, 0.99867363, 0.99929979, 0.49322763],
                             [- 6.27312927, -0.00043413, 0.00026597, 7.27269513, 0.99867381, 0.99929989, 0.49322751],
                             [- 6.27312941, -0.00043407, 0.00026593, 7.27269534, 0.99867400, 0.99929999, 0.49322758],
                             [- 6.27312956, -0.00043401, 0.00026590, 7.27269555, 0.99867418, 0.99930009, 0.49322771],
                             [- 6.27312970, -0.00043395, 0.00026586, 7.27269575, 0.99867437, 0.99930019, 0.49322770],
                             [- 6.27312985, -0.00043389, 0.00026582, 7.27269596, 0.99867456, 0.99930029, 0.49322772],
                             [- 6.27313000, -0.00043383, 0.00026579, 7.27269616, 0.99867474, 0.99930038, 0.49322775],
                             [- 6.27313014, -0.00043377, 0.00026575, 7.27269637, 0.99867493, 0.99930048, 0.49322776],
                             [- 6.27313029, -0.00043371, 0.00026571, 7.27269658, 0.99867511, 0.99930058, 0.49322779],
                             [- 6.27313043, -0.00043365, 0.00026567, 7.27269678, 0.99867530, 0.99930068, 0.49322775],
                             [- 6.27313058, -0.00043359, 0.00026564, 7.27269699, 0.99867548, 0.99930078, 0.49322782],
                             [- 6.27313072, -0.00043353, 0.00026560, 7.27269720, 0.99867567, 0.99930087, 0.49322777],
                             [- 6.27313087, -0.00043347, 0.00026556, 7.27269740, 0.99867586, 0.99930097, 0.49322785],
                             [- 6.27313101, -0.00043341, 0.00026552, 7.27269761, 0.99867604, 0.99930107, 0.49322785],
                             [- 6.27313116, -0.00043335, 0.00026549, 7.27269781, 0.99867623, 0.99930117, 0.49322792],
                             [- 6.27313130, -0.00043328, 0.00026545, 7.27269802, 0.99867641, 0.99930127, 0.49322802],
                             [- 6.27313145, -0.00043322, 0.00026541, 7.27269823, 0.99867660, 0.99930136, 0.49322800],
                             [- 6.27313159, -0.00043316, 0.00026538, 7.27269843, 0.99867678, 0.99930146, 0.49322801],
                             [- 6.27313174, -0.00043310, 0.00026534, 7.27269864, 0.99867697, 0.99930156, 0.49322793],
                             [- 6.27313188, -0.00043304, 0.00026530, 7.27269884, 0.99867715, 0.99930166, 0.49322797],
                             [- 6.27313203, -0.00043298, 0.00026526, 7.27269905, 0.99867734, 0.99930175, 0.49322808],
                             [- 6.27313217, -0.00043292, 0.00026523, 7.27269925, 0.99867752, 0.99930185, 0.49322805],
                             [- 6.27313232, -0.00043286, 0.00026519, 7.27269946, 0.99867771, 0.99930195, 0.49322811],
                             [- 6.27313246, -0.00043280, 0.00026515, 7.27269966, 0.99867789, 0.99930205, 0.49322809],
                             [- 6.27313261, -0.00043274, 0.00026512, 7.27269987, 0.99867808, 0.99930214, 0.49322806],
                             [- 6.27313275, -0.00043268, 0.00026508, 7.27270007, 0.99867826, 0.99930224, 0.49322819],
                             [- 6.27313290, -0.00043262, 0.00026504, 7.27270028, 0.99867845, 0.99930234, 0.49322820],
                             [- 6.27313304, -0.00043256, 0.00026501, 7.27270049, 0.99867863, 0.99930244, 0.49322821],
                             [- 6.27313319, -0.00043250, 0.00026497, 7.27270069, 0.99867882, 0.99930253, 0.49322826],
                             [- 6.27313333, -0.00043244, 0.00026493, 7.27270090, 0.99867900, 0.99930263, 0.49322825],
                             [- 6.27313348, -0.00043238, 0.00026489, 7.27270110, 0.99867919, 0.99930273, 0.49322828],
                             [- 6.27313362, -0.00043232, 0.00026486, 7.27270131, 0.99867937, 0.99930283, 0.49322824],
                             [- 6.27313377, -0.00043226, 0.00026482, 7.27270151, 0.99867955, 0.99930292, 0.49322829],
                             [- 6.27313391, -0.00043220, 0.00026478, 7.27270171, 0.99867974, 0.99930302, 0.49322833],
                             [- 6.27313405, -0.00043214, 0.00026475, 7.27270192, 0.99867992, 0.99930312, 0.49322832],
                             [- 6.27313420, -0.00043207, 0.00026471, 7.27270212, 0.99868011, 0.99930322, 0.49322832],
                             [- 6.27313434, -0.00043201, 0.00026467, 7.27270233, 0.99868029, 0.99930331, 0.49322841],
                             [- 6.27313449, -0.00043195, 0.00026464, 7.27270253, 0.99868048, 0.99930341, 0.49322842],
                             [- 6.27313463, -0.00043189, 0.00026460, 7.27270274, 0.99868066, 0.99930351, 0.49322850],
                             [- 6.27313478, -0.00043183, 0.00026456, 7.27270294, 0.99868084, 0.99930360, 0.49322849],
                             [- 6.27313492, -0.00043177, 0.00026452, 7.27270315, 0.99868103, 0.99930370, 0.49322832],
                             [- 6.27313506, -0.00043171, 0.00026449, 7.27270335, 0.99868121, 0.99930380, 0.49322850],
                             [- 6.27313521, -0.00043165, 0.00026445, 7.27270356, 0.99868140, 0.99930390, 0.49322849],
                             [- 6.27313535, -0.00043159, 0.00026441, 7.27270376, 0.99868158, 0.99930399, 0.49322851],
                             [- 6.27313550, -0.00043153, 0.00026438, 7.27270396, 0.99868176, 0.99930409, 0.49322842],
                             [- 6.27313564, -0.00043147, 0.00026434, 7.27270417, 0.99868195, 0.99930419, 0.49322856],
                             [- 6.27313578, -0.00043141, 0.00026430, 7.27270437, 0.99868213, 0.99930428, 0.49322852],
                             [- 6.27313593, -0.00043135, 0.00026427, 7.27270458, 0.99868232, 0.99930438, 0.49322857],
                             [- 6.27313607, -0.00043129, 0.00026423, 7.27270478, 0.99868250, 0.99930448, 0.49322857],
                             [- 6.27313622, -0.00043123, 0.00026419, 7.27270498, 0.99868268, 0.99930458, 0.49322862],
                             [- 6.27313636, -0.00043117, 0.00026416, 7.27270519, 0.99868287, 0.99930467, 0.49322867],
                             [- 6.27313650, -0.00043111, 0.00026412, 7.27270539, 0.99868305, 0.99930477, 0.49322876],
                             [- 6.27313665, -0.00043105, 0.00026408, 7.27270560, 0.99868323, 0.99930487, 0.49322877],
                             [- 6.27313679, -0.00043099, 0.00026405, 7.27270580, 0.99868342, 0.99930496, 0.49322873],
                             [- 6.27313693, -0.00043093, 0.00026401, 7.27270600, 0.99868360, 0.99930506, 0.49322870],
                             [- 6.27313708, -0.00043087, 0.00026397, 7.27270621, 0.99868378, 0.99930516, 0.49322877],
                             [- 6.27313722, -0.00043081, 0.00026394, 7.27270641, 0.99868397, 0.99930525, 0.49322879],
                             [- 6.27313737, -0.00043075, 0.00026390, 7.27270661, 0.99868415, 0.99930535, 0.49322891],
                             [- 6.27313751, -0.00043069, 0.00026386, 7.27270682, 0.99868433, 0.99930545, 0.49322887],
                             [- 6.27313765, -0.00043063, 0.00026382, 7.27270702, 0.99868452, 0.99930554, 0.49322881],
                             [- 6.27313780, -0.00043057, 0.00026379, 7.27270722, 0.99868470, 0.99930564, 0.49322892],
                             [- 6.27313794, -0.00043051, 0.00026375, 7.27270743, 0.99868488, 0.99930574, 0.49322891],
                             [- 6.27313808, -0.00043045, 0.00026371, 7.27270763, 0.99868507, 0.99930583, 0.49322891],
                             [- 6.27313823, -0.00043039, 0.00026368, 7.27270783, 0.99868525, 0.99930593, 0.49322893],
                             [- 6.27313837, -0.00043033, 0.00026364, 7.27270804, 0.99868543, 0.99930603, 0.49322886],
                             [- 6.27313851, -0.00043027, 0.00026360, 7.27270824, 0.99868561, 0.99930612, 0.49322902],
                             [- 6.27313865, -0.00043021, 0.00026357, 7.27270844, 0.99868580, 0.99930622, 0.49322900],
                             [- 6.27313880, -0.00043015, 0.00026353, 7.27270864, 0.99868598, 0.99930632, 0.49322904],
                             [- 6.27313894, -0.00043009, 0.00026349, 7.27270885, 0.99868616, 0.99930641, 0.49322903],
                             [- 6.27313908, -0.00043003, 0.00026346, 7.27270905, 0.99868634, 0.99930651, 0.49322900],
                             [- 6.27313923, -0.00042997, 0.00026342, 7.27270925, 0.99868653, 0.99930660, 0.49322904],
                             [- 6.27313937, -0.00042991, 0.00026338, 7.27270946, 0.99868671, 0.99930670, 0.49322912],
                             [- 6.27313951, -0.00042985, 0.00026335, 7.27270966, 0.99868689, 0.99930680, 0.49322917],
                             [- 6.27313966, -0.00042980, 0.00026331, 7.27270986, 0.99868707, 0.99930689, 0.49322903],
                             [- 6.27313980, -0.00042974, 0.00026328, 7.27271006, 0.99868726, 0.99930699, 0.49322919],
                             [- 6.27313994, -0.00042968, 0.00026324, 7.27271027, 0.99868744, 0.99930709, 0.49322915],
                             [- 6.27314008, -0.00042962, 0.00026320, 7.27271047, 0.99868762, 0.99930718, 0.49322921],
                             [- 6.27314023, -0.00042956, 0.00026317, 7.27271067, 0.99868780, 0.99930728, 0.49322918],
                             [- 6.27314037, -0.00042950, 0.00026313, 7.27271087, 0.99868799, 0.99930737, 0.49322930],
                             [- 6.27314051, -0.00042944, 0.00026309, 7.27271107, 0.99868817, 0.99930747, 0.49322927],
                             [- 6.27314065, -0.00042938, 0.00026306, 7.27271128, 0.99868835, 0.99930757, 0.49322931],
                             [- 6.27314080, -0.00042932, 0.00026302, 7.27271148, 0.99868853, 0.99930766, 0.49322934],
                             [- 6.27314094, -0.00042926, 0.00026298, 7.27271168, 0.99868871, 0.99930776, 0.49322928],
                             [- 6.27314108, -0.00042920, 0.00026295, 7.27271188, 0.99868890, 0.99930785, 0.49322942],
                             [- 6.27314122, -0.00042914, 0.00026291, 7.27271208, 0.99868908, 0.99930795, 0.49322942],
                             [- 6.27314137, -0.00042908, 0.00026287, 7.27271229, 0.99868926, 0.99930805, 0.49322944],
                             [- 6.27314151, -0.00042902, 0.00026284, 7.27271249, 0.99868944, 0.99930814, 0.49322951],
                             [- 6.27314165, -0.00042896, 0.00026280, 7.27271269, 0.99868962, 0.99930824, 0.49322944],
                             [- 6.27314179, -0.00042890, 0.00026276, 7.27271289, 0.99868980, 0.99930833, 0.49322951],
                             [- 6.27314194, -0.00042884, 0.00026273, 7.27271309, 0.99868999, 0.99930843, 0.49322954],
                             [- 6.27314208, -0.00042878, 0.00026269, 7.27271329, 0.99869017, 0.99930853, 0.49322961],
                             [- 6.27314222, -0.00042872, 0.00026265, 7.27271350, 0.99869035, 0.99930862, 0.49322958],
                             [- 6.27314236, -0.00042866, 0.00026262, 7.27271370, 0.99869053, 0.99930872, 0.49322958],
                             [- 6.27314250, -0.00042861, 0.00026258, 7.27271390, 0.99869071, 0.99930881, 0.49322960],
                             [- 6.27314265, -0.00042855, 0.00026255, 7.27271410, 0.99869089, 0.99930891, 0.49322954],
                             [- 6.27314279, -0.00042849, 0.00026251, 7.27271430, 0.99869107, 0.99930900, 0.49322960],
                             [- 6.27314293, -0.00042843, 0.00026247, 7.27271450, 0.99869125, 0.99930910, 0.49322964],
                             [- 6.27314307, -0.00042837, 0.00026244, 7.27271470, 0.99869144, 0.99930920, 0.49322967],
                             [- 6.27314321, -0.00042831, 0.00026240, 7.27271491, 0.99869162, 0.99930929, 0.49322963],
                             [- 6.27314336, -0.00042825, 0.00026236, 7.27271511, 0.99869180, 0.99930939, 0.49322972],
                             [- 6.27314350, -0.00042819, 0.00026233, 7.27271531, 0.99869198, 0.99930948, 0.49322962],
                             [- 6.27314364, -0.00042813, 0.00026229, 7.27271551, 0.99869216, 0.99930958, 0.49322977],
                             [- 6.27314378, -0.00042807, 0.00026226, 7.27271571, 0.99869234, 0.99930967, 0.49322982],
                             [- 6.27314392, -0.00042801, 0.00026222, 7.27271591, 0.99869252, 0.99930977, 0.49322974],
                             [- 6.27314406, -0.00042795, 0.00026218, 7.27271611, 0.99869270, 0.99930986, 0.49322983],
                             [- 6.27314421, -0.00042789, 0.00026215, 7.27271631, 0.99869288, 0.99930996, 0.49322981],
                             [- 6.27314435, -0.00042783, 0.00026211, 7.27271651, 0.99869306, 0.99931005, 0.49322987],
                             [- 6.27314449, -0.00042778, 0.00026207, 7.27271671, 0.99869325, 0.99931015, 0.49322991],
                             [- 6.27314463, -0.00042772, 0.00026204, 7.27271691, 0.99869343, 0.99931025, 0.49322985],
                             [- 6.27314477, -0.00042766, 0.00026200, 7.27271711, 0.99869361, 0.99931034, 0.49322989],
                             [- 6.27314491, -0.00042760, 0.00026197, 7.27271731, 0.99869379, 0.99931044, 0.49322995],
                             [- 6.27314505, -0.00042754, 0.00026193, 7.27271752, 0.99869397, 0.99931053, 0.49322998],
                             [- 6.27314520, -0.00042748, 0.00026189, 7.27271772, 0.99869415, 0.99931063, 0.49322992],
                             [- 6.27314534, -0.00042742, 0.00026186, 7.27271792, 0.99869433, 0.99931072, 0.49322999],
                             [- 6.27314548, -0.00042736, 0.00026182, 7.27271812, 0.99869451, 0.99931082, 0.49323014],
                             [- 6.27314562, -0.00042730, 0.00026178, 7.27271832, 0.99869469, 0.99931091, 0.49323001],
                             [- 6.27314576, -0.00042724, 0.00026175, 7.27271852, 0.99869487, 0.99931101, 0.49323007],
                             [- 6.27314590, -0.00042719, 0.00026171, 7.27271872, 0.99869505, 0.99931110, 0.49323010],
                             [- 6.27314604, -0.00042713, 0.00026168, 7.27271892, 0.99869523, 0.99931120, 0.49323018],
                             [- 6.27314618, -0.00042707, 0.00026164, 7.27271912, 0.99869541, 0.99931129, 0.49323018],
                             [- 6.27314633, -0.00042701, 0.00026160, 7.27271932, 0.99869559, 0.99931139, 0.49323016],
                             [- 6.27314647, -0.00042695, 0.00026157, 7.27271952, 0.99869577, 0.99931148, 0.49323007],
                             [- 6.27314661, -0.00042689, 0.00026153, 7.27271972, 0.99869595, 0.99931158, 0.49323022],
                             [- 6.27314675, -0.00042683, 0.00026150, 7.27271992, 0.99869613, 0.99931167, 0.49323025],
                             [- 6.27314689, -0.00042677, 0.00026146, 7.27272012, 0.99869631, 0.99931177, 0.49323030],
                             [- 6.27314703, -0.00042671, 0.00026142, 7.27272032, 0.99869649, 0.99931186, 0.49323027],
                             [- 6.27314717, -0.00042666, 0.00026139, 7.27272051, 0.99869667, 0.99931196, 0.49323031],
                             [- 6.27314731, -0.00042660, 0.00026135, 7.27272071, 0.99869685, 0.99931205, 0.49323029],
                             [- 6.27314745, -0.00042654, 0.00026132, 7.27272091, 0.99869703, 0.99931215, 0.49323033],
                             [- 6.27314759, -0.00042648, 0.00026128, 7.27272111, 0.99869721, 0.99931224, 0.49323043],
                             [- 6.27314773, -0.00042642, 0.00026124, 7.27272131, 0.99869739, 0.99931234, 0.49323035],
                             [- 6.27314787, -0.00042636, 0.00026121, 7.27272151, 0.99869757, 0.99931243, 0.49323032],
                             [- 6.27314801, -0.00042630, 0.00026117, 7.27272171, 0.99869775, 0.99931253, 0.49323043],
                             [- 6.27314815, -0.00042624, 0.00026114, 7.27272191, 0.99869792, 0.99931262, 0.49323038],
                             [- 6.27314830, -0.00042619, 0.00026110, 7.27272211, 0.99869810, 0.99931272, 0.49323055],
                             [- 6.27314844, -0.00042613, 0.00026106, 7.27272231, 0.99869828, 0.99931281, 0.49323047],
                             [- 6.27314858, -0.00042607, 0.00026103, 7.27272251, 0.99869846, 0.99931290, 0.49323039],
                             [- 6.27314872, -0.00042601, 0.00026099, 7.27272271, 0.99869864, 0.99931300, 0.49323039],
                             [- 6.27314886, -0.00042595, 0.00026096, 7.27272291, 0.99869882, 0.99931309, 0.49323057],
                             [- 6.27314900, -0.00042589, 0.00026092, 7.27272310, 0.99869900, 0.99931319, 0.49323051],
                             [- 6.27314914, -0.00042583, 0.00026088, 7.27272330, 0.99869918, 0.99931328, 0.49323056],
                             [- 6.27314928, -0.00042578, 0.00026085, 7.27272350, 0.99869936, 0.99931338, 0.49323064],
                             [- 6.27314942, -0.00042572, 0.00026081, 7.27272370, 0.99869954, 0.99931347, 0.49323065],
                             [- 6.27314956, -0.00042566, 0.00026078, 7.27272390, 0.99869972, 0.99931357, 0.49323065],
                             [- 6.27314970, -0.00042560, 0.00026074, 7.27272410, 0.99869989, 0.99931366, 0.49323070],
                             [- 6.27314984, -0.00042554, 0.00026070, 7.27272430, 0.99870007, 0.99931375, 0.49323062],
                             [- 6.27314998, -0.00042548, 0.00026067, 7.27272450, 0.99870025, 0.99931385, 0.49323063],
                             [- 6.27315012, -0.00042542, 0.00026063, 7.27272469, 0.99870043, 0.99931394, 0.49323075],
                             [- 6.27315026, -0.00042537, 0.00026060, 7.27272489, 0.99870061, 0.99931404, 0.49323074],
                             [- 6.27315040, -0.00042531, 0.00026056, 7.27272509, 0.99870079, 0.99931413, 0.49323081],
                             [- 6.27315054, -0.00042525, 0.00026052, 7.27272529, 0.99870097, 0.99931423, 0.49323082],
                             [- 6.27315068, -0.00042519, 0.00026049, 7.27272549, 0.99870115, 0.99931432, 0.49323079],
                             [- 6.27315082, -0.00042513, 0.00026045, 7.27272569, 0.99870132, 0.99931441, 0.49323072],
                             [- 6.27315096, -0.00042507, 0.00026042, 7.27272588, 0.99870150, 0.99931451, 0.49323088],
                             [- 6.27315110, -0.00042502, 0.00026038, 7.27272608, 0.99870168, 0.99931460, 0.49323096],
                             [- 6.27315124, -0.00042496, 0.00026035, 7.27272628, 0.99870186, 0.99931470, 0.49323088],
                             [- 6.27315138, -0.00042490, 0.00026031, 7.27272648, 0.99870204, 0.99931479, 0.49323085],
                             [- 6.27315152, -0.00042484, 0.00026027, 7.27272668, 0.99870222, 0.99931489, 0.49323089],
                             [- 6.27315165, -0.00042478, 0.00026024, 7.27272687, 0.99870239, 0.99931498, 0.49323085],
                             [- 6.27315179, -0.00042472, 0.00026020, 7.27272707, 0.99870257, 0.99931507, 0.49323101],
                             [- 6.27315193, -0.00042467, 0.00026017, 7.27272727, 0.99870275, 0.99931517, 0.49323094],
                             [- 6.27315207, -0.00042461, 0.00026013, 7.27272747, 0.99870293, 0.99931526, 0.49323104],
                             [- 6.27315221, -0.00042455, 0.00026010, 7.27272766, 0.99870311, 0.99931535, 0.49323100],
                             [- 6.27315235, -0.00042449, 0.00026006, 7.27272786, 0.99870328, 0.99931545, 0.49323105],
                             [- 6.27315249, -0.00042443, 0.00026002, 7.27272806, 0.99870346, 0.99931554, 0.49323112],
                             [- 6.27315263, -0.00042437, 0.00025999, 7.27272826, 0.99870364, 0.99931564, 0.49323100],
                             [- 6.27315277, -0.00042432, 0.00025995, 7.27272845, 0.99870382, 0.99931573, 0.49323104],
                             [- 6.27315291, -0.00042426, 0.00025992, 7.27272865, 0.99870399, 0.99931582, 0.49323113],
                             [- 6.27315305, -0.00042420, 0.00025988, 7.27272885, 0.99870417, 0.99931592, 0.49323100],
                             [- 6.27315319, -0.00042414, 0.00025985, 7.27272905, 0.99870435, 0.99931601, 0.49323118],
                             [- 6.27315333, -0.00042408, 0.00025981, 7.27272924, 0.99870453, 0.99931611, 0.49323122],
                             [- 6.27315347, -0.00042403, 0.00025978, 7.27272944, 0.99870470, 0.99931620, 0.49323123],
                             [- 6.27315360, -0.00042397, 0.00025974, 7.27272964, 0.99870488, 0.99931629, 0.49323124],
                             [- 6.27315374, -0.00042391, 0.00025970, 7.27272983, 0.99870506, 0.99931639, 0.49323127],
                             [- 6.27315388, -0.00042385, 0.00025967, 7.27273003, 0.99870524, 0.99931648, 0.49323133],
                             [- 6.27315402, -0.00042379, 0.00025963, 7.27273023, 0.99870541, 0.99931657, 0.49323140],
                             [- 6.27315416, -0.00042374, 0.00025960, 7.27273042, 0.99870559, 0.99931667, 0.49323133],
                             [- 6.27315430, -0.00042368, 0.00025956, 7.27273062, 0.99870577, 0.99931676, 0.49323138],
                             [- 6.27315444, -0.00042362, 0.00025953, 7.27273082, 0.99870595, 0.99931685, 0.49323140],
                             [- 6.27315458, -0.00042356, 0.00025949, 7.27273101, 0.99870612, 0.99931695, 0.49323145],
                             [- 6.27315472, -0.00042350, 0.00025946, 7.27273121, 0.99870630, 0.99931704, 0.49323139],
                             [- 6.27315485, -0.00042345, 0.00025942, 7.27273141, 0.99870648, 0.99931713, 0.49323136],
                             [- 6.27315499, -0.00042339, 0.00025938, 7.27273160, 0.99870665, 0.99931723, 0.49323144],
                             [- 6.27315513, -0.00042333, 0.00025935, 7.27273180, 0.99870683, 0.99931732, 0.49323141],
                             [- 6.27315527, -0.00042327, 0.00025931, 7.27273200, 0.99870701, 0.99931741, 0.49323140],
                             [- 6.27315541, -0.00042321, 0.00025928, 7.27273219, 0.99870719, 0.99931751, 0.49323155],
                             [- 6.27315555, -0.00042316, 0.00025924, 7.27273239, 0.99870736, 0.99931760, 0.49323157],
                             [- 6.27315569, -0.00042310, 0.00025921, 7.27273259, 0.99870754, 0.99931769, 0.49323161],
                             [- 6.27315582, -0.00042304, 0.00025917, 7.27273278, 0.99870772, 0.99931779, 0.49323157],
                             [- 6.27315596, -0.00042298, 0.00025914, 7.27273298, 0.99870789, 0.99931788, 0.49323162],
                             [- 6.27315610, -0.00042292, 0.00025910, 7.27273318, 0.99870807, 0.99931797, 0.49323164],
                             [- 6.27315624, -0.00042287, 0.00025907, 7.27273337, 0.99870825, 0.99931807, 0.49323172],
                             [- 6.27315638, -0.00042281, 0.00025903, 7.27273357, 0.99870842, 0.99931816, 0.49323163],
                             [- 6.27315652, -0.00042275, 0.00025899, 7.27273376, 0.99870860, 0.99931825, 0.49323158],
                             [- 6.27315665, -0.00042269, 0.00025896, 7.27273396, 0.99870877, 0.99931835, 0.49323168],
                             [- 6.27315679, -0.00042264, 0.00025892, 7.27273416, 0.99870895, 0.99931844, 0.49323175],
                             [- 6.27315693, -0.00042258, 0.00025889, 7.27273435, 0.99870913, 0.99931853, 0.49323172],
                             [- 6.27315707, -0.00042252, 0.00025885, 7.27273455, 0.99870930, 0.99931863, 0.49323175],
                             [- 6.27315721, -0.00042246, 0.00025882, 7.27273474, 0.99870948, 0.99931872, 0.49323179],
                             [- 6.27315734, -0.00042241, 0.00025878, 7.27273494, 0.99870966, 0.99931881, 0.49323178],
                             [- 6.27315748, -0.00042235, 0.00025875, 7.27273513, 0.99870983, 0.99931891, 0.49323183],
                             [- 6.27315762, -0.00042229, 0.00025871, 7.27273533, 0.99871001, 0.99931900, 0.49323180],
                             [- 6.27315776, -0.00042223, 0.00025868, 7.27273552, 0.99871018, 0.99931909, 0.49323181],
                             [- 6.27315790, -0.00042218, 0.00025864, 7.27273572, 0.99871036, 0.99931918, 0.49323188],
                             [- 6.27315803, -0.00042212, 0.00025861, 7.27273592, 0.99871054, 0.99931928, 0.49323197],
                             [- 6.27315817, -0.00042206, 0.00025857, 7.27273611, 0.99871071, 0.99931937, 0.49323190],
                             [- 6.27315831, -0.00042200, 0.00025854, 7.27273631, 0.99871089, 0.99931946, 0.49323185],
                             [- 6.27315845, -0.00042194, 0.00025850, 7.27273650, 0.99871106, 0.99931956, 0.49323199],
                             [- 6.27315858, -0.00042189, 0.00025846, 7.27273670, 0.99871124, 0.99931965, 0.49323204],
                             [- 6.27315872, -0.00042183, 0.00025843, 7.27273689, 0.99871141, 0.99931974, 0.49323191],
                             [- 6.27315886, -0.00042177, 0.00025839, 7.27273709, 0.99871159, 0.99931983, 0.49323202],
                             [- 6.27315900, -0.00042171, 0.00025836, 7.27273728, 0.99871177, 0.99931993, 0.49323199],
                             [- 6.27315913, -0.00042166, 0.00025832, 7.27273748, 0.99871194, 0.99932002, 0.49323200],
                             [- 6.27315927, -0.00042160, 0.00025829, 7.27273767, 0.99871212, 0.99932011, 0.49323212],
                             [- 6.27315941, -0.00042154, 0.00025825, 7.27273787, 0.99871229, 0.99932020, 0.49323215],
                             [- 6.27315955, -0.00042149, 0.00025822, 7.27273806, 0.99871247, 0.99932030, 0.49323211],
                             [- 6.27315968, -0.00042143, 0.00025818, 7.27273826, 0.99871264, 0.99932039, 0.49323217],
                             [- 6.27315982, -0.00042137, 0.00025815, 7.27273845, 0.99871282, 0.99932048, 0.49323219],
                             [- 6.27315996, -0.00042131, 0.00025811, 7.27273864, 0.99871299, 0.99932057, 0.49323223],
                             [- 6.27316010, -0.00042126, 0.00025808, 7.27273884, 0.99871317, 0.99932067, 0.49323224],
                             [- 6.27316023, -0.00042120, 0.00025804, 7.27273903, 0.99871334, 0.99932076, 0.49323221],
                             [- 6.27316037, -0.00042114, 0.00025801, 7.27273923, 0.99871352, 0.99932085, 0.49323224],
                             [- 6.27316051, -0.00042108, 0.00025797, 7.27273942, 0.99871369, 0.99932094, 0.49323220],
                             [- 6.27316064, -0.00042103, 0.00025794, 7.27273962, 0.99871387, 0.99932104, 0.49323229],
                             [- 6.27316078, -0.00042097, 0.00025790, 7.27273981, 0.99871404, 0.99932113, 0.49323228],
                             [- 6.27316092, -0.00042091, 0.00025787, 7.27274001, 0.99871422, 0.99932122, 0.49323228],
                             [- 6.27316105, -0.00042085, 0.00025783, 7.27274020, 0.99871439, 0.99932131, 0.49323234],
                             [- 6.27316119, -0.00042080, 0.00025780, 7.27274039, 0.99871457, 0.99932141, 0.49323243],
                             [- 6.27316133, -0.00042074, 0.00025776, 7.27274059, 0.99871474, 0.99932150, 0.49323241],
                             [- 6.27316147, -0.00042068, 0.00025773, 7.27274078, 0.99871492, 0.99932159, 0.49323242],
                             [- 6.27316160, -0.00042063, 0.00025769, 7.27274098, 0.99871509, 0.99932168, 0.49323234],
                             [- 6.27316174, -0.00042057, 0.00025766, 7.27274117, 0.99871527, 0.99932177, 0.49323243],
                             [- 6.27316188, -0.00042051, 0.00025762, 7.27274136, 0.99871544, 0.99932187, 0.49323249],
                             [- 6.27316201, -0.00042045, 0.00025759, 7.27274156, 0.99871562, 0.99932196, 0.49323249],
                             [- 6.27316215, -0.00042040, 0.00025755, 7.27274175, 0.99871579, 0.99932205, 0.49323255],
                             [- 6.27316229, -0.00042034, 0.00025752, 7.27274195, 0.99871597, 0.99932214, 0.49323261],
                             [- 6.27316242, -0.00042028, 0.00025748, 7.27274214, 0.99871614, 0.99932223, 0.49323266],
                             [- 6.27316256, -0.00042023, 0.00025745, 7.27274233, 0.99871631, 0.99932233, 0.49323261],
                             [- 6.27316270, -0.00042017, 0.00025741, 7.27274253, 0.99871649, 0.99932242, 0.49323265],
                             [- 6.27316283, -0.00042011, 0.00025738, 7.27274272, 0.99871666, 0.99932251, 0.49323261],
                             [- 6.27316297, -0.00042006, 0.00025734, 7.27274291, 0.99871684, 0.99932260, 0.49323268],
                             [- 6.27316311, -0.00042000, 0.00025731, 7.27274311, 0.99871701, 0.99932269, 0.49323261],
                             [- 6.27316324, -0.00041994, 0.00025727, 7.27274330, 0.99871719, 0.99932279, 0.49323277],
                             [- 6.27316338, -0.00041988, 0.00025724, 7.27274349, 0.99871736, 0.99932288, 0.49323278],
                             [- 6.27316351, -0.00041983, 0.00025720, 7.27274369, 0.99871753, 0.99932297, 0.49323270],
                             [- 6.27316365, -0.00041977, 0.00025717, 7.27274388, 0.99871771, 0.99932306, 0.49323262],
                             [- 6.27316379, -0.00041971, 0.00025713, 7.27274407, 0.99871788, 0.99932315, 0.49323267],
                             [- 6.27316392, -0.00041966, 0.00025710, 7.27274427, 0.99871806, 0.99932325, 0.49323282],
                             [- 6.27316406, -0.00041960, 0.00025706, 7.27274446, 0.99871823, 0.99932334, 0.49323274],
                             [- 6.27316420, -0.00041954, 0.00025703, 7.27274465, 0.99871840, 0.99932343, 0.49323276],
                             [- 6.27316433, -0.00041949, 0.00025699, 7.27274485, 0.99871858, 0.99932352, 0.49323279],
                             [- 6.27316447, -0.00041943, 0.00025696, 7.27274504, 0.99871875, 0.99932361, 0.49323285],
                             [- 6.27316460, -0.00041937, 0.00025692, 7.27274523, 0.99871892, 0.99932370, 0.49323286],
                             [- 6.27316474, -0.00041932, 0.00025689, 7.27274542, 0.99871910, 0.99932380, 0.49323296],
                             [- 6.27316488, -0.00041926, 0.00025685, 7.27274562, 0.99871927, 0.99932389, 0.49323294],
                             [- 6.27316501, -0.00041920, 0.00025682, 7.27274581, 0.99871944, 0.99932398, 0.49323292],
                             [- 6.27316515, -0.00041915, 0.00025678, 7.27274600, 0.99871962, 0.99932407, 0.49323285],
                             [- 6.27316528, -0.00041909, 0.00025675, 7.27274619, 0.99871979, 0.99932416, 0.49323292],
                             [- 6.27316542, -0.00041903, 0.00025671, 7.27274639, 0.99871996, 0.99932425, 0.49323303],
                             [- 6.27316555, -0.00041898, 0.00025668, 7.27274658, 0.99872014, 0.99932435, 0.49323312],
                             [- 6.27316569, -0.00041892, 0.00025664, 7.27274677, 0.99872031, 0.99932444, 0.49323326],
                             [- 6.27316583, -0.00041886, 0.00025661, 7.27274696, 0.99872048, 0.99932453, 0.49323308],
                             [- 6.27316596, -0.00041881, 0.00025658, 7.27274716, 0.99872066, 0.99932462, 0.49323307],
                             [- 6.27316610, -0.00041875, 0.00025654, 7.27274735, 0.99872083, 0.99932471, 0.49323314],
                             [- 6.27316623, -0.00041869, 0.00025651, 7.27274754, 0.99872100, 0.99932480, 0.49323312],
                             [- 6.27316637, -0.00041864, 0.00025647, 7.27274773, 0.99872118, 0.99932489, 0.49323320],
                             [- 6.27316650, -0.00041858, 0.00025644, 7.27274793, 0.99872135, 0.99932498, 0.49323311],
                             [- 6.27316664, -0.00041852, 0.00025640, 7.27274812, 0.99872152, 0.99932508, 0.49323311],
                             [- 6.27316677, -0.00041847, 0.00025637, 7.27274831, 0.99872170, 0.99932517, 0.49323317],
                             [- 6.27316691, -0.00041841, 0.00025633, 7.27274850, 0.99872187, 0.99932526, 0.49323323],
                             [- 6.27316705, -0.00041835, 0.00025630, 7.27274869, 0.99872204, 0.99932535, 0.49323325],
                             [- 6.27316718, -0.00041830, 0.00025626, 7.27274888, 0.99872221, 0.99932544, 0.49323334],
                             [- 6.27316732, -0.00041824, 0.00025623, 7.27274908, 0.99872239, 0.99932553, 0.49323332],
                             [- 6.27316745, -0.00041818, 0.00025619, 7.27274927, 0.99872256, 0.99932562, 0.49323331],
                             [- 6.27316759, -0.00041813, 0.00025616, 7.27274946, 0.99872273, 0.99932571, 0.49323331],
                             [- 6.27316772, -0.00041807, 0.00025612, 7.27274965, 0.99872290, 0.99932581, 0.49323346],
                             [- 6.27316786, -0.00041801, 0.00025609, 7.27274984, 0.99872308, 0.99932590, 0.49323331],
                             [- 6.27316799, -0.00041796, 0.00025606, 7.27275003, 0.99872325, 0.99932599, 0.49323339],
                             [- 6.27316813, -0.00041790, 0.00025602, 7.27275023, 0.99872342, 0.99932608, 0.49323343],
                             [- 6.27316826, -0.00041784, 0.00025599, 7.27275042, 0.99872359, 0.99932617, 0.49323337],
                             [- 6.27316840, -0.00041779, 0.00025595, 7.27275061, 0.99872377, 0.99932626, 0.49323342],
                             [- 6.27316853, -0.00041773, 0.00025592, 7.27275080, 0.99872394, 0.99932635, 0.49323347],
                             [- 6.27316867, -0.00041768, 0.00025588, 7.27275099, 0.99872411, 0.99932644, 0.49323353],
                             [- 6.27316880, -0.00041762, 0.00025585, 7.27275118, 0.99872428, 0.99932653, 0.49323349],
                             [- 6.27316894, -0.00041756, 0.00025581, 7.27275137, 0.99872446, 0.99932662, 0.49323358],
                             [- 6.27316907, -0.00041751, 0.00025578, 7.27275157, 0.99872463, 0.99932671, 0.49323354],
                             [- 6.27316921, -0.00041745, 0.00025574, 7.27275176, 0.99872480, 0.99932681, 0.49323350],
                             [- 6.27316934, -0.00041739, 0.00025571, 7.27275195, 0.99872497, 0.99932690, 0.49323356],
                             [- 6.27316948, -0.00041734, 0.00025568, 7.27275214, 0.99872514, 0.99932699, 0.49323355],
                             [- 6.27316961, -0.00041728, 0.00025564, 7.27275233, 0.99872532, 0.99932708, 0.49323359],
                             [- 6.27316974, -0.00041722, 0.00025561, 7.27275252, 0.99872549, 0.99932717, 0.49323360],
                             [- 6.27316988, -0.00041717, 0.00025557, 7.27275271, 0.99872566, 0.99932726, 0.49323362],
                             [- 6.27317001, -0.00041711, 0.00025554, 7.27275290, 0.99872583, 0.99932735, 0.49323369],
                             [- 6.27317015, -0.00041706, 0.00025550, 7.27275309, 0.99872600, 0.99932744, 0.49323369],
                             [- 6.27317028, -0.00041700, 0.00025547, 7.27275328, 0.99872617, 0.99932753, 0.49323368],
                             [- 6.27317042, -0.00041694, 0.00025543, 7.27275347, 0.99872635, 0.99932762, 0.49323375],
                             [- 6.27317055, -0.00041689, 0.00025540, 7.27275366, 0.99872652, 0.99932771, 0.49323378],
                             [- 6.27317069, -0.00041683, 0.00025537, 7.27275385, 0.99872669, 0.99932780, 0.49323373],
                             [- 6.27317082, -0.00041678, 0.00025533, 7.27275405, 0.99872686, 0.99932789, 0.49323370],
                             [- 6.27317095, -0.00041672, 0.00025530, 7.27275424, 0.99872703, 0.99932798, 0.49323379],
                             [- 6.27317109, -0.00041666, 0.00025526, 7.27275443, 0.99872720, 0.99932807, 0.49323375],
                             [- 6.27317122, -0.00041661, 0.00025523, 7.27275462, 0.99872738, 0.99932816, 0.49323392],
                             [- 6.27317136, -0.00041655, 0.00025519, 7.27275481, 0.99872755, 0.99932826, 0.49323391],
                             [- 6.27317149, -0.00041649, 0.00025516, 7.27275500, 0.99872772, 0.99932835, 0.49323391],
                             [- 6.27317163, -0.00041644, 0.00025512, 7.27275519, 0.99872789, 0.99932844, 0.49323388],
                             [- 6.27317176, -0.00041638, 0.00025509, 7.27275538, 0.99872806, 0.99932853, 0.49323395],
                             [- 6.27317189, -0.00041633, 0.00025506, 7.27275557, 0.99872823, 0.99932862, 0.49323385],
                             [- 6.27317203, -0.00041627, 0.00025502, 7.27275576, 0.99872840, 0.99932871, 0.49323392],
                             [- 6.27317216, -0.00041621, 0.00025499, 7.27275595, 0.99872857, 0.99932880, 0.49323402],
                             [- 6.27317230, -0.00041616, 0.00025495, 7.27275614, 0.99872874, 0.99932889, 0.49323405],
                             [- 6.27317243, -0.00041610, 0.00025492, 7.27275633, 0.99872892, 0.99932898, 0.49323405],
                             [- 6.27317256, -0.00041605, 0.00025488, 7.27275652, 0.99872909, 0.99932907, 0.49323408],
                             [- 6.27317270, -0.00041599, 0.00025485, 7.27275671, 0.99872926, 0.99932916, 0.49323410],
                             [- 6.27317283, -0.00041594, 0.00025482, 7.27275690, 0.99872943, 0.99932925, 0.49323410],
                             [- 6.27317297, -0.00041588, 0.00025478, 7.27275709, 0.99872960, 0.99932934, 0.49323416],
                             [- 6.27317310, -0.00041582, 0.00025475, 7.27275728, 0.99872977, 0.99932943, 0.49323413],
                             [- 6.27317323, -0.00041577, 0.00025471, 7.27275746, 0.99872994, 0.99932952, 0.49323409],
                             [- 6.27317337, -0.00041571, 0.00025468, 7.27275765, 0.99873011, 0.99932961, 0.49323425],
                             [- 6.27317350, -0.00041566, 0.00025465, 7.27275784, 0.99873028, 0.99932970, 0.49323426],
                             [- 6.27317363, -0.00041560, 0.00025461, 7.27275803, 0.99873045, 0.99932979, 0.49323421],
                             [- 6.27317377, -0.00041554, 0.00025458, 7.27275822, 0.99873062, 0.99932988, 0.49323430],
                             [- 6.27317390, -0.00041549, 0.00025454, 7.27275841, 0.99873079, 0.99932997, 0.49323422],
                             [- 6.27317403, -0.00041543, 0.00025451, 7.27275860, 0.99873096, 0.99933006, 0.49323429],
                             [- 6.27317417, -0.00041538, 0.00025447, 7.27275879, 0.99873113, 0.99933015, 0.49323420],
                             [- 6.27317430, -0.00041532, 0.00025444, 7.27275898, 0.99873130, 0.99933024, 0.49323427],
                             [- 6.27317443, -0.00041527, 0.00025441, 7.27275917, 0.99873147, 0.99933033, 0.49323431],
                             [- 6.27317457, -0.00041521, 0.00025437, 7.27275936, 0.99873164, 0.99933042, 0.49323443],
                             [- 6.27317470, -0.00041515, 0.00025434, 7.27275955, 0.99873181, 0.99933051, 0.49323428],
                             [- 6.27317483, -0.00041510, 0.00025430, 7.27275974, 0.99873198, 0.99933060, 0.49323443],
                             [- 6.27317497, -0.00041504, 0.00025427, 7.27275992, 0.99873215, 0.99933069, 0.49323437],
                             [- 6.27317510, -0.00041499, 0.00025424, 7.27276011, 0.99873232, 0.99933078, 0.49323454],
                             [- 6.27317523, -0.00041493, 0.00025420, 7.27276030, 0.99873249, 0.99933087, 0.49323446],
                             [- 6.27317537, -0.00041488, 0.00025417, 7.27276049, 0.99873266, 0.99933096, 0.49323436],
                             [- 6.27317550, -0.00041482, 0.00025413, 7.27276068, 0.99873283, 0.99933105, 0.49323455],
                             [- 6.27317563, -0.00041476, 0.00025410, 7.27276087, 0.99873300, 0.99933114, 0.49323443],
                             [- 6.27317577, -0.00041471, 0.00025406, 7.27276106, 0.99873317, 0.99933123, 0.49323444],
                             [- 6.27317590, -0.00041465, 0.00025403, 7.27276124, 0.99873334, 0.99933132, 0.49323452],
                             [- 6.27317603, -0.00041460, 0.00025400, 7.27276143, 0.99873351, 0.99933140, 0.49323465],
                             [- 6.27317616, -0.00041454, 0.00025396, 7.27276162, 0.99873368, 0.99933149, 0.49323452],
                             [- 6.27317630, -0.00041449, 0.00025393, 7.27276181, 0.99873385, 0.99933158, 0.49323459],
                             [- 6.27317643, -0.00041443, 0.00025389, 7.27276200, 0.99873402, 0.99933167, 0.49323462],
                             [- 6.27317656, -0.00041438, 0.00025386, 7.27276219, 0.99873419, 0.99933176, 0.49323457],
                             [- 6.27317670, -0.00041432, 0.00025383, 7.27276237, 0.99873436, 0.99933185, 0.49323468],
                             [- 6.27317683, -0.00041427, 0.00025379, 7.27276256, 0.99873453, 0.99933194, 0.49323479],
                             [- 6.27317696, -0.00041421, 0.00025376, 7.27276275, 0.99873470, 0.99933203, 0.49323481],
                             [- 6.27317709, -0.00041415, 0.00025372, 7.27276294, 0.99873487, 0.99933212, 0.49323465],
                             [- 6.27317723, -0.00041410, 0.00025369, 7.27276313, 0.99873504, 0.99933221, 0.49323467],
                             [- 6.27317736, -0.00041404, 0.00025366, 7.27276332, 0.99873521, 0.99933230, 0.49323474],
                             [- 6.27317749, -0.00041399, 0.00025362, 7.27276350, 0.99873538, 0.99933239, 0.49323475],
                             [- 6.27317762, -0.00041393, 0.00025359, 7.27276369, 0.99873555, 0.99933248, 0.49323489],
                             [- 6.27317776, -0.00041388, 0.00025355, 7.27276388, 0.99873572, 0.99933257, 0.49323478],
                             [- 6.27317789, -0.00041382, 0.00025352, 7.27276407, 0.99873588, 0.99933266, 0.49323490],
                             [- 6.27317802, -0.00041377, 0.00025349, 7.27276425, 0.99873605, 0.99933275, 0.49323500],
                             [- 6.27317815, -0.00041371, 0.00025345, 7.27276444, 0.99873622, 0.99933283, 0.49323486],
                             [- 6.27317829, -0.00041366, 0.00025342, 7.27276463, 0.99873639, 0.99933292, 0.49323491],
                             [- 6.27317842, -0.00041360, 0.00025339, 7.27276482, 0.99873656, 0.99933301, 0.49323473],
                             [- 6.27317855, -0.00041355, 0.00025335, 7.27276500, 0.99873673, 0.99933310, 0.49323495],
                             [- 6.27317868, -0.00041349, 0.00025332, 7.27276519, 0.99873690, 0.99933319, 0.49323495],
                             [- 6.27317882, -0.00041344, 0.00025328, 7.27276538, 0.99873707, 0.99933328, 0.49323487],
                             [- 6.27317895, -0.00041338, 0.00025325, 7.27276557, 0.99873724, 0.99933337, 0.49323500],
                             [- 6.27317908, -0.00041333, 0.00025322, 7.27276575, 0.99873740, 0.99933346, 0.49323493],
                             [- 6.27317921, -0.00041327, 0.00025318, 7.27276594, 0.99873757, 0.99933355, 0.49323497],
                             [- 6.27317934, -0.00041321, 0.00025315, 7.27276613, 0.99873774, 0.99933364, 0.49323507],
                             [- 6.27317948, -0.00041316, 0.00025311, 7.27276632, 0.99873791, 0.99933373, 0.49323512],
                             [- 6.27317961, -0.00041310, 0.00025308, 7.27276650, 0.99873808, 0.99933381, 0.49323500],
                             [- 6.27317974, -0.00041305, 0.00025305, 7.27276669, 0.99873825, 0.99933390, 0.49323510],
                             [- 6.27317987, -0.00041299, 0.00025301, 7.27276688, 0.99873842, 0.99933399, 0.49323520],
                             [- 6.27318000, -0.00041294, 0.00025298, 7.27276706, 0.99873858, 0.99933408, 0.49323515],
                             [- 6.27318014, -0.00041288, 0.00025295, 7.27276725, 0.99873875, 0.99933417, 0.49323523],
                             [- 6.27318027, -0.00041283, 0.00025291, 7.27276744, 0.99873892, 0.99933426, 0.49323516],
                             [- 6.27318040, -0.00041277, 0.00025288, 7.27276762, 0.99873909, 0.99933435, 0.49323517],
                             [- 6.27318053, -0.00041272, 0.00025284, 7.27276781, 0.99873926, 0.99933444, 0.49323519],
                             [- 6.27318066, -0.00041266, 0.00025281, 7.27276800, 0.99873942, 0.99933453, 0.49323515],
                             [- 6.27318079, -0.00041261, 0.00025278, 7.27276818, 0.99873959, 0.99933461, 0.49323538],
                             [- 6.27318093, -0.00041255, 0.00025274, 7.27276837, 0.99873976, 0.99933470, 0.49323529],
                             [- 6.27318106, -0.00041250, 0.00025271, 7.27276856, 0.99873993, 0.99933479, 0.49323537],
                             [- 6.27318119, -0.00041244, 0.00025268, 7.27276874, 0.99874010, 0.99933488, 0.49323534],
                             [- 6.27318132, -0.00041239, 0.00025264, 7.27276893, 0.99874026, 0.99933497, 0.49323537],
                             [- 6.27318145, -0.00041233, 0.00025261, 7.27276912, 0.99874043, 0.99933506, 0.49323540],
                             [- 6.27318158, -0.00041228, 0.00025258, 7.27276930, 0.99874060, 0.99933515, 0.49323539],
                             [- 6.27318171, -0.00041222, 0.00025254, 7.27276949, 0.99874077, 0.99933523, 0.49323536],
                             [- 6.27318185, -0.00041217, 0.00025251, 7.27276968, 0.99874094, 0.99933532, 0.49323541],
                             [- 6.27318198, -0.00041211, 0.00025247, 7.27276986, 0.99874110, 0.99933541, 0.49323529],
                             [- 6.27318211, -0.00041206, 0.00025244, 7.27277005, 0.99874127, 0.99933550, 0.49323554],
                             [- 6.27318224, -0.00041200, 0.00025241, 7.27277023, 0.99874144, 0.99933559, 0.49323556],
                             [- 6.27318237, -0.00041195, 0.00025237, 7.27277042, 0.99874161, 0.99933568, 0.49323548],
                             [- 6.27318250, -0.00041190, 0.00025234, 7.27277061, 0.99874177, 0.99933577, 0.49323555],
                             [- 6.27318263, -0.00041184, 0.00025231, 7.27277079, 0.99874194, 0.99933585, 0.49323555],
                             [- 6.27318276, -0.00041179, 0.00025227, 7.27277098, 0.99874211, 0.99933594, 0.49323566],
                             [- 6.27318290, -0.00041173, 0.00025224, 7.27277117, 0.99874228, 0.99933603, 0.49323552],
                             [- 6.27318303, -0.00041168, 0.00025221, 7.27277135, 0.99874244, 0.99933612, 0.49323561],
                             [- 6.27318316, -0.00041162, 0.00025217, 7.27277154, 0.99874261, 0.99933621, 0.49323566],
                             [- 6.27318329, -0.00041157, 0.00025214, 7.27277172, 0.99874278, 0.99933630, 0.49323555],
                             [- 6.27318342, -0.00041151, 0.00025210, 7.27277191, 0.99874294, 0.99933638, 0.49323560],
                             [- 6.27318355, -0.00041146, 0.00025207, 7.27277209, 0.99874311, 0.99933647, 0.49323570],
                             [- 6.27318368, -0.00041140, 0.00025204, 7.27277228, 0.99874328, 0.99933656, 0.49323566],
                             [- 6.27318381, -0.00041135, 0.00025200, 7.27277246, 0.99874345, 0.99933665, 0.49323580],
                             [- 6.27318394, -0.00041129, 0.00025197, 7.27277265, 0.99874361, 0.99933674, 0.49323573],
                             [- 6.27318407, -0.00041124, 0.00025194, 7.27277284, 0.99874378, 0.99933682, 0.49323588],
                             [- 6.27318420, -0.00041118, 0.00025190, 7.27277302, 0.99874395, 0.99933691, 0.49323574],
                             [- 6.27318434, -0.00041113, 0.00025187, 7.27277321, 0.99874411, 0.99933700, 0.49323584],
                             [- 6.27318447, -0.00041107, 0.00025184, 7.27277339, 0.99874428, 0.99933709, 0.49323574],
                             [- 6.27318460, -0.00041102, 0.00025180, 7.27277358, 0.99874445, 0.99933718, 0.49323586],
                             [- 6.27318473, -0.00041097, 0.00025177, 7.27277376, 0.99874461, 0.99933726, 0.49323593],
                             [- 6.27318486, -0.00041091, 0.00025174, 7.27277395, 0.99874478, 0.99933735, 0.49323592],
                             [- 6.27318499, -0.00041086, 0.00025170, 7.27277413, 0.99874495, 0.99933744, 0.49323598],
                             [- 6.27318512, -0.00041080, 0.00025167, 7.27277432, 0.99874511, 0.99933753, 0.49323589],
                             [- 6.27318525, -0.00041075, 0.00025164, 7.27277450, 0.99874528, 0.99933762, 0.49323587],
                             [- 6.27318538, -0.00041069, 0.00025160, 7.27277469, 0.99874545, 0.99933770, 0.49323605],
                             [- 6.27318551, -0.00041064, 0.00025157, 7.27277487, 0.99874561, 0.99933779, 0.49323594],
                             [- 6.27318564, -0.00041058, 0.00025154, 7.27277506, 0.99874578, 0.99933788, 0.49323599],
                             [- 6.27318577, -0.00041053, 0.00025150, 7.27277524, 0.99874595, 0.99933797, 0.49323604],
                             [- 6.27318590, -0.00041048, 0.00025147, 7.27277543, 0.99874611, 0.99933806, 0.49323602],
                             [- 6.27318603, -0.00041042, 0.00025144, 7.27277561, 0.99874628, 0.99933814, 0.49323601],
                             [- 6.27318616, -0.00041037, 0.00025140, 7.27277580, 0.99874645, 0.99933823, 0.49323601],
                             [- 6.27318629, -0.00041031, 0.00025137, 7.27277598, 0.99874661, 0.99933832, 0.49323608],
                             [- 6.27318642, -0.00041026, 0.00025134, 7.27277616, 0.99874678, 0.99933841, 0.49323616],
                             [- 6.27318655, -0.00041020, 0.00025130, 7.27277635, 0.99874694, 0.99933849, 0.49323617],
                             [- 6.27318668, -0.00041015, 0.00025127, 7.27277653, 0.99874711, 0.99933858, 0.49323615],
                             [- 6.27318681, -0.00041009, 0.00025124, 7.27277672, 0.99874728, 0.99933867, 0.49323614],
                             [- 6.27318694, -0.00041004, 0.00025120, 7.27277690, 0.99874744, 0.99933876, 0.49323615],
                             [- 6.27318707, -0.00040999, 0.00025117, 7.27277709, 0.99874761, 0.99933884, 0.49323610],
                             [- 6.27318720, -0.00040993, 0.00025114, 7.27277727, 0.99874777, 0.99933893, 0.49323628],
                             [- 6.27318733, -0.00040988, 0.00025110, 7.27277746, 0.99874794, 0.99933902, 0.49323634],
                             [- 6.27318746, -0.00040982, 0.00025107, 7.27277764, 0.99874811, 0.99933911, 0.49323627],
                             [- 6.27318759, -0.00040977, 0.00025104, 7.27277782, 0.99874827, 0.99933919, 0.49323622],
                             [- 6.27318772, -0.00040971, 0.00025100, 7.27277801, 0.99874844, 0.99933928, 0.49323631],
                             [- 6.27318785, -0.00040966, 0.00025097, 7.27277819, 0.99874860, 0.99933937, 0.49323628],
                             [- 6.27318798, -0.00040961, 0.00025094, 7.27277838, 0.99874877, 0.99933946, 0.49323627],
                             [- 6.27318811, -0.00040955, 0.00025090, 7.27277856, 0.99874893, 0.99933954, 0.49323637],
                             [- 6.27318824, -0.00040950, 0.00025087, 7.27277874, 0.99874910, 0.99933963, 0.49323632],
                             [- 6.27318837, -0.00040944, 0.00025084, 7.27277893, 0.99874927, 0.99933972, 0.49323640],
                             [- 6.27318850, -0.00040939, 0.00025080, 7.27277911, 0.99874943, 0.99933981, 0.49323651],
                             [- 6.27318863, -0.00040934, 0.00025077, 7.27277929, 0.99874960, 0.99933989, 0.49323644],
                             [- 6.27318876, -0.00040928, 0.00025074, 7.27277948, 0.99874976, 0.99933998, 0.49323637],
                             [- 6.27318889, -0.00040923, 0.00025070, 7.27277966, 0.99874993, 0.99934007, 0.49323647],
                             [- 6.27318902, -0.00040917, 0.00025067, 7.27277985, 0.99875009, 0.99934016, 0.49323644],
                             [- 6.27318915, -0.00040912, 0.00025064, 7.27278003, 0.99875026, 0.99934024, 0.49323643],
                             [- 6.27318928, -0.00040906, 0.00025060, 7.27278021, 0.99875042, 0.99934033, 0.49323648],
                             [- 6.27318941, -0.00040901, 0.00025057, 7.27278040, 0.99875059, 0.99934042, 0.49323655],
                             [- 6.27318954, -0.00040896, 0.00025054, 7.27278058, 0.99875075, 0.99934050, 0.49323657],
                             [- 6.27318967, -0.00040890, 0.00025051, 7.27278076, 0.99875092, 0.99934059, 0.49323664],
                             [- 6.27318979, -0.00040885, 0.00025047, 7.27278095, 0.99875108, 0.99934068, 0.49323657],
                             [- 6.27318992, -0.00040879, 0.00025044, 7.27278113, 0.99875125, 0.99934077, 0.49323666],
                             [- 6.27319005, -0.00040874, 0.00025041, 7.27278131, 0.99875141, 0.99934085, 0.49323663],
                             [- 6.27319018, -0.00040869, 0.00025037, 7.27278150, 0.99875158, 0.99934094, 0.49323672],
                             [- 6.27319031, -0.00040863, 0.00025034, 7.27278168, 0.99875174, 0.99934103, 0.49323665],
                             [- 6.27319044, -0.00040858, 0.00025031, 7.27278186, 0.99875191, 0.99934111, 0.49323670],
                             [- 6.27319057, -0.00040852, 0.00025027, 7.27278204, 0.99875207, 0.99934120, 0.49323671],
                             [- 6.27319070, -0.00040847, 0.00025024, 7.27278223, 0.99875224, 0.99934129, 0.49323673],
                             [- 6.27319083, -0.00040842, 0.00025021, 7.27278241, 0.99875240, 0.99934137, 0.49323684],
                             [- 6.27319096, -0.00040836, 0.00025017, 7.27278259, 0.99875257, 0.99934146, 0.49323687],
                             [- 6.27319109, -0.00040831, 0.00025014, 7.27278278, 0.99875273, 0.99934155, 0.49323685],
                             [- 6.27319121, -0.00040826, 0.00025011, 7.27278296, 0.99875290, 0.99934164, 0.49323688],
                             [- 6.27319134, -0.00040820, 0.00025008, 7.27278314, 0.99875306, 0.99934172, 0.49323675],
                             [- 6.27319147, -0.00040815, 0.00025004, 7.27278332, 0.99875322, 0.99934181, 0.49323678],
                             [- 6.27319160, -0.00040809, 0.00025001, 7.27278351, 0.99875339, 0.99934190, 0.49323687],
                             [- 6.27319173, -0.00040804, 0.00024998, 7.27278369, 0.99875355, 0.99934198, 0.49323686],
                             [- 6.27319186, -0.00040799, 0.00024994, 7.27278387, 0.99875372, 0.99934207, 0.49323698],
                             [- 6.27319199, -0.00040793, 0.00024991, 7.27278405, 0.99875388, 0.99934216, 0.49323692],
                             [- 6.27319212, -0.00040788, 0.00024988, 7.27278424, 0.99875405, 0.99934224, 0.49323708],
                             [- 6.27319224, -0.00040783, 0.00024985, 7.27278442, 0.99875421, 0.99934233, 0.49323704],
                             [- 6.27319237, -0.00040777, 0.00024981, 7.27278460, 0.99875437, 0.99934242, 0.49323700],
                             [- 6.27319250, -0.00040772, 0.00024978, 7.27278478, 0.99875454, 0.99934250, 0.49323693],
                             [- 6.27319263, -0.00040766, 0.00024975, 7.27278497, 0.99875470, 0.99934259, 0.49323702],
                             [- 6.27319276, -0.00040761, 0.00024971, 7.27278515, 0.99875487, 0.99934268, 0.49323716],
                             [- 6.27319289, -0.00040756, 0.00024968, 7.27278533, 0.99875503, 0.99934276, 0.49323703],
                             [- 6.27319302, -0.00040750, 0.00024965, 7.27278551, 0.99875519, 0.99934285, 0.49323706],
                             [- 6.27319314, -0.00040745, 0.00024961, 7.27278569, 0.99875536, 0.99934294, 0.49323706],
                             [- 6.27319327, -0.00040740, 0.00024958, 7.27278588, 0.99875552, 0.99934302, 0.49323714],
                             [- 6.27319340, -0.00040734, 0.00024955, 7.27278606, 0.99875569, 0.99934311, 0.49323706],
                             [- 6.27319353, -0.00040729, 0.00024952, 7.27278624, 0.99875585, 0.99934320, 0.49323708],
                             [- 6.27319366, -0.00040724, 0.00024948, 7.27278642, 0.99875601, 0.99934328, 0.49323729],
                             [- 6.27319379, -0.00040718, 0.00024945, 7.27278660, 0.99875618, 0.99934337, 0.49323724],
                             [- 6.27319391, -0.00040713, 0.00024942, 7.27278679, 0.99875634, 0.99934345, 0.49323721],
                             [- 6.27319404, -0.00040707, 0.00024938, 7.27278697, 0.99875650, 0.99934354, 0.49323728],
                             [- 6.27319417, -0.00040702, 0.00024935, 7.27278715, 0.99875667, 0.99934363, 0.49323732],
                             [- 6.27319430, -0.00040697, 0.00024932, 7.27278733, 0.99875683, 0.99934371, 0.49323735],
                             [- 6.27319443, -0.00040691, 0.00024929, 7.27278751, 0.99875700, 0.99934380, 0.49323736],
                             [- 6.27319455, -0.00040686, 0.00024925, 7.27278769, 0.99875716, 0.99934389, 0.49323725],
                             [- 6.27319468, -0.00040681, 0.00024922, 7.27278788, 0.99875732, 0.99934397, 0.49323738],
                             [- 6.27319481, -0.00040675, 0.00024919, 7.27278806, 0.99875749, 0.99934406, 0.49323735],
                             [- 6.27319494, -0.00040670, 0.00024916, 7.27278824, 0.99875765, 0.99934414, 0.49323731],
                             [- 6.27319507, -0.00040665, 0.00024912, 7.27278842, 0.99875781, 0.99934423, 0.49323743],
                             [- 6.27319519, -0.00040659, 0.00024909, 7.27278860, 0.99875798, 0.99934432, 0.49323750],
                             [- 6.27319532, -0.00040654, 0.00024906, 7.27278878, 0.99875814, 0.99934440, 0.49323738],
                             [- 6.27319545, -0.00040649, 0.00024902, 7.27278896, 0.99875830, 0.99934449, 0.49323751],
                             [- 6.27319558, -0.00040643, 0.00024899, 7.27278914, 0.99875846, 0.99934457, 0.49323752],
                             [- 6.27319571, -0.00040638, 0.00024896, 7.27278933, 0.99875863, 0.99934466, 0.49323741],
                             [- 6.27319583, -0.00040633, 0.00024893, 7.27278951, 0.99875879, 0.99934475, 0.49323749],
                             [- 6.27319596, -0.00040627, 0.00024889, 7.27278969, 0.99875895, 0.99934483, 0.49323755],
                             [- 6.27319609, -0.00040622, 0.00024886, 7.27278987, 0.99875912, 0.99934492, 0.49323762],
                             [- 6.27319622, -0.00040617, 0.00024883, 7.27279005, 0.99875928, 0.99934501, 0.49323759],
                             [- 6.27319634, -0.00040611, 0.00024880, 7.27279023, 0.99875944, 0.99934509, 0.49323760],
                             [- 6.27319647, -0.00040606, 0.00024876, 7.27279041, 0.99875960, 0.99934518, 0.49323757],
                             [- 6.27319660, -0.00040601, 0.00024873, 7.27279059, 0.99875977, 0.99934526, 0.49323761],
                             [- 6.27319673, -0.00040595, 0.00024870, 7.27279077, 0.99875993, 0.99934535, 0.49323762],
                             [- 6.27319685, -0.00040590, 0.00024867, 7.27279095, 0.99876009, 0.99934543, 0.49323771],
                             [- 6.27319698, -0.00040585, 0.00024863, 7.27279113, 0.99876026, 0.99934552, 0.49323775],
                             [- 6.27319711, -0.00040579, 0.00024860, 7.27279131, 0.99876042, 0.99934561, 0.49323777],
                             [- 6.27319724, -0.00040574, 0.00024857, 7.27279149, 0.99876058, 0.99934569, 0.49323761],
                             [- 6.27319736, -0.00040569, 0.00024853, 7.27279168, 0.99876074, 0.99934578, 0.49323772],
                             [- 6.27319749, -0.00040563, 0.00024850, 7.27279186, 0.99876091, 0.99934586, 0.49323767],
                             [- 6.27319762, -0.00040558, 0.00024847, 7.27279204, 0.99876107, 0.99934595, 0.49323779],
                             [- 6.27319774, -0.00040553, 0.00024844, 7.27279222, 0.99876123, 0.99934603, 0.49323782],
                             [- 6.27319787, -0.00040547, 0.00024840, 7.27279240, 0.99876139, 0.99934612, 0.49323776],
                             [- 6.27319800, -0.00040542, 0.00024837, 7.27279258, 0.99876156, 0.99934621, 0.49323782],
                             [- 6.27319813, -0.00040537, 0.00024834, 7.27279276, 0.99876172, 0.99934629, 0.49323787],
                             [- 6.27319825, -0.00040532, 0.00024831, 7.27279294, 0.99876188, 0.99934638, 0.49323783],
                             [- 6.27319838, -0.00040526, 0.00024827, 7.27279312, 0.99876204, 0.99934646, 0.49323795],
                             [- 6.27319851, -0.00040521, 0.00024824, 7.27279330, 0.99876220, 0.99934655, 0.49323788],
                             [- 6.27319863, -0.00040516, 0.00024821, 7.27279348, 0.99876237, 0.99934663, 0.49323798],
                             [- 6.27319876, -0.00040510, 0.00024818, 7.27279366, 0.99876253, 0.99934672, 0.49323799],
                             [- 6.27319889, -0.00040505, 0.00024814, 7.27279384, 0.99876269, 0.99934681, 0.49323794],
                             [- 6.27319901, -0.00040500, 0.00024811, 7.27279402, 0.99876285, 0.99934689, 0.49323812],
                             [- 6.27319914, -0.00040494, 0.00024808, 7.27279420, 0.99876301, 0.99934698, 0.49323800],
                             [- 6.27319927, -0.00040489, 0.00024805, 7.27279438, 0.99876318, 0.99934706, 0.49323799],
                             [- 6.27319939, -0.00040484, 0.00024801, 7.27279456, 0.99876334, 0.99934715, 0.49323807],
                             [- 6.27319952, -0.00040479, 0.00024798, 7.27279474, 0.99876350, 0.99934723, 0.49323804],
                             [- 6.27319965, -0.00040473, 0.00024795, 7.27279492, 0.99876366, 0.99934732, 0.49323807],
                             [- 6.27319977, -0.00040468, 0.00024792, 7.27279510, 0.99876382, 0.99934740, 0.49323796],
                             [- 6.27319990, -0.00040463, 0.00024788, 7.27279527, 0.99876398, 0.99934749, 0.49323820],
                             [- 6.27320003, -0.00040457, 0.00024785, 7.27279545, 0.99876415, 0.99934757, 0.49323815],
                             [- 6.27320015, -0.00040452, 0.00024782, 7.27279563, 0.99876431, 0.99934766, 0.49323813],
                             [- 6.27320028, -0.00040447, 0.00024779, 7.27279581, 0.99876447, 0.99934774, 0.49323819],
                             [- 6.27320041, -0.00040442, 0.00024776, 7.27279599, 0.99876463, 0.99934783, 0.49323817],
                             [- 6.27320053, -0.00040436, 0.00024772, 7.27279617, 0.99876479, 0.99934791, 0.49323809],
                             [- 6.27320066, -0.00040431, 0.00024769, 7.27279635, 0.99876495, 0.99934800, 0.49323823],
                             [- 6.27320079, -0.00040426, 0.00024766, 7.27279653, 0.99876511, 0.99934809, 0.49323818],
                             [- 6.27320091, -0.00040420, 0.00024763, 7.27279671, 0.99876528, 0.99934817, 0.49323822],
                             [- 6.27320104, -0.00040415, 0.00024759, 7.27279689, 0.99876544, 0.99934826, 0.49323828],
                             [- 6.27320117, -0.00040410, 0.00024756, 7.27279707, 0.99876560, 0.99934834, 0.49323852],
                             [- 6.27320129, -0.00040405, 0.00024753, 7.27279725, 0.99876576, 0.99934843, 0.49323826],
                             [- 6.27320142, -0.00040399, 0.00024750, 7.27279743, 0.99876592, 0.99934851, 0.49323833],
                             [- 6.27320154, -0.00040394, 0.00024746, 7.27279760, 0.99876608, 0.99934860, 0.49323832],
                             [- 6.27320167, -0.00040389, 0.00024743, 7.27279778, 0.99876624, 0.99934868, 0.49323839],
                             [- 6.27320180, -0.00040383, 0.00024740, 7.27279796, 0.99876640, 0.99934877, 0.49323831],
                             [- 6.27320192, -0.00040378, 0.00024737, 7.27279814, 0.99876656, 0.99934885, 0.49323847],
                             [- 6.27320205, -0.00040373, 0.00024733, 7.27279832, 0.99876673, 0.99934894, 0.49323846],
                             [- 6.27320217, -0.00040368, 0.00024730, 7.27279850, 0.99876689, 0.99934902, 0.49323853],
                             [- 6.27320230, -0.00040362, 0.00024727, 7.27279868, 0.99876705, 0.99934911, 0.49323849],
                             [- 6.27320243, -0.00040357, 0.00024724, 7.27279886, 0.99876721, 0.99934919, 0.49323847],
                             [- 6.27320255, -0.00040352, 0.00024721, 7.27279903, 0.99876737, 0.99934928, 0.49323843],
                             [- 6.27320268, -0.00040347, 0.00024717, 7.27279921, 0.99876753, 0.99934936, 0.49323854],
                             [- 6.27320280, -0.00040341, 0.00024714, 7.27279939, 0.99876769, 0.99934944, 0.49323862],
                             [- 6.27320293, -0.00040336, 0.00024711, 7.27279957, 0.99876785, 0.99934953, 0.49323861],
                             [- 6.27320306, -0.00040331, 0.00024708, 7.27279975, 0.99876801, 0.99934961, 0.49323862],
                             [- 6.27320318, -0.00040326, 0.00024704, 7.27279993, 0.99876817, 0.99934970, 0.49323859],
                             [- 6.27320331, -0.00040320, 0.00024701, 7.27280010, 0.99876833, 0.99934978, 0.49323864],
                             [- 6.27320343, -0.00040315, 0.00024698, 7.27280028, 0.99876849, 0.99934987, 0.49323872],
                             [- 6.27320356, -0.00040310, 0.00024695, 7.27280046, 0.99876865, 0.99934995, 0.49323853],
                             [- 6.27320368, -0.00040305, 0.00024692, 7.27280064, 0.99876881, 0.99935004, 0.49323874],
                             [- 6.27320381, -0.00040299, 0.00024688, 7.27280082, 0.99876897, 0.99935012, 0.49323866],
                             [- 6.27320394, -0.00040294, 0.00024685, 7.27280099, 0.99876913, 0.99935021, 0.49323878],
                             [- 6.27320406, -0.00040289, 0.00024682, 7.27280117, 0.99876929, 0.99935029, 0.49323872],
                             [- 6.27320419, -0.00040284, 0.00024679, 7.27280135, 0.99876945, 0.99935038, 0.49323872],
                             [- 6.27320431, -0.00040278, 0.00024676, 7.27280153, 0.99876961, 0.99935046, 0.49323876],
                             [- 6.27320444, -0.00040273, 0.00024672, 7.27280171, 0.99876978, 0.99935055, 0.49323884],
                             [- 6.27320456, -0.00040268, 0.00024669, 7.27280188, 0.99876994, 0.99935063, 0.49323875],
                             [- 6.27320469, -0.00040263, 0.00024666, 7.27280206, 0.99877010, 0.99935071, 0.49323881],
                             [- 6.27320481, -0.00040257, 0.00024663, 7.27280224, 0.99877026, 0.99935080, 0.49323883],
                             [- 6.27320494, -0.00040252, 0.00024659, 7.27280242, 0.99877042, 0.99935088, 0.49323893],
                             [- 6.27320506, -0.00040247, 0.00024656, 7.27280259, 0.99877058, 0.99935097, 0.49323897],
                             [- 6.27320519, -0.00040242, 0.00024653, 7.27280277, 0.99877073, 0.99935105, 0.49323888],
                             [- 6.27320532, -0.00040237, 0.00024650, 7.27280295, 0.99877089, 0.99935114, 0.49323894],
                             [- 6.27320544, -0.00040231, 0.00024647, 7.27280313, 0.99877105, 0.99935122, 0.49323891],
                             [- 6.27320557, -0.00040226, 0.00024643, 7.27280330, 0.99877121, 0.99935130, 0.49323883],
                             [- 6.27320569, -0.00040221, 0.00024640, 7.27280348, 0.99877137, 0.99935139, 0.49323900],
                             [- 6.27320582, -0.00040216, 0.00024637, 7.27280366, 0.99877153, 0.99935147, 0.49323894],
                             [- 6.27320594, -0.00040210, 0.00024634, 7.27280384, 0.99877169, 0.99935156, 0.49323913],
                             [- 6.27320607, -0.00040205, 0.00024631, 7.27280401, 0.99877185, 0.99935164, 0.49323892],
                             [- 6.27320619, -0.00040200, 0.00024627, 7.27280419, 0.99877201, 0.99935173, 0.49323905],
                             [- 6.27320632, -0.00040195, 0.00024624, 7.27280437, 0.99877217, 0.99935181, 0.49323908],
                             [- 6.27320644, -0.00040190, 0.00024621, 7.27280455, 0.99877233, 0.99935189, 0.49323912],
                             [- 6.27320657, -0.00040184, 0.00024618, 7.27280472, 0.99877249, 0.99935198, 0.49323907],
                             [- 6.27320669, -0.00040179, 0.00024615, 7.27280490, 0.99877265, 0.99935206, 0.49323901],
                             [- 6.27320682, -0.00040174, 0.00024611, 7.27280508, 0.99877281, 0.99935215, 0.49323923],
                             [- 6.27320694, -0.00040169, 0.00024608, 7.27280525, 0.99877297, 0.99935223, 0.49323914],
                             [- 6.27320706, -0.00040163, 0.00024605, 7.27280543, 0.99877313, 0.99935231, 0.49323921],
                             [- 6.27320719, -0.00040158, 0.00024602, 7.27280561, 0.99877329, 0.99935240, 0.49323926],
                             [- 6.27320731, -0.00040153, 0.00024599, 7.27280578, 0.99877345, 0.99935248, 0.49323926],
                             [- 6.27320744, -0.00040148, 0.00024595, 7.27280596, 0.99877361, 0.99935257, 0.49323918],
                             [- 6.27320756, -0.00040143, 0.00024592, 7.27280614, 0.99877376, 0.99935265, 0.49323936],
                             [- 6.27320769, -0.00040137, 0.00024589, 7.27280631, 0.99877392, 0.99935273, 0.49323929],
                             [- 6.27320781, -0.00040132, 0.00024586, 7.27280649, 0.99877408, 0.99935282, 0.49323934],
                             [- 6.27320794, -0.00040127, 0.00024583, 7.27280667, 0.99877424, 0.99935290, 0.49323932],
                             [- 6.27320806, -0.00040122, 0.00024580, 7.27280684, 0.99877440, 0.99935299, 0.49323931],
                             [- 6.27320819, -0.00040117, 0.00024576, 7.27280702, 0.99877456, 0.99935307, 0.49323926],
                             [- 6.27320831, -0.00040111, 0.00024573, 7.27280720, 0.99877472, 0.99935315, 0.49323938],
                             [- 6.27320843, -0.00040106, 0.00024570, 7.27280737, 0.99877488, 0.99935324, 0.49323939],
                             [- 6.27320856, -0.00040101, 0.00024567, 7.27280755, 0.99877504, 0.99935332, 0.49323934],
                             [- 6.27320868, -0.00040096, 0.00024564, 7.27280773, 0.99877519, 0.99935341, 0.49323939],
                             [- 6.27320881, -0.00040091, 0.00024560, 7.27280790, 0.99877535, 0.99935349, 0.49323943],
                             [- 6.27320893, -0.00040085, 0.00024557, 7.27280808, 0.99877551, 0.99935357, 0.49323942],
                             [- 6.27320906, -0.00040080, 0.00024554, 7.27280825, 0.99877567, 0.99935366, 0.49323947],
                             [- 6.27320918, -0.00040075, 0.00024551, 7.27280843, 0.99877583, 0.99935374, 0.49323951],
                             [- 6.27320930, -0.00040070, 0.00024548, 7.27280861, 0.99877599, 0.99935382, 0.49323953],
                             [- 6.27320943, -0.00040065, 0.00024545, 7.27280878, 0.99877615, 0.99935391, 0.49323955],
                             [- 6.27320955, -0.00040059, 0.00024541, 7.27280896, 0.99877630, 0.99935399, 0.49323962],
                             [- 6.27320968, -0.00040054, 0.00024538, 7.27280913, 0.99877646, 0.99935408, 0.49323956],
                             [- 6.27320980, -0.00040049, 0.00024535, 7.27280931, 0.99877662, 0.99935416, 0.49323962],
                             [- 6.27320993, -0.00040044, 0.00024532, 7.27280949, 0.99877678, 0.99935424, 0.49323958],
                             [- 6.27321005, -0.00040039, 0.00024529, 7.27280966, 0.99877694, 0.99935433, 0.49323965],
                             [- 6.27321017, -0.00040034, 0.00024525, 7.27280984, 0.99877710, 0.99935441, 0.49323969],
                             [- 6.27321030, -0.00040028, 0.00024522, 7.27281001, 0.99877725, 0.99935449, 0.49323976],
                             [- 6.27321042, -0.00040023, 0.00024519, 7.27281019, 0.99877741, 0.99935458, 0.49323970],
                             [- 6.27321054, -0.00040018, 0.00024516, 7.27281036, 0.99877757, 0.99935466, 0.49323974],
                             [- 6.27321067, -0.00040013, 0.00024513, 7.27281054, 0.99877773, 0.99935474, 0.49323974],
                             [- 6.27321079, -0.00040008, 0.00024510, 7.27281072, 0.99877789, 0.99935483, 0.49323966],
                             [- 6.27321092, -0.00040003, 0.00024506, 7.27281089, 0.99877804, 0.99935491, 0.49323980],
                             [- 6.27321104, -0.00039997, 0.00024503, 7.27281107, 0.99877820, 0.99935499, 0.49323982],
                             [- 6.27321116, -0.00039992, 0.00024500, 7.27281124, 0.99877836, 0.99935508, 0.49323984],
                             [- 6.27321129, -0.00039987, 0.00024497, 7.27281142, 0.99877852, 0.99935516, 0.49323979],
                             [- 6.27321141, -0.00039982, 0.00024494, 7.27281159, 0.99877868, 0.99935524, 0.49323984],
                             [- 6.27321153, -0.00039977, 0.00024491, 7.27281177, 0.99877883, 0.99935533, 0.49323987],
                             [- 6.27321166, -0.00039972, 0.00024487, 7.27281194, 0.99877899, 0.99935541, 0.49323994],
                             [- 6.27321178, -0.00039966, 0.00024484, 7.27281212, 0.99877915, 0.99935549, 0.49323989],
                             [- 6.27321191, -0.00039961, 0.00024481, 7.27281229, 0.99877931, 0.99935558, 0.49323986],
                             [- 6.27321203, -0.00039956, 0.00024478, 7.27281247, 0.99877946, 0.99935566, 0.49323995],
                             [- 6.27321215, -0.00039951, 0.00024475, 7.27281264, 0.99877962, 0.99935574, 0.49323997],
                             [- 6.27321228, -0.00039946, 0.00024472, 7.27281282, 0.99877978, 0.99935583, 0.49323996],
                             [- 6.27321240, -0.00039941, 0.00024468, 7.27281299, 0.99877994, 0.99935591, 0.49323998],
                             [- 6.27321252, -0.00039935, 0.00024465, 7.27281317, 0.99878009, 0.99935599, 0.49323996],
                             [- 6.27321265, -0.00039930, 0.00024462, 7.27281334, 0.99878025, 0.99935608, 0.49324002],
                             [- 6.27321277, -0.00039925, 0.00024459, 7.27281352, 0.99878041, 0.99935616, 0.49323998],
                             [- 6.27321289, -0.00039920, 0.00024456, 7.27281369, 0.99878057, 0.99935624, 0.49324001],
                             [- 6.27321302, -0.00039915, 0.00024453, 7.27281387, 0.99878072, 0.99935632, 0.49324014],
                             [- 6.27321314, -0.00039910, 0.00024450, 7.27281404, 0.99878088, 0.99935641, 0.49324019],
                             [- 6.27321326, -0.00039905, 0.00024446, 7.27281422, 0.99878104, 0.99935649, 0.49324012],
                             [- 6.27321338, -0.00039899, 0.00024443, 7.27281439, 0.99878120, 0.99935657, 0.49324003],
                             [- 6.27321351, -0.00039894, 0.00024440, 7.27281457, 0.99878135, 0.99935666, 0.49324013],
                             [- 6.27321363, -0.00039889, 0.00024437, 7.27281474, 0.99878151, 0.99935674, 0.49324013],
                             [- 6.27321375, -0.00039884, 0.00024434, 7.27281491, 0.99878167, 0.99935682, 0.49324017],
                             [- 6.27321388, -0.00039879, 0.00024431, 7.27281509, 0.99878182, 0.99935690, 0.49324022],
                             [- 6.27321400, -0.00039874, 0.00024427, 7.27281526, 0.99878198, 0.99935699, 0.49324015],
                             [- 6.27321412, -0.00039869, 0.00024424, 7.27281544, 0.99878214, 0.99935707, 0.49324036],
                             [- 6.27321425, -0.00039863, 0.00024421, 7.27281561, 0.99878229, 0.99935715, 0.49324026],
                             [- 6.27321437, -0.00039858, 0.00024418, 7.27281579, 0.99878245, 0.99935724, 0.49324022],
                             [- 6.27321449, -0.00039853, 0.00024415, 7.27281596, 0.99878261, 0.99935732, 0.49324031],
                             [- 6.27321461, -0.00039848, 0.00024412, 7.27281613, 0.99878277, 0.99935740, 0.49324028],
                             [- 6.27321474, -0.00039843, 0.00024409, 7.27281631, 0.99878292, 0.99935748, 0.49324030],
                             [- 6.27321486, -0.00039838, 0.00024405, 7.27281648, 0.99878308, 0.99935757, 0.49324023],
                             [- 6.27321498, -0.00039833, 0.00024402, 7.27281666, 0.99878324, 0.99935765, 0.49324037],
                             [- 6.27321511, -0.00039828, 0.00024399, 7.27281683, 0.99878339, 0.99935773, 0.49324036],
                             [- 6.27321523, -0.00039822, 0.00024396, 7.27281700, 0.99878355, 0.99935782, 0.49324044],
                             [- 6.27321535, -0.00039817, 0.00024393, 7.27281718, 0.99878370, 0.99935790, 0.49324039],
                             [- 6.27321547, -0.00039812, 0.00024390, 7.27281735, 0.99878386, 0.99935798, 0.49324042],
                             [- 6.27321560, -0.00039807, 0.00024387, 7.27281752, 0.99878402, 0.99935806, 0.49324055],
                             [- 6.27321572, -0.00039802, 0.00024383, 7.27281770, 0.99878417, 0.99935815, 0.49324042],
                             [- 6.27321584, -0.00039797, 0.00024380, 7.27281787, 0.99878433, 0.99935823, 0.49324044],
                             [- 6.27321596, -0.00039792, 0.00024377, 7.27281805, 0.99878449, 0.99935831, 0.49324041],
                             [- 6.27321609, -0.00039787, 0.00024374, 7.27281822, 0.99878464, 0.99935839, 0.49324060],
                             [- 6.27321621, -0.00039781, 0.00024371, 7.27281839, 0.99878480, 0.99935848, 0.49324044],
                             [- 6.27321633, -0.00039776, 0.00024368, 7.27281857, 0.99878496, 0.99935856, 0.49324054],
                             [- 6.27321645, -0.00039771, 0.00024365, 7.27281874, 0.99878511, 0.99935864, 0.49324046],
                             [- 6.27321658, -0.00039766, 0.00024362, 7.27281891, 0.99878527, 0.99935872, 0.49324052],
                             [- 6.27321670, -0.00039761, 0.00024358, 7.27281909, 0.99878542, 0.99935881, 0.49324063],
                             [- 6.27321682, -0.00039756, 0.00024355, 7.27281926, 0.99878558, 0.99935889, 0.49324081],
                             [- 6.27321694, -0.00039751, 0.00024352, 7.27281943, 0.99878574, 0.99935897, 0.49324067],
                             [- 6.27321706, -0.00039746, 0.00024349, 7.27281961, 0.99878589, 0.99935905, 0.49324080],
                             [- 6.27321719, -0.00039741, 0.00024346, 7.27281978, 0.99878605, 0.99935913, 0.49324070],
                             [- 6.27321731, -0.00039736, 0.00024343, 7.27281995, 0.99878620, 0.99935922, 0.49324061],
                             [- 6.27321743, -0.00039730, 0.00024340, 7.27282013, 0.99878636, 0.99935930, 0.49324072],
                             [- 6.27321755, -0.00039725, 0.00024337, 7.27282030, 0.99878652, 0.99935938, 0.49324071],
                             [- 6.27321767, -0.00039720, 0.00024333, 7.27282047, 0.99878667, 0.99935946, 0.49324076],
                             [- 6.27321780, -0.00039715, 0.00024330, 7.27282064, 0.99878683, 0.99935955, 0.49324084],
                             [- 6.27321792, -0.00039710, 0.00024327, 7.27282082, 0.99878698, 0.99935963, 0.49324086],
                             [- 6.27321804, -0.00039705, 0.00024324, 7.27282099, 0.99878714, 0.99935971, 0.49324068],
                             [- 6.27321816, -0.00039700, 0.00024321, 7.27282116, 0.99878729, 0.99935979, 0.49324078],
                             [- 6.27321828, -0.00039695, 0.00024318, 7.27282134, 0.99878745, 0.99935987, 0.49324088],
                             [- 6.27321841, -0.00039690, 0.00024315, 7.27282151, 0.99878760, 0.99935996, 0.49324077],
                             [- 6.27321853, -0.00039685, 0.00024312, 7.27282168, 0.99878776, 0.99936004, 0.49324091],
                             [- 6.27321865, -0.00039680, 0.00024308, 7.27282185, 0.99878792, 0.99936012, 0.49324089],
                             [- 6.27321877, -0.00039674, 0.00024305, 7.27282203, 0.99878807, 0.99936020, 0.49324098],
                             [- 6.27321889, -0.00039669, 0.00024302, 7.27282220, 0.99878823, 0.99936028, 0.49324089],
                             [- 6.27321901, -0.00039664, 0.00024299, 7.27282237, 0.99878838, 0.99936037, 0.49324096],
                             [- 6.27321914, -0.00039659, 0.00024296, 7.27282254, 0.99878854, 0.99936045, 0.49324110],
                             [- 6.27321926, -0.00039654, 0.00024293, 7.27282272, 0.99878869, 0.99936053, 0.49324104],
                             [- 6.27321938, -0.00039649, 0.00024290, 7.27282289, 0.99878885, 0.99936061, 0.49324115],
                             [- 6.27321950, -0.00039644, 0.00024287, 7.27282306, 0.99878900, 0.99936069, 0.49324105],
                             [- 6.27321962, -0.00039639, 0.00024284, 7.27282323, 0.99878916, 0.99936078, 0.49324108],
                             [- 6.27321974, -0.00039634, 0.00024280, 7.27282341, 0.99878931, 0.99936086, 0.49324101],
                             [- 6.27321987, -0.00039629, 0.00024277, 7.27282358, 0.99878947, 0.99936094, 0.49324102],
                             [- 6.27321999, -0.00039624, 0.00024274, 7.27282375, 0.99878962, 0.99936102, 0.49324113],
                             [- 6.27322011, -0.00039619, 0.00024271, 7.27282392, 0.99878978, 0.99936110, 0.49324108],
                             [- 6.27322023, -0.00039613, 0.00024268, 7.27282409, 0.99878993, 0.99936119, 0.49324117],
                             [- 6.27322035, -0.00039608, 0.00024265, 7.27282427, 0.99879009, 0.99936127, 0.49324125],
                             [- 6.27322047, -0.00039603, 0.00024262, 7.27282444, 0.99879024, 0.99936135, 0.49324106],
                             [- 6.27322059, -0.00039598, 0.00024259, 7.27282461, 0.99879040, 0.99936143, 0.49324119],
                             [- 6.27322071, -0.00039593, 0.00024256, 7.27282478, 0.99879055, 0.99936151, 0.49324110],
                             [- 6.27322084, -0.00039588, 0.00024252, 7.27282495, 0.99879071, 0.99936159, 0.49324126],
                             [- 6.27322096, -0.00039583, 0.00024249, 7.27282513, 0.99879086, 0.99936168, 0.49324122],
                             [- 6.27322108, -0.00039578, 0.00024246, 7.27282530, 0.99879102, 0.99936176, 0.49324129],
                             [- 6.27322120, -0.00039573, 0.00024243, 7.27282547, 0.99879117, 0.99936184, 0.49324129],
                             [- 6.27322132, -0.00039568, 0.00024240, 7.27282564, 0.99879133, 0.99936192, 0.49324124],
                             [- 6.27322144, -0.00039563, 0.00024237, 7.27282581, 0.99879148, 0.99936200, 0.49324135],
                             [- 6.27322156, -0.00039558, 0.00024234, 7.27282598, 0.99879163, 0.99936208, 0.49324138],
                             [- 6.27322168, -0.00039553, 0.00024231, 7.27282616, 0.99879179, 0.99936216, 0.49324133],
                             [- 6.27322180, -0.00039548, 0.00024228, 7.27282633, 0.99879194, 0.99936225, 0.49324140],
                             [- 6.27322193, -0.00039543, 0.00024225, 7.27282650, 0.99879210, 0.99936233, 0.49324130],
                             [- 6.27322205, -0.00039538, 0.00024221, 7.27282667, 0.99879225, 0.99936241, 0.49324149],
                             [- 6.27322217, -0.00039533, 0.00024218, 7.27282684, 0.99879241, 0.99936249, 0.49324143],
                             [- 6.27322229, -0.00039528, 0.00024215, 7.27282701, 0.99879256, 0.99936257, 0.49324140],
                             [- 6.27322241, -0.00039522, 0.00024212, 7.27282718, 0.99879271, 0.99936265, 0.49324158],
                             [- 6.27322253, -0.00039517, 0.00024209, 7.27282736, 0.99879287, 0.99936273, 0.49324147],
                             [- 6.27322265, -0.00039512, 0.00024206, 7.27282753, 0.99879302, 0.99936282, 0.49324156],
                             [- 6.27322277, -0.00039507, 0.00024203, 7.27282770, 0.99879318, 0.99936290, 0.49324153],
                             [- 6.27322289, -0.00039502, 0.00024200, 7.27282787, 0.99879333, 0.99936298, 0.49324143],
                             [- 6.27322301, -0.00039497, 0.00024197, 7.27282804, 0.99879349, 0.99936306, 0.49324134],
                             [- 6.27322313, -0.00039492, 0.00024194, 7.27282821, 0.99879364, 0.99936314, 0.49324161],
                             [- 6.27322325, -0.00039487, 0.00024191, 7.27282838, 0.99879379, 0.99936322, 0.49324161],
                             [- 6.27322337, -0.00039482, 0.00024187, 7.27282855, 0.99879395, 0.99936330, 0.49324164],
                             [- 6.27322349, -0.00039477, 0.00024184, 7.27282872, 0.99879410, 0.99936339, 0.49324176],
                             [- 6.27322362, -0.00039472, 0.00024181, 7.27282889, 0.99879425, 0.99936347, 0.49324160],
                             [- 6.27322374, -0.00039467, 0.00024178, 7.27282907, 0.99879441, 0.99936355, 0.49324157],
                             [- 6.27322386, -0.00039462, 0.00024175, 7.27282924, 0.99879456, 0.99936363, 0.49324158],
                             [- 6.27322398, -0.00039457, 0.00024172, 7.27282941, 0.99879472, 0.99936371, 0.49324166],
                             [- 6.27322410, -0.00039452, 0.00024169, 7.27282958, 0.99879487, 0.99936379, 0.49324172],
                             [- 6.27322422, -0.00039447, 0.00024166, 7.27282975, 0.99879502, 0.99936387, 0.49324177],
                             [- 6.27322434, -0.00039442, 0.00024163, 7.27282992, 0.99879518, 0.99936395, 0.49324175],
                             [- 6.27322446, -0.00039437, 0.00024160, 7.27283009, 0.99879533, 0.99936403, 0.49324173],
                             [- 6.27322458, -0.00039432, 0.00024157, 7.27283026, 0.99879548, 0.99936412, 0.49324183],
                             [- 6.27322470, -0.00039427, 0.00024154, 7.27283043, 0.99879564, 0.99936420, 0.49324189],
                             [- 6.27322482, -0.00039422, 0.00024150, 7.27283060, 0.99879579, 0.99936428, 0.49324183],
                             [- 6.27322494, -0.00039417, 0.00024147, 7.27283077, 0.99879594, 0.99936436, 0.49324176],
                             [- 6.27322506, -0.00039412, 0.00024144, 7.27283094, 0.99879610, 0.99936444, 0.49324183],
                             [- 6.27322518, -0.00039407, 0.00024141, 7.27283111, 0.99879625, 0.99936452, 0.49324186],
                             [- 6.27322530, -0.00039402, 0.00024138, 7.27283128, 0.99879640, 0.99936460, 0.49324198],
                             [- 6.27322542, -0.00039397, 0.00024135, 7.27283145, 0.99879656, 0.99936468, 0.49324210],
                             [- 6.27322554, -0.00039392, 0.00024132, 7.27283162, 0.99879671, 0.99936476, 0.49324198],
                             [- 6.27322566, -0.00039387, 0.00024129, 7.27283179, 0.99879686, 0.99936484, 0.49324205],
                             [- 6.27322578, -0.00039382, 0.00024126, 7.27283196, 0.99879702, 0.99936492, 0.49324207],
                             [- 6.27322590, -0.00039377, 0.00024123, 7.27283213, 0.99879717, 0.99936501, 0.49324191],
                             [- 6.27322602, -0.00039372, 0.00024120, 7.27283230, 0.99879732, 0.99936509, 0.49324201],
                             [- 6.27322614, -0.00039367, 0.00024117, 7.27283247, 0.99879748, 0.99936517, 0.49324194],
                             [- 6.27322626, -0.00039362, 0.00024114, 7.27283264, 0.99879763, 0.99936525, 0.49324211],
                             [- 6.27322638, -0.00039357, 0.00024111, 7.27283281, 0.99879778, 0.99936533, 0.49324210],
                             [- 6.27322650, -0.00039352, 0.00024107, 7.27283298, 0.99879793, 0.99936541, 0.49324195],
                             [- 6.27322662, -0.00039347, 0.00024104, 7.27283315, 0.99879809, 0.99936549, 0.49324205],
                             [- 6.27322674, -0.00039342, 0.00024101, 7.27283332, 0.99879824, 0.99936557, 0.49324210],
                             [- 6.27322686, -0.00039337, 0.00024098, 7.27283349, 0.99879839, 0.99936565, 0.49324214],
                             [- 6.27322698, -0.00039332, 0.00024095, 7.27283366, 0.99879855, 0.99936573, 0.49324202],
                             [- 6.27322710, -0.00039327, 0.00024092, 7.27283383, 0.99879870, 0.99936581, 0.49324209],
                             [- 6.27322722, -0.00039322, 0.00024089, 7.27283400, 0.99879885, 0.99936589, 0.49324208],
                             [- 6.27322734, -0.00039317, 0.00024086, 7.27283417, 0.99879900, 0.99936597, 0.49324207],
                             [- 6.27322746, -0.00039312, 0.00024083, 7.27283434, 0.99879916, 0.99936605, 0.49324215],
                             [- 6.27322757, -0.00039307, 0.00024080, 7.27283451, 0.99879931, 0.99936613, 0.49324220],
                             [- 6.27322769, -0.00039302, 0.00024077, 7.27283468, 0.99879946, 0.99936621, 0.49324229],
                             [- 6.27322781, -0.00039297, 0.00024074, 7.27283485, 0.99879961, 0.99936630, 0.49324226],
                             [- 6.27322793, -0.00039292, 0.00024071, 7.27283502, 0.99879977, 0.99936638, 0.49324230],
                             [- 6.27322805, -0.00039287, 0.00024068, 7.27283519, 0.99879992, 0.99936646, 0.49324236],
                             [- 6.27322817, -0.00039282, 0.00024065, 7.27283535, 0.99880007, 0.99936654, 0.49324229],
                             [- 6.27322829, -0.00039277, 0.00024062, 7.27283552, 0.99880022, 0.99936662, 0.49324245],
                             [- 6.27322841, -0.00039272, 0.00024059, 7.27283569, 0.99880038, 0.99936670, 0.49324230],
                             [- 6.27322853, -0.00039267, 0.00024055, 7.27283586, 0.99880053, 0.99936678, 0.49324241],
                             [- 6.27322865, -0.00039262, 0.00024052, 7.27283603, 0.99880068, 0.99936686, 0.49324234],
                             [- 6.27322877, -0.00039257, 0.00024049, 7.27283620, 0.99880083, 0.99936694, 0.49324231],
                             [- 6.27322889, -0.00039252, 0.00024046, 7.27283637, 0.99880098, 0.99936702, 0.49324244],
                             [- 6.27322901, -0.00039247, 0.00024043, 7.27283654, 0.99880114, 0.99936710, 0.49324243],
                             [- 6.27322913, -0.00039242, 0.00024040, 7.27283671, 0.99880129, 0.99936718, 0.49324251],
                             [- 6.27322924, -0.00039237, 0.00024037, 7.27283688, 0.99880144, 0.99936726, 0.49324242],
                             [- 6.27322936, -0.00039232, 0.00024034, 7.27283704, 0.99880159, 0.99936734, 0.49324245],
                             [- 6.27322948, -0.00039227, 0.00024031, 7.27283721, 0.99880174, 0.99936742, 0.49324248],
                             [- 6.27322960, -0.00039222, 0.00024028, 7.27283738, 0.99880190, 0.99936750, 0.49324240],
                             [- 6.27322972, -0.00039217, 0.00024025, 7.27283755, 0.99880205, 0.99936758, 0.49324256],
                             [- 6.27322984, -0.00039212, 0.00024022, 7.27283772, 0.99880220, 0.99936766, 0.49324270],
                             [- 6.27322996, -0.00039207, 0.00024019, 7.27283789, 0.99880235, 0.99936774, 0.49324265],
                             [- 6.27323008, -0.00039202, 0.00024016, 7.27283806, 0.99880250, 0.99936782, 0.49324254],
                             [- 6.27323020, -0.00039197, 0.00024013, 7.27283822, 0.99880265, 0.99936790, 0.49324268],
                             [- 6.27323032, -0.00039192, 0.00024010, 7.27283839, 0.99880281, 0.99936798, 0.49324263],
                             [- 6.27323043, -0.00039187, 0.00024007, 7.27283856, 0.99880296, 0.99936806, 0.49324257],
                             [- 6.27323055, -0.00039182, 0.00024004, 7.27283873, 0.99880311, 0.99936814, 0.49324271],
                             [- 6.27323067, -0.00039177, 0.00024001, 7.27283890, 0.99880326, 0.99936822, 0.49324263],
                             [- 6.27323079, -0.00039172, 0.00023998, 7.27283907, 0.99880341, 0.99936830, 0.49324259],
                             [- 6.27323091, -0.00039167, 0.00023995, 7.27283923, 0.99880356, 0.99936838, 0.49324272],
                             [- 6.27323103, -0.00039162, 0.00023992, 7.27283940, 0.99880372, 0.99936846, 0.49324266],
                             [- 6.27323115, -0.00039158, 0.00023989, 7.27283957, 0.99880387, 0.99936854, 0.49324277],
                             [- 6.27323126, -0.00039153, 0.00023985, 7.27283974, 0.99880402, 0.99936862, 0.49324283],
                             [- 6.27323138, -0.00039148, 0.00023982, 7.27283991, 0.99880417, 0.99936870, 0.49324288],
                             [- 6.27323150, -0.00039143, 0.00023979, 7.27284008, 0.99880432, 0.99936878, 0.49324274],
                             [- 6.27323162, -0.00039138, 0.00023976, 7.27284024, 0.99880447, 0.99936886, 0.49324289],
                             [- 6.27323174, -0.00039133, 0.00023973, 7.27284041, 0.99880462, 0.99936894, 0.49324287],
                             [- 6.27323186, -0.00039128, 0.00023970, 7.27284058, 0.99880477, 0.99936902, 0.49324285],
                             [- 6.27323198, -0.00039123, 0.00023967, 7.27284075, 0.99880493, 0.99936910, 0.49324283],
                             [- 6.27323209, -0.00039118, 0.00023964, 7.27284091, 0.99880508, 0.99936918, 0.49324290],
                             [- 6.27323221, -0.00039113, 0.00023961, 7.27284108, 0.99880523, 0.99936926, 0.49324280],
                             [- 6.27323233, -0.00039108, 0.00023958, 7.27284125, 0.99880538, 0.99936934, 0.49324296],
                             [- 6.27323245, -0.00039103, 0.00023955, 7.27284142, 0.99880553, 0.99936942, 0.49324284],
                             [- 6.27323257, -0.00039098, 0.00023952, 7.27284159, 0.99880568, 0.99936950, 0.49324285],
                             [- 6.27323268, -0.00039093, 0.00023949, 7.27284175, 0.99880583, 0.99936958, 0.49324302],
                             [- 6.27323280, -0.00039088, 0.00023946, 7.27284192, 0.99880598, 0.99936966, 0.49324295],
                             [- 6.27323292, -0.00039083, 0.00023943, 7.27284209, 0.99880613, 0.99936974, 0.49324320],
                             [- 6.27323304, -0.00039078, 0.00023940, 7.27284226, 0.99880628, 0.99936982, 0.49324315],
                             [- 6.27323316, -0.00039073, 0.00023937, 7.27284242, 0.99880643, 0.99936990, 0.49324296],
                             [- 6.27323328, -0.00039069, 0.00023934, 7.27284259, 0.99880658, 0.99936997, 0.49324298],
                             [- 6.27323339, -0.00039064, 0.00023931, 7.27284276, 0.99880674, 0.99937005, 0.49324310],
                             [- 6.27323351, -0.00039059, 0.00023928, 7.27284292, 0.99880689, 0.99937013, 0.49324316],
                             [- 6.27323363, -0.00039054, 0.00023925, 7.27284309, 0.99880704, 0.99937021, 0.49324305],
                             [- 6.27323375, -0.00039049, 0.00023922, 7.27284326, 0.99880719, 0.99937029, 0.49324317],
                             [- 6.27323387, -0.00039044, 0.00023919, 7.27284343, 0.99880734, 0.99937037, 0.49324316],
                             [- 6.27323398, -0.00039039, 0.00023916, 7.27284359, 0.99880749, 0.99937045, 0.49324322],
                             [- 6.27323410, -0.00039034, 0.00023913, 7.27284376, 0.99880764, 0.99937053, 0.49324310],
                             [- 6.27323422, -0.00039029, 0.00023910, 7.27284393, 0.99880779, 0.99937061, 0.49324325],
                             [- 6.27323434, -0.00039024, 0.00023907, 7.27284409, 0.99880794, 0.99937069, 0.49324319],
                             [- 6.27323445, -0.00039019, 0.00023904, 7.27284426, 0.99880809, 0.99937077, 0.49324326],
                             [- 6.27323457, -0.00039014, 0.00023901, 7.27284443, 0.99880824, 0.99937085, 0.49324329],
                             [- 6.27323469, -0.00039009, 0.00023898, 7.27284460, 0.99880839, 0.99937093, 0.49324333],
                             [- 6.27323481, -0.00039005, 0.00023895, 7.27284476, 0.99880854, 0.99937101, 0.49324318],
                             [- 6.27323493, -0.00039000, 0.00023892, 7.27284493, 0.99880869, 0.99937109, 0.49324318],
                             [- 6.27323504, -0.00038995, 0.00023889, 7.27284510, 0.99880884, 0.99937117, 0.49324337],
                             [- 6.27323516, -0.00038990, 0.00023886, 7.27284526, 0.99880899, 0.99937124, 0.49324337],
                             [- 6.27323528, -0.00038985, 0.00023883, 7.27284543, 0.99880914, 0.99937132, 0.49324350],
                             [- 6.27323540, -0.00038980, 0.00023880, 7.27284560, 0.99880929, 0.99937140, 0.49324334],
                             [- 6.27323551, -0.00038975, 0.00023877, 7.27284576, 0.99880944, 0.99937148, 0.49324341],
                             [- 6.27323563, -0.00038970, 0.00023874, 7.27284593, 0.99880959, 0.99937156, 0.49324338],
                             [- 6.27323575, -0.00038965, 0.00023871, 7.27284609, 0.99880974, 0.99937164, 0.49324344],
                             [- 6.27323587, -0.00038960, 0.00023868, 7.27284626, 0.99880989, 0.99937172, 0.49324362],
                             [- 6.27323598, -0.00038955, 0.00023865, 7.27284643, 0.99881004, 0.99937180, 0.49324347],
                             [- 6.27323610, -0.00038951, 0.00023862, 7.27284659, 0.99881019, 0.99937188, 0.49324345],
                             [- 6.27323622, -0.00038946, 0.00023859, 7.27284676, 0.99881034, 0.99937196, 0.49324353],
                             [- 6.27323633, -0.00038941, 0.00023856, 7.27284693, 0.99881049, 0.99937204, 0.49324361],
                             [- 6.27323645, -0.00038936, 0.00023853, 7.27284709, 0.99881064, 0.99937211, 0.49324364],
                             [- 6.27323657, -0.00038931, 0.00023850, 7.27284726, 0.99881079, 0.99937219, 0.49324360],
                             [- 6.27323669, -0.00038926, 0.00023847, 7.27284743, 0.99881094, 0.99937227, 0.49324354],
                             [- 6.27323680, -0.00038921, 0.00023844, 7.27284759, 0.99881109, 0.99937235, 0.49324359],
                             [- 6.27323692, -0.00038916, 0.00023841, 7.27284776, 0.99881124, 0.99937243, 0.49324355],
                             [- 6.27323704, -0.00038911, 0.00023838, 7.27284792, 0.99881139, 0.99937251, 0.49324369],
                             [- 6.27323715, -0.00038907, 0.00023835, 7.27284809, 0.99881154, 0.99937259, 0.49324366],
                             [- 6.27323727, -0.00038902, 0.00023832, 7.27284826, 0.99881169, 0.99937267, 0.49324363],
                             [- 6.27323739, -0.00038897, 0.00023829, 7.27284842, 0.99881184, 0.99937275, 0.49324368],
                             [- 6.27323751, -0.00038892, 0.00023826, 7.27284859, 0.99881198, 0.99937282, 0.49324385],
                             [- 6.27323762, -0.00038887, 0.00023823, 7.27284875, 0.99881213, 0.99937290, 0.49324375],
                             [- 6.27323774, -0.00038882, 0.00023820, 7.27284892, 0.99881228, 0.99937298, 0.49324381],
                             [- 6.27323786, -0.00038877, 0.00023817, 7.27284908, 0.99881243, 0.99937306, 0.49324383],
                             [- 6.27323797, -0.00038872, 0.00023814, 7.27284925, 0.99881258, 0.99937314, 0.49324389],
                             [- 6.27323809, -0.00038867, 0.00023811, 7.27284942, 0.99881273, 0.99937322, 0.49324387],
                             [- 6.27323821, -0.00038863, 0.00023808, 7.27284958, 0.99881288, 0.99937330, 0.49324385],
                             [- 6.27323832, -0.00038858, 0.00023805, 7.27284975, 0.99881303, 0.99937338, 0.49324378],
                             [- 6.27323844, -0.00038853, 0.00023802, 7.27284991, 0.99881318, 0.99937345, 0.49324377],
                             [- 6.27323856, -0.00038848, 0.00023799, 7.27285008, 0.99881333, 0.99937353, 0.49324388],
                             [- 6.27323867, -0.00038843, 0.00023796, 7.27285024, 0.99881348, 0.99937361, 0.49324392],
                             [- 6.27323879, -0.00038838, 0.00023793, 7.27285041, 0.99881362, 0.99937369, 0.49324396],
                             [- 6.27323891, -0.00038833, 0.00023790, 7.27285057, 0.99881377, 0.99937377, 0.49324387],
                             [- 6.27323902, -0.00038828, 0.00023787, 7.27285074, 0.99881392, 0.99937385, 0.49324395],
                             [- 6.27323914, -0.00038824, 0.00023784, 7.27285090, 0.99881407, 0.99937393, 0.49324401],
                             [- 6.27323926, -0.00038819, 0.00023781, 7.27285107, 0.99881422, 0.99937400, 0.49324397],
                             [- 6.27323937, -0.00038814, 0.00023778, 7.27285124, 0.99881437, 0.99937408, 0.49324409],
                             [- 6.27323949, -0.00038809, 0.00023775, 7.27285140, 0.99881452, 0.99937416, 0.49324390],
                             [- 6.27323961, -0.00038804, 0.00023772, 7.27285157, 0.99881467, 0.99937424, 0.49324412],
                             [- 6.27323972, -0.00038799, 0.00023769, 7.27285173, 0.99881481, 0.99937432, 0.49324405],
                             [- 6.27323984, -0.00038794, 0.00023766, 7.27285190, 0.99881496, 0.99937440, 0.49324413],
                             [- 6.27323996, -0.00038789, 0.00023763, 7.27285206, 0.99881511, 0.99937448, 0.49324398],
                             [- 6.27324007, -0.00038785, 0.00023760, 7.27285223, 0.99881526, 0.99937455, 0.49324408],
                             [- 6.27324019, -0.00038780, 0.00023757, 7.27285239, 0.99881541, 0.99937463, 0.49324411],
                             [- 6.27324030, -0.00038775, 0.00023754, 7.27285256, 0.99881556, 0.99937471, 0.49324404],
                             [- 6.27324042, -0.00038770, 0.00023751, 7.27285272, 0.99881571, 0.99937479, 0.49324417],
                             [- 6.27324054, -0.00038765, 0.00023748, 7.27285289, 0.99881585, 0.99937487, 0.49324417],
                             [- 6.27324065, -0.00038760, 0.00023745, 7.27285305, 0.99881600, 0.99937495, 0.49324417],
                             [- 6.27324077, -0.00038755, 0.00023742, 7.27285321, 0.99881615, 0.99937502, 0.49324410],
                             [- 6.27324089, -0.00038751, 0.00023739, 7.27285338, 0.99881630, 0.99937510, 0.49324415],
                             [- 6.27324100, -0.00038746, 0.00023736, 7.27285354, 0.99881645, 0.99937518, 0.49324415],
                             [- 6.27324112, -0.00038741, 0.00023733, 7.27285371, 0.99881660, 0.99937526, 0.49324428],
                             [- 6.27324123, -0.00038736, 0.00023730, 7.27285387, 0.99881674, 0.99937534, 0.49324427],
                             [- 6.27324135, -0.00038731, 0.00023727, 7.27285404, 0.99881689, 0.99937541, 0.49324429],
                             [- 6.27324147, -0.00038726, 0.00023724, 7.27285420, 0.99881704, 0.99937549, 0.49324429],
                             [- 6.27324158, -0.00038722, 0.00023721, 7.27285437, 0.99881719, 0.99937557, 0.49324431],
                             [- 6.27324170, -0.00038717, 0.00023718, 7.27285453, 0.99881734, 0.99937565, 0.49324418],
                             [- 6.27324181, -0.00038712, 0.00023715, 7.27285469, 0.99881748, 0.99937573, 0.49324436],
                             [- 6.27324193, -0.00038707, 0.00023712, 7.27285486, 0.99881763, 0.99937581, 0.49324427],
                             [- 6.27324205, -0.00038702, 0.00023709, 7.27285502, 0.99881778, 0.99937588, 0.49324438],
                             [- 6.27324216, -0.00038697, 0.00023706, 7.27285519, 0.99881793, 0.99937596, 0.49324441],
                             [- 6.27324228, -0.00038693, 0.00023704, 7.27285535, 0.99881807, 0.99937604, 0.49324425],
                             [- 6.27324239, -0.00038688, 0.00023701, 7.27285552, 0.99881822, 0.99937612, 0.49324444],
                             [- 6.27324251, -0.00038683, 0.00023698, 7.27285568, 0.99881837, 0.99937620, 0.49324437],
                             [- 6.27324262, -0.00038678, 0.00023695, 7.27285584, 0.99881852, 0.99937627, 0.49324451],
                             [- 6.27324274, -0.00038673, 0.00023692, 7.27285601, 0.99881867, 0.99937635, 0.49324443],
                             [- 6.27324286, -0.00038668, 0.00023689, 7.27285617, 0.99881881, 0.99937643, 0.49324463],
                             [- 6.27324297, -0.00038664, 0.00023686, 7.27285634, 0.99881896, 0.99937651, 0.49324464],
                             [- 6.27324309, -0.00038659, 0.00023683, 7.27285650, 0.99881911, 0.99937659, 0.49324460],
                             [- 6.27324320, -0.00038654, 0.00023680, 7.27285666, 0.99881926, 0.99937666, 0.49324474],
                             [- 6.27324332, -0.00038649, 0.00023677, 7.27285683, 0.99881940, 0.99937674, 0.49324457],
                             [- 6.27324343, -0.00038644, 0.00023674, 7.27285699, 0.99881955, 0.99937682, 0.49324464],
                             [- 6.27324355, -0.00038639, 0.00023671, 7.27285716, 0.99881970, 0.99937690, 0.49324452],
                             [- 6.27324366, -0.00038635, 0.00023668, 7.27285732, 0.99881985, 0.99937697, 0.49324451],
                             [- 6.27324378, -0.00038630, 0.00023665, 7.27285748, 0.99881999, 0.99937705, 0.49324463],
                             [- 6.27324390, -0.00038625, 0.00023662, 7.27285765, 0.99882014, 0.99937713, 0.49324460],
                             [- 6.27324401, -0.00038620, 0.00023659, 7.27285781, 0.99882029, 0.99937721, 0.49324461],
                             [- 6.27324413, -0.00038615, 0.00023656, 7.27285797, 0.99882043, 0.99937729, 0.49324473],
                             [- 6.27324424, -0.00038610, 0.00023653, 7.27285814, 0.99882058, 0.99937736, 0.49324474],
                             [- 6.27324436, -0.00038606, 0.00023650, 7.27285830, 0.99882073, 0.99937744, 0.49324465],
                             [- 6.27324447, -0.00038601, 0.00023647, 7.27285846, 0.99882088, 0.99937752, 0.49324475],
                             [- 6.27324459, -0.00038596, 0.00023644, 7.27285863, 0.99882102, 0.99937760, 0.49324482],
                             [- 6.27324470, -0.00038591, 0.00023641, 7.27285879, 0.99882117, 0.99937767, 0.49324468],
                             [- 6.27324482, -0.00038586, 0.00023638, 7.27285895, 0.99882132, 0.99937775, 0.49324472],
                             [- 6.27324493, -0.00038582, 0.00023636, 7.27285912, 0.99882146, 0.99937783, 0.49324478],
                             [- 6.27324505, -0.00038577, 0.00023633, 7.27285928, 0.99882161, 0.99937791, 0.49324486],
                             [- 6.27324516, -0.00038572, 0.00023630, 7.27285944, 0.99882176, 0.99937798, 0.49324500],
                             [- 6.27324528, -0.00038567, 0.00023627, 7.27285961, 0.99882190, 0.99937806, 0.49324482],
                             [- 6.27324539, -0.00038562, 0.00023624, 7.27285977, 0.99882205, 0.99937814, 0.49324496],
                             [- 6.27324551, -0.00038558, 0.00023621, 7.27285993, 0.99882220, 0.99937822, 0.49324495],
                             [- 6.27324562, -0.00038553, 0.00023618, 7.27286009, 0.99882234, 0.99937829, 0.49324492],
                             [- 6.27324574, -0.00038548, 0.00023615, 7.27286026, 0.99882249, 0.99937837, 0.49324476],
                             [- 6.27324585, -0.00038543, 0.00023612, 7.27286042, 0.99882264, 0.99937845, 0.49324499],
                             [- 6.27324597, -0.00038538, 0.00023609, 7.27286058, 0.99882278, 0.99937853, 0.49324495],
                             [- 6.27324608, -0.00038534, 0.00023606, 7.27286075, 0.99882293, 0.99937860, 0.49324497],
                             [- 6.27324620, -0.00038529, 0.00023603, 7.27286091, 0.99882308, 0.99937868, 0.49324499],
                             [- 6.27324631, -0.00038524, 0.00023600, 7.27286107, 0.99882322, 0.99937876, 0.49324496],
                             [- 6.27324643, -0.00038519, 0.00023597, 7.27286123, 0.99882337, 0.99937884, 0.49324507],
                             [- 6.27324654, -0.00038514, 0.00023594, 7.27286140, 0.99882352, 0.99937891, 0.49324506],
                             [- 6.27324666, -0.00038510, 0.00023591, 7.27286156, 0.99882366, 0.99937899, 0.49324519],
                             [- 6.27324677, -0.00038505, 0.00023588, 7.27286172, 0.99882381, 0.99937907, 0.49324510],
                             [- 6.27324689, -0.00038500, 0.00023586, 7.27286188, 0.99882396, 0.99937914, 0.49324509],
                             [- 6.27324700, -0.00038495, 0.00023583, 7.27286205, 0.99882410, 0.99937922, 0.49324510],
                             [- 6.27324711, -0.00038490, 0.00023580, 7.27286221, 0.99882425, 0.99937930, 0.49324517],
                             [- 6.27324723, -0.00038486, 0.00023577, 7.27286237, 0.99882440, 0.99937938, 0.49324527],
                             [- 6.27324734, -0.00038481, 0.00023574, 7.27286253, 0.99882454, 0.99937945, 0.49324519],
                             [- 6.27324746, -0.00038476, 0.00023571, 7.27286270, 0.99882469, 0.99937953, 0.49324516],
                             [- 6.27324757, -0.00038471, 0.00023568, 7.27286286, 0.99882483, 0.99937961, 0.49324513],
                             [- 6.27324769, -0.00038467, 0.00023565, 7.27286302, 0.99882498, 0.99937968, 0.49324519],
                             [- 6.27324780, -0.00038462, 0.00023562, 7.27286318, 0.99882513, 0.99937976, 0.49324509],
                             [- 6.27324792, -0.00038457, 0.00023559, 7.27286335, 0.99882527, 0.99937984, 0.49324532],
                             [- 6.27324803, -0.00038452, 0.00023556, 7.27286351, 0.99882542, 0.99937992, 0.49324518],
                             [- 6.27324814, -0.00038447, 0.00023553, 7.27286367, 0.99882556, 0.99937999, 0.49324537],
                             [- 6.27324826, -0.00038443, 0.00023550, 7.27286383, 0.99882571, 0.99938007, 0.49324536],
                             [- 6.27324837, -0.00038438, 0.00023547, 7.27286399, 0.99882586, 0.99938015, 0.49324540],
                             [- 6.27324849, -0.00038433, 0.00023545, 7.27286416, 0.99882600, 0.99938022, 0.49324531],
                             [- 6.27324860, -0.00038428, 0.00023542, 7.27286432, 0.99882615, 0.99938030, 0.49324544],
                             [- 6.27324872, -0.00038424, 0.00023539, 7.27286448, 0.99882629, 0.99938038, 0.49324534],
                             [- 6.27324883, -0.00038419, 0.00023536, 7.27286464, 0.99882644, 0.99938045, 0.49324551],
                             [- 6.27324894, -0.00038414, 0.00023533, 7.27286480, 0.99882658, 0.99938053, 0.49324541],
                             [- 6.27324906, -0.00038409, 0.00023530, 7.27286497, 0.99882673, 0.99938061, 0.49324551],
                             [- 6.27324917, -0.00038405, 0.00023527, 7.27286513, 0.99882688, 0.99938068, 0.49324541],
                             [- 6.27324929, -0.00038400, 0.00023524, 7.27286529, 0.99882702, 0.99938076, 0.49324543],
                             [- 6.27324940, -0.00038395, 0.00023521, 7.27286545, 0.99882717, 0.99938084, 0.49324548],
                             [- 6.27324951, -0.00038390, 0.00023518, 7.27286561, 0.99882731, 0.99938092, 0.49324547],
                             [- 6.27324963, -0.00038385, 0.00023515, 7.27286577, 0.99882746, 0.99938099, 0.49324552],
                             [- 6.27324974, -0.00038381, 0.00023512, 7.27286594, 0.99882760, 0.99938107, 0.49324561],
                             [- 6.27324986, -0.00038376, 0.00023509, 7.27286610, 0.99882775, 0.99938115, 0.49324575],
                             [- 6.27324997, -0.00038371, 0.00023507, 7.27286626, 0.99882789, 0.99938122, 0.49324555],
                             [- 6.27325008, -0.00038366, 0.00023504, 7.27286642, 0.99882804, 0.99938130, 0.49324561],
                             [- 6.27325020, -0.00038362, 0.00023501, 7.27286658, 0.99882818, 0.99938138, 0.49324560],
                             [- 6.27325031, -0.00038357, 0.00023498, 7.27286674, 0.99882833, 0.99938145, 0.49324567],
                             [- 6.27325043, -0.00038352, 0.00023495, 7.27286690, 0.99882847, 0.99938153, 0.49324557],
                             [- 6.27325054, -0.00038347, 0.00023492, 7.27286706, 0.99882862, 0.99938161, 0.49324572],
                             [- 6.27325065, -0.00038343, 0.00023489, 7.27286723, 0.99882877, 0.99938168, 0.49324570],
                             [- 6.27325077, -0.00038338, 0.00023486, 7.27286739, 0.99882891, 0.99938176, 0.49324576],
                             [- 6.27325088, -0.00038333, 0.00023483, 7.27286755, 0.99882906, 0.99938184, 0.49324591],
                             [- 6.27325099, -0.00038328, 0.00023480, 7.27286771, 0.99882920, 0.99938191, 0.49324569],
                             [- 6.27325111, -0.00038324, 0.00023477, 7.27286787, 0.99882935, 0.99938199, 0.49324559],
                             [- 6.27325122, -0.00038319, 0.00023475, 7.27286803, 0.99882949, 0.99938207, 0.49324564],
                             [- 6.27325133, -0.00038314, 0.00023472, 7.27286819, 0.99882964, 0.99938214, 0.49324576],
                             [- 6.27325145, -0.00038309, 0.00023469, 7.27286835, 0.99882978, 0.99938222, 0.49324581],
                             [- 6.27325156, -0.00038305, 0.00023466, 7.27286851, 0.99882993, 0.99938229, 0.49324590],
                             [- 6.27325168, -0.00038300, 0.00023463, 7.27286868, 0.99883007, 0.99938237, 0.49324573],
                             [- 6.27325179, -0.00038295, 0.00023460, 7.27286884, 0.99883021, 0.99938245, 0.49324578],
                             [- 6.27325190, -0.00038291, 0.00023457, 7.27286900, 0.99883036, 0.99938252, 0.49324585],
                             [- 6.27325202, -0.00038286, 0.00023454, 7.27286916, 0.99883050, 0.99938260, 0.49324579],
                             [- 6.27325213, -0.00038281, 0.00023451, 7.27286932, 0.99883065, 0.99938268, 0.49324589],
                             [- 6.27325224, -0.00038276, 0.00023448, 7.27286948, 0.99883079, 0.99938275, 0.49324588],
                             [- 6.27325236, -0.00038272, 0.00023446, 7.27286964, 0.99883094, 0.99938283, 0.49324586],
                             [- 6.27325247, -0.00038267, 0.00023443, 7.27286980, 0.99883108, 0.99938291, 0.49324588],
                             [- 6.27325258, -0.00038262, 0.00023440, 7.27286996, 0.99883123, 0.99938298, 0.49324600],
                             [- 6.27325269, -0.00038257, 0.00023437, 7.27287012, 0.99883137, 0.99938306, 0.49324602],
                             [- 6.27325281, -0.00038253, 0.00023434, 7.27287028, 0.99883152, 0.99938313, 0.49324593],
                             [- 6.27325292, -0.00038248, 0.00023431, 7.27287044, 0.99883166, 0.99938321, 0.49324600],
                             [- 6.27325303, -0.00038243, 0.00023428, 7.27287060, 0.99883180, 0.99938329, 0.49324594],
                             [- 6.27325315, -0.00038238, 0.00023425, 7.27287076, 0.99883195, 0.99938336, 0.49324594],
                             [- 6.27325326, -0.00038234, 0.00023422, 7.27287092, 0.99883209, 0.99938344, 0.49324616],
                             [- 6.27325337, -0.00038229, 0.00023419, 7.27287108, 0.99883224, 0.99938352, 0.49324609],
                             [- 6.27325349, -0.00038224, 0.00023417, 7.27287124, 0.99883238, 0.99938359, 0.49324615],
                             [- 6.27325360, -0.00038220, 0.00023414, 7.27287140, 0.99883253, 0.99938367, 0.49324616],
                             [- 6.27325371, -0.00038215, 0.00023411, 7.27287156, 0.99883267, 0.99938374, 0.49324616],
                             [- 6.27325383, -0.00038210, 0.00023408, 7.27287172, 0.99883281, 0.99938382, 0.49324603],
                             [- 6.27325394, -0.00038205, 0.00023405, 7.27287188, 0.99883296, 0.99938390, 0.49324614],
                             [- 6.27325405, -0.00038201, 0.00023402, 7.27287204, 0.99883310, 0.99938397, 0.49324629],
                             [- 6.27325416, -0.00038196, 0.00023399, 7.27287220, 0.99883325, 0.99938405, 0.49324628],
                             [- 6.27325428, -0.00038191, 0.00023396, 7.27287236, 0.99883339, 0.99938412, 0.49324634],
                             [- 6.27325439, -0.00038187, 0.00023393, 7.27287252, 0.99883353, 0.99938420, 0.49324618],
                             [- 6.27325450, -0.00038182, 0.00023391, 7.27287268, 0.99883368, 0.99938428, 0.49324620],
                             [- 6.27325462, -0.00038177, 0.00023388, 7.27287284, 0.99883382, 0.99938435, 0.49324621],
                             [- 6.27325473, -0.00038172, 0.00023385, 7.27287300, 0.99883397, 0.99938443, 0.49324620],
                             [- 6.27325484, -0.00038168, 0.00023382, 7.27287316, 0.99883411, 0.99938450, 0.49324631],
                             [- 6.27325495, -0.00038163, 0.00023379, 7.27287332, 0.99883425, 0.99938458, 0.49324616],
                             [- 6.27325507, -0.00038158, 0.00023376, 7.27287348, 0.99883440, 0.99938466, 0.49324629],
                             [- 6.27325518, -0.00038154, 0.00023373, 7.27287364, 0.99883454, 0.99938473, 0.49324635],
                             [- 6.27325529, -0.00038149, 0.00023370, 7.27287380, 0.99883469, 0.99938481, 0.49324648],
                             [- 6.27325540, -0.00038144, 0.00023367, 7.27287396, 0.99883483, 0.99938488, 0.49324638],
                             [- 6.27325552, -0.00038140, 0.00023365, 7.27287412, 0.99883497, 0.99938496, 0.49324637],
                             [- 6.27325563, -0.00038135, 0.00023362, 7.27287428, 0.99883512, 0.99938503, 0.49324641],
                             [- 6.27325574, -0.00038130, 0.00023359, 7.27287444, 0.99883526, 0.99938511, 0.49324645],
                             [- 6.27325585, -0.00038125, 0.00023356, 7.27287460, 0.99883540, 0.99938519, 0.49324647],
                             [- 6.27325597, -0.00038121, 0.00023353, 7.27287476, 0.99883555, 0.99938526, 0.49324642],
                             [- 6.27325608, -0.00038116, 0.00023350, 7.27287492, 0.99883569, 0.99938534, 0.49324651],
                             [- 6.27325619, -0.00038111, 0.00023347, 7.27287508, 0.99883583, 0.99938541, 0.49324649],
                             [- 6.27325630, -0.00038107, 0.00023344, 7.27287524, 0.99883598, 0.99938549, 0.49324647],
                             [- 6.27325642, -0.00038102, 0.00023342, 7.27287540, 0.99883612, 0.99938556, 0.49324658],
                             [- 6.27325653, -0.00038097, 0.00023339, 7.27287555, 0.99883626, 0.99938564, 0.49324664],
                             [- 6.27325664, -0.00038093, 0.00023336, 7.27287571, 0.99883641, 0.99938572, 0.49324649],
                             [- 6.27325675, -0.00038088, 0.00023333, 7.27287587, 0.99883655, 0.99938579, 0.49324658],
                             [- 6.27325686, -0.00038083, 0.00023330, 7.27287603, 0.99883669, 0.99938587, 0.49324653],
                             [- 6.27325698, -0.00038079, 0.00023327, 7.27287619, 0.99883684, 0.99938594, 0.49324658],
                             [- 6.27325709, -0.00038074, 0.00023324, 7.27287635, 0.99883698, 0.99938602, 0.49324649],
                             [- 6.27325720, -0.00038069, 0.00023321, 7.27287651, 0.99883712, 0.99938609, 0.49324663],
                             [- 6.27325731, -0.00038065, 0.00023319, 7.27287667, 0.99883727, 0.99938617, 0.49324670],
                             [- 6.27325742, -0.00038060, 0.00023316, 7.27287683, 0.99883741, 0.99938624, 0.49324665],
                             [- 6.27325754, -0.00038055, 0.00023313, 7.27287699, 0.99883755, 0.99938632, 0.49324681],
                             [- 6.27325765, -0.00038050, 0.00023310, 7.27287714, 0.99883769, 0.99938640, 0.49324678],
                             [- 6.27325776, -0.00038046, 0.00023307, 7.27287730, 0.99883784, 0.99938647, 0.49324668],
                             [- 6.27325787, -0.00038041, 0.00023304, 7.27287746, 0.99883798, 0.99938655, 0.49324676],
                             [- 6.27325798, -0.00038036, 0.00023301, 7.27287762, 0.99883812, 0.99938662, 0.49324679],
                             [- 6.27325810, -0.00038032, 0.00023299, 7.27287778, 0.99883827, 0.99938670, 0.49324680],
                             [- 6.27325821, -0.00038027, 0.00023296, 7.27287794, 0.99883841, 0.99938677, 0.49324686],
                             [- 6.27325832, -0.00038022, 0.00023293, 7.27287810, 0.99883855, 0.99938685, 0.49324703],
                             [- 6.27325843, -0.00038018, 0.00023290, 7.27287825, 0.99883869, 0.99938692, 0.49324683],
                             [- 6.27325854, -0.00038013, 0.00023287, 7.27287841, 0.99883884, 0.99938700, 0.49324688],
                             [- 6.27325866, -0.00038008, 0.00023284, 7.27287857, 0.99883898, 0.99938707, 0.49324687],
                             [- 6.27325877, -0.00038004, 0.00023281, 7.27287873, 0.99883912, 0.99938715, 0.49324692],
                             [- 6.27325888, -0.00037999, 0.00023279, 7.27287889, 0.99883926, 0.99938722, 0.49324678],
                             [- 6.27325899, -0.00037994, 0.00023276, 7.27287905, 0.99883941, 0.99938730, 0.49324693],
                             [- 6.27325910, -0.00037990, 0.00023273, 7.27287920, 0.99883955, 0.99938737, 0.49324693],
                             [- 6.27325921, -0.00037985, 0.00023270, 7.27287936, 0.99883969, 0.99938745, 0.49324695],
                             [- 6.27325932, -0.00037980, 0.00023267, 7.27287952, 0.99883983, 0.99938752, 0.49324704],
                             [- 6.27325944, -0.00037976, 0.00023264, 7.27287968, 0.99883998, 0.99938760, 0.49324699],
                             [- 6.27325955, -0.00037971, 0.00023261, 7.27287984, 0.99884012, 0.99938768, 0.49324702],
                             [- 6.27325966, -0.00037966, 0.00023259, 7.27287999, 0.99884026, 0.99938775, 0.49324694],
                             [- 6.27325977, -0.00037962, 0.00023256, 7.27288015, 0.99884040, 0.99938783, 0.49324708],
                             [- 6.27325988, -0.00037957, 0.00023253, 7.27288031, 0.99884055, 0.99938790, 0.49324716],
                             [- 6.27325999, -0.00037953, 0.00023250, 7.27288047, 0.99884069, 0.99938798, 0.49324712],
                             [- 6.27326011, -0.00037948, 0.00023247, 7.27288063, 0.99884083, 0.99938805, 0.49324715],
                             [- 6.27326022, -0.00037943, 0.00023244, 7.27288078, 0.99884097, 0.99938813, 0.49324707],
                             [- 6.27326033, -0.00037939, 0.00023241, 7.27288094, 0.99884111, 0.99938820, 0.49324704],
                             [- 6.27326044, -0.00037934, 0.00023239, 7.27288110, 0.99884126, 0.99938828, 0.49324718],
                             [- 6.27326055, -0.00037929, 0.00023236, 7.27288126, 0.99884140, 0.99938835, 0.49324714],
                             [- 6.27326066, -0.00037925, 0.00023233, 7.27288142, 0.99884154, 0.99938843, 0.49324726],
                             [- 6.27326077, -0.00037920, 0.00023230, 7.27288157, 0.99884168, 0.99938850, 0.49324702],
                             [- 6.27326088, -0.00037915, 0.00023227, 7.27288173, 0.99884182, 0.99938858, 0.49324721],
                             [- 6.27326100, -0.00037911, 0.00023224, 7.27288189, 0.99884197, 0.99938865, 0.49324723],
                             [- 6.27326111, -0.00037906, 0.00023221, 7.27288205, 0.99884211, 0.99938873, 0.49324718],
                             [- 6.27326122, -0.00037901, 0.00023219, 7.27288220, 0.99884225, 0.99938880, 0.49324712],
                             [- 6.27326133, -0.00037897, 0.00023216, 7.27288236, 0.99884239, 0.99938887, 0.49324742],
                             [- 6.27326144, -0.00037892, 0.00023213, 7.27288252, 0.99884253, 0.99938895, 0.49324731],
                             [- 6.27326155, -0.00037887, 0.00023210, 7.27288268, 0.99884267, 0.99938902, 0.49324731],
                             [- 6.27326166, -0.00037883, 0.00023207, 7.27288283, 0.99884282, 0.99938910, 0.49324735],
                             [- 6.27326177, -0.00037878, 0.00023204, 7.27288299, 0.99884296, 0.99938917, 0.49324724],
                             [- 6.27326188, -0.00037874, 0.00023202, 7.27288315, 0.99884310, 0.99938925, 0.49324733],
                             [- 6.27326199, -0.00037869, 0.00023199, 7.27288331, 0.99884324, 0.99938932, 0.49324726],
                             [- 6.27326211, -0.00037864, 0.00023196, 7.27288346, 0.99884338, 0.99938940, 0.49324734],
                             [- 6.27326222, -0.00037860, 0.00023193, 7.27288362, 0.99884352, 0.99938947, 0.49324737],
                             [- 6.27326233, -0.00037855, 0.00023190, 7.27288378, 0.99884367, 0.99938955, 0.49324746],
                             [- 6.27326244, -0.00037850, 0.00023187, 7.27288393, 0.99884381, 0.99938962, 0.49324742],
                             [- 6.27326255, -0.00037846, 0.00023185, 7.27288409, 0.99884395, 0.99938970, 0.49324750],
                             [- 6.27326266, -0.00037841, 0.00023182, 7.27288425, 0.99884409, 0.99938977, 0.49324754],
                             [- 6.27326277, -0.00037837, 0.00023179, 7.27288441, 0.99884423, 0.99938985, 0.49324741],
                             [- 6.27326288, -0.00037832, 0.00023176, 7.27288456, 0.99884437, 0.99938992, 0.49324760],
                             [- 6.27326299, -0.00037827, 0.00023173, 7.27288472, 0.99884451, 0.99939000, 0.49324751],
                             [- 6.27326310, -0.00037823, 0.00023170, 7.27288488, 0.99884466, 0.99939007, 0.49324747],
                             [- 6.27326321, -0.00037818, 0.00023168, 7.27288503, 0.99884480, 0.99939014, 0.49324756],
                             [- 6.27326332, -0.00037813, 0.00023165, 7.27288519, 0.99884494, 0.99939022, 0.49324744],
                             [- 6.27326343, -0.00037809, 0.00023162, 7.27288535, 0.99884508, 0.99939029, 0.49324765],
                             [- 6.27326354, -0.00037804, 0.00023159, 7.27288550, 0.99884522, 0.99939037, 0.49324755],
                             [- 6.27326366, -0.00037800, 0.00023156, 7.27288566, 0.99884536, 0.99939044, 0.49324760],
                             [- 6.27326377, -0.00037795, 0.00023153, 7.27288582, 0.99884550, 0.99939052, 0.49324764],
                             [- 6.27326388, -0.00037790, 0.00023151, 7.27288597, 0.99884564, 0.99939059, 0.49324764],
                             [- 6.27326399, -0.00037786, 0.00023148, 7.27288613, 0.99884578, 0.99939067, 0.49324768],
                             [- 6.27326410, -0.00037781, 0.00023145, 7.27288629, 0.99884592, 0.99939074, 0.49324772],
                             [- 6.27326421, -0.00037777, 0.00023142, 7.27288644, 0.99884607, 0.99939081, 0.49324775],
                             [- 6.27326432, -0.00037772, 0.00023139, 7.27288660, 0.99884621, 0.99939089, 0.49324763],
                             [- 6.27326443, -0.00037767, 0.00023136, 7.27288676, 0.99884635, 0.99939096, 0.49324773],
                             [- 6.27326454, -0.00037763, 0.00023134, 7.27288691, 0.99884649, 0.99939104, 0.49324770],
                             [- 6.27326465, -0.00037758, 0.00023131, 7.27288707, 0.99884663, 0.99939111, 0.49324772],
                             [- 6.27326476, -0.00037753, 0.00023128, 7.27288722, 0.99884677, 0.99939119, 0.49324778],
                             [- 6.27326487, -0.00037749, 0.00023125, 7.27288738, 0.99884691, 0.99939126, 0.49324783],
                             [- 6.27326498, -0.00037744, 0.00023122, 7.27288754, 0.99884705, 0.99939133, 0.49324780],
                             [- 6.27326509, -0.00037740, 0.00023120, 7.27288769, 0.99884719, 0.99939141, 0.49324776],
                             [- 6.27326520, -0.00037735, 0.00023117, 7.27288785, 0.99884733, 0.99939148, 0.49324786],
                             [- 6.27326531, -0.00037730, 0.00023114, 7.27288801, 0.99884747, 0.99939156, 0.49324788],
                             [- 6.27326542, -0.00037726, 0.00023111, 7.27288816, 0.99884761, 0.99939163, 0.49324776],
                             [- 6.27326553, -0.00037721, 0.00023108, 7.27288832, 0.99884775, 0.99939171, 0.49324788],
                             [- 6.27326564, -0.00037717, 0.00023105, 7.27288847, 0.99884789, 0.99939178, 0.49324801],
                             [- 6.27326575, -0.00037712, 0.00023103, 7.27288863, 0.99884803, 0.99939185, 0.49324790],
                             [- 6.27326586, -0.00037707, 0.00023100, 7.27288879, 0.99884817, 0.99939193, 0.49324783],
                             [- 6.27326597, -0.00037703, 0.00023097, 7.27288894, 0.99884831, 0.99939200, 0.49324795],
                             [- 6.27326608, -0.00037698, 0.00023094, 7.27288910, 0.99884846, 0.99939208, 0.49324808],
                             [- 6.27326619, -0.00037694, 0.00023091, 7.27288925, 0.99884860, 0.99939215, 0.49324796],
                             [- 6.27326630, -0.00037689, 0.00023089, 7.27288941, 0.99884874, 0.99939222, 0.49324802],
                             [- 6.27326641, -0.00037685, 0.00023086, 7.27288956, 0.99884888, 0.99939230, 0.49324786],
                             [- 6.27326652, -0.00037680, 0.00023083, 7.27288972, 0.99884902, 0.99939237, 0.49324795],
                             [- 6.27326663, -0.00037675, 0.00023080, 7.27288988, 0.99884916, 0.99939245, 0.49324822],
                             [- 6.27326674, -0.00037671, 0.00023077, 7.27289003, 0.99884930, 0.99939252, 0.49324803],
                             [- 6.27326685, -0.00037666, 0.00023074, 7.27289019, 0.99884944, 0.99939259, 0.49324799],
                             [- 6.27326696, -0.00037662, 0.00023072, 7.27289034, 0.99884958, 0.99939267, 0.49324798],
                             [- 6.27326707, -0.00037657, 0.00023069, 7.27289050, 0.99884972, 0.99939274, 0.49324802],
                             [- 6.27326718, -0.00037652, 0.00023066, 7.27289065, 0.99884986, 0.99939282, 0.49324803],
                             [- 6.27326729, -0.00037648, 0.00023063, 7.27289081, 0.99885000, 0.99939289, 0.49324811],
                             [- 6.27326740, -0.00037643, 0.00023060, 7.27289096, 0.99885014, 0.99939296, 0.49324817],
                             [- 6.27326751, -0.00037639, 0.00023058, 7.27289112, 0.99885028, 0.99939304, 0.49324821],
                             [- 6.27326762, -0.00037634, 0.00023055, 7.27289127, 0.99885042, 0.99939311, 0.49324821],
                             [- 6.27326773, -0.00037630, 0.00023052, 7.27289143, 0.99885056, 0.99939318, 0.49324827],
                             [- 6.27326783, -0.00037625, 0.00023049, 7.27289159, 0.99885070, 0.99939326, 0.49324820],
                             [- 6.27326794, -0.00037620, 0.00023046, 7.27289174, 0.99885084, 0.99939333, 0.49324832],
                             [- 6.27326805, -0.00037616, 0.00023044, 7.27289190, 0.99885097, 0.99939341, 0.49324834],
                             [- 6.27326816, -0.00037611, 0.00023041, 7.27289205, 0.99885111, 0.99939348, 0.49324841],
                             [- 6.27326827, -0.00037607, 0.00023038, 7.27289221, 0.99885125, 0.99939355, 0.49324834],
                             [- 6.27326838, -0.00037602, 0.00023035, 7.27289236, 0.99885139, 0.99939363, 0.49324834],
                             [- 6.27326849, -0.00037598, 0.00023032, 7.27289252, 0.99885153, 0.99939370, 0.49324836],
                             [- 6.27326860, -0.00037593, 0.00023030, 7.27289267, 0.99885167, 0.99939377, 0.49324838],
                             [- 6.27326871, -0.00037588, 0.00023027, 7.27289283, 0.99885181, 0.99939385, 0.49324843],
                             [- 6.27326882, -0.00037584, 0.00023024, 7.27289298, 0.99885195, 0.99939392, 0.49324818],
                             [- 6.27326893, -0.00037579, 0.00023021, 7.27289314, 0.99885209, 0.99939399, 0.49324852],
                             [- 6.27326904, -0.00037575, 0.00023018, 7.27289329, 0.99885223, 0.99939407, 0.49324850],
                             [- 6.27326915, -0.00037570, 0.00023016, 7.27289345, 0.99885237, 0.99939414, 0.49324851],
                             [- 6.27326926, -0.00037566, 0.00023013, 7.27289360, 0.99885251, 0.99939422, 0.49324838],
                             [- 6.27326937, -0.00037561, 0.00023010, 7.27289375, 0.99885265, 0.99939429, 0.49324847],
                             [- 6.27326947, -0.00037557, 0.00023007, 7.27289391, 0.99885279, 0.99939436, 0.49324849],
                             [- 6.27326958, -0.00037552, 0.00023004, 7.27289406, 0.99885293, 0.99939444, 0.49324851],
                             [- 6.27326969, -0.00037547, 0.00023002, 7.27289422, 0.99885307, 0.99939451, 0.49324844],
                             [- 6.27326980, -0.00037543, 0.00022999, 7.27289437, 0.99885321, 0.99939458, 0.49324862],
                             [- 6.27326991, -0.00037538, 0.00022996, 7.27289453, 0.99885334, 0.99939466, 0.49324845],
                             [- 6.27327002, -0.00037534, 0.00022993, 7.27289468, 0.99885348, 0.99939473, 0.49324866],
                             [- 6.27327013, -0.00037529, 0.00022991, 7.27289484, 0.99885362, 0.99939480, 0.49324850],
                             [- 6.27327024, -0.00037525, 0.00022988, 7.27289499, 0.99885376, 0.99939488, 0.49324874],
                             [- 6.27327035, -0.00037520, 0.00022985, 7.27289514, 0.99885390, 0.99939495, 0.49324863],
                             [- 6.27327045, -0.00037516, 0.00022982, 7.27289530, 0.99885404, 0.99939502, 0.49324861],
                             [- 6.27327056, -0.00037511, 0.00022979, 7.27289545, 0.99885418, 0.99939510, 0.49324863],
                             [- 6.27327067, -0.00037506, 0.00022977, 7.27289561, 0.99885432, 0.99939517, 0.49324878],
                             [- 6.27327078, -0.00037502, 0.00022974, 7.27289576, 0.99885446, 0.99939524, 0.49324852],
                             [- 6.27327089, -0.00037497, 0.00022971, 7.27289592, 0.99885459, 0.99939532, 0.49324868],
                             [- 6.27327100, -0.00037493, 0.00022968, 7.27289607, 0.99885473, 0.99939539, 0.49324870],
                             [- 6.27327111, -0.00037488, 0.00022965, 7.27289622, 0.99885487, 0.99939546, 0.49324867],
                             [- 6.27327122, -0.00037484, 0.00022963, 7.27289638, 0.99885501, 0.99939554, 0.49324861],
                             [- 6.27327132, -0.00037479, 0.00022960, 7.27289653, 0.99885515, 0.99939561, 0.49324879],
                             [- 6.27327143, -0.00037475, 0.00022957, 7.27289669, 0.99885529, 0.99939568, 0.49324878],
                             [- 6.27327154, -0.00037470, 0.00022954, 7.27289684, 0.99885543, 0.99939576, 0.49324873],
                             [- 6.27327165, -0.00037466, 0.00022952, 7.27289699, 0.99885557, 0.99939583, 0.49324877],
                             [- 6.27327176, -0.00037461, 0.00022949, 7.27289715, 0.99885570, 0.99939590, 0.49324893],
                             [- 6.27327187, -0.00037457, 0.00022946, 7.27289730, 0.99885584, 0.99939597, 0.49324876],
                             [- 6.27327198, -0.00037452, 0.00022943, 7.27289746, 0.99885598, 0.99939605, 0.49324876],
                             [- 6.27327208, -0.00037447, 0.00022940, 7.27289761, 0.99885612, 0.99939612, 0.49324883],
                             [- 6.27327219, -0.00037443, 0.00022938, 7.27289776, 0.99885626, 0.99939619, 0.49324897],
                             [- 6.27327230, -0.00037438, 0.00022935, 7.27289792, 0.99885640, 0.99939627, 0.49324903],
                             [- 6.27327241, -0.00037434, 0.00022932, 7.27289807, 0.99885653, 0.99939634, 0.49324897],
                             [- 6.27327252, -0.00037429, 0.00022929, 7.27289822, 0.99885667, 0.99939641, 0.49324915],
                             [- 6.27327263, -0.00037425, 0.00022927, 7.27289838, 0.99885681, 0.99939649, 0.49324899],
                             [- 6.27327273, -0.00037420, 0.00022924, 7.27289853, 0.99885695, 0.99939656, 0.49324895],
                             [- 6.27327284, -0.00037416, 0.00022921, 7.27289869, 0.99885709, 0.99939663, 0.49324901],
                             [- 6.27327295, -0.00037411, 0.00022918, 7.27289884, 0.99885723, 0.99939671, 0.49324899],
                             [- 6.27327306, -0.00037407, 0.00022915, 7.27289899, 0.99885736, 0.99939678, 0.49324910],
                             [- 6.27327317, -0.00037402, 0.00022913, 7.27289915, 0.99885750, 0.99939685, 0.49324893],
                             [- 6.27327328, -0.00037398, 0.00022910, 7.27289930, 0.99885764, 0.99939692, 0.49324897],
                             [- 6.27327338, -0.00037393, 0.00022907, 7.27289945, 0.99885778, 0.99939700, 0.49324903],
                             [- 6.27327349, -0.00037389, 0.00022904, 7.27289961, 0.99885792, 0.99939707, 0.49324907],
                             [- 6.27327360, -0.00037384, 0.00022902, 7.27289976, 0.99885805, 0.99939714, 0.49324912],
                             [- 6.27327371, -0.00037380, 0.00022899, 7.27289991, 0.99885819, 0.99939722, 0.49324911],
                             [- 6.27327382, -0.00037375, 0.00022896, 7.27290007, 0.99885833, 0.99939729, 0.49324918],
                             [- 6.27327392, -0.00037371, 0.00022893, 7.27290022, 0.99885847, 0.99939736, 0.49324911],
                             [- 6.27327403, -0.00037366, 0.00022891, 7.27290037, 0.99885861, 0.99939743, 0.49324922],
                             [- 6.27327414, -0.00037362, 0.00022888, 7.27290052, 0.99885874, 0.99939751, 0.49324912],
                             [- 6.27327425, -0.00037357, 0.00022885, 7.27290068, 0.99885888, 0.99939758, 0.49324914],
                             [- 6.27327436, -0.00037353, 0.00022882, 7.27290083, 0.99885902, 0.99939765, 0.49324919],
                             [- 6.27327446, -0.00037348, 0.00022880, 7.27290098, 0.99885916, 0.99939772, 0.49324915],
                             [- 6.27327457, -0.00037344, 0.00022877, 7.27290114, 0.99885929, 0.99939780, 0.49324920],
                             [- 6.27327468, -0.00037339, 0.00022874, 7.27290129, 0.99885943, 0.99939787, 0.49324914],
                             [- 6.27327479, -0.00037335, 0.00022871, 7.27290144, 0.99885957, 0.99939794, 0.49324927],
                             [- 6.27327490, -0.00037330, 0.00022868, 7.27290160, 0.99885971, 0.99939801, 0.49324924],
                             [- 6.27327500, -0.00037326, 0.00022866, 7.27290175, 0.99885984, 0.99939809, 0.49324926],
                             [- 6.27327511, -0.00037321, 0.00022863, 7.27290190, 0.99885998, 0.99939816, 0.49324920],
                             [- 6.27327522, -0.00037317, 0.00022860, 7.27290205, 0.99886012, 0.99939823, 0.49324932],
                             [- 6.27327533, -0.00037312, 0.00022857, 7.27290221, 0.99886026, 0.99939831, 0.49324939],
                             [- 6.27327543, -0.00037308, 0.00022855, 7.27290236, 0.99886039, 0.99939838, 0.49324946],
                             [- 6.27327554, -0.00037303, 0.00022852, 7.27290251, 0.99886053, 0.99939845, 0.49324935],
                             [- 6.27327565, -0.00037299, 0.00022849, 7.27290266, 0.99886067, 0.99939852, 0.49324930],
                             [- 6.27327576, -0.00037294, 0.00022846, 7.27290282, 0.99886081, 0.99939860, 0.49324950],
                             [- 6.27327586, -0.00037290, 0.00022844, 7.27290297, 0.99886094, 0.99939867, 0.49324938],
                             [- 6.27327597, -0.00037285, 0.00022841, 7.27290312, 0.99886108, 0.99939874, 0.49324942],
                             [- 6.27327608, -0.00037281, 0.00022838, 7.27290327, 0.99886122, 0.99939881, 0.49324944],
                             [- 6.27327619, -0.00037276, 0.00022835, 7.27290343, 0.99886136, 0.99939888, 0.49324942],
                             [- 6.27327629, -0.00037272, 0.00022833, 7.27290358, 0.99886149, 0.99939896, 0.49324940],
                             [- 6.27327640, -0.00037267, 0.00022830, 7.27290373, 0.99886163, 0.99939903, 0.49324954],
                             [- 6.27327651, -0.00037263, 0.00022827, 7.27290388, 0.99886177, 0.99939910, 0.49324936],
                             [- 6.27327662, -0.00037258, 0.00022824, 7.27290404, 0.99886190, 0.99939917, 0.49324945],
                             [- 6.27327672, -0.00037254, 0.00022822, 7.27290419, 0.99886204, 0.99939925, 0.49324953],
                             [- 6.27327683, -0.00037249, 0.00022819, 7.27290434, 0.99886218, 0.99939932, 0.49324957],
                             [- 6.27327694, -0.00037245, 0.00022816, 7.27290449, 0.99886231, 0.99939939, 0.49324952],
                             [- 6.27327705, -0.00037240, 0.00022813, 7.27290464, 0.99886245, 0.99939946, 0.49324959],
                             [- 6.27327715, -0.00037236, 0.00022811, 7.27290480, 0.99886259, 0.99939954, 0.49324957],
                             [- 6.27327726, -0.00037231, 0.00022808, 7.27290495, 0.99886273, 0.99939961, 0.49324964],
                             [- 6.27327737, -0.00037227, 0.00022805, 7.27290510, 0.99886286, 0.99939968, 0.49324963],
                             [- 6.27327747, -0.00037222, 0.00022802, 7.27290525, 0.99886300, 0.99939975, 0.49324974],
                             [- 6.27327758, -0.00037218, 0.00022800, 7.27290540, 0.99886314, 0.99939982, 0.49324966],
                             [- 6.27327769, -0.00037213, 0.00022797, 7.27290556, 0.99886327, 0.99939990, 0.49324952],
                             [- 6.27327780, -0.00037209, 0.00022794, 7.27290571, 0.99886341, 0.99939997, 0.49324969],
                             [- 6.27327790, -0.00037204, 0.00022791, 7.27290586, 0.99886355, 0.99940004, 0.49324970],
                             [- 6.27327801, -0.00037200, 0.00022789, 7.27290601, 0.99886368, 0.99940011, 0.49324976],
                             [- 6.27327812, -0.00037195, 0.00022786, 7.27290616, 0.99886382, 0.99940019, 0.49324984],
                             [- 6.27327822, -0.00037191, 0.00022783, 7.27290631, 0.99886396, 0.99940026, 0.49324963],
                             [- 6.27327833, -0.00037187, 0.00022781, 7.27290647, 0.99886409, 0.99940033, 0.49324968],
                             [- 6.27327844, -0.00037182, 0.00022778, 7.27290662, 0.99886423, 0.99940040, 0.49324978],
                             [- 6.27327855, -0.00037178, 0.00022775, 7.27290677, 0.99886436, 0.99940047, 0.49324985],
                             [- 6.27327865, -0.00037173, 0.00022772, 7.27290692, 0.99886450, 0.99940055, 0.49325000],
                             [- 6.27327876, -0.00037169, 0.00022770, 7.27290707, 0.99886464, 0.99940062, 0.49324984],
                             [- 6.27327887, -0.00037164, 0.00022767, 7.27290722, 0.99886477, 0.99940069, 0.49324982],
                             [- 6.27327897, -0.00037160, 0.00022764, 7.27290737, 0.99886491, 0.99940076, 0.49324988],
                             [- 6.27327908, -0.00037155, 0.00022761, 7.27290753, 0.99886505, 0.99940083, 0.49325015],
                             [- 6.27327919, -0.00037151, 0.00022759, 7.27290768, 0.99886518, 0.99940091, 0.49324977],
                             [- 6.27327929, -0.00037146, 0.00022756, 7.27290783, 0.99886532, 0.99940098, 0.49324993],
                             [- 6.27327940, -0.00037142, 0.00022753, 7.27290798, 0.99886546, 0.99940105, 0.49325002],
                             [- 6.27327951, -0.00037137, 0.00022750, 7.27290813, 0.99886559, 0.99940112, 0.49324999],
                             [- 6.27327961, -0.00037133, 0.00022748, 7.27290828, 0.99886573, 0.99940119, 0.49325000],
                             [- 6.27327972, -0.00037129, 0.00022745, 7.27290843, 0.99886586, 0.99940126, 0.49325006],
                             [- 6.27327983, -0.00037124, 0.00022742, 7.27290858, 0.99886600, 0.99940134, 0.49324994],
                             [- 6.27327993, -0.00037120, 0.00022740, 7.27290874, 0.99886614, 0.99940141, 0.49324986],
                             [- 6.27328004, -0.00037115, 0.00022737, 7.27290889, 0.99886627, 0.99940148, 0.49325009],
                             [- 6.27328015, -0.00037111, 0.00022734, 7.27290904, 0.99886641, 0.99940155, 0.49325012],
                             [- 6.27328025, -0.00037106, 0.00022731, 7.27290919, 0.99886654, 0.99940162, 0.49325015],
                             [- 6.27328036, -0.00037102, 0.00022729, 7.27290934, 0.99886668, 0.99940170, 0.49325010],
                             [- 6.27328047, -0.00037097, 0.00022726, 7.27290949, 0.99886682, 0.99940177, 0.49325020],
                             [- 6.27328057, -0.00037093, 0.00022723, 7.27290964, 0.99886695, 0.99940184, 0.49325004],
                             [- 6.27328068, -0.00037089, 0.00022720, 7.27290979, 0.99886709, 0.99940191, 0.49325014],
                             [- 6.27328078, -0.00037084, 0.00022718, 7.27290994, 0.99886722, 0.99940198, 0.49325025],
                             [- 6.27328089, -0.00037080, 0.00022715, 7.27291009, 0.99886736, 0.99940205, 0.49325008],
                             [- 6.27328100, -0.00037075, 0.00022712, 7.27291024, 0.99886749, 0.99940213, 0.49325032],
                             [- 6.27328110, -0.00037071, 0.00022710, 7.27291040, 0.99886763, 0.99940220, 0.49325016],
                             [- 6.27328121, -0.00037066, 0.00022707, 7.27291055, 0.99886777, 0.99940227, 0.49325010],
                             [- 6.27328132, -0.00037062, 0.00022704, 7.27291070, 0.99886790, 0.99940234, 0.49325006],
                             [- 6.27328142, -0.00037057, 0.00022701, 7.27291085, 0.99886804, 0.99940241, 0.49325022],
                             [- 6.27328153, -0.00037053, 0.00022699, 7.27291100, 0.99886817, 0.99940248, 0.49325031],
                             [- 6.27328163, -0.00037049, 0.00022696, 7.27291115, 0.99886831, 0.99940255, 0.49325037],
                             [- 6.27328174, -0.00037044, 0.00022693, 7.27291130, 0.99886844, 0.99940263, 0.49325027],
                             [- 6.27328185, -0.00037040, 0.00022691, 7.27291145, 0.99886858, 0.99940270, 0.49325034],
                             [- 6.27328195, -0.00037035, 0.00022688, 7.27291160, 0.99886871, 0.99940277, 0.49325020],
                             [- 6.27328206, -0.00037031, 0.00022685, 7.27291175, 0.99886885, 0.99940284, 0.49325035],
                             [- 6.27328217, -0.00037026, 0.00022682, 7.27291190, 0.99886898, 0.99940291, 0.49325022],
                             [- 6.27328227, -0.00037022, 0.00022680, 7.27291205, 0.99886912, 0.99940298, 0.49325038],
                             [- 6.27328238, -0.00037018, 0.00022677, 7.27291220, 0.99886926, 0.99940305, 0.49325045],
                             [- 6.27328248, -0.00037013, 0.00022674, 7.27291235, 0.99886939, 0.99940313, 0.49325055],
                             [- 6.27328259, -0.00037009, 0.00022672, 7.27291250, 0.99886953, 0.99940320, 0.49325042],
                             [- 6.27328269, -0.00037004, 0.00022669, 7.27291265, 0.99886966, 0.99940327, 0.49325036],
                             [- 6.27328280, -0.00037000, 0.00022666, 7.27291280, 0.99886980, 0.99940334, 0.49325057],
                             [- 6.27328291, -0.00036995, 0.00022663, 7.27291295, 0.99886993, 0.99940341, 0.49325047],
                             [- 6.27328301, -0.00036991, 0.00022661, 7.27291310, 0.99887007, 0.99940348, 0.49325050],
                             [- 6.27328312, -0.00036987, 0.00022658, 7.27291325, 0.99887020, 0.99940355, 0.49325044],
                             [- 6.27328322, -0.00036982, 0.00022655, 7.27291340, 0.99887034, 0.99940363, 0.49325060],
                             [- 6.27328333, -0.00036978, 0.00022653, 7.27291355, 0.99887047, 0.99940370, 0.49325046],
                             [- 6.27328344, -0.00036973, 0.00022650, 7.27291370, 0.99887061, 0.99940377, 0.49325049],
                             [- 6.27328354, -0.00036969, 0.00022647, 7.27291385, 0.99887074, 0.99940384, 0.49325054],
                             [- 6.27328365, -0.00036965, 0.00022644, 7.27291400, 0.99887088, 0.99940391, 0.49325051],
                             [- 6.27328375, -0.00036960, 0.00022642, 7.27291415, 0.99887101, 0.99940398, 0.49325068],
                             [- 6.27328386, -0.00036956, 0.00022639, 7.27291430, 0.99887115, 0.99940405, 0.49325059],
                             [- 6.27328396, -0.00036951, 0.00022636, 7.27291445, 0.99887128, 0.99940412, 0.49325082],
                             [- 6.27328407, -0.00036947, 0.00022634, 7.27291460, 0.99887142, 0.99940419, 0.49325077],
                             [- 6.27328418, -0.00036942, 0.00022631, 7.27291475, 0.99887155, 0.99940427, 0.49325076],
                             [- 6.27328428, -0.00036938, 0.00022628, 7.27291490, 0.99887168, 0.99940434, 0.49325073],
                             [- 6.27328439, -0.00036934, 0.00022626, 7.27291505, 0.99887182, 0.99940441, 0.49325073],
                             [- 6.27328449, -0.00036929, 0.00022623, 7.27291520, 0.99887195, 0.99940448, 0.49325059],
                             [- 6.27328460, -0.00036925, 0.00022620, 7.27291535, 0.99887209, 0.99940455, 0.49325071],
                             [- 6.27328470, -0.00036920, 0.00022617, 7.27291550, 0.99887222, 0.99940462, 0.49325075],
                             [- 6.27328481, -0.00036916, 0.00022615, 7.27291565, 0.99887236, 0.99940469, 0.49325074],
                             [- 6.27328491, -0.00036912, 0.00022612, 7.27291580, 0.99887249, 0.99940476, 0.49325076],
                             [- 6.27328502, -0.00036907, 0.00022609, 7.27291595, 0.99887263, 0.99940483, 0.49325069],
                             [- 6.27328512, -0.00036903, 0.00022607, 7.27291610, 0.99887276, 0.99940491, 0.49325089],
                             [- 6.27328523, -0.00036898, 0.00022604, 7.27291624, 0.99887290, 0.99940498, 0.49325087],
                             [- 6.27328533, -0.00036894, 0.00022601, 7.27291639, 0.99887303, 0.99940505, 0.49325089],
                             [- 6.27328544, -0.00036890, 0.00022599, 7.27291654, 0.99887316, 0.99940512, 0.49325087],
                             [- 6.27328555, -0.00036885, 0.00022596, 7.27291669, 0.99887330, 0.99940519, 0.49325085],
                             [- 6.27328565, -0.00036881, 0.00022593, 7.27291684, 0.99887343, 0.99940526, 0.49325080],
                             [- 6.27328576, -0.00036876, 0.00022590, 7.27291699, 0.99887357, 0.99940533, 0.49325071],
                             [- 6.27328586, -0.00036872, 0.00022588, 7.27291714, 0.99887370, 0.99940540, 0.49325095],
                             [- 6.27328597, -0.00036868, 0.00022585, 7.27291729, 0.99887384, 0.99940547, 0.49325080],
                             [- 6.27328607, -0.00036863, 0.00022582, 7.27291744, 0.99887397, 0.99940554, 0.49325088],
                             [- 6.27328618, -0.00036859, 0.00022580, 7.27291759, 0.99887410, 0.99940561, 0.49325100],
                             [- 6.27328628, -0.00036855, 0.00022577, 7.27291774, 0.99887424, 0.99940568, 0.49325106],
                             [- 6.27328639, -0.00036850, 0.00022574, 7.27291789, 0.99887437, 0.99940576, 0.49325085],
                             [- 6.27328649, -0.00036846, 0.00022572, 7.27291803, 0.99887451, 0.99940583, 0.49325087],
                             [- 6.27328660, -0.00036841, 0.00022569, 7.27291818, 0.99887464, 0.99940590, 0.49325084],
                             [- 6.27328670, -0.00036837, 0.00022566, 7.27291833, 0.99887477, 0.99940597, 0.49325118],
                             [- 6.27328681, -0.00036833, 0.00022564, 7.27291848, 0.99887491, 0.99940604, 0.49325112],
                             [- 6.27328691, -0.00036828, 0.00022561, 7.27291863, 0.99887504, 0.99940611, 0.49325104],
                             [- 6.27328702, -0.00036824, 0.00022558, 7.27291878, 0.99887518, 0.99940618, 0.49325104],
                             [- 6.27328712, -0.00036819, 0.00022556, 7.27291893, 0.99887531, 0.99940625, 0.49325116],
                             [- 6.27328723, -0.00036815, 0.00022553, 7.27291907, 0.99887544, 0.99940632, 0.49325111],
                             [- 6.27328733, -0.00036811, 0.00022550, 7.27291922, 0.99887558, 0.99940639, 0.49325116],
                             [- 6.27328744, -0.00036806, 0.00022547, 7.27291937, 0.99887571, 0.99940646, 0.49325108],
                             [- 6.27328754, -0.00036802, 0.00022545, 7.27291952, 0.99887584, 0.99940653, 0.49325111],
                             [- 6.27328764, -0.00036798, 0.00022542, 7.27291967, 0.99887598, 0.99940660, 0.49325113],
                             [- 6.27328775, -0.00036793, 0.00022539, 7.27291982, 0.99887611, 0.99940667, 0.49325130],
                             [- 6.27328785, -0.00036789, 0.00022537, 7.27291997, 0.99887625, 0.99940674, 0.49325114],
                             [- 6.27328796, -0.00036784, 0.00022534, 7.27292011, 0.99887638, 0.99940681, 0.49325125],
                             [- 6.27328806, -0.00036780, 0.00022531, 7.27292026, 0.99887651, 0.99940689, 0.49325120],
                             [- 6.27328817, -0.00036776, 0.00022529, 7.27292041, 0.99887665, 0.99940696, 0.49325118],
                             [- 6.27328827, -0.00036771, 0.00022526, 7.27292056, 0.99887678, 0.99940703, 0.49325120],
                             [- 6.27328838, -0.00036767, 0.00022523, 7.27292071, 0.99887691, 0.99940710, 0.49325135],
                             [- 6.27328848, -0.00036763, 0.00022521, 7.27292086, 0.99887705, 0.99940717, 0.49325134],
                             [- 6.27328859, -0.00036758, 0.00022518, 7.27292100, 0.99887718, 0.99940724, 0.49325139],
                             [- 6.27328869, -0.00036754, 0.00022515, 7.27292115, 0.99887731, 0.99940731, 0.49325135],
                             [- 6.27328880, -0.00036749, 0.00022513, 7.27292130, 0.99887745, 0.99940738, 0.49325126],
                             [- 6.27328890, -0.00036745, 0.00022510, 7.27292145, 0.99887758, 0.99940745, 0.49325131],
                             [- 6.27328900, -0.00036741, 0.00022507, 7.27292160, 0.99887771, 0.99940752, 0.49325136],
                             [- 6.27328911, -0.00036736, 0.00022505, 7.27292174, 0.99887785, 0.99940759, 0.49325152],
                             [- 6.27328921, -0.00036732, 0.00022502, 7.27292189, 0.99887798, 0.99940766, 0.49325159],
                             [- 6.27328932, -0.00036728, 0.00022499, 7.27292204, 0.99887811, 0.99940773, 0.49325140],
                             [- 6.27328942, -0.00036723, 0.00022497, 7.27292219, 0.99887825, 0.99940780, 0.49325141],
                             [- 6.27328953, -0.00036719, 0.00022494, 7.27292234, 0.99887838, 0.99940787, 0.49325131],
                             [- 6.27328963, -0.00036715, 0.00022491, 7.27292248, 0.99887851, 0.99940794, 0.49325145],
                             [- 6.27328973, -0.00036710, 0.00022489, 7.27292263, 0.99887864, 0.99940801, 0.49325145],
                             [- 6.27328984, -0.00036706, 0.00022486, 7.27292278, 0.99887878, 0.99940808, 0.49325157],
                             [- 6.27328994, -0.00036702, 0.00022483, 7.27292293, 0.99887891, 0.99940815, 0.49325149],
                             [- 6.27329005, -0.00036697, 0.00022481, 7.27292307, 0.99887904, 0.99940822, 0.49325164],
                             [- 6.27329015, -0.00036693, 0.00022478, 7.27292322, 0.99887918, 0.99940829, 0.49325155],
                             [- 6.27329025, -0.00036689, 0.00022475, 7.27292337, 0.99887931, 0.99940836, 0.49325164],
                             [- 6.27329036, -0.00036684, 0.00022473, 7.27292352, 0.99887944, 0.99940843, 0.49325151],
                             [- 6.27329046, -0.00036680, 0.00022470, 7.27292366, 0.99887958, 0.99940850, 0.49325168],
                             [- 6.27329057, -0.00036675, 0.00022467, 7.27292381, 0.99887971, 0.99940857, 0.49325138],
                             [- 6.27329067, -0.00036671, 0.00022465, 7.27292396, 0.99887984, 0.99940864, 0.49325171],
                             [- 6.27329078, -0.00036667, 0.00022462, 7.27292411, 0.99887997, 0.99940871, 0.49325159],
                             [- 6.27329088, -0.00036662, 0.00022459, 7.27292425, 0.99888011, 0.99940878, 0.49325160],
                             [- 6.27329098, -0.00036658, 0.00022457, 7.27292440, 0.99888024, 0.99940885, 0.49325177],
                             [- 6.27329109, -0.00036654, 0.00022454, 7.27292455, 0.99888037, 0.99940892, 0.49325177],
                             [- 6.27329119, -0.00036649, 0.00022451, 7.27292470, 0.99888050, 0.99940899, 0.49325164],
                             [- 6.27329129, -0.00036645, 0.00022449, 7.27292484, 0.99888064, 0.99940906, 0.49325172],
                             [- 6.27329140, -0.00036641, 0.00022446, 7.27292499, 0.99888077, 0.99940913, 0.49325179],
                             [- 6.27329150, -0.00036636, 0.00022443, 7.27292514, 0.99888090, 0.99940920, 0.49325181],
                             [- 6.27329161, -0.00036632, 0.00022441, 7.27292529, 0.99888103, 0.99940927, 0.49325178],
                             [- 6.27329171, -0.00036628, 0.00022438, 7.27292543, 0.99888117, 0.99940934, 0.49325178],
                             [- 6.27329181, -0.00036623, 0.00022435, 7.27292558, 0.99888130, 0.99940941, 0.49325183],
                             [- 6.27329192, -0.00036619, 0.00022433, 7.27292573, 0.99888143, 0.99940948, 0.49325179],
                             [- 6.27329202, -0.00036615, 0.00022430, 7.27292587, 0.99888156, 0.99940955, 0.49325183],
                             [- 6.27329212, -0.00036610, 0.00022427, 7.27292602, 0.99888170, 0.99940962, 0.49325178],
                             [- 6.27329223, -0.00036606, 0.00022425, 7.27292617, 0.99888183, 0.99940969, 0.49325167],
                             [- 6.27329233, -0.00036602, 0.00022422, 7.27292631, 0.99888196, 0.99940976, 0.49325180],
                             [- 6.27329244, -0.00036597, 0.00022419, 7.27292646, 0.99888209, 0.99940983, 0.49325201],
                             [- 6.27329254, -0.00036593, 0.00022417, 7.27292661, 0.99888222, 0.99940990, 0.49325190],
                             [- 6.27329264, -0.00036589, 0.00022414, 7.27292675, 0.99888236, 0.99940997, 0.49325201],
                             [- 6.27329275, -0.00036584, 0.00022412, 7.27292690, 0.99888249, 0.99941004, 0.49325203],
                             [- 6.27329285, -0.00036580, 0.00022409, 7.27292705, 0.99888262, 0.99941011, 0.49325195],
                             [- 6.27329295, -0.00036576, 0.00022406, 7.27292720, 0.99888275, 0.99941018, 0.49325194],
                             [- 6.27329306, -0.00036571, 0.00022404, 7.27292734, 0.99888289, 0.99941025, 0.49325187],
                             [- 6.27329316, -0.00036567, 0.00022401, 7.27292749, 0.99888302, 0.99941032, 0.49325203],
                             [- 6.27329326, -0.00036563, 0.00022398, 7.27292764, 0.99888315, 0.99941039, 0.49325193],
                             [- 6