function rigidity = nye(temperature,ice_type)
%NYE - Nye viscosity coefficient
%
%   Compute rigidity of ice (either CO2 or H2O) for a given temperature
%   rigidity (in s^(1/3)Pa) is the flow law parameter in the flow law
%   sigma=B*e(1/3) (Nye, p2000).  temperature is in Kelvin degrees
%
%   Usage:
%      rigidity=nye(temperature,ice_type) % ice_type = 1: CO2 ice // ice_type = 2: H2O ice
	if ((ice_type == 1) && (any(temperature > 195)))
		error('Input temperature for CO2 ice out of bounds (T>195)');
	elseif ((ice_type == 2) && (any(temperature > 273.15)))
		error('Input temperature for H2O ice out of bounds (T>273.15)');
	end

	Rg = 8.3144598; % J mol^-1 K^-1

	if(ice_type==1) %CO2 ice
		A_const = 10^(10.8); % s^-1 MPa
		Q       = 63000;     % J mol^-1
		n       = 7;         % Glen's exponent

	elseif(ice_type==2) % H2O ice
		A_const = 9e4;   % s^-1 MPa
		Q       = 60000; % J mol^-1
		n       = 3;     % Glen's exponent

	else
		error('ice type not supported');
	end

	%Arhenius law
	A = A_const*exp(-Q./(temperature*Rg)); % s^-1 MPa
	rigidity = A.^(-1/n)*1e6; % s^(1/n) Pa

end
