/*!\file KillIcebergsx
 * \brief: compute inverse method gradient
 */

#include "./KillIcebergsx.h"
#include "../../shared/shared.h"
#include "../../toolkits/toolkits.h"

void KillIcebergsx(FemModel* femmodel){

	/*Intermediaries*/
	IssmDouble* local_mask = NULL;
	const int MAXVERTICES = 6;
	bool      found1;
	int       sidlist[MAXVERTICES];
	int       lidlist[MAXVERTICES];

	/*retrieve vertex info*/
	int nbv_global = femmodel->vertices->NumberOfVerticesLocalAll();
	int nbv_local  = femmodel->vertices->NumberOfVerticesLocal();
	if(nbv_global==0)  return;
	Vector<IssmDouble>* vec_connected_to_land=new Vector<IssmDouble>(nbv_local,nbv_global);

	/*Prepare element flag to speed up process*/
	bool* element_flag = xNewZeroInit<bool>(femmodel->elements->Size());

	/*Fill vector with 1 once for all*/
	IssmDouble eflags[MAXVERTICES];
	for(int i=0;i<MAXVERTICES;i++) eflags[i] = 1.;

	/*Step 1, go through all elements and put 1 in vec_connected_to_land if the element is grounded*/
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));

		if(!element->IsIceInElement()){
			/*Nothing to do, just flag element to speed up the computation*/
			element_flag[i] = true;
		}
		else{
			if(element->IsGrounded()){
				int  numvertices = element->GetNumberOfVertices();
				if(numvertices>MAXVERTICES) _error_("need to increase MAXVERTICES");
				element->GetVerticesSidList(&sidlist[0]);
				vec_connected_to_land->SetValues(numvertices,&sidlist[0],&eflags[0],ADD_VAL);
			}
		}
	}
	vec_connected_to_land->Assemble();

	/*Now we have 2 loops, one across cpus, and one for each cpus: we are going to propagate the mask if an element
	 * is connected to a positive mask already.
	 * We then communicate to the other partitions. We stop when the mask stops changing*/
	bool keepsyncing = true;
	while(keepsyncing){

		/*Get local mask from parallel vector*/
		if(local_mask) xDelete<IssmDouble>(local_mask);
		femmodel->GetLocalVectorWithClonesVertices(&local_mask,vec_connected_to_land);

		/*Local iterations on partition*/
		int  keepgoing = 1;
		int  iter      = 1;
		while(keepgoing){
			_printf0_("   -- Kill icebergs: iteration "<<iter);

			keepgoing = 0;
			for(int i=0;i<femmodel->elements->Size();i++){
				Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));

				if(!element_flag[i]){
					int numvertices = element->GetNumberOfVertices();
					element->GetVerticesLidList(&lidlist[0]);
					bool found1 = false;
					for(int j=0;j<numvertices;j++){
						if(local_mask[lidlist[j]]>0.){
							found1 = true;
							break;
						}
					}
					if(found1){
						element_flag[i] = true;
						for(int j=0;j<numvertices;j++){
							if(local_mask[lidlist[j]]==0.){
								local_mask[lidlist[j]]=1.;
								keepgoing = true;
							}
						}
					}
				}
			}
			iter++;
		}

		/*Check how many iterations all cpus did*/
		int iter_max;
		ISSM_MPI_Reduce(&iter,&iter_max,1,ISSM_MPI_INT,ISSM_MPI_MAX,0,IssmComm::GetComm());
		if(iter_max==2){
			keepsyncing = false;
		}
		else{
			for(int i=0;i<femmodel->elements->Size();i++){
				Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));

				if(element->IsIceInElement()){
						int  numvertices = element->GetNumberOfVertices();
						element->GetVerticesSidList(&sidlist[0]);
						element->GetVerticesLidList(&lidlist[0]);
						for(int j=0;j<numvertices;j++) eflags[j] = local_mask[lidlist[j]];
						vec_connected_to_land->SetValues(numvertices,&sidlist[0],&eflags[0],ADD_VAL);
					}
			}
			vec_connected_to_land->Assemble();
		}
	}

	/*Cleanup*/
	xDelete<bool>(element_flag);
	delete vec_connected_to_land;

	/*OK, now deactivate iceberg and count the number of deactivated vertices*/
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));

		if(element->IsIceInElement()){
			int  numvertices = element->GetNumberOfVertices();
			element->GetVerticesLidList(&lidlist[0]);
			bool deactivate = false;
			for(int j=0;j<numvertices;j++){
				if(local_mask[lidlist[j]]==0.){
					deactivate = true;
					break;
				}
			}

			if(deactivate){
				_error_("Don't know what to do....");
			}
		}
	}

	/*cleanup*/
	xDelete<IssmDouble>(local_mask);
}
