/*!\file Channel.c
 * \brief: implementation of the Channel object
 */

/*Headers:*/
/*{{{*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "shared/shared.h"
#include "../classes.h"
/*}}}*/	
#define NUMNODES 2

/*Channel constructors and destructor*/
Channel::Channel(){/*{{{*/
	this->parameters = NULL;
	this->helement   = NULL;
	this->element    = NULL;
	this->hnodes     = NULL;
	this->hvertices  = NULL;
	this->nodes      = NULL;
}
/*}}}*/
Channel::Channel(int channel_id,int i,int index,IoModel* iomodel){/*{{{*/

	this->id=channel_id;
	this->parameters = NULL;
	this->element    = NULL;
	this->nodes      = NULL;

	/*Get edge info*/
	int i1 = iomodel->faces[4*index+0];
	int i2 = iomodel->faces[4*index+1];
	int e1 = iomodel->faces[4*index+2];
	int e2 = iomodel->faces[4*index+3];

	/*Set Element hook (4th column may be -1 for boundary edges)*/
	this->helement  = new Hook(&e1,1);

	/*Set Vertices hooks (4th column may be -1 for boundary edges)*/
	int channel_vertex_ids[2];
	channel_vertex_ids[0]=i1;
	channel_vertex_ids[1]=i2;
	this->hvertices =new Hook(&channel_vertex_ids[0],2);

	/*Set Nodes hooks (!! Assumes P1 CG)*/
	int channel_node_ids[2];
	channel_node_ids[0]=i1;
	channel_node_ids[1]=i2;
	this->hnodes=new Hook(&channel_node_ids[0],2);

}/*}}}*/
Channel::~Channel(){/*{{{*/
	this->parameters=NULL;
	delete helement;
	delete hnodes;
	delete hvertices;
}/*}}}*/

/*Object virtual functions definitions:*/
Object* Channel::copy() {/*{{{*/

	Channel* channel=NULL;

	channel=new Channel();

	/*copy fields: */
	channel->id=this->id;

	/*point parameters: */
	channel->parameters=this->parameters;

	/*now deal with hooks and objects: */
	channel->hnodes    = (Hook*)this->hnodes->copy();
	channel->hvertices = (Hook*)this->hvertices->copy();
	channel->helement  = (Hook*)this->helement->copy();

	/*corresponding fields*/
	channel->nodes    = (Node**)channel->hnodes->deliverp();
	channel->vertices = (Vertex**)channel->hvertices->deliverp();
	channel->element  = (Element*)channel->helement->delivers();

	return channel;
}
/*}}}*/
void    Channel::DeepEcho(void){/*{{{*/

	_printf_("Channel:\n");
	_printf_("   id: " << id << "\n");
	hnodes->DeepEcho();
	hvertices->DeepEcho();
	helement->DeepEcho();
	_printf_("   parameters\n");
	if(parameters)
	 parameters->DeepEcho();
	else
	 _printf_("      NULL\n");
}		
/*}}}*/
void    Channel::Echo(void){/*{{{*/
	_printf_("Channel:\n");
	_printf_("   id: " << id << "\n");
	hnodes->Echo();
	hvertices->Echo();
	helement->Echo();
	_printf_("   parameters: " << parameters << "\n");
}
/*}}}*/
int     Channel::Id(void){/*{{{*/
	return id;
}
/*}}}*/
void    Channel::Marshall(char** pmarshalled_data,int* pmarshalled_data_size, int marshall_direction){ /*{{{*/

	_assert_(this);

	/*ok, marshall operations: */
	MARSHALLING_ENUM(ChannelEnum);
	MARSHALLING(id);

	if(marshall_direction==MARSHALLING_BACKWARD){
		this->hnodes      = new Hook();
		this->hvertices   = new Hook();
		this->helement    = new Hook();
	}

	this->hnodes->Marshall(pmarshalled_data,pmarshalled_data_size,marshall_direction);
	this->helement->Marshall(pmarshalled_data,pmarshalled_data_size,marshall_direction);
	this->hvertices->Marshall(pmarshalled_data,pmarshalled_data_size,marshall_direction);

	/*corresponding fields*/
	nodes    =(Node**)this->hnodes->deliverp();
	vertices =(Vertex**)this->hvertices->deliverp();
	element  =(Element*)this->helement->delivers();

}
/*}}}*/
int     Channel::ObjectEnum(void){/*{{{*/

	return ChannelEnum;

}
/*}}}*/

/*Load virtual functions definitions:*/
void  Channel::Configure(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){/*{{{*/

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnodes->configure((DataSet*)nodesin);
	hvertices->configure((DataSet*)verticesin);
	helement->configure((DataSet*)elementsin);

	/*Initialize hooked fields*/
	this->nodes    = (Node**)hnodes->deliverp();
	this->vertices = (Vertex**)hvertices->deliverp();
	this->element  = (Element*)helement->delivers();

	/*point parameters to real dataset: */
	this->parameters=parametersin;
}
/*}}}*/
void  Channel::CreateKMatrix(Matrix<IssmDouble>* Kff, Matrix<IssmDouble>* Kfs){/*{{{*/

	/*recover some parameters*/
	ElementMatrix* Ke=NULL;
	int analysis_type;
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	switch(analysis_type){
		case HydrologyGlaDSAnalysisEnum:
			Ke = this->CreateKMatrixHydrologyGlaDS();
			break;
		default:
			_error_("Don't know why we should be here");
	}

	/*Add to global matrix*/
	if(Ke){
		Ke->AddToGlobal(Kff,Kfs);
		delete Ke;
	}

}
/*}}}*/
void  Channel::CreatePVector(Vector<IssmDouble>* pf){/*{{{*/

	/*recover some parameters*/
	ElementVector* pe=NULL;
	int analysis_type;
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	switch(analysis_type){
		case HydrologyGlaDSAnalysisEnum:
			pe = this->CreatePVectorHydrologyGlaDS();
			break;
		default:
			_error_("Don't know why we should be here");
	}

	/*Add to global matrix*/
	if(pe){
		pe->AddToGlobal(pf);
		delete pe;
	}

}
/*}}}*/
void  Channel::GetNodesLidList(int* lidlist){/*{{{*/

	_assert_(lidlist);
	_assert_(nodes);

	for(int i=0;i<NUMNODES;i++) lidlist[i]=nodes[i]->Lid();
}
/*}}}*/
void  Channel::GetNodesSidList(int* sidlist){/*{{{*/

	_assert_(sidlist);
	_assert_(nodes);

	for(int i=0;i<NUMNODES;i++) sidlist[i]=nodes[i]->Sid();
}
/*}}}*/
int   Channel::GetNumberOfNodes(void){/*{{{*/
	return NUMNODES;
}
/*}}}*/
bool  Channel::IsPenalty(void){/*{{{*/
	return false;
}
/*}}}*/
void  Channel::PenaltyCreateKMatrix(Matrix<IssmDouble>* Kff, Matrix<IssmDouble>* Kfs,IssmDouble kmax){/*{{{*/

	/*No stiffness loads applied, do nothing: */
	return;

}
/*}}}*/
void  Channel::PenaltyCreatePVector(Vector<IssmDouble>* pf,IssmDouble kmax){/*{{{*/

	/*No penalty loads applied, do nothing: */
	return;

}
/*}}}*/
void  Channel::ResetHooks(){/*{{{*/

	this->nodes=NULL;
	this->vertices=NULL;
	this->element=NULL;
	this->parameters=NULL;

	/*Get Element type*/
	this->hnodes->reset();
	this->hvertices->reset();
	this->helement->reset();

}
/*}}}*/
void  Channel::SetCurrentConfiguration(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){/*{{{*/

}
/*}}}*/
void  Channel::SetwiseNodeConnectivity(int* pd_nz,int* po_nz,Node* node,bool* flags,int* flagsindices,int set1_enum,int set2_enum){/*{{{*/

	/*Output */
	int d_nz = 0;
	int o_nz = 0;

	/*Loop over all nodes*/
	for(int i=0;i<this->GetNumberOfNodes();i++){

		if(!flags[this->nodes[i]->Lid()]){

			/*flag current node so that no other element processes it*/
			flags[this->nodes[i]->Lid()]=true;

			int counter=0;
			while(flagsindices[counter]>=0) counter++;
			flagsindices[counter]=this->nodes[i]->Lid();

			/*if node is clone, we have an off-diagonal non-zero, else it is a diagonal non-zero*/
			switch(set2_enum){
				case FsetEnum:
					if(nodes[i]->fsize){
						if(this->nodes[i]->IsClone())
						 o_nz += 1;
						else
						 d_nz += 1;
					}
					break;
				case GsetEnum:
					if(nodes[i]->gsize){
						if(this->nodes[i]->IsClone())
						 o_nz += 1;
						else
						 d_nz += 1;
					}
					break;
				case SsetEnum:
					if(nodes[i]->ssize){
						if(this->nodes[i]->IsClone())
						 o_nz += 1;
						else
						 d_nz += 1;
					}
					break;
				default: _error_("not supported");
			}
		}
	}

	/*Assign output pointers: */
	*pd_nz=d_nz;
	*po_nz=o_nz;
}
/*}}}*/

/*Channel specific functions*/
ElementVector* Channel::CreatePVectorHydrologyGlaDS(void){/*{{{*/

	_error_("not implemented :( ");

	/*Initialize Element matrix and return if necessary*/
	Tria*  tria=(Tria*)element;
	if(!tria->IsIceInElement()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMNODES,this->parameters);

	/*Clean up and return*/
	return pe;
}
/*}}}*/
ElementMatrix* Channel::CreateKMatrixHydrologyGlaDS(void){/*{{{*/

	_error_("not implemented :( ");

	/*Initialize Element matrix and return if necessary*/
	Tria*  tria=(Tria*)element;
	if(!tria->IsIceInElement()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMNODES,this->parameters);

	/*Clean up and return*/
	return Ke;
}
/*}}}*/
