/*!\file: hydrology_core.cpp
 * \brief: core of the hydrology solution
 */

#include "./cores.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"
#include "../solutionsequences/solutionsequences.h"

void hydrology_core(FemModel* femmodel){

	/*Start profiler*/
	femmodel->profiler->Start(HYDROLOGYCORE);

	/*intermediary*/
	int          hydrology_model;
	int          solution_type;
	int          numoutputs        = 0;
	int          smboutputs;
	bool         save_results;
	bool         modify_loads      = true;
	bool         issmb;
	char       **requested_outputs = NULL;
	char       **requested_smb_outputs = NULL;
	IssmDouble   ThawedNodes;

	/*first recover parameters common to all solutions*/
	femmodel->parameters->FindParam(&save_results,SaveResultsEnum);
	femmodel->parameters->FindParam(&hydrology_model,HydrologyModelEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
	femmodel->parameters->FindParam(&numoutputs,HydrologyNumRequestedOutputsEnum);
	if(numoutputs) femmodel->parameters->FindParam(&requested_outputs,&numoutputs,HydrologyRequestedOutputsEnum);
	femmodel->parameters->FindParam(&issmb,TransientIssmbEnum);

	/*Using the Shreve based Model*/
	if (hydrology_model==HydrologyshreveEnum){
		if(VerboseSolution()) _printf0_("   computing water heads\n");
		/*first compute slopes: */
		surfaceslope_core(femmodel);
		bedslope_core(femmodel);
		/*and then go to water column*/
		if(VerboseSolution()) _printf0_("   computing water column\n");
		femmodel->SetCurrentConfiguration(HydrologyShreveAnalysisEnum);
		solutionsequence_nonlinear(femmodel,modify_loads);
		/*transfer water column thickness to old water column thickness: */
		InputDuplicatex(femmodel,WatercolumnEnum,WaterColumnOldEnum);

	}

	/*Using the double continuum model*/
	else if (hydrology_model==HydrologydcEnum){
		/*intermediary: */
		bool       isefficientlayer;
		int        hydrostep,hydroslices,numaveragedinput;
		IssmDouble time,hydrotime,yts;
		IssmDouble dt,hydrodt;
		/*SMB related */
		int    smb_model;

		/*recover parameters: */
		femmodel->parameters->FindParam(&isefficientlayer,HydrologydcIsefficientlayerEnum);
		femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
		femmodel->parameters->FindParam(&time,TimeEnum);
		femmodel->parameters->FindParam(&hydroslices,HydrologyStepsPerStepEnum);
		femmodel->parameters->FindParam(&yts,ConstantsYtsEnum);

		/*recover SMB related parameters: */
		if(issmb){
			femmodel->parameters->FindParam(&smb_model,SmbEnum);
			femmodel->parameters->FindParam(&smboutputs,SmbNumRequestedOutputsEnum);
			if(smboutputs) femmodel->parameters->FindParam(&requested_smb_outputs,&smboutputs,SmbRequestedOutputsEnum);
		}

		/*first we exclude frozen nodes of the solved nodes*/
		femmodel->SetCurrentConfiguration(HydrologyDCInefficientAnalysisEnum);
		femmodel->HydrologyIDSupdateDomainx(&ThawedNodes);

		if(ThawedNodes>0){
			hydrotime=time-dt; //getting the time back to the start of the timestep
			hydrodt=dt/hydroslices; //computing hydro dt from dt and a divider
			hydrostep=0;
			femmodel->parameters->AddObject(new DoubleParam(HydrologydtEnum,hydrodt));

			if(hydroslices>1){
				if (isefficientlayer){
					/*define which variable needs to be averaged on the sub-timestep and initialize as needed*/
					numaveragedinput = 4;
					int inputtostack[4]	=	{EffectivePressureHydrostepEnum,SedimentHeadHydrostepEnum,EplHeadHydrostepEnum,HydrologydcEplThicknessHydrostepEnum};
					int stackedinput[4]	=	{EffectivePressureStackedEnum,SedimentHeadStackedEnum,EplHeadStackedEnum,HydrologydcEplThicknessStackedEnum};
					int averagedinput[4]	=	{EffectivePressureEnum,SedimentHeadEnum,EplHeadEnum,HydrologydcEplThicknessEnum};
					femmodel->InitMeanOutputx(&stackedinput[0],numaveragedinput);

					//while(hydrotime<time-(yts*DBL_EPSILON)){ //loop on hydro dts
					while(hydrostep<hydroslices){ //loop on hydro dts
						hydrostep+=1;
						hydrotime+=hydrodt;
						/*Setting substep time as global time*/
						femmodel->parameters->SetParam(hydrotime,TimeEnum);
						if(VerboseSolution()) _printf0_("sub iteration " << hydrostep << "/" << hydroslices << "  time [yr]: " << setprecision(4) << hydrotime/yts << " (time step: " << hydrodt/yts << ")\n");
						if(issmb){
							if(VerboseSolution()) _printf0_("   computing mass balance\n");
							SmbAnalysis* analysis = new SmbAnalysis();
							analysis->Core(femmodel);
							delete analysis;
						}
						if(VerboseSolution()) _printf0_("   computing water heads\n");
						/*save preceding timestep*/
						InputDuplicatex(femmodel,SedimentHeadHydrostepEnum,SedimentHeadOldEnum);
						InputDuplicatex(femmodel,EplHeadHydrostepEnum,EplHeadOldEnum);
						InputDuplicatex(femmodel,HydrologydcEplThicknessHydrostepEnum,HydrologydcEplThicknessOldEnum);
						/*Proceed now to heads computations*/
						solutionsequence_hydro_nonlinear(femmodel);
						/*If we have a sub-timestep we stack the variables here*/
						femmodel->SumOutputx(&inputtostack[0],&stackedinput[0],numaveragedinput);
					}
					/*Reseting to global time*/
					femmodel->parameters->SetParam(time,TimeEnum);
					femmodel->AverageSumOutputx(&stackedinput[0],&averagedinput[0],numaveragedinput);
				}
				else{
					/*define which variable needs to be averaged on the sub-timestep and initialize as needed*/
					numaveragedinput = 2;
					int inputtostack[2]	=	{EffectivePressureHydrostepEnum,SedimentHeadHydrostepEnum};
					int stackedinput[2]	=	{EffectivePressureStackedEnum,SedimentHeadStackedEnum};
					int averagedinput[2]	=	{EffectivePressureEnum,SedimentHeadEnum};
					femmodel->InitMeanOutputx(&stackedinput[0],numaveragedinput);
					while(hydrotime<time-(yts*DBL_EPSILON)){ //loop on hydro dts
						hydrostep+=1;
						hydrotime+=hydrodt;
						/*Setting substep time as global time*/
						femmodel->parameters->SetParam(hydrotime,TimeEnum);
						if(VerboseSolution()) _printf0_("sub iteration " << hydrostep << "/" << hydroslices << "  time [yr]: " << setprecision(4) << hydrotime/yts << " (time step: " << hydrodt/yts << ")\n");
						if(issmb){
							if(VerboseSolution()) _printf0_("   computing mass balance\n");
							SmbAnalysis* analysis = new SmbAnalysis();
							analysis->Core(femmodel);
							delete analysis;
						}
						if(VerboseSolution()) _printf0_("   computing water heads\n");
						/*save preceding timestep*/
						InputDuplicatex(femmodel,SedimentHeadHydrostepEnum,SedimentHeadOldEnum);
						/*Proceed now to heads computations*/
						solutionsequence_hydro_nonlinear(femmodel);
						/*If we have a sub-timestep we stack the variables here*/
						femmodel->SumOutputx(&inputtostack[0],&stackedinput[0],numaveragedinput);
					}
					/*Reseting to global time*/
					femmodel->parameters->SetParam(time,TimeEnum);
					femmodel->AverageSumOutputx(&stackedinput[0],&averagedinput[0],numaveragedinput);
				}
			}
			else{
				if(issmb){
					if(VerboseSolution()) _printf0_("   computing mass balance\n");
					SmbAnalysis* analysis = new SmbAnalysis();
					analysis->Core(femmodel);
					delete analysis;
				}
				InputDuplicatex(femmodel,SedimentHeadHydrostepEnum,SedimentHeadOldEnum);
				if (isefficientlayer){
					InputDuplicatex(femmodel,EplHeadHydrostepEnum,EplHeadOldEnum);
					InputDuplicatex(femmodel,HydrologydcEplThicknessHydrostepEnum,HydrologydcEplThicknessOldEnum);
				}
				/*Proceed now to heads computations*/
				if(VerboseSolution()) _printf0_("   computing water heads\n");
				solutionsequence_hydro_nonlinear(femmodel);
			}
		}
		else{
			/* If everything is frozen we still need smb */
			if(issmb){
				if(VerboseSolution()) _printf0_("   computing mass balance\n");
				SmbAnalysis* analysis = new SmbAnalysis();
				analysis->Core(femmodel);
				delete analysis;
			}
		}
	}

	/*Using the SHAKTI model*/
	else if (hydrology_model==HydrologyshaktiEnum){
		femmodel->SetCurrentConfiguration(HydrologyShaktiAnalysisEnum);
		InputDuplicatex(femmodel,HydrologyHeadEnum,HydrologyHeadOldEnum);
		solutionsequence_shakti_nonlinear(femmodel);
		if(VerboseSolution()) _printf0_("   updating gap height\n");
		HydrologyShaktiAnalysis* analysis = new HydrologyShaktiAnalysis();
		analysis->UpdateGapHeight(femmodel);
		delete analysis;
	}

	/*Using the GlaDS model*/
	else if (hydrology_model==HydrologyGlaDSAnalysisEnum){
		femmodel->SetCurrentConfiguration(HydrologyGlaDSAnalysisEnum);
		InputDuplicatex(femmodel,HydraulicPotentialEnum,HydraulicPotentialOldEnum);
		solutionsequence_shakti_nonlinear(femmodel);
		if(VerboseSolution()) _printf0_("   updating sheet thickness\n");
		HydrologyGlaDSAnalysis* analysis = new HydrologyGlaDSAnalysis();
		analysis->UpdateSheetThickness(femmodel);
		delete analysis;
	}

	/*Using the PISM hydrology model*/
	else if (hydrology_model==HydrologypismEnum){
		femmodel->SetCurrentConfiguration(HydrologyPismAnalysisEnum);
		if(VerboseSolution()) _printf0_("   updating water column\n");
		HydrologyPismAnalysis* analysis = new HydrologyPismAnalysis();
		analysis->UpdateWaterColumn(femmodel);
		delete analysis;
	}
	else{
		_error_("Hydrology model "<< EnumToStringx(hydrology_model) <<" not supported yet");
	}
	if(save_results){
		if(VerboseSolution()) _printf0_("   saving results \n");
		if(hydrology_model==HydrologydcEnum && ThawedNodes==0){
			if(VerboseSolution()) _printf0_("   No thawed node hydro is skiped \n");}
		else if (hydrology_model==HydrologydcEnum && issmb){
			femmodel->RequestedOutputsx(&femmodel->results,requested_outputs,numoutputs);
			femmodel->RequestedOutputsx(&femmodel->results,requested_smb_outputs,smboutputs);
		}
		else{
			femmodel->RequestedOutputsx(&femmodel->results,requested_outputs,numoutputs);
		}
	}
	/*Free ressources:*/
	if(numoutputs){
		for(int i=0;i<numoutputs;i++){
			xDelete<char>(requested_outputs[i]);
		}
		xDelete<char*>(requested_outputs);
	}
	if(issmb){
		if(smboutputs){
			for(int i=0;i<smboutputs;i++){
				xDelete<char>(requested_smb_outputs[i]);
			}
			xDelete<char*>(requested_smb_outputs);
		}
	}
	/*End profiler*/
	femmodel->profiler->Stop(HYDROLOGYCORE);
}
