%FRICTIONSCHOOF class definition
%
%   Usage:
%      frictionschoof=frictionschoof();

classdef frictionschoof
	properties (SetAccess=public) 
		C    = NaN;
		Cmax = NaN;
		m    = 0.;
	end
	methods
		function self = frictionschoof(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = extrude(self,md) % {{{
			md.friction.C    = project3d(md,'vector',md.friction.C,'type','node','layer',1);
			md.friction.Cmax = project3d(md,'vector',md.friction.Cmax,'type','node','layer',1);
		end % }}}
		function self = setdefaultparameters(self) % {{{

			%default m is 1/n = 1/3
			self.m = 1./3.;

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			%Early return
			if ~ismember('StressbalanceAnalysis',analyses) & ~ismember('ThermalAnalysis',analyses), return; end
			md = checkfield(md,'fieldname','friction.C','timeseries',1,'NaN',1,'Inf',1,'>',0.);
			md = checkfield(md,'fieldname','friction.Cmax','timeseries',1,'NaN',1,'Inf',1,'>',0.);
			md = checkfield(md,'fieldname','friction.m','NaN',1,'Inf',1,'numel',1,'>',0.);
		end % }}}
		function disp(self) % {{{
			%See Brondex et al. 2019 https://www.the-cryosphere.net/13/177/2019/
			disp('Schoof sliding law parameters:');
			disp('   Schoof''s sliding law reads:');
			disp('                         C |u_b|^(m-1)                ');
			disp('      tau_b = - _____________________________   u_b   ');
			disp('               (1+(C/(Cmax N))^1/m |u_b| )^m          ');
			disp(' ');
			fielddisplay(self,'C','friction coefficient [SI]');
			fielddisplay(self,'Cmax','Iken''s bound (typically between 0.17 and 0.84) [SI]');
			fielddisplay(self,'m','m exponent (generally taken as m = 1/n = 1/3)');
		end % }}}
		function marshall(self,prefix,md,fid) % {{{
			yts=md.constants.yts;

			WriteData(fid,prefix,'name','md.friction.law','data',11,'format','Integer');
			WriteData(fid,prefix,'class','friction','object',self,'fieldname','C','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1,'yts',md.constants.yts);
			WriteData(fid,prefix,'class','friction','object',self,'fieldname','Cmax','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1,'yts',md.constants.yts);
			WriteData(fid,prefix,'class','friction','object',self,'fieldname','m','format','Double');
			

		end % }}}
	end
end
