/*!\file FloatingiceMeltingRatex
 * \brief: calculates Floating ice melting rate
 */

#include "./FloatingiceMeltingRatex.h"
#include "../../shared/shared.h"
#include "../../toolkits/toolkits.h"

void FloatingiceMeltingRatex(FemModel* femmodel){/*{{{*/

	/*Intermediaties*/
	int  basalforcing_model;

	/*First, get BMB model from parameters*/
	femmodel->parameters->FindParam(&basalforcing_model,BasalforcingsEnum);

	/*branch to correct module*/
	switch(basalforcing_model){
		case FloatingMeltRateEnum:
		case MantlePlumeGeothermalFluxEnum:
			/*Nothing to be done*/
			break;
		case LinearFloatingMeltRateEnum:
			if(VerboseSolution())_printf0_("	call Linear Floating melting rate module\n");
			LinearFloatingiceMeltingRatex(femmodel);
			break;
		case MismipFloatingMeltRateEnum:
			if(VerboseSolution())_printf0_("	call Mismip Floating melting rate module\n");
			MismipFloatingiceMeltingRatex(femmodel);
			break;
		case SpatialLinearFloatingMeltRateEnum:
			if(VerboseSolution())_printf0_("	call Spatial Linear Floating melting rate module\n");
			SpatialLinearFloatingiceMeltingRatex(femmodel);
			break;
		case BasalforcingsPicoEnum:
			if(VerboseSolution())_printf0_(" call Pico Floating melting rate module\n");
			FloatingiceMeltingRatePicox(femmodel);
			break;
		case BasalforcingsIsmip6Enum:
			if(VerboseSolution())_printf0_(" call ISMIP 6 Floating melting rate module\n");
			FloatingiceMeltingRateIsmip6x(femmodel);
			break;
		default:
			_error_("Basal forcing model "<<EnumToStringx(basalforcing_model)<<" not supported yet");
	}

}/*}}}*/

void LinearFloatingiceMeltingRatex(FemModel* femmodel){/*{{{*/

	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
		element->LinearFloatingiceMeltingRate();
	}

}/*}}}*/
void SpatialLinearFloatingiceMeltingRatex(FemModel* femmodel){/*{{{*/

	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
		element->SpatialLinearFloatingiceMeltingRate();
	}

}/*}}}*/
void MismipFloatingiceMeltingRatex(FemModel* femmodel){/*{{{*/

	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
		element->MismipFloatingiceMeltingRate();
	}
}
/*}}}*/
void FloatingiceMeltingRateIsmip6x(FemModel* femmodel){/*{{{*/


	int         num_basins, basinid;
	IssmDouble  area, tf, base, time;
	IssmDouble  tf_test[3];

   femmodel->parameters->FindParam(&num_basins,BasalforcingsIsmp6NumBasinsEnum);

   IssmDouble* tf_weighted_avg     = xNewZeroInit<IssmDouble>(num_basins);
	IssmDouble* tf_weighted_avg_cpu = xNewZeroInit<IssmDouble>(num_basins);
   IssmDouble* areas_summed        = xNewZeroInit<IssmDouble>(num_basins);
   IssmDouble* areas_summed_cpu    = xNewZeroInit<IssmDouble>(num_basins);

	/*Find and save TF at each ice shelf point*/
	//	element->parameters->FindParam(&time,TimeEnum);
	//	Input* tf_input = element->GetInput(BasalforcingsIsmp6TfEnum); _assert_(tf_input);
	//	tf_input->Echo();
	//	Input* base_input = element->GetInput(BaseEnum);               _assert_(base_input);
	//	Gauss* gauss=element->NewGauss(1); gauss->GaussPoint(0);
	//	base_input->GetInputValue(&base,gauss);
	//	//tf_input->GetInputValue(&tf,gauss,BasalforcingsIsmp6TfEnum);

	/*TEST: Set tf=2 for all ice shelf elements*/
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
		if(!element->IsIceInElement() || !element->IsFloating()) continue;
		for(int k=0;k<3;k++) {tf_test[k] = 2.;}
		element->AddInput(BasalforcingsIsmp6TfShelfEnum,tf_test,P1Enum);
	}

	/*Compute sums of tf*area and shelf-area per cpu*/
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
		if(!element->IsIceInElement() || !element->IsFloating()) continue;
		Input* tf_input=element->GetInput(BasalforcingsIsmp6TfShelfEnum); _assert_(tf_input);
		element->inputs->GetInputValue(&basinid,BasalforcingsIsmp6BasinIdEnum);
		Gauss* gauss=element->NewGauss(1); gauss->GaussPoint(0);
		tf_input->GetInputValue(&tf,gauss);
		delete gauss;
		area=element->GetHorizontalSurfaceArea();
		tf_weighted_avg[basinid]+=tf*area;
		areas_summed[basinid]   +=area;
	}

	/*Syncronize across cpus*/
	ISSM_MPI_Allreduce(tf_weighted_avg,tf_weighted_avg_cpu,num_basins,ISSM_MPI_DOUBLE,ISSM_MPI_SUM,IssmComm::GetComm());
	ISSM_MPI_Allreduce(areas_summed,areas_summed_cpu,num_basins,ISSM_MPI_DOUBLE,ISSM_MPI_SUM,IssmComm::GetComm());

	/*Compute weighted means and save*/
	for(int k=0;k<num_basins;k++){tf_weighted_avg_cpu[k] = tf_weighted_avg_cpu[k]/areas_summed_cpu[k];}
	femmodel->parameters->AddObject(new DoubleVecParam(BasalforcingsIsmp6AverageTfEnum,tf_weighted_avg_cpu,num_basins));

	/*Cleanup and return */
	xDelete<IssmDouble>(tf_weighted_avg);
	xDelete<IssmDouble>(tf_weighted_avg_cpu);
	xDelete<IssmDouble>(areas_summed);
	xDelete<IssmDouble>(areas_summed_cpu);

   /*Compute meltrates*/
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
		element->Ismip6FloatingiceMeltingRate();
	}
}
/*}}}*/
