import numpy as np
from struct import pack,unpack
import pairoptions

def WriteData(fid,prefix,*args):
	"""
	WRITEDATA - write model field in binary file

	   Usage:
	      WriteData(fid,varargin)
	"""

	#process options
	options=pairoptions.pairoptions(*args)

	#Get data properties
	if options.exist('object'):
		#This is an object field, construct enum and data
		obj       = options.getfieldvalue('object')
		fieldname = options.getfieldvalue('fieldname')
		classname = options.getfieldvalue('class',str(type(obj)).rsplit('.')[-1].split("'")[0])
		name      = options.getfieldvalue('name',prefix+'.'+fieldname);
		if options.exist('data'):
			data = options.getfieldvalue('data')
		else:
			data      = getattr(obj,fieldname)
	else:
		#No processing required
		data = options.getfieldvalue('data')
		name = options.getfieldvalue('name')

	datatype  = options.getfieldvalue('format')
	mattype = options.getfieldvalue('mattype',0)    #only required for matrices
	timeserieslength = options.getfieldvalue('timeserieslength',-1)

	#Process sparse matrices
#	if issparse(data),
#		data=full(data);
#	end

	#Scale data if necesarry
	if options.exist('scale'):
		data=np.array(data)
		scale = options.getfieldvalue('scale')
		if np.size(data) > 1 and np.ndim(data) > 1 and np.size(data,0)==timeserieslength:
			data[0:-1,:] = scale*data[0:-1,:]
		else:
			data  = scale*data
	if np.size(data) > 1 and np.size(data,0)==timeserieslength:
		yts = options.getfieldvalue('yts')
		if np.ndim(data) > 1:
			data[-1,:] = yts*data[-1,:]
		else:
			data[-1] = yts*data[-1]

	#Step 1: write the enum to identify this record uniquely
	fid.write(pack('i',len(name)))
	fid.write(pack('{}s'.format(len(name)),name.encode()))


	#Step 2: write the data itself.
	if datatype=='Boolean':    # {{{
#		if len(data) !=1:
#			raise ValueError('fi eld %s cannot be marshalled as it has more than one element!' % name[0])

		#first write length of record
		fid.write(pack('i',4+4))  #1 bool (disguised as an int)+code

		#write data code:
		fid.write(pack('i',FormatToCode(datatype)))

		#now write integer
		fid.write(pack('i',int(data)))  #send an int, not easy to send a bool
		# }}}

	elif datatype=='Integer':    # {{{
#		if len(data) !=1:
#			raise ValueError('field %s cannot be marshalled as it has more than one element!' % name[0])

		#first write length of record
		fid.write(pack('i',4+4))  #1 integer + code

		#write data code:
		fid.write(pack('i',FormatToCode(datatype)))

		#now write integer
		fid.write(pack('i',int(data))) #force an int,
		# }}}

	elif datatype=='Double':    # {{{
#		if len(data) !=1:
#			raise ValueError('field %s cannot be marshalled as it has more than one element!' % name[0])

		#first write length of record
		fid.write(pack('i',8+4))  #1 double+code

		#write data code:
		fid.write(pack('i',FormatToCode(datatype)))

		#now write double
		fid.write(pack('d',data))
		# }}}

	elif datatype=='String':    # {{{
		#first write length of record
		fid.write(pack('i',len(data)+4+4))  #string + string size + code

		#write data code:
		fid.write(pack('i',FormatToCode(datatype)))

		#now write string
		fid.write(pack('i',len(data)))
		fid.write(pack('{}s'.format(len(data)),data.encode()))
		# }}}

	elif datatype in ['IntMat','BooleanMat']:    # {{{

		if isinstance(data,(int,bool)):
			data=np.array([data])
		elif isinstance(data,(list,tuple)):
			data=np.array(data).reshape(-1,)
		if np.ndim(data) == 1:
			if np.size(data):
				data=data.reshape(np.size(data),)
			else:
				data=data.reshape(0,0)

		#Get size
		s=data.shape
		#if matrix = NaN, then do not write anything
		if np.ndim(data)==2 and np.product(s)==1 and np.all(np.isnan(data)):
			s=(0,0)

		#first write length of record
		fid.write(pack('i',4+4+8*np.product(s)+4+4))    #2 integers (32 bits) + the double matrix + code + matrix type

		#write data code and matrix type:
		fid.write(pack('i',FormatToCode(datatype)))
		fid.write(pack('i',mattype))

		#now write matrix
		if np.ndim(data) == 1:
			fid.write(pack('i',s[0]))
			fid.write(pack('i',1))
			for i in range(s[0]):
				fid.write(pack('d',float(data[i])))    #get to the "c" convention, hence the transpose
		else:
			fid.write(pack('i',s[0]))
			fid.write(pack('i',s[1]))
			for i in range(s[0]):
				for j in range(s[1]):
					fid.write(pack('d',float(data[i][j])))    #get to the "c" convention, hence the transpose
		# }}}

	elif datatype=='DoubleMat':    # {{{

		if   isinstance(data,(bool,int,float)):
			data=np.array([data])
		elif isinstance(data,(list,tuple)):
			data=np.array(data).reshape(-1,)
		if np.ndim(data) == 1:
			if np.size(data):
				data=data.reshape(np.size(data),)
			else:
				data=data.reshape(0,0)

		#Get size
		s=data.shape
		#if matrix = NaN, then do not write anything
		if np.ndim(data)==1 and np.product(s)==1 and np.all(np.isnan(data)):
			s=(0,0)

		#first write length of record
		recordlength=4+4+8*np.product(s)+4+4; #2 integers (32 bits) + the double matrix + code + matrix type
		if recordlength > 4**31 :
			raise ValueError('field {} cannot be marshalled because it is larger than 4^31 bytes!'.format(enum))

		fid.write(pack('i',recordlength))  #2 integers (32 bits) + the double matrix + code + matrix type

		#write data code and matrix type:
		fid.write(pack('i',FormatToCode(datatype)))
		fid.write(pack('i',mattype))

		#now write matrix
		if np.ndim(data) == 1:
			fid.write(pack('i',s[0]))
			fid.write(pack('i',1))
			for i in range(s[0]):
				fid.write(pack('d',float(data[i])))    #get to the "c" convention, hence the transpose
		else:
			fid.write(pack('i',s[0]))
			fid.write(pack('i',s[1]))
			for i in range(s[0]):
				for j in range(s[1]):
					fid.write(pack('d',float(data[i][j])))    #get to the "c" convention, hence the transpose
		# }}}

	elif datatype=='CompressedMat':    # {{{

		if   isinstance(data,(bool,int,float)):
			data=np.array([data])
		elif isinstance(data,(list,tuple)):
			data=np.array(data).reshape(-1,)
		if np.ndim(data) == 1:
			if np.size(data):
				data=data.reshape(np.size(data),)
			else:
				data=data.reshape(0,0)

		#Get size
		s=data.shape
		if np.ndim(data) == 1:
		   n2=1
		else:
			n2=s[1]

		#if matrix = NaN, then do not write anything
		if np.ndim(data)==1 and np.product(s)==1 and np.all(np.isnan(data)):
			s=(0,0)
			n2=0

		#first write length of record
		recordlength=4+4+8+8+1*(s[0]-1)*n2+8*n2+4+4 #2 integers (32 bits) + the matrix + code + matrix type
		if recordlength > 4**31 :
			raise ValueError('field %s cannot be marshalled because it is larger than 4^31 bytes!' % enum)

		fid.write(pack('i',recordlength))  #2 integers (32 bits) + the matrix + code + matrix type

		#write data code and matrix type:
		fid.write(pack('i',FormatToCode(datatype)))
		fid.write(pack('i',mattype))

		#Write offset and range
		A = data[0:s[0]-1]
		offsetA = A.min()
		rangeA = A.max() - offsetA

		if rangeA == 0:
			A = A*0
		else:
			A = (A-offsetA)/rangeA*255.

		#now write matrix
		if np.ndim(data) == 1:
			fid.write(pack('i',s[0]))
			fid.write(pack('i',1))
			fid.write(pack('d',float(offsetA)))
			fid.write(pack('d',float(rangeA)))
			for i in range(s[0]-1):
				fid.write(pack('B',int(A[i])))

			fid.write(pack('d',float(data[s[0]-1])))    #get to the "c" convention, hence the transpose

		elif np.product(s) > 0:
			fid.write(pack('i',s[0]))
			fid.write(pack('i',s[1]))
			fid.write(pack('d',float(offsetA)))
			fid.write(pack('d',float(rangeA)))
			for i in range(s[0]-1):
				for j in range(s[1]):
					fid.write(pack('B',int(A[i][j])))    #get to the "c" convention, hence the transpose

			for j in range(s[1]):
				fid.write(pack('d',float(data[s[0]-1][j])))

		# }}}

	elif datatype=='MatArray':    # {{{

		#first get length of record
		recordlength=4+4    #number of records + code
		for matrix in data:
			if   isinstance(matrix,(bool,int,float)):
				matrix=np.array([matrix])
			elif isinstance(matrix,(list,tuple)):
				matrix=np.array(matrix).reshape(-1,)
			if np.ndim(matrix) == 1:
				if np.size(matrix):
					matrix=matrix.reshape(np.size(matrix),)
				else:
					matrix=matrix.reshape(0,0)

			s=matrix.shape
			recordlength+=4*2+np.product(s)*8    #row and col of matrix + matrix of doubles

		#write length of record
		fid.write(pack('i',recordlength))

		#write data code:
		fid.write(pack('i',FormatToCode(datatype)))

		#write data, first number of records
		fid.write(pack('i',len(data)))

		for matrix in data:
			if   isinstance(matrix,(bool,int,float)):
				matrix=np.array([matrix])
			elif isinstance(matrix,(list,tuple)):
				matrix=np.array(matrix).reshape(-1,)
			if np.ndim(matrix) == 1:
				matrix=matrix.reshape(np.size(matrix),)

			s=matrix.shape

			if np.ndim(matrix) == 1:
				fid.write(pack('i',s[0]))
				fid.write(pack('i',1))
				for i in range(s[0]):
					fid.write(pack('d',float(matrix[i])))    #get to the "c" convention, hence the transpose
			else:
				fid.write(pack('i',s[0]))
				fid.write(pack('i',s[1]))
				for i in range(s[0]):
					for j in range(s[1]):
						fid.write(pack('d',float(matrix[i][j])))
		# }}}

	elif datatype=='StringArray':    # {{{

		#first get length of record
		recordlength=4+4    #for length of array + code
		for string in data:
			recordlength+=4+len(string)    #for each string

		#write length of record
		fid.write(pack('i',recordlength))

		#write data code:
		fid.write(pack('i',FormatToCode(datatype)))

		#now write length of string array
		fid.write(pack('i',len(data)))

		#now write the strings
		for string in data:
			fid.write(pack('i',len(string)))
			fid.write(pack('{}s'.format(len(string)),string.encode()))
		# }}}

	else:    # {{{
		raise TypeError('WriteData error message: data type: {} not supported yet! ({})'.format(datatype,enum))
	# }}}

def FormatToCode(datatype): # {{{
	"""
	This routine takes the datatype string, and hardcodes it into an integer, which
	is passed along the record, in order to identify the nature of the dataset being
	sent.
	"""

	if datatype=='Boolean':
		code=1
	elif datatype=='Integer':
		code=2
	elif datatype=='Double':
		code=3
	elif datatype=='String':
		code=4
	elif datatype=='BooleanMat':
		code=5
	elif datatype=='IntMat':
		code=6
	elif datatype=='DoubleMat':
		code=7
	elif datatype=='MatArray':
		code=8
	elif datatype=='StringArray':
		code=9
	elif datatype=='CompressedMat':
		code=10
	else:
		raise InputError('FormatToCode error message: data type not supported yet!')

	return code
# }}}
