import numpy as np
import os
import model
import glob

def exportVTK(filename,model,enveloppe=False,*args):
	'''
	vtk export
	function exportVTK(filename,model)
	creates a directory with the vtk files for displays in paraview
	(only work for triangle and wedges based on their number of nodes)

	Usage:
	exportVTK('DirName',md)
	exportVTK('DirName',md,'geometry','mesh')
	exportVTK('DirName',md,'geometry','mesh',enveloppe=True)

	DirName is the name of the output directory, each timestep then has it
	own file ('Timestep.vtkX.vtk') with X the number of the output step
	enveloppe is an option keeping only the enveloppe of the model (it is False by default)

	TODO: - make time easily accessible
	      - make evolving geometry
        - make enveloppe an option?

	Basile de Fleurian:
	'''

	# {{{File checking and creation
	Dir=os.path.basename(filename)
	Path=filename[:-len(Dir)]
	if os.path.exists(filename):
		print(('File {} allready exist'.format(filename)))
		newname=input('Give a new name or "delete" to replace: ')
		if newname=='delete':
			filelist = glob.glob(filename+'/*')
			for oldfile in filelist:
				os.remove(oldfile)
		else:
			print(('New file name is {}'.format(newname)))
			filename=newname
			os.mkdir(filename)
	else:
		os.mkdir(filename)
	# }}}

	# {{{this is the result structure
	res_struct=model.results
	moving_mesh=False
	if(type(res_struct)!=list):
		#Getting all the solutions of the model
		solnames=(dict.keys(res_struct.__dict__))
		num_of_sols=len(solnames)
		num_of_timesteps=1
		#%building solutionstructure
		for solution in solnames:
			#looking for multiple time steps
			if (np.size(res_struct.__dict__[solution])>num_of_timesteps):
				num_of_timesteps=np.size(res_struct.__dict__[solution])
				num_of_timesteps=int(num_of_timesteps)
			if 'Surface' in dict.keys(res_struct.__dict__[solution][0].__dict__):
				moving_mesh=True
	else:
		num_of_timesteps=1
	# }}}

	# {{{get the element related variables
	#first get the general things
	dim=int(model.mesh.domaintype()[0])
	every_nodes=model.mesh.numberofvertices
	every_cells=model.mesh.numberofelements

	point_per_elt=np.shape(model.mesh.elements)[1]
	if point_per_elt==3:
		celltype=5 #triangles
	elif point_per_elt==6:
		celltype=13 #wedges

	if enveloppe:
		if dim==3:
			is_enveloppe=np.logical_or(model.mesh.vertexonbase,model.mesh.vertexonsurface)
			enveloppe_index=np.where(is_enveloppe)[0]
			convert_index=np.nan*np.ones(np.shape(model.mesh.x))
			convert_index=np.asarray([[i,np.where(enveloppe_index==i)[0][0]] for i,val in enumerate(convert_index) if any(enveloppe_index==i)])
			points=np.column_stack((model.mesh.x[enveloppe_index],
															model.mesh.y[enveloppe_index],
															model.mesh.z[enveloppe_index]))

			num_of_elt=np.size(np.where(np.isnan(model.mesh.lowerelements)))+np.size(np.where(np.isnan(model.mesh.upperelements)))
			connect=model.mesh.elements[np.where(is_enveloppe[model.mesh.elements-1])].reshape(int(num_of_elt),3)-1

			for elt in range(0, num_of_elt):
				connect[elt,0]=convert_index[np.where(convert_index==connect[elt,0])[0],1][0]
				connect[elt,1]=convert_index[np.where(convert_index==connect[elt,1])[0],1][0]
				connect[elt,2]=convert_index[np.where(convert_index==connect[elt,2])[0],1][0]

			num_of_points=np.size(enveloppe_index)
		else:
			raise BadDimension("exportVTK can't get an enveloppe for  dimension {}".format(dim))

	else:
		#we get all the mesh, mainly defining dummies
		num_of_elt=every_cells
		connect=model.mesh.elements-1
		enveloppe_index=np.arange(0,np.size(model.mesh.x))
		num_of_points=every_nodes
		if dim==2:
			points=np.column_stack((model.mesh.x,
															model.mesh.y,
															np.zeros(np.shape(model.mesh.x))))
		elif dim==3:
			points=np.column_stack((model.mesh.x,
															model.mesh.y,
															model.mesh.z))
		else:
			raise BadDimension('exportVTK does not support dimension {}'.format(dim))
	# }}}
	# {{{write header and mesh
	for step in range(0,num_of_timesteps):
		saved_cells={}
		timestep=step
		fid=open((filename +'/Timestep.vtk'+str(timestep)+'.vtk'),'w+')
		fid.write('# vtk DataFile Version 3.0 \n')
		fid.write('Data for run %s \n' % model.miscellaneous.name)
		fid.write('ASCII \n')
		fid.write('DATASET UNSTRUCTURED_GRID \n')
		fid.write('POINTS %d float\n' % num_of_points)
		#updating z for mesh evolution
		if moving_mesh:
			altitude=(np.squeeze(res_struct.__dict__['TransientSolution'][step].__dict__['Base'][enveloppe_index])+
								(np.squeeze(res_struct.__dict__['TransientSolution'][step].__dict__['Thickness'][enveloppe_index])/
								 model.geometry.thickness[enveloppe_index])*
								(points[:,2]-model.geometry.thickness[enveloppe_index]))
		else:
			altitude=points[:,2]
		for index,point in enumerate(points):
			fid.write('%f %f %f \n'%(point[0], point[1], altitude[index]))

		fid.write('CELLS %d %d\n' %(num_of_elt, num_of_elt*(point_per_elt+1)))

		for elt in range(0, num_of_elt):
			fid.write('3 %d %d %d\n' %(connect[elt,0],
																 connect[elt,1],
																 connect[elt,2]))

		fid.write('CELL_TYPES %d\n' %num_of_elt)
		for elt in range(0, num_of_elt):
			fid.write('%d\n' % celltype)

		fid.write('POINT_DATA %s \n' %str(num_of_points))
		# }}}
		# {{{loop over the different solution structures
		if solnames in locals():
			for sol in solnames:
				treated_res=[]
				#dealing with results on different timesteps
				if(np.size(res_struct.__dict__[sol])>timestep):
					timestep = step
				else:
					timestep = np.size(res_struct.__dict__[sol])

				#getting the  fields in the solution
				if(type(res_struct.__dict__[sol])==list):
					spe_res_struct=res_struct.__dict__[sol].__getitem__(timestep)
					fieldnames=dict.keys(spe_res_struct.__dict__)
				else:
					spe_res_struct=res_struct.__dict__[sol]
					fieldnames=dict.keys(spe_res_struct.__dict__)

				#Sorting scalars, vectors and tensors
				tensors=[field for field in fieldnames if field[-2:] in ['xx','yy','xy','zz','xz','yz']]
				non_tensor=[field for field in fieldnames if field not in tensors]
				vectors=[field for field in non_tensor if field[-1] in ['x','y','z']]

				#check which field is a real result and print
				for field in fieldnames:
					if field in treated_res:
						continue
					elif field in vectors:
						try:
							Vxstruct=spe_res_struct.__dict__[field[:-1]+'x']
							Vystruct=spe_res_struct.__dict__[field[:-1]+'y']
							if dim==3:
								Vzstruct=spe_res_struct.__dict__[field[:-1]+'z']
								treated_res+=[field[:-1]+'x',field[:-1]+'y',field[:-1]+'z']
							elif dim==2:
								treated_res+=[field[:-1]+'x',field[:-1]+'y']
						except KeyError:
							fieldnames+=field
							vectors.remove(field)

						fid.write('VECTORS {} float \n'.format(field[:-1]))
						for node in range(0,num_of_points):
							Vx=cleanOutliers(Vxstruct[enveloppe_index[node]])
							Vy=cleanOutliers(Vystruct[enveloppe_index[node]])
							if dim==3:
								Vz=cleanOutliers(Vzstruct[enveloppe_index[node]])
								fid.write('%e %e %e\n' %(Vx,Vy,Vz))
							elif dim==2:
								fid.write('%e %e %e\n' %(Vx,Vy,0))

					elif field in tensors:
						print("nothing")
					else:
						if ((np.size(spe_res_struct.__dict__[field]))==every_nodes):
							fid.write('SCALARS %s float 1 \n' % field)
							fid.write('LOOKUP_TABLE default\n')
							for node in range(0,num_of_points):
								outval=cleanOutliers(spe_res_struct.__dict__[field][enveloppe_index[node]])
								fid.write('%e\n' % outval)
						elif ((np.size(spe_res_struct.__dict__[field]))==every_cells):
							saved_cells[field]=spe_res_struct.__dict__[field]
		# }}}
		# {{{loop on arguments, if something other than result is asked, do it now
		for other in args:
			other_struct=model.__dict__[other]
			othernames=(dict.keys(other_struct.__dict__))
			for field in othernames:
				if (np.size(other_struct.__dict__[field])==every_nodes):
					fid.write('SCALARS %s float 1 \n' % field)
					fid.write('LOOKUP_TABLE default\n')
					for node in range(0,num_of_points):
						outval=cleanOutliers(other_struct.__dict__[field][enveloppe_index[node]])
						fid.write('%e\n' % outval)
				elif (np.size(other_struct.__dict__[field])==every_cells):
					saved_cells[field]=other_struct.__dict__[field]
			# }}}
		# {{{Now writting cell variables
		if np.size(list(saved_cells.keys()))>0:
			fid.write('CELL_DATA %s \n' %str(num_of_elt))
			for key in list(saved_cells.keys()):
				fid.write('SCALARS %s float 1 \n' % key)
				fid.write('LOOKUP_TABLE default\n')
				for cell in range(0,num_of_elt):
					outval=cleanOutliers(saved_cells[key][cell])
					fid.write('%e\n' % outval)
		# }}}

	fid.close();

def cleanOutliers(Val):
	#paraview does not like NaN, replacing
	if np.isnan(Val):
		CleanVal=-9999.999
		#also checking for very small value that mess up
	elif (Val<1.0e-20):
		CleanVal= 0.0
	else:
		CleanVal=Val
	return CleanVal

class BadDimension(Exception):
	"""The required dimension is not supported yet."""
