/*!\file GetMaskOfIceVerticesLSMx 
 * \brief: Return a mask for all the vertices determining whether the node should be active or not. 
 */

#include "./SetActiveNodesLSMx.h"

#include "../../classes/classes.h"
#include "../../shared/shared.h"
#include "../../toolkits/toolkits.h"
#include "../modules.h"

void SetActiveNodesLSMx(FemModel* femmodel){/*{{{*/
	/* activate/deactivate nodes for levelset method according to IceMaskNodeActivation */

	/* find parameters */
	int domaintype;
	femmodel->parameters->FindParam(&domaintype,DomainTypeEnum);

	for(int i=0;i<femmodel->elements->Size();i++){
		Element    *element  = xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
		int         numnodes = element->GetNumberOfNodes();
		IssmDouble *mask     = xNew<IssmDouble>(numnodes);

		/*include switch for elements with multiple different sets of nodes*/
		switch(element->GetElementType()){
			case MINIEnum:case MINIcondensedEnum:
			case TaylorHoodEnum:case XTaylorHoodEnum:case LATaylorHoodEnum:
			case CrouzeixRaviartEnum:case LACrouzeixRaviartEnum:case OneLayerP4zEnum:{
				Input* input=element->GetInput(IceMaskNodeActivationEnum);
				if(!input) _error_("Input " << EnumToStringx(IceMaskNodeActivationEnum) << " not found in element");

				/* Start looping on the number of vertices: */
				Gauss* gauss=element->NewGauss();
				for(int iv=0;iv<element->NumberofNodesVelocity();iv++){
					gauss->GaussNode(element->VelocityInterpolation(),iv);
					input->GetInputValue(&mask[iv],gauss);
				}
				for(int iv=0;iv<element->NumberofNodesPressure();iv++){
					gauss->GaussNode(element->PressureInterpolation(),iv);
					input->GetInputValue(&mask[element->NumberofNodesVelocity()+iv],gauss);
				}
				delete gauss;
				break;
			}
			default:
				element->GetInputListOnNodes(&mask[0],IceMaskNodeActivationEnum);
				break;
		}

		for(int in=0;in<numnodes;in++){
			Node* node=element->GetNode(in);
			if(mask[in]==1.) node->Activate();
			else             node->Deactivate();
		}
		xDelete<IssmDouble>(mask);
	}
}/*}}}*/

void GetMaskOfIceVerticesLSMx0(FemModel* femmodel){/*{{{*/

	/*Initialize vector with number of vertices*/
	int numvertices=femmodel->vertices->NumberOfVertices();
	if(numvertices==0)  return;
	Vector<IssmDouble>* vec_mask_ice=new Vector<IssmDouble>(numvertices);

	/*Fill vector with values: */
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
		if(element->IsIceInElement()){
			int nbv = element->GetNumberOfVertices();
			for(int iv=0;iv<nbv;iv++){
				vec_mask_ice->SetValue(element->vertices[iv]->Sid(),1.,INS_VAL);
			}
		}
	}

	/*Assemble vector and serialize */
	vec_mask_ice->Assemble();
	InputUpdateFromVectorx(femmodel,vec_mask_ice,IceMaskNodeActivationEnum,VertexSIdEnum);
	delete vec_mask_ice;
}/*}}}*/
void GetMaskOfIceVerticesLSMx(FemModel* femmodel){/*{{{*/

	femmodel->SetCurrentConfiguration(LevelsetAnalysisEnum);

	/*Create vector on gset*/
	int gsize              = femmodel->nodes->NumberOfDofs(GsetEnum);
	int glocalsize_masters = femmodel->nodes->NumberOfDofsLocal(GsetEnum);
	if(gsize==0)  return;
	Vector<IssmDouble>* vec_mask_ice=new Vector<IssmDouble>(glocalsize_masters,gsize);

	/*Fill vector with values: */
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));

		if(element->IsIceInElement()){
			int numnodes = element->GetNumberOfNodes();
			int  gsize_local=GetNumberOfDofs(element->nodes,numnodes,GsetEnum,NoneEnum);
			int* glist_local=GetGlobalDofList(element->nodes,numnodes,GsetEnum,NoneEnum);
			IssmDouble* ones = xNew<IssmDouble>(gsize_local);
			for(int n=0;n<gsize_local;n++) ones[n] = 1.;
			vec_mask_ice->SetValues(gsize_local,glist_local,ones,INS_VAL);
			xDelete<IssmDouble>(ones);
			xDelete<int>(glist_local);
		}
	}

	/*Assemble vector and serialize */
	vec_mask_ice->Assemble();


	/*FIXME: What follows should be copied and pasted into InputUpdateFromSolution Nodes*/

	/*recover my_rank:*/
	ISSM_MPI_Status status;
	int my_rank   = IssmComm::GetRank();
	int num_procs = IssmComm::GetSize();

	/*retrieve node info*/
	int glocalsize         = femmodel->nodes->NumberOfDofsLocalAll(GsetEnum);
	int maxdofspernode     = femmodel->nodes->MaxNumDofs(GsetEnum);

	/*Get local vector of ug*/
	int        *indices_ug_masters = NULL;
	IssmDouble *local_ug_masters   = NULL;
	vec_mask_ice->GetLocalVector(&local_ug_masters,&indices_ug_masters);
	_assert_(glocalsize_masters==indices_ug_masters[glocalsize_masters-1] - indices_ug_masters[0]+1);
	xDelete<int>(indices_ug_masters);
	delete vec_mask_ice;

	/*Now, extend vectors to account for clones (make vectors longer, for clones at the end)*/
	IssmDouble *local_ug  = xNew<IssmDouble>(glocalsize);
	xMemCpy<IssmDouble>(local_ug,local_ug_masters,glocalsize_masters);
	xDelete<IssmDouble>(local_ug_masters);

	/*Now send and receive ug for nodes on partition edge*/
	IssmDouble* buffer = xNew<IssmDouble>(femmodel->nodes->Size()*maxdofspernode); //only one alloc
	for(int rank=0;rank<num_procs;rank++){
		if(femmodel->nodes->common_send[rank]){
			int  numids = femmodel->nodes->common_send[rank];
			for(int i=0;i<numids;i++){
				int   master_lid = femmodel->nodes->common_send_ids[rank][i];
				Node* node=xDynamicCast<Node*>(femmodel->nodes->GetObjectByOffset(master_lid));
				_assert_(!node->IsClone());
				for(int j=0;j<node->gsize;j++) buffer[i*maxdofspernode+j]=local_ug[node->gdoflist_local[j]];
			}
			ISSM_MPI_Send(buffer,numids*maxdofspernode,ISSM_MPI_DOUBLE,rank,0,IssmComm::GetComm());
		}
	}
	for(int rank=0;rank<num_procs;rank++){
		if(femmodel->nodes->common_recv[rank]){
			int  numids = femmodel->nodes->common_recv[rank];
			ISSM_MPI_Recv(buffer,numids*maxdofspernode,ISSM_MPI_DOUBLE,rank,0,IssmComm::GetComm(),&status);
			for(int i=0;i<numids;i++){
				int   master_lid = femmodel->nodes->common_recv_ids[rank][i];
				Node* node=xDynamicCast<Node*>(femmodel->nodes->GetObjectByOffset(master_lid));
				for(int j=0;j<node->gsize;j++) local_ug[node->gdoflist_local[j]] = buffer[i*maxdofspernode+j];
			}
		}
	}
	xDelete<IssmDouble>(buffer);

	/*Now update inputs (analysis specific)*/
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
		element->InputUpdateFromSolutionOneDof(local_ug,IceMaskNodeActivationEnum);
	}

	/*cleanup and return*/
	xDelete<IssmDouble>(local_ug);
}/*}}}*/
