/*!\file InputUpdateFromSolutionx
 * \brief: update datasets using  parameter inputs
 */

#include "./InputUpdateFromSolutionx.h"
#include "../../shared/shared.h"
#include "../../toolkits/toolkits.h"

void InputUpdateFromSolutionx(FemModel* femmodel,Vector<IssmDouble>* solution){

	/*Display message*/
	if(VerboseModule()) _printf0_("   Updating inputs from solution\n");

	/*GetAnalysis*/
	int analysisenum;
	femmodel->parameters->FindParam(&analysisenum,AnalysisTypeEnum);
	Analysis* analysis = EnumToAnalysis(analysisenum);

	/*recover my_rank:*/
	ISSM_MPI_Status status;
	int my_rank   = IssmComm::GetRank();
	int num_procs = IssmComm::GetSize();

	/*retrieve node info*/
	int gsize              = femmodel->nodes->NumberOfDofs(GsetEnum);
	int glocalsize_masters = femmodel->nodes->NumberOfDofsLocal(GsetEnum);
	int glocalsize         = femmodel->nodes->NumberOfDofsLocalAll(GsetEnum);
	int maxdofspernode     = femmodel->nodes->MaxNumDofs(GsetEnum);

	/*Get local vector of ug*/
	int        *indices_ug_masters = NULL;
	IssmDouble *local_ug_masters   = NULL;
	solution->GetLocalVector(&local_ug_masters,&indices_ug_masters);
	_assert_(glocalsize_masters==indices_ug_masters[glocalsize_masters-1] - indices_ug_masters[0]+1);
	xDelete<int>(indices_ug_masters);

	/*Now, extend vectors to account for clones (make vectors longer, for clones at the end)*/
	IssmDouble *local_ug  = xNew<IssmDouble>(glocalsize);
	xMemCpy<IssmDouble>(local_ug,local_ug_masters,glocalsize_masters);
	xDelete<IssmDouble>(local_ug_masters);

	/*Now send and receive ug for nodes on partition edge*/
	IssmDouble* buffer = xNew<IssmDouble>(femmodel->nodes->Size()*maxdofspernode); //only one alloc
	for(int rank=0;rank<num_procs;rank++){
		if(femmodel->nodes->common_send[rank]){
			int  numids = femmodel->nodes->common_send[rank];
			for(int i=0;i<numids;i++){
				int   master_lid = femmodel->nodes->common_send_ids[rank][i];
				Node* node=xDynamicCast<Node*>(femmodel->nodes->GetObjectByOffset(master_lid));
				_assert_(!node->IsClone());
				for(int j=0;j<node->gsize;j++) buffer[i*maxdofspernode+j]=local_ug[node->gdoflist_local[j]];
			}
			ISSM_MPI_Send(buffer,numids*maxdofspernode,ISSM_MPI_DOUBLE,rank,0,IssmComm::GetComm());
		}
	}
	for(int rank=0;rank<num_procs;rank++){
		if(femmodel->nodes->common_recv[rank]){
			int  numids = femmodel->nodes->common_recv[rank];
			ISSM_MPI_Recv(buffer,numids*maxdofspernode,ISSM_MPI_DOUBLE,rank,0,IssmComm::GetComm(),&status);
			for(int i=0;i<numids;i++){
				int   master_lid = femmodel->nodes->common_recv_ids[rank][i];
				Node* node=xDynamicCast<Node*>(femmodel->nodes->GetObjectByOffset(master_lid));
				for(int j=0;j<node->gsize;j++) local_ug[node->gdoflist_local[j]] = buffer[i*maxdofspernode+j];
			}
		}
	}
	xDelete<IssmDouble>(buffer);

	/*Now update inputs (analysis specific)*/
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
		analysis->InputUpdateFromSolution(local_ug,element);
	}

	/*cleanup and return*/
	delete analysis;
	xDelete<IssmDouble>(local_ug);
}
