/*
 * \file Vertices.cpp
 * \brief: Implementation of Vertices class, derived from DataSet class.
 */

/*Headers: {{{*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <vector>
#include <functional>
#include <algorithm>
#include <iostream>

#include "./Vertices.h"
#include "../shared/shared.h"
#include "./Vertex.h"

using namespace std;
/*}}}*/

/*Object constructors and destructor*/
Vertices::Vertices(){/*{{{*/
	this->enum_type       = VerticesEnum;
	this->common_recv     = NULL;
	this->common_recv_ids = NULL;
	this->common_send     = NULL;
	this->common_send_ids = NULL;
	return;
}
/*}}}*/
Vertices::~Vertices(){/*{{{*/

	int num_proc=IssmComm::GetSize();

	if(this->common_recv) xDelete<int>(common_recv);
	if(this->common_send) xDelete<int>(common_send);
	if(this->common_recv_ids){
		for(int i=0;i<num_proc;i++) if(common_recv_ids[i]) xDelete<int>(common_recv_ids[i]);
		xDelete<int*>(common_recv_ids);
	}
	if(this->common_send_ids){
		for(int i=0;i<num_proc;i++) if(common_send_ids[i]) xDelete<int>(common_send_ids[i]);
		xDelete<int*>(common_send_ids);
	}
	return;
}
/*}}}*/

/*Numerics management*/
Vertices* Vertices::Copy() {/*{{{*/

	int num_proc = IssmComm::GetSize();

	/*Copy dataset*/
	Vertices* output=new Vertices();
	output->sorted=this->sorted;
	output->numsorted=this->numsorted;
	output->presorted=this->presorted;
	for(vector<Object*>::iterator obj=this->objects.begin() ; obj < this->objects.end(); obj++ ){
		output->AddObject((*obj)->copy());
	}

	/*Build id_offsets and sorted_ids*/
	int objsize = this->numsorted;
	if(this->sorted && objsize>0 && this->id_offsets){	
		/*Allocate new ids*/
		output->id_offsets=xNew<int>(objsize);
		xMemCpy<int>(output->id_offsets,this->id_offsets,objsize);
	}
	else output->id_offsets=NULL;
	if(this->sorted && objsize>0 && this->sorted_ids){
		/*Allocate new ids*/
		output->sorted_ids=xNew<int>(objsize);
		xMemCpy<int>(output->sorted_ids,this->sorted_ids,objsize);
	}
	else output->sorted_ids=NULL;


	if(this->common_recv){
		output->common_recv=xNew<int>(num_proc);
		for(int i=0;i<num_proc;i++) output->common_recv[i]=this->common_recv[i];
	}
	if(this->common_send){
		output->common_send=xNew<int>(num_proc);
		for(int i=0;i<num_proc;i++) output->common_send[i]=this->common_send[i];
	}
	if(this->common_recv_ids){
		output->common_recv_ids = xNew<int*>(num_proc);
		for(int i=0;i<num_proc;i++){
			output->common_recv_ids[i]=xNew<int>(this->common_recv[i]);
			for(int j=0;j<this->common_recv[i];j++) output->common_recv_ids[i][j]=this->common_recv_ids[i][j];
		}
	}
	if(this->common_send_ids){
		output->common_send_ids = xNew<int*>(num_proc);
		for(int i=0;i<num_proc;i++){
			output->common_send_ids[i]=xNew<int>(this->common_send[i]);
			for(int j=0;j<this->common_send[i];j++) output->common_send_ids[i][j]=this->common_send_ids[i][j];
		}
	}

	return output;
}
/*}}}*/
void Vertices::Marshall(char** pmarshalled_data,int* pmarshalled_data_size, int marshall_direction){ /*{{{*/

	int num_procs=IssmComm::GetSize();
	int test = num_procs;
	MARSHALLING_ENUM(VerticesEnum);
	MARSHALLING(test);
	if(test!=num_procs) _error_("number of cores is not the same as before");
	MARSHALLING_DYNAMIC(this->common_recv,int,num_procs);
	MARSHALLING_DYNAMIC(this->common_send,int,num_procs);
	if(marshall_direction == MARSHALLING_BACKWARD){
		this->common_recv_ids = xNew<int*>(num_procs);
		this->common_send_ids = xNew<int*>(num_procs);
	}
	for(int i=0;i<num_procs;i++){
		MARSHALLING_DYNAMIC(this->common_recv_ids[i],int,this->common_recv[i]);
		MARSHALLING_DYNAMIC(this->common_send_ids[i],int,this->common_send[i]);
	}
	DataSet::Marshall(pmarshalled_data,pmarshalled_data_size,marshall_direction);
}
/*}}}*/
int Vertices::NumberOfVertices(void){/*{{{*/

	int i,sid;
	int max_sid=0;
	int vertex_max_sid;

	if(this->Size()==0) return 0;

	for(i=0;i<this->Size();i++){
		Vertex* vertex=xDynamicCast<Vertex*>(this->GetObjectByOffset(i));
		sid=vertex->Sid();
		if (sid>max_sid)max_sid=sid;
	}

	ISSM_MPI_Reduce (&max_sid,&vertex_max_sid,1,ISSM_MPI_INT,ISSM_MPI_MAX,0,IssmComm::GetComm() );
	ISSM_MPI_Bcast(&vertex_max_sid,1,ISSM_MPI_INT,0,IssmComm::GetComm());
	max_sid=vertex_max_sid;

	/*sid starts at 0*/
	max_sid++;

	/*return:*/
	return max_sid;
}
/*}}}*/
void Vertices::LatLonList(IssmDouble** plat,IssmDouble** plon){/*{{{*/

	/*output: */
	IssmDouble* xyz_serial=NULL;

	/*recover my_rank:*/
	int my_rank=IssmComm::GetRank();

	/*First, figure out number of vertices: */
	int num_vertices=this->NumberOfVertices();

	/*Now, allocate vectors*/
	Vector<IssmDouble>* lat = new Vector<IssmDouble>(num_vertices);
	Vector<IssmDouble>* lon = new Vector<IssmDouble>(num_vertices);

	/*Go through vertices, and for each vertex, object, report it cpu: */
	for(int i=0;i<this->Size();i++){
		Vertex* vertex=xDynamicCast<Vertex*>(this->GetObjectByOffset(i));
		lat->SetValue(vertex->sid,vertex->GetLatitude() ,INS_VAL);
		lon->SetValue(vertex->sid,vertex->GetLongitude(),INS_VAL);
	}

	/*Assemble:*/
	lat->Assemble();
	lon->Assemble();

	/*gather on cpu 0: */
	IssmDouble* lat_serial=lat->ToMPISerial();
	IssmDouble* lon_serial=lon->ToMPISerial();

	/*free ressources: */
	*plat = lat_serial;
	*plon = lon_serial;
	delete lat;
	delete lon;
}
/*}}}*/
