/*
 * \file Nodes.cpp
 * \brief: Implementation of Nodes class, derived from DataSet class.
 */

/*Headers: {{{*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../shared/io/Comm/IssmComm.h"
#include "./Nodes.h"
#include "./Node.h"

using namespace std;
/*}}}*/

/*Object constructors and destructor*/
Nodes::Nodes(){/*{{{*/
	enum_type=NodesEnum;
	this->common_recv     = NULL;
	this->common_recv_ids = NULL;
	this->common_send     = NULL;
	this->common_send_ids = NULL;
	return;
}
/*}}}*/
Nodes::~Nodes(){/*{{{*/
	int num_proc=IssmComm::GetSize();

	if(this->common_recv) xDelete<int>(common_recv);
	if(this->common_send) xDelete<int>(common_send);
	if(this->common_recv_ids){
		for(int i=0;i<num_proc;i++) if(common_recv_ids[i]) xDelete<int>(common_recv_ids[i]);
		xDelete<int*>(common_recv_ids);
	}
	if(this->common_send_ids){
		for(int i=0;i<num_proc;i++) if(common_send_ids[i]) xDelete<int>(common_send_ids[i]);
		xDelete<int*>(common_send_ids);
	}
	return;
}
/*}}}*/

/*Numerics*/
Nodes* Nodes::Copy() {/*{{{*/

	int num_proc = IssmComm::GetSize();

	/*Copy dataset*/
	Nodes* output=new Nodes();
	output->sorted=this->sorted;
	output->numsorted=this->numsorted;
	output->presorted=this->presorted;
	for(vector<Object*>::iterator obj=this->objects.begin() ; obj < this->objects.end(); obj++ ){
		output->AddObject((*obj)->copy());
	}

	/*Build id_offsets and sorted_ids*/
	int objsize = this->numsorted;
	if(this->sorted && objsize>0 && this->id_offsets){	
		/*Allocate new ids*/
		output->id_offsets=xNew<int>(objsize);
		xMemCpy<int>(output->id_offsets,this->id_offsets,objsize);
	}
	else output->id_offsets=NULL;
	if(this->sorted && objsize>0 && this->sorted_ids){
		/*Allocate new ids*/
		output->sorted_ids=xNew<int>(objsize);
		xMemCpy<int>(output->sorted_ids,this->sorted_ids,objsize);
	}
	else output->sorted_ids=NULL;


	if(this->common_recv){
		output->common_recv=xNew<int>(num_proc);
		for(int i=0;i<num_proc;i++) output->common_recv[i]=this->common_recv[i];
	}
	if(this->common_send){
		output->common_send=xNew<int>(num_proc);
		for(int i=0;i<num_proc;i++) output->common_send[i]=this->common_send[i];
	}
	if(this->common_recv_ids){
		output->common_recv_ids = xNew<int*>(num_proc);
		for(int i=0;i<num_proc;i++){
			output->common_recv_ids[i]=xNew<int>(this->common_recv[i]);
			for(int j=0;j<this->common_recv[i];j++) output->common_recv_ids[i][j]=this->common_recv_ids[i][j];
		}
	}
	if(this->common_send_ids){
		output->common_send_ids = xNew<int*>(num_proc);
		for(int i=0;i<num_proc;i++){
			output->common_send_ids[i]=xNew<int>(this->common_send[i]);
			for(int j=0;j<this->common_send[i];j++) output->common_send_ids[i][j]=this->common_send_ids[i][j];
		}
	}

	return output;
}
/*}}}*/
void  Nodes::Marshall(char** pmarshalled_data,int* pmarshalled_data_size, int marshall_direction){ /*{{{*/

	int num_procs=IssmComm::GetSize();
	int test = num_procs;
	MARSHALLING_ENUM(NodesEnum);
	MARSHALLING(test);
	if(test!=num_procs) _error_("number of cores is not the same as before");

	DataSet::Marshall(pmarshalled_data,pmarshalled_data_size,marshall_direction);

	if(marshall_direction == MARSHALLING_BACKWARD){
		this->common_recv_ids = xNew<int*>(num_procs);
		this->common_send_ids = xNew<int*>(num_procs);
		for(int i=0;i<num_procs;i++){
			this->common_recv_ids[i] = NULL;
			this->common_send_ids[i] = NULL;
		}
	}

	/*Stop here if no nodes*/
	if(this->Size()==0) return;

	MARSHALLING_DYNAMIC(this->common_recv,int,num_procs);
	MARSHALLING_DYNAMIC(this->common_send,int,num_procs);
	for(int i=0;i<num_procs;i++){
		if(this->common_recv[i]) MARSHALLING_DYNAMIC(this->common_recv_ids[i],int,this->common_recv[i]);
		if(this->common_send[i]) MARSHALLING_DYNAMIC(this->common_send_ids[i],int,this->common_send[i]);
	}
}
/*}}}*/
void  Nodes::DistributeDofs(int setenum){/*{{{*/

	/*some check: */
	_assert_(setenum==GsetEnum || setenum==FsetEnum || setenum==SsetEnum);

	int  i;
	int  dofcount=0;
	int  maxdofspernode=0;
	int* alldofcount=NULL;
	int* truedofs=NULL;
	int* alltruedofs=NULL;
	int  numnodes=0;

	/*recover my_rank:*/
	int my_rank   = IssmComm::GetRank();
	int num_procs = IssmComm::GetSize();

	/*Go through objects, and distribute dofs locally, from 0 to numberofdofsperobject*/
	for(i=0;i<this->Size();i++){
		Node* node=xDynamicCast<Node*>(this->GetObjectByOffset(i));
		node->DistributeDofs(&dofcount,setenum);
	}

	/* Now every object has distributed dofs, but locally, and with a dof count starting from 
	 * 0. This means the dofs between all the cpus are not unique. We now offset the dofs of each
	 * cpus by the total last dofs of the previus cpu, starting from 0.
	 * First: get number of dofs for each cpu*/
	alldofcount=xNew<int>(num_procs);
	ISSM_MPI_Gather(&dofcount,1,ISSM_MPI_INT,alldofcount,1,ISSM_MPI_INT,0,IssmComm::GetComm());
	ISSM_MPI_Bcast(alldofcount,num_procs,ISSM_MPI_INT,0,IssmComm::GetComm());

	/* Every cpu should start its own dof count at the end of the dofcount from cpu-1*/
	dofcount=0;
	for(i=0;i<my_rank;i++){
		dofcount+=alldofcount[i];
	}
	for(i=0;i<this->Size();i++){
		/*Check that this node corresponds to our analysis currently being carried out: */
		Node* node=xDynamicCast<Node*>(this->GetObjectByOffset(i));
		node->OffsetDofs(dofcount,setenum);
	}

	/* Finally, remember that cpus may have skipped some objects, because they were clones. For every 
	 * object that is not a clone, tell them to show their dofs, so that later on, they can get picked 
	 * up by their clones: */
	maxdofspernode=this->MaxNumDofs(setenum);
	numnodes=this->NumberOfNodes();
	if(numnodes*maxdofspernode){
		truedofs=   xNewZeroInit<int>(numnodes*maxdofspernode); //initialize to 0, so that we can pick up the max
		alltruedofs=xNewZeroInit<int>(numnodes*maxdofspernode);
	}

	for(i=0;i<this->Size();i++){
		Node* node=xDynamicCast<Node*>(this->GetObjectByOffset(i));
		node->ShowTrueDofs(truedofs,maxdofspernode,setenum);//give maxdofspernode, column size, so that nodes can index into truedofs
	}

	ISSM_MPI_Allreduce((void*)truedofs,(void*)alltruedofs,numnodes*maxdofspernode,ISSM_MPI_INT,ISSM_MPI_MAX,IssmComm::GetComm());

	/* Now every cpu knows the true dofs of everyone else that is not a clone*/
	for(i=0;i<this->Size();i++){
		Node* node=xDynamicCast<Node*>(this->GetObjectByOffset(i));
		node->UpdateCloneDofs(alltruedofs,maxdofspernode,setenum);
	}

	/*Update indexingupdateflag*/
	for(i=0;i<this->Size();i++){
		Node* node=xDynamicCast<Node*>(this->GetObjectByOffset(i));
		node->ReindexingDone();
	}

	/* Free ressources: */
	xDelete<int>(alldofcount);
	xDelete<int>(truedofs);
	xDelete<int>(alltruedofs);
}
/*}}}*/
int   Nodes::MaxNumDofs(int setenum){/*{{{*/

	int max=0;
	int allmax;

	/*Now go through all nodes, and get how many dofs they own, unless they are clone nodes: */
	for(int i=0;i<this->Size();i++){
		Node* node=xDynamicCast<Node*>(this->GetObjectByOffset(i));

		int numdofs=node->GetNumberOfDofs(NoneApproximationEnum,setenum);
		if(numdofs>max) max=numdofs;
	}

	/*Grab max of all cpus: */
	ISSM_MPI_Allreduce((void*)&max,(void*)&allmax,1,ISSM_MPI_INT,ISSM_MPI_MAX,IssmComm::GetComm());
	max=allmax;

	return max;
}
/*}}}*/
int   Nodes::NumberOfDofs(int setenum){/*{{{*/

	int   allnumdofs;

	/*Get number of dofs on current cpu (excluding clones)*/
	int numdofs=this->NumberOfDofsLocal(setenum);

	/*Gather from all cpus: */
	ISSM_MPI_Allreduce ((void*)&numdofs,(void*)&allnumdofs,1,ISSM_MPI_INT,ISSM_MPI_SUM,IssmComm::GetComm());
	return allnumdofs;
}
/*}}}*/
int   Nodes::NumberOfDofsLocal(int setenum){/*{{{*/

	int   numdofs=0;

	/*Now go through all nodes, and get how many dofs they own, unless they are clone nodes: */
	for(int i=0;i<this->Size();i++){
		Node* node=xDynamicCast<Node*>(this->GetObjectByOffset(i));

		/*Ok, this object is a node, ask it to plug values into partition: */
		if (!node->IsClone()){
			numdofs+=node->GetNumberOfDofs(NoneApproximationEnum,setenum);
		}
	}

	return numdofs;
}
/*}}}*/
int   Nodes::NumberOfNodes(void){/*{{{*/

	/*Careful! only use once all clones have been setup for all nodes!: */

	int numnodes=0;
	int allnumnodes=0;

	/*Now go through all nodes, and get how many dofs they own, unless they are clone nodes: */
	for(int i=0;i<this->Size();i++){
		Node* node=xDynamicCast<Node*>(this->GetObjectByOffset(i));

		/*Ok, this object is a node, ask it to plug values into partition: */
		if (!node->IsClone()) numnodes++;
	}

	/*Gather from all cpus: */
	ISSM_MPI_Allreduce((void*)&numnodes,(void*)&allnumnodes,1,ISSM_MPI_INT,ISSM_MPI_SUM,IssmComm::GetComm());

	return allnumnodes;
}
/*}}}*/
bool  Nodes::RequiresDofReindexing(void){/*{{{*/

	int flag = 0;
	int allflag;

	/*Now go through all nodes, and get how many dofs they own, unless they are clone nodes: */
	for(int i=0;i<this->Size();i++){
		Node* node=xDynamicCast<Node*>(this->GetObjectByOffset(i));
		if(node->RequiresDofReindexing()){
			flag = 1;
			break;
		}
	}

	/*Grab max of all cpus: */
	ISSM_MPI_Allreduce((void*)&flag,(void*)&allflag,1,ISSM_MPI_INT,ISSM_MPI_MAX,IssmComm::GetComm());

	if(allflag) return true;
	else        return false;
}
/*}}}*/
