/*!\file GetMaskOfIceVerticesLSMx 
 * \brief: Return a mask for all the vertices determining whether the node should be active or not. 
 */

#include "./SetActiveNodesLSMx.h"

#include "../../classes/classes.h"
#include "../../shared/shared.h"
#include "../../toolkits/toolkits.h"
#include "../modules.h"

void SetActiveNodesLSMx(FemModel* femmodel){/*{{{*/
	/* activate/deactivate nodes for levelset method according to IceMaskNodeActivation */

	/* find parameters */
	int domaintype;
	femmodel->parameters->FindParam(&domaintype,DomainTypeEnum);

	for(int i=0;i<femmodel->elements->Size();i++){
		Element    *element  = xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
		int         numnodes = element->GetNumberOfNodes();
		IssmDouble *mask     = xNew<IssmDouble>(numnodes);

		/*include switch for elements with multiple different sets of nodes*/
		switch(element->GetElementType()){
			case MINIEnum:case MINIcondensedEnum:
			case TaylorHoodEnum:case XTaylorHoodEnum:case LATaylorHoodEnum:
			case CrouzeixRaviartEnum:case LACrouzeixRaviartEnum:case OneLayerP4zEnum:{
				Input* input=element->GetInput(IceMaskNodeActivationEnum);
				if(!input) _error_("Input " << EnumToStringx(IceMaskNodeActivationEnum) << " not found in element");

				/* Start looping on the number of vertices: */
				Gauss* gauss=element->NewGauss();
				for(int iv=0;iv<element->NumberofNodesVelocity();iv++){
					gauss->GaussNode(element->VelocityInterpolation(),iv);
					input->GetInputValue(&mask[iv],gauss);
				}
				for(int iv=0;iv<element->NumberofNodesPressure();iv++){
					gauss->GaussNode(element->PressureInterpolation(),iv);
					input->GetInputValue(&mask[element->NumberofNodesVelocity()+iv],gauss);
				}
				delete gauss;
				break;
			}
			default:
				element->GetInputListOnNodes(&mask[0],IceMaskNodeActivationEnum);
				break;
		}

		for(int in=0;in<numnodes;in++){
			Node* node=element->GetNode(in);
			if(mask[in]==1.) node->Activate();
			else             node->Deactivate();
		}
		xDelete<IssmDouble>(mask);
	}
}/*}}}*/

void GetMaskOfIceVerticesLSMx0(FemModel* femmodel){/*{{{*/

	/*Initialize vector with number of vertices*/
	int numvertices=femmodel->vertices->NumberOfVertices();
	if(numvertices==0)  return;
	Vector<IssmDouble>* vec_mask_ice=new Vector<IssmDouble>(numvertices);

	/*Fill vector with values: */
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
		if(element->IsIceInElement()){
			int nbv = element->GetNumberOfVertices();
			for(int iv=0;iv<nbv;iv++){
				vec_mask_ice->SetValue(element->vertices[iv]->Sid(),1.,INS_VAL);
			}
		}
	}

	/*Assemble vector and serialize */
	vec_mask_ice->Assemble();
	InputUpdateFromVectorx(femmodel,vec_mask_ice,IceMaskNodeActivationEnum,VertexSIdEnum);
	delete vec_mask_ice;
}/*}}}*/
void GetMaskOfIceVerticesLSMx(FemModel* femmodel){/*{{{*/

	femmodel->SetCurrentConfiguration(LevelsetAnalysisEnum);

	/*Create vector on gset*/
	int gsize=femmodel->nodes->NumberOfDofs(GsetEnum);
	if(gsize==0)  return;
	Vector<IssmDouble>* vec_mask_ice=new Vector<IssmDouble>(gsize);

	/*Fill vector with values: */
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));

		if(element->IsIceInElement()){
			int numnodes = element->GetNumberOfNodes();
			int  gsize_local=GetNumberOfDofs(element->nodes,numnodes,GsetEnum,NoneEnum);
			int* glist_local=GetGlobalDofList(element->nodes,numnodes,GsetEnum,NoneEnum);
			IssmDouble* ones = xNew<IssmDouble>(gsize_local);
			for(int n=0;n<gsize_local;n++) ones[n] = 1.;
			vec_mask_ice->SetValues(gsize_local,glist_local,ones,INS_VAL);
			xDelete<IssmDouble>(ones);
			xDelete<int>(glist_local);
		}
	}

	/*Assemble vector and serialize */
	vec_mask_ice->Assemble();
	IssmDouble* mask_ice=vec_mask_ice->ToMPISerial();
	delete vec_mask_ice;
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
		element->InputUpdateFromSolutionOneDof(mask_ice,IceMaskNodeActivationEnum);
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(mask_ice);
}/*}}}*/
