/*! \file IoModel.cpp
 * \brief  IoModel structure that mirrors the matlab workspace structure. Servers for the serial 
 * and parallel runs.
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../shared/shared.h"
#include "../io/io.h"
#include "../include/globals.h"
#include "../include/macros.h"

#include <string.h>
#include "stdio.h"

#include "./IoModel.h"


/*!--------------------------------------------------
	NewIoModel
  --------------------------------------------------*/

IoModel* NewIoModel(void) {
	/*! create a new IoModel object */
	IoModel* iomodel=NULL;

	iomodel=(IoModel*)xmalloc(sizeof(IoModel));

	/*!initialize all pointers to 0: */
	iomodel->name=NULL;
	iomodel->repository=NULL;
	iomodel->meshtype=NULL;
	iomodel->analysis_type=0;
	iomodel->sub_analysis_type=0;
	iomodel->qmu_analysis=0;
	iomodel->control_analysis=0;
	iomodel->solverstring=NULL;
	iomodel->numberofresponses=0;
	iomodel->numberofvariables=0;
	iomodel->qmu_npart=0; 
	iomodel->numberofelements=0;
	iomodel->numberofnodes=0;
	iomodel->x=NULL; 
	iomodel->y=NULL;
	iomodel->z=NULL;
	iomodel->elements=NULL;
	iomodel->elements_type=NULL;
	iomodel->numberofnodes2d=0;
	iomodel->elements2d=NULL;
	iomodel->deadgrids=NULL;
	iomodel->numlayers=0;
	iomodel->uppernodes=NULL;
	iomodel->gridonhutter=NULL;
	iomodel->gridonmacayeal=NULL;
	iomodel->gridonpattyn=NULL;
	
	iomodel->vx_obs=NULL;
	iomodel->vy_obs=NULL;
	iomodel->vx=NULL;
	iomodel->vy=NULL;
	iomodel->vz=NULL;
	iomodel->pressure=NULL;
	iomodel->temperature=NULL;
	iomodel->melting=NULL;
	iomodel->geothermalflux=NULL;
	iomodel->elementonbed=NULL;
	iomodel->elementonsurface=NULL;
	iomodel->gridonbed=NULL;
	iomodel->gridonsurface=NULL;
	iomodel->gridonstokes=NULL;
	iomodel->borderstokes=NULL;
	iomodel->thickness=NULL;
	iomodel->surface=NULL;
	iomodel->bed=NULL;
	iomodel->elementoniceshelf=NULL;
	iomodel->elementonwater=NULL;
	iomodel->gridonicesheet=NULL;
	iomodel->gridoniceshelf=NULL;

	iomodel->drag_type=0;
	iomodel->drag=NULL;
	iomodel->p=NULL;
	iomodel->q=NULL;
	
	
	iomodel->numberofpressureloads=0;
	iomodel->numberofpressureloads_stokes=0;
	iomodel->pressureload=NULL;
	iomodel->pressureload_stokes=NULL;
	iomodel-> spcvelocity=NULL;
	iomodel-> spctemperature=NULL;
	iomodel-> spcthickness=NULL;
	
	/*!materials: */
	iomodel->rho_water=0;
	iomodel->rho_ice=0;
	iomodel->g=0;
	iomodel->n=NULL;
	iomodel->B=NULL;

	/*!control methods: */
	iomodel->control_type=NULL;

	/*!solution parameters: */
	iomodel->fit=NULL;
	iomodel->cm_jump=NULL;
	iomodel->meanvel=0;
	iomodel->epsvel=0;
	iomodel->artificial_diffusivity=0;
	iomodel->nsteps=0;
	iomodel->eps_cm=0;
	iomodel->tolx=0;
	iomodel->maxiter=NULL;
	iomodel->cm_noisedmp=0;
	iomodel->cm_mindmp_value=0;
	iomodel->cm_mindmp_slope=0;
	iomodel->cm_maxdmp_value=0;
	iomodel->cm_maxdmp_slope=0;
	iomodel->cm_min=0;
	iomodel->cm_max=0;
	iomodel->verbose=0;
	iomodel->plot=0;
	iomodel->eps_res=0;
	iomodel->eps_rel=0;
	iomodel->eps_abs=0;
	iomodel->dt=0;
	iomodel->ndt=0;
	iomodel->penalty_offset=0;
	iomodel->penalty_melting=0;
	iomodel->penalty_lock=0;
	iomodel->sparsity=0;
	iomodel->connectivity=0;
	iomodel->lowmem=0;
	iomodel->optscal=NULL;
	iomodel->yts=0;
	iomodel->viscosity_overshoot=0;
	iomodel->artdiff=0;
	iomodel->stokesreconditioning=0;
	iomodel->waitonlock=0;

	/*!thermal parameters: */
	iomodel->beta=0;
	iomodel->meltingpoint=0;
	iomodel->latentheat=0;
	iomodel->heatcapacity=0;
	iomodel->thermalconductivity=0;
	iomodel->min_thermal_constraints=0;
	iomodel->stabilize_constraints=0;
	iomodel->mixed_layer_capacity=0;
	iomodel->thermal_exchange_velocity=0;

	
	iomodel->numrifts=0;
	iomodel->riftinfo=NULL;

	/*!penalties: */
	iomodel->numpenalties=0;
	iomodel->penalties=NULL;
	iomodel->penaltypartitioning=NULL;

	/*!basal: */
	iomodel->accumulation=NULL;
	
	/*parameter output: */
	iomodel->numoutput=0;

	/*elements type: */
	iomodel->ishutter=0;
	iomodel->ismacayealpattyn=0;
	iomodel->isstokes=0;


	iomodel->epart=NULL;
	iomodel->npart=NULL;
	iomodel->my_grids=NULL;
	iomodel->my_bordergrids=NULL;

	return iomodel;
}


/*!--------------------------------------------------
	DeleteIoModel
  --------------------------------------------------*/

void DeleteIoModel(IoModel** piomodel){

	/*!Recover structure: */
	IoModel* iomodel = *piomodel;
	
	int i;

	/*!Two cases here: 
	 * - serial mode: matlab's memory manager will take care of delete iomodel when returning from Imp. Do nothing here, so as not to confuse 
	 *                the memory manager.
     * - in parallel, anything the io layer does (FetchData) did needs to be erased explicitely in the iomodel.
	 */

	#ifdef _PARALLEL_
	xfree((void**)&iomodel->x);
	xfree((void**)&iomodel->y);
	xfree((void**)&iomodel->z);
	xfree((void**)&iomodel->elements);
	xfree((void**)&iomodel->elements_type);
	xfree((void**)&iomodel->gridonhutter);
	xfree((void**)&iomodel->gridonmacayeal);
	if (strcmp(iomodel->meshtype,"3d")==0){
		xfree((void**)&iomodel->elements2d);
		xfree((void**)&iomodel->deadgrids);
		xfree((void**)&iomodel->uppernodes);
		xfree((void**)&iomodel->gridonpattyn);
	}
	xfree((void**)&iomodel->solverstring);
	xfree((void**)&iomodel->elementonbed);
	xfree((void**)&iomodel->elementonsurface);
	xfree((void**)&iomodel->gridonbed);
	xfree((void**)&iomodel->gridonsurface);
	xfree((void**)&iomodel->gridonstokes);
	xfree((void**)&iomodel->borderstokes);
	xfree((void**)&iomodel->thickness);
	xfree((void**)&iomodel->surface);
	xfree((void**)&iomodel->bed);
	xfree((void**)&iomodel->vx_obs);
	xfree((void**)&iomodel->vy_obs);
	xfree((void**)&iomodel->vx);
	xfree((void**)&iomodel->vy);
	xfree((void**)&iomodel->vz);
	xfree((void**)&iomodel->pressure);
	xfree((void**)&iomodel->temperature);
	xfree((void**)&iomodel->drag);
	xfree((void**)&iomodel->p);
	xfree((void**)&iomodel->q);
	xfree((void**)&iomodel->elementoniceshelf);
	xfree((void**)&iomodel->elementonwater);
	xfree((void**)&iomodel->gridonicesheet);
	xfree((void**)&iomodel->gridoniceshelf);
	xfree((void**)&iomodel->pressureload);
	xfree((void**)&iomodel->pressureload_stokes);
	xfree((void**)&iomodel->spcvelocity);
	xfree((void**)&iomodel->spcthickness);
	xfree((void**)&iomodel->spctemperature);
	xfree((void**)&iomodel->geothermalflux);
	xfree((void**)&iomodel->melting);
	xfree((void**)&iomodel->accumulation);
	xfree((void**)&iomodel->B);
	xfree((void**)&iomodel->n);
	xfree((void**)&iomodel->fit);
	xfree((void**)&iomodel->cm_jump);
	xfree((void**)&iomodel->optscal);
	xfree((void**)&iomodel->maxiter);


	/*!Delete structure fields: */
	xfree((void**)&iomodel->repository);
	xfree((void**)&iomodel->meshtype);
	xfree((void**)&iomodel->name);
	
	xfree((void**)&iomodel->riftinfo);
	
	xfree((void**)&iomodel->penalties);
	xfree((void**)&iomodel->penaltypartitioning);
	
	xfree((void**)&iomodel->control_type);
	
	xfree((void**)&iomodel->epart);
	xfree((void**)&iomodel->npart);
	xfree((void**)&iomodel->my_grids);
	xfree((void**)&iomodel->my_bordergrids);
	
	/*!Delete entire structure: */
	xfree((void**)piomodel);
	#endif
}

/*!--------------------------------------------------
	IoModelInit
  --------------------------------------------------*/

#undef __FUNCT__ 
#define __FUNCT__ "IoModelInit"

int	IoModelInit(IoModel** piomodel,ConstDataHandle iomodel_handle){

	int i,j;
	
	/*output: */
	IoModel* iomodel=NULL;

	/*Allocate iomodel: */
	iomodel=NewIoModel();

	/*In IoModelInit, we get all the data that is not difficult to get, and that is small: */
	IoModelFetchData(&iomodel->name,iomodel_handle,"name"); 
	IoModelFetchData(&iomodel->analysis_type,iomodel_handle,"analysis_type"); 
	IoModelFetchData(&iomodel->sub_analysis_type,iomodel_handle,"sub_analysis_type"); 
	IoModelFetchData(&iomodel->qmu_analysis,iomodel_handle,"qmu_analysis"); 
	IoModelFetchData(&iomodel->control_analysis,iomodel_handle,"control_analysis"); 
	IoModelFetchData(&iomodel->meshtype,iomodel_handle,"type");
	/*!Get numberofelements and numberofnodes: */
	IoModelFetchData(&iomodel->numberofnodes,iomodel_handle,"numberofgrids");
	IoModelFetchData(&iomodel->numberofelements,iomodel_handle,"numberofelements");
	/*!In case we are running 3d, we are going to need the collapsed and non-collapsed 2d meshes, from which the 3d mesh was extruded: */
	if (strcmp(iomodel->meshtype,"3d")==0){
	
		/*!Deal with 2d mesh: */
		IoModelFetchData(&iomodel->numberofelements2d,iomodel_handle,"numberofelements2d");
		IoModelFetchData(&iomodel->numberofnodes2d,iomodel_handle,"numberofgrids2d");
		IoModelFetchData(&iomodel->numlayers,iomodel_handle,"numlayers");
	}


	/*elements type: */
	IoModelFetchData(&iomodel->ishutter,iomodel_handle,"ishutter");
	IoModelFetchData(&iomodel->ismacayealpattyn,iomodel_handle,"ismacayealpattyn");
	IoModelFetchData(&iomodel->isstokes,iomodel_handle,"isstokes");

	/*!Get drag_type, drag and p,q: */
	IoModelFetchData(&iomodel->drag_type,iomodel_handle,"drag_type");

	/*!Get materials: */
	IoModelFetchData(&iomodel->rho_water,iomodel_handle,"rho_water");
	IoModelFetchData(&iomodel->rho_ice,iomodel_handle,"rho_ice");
	IoModelFetchData(&iomodel->g,iomodel_handle,"g");

	/*Get control parameters: */
	IoModelFetchData(&iomodel->control_type,iomodel_handle,"control_type"); 

	/*!Get solution parameters: */
	IoModelFetchData(&iomodel->yts,iomodel_handle,"yts");
	IoModelFetchData(&iomodel->meanvel,iomodel_handle,"meanvel");
	IoModelFetchData(&iomodel->epsvel,iomodel_handle,"epsvel");
	IoModelFetchData(&iomodel->verbose,iomodel_handle,"verbose");
	IoModelFetchData(&iomodel->plot,iomodel_handle,"plot");
	IoModelFetchData(&iomodel->artificial_diffusivity,iomodel_handle,"artificial_diffusivity");
	IoModelFetchData(&iomodel->nsteps,iomodel_handle,"nsteps");
	IoModelFetchData(&iomodel->eps_cm,iomodel_handle,"eps_cm");
	IoModelFetchData(&iomodel->tolx,iomodel_handle,"tolx");
	IoModelFetchData(&iomodel->cm_noisedmp,iomodel_handle,"cm_noisedmp");
	IoModelFetchData(&iomodel->cm_mindmp_value,iomodel_handle,"cm_mindmp_value");
	IoModelFetchData(&iomodel->cm_mindmp_slope,iomodel_handle,"cm_mindmp_slope");
	IoModelFetchData(&iomodel->cm_maxdmp_value,iomodel_handle,"cm_maxdmp_value");
	IoModelFetchData(&iomodel->cm_maxdmp_slope,iomodel_handle,"cm_maxdmp_slope");
	IoModelFetchData(&iomodel->cm_min,iomodel_handle,"cm_min");
	IoModelFetchData(&iomodel->cm_max,iomodel_handle,"cm_max");
	IoModelFetchData(&iomodel->eps_res,iomodel_handle,"eps_res");
	IoModelFetchData(&iomodel->eps_rel,iomodel_handle,"eps_rel");
	IoModelFetchData(&iomodel->eps_abs,iomodel_handle,"eps_abs");
	IoModelFetchData(&iomodel->dt,iomodel_handle,"dt");
	IoModelFetchData(&iomodel->ndt,iomodel_handle,"ndt");
	IoModelFetchData(&iomodel->penalty_offset,iomodel_handle,"penalty_offset");
	IoModelFetchData(&iomodel->penalty_melting,iomodel_handle,"penalty_melting");
	IoModelFetchData(&iomodel->penalty_lock,iomodel_handle,"penalty_lock");
	IoModelFetchData(&iomodel->sparsity,iomodel_handle,"sparsity");
	IoModelFetchData(&iomodel->connectivity,iomodel_handle,"connectivity");
	IoModelFetchData(&iomodel->lowmem,iomodel_handle,"lowmem");
	IoModelFetchData(&iomodel->solverstring,iomodel_handle,"solverstring");
	IoModelFetchData(&iomodel->viscosity_overshoot,iomodel_handle,"viscosity_overshoot");
	IoModelFetchData(&iomodel->artdiff,iomodel_handle,"artificial_diffusivity");
	IoModelFetchData(&iomodel->stokesreconditioning,iomodel_handle,"stokesreconditioning");
	IoModelFetchData(&iomodel->waitonlock,iomodel_handle,"waitonlock");

	/*!Get thermal parameters: */
	IoModelFetchData(&iomodel->beta,iomodel_handle,"beta");
	IoModelFetchData(&iomodel->meltingpoint,iomodel_handle,"meltingpoint");
	IoModelFetchData(&iomodel->latentheat,iomodel_handle,"latentheat");
	IoModelFetchData(&iomodel->heatcapacity,iomodel_handle,"heatcapacity");
	IoModelFetchData(&iomodel->thermalconductivity,iomodel_handle,"thermalconductivity");
	IoModelFetchData(&iomodel->min_thermal_constraints,iomodel_handle,"min_thermal_constraints");
	IoModelFetchData(&iomodel->stabilize_constraints,iomodel_handle,"stabilize_constraints");
	IoModelFetchData(&iomodel->mixed_layer_capacity,iomodel_handle,"mixed_layer_capacity");
	IoModelFetchData(&iomodel->thermal_exchange_velocity,iomodel_handle,"thermal_exchange_velocity");
	
	/*qmu: */
	if(iomodel->qmu_analysis){
		IoModelFetchData(&iomodel->numberofvariables,iomodel_handle,"numberofvariables");
		IoModelFetchData(&iomodel->numberofresponses,iomodel_handle,"numberofresponses");
		IoModelFetchData(&iomodel->qmu_npart,iomodel_handle,"npart");
	}
	
	/*parameter output : */
	IoModelFetchData(&iomodel->numoutput,iomodel_handle,"numoutput");

	/*Assign output pointers: */
	*piomodel=iomodel;

	return 1;
}

/*!--------------------------------------------------
	IoModelEcho
  --------------------------------------------------*/
void IoModelEcho(IoModel* iomodel,int which_part,int rank) {

	//which_part  determines what gets echoed, otherwise, we'll get too much output.
	//1-> penalties

	int i,j;

	if(which_part==1 && my_rank==rank && (strcmp(iomodel->meshtype,"3d")==0)){
		printf("IoModel penalties: \n");
		printf("   number of penalties: %i\n",iomodel->numpenalties);
		printf("   grids: \n");

		for(i=0;i<iomodel->numpenalties;i++){
			for(j=0;j<iomodel->numlayers;j++){
				printf("%i ",(int)*(iomodel->penalties+iomodel->numlayers*i+j));
			}
			printf("\n");
		}
	}
	
	return;
}
