/*!\file: ControlInitialization.cpp
 * \brief: ...
 */ 

#undef __FUNCT__ 
#define __FUNCT__ "ControlInitialization"

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./parallel.h"
#include "../issm.h"

void ControlInitialization(Model* model, ParameterInputs* inputs){

	extern int my_rank;

	/*fem models: */
	FemModel* fem_dh=NULL;
	FemModel* fem_dv=NULL;
	FemModel* fem_dhu=NULL;
	FemModel* fem_ds=NULL;
	FemModel* fem_sl=NULL;

	/*solutions: */
	Vec ug=NULL;
	Vec ug_horiz=NULL;
	Vec ug_vert=NULL;
	Vec ug_stokes=NULL;
	Vec pg=NULL;
	Vec slopex=NULL;
	Vec slopey=NULL;

	/*flags: */
	int verbose=0;
	int dim=-1;
	int ishutter=0;
	int ismacayealpattyn=0;
	int isstokes=0;
	int numberofdofspernode_sl;
	int numberofdofspernode_dh;
	int numberofdofspernode_ds;
	int numberofnodes;

	double stokesreconditioning;

	/*dof recovery: */
	int dof01[2]={0,1};
	int dof2[1]={2};
	int dof012[3]={0,1,2};
	int dof3[1]={3};
	double* dofset=NULL;

	/*first recover parameters common to all solutions:*/
	model->FindParam(&verbose,"verbose");
	model->FindParam(&dim,"dim");
	model->FindParam(&ishutter,"ishutter");
	model->FindParam(&ismacayealpattyn,"ismacayealpattyn");
	model->FindParam(&numberofnodes,"numberofnodes");
	model->FindParam(&isstokes,"isstokes");
	model->FindParam(&stokesreconditioning,"stokesreconditioning");

	/*recover fem models: */
	fem_dh=model->GetFormulation(DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	fem_dv=model->GetFormulation(DiagnosticAnalysisEnum(),VertAnalysisEnum());
	fem_ds=model->GetFormulation(DiagnosticAnalysisEnum(),StokesAnalysisEnum());
	fem_dhu=model->GetFormulation(DiagnosticAnalysisEnum(),HutterAnalysisEnum());
	fem_sl=model->GetFormulation(SlopeComputeAnalysisEnum());

	//specific parameters for specific models
	fem_dh->FindParam(&numberofdofspernode_dh,"numberofdofspernode");
	fem_sl->FindParam(&numberofdofspernode_sl,"numberofdofspernode");
	fem_ds->FindParam(&numberofdofspernode_ds,"numberofdofspernode");

	/*if no Stokes, assign output and return*/
	if (!isstokes){
		model->SetActiveFormulation(fem_dh);
		return;
	}

	/*1: compute slopes once for all*/

	//compute slopes
	if(verbose)_printf_("%s\n","computing bed slope (x and y derivatives)...");
	diagnostic_core_linear(&slopex,fem_sl,inputs,SlopeComputeAnalysisEnum(),BedXAnalysisEnum());
	diagnostic_core_linear(&slopey,fem_sl,inputs,SlopeComputeAnalysisEnum(),BedYAnalysisEnum());
	FieldExtrudex( slopex, fem_sl->elements,fem_sl->nodes,fem_sl->loads,fem_sl->materials,fem_sl->parameters,"slopex",0);
	FieldExtrudex( slopey, fem_sl->elements,fem_sl->nodes,fem_sl->loads,fem_sl->materials,fem_sl->parameters,"slopey",0);

	//Add in inputs
	inputs->Add("bedslopex",slopex,numberofdofspernode_sl,numberofnodes);
	inputs->Add("bedslopey",slopey,numberofdofspernode_sl,numberofnodes);
	VecFree(&slopex); VecFree(&slopey);

	/*2: run a complete diagnostic to update spcs*/

	//horizontal velocity
	if(verbose)_printf_("%s\n"," computing horizontal velocities...");
	diagnostic_core_nonlinear(&ug,NULL,NULL,NULL,fem_dh,inputs,DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	if(verbose)_printf_("%s\n"," extruding horizontal velocities...");
	VecDuplicatePatch(&ug_horiz,ug); FieldExtrudex( ug_horiz,fem_dh->elements,fem_dh->nodes, fem_dh->loads,fem_dh-> materials,fem_dh->parameters,"velocity",1);

	//vertical velocity
	if(verbose)_printf_("%s\n"," computing vertical velocities...");
	inputs->Add("velocity",ug_horiz,numberofdofspernode_dh,numberofnodes);
	diagnostic_core_linear(&ug_vert,fem_dv,inputs,DiagnosticAnalysisEnum(),VertAnalysisEnum());

	//Create 3d u_g
	if(verbose)_printf_("%s\n"," combining horizontal and vertical velocities...");
	VecFree(&ug); ug=NewVec(numberofnodes*3);
	xfree((void**)&dofset);dofset=dofsetgen(2,&dof01[0],3,numberofnodes*3); VecMerge(ug,ug_horiz,dofset,numberofnodes*2);
	xfree((void**)&dofset);dofset=dofsetgen(1,&dof2[0],3,numberofnodes*3); VecMerge(ug,ug_vert,dofset,numberofnodes*1);
	VecFree(&ug_vert); VecFree(&ug_horiz);

	//Create 4d u_g
	if(verbose)_printf_("%s\n"," computing pressure according to Pattyn...");
	ComputePressurex( &pg,fem_dh->elements, fem_dh->nodes, fem_dh->loads,  fem_dh->materials, fem_dh->parameters);
	VecScale(pg,1.0/stokesreconditioning);
	ug_stokes=NewVec(fem_ds->nodesets->GetGSize());
	xfree((void**)&dofset);dofset=dofsetgen(3,dof012,4,numberofnodes*4); VecMerge(ug_stokes,ug,dofset,numberofnodes*3);
	xfree((void**)&dofset);dofset=dofsetgen(1,dof3,4,numberofnodes*4); VecMerge(ug_stokes,pg,dofset,numberofnodes);

	//Add in inputs
	inputs->Add("velocity",ug_stokes,numberofdofspernode_ds,numberofnodes);
	VecFree(&ug_stokes);

	//update spcs
	if(verbose)_printf_("%s\n"," update boundary conditions for stokes using velocities previously computed...");
	xfree((void**)&dofset);dofset=dofsetgen(3,dof012,4,numberofnodes*4); VecMerge(fem_ds->yg->vector,ug,dofset,3*numberofnodes);
	
	VecFree(&fem_ds->ys); VecFree(&fem_ds->ys0);
	Reducevectorgtosx(&fem_ds->ys,&fem_ds->ys0, fem_ds->yg->vector,fem_ds->nodesets);

	//Compute Stokes velocities to speed up later runs
	if(verbose)_printf_("%s\n"," computing stokes velocities and pressure ...");
	VecFree(&ug);
	diagnostic_core_nonlinear(&ug,NULL,NULL,NULL,fem_ds,inputs,DiagnosticAnalysisEnum(),StokesAnalysisEnum());

	//Add in inputs
	inputs->Add("velocity",ug,numberofdofspernode_ds,numberofnodes);

	/*Assign output*/
	model->SetActiveFormulation(fem_ds);

	/*Free ressources:*/
	xfree((void**)&dofset);
	VecFree(&ug);
	VecFree(&ug_horiz);
	VecFree(&ug_vert);
	VecFree(&ug_stokes);
	VecFree(&pg);
	VecFree(&slopex);
	VecFree(&slopey);

}
