/*!\file:  prognostic.cpp
 * \brief: prognostic solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "prognostic"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif


int main(int argc,char* *argv){
	
	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   numberofnodes;
	int   waitonlock=0;

	Model* model=NULL;

	Vec     h_g=NULL;
	Vec     u_g=NULL;
	double* u_g_serial=NULL;
	double* h_g_initial=NULL;
	double* melting_g=NULL;
	double* accumulation_g=NULL;
	double  dt;
	double  yts;
	int     qmu_analysis;

	/*Results: */
	DataSet* results=NULL;
	DataSet* processedresults=NULL;
	Result*  result=NULL;

	ParameterInputs* inputs=NULL;
	Param*   param=NULL;

	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	PetscInitialize(&argc,&argv,(char *)0,"");  

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover , input file name and output file name:\n");
	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Initialize model structure: */
	model=new Model();

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	/*Initialize model structure: */
	model=new Model();

	_printf_("read and create finite element model:\n");
	model->AddFormulation(fid,PrognosticAnalysisEnum());

	/*recover parameters: */
	model->FindParam(&waitonlock,"waitonlock");
	model->FindParam(&qmu_analysis,"qmu_analysis");

	_printf_("initialize inputs:\n");
	
	model->FindParam(&u_g_serial,NULL,NULL,"u_g",PrognosticAnalysisEnum());
	model->FindParam(&h_g_initial,NULL,NULL,"h_g",PrognosticAnalysisEnum());
	model->FindParam(&melting_g,NULL,NULL,"m_g",PrognosticAnalysisEnum());
	model->FindParam(&accumulation_g,NULL,NULL,"a_g",PrognosticAnalysisEnum());
	model->FindParam(&dt,"dt");
	model->FindParam(&yts,"yts");
	model->FindParam(&numberofnodes,"numberofnodes");
	
	inputs=new ParameterInputs;
	inputs->Add("velocity",u_g_serial,3,numberofnodes);
	inputs->Add("thickness",h_g_initial,1,numberofnodes);
	inputs->Add("melting",melting_g,1,numberofnodes);
	inputs->Add("accumulation",accumulation_g,1,numberofnodes);
	inputs->Add("dt",dt*yts);

	_printf_("initialize results:\n");
	results=new DataSet(ResultsEnum());

	/*are we running the solutoin sequence, or a qmu wrapper around it? : */
	if(!qmu_analysis){

		/*run prognostic analysis: */
		_printf_("call computational core:\n");
		prognostic_core(results,model,inputs);

	}
	else{

		/*run qmu analysis: */
		_printf_("calling qmu analysis on prognostic core:\n");
	
		#ifdef _HAVE_DAKOTA_ 
		Qmux(model,inputs,PrognosticAnalysisEnum(),NoneAnalysisEnum());
	 	#else
		throw ErrorException(__FUNCT__," Dakota not present, cannot do qmu!");
		#endif
	}

	/*Add analysis_type to results: */
	result=new Result(results->Size()+1,0,1,"analysis_type","prognostic");
	results->AddObject(result);
	
	_printf_("process results:\n");
	ProcessResults(&processedresults,results,model,PrognosticAnalysisEnum());
	
	_printf_("write results to disk:\n");
	OutputResults(processedresults,outputfilename);

	_printf_("write lock file:\n");
	if (waitonlock){
		WriteLockFile(lockname);
	}
		
	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 

	/*end module: */
	MODULEEND();

	/*Free ressources:*/
	delete processedresults;
	delete results;
	delete model;
	delete inputs;

	return 0; //unix success return;
}
