/*!\file Param.c
 * \brief: implementation of the Param object
 */


#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../include/macros.h"
#include "../shared/shared.h"
#include "./objects.h"

		
Param::Param(){
	stringarray=NULL;
	doublevec=NULL;
	doublemat=NULL;
	vec=NULL;
	mat=NULL;

	return;
}

#undef __FUNCT__ 
#define __FUNCT__  "Param constructor"
Param::Param(int param_id, char* param_name, int param_type){

	id=param_id;
	strcpy(name,param_name);
	type=param_type;
	if ((param_type!=STRING) & (param_type!=STRINGARRAY) & (param_type!=INTEGER) & (param_type!=DOUBLE) & 
		(param_type!=DOUBLEVEC) & (param_type!=DOUBLEMAT) & (param_type!=PETSCVEC) & (param_type!=PETSCMAT) 
		){
		throw ErrorException(__FUNCT__,exprintf("%s%i"," unknow parameter type ",param_type));
	}
	stringarray=NULL;
	doublevec=NULL;
	doublemat=NULL;
	vec=NULL;
	mat=NULL;

}

Param::~Param(){
	
	int i;

	switch(type){
	
		case STRING:
			break;

		case STRINGARRAY:
			for(i=0;i<M;i++){
				char* descriptor=stringarray[i];
				xfree((void**)&descriptor);
			}
			xfree((void**)&stringarray);

			break;
	
		case INTEGER:
			break;

	
		case DOUBLE:
			break;
		
		case DOUBLEVEC:
			xfree((void**)&doublevec);
			break;
	
		case DOUBLEMAT:
			xfree((void**)&doublemat);
			break;

		case PETSCVEC:
			VecFree(&vec);
			break;

		case  PETSCMAT:
			MatFree(&mat);
			break;

		default:
			throw ErrorException(__FUNCT__,exprintf("%s%i","unknow parameter type ",type));
	}
}

#undef __FUNCT__ 
#define __FUNCT__  "Param echo"
void Param::Echo(void){

	int i,j;
	
	printf("Param:\n");
	printf("   id: %i\n",id);
	printf("   name: %s\n",name);
	
	switch(type){
		case STRING:
			printf("   string value: %s\n",string);
			break;
			
		case  STRINGARRAY:
			printf("   string array: %i strings\n",M);
			for(i=0;i<M;i++){
				printf("      %i: %s\n",i,stringarray[i]);
			}
			break;
	
		case DOUBLE:
			printf("   double value: %g\n",ddouble);
			break;
		
		case DOUBLEVEC:
			/*printf("   double vector. size: %i ndof: %i\n",M,ndof);
			for(i=0;i<M;i++)printf("%g\n",doublevec[i]);*/
			break;
	
		case DOUBLEMAT:
			printf("   double matrix. size: %i,%i\n",M,N);
			for(i=0;i<M;i++){
				for(j=0;j<N;j++){
					printf("%g ",*(doublemat+N*i+j));
				}
				printf("\n");
			}
			break;

		case PETSCVEC:
			/*printf("   Petsc vector: \n");
			VecView(vec,PETSC_VIEWER_STDOUT_WORLD);*/
			break;

		case  PETSCMAT:
			/*printf("   Petsc matrix: \n");
			MatView(mat,PETSC_VIEWER_STDOUT_WORLD);*/
			break;

		default:
			throw ErrorException(__FUNCT__,exprintf("%s%i","unknow parameter type ",type));
	}
}

#undef __FUNCT__ 
#define __FUNCT__  "Param Deep Echo"
void Param::DeepEcho(void){

	int i,j;
	
	printf("Param:\n");
	printf("   id: %i\n",id);
	printf("   name: %s\n",name);
	
	switch(type){
		case STRING:
			printf("   string value: %s\n",string);
			break;
			
		case  STRINGARRAY:
			printf("   string array: %i strings\n",M);
			for(i=0;i<M;i++){
				printf("      %i: %s\n",i,stringarray[i]);
			}
	
		case DOUBLE:
			printf("   double value: %g\n",ddouble);
			break;
		
		case DOUBLEVEC:
			printf("   double vector. size: %i ndof: %i\n",M,ndof);
			for(i=0;i<M;i++)printf("%g\n",doublevec[i]);
			break;
	
		case DOUBLEMAT:
			printf("   double matrix. size: %i,%i\n",M,N);
			for(i=0;i<M;i++){
				for(j=0;j<N;j++){
					printf("%g ",*(doublemat+N*i+j));
				}
				printf("\n");
			}
			break;

		case PETSCVEC:
			printf("   Petsc vector: \n");
			VecView(vec,PETSC_VIEWER_STDOUT_WORLD);
			break;

		case  PETSCMAT:
			printf("   Petsc matrix: \n");
			MatView(mat,PETSC_VIEWER_STDOUT_WORLD);
			break;

		default:
			throw ErrorException(__FUNCT__,exprintf("%s%i","unknow parameter type ",type));
	}
}	
#undef __FUNCT__
#define __FUNCT__ "Marshall"

void  Param::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	double* serial_vec=NULL; 
	double* serial_mat=NULL;
	int i;
	char* tempstring=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Param: */
	enum_type=ParamEnum();
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Param data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&name,sizeof(name));marshalled_dataset+=sizeof(name);
	memcpy(marshalled_dataset,&type,sizeof(type));marshalled_dataset+=sizeof(type);
	memcpy(marshalled_dataset,&ndof,sizeof(ndof));marshalled_dataset+=sizeof(ndof);

	switch(type){
		case STRING:
			memcpy(marshalled_dataset,&string,sizeof(string));marshalled_dataset+=sizeof(string);
			break;

		case STRINGARRAY:
			memcpy(marshalled_dataset,&M,sizeof(M));marshalled_dataset+=sizeof(M);
			for(i=0;i<M;i++){
				tempstring=stringarray[i];
				int size=(strlen(tempstring)+1)*sizeof(char);
				memcpy(marshalled_dataset,&size,sizeof(size));marshalled_dataset+=sizeof(size);
				memcpy(marshalled_dataset,tempstring,size);marshalled_dataset+=size;
			}
			break;

		case DOUBLE:
			memcpy(marshalled_dataset,&ddouble,sizeof(ddouble));marshalled_dataset+=sizeof(ddouble);
			break;
		case DOUBLEVEC:
			memcpy(marshalled_dataset,&M,sizeof(M));marshalled_dataset+=sizeof(M);
			if(M){
				memcpy(marshalled_dataset,doublevec,M*sizeof(double));
				marshalled_dataset+=(M*sizeof(double));
			}
			break;

		case DOUBLEMAT:
			memcpy(marshalled_dataset,&M,sizeof(M));marshalled_dataset+=sizeof(M);
			memcpy(marshalled_dataset,&N,sizeof(N));marshalled_dataset+=sizeof(N);
			if(M*N){
				memcpy(marshalled_dataset,doublemat,M*N*sizeof(double));
				marshalled_dataset+=(M*N*sizeof(double));
			}
			break;

		case PETSCVEC:
			if(vec){
				VecGetSize(vec,&M);
				VecToMPISerial(&serial_vec,vec);
				memcpy(marshalled_dataset,&M,sizeof(M));marshalled_dataset+=sizeof(M);
				memcpy(marshalled_dataset,serial_vec,M*sizeof(double));marshalled_dataset+=(M*sizeof(double));
				xfree((void**)&serial_vec);
			}
			else{
				M=0;
				memcpy(marshalled_dataset,&M,sizeof(M));marshalled_dataset+=sizeof(M);
			}
			break;

		case PETSCMAT:
			if(mat){
				MatGetSize(mat,&M,&N);
				MatToSerial(&serial_mat,mat);
				memcpy(marshalled_dataset,&M,sizeof(M));marshalled_dataset+=sizeof(M);
				memcpy(marshalled_dataset,&N,sizeof(N));marshalled_dataset+=sizeof(N);
				memcpy(marshalled_dataset,serial_mat,M*N*sizeof(double));marshalled_dataset+=(M*N*sizeof(double));
				xfree((void**)&serial_mat);
			}
			else{
				M=0;
				N=0;
				memcpy(marshalled_dataset,&M,sizeof(M));marshalled_dataset+=sizeof(M);
				memcpy(marshalled_dataset,&N,sizeof(N));marshalled_dataset+=sizeof(N);
			}
			break;
		
		default:
			throw ErrorException(__FUNCT__,exprintf("%s%i","unknown parameter type",type));
			break;
	}

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
		
int   Param::MarshallSize(){

	int size;
	int i;
	char* tempstring=NULL;

	size=sizeof(id)+
		sizeof(name)+
		sizeof(type)+
		sizeof(ndof)+
		sizeof(int); //sizeof(int) for enum type

	switch(type){
		case STRING:
			size+=sizeof(string);
			break;

		case STRINGARRAY:
			size+=sizeof(M);
			for(i=0;i<M;i++){
				tempstring=stringarray[i];
				size+=sizeof(M);
				size+=(strlen(tempstring)+1)*sizeof(char);
			}
			break;

		case DOUBLE:
			size+= sizeof(ddouble);
			break;
		case DOUBLEVEC:
			size+= sizeof(M)+M*sizeof(double);
			break;

		case DOUBLEMAT:
			size+= sizeof(M)+sizeof(N)+M*N*sizeof(double);
			break;

		case PETSCVEC:
			size+= sizeof(M)+M*sizeof(double);
			break;

		case PETSCMAT:
			size+= sizeof(M)+sizeof(N)+M*N*sizeof(double);
			break;

		default:
			throw ErrorException(__FUNCT__,exprintf("%s%i","unknown parameter type",type));
	}

	return size;
}

char* Param::GetName(void){
	return "param";
}
		

void  Param::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	double* serial_vec=NULL;
	double* serial_mat=NULL;
	int* idxm=NULL;
	int* idxn=NULL; 
	int i;
	double sparsity=.001;
	char* tempstring=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&name,marshalled_dataset,sizeof(name));marshalled_dataset+=sizeof(name);
	memcpy(&type,marshalled_dataset,sizeof(type));marshalled_dataset+=sizeof(type);
	memcpy(&ndof,marshalled_dataset,sizeof(ndof));marshalled_dataset+=sizeof(ndof);

	switch(type){

		case STRING:
			memcpy(&string,marshalled_dataset,sizeof(string));marshalled_dataset+=sizeof(string);
			break;
		
		case STRINGARRAY:
			memcpy(&M,marshalled_dataset,sizeof(M));marshalled_dataset+=sizeof(M);
			if(M){
				stringarray=(char**)xmalloc(M*sizeof(char*));
				for(i=0;i<M;i++){
					int size;
					memcpy(&size,marshalled_dataset,sizeof(int));marshalled_dataset+=sizeof(int);
					tempstring=(char*)xmalloc(size);
					memcpy(tempstring,marshalled_dataset,size);marshalled_dataset+=size;
					stringarray[i]=tempstring;
				}
			}
			break;

		case DOUBLE:
			memcpy(&ddouble,marshalled_dataset,sizeof(ddouble));marshalled_dataset+=sizeof(ddouble);
			break;

		case DOUBLEVEC:
	
			memcpy(&M,marshalled_dataset,sizeof(M));marshalled_dataset+=sizeof(M);
			if(M){
				doublevec=(double*)xmalloc(M*sizeof(double));
				memcpy(doublevec,marshalled_dataset,M*sizeof(double));marshalled_dataset+=(M*sizeof(double));
			}
			break;

		case DOUBLEMAT:
			memcpy(&M,marshalled_dataset,sizeof(M));marshalled_dataset+=sizeof(M);
			memcpy(&N,marshalled_dataset,sizeof(N));marshalled_dataset+=sizeof(N);
			if(M*N){
				doublemat=(double*)xmalloc(M*N*sizeof(double));
				memcpy(doublemat,marshalled_dataset,M*N*sizeof(double));marshalled_dataset+=(M*N*sizeof(double));
			}
			break;

		case PETSCVEC:
			memcpy(&M,marshalled_dataset,sizeof(M));marshalled_dataset+=sizeof(M);
			if(M){
				serial_vec=(double*)xmalloc(M*sizeof(double));
				memcpy(serial_vec,marshalled_dataset,M*sizeof(double));marshalled_dataset+=(M*sizeof(double));

				vec=NewVec(M);
				idxm=(int*)xmalloc(M*sizeof(int));
				for(i=0;i<M;i++)idxm[i]=i;
				VecSetValues(vec,M,idxm,serial_vec,INSERT_VALUES);

				VecAssemblyBegin(vec);
				VecAssemblyEnd(vec);
				
				xfree((void**)&serial_vec);
				xfree((void**)&idxm);
			}
			else{
				vec=NULL;
			}
			break;

		case PETSCMAT:
			memcpy(&M,marshalled_dataset,sizeof(M));marshalled_dataset+=sizeof(M);
			memcpy(&N,marshalled_dataset,sizeof(N));marshalled_dataset+=sizeof(N);
			if(M!=0 && N!=0){
				serial_mat=(double*)xmalloc(M*N*sizeof(double));
				memcpy(serial_mat,marshalled_dataset,M*N*sizeof(double));marshalled_dataset+=(M*N*sizeof(double));
				mat=NewMat(M,N,&sparsity,NULL,NULL);
				idxm=(int*)xmalloc(M*sizeof(int));
				idxn=(int*)xmalloc(N*sizeof(int));
				for(i=0;i<M;i++)idxm[i]=i;
				for(i=0;i<N;i++)idxn[i]=i;
				MatSetValues(mat,M,idxm,N,idxn,serial_mat,INSERT_VALUES); 
				MatAssemblyBegin(mat,MAT_FINAL_ASSEMBLY); 
				MatAssemblyEnd(mat,MAT_FINAL_ASSEMBLY);

				xfree((void**)&serial_mat);
				xfree((void**)&idxm);
				xfree((void**)&idxn);
			}
			else{
				mat=NULL;
			}
			break;

		default:
			throw ErrorException(__FUNCT__,exprintf("%s%i","unknown parameter type",type));
	}

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}


int Param::Enum(void){

	return ParamEnum();
}

char* Param::GetParameterName(void){ 
	return name; 
}

#undef __FUNCT__ 
#define __FUNCT__ "GetParameterValue"
void  Param::GetParameterValue(double* pdouble){

	if(type!=DOUBLE)throw ErrorException(__FUNCT__,exprintf("%s%i"," error message: trying to recover a double from a param with type",type));
	*pdouble=ddouble;
}

#undef __FUNCT__ 
#define __FUNCT__ "GetParameterValue"
void  Param::GetParameterValue(int* pinteger){
	
	if(type!=DOUBLE)throw ErrorException(__FUNCT__,exprintf("%s%i"," error message: trying to recover an integer from a param with type",type));
	*pinteger=(int)ddouble;
}

#undef __FUNCT__ 
#define __FUNCT__ "GetParameterValue"
void  Param::GetParameterValue(char** pstring){
	
	char*  outstring=NULL;
	
	if(type!=STRING)throw ErrorException(__FUNCT__,exprintf("%s%i"," error message: trying to recover a string from a param with type",type));
	outstring=(char*)xmalloc((strlen(string)+1)*sizeof(char));
	strcpy(outstring,string);
	*pstring=outstring;

}

#undef __FUNCT__ 
#define __FUNCT__ "GetParameterValue"
void  Param::GetParameterValue(char*** pstringarray){

	int i;
	if(type!=STRINGARRAY)throw ErrorException(__FUNCT__,exprintf("%s%i"," error message: trying to recover a string array from a param with type",type));

	char**  outstringarray=NULL;
	outstringarray=(char**)xmalloc(M*sizeof(char*));
	for(i=0;i<M;i++){
		char* outstring=(char*)xmalloc((strlen(stringarray[i])+1)*sizeof(char));
		strcpy(outstring,stringarray[i]);
		outstringarray[i]=outstring;
	}
	*pstringarray=outstringarray;
}

#undef __FUNCT__ 
#define __FUNCT__ "GetParameterValue"
	
void  Param::GetParameterValue(double** pdoublearray){
	

	double* outdoublearray=NULL;
	
	if((type!=DOUBLEVEC) & (type!=DOUBLEMAT)) throw ErrorException(__FUNCT__,exprintf("%s%i"," error message: trying to recover a double array from a param with type",type));
	
	if(type==DOUBLEVEC){

		if(M){
			outdoublearray=(double*)xmalloc(M*sizeof(double));
			memcpy(outdoublearray,doublevec,M*sizeof(double));
		}
	}
	
	if(type==DOUBLEMAT){
	
		if(M*N){
			outdoublearray=(double*)xmalloc(M*N*sizeof(double));
			memcpy(outdoublearray,doublemat,M*N*sizeof(double));
		}
	}

	*pdoublearray=outdoublearray;
}

#undef __FUNCT__ 
#define __FUNCT__ "GetParameterValue"
void  Param::GetParameterValue(Vec* pvec){
	
	Vec  outvec=NULL;
	
	if(type!=PETSCVEC)  throw ErrorException(__FUNCT__,exprintf("%s%i"," error message: trying to recover a Petsc vec from a param with type",type));

	if(vec){
		VecDuplicate(vec,&outvec);
		VecCopy(vec,outvec);
	}
	*pvec=outvec;
}

#undef __FUNCT__ 
#define __FUNCT__ "GetParameterValue"

void  Param::GetParameterValue(Mat* pmat){
	
	Mat  outmat=NULL;
	
	if(type!=PETSCMAT)  throw ErrorException(__FUNCT__,exprintf("%s%i"," error message: trying to recover a Petsc mat from a param with type",type));
	
	if(mat){
		MatDuplicate(mat,MAT_COPY_VALUES,&outmat);
	}
	*pmat=outmat;
}

int    Param::GetId(void){ return id; }

int    Param::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}

void  Param::DistributeNumDofs(int* numdofspernode,int analysis_type){return;}
		

int   Param::GetType(){
	return type;
}

Object* Param::copy() {
	return new Param(*this); 
}

#undef __FUNCT__
#define __FUNCT__ "SetDouble"
void  Param::SetDouble(double value){
	if (type!=DOUBLE) throw ErrorException(__FUNCT__,exprintf("%s%i"," trying to set double for type",type));
	ddouble=value;
}

#undef __FUNCT__
#define __FUNCT__ "SetDouble"
void  Param::SetDouble(int value){
	if (type!=DOUBLE) throw ErrorException(__FUNCT__,exprintf("%s%i"," trying to set double for type",type));
	ddouble=(double)value;
}


#undef __FUNCT__
#define __FUNCT__ "SetString"
void  Param::SetString(char* value){
	if (type!=STRING) throw ErrorException(__FUNCT__,exprintf("%s%i"," trying to set string for type",type));
	strcpy(string,value);
}

#undef __FUNCT__
#define __FUNCT__ "SetStringArray"
void  Param::SetStringArray(char** value,int size){
	
	int i;
	if (type!=STRINGARRAY) throw ErrorException(__FUNCT__,exprintf("%s%i"," trying to set string for type",type));
	M=size;
	stringarray=(char**)xmalloc(M*sizeof(char*));
	for(i=0;i<M;i++){
		char* instring=(char*)xmalloc((strlen(value[i])+1)*sizeof(char));
		strcpy(instring,value[i]);
		stringarray[i]=instring;
	}
}


int   Param::GetM(){
	return M;
}

int   Param::GetNdof(){
	return ndof;
}

int   Param::GetN(){
	return N;
}

#undef __FUNCT__
#define __FUNCT__ "SetDoubleVec"
void  Param::SetDoubleVec(double* value,int size){
	if (type!=DOUBLEVEC) throw ErrorException(__FUNCT__,exprintf("%s%i"," trying to set doublevecfor type",type));
	
	M=size;
	if(M){
		xfree((void**)&doublevec); doublevec=(double*)xmalloc(M*sizeof(double));
		memcpy(doublevec,value,M*sizeof(double));
	}
	ndof=0; //this vector will not be repartitioned.

}

#undef __FUNCT__
#define __FUNCT__ "SetDoubleVec"
void  Param::SetDoubleVec(double* value,int size, int numberofdofs){
	if (type!=DOUBLEVEC) throw ErrorException(__FUNCT__,exprintf("%s%i"," trying to set doublevecfor type",type));
	
	M=size;
	if(M){
		xfree((void**)&doublevec); doublevec=(double*)xmalloc(M*sizeof(double));
		memcpy(doublevec,value,M*sizeof(double));
	}
	ndof=numberofdofs;
}


#undef __FUNCT__
#define __FUNCT__ "SetVec"
void  Param::SetVec(Vec value){

	if(value){
		VecDuplicate(value,&vec);
		VecCopy(value,vec);
		VecGetSize(value,&M);
		N=1;
	}
	else{
		vec=NULL;
		M=0;
		N=0;
	}

}
		

#undef __FUNCT__
#define __FUNCT__ "SetDoubleMat"
void  Param::SetDoubleMat(double* value,int pM,int pN){
	
	if (type!=DOUBLEMAT) throw ErrorException(__FUNCT__,exprintf("%s%i"," trying to set doublematrix type",type));

	this->M=pM;
	this->N=pN;
	if(this->M*this->N){
		xfree((void**)&doublemat); doublemat=(double*)xcalloc(M*N,sizeof(double));
		memcpy(doublemat,value,M*N*sizeof(double));
	}
}
