/*!\file: convergence.cpp
 * \brief: figure out if convergence has been reached
 */ 

#undef __FUNCT__ 
#define __FUNCT__ "convergence"

#include "../objects/objects.h"
#include "../issm.h"

void convergence(int* pconverged, Mat Kff,Vec pf,Vec uf,Vec old_uf,DataSet* parameters){

	/*output*/
	int converged=0;

	/*intermediary*/
	Vec KU=NULL;
	Vec KUF=NULL;
	Vec KUold=NULL;
	Vec KUoldF=NULL;
	Vec duf=NULL;
	double ndu,nduinf,nu;
	double nKUF;
	double nKUoldF;
	double nF;
	double solver_residue,res;

	/*convergence options*/
	double eps_res;
	double eps_rel;
	double eps_abs;
	int    verbose;
	double yts;

	/*get convergence options*/
	parameters->FindParam((void*)&eps_res,"eps_res");
	parameters->FindParam((void*)&eps_rel,"eps_rel");
	parameters->FindParam((void*)&eps_abs,"eps_abs");
	parameters->FindParam((void*)&yts,"yts");
	parameters->FindParam((void*)&verbose,"verbose");

	/*Display solver caracteristics*/
	if (verbose>1){

		//compute KUF = KU - F = K*U - F
		VecDuplicate(uf,&KU); MatMultPatch(Kff,uf,KU);
		VecDuplicate(KU,&KUF);VecCopy(KU,KUF); VecAYPX(KUF,-1.0,pf);

		//compute norm(KUF), norm(F) and residue
		VecNorm(KUF,NORM_2,&nKUF);
		VecNorm(pf,NORM_2,&nF);
		solver_residue=nKUF/nF;
		_printf_("\n%s%g\n","   solver residue: norm(KU-F)/norm(F)=",solver_residue);

		//clean up
		VecFree(&KU);
		VecFree(&KUF);
	}

	/*Force equilibrium (Mandatory)*/

	//compute K[n]U[n-1]F = K[n]U[n-1] - F
	VecDuplicate(uf,&KUold); MatMultPatch(Kff,old_uf,KUold);
	VecDuplicate(KUold,&KUoldF);VecCopy(KUold,KUoldF); VecAYPX(KUoldF,-1.0,pf);
	VecNorm(KUoldF,NORM_2,&nKUoldF);
	VecNorm(pf,NORM_2,&nF);
	res=nKUoldF/nF;
	if (isnan(res)){
		PetscSynchronizedPrintf(MPI_COMM_WORLD,"norm nf %lf \n",nF);
		PetscSynchronizedFlush(MPI_COMM_WORLD);

		PetscSynchronizedPrintf(MPI_COMM_WORLD,"norm kuold %lf \n",nKUoldF);
		PetscSynchronizedFlush(MPI_COMM_WORLD);
		throw ErrorException(__FUNCT__,exprintf("mechanical equilibrium convergence criterion is NaN! "));
	}

	//clean up
	VecFree(&KUold);
	VecFree(&KUoldF);

	//print
	if(res<eps_res){
		if (verbose) _printf_("%-50s%g%s%g%s\n","   mechanical equilibrium convergence criterion",res*100," < ",eps_res*100," \%");
		converged=1;
	}
	else{ 
		if (verbose) _printf_("%-50s%g%s%g%s\n","   mechanical equilibrium convergence criterion",res*100," > ",eps_res*100," \%");
		converged=0;
	}

	/*Relative criterion (optional)*/
	if (!isnan(eps_rel) || (verbose>1)){

		//compute norm(du)/norm(u)
		VecDuplicate(old_uf,&duf);VecCopy(old_uf,duf); VecAYPX(duf,-1.0,uf);
		VecNorm(duf,NORM_2,&ndu); VecNorm(old_uf,NORM_2,&nu);

		if (isnan(ndu) || isnan(nu)) throw ErrorException(__FUNCT__,exprintf("convergence criterion is NaN! "));

		//clean up
		VecFree(&duf);

		//print
		if (!isnan(eps_rel)){
			if((ndu/nu)<eps_rel){
				if (verbose) _printf_("%-50s%g%s%g%s\n","   Convergence criterion: norm(du)/norm(u)",ndu/nu*100," < ",eps_rel*100," \%");
			}
			else{ 
				if (verbose) _printf_("%-50s%g%s%g%s\n","   Convergence criterion: norm(du)/norm(u)",ndu/nu*100," > ",eps_rel*100," \%");
				converged=0;
			}
		}
		else _printf_("%-50s%g%s\n","   Convergence criterion: norm(du)/norm(u)",ndu/nu*100," \%");

	}

	/*Absolute criterion (Optional) = max(du)*/
	if (!isnan(eps_abs) || (verbose>1)){

		//compute max(du)
		VecDuplicate(old_uf,&duf);VecCopy(old_uf,duf); VecAYPX(duf,-1.0,uf);
		VecNorm(duf,NORM_2,&ndu); VecNorm(duf,NORM_INFINITY,&nduinf);
		if (isnan(ndu) || isnan(nu)) throw ErrorException(__FUNCT__,exprintf("convergence criterion is NaN! "));

		//clean up
		VecFree(&duf);

		//print
		if (!isnan(eps_abs)){
			if ((nduinf*yts)<eps_abs){
				if (verbose) _printf_("%-50s%g%s%g%s\n","   Convergence criterion: max(du)",nduinf*yts," < ",eps_abs," m/yr");
			}
			else{
				if (verbose) _printf_("%-50s%g%s%g%s\n","   Convergence criterion: max(du)",nduinf*yts," > ",eps_abs," m/yr");
				converged=0;
			}
		}
		else  _printf_("%-50s%g%s\n","   Convergence criterion: max(du)",nduinf*yts," m/yr");

	}

	/*assign output*/
	*pconverged=converged;
}
