/*!\file: control_core.cpp
 * \brief: core of the control solution 
 */ 

#undef __FUNCT__ 
#define __FUNCT__ "control_core"

#include "./parallel.h"
#include "../issm.h"

void control_core(DataSet* results,Model* model, ParameterInputs* inputs){

	extern int my_rank;

	/*fem model: */
	FemModel* fem_model=NULL;

	/*output: */
	Result* result=NULL;

	/*Intermediary: */
	DataSet* diagnostic_results=NULL;
	DataSet* gradjcompute_results=NULL;
	DataSet* steadystate_results=NULL;
	Vec     u_g=NULL;
	Vec     t_g=NULL;
	Vec     m_g=NULL;
	double  search_scalar;
	char*   control_type=NULL;
	double* fit=NULL;
	double* optscal=NULL;
	int     gsize;
	double* maxiter=NULL;
	double* cm_jump=NULL;
	double  eps_cm;
	double  tolx;
	double* param_g=NULL;
	Vec     grad_g=NULL;
	Vec     new_grad_g=NULL;
	Vec     grad_g_old=NULL;
	double* grad_g_double=NULL;
	double  mincontrolconstraint;
	double  maxcontrolconstraint;
	int     nsteps,n,i;
	double* J=NULL;
	OptArgs optargs;
	OptPars optpars;
	Param*  param=NULL;

	/*flags: */
	int analysis_type;
	int sub_analysis_type;
	int     control_steady;
	int verbose=0;
	int convergence=0;
	int numberofnodes;

	/*Process models*/
	ControlInitialization(model,inputs);
	fem_model=model->GetActiveFormulation();

	/*Recover parameters used throughout the solution:*/
	model->FindParam(&nsteps,"nsteps");
	model->FindParam(&control_type,"control_type");
	model->FindParam(&fit,"fit");
	model->FindParam(&optscal,"optscal");
	model->FindParam(&maxiter,"maxiter");
	model->FindParam(&cm_jump,"cm_jump");
	model->FindParam(&eps_cm,"eps_cm");
	model->FindParam(&tolx,"tolx");
	model->FindParam(&mincontrolconstraint,"mincontrolconstraint");
	model->FindParam(&maxcontrolconstraint,"maxcontrolconstraint");
	model->FindParam(&param_g,"param_g");
	model->FindParam(&analysis_type,"analysis_type");
	model->FindParam(&sub_analysis_type,"sub_analysis_type");
	model->FindParam(&numberofnodes,"numberofnodes");
	model->FindParam(&control_steady,"control_steady");
	gsize=fem_model->nodes->NumberOfDofs();

	/*Initialize misfit: */
	J=(double*)xmalloc(nsteps*sizeof(double));

	/*erase useless parameters: */
	param=(Param*)fem_model->parameters->FindParamObject("param_g");
	fem_model->parameters->DeleteObject((Object*)param);

	param=(Param*)fem_model->parameters->FindParamObject("u_g");
	fem_model->parameters->DeleteObject((Object*)param);

	/*Start looping: */
	for(n=0;n<nsteps;n++){

		_printf_("\n%s%i%s%i\n","   control method step ",n+1,"/",nsteps);
		inputs->Add(control_type,param_g,1,numberofnodes);
		inputs->Add("fit",fit[n]);

		/*In case we are running a steady state control method, compute new temperature field using new parameter 
		 * distribution: */
		if (control_steady){
			steadystate_results=new DataSet(ResultsEnum());
			steadystate_core(steadystate_results,model,inputs);
			VecFree(&t_g); steadystate_results->FindResult(&t_g,"t_g");
			delete steadystate_results;
			inputs->Add("temperature",t_g,1,numberofnodes);
		}
	
		/*Update parameters: */
		UpdateFromInputsx(fem_model->elements,fem_model->nodes,fem_model->loads, fem_model->materials,inputs);

		_printf_("%s\n","      computing gradJ...");
		gradjcompute_results=new DataSet(ResultsEnum()); 
		gradjcompute_core(gradjcompute_results,model, inputs);
		gradjcompute_results->FindResult(&grad_g,"grad_g");
		delete gradjcompute_results;
		_printf_("%s\n","      done.");

		_printf_("%s\n","      normalizing directions...");
		Orthx(&new_grad_g,grad_g,grad_g_old);
		VecFree(&grad_g); VecFree(&grad_g_old); 
		grad_g_old=new_grad_g;
		VecToMPISerial(&grad_g_double,new_grad_g);
		_printf_("%s\n","      done.");

		_printf_("%s\n","      optimizing along gradient direction...");
		optargs.model=model;
		optargs.param_g=param_g; optargs.grad_g=grad_g_double; optargs.inputs=inputs;optargs.n=n;
		optpars.xmin=0; optpars.xmax=1; optpars.tolerance=tolx; optpars.maxiter=(int)maxiter[n];optpars.cm_jump=cm_jump[n];
		BrentSearch(&search_scalar,J+n,&optpars,&objectivefunctionC,&optargs);
		_printf_("%s\n","      done.");

		_printf_("%s\n","      updating parameter using optimized search scalar...");
		for(i=0;i<numberofnodes;i++)param_g[i]=param_g[i]+search_scalar*optscal[n]*grad_g_double[i];
		_printf_("%s\n","      done.");

		_printf_("%s\n","      constraining the new distribution...");    
		ControlConstrainx(param_g,numberofnodes,mincontrolconstraint,maxcontrolconstraint,control_type);
		_printf_("%s\n","      done.");
		
		_printf_("%s%i%s%g\n","      value of misfit J after optimization #",n+1,": ",J[n]);

		/*some freeing:*/
		xfree((void**)&grad_g_double);

		/*Has convergence been reached?*/
		if (!isnan(eps_cm)){
			i=n-2;
			//go through the previous misfits(starting from n-2)
			while(i>=0){
				if (fit[i]==fit[n]){
					//convergence test only if we have the same misfits
					if ((J[i]-J[n])/J[n] <= eps_cm){
						//convergence if convergence criteria fullfilled
						convergence=1;
						_printf_("%s%g%s%g\n","      Convergence criterion: dJ/J = ",(J[i]-J[n])/J[n],"<",eps_cm);
					}
					else{
						_printf_("%s%g%s%g\n","      Convergence criterion: dJ/J = ",(J[i]-J[n])/J[n],">",eps_cm);
					}
					break;
				}
				i=i-1;
			}
		}
		//stop if convergence has been reached
		if(convergence) break;

		//some temporary saving
		/*if (((n+1)%5)==0){
			_printf_("%s\n","      saving temporary results...");
			ControlTemporaryResults(model,param_g,J,n,inputs);
			_printf_("%s\n","      done.");
		}*/
	}

	/*Write results to disk: */
	_printf_("%s\n","      preparing final velocity solution...");
	/*Launch diagnostic with the last parameter distribution*/
	if (control_steady){
		inputs->Add(control_type,param_g,1,numberofnodes);
		steadystate_results=new DataSet(ResultsEnum());
		steadystate_core(steadystate_results,model,inputs);

		//extract u_g ,t_g and m_g from steadystate results, and erase diagnostic_results;
		steadystate_results->FindResult(&u_g,"u_g");
		steadystate_results->FindResult(&m_g,"m_g");
		steadystate_results->FindResult(&t_g,"t_g");
		delete steadystate_results;
	}
	else{
		inputs->Add(control_type,param_g,1,numberofnodes);
		diagnostic_results=new DataSet(ResultsEnum()); 
		diagnostic_core(diagnostic_results,model, inputs);

		//extract u_g from diagnostic_results, and erase diagnostic_results;
		diagnostic_results->FindResult(&u_g,"u_g");
		delete diagnostic_results;
	}

	/*Plug results into output dataset: */
	result=new Result(results->Size()+1,0,1,"u_g",u_g);
	results->AddObject(result);
	result=new Result(results->Size()+1,0,1,"param_g",param_g,numberofnodes);
	results->AddObject(result);
	result=new Result(results->Size()+1,0,1,"J",J,nsteps);
	results->AddObject(result);
	if (control_steady){
		result=new Result(results->Size()+1,0,1,"t_g",t_g);
		results->AddObject(result);
		result=new Result(results->Size()+1,0,1,"m_g",m_g);
		results->AddObject(result);
	}

	/*Free ressources: */
	xfree((void**)&control_type);
	xfree((void**)&fit);
	xfree((void**)&optscal);
	xfree((void**)&maxiter);
	xfree((void**)&cm_jump);
	VecFree(&new_grad_g); //do not VecFree grad_g and grad_g_old, they point to new_grad_g
	xfree((void**)&grad_g_double);
	xfree((void**)&param_g);
	VecFree(&u_g);
	VecFree(&t_g);
	VecFree(&m_g);
	xfree((void**)&J);
}
