/*!\file: transient_3d_core.cpp
 * \brief: core of the transient_3d solution
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <float.h>
#include "./cores.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"
#include "../solutionsequences/solutionsequences.h"

void transient_core(FemModel* femmodel){

	/*parameters: */
	IssmDouble finaltime,dt,yts,starttime;
	bool       isstressbalance,ismasstransport,issmb,isFS,isthermal,isgroundingline,isgia,isesa,isslr,iscoupler,ismovingfront,isdamageevolution,ishydrology,isoceancoupling,iscontrol,isautodiff;
	bool       save_results,dakota_analysis;
	int        timestepping;
	int        output_frequency;
	int        recording_frequency;
	int        domaintype,groundingline_migration,smb_model,amr_frequency,amr_restart;
	int        numoutputs;
	Analysis  *analysis          = NULL;
	char     **requested_outputs = NULL;

	/*intermediary: */
	int        step;
	IssmDouble time;

	/*first, figure out if there was a check point, if so, do a reset of the FemModel* femmodel structure. */
	femmodel->parameters->FindParam(&recording_frequency,SettingsRecordingFrequencyEnum);
	if(recording_frequency) femmodel->Restart();

	/*then recover parameters common to all solutions*/
	femmodel->parameters->FindParam(&domaintype,DomainTypeEnum);
	femmodel->parameters->FindParam(&step,StepEnum);
	femmodel->parameters->FindParam(&time,TimeEnum);
	femmodel->parameters->FindParam(&finaltime,TimesteppingFinalTimeEnum);
	femmodel->parameters->FindParam(&starttime,TimesteppingStartTimeEnum);
	femmodel->parameters->FindParam(&yts,ConstantsYtsEnum);
	femmodel->parameters->FindParam(&dakota_analysis,QmuIsdakotaEnum);
	femmodel->parameters->FindParam(&output_frequency,SettingsOutputFrequencyEnum);
	femmodel->parameters->FindParam(&timestepping,TimesteppingTypeEnum);
	femmodel->parameters->FindParam(&isstressbalance,TransientIsstressbalanceEnum);
	femmodel->parameters->FindParam(&ismasstransport,TransientIsmasstransportEnum);
	femmodel->parameters->FindParam(&issmb,TransientIssmbEnum);
	femmodel->parameters->FindParam(&isthermal,TransientIsthermalEnum);
	femmodel->parameters->FindParam(&isgia,TransientIsgiaEnum);
	femmodel->parameters->FindParam(&isesa,TransientIsesaEnum);
	femmodel->parameters->FindParam(&isslr,TransientIsslrEnum);
	femmodel->parameters->FindParam(&iscoupler,TransientIscouplerEnum);
	femmodel->parameters->FindParam(&isgroundingline,TransientIsgroundinglineEnum);
	femmodel->parameters->FindParam(&ismovingfront,TransientIsmovingfrontEnum);
	femmodel->parameters->FindParam(&isoceancoupling,TransientIsoceancouplingEnum);
	femmodel->parameters->FindParam(&isdamageevolution,TransientIsdamageevolutionEnum);
	femmodel->parameters->FindParam(&ishydrology,TransientIshydrologyEnum);
	femmodel->parameters->FindParam(&amr_frequency,TransientAmrFrequencyEnum);
	femmodel->parameters->FindParam(&isFS,FlowequationIsFSEnum);
	femmodel->parameters->FindParam(&iscontrol,InversionIscontrolEnum);
	femmodel->parameters->FindParam(&isautodiff,AutodiffIsautodiffEnum);
	if(isgroundingline) femmodel->parameters->FindParam(&groundingline_migration,GroundinglineMigrationEnum);
	femmodel->parameters->FindParam(&numoutputs,TransientNumRequestedOutputsEnum);
	if(numoutputs) femmodel->parameters->FindParam(&requested_outputs,&numoutputs,TransientRequestedOutputsEnum);

	#if defined(_HAVE_BAMG_) && !defined(_HAVE_AD_)
	if(amr_frequency){
		femmodel->parameters->FindParam(&amr_restart,AmrRestartEnum);
		if(amr_restart) femmodel->ReMesh();
	}
	#endif

	if(isoceancoupling){ /*{{{*/
		#ifndef _HAVE_AD_
		if(VerboseSolution()) _printf0_("   ocean coupling: initialization \n");
		int my_rank;
		ISSM_MPI_Comm tomitgcmcomm;
		ISSM_MPI_Status status;

		my_rank=IssmComm::GetRank();
		GenericParam<ISSM_MPI_Comm>* parcom = dynamic_cast<GenericParam<ISSM_MPI_Comm>*>(femmodel->parameters->FindParamObject(ToMITgcmCommEnum));
		if(!parcom)_error_("TransferForcing error message: could not find ToMITgcmCommEnum communicator");
		tomitgcmcomm=parcom->GetParameterValue();

		int oceangridnxsize,oceangridnysize,ngrids_ocean,nels_ocean;
		IssmDouble  oceantime,coupling_time;
		IssmDouble *oceangridx;
		IssmDouble *oceangridy;
		IssmDouble *icebase_oceangrid = NULL;
		IssmDouble *icemask_oceangrid = NULL;
		IssmDouble* x_ice             = NULL;
		IssmDouble* y_ice             = NULL;
		IssmDouble* lat_ice           = NULL;
		IssmDouble* lon_ice           = NULL;
		IssmDouble* z_ice             = NULL;
		IssmDouble* icebase           = NULL;
		IssmDouble* icemask           = NULL;
		int*        index_ice         = NULL;
		int*        index_ocean       = NULL;
		int         ngrids_ice=femmodel->vertices->NumberOfVertices();
		int         nels_ice=femmodel->elements->NumberOfElements();

		/*Recover fixed parameters and store them*/
		femmodel->parameters->FindParam(&coupling_time,TimesteppingCouplingTimeEnum);
		if(my_rank==0){
			ISSM_MPI_Send(&coupling_time,1,ISSM_MPI_DOUBLE,0,10001000,tomitgcmcomm);
			ISSM_MPI_Recv(&oceangridnxsize,1,ISSM_MPI_INT,0,10001003,tomitgcmcomm,&status);
			ISSM_MPI_Recv(&oceangridnysize,1,ISSM_MPI_INT,0,10001004,tomitgcmcomm,&status);
		}
		ngrids_ocean=oceangridnxsize*oceangridnysize;
		ISSM_MPI_Bcast(&oceangridnxsize,1,ISSM_MPI_INT,0,IssmComm::GetComm());
		ISSM_MPI_Bcast(&oceangridnysize,1,ISSM_MPI_INT,0,IssmComm::GetComm());
		ISSM_MPI_Bcast(&ngrids_ocean,1,ISSM_MPI_INT,0,IssmComm::GetComm());
		ISSM_MPI_Bcast(&oceantime,1,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());
		femmodel->parameters->SetParam(oceangridnxsize,OceanGridNxEnum);
		femmodel->parameters->SetParam(oceangridnysize,OceanGridNyEnum);
		if(my_rank==0){
			oceangridx = xNew<IssmDouble>(ngrids_ocean);
			ISSM_MPI_Recv(oceangridx,ngrids_ocean,ISSM_MPI_DOUBLE,0,10001005,tomitgcmcomm,&status);
			oceangridy = xNew<IssmDouble>(ngrids_ocean);
			ISSM_MPI_Recv(oceangridy,ngrids_ocean,ISSM_MPI_DOUBLE,0,10001006,tomitgcmcomm,&status);

			/*Exchange varying parameters for the initialization*/
			ISSM_MPI_Send(&time,1,ISSM_MPI_DOUBLE,0,10001001,tomitgcmcomm);
			ISSM_MPI_Recv(&oceantime,1,ISSM_MPI_DOUBLE,0,10001002,tomitgcmcomm,&status);
		}
		if(my_rank!=0){
			oceangridx=xNew<IssmDouble>(ngrids_ocean);
			oceangridy=xNew<IssmDouble>(ngrids_ocean);
		}
		ISSM_MPI_Bcast(oceangridx,ngrids_ocean,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());
		ISSM_MPI_Bcast(oceangridy,ngrids_ocean,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());
		femmodel->parameters->SetParam(oceangridx,ngrids_ocean,OceanGridXEnum);
		femmodel->parameters->SetParam(oceangridy,ngrids_ocean,OceanGridYEnum);

		/*Interpolate ice base and mask onto ocean grid*/
		femmodel->GetMesh(femmodel->vertices,femmodel->elements,&x_ice,&y_ice,&z_ice,&index_ice);
		BamgTriangulatex(&index_ocean,&nels_ocean,oceangridx,oceangridy,ngrids_ocean);
		femmodel->vertices->LatLonList(&lat_ice,&lon_ice);
		GetVectorFromInputsx(&icebase,femmodel,BaseEnum,VertexSIdEnum);
		Options* options = new Options();
		GenericOption<double> *odouble = new GenericOption<double>();
		const char* name = "default";
		odouble->name =xNew<char>(strlen(name)+1);
		memcpy(odouble->name,name,(strlen(name)+1)*sizeof(char));
		odouble->value=+9999.;
		odouble->numel=1;
		odouble->ndims=1;
		odouble->size=NULL;
		options->AddOption(odouble);
		InterpFromMeshToMesh2dx(&icebase_oceangrid,index_ice,lon_ice,lat_ice,ngrids_ice,nels_ice,
						icebase,ngrids_ice,1,
						oceangridx,oceangridy,ngrids_ocean,options);
		delete options;
		xDelete<IssmDouble>(icebase);

		GetVectorFromInputsx(&icemask,femmodel,MaskIceLevelsetEnum,VertexSIdEnum);
		Options* options2 = new Options();
		GenericOption<double> *odouble2 = new GenericOption<double>();
		const char* name2 = "default";
		odouble2->name =xNew<char>(strlen(name2)+1);
		memcpy(odouble2->name,name2,(strlen(name2)+1)*sizeof(char));
		odouble2->value=+1.;
		odouble2->numel=1;
		odouble2->ndims=1;
		odouble2->size=NULL;
		options2->AddOption(odouble2);
		InterpFromMeshToMesh2dx(&icemask_oceangrid,index_ice,lon_ice,lat_ice,ngrids_ice,nels_ice,
					icemask,ngrids_ice,1,oceangridx,oceangridy,ngrids_ocean,options2);
		delete options2;
		xDelete<IssmDouble>(icemask);

		/*Put +9999 for places where there is no ice!*/
		for(int i=0;i<ngrids_ocean;i++) if(icemask_oceangrid[i]>0.) icebase_oceangrid[i]=+9999.;
		xDelete<IssmDouble>(icemask_oceangrid);
			
		if(my_rank==0){
			ISSM_MPI_Send(icebase_oceangrid,ngrids_ocean,ISSM_MPI_DOUBLE,0,10001008,tomitgcmcomm);
		}

		/*Delete*/
		xDelete<int>(index_ice);
		xDelete<int>(index_ocean);
		xDelete<IssmDouble>(lat_ice);
		xDelete<IssmDouble>(lon_ice);
		xDelete<IssmDouble>(x_ice);
		xDelete<IssmDouble>(y_ice);
		xDelete<IssmDouble>(z_ice);
		xDelete<IssmDouble>(icebase_oceangrid);
		xDelete<IssmDouble>(oceangridx);
		xDelete<IssmDouble>(oceangridy);
	#else
	_error_("not supported");
	#endif
	}
	/*}}}*/

		IssmDouble  output_value;
		int         num_dependents;
		IssmPDouble *dependents;
		DataSet*    dependent_objects=NULL;
		IssmDouble  J=0.;

	if(iscontrol && isautodiff){

		femmodel->parameters->SetParam(starttime,TimeEnum);
		femmodel->parameters->FindParam(&num_dependents,AutodiffNumDependentsEnum);
		femmodel->parameters->FindParam(&dependent_objects,AutodiffDependentObjectsEnum);

		/*Go through our dependent variables, and compute the response:*/
		dependents=xNew<IssmPDouble>(num_dependents);
	}

	while(time < finaltime - (yts*DBL_EPSILON)){ //make sure we run up to finaltime.

		/*Time Increment*/
		switch(timestepping){
			case AdaptiveTimesteppingEnum:
				femmodel->TimeAdaptx(&dt);
				if(time+dt>finaltime) dt=finaltime-time;
				femmodel->parameters->SetParam(dt,TimesteppingTimeStepEnum);
				break;
			case FixedTimesteppingEnum:
				femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
				break;
			default:
				_error_("Time stepping \""<<EnumToStringx(timestepping)<<"\" not supported yet");
		}
		step+=1;
		time+=dt;
		femmodel->parameters->SetParam(time,TimeEnum);
		femmodel->parameters->SetParam(step,StepEnum);

		if(VerboseSolution()) _printf0_("iteration " << step << "/" << ceil((finaltime-time)/dt)+step << "  time [yr]: " << setprecision(4) << time/yts << " (time step: " << dt/yts << ")\n");
		if(step%output_frequency==0 || (time >= finaltime - (yts*DBL_EPSILON)) || step==1)
		 save_results=true;
		else
		 save_results=false;
		femmodel->parameters->SetParam(save_results,SaveResultsEnum);

		if(isoceancoupling){ /*{{{*/

			#ifndef _HAVE_AD_
			if(VerboseSolution()) _printf0_("   ocean coupling: exchanging information\n");
			int my_rank;
			ISSM_MPI_Comm tomitgcmcomm;
			ISSM_MPI_Status status;

			my_rank=IssmComm::GetRank();
			GenericParam<ISSM_MPI_Comm>* parcom = dynamic_cast<GenericParam<ISSM_MPI_Comm>*>(femmodel->parameters->FindParamObject(ToMITgcmCommEnum));
			if(!parcom)_error_("TransferForcing error message: could not find ToMITgcmCommEnum communicator");
			tomitgcmcomm=parcom->GetParameterValue();
			int ngrids_ocean, nels_ocean;
			IssmDouble oceantime;
			IssmDouble rho_ice;
			IssmDouble *oceanmelt         = NULL;
			IssmDouble *icebase_oceangrid = NULL;
			IssmDouble *icemask_oceangrid = NULL;
			IssmDouble *oceangridx        = NULL;
			IssmDouble *oceangridy        = NULL;
			IssmDouble *x_ice             = NULL;
			IssmDouble *y_ice             = NULL;
			IssmDouble *lat_ice           = NULL;
			IssmDouble *lon_ice           = NULL;
			IssmDouble *z_ice             = NULL;
			IssmDouble *icebase           = NULL;
			IssmDouble *icemask           = NULL;
			IssmDouble *melt_mesh         = NULL;
			int        *index_ice         = NULL;
			int        *index_ocean       = NULL;
			int         ngrids_ice=femmodel->vertices->NumberOfVertices();
			int         nels_ice=femmodel->elements->NumberOfElements();

			/*Recover mesh and inputs needed*/
			femmodel->parameters->FindParam(&rho_ice,MaterialsRhoIceEnum);
			femmodel->GetMesh(femmodel->vertices,femmodel->elements,&x_ice,&y_ice,&z_ice,&index_ice);
			femmodel->parameters->FindParam(&oceangridx,&ngrids_ocean,OceanGridXEnum);
			femmodel->parameters->FindParam(&oceangridy,&ngrids_ocean,OceanGridYEnum);
			BamgTriangulatex(&index_ocean,&nels_ocean,oceangridx,oceangridy,ngrids_ocean);

			femmodel->vertices->LatLonList(&lat_ice,&lon_ice);

			/*Interpolate ice base and mask onto ocean grid*/
			GetVectorFromInputsx(&icebase,femmodel,BaseEnum,VertexSIdEnum);
			Options* options = new Options();
			GenericOption<double> *odouble = new GenericOption<double>();
			const char* name = "default";
			odouble->name =xNew<char>(strlen(name)+1);
			memcpy(odouble->name,name,(strlen(name)+1)*sizeof(char));
			odouble->value=+9999.;
			odouble->numel=1;
			odouble->ndims=1;
			odouble->size=NULL;
			options->AddOption(odouble);
			InterpFromMeshToMesh2dx(&icebase_oceangrid,index_ice,lon_ice,lat_ice,ngrids_ice,nels_ice,
						icebase,ngrids_ice,1,oceangridx,oceangridy,ngrids_ocean,options);
			delete options;
			xDelete<IssmDouble>(icebase);

			GetVectorFromInputsx(&icemask,femmodel,MaskIceLevelsetEnum,VertexSIdEnum);
			Options* options2 = new Options();
			GenericOption<double> *odouble2 = new GenericOption<double>();
			const char* name2 = "default";
			odouble2->name =xNew<char>(strlen(name2)+1);
			memcpy(odouble2->name,name2,(strlen(name2)+1)*sizeof(char));
			odouble2->value=+1.;
			odouble2->numel=1;
			odouble2->ndims=1;
			odouble2->size=NULL;
			options2->AddOption(odouble2);
			InterpFromMeshToMesh2dx(&icemask_oceangrid,index_ice,lon_ice,lat_ice,ngrids_ice,nels_ice,
						icemask,ngrids_ice,1,oceangridx,oceangridy,ngrids_ocean,options2);
			delete options2;
			xDelete<IssmDouble>(icemask);

			/*Put +9999 for places where there is no ice!*/
			for(int i=0;i<ngrids_ocean;i++) if(icemask_oceangrid[i]>0.) icebase_oceangrid[i]=+9999.;
			xDelete<IssmDouble>(icemask_oceangrid);

			/*Send and receive data*/
			if(my_rank==0){
				ISSM_MPI_Send(&time,1,ISSM_MPI_DOUBLE,0,10001001,tomitgcmcomm);
				ISSM_MPI_Recv(&oceantime,1,ISSM_MPI_DOUBLE,0,10001002,tomitgcmcomm,&status);
				if((oceantime - time > 0.1*yts) & (oceantime - time < -0.1*yts)) _error_("Ocean and ice time are starting to diverge");
				oceanmelt = xNew<IssmDouble>(ngrids_ocean);
				ISSM_MPI_Recv(oceanmelt,ngrids_ocean,ISSM_MPI_DOUBLE,0,10001007,tomitgcmcomm,&status);
				ISSM_MPI_Send(icebase_oceangrid,ngrids_ocean,ISSM_MPI_DOUBLE,0,10001008,tomitgcmcomm);
			}
			ISSM_MPI_Bcast(&oceantime,1,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());
			if(my_rank!=0) oceanmelt=xNew<IssmDouble>(ngrids_ocean);
			ISSM_MPI_Bcast(oceanmelt,ngrids_ocean,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());

			/*Interp melt onto ice grid*/
			InterpFromMeshToMesh2dx(&melt_mesh,index_ocean,oceangridx,oceangridy,ngrids_ocean,nels_ocean,
						oceanmelt,ngrids_ocean,1,
						lon_ice,lat_ice,ngrids_ice,NULL);

			for(int i=0;i<ngrids_ice;i++) melt_mesh[i]=-melt_mesh[i]/rho_ice; //heat flux provided by ocean is in kg/m^2/s
			InputUpdateFromVectorx(femmodel,melt_mesh,BasalforcingsFloatingiceMeltingRateEnum,VertexSIdEnum);

			/*Delete*/
			xDelete<int>(index_ice);
			xDelete<int>(index_ocean);
			xDelete<IssmDouble>(lat_ice);
			xDelete<IssmDouble>(lon_ice);
			xDelete<IssmDouble>(x_ice);
			xDelete<IssmDouble>(y_ice);
			xDelete<IssmDouble>(z_ice);
			xDelete<IssmDouble>(melt_mesh);
			xDelete<IssmDouble>(oceangridx);
			xDelete<IssmDouble>(oceangridy);
			xDelete<IssmDouble>(oceanmelt);
			xDelete<IssmDouble>(icebase_oceangrid);

		#else
		_error_("not supported");
		#endif
		}
		/*}}}*/

		if(isthermal && domaintype==Domain3DEnum){
			if(issmb){
				bool isenthalpy;
				femmodel->parameters->FindParam(&isenthalpy,ThermalIsenthalpyEnum);
				femmodel->parameters->FindParam(&smb_model,SmbEnum);
				if(isenthalpy){
					if(smb_model==SMBpddEnum)     ResetBoundaryConditions(femmodel,EnthalpyAnalysisEnum);
					if(smb_model==SMBd18opddEnum) ResetBoundaryConditions(femmodel,EnthalpyAnalysisEnum);
				}
				else{
					if(smb_model==SMBpddEnum)     ResetBoundaryConditions(femmodel,ThermalAnalysisEnum);
					if(smb_model==SMBd18opddEnum) ResetBoundaryConditions(femmodel,ThermalAnalysisEnum);
				}
			}
			thermal_core(femmodel);
		}

		if(ishydrology) hydrology_core(femmodel);

		if(isstressbalance) stressbalance_core(femmodel);

		if(isdamageevolution) damage_core(femmodel);

		if(ismovingfront)	movingfront_core(femmodel);

		/* from here on, prepare geometry for next time step*/

		if(issmb) smb_core(femmodel);

		if(ismasstransport){
			bmb_core(femmodel);
			masstransport_core(femmodel);
			femmodel->UpdateVertexPositionsx();
		}

		if(isgroundingline){
		
			/*Start profiler*/
			femmodel->profiler->Start(GROUNDINGLINECORE);
		
			if(VerboseSolution()) _printf0_("   computing new grounding line position\n");
			GroundinglineMigrationx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);

			femmodel->parameters->SetParam(MaskGroundediceLevelsetEnum,InputToExtrudeEnum);
			extrudefrombase_core(femmodel);
			femmodel->parameters->SetParam(BaseEnum,InputToExtrudeEnum);
			extrudefrombase_core(femmodel);
			femmodel->parameters->SetParam(SurfaceEnum,InputToExtrudeEnum);
			extrudefrombase_core(femmodel);
			
			/*Stop profiler*/
			femmodel->profiler->Stop(GROUNDINGLINECORE);

			if(save_results){
				int outputs[3] = {SurfaceEnum,BaseEnum,MaskGroundediceLevelsetEnum};
				femmodel->RequestedOutputsx(&femmodel->results,&outputs[0],3);
			}
		}

		if(isgia){
			if(VerboseSolution()) _printf0_("   computing glacial isostatic adjustment\n");
			#ifdef _HAVE_GIAIVINS_
			gia_core(femmodel);
			#else
			_error_("ISSM was not compiled with gia capabilities. Exiting");
			#endif
		}

		/*esa: */
		if(isesa) esa_core(femmodel);

		/*Sea level rise: */
		if(isslr) sealevelrise_core(femmodel);

		/*unload results*/
		if(VerboseSolution()) _printf0_("   computing requested outputs\n");
		femmodel->RequestedOutputsx(&femmodel->results,requested_outputs,numoutputs,save_results);
		if(isgroundingline && (groundingline_migration==AggressiveMigrationEnum || groundingline_migration==ContactEnum)){
			int outputs[1] = {MaskGroundediceLevelsetEnum};
			femmodel->RequestedOutputsx(&femmodel->results,&outputs[0],1,save_results);
		}

		if(save_results){
			if(VerboseSolution()) _printf0_("   saving temporary results\n");
			OutputResultsx(femmodel);
		}

		if(recording_frequency && (step%recording_frequency==0)){
			if(VerboseSolution()) _printf0_("   checkpointing model \n");
			femmodel->CheckPoint();
		}

		/*Adaptive mesh refinement*/
		if(amr_frequency){

			#if !defined(_HAVE_AD_)
			if(save_results) femmodel->WriteMeshInResults();
			if(step%amr_frequency==0 && time<finaltime){
				if(VerboseSolution()) _printf0_("   refining mesh\n");
				femmodel->ReMesh();//Do not refine the last step
			}

			#else
			_error_("AMR not suppored with AD");
			#endif
		}

		if (iscontrol && isautodiff) {
			/*Go through our dependent variables, and compute the response:*/
			for(int i=0;i<dependent_objects->Size();i++){
				DependentObject* dep=(DependentObject*)dependent_objects->GetObjectByOffset(i);
				 dep->Responsex(&output_value,femmodel);
				 dep->AddValue(output_value);
				}
			}
	}

	if(!iscontrol || !isautodiff) femmodel->RequestedDependentsx();
	if(iscontrol && isautodiff) femmodel->parameters->SetParam(dependent_objects,AutodiffDependentObjectsEnum);

	/*Free ressources:*/
	if(numoutputs){for(int i=0;i<numoutputs;i++){xDelete<char>(requested_outputs[i]);} xDelete<char*>(requested_outputs);}
}
