/*!\file: solutionsequence_schurcg.cpp
 * \brief: numerical core of 
 */ 

#include "./solutionsequences.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"
#include "../analyses/analyses.h"


#ifdef _HAVE_PETSC_


void SchurCGSolver(Vector<IssmDouble>** puf,Mat Kff,Vec pf, Vec uf0,IS isv,IS isp,Parameters* parameters){/*{{{*/

	Mat                  A, B, BT;				/* Saddle point block matrices */
	Mat						IP;						/* Preconditioner matrix */
	Mat						IP2;
	int                  nu, np;					/* No of. free nodes in velocity / pressure space */
   Vec                  p,uold,unew;			/* Solution vectors for pressure / vel. */ 
	Vec						tmpu, tmpp, rhsu,rhsp; /* temp. vectors, arbitrary RHS in vel. / pressure space */
	Vec						gold,gnew,wold,wnew,chi,thetaold,thetanew,eta; /* CG intermediaries */
	Vec						f1,f2;					/* RHS of the global system */
	double					rho,gamma,tmpScalar; /* Step sizes, arbitrary double */
	KSP						kspu,kspp;				/* KSP contexts for vel. / pressure systems*/
	KSPConvergedReason	reason;					/* Convergence reason for troubleshooting */
	int						its;						/* No. of iterations for troubleshooting */
	double					initRnorm, rnorm, TOL; /* residual norms, STOP tolerance */
	PC							pcu,pcp;					/* Preconditioner contexts pertaining the KSP contexts*/
	PetscViewer				viewer;					/* Viewer for troubleshooting */
	IssmPDouble				t1,t2;					/* Time measurement for bottleneck analysis */

	/*STOP tolerance for the rel. residual*/
	TOL = 0.4;

	/*Initialize output*/
	Vector<IssmDouble>* out_uf=new Vector<IssmDouble>(uf0);
	
	/* Extract block matrices from the saddle point matrix */
	/* [ A   B ] = Kff
    * [ B^T 0 ] 
	 *         */
	MatGetSubMatrix(Kff,isv,isv,MAT_INITIAL_MATRIX,&A);
	MatGetSubMatrix(Kff,isv,isp,MAT_INITIAL_MATRIX,&B);
	MatGetSubMatrix(Kff,isp,isv,MAT_INITIAL_MATRIX,&BT);
	
	/* Extract preconditioner matrix on the pressure space*/
	MatGetSubMatrix(Kff,isp,isp,MAT_INITIAL_MATRIX,&IP);

	/* Get number of velocity / pressure nodes */
	MatGetSize(B,&nu,&np);

	/* Extract initial guesses for uold and pold */
	VecCreate(IssmComm::GetComm(),&p);VecSetSizes(p,PETSC_DECIDE,np);VecSetFromOptions(p);
	VecAssemblyBegin(p);VecAssemblyEnd(p);
	VecCreate(IssmComm::GetComm(),&uold);VecSetSizes(uold,PETSC_DECIDE,nu);VecSetFromOptions(uold);
	VecAssemblyBegin(uold);VecAssemblyEnd(uold);

	VecGetSubVector(out_uf->pvector->vector,isv,&uold);
	VecGetSubVector(out_uf->pvector->vector,isp,&p);


	/* Set up intermediaries */
	VecDuplicate(uold,&f1);VecSet(f1,0.0);
	VecDuplicate(p,&f2);VecSet(f2,0.0);
	VecDuplicate(uold,&tmpu);VecSet(tmpu,0.0);
	VecDuplicate(p,&tmpp);VecSet(tmpp,0.0);
	VecDuplicate(p,&rhsp);VecSet(rhsp,0.0);
	VecDuplicate(uold,&rhsu);VecSet(rhsu,0.0);
	VecDuplicate(p,&gold);VecSet(gold,0.0);
	VecDuplicate(p,&wnew);VecSet(wnew,0.0);
	VecDuplicate(uold,&chi);VecSet(chi,0.0);
	VecDuplicate(p,&thetanew);VecSet(thetanew,0.0);
	VecDuplicate(p,&thetaold);VecSet(thetaold,0.0);
	VecDuplicate(p,&eta);VecSet(eta,0.0);
	
	/* Get global RHS (for each block sub-problem respectively)*/
	VecGetSubVector(pf,isv,&f1);
	VecGetSubVector(pf,isp,&f2);

   /* ------------------------------------------------------------ */

	/* Generate initial value for the velocity from the pressure */
	/* a(u0,v) = f1(v)-b(p0,v)  i.e.  Au0 = F1-Bp0 */
	/* u0 = u_DIR on \Gamma_DIR */
	
	/* Create KSP context */
	KSPCreate(IssmComm::GetComm(),&kspu);
	KSPSetOperators(kspu,A,A);
	KSPSetType(kspu,KSPCG);
	KSPSetInitialGuessNonzero(kspu,PETSC_TRUE);
	//KSPSetTolerances(kspu,1e-12,PETSC_DEFAULT,PETSC_DEFAULT,PETSC_DEFAULT);
	//KSPMonitorSet(kspu,KSPMonitorDefault,NULL,NULL);
	KSPGetPC(kspu,&pcu);
	PCSetType(pcu,PCSOR);
	KSPSetUp(kspu);

	
	/* Create RHS */
	/* RHS = F1-B * pold */
	VecScale(p,-1.);MatMultAdd(B,p,f1,rhsu);VecScale(p,-1.);

	/* Go solve Au0 = F1-Bp0*/
	KSPSolve(kspu,rhsu,uold);
	

	/* Set up u_new */
	VecDuplicate(uold,&unew);VecCopy(uold,unew);
	VecAssemblyBegin(unew);VecAssemblyEnd(unew);



	/* ------------------------------------------------------------- */

	/*Get initial residual*/
	/*(1/mu(x) * g0, q) = b(q,u0) - (f2,q)  i.e.  IP * g0 = BT * u0 - F2*/
	
	/* Create KSP context */
	KSPCreate(IssmComm::GetComm(),&kspp);
	KSPSetOperators(kspp,IP,IP);
	
	/* Create RHS */
	/* RHS = BT * uold - F2 */
	VecScale(f2,-1.);MatMultAdd(BT,uold,f2,rhsp);VecScale(f2,-1.);

	/* Set KSP & PC options */
	KSPSetType(kspp,KSPCG);
	KSPSetInitialGuessNonzero(kspp,PETSC_TRUE);
	KSPGetPC(kspp,&pcp);
	PCSetType(pcp,PCJACOBI);
	/* Note: Systems in the pressure space are cheap, so we can afford a better tolerance */
	KSPSetTolerances(kspp,1e-10,PETSC_DEFAULT,PETSC_DEFAULT,PETSC_DEFAULT);
	KSPSetUp(kspp);
	
	/* Go solve */
	KSPSolve(kspp,rhsp,gold);
	
	/*Initial residual*/
	VecNorm(gold,NORM_INFINITY,&initRnorm);
	
	/* Further setup */
	VecDuplicate(gold,&gnew);VecCopy(gold,gnew);
	VecAssemblyBegin(gnew);VecAssemblyEnd(gnew);


	/* ------------------------------------------------------------ */

	/*Set initial search direction*/
	/*w0 = g0*/
	VecDuplicate(gold,&wold);VecCopy(gold,wold);
	VecAssemblyBegin(wold);VecAssemblyEnd(wold);

	/*Realizing the step size part 1: thetam */
	/*IP * theta = BT * uold - F2*/
	VecScale(f2,-1.);MatMultAdd(BT,uold,f2,rhsp);VecScale(f2,-1.);
	KSPSolve(kspp,rhsp,thetaold);


	/* Count number of iterations */
	int count = 0;

	/* CG iteration*/
	for(;;){

		/*Realizing the step size part 2: chim */
		/*a(chim,v) = -b(wm,v)  i.e.  A * chim = -B * wm */
		/*chim_DIR = 0*/
		VecScale(wold,-1.);MatMult(B,wold,rhsu);VecScale(wold,-1.);
		KSPSolve(kspu,rhsu,chi);

		/*Realizing the step size part 3: etam */
		MatMult(BT,chi,rhsp);
		KSPSolve(kspp,rhsp,eta);
	
		/* ---------------------------------------------------------- */


		/*Set step size*/
		/*rhom = [(wm)^T * IP^-1 * (BT * um - F2)]/[(wm)^T * IP^-1 * BT * chim]*/
		VecDot(wold,thetaold,&rho);
		VecDot(wold,eta,&tmpScalar);
		rho = rho/tmpScalar;


		/* ---------------------------------------------------------- */


		/*Pressure update*/
		/*p(m+1) = pm - rhom * wm*/
		VecAXPY(p,-1.*rho,wold);


		/*Velocity update*/
		/*u(m+1) = um - rhom * chim*/
		VecWAXPY(unew,-1.*rho,chi,uold);


		/* ---------------------------------------------------------- */

		/*Theta update*/
		/*IP * theta = BT * uold - F2*/
		VecScale(f2,-1.);MatMultAdd(BT,unew,f2,rhsp);VecScale(f2,-1.);
		KSPSolve(kspp,rhsp,thetanew);


		/* ---------------------------------------------------------- */

		/*Residual update*/
		/*g(m+1) = gm - rhom * BT * chim*/
		VecWAXPY(gnew,-1.*rho,eta,gold);

		/* ---------------------------------------------------------- */


		/*BREAK if norm(g(m+0),2) < TOL or pressure space has been full searched*/
		VecNorm(gnew,NORM_INFINITY,&rnorm);
		if(rnorm < TOL*initRnorm) 
		 break;
		else if(rnorm > 100*initRnorm)
		 _error_("Solver diverged. This shouldn't happen\n");
		//else
		// PetscPrintf(PETSC_COMM_WORLD,"rel. residual at step %d: %g, at TOL = %g\n",count,rnorm/initRnorm,TOL);
	



		if(count > np-1) break;
	

		/* ---------------------------------------------------------- */


		/*Directional update*/
		/*gamma = [g(m+1)^T * theta(m+1)]/[g(m)^T * thetam]*/
		VecDot(gnew,thetanew,&gamma);
		VecDot(gold,thetaold,&tmpScalar);
		gamma = gamma/tmpScalar;

		/*w(m+1) = g(m+1) + gamma * w(m)*/
		VecWAXPY(wnew,gamma,wold,gnew);

		/* Assign new to old iterates */
		VecCopy(wnew,wold);VecCopy(gnew,gold);VecCopy(unew,uold);VecCopy(thetanew,thetaold);
		
		count++;
	}


	/* Restore pressure and velocity sol. vectors to its global form */
	VecRestoreSubVector(out_uf->pvector->vector,isv,&unew);
	VecRestoreSubVector(out_uf->pvector->vector,isp,&p);

	/*return output pointer*/
	*puf=out_uf;


	/* Cleanup */
	KSPDestroy(&kspu);KSPDestroy(&kspp);

	MatDestroy(&A);MatDestroy(&B);MatDestroy(&BT);MatDestroy(&IP);
	
	VecDestroy(&p);VecDestroy(&uold);VecDestroy(&unew);VecDestroy(&rhsu);VecDestroy(&rhsp);
	VecDestroy(&gold);VecDestroy(&gnew);VecDestroy(&wold);VecDestroy(&wnew);VecDestroy(&chi);
	VecDestroy(&tmpp);VecDestroy(&tmpu);VecDestroy(&f1);VecDestroy(&f2);VecDestroy(&eta);
	VecDestroy(&thetanew);VecDestroy(&thetaold);

}/*}}}*/
void convergence_schurcg(bool* pconverged, Matrix<IssmDouble>* Kff,Vector<IssmDouble>* pf,Vector<IssmDouble>* uf,Vector<IssmDouble>* old_uf,IssmDouble eps_res,IssmDouble eps_rel,IssmDouble eps_abs,IS isv,IS isp){/*{{{*/

	/*output*/
	bool converged=false;

	/*intermediary*/
	//Vector<IssmDouble>* KU=NULL;
	//Vector<IssmDouble>* KUF=NULL;
	//Vector<IssmDouble>* KUold=NULL;
	//Vector<IssmDouble>* KUoldF=NULL;
	Vector<IssmDouble>* duf=NULL;
	IssmDouble ndu,nduinf,nu;
	IssmDouble nKUF;
	IssmDouble nKUoldF;
	IssmDouble nF;
	IssmDouble solver_residue,res;
	int analysis_type;

	Mat A, B, BT;
	Vec u,p,uold,pold,f1,f2,tmp,res1,res2;
	int n_u,n_p;
	double rnorm1, rnorm2;


	if(VerboseModule()) _printf0_("   checking convergence\n");

	/*If uf is NULL in input, f-set is nil, model is fully constrained, therefore converged from 
	 * the get go: */
	if(uf->IsEmpty()){
		*pconverged=true;
		return;
	}

  /* Note: SchurCG also constructs the Schur preconditioner and stores it in the free block of Kff */
  /*			[A    B]
	* Kff =  |      |
	*			[B^T IP]
  /* To calculate the residual, only the necessary blocks need to be extracted */

		/*Extract A, B, B^T */
		MatGetSubMatrix(Kff->pmatrix->matrix,isv,isv,MAT_INITIAL_MATRIX,&A);
		MatGetSubMatrix(Kff->pmatrix->matrix,isv,isp,MAT_INITIAL_MATRIX,&B);
		MatGetSubMatrix(Kff->pmatrix->matrix,isp,isv,MAT_INITIAL_MATRIX,&BT);
	
		/*no. of free nodes in velocity/pressure space*/
		MatGetSize(B,&n_u,&n_p);

		/*Extract values corresponding to the free velocity/pressure nodes*/
		VecCreate(IssmComm::GetComm(),&p);VecSetSizes(p,PETSC_DECIDE,n_p);VecSetFromOptions(p);
		VecAssemblyBegin(p);VecAssemblyEnd(p);
		VecCreate(IssmComm::GetComm(),&u);VecSetSizes(u,PETSC_DECIDE,n_u);VecSetFromOptions(u);
		VecAssemblyBegin(u);VecAssemblyEnd(u);

		VecGetSubVector(uf->pvector->vector,isv,&u);
		VecGetSubVector(uf->pvector->vector,isp,&p);
		

		/*Extract values of the RHS corresponding to the first/second block*/
		VecDuplicate(u,&f1);VecSet(f1,1.0);
		VecDuplicate(p,&f2);VecSet(f2,1.0);
		VecGetSubVector(pf->pvector->vector,isv,&f1);
		VecGetSubVector(pf->pvector->vector,isp,&f2);

		/*Allocate intermediaries*/
		VecDuplicate(u,&res1);VecSet(res1,1.0);
		VecDuplicate(u,&tmp);VecSet(tmp,1.0);
		VecDuplicate(p,&res2);VecSet(res2,1.0);


	/*Display solver caracteristics*/
	if (VerboseConvergence()){
		
		/*Calculate res1 = A*u + B*p - f1*/
		VecScale(f1,-1.);MatMultAdd(A,u,f1,tmp);MatMultAdd(B,p,tmp,res1);VecScale(f1,-1.);
		/*Calculate res2 = B^T * u - f2*/
		VecScale(f2,-1.);MatMultAdd(BT,u,f2,res2);VecScale(f2,-1.);


		/*compute norm(res1), norm(res2), norm(F) and residue*/
		VecNorm(res1,NORM_2,&rnorm1);VecNorm(res2,NORM_2,&rnorm2);
		nKUF=sqrt(rnorm1*rnorm1 + rnorm2*rnorm2);
		nF=pf->Norm(NORM_TWO);
		solver_residue=nKUF/nF;
		_printf0_("\n" << "   solver residue: norm(KU-F)/norm(F)=" << solver_residue << "\n");
		if(xIsNan<IssmDouble>(solver_residue)){
			//Kff->Echo();
		}

	}
	/*clean up*/
	VecRestoreSubVector(uf->pvector->vector,isv,&u);
	VecRestoreSubVector(uf->pvector->vector,isp,&p);
	
	/*Extract values corresponding to velocity/pressure on the old solution*/
	VecGetSubVector(old_uf->pvector->vector,isv,&uold);
	VecGetSubVector(old_uf->pvector->vector,isp,&pold);
		

	/*Force equilibrium (Mandatory)*/

	/*Calculate res1 = A*uold + B*pold - f1*/
	VecScale(f1,-1.);MatMultAdd(A,uold,f1,tmp);MatMultAdd(B,pold,tmp,res1);VecScale(f1,-1.);
	/*Calculate res2 = B^T * uold - f2*/
	VecScale(f2,-1.);MatMultAdd(BT,uold,f2,res2);VecScale(f2,-1.);
	
	/*compute norm(res1), norm(res2), norm(F) and residue*/
	VecNorm(res1,NORM_2,&rnorm1);VecNorm(res2,NORM_2,&rnorm2);
	nKUoldF=sqrt(rnorm1*rnorm1 + rnorm2*rnorm2);
	nF=pf->Norm(NORM_TWO);
	res=nKUoldF/nF;
	if (xIsNan<IssmDouble>(res)){
		_printf0_("norm nf = " << nF << "f and norm kuold = " << nKUoldF << "f\n");
		_error_("mechanical equilibrium convergence criterion is NaN!");
	}

	MatDestroy(&A);MatDestroy(&B);MatDestroy(&BT);
	VecRestoreSubVector(pf->pvector->vector,isv,&f1);
	VecRestoreSubVector(pf->pvector->vector,isp,&f2);
	VecDestroy(&res1);VecDestroy(&res2);VecDestroy(&tmp);
	VecRestoreSubVector(old_uf->pvector->vector,isv,&uold);
	VecRestoreSubVector(old_uf->pvector->vector,isp,&pold);
	


	//print
	if(res<eps_res){
		if(VerboseConvergence()) _printf0_(setw(50)<<left<<"   mechanical equilibrium convergence criterion"<<res*100<< " < "<<eps_res*100<<" %\n");
		converged=true;
	}
	else{ 
		if(VerboseConvergence()) _printf0_(setw(50)<<left<<"   mechanical equilibrium convergence criterion"<<res*100<<" > "<<eps_res*100<<" %\n");
		converged=false;
	}

	/*Relative criterion (optional)*/
	if (!xIsNan<IssmDouble>(eps_rel) || (VerboseConvergence())){

		//compute norm(du)/norm(u)
		duf=old_uf->Duplicate(); old_uf->Copy(duf); duf->AYPX(uf,-1.0);
		ndu=duf->Norm(NORM_TWO); nu=old_uf->Norm(NORM_TWO);

		if (xIsNan<IssmDouble>(ndu) || xIsNan<IssmDouble>(nu)) _error_("convergence criterion is NaN!");

		//clean up
		delete duf;

		//print
		if (!xIsNan<IssmDouble>(eps_rel)){
			if((ndu/nu)<eps_rel){
				if(VerboseConvergence()) _printf0_(setw(50) << left << "   Convergence criterion: norm(du)/norm(u)" << ndu/nu*100 << " < " << eps_rel*100 << " %\n");
			}
			else{ 
				if(VerboseConvergence()) _printf0_(setw(50) << left << "   Convergence criterion: norm(du)/norm(u)" << ndu/nu*100 << " > " << eps_rel*100 << " %\n");
				converged=false;
			}
		}
		else _printf0_(setw(50) << left << "   Convergence criterion: norm(du)/norm(u)" << ndu/nu*100 << " %\n");

	}

	/*Absolute criterion (Optional) = max(du)*/
	if (!xIsNan<IssmDouble>(eps_abs) || (VerboseConvergence())){

		//compute max(du)
		duf=old_uf->Duplicate(); old_uf->Copy(duf); duf->AYPX(uf,-1.0);
		ndu=duf->Norm(NORM_TWO); nduinf=duf->Norm(NORM_INF);
		if (xIsNan<IssmDouble>(ndu) || xIsNan<IssmDouble>(nu)) _error_("convergence criterion is NaN!");

		//clean up
		delete duf;

		//print
		if (!xIsNan<IssmDouble>(eps_abs)){
			if ((nduinf)<eps_abs){
				if(VerboseConvergence()) _printf0_(setw(50) << left << "   Convergence criterion: max(du)" << nduinf << " < " << eps_abs << "\n");
			}
			else{
				if(VerboseConvergence()) _printf0_(setw(50) << left << "   Convergence criterion: max(du)" << nduinf << " > " << eps_abs << "\n");
				converged=false;
			}
		}
		else  _printf0_(setw(50) << left << "   Convergence criterion: max(du)" << nduinf << "\n");

	}

	/*assign output*/
	*pconverged=converged;
}/*}}}*/
void solutionsequence_schurcg(FemModel* femmodel){/*{{{*/

	/*intermediary: */
	Matrix<IssmDouble>* Kff = NULL;
	Matrix<IssmDouble>* Kfs = NULL;
	Vector<IssmDouble>* ug  = NULL;
	Vector<IssmDouble>* uf  = NULL;
	Vector<IssmDouble>* old_uf = NULL;
	Vector<IssmDouble>* pf  = NULL;
	Vector<IssmDouble>* df  = NULL;
	Vector<IssmDouble>* ys  = NULL;


	/*parameters:*/
	int max_nonlinear_iterations;
	int configuration_type;
	IssmDouble eps_res,eps_rel,eps_abs;

	/*Recover parameters: */
	femmodel->parameters->FindParam(&max_nonlinear_iterations,StressbalanceMaxiterEnum);
	femmodel->parameters->FindParam(&eps_res,StressbalanceRestolEnum);
	femmodel->parameters->FindParam(&eps_rel,StressbalanceReltolEnum);
	femmodel->parameters->FindParam(&eps_abs,StressbalanceAbstolEnum);
	femmodel->parameters->FindParam(&configuration_type,ConfigurationTypeEnum);
	femmodel->UpdateConstraintsx();
	int size;
	int  count=0;
	bool converged=false;

	/*Start non-linear iteration using input velocity: */
	GetSolutionFromInputsx(&ug,femmodel);
	Reducevectorgtofx(&uf, ug, femmodel->nodes,femmodel->parameters);

	/*Update once again the solution to make sure that vx and vxold are similar*/
	InputUpdateFromConstantx(femmodel,converged,ConvergedEnum);
	InputUpdateFromSolutionx(femmodel,ug);

	for(;;){

		/*save pointer to old velocity*/
		delete old_uf; old_uf=uf;
		delete ug;

		/*Get stiffness matrix and Load vector*/
		SystemMatricesx(&Kff,&Kfs,&pf,&df,NULL,femmodel);
		CreateNodalConstraintsx(&ys,femmodel->nodes,configuration_type);
		Reduceloadx(pf, Kfs, ys); delete Kfs;

		/*Create mass matrix*/
		StressbalanceAnalysis* analysis = new StressbalanceAnalysis();
		/*Get complete stiffness matrix without penalties*/
		for(int i=0;i<femmodel->elements->Size();i++){
			Element* element=xDynamicCast<Element*>(femmodel->elements->GetObjectByOffset(i));
			ElementMatrix* Ie = analysis->CreateSchurPrecondMatrix(element);
			if(Ie) Ie->AddToGlobal(Kff,NULL);
			delete Ie;
		}
		Kff->Assemble();
		delete analysis;

		/*Obtain index sets for velocity and pressure components */
		IS isv = NULL;
		IS isp = NULL;
		#if _PETSC_MAJOR_==3
			/*Make indices out of doftypes: */
			if(!(df->pvector->vector))_error_("need doftypes for FS solver!\n");
			DofTypesToIndexSet(&isv,&isp,df->pvector->vector,FSSolverEnum);
		#else
			_error_("Petsc 3.X required");
		#endif


		/*Solve*/
		femmodel->profiler->Start(SOLVER);
		_assert_(Kff->type==PetscMatType); 
		
		SchurCGSolver(&uf,
					Kff->pmatrix->matrix,
					pf->pvector->vector,
					old_uf->pvector->vector,
					isv,
					isp,
					femmodel->parameters);
		femmodel->profiler->Stop(SOLVER);
	

		/*Merge solution from f set to g set*/
		Mergesolutionfromftogx(&ug, uf,ys,femmodel->nodes,femmodel->parameters);delete ys;

		/*Check for convergence and update inputs accordingly*/
		convergence_schurcg(&converged,Kff,pf,uf,old_uf,eps_res,eps_rel,eps_abs,isv,isp); delete Kff; delete pf; delete df;
		count++;

		if(count>=max_nonlinear_iterations){
			_printf0_("   maximum number of nonlinear iterations (" << max_nonlinear_iterations << ") exceeded\n"); 
			converged=true;
		}
		InputUpdateFromConstantx(femmodel,converged,ConvergedEnum);
		InputUpdateFromSolutionx(femmodel,ug);

		/*Increase count: */
		if(converged==true){
			femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,StressbalanceConvergenceNumStepsEnum,count));
			break;
		}
	}

	if(VerboseConvergence()) _printf0_("\n   total number of iterations: " << count << "\n");

	/*clean-up*/
	delete uf;
	delete ug;
	delete old_uf;

}/*}}}*/

#else
void solutionsequence_schurcg(FemModel* femmodel){_error_("PETSc needs to be installed");}
#endif
