/* \file WriteData.c:
 * \brief: general interface for writing data
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../include/macros.h"
#include "./shared/shared.h"
#include "./io.h"

#undef __FUNCT__ 
#define __FUNCT__  "WriteData"

#ifdef _SERIAL_

#include <mex.h>


/*Several prototypes for WriteData, according to type: */

/*DataSet: */
void WriteData(mxArray** pdataref,DataSet* dataset){

	mxArray* dataref=NULL;
	char* marshalled_dataset=NULL;
	int   marshalled_dataset_size;

	/*Write a dataset: */
	if(dataset){
			/* marshall the dataset: */
			marshalled_dataset=dataset->Marshall();
			marshalled_dataset_size=dataset->MarshallSize();
			
			dataref = mxCreateDoubleMatrix(0,0,mxREAL);
			mxSetM(dataref,(mwSize)(marshalled_dataset_size/sizeof(double)));
			mxSetN(dataref,(mwSize)1);
			mxSetPr(dataref,(double*)marshalled_dataset);	
	}
	else{
		/* return empty matrix: */
		dataref=mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
	
}

void WriteData(mxArray** pdataref,Mat matrix){
		
	mxArray* dataref=NULL;
	
	if(matrix){
		
		/*call toolkit routine: */
		PetscMatrixToMatlabMatrix(&dataref,matrix);
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}

	*pdataref=dataref;
}

void WriteData(mxArray** pdataref,double* matrix, int M,int N){
	
	mxArray* dataref=NULL;
	mxArray* tdataref=NULL;
		
	if(matrix){
		
		/*data is a double* pointer. Copy into a matrix: */
		tdataref = mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(tdataref,(mwSize)N);
		mxSetN(tdataref,(mwSize)M);
		mxSetPr(tdataref,(double*)matrix);

		//transpose
		mexCallMATLAB(1,&dataref,1,&tdataref, "'");
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
}


void WriteData(mxArray** pdataref,Vec vector){
	
	mxArray* dataref=NULL;
	
	if(vector){
		
		/*call toolkit routine: */
		PetscVectorToMatlabVector(&dataref,vector);
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;

}


void WriteData(mxArray** pdataref,DofVec* dofvec){
	
	mxArray* mxvector=NULL;
	mxArray* structdataref=NULL;
	mxArray* dataref=NULL;

	int nfields=4;
	const	char*	fnames[nfields];
	mwSize		onebyone[2] = {1,1};
	mwSize		ndim=2;

	fnames[0] = "name";
	fnames[1] = "numdofs";
	fnames[2] = "numentries";
	fnames[3] = "vector";
	
	if(dofvec){
		
		/*call toolkit routine: */
		if(dofvec->vector)PetscVectorToMatlabVector(&mxvector,dofvec->vector);
		else mxvector=mxCreateDoubleMatrix(0,0,mxREAL);

		/*use the mxvector to create a dofvec object: */
		structdataref=mxCreateStructArray( ndim,onebyone,nfields,fnames);
	
		mxSetField( structdataref, 0, "name",mxCreateString(dofvec->name));
		mxSetField( structdataref, 0, "numdofs",mxCreateDoubleScalar(dofvec->numdofs));
		mxSetField( structdataref, 0, "numentries",mxCreateDoubleScalar(dofvec->numentries));
		mxSetField( structdataref, 0, "vector",mxvector);


		/*transform structure into dofvec class: */
		mexCallMATLAB( 1, &dataref, 1, &structdataref, "dofvec");
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
}

void WriteData(mxArray** pdataref,double* vector, int M){
	
	mxArray* dataref=NULL;

	if(vector){

		/*data is a double* pointer. Copy into a vector: */
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(dataref,(mwSize)M);
		mxSetN(dataref,(mwSize)1);
		mxSetPr(dataref,vector);
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}

	*pdataref=dataref;
}

void WriteData(mxArray** pdataref,double scalar){

	*pdataref=mxCreateDoubleScalar(scalar);
}

void WriteData(mxArray** pdataref,int integer){

		*pdataref=mxCreateDoubleScalar((double)integer);

}

void WriteData(mxArray** pdataref,char* string){

		*pdataref=mxCreateString(string);
}

#else
void WriteData(int* pdummy,void* data,char* data_type){
	
	/*In parallel mode, WriteData is not used, instead we access the data directly through pointers in the solution sequences. */
}
#endif
