%GROUNDINGLINE class definition
%
%   Usage:
%      groundingline=groundingline();

classdef groundingline
	properties (SetAccess=public) 
		migration              = '';
		friction_interpolation = '';
		melt_interpolation     = '';
	end
	methods
		function self = groundingline(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{

			%Type of migration
			self.migration             = 'SubelementMigration';
			self.friction_interpolation= 'SubelementFriction1';
			self.melt_interpolation    = 'NoMeltOnPartiallyFloating';

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			md = checkfield(md,'fieldname','groundingline.migration','values',{'None' 'SubelementMigration' 'AggressiveMigration' 'SoftMigration' 'Contact' 'GroundingOnly'});
			md = checkfield(md,'fieldname','groundingline.friction_interpolation','values',{'NoFrictionOnPartiallyFloating' 'SubelementFriction1' 'SubelementFriction2'});
			md = checkfield(md,'fieldname','groundingline.melt_interpolation','values',{'NoMeltOnPartiallyFloating' 'SubelementMelt1' 'SubelementMelt2' 'FullMeltOnPartiallyFloating'});

			if ~strcmp(self.migration,'None'),
				if isnan(md.geometry.bed),
					md = checkmessage(md,['requesting grounding line migration, but bathymetry is absent!']);
				end
				pos=find(md.mask.groundedice_levelset>0. & md.mask.ice_levelset<=0);
				if any(abs(md.geometry.base(pos)-md.geometry.bed(pos))>10^-10),
					md = checkmessage(md,['base not equal to bed on grounded ice!']);
				end
				pos=find(md.mask.groundedice_levelset<=0. & md.mask.ice_levelset<=0);
				if any(md.geometry.bed(pos) - md.geometry.base(pos) > 10^-9),
					md = checkmessage(md,['bed superior to base on floating ice!']);
				end
			end

		end % }}}
		function disp(self) % {{{
			disp(sprintf('   grounding line migration parameters:'));
			fielddisplay(self,'migration','type of grounding line migration: ''SoftMigration'',''SubelementMigration'',''AggressiveMigration'',''Contact'' or ''None''');
			fielddisplay(self,'friction_interpolation','type of friction interpolation for partially floating elements: ''NoFrictionOnPartiallyFloating'',''SubelementFriction1'', or ''SubelementFriction2''');
			fielddisplay(self,'melt_interpolation','type of melt interpolation for partially floating elements: ''NoMeltOnPartiallyFloating'',''SubelementMelt1'',''SubelementMelt2'', or ''FullMeltOnPartiallyFloating''');

		end % }}}
		function marshall(self,prefix,md,fid) % {{{
			WriteData(fid,prefix,'data',self.migration,'name','md.groundingline.migration','format','String');
			WriteData(fid,prefix,'data',self.friction_interpolation,'name','md.groundingline.friction_interpolation','format','String');
			WriteData(fid,prefix,'data',self.melt_interpolation,'name','md.groundingline.melt_interpolation','format','String');
		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
		
			writejsstring(fid,[modelname '.groundingline.migration'],self.migration);
			writejsstring(fid,[modelname '.groundingline.friction_interpolation'],self.migration);
			writejsstring(fid,[modelname '.groundingline.melt_interpolation'],self.migration);

		end % }}}
	end
end
