/*!\file InterpFromMeshToMesh2d.c
 * \brief: data interpolation from a list of (x,y,values) into mesh grids
 
	InterpFromMeshToMesh2d.c

	usage:
	data_mesh=InterpFromMeshToMesh2d(index,x,y,data,x_mesh,y_mesh);
	
	where:

		input:
		x,y: coordinates of matrix data
		data - matrix holding the data to be interpolated onto the mesh.
		x_mesh,y_mesh: coordinates of the mesh grids onto which we interpolate.
		
		output: 
		data_mesh:  vector of mesh interpolated data.

*/
	
#include "./InterpFromMeshToMesh2d.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]) {

	/*input: */
	double* index_data=NULL;
	int     index_data_rows;

	double* x_data=NULL;
	int     x_data_rows;
	
	double* y_data=NULL;
	int     y_data_rows;

	double* data=NULL; 
	int     data_rows;
	int     data_cols;

	double* x_prime=NULL;
	double* y_prime=NULL;
	
	int     x_prime_rows;
	int     y_prime_rows;

	double default_value;

	/*Intermediary*/
	int nods_data;
	int nels_data;
	int nods_prime;

	/* output: */
	Vec  data_prime=NULL;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	CheckNumMatlabArguments(nlhs,NLHS,nrhs,NRHS,__FUNCT__,&InterpFromMeshToMesh2dUsage);

	/*Input datasets: */
	FetchData((void**)&index_data,&index_data_rows,NULL,INDEXHANDLE,"Matrix","Mat");
	FetchData((void**)&x_data,&x_data_rows,NULL,XHANDLE,"Matrix","Mat");
	FetchData((void**)&y_data,&y_data_rows,NULL,YHANDLE,"Matrix","Mat");
	FetchData((void**)&data,&data_rows,&data_cols,DATAHANDLE,"Matrix","Mat");
	FetchData((void**)&x_prime,&x_prime_rows,NULL,XPRIMEHANDLE,"Matrix","Mat");
	FetchData((void**)&y_prime,&y_prime_rows,NULL,YPRIMEHANDLE,"Matrix","Mat");
	FetchData((void**)&default_value,NULL,NULL,DEFAULTHANDLE,"Scalar",NULL);

	/*some checks*/
	if (x_data_rows!=y_data_rows){
		throw ErrorException(__FUNCT__,"vectors x and y should have the same length!");
	}
	if (x_prime_rows!=y_prime_rows){
		throw ErrorException(__FUNCT__,"vectors x_prime and y_prime should have the same length!");
	}
	
	/*get number of elements and number of nodes in the data*/
	nels_data=index_data_rows;
	nods_data=x_data_rows;
	nods_prime=x_prime_rows;

	/* Run core computations: */
	InterpFromMeshToMesh2dx(&data_prime,index_data,x_data,y_data,nods_data,nels_data,data,data_rows,x_prime,y_prime,nods_prime,default_value);

	/*Write data: */
	WriteData(DATAPRIME,data_prime,0,0,"Vector",NULL);

	/*end module: */
	MODULEEND();
}

void InterpFromMeshToMesh2dUsage(void)
{
	_printf_("   usage:\n");
	_printf_("      data_prime=InterpFromMeshToMesh2d(index,x,y,data,x_prime,y_prime,default_value);\n\n");
	_printf_("   where:\n");
	_printf_("      x,y: coordinates of the nodes where data is defined\n");
	_printf_("      index: index of the mesh where data is defined\n");
	_printf_("      data - vector holding the data to be interpolated onto the points.\n");
	_printf_("      x_prime,y_prime: coordinates of the mesh grids onto which we interpolate.\n");
	_printf_("      default_value - default value if no interpolation is found.\n");
	_printf_("      data_prime:  vector of prime interpolated data.\n");
	_printf_("\n");
}
