/*!\file: hydrology_core.cpp
 * \brief: core of the hydrology solution
 */

#include "./cores.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"
#include "../solutionsequences/solutionsequences.h"

void hydrology_core(FemModel* femmodel){

	/*intermediary*/
	int  hydrology_model;
	int  solution_type;
	int  numoutputs=0;
	bool save_results;
	bool modify_loads=true;
	char **requested_outputs = NULL;

	/*first recover parameters common to all solutions*/
	femmodel->parameters->FindParam(&save_results,SaveResultsEnum);
	femmodel->parameters->FindParam(&hydrology_model,HydrologyModelEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
	femmodel->parameters->FindParam(&numoutputs,HydrologyNumRequestedOutputsEnum);
	if(numoutputs) femmodel->parameters->FindParam(&requested_outputs,&numoutputs,HydrologyRequestedOutputsEnum);
	/*Using the Shreve based Model*/
	if (hydrology_model==HydrologyshreveEnum){
		if(VerboseSolution()) _printf0_("   computing water heads\n");
		/*first compute slopes: */
		surfaceslope_core(femmodel);
		bedslope_core(femmodel);
		/*and then go to water column*/
		if(VerboseSolution()) _printf0_("   computing water column\n");
		femmodel->SetCurrentConfiguration(HydrologyShreveAnalysisEnum);
		solutionsequence_nonlinear(femmodel,modify_loads);

		/*transfer water column thickness to old water column thickness: */
		InputDuplicatex(femmodel,WatercolumnEnum,WaterColumnOldEnum);

	}

	/*Using the double continuum model*/
	else if (hydrology_model==HydrologydcEnum){
		/*intermediary: */
		bool       isefficientlayer;
		int        step,hydroslices;
		IssmDouble time,init_time,hydrotime,yts;
		IssmDouble dt,hydrodt;

		femmodel->parameters->FindParam(&isefficientlayer,HydrologydcIsefficientlayerEnum);
		femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
		femmodel->parameters->FindParam(&step,StepEnum);
		femmodel->parameters->FindParam(&time,TimeEnum);
		femmodel->parameters->FindParam(&hydroslices,HydrologyStepsPerStepEnum);
		femmodel->parameters->FindParam(&yts,ConstantsYtsEnum);

		init_time = time-dt; //getting the time back to the start of the timestep
		hydrotime=init_time;
		hydrodt=dt/hydroslices; //computing hydro dt from dt and a divider
		femmodel->parameters->AddObject(new DoubleParam(HydrologydtEnum,hydrodt));
		if(hydroslices>1){
			/*define which variable needs to be averaged on the sub-timestep and initialize as needed*/
			if (isefficientlayer){
				int inputtostack[4]={EffectivePressureEnum,SedimentHeadEnum,EplHeadEnum,HydrologydcEplThicknessEnum};
				int stackedinput[4]={EffectivePressureStackedEnum,SedimentHeadStackedEnum,EplHeadStackedEnum,HydrologydcEplThicknessStackedEnum};
				int averagedinput[4]={EffectivePressureTimeAverageEnum,SedimentHeadTimeAverageEnum,EplHeadTimeAverageEnum,HydrologydcEplThicknessTimeAverageEnum};
				femmodel->InitTransientOutputx(&stackedinput[0],4);
				while(hydrotime<time-(yts*DBL_EPSILON)){ //loop on hydro dts
					hydrotime+=hydrodt;
					/*save preceding timestep*/
					InputDuplicatex(femmodel,SedimentHeadEnum,SedimentHeadOldEnum);
					InputDuplicatex(femmodel,EplHeadEnum,EplHeadOldEnum);
					InputDuplicatex(femmodel,HydrologydcEplThicknessEnum,HydrologydcEplThicknessOldEnum);
					/*Proceed now to heads computations*/
					solutionsequence_hydro_nonlinear(femmodel);
					/*If we have a sub-timestep we stack the variables here*/
					femmodel->StackTransientOutputx(&inputtostack[0],&stackedinput[0],hydrotime,4);
				}
				femmodel->AverageTransientOutputx(&stackedinput[0],&averagedinput[0],init_time,4);
			}
			else{
				int inputtostack[2]={EffectivePressureEnum,SedimentHeadEnum};
				int stackedinput[2]={EffectivePressureStackedEnum,SedimentHeadStackedEnum};
				int averagedinput[2]={EffectivePressureTimeAverageEnum,SedimentHeadTimeAverageEnum};
				femmodel->InitTransientOutputx(&stackedinput[0],2);
				while(hydrotime<time-(yts*DBL_EPSILON)){ //loop on hydro dts
					hydrotime+=hydrodt;
					/*save preceding timestep*/
					InputDuplicatex(femmodel,SedimentHeadEnum,SedimentHeadOldEnum);
					/*Proceed now to heads computations*/
					solutionsequence_hydro_nonlinear(femmodel);
					/*If we have a sub-timestep we stack the variables here*/
					femmodel->StackTransientOutputx(&inputtostack[0],&stackedinput[0],hydrotime,2);
				}
				femmodel->AverageTransientOutputx(&stackedinput[0],&averagedinput[0],init_time,2);
			}
		}
		else{
			InputDuplicatex(femmodel,SedimentHeadEnum,SedimentHeadOldEnum);
			if (isefficientlayer){
				InputDuplicatex(femmodel,EplHeadEnum,EplHeadOldEnum);
				InputDuplicatex(femmodel,HydrologydcEplThicknessEnum,HydrologydcEplThicknessOldEnum);
			}
			/*Proceed now to heads computations*/
			solutionsequence_hydro_nonlinear(femmodel);
		}
	}
	else if (hydrology_model==HydrologysommersEnum){
		femmodel->SetCurrentConfiguration(HydrologySommersAnalysisEnum);
      InputDuplicatex(femmodel,HydrologyHeadEnum,HydrologyHeadOldEnum);
		solutionsequence_shakti_nonlinear(femmodel);
		if(VerboseSolution()) _printf0_("   updating gap height\n");
		HydrologySommersAnalysis* analysis = new HydrologySommersAnalysis();
		analysis->UpdateGapHeight(femmodel);
		delete analysis;
	}
	else{
		_error_("Hydrology model "<< EnumToStringx(hydrology_model) <<" not supported yet");
	}
	if(save_results){
		if(VerboseSolution()) _printf0_("   saving results \n");
		femmodel->RequestedOutputsx(&femmodel->results,requested_outputs,numoutputs);
	}
	/*Free ressources:*/
	if(numoutputs){
		for(int i=0;i<numoutputs;i++){
			xDelete<char>(requested_outputs[i]);
		}
		xDelete<char*>(requested_outputs);
	}
}
