
clear all;
steps=[5]; % [1:6]; 

if any(steps==0) % Download GRACE land_mass data {{{
	disp('   Step 0: Download GRACE land_mass data');

	% Connect to ftp server and download 
	f = ftp('podaac-ftp.jpl.nasa.gov');
	%dir(f)
	cd(f,'allData/tellus/L3/land_mass/RL05/netcdf/');
   mget(f,'GRCTellus.JPL.200204_201701.LND.RL05_1.DSTvSCS1411.nc');

	% display the content of the netcdf file. 
	%ncdisp('GRCTellus.JPL.200204_201701.LND.RL05_1.DSTvSCS1411.nc')

end % }}}

if any(steps==1) % Global mesh creation {{{ 
	disp('   Step 1: Global mesh creation');

	resolution=300;	% km. uniform meshing... 
	radius = 6.371012*10^3;	% mean radius of Earth, km

	%mesh earth: 
	md=model; 
	md.mask=maskpsl(); % use maskpsl class (instead of mask) to store the ocean function as a ocean_levelset 
	md.mesh=gmshplanet('radius',radius,'resolution',resolution);

	%figure out mask: 
	md.mask.ocean_levelset=gmtmask(md.mesh.lat,md.mesh.long); 

	save ./Models/EsaGRACE.Mesh md;
	
	plotmodel (md,'data',md.mask.ocean_levelset,'edgecolor','k');
	%export_fig('Fig1.pdf'); 

end % }}} 

if any(steps==2) % Define loads {{{ 
	disp('   Step 2: Define loads in meters of ice height equivalent');
	md = loadmodel('./Models/EsaGRACE.Mesh');

	year_month = 2007+15/365;
	time_min=year_month; 
	time_max=year_month; 

	% map GRACE water load on to the mesh for the seleted month(s) 
	water_load = grace(md.mesh.elements,md.mesh.lat,md.mesh.long,time_min,time_max); 
	
	md.esa.deltathickness = water_load*md.materials.rho_freshwater/md.materials.rho_ice; % ice height equivalent 

	save ./Models/EsaGRACE.Loads md; 
	
	plotmodel (md,'data',md.esa.deltathickness,...
		'view',[90 -90],'caxis',[-.1 .1],...
		'title','Ice height equivalent [m]');
	%export_fig('Fig2.pdf'); 

end % }}} 

if any(steps==3) % Parameterization {{{ 
	disp('   Step 3: Parameterization');
	md = loadmodel('./Models/EsaGRACE.Loads');

	% Love numbers and reference frame: CF or CM (choose one!)  
	nlove=10001;	% up to 10,000 degree 
	md.esa.love_h = love_numbers('h','CF'); md.esa.love_h(nlove+1:end)=[];
	md.esa.love_l = love_numbers('l','CF'); md.esa.love_l(nlove+1:end)=[];

	% Mask: for computational efficiency only those elements that have loads are convolved! 
	md.mask.groundedice_levelset = ones(md.mesh.numberofvertices,1); % 1 = ice is grounnded 
	md.mask.ice_levelset = ones(md.mesh.numberofvertices,1); 
	pos=find(md.esa.deltathickness~=0);
	md.mask.ice_levelset(md.mesh.elements(pos,:))=-1; % -1 = ice loads 
	md.mask.land_levelset = 1-md.mask.ocean_levelset; 

	%% IGNORE BUT DO NOT DELETE %% {{{
	% Geometry: Important only when you want to couple with Ice Flow Model 
	di=md.materials.rho_ice/md.materials.rho_water;
	md.geometry.thickness=ones(md.mesh.numberofvertices,1);
	md.geometry.surface=(1-di)*zeros(md.mesh.numberofvertices,1);
	md.geometry.base=md.geometry.surface-md.geometry.thickness;
	md.geometry.bed=md.geometry.base;
	% Materials: 
	md.initialization.temperature=273.25*ones(md.mesh.numberofvertices,1);
	md.materials.rheology_B=paterson(md.initialization.temperature);
	md.materials.rheology_n=3*ones(md.mesh.numberofelements,1);
	% Miscellaneous: 
	md.miscellaneous.name='EsaGRACE';
	%% IGNORE BUT DO NOT DELETE %% }}}  
	
	save ./Models/EsaGRACE.Parameterization md; 

end % }}} 

if any(steps==4) % Solve {{{ 
	disp('   Step 4: Solve Esa solver');
	md = loadmodel('./Models/EsaGRACE.Parameterization');

	% Request outputs 
	md.esa.requested_outputs = {'EsaUmotion','EsaNmotion','EsaEmotion'};
	
	% Cluster info 
	md.cluster=generic('name',oshostname(),'np',3); 
	md.verbose=verbose('111111111');

	% Solve 
	md=solve(md,'Esa');

	save ./Models/EsaGRACE.Solution md; 

end % }}} 

if any(steps==5) % Plot solutions {{{ 
	disp('   Step 5: Plot solutions');
	md = loadmodel('./Models/EsaGRACE.Solution');

	% solutions. 
	ur = md.results.EsaSolution.EsaUmotion*1000; % [mm] 
	un = md.results.EsaSolution.EsaNmotion*1000; % [mm] 
	ue = md.results.EsaSolution.EsaEmotion*1000; % [mm] 

	res = 1.0; % degree 

	% Make a grid of lats and lons, based on the min and max of the original vectors
	[lat_grid, lon_grid] = meshgrid(linspace(-90,90,180/res), linspace(-180,180,360/res));
	sol_grid = zeros(size(lat_grid)); 

	sol = ue; 

	% Make a interpolation object
	F = scatteredInterpolant(md.mesh.lat,md.mesh.long,sol); 
	F.Method = 'linear';
	F.ExtrapolationMethod = 'linear'; 

	% Do the interpolation to get gridded solutions... 
	sol_grid = F(lat_grid, lon_grid);
	sol_grid(isnan(sol_grid))=0; 

	% set polar unphysical 0s to Nan 
	sol_grid(lat_grid>85 & sol_grid==0) =NaN; 

	set(0,'DefaultAxesFontSize',18,'DefaultAxesLineWidth',1,'DefaultTextFontSize',18,'DefaultLineMarkerSize',8)
	figure1=figure('Position', [100, 100, 1000, 500]); 
	gcf; 
	load coast; 
	cla; 
	pcolor(lon_grid,lat_grid,sol_grid); shading flat; 
	%caxis([-0.3 0.3])
	hold on
	plot(long,lat,'k');
	%geoshow(lat,long,'DisplayType','polygon','FaceColor','white'); 	
	hold off; 
	c1=colorbar;
	colormap(jet); 
	xlim([-180 180]); 
	ylim([-90 90]); 
	grid on; 
	title(['Average change in relative sea-level [mm/yr]']);
	set(gcf,'color','w');

	%export_fig('Fig5.pdf'); 

end % }}} 

if any(steps==6) % {{{ Transient 
	disp('   Step 6: Transient run');

end % }}} 

