/*!\file ControlInput.c
 * \brief: implementation of the ControlInput object
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../classes.h"
#include "../../shared/shared.h"

/*ControlInput constructors and destructor*/
ControlInput::ControlInput(){/*{{{*/
	control_id  = 0;
	values      = NULL;
	savedvalues = NULL;
	minvalues   = NULL;
	maxvalues   = NULL;
	gradient    = NULL;
}
/*}}}*/
ControlInput::ControlInput(int in_enum_type,int input_layout_enum,IssmDouble* pvalues,IssmDouble* pmin,IssmDouble* pmax,int interp,int id){/*{{{*/

	control_id=id;
	enum_type=in_enum_type;
	layout_enum = input_layout_enum;

	_assert_(interp==P1Enum);

	switch(input_layout_enum){
		case TriaInputEnum:
			values     =new TriaInput(enum_type,pvalues,interp);
			savedvalues=new TriaInput(enum_type,pvalues,interp);
			minvalues  =new TriaInput(enum_type,pmin,interp);
			maxvalues  =new TriaInput(enum_type,pmax,interp);
			break;
		case PentaInputEnum:
			values     =new PentaInput(enum_type,pvalues,interp);
			savedvalues=new PentaInput(enum_type,pvalues,interp);
			minvalues  =new PentaInput(enum_type,pmin,interp);
			maxvalues  =new PentaInput(enum_type,pmax,interp);
			break;
		default:
			_error_("Input of Enum " << EnumToStringx(input_layout_enum) << " not supported yet by ControlInput");
	}
	gradient   =NULL;
}
/*}}}*/
ControlInput::ControlInput(int in_enum_type,int input_layout_enum,Input* input_pvalues,Input* input_pmin,Input* input_pmax,Input* input_pgrad,int interp,int id){/*{{{*/

	this->control_id=id;
	this->enum_type=in_enum_type;
	this->layout_enum = input_layout_enum;

	_assert_(interp==P1Enum);
	if(input_layout_enum!=TransientInputEnum) _error_("Wrong type of layout_enum, needs to be a TransientInputEnum");

			this->values = input_pvalues;
			this->savedvalues= NULL;
			this->minvalues  = input_pmin;
			this->maxvalues  = input_pmax;
			this->gradient =  input_pgrad;
	}
/*}}}*/
ControlInput::~ControlInput(){/*{{{*/
	if(values)      delete values;
	if(savedvalues) delete savedvalues;
	if(minvalues)   delete minvalues;
	if(maxvalues)   delete maxvalues;
	if(gradient)    delete gradient;
}
/*}}}*/

/*Object virtual functions definitions:*/
Object* ControlInput::copy() {/*{{{*/

	ControlInput* output=NULL;

	output = new ControlInput();
	output->enum_type=this->enum_type;
	output->control_id=this->control_id;
	output->layout_enum = this->control_id;

	if(values)      output->values      = xDynamicCast<Input*>(this->values->copy());
	if(savedvalues) output->savedvalues = xDynamicCast<Input*>(this->savedvalues->copy());
	if(minvalues)   output->minvalues   = xDynamicCast<Input*>(this->minvalues->copy());
	if(maxvalues)   output->maxvalues   = xDynamicCast<Input*>(this->maxvalues->copy());
	if(gradient)    output->gradient    = xDynamicCast<Input*>(this->gradient->copy());

	return output;
}
/*}}}*/
void ControlInput::DeepEcho(void){/*{{{*/

	_printf_("ControlInput:\n");
	_printf_(setw(15)<<"   ControlInput "<<setw(25)<<left<<EnumToStringx(this->enum_type)<<"\n");
	_printf_(setw(15)<<"   ControlInput "<<setw(25)<<left<<EnumToStringx(this->layout_enum)<<"\n");
	_printf_("---values: \n");     if (values)      values->Echo();
	_printf_("---savedvalues: \n");if (savedvalues) savedvalues->Echo();
	_printf_("---minvalues: \n");  if (minvalues)   minvalues->Echo();
	_printf_("---maxvalues: \n");  if (maxvalues)   maxvalues->Echo();
	_printf_("---gradient: \n");   if (gradient){    gradient->Echo();} else{_printf_("     Not set yet\n");}
}
/*}}}*/
void ControlInput::Echo(void){/*{{{*/
	this->DeepEcho();
}
/*}}}*/
int  ControlInput::Id(void){ return -1; }/*{{{*/
/*}}}*/
void ControlInput::Marshall(char** pmarshalled_data,int* pmarshalled_data_size, int marshall_direction){ /*{{{*/

	MARSHALLING_ENUM(ControlInputEnum);

	MARSHALLING(enum_type);
	MARSHALLING(control_id);
	MARSHALLING(layout_enum);

	if (marshall_direction == MARSHALLING_BACKWARD){
		switch(enum_type){
			case TriaInputEnum:
				values     =new TriaInput();
				savedvalues=new TriaInput();
				minvalues  =new TriaInput();
				maxvalues  =new TriaInput();
				gradient   =new TriaInput();
				break;
			case PentaInputEnum:
				values     =new PentaInput();
				savedvalues=new PentaInput();
				minvalues  =new PentaInput();
				maxvalues  =new PentaInput();
				gradient   =new PentaInput();
				break;
			case TransientInputEnum:
				values	  =new TransientInput();
				savedvalues=new TransientInput();
				minvalues  =new TransientInput();
				maxvalues  =new TransientInput();
				gradient   =new TransientInput();
				break;
			default:
				_error_("Input of Enum " << EnumToStringx(enum_type) << " not supported yet by ControlInput");
		}
	}
	if(values) this->values->Marshall(pmarshalled_data,pmarshalled_data_size,marshall_direction);
	if(savedvalues) this->savedvalues->Marshall(pmarshalled_data,pmarshalled_data_size,marshall_direction);
	if(minvalues) this->minvalues->Marshall(pmarshalled_data,pmarshalled_data_size,marshall_direction);
	if(maxvalues) this->maxvalues->Marshall(pmarshalled_data,pmarshalled_data_size,marshall_direction);
	if(gradient) this->gradient->Marshall(pmarshalled_data,pmarshalled_data_size,marshall_direction);
}
/*}}}*/
int  ControlInput::ObjectEnum(void){/*{{{*/

	return ControlInputEnum;

}
/*}}}*/

/*ControlInput management*/
int ControlInput::InstanceEnum(void){/*{{{*/

	return this->enum_type;

}
/*}}}*/

/*Object functions*/
void ControlInput::AXPY(Input* xinput,IssmDouble scalar){/*{{{*/
	values->AXPY(xinput,scalar);
}/*}}}*/
void ControlInput::Configure(Parameters* parameters){/*{{{*/
	if(this->values->ObjectEnum()==TransientInputEnum){
		this->values->Configure(parameters);
		this->minvalues->Configure(parameters);
		this->maxvalues->Configure(parameters);
		this->gradient->Configure(parameters);
	}
	/*do nothing: */
}
/*}}}*/
void ControlInput::Extrude(int start){/*{{{*/
	values->Extrude(start);
	savedvalues->Extrude(start);
	//gradient->Extrude();
}/*}}}*/
void ControlInput::GetGradient(Vector<IssmDouble>* gradient_vec,int* doflist){/*{{{*/
	if(gradient) gradient->GetVectorFromInputs(gradient_vec,doflist);
}/*}}}*/
void ControlInput::GetGradientValue(IssmDouble* pvalue,Gauss* gauss){/*{{{*/
	gradient->GetInputValue(pvalue,gauss);
}/*}}}*/
void ControlInput::GetInputAverage(IssmDouble* pvalue){/*{{{*/
	values->GetInputAverage(pvalue);
}/*}}}*/
void ControlInput::GetInputDerivativeValue(IssmDouble* derivativevalues, IssmDouble* xyz_list, Gauss* gauss){/*{{{*/
	values->GetInputDerivativeValue(derivativevalues,xyz_list,gauss);
}/*}}}*/
void ControlInput::GetInputValue(bool* pvalue){/*{{{*/
	values->GetInputValue(pvalue);
}/*}}}*/
void ControlInput::GetInputValue(int* pvalue){/*{{{*/
	values->GetInputValue(pvalue);
}/*}}}*/
void ControlInput::GetInputValue(IssmDouble* pvalue){/*{{{*/
	values->GetInputValue(pvalue);
}/*}}}*/
void ControlInput::GetInputValue(IssmDouble* pvalue,Gauss* gauss){/*{{{*/
	values->GetInputValue(pvalue,gauss);
}/*}}}*/
int  ControlInput::GetResultInterpolation(void){/*{{{*/

	return values->GetResultInterpolation();

}
/*}}}*/
int  ControlInput::GetResultNumberOfNodes(void){/*{{{*/

	return values->GetResultNumberOfNodes();

}
/*}}}*/
void ControlInput::GetVectorFromInputs(Vector<IssmDouble>* vector,int* doflist){/*{{{*/
	values->GetVectorFromInputs(vector,doflist);
}/*}}}*/
void ControlInput::GetVectorFromInputs(Vector<IssmDouble>* vector,int* doflist,const char* data){/*{{{*/
	if(strcmp(data,"value")==0){
		 _assert_(values);
		 values->GetVectorFromInputs(vector,doflist);
	 }
	 else if (strcmp(data,"lowerbound")==0){
		 _assert_(minvalues);
		 minvalues->GetVectorFromInputs(vector,doflist);
	 }
	 else if (strcmp(data,"upperbound")==0){
		 _assert_(maxvalues);
		 maxvalues->GetVectorFromInputs(vector,doflist);
	 }
	 else if (strcmp(data,"gradient")==0){
		 _assert_(gradient);
		 gradient->GetVectorFromInputs(vector,doflist);
	 }
	 else{
		 _error_("Data " << data << " not supported yet");
	 }
}/*}}}*/
IssmDouble ControlInput::Min(void){/*{{{*/

	return values->Min();

}
/*}}}*/
void ControlInput::SaveValue(void){/*{{{*/
	if(!values) _error_("Values of " << EnumToStringx(this->enum_type) << " not found");

	if(savedvalues) delete this->savedvalues;
	this->savedvalues=xDynamicCast<Input*>(this->values->copy());
}/*}}}*/
void ControlInput::SetGradient(Input* gradient_in,int timestep){/*{{{*/
if(this->values->ObjectEnum()!=TransientInputEnum)_error_("you are in the wrong place, go home");

	/*Get enum for current gradient*/
	switch(this->control_id){
		case 1:
			gradient_in->ChangeEnum(Gradient1Enum);
			break;
		case 2:
			gradient_in->ChangeEnum(Gradient2Enum);
			break;
		case 3:
			gradient_in->ChangeEnum(Gradient3Enum);
			break;
		default:
			_error_("more than 3 controls not implemented yet (Gradient " << this->control_id << " was requested). EnumDefinitions.h needs to be updated.");
	}

	TransientInput* transient_input = xDynamicCast<TransientInput*>(this->gradient);
	TransientInput* values_input = xDynamicCast<TransientInput*>(this->values);
	if(values_input->numtimesteps==transient_input->numtimesteps){
		TransientInput* new_trans_input = new TransientInput(ControlInputGradEnum);
		IssmDouble time = transient_input->GetTimeByOffset(timestep);
		for(int i=0;i<transient_input->numtimesteps;i++){
			if(transient_input->timesteps[i]==time) new_trans_input->AddTimeInput(xDynamicCast<TriaInput*>(gradient_in),time);
			else {
				Input* input = transient_input->GetTimeInput(transient_input->timesteps[i]);
				new_trans_input->AddTimeInput(xDynamicCast<TriaInput*>(input),transient_input->timesteps[i]);
			}
		}
		this->gradient=new_trans_input;
	}
	else{
		IssmDouble time = values_input->GetTimeByOffset(timestep);
		transient_input->AddTimeInput(gradient_in,time);
	}
}/*}}}*/
void ControlInput::SetGradient(Input* gradient_in){/*{{{*/

	/*Get enum for current gradient*/
	switch(this->control_id){
		case 1:
			gradient_in->ChangeEnum(Gradient1Enum);
			break;
		case 2:
			gradient_in->ChangeEnum(Gradient2Enum);
			break;
		case 3:
			gradient_in->ChangeEnum(Gradient3Enum);
			break;
		default:
			_error_("more than 3 controls not implemented yet (Gradient " << this->control_id << " was requested). EnumDefinitions.h needs to be updated.");
	}

	/*Delete old gradient and assign new gradient*/
	if(gradient) delete gradient;
	gradient=gradient_in;

}/*}}}*/
void ControlInput::SetInput(Input* in_input){/*{{{*/

	if(layout_enum==TransientInputEnum)_error_("need two arguments in SetInput for TransientInput Controls");
	delete values; this->values=in_input;
	this->SaveValue(); //because this is what SpawnResult saves FIXME

}/*}}}*/
void ControlInput::SetInput(Input* in_input,int timeoffset){/*{{{*/
	Input* input = this->values;
	if(input->ObjectEnum()!=TransientInputEnum)_error_("cannot have timeoffset argument if not TransientInput Control");
	TransientInput* transient_input = xDynamicCast<TransientInput*>(input);
	IssmDouble time = transient_input->GetTimeByOffset(timeoffset);
	TransientInput* new_trans_input = new TransientInput(ControlInputValuesEnum);
	for(int i=0;i<transient_input->numtimesteps;i++){
		if(transient_input->timesteps[i]==time) new_trans_input->AddTimeInput(xDynamicCast<TriaInput*>(in_input),time);
		else {
			input = transient_input->GetTimeInput(transient_input->timesteps[i]);
			new_trans_input->AddTimeInput(xDynamicCast<TriaInput*>(input),transient_input->timesteps[i]);
		}
	}
	this->values=new_trans_input;

	//	this->values->Echo();
	//this->values->Echo();
	//new_trans_input->Echo();

}/*}}}*/
Input* ControlInput::SpawnSegInput(int index1,int index2){/*{{{*/
	return values->SpawnSegInput(index1,index2);
}/*}}}*/
Input* ControlInput::SpawnTriaInput(int index1,int index2,int index3){/*{{{*/
	return values->SpawnTriaInput(index1,index2,index3);
}/*}}}*/
void ControlInput::UpdateValue(IssmDouble scalar){/*{{{*/
	if(!gradient)    _error_("Gradient of " << EnumToStringx(this->enum_type) << " not found");
	if(!savedvalues) _error_("Values of " << EnumToStringx(this->enum_type) << " not found");

	if(values) delete this->values;
	this->values=xDynamicCast<Input*>(this->savedvalues->copy());
	this->values->AXPY(gradient,scalar);
}/*}}}*/
