%SEALEVELMODEL class definition
%
%   Usage:
%      slm = sealevelmodel(varargin)
%
%      where varargin is a variable list of options: 
%
%   Example: 
%      slm = sealevel('icecap',md_greenland,'icecap',md_antarctica,'earth',md_earth);

classdef sealevelmodel
	properties (SetAccess=public) %Model fields
		% {{{
		icecaps          = {}; % list of ice cap models
		earth            = 0;  % model for the whole earth
		cluster          = 0;
		miscellaneous    = 0;
		settings         = 0;
		private          = 0;
		range            = 0;
		mergedcaps     = 0;
		%}}}
	end
	methods
		function slm = sealevelmodel(varargin) % {{{

			if nargin==0, 
				slm=setdefaultparameters(slm);
			else 
				slm=setdefaultparameters(slm);

				options=pairoptions(varargin{:}); 
			
				%recover all the icecap models: 
				slm.icecaps=getfieldvalues(options,'ice_cap',{}); 
				
				%recover the earth model:
				slm.earth = getfieldvalue(options,'earth');
			end
		end
		%}}}
		function checkconsistency(slm,solutiontype) % {{{

			%is the coupler turned on? 
			for i=1:length(slm.icecaps),
				if slm.icecaps{i}.transient.iscoupler==0,
					warning(sprintf('sealevelmodel checkconsistenty error:  icecap model %s should have the transient coupler option turned on!',slm.icecaps{i}.miscellaneous.name));
				end
			end
				
			if slm.earth.transient.iscoupler==0,
				warning('sealevelmodel checkconsistenty error:  earth model should have the transient coupler option turned on!');
			end

			%check that the transition vectors have the right size: 
			for i=1:length(slm.icecaps),
				if slm.icecaps{i}.mesh.numberofvertices ~= length(slm.earth.slr.transitions{i}),
					error(['sealevelmodel checkconsistenty issue with size of transition vector for ice cap: ' num2str(i) ' name: ' slm.icecaps{i}.miscellaneous.name]);
				end
			end
			
			%check that run_frequency is the same everywhere: 
			for i=1:length(slm.icecaps),
				if slm.icecaps{i}.slr.geodetic_run_frequency~=slm.earth.slr.geodetic_run_frequency,
					error(sprintf('sealevelmodel checkconsistenty error:  icecap model %s should have the same run frequency as earth!',slm.icecaps{i}.miscellaneous.name));
				end
			end

			%make sure steric_rate is the same everywhere: 
			for i=1:length(slm.icecaps),
				md= slm.icecaps{i}; 
				if ~isempty(find(md.slr.steric_rate - slm.earth.slr.steric_rate(slm.earth.slr.transitions{i}))),
					error(sprintf('steric rate on ice cap %s is not the same as for the earth\n',md.miscellaneous.name));
				end
			end

		end
		%}}}
		function slm = setdefaultparameters(slm) % {{{

			%initialize subclasses
			slm.icecaps           = {};
			slm.earth             = {};
			slm.miscellaneous     = miscellaneous();
			slm.settings          = settings();
			slm.private           = private();
			slm.cluster           = generic();
			slm.range             = {};
		end
		%}}}
		function disp(self) % {{{
			disp(sprintf('%19s: %-22s -- %s','icecaps'         ,['[' num2str(length(self.icecaps)) 'x1 ' class(self.icecaps) ']'],'ice caps'));
			disp(sprintf('%19s: %-22s -- %s','earth'           ,['[1x1 ' class(self.earth) ']'],'earth'));
			disp(sprintf('%19s: %-22s -- %s','settings'        ,['[1x1 ' class(self.settings) ']'],'settings properties'));
			disp(sprintf('%19s: %-22s -- %s','cluster'         ,['[1x1 ' class(self.cluster) ']'],'cluster parameters (number of cpus...)'));
			disp(sprintf('%19s: %-22s -- %s','miscellaneous'   ,['[1x1 ' class(self.miscellaneous) ']'],'miscellaneous fields'));
			disp(sprintf('%19s: %-22s -- %s','range'   ,['[1x1 ' class(self.range) ']'],'ranges'));
		end % }}}
		function self=mergeresults(self) % {{{
			champs=fields(self.icecaps{1}.results.TransientSolution);
			for i=1:length(self.mergedcaps)/2,
				md=self.mergedcaps{2*(i-1)+1}; trans=self.mergedcaps{2*(i-1)+2};
				icecaps=self.icecaps(self.range{2*(i-1)+2});
				for j=1:length(self.icecaps{1}.results.TransientSolution),
					for k=1:length(champs),
						if strcmpi(class(icecaps{1}.results.TransientSolution(j).(champs{k})),'double'),
							%vertex or element? 
							if length(icecaps{1}.results.TransientSolution(j).(champs{k}))==icecaps{1}.mesh.numberofvertices,
								md.results.TransientSolution(j).(champs{k})=zeros(md.mesh.numberofvertices,1);
								for l=1:length(trans),
									resultcap=icecaps{l}.results.TransientSolution(j).(champs{k});
									md.results.TransientSolution(j).(champs{k})(trans{l})=resultcap;
								end
							else 
								if strcmpi(champs{k},'IceVolume') | strcmpi(champs{k},'IceVolumeAboveFloatation') ,
									md.results.TransientSolution(j).(champs{k})=0;
									for l=1:length(trans),
										resultcap=icecaps{l}.results.TransientSolution(j).(champs{k});
										md.results.TransientSolution(j).(champs{k})= md.results.TransientSolution(j).(champs{k})+resultcap;
									end
								elseif strcmpi(champs{k},'time'),
									md.results.TransientSolution(j).(champs{k})= icecaps{1}.results.TransientSolution(j).(champs{k});
								else
									continue;
								end
							end
						else
							continue;
						end
											end
				end
				self.mergedcaps{2*(i-1)+1}=md;
			end
		end % }}}
		function listcaps(self) % {{{
			for  i=1:length(self.icecaps),
				disp(sprintf('%i: %s',i,self.icecaps{i}.miscellaneous.name));
			end
		end % }}}
		function viscousiterations(self) % {{{
			for  i=1:length(self.icecaps),
				ic=self.icecaps{i}; 
				mvi=ic.results.TransientSolution(1).StressbalanceConvergenceNumSteps;
				for j=2:length(ic.results.TransientSolution)-1,
					mvi=max(mvi,ic.results.TransientSolution(j).StressbalanceConvergenceNumSteps);
				end
				disp(sprintf('%i, %s: %i',i,self.icecaps{i}.miscellaneous.name,mvi));
			end
		end % }}}
		function maxtimestep(self) % {{{
			for  i=1:length(self.icecaps),
				ic=self.icecaps{i}; 
				mvi=length(ic.results.TransientSolution);
				timei=ic.results.TransientSolution(end).time;
				disp(sprintf('%i, %s: %i/%g',i,self.icecaps{i}.miscellaneous.name,mvi,timei));
			end
			mvi=length(self.earth.results.TransientSolution);
			timei=self.earth.results.TransientSolution(end).time;
			disp(sprintf('Earth: %i/%g',mvi,timei));
		end % }}}
		function self=homogeneize(self) % {{{
			mintimestep=Inf;
			for  i=1:length(self.icecaps),
				ic=self.icecaps{i}; 
				mintimestep=min(mintimestep, length(ic.results.TransientSolution));
			end
			mintimestep=min(mintimestep, length(self.earth.results.TransientSolution));
			
			for  i=1:length(self.icecaps),
				ic=self.icecaps{i}; 
				ic.results.TransientSolution=ic.results.TransientSolution(1:mintimestep);
				self.icecaps{i}=ic;
			end
			ic=self.earth;
			ic.results.TransientSolution=ic.results.TransientSolution(1:mintimestep);
			self.earth=ic;
		end % }}}
	end
end
