//NODE class definition //{{{
//	Description:
//		Contains all information for rendering an object in WebGL, including mesh, materials, shaders, and other attributes.	
//	Usage:
//		node=new Node('canvas', canvas);

function Node() { //{{{
	//properties
	// {{{
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(0,args.length));

	this.canvas = options.getfieldvalue('canvas', 								null),
	this.options = options.getfieldvalue('options', 							null),
	this.gl = canvas.gl;
	
	this.alpha = options.getfieldvalue('alpha', 								1.0);									//Shading transparency.
	this.ambientColor = options.getfieldvalue('ambientColor', 					[0.0, 0.0, 0.0, 1.0]);					//Ambient color used in lighting.
	this.animation = options.getfieldvalue('animation', 						{});									//Animtation parameters.
	this.arrays = options.getfieldvalue('arrays', 								{});									//Storage for webgl arrays.
	this.caxis = options.getfieldvalue('caxis', 								[0.0, 1.0]);							//Color axis for texturing.
	this.center = options.getfieldvalue('center', 								vec3.create());							//Center of rotation/scaling.
	this.cullFace = options.getfieldvalue('cullFace', 							this.gl.BACK);							//GL enum for face culling back/front faces.
	this.computeIndices = options.getfieldvalue('computeIndices', 				true);									//Specifies whether or not indices need to be computed during patching.
	this.disableDepthTest = options.getfieldvalue('disableDepthTest', 			false);									//GL enum for enabling/disabling depth testing.
	this.diffuseColor = options.getfieldvalue('diffuseColor', 					[0.0, 0.0, 0.0, 1.0]);					//Diffuse color in vec4 rgba format.
	this.drawMode = options.getfieldvalue('drawMode', 							this.gl.TRIANGLES);						//GL enum for draw mode.
	this.drawOrder = options.getfieldvalue('drawOrder', 						1);										//Drawing order for non-depth tested/transparent objects. Higher numbers are drawn. first.
	this.enabled = options.getfieldvalue('enabled', 							true);									//Toggles display of this nodde.
	this.enableCullFace = options.getfieldvalue('enableCullFace', 				true);									//Toggles use of face culling.
	this.hideOcean = options.getfieldvalue('hideOcean', 						false);									//ISSM shader uniform controlling ocean masking
	this.lineWidth = options.getfieldvalue('lineWidth', 						1.0);									//Controls width of gl lines. No reliable support across windows platforms.
	this.lightingBias = options.getfieldvalue('lightingBias', 					0.0);									//Controls width of gl lines. No reliable support across windows platforms.
	this.log = options.getfieldvalue('log', 									false);									//Controls logarithmic color axis scaling for texturing.
	this.maskAll = options.getfieldvalue('maskAll', 							[]);									//Masking array used for marking all region elements.
	this.maskBoundary = options.getfieldvalue('maskBoundary', 					[]);									//Masking array used for marking region boundary edges.
	this.maskColor = options.getfieldvalue('maskColor', 						vec4.fromValues(0.0, 0.0, 1.0, 1.0));	//ISSM shader uniform controlling ocean masking color.
	this.maskEnabled = options.getfieldvalue('maskEnabled', 					false);									//ISSM shader uniform toggling ocean masking.
	this.maskHeight = options.getfieldvalue('maskHeight', 						150.0);									//ISSM shader uniform controlling height at which ocean masking is cut off.
	this.maskObject = options.getfieldvalue('maskObject', 						{'enabled':false});						//Masking array options object.
	this.maskZerosColor = options.getfieldvalue('maskZerosColor', 				[1.0, 1.0, 1.0, 1.0]);					//ISSM shader uniform controlling value masking color.
	this.maskZerosEnabled = options.getfieldvalue('maskZerosEnabled', 			false);									//ISSM shader uniform toggling value masking.
	this.maskZerosTolerance = options.getfieldvalue('maskZerosTolerance', 		1e-6);									//ISSM shader uniform controlling tolerance of value to zero value when masking.
	this.maskZerosZeroValue = options.getfieldvalue('maskZerosZeroValue', 		0.5);									//ISSM shader uniform controlling zero value masking offset.
	this.mesh = options.getfieldvalue('mesh', 									undefined);								//Litegl GL.Mesh class used for drawing.
	this.name = options.getfieldvalue('name', 									'node');								//Name of node.
	this.nanIndices = options.getfieldvalue('nanIndices', 						new Set());								//Set of NaN indices to mask out of mesh.
	this.octree = options.getfieldvalue('octree', 								undefined);								//Fast raytracing octree object.
	this.pointSize = options.getfieldvalue('pointSize', 						15.0);									//Size of points when displaying GL.POINTS.
	this.shaderName = options.getfieldvalue('shaderName', 						'Colored');								//Name of shader to use.
	this.shader = options.getfieldvalue('shader', 								this.gl.shaders[this.shaderName]);		//Compiled shader object.
	this.specularColor = options.getfieldvalue('specularColor', 				[1.0, 1.0, 1.0, 1.0]);					//Specular reflection color used in lighting.
	this.specularStrength = options.getfieldvalue('specularStrength', 			1.0);									//Specular reflection power - represents specular highlight sharpness.
	this.specularPower = options.getfieldvalue('specularPower', 				5);										//Specular reflection power - represents specular highlight sharpness.
	this.texture = options.getfieldvalue('texture', 							undefined);								//GL texture object.
	this.scale = options.getfieldvalue('scale', 								vec3.fromValues(1, 1, 1));				//XYZ scaling of the node.
	this.renderObject = options.getfieldvalue('renderObject', 					{});									//Field for additional render object state information not necessarily related to node display.
	this.rotation = options.getfieldvalue('rotation', 							vec3.create());							//XYZ rotation of the node.
	this.rotationQuaternion = options.getfieldvalue('rotationQuaternion',		quat.create());							//Quaternion rotation of the node. Not currently in use.
	this.translation = options.getfieldvalue('translation', 					vec3.create());							//XYZ translation of the node.
	this.modelMatrix = options.getfieldvalue('modelMatrix', 					mat4.create());							//Model matrix computed from updateModelMatrix function.
	this.translationMatrix = options.getfieldvalue('translationMatrix', 		mat4.create());							//Intermediate translation matrix computed from updateModelMatrix function.
	this.rotationMatrix = options.getfieldvalue('rotationMatrix', 				mat4.create());							//Intermediate rotation matrix computed from updateModelMatrix function.
	this.scaleMatrix = options.getfieldvalue('scaleMatrix', 					mat4.create());							//Intermediate scale matrix computed from updateModelMatrix function.
	this.inverseModelMatrix = options.getfieldvalue('inverseModelMatrix', 		mat4.create());							//Inverse model matrix.
	this.inverseRotationMatrix = options.getfieldvalue('inverseRotationMatrix', mat4.create());							//Inverse rotation matrix.
	//}}}
	//initialize {{{
	//if (this.name in canvas.nodes) abort?
	this.updateModelMatrix();
	this.updateDiffuseColor();
	canvas.nodes[this.name] = this;
	//}}}
} //}}}
Node.prototype.eulerToQuaternion = function(pitch, roll, yaw) { //{{{
	var t0 = Math.cos(yaw * 0.5);
	var t1 = Math.sin(yaw * 0.5);
	var t2 = Math.cos(roll * 0.5);
	var t3 = Math.sin(roll * 0.5);
	var t4 = Math.cos(pitch * 0.5);
	var t5 = Math.sin(pitch * 0.5);

	var w = t0 * t2 * t4 + t1 * t3 * t5;
	var x = t0 * t3 * t4 - t1 * t2 * t5;
	var y = t0 * t2 * t5 + t1 * t3 * t4;
	var z = t1 * t2 * t4 - t0 * t3 * t5;
	return quat.fromValues(x, y, z, w);
} //}}}
Node.prototype.quaternionToEuler = function(q) { //{{{
	var ysqr = q[1] * q[1];

	// roll (x-axis rotation)
	var t0 = +2.0 * (q[3] * q[0] + q[1] * q[2]);
	var t1 = +1.0 - 2.0 * (q[0] * q[0] + ysqr);
	var roll = Math.atan2(t0, t1);

	// pitch (y-axis rotation)
	var t2 = +2.0 * (q[3] * q[1] - q[2] * q[0]);
	t2 = t2 > 1.0 ? 1.0 : t2;
	t2 = t2 < -1.0 ? -1.0 : t2;
	var pitch = Math.asin(t2);

	// yaw (z-axis rotation)
	var t3 = +2.0 * (q[3] * q[2] + q[0] * q[1]);
	var t4 = +1.0 - 2.0 * (ysqr + q[2] * q[2]);
	var yaw = Math.atan2(t3, t4);
	
	return [pitch * RAD2DEG, roll * RAD2DEG, yaw * RAD2DEG];
} //}}}
Node.prototype.updateModelMatrix = function() { //{{{
	//Update the model matrix if rotation, scale, or translation have been manually modified.
	var modelMatrix = mat4.create();

	var translationMatrix = mat4.create();
	mat4.translate(translationMatrix, translationMatrix, vec3.negate(vec3.create(), this.center)); //scale/rotation centering
	mat4.multiply(modelMatrix, translationMatrix, modelMatrix);
	
	var scaleMatrix = mat4.create();
	mat4.scale(scaleMatrix, scaleMatrix, this.scale);
	mat4.multiply(modelMatrix, scaleMatrix, modelMatrix);
	
	var rotationMatrix = mat4.create();
	var zRotationMatrix = mat4.create();	
	mat4.rotate(zRotationMatrix, zRotationMatrix, DEG2RAD * this.rotation[2], [0.0, 0.0, 1.0]);
	mat4.multiply(rotationMatrix, zRotationMatrix, rotationMatrix);
	var yRotationMatrix = mat4.create();	
	mat4.rotate(yRotationMatrix, yRotationMatrix, DEG2RAD * this.rotation[1], [0.0, 1.0, 0.0]);
	mat4.multiply(rotationMatrix, yRotationMatrix, rotationMatrix);
	var xRotationMatrix = mat4.create();	
	mat4.rotate(xRotationMatrix, xRotationMatrix, DEG2RAD * this.rotation[0], [1.0, 0.0, 0.0]);
	mat4.multiply(rotationMatrix, xRotationMatrix, rotationMatrix);
	mat4.multiply(modelMatrix, rotationMatrix, modelMatrix);

	//var rotationQuaternionX = this.eulerToQuaternion(0, -DEG2RAD * this.rotation[0], 0);
	//var rotationQuaternionY = this.eulerToQuaternion(DEG2RAD * this.rotation[1], 0, 0);
	//mat4.fromQuat(this.rotationMatrix, quat.multiply(quat.create(), rotationQuaternionY, rotationQuaternionX));

	//mat4.multiply(this.modelMatrix, this.rotationMatrix, this.modelMatrix);	
		
	
	mat4.identity(translationMatrix);
	mat4.translate(translationMatrix, translationMatrix, this.center); //relative translation
	mat4.multiply(modelMatrix, translationMatrix, modelMatrix);
	
	mat4.identity(translationMatrix);
	mat4.translate(translationMatrix, translationMatrix, this.translation); //absolute translation
	mat4.multiply(modelMatrix, translationMatrix, modelMatrix);
	
	this.modelMatrix = modelMatrix;
	this.inverseModelMatrix = mat4.invert(mat4.create(), modelMatrix);
	this.rotationMatrix = rotationMatrix;
	this.inverseRotationMatrix = mat4.invert(mat4.create(), rotationMatrix);
} //}}}
Node.prototype.updateDiffuseColor = function() { //{{{
	//Update the diffuse color with an RGB color name or vec4 containing r, g, b, and alpha values from 0.0 to 1.0
	var color = this.diffuseColor;
	if (typeof color === 'string') {
		color = new RGBColor(color);
		if (color.ok) color = [color.r/255.0, color.g/255.0, color.b/255.0, 1.0];
		else throw Error(sprintf("s%s%s\n","initWebGL error message: cound not find out edgecolor color for curent canvas ", canvas));
	}
	this.diffuseColor = color;
} //}}}
Node.prototype.transform = function() { //{{{
	//Transforms the translation, rotation, or scle fo the node and updates the model matrix.
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(0,args.length));
	
	var translation = options.getfieldvalue('translation', undefined);
	var rotation = options.getfieldvalue('rotation', undefined);
	var scale = options.getfieldvalue('scale', undefined);
	
	if (!VESL.Helpers.isEmptyOrUndefined(translation)) this.translation = translation;
	if (!VESL.Helpers.isEmptyOrUndefined(rotation)) this.rotation = rotation;
	if (!VESL.Helpers.isEmptyOrUndefined(scale)) this.scale = scale;
	this.updateModelMatrix();
} //}}}
Node.prototype.patch = function() { //{{{
	//Emulates the behavior of MATLAB patch function by constructing a mesh from arguments.
	//Limitations:
	//	-Expects pair labeled arguments ('FaceColor','none',...).
	//	-Only handles Face/Vertices/FaceVertexCData element/node plots.
	//	-Only supports FaceColor 'interp' and 'none'.
	
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(0,args.length));

	var faces = options.getfieldvalue('Faces', undefined);
	var vertices = options.getfieldvalue('Vertices', undefined);
	var faceVertexCData = options.getfieldvalue('FaceVertexCData', undefined);
	var faceColor = options.getfieldvalue('FaceColor', 'interp');
	var edgeColor = options.getfieldvalue('EdgeColor', this.diffuseColor);
	var lineWidth = options.getfieldvalue('linewidth', 1);
	
	this.faces = faces;
	this.vertices = vertices;
	this.diffuseColor = edgeColor;
	this.updateDiffuseColor();
	this.computeMasks();
	
	this.patchVertices(faceVertexCData, faces, vertices);
	this.patchCoords(faceVertexCData, faces, vertices);
	this.patchIndices(faces, faceColor);

	this.mesh = GL.Mesh.load(this.arrays, null, null, this.gl);
	this.mesh.computeNormals();
	this.computeOctree();
} //}}}
Node.prototype.patchVertices = function(faceVertexCData, faces, vertices) { //{{{
	//Patch subfunction for processing xyz vertices.
	var vertexArray;
	var face;
	
	if (VESL.Helpers.isEmptyOrUndefined(faceVertexCData)) {
		vertexArray = new Float32Array(vertices[0].length * 3);
		for(var i = 0, v = 0; i < vertices[0].length; i++) {	
			vertexArray[v++] = vertices[0][i];
			vertexArray[v++] = vertices[1][i];
			vertexArray[v++] = vertices[2][i];
		}
	}
	else if (!Array.isArray(faceVertexCData[0])) { //indexed plot - faceVertexCData = [0, 2.32, 231.1, ...]
		if (faceVertexCData.length === faces.length) { //element plot
			vertexArray = new Float32Array(faces.length * 3 * 3);
			for(var i = 0, v = 0, t = 0; i < faces.length; i++) {
				face = [faces[i][0] - 1, faces[i][1] - 1, faces[i][2] - 1];
				for(var j = 0; j < face.length; j++) {
					if (isNaN(faceVertexCData[i])) {
						this.nanIndices.add(i);
						vertexArray[v++] = 0.0;
						vertexArray[v++] = 0.0;
						vertexArray[v++] = 0.0;
					}
					else {
						vertexArray[v++] = vertices[0][face[j]];
						vertexArray[v++] = vertices[1][face[j]];
						vertexArray[v++] = vertices[2][face[j]];
					}
				}
			}
			this.computeIndices = false;
		}
		else if (faceVertexCData.length === vertices[0].length) { //node plot
			vertexArray = new Float32Array(vertices[0].length * 3);
			for(var i = 0, v = 0, t = 0; i < vertices[0].length; i++) {
				if (isNaN(faceVertexCData[i])) {
					this.nanIndices.add(i);
					vertexArray[v++] = 0.0;
					vertexArray[v++] = 0.0;
					vertexArray[v++] = 0.0;
				}
				else {
					vertexArray[v++] = vertices[0][i];
					vertexArray[v++] = vertices[1][i];
					vertexArray[v++] = vertices[2][i];
				}
			}
		}
	}
	else if (Array.isArray(faceVertexCData[0])) { //JS specific: precomputed UV coord plot - faceVertexCData = [[0, 0.99, 0.4, ...],[0, 0.99, 0.4, ...]]
		if (faceVertexCData[0].length === faces.length) { //element plot
			vertexArray = new Float32Array(faces.length * 3 * 3);
			for(var i = 0, v = 0, t = 0; i < faces.length; i++) {
				face = [faces[i][0] - 1, faces[i][1] - 1, faces[i][2] - 1];
				for(var j = 0; j < face.length; j++) {					
					vertexArray[v++] = vertices[0][face[j]];
					vertexArray[v++] = vertices[1][face[j]];
					vertexArray[v++] = vertices[2][face[j]];
				}
			}
			this.computeIndices = false;
		}
		else if (faceVertexCData[0].length === vertices[0].length) { //node plot
			vertexArray = new Float32Array(vertices[0].length * 3);
			for(var i = 0, v = 0, t = 0; i < vertices[0].length; i++) {
				vertexArray[v++] = vertices[0][i];
				vertexArray[v++] = vertices[1][i];
				vertexArray[v++] = vertices[2][i];
			}
		}
	}
	this.arrays.vertices = vertexArray;
} //}}}
Node.prototype.patchCoords = function(faceVertexCData, faces, vertices) { //{{{
	//Patch subfunction for processing texture coords/UVs.
	var coordArray;
	var cramge;
	var caxis = this.caxis;
	var face;
	
	if (VESL.Helpers.isEmptyOrUndefined(faceVertexCData)) { return; }
	
	//Use logarithmic scaling if it is valid
	if (this.log !== false && this.log !== 'off') {	
		caxis = [
			Math.log10(caxis[0]) / Math.log10(this.log),
			Math.log10(caxis[1]) / Math.log10(this.log)
		];
	}
	
	if (!Array.isArray(faceVertexCData[0])) { //indexed plot - faceVertexCData = [0, 2.32, 231.1, ...]
		if (faceVertexCData.length === faces.length) { //element plot
			coordArray = new Float32Array(faces.length * 3 * 2);
			crange = caxis[1] - caxis[0];
			for(var i = 0, v = 0, t = 0; i < faces.length; i++) {
				face = [faces[i][0] - 1, faces[i][1] - 1, faces[i][2] - 1];
				for(var j = 0; j < face.length; j++) {
					if (isNaN(faceVertexCData[i])) {
						this.nanIndices.add(i);
						coordArray[t++] = 0.0;
						coordArray[t++] = 0.0;
					}
					else {
						coordArray[t++] = this.queryCoordMask(i); //Account for special texturing
						coordArray[t++] = clamp((faceVertexCData[i] - caxis[0]) / crange, 0.0, 1.0);
					}
				}
			}
			this.computeIndices = false;
		}
		else if (faceVertexCData.length === vertices[0].length) { //node plot
			coordArray = new Float32Array(vertices[0].length * 2);
			crange = caxis[1] - caxis[0];
			for(var i = 0, v = 0, t = 0; i < vertices[0].length; i++) {
				if (isNaN(faceVertexCData[i])) {
					this.nanIndices.add(i);
					coordArray[t++] = 0.0;
					coordArray[t++] = 0.0;
				}
				else {
					coordArray[t++] = this.queryCoordMask(i); //Account for special texturing
					coordArray[t++] = clamp((faceVertexCData[i] - caxis[0]) / crange, 0.0, 1.0);
				}
			}
		}
	}
	else if (Array.isArray(faceVertexCData[0])) { //JS specific: precomputed UV coord plot - faceVertexCData = [[0, 0.99, 0.4, ...],[0, 0.99, 0.4, ...]]
		if (faceVertexCData[0].length === faces.length) { //element plot
			coordArray = new Float32Array(faces.length * 3 * 2);
			for(var i = 0, v = 0, t = 0; i < faces.length; i++) {
				face = [faces[i][0] - 1, faces[i][1] - 1, faces[i][2] - 1];
				for(var j = 0; j < face.length; j++) {					
					coordArray[t++] = faceVertexCData[0][i];
					coordArray[t++] = faceVertexCData[1][i];
				}
			}
			this.computeIndices = false;
		}
		else if (faceVertexCData[0].length === vertices[0].length) { //node plot
			coordArray = new Float32Array(vertices[0].length * 2);
			for(var i = 0, v = 0, t = 0; i < vertices[0].length; i++) {
				coordArray[t++] = faceVertexCData[0][i];
				coordArray[t++] = faceVertexCData[1][i];
			}
		}
	}
	this.arrays.coords = coordArray;
} //}}}
Node.prototype.patchIndices = function(faces, faceColor) { //{{{
	//Patch subfunction for processing faces/elements/triangles/indices.
	var indexArray;
	var face;
	
	if (faceColor === 'none') { //Check for wireframe mesh rendering
		if (this.drawMode === this.gl.TRIANGLES) { //NOTE: Stopgap to allow gl.LINE_STRIP nodes render normally. Only use case for faceColor === 'none' is for plot_mesh
			this.drawMode = this.gl.LINES;
		}
	}
	
	if (this.computeIndices === true && !VESL.Helpers.isEmptyOrUndefined(faces)) {
		if (!VESL.Helpers.isEmptyOrUndefined(faces[0])) { //Check for 2D format and process if needed
			if (faceColor !== 'none') { //Check for triangle rendering
				indexArray = new Uint16Array(faces.length * 3);
				for(var i = 0, f = 0; i < faces.length; i++) {
					face = [faces[i][0] - 1, faces[i][1] - 1, faces[i][2] - 1];
					if (this.nanIndices.has(face[0]) || this.nanIndices.has(face[1]) || this.nanIndices.has(face[2])) continue; //Skip triangle if contains NaN value.
					indexArray[f++] = faces[i][0] - 1;
					indexArray[f++] = faces[i][1] - 1;
					indexArray[f++] = faces[i][2] - 1;
				}
			}
			else { //Check for wireframe mesh rendering
				indexArray = new Uint16Array(faces.length * 6);
				for(var i = 0, f = 0; i < faces.length; i++) {
					indexArray[f++] = faces[i][0] - 1;
					indexArray[f++] = faces[i][1] - 1;
					indexArray[f++] = faces[i][1] - 1;
					indexArray[f++] = faces[i][2] - 1;
					indexArray[f++] = faces[i][2] - 1;
					indexArray[f++] = faces[i][0] - 1;
				}
			}
		}
		else { //Else, assume face indices have already been processed
			indexArray = faces;
		}
		this.arrays.triangles = indexArray;
	}
} //}}}
Node.prototype.updateBuffer = function() { //{{{
	//Updates the mesh buffers provided in place.
	//NOTE: Only support coord buffers currently.
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(0,args.length));
	
	var coords = options.getfieldvalue('Coords', undefined);
	if (!VESL.Helpers.isEmptyOrUndefined(coords)) {
		this.patchCoords(coords, this.faces, this.vertices);
		var buffer = this.mesh.getBuffer("coords");
		buffer.data = this.arrays.coords;
		buffer.upload(this.gl.DYNAMIC_DRAW);
	}
} //}}}
Node.prototype.computeOctree = function() { //{{{
	//Computes and caches octrees for a node.
	var octree = this.canvas.octrees[this.name];
	if (VESL.Helpers.isEmptyOrUndefined(octree)) {
		octree = new GL.Octree(this.mesh);
	}
	this.canvas.octrees[this.name] = octree;
	this.octree = octree;
} //}}}
Node.prototype.geometryShader = function() { //{{{
	//Emulates OpenGL geometry shaders by rendering each point as a mesh.
	//Parameters:
	//	Mesh - the geometry to duplicate for each point.
	//	Vertices - the list of points to shade.
	//	Indices - (optional) ordered list for non-ordered Vertices objects.
	
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(0,args.length));

	var mesh = options.getfieldvalue('Mesh', undefined);
	var vertices = options.getfieldvalue('Vertices', undefined);
	var indices = options.getfieldvalue('Indices', undefined);
	
	//For handling key-value object arrays, like xcity
	for (var i = 0; i < vertices.length; i++) {
		if (!Array.isArray(vertices[i])) {
			var array = [];
			var coordinateObject = vertices[i];
			var j = 0;
			if (VESL.Helpers.isEmptyOrUndefined(indices)) {
				for (var key in coordinateObject) {
					console.log(key);
					array[j++] = coordinateObject[key];
				}
			}
			else {
				for (var k = 0; k < indices.length; k++) {
					array[j++] = coordinateObject[indices[k]];
				}
			}
			vertices[i] = array;
		}
	}
	
	var x = vertices[0];
	var y = vertices[1];
	var z = vertices[2];
	var meshVertices = mesh.getBuffer('vertices').data;
	var meshIndicies = mesh.getIndexBuffer('triangles').data;
	var indices = new Uint16Array(meshIndicies.length * x.length);
	var size = meshVertices.length * x.length / 3;
	newX = new Float32Array(size);
	newY = new Float32Array(size);
	newZ = new Float32Array(size);
	
	//For each vertex in vertices, instantiate mesh geomtry centered around that point.
	for(var i = 0, v = 0, e = 0; i < x.length; i++){
		var vector = [x[i], y[i], z[i]];
		for (var j = 0; j < meshVertices.length;) {
			newX[v] = meshVertices[j++] + x[i];
			newY[v] = meshVertices[j++] + y[i];
			newZ[v++] = meshVertices[j++] + z[i];
		}
		var offset = i * meshVertices.length / 3;
		for (var j = 0; j < meshIndicies.length;) {
			indices[e++] = meshIndicies[j++] + offset;
		}
	}
	
	this.patch('Faces', indices, 'Vertices', [newX, newY, newZ], 'FaceColor', 'interp');
} //}}}
Node.prototype.scaleVertices = function(md, x, y, z, elements, scale, maskObject) { //{{{
	//Scales and returns vertices x, y, and z by factor scale. Uses md.geometry.scale for heightscaling in 3d meshes.
	var region = maskObject.region;
	var mask = maskObject.enabled ? maskObject.heightmask[region] : [];
	var maskScale = maskObject.enabled ? maskObject.scale : 1;
	
	//The maskScaleField is the height scaling array. Use one if provided by maskObject.field, otherwise use md.geometry.surface
	var maskScaleField = !VESL.Helpers.isEmptyOrUndefined(maskObject.scaleField) ? maskObject.scaleField : md.geometry.surface;
	//If md.geometry.surface was empty and was used as maskScaleField, assign an empty array to both.
	if (!maskScaleField) {
		md.geometry.surface = NewArrayFill(md.mesh.x.length,0);
		maskScaleField = md.geometry.surface;
	}
	
	//Scale in 3D if using globe model, or in 2D otherwise.
	if (md.mesh.classname() === 'mesh3dsurface') {
		var element;
		if (x.length === md.mesh.numberofelements) { //element plot
			for (var i = 0; i < x.length; i++) {
				if (maskObject.enabled) { //Scale the element if mask is not enabled, or if it is, only if it is also in the mask
					element = elements[i];
					for (var j = 0; j < 3; j++) {
						xyz = vec3.fromValues(x[element[j] - 1], y[element[j] - 1], z[element[j] - 1]);
						magnitude = 1 + maskScaleField[element[j] - 1] * mask[i] * scale * maskScale / vec3.length(xyz);
						vec3.scale(xyz, xyz, magnitude);
						x[element[j] - 1] = xyz[0];
						y[element[j] - 1] = xyz[1];
						z[element[j] - 1] = xyz[2];
					}
					
				}
			}
		}
		else if (x.length === md.mesh.numberofvertices) { //node plot
			for (var i = 0; i < x.length; i++) {
				if (!maskObject.enabled || mask[i] === 1) { //Scale the node if mask is not enabled, or if it is, only if it is also in the mask
					xyz = vec3.fromValues(x[i], y[i], z[i]);
					magnitude = 1 + maskScaleField[i] * scale * maskScale / vec3.length(xyz);
					vec3.scale(xyz, xyz, magnitude);
					x[i] = xyz[0];
					y[i] = xyz[1];
					z[i] = xyz[2];
				}
			}
		}
	} else {
		z = z.slice();
		var zMin = ArrayMin(maskScaleField);
		for(var i = 0; i < z.length; i++) {
			if (!maskObject.enabled || mask[i] === 1) { //Scale the element if mask is not enabled, or if it is, only if it is also in the mask
				z[i] = (z[i] - zMin) * scale + zMin;
			}
		}
	}
	return [x, y, z];
} //}}}
Node.prototype.computeMasks = function() { //{{{
	//NOTE: Since sets are not yet widely supported, use array with binary values to represent if object is in set.
	//NOTE: Only support element wise masks for now.
	var maskObject = this.maskObject;
	if (!maskObject.enabled) { return; }
	var region = maskObject.region;
	var mask = maskObject.mask[region];
	var maskRegion = []; //new Set()
	var maskAdjacency = []; //new Set()
	var adjacentElements;
	var adjacentElements2;
	var adjacentNodes;
	var adjacentNodes2;
	var adjacencyLength; //length of connectivity array - 3 for elements, many for nodes
	var index;
	//For each element in the mask elements array,
	if (mask.length === md.mesh.numberofelements) { //element plot
		if (md.mesh.elementconnectivity) {
			var adjacencyLength = 3;
			for (var i = 0; i < mask.length; i++) {
				//Determine if element is in the mask
				if (mask[i] === 1) {
					//Then add each element to the adjacency set and mask region set
					adjacentElements = md.mesh.elementconnectivity[i];
					for (var j = 0; j < adjacencyLength; j++) {
						index = adjacentElements[j] - 1;
						maskAdjacency[index] = true;
						
						//Get second level of adjacent elemnents to form solid boundary (first layer look like this: /\/\/\/\/\; second layer looks like this \/\/\/\/\/; combined = |       |
						adjacentElements2 = md.mesh.elementconnectivity[index];
						for (var k = 0; k < adjacencyLength; k++) {
							index = adjacentElements2[k] - 1;
							maskAdjacency[index] = true;
						}
					}
					maskRegion[i] = true;
				}
			}
		}
	}
	else if (mask.length === md.mesh.numberofvertices) { //node plot
		if (md.mesh.nodeconnectivity) {
			var adjacencyLength = md.mesh.nodeconnectivity[0].length;
			for (var i = 0; i < mask.length; i++) {
				//Determine if node is in the mask
				if (mask[i] === 1) {
					//Then add each node to the adjacency set and mask region set
					adjacentNodes = md.mesh.nodeconnectivity[i];
					for (var j = 0; j < adjacencyLength; j++) {
						index = adjacentNode[j];
						maskAdjacency[index] = true;
						
						//Get second level of adjacent elemnents to form solid boundary (first layer look like this: /\/\/\/\/\; second layer looks like this \/\/\/\/\/; combined = |       |
						adjacentNodes2 = md.mesh.nodeconnectivity[index];
						for (var k = 0; k < adjacencyLength; k++) {
							index = adjacentNodes2[k];
							maskAdjacency[index] = true;
						}
					}
					maskRegion[i] = true;
				}
			}
		}
	}
	
	//Then subtract the mask elements array from the mask adjacency array.
	var maskBoundary = [];
	for (var i in maskAdjacency) {
		if (!(i in maskRegion)) {
			maskBoundary[i] = true;
		}
	}
	this.maskBoundary = maskBoundary;
	
	//For each element not in any mask region (default to mask[0] in slr-gfm), change color to mask color.
	var maskAll = maskObject.mask[0];
	this.maskAll = maskAll;
} //}}}
Node.prototype.queryCoordMask = function(index) { //{{{
	//Calculate UV coordinate offsets for special color texture lookup.
	if (!this.maskObject.enabled) {
		return 1.0 
	}
	else {
		//var sections = 3; //Object.keys(maskObject.colors).length + 1
		//var boundaryCoord = 0.5 / sections;
		//var maskCoord = 1.5 / sections;
		//var textureCoord = 2.5 / sections;
		if (index in this.maskBoundary) {
			return 0.166666667; //boundaryCoord - edge color
		}
		else if (this.maskAll[index] === 0) {
			return 0.5; //maskCoord - masked color
		}
		else {
			return 0.833333333; //textureCoord - default colormap
		}
	}
} //}}}
Node.prototype.mergeVertices = function(x1, y1, z1, elements1, x2, y2, z2, elements2) { //{{{
	//Merges and returns two sets of indexed xyz vertices.
	elements2 = elements2.slice();
	for (var i = 0, offset = x1.length; i < elements2.length; i++) {
		elements2[i] = [elements2[i][0] + offset, elements2[i][1] + offset, elements2[i][2] + offset];
	}
	return {x:x1.concat(x2), y:y1.concat(y2), z:z1.concat(z2), elements:elements1.concat(elements2)};
} //}}}
//}}}
