/*This is where we have all our webgl relevant functionality for the plotting routines: */
//{{{ Canvas Initialization
function initCanvas(options) {
	//Initialize open Gl for each canvas and clear any previous animation handlers, once per plotmodel call:
	canvas = document.getElementById(options.getfieldvalue('canvasid'));
	//var canvas = document.getElementById(options.getfieldvalue('canvasid'));
	if (!canvas.initialized) {
		if (!VESL.Helpers.isEmptyOrUndefined(canvas.draw) && canvas.draw.handler !== 0)	{ window.cancelAnimationFrame(canvas.draw.handler); }
		if (!VESL.Helpers.isEmptyOrUndefined(canvas.animation) && canvas.animation.handler !== 0) { clearInterval(canvas.animation.handler); }
		initWebGL(canvas, options);
		draw(canvas);
		canvas.initialized = true;
		
		//The onStart event triggers once per plotmodel call load after WebGL and canvas initialization are complete
		canvas.selector.trigger('onStart', [canvas]);
	}
	return canvas;
}
function initWebGL(canvas, options) { //{{{
	//Initialize canvas.gl on page load, reusing gl context on additional runs
	var gl = canvas.gl;
	if (VESL.Helpers.isEmptyOrUndefined(gl)) {
		gl = GL.create({canvas: canvas});
		gl.enable(gl.DEPTH_TEST); // Enable depth testing
		gl.depthFunc(gl.LEQUAL); // Near things obscure far things
		gl.enable(gl.BLEND); // Enable color blending/overlay
		gl.enable(gl.CULL_FACE); // Enable face culling
		gl.cullFace(gl.FRONT);
		gl.shaders = loadShaders(gl, options.getfieldvalue('rootpath', '/canvas')); // Load shaders and store them in gl object.
		gl.textures = {};
		
		// Add event listeners for canvas
		var displayview = options.getfieldvalue('displayview', 'off') === 'on';
		var displayzoom = options.getfieldvalue('displayzoom', 'off') === 'on';
		var mc = new Hammer(canvas);	
		mc.add(new Hammer.Pan({threshold: 0, pointers: 0}));
		mc.add(new Hammer.Pinch({threshold: 0})).recognizeWith(mc.get('pan'));
		mc.on('tap', function (ev) {onTap(ev, canvas);});
		mc.on('panstart panmove', function (ev) {onPan(ev, canvas, displayview);});
		mc.on('pinchstart pinchmove', function (ev) {onPinch(ev, canvas, displayview);});
		canvas.addEventListener('mousewheel', function (ev) {onZoom(ev, canvas, displayzoom)}, false);
		canvas.addEventListener('DOMMouseScroll', function (ev) {onZoom(ev, canvas, displayzoom)}, false);
		
		//Add persistent state variables
		canvas.nodes = {};
		canvas.octrees = {};
		canvas.unitNode = {};
		canvas.unitData = {};
		canvas.unitMovieData = {};
		
		canvas.gl = gl;
		canvas.assetsPath = options.getfieldvalue('rootpath', '/canvas');
		canvas.id = options.getfieldvalue('canvasid', '.sim-canvas');
		canvas.selector = $('#' + canvas.id);
		canvas.textcanvas = null;
		canvas.overlaycanvas = null;
		canvas.permalinkUsed = false;
		
		typedArraySliceSupport();
	}
	
	if (options.getfieldvalue('clf','on')=='on') {
		//Add context state variables
		canvas.render = options.getfieldvalue('render', {});
		canvas.controlSensitivity = options.getfieldvalue('controlsensitivity', 1);
		if (options.getfieldvalue('clf','on')=='on') canvas.overlayHandlers = {};
		var backgroundcolor = new RGBColor(options.getfieldvalue('backgroundcolor', 'lightcyan'));
		if (backgroundcolor.ok) { canvas.backgroundcolor = [backgroundcolor.r/255.0, backgroundcolor.g/255.0, backgroundcolor.b/255.0, 1.0]; }
		else { throw Error(sprintf('s%s%s\n','initWebGL error message: cound not find out background color for current canvas ', canvas)); }
		
		//Property intiialization, using values from options first, then from default values.
		var atmosphere = options.getfieldvalue('atmosphere', {});
		canvas.atmosphere = { 														//Default Values
			wavelength_r: defaultFor(atmosphere.wavelength_r, 			0.65), 		//0.65		Red wavelength (micrometers)
			wavelength_g: defaultFor(atmosphere.wavelength_g, 			0.57),		//0.57		Green wavelength (micrometers)
			wavelength_b: defaultFor(atmosphere.wavelength_b, 			0.475),		//0.475		Green wavelength (micrometers)
			eSun: 	defaultFor(atmosphere.eSun, 						100.0),		//20.0		Sun intensity	
			kRayleigh: defaultFor(atmosphere.kRayleigh, 				0.0025),	//0.0025	Rayleigh scattering amount
			kMie: defaultFor(atmosphere.kMie, 							0.000), 	//0.01		Mie scattering amount
			g: defaultFor(atmosphere.g, 								-0.99),		//-0.99		Mie phase asymmetry/direction factor
			hdr_exposure: defaultFor(atmosphere.hdr_exposure, 			0.8),		//0.8		High Dynamic Range Exposure
			scaleHeight: defaultFor(atmosphere.scaleHeight, 			1.25), 		//1.025		Scale of height of atmosphere to earth radius.
			scaleDepth: defaultFor(atmosphere.scaleDepth, 				0.25), 		//0.25		Percentage altitude at which the atmosphere's average density is found
			a: defaultFor(atmosphere.a, 								-0.00287),	//-0.00287	Scaling constant a
			b: defaultFor(atmosphere.b, 								0.459),		//0.459		Scaling constant b
			c: defaultFor(atmosphere.c, 								3.83),		//3.83		Scaling constant c
			d: defaultFor(atmosphere.d, 								-6.80),		//-6.80		Scaling constant d
			e: defaultFor(atmosphere.e, 								3.6),		//5.25		Scaling constant e. Lower when increasing atmosphere scale.
			attenuation: defaultFor(atmosphere.attenuation, 			0.5)		//0.5		Strength of atmospheric scattering on ground shading.
		};
		updateAtmosphereParameters(canvas);
			
		var animation = options.getfieldvalue('movies', {});
		canvas.animation = {
			frame: defaultFor(animation.frame, 							0),
			play: defaultFor(animation.play, 							true),
			increment: defaultFor(animation.increment, 					true),
			fps: defaultFor(animation.fps, 								4),
			interval: defaultFor(animation.interval, 					1000 / defaultFor(animation.fps, 4)),
			loop: defaultFor(animation.loop, 							true),
			handler: defaultFor(animation.handler, 						0)
		}
		var brush = options.getfieldvalue('brush', {});
		canvas.brush = {
			enabled: defaultFor(brush.enabled, 							false),
			strength: defaultFor(brush.strength, 						0.075),
			falloff: defaultFor(brush.falloff, 							0.5),
			hit: defaultFor(brush.hit, 									{})
		};
		var camera = options.getfieldvalue('camera', {});
		canvas.camera = {
			position: defaultFor(camera.position, 						vec3.create()),
			rotation: defaultFor(camera.rotation, 						quat.create()),
			relativePosition: defaultFor(camera.relativePosition, 		vec3.create()),
			direction: defaultFor(camera.direction, 					vec3.create()),
			near: defaultFor(camera.near, 								1e3),
			far: defaultFor(camera.far, 								1e10),
			fov: defaultFor(camera.fov, 								45),
			vMatrix: defaultFor(camera.vMatrix, 						mat4.create()),
			pMatrix: defaultFor(camera.pMatrix, 						mat4.create()),
			vpMatrix: defaultFor(camera.vpMatrix, 						mat4.create()),
			vInverseMatrix: defaultFor(camera.vInverseMatrix, 			mat4.create()),
			pInverseMatrix: defaultFor(camera.pInverseMatrix, 			mat4.create()),
			vpInverseMatrix: defaultFor(camera.vpInverseMatrix, 		mat4.create()),
			ready: defaultFor(camera.ready, 							false)
		};
		var dataMarkers = options.getfieldvalue('datamarkers', {});
		canvas.dataMarkers = {
			enabled: defaultFor(dataMarkers.enabled, 					true),
			values: defaultFor(dataMarkers.values, 						[]),
			image: defaultFor(dataMarkers.image, 						canvas.assetsPath + '/data-markers/data_marker.svg'),
			size: defaultFor(dataMarkers.size, 							[32, 32]),
			format: defaultFor(dataMarkers.format, 						['X: %.2em<br>Y: %.2em<br>Z: %.2em<br>Value: %0.1f']),
			animated: defaultFor(dataMarkers.animated, 					false),
			labels: defaultFor(dataMarkers.labels, 						['x', 'y', 'z', 'value']),
			font: defaultFor(dataMarkers.font, 							''),
			marker: defaultFor(dataMarkers.marker, 						document.getElementById('sim-data-marker-' + canvas.id)),
			reposition: defaultFor(dataMarkers.reposition, 				true)
		};
		var draw = options.getfieldvalue('draw', {});
		canvas.draw = {
			ready: defaultFor(draw.ready, 								false),
			handler: defaultFor(draw.handler, 							null)
		};
		var view = options.getfieldvalue('view', {});
		canvas.view = {
			position: defaultFor(view.position, 						[0.0, 0.0, 0.0]),
			rotation: defaultFor(view.rotation, 						[0, 90]),
			zoom: defaultFor(view.zoom, 								1.0),
			zoomLimits: defaultFor(view.zoomLimits, 					[0.001, 100.0]),
			lastZoom: defaultFor(view.lastZoom, 						1.0),
			lightingBias: defaultFor(view.lightingBias, 				0.75),
			azimuthLimits: defaultFor(view.azimuthLimits, 				[0, 360]),
			elevationLimits: defaultFor(view.elevationLimits, 			[-180, 180]),
			panningEnabled: defaultFor(view.panningEnabled, 			false),
			preventDefaultOnPan: defaultFor(view.preventDefaultOnPan, 	false),
			twod: defaultFor(view.twod, 								false)
		};

		// Override with parameters from URL, if any
		VESL.UI.parsePermalinkCanvas(canvas);
	}
} //}}}
function loadShaders(gl, assetsPath) { //{{{
	var shaders = {};
	// NOTE: Consider changing fromURL() to XMLHttpRequest with responseType = 'text' to avoid XML Parsing Error (shader files are not encoded as XML).
	shaders.Colored = new GL.Shader.fromURL(assetsPath + '/shaders/Colored.vsh', assetsPath + '/shaders/Colored.fsh', null, gl);
	shaders.ColoredDiffuse = new GL.Shader.fromURL(assetsPath + '/shaders/ColoredDiffuse.vsh', assetsPath + '/shaders/ColoredDiffuse.fsh', null, gl);
	shaders.Textured = new GL.Shader.fromURL(assetsPath + '/shaders/Textured.vsh', assetsPath + '/shaders/Textured.fsh', null, gl);
	shaders.TexturedDiffuse = new GL.Shader.fromURL(assetsPath + '/shaders/TexturedDiffuse.vsh', assetsPath + '/shaders/TexturedDiffuse.fsh', null, gl);
	shaders.SkyFromSpace = new GL.Shader.fromURL(assetsPath + '/shaders/SkyFromSpace.vert', assetsPath + '/shaders/SkyFromSpace.frag', null, gl);
	shaders.GroundFromSpace = new GL.Shader.fromURL(assetsPath + '/shaders/GroundFromSpace.vert', assetsPath + '/shaders/GroundFromSpace.frag', null, gl);
	return shaders;
} //}}}
function initTexture(gl, imageSource) { //{{{
	//Initialize textures, or load from memory if they already exist.
	if (VESL.Helpers.isEmptyOrUndefined(gl.textures[imageSource])) {
		gl.textures[imageSource] = GL.Texture.fromURL(imageSource, {minFilter: gl.LINEAR_MIPMAP_LINEAR, magFilter: gl.LINEAR}, null, gl);
	}
	return gl.textures[imageSource];
} //}}}
function updateAtmosphereParameters(canvas) {
	//Precalculate derived atmosphere shader parameters
	//TODO: Find a better way to structure this
	var atm = canvas.atmosphere;
	atm.inv_wavelength4 = [1.0 / Math.pow(atm.wavelength_r, 4), 1.0 / Math.pow(atm.wavelength_g, 4), 1.0 / Math.pow(atm.wavelength_b, 4)];
	atm.innerRadius = 6.371e6;
	atm.innerRadius2 = atm.innerRadius * atm.innerRadius;
	atm.outerRadius = atm.innerRadius * atm.scaleHeight;
	atm.outerRadius2 = atm.outerRadius * atm.outerRadius;
	atm.krESun = atm.kRayleigh * atm.eSun;
	atm.kmESun = atm.kMie * atm.eSun;
	atm.kr4PI = atm.kRayleigh * 4 * Math.PI;
	atm.km4PI = atm.kMie * 4 * Math.PI;
	atm.scale = 1.0 / (atm.outerRadius - atm.innerRadius);
	atm.scaleOverScaleDepth = atm.scale / atm.scaleDepth;
	atm.g2 = atm.g * atm.g;
	canvas.atmosphere = atm;
}
function clamp(value, min, max) { //{{{
	return Math.max(min, Math.min(value, max));
} //}}}
function defaultFor(name, value) { //{{{
	return typeof name !== 'undefined' ? name : value;
} //}}}
function recover(canvasid, name, value) { //{{{
	//Traverse canvas object tree for property defined by dot delimited string, returning it, or a default value if it is not found.
	var object = document.getElementById(canvasid);
	var properties = name.split('.');
	for (var i = 0; i < properties.length; i++) {
		object = object[properties[i]];
		if (VESL.Helpers.isEmptyOrUndefined(object)) { break; }
    }
	return defaultFor(object, value);
} //}}}
//}}}
//{{{ Interaction Functions
function onTap(ev, canvas) { //{{{
	ev.preventDefault();
	
	var hit = raycastXY(canvas, ev.srcEvent.layerX, ev.srcEvent.layerY, canvas.unitNode);

	//Trigger any handlers attatched to this canvas event.
	canvas.selector.trigger('onTap', [ev, canvas, hit]);
} //}}}
function onPan(ev, canvas, displaylog) { //{{{
	ev.preventDefault();
	
	if (ev.type === 'panstart') {
		canvas.lastDeltaX = 0;
		canvas.lastDeltaY = 0;
	}
	
	//Trigger any handlers attatched to this canvas event.
	canvas.selector.trigger('onPan', [ev, canvas]);
	
	//If any onPan handler sets preventDefaultOnPan to true, skips default onPan camera behavior
	if (!canvas.view.preventDefaultOnPan) {
		//If panning with two fingers or shift key, translate camera center
		if (ev.srcEvent.shiftKey || ev.pointers.length === 2) {
			if (canvas.view.panningEnabled) {
				var deltaX = (canvas.lastDeltaX - ev.deltaX) / canvas.clientWidth / canvas.view.zoom * 2 * canvas.controlSensitivity * 6.371e6;
				var deltaY = (canvas.lastDeltaY - ev.deltaY) / canvas.clientHeight / canvas.view.zoom * 2 * canvas.controlSensitivity * 6.371e6;
				
				//TODO: convert canvas.view.rotation from az/el euler to quaternion
				if (canvas.view.twod) {
					canvas.view.position[0] += Math.cos(DEG2RAD * canvas.view.rotation[0]) * deltaX - Math.sin(DEG2RAD * 0) * deltaY;
					canvas.view.position[2] += Math.sin(DEG2RAD * canvas.view.rotation[0]) * deltaX + Math.cos(DEG2RAD * 0) * deltaY;
				}
				else {
					canvas.view.position[0] += Math.cos(DEG2RAD * canvas.view.rotation[0]) * deltaX - Math.sin(DEG2RAD * canvas.view.rotation[0]) * deltaY;
					canvas.view.position[2] += Math.sin(DEG2RAD * canvas.view.rotation[0]) * deltaX + Math.cos(DEG2RAD * canvas.view.rotation[0]) * deltaY;
				}
			}
		}
		//Else, rotate around camera center
		else {
			canvas.view.rotation[0] += (canvas.lastDeltaX - ev.deltaX) / canvas.clientWidth * 2 * canvas.controlSensitivity * RAD2DEG;
			canvas.view.rotation[1] += (canvas.lastDeltaY - ev.deltaY) / canvas.clientHeight * -2 * canvas.controlSensitivity * RAD2DEG;
			
			if (canvas.view.rotation[0] > 360) { canvas.view.rotation[0] -= 360; };
			if (canvas.view.rotation[0] < -360) { canvas.view.rotation[0] += 360; };
			if (canvas.view.rotation[1] > 180) { canvas.view.rotation[1] -= 360; };
			if (canvas.view.rotation[1] < -180) { canvas.view.rotation[1] += 360; };
			
			canvas.view.rotation[0] = clamp(canvas.view.rotation[0], canvas.view.azimuthLimits[0], canvas.view.azimuthLimits[1]);
			canvas.view.rotation[1] = clamp(canvas.view.rotation[1], canvas.view.elevationLimits[0], canvas.view.elevationLimits[1]);
			
			if (displaylog) { console.log(canvas.view.rotation); }
		}
	}	
	
	canvas.view.preventDefaultOnPan = false;
	canvas.lastDeltaX = ev.deltaX;
	canvas.lastDeltaY = ev.deltaY;
} //}}}
function onPinch(ev, canvas, displaylog) { //{{{
	ev.preventDefault();
	if (ev.type === 'pinchstart') { canvas.view.lastZoom = canvas.view.zoom; }
	else { 
		canvas.view.zoom = ev.scale * canvas.view.lastZoom;
		if (displaylog) { console.log(canvas.view.zoom); }
	}
} //}}}
function onZoom(ev, canvas, displaylog) { //{{{
	ev.preventDefault();
	var delta = clamp(ev.scale || ev.wheelDelta || -ev.detail, -1, 1) * canvas.controlSensitivity * canvas.view.zoom / 2;
	modifyZoom(canvas.view.zoom + delta, canvas, displaylog, ev, 0);
} //}}}
function modifyZoom(value, canvas, displaylog, ev, duration) { //{{{
	if (VESL.Helpers.isEmptyOrUndefined(duration)) duration = 200;
	var targetZoom = clamp(value, canvas.view.zoomLimits[0], canvas.view.zoomLimits[1]);
	var currentZoom = canvas.view.zoom;
	animateValue(
		0,
		1.0,
		duration,
		'swing',
		function(value, info) {
			canvas.view.zoom = currentZoom * (1 - value) + targetZoom * value;
			if (displaylog) { console.log(canvas.view.zoom); }
			
			//Trigger any handlers attatched to this canvas event.
			canvas.selector.trigger('onZoom', [ev, canvas]);
		}
	);
} //}}}
function toggleMoviePlay(canvas) { //{{{
	canvas.animation.play = !canvas.animation.play;
} //}}}
function screenToWorldPoint(canvas, x, y) { //{{{
	var viewportX = (x - canvas.width / 2) / (canvas.width / 2);
	var viewportY = (canvas.height / 2 - y) / (canvas.height / 2);
	var origin = vec3.transformMat4(vec3.create(), [viewportX, viewportY, 0], canvas.camera.vpInverseMatrix);
	return origin;
} //}}}
function screenToModelRay(canvas, x, y, node) { //{{{
	var inverseMVPMatrix = mat4.invert(mat4.create(), mat4.multiply(mat4.create(), canvas.camera.vpMatrix, node.modelMatrix));
	var viewportX = (x - canvas.width / 2) / (canvas.width / 2);
	var viewportY = (canvas.height / 2 - y) / (canvas.height / 2);
	var origin = vec3.transformMat4(vec3.create(), [viewportX, viewportY, 0], inverseMVPMatrix);
	var far = vec3.transformMat4(vec3.create(), [viewportX, viewportY, 1.0], inverseMVPMatrix);
	var direction = vec3.normalize(vec3.create(), vec3.subtract(vec3.create(), far, origin));
	return {'origin':origin, 'direction':direction};
} //}}}
function raycast(canvas, origin, direction, node) { //{{{
	//Performs raycast on given node using ray origin and direction vectors.
	//Returns hit objects with hit position, normals, barycentric coordinates, element number, and indices of ray-triangle intersection.
	//TODO: Diagnose marker issues with orthographic views and slr-eustatic updates when switching between basins.
	if (!node.octree) { node.octree = new GL.Octree(node.mesh); }
	
	var hit = node.octree.testRay(origin, direction, 1e3, 1e10);
	if (!hit) { return; }

	hit.modelPos = vec3.copy(vec3.create(), hit.pos);
	vec3.transformMat4(hit.pos, hit.pos, node.modelMatrix);
	
	return hit;
} //}}}
function raycastXY(canvas, x, y, node, faceWinding) { //{{{
	//Performs raycast on given node using x and y screenspace coordinates.
	//Returns hit objects with hit position, normals, barycentric coordinates, element number, and indices of ray-triangle intersection.
	//TODO: Diagnose marker issues with orthographic views and slr-eustatic updates when switching between basins.
	var ray = screenToModelRay(canvas, x, y, node);
	return raycast(canvas, ray.origin, ray.direction, node);
} //}}}
function animateValue(current, target, duration, easing, stepCallback, doneCallback) { //{{{
	//Animates scalar value for length duration, calling callback each step. Specify smooth easing as a string ('swing', 'linear').
	$({'value':current}).animate({'value':target}, {
		duration: duration,
		easing: easing,
		step: stepCallback,
		done: doneCallback
	});
} //}}}
//}}}
//{{{ Drawing Functions
function updateCameraMatrix(canvas) { //{{{
    //Update view matrix and multiply with projection matrix to get the view-projection matrix.
	var vMatrix = mat4.create();
	var pMatrix = mat4.create();
	var translateMatrix = mat4.create();
	var rotationMatrix = mat4.create();
	var azimuthRotationMatrix = mat4.create();
	var elevationRotationMatrix = mat4.create();
	var aspectRatio = canvas.clientWidth / canvas.clientHeight;
	var camera = canvas.camera;
	var view = canvas.view;

	if (view.twod) { mat4.ortho(pMatrix, -aspectRatio*6.371e6/view.zoom, aspectRatio*6.371e6/view.zoom, -6.371e6/view.zoom, 6.371e6/view.zoom, camera.near, camera.far); }
	else { mat4.perspective(pMatrix, camera.fov * DEG2RAD, aspectRatio, camera.near, camera.far); }
	
	//Apply worldspace translation
	mat4.translate(vMatrix, translateMatrix, vec3.negate(vec3.create(), view.position));
	
	//Calculate rotation around camera focal point about worldspace origin
	if (view.twod) {
		mat4.rotate(azimuthRotationMatrix, azimuthRotationMatrix, -DEG2RAD * 0, [0, 1, 0]);
		mat4.rotate(elevationRotationMatrix, elevationRotationMatrix, DEG2RAD * 90, [1, 0, 0]);
		mat4.multiply(rotationMatrix, elevationRotationMatrix, azimuthRotationMatrix);
	}
	else {
		mat4.rotate(azimuthRotationMatrix, azimuthRotationMatrix, -DEG2RAD * (view.rotation[0] + 90), [0, 1, 0]);
		mat4.rotate(elevationRotationMatrix, elevationRotationMatrix, DEG2RAD * view.rotation[1], [1, 0, 0]);
		mat4.multiply(rotationMatrix, elevationRotationMatrix, azimuthRotationMatrix);
		//var quaternionWorldX = Node.prototype.eulerToQuaternion(0, 0, DEG2RAD * (view.rotation[0]));
		//var quaternionWorldY = Node.prototype.eulerToQuaternion(0, DEG2RAD * (view.rotation[1]), 0);
		//var quaternionWorldZ = Node.prototype.eulerToQuaternion(DEG2RAD * (view.rotation[2]), 0, 0);
		//var quaternionTemp = quat.multiply(quat.create(), quaternionWorldY, quaternionWorldX);
		//quat.multiply(camera.rotation, quaternionWorldZ, quaternionTemp);
		//mat4.fromQuat(rotationMatrix, camera.rotation);	
	}

	//Apply rotation transform
	mat4.multiply(vMatrix, rotationMatrix, vMatrix);
	
	//Apply screenspace translation to emulate rotation around point
	mat4.identity(translateMatrix);
	mat4.translate(translateMatrix, translateMatrix, [0.0, 0.0, -6.371e6/view.zoom]);
	mat4.multiply(vMatrix, translateMatrix, vMatrix);
	
	//Apply projection matrix to get camera matrix
	mat4.copy(camera.vMatrix, vMatrix);
	mat4.multiply(camera.vpMatrix, pMatrix, vMatrix);
	
	//Calculate inverse view matrix fields for lighting and raycasts
	mat4.invert(camera.vInverseMatrix, camera.vMatrix);
	mat4.invert(camera.vpInverseMatrix, camera.vpMatrix);
	
	vec3.transformMat4(camera.position, vec3.create(), camera.vInverseMatrix);
	vec3.sub(camera.relativePosition, camera.position, view.position);
	vec3.normalize(camera.direction, camera.relativePosition);
	
	camera.ready = true;
}//}}}
function drawSceneGraphNode(canvas, node) { //{{{
	if (!node.enabled) { return; }

	var gl = canvas.gl;
	gl.makeCurrent();
	
	var mvpMatrix = mat4.create();
	mat4.multiply(mvpMatrix, canvas.camera.vpMatrix, node.modelMatrix);
	
	var normalMatrix = mat3.create();
	var tempMatrix = mat4.create();
	mat4.invert(tempMatrix, node.modelMatrix);
	mat4.transpose(tempMatrix, tempMatrix);
	mat3.fromMat4(normalMatrix, tempMatrix);
	
	if (node.texture) { node.texture.bind(0); }
	if (node.disableDepthTest) { gl.disable(gl.DEPTH_TEST); }
	if (node.enableCullFace) { gl.enable(gl.CULL_FACE); }

	gl.cullFace(node.cullFace);
	gl.lineWidth(node.lineWidth);
	gl.blendFunc(gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA);
	
	//Setup for light that originates from camera
	var atm = canvas.atmosphere;
	var lightOrigin = vec3.create();
	
	node.shader.uniforms({
		m4MVP: 					mvpMatrix,
		m3Normal: 				normalMatrix,
		m4Model: 				node.modelMatrix,
		u_alpha: 				node.alpha,
		u_ambientColor: 		node.ambientColor,
		u_cameraPosition: 		canvas.camera.position,
		u_diffuseColor: 		node.diffuseColor,
		u_lightDirection: 		canvas.camera.direction,
		u_lightingBias: 		node.lightingBias,
		u_maskZerosColor: 		node.maskZerosColor,
		u_maskZerosEnabled: 	node.maskZerosEnabled,
		u_maskZerosTolerance: 	node.maskZerosTolerance,
		u_maskZerosZeroValue: 	node.maskZerosZeroValue,
		u_maskEnabled: 			node.maskEnabled,
		u_maskHeight: 			node.maskHeight,
		u_maskColor: 			node.maskColor,
		u_pointSize: 			node.pointSize,
		u_specularColor: 		node.specularColor,
		u_specularPower: 		node.specularPower,
		u_specularStrength: 	node.specularStrength,
		u_texture: 				0,
		v3CameraPosition: 		canvas.camera.position,
		v3Translate: 			node.translation,
		v3LightPos: 			lightOrigin,
		v3InvWavelength: 		atm.inv_wavelength4,
		fOuterRadius: 			atm.outerRadius,
		fOuterRadius2: 			atm.outerRadius2,
		fInnerRadius: 			atm.innerRadius,
		fInnerRadius2: 			atm.innerRadius2,
		fKrESun: 				atm.krESun,
		fKmESun: 				atm.kmESun,
		fKr4PI: 				atm.kr4PI,
		fKm4PI: 				atm.km4PI,
		fScale: 				atm.scale, 
		fScaleDepth: 			atm.scaleDepth,
		fScaleOverScaleDepth: 	atm.scaleOverScaleDepth, 
		v3LightPosFrag: 		lightOrigin,
		fHdrExposure: 			atm.hdr_exposure,	
		g: 						atm.g,			
		g2: 					atm.g2,
		a: 						atm.a,
		b: 						atm.b,
		c: 						atm.c,
		d: 						atm.d,		
		e: 						atm.e,
		attenuation: 			atm.attenuation
	}).draw(node.mesh, node.drawMode, 'triangles');
	
	gl.enable(gl.DEPTH_TEST);
	gl.disable(gl.CULL_FACE);
} //}}}
function canvasResize(canvas) {
	var rect = canvas.getBoundingClientRect();
	canvas.width  = rect.width;
	canvas.height = rect.height;
	canvas.gl.viewport(0, 0, canvas.width, canvas.height);
	
	if (!VESL.Helpers.isEmptyOrUndefined(canvas.overlaycanvas)) {
		rect = canvas.overlaycanvas.getBoundingClientRect();
		canvas.overlaycanvas.width  = rect.width;
		canvas.overlaycanvas.height = rect.height;
	}
}
function draw(canvas) { //{{{
	//Ensure all nodes are ready to render
	//TODO: Come up with better way to check if shaders are ready, or move outside of main draw function
	var nodes = canvas.nodes;
	if (!canvas.draw.ready) {
		if (Object.keys(nodes).length !== 0) {
			canvas.draw.ready = true;
			for (var node in nodes) {
				if (nodes[node].shader.ready === false) {
					canvas.draw.ready = false;
					break;
				}
			}
			
		}
	}
	
	//Begin rendering nodes
	if (canvas.draw.ready) {
		//Handle canvas resizing and viewport/screenspace coordinate synchronization
		canvasResize(canvas);	
		
		var gl = canvas.gl;
		gl.makeCurrent(); //litegl function to handle switching between multiple canvases
		gl.clearColor(canvas.backgroundcolor[0], canvas.backgroundcolor[1], canvas.backgroundcolor[2], canvas.backgroundcolor[3]);
		gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
		
		updateCameraMatrix(canvas);
		
		//Trigger any handlers attatched to this canvas event.
		canvas.selector.trigger('onPreRender', [canvas]);
		
		for (var handler in canvas.overlayHandlers) { canvas.overlayHandlers[handler](canvas); }
		
		var drawPassNumber = 3;
		for (var i = drawPassNumber - 1; i >= 0; --i) {
			for (var node in nodes) {
				if (nodes[node].drawOrder === i) { drawSceneGraphNode(canvas, nodes[node]); }
			}
		}
	}
	
	//Regardless of ready state, schedule next frame to check for ready state and render
	canvas.draw.handler = window.requestAnimationFrame(function(time) { draw(canvas); });
} //}}}
//}}}
