/*!\file: steadystate_core.cpp
 * \brief: core of the steadystate solution 
 */ 

#undef __FUNCT__ 
#define __FUNCT__ "steadystate_core"

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./parallel.h"
#include "../issm.h"

void steadystate_core(DataSet* results,Model* model, ParameterInputs* inputs){

	extern int my_rank;

	/*fem models: */
	FemModel* fem_dh=NULL;
	FemModel* fem_dv=NULL;
	FemModel* fem_dhu=NULL;
	FemModel* fem_ds=NULL;
	FemModel* fem_sl=NULL;
	FemModel* fem_t=NULL;
	FemModel* fem_m=NULL;

	/*output: */
	Result* result=NULL;
	DataSet* results_thermal=NULL;
	DataSet* results_diagnostic=NULL;

	/*solutions: */
	Vec u_g=NULL;
	Vec old_u_g=NULL;
	Vec t_g=NULL;
	Vec t_g_average=NULL;
	Vec old_t_g=NULL;
	Vec p_g=NULL;
	Vec m_g=NULL;
	Vec du_g=NULL;
	Vec dt_g=NULL;
	double ndu,nu;
	double normdt,normt;
	double eps_rel;

	/*flags: */
	int debug=0;
	int isstokes=0;
	int numberofnodes;
	int ndof;
	int converged;
	int step;

	/*recover fem models: */
	fem_dh=model->GetFormulation(DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	fem_dv=model->GetFormulation(DiagnosticAnalysisEnum(),VertAnalysisEnum());
	fem_ds=model->GetFormulation(DiagnosticAnalysisEnum(),StokesAnalysisEnum());
	fem_dhu=model->GetFormulation(DiagnosticAnalysisEnum(),HutterAnalysisEnum());
	fem_sl=model->GetFormulation(SlopeComputeAnalysisEnum());
	fem_t=model->GetFormulation(ThermalAnalysisEnum());
	fem_m=model->GetFormulation(MeltingAnalysisEnum());


	//first recover parameters common to all solutions
	model->FindParam(&debug,"debug");debug=1;
	model->FindParam(&numberofnodes,"numberofnodes");
	model->FindParam(&eps_rel,"eps_rel");
	model->FindParam(&isstokes,"isstokes");

	//initialize: 
	converged=0;
	step=1;

	if (isstokes)ndof=4;
	else ndof=3;

	for(;;){
	
		if(debug)_printf_("%s%i\n","   computing temperature and velocity for step: ",step);

		//first compute temperature at steady state.
		if (step>1){
			inputs->Add("velocity",u_g,ndof,numberofnodes);
		}
		results_thermal=new DataSet(ResultsEnum()); 
		thermal_core(results_thermal,model,inputs);
	
		//get t_g and m_g;
		VecFree(&t_g);results_thermal->FindResult(&t_g,"t_g");
		VecFree(&m_g);results_thermal->FindResult(&m_g,"m_g");
		delete results_thermal;

		//Add temperature to inputs.
		//compute depth averaged temperature and add to inputs
		VecDuplicatePatch(&t_g_average,t_g); 
		FieldDepthAveragex( t_g_average, fem_t->elements,fem_t->nodes, fem_t->loads, fem_t->materials,"temperature");
		inputs->Add("temperature_average",t_g_average,1,numberofnodes);
		inputs->Add("temperature",t_g,1,numberofnodes);
		VecFree(&t_g_average); //not needed anymore

		//now compute diagnostic velocity using the steady state temperature.
		results_diagnostic=new DataSet(ResultsEnum());
		diagnostic_core(results_diagnostic,model, inputs);

		//get p_g and u_g
		VecFree(&u_g);results_diagnostic->FindResult(&u_g,"u_g");
		VecFree(&p_g);results_diagnostic->FindResult(&p_g,"p_g");
		delete results_diagnostic;

		//convergence? 
		if(step>1){
			VecDuplicatePatch(&du_g,old_u_g);VecAYPX(du_g,-1.0,u_g);
			VecNorm(du_g,NORM_2,&ndu); VecNorm(old_u_g,NORM_2,&nu); VecFree(&du_g);

			VecDuplicatePatch(&dt_g,old_t_g); VecAYPX(dt_g,-1.0,t_g);
			VecNorm(dt_g,NORM_2,&normdt); VecNorm(old_t_g,NORM_2,&normt);VecFree(&dt_g);
					
			if (debug) _printf_("%-60s%g\n                                     %s%g\n                                     %s%g%s\n",
					  "      relative convergence criterion: velocity -> norm(du)/norm(u)=   ",ndu/nu*100," temperature -> norm(dt)/norm(t)=",normdt/normt*100," eps_rel:                        ",eps_rel*100," %");
		
			if ((ndu/nu<=eps_rel)  && (normdt/normt<=eps_rel)) converged=1;
			else converged=0;
		}
		else{
			converged=0;
		}

		VecFree(&old_u_g);VecDuplicatePatch(&old_u_g,u_g);
		VecFree(&old_t_g);VecDuplicatePatch(&old_t_g,t_g);

		step++;
		if (converged)break;
	}

	/*Plug results into output dataset: */
	result=new Result(results->Size()+1,0,1,"u_g",u_g);
	results->AddObject(result);
	result=new Result(results->Size()+1,0,1,"p_g",p_g);
	results->AddObject(result);
	result=new Result(results->Size()+1,0,1,"t_g",t_g);
	results->AddObject(result);
	result=new Result(results->Size()+1,0,1,"m_g",m_g);
	results->AddObject(result);

	/*Free ressource*/
	VecFree(&old_u_g);
	VecFree(&old_t_g);
	VecFree(&u_g);
	VecFree(&p_g);
	VecFree(&t_g);
	VecFree(&m_g);
}
