/*!\file: hydrology_core.cpp
 * \brief: core of the hydrology solution 
 */ 

#include "./cores.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"
#include "../solutionsequences/solutionsequences.h"

void hydrology_core(FemModel* femmodel){

	/*intermediary*/
	int  hydrology_model;
	int  solution_type;
	bool save_results;
	bool modify_loads=true;
	bool isefficientlayer;

	/*first recover parameters common to all solutions*/
	femmodel->parameters->FindParam(&save_results,SaveResultsEnum);
	femmodel->parameters->FindParam(&hydrology_model,HydrologyModelEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);	

	if(VerboseSolution()) _printf0_("   computing water heads\n");
			
	/*first compute slopes: */
	if (hydrology_model==HydrologyshreveEnum){
		surfaceslope_core(femmodel);
		bedslope_core(femmodel);
	}

	/*Using the Shreve based Model*/
	if (hydrology_model==HydrologyshreveEnum){
		if(VerboseSolution()) _printf0_("   computing water column\n");
		femmodel->SetCurrentConfiguration(HydrologyShreveAnalysisEnum);
		solutionsequence_nonlinear(femmodel,modify_loads);
		
		/*transfer water column thickness to old water column thickness: */
		InputDuplicatex(femmodel,WatercolumnEnum,WaterColumnOldEnum);
		
		if(save_results){
			if(VerboseSolution()) _printf0_("   saving results \n");
			int outputs[3] = {WatercolumnEnum,HydrologyWaterVxEnum,HydrologyWaterVyEnum};
			femmodel->RequestedOutputsx(&femmodel->results,&outputs[0],3);
			
			/*unload results*/
			if(VerboseSolution()) _printf0_("   saving temporary results\n");
			OutputResultsx(femmodel);
		}
	}

	/*Using the double continuum model*/
	else if (hydrology_model==HydrologydcEnum){
		/*intermediary: */
		int        step,hydroslices;
		int        numoutputs;
		char       **requested_outputs = NULL;
		IssmDouble time,init_time,hydrotime,yts;
		IssmDouble dt,hydrodt;

		femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
		femmodel->parameters->FindParam(&step,StepEnum);
		femmodel->parameters->FindParam(&time,TimeEnum);
		femmodel->parameters->FindParam(&hydroslices,HydrologydcStepsPerStepEnum);
		femmodel->parameters->FindParam(&yts,ConstantsYtsEnum);
		init_time = time-dt; //getting the time back to the start of the timestep
		hydrotime=init_time;
		hydrodt=dt/hydroslices; //computing hydro dt from dt and a divider
		femmodel->parameters->AddObject(new DoubleParam(HydrologydcHydrodtEnum,hydrodt));
		femmodel->parameters->FindParam(&numoutputs,HydrologyNumRequestedOutputsEnum);
		if(numoutputs) femmodel->parameters->FindParam(&requested_outputs,&numoutputs,HydrologyRequestedOutputsEnum);

		if(dt>0){
			if(hydroslices>1){
				int trans_input[1]={HydrologydcStackedNEnum};
				femmodel->InitTransientOutputx(&trans_input[0],1);
			}
			while(hydrotime<time-(yts*DBL_EPSILON)){ //loop on hydro dts
				hydrotime+=hydrodt;
				InputDuplicatex(femmodel,SedimentHeadEnum,SedimentHeadOldEnum);
				femmodel->parameters->FindParam(&isefficientlayer,HydrologydcIsefficientlayerEnum);
				if (isefficientlayer){
					InputDuplicatex(femmodel,EplHeadEnum,EplHeadOldEnum);
					InputDuplicatex(femmodel,HydrologydcEplThicknessEnum,HydrologydcEplThicknessOldEnum);
				}
				/*Proceed now to heads computations*/
				solutionsequence_hydro_nonlinear(femmodel);
				if (hydroslices>1){
					int output[1]={EffectivePressureEnum};
					femmodel->StackTransientOutputx(&output[0],hydrotime,1);
				}
			}
			if(hydroslices>1){
				int output[1]={HydrologydcStackedNEnum};
				femmodel->AverageTransientOutputx(&output[0],init_time,1);
			}		
		}
		else{
			InputDuplicatex(femmodel,SedimentHeadEnum,SedimentHeadOldEnum);
			femmodel->parameters->FindParam(&isefficientlayer,HydrologydcIsefficientlayerEnum);
			if (isefficientlayer){
				InputDuplicatex(femmodel,EplHeadEnum,EplHeadOldEnum);
				InputDuplicatex(femmodel,HydrologydcEplThicknessEnum,HydrologydcEplThicknessOldEnum);
			}
			/*Proceed now to heads computations*/
			solutionsequence_hydro_nonlinear(femmodel);
		}
		if(save_results){
			if(VerboseSolution()) _printf0_("   saving results \n");
			femmodel->RequestedOutputsx(&femmodel->results,requested_outputs,numoutputs);
			/*unload results removed 23/11 needs checking*/
			/* if(VerboseSolution()) _printf0_("   saving temporary results\n"); */
			/* OutputResultsx(femmodel); */
		}
		/*Free ressources:*/	
		if(numoutputs){
			for(int i=0;i<numoutputs;i++){
				xDelete<char>(requested_outputs[i]);
			} 
			xDelete<char*>(requested_outputs);
		}
	}
	

	else if (hydrology_model==HydrologysommersEnum){	
		femmodel->SetCurrentConfiguration(HydrologySommersAnalysisEnum);	
      InputDuplicatex(femmodel,HydrologyHeadEnum,HydrologyHeadOldEnum);	
		solutionsequence_nonlinear(femmodel,modify_loads); 
		if(VerboseSolution()) _printf0_("   updating gap height\n");
		HydrologySommersAnalysis* analysis = new HydrologySommersAnalysis();
		analysis->UpdateGapHeight(femmodel);
		delete analysis;
		
		if(save_results){
			if(VerboseSolution()) _printf0_("   saving results \n");
			int outputs[5] = {HydrologyHeadEnum,HydrologyGapHeightEnum,EffectivePressureEnum,HydrologyBasalFluxEnum,DegreeOfChannelizationEnum};
			femmodel->RequestedOutputsx(&femmodel->results,&outputs[0],5);
			
			/*unload results*/
			if(VerboseSolution()) _printf0_("   saving temporary results\n");
			OutputResultsx(femmodel);
		}
	}

	else{
		_error_("Hydrology model "<< EnumToStringx(hydrology_model) <<" not supported yet");
	}
}

