/*!\file: sealevelrise_core.cpp
 * \brief: core of the SLR solution 
 */ 

#include "./cores.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"
#include "../solutionsequences/solutionsequences.h"

/*cores:*/
void sealevelrise_core(FemModel* femmodel){ /*{{{*/


	/*Parameters, variables:*/
	bool save_results;
	bool isslr=0;
	int solution_type;
		
	/*Retrieve parameters:*/
	femmodel->parameters->FindParam(&isslr,TransientIsslrEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
	femmodel->parameters->FindParam(&save_results,SaveResultsEnum);
	
	/*in case we are running SealevelriseSolutionEnum, then bypass transient settings:*/
	if(solution_type==SealevelriseSolutionEnum)isslr=1;

	/*Should we be here?:*/
	if(!isslr)return;

	/*Verbose: */
	if(VerboseSolution()) _printf0_("   computing sea level rise\n");

	/*set SLR configuration: */
	femmodel->SetCurrentConfiguration(SealevelriseAnalysisEnum);

	/*Run geodetic:*/
	geodetic_core(femmodel);

	/*Run steric core for sure:*/
	steric_core(femmodel);
	
	/*Save results: */
	if(save_results){
		int        numoutputs        = 0;
		char     **requested_outputs = NULL;

		if(VerboseSolution()) _printf0_("   saving results\n");
		femmodel->parameters->FindParam(&requested_outputs,&numoutputs,SealevelriseRequestedOutputsEnum);
		femmodel->RequestedOutputsx(&femmodel->results,requested_outputs,numoutputs);
		if(numoutputs){for(int i=0;i<numoutputs;i++){xDelete<char>(requested_outputs[i]);} xDelete<char*>(requested_outputs);}
	}

	/*requested dependents: */
	if(solution_type==SealevelriseSolutionEnum)femmodel->RequestedDependentsx();
}
/*}}}*/
void geodetic_core(FemModel* femmodel){ /*{{{*/


	/*variables:*/
	Vector<IssmDouble> *Sg    = NULL;
	Vector<IssmDouble> *Sg_rate    = NULL;
	Vector<IssmDouble> *Sg_eustatic  = NULL; 
	Vector<IssmDouble> *U_radial  = NULL; 
	Vector<IssmDouble> *U_radial_rate  = NULL; 
	Vector<IssmDouble> *U_north   = NULL; 
	Vector<IssmDouble> *U_east    = NULL; 
	Vector<IssmDouble>* cumdeltathickness=NULL; 

	/*parameters:*/
	bool iscoupler;
	int  configuration_type;
	int  modelid,earthid;
	bool istransientmasstransport;
	int  frequency,count;
	int  horiz;
	int  geodetic=0;
	IssmDouble          dt;

	/*Should we even be here?:*/
	femmodel->parameters->FindParam(&geodetic,SealevelriseGeodeticEnum); if(!geodetic)return;
	
	/*Verbose: */
	if(VerboseSolution()) _printf0_("	  computing geodetic sea level rise\n");

	/*retrieve more parameters:*/
	femmodel->parameters->FindParam(&iscoupler,TransientIscouplerEnum);
	femmodel->parameters->FindParam(&frequency,SealevelriseGeodeticRunFrequencyEnum);
	femmodel->parameters->FindParam(&count,SealevelriseRunCountEnum);
	femmodel->parameters->FindParam(&configuration_type,ConfigurationTypeEnum);
	femmodel->parameters->FindParam(&horiz,SealevelriseHorizEnum);
	femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);

	if(iscoupler){
		femmodel->parameters->FindParam(&modelid,ModelIdEnum);
		femmodel->parameters->FindParam(&earthid,EarthIdEnum);
		femmodel->parameters->FindParam(&istransientmasstransport,TransientIsmasstransportEnum);
	}
	else{
		/* we are here, we are not running in a coupler, so we will indeed compute SLR,
		 * so make sure we are identified as being the Earth.:*/
		modelid=1; earthid=1; 
		/* in addition, if we are running solution_type SealevelriseSolutionEnum, make sure we 
		 * run, irresepective of the time settings:*/
		count=frequency;
	}

	/*If we are running in coupled mode, the Earth model needs to run its own mass transport (if 
	 * not already done by the mass trasnport module. For ice caps, they rely on the transient mass 
	 * transport module exclusively:*/
	if(iscoupler) if(modelid==earthid) if(!istransientmasstransport) EarthMassTransport(femmodel);

	/*increment counter, or call solution core if count==frequency:*/
	if (count<frequency){
		count++; femmodel->parameters->SetParam(count,SealevelriseRunCountEnum); 
		return;
	}

	/*call sea-level rise sub cores:*/
	if(iscoupler){
		/*transfer cumulated deltathickness forcing from ice caps to earth model: */
		TransferForcing(femmodel,SealevelriseCumDeltathicknessEnum);

		/*we have accumulated thicknesses, dump them in deltathcikness: */
		if(modelid==earthid)InputDuplicatex(femmodel,SealevelriseCumDeltathicknessEnum,SealevelriseDeltathicknessEnum);
	}

	/*run cores:*/
	if(modelid==earthid){

		/*call eustatic core  (generalized eustatic - Farrel and Clark, Eq 4, 1st, 3rd and 4rd terms on the RHS) */
		Sg_eustatic=sealevelrise_core_eustatic(femmodel); 

		/*call non-eustatic core (ocean loading tems  - 2nd and 5th terms on the RHS of Farrel and Clark) */
		Sg=sealevelrise_core_noneustatic(femmodel,Sg_eustatic); 
		

		/*compute other elastic geodetic signatures, such as components of 3-D crustal motion: */
		sealevelrise_core_elastic(&U_radial,&U_north,&U_east,femmodel,Sg);
		
		/*transform these values into rates (as we only run this once each frequency turn:*/
		Sg_rate=Sg->Duplicate(); Sg->Copy(Sg_rate); Sg_rate->Scale(1/(dt*frequency));
		U_radial_rate=U_radial->Duplicate(); U_radial->Copy(U_radial_rate); U_radial_rate->Scale(1/(dt*frequency));
		

		/*get some results into elements:*/
		InputUpdateFromVectorx(femmodel,Sg_rate,SealevelRateEnum,VertexSIdEnum); 
		InputUpdateFromVectorx(femmodel,U_radial_rate,SealevelUmotionRateEnum,VertexSIdEnum); 
		if (horiz){
			InputUpdateFromVectorx(femmodel,U_north,SealevelNmotionEnum,VertexSIdEnum);	// north motion 
			InputUpdateFromVectorx(femmodel,U_east,SealevelEmotionEnum,VertexSIdEnum);		// east motion 
		}
	}


	if(iscoupler){
		/*transfer sea level back to ice caps:*/
		TransferSealevel(femmodel,SealevelRateEnum);
		TransferSealevel(femmodel,SealevelUmotionRateEnum);

		//reset cumdeltathickness  to 0: 
		InputUpdateFromConstantx(femmodel,0.0,SealevelriseCumDeltathicknessEnum);
	}

	/*reset counter to 1:*/
	femmodel->parameters->SetParam(1,SealevelriseRunCountEnum); //reset counter.

	/*Free ressources:*/	
	delete Sg_eustatic;
	delete Sg;
	delete Sg_rate;
	delete U_radial;
	delete U_radial_rate;
	if(horiz){
		delete U_north;
		delete U_east;
	}
} 
/*}}}*/
void steric_core(FemModel* femmodel){ /*{{{*/

	/*variables:*/
	Vector<IssmDouble> *bedrock  = NULL; 
	Vector<IssmDouble> *Sg_absolute  = NULL; 
	Vector<IssmDouble> *steric_rate_g  = NULL; 
	Vector<IssmDouble> *rsl_g    = NULL;
	Vector<IssmDouble> *bedrock_rate_g  = NULL; 
		
	/*parameters: */
	bool isslr=0;
	int  solution_type;
	IssmDouble          dt;
	int  geodetic=0;

	/*Retrieve parameters:*/
	femmodel->parameters->FindParam(&isslr,TransientIsslrEnum);
	femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
	femmodel->parameters->FindParam(&geodetic,SealevelriseGeodeticEnum); 

	/*in case we are running SealevelriseSolutionEnum, then bypass transient settings:*/
	if(solution_type==SealevelriseSolutionEnum)isslr=1;
	
	/*Should we be here?:*/
	if(!isslr)return;

	/*Verbose: */
	if(VerboseSolution()) _printf0_("	  computing steric sea level rise\n");

	/*Retrieve absolute sea level, RSL rate, bedrock uplift rate and steric rate, as vectors:*/
	GetVectorFromInputsx(&bedrock,femmodel,BedEnum,VertexSIdEnum);
	GetVectorFromInputsx(&Sg_absolute,femmodel,SealevelEnum,VertexSIdEnum);
	GetVectorFromInputsx(&steric_rate_g,femmodel,SealevelriseStericRateEnum,VertexSIdEnum);
	if(geodetic){
		GetVectorFromInputsx(&rsl_g,femmodel,SealevelRateEnum,VertexSIdEnum);
		GetVectorFromInputsx(&bedrock_rate_g,femmodel,SealevelUmotionRateEnum,VertexSIdEnum);
	}

	/*compute: absolute sea level change = initial absolute sea level + relative sea level change rate * dt + vertical motion rate * dt + steric_rate * dt*/
	if(rsl_g) Sg_absolute->AXPY(rsl_g,dt);
	if(bedrock_rate_g)Sg_absolute->AXPY(bedrock_rate_g,dt);
	Sg_absolute->AXPY(steric_rate_g,dt);

	/*compute new bedrock position: */
	if(bedrock_rate_g)bedrock->AXPY(bedrock_rate_g,dt);

	/*update element inputs:*/
	InputUpdateFromVectorx(femmodel,bedrock,BedEnum,VertexSIdEnum);	
	InputUpdateFromVectorx(femmodel,bedrock,SealevelUmotionEnum,VertexSIdEnum);	
	InputUpdateFromVectorx(femmodel,Sg_absolute,SealevelEnum,VertexSIdEnum);	

	/*Free ressources:*/	
	delete bedrock;
	delete Sg_absolute;
	delete steric_rate_g;
	if(rsl_g)delete rsl_g;
	if(bedrock_rate_g)delete bedrock_rate_g;

}
/*}}}*/
Vector<IssmDouble>* sealevelrise_core_eustatic(FemModel* femmodel){ /*{{{*/
 
	/*Eustatic core of the SLR solution (terms that are constant with respect to sea-level)*/

	Vector<IssmDouble> *Sgi    = NULL;
	IssmDouble          Sgi_oceanaverage   = 0;

	/*parameters: */
	int  configuration_type;
	int  gsize;
	bool spherical=true;
	IssmDouble *latitude  = NULL;
	IssmDouble *longitude = NULL;
	IssmDouble *radius    = NULL;
	int         loop;

	/*outputs:*/
	IssmDouble eustatic;

	/*recover parameters:*/
	femmodel->parameters->FindParam(&configuration_type,ConfigurationTypeEnum);
	femmodel->parameters->FindParam(&loop,SealevelriseLoopIncrementEnum);

	/*first, recover lat,long and radius vectors from vertices: */
	VertexCoordinatesx(&latitude,&longitude,&radius,femmodel->vertices,spherical); 

	/*Figure out size of g-set deflection vector and allocate solution vector: */
	gsize = femmodel->nodes->NumberOfDofs(configuration_type,GsetEnum);
	
	/*Initialize:*/
	Sgi = new Vector<IssmDouble>(gsize);

	/*call the eustatic main module: */
	femmodel->SealevelriseEustatic(Sgi,&eustatic, latitude, longitude, radius,loop); //this computes 

	/*we need to average Sgi over the ocean: RHS term  4 in Eq.4 of Farrel and clarke. Only the elements can do that: */
	Sgi_oceanaverage=femmodel->SealevelriseOceanAverage(Sgi);

	/*Sg is the sum of the pure eustatic component (term 3) and the contribution from the perturbation to the graviation potential due to the 
	 * presence of ice (terms 1 and 4 in Eq.4 of Farrel and Clarke):*/
	Sgi->Shift(-eustatic-Sgi_oceanaverage);

	/*save eustatic value for results: */
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,SealevelEustaticEnum,-eustatic));

	/*clean up and return:*/
	xDelete<IssmDouble>(latitude);
	xDelete<IssmDouble>(longitude);
	xDelete<IssmDouble>(radius);
	return Sgi;
}/*}}}*/
Vector<IssmDouble>* sealevelrise_core_noneustatic(FemModel* femmodel,Vector<IssmDouble>* Sg_eustatic){ /*{{{*/

	/*sealevelrise_core_noneustatic.cpp //this computes the contributions from Eq.4 of Farrel and Clarke, rhs terms 2 and 5.
	  non eustatic core of the SLR solution */


	Vector<IssmDouble> *Sg    = NULL;
	Vector<IssmDouble> *Sg_old    = NULL;

	Vector<IssmDouble> *Sgo    = NULL; //ocean convolution of the perturbation to gravity potential.
	Vector<IssmDouble> *Sgo_rot= NULL; // rotational feedback 
	IssmDouble          Sgo_oceanaverage = 0;  //average of Sgo over the ocean.

	/*parameters: */
	int count;
	bool save_results;
	int  gsize;
	int  configuration_type;
	bool spherical=true;
	bool converged=true;
	bool rotation=true;
	bool verboseconvolution=true;
	int max_nonlinear_iterations;
	IssmDouble           eps_rel;
	IssmDouble           eps_abs;
	IssmDouble          *latitude    = NULL;
	IssmDouble          *longitude    = NULL;
	IssmDouble          *radius    = NULL;
	IssmDouble           eustatic;
	int	                 loop;

	/*Recover some parameters: */
	femmodel->parameters->FindParam(&max_nonlinear_iterations,SealevelriseMaxiterEnum);
	femmodel->parameters->FindParam(&eps_rel,SealevelriseReltolEnum);
	femmodel->parameters->FindParam(&eps_abs,SealevelriseAbstolEnum);
	femmodel->parameters->FindParam(&configuration_type,ConfigurationTypeEnum);
	femmodel->parameters->FindParam(&loop,SealevelriseLoopIncrementEnum);
	
	/*computational flag: */
	femmodel->parameters->FindParam(&rotation,SealevelriseRotationEnum);

	/*first, recover lat,long and radius vectors from vertices: */
	VertexCoordinatesx(&latitude,&longitude,&radius,femmodel->vertices,spherical); 

	/*Figure out size of g-set deflection vector and allocate solution vector: */
	gsize = femmodel->nodes->NumberOfDofs(configuration_type,GsetEnum);
	
	/*Initialize:*/
	Sg = new Vector<IssmDouble>(gsize);
	Sg->Assemble();
	Sg_eustatic->Copy(Sg);  //first initialize Sg with the eustatic component computed in sealevelrise_core_eustatic.

	Sg_old = new Vector<IssmDouble>(gsize);
	Sg_old->Assemble();

	count=1;
	converged=false;
	
	/*Start loop: */
	for(;;){

		//save pointer to old sea level rise
		delete Sg_old; Sg_old=Sg; 

		/*Initialize solution vector: */
		Sg  = new Vector<IssmDouble>(gsize); Sg->Assemble();
		Sgo = new Vector<IssmDouble>(gsize); Sgo->Assemble();

		/*call the non eustatic module: */
		femmodel->SealevelriseNonEustatic(Sgo, Sg_old, latitude, longitude, radius,verboseconvolution,loop);
	
		/*assemble solution vector: */
		Sgo->Assemble(); 

		if(rotation){
			/*call rotational feedback  module: */
			Sgo_rot = new Vector<IssmDouble>(gsize); Sgo_rot->Assemble();
			femmodel->SealevelriseRotationalFeedback(Sgo_rot,Sg_old,latitude,longitude,radius); 
			Sgo_rot->Assemble(); 

			Sgo->AXPY(Sgo_rot,1); 
		}
		
		/*we need to average Sgo over the ocean: RHS term  5 in Eq.4 of Farrel and clarke. Only the elements can do that: */
		Sgo_oceanaverage=femmodel->SealevelriseOceanAverage(Sgo);
	
		/*Sg is the sum of the eustatic term, and the ocean terms: */
		Sg_eustatic->Copy(Sg); Sg->AXPY(Sgo,1); 
		Sg->Shift(-Sgo_oceanaverage);

		/*convergence criterion:*/
		slrconvergence(&converged,Sg,Sg_old,eps_rel,eps_abs);

		/*free ressources: */
		delete Sgo;

		/*Increase count: */
		count++;
		if(converged==true){
			break;
		}
		if(count>=max_nonlinear_iterations){
			_printf0_("   maximum number of nonlinear iterations (" << max_nonlinear_iterations << ") exceeded\n"); 
			converged=true;
			break;
		}	
		
		/*some minor verbosing adjustment:*/
		if(count>1)verboseconvolution=false;
		
	}
	if(VerboseConvergence()) _printf0_("\n   total number of iterations: " << count-1 << "\n");

	xDelete<IssmDouble>(latitude);
	xDelete<IssmDouble>(longitude);
	xDelete<IssmDouble>(radius);
	delete Sg_old;

	return Sg;
} /*}}}*/
void sealevelrise_core_elastic(Vector<IssmDouble>** pU_radial, Vector<IssmDouble>** pU_north,Vector<IssmDouble>** pU_east,FemModel* femmodel,Vector<IssmDouble>* Sg){ /*{{{*/

	Vector<IssmDouble> *U_radial  = NULL; 
	Vector<IssmDouble> *U_north   = NULL; 
	Vector<IssmDouble> *U_east    = NULL; 
		
	/*parameters: */
	int configuration_type;
	int  gsize;
	bool spherical=true;

	IssmDouble          *latitude   = NULL;
	IssmDouble          *longitude  = NULL;
	IssmDouble          *radius     = NULL;
	IssmDouble          *xx     = NULL;
	IssmDouble          *yy     = NULL;
	IssmDouble          *zz     = NULL;
	int  loop;
	int  horiz;

	/*retrieve some parameters:*/
	femmodel->parameters->FindParam(&configuration_type,ConfigurationTypeEnum);
	femmodel->parameters->FindParam(&loop,SealevelriseLoopIncrementEnum);
	femmodel->parameters->FindParam(&horiz,SealevelriseHorizEnum);

	/*find size of vectors:*/
	gsize      = femmodel->nodes->NumberOfDofs(configuration_type,GsetEnum);

	/*intialize vectors:*/
	U_radial = new Vector<IssmDouble>(gsize);
	if (horiz){
		U_north = new Vector<IssmDouble>(gsize);
		U_east = new Vector<IssmDouble>(gsize);
	}

	/*retrieve geometric information: */
	VertexCoordinatesx(&latitude,&longitude,&radius,femmodel->vertices,spherical); 
	VertexCoordinatesx(&xx,&yy,&zz,femmodel->vertices); 
	
	/*call the elastic main modlule:*/ 
	femmodel->SealevelriseGeodetic(U_radial,U_north,U_east,Sg,latitude,longitude,radius,xx,yy,zz,loop,horiz);

	/*Assign output pointers:*/
	*pU_radial=U_radial;
	if(horiz){
		*pU_east=U_east;
		*pU_north=U_north;
	}

	/*Free ressources: */
	delete longitude; 
	delete latitude; 
	delete radius; 
	delete xx; 
	delete yy; 
	delete zz; 
}
/*}}}*/

/*support routines:*/
void TransferForcing(FemModel* femmodel,int forcingenum){ /*{{{*/

	/*forcing being transferred from models to earth: */
	IssmDouble** forcings=NULL;
	IssmDouble*  forcing=NULL; 
	Vector<IssmDouble>* forcingglobal=NULL; 
	int*         nvs=NULL;
	
	/*transition vectors:*/
	IssmDouble** transitions=NULL;
	int          ntransitions; 
	int*         transitions_m=NULL;
	int*         transitions_n=NULL;
	int          nv;
	
	/*communicators:*/
	ISSM_MPI_Comm tocomm;
	ISSM_MPI_Comm* fromcomms=NULL;
	ISSM_MPI_Status status;
	int         my_rank;
	int         modelid,earthid;
	int         nummodels;

	/*Recover some parameters: */
	femmodel->parameters->FindParam(&modelid,ModelIdEnum);
	femmodel->parameters->FindParam(&earthid,EarthIdEnum);
	femmodel->parameters->FindParam(&nummodels,NumModelsEnum);
	my_rank=IssmComm::GetRank();
	
	/*retrieve the inter communicators that will be used to send data from each ice cap to the earth: */
	if(modelid==earthid){
		GenericParam<ISSM_MPI_Comm*>* parcoms = dynamic_cast<GenericParam<ISSM_MPI_Comm*>*>(femmodel->parameters->FindParamObject(IcecapToEarthCommEnum));
		if(!parcoms)_error_("TransferForcing error message: could not find IcecapToEarthComm communicator");
		fromcomms=parcoms->GetParameterValue();
	}
	else {
		GenericParam<ISSM_MPI_Comm>* parcom = dynamic_cast<GenericParam<ISSM_MPI_Comm>*>(femmodel->parameters->FindParamObject(IcecapToEarthCommEnum));
		if(!parcom)_error_("TransferForcing error message: could not find IcecapToEarthComm communicator");
		tocomm=parcom->GetParameterValue();
	}

	/*For each icecap, retrieve the forcing vector that will be sent to the earth model: */
	if(modelid!=earthid){
		nv=femmodel->vertices->NumberOfVertices();
		GetVectorFromInputsx(&forcing,femmodel,forcingenum,VertexSIdEnum);
	}

	/*Send the forcing to the earth model:{{{*/
	if(my_rank==0){
		if(modelid==earthid){
			forcings=xNew<IssmDouble*>(nummodels-1);
			nvs=xNew<int>(nummodels-1);
			for(int i=0;i<earthid;i++){
				ISSM_MPI_Recv(nvs+i, 1, ISSM_MPI_INT, 0,i, fromcomms[i], &status);
				forcings[i]=xNew<IssmDouble>(nvs[i]);
				ISSM_MPI_Recv(forcings[i], nvs[i], ISSM_MPI_DOUBLE, 0,i, fromcomms[i], &status);
			}
			
		}
		else{
			ISSM_MPI_Send(&nv, 1, ISSM_MPI_INT, 0, modelid, tocomm);
			ISSM_MPI_Send(forcing, nv, ISSM_MPI_DOUBLE, 0, modelid, tocomm);
		}
	}
	/*}}}*/

	/*On the earth model, consolidate all the forcings into one, and update the elements dataset accordingly: {{{*/
	if(modelid==earthid){
		
		/*Out of all the delta thicknesses, build one delta thickness vector made of all the ice cap contributions. 
		 *First, build the global delta thickness vector in the earth model: */
		nv=femmodel->vertices->NumberOfVertices();
		GetVectorFromInputsx(&forcingglobal,femmodel,forcingenum,VertexSIdEnum);

		/*Retrieve transition vectors, used to plug from each ice cap into the global forcing:*/
		femmodel->parameters->FindParam(&transitions,&ntransitions,&transitions_m,&transitions_n,SealevelriseTransitionsEnum);

		if(ntransitions!=earthid)_error_("TransferForcing error message: number of transition vectors is not equal to the number of icecaps!");

		/*Go through all the delta thicknesses coming from each ice cap: */
		if(my_rank==0){
			for(int i=0;i<earthid;i++){

				IssmDouble* forcingfromcap= forcings[i]; //careful, this only exists on rank 0 of the earth model!
				IssmDouble* transition=transitions[i];
				int         M=transitions_m[i];

				/*build index to plug values: */
				int*        index=xNew<int>(M); for(int i=0;i<M;i++)index[i]=reCast<int>(transition[i])-1; //matlab indexing!


				/*We are going to plug this vector into the earth model, at the right vertices corresponding to this particular 
				 * ice cap: */
				forcingglobal->SetValues(M,index,forcingfromcap,ADD_VAL);
				xDelete<int>(index);
			}
		}

		/*Assemble vector:*/
		forcingglobal->Assemble();
		
		/*Plug into elements:*/
		InputUpdateFromVectorx(femmodel,forcingglobal,forcingenum,VertexSIdEnum);
	} 
	/*}}}*/

	/*Free ressources:{{{*/
	if(forcings){
		for(int i=0;i<nummodels-1;i++){
			IssmDouble* temp=forcings[i]; xDelete<IssmDouble>(temp);
		}
		xDelete<IssmDouble*>(forcings);
	}
	if(forcing)xDelete<IssmDouble>(forcing);
	if(forcingglobal)delete forcingglobal;
	if(transitions){
		for(int i=0;i<earthid;i++){
			IssmDouble* temp=transitions[i];
			xDelete<IssmDouble>(temp);
		}
		xDelete<IssmDouble*>(transitions);
		xDelete<int>(transitions_m);
		xDelete<int>(transitions_n);
	}
	if(nvs)xDelete<int>(nvs);
	/*}}}*/

} /*}}}*/
void TransferSealevel(FemModel* femmodel,int forcingenum){ /*{{{*/

	/*forcing being transferred from earth to ice caps: */
	IssmDouble*  forcing=NULL; 
	IssmDouble*  forcingglobal=NULL; 
	
	/*transition vectors:*/
	IssmDouble** transitions=NULL;
	int          ntransitions; 
	int*         transitions_m=NULL;
	int*         transitions_n=NULL;
	int          nv;
	
	/*communicators:*/
	ISSM_MPI_Comm fromcomm;
	ISSM_MPI_Comm* tocomms=NULL;
	ISSM_MPI_Status status;
	int         my_rank;
	int         modelid,earthid;
	int         nummodels;
	int         numcoms;

	/*Recover some parameters: */
	femmodel->parameters->FindParam(&modelid,ModelIdEnum);
	femmodel->parameters->FindParam(&earthid,EarthIdEnum);
	femmodel->parameters->FindParam(&nummodels,NumModelsEnum);
	my_rank=IssmComm::GetRank();
	
	/*retrieve the inter communicators that will be used to send data from earth to ice caps:*/
	if(modelid==earthid){
		GenericParam<ISSM_MPI_Comm*>* parcoms = dynamic_cast<GenericParam<ISSM_MPI_Comm*>*>(femmodel->parameters->FindParamObject(IcecapToEarthCommEnum));
		if(!parcoms)_error_("TransferSealevel error message: could not find IcecapToEarthComm communicator");
		tocomms=parcoms->GetParameterValue();
		//femmodel->parameters->FindParam((int**)(&tocomms),&numcoms,IcecapToEarthCommEnum);
	}
	else{
		GenericParam<ISSM_MPI_Comm>* parcom = dynamic_cast<GenericParam<ISSM_MPI_Comm>*>(femmodel->parameters->FindParamObject(IcecapToEarthCommEnum));
		if(!parcom)_error_("TransferSealevel error message: could not find IcecapToEarthComm communicator");
		fromcomm=parcom->GetParameterValue();
		//femmodel->parameters->FindParam((int*)(&fromcomm), IcecapToEarthCommEnum);
	}


	/*Retrieve sea-level on earth model: */
	if(modelid==earthid){
		nv=femmodel->vertices->NumberOfVertices();
		GetVectorFromInputsx(&forcingglobal,femmodel,forcingenum,VertexSIdEnum);
	}

	/*Send the forcing to the ice caps:{{{*/
	if(my_rank==0){
		
		if(modelid==earthid){
			
			/*Retrieve transition vectors, used to figure out global forcing contribution to each ice cap's own elements: */
			femmodel->parameters->FindParam(&transitions,&ntransitions,&transitions_m,&transitions_n,SealevelriseTransitionsEnum);
			
			if(ntransitions!=earthid)_error_("TransferSeaLevel error message: number of transition vectors is not equal to the number of icecaps!");

			for(int i=0;i<earthid;i++){
				nv=transitions_m[i];
				forcing=xNew<IssmDouble>(nv);
				IssmDouble* transition=transitions[i];
				for(int j=0;j<nv;j++){
					forcing[j]=forcingglobal[reCast<int>(transition[j])-1];
				}
				ISSM_MPI_Send(&nv, 1, ISSM_MPI_INT, 0, i, tocomms[i]);
				ISSM_MPI_Send(forcing, nv, ISSM_MPI_DOUBLE, 0, i, tocomms[i]);
			}
		}
		else{
			ISSM_MPI_Recv(&nv, 1, ISSM_MPI_INT, 0, modelid, fromcomm, &status);
			forcing=xNew<IssmDouble>(nv);
			ISSM_MPI_Recv(forcing, nv, ISSM_MPI_DOUBLE, 0, modelid, fromcomm, &status);
		}
	}
	/*}}}*/

	/*On each ice cap, spread the forcing across cpus, and update the elements dataset accordingly: {{{*/
	if(modelid!=earthid){

		ISSM_MPI_Bcast(&nv,1,ISSM_MPI_INT,0,IssmComm::GetComm());
		if(my_rank!=0)forcing=xNew<IssmDouble>(nv);
		ISSM_MPI_Bcast(forcing,nv,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());

		/*Plug into elements:*/
		InputUpdateFromVectorx(femmodel,forcing,forcingenum,VertexSIdEnum);
	} 
	/*}}}*/

	/*Free ressources:{{{*/
	if(forcingglobal)xDelete<IssmDouble>(forcingglobal);
	if(forcing)xDelete<IssmDouble>(forcing);
	if(transitions){
		for(int i=0;i<ntransitions;i++){
			IssmDouble* temp=transitions[i];
			xDelete<IssmDouble>(temp);
		}
		xDelete<IssmDouble*>(transitions);
		xDelete<int>(transitions_m);
		xDelete<int>(transitions_n);
	}
	/*}}}*/

} /*}}}*/
void EarthMassTransport(FemModel* femmodel){ /*{{{*/

	IssmDouble time,dt;
	Vector<IssmDouble> *oldthickness    = NULL;
	Vector<IssmDouble> *newthickness    = NULL;
	Vector<IssmDouble> *deltathickness    = NULL;
	Vector<IssmDouble> *cumdeltathickness    = NULL;
	int nv;
	
	if(VerboseSolution()) _printf0_("      computing earth mass transport\n");

	/*This mass transport module for the Earth is because we might have thickness variations as spcs 
	 * specified in the md.slr class, outside of what we will get from the icecaps. That's why we get t
	 * the thickness variations from SealevelriseSpcthicknessEnum.*/

	/*No mass transport module was called, so we are just going to retrieve the geometry thickness 
	 * at this time step, at prior time step, and plug the difference as deltathickness: */
	femmodel->parameters->FindParam(&time,TimeEnum);
	femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	nv=femmodel->vertices->NumberOfVertices();

	GetVectorFromInputsx(&newthickness,femmodel,SealevelriseSpcthicknessEnum,VertexSIdEnum);
	GetVectorFromInputsx(&oldthickness,femmodel,SealevelriseSpcthicknessEnum,VertexSIdEnum,time-dt);

	/*compute deltathickness: */
	deltathickness = new Vector<IssmDouble>(nv); 
	newthickness->Copy(deltathickness); deltathickness->AXPY(oldthickness,-1); 

	/*plug into elements:*/
	InputUpdateFromVectorx(femmodel,deltathickness,SealevelriseDeltathicknessEnum,VertexSIdEnum);

	/*add to cumulated delta thickness: */
	GetVectorFromInputsx(&cumdeltathickness,femmodel,SealevelriseCumDeltathicknessEnum,VertexSIdEnum);
	cumdeltathickness->AXPY(deltathickness,1); 
	InputUpdateFromVectorx(femmodel,cumdeltathickness,SealevelriseCumDeltathicknessEnum,VertexSIdEnum);

	/*free ressources:*/
	delete oldthickness;
	delete newthickness;
	delete deltathickness;
	delete cumdeltathickness;

} /*}}}*/
void slrconvergence(bool* pconverged, Vector<IssmDouble>* Sg,Vector<IssmDouble>* Sg_old,IssmDouble eps_rel,IssmDouble eps_abs){ /*{{{*/
	
	bool converged=true;
	IssmDouble ndS,nS; 
	Vector<IssmDouble> *dSg    = NULL;

	//compute norm(du) and norm(u) if requested
	dSg=Sg_old->Duplicate(); Sg_old->Copy(dSg); dSg->AYPX(Sg,-1.0);
	ndS=dSg->Norm(NORM_TWO); 
	
	if (xIsNan<IssmDouble>(ndS)) _error_("convergence criterion is NaN!");
	
	if(!xIsNan<IssmDouble>(eps_rel)){
		nS=Sg_old->Norm(NORM_TWO);
		if (xIsNan<IssmDouble>(nS)) _error_("convergence criterion is NaN!");
	}


	//clean up
	delete dSg;

	//print
	if(!xIsNan<IssmDouble>(eps_rel)){
		if((ndS/nS)<eps_rel){
			if(VerboseConvergence()) _printf0_(setw(50) << left << "      convergence criterion: norm(dS)/norm(S)" << ndS/nS*100 << " < " << eps_rel*100 << " %\n");
		}
		else{ 
			if(VerboseConvergence()) _printf0_(setw(50) << left << "      convergence criterion: norm(dS)/norm(S)" << ndS/nS*100 << " > " << eps_rel*100 << " %\n");
			converged=false;
		}
	}
	if(!xIsNan<IssmDouble>(eps_abs)){
		if(ndS<eps_abs){
			if(VerboseConvergence()) _printf0_(setw(50) << left << "      convergence criterion: norm(dS)" << ndS << " < " << eps_abs << " \n");
		}
		else{ 
			if(VerboseConvergence()) _printf0_(setw(50) << left << "      convergence criterion: norm(dS)" << ndS << " > " << eps_abs << " \n");
			converged=false;
		}
	}

	/*assign output*/
	*pconverged=converged;

} /*}}}*/
