/*!\file AdaptiveMeshrefinement.cpp
 * \brief: implementation of the adaptive mesh refinement tool based on NeoPZ library: github.com/labmec/neopz
 */

#ifdef HAVE_CONFIG_H
    #include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./AdaptiveMeshRefinement.h"

using namespace pzgeom;

/*Constructor, copy, clean up and destructor*/
AdaptiveMeshRefinement::AdaptiveMeshRefinement(){/*{{{*/
	this->Initialize();
}
/*}}}*/
AdaptiveMeshRefinement::AdaptiveMeshRefinement(const AdaptiveMeshRefinement &cp){/*{{{*/
	this->Initialize(); 
	this->operator =(cp);
}
/*}}}*/
AdaptiveMeshRefinement & AdaptiveMeshRefinement::operator =(const AdaptiveMeshRefinement &cp){/*{{{*/

	/*Clean all attributes*/
	this->CleanUp();
	/*Copy all data*/
	this->fathermesh						= new TPZGeoMesh(*cp.fathermesh);
	this->previousmesh					= new TPZGeoMesh(*cp.previousmesh);
	this->level_max						= cp.level_max;
	this->radius_level_max				= cp.radius_level_max;
	this->gradation						= cp.gradation;
	this->lag								= cp.lag;
   this->groundingline_distance		= cp.groundingline_distance;
	this->icefront_distance				= cp.icefront_distance;
	this->thicknesserror_threshold	= cp.thicknesserror_threshold;
	this->deviatoricerror_threshold	= cp.deviatoricerror_threshold;
	this->sid2index.clear();
	this->sid2index.resize(cp.sid2index.size());
	for(int i=0;i<cp.sid2index.size();i++) this->sid2index[i]=cp.sid2index[i];
	this->index2sid.clear();
	this->index2sid.resize(cp.index2sid.size());
	for(int i=0;i<cp.index2sid.size();i++) this->index2sid[i]=cp.index2sid[i];
	this->specialelementsindex.clear();
	this->specialelementsindex.resize(cp.specialelementsindex.size());
	for(int i=0;i<cp.specialelementsindex.size();i++) this->specialelementsindex[i]=cp.specialelementsindex[i];

	return *this;
}
/*}}}*/
AdaptiveMeshRefinement::~AdaptiveMeshRefinement(){/*{{{*/
	this->CleanUp();
	gRefDBase.clear();
}
/*}}}*/
void AdaptiveMeshRefinement::CleanUp(){/*{{{*/

	/*Verify and delete all data*/
	if(this->fathermesh)    delete this->fathermesh;
	if(this->previousmesh)  delete this->previousmesh;
	this->level_max						= -1;
	this->radius_level_max				= -1;
	this->gradation						= -1;
	this->lag								= -1;
   this->groundingline_distance		= -1;
	this->icefront_distance				= -1;
	this->thicknesserror_threshold	= -1;
	this->deviatoricerror_threshold	= -1;
	this->sid2index.clear();
	this->index2sid.clear();
	this->specialelementsindex.clear();
}
/*}}}*/
void AdaptiveMeshRefinement::Initialize(){/*{{{*/

	/*Set pointers to NULL*/
	this->fathermesh						= NULL;
	this->previousmesh					= NULL;
	this->level_max						= -1;
	this->radius_level_max				= -1;
	this->gradation						= -1;
	this->lag								= -1;
   this->groundingline_distance		= -1;
	this->icefront_distance				= -1;
	this->thicknesserror_threshold	= -1;
	this->deviatoricerror_threshold	= -1;
	this->sid2index.clear();
	this->index2sid.clear();
	this->specialelementsindex.clear();
}
/*}}}*/

/*Mesh refinement methods*/
void AdaptiveMeshRefinement::ExecuteRefinement(int numberofpoints,double* xylist,int* pnewnumberofvertices,int* pnewnumberofelements,double** px,double** py,int** pelementslist){/*{{{*/

	/*IMPORTANT! pelementslist are in Matlab indexing*/
	/*NEOPZ works only in C indexing*/
	if(!this->fathermesh || !this->previousmesh) _error_("Impossible to execute refinement: fathermesh or previousmesh is NULL!\n");

	/*Intermediaries*/
	bool verbose=VerboseSolution();

	/*Refine the mesh using level max*/
	this->RefineMesh(verbose,this->previousmesh,numberofpoints,xylist);
	
	/*Get new geometric mesh in ISSM data structure*/
	this->GetMesh(this->previousmesh,pnewnumberofvertices,pnewnumberofelements,px,py,pelementslist);

	/*Verify the new geometry*/
	this->CheckMesh(pnewnumberofvertices,pnewnumberofelements,px,py,pelementslist);
	
	if(verbose) _printf_("\trefinement process done!\n");
}
/*}}}*/
void AdaptiveMeshRefinement::ExecuteRefinement(double* gl_elementdistance,double* if_elementdistance,double* deviatoricerror,double* thicknesserror,int* pnewnumberofvertices,int* pnewnumberofelements,double** px,double** py,int** pelementslist){/*{{{*/

	/*IMPORTANT! pelementslist are in Matlab indexing*/
	/*NEOPZ works only in C indexing*/
	if(!this->fathermesh || !this->previousmesh) _error_("Impossible to execute refinement: fathermesh or previousmesh is NULL!\n");

	/*Intermediaries*/
	bool verbose=true;
	
	/*Execute refinement*/
	this->RefinementProcess(verbose,gl_elementdistance,if_elementdistance,deviatoricerror,thicknesserror);
   
	/*Get new geometric mesh in ISSM data structure*/
	this->GetMesh(this->previousmesh,pnewnumberofvertices,pnewnumberofelements,px,py,pelementslist);

	/*Verify the new geometry*/
	this->CheckMesh(pnewnumberofvertices,pnewnumberofelements,px,py,pelementslist);
}
/*}}}*/
void AdaptiveMeshRefinement::RefinementProcess(bool &verbose,double* gl_elementdistance,double* if_elementdistance,double* deviatoricerror,double* thicknesserror){/*{{{*/
   
	if(verbose) _printf_("\n\trefinement process started (level max = " << this->level_max << ")\n");
	
	/*Intermediaries*/
	TPZGeoMesh* nohangingnodesmesh=NULL;

	double mean_mask		= 0;
	double mean_tauerror = 0;
	double mean_Herror	= 0;
	double group_Herror	= 0;
	int index,sid;
	std::vector<int> specialfatherindex; specialfatherindex.clear();

	/*Calculate mean values*/
	/*
	for(int i=0;i<this->sid2index.size();i++){
		mean_mask		+= masklevelset[i]; 
		mean_tauerror	+= deviatorictensorerror[i]; 
		mean_Herror		+= thicknesserror[i];
	}
	mean_mask		/= this->sid2index.size();
	mean_tauerror	/= this->sid2index.size();
	mean_Herror		/= this->sid2index.size();
	*/
	if(verbose) _printf_("\t\tdeal with special elements...\n");
	/*Deal with special elements*/
	for(int i=0;i<this->specialelementsindex.size();i++){
		if(this->specialelementsindex[i]==-1) continue;
		/*Get special element and verify*/
		TPZGeoEl* geoel=this->previousmesh->Element(this->specialelementsindex[i]);
		if(!geoel)_error_("special element (sid) "<<i<<" is null!\n");
		if(geoel->HasSubElement())_error_("special element (sid) "<<i<<" has "<<geoel->NSubElements()<<" subelements!\n");
		if(geoel->MaterialId()!=this->GetElemMaterialID()) _error_("geoel->MaterialId is not GetElemMaterialID!\n");
		if(!geoel->Father())_error_("father of special element (sid) "<<i<<" is null!\n");
		
		/*Get element's siblings and verify*/
		TPZGeoEl* father=geoel->Father();
		TPZVec<TPZGeoEl *> siblings;
		father->GetHigherSubElements(siblings);
		std::vector<int> sidvec; sidvec.resize(siblings.size());
		for (int j=0;j<siblings.size();j++){
			if(!siblings[j]) _error_("special element (sid) "<<i<<" has a null siblings null!\n"); 
			sidvec[j]=this->index2sid[siblings[j]->Index()];
		}
		
		/*Now, reset the data strucure and verify if the siblings should be deleted*/	
		if(siblings.size()<4){
			/*Reset subelements in the father*/
			father->ResetSubElements();
		}else{
			if(siblings.size()!=4) _error_("element (index) "<<father->Index()<<" has "<<father->NSubElements()<<" subelements!\n");
		}
		for (int j=0;j<siblings.size();j++){
			for(int k=0;k<this->specialelementsindex.size();k++){
				if(this->specialelementsindex[k]==siblings[j]->Index()){ 
					index									= siblings[j]->Index();
					if(index<0) _error_("index is null!\n");
					sid									= this->index2sid[index];
					if(sid<0) _error_("sid is null!\n");
					this->specialelementsindex[k]	= -1;
					this->index2sid[index]			= -1;
					this->sid2index[sid]				= -1;
				}
			}
			if(siblings.size()<4){
				/*Ok, the special element can be deleted*/
				siblings[j]->ResetSubElements();
				this->previousmesh->DeleteElement(siblings[j],siblings[j]->Index());
			}
		}
		
		/*Now, verify if the father should be refined with uniform pattern (smoother)*/
		if(siblings.size()==3){//it keeps the mesh with uniform elements
			/*Father has uniform subelements now*/
			TPZVec<TPZGeoEl *> sons;
			father->Divide(sons);
		}else{
			specialfatherindex.push_back(father->Index());
		}
		if(this->specialelementsindex[i]!=-1) _error_("special element "<<i<<" was not deleted!\n");	
	}
	this->previousmesh->BuildConnectivity();
	
	if(verbose) _printf_("\t\tuniform refinement...\n");
	/*Deal with uniform elemnts*/
	for(int i=0;i<this->sid2index.size();i++){
		if(this->sid2index[i]==-1) continue;
		/*Get element and verify*/
		TPZGeoEl* geoel=this->previousmesh->Element(this->sid2index[i]);
		if(geoel->HasSubElement()) _error_("element (sid) "<<i<<" has "<<geoel->NSubElements()<<" subelements!\n");
		if(geoel->MaterialId()!=this->GetElemMaterialID()) _error_("geoel->MaterialId is not GetElemMaterialID!\n");

		/*Refine process*/
		if(thicknesserror[i]>mean_Herror)
		{	
			int count=0;
			TPZGeoEl* father=geoel->Father();
			if(father){
				for(int j=3;j<6;j++){
					index=father->Neighbour(j).Element()->Index();
					for(int k=0;k<specialfatherindex.size();k++) if(specialfatherindex[k]==index) count++;
				}
			}
			TPZVec<TPZGeoEl *> sons;
			if(geoel->Level()<this->level_max && count==0) geoel->Divide(sons);
		} 
		else if(geoel->Level()>0)
		{/*Unrefine process*/
			
			/*Get siblings and verify*/
			TPZVec<TPZGeoEl *> siblings;
			geoel->Father()->GetHigherSubElements(siblings);
			//if(siblings.size()<4) _error_("Impossible to refine: geoel (index) "<<this->sid2index[i]<<" has less than 3 siblings!\n");	
			if(siblings.size()>4) continue;//Father has more then 4 sons, this group should not be unrefined.
			
			/*Compute the error of the group*/
			group_Herror=0;
			for(int j=0;j<siblings.size();j++){
				index		= siblings[j]->Index();
				sid		= this->index2sid[index];
				if(sid==-1) continue;
				group_Herror+=thicknesserror[sid];
			}
			/*Verify if this group should be unrefined*/
			if(group_Herror>0 && group_Herror<0*mean_Herror){ //itapopo
				/*Reset subelements in the father*/
				this->previousmesh->Element(geoel->Father()->Index())->ResetSubElements();
				/*Delete the elements and set their indexes in the index2sid and sid2index*/
				for (int j=0;j<siblings.size();j++){
					index	= siblings[j]->Index();
					sid	= this->index2sid[index];
					this->index2sid[index]=-1;
					if(sid!=-1) this->sid2index[sid]=-1;
					this->previousmesh->DeleteElement(siblings[j],siblings[j]->Index());
				}//for j
			}//if
		}/*Unrefine process*/
	}//for i
	this->previousmesh->BuildConnectivity();
	
	if(verbose) _printf_("\t\trefine to avoid hanging nodes...\n");
	this->RefineMeshToAvoidHangingNodes(verbose,this->previousmesh);
	
		//nohangingnodesmesh = this->CreateRefPatternMesh(newmesh); itapopo tentar otimizar

	if(verbose) _printf_("\trefinement process done!\n");
}
/*}}}*/
void AdaptiveMeshRefinement::RefineMesh(bool &verbose,TPZGeoMesh* gmesh,int numberofpoints,double* xylist){/*{{{*/
	
	/*Verify if there are points*/
	if(numberofpoints==0) return;	

	/*Intermediaries*/
	int nelem			=-1;
	int side2D			= 6;
	double radius_h	=-1;
	double radius_hmax=std::max(this->radius_level_max,std::max(this->groundingline_distance,this->icefront_distance));
	int count			=-1;
	double mindistance=0.;
	double distance	=0.;;
	TPZVec<REAL> qsi(2,0.),cp(3,0.);
	TPZVec<TPZGeoEl*> sons;
 
	/*First, delete the special elements*/
	this->DeleteSpecialElements(verbose,gmesh);
	
	/*Refinement process: loop over level of refinements{{{*/
	if(verbose) _printf_("\trefinement process (level max = "<<this->level_max<<")\n");
	for(int h=1;h<=this->level_max;h++){
		if(verbose) _printf_("\tlevel "<<h<<" (total: ");
		count=0;

		/*Filter: set the region/radius for level h*/
		radius_h=radius_hmax*std::pow(this->gradation,this->level_max-h);

		/*Find the minimal distance of the elements (center point) to the points */ 
		nelem=gmesh->NElements();//must keep here
		for(int i=0;i<nelem;i++){//itapopo pode-se reduzir o espaço de busca aqui
			if(!gmesh->Element(i)) continue;
			if(gmesh->Element(i)->MaterialId()!=this->GetElemMaterialID()) continue;
			if(gmesh->Element(i)->HasSubElement()) continue;
			if(gmesh->Element(i)->Level()>=h) continue;
			gmesh->Element(i)->CenterPoint(side2D,qsi);
			gmesh->Element(i)->X(qsi,cp);
			mindistance=radius_h;
			for (int j=0;j<numberofpoints;j++){
				distance		= std::sqrt( (xylist[2*j]-cp[0])*(xylist[2*j]-cp[0])+(xylist[2*j+1]-cp[1] )*(xylist[2*j+1]-cp[1]) );
				mindistance = std::min(mindistance,distance);//min distance to the point
			}
			/*If the element is inside the region, refine it*/
			if(mindistance<radius_h){ 
				gmesh->Element(i)->Divide(sons);
				count++;
			}
		}
		if(verbose) _printf_(""<<count<<")\n");
	}
	/*Adjust the connectivities before continue*/
	gmesh->BuildConnectivity();
	/*}}}*/
	
	/*Unrefinement process: loop over level of refinements{{{*/
	if(verbose) _printf_("\tunrefinement process...\n");
	for(int h=this->level_max;h>=1;h--){
		if(verbose) _printf_("\tlevel "<<h<<" (total: ");
		count=0;
		
		/*Filter with lag: set the region/radius for level h*/
		radius_h=this->lag*radius_hmax*std::pow(this->gradation,this->level_max-h);
		
		/*Find the minimal distance of the elements (center point) to the points */ 
		nelem=gmesh->NElements();//must keep here
		for(int i=0;i<nelem;i++){//itapopo pode-se reduzir o espaço de busca aqui
			if(!gmesh->Element(i)) continue;
			if(gmesh->Element(i)->MaterialId()!=this->GetElemMaterialID()) continue;
			if(gmesh->Element(i)->HasSubElement()) continue;
			if(gmesh->Element(i)->Level()!=h) continue;
			if(!gmesh->Element(i)->Father()) _error_("father is NULL!\n");
			/*Get the sons of the father (sibilings)*/	
			sons.clear();
			gmesh->Element(i)->Father()->GetHigherSubElements(sons);
			if(sons.size()!=4) continue;//delete just group of 4 elements. This avoids big holes in the mesh
			/*Find the minimal distance of the group*/	
			mindistance=INFINITY;
			for(int s=0;s<sons.size();s++){
				sons[s]->CenterPoint(side2D,qsi);
				sons[s]->X(qsi,cp);
				for (int j=0;j<numberofpoints;j++){
					distance		= std::sqrt( (xylist[2*j]-cp[0])*(xylist[2*j]-cp[0])+(xylist[2*j+1]-cp[1] )*(xylist[2*j+1]-cp[1]) );
					mindistance = std::min(mindistance,distance);//min distance to the point
				}
			}
			/*If the group is outside the region, unrefine the group*/
			if(mindistance>radius_h){ 
				gmesh->Element(i)->Father()->ResetSubElements();
				count++;
				for(int s=0;s<sons.size();s++){
					gmesh->DeleteElement(sons[s],sons[s]->Index());
				}
			}
		}
		if(verbose) _printf_(""<<count<<")\n");
	}
	/*Adjust the connectivities before continue*/
	gmesh->BuildConnectivity();
	/*}}}*/
	
	/*Now, insert special elements to avoid hanging nodes*/
	this->RefineMeshToAvoidHangingNodes(verbose,gmesh);
}
/*}}}*/
void AdaptiveMeshRefinement::RefineMesh(TPZGeoMesh* gmesh,std::vector<int> &elements){/*{{{*/

	/*Refine elements: uniform pattern refinement*/
	for(int i=0;i<elements.size();i++){
		/*Get geometric element and verify if it has already been refined*/
		int index = elements[i];
		TPZGeoEl * geoel = gmesh->Element(index);
		if(geoel->HasSubElement()) _error_("Impossible to refine: geoel (index) " << index << " has subelements!\n");
		if(geoel->MaterialId()!=this->GetElemMaterialID()) _error_("Impossible to refine: geoel->MaterialId is not GetElemMaterialID!\n");
		/*Divide geoel*/
		TPZVec<TPZGeoEl *> Sons;
		geoel->Divide(Sons);
	}
	gmesh->BuildConnectivity();
}
/*}}}*/
void AdaptiveMeshRefinement::RefineMeshToAvoidHangingNodes(bool &verbose,TPZGeoMesh* gmesh){/*{{{*/
   
	/*Now, insert special elements to avoid hanging nodes*/
	if(verbose) _printf_("\trefining to avoid hanging nodes (total: ");
	
	/*Intermediaries*/
	int nelem=-1;
	int count= 1;
	
	while(count>0){
		nelem=gmesh->NElements();//must keep here
		count=0;
		for(int i=0;i<nelem;i++){
			/*Get geometric element and verify if it has already been refined. Geoel may not have been previously refined*/
			TPZGeoEl * geoel=gmesh->Element(i);
			if(!geoel) continue;
			if(geoel->HasSubElement()) continue;
			if(geoel->MaterialId() != this->GetElemMaterialID()) continue;
			/*Get the refinement pattern for this element and refine it*/
			TPZAutoPointer<TPZRefPattern> refp=TPZRefPatternTools::PerfectMatchRefPattern(geoel);
			if(refp){
				/*Non-uniform refinement*/
				TPZVec<TPZGeoEl *> sons;
				geoel->SetRefPattern(refp);
				geoel->Divide(sons);
				count++;
				/*Keep the index of the special elements*/
				for(int j=0;j<sons.size();j++) this->specialelementsindex.push_back(sons[j]->Index());
			}
		}
		if(verbose){
			if(count==0) _printf_(""<<count<<")\n");
			else _printf_(""<<count<<", ");
		}
		gmesh->BuildConnectivity();
	}
}
/*}}}*/
void AdaptiveMeshRefinement::DeleteSpecialElements(bool &verbose,TPZGeoMesh* gmesh){/*{{{*/

	/*Intermediaries*/
	int count=0;

	if(verbose) _printf_("\tdelete "<<this->specialelementsindex.size()<<" special elements (total: ");
	for(int i=0;i<this->specialelementsindex.size();i++){
		if(this->specialelementsindex[i]==-1) continue;
		if(!gmesh->Element(this->specialelementsindex[i])) continue;
		if(gmesh->Element(this->specialelementsindex[i])->Father()) gmesh->Element(this->specialelementsindex[i])->Father()->ResetSubElements();
		gmesh->DeleteElement(gmesh->Element(this->specialelementsindex[i]),this->specialelementsindex[i]);
		count++;
	}
	if(verbose) _printf_(""<<count<<")\n");
	/*Cleanup*/
	this->specialelementsindex.clear();
	/*Adjust connectivities*/
	gmesh->BuildConnectivity();
}
/*}}}*/
void AdaptiveMeshRefinement::GetMesh(TPZGeoMesh* gmesh,int* nvertices,int* nelements,double** px,double** py, int** pelements){/*{{{*/

	/* IMPORTANT! pelements are in Matlab indexing
	   NEOPZ works only in C indexing.
		This method cleans up and updated the this->sid2index
		and this->index2sid and fills in it with the new mesh.
		Avoid to call this method before Refinement Process.*/

	/*Intermediaries */
	long sid,nodeindex;
	int nconformelements,nconformvertices;
	int ntotalvertices		= gmesh->NNodes();
	int* newelements			= NULL;
	double* newmeshX			= NULL;
	double* newmeshY			= NULL;
	TPZGeoEl* geoel			= NULL;
	long* vertex_index2sid 	= xNew<long>(ntotalvertices);
	this->index2sid.clear(); this->index2sid.resize(gmesh->NElements());
	this->sid2index.clear();
	
	/*Fill in the vertex_index2sid vector with non usual index value*/
	for(int i=0;i<gmesh->NNodes();i++) vertex_index2sid[i]=-1;
	
	/*Fill in the this->index2sid vector with non usual index value*/
	for(int i=0;i<gmesh->NElements();i++) this->index2sid[i]=-1;
	
	/*Get elements without sons and fill in the vertex_index2sid with used vertices (indexes) */
	sid=0;
	for(int i=0;i<gmesh->NElements();i++){//over gmesh elements index 
		geoel=gmesh->ElementVec()[i];
		if(!geoel) continue;
		if(geoel->HasSubElement()) continue;
		if(geoel->MaterialId() != this->GetElemMaterialID()) continue;
		this->sid2index.push_back(geoel->Index());//keep the element index
		this->index2sid[geoel->Index()]=this->sid2index.size()-1;//keep the element sid
		for(int j=0;j<this->GetNumberOfNodes();j++){
      	nodeindex=geoel->NodeIndex(j);
      	if(nodeindex<0) _error_("nodeindex is <0\n");
			if(vertex_index2sid[nodeindex]==-1){
      		vertex_index2sid[nodeindex]=sid; 
				sid++;
			}
      }	
	}

	nconformelements	= (int)this->sid2index.size();
	nconformvertices	= (int)sid;
	newelements			= xNew<int>(nconformelements*this->GetNumberOfNodes());
	newmeshX				= xNew<double>(nconformvertices);
   newmeshY				= xNew<double>(nconformvertices);

	for(int i=0;i<ntotalvertices;i++){//over the TPZNode index (fill in the ISSM vertices coords)
		sid=vertex_index2sid[i];
		if(sid==-1) continue;//skip this index (node no used)
		TPZVec<REAL> coords(3,0.);
		gmesh->NodeVec()[i].GetCoordinates(coords);
		newmeshX[sid] = coords[0];
		newmeshY[sid] = coords[1];
	}
		
	for(int i=0;i<this->sid2index.size();i++){//over the sid (fill the ISSM elements)
		for(int j=0;j<this->GetNumberOfNodes();j++) {
			geoel	= gmesh->ElementVec()[this->sid2index[i]];
			sid	= vertex_index2sid[geoel->NodeIndex(j)];
			newelements[i*this->GetNumberOfNodes()+j]=(int)sid+1;//C to Matlab indexing
		}
		/*Verify the Jacobian determinant. If detJ<0, swap the 2 first postions:
		  a -> b
		  b -> a */
		double detJ,xa,xb,xc,ya,yb,yc;
		int a,b,c;

		a=newelements[i*this->GetNumberOfNodes()+0]-1;
		b=newelements[i*this->GetNumberOfNodes()+1]-1;
		c=newelements[i*this->GetNumberOfNodes()+2]-1;

		xa=newmeshX[a]; ya=newmeshY[a];
		xb=newmeshX[b]; yb=newmeshY[b];
		xc=newmeshX[c]; yc=newmeshY[c];

		detJ=(xb-xa)*(yc-ya)-(xc-xa)*(yb-ya);
	
		/*verify and swap, if necessary*/
		if(detJ<0) {
			newelements[i*this->GetNumberOfNodes()+0]=b+1;//a->b
			newelements[i*this->GetNumberOfNodes()+1]=a+1;//b->a
		}
	}

	/*Setting outputs*/
	*nvertices	= nconformvertices;
	*nelements	= nconformelements;
	*px			= newmeshX;
	*py		   = newmeshY;
	*pelements	= newelements;
   
	/*Cleanup*/
	xDelete<long>(vertex_index2sid);
}
/*}}}*/
void AdaptiveMeshRefinement::CreateInitialMesh(int &nvertices,int &nelements,double* x,double* y,int* elements){/*{{{*/

	/* IMPORTANT! elements come in Matlab indexing
		NEOPZ works only in C indexing*/
	
	if(nvertices<=0) _error_("Impossible to create initial mesh: nvertices is <= 0!\n");
   if(nelements<=0) _error_("Impossible to create initial mesh: nelements is <= 0!\n");

    /*Verify and creating initial mesh*/
   if(this->fathermesh || this->previousmesh) _error_("Initial mesh already exists!");
    
   this->fathermesh = new TPZGeoMesh();
	this->fathermesh->NodeVec().Resize(nvertices);

	/*Set the vertices (geometric nodes in NeoPZ context)*/
	for(int i=0;i<nvertices;i++){  
      /*x,y,z coords*/
		TPZManVector<REAL,3> coord(3,0.);
      coord[0]= x[i];
      coord[1]= y[i];
      coord[2]= 0.;
      /*Insert in the mesh*/
      this->fathermesh->NodeVec()[i].SetCoord(coord);
		this->fathermesh->NodeVec()[i].SetNodeId(i);
	}
	
	/*Generate the elements*/
	long index;
   const int mat = this->GetElemMaterialID();
   TPZManVector<long> elem(this->GetNumberOfNodes(),0);
   this->sid2index.clear();

	for(int i=0;i<nelements;i++){
		for(int j=0;j<this->GetNumberOfNodes();j++) elem[j]=elements[i*this->GetNumberOfNodes()+j]-1;//Convert Matlab to C indexing
      /*reftype = 0: uniform, fast / reftype = 1: uniform and non-uniform (avoid hanging nodes), it is not too fast */
      const int reftype = 1;
      switch(this->GetNumberOfNodes()){
			case 3: this->fathermesh->CreateGeoElement(ETriangle,elem,mat,index,reftype);	break;
         default:	_error_("mesh not supported yet");
		}
      /*Define the element ID*/        
      this->fathermesh->ElementVec()[index]->SetId(i);
		/*Initialize sid2index*/
		this->sid2index.push_back((int)index);
	}
   /*Build element and node connectivities*/
   this->fathermesh->BuildConnectivity();
	/*Set previous mesh*/
	this->previousmesh=new TPZGeoMesh(*this->fathermesh);
}
/*}}}*/
TPZGeoMesh* AdaptiveMeshRefinement::CreateRefPatternMesh(TPZGeoMesh* gmesh){/*{{{*/
	
	TPZGeoMesh *newgmesh = new TPZGeoMesh();
   newgmesh->CleanUp();
    
   int nnodes  = gmesh->NNodes();
	int nelem   = gmesh->NElements();
   int mat     = this->GetElemMaterialID();;
   int reftype = 1;
   long index; 
   
	//nodes
	newgmesh->NodeVec().Resize(nnodes);
   for(int i=0;i<nnodes;i++) newgmesh->NodeVec()[i] = gmesh->NodeVec()[i];
    
   //elements
   for(int i=0;i<nelem;i++){
   	TPZGeoEl * geoel = gmesh->Element(i);
      TPZManVector<long> elem(3,0);
      for(int j=0;j<3;j++) elem[j] = geoel->NodeIndex(j);
     
      newgmesh->CreateGeoElement(ETriangle,elem,mat,index,reftype);
      newgmesh->ElementVec()[index]->SetId(geoel->Id());
        
      TPZGeoElRefPattern<TPZGeoTriangle>* newgeoel = dynamic_cast<TPZGeoElRefPattern<TPZGeoTriangle>*>(newgmesh->ElementVec()[index]);
        
      //old neighbourhood
      const int nsides = TPZGeoTriangle::NSides;
      TPZVec< std::vector<TPZGeoElSide> > neighbourhood(nsides);
      TPZVec<long> NodesSequence(0);
      for(int s = 0; s < nsides; s++){
      	neighbourhood[s].resize(0);
      	TPZGeoElSide mySide(geoel,s);
      	TPZGeoElSide neighS = mySide.Neighbour();
         if(mySide.Dimension() == 0){
         	long oldSz = NodesSequence.NElements();
            NodesSequence.resize(oldSz+1);
            NodesSequence[oldSz] = geoel->NodeIndex(s);
         }
      	while(mySide != neighS){
         	neighbourhood[s].push_back(neighS);
            neighS = neighS.Neighbour();
         }
      }
        
      //inserting in new element
      for(int s = 0; s < nsides; s++){
      	TPZGeoEl * tempEl = newgeoel;
         TPZGeoElSide tempSide(newgeoel,s);
         int byside = s;
         for(unsigned long n = 0; n < neighbourhood[s].size(); n++){
         	TPZGeoElSide neighS = neighbourhood[s][n];
            tempEl->SetNeighbour(byside, neighS);
            tempEl = neighS.Element();
            byside = neighS.Side();
         }
         tempEl->SetNeighbour(byside, tempSide);
      }
        
      long fatherindex = geoel->FatherIndex();
      if(fatherindex>-1) newgeoel->SetFather(fatherindex);
        
      if(!geoel->HasSubElement()) continue;
        
      int nsons = geoel->NSubElements();

      TPZAutoPointer<TPZRefPattern> ref = gRefDBase.GetUniformRefPattern(ETriangle);
      newgeoel->SetRefPattern(ref);
        
      for(int j=0;j<nsons;j++){
      	TPZGeoEl* son = geoel->SubElement(j);
         if(!son){
             DebugStop();
         }
         newgeoel->SetSubElement(j,son);
      }
   }
	newgmesh->BuildConnectivity();
    
	return newgmesh;
}
/*}}}*/
void AdaptiveMeshRefinement::CheckMesh(int* nvertices,int* nelements,double** px,double** py,int** pelements){/*{{{*/

	/*Basic verification*/
	if(nvertices<=0) _error_("Impossible to continue: nvertices <=0!\n");
	if(nelements<=0) _error_("Impossible to continue: nelements <=0!\n");
	if(!px) _error_("Impossible to continue: px is NULL!\n");
	if(!py) _error_("Impossible to continue: py is NULL!\n");
	if(!pelements) _error_("Impossible to continue: pelements is NULL!\n");

	/*Verify if there are orphan nodes*/
	std::set<int> elemvertices;
	elemvertices.clear(); 
	for(int i=0;i<*nelements;i++){
		for(int j=0;j<this->GetNumberOfNodes();j++) {
			elemvertices.insert((*pelements)[i*this->GetNumberOfNodes()+j]);
		}
	}
	if(elemvertices.size()!=*nvertices) _error_("Impossible to continue: elemvertices.size() != nvertices!\n");
	
	//Verify if there are inf or NaN in coords
	for(int i=0;i<*nvertices;i++){
		if(isnan((*px)[i]) || isinf((*px)[i])) _error_("Impossible to continue: px i=" << i <<" is NaN or Inf!\n"); 
		if(isnan((*py)[i]) || isinf((*py)[i])) _error_("Impossible to continue: py i=" << i <<" is NaN or Inf!\n");
	}
	for(int i=0;i<*nelements;i++){
		for(int j=0;j<this->GetNumberOfNodes();j++){
			if(isnan((*pelements)[i*GetNumberOfNodes()+j])) _error_("Impossible to continue: px i=" << i <<" is NaN!\n");
			if(isinf((*pelements)[i*GetNumberOfNodes()+j])) _error_("Impossible to continue: px i=" << i <<" is Inf!\n");
		}
	}
    
}
/*}}}*/
void AdaptiveMeshRefinement::PrintGMeshVTK(TPZGeoMesh* gmesh,std::ofstream &file,bool matColor){/*{{{*/
    
	file.clear();
	long nelements = gmesh->NElements();
	TPZGeoEl *gel;
	std::stringstream node, connectivity, type, material;

	//Header
	file << "# vtk DataFile Version 3.0" << std::endl;
	file << "TPZGeoMesh VTK Visualization" << std::endl;
	file << "ASCII" << std::endl << std::endl;
	file << "DATASET UNSTRUCTURED_GRID" << std::endl;
	file << "POINTS ";

	long actualNode = -1, size = 0, nVALIDelements = 0;
	for(long el = 0; el < nelements; el++){
	  gel = gmesh->ElementVec()[el];
	  if(!gel )//|| (gel->Type() == EOned && !gel->IsLinearMapping()))//Exclude Arc3D and Ellipse3D
	  {
			continue;
	  }
	  if(gel->HasSubElement())
	  {
			continue;
	  }
	  MElementType elt = gel->Type();
	  int elNnodes = MElementType_NNodes(elt);
	  
	  size += (1+elNnodes);
	  connectivity << elNnodes;
	  
	  for(int t = 0; t < elNnodes; t++)
	  {
			for(int c = 0; c < 3; c++)
			{
				 double coord = gmesh->NodeVec()[gel->NodeIndex(t)].Coord(c);
				 node << coord << " ";
			}
			node << std::endl;
			
			actualNode++;
			connectivity << " " << actualNode;
	  }
	  connectivity << std::endl;
	  
	  int elType = this->GetVTK_ElType(gel);
	  type << elType << std::endl;
	  
	  if(matColor == true)
	  {
			material << gel->MaterialId() << std::endl;
	  }
	  else
	  {
			material << gel->Index() << std::endl;
	  }
	  
	  nVALIDelements++;
	}
	node << std::endl;
	actualNode++;
	file << actualNode << " float" << std::endl << node.str();

	file << "CELLS " << nVALIDelements << " ";

	file << size << std::endl;
	file << connectivity.str() << std::endl;

	file << "CELL_TYPES " << nVALIDelements << std::endl;
	file << type.str() << std::endl;

	file << "CELL_DATA" << " " << nVALIDelements << std::endl;
	file << "FIELD FieldData 1" << std::endl;
	if(matColor == true)
	{
	  file << "material 1 " << nVALIDelements << " int" << std::endl;
	}
	else
	{
	  file << "ElementIndex 1 " << nVALIDelements << " int" << std::endl;
	}
	file << material.str();
	file.close();
}
/*}}}*/
int AdaptiveMeshRefinement::GetVTK_ElType(TPZGeoEl * gel){/*{{{*/
    
	MElementType pzElType = gel->Type();
    
    int elType = -1;
    switch (pzElType)
    {
        case(EPoint):
        {
            elType = 1;
            break;
        }
        case(EOned):
        {
            elType = 3;    
            break;
        }
        case (ETriangle):
        {
            elType = 5;
            break;                
        }
        case (EQuadrilateral):
        {
            elType = 9;
            break;                
        }
        case (ETetraedro):
        {
            elType = 10;
            break;                
        }
        case (EPiramide):
        {
            elType = 14;
            break;                
        }
        case (EPrisma):
        {
            elType = 13;
            break;                
        }
        case (ECube):
        {
            elType = 12;
            break;                
        }
        default:
        {
            std::cout << "Element type not found on " << __PRETTY_FUNCTION__ << std::endl;
            DebugStop();
            break;    
        }
    }
    if(elType == -1)
    {
        std::cout << "Element type not found on " << __PRETTY_FUNCTION__ << std::endl;
        std::cout << "MIGHT BE CURVED ELEMENT (quadratic or quarter point)" << std::endl;
        DebugStop();
    }
    
    return elType;
}
/*}}}*/

