/*!\file AmrBamg.cpp
 * \brief: implementation of the adaptive mesh refinement tool based on bamg
 */

#ifdef HAVE_CONFIG_H
    #include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./AmrBamg.h"
#include "../bamg/bamgobjects.h"
#include "../modules/Bamgx/Bamgx.h"

using namespace bamg;
using namespace std;

/*Constructor, copy, clean up and destructor*/
AmrBamg::AmrBamg(IssmDouble hmin,IssmDouble hmax,int fieldenum_in,IssmDouble err_in,int keepmetric_in,
						IssmDouble groundingline_resolution_in,IssmDouble groundingline_distance_in,
						IssmDouble icefront_resolution_in,IssmDouble icefront_distance_in,
						IssmDouble thicknesserror_resolution_in,IssmDouble thicknesserror_threshold_in,
						IssmDouble deviatoricerror_resolution_in,IssmDouble deviatoricerror_threshold_in){/*{{{*/

	this->fieldenum    					= fieldenum_in;
	this->keepmetric   					= keepmetric_in;
	this->groundingline_resolution	= groundingline_resolution_in;
	this->groundingline_distance 		= groundingline_distance_in;
	this->icefront_resolution 			= icefront_resolution_in;
	this->icefront_distance 			= icefront_distance_in;
	this->thicknesserror_resolution 	= thicknesserror_resolution_in;
	this->thicknesserror_threshold 	= thicknesserror_threshold_in;
	this->deviatoricerror_resolution = deviatoricerror_resolution_in;
	this->deviatoricerror_threshold  = deviatoricerror_threshold_in;
	this->geometry     					= NULL;
	this->fathermesh   					= NULL;
	this->previousmesh 					= NULL;

	/*Only initialize options for now (same as bamg.m)*/
	this->options = new BamgOpts();
	this->options->anisomax          = 10.e30;
	this->options->cutoff            = 10.e-5;
	this->options->coeff             = 1;
	this->options->errg              = 0.1;
	this->options->gradation         = 1.5;
	this->options->Hessiantype       = 0;
	this->options->maxnbv            = 1e6;
	this->options->maxsubdiv         = 10;
	this->options->Metrictype        = 0;
	this->options->nbjacobi          = 1;
	this->options->nbsmooth          = 3;
	this->options->omega             = 1.8;
	this->options->power             = 1;
	this->options->verbose           = 0;
	this->options->Crack             = 0;
	this->options->KeepVertices      = 1; /*!!!!! VERY IMPORTANT !!!!!*/
	this->options->splitcorners      = 1;
	this->options->hmin              = hmin;
	this->options->hmax              = hmax;

	this->options->err=xNew<IssmDouble>(1);
	this->options->errSize[0]=1;
	this->options->errSize[1]=1;
	this->options->err[0] = err_in;
}
/*}}}*/
AmrBamg::~AmrBamg(){/*{{{*/

	if(this->geometry) delete this->geometry;
	if(this->fathermesh) delete this->fathermesh;
	if(this->previousmesh) delete this->previousmesh;
	if(this->options) delete this->options;

}
/*}}}*/

/*Methods*/
void AmrBamg::Initialize(int* elements,IssmDouble* x,IssmDouble* y,int numberofvertices,int numberofelements){/*{{{*/

	/*Check options*/
	_assert_(this->options);
	this->options->Check();

	/*Read father mesh and create geometry*/
	Mesh* Th=new Mesh(elements,x,y,numberofvertices,numberofelements);

	/*Write geometry*/
	this->geometry = new BamgGeom();
	Th->Gh.WriteGeometry(this->geometry,this->options);

	/*Write father mesh*/
	this->fathermesh = new BamgMesh();
	Th->WriteMesh(this->fathermesh,this->options);

	/*Cleanup and return*/
	delete Th;
}/*}}}*/
void AmrBamg::ExecuteRefinementBamg(IssmDouble* field,IssmDouble* hmaxVertices,int* pnewnumberofvertices,int *pnewnumberofelements,IssmDouble** px,IssmDouble** py,IssmDouble** pz,int** pelementslist){/*{{{*/

	/*Intermediaries*/
	BamgGeom* geomout=new BamgGeom();
	BamgMesh* meshout=new BamgMesh();

	/*Some checks*/
	_assert_(this->geometry);
	_assert_(this->options);
	_assert_(this->fathermesh);
	_assert_(field || hmaxVertices);//at least one is necessary

	/*Prepare field for metric*/
	this->options->field			 = field;
	this->options->hmaxVertices = hmaxVertices;

	/*remesh*/
	if(this->previousmesh){
		this->options->fieldSize[0]			= this->previousmesh->VerticesSize[0];
		this->options->fieldSize[1]			= 1;
		this->options->hmaxVerticesSize[0]	= this->previousmesh->VerticesSize[0];
		this->options->hmaxVerticesSize[1]	= 1;
		Bamgx(meshout,geomout,this->previousmesh,this->geometry,this->options);
	}
	else{
		this->options->fieldSize[0]			= this->fathermesh->VerticesSize[0];
		this->options->fieldSize[1]			= 1;
		this->options->hmaxVerticesSize[0]	= this->fathermesh->VerticesSize[0];
		this->options->hmaxVerticesSize[1]	= 1;
		Bamgx(meshout,geomout,this->fathermesh,this->geometry,this->options);
	}

	/*remove field and hmaxVertices for memory management (FemModel is taking care of deleting it)*/
	this->options->field = NULL;
	this->options->fieldSize[0] = 0;
	this->options->fieldSize[1] = 0;
	this->options->hmaxVertices = NULL;
	this->options->hmaxVerticesSize[0] = 0;
	this->options->hmaxVerticesSize[1] = 0;
	
	/*verify if the metric will be reseted or not*/
	if(!this->keepmetric){
		if(this->options->metric) xDelete<IssmDouble>(this->options->metric);
		this->options->metricSize[0] = 0;
		this->options->metricSize[1] = 0;
	}

	/*Change previous mesh*/
	if(this->previousmesh) delete this->previousmesh;
	this->previousmesh = meshout;

	/*Prepare output*/
	int nbv = meshout->VerticesSize[0];
	int nbt = meshout->TrianglesSize[0];
	IssmDouble *x = xNew<IssmDouble>(nbv);
	IssmDouble *y = xNew<IssmDouble>(nbv);
	IssmDouble *z = xNew<IssmDouble>(nbv);
	for(int i=0;i<nbv;i++){
		x[i] = meshout->Vertices[i*3+0];
		y[i] = meshout->Vertices[i*3+1];
		z[i] = 0.;
	}
	int* elementslist= xNew<int>(nbt*3);
	for(int i=0;i<nbt;i++){
		elementslist[3*i+0] = reCast<int>(meshout->Triangles[4*i+0]);
		elementslist[3*i+1] = reCast<int>(meshout->Triangles[4*i+1]);
		elementslist[3*i+2] = reCast<int>(meshout->Triangles[4*i+2]);
	}

	/*Cleanup and return*/
	delete geomout;
	*pnewnumberofvertices = nbv;
	*pnewnumberofelements = nbt;
	*px = x;
	*py = y;
	*pz = z;
	*pelementslist = elementslist;
}/*}}}*/
