/*!\file AdaptiveMeshrefinement.cpp
 * \brief: implementation of the adaptive mesh refinement tool based on NeoPZ library: github.com/labmec/neopz
 */

#ifdef HAVE_CONFIG_H
    #include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./AdaptiveMeshRefinement.h"
#include "TPZVTKGeoMesh.h"
#include "../shared/shared.h"
#include "pzgeotriangle.h"
#include "pzreftriangle.h"
using namespace pzgeom;

/*Constructor, copy, clean up and destructor*/
AdaptiveMeshRefinement::AdaptiveMeshRefinement(){/*{{{*/
	this->Initialize();
}
/*}}}*/
AdaptiveMeshRefinement::AdaptiveMeshRefinement(const AdaptiveMeshRefinement &cp){/*{{{*/
	this->Initialize(); 
	this->operator =(cp);
}
/*}}}*/
AdaptiveMeshRefinement & AdaptiveMeshRefinement::operator =(const AdaptiveMeshRefinement &cp){/*{{{*/

	/*Clean all attributes*/
	this->CleanUp();
	/*Copy all data*/
	this->fathermesh			= new TPZGeoMesh(*cp.fathermesh);
	this->currentmesh			= new TPZGeoMesh(*cp.currentmesh);
	this->levelmax				= cp.levelmax;
	this->elementswidth		= cp.elementswidth;
	this->regionlevel1		= cp.regionlevel1;
	this->regionlevelmax		= cp.regionlevelmax;
	this->sid2index.clear();
	this->sid2index.resize(cp.sid2index.size());
	for(int i=0;i<cp.sid2index.size();i++) this->sid2index[i]=cp.sid2index[i];
	
	return *this;

}
/*}}}*/
AdaptiveMeshRefinement::~AdaptiveMeshRefinement(){/*{{{*/
	
	bool ismismip = false;
	if(ismismip){//itapopo
		TPZFileStream fstr;
		std::stringstream ss;
	    
		ss << this->levelmax;
		std::string AMRfile	= "/home/santos/L" + ss.str() + "_amr.txt"; 
	
		fstr.OpenWrite(AMRfile.c_str());
		int withclassid = 1;
		this->Write(fstr,withclassid);
	}
	this->CleanUp();
	gRefDBase.clear();
}
/*}}}*/
void AdaptiveMeshRefinement::CleanUp(){/*{{{*/

	/*Verify and delete all data*/
	if(this->fathermesh)    delete this->fathermesh;
	if(this->currentmesh)   delete this->currentmesh;
	this->levelmax				= -1;
	this->elementswidth		= -1;
	this->regionlevel1		= -1;
	this->regionlevelmax		= -1;
	this->sid2index.clear();
}
/*}}}*/
void AdaptiveMeshRefinement::Initialize(){/*{{{*/

	/*Set pointers to NULL*/
	this->fathermesh			= NULL;
	this->currentmesh			= NULL;
	this->levelmax				= -1;
	this->elementswidth		= -1;
	this->regionlevel1		= -1;
	this->regionlevelmax		= -1;
	this->sid2index.clear();
}
/*}}}*/
int AdaptiveMeshRefinement::ClassId() const{/*{{{*/
    return 13829430; //Antartic area with ice shelves (km^2)
}
/*}}}*/
void AdaptiveMeshRefinement::Read(TPZStream &buf,void *context){/*{{{*/

	try
	{
		/* Read the id context*/
		TPZSaveable::Read(buf,context);
		/* Read class id*/
		int classid;
		buf.Read(&classid,1); 
		/* Verify the class id*/
      if(classid!=this->ClassId()) _error_("AdaptiveMeshRefinement::Read: Error in restoring AdaptiveMeshRefinement!\n"); 
		/* Read simple attributes */
		buf.Read(&this->levelmax,1);
		buf.Read(&this->elementswidth,1);
		buf.Read(&this->regionlevel1,1);
		buf.Read(&this->regionlevelmax,1);
		/* Read vector attributes*/
		int size;
		buf.Read(&size,1);
		int* psid2index=xNew<int>(size);
		buf.Read(psid2index,size);
		this->sid2index.clear();
		this->sid2index.assign(psid2index,psid2index+size);
		/* Read geometric mesh (father)*/
		TPZSaveable *sv1 = TPZSaveable::Restore(buf,0);
		this->fathermesh = dynamic_cast<TPZGeoMesh*>(sv1);
		/* Read geometric mesh (current)*/
		TPZSaveable *sv2 = TPZSaveable::Restore(buf,0);
		this->currentmesh = dynamic_cast<TPZGeoMesh*>(sv2);
		/* Cleanup*/
		xDelete<int>(psid2index);
	}
	catch(const std::exception& e)
	{
		_error_("AdaptiveMeshRefinement::Read: Exception catched!\n");
	}

}
/*}}}*/
template class TPZRestoreClass<AdaptiveMeshRefinement,13829430>;/*{{{*/
/*}}}*/
void AdaptiveMeshRefinement::Write(TPZStream &buf,int withclassid){/*{{{*/
    
	try
	{
		/* Write context (this class) class ID*/
		TPZSaveable::Write(buf,withclassid);
		/* Write this class id*/
		int classid = this->ClassId();
		buf.Write(&classid,1);
		/* Write simple attributes */
		buf.Write(&this->levelmax,1);
		buf.Write(&this->elementswidth,1);
		buf.Write(&this->regionlevel1,1);
		buf.Write(&this->regionlevelmax,1);
		/* Write vector attributes*/
		int size=this->sid2index.size();
		int* psid2index=&this->sid2index[0];
		buf.Write(&size,1);//vector size
		buf.Write(psid2index,this->sid2index.size());
		/* Write the geometric mesh*/
		this->fathermesh->Write(buf,this->ClassId());
		this->currentmesh->Write(buf,this->ClassId());
    }
    catch(const std::exception& e)
    {
		_error_("AdaptiveMeshRefinement::Write: Exception catched!\n");
    }
}
/*}}}*/

/*Mesh refinement methods*/
void AdaptiveMeshRefinement::Execute(bool &amr_verbose,
												int &numberofelements,
												double* partiallyfloatedelements,
												double *masklevelset,
												double* deviatorictensorerror,
												double* thicknesserror,
												int &newnumberofvertices,int &newnumberofelements,double** x,double** y,int** elementslist){/*{{{*/

	/*IMPORTANT! newelements are in Matlab indexing*/
	/*NEOPZ works only in C indexing*/
	if(!this->fathermesh || !this->currentmesh) _error_("Impossible to execute refinement: fathermesh or currentmesh is NULL!\n");
	if(numberofelements!=this->sid2index.size()) _error_("Impossible to execute refinement: sid2index.size is not equal to numberofelements!\n");

	/*Execute the refinement.*/
	this->RefinementProcess(amr_verbose,partiallyfloatedelements,masklevelset,deviatorictensorerror,thicknesserror);
    
	/*Get new geometric mesh in ISSM data structure*/
	this->GetMesh(newnumberofvertices,newnumberofelements,x,y,elementslist);
	
	/*Verify the new geometry*/
	this->CheckMesh(newnumberofvertices,newnumberofelements,this->elementswidth,x,y,elementslist);

}
/*}}}*/
void AdaptiveMeshRefinement::RefinementProcess(bool &amr_verbose,double* partiallyfloatedelements,double* masklevelset,
																double* deviatorictensorerror,double* thicknesserror){/*{{{*/
   
	if(amr_verbose) _printf_("\n\trefinement process started (level max = " << this->levelmax << ")\n");
	
	/*Intermediaries*/
	TPZGeoMesh* nohangingnodesmesh=NULL;
	double mean_mask		= 0;
	double mean_tauerror = 0;
	double mean_Herror	= 0;
	double group_error	= 0;
   
	/*Calculate mean values*/
	for(int i=0;i<this->sid2index.size();i++){
		mean_mask		+= masklevelset[i]; 
		mean_tauerror	+= deviatorictensorerror[i]; 
		mean_Herror		+= thicknesserror[i];
	}
	mean_mask		/= this->sid2index.size();
	mean_tauerror	/= this->sid2index.size();
	mean_Herror		/= this->sid2index.size();

	if(amr_verbose) _printf_("\t\tuniform refinement...\n");
	for(int i=0;i<this->sid2index.size();i++){
		TPZGeoEl* geoel=this->currentmesh->Element(this->sid2index[i]);
		if(geoel->HasSubElement()) _error_("Impossible to refine: geoel (index) "<<this->sid2index[i]<<" has subelements!\n");
		if(geoel->MaterialId()!=this->GetElemMaterialID()) _error_("Impossible to refine: geoel->MaterialId is not GetElemMaterialID!\n");
		/*Refine*/
		if(thicknesserror[i]>mean_Herror){
			TPZVec<TPZGeoEl *> sons;
			if(geoel->Level()<this->levelmax) geoel->Divide(sons);
		} 
		else if(geoel->Level()>0){ /*try to unrefine*/
			TPZVec<TPZGeoEl *> sons;
			geoel->Father()->GetHigherSubElements(sons);
			group_error=0;
			for(int j=0;j<sons.size();j++){
				sons[j]->Index();
			}
		}
	}
	this->currentmesh->BuildConnectivity();
	
	if(amr_verbose) _printf_("\t\trefine to avoid hanging nodes...\n");
	this->RefineMeshToAvoidHangingNodes(this->currentmesh);
	this->currentmesh->BuildConnectivity();
	
		//nohangingnodesmesh = this->CreateRefPatternMesh(newmesh); itapopo tentar otimizar
	
	if(amr_verbose) _printf_("\trefinement process done!\n");
}
/*}}}*/
void AdaptiveMeshRefinement::RefineMesh(TPZGeoMesh *gmesh,std::vector<int> &elements){/*{{{*/

	/*Refine elements: uniform pattern refinement*/
	for(int i=0;i<elements.size();i++){
		/*Get geometric element and verify if it has already been refined*/
		int index = elements[i];
		TPZGeoEl * geoel = gmesh->Element(index);
		if(geoel->HasSubElement()) _error_("Impossible to refine: geoel (index) " << index << " has subelements!\n");
		if(geoel->MaterialId()!=this->GetElemMaterialID()) _error_("Impossible to refine: geoel->MaterialId is not GetElemMaterialID!\n");
		/*Divide geoel*/
		TPZVec<TPZGeoEl *> Sons;
		geoel->Divide(Sons);
	}
	gmesh->BuildConnectivity();
}
/*}}}*/
void AdaptiveMeshRefinement::RefineMeshToAvoidHangingNodes(TPZGeoMesh *gmesh){/*{{{*/
   
	/*Refine elements to avoid hanging nodes: non-uniform refinement*/
	const int NElem = gmesh->NElements();
	for(int i=0;i<NElem;i++){
		/*Get geometric element and verify if it has already been refined. Geoel may not have been previously refined*/
		TPZGeoEl * geoel=gmesh->Element(i);
		if(!geoel) continue;
		if(geoel->HasSubElement()) continue;
		if(geoel->MaterialId() != this->GetElemMaterialID()) continue;
		/*Get the refinement pattern for this element and refine it*/
		TPZAutoPointer<TPZRefPattern> refp=TPZRefPatternTools::PerfectMatchRefPattern(geoel);
		if(refp){
			TPZVec<TPZGeoEl *> Sons;
			geoel->SetRefPattern(refp);
			geoel->Divide(Sons);
      }
	}
   gmesh->BuildConnectivity();
    
}
/*}}}*/
void AdaptiveMeshRefinement::GetMesh(int &nvertices,int &nelements,double** px,double** py, int** pelements){/*{{{*/

	/* IMPORTANT! pelements are in Matlab indexing
	   NEOPZ works only in C indexing.
		This method cleans up and updated the this->sid2index 
		and fills in it with the new mesh.
		Avoid to call this method before Refinement Process.*/

	/*Intermediaries */
	TPZGeoMesh* gmesh = this->currentmesh;//itapopo confirmar
	
	long sid,nodeindex;
	int nconformelements,nconformvertices;	
	int* newelements			= NULL;
	double* newmeshX			= NULL;//xNew<double>(ntotalvertices);
	double* newmeshY			= NULL;//xNew<double>(ntotalvertices);
	TPZGeoEl* geoel			= NULL;
	long* vertex_index2sid 	= xNew<long>(gmesh->NNodes());
	this->sid2index.clear();
	
	/*Get mesh coords */
	//for(int i=0;i<ntotalvertices;i++ ){
	//	TPZVec<REAL> coords(3,0.);
	//	gmesh->NodeVec()[i].GetCoordinates(coords);
	//	newmeshX[i] = coords[0];
	//	newmeshY[i] = coords[1];
	//}

	/*Fill in the vertex_index2sid vector with non usual index value*/
	for(int i=0;i<gmesh->NNodes();i++) vertex_index2sid[i]=-1;
	
	/*Get elements without sons and fill in the vertex_index2sid with used vertices (indexes) */
	sid=0;
	for(int i=0;i<gmesh->NElements();i++){//over gmesh elements index 
		geoel=gmesh->ElementVec()[i];
		if(!geoel) continue;
		if(geoel->HasSubElement()) continue;
		if(geoel->MaterialId() != this->GetElemMaterialID()) continue;
		this->sid2index.push_back(i);//keep the element index
		for(int j=0;j<this->elementswidth;j++){
      	nodeindex=geoel->NodeIndex(j);
      	if(vertex_index2sid[nodeindex]==-1){
      		vertex_index2sid[nodeindex]=sid; 
				sid++;
			}
      }	
	}

	nconformelements	= (int)this->sid2index.size();
	nconformvertices	= (int)sid;
	newelements			= xNew<int>(nconformelements*this->elementswidth);
	newmeshX				= xNew<double>(nconformvertices);
   newmeshY				= xNew<double>(nconformvertices);

	for(int i=0;i<nconformvertices;i++){//over the TPZNode index (fill in the ISSM vertices coords)
		sid = vertex_index2sid[i];
		if(sid!=-1){
			TPZVec<REAL> coords(3,0.);
			gmesh->NodeVec()[i].GetCoordinates(coords);
			newmeshX[sid] = coords[0];
			newmeshY[sid] = coords[1];
		}
	}
		
	for(int i=0;i<this->sid2index.size();i++){//over the sid (fill the ISSM elements)
		for(int j=0;j<this->elementswidth;j++) {
			geoel	= gmesh->ElementVec()[this->sid2index[i]];
			sid	= vertex_index2sid[geoel->NodeIndex(j)];
			newelements[i*this->elementswidth+j]=(int)sid+1;//C to Matlab indexing
		}
	}
 
	/*Setting outputs*/
	nvertices	= nconformvertices;
	nelements	= nconformelements;
	*px			= newmeshX;
	*py		   = newmeshY;
	*pelements	= newelements;
   
	/*Cleanup*/
	xDelete<long>(vertex_index2sid);

}
/*}}}*/
void AdaptiveMeshRefinement::FindElements(int &numberofpoints,double* xp,double* yp,TPZGeoMesh *gmesh,int &hlevel,std::vector<int> &elements){/*{{{*/

	if(!gmesh) _error_("Impossible to set elements: gmesh is NULL!\n");

	if(false){
		this->AllElements(gmesh,elements); //uniform, refine all elements!
		return;
	}

	/*Intermediaries*/
	elements.clear();
	double D1		= this->regionlevel1;
	double Dhmax	= this->regionlevelmax;
	int hmax			= this->levelmax;
	double alpha	= (hmax==1) ? 0. : log(D1/Dhmax)/(hmax-1.);
	double Di		= D1/exp(alpha*(hlevel-1));
	int side2D		= 6;
	double distance,value;
    
	/*Find elements near the points */ 
	for(int i=0;i<gmesh->NElements();i++){
		if(gmesh->Element(i)->MaterialId()!=this->GetElemMaterialID()) continue;
		if(gmesh->Element(i)->HasSubElement()) continue;
		if(gmesh->Element(i)->Level()>=hlevel) continue;
		TPZVec<REAL> qsi(2,0.);
		TPZVec<REAL> centerPoint(3,0.);
		gmesh->Element(i)->CenterPoint(side2D, qsi);
		gmesh->Element(i)->X(qsi, centerPoint);
		distance = Di;
		for (int j=0;j<numberofpoints;j++){
			value = std::sqrt( (xp[j]-centerPoint[0])*(xp[j]-centerPoint[0])+(yp[j]-centerPoint[1] )*(yp[j]-centerPoint[1]) );//sqrt( (x2-x1)^2 + (y2-y1)^2 )
			if(value<distance) distance=value; //min distance to the point
		}  
		if(distance<Di) elements.push_back(i);
	}

}
/*}}}*/
void AdaptiveMeshRefinement::AllElements(TPZGeoMesh *gmesh,std::vector<int> &elements){/*{{{*/
    /* Uniform refinement. This refines the entire mesh */
    int nelements = gmesh->NElements();
	 elements.clear();
    for(int i=0;i<nelements;i++){
        if(gmesh->Element(i)->MaterialId()!=this->GetElemMaterialID()) continue;
        if(gmesh->Element(i)->HasSubElement()) continue;
        elements.push_back(i);
    }
}
/*}}}*/
void AdaptiveMeshRefinement::CreateInitialMesh(int &nvertices,int &nelements,int &width,double* x,double* y,int* elements){/*{{{*/

	/* IMPORTANT! elements come in Matlab indexing
		NEOPZ works only in C indexing*/
	
	if(nvertices<=0) _error_("Impossible to create initial mesh: nvertices is <= 0!\n");
   if(nelements<=0) _error_("Impossible to create initial mesh: nelements is <= 0!\n");
	this->SetElementWidth(width);

    /*Verify and creating initial mesh*/
   if(this->fathermesh || this->currentmesh) _error_("Initial mesh already exists!");
    
   this->fathermesh = new TPZGeoMesh();
	this->fathermesh->NodeVec().Resize(nvertices);

	/*Set the vertices (geometric nodes in NeoPZ context)*/
	for(int i=0;i<nvertices;i++){  
      /*x,y,z coords*/
		TPZManVector<REAL,3> coord(3,0.);
      coord[0]= x[i];
      coord[1]= y[i];
      coord[2]= 0.;
      /*Insert in the mesh*/
      this->fathermesh->NodeVec()[i].SetCoord(coord);
		this->fathermesh->NodeVec()[i].SetNodeId(i);
	}
	
	/*Generate the elements*/
	long index;
   const int mat = this->GetElemMaterialID();
   TPZManVector<long> elem(this->elementswidth,0);
   this->sid2index.clear();

	for(int i=0;i<nelements;i++){
		for(int j=0;j<this->elementswidth;j++) elem[j]=elements[i*this->elementswidth+j]-1;//Convert Matlab to C indexing
      /*reftype = 0: uniform, fast / reftype = 1: uniform and non-uniform (avoid hanging nodes), it is not too fast */
      const int reftype = 1;
      switch(this->elementswidth){
			case 3: this->fathermesh->CreateGeoElement(ETriangle,elem,mat,index,reftype);	break;
         default:	_error_("mesh not supported yet");
		}
      /*Define the element ID*/        
      this->fathermesh->ElementVec()[index]->SetId(i);
		/*Initialize sid2index*/
		this->sid2index.push_back((int)index);
	}
   /*Build element and node connectivities*/
   this->fathermesh->BuildConnectivity();
	/*Set current mesh*/
	this->currentmesh=new TPZGeoMesh(*this->fathermesh);
}
/*}}}*/
TPZGeoMesh* AdaptiveMeshRefinement::CreateRefPatternMesh(TPZGeoMesh* gmesh){/*{{{*/
	
	TPZGeoMesh *newgmesh = new TPZGeoMesh();
   newgmesh->CleanUp();
    
   int nnodes  = gmesh->NNodes();
	int nelem   = gmesh->NElements();
   int mat     = this->GetElemMaterialID();;
   int reftype = 1;
   long index; 
   
	//nodes
	newgmesh->NodeVec().Resize(nnodes);
   for(int i=0;i<nnodes;i++) newgmesh->NodeVec()[i] = gmesh->NodeVec()[i];
    
   //elements
   for(int i=0;i<nelem;i++){
   	TPZGeoEl * geoel = gmesh->Element(i);
      TPZManVector<long> elem(3,0);
      for(int j=0;j<3;j++) elem[j] = geoel->NodeIndex(j);
     
      newgmesh->CreateGeoElement(ETriangle,elem,mat,index,reftype);
      newgmesh->ElementVec()[index]->SetId(geoel->Id());
        
      TPZGeoElRefPattern<TPZGeoTriangle>* newgeoel = dynamic_cast<TPZGeoElRefPattern<TPZGeoTriangle>*>(newgmesh->ElementVec()[index]);
        
      //old neighbourhood
      const int nsides = TPZGeoTriangle::NSides;
      TPZVec< std::vector<TPZGeoElSide> > neighbourhood(nsides);
      TPZVec<long> NodesSequence(0);
      for(int s = 0; s < nsides; s++){
      	neighbourhood[s].resize(0);
      	TPZGeoElSide mySide(geoel,s);
      	TPZGeoElSide neighS = mySide.Neighbour();
         if(mySide.Dimension() == 0){
         	long oldSz = NodesSequence.NElements();
            NodesSequence.resize(oldSz+1);
            NodesSequence[oldSz] = geoel->NodeIndex(s);
         }
      	while(mySide != neighS){
         	neighbourhood[s].push_back(neighS);
            neighS = neighS.Neighbour();
         }
      }
        
      //inserting in new element
      for(int s = 0; s < nsides; s++){
      	TPZGeoEl * tempEl = newgeoel;
         TPZGeoElSide tempSide(newgeoel,s);
         int byside = s;
         for(unsigned long n = 0; n < neighbourhood[s].size(); n++){
         	TPZGeoElSide neighS = neighbourhood[s][n];
            tempEl->SetNeighbour(byside, neighS);
            tempEl = neighS.Element();
            byside = neighS.Side();
         }
         tempEl->SetNeighbour(byside, tempSide);
      }
        
      long fatherindex = geoel->FatherIndex();
      if(fatherindex>-1) newgeoel->SetFather(fatherindex);
        
      if(!geoel->HasSubElement()) continue;
        
      int nsons = geoel->NSubElements();

      TPZAutoPointer<TPZRefPattern> ref = gRefDBase.GetUniformRefPattern(ETriangle);
      newgeoel->SetRefPattern(ref);
        
      for(int j=0;j<nsons;j++){
      	TPZGeoEl* son = geoel->SubElement(j);
         if(!son){
             DebugStop();
         }
         newgeoel->SetSubElement(j,son);
      }
   }
	newgmesh->BuildConnectivity();
    
	return newgmesh;
}
/*}}}*/
void AdaptiveMeshRefinement::SetLevelMax(int &h){/*{{{*/
    this->levelmax = h;
}
/*}}}*/
void AdaptiveMeshRefinement::SetRegions(double &D1,double Dhmax){/*{{{*/
    this->regionlevel1	 = D1;
    this->regionlevelmax = Dhmax;
}
/*}}}*/
void AdaptiveMeshRefinement::SetElementWidth(int &width){/*{{{*/
	if(width!=3) _error_("elementswidth not supported yet!");
   this->elementswidth = width;
}
/*}}}*/
void AdaptiveMeshRefinement::CheckMesh(int &nvertices,int &nelements,int &width,double** px,double** py,int** pelements){/*{{{*/

	/*Basic verification*/
	if(nvertices<=0) _error_("Impossible to continue: nvertices <=0!\n");
	if(nelements<=0) _error_("Impossible to continue: nelements <=0!\n");
	if(width!=3) _error_("Impossible to continue: width !=3!\n"); 
	if(!px) _error_("Impossible to continue: px is NULL!\n");
	if(!py) _error_("Impossible to continue: py is NULL!\n");
	if(!pelements) _error_("Impossible to continue: pelements is NULL!\n");

	/*Verify if there are orphan nodes*/
	std::set<int> elemvertices;
	elemvertices.clear(); 
	for(int i=0;i<nelements;i++){
		for(int j=0;j<width;j++) {
			elemvertices.insert((*pelements)[i*width+j]);
		}
	}
	if(elemvertices.size()!=nvertices) _error_("Impossible to continue: elemvertices.size() != nvertices!\n");
	
	//Verify if there are inf or NaN in coords
	for(int i=0;i<nvertices;i++){
		if(isnan((*px)[i]) || isinf((*px)[i])) _error_("Impossible to continue: px i=" << i <<" is NaN or Inf!\n"); 
		if(isnan((*py)[i]) || isinf((*py)[i])) _error_("Impossible to continue: py i=" << i <<" is NaN or Inf!\n");
	}
	for(int i=0;i<nelements;i++){
		for(int j=0;j<width;j++){
			if(isnan((*pelements)[i*width+j]) || isinf((*pelements)[i*width+j]) ) _error_("Impossible to continue: px i=" << i <<" is NaN or Inf!\n");
		}
	}
    
}
/*}}}*/
