/*!\file AmrBamg.cpp
 * \brief: implementation of the adaptive mesh refinement tool based on bamg
 */

#ifdef HAVE_CONFIG_H
    #include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./AmrBamg.h"
#include "../bamg/bamgobjects.h"
#include "../modules/Bamgx/Bamgx.h"

using namespace bamg;
using namespace std;

/*Constructor, copy, clean up and destructor*/
AmrBamg::AmrBamg(IssmDouble hmin, IssmDouble hmax){/*{{{*/

	this->geometry     = NULL;
	this->fathermesh   = NULL;
	this->previousmesh = NULL;

	/*Only initialize options for now (same as bamg.m)*/
	this->options = new BamgOpts();
	this->options->anisomax          = 10.e30;
	this->options->cutoff            = 10.e-5;
	this->options->coeff             = 1;
	this->options->errg              = 0.1;
	this->options->gradation         = 1.5;
	this->options->Hessiantype       = 0;
	this->options->MaxCornerAngle    = 10;
	this->options->maxnbv            = 1e6;
	this->options->maxsubdiv         = 10;
	this->options->Metrictype        = 0;
	this->options->nbjacobi          = 1;
	this->options->nbsmooth          = 3;
	this->options->omega             = 1.8;
	this->options->power             = 1;
	this->options->random            = 0;
	this->options->verbose           = 0;
	this->options->Crack             = 0;
	this->options->geometricalmetric = 0;
	this->options->KeepVertices      = 1; /*!!!!! VERY IMPORTANT !!!!!*/
	this->options->splitcorners      = 1;
	this->options->hmin              = hmin;
	this->options->hmax              = hmax;
}
/*}}}*/
AmrBamg::~AmrBamg(){/*{{{*/

	if(this->geometry) delete this->geometry;
	if(this->fathermesh) delete this->fathermesh;
	if(this->previousmesh) delete this->previousmesh;
	if(this->options) delete this->options;

}
/*}}}*/

/*Methods*/
void AmrBamg::Initialize(int* elements,IssmDouble* x,IssmDouble* y,int numberofvertices,int numberofelements){/*{{{*/

	/*Check options*/
	_assert_(this->options);
	this->options->Check();

	/*Read father mesh and create geometry*/
	Mesh* Th=new Mesh(elements,x,y,numberofvertices,numberofelements);

	/*Write geometry*/
	this->geometry = new BamgGeom();
	Th->Gh.WriteGeometry(this->geometry,this->options);

	/*Write father mesh*/
	this->fathermesh = new BamgMesh();
	Th->WriteMesh(this->fathermesh,this->options);

	/*Cleanup and return*/
	delete Th;
}/*}}}*/
void AmrBamg::ExecuteRefinementBamg(int* pnewnumberofvertices,int *pnewnumberofelements,IssmDouble** px,IssmDouble** py,IssmDouble** pz,int** pelementslist){/*{{{*/

	/*Intermediaries*/
	BamgGeom* geomout=new BamgGeom();
	BamgMesh* meshout=new BamgMesh();

	/*Some checks*/
	_assert_(this->geometry);
	_assert_(this->options);
	_assert_(this->fathermesh);

	/*remesh*/
	if(this->previousmesh){
		Bamgx(meshout,geomout,this->previousmesh,this->geometry,this->options);
	}
	else{
		Bamgx(meshout,geomout,this->fathermesh,this->geometry,this->options);
	}

	/*Change previous mesh*/
	if(this->previousmesh) delete this->previousmesh;
	this->previousmesh = meshout;

	/*Prepare output*/
	int nbv = meshout->VerticesSize[0];
	int nbt = meshout->TrianglesSize[0];
	IssmDouble *x = xNew<IssmDouble>(nbv);
	IssmDouble *y = xNew<IssmDouble>(nbv);
	IssmDouble *z = xNew<IssmDouble>(nbv);
	for(int i=0;i<nbv;i++){
		x[i] = meshout->Vertices[i*3+0];
		y[i] = meshout->Vertices[i*3+1];
		z[i] = 0.;
	}
	int* elementslist= xNew<int>(nbt*3);
	for(int i=0;i<nbt;i++){
		elementslist[3*i+0] = reCast<int>(meshout->Triangles[4*i+0]);
		elementslist[3*i+1] = reCast<int>(meshout->Triangles[4*i+1]);
		elementslist[3*i+2] = reCast<int>(meshout->Triangles[4*i+2]);
	}

	/*Cleanup and return*/
	delete geomout;
	*pnewnumberofvertices = nbv;
	*pnewnumberofelements = nbt;
	*px = x;
	*py = y;
	*pz = z;
	*pelementslist = elementslist;
}/*}}}*/
