//NODE class definition //{{{
//	Description:
//		Contains all information for rendering an object in WebGL, including mesh, materials, shaders, and other attributes.	
//	Usage:
//		node=new Node('canvas', canvas);

function Node() { //{{{
	//properties 
	// {{{
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(0,args.length));

	this.canvas = options.getfieldvalue('canvas', 								null),
	this.options = options.getfieldvalue('options', 							null),
	this.gl = canvas.gl;
	
	this.alpha = options.getfieldvalue('alpha', 								1.0),
	this.animation = options.getfieldvalue('animation', 						{}),
	this.arrays = options.getfieldvalue('arrays', 								{}),
	this.caxis = options.getfieldvalue('caxis', 								[0.0, 1.0]),
	this.center = options.getfieldvalue('center', 								vec3.create()),
	this.cullFace = options.getfieldvalue('cullFace', 							this.gl.BACK),
	this.computeIndices = options.getfieldvalue('computeIndices', 				true),
	this.disableDepthTest = options.getfieldvalue('disableDepthTest', 			false),
	this.diffuseColor = options.getfieldvalue('diffuseColor', 					[0.0, 0.0, 0.0, 1.0]),
	this.drawMode = options.getfieldvalue('drawMode', 							this.gl.TRIANGLES),
	this.drawOrder = options.getfieldvalue('drawOrder', 						1),
	this.enabled = options.getfieldvalue('enabled', 							true),
	this.enableCullFace = options.getfieldvalue('enableCullFace', 				true),
	this.hideOcean = options.getfieldvalue('hideOcean', 						false),
	this.lineWidth = options.getfieldvalue('lineWidth', 						1.0),
	this.log = options.getfieldvalue('log', 									false),
	this.maskColor = options.getfieldvalue('maskColor', 						vec4.fromValues(0.0, 0.0, 1.0, 1.0)),
	this.maskEnabled = options.getfieldvalue('maskEnabled', 					false),
	this.maskHeight = options.getfieldvalue('maskHeight', 						150.0),
	this.maskZerosColor = options.getfieldvalue('maskZerosColor', 				[1.0, 1.0, 1.0, 1.0]),
	this.maskZerosEnabled = options.getfieldvalue('maskZerosEnabled', 			false),
	this.maskZerosTolerance = options.getfieldvalue('maskZerosTolerance', 		1e-6),
	this.maskZerosZeroValue = options.getfieldvalue('maskZerosZeroValue', 		0.5),
	this.mesh = options.getfieldvalue('mesh', 									undefined),
	this.name = options.getfieldvalue('name', 									'node'),
	this.nanIndices = options.getfieldvalue('nanIndices', 						new Set()),
	this.octree = options.getfieldvalue('octree', 								undefined),
	this.pointSize = options.getfieldvalue('pointSize', 						15.0),
	this.shaderName = options.getfieldvalue('shaderName', 						'Colored'),
	this.shader = options.getfieldvalue('shader', 								this.gl.shaders[this.shaderName]),
	this.texture = options.getfieldvalue('texture', 							undefined),
	this.scale = options.getfieldvalue('scale', 								vec3.fromValues(1, 1, 1)),
	this.rotation = options.getfieldvalue('rotation', 							vec3.create()),
	this.translation = options.getfieldvalue('translation', 					vec3.create()),
	this.modelMatrix = options.getfieldvalue('modelMatrix', 					mat4.create()),
	this.rotationMatrix = options.getfieldvalue('rotationMatrix', 				mat4.create()),
	this.inverseModelMatrix = options.getfieldvalue('inverseModelMatrix', 		mat4.create()),
	this.inverseRotationMatrix = options.getfieldvalue('inverseRotationMatrix', mat4.create())
	//}}}
	//initialize {{{
	//if (this.name in canvas.nodes) abort? 
	this.updateModelMatrix();
	this.updateDiffuseColor();
	canvas.nodes[this.name] = this;
	//}}}
} //}}}
Node.prototype.updateModelMatrix = function() { //{{{
	//Update the model matrix if rotation, scale, or translation have been manually modified.
	var modelMatrix = mat4.create();

	var translationMatrix = mat4.create();
	mat4.translate(translationMatrix, translationMatrix, vec3.negate(vec3.create(), this.center)); //scale/rotation centering
	mat4.multiply(modelMatrix, translationMatrix, modelMatrix);
	
	var scaleMatrix = mat4.create();
	mat4.scale(scaleMatrix, scaleMatrix, this.scale);
	mat4.multiply(modelMatrix, scaleMatrix, modelMatrix);
	
	var rotationMatrix = mat4.create();
	var zRotationMatrix = mat4.create();	
	mat4.rotate(zRotationMatrix, zRotationMatrix, DEG2RAD * this.rotation[2], [0.0, 0.0, 1.0]);
	mat4.multiply(rotationMatrix, zRotationMatrix, rotationMatrix);
	var yRotationMatrix = mat4.create();	
	mat4.rotate(yRotationMatrix, yRotationMatrix, DEG2RAD * this.rotation[1], [0.0, 1.0, 0.0]);
	mat4.multiply(rotationMatrix, yRotationMatrix, rotationMatrix);
	var xRotationMatrix = mat4.create();	
	mat4.rotate(xRotationMatrix, xRotationMatrix, DEG2RAD * this.rotation[0], [1.0, 0.0, 0.0]);
	mat4.multiply(rotationMatrix, xRotationMatrix, rotationMatrix);
	mat4.multiply(modelMatrix, rotationMatrix, modelMatrix);	
	
	mat4.identity(translationMatrix);
	mat4.translate(translationMatrix, translationMatrix, this.center); //relative translation
	mat4.multiply(modelMatrix, translationMatrix, modelMatrix);
	
	mat4.identity(translationMatrix);
	mat4.translate(translationMatrix, translationMatrix, this.translation); //absolute translation
	mat4.multiply(modelMatrix, translationMatrix, modelMatrix);
	
	this.modelMatrix = modelMatrix;
	this.inverseModelMatrix = mat4.invert(mat4.create(), modelMatrix);
	this.rotationMatrix = rotationMatrix;
	this.inverseRotationMatrix = mat4.invert(mat4.create(), rotationMatrix);
} //}}}
Node.prototype.updateDiffuseColor = function() { //{{{
	//Update the diffuse color with an RGB color name or vec4 containing r, g, b, and alpha values from 0.0 to 1.0
	var color = this.diffuseColor;
	if (typeof color === 'string') {
		color = new RGBColor(color);
		if (color.ok) color = [color.r/255.0, color.g/255.0, color.b/255.0, 1.0];
		else throw Error(sprintf("s%s%s\n","initWebGL error message: cound not find out edgecolor color for curent canvas ", canvas));
	}
	this.diffuseColor = color;
} //}}}
Node.prototype.transform = function() { //{{{
	//Transforms the translation, rotation, or scle fo the node and updates the model matrix.
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(0,args.length));
	
	var translation = options.getfieldvalue('translation', undefined);
	var rotation = options.getfieldvalue('rotation', undefined);
	var scale = options.getfieldvalue('scale', undefined);
	
	if (!isEmptyOrUndefined(translation)) this.translation = translation;
	if (!isEmptyOrUndefined(rotation)) this.rotation = rotation;
	if (!isEmptyOrUndefined(scale)) this.scale = scale;
	this.updateModelMatrix();
} //}}}
Node.prototype.patch = function() { //{{{
	//Emulates the behavior of MATLAB patch function by constructing a mesh from arguments.
	//Limitations: 
	//	-Expects pair labeled arguments ('FaceColor','none',...).
	//	-Only handles Face/Vertices/FaceVertexCData element/node plots.
	//	-Only supports FaceColor 'interp' and 'none'.
	
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(0,args.length));

	var faces = options.getfieldvalue('Faces', undefined);
	var vertices = options.getfieldvalue('Vertices', undefined);
	var faceVertexCData = options.getfieldvalue('FaceVertexCData', undefined);
	var faceColor = options.getfieldvalue('FaceColor', 'interp');
	var edgeColor = options.getfieldvalue('EdgeColor', this.diffuseColor);
	var lineWidth = options.getfieldvalue('linewidth', 1);
	
	this.faces = faces;
	this.vertices = vertices;
	this.diffuseColor = edgeColor;
	this.updateDiffuseColor();
	
	this.patchVertices(faceVertexCData, faces, vertices);
	this.patchCoords(faceVertexCData, faces, vertices);
	this.patchIndices(faces, faceColor);

	this.mesh = GL.Mesh.load(this.arrays, null, null, this.gl);
	//this.mesh.computeNormals();
	this.computeOctree();
} //}}}
Node.prototype.patchVertices = function(faceVertexCData, faces, vertices) { //{{{
	//Patch subfunction for processing xyz vertices.
	var vertexArray;
	var face;
	
	if (isEmptyOrUndefined(faceVertexCData)) {
		vertexArray = new Float32Array(vertices[0].length * 3);
		for(var i = 0, v = 0; i < vertices[0].length; i++) {	
			vertexArray[v++] = vertices[0][i];
			vertexArray[v++] = vertices[1][i];
			vertexArray[v++] = vertices[2][i];
		}
	}
	else if (!Array.isArray(faceVertexCData[0])) { //indexed plot - faceVertexCData = [0, 2.32, 231.1, ...]
		if (faceVertexCData.length === faces.length) { //element plot
			vertexArray = new Float32Array(faces.length * 3 * 3);
			for(var i = 0, v = 0, t = 0; i < faces.length; i++) {
				face = [faces[i][0] - 1, faces[i][1] - 1, faces[i][2] - 1];
				for(var j = 0; j < face.length; j++) {
					if (isNaN(faceVertexCData[i])) {
						this.nanIndices.add(i);
						vertexArray[v++] = 0.0;
						vertexArray[v++] = 0.0;
						vertexArray[v++] = 0.0;
					}
					else {
						vertexArray[v++] = vertices[0][face[j]];
						vertexArray[v++] = vertices[1][face[j]];
						vertexArray[v++] = vertices[2][face[j]];
					}
				}
			}
			this.computeIndices = false;
		}
		else if (faceVertexCData.length === vertices[0].length) { //node plot
			vertexArray = new Float32Array(vertices[0].length * 3);
			for(var i = 0, v = 0, t = 0; i < vertices[0].length; i++) {
				if (isNaN(faceVertexCData[i])) {
					this.nanIndices.add(i);
					vertexArray[v++] = 0.0;
					vertexArray[v++] = 0.0;
					vertexArray[v++] = 0.0;
				}
				else {
					vertexArray[v++] = vertices[0][i];
					vertexArray[v++] = vertices[1][i];
					vertexArray[v++] = vertices[2][i];
				}
			}
		}
	}
	else if (Array.isArray(faceVertexCData[0])) { //JS specific: precomputed UV coord plot - faceVertexCData = [[0, 0.99, 0.4, ...],[0, 0.99, 0.4, ...]]
		if (faceVertexCData[0].length === faces.length) { //element plot
			vertexArray = new Float32Array(faces.length * 3 * 3);
			for(var i = 0, v = 0, t = 0; i < faces.length; i++) {
				face = [faces[i][0] - 1, faces[i][1] - 1, faces[i][2] - 1];
				for(var j = 0; j < face.length; j++) {					
					vertexArray[v++] = vertices[0][face[j]];
					vertexArray[v++] = vertices[1][face[j]];
					vertexArray[v++] = vertices[2][face[j]];
				}
			}
			this.computeIndices = false;
		}
		else if (faceVertexCData[0].length === vertices[0].length) { //node plot
			vertexArray = new Float32Array(vertices[0].length * 3);
			for(var i = 0, v = 0, t = 0; i < vertices[0].length; i++) {
				vertexArray[v++] = vertices[0][i];
				vertexArray[v++] = vertices[1][i];
				vertexArray[v++] = vertices[2][i];
			}
		}
	}
	this.arrays.vertices = vertexArray;
} //}}}
Node.prototype.patchCoords = function(faceVertexCData, faces, vertices) { //{{{
	//Patch subfunction for processing texture coords/UVs.
	var coordArray;
	var cramge;
	var caxis = this.caxis;
	var face;
	
	if (isEmptyOrUndefined(faceVertexCData)) { return; }
	
	//Use logarithmic scaling if it is valid
	if (this.log !== false && this.log !== 'off') {	
		caxis = [
			Math.log10(caxis[0]) / Math.log10(this.log), 
			Math.log10(caxis[1]) / Math.log10(this.log)
		];
	}
	
	if (!Array.isArray(faceVertexCData[0])) { //indexed plot - faceVertexCData = [0, 2.32, 231.1, ...]
		if (faceVertexCData.length === faces.length) { //element plot
			coordArray = new Float32Array(faces.length * 3 * 2);
			crange = caxis[1] - caxis[0];
			for(var i = 0, v = 0, t = 0; i < faces.length; i++) {
				face = [faces[i][0] - 1, faces[i][1] - 1, faces[i][2] - 1];
				for(var j = 0; j < face.length; j++) {
					if (isNaN(faceVertexCData[i])) {
						this.nanIndices.add(i);
						coordArray[t++] = 0.0;
						coordArray[t++] = 0.0;
					}
					else {
						coordArray[t++] = 0.5;
						coordArray[t++] = clamp((faceVertexCData[i] - caxis[0]) / crange, 0.0, 1.0);
					}
				}
			}
			this.computeIndices = false;
		}
		else if (faceVertexCData.length === vertices[0].length) { //node plot
			coordArray = new Float32Array(vertices[0].length * 2);
			crange = caxis[1] - caxis[0];
			for(var i = 0, v = 0, t = 0; i < vertices[0].length; i++) {
				if (isNaN(faceVertexCData[i])) {
					this.nanIndices.add(i);
					coordArray[t++] = 0.0;
					coordArray[t++] = 0.0;
				}
				else {
					coordArray[t++] = 0.5;
					coordArray[t++] = clamp((faceVertexCData[i] - caxis[0]) / crange, 0.0, 1.0);
				}
			}
		}
	}
	else if (Array.isArray(faceVertexCData[0])) { //JS specific: precomputed UV coord plot - faceVertexCData = [[0, 0.99, 0.4, ...],[0, 0.99, 0.4, ...]]
		if (faceVertexCData[0].length === faces.length) { //element plot
			coordArray = new Float32Array(faces.length * 3 * 2);
			for(var i = 0, v = 0, t = 0; i < faces.length; i++) {
				face = [faces[i][0] - 1, faces[i][1] - 1, faces[i][2] - 1];
				for(var j = 0; j < face.length; j++) {					
					coordArray[t++] = faceVertexCData[0][i];
					coordArray[t++] = faceVertexCData[1][i];
				}
			}
			this.computeIndices = false;
		}
		else if (faceVertexCData[0].length === vertices[0].length) { //node plot
			coordArray = new Float32Array(vertices[0].length * 2);
			for(var i = 0, v = 0, t = 0; i < vertices[0].length; i++) {
				coordArray[t++] = faceVertexCData[0][i];
				coordArray[t++] = faceVertexCData[1][i];
			}
		}
	}
	this.arrays.coords = coordArray;
} //}}}
Node.prototype.patchIndices = function(faces, faceColor) { //{{{
	//Patch subfunction for processing faces/elements/triangles/indices.
	var indexArray;
	var face;
	
	if (faceColor === 'none') { //Check for wireframe mesh rendering
		if (this.drawMode === this.gl.TRIANGLES) { //NOTE: Stopgap to allow gl.LINE_STRIP nodes render normally. Only use case for faceColor === 'none' is for plot_mesh
			this.drawMode = this.gl.LINES;
		}
	}
	
	if (this.computeIndices === true && !isEmptyOrUndefined(faces)) {
		if (!isEmptyOrUndefined(faces[0])) { //Check for 2D format and process if needed
			if (faceColor !== 'none') { //Check for triangle rendering
				indexArray = new Uint16Array(faces.length * 3);
				for(var i = 0, f = 0; i < faces.length; i++) {
					face = [faces[i][0] - 1, faces[i][1] - 1, faces[i][2] - 1];
					if (this.nanIndices.has(face[0]) || this.nanIndices.has(face[1]) || this.nanIndices.has(face[2])) continue; //Skip triangle if contains NaN value.
					indexArray[f++] = faces[i][0] - 1;
					indexArray[f++] = faces[i][1] - 1;
					indexArray[f++] = faces[i][2] - 1;
				}
			}
			else { //Check for wireframe mesh rendering
				indexArray = new Uint16Array(faces.length * 6);
				for(var i = 0, f = 0; i < faces.length; i++) {
					indexArray[f++] = faces[i][0] - 1;
					indexArray[f++] = faces[i][1] - 1;
					indexArray[f++] = faces[i][1] - 1;
					indexArray[f++] = faces[i][2] - 1;
					indexArray[f++] = faces[i][2] - 1;
					indexArray[f++] = faces[i][0] - 1;
				}
			}
		}
		else { //Else, assume face indices have already been processed
			indexArray = faces;
		}
		this.arrays.triangles = indexArray;
	}
} //}}}
Node.prototype.updateBuffer = function() { //{{{
	//Updates the mesh buffers provided in place.
	//NOTE: Only support coord buffers currently.
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(0,args.length));

	var coords = options.getfieldvalue('Coords', undefined);
	var cacheIndex = options.getfieldvalue('CacheIndex', false);
	
	if (!isEmptyOrUndefined(coords)) {
		this.patchCoords(coords, this.faces, this.vertices);
		var buffer = this.mesh.getBuffer("coords");
		buffer.data = this.arrays.coords;
		buffer.upload(this.gl.DYNAMIC_DRAW);
	}
} //}}}
Node.prototype.computeOctree = function() { //{{{
	//Computes and caches octrees for a node.
	var octree = this.canvas.octrees[this.name];
	if (isEmptyOrUndefined(octree)) {
		octree = new GL.Octree(this.mesh);
	}
	this.canvas.octrees[this.name] = octree;
	this.octree = octree;
} //}}}
Node.prototype.geometryShader = function() { //{{{
	//Emulates OpenGL geometry shaders by rendering each point as a mesh.
	//Parameters:
	//	Mesh - the geometry to duplicate for each point.
	//	Vertices - the list of points to shade.
	//	Indices - (optional) ordered list for non-ordered Vertices objects.
	
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(0,args.length));

	var mesh = options.getfieldvalue('Mesh', undefined);
	var vertices = options.getfieldvalue('Vertices', undefined);
	var indices = options.getfieldvalue('Indices', undefined);
	
	//For handling key-value object arrays, like xcity
	for (var i = 0; i < vertices.length; i++) {
		if (!Array.isArray(vertices[i])) {
			var array = [];
			var coordinateObject = vertices[i];
			var j = 0;
			if (isEmptyOrUndefined(indices)) {
				for (var key in object) {
					array[j++] = coordinateObject[key];
				}
			}
			else {
				for (var k = 0; k < indices.length; k++) {
					array[j++] = coordinateObject[indices[k]];
				}
			}
			vertices[i] = array;
		}
	}
	
	var x = vertices[0];
	var y = vertices[1];
	var z = vertices[2];
	var meshVertices = mesh.getBuffer('vertices').data;
	var meshIndicies = mesh.getIndexBuffer('triangles').data;
	var indicies = new Uint16Array(meshIndicies.length * x.length);
	var size = meshVertices.length * x.length / 3;
	newX = new Float32Array(size);
	newY = new Float32Array(size);
	newZ = new Float32Array(size);
	
	//For each vertex in vertices, instantiate mesh geomtry centered around that point.
	for(var i = 0, v = 0, e = 0; i < x.length; i++){
		var vector = [x[i], y[i], z[i]];
		for (var j = 0; j < meshVertices.length;) {
			newX[v] = meshVertices[j++] + x[i];
			newY[v] = meshVertices[j++] + y[i];
			newZ[v++] = meshVertices[j++] + z[i];
		}
		var offset = i * meshVertices.length / 3;
		for (var j = 0; j < meshIndicies.length;) {
			indicies[e++] = meshIndicies[j++] + offset;
		}
	}
	
	this.patch('Faces', indicies, 'Vertices', [newX, newY, newZ], 'FaceColor', 'interp');
} //}}}
Node.prototype.scaleVertices = function(md, x, y, z, scale) { //{{{
	//Scales and returns vertices x, y, and z by factor scale. Uses md.geometry.scale for heightscaling in 3d meshes.
	if (md.mesh.classname() === 'mesh3dsurface') {
		var xyz, magnitude;
		x = x.slice();
		y = y.slice();
		z = z.slice();
		for(var i = 0; i < x.length; i++) {
			xyz = vec3.fromValues(x[i], y[i], z[i]);
			magnitude = 1 + md.geometry.surface[i] * scale / vec3.length(xyz);
			vec3.scale(xyz, xyz, magnitude);
			x[i] = xyz[0];
			y[i] = xyz[1];
			z[i] = xyz[2];
		}
	}
	else {
		z = z.slice();
		var zMin = ArrayMin(md.geometry.surface);
		for(var i = 0; i < z.length; i++) {
			z[i] = (z[i] - zMin) * scale + zMin;
		}
	}
	return [x, y, z];
} //}}}
Node.prototype.mergeVertices = function(x1, y1, z1, elements1, x2, y2, z2, elements2) { //{{{
	//Merges and returns two sets of indexed xyz vertices.
	elements2 = elements2.slice();
	for (var i = 0, offset = x1.length; i < elements2.length; i++) {
		elements2[i] = [elements2[i][0] + offset, elements2[i][1] + offset, elements2[i][2] + offset];
	}
	return {x:x1.concat(x2), y:y1.concat(y2), z:z1.concat(z2), elements:elements1.concat(elements2)};
} //}}}
//}}}
