function md=runme_core(varargin)
%RUNME_CORE - test deck for ISSM nightly runs
%
%   Usage:
%      md=runme_core(varargin);
%
%   Examples:
%      md=runme_core('parallel',1,'update');
%
%   See Also: UPDATEARCHIVE RUNME

% This file can be run to check that the current version of issm is giving 
% coherent results.
% is square. Just run this file in Matlab, with a properly setup ISSM code. 
% The results of this test will indicate if there is a difference between current computations 
% and archived results.

% Errors  between archived results and the current version will get flagged if they are not within 
% a certain tolerance.

%recover options
options=recover_options(varargin{:});

%check arguments
if nargout~=1
	help runme_core
	error('runme_core error message: bad usage');
end

%warning if a runme_core already exists:
if exist('./runme.m')
	disp('this is the nighlty run runme_core, located in src/m/utils/Nightly/runme_core.m')
	disp('Another local runme exist but is not executed')
end

%read configuration
if ~exist('./configuration.m')
	error('runme_core error message: configuration file ''configuration.m'' not found');
else
	configuration;
end

%go through the sequences requested.
for i=1:size(sequences,1),
	sequence=sequences{i};

	%recover what is being requested.
	analysis_type=sequence{1};
	sub_analysis_type=sequence{2};
	qmu=sequence{3};
	control=sequence{4};
	control_fit=sequence{5};
	parallel=sequence{6};

	%CHECK analysis_type
	analysis_type_arg=find_option(options,'analysis_type');
	if ~isempty(analysis_type_arg),
		if ~ismember(analysis_type,analysis_type_arg)
			continue
		end
	end

	%CHECK sub_analysis_type
	sub_analysis_type_arg=find_option(options,'sub_analysis_type');
	if ~isempty(sub_analysis_type_arg),
		if ~ismember(sub_analysis_type,sub_analysis_type_arg)
			continue
		end
	end

	%CHECK qmu
	qmu_arg=find_option(options,'qmu');
	if ~isempty(qmu_arg),
		if qmu~=qmu_arg,
			continue
		end
	end

	%CHECK control
	control_arg=find_option(options,'control');
	if ~isempty(control_arg),
		if control~=control_arg,
			continue
		end
	end

	%CHECK control_fit
	control_fit_arg=find_option(options,'control_fit');
	if ~isempty(control_fit_arg),
		if ~ismember(control_fit,control_fit_arg)
			continue
		end
	end

	%CHECK parallel
	parallel_arg=find_option(options,'parallel');
	if ~isempty(parallel_arg),
		if parallel~=parallel_arg,
			continue
		end
	end

	%CHECK procedure
	procedure=find_option(options,'procedure');
	if isempty(procedure),
		disp('runme_core warning: no procedure found, defaulting to test checking')
		procedure='check';
	end

	%initialize model, using presolve if need be, or just the piece of generic code below.
	md=model;
	if ~rifts,
		md=mesh(md,domainname,resolution);
	else
		md=mesh(md,domainname,riftname,resolution);
		md=meshprocessrifts(md);
	end
	md=geography(md,iceshelfname,icesheetname);
	md=parameterize(md,parametername);
	md=extrude(md,numlayers,extrusionexponent);
	eval(['md=setelementstype(md,' elementstype ');']);

	if parallel, 
		%this run will be done in parallel
		md.cluster=oshostname;
		%check and modify number of cpus if necessary
		np=feature('numCores');
		if np<md.np;
			disp(['warning: changing number of cpus to ' num2str(np) ]);
			md.np=np;
		end
		runtype='parallel';
	else
		runtype='serial';
	end

	%some solution checks;
	if (  (strcmpi(analysis_type,'thermal') & strcmpi(sub_analysis_type,'steady')) | ...
			(strcmpi(analysis_type,'steadystate'))...
		)
		md.dt=0;
	end

	if control,
		md.control_analysis=1;
		if strcmpi(control_fit,'absolute'),
			md.fit(:)=0;
		elseif strcmpi(control_fit,'relative'),
			md.fit(:)=1;
		elseif strcmpi(control_fit,'logarithmic'),
			md.fit(:)=2;
		else
			error(['runme_core error message: for control solution, ' control_fit ' fit is not supported!']);
		end
	end

	%try to run the test
	try,

		%presolve
		if exist('./testpresolve.m','file')==2,
			testpresolve;
		end

		%solve
		md=solve(md,'analysis_type',analysis_type,'sub_analysis_type',sub_analysis_type);

		%postsolve
		if exist('./testpostsolve.m','file')==2,
			testpostsolve;
		end

		%compute fields to be checked
		fields=testsgetfields2(md,sequence);

		%get Archive name
		archive_name='Archive';
		for i=1:length(sequence),
			if ischar(sequence{i}),
				archive_name=[archive_name '_' sequence{i}];
			else
				archive_name=[archive_name '_' num2str(sequence{i})];
			end
		end

		%CHECK TEST?
		if strcmpi(testtype,'check'),

			%load archive
			eval(['load ' archive_name ]);

			for k=1:length(fields),

				%Get field and tolerance
				field=fields{k};
				tolerance=testsgettolerance2(md,sequence,field);

				%compare to archive
				eval(['Archive=' archive_name '_field' num2str(k) ';']);
				eval(['error_diff=full(max(abs(Archive-md.results.' field '))/(max(abs(Archive))+eps));']);

				%disp test result
				if (error_diff>tolerance);
					disp(sprintf(['\nERROR   difference: %-7.2g > %7.2g test: %-25s solution: %-16s type: %-9s field: ' field  '.\n\n'],error_diff,tolerance,testname,[analysis_type '_' sub_analysis_type],runtype));
				else
					disp(sprintf(['\nSUCCESS difference: %-7.2g < %7.2g test: %-25s solution: %-16s type: %-9s field: ' field  '.\n\n'],error_diff,tolerance,testname,[analysis_type '_' sub_analysis_type],runtype));
				end
			end

		%UPDATE ARCHIVE?
		elseif strcmpi(testtype,'update'),

			for k=1:length(fields),
				field=fields{k};
				eval([ archive_name '_field' num2str(k) ' = md.results. ' field  ';']);
			end
			eval(['save ' archive_name ' ' archive_name '_field*']);
			disp(sprintf(['File ' archive_name ' saved\n']));

		else

			error('runme_core error message: unknown runtype. should be ''check'' or ''update''');

		end

	catch me,

		%something went wrong, print failure message:
		message=getReport(me)
		disp(sprintf(['\n\nFAILURE difference: %-16s test: %-25s solution: %-16s type: %-9s field: N/A.\n\n'],'N/A',testname,[analysis_type '_' sub_analysis_type],runtype));
	end
end
