
/*
	Matice.c
*/


#include <stdio.h>
#include <string.h>	
#include <math.h>

/* environment: */
#include "../include/cielo_types.h"
#include "../include/cielo_macros.h"
#include "../include/cielo_externals.h"

/* hosting environment: */
#include "../include/matlab_includes.h"

/* object handlers: */
#include "../DataSet/DataSet.h"
#include "../EnumDefinitions/EnumDefinitions.h"

/* shared utilities (includes alloc.h memory management): */
#include "../shared/shared.h"


/* "virtual function" access: */
#include "./VirtualFunctions.h"

#include "./Matice.h"


/*--------------------------------------------------
	NewMatice
  --------------------------------------------------*/

Matice* NewMatice(void) {

	return (Matice*)xmalloc(sizeof(Matice));
}


/*--------------------------------------------------
	MaticeInit
  --------------------------------------------------*/

int MaticeInit(Matice* this)
{

	this->name[0]	= '\0';
	this->mid 		= UNDEF;
	this->B			= UNDEF;
	this->n			= UNDEF;

	return 1;
}


/*--------------------------------------------------
	DeleteMatice
  --------------------------------------------------*/

void DeleteMatice(void* *vpthis)
{
	/* vpthis for polymorphic function compatibility */

	Matice* *pthis = (Matice**)vpthis;
	Matice*   this = *pthis;

	#ifdef _DEBUG_
	_printf_("   DeleteMatice message:\n");
	_printf_("   deleting Matice with mid = %d\n", this->mid);
	#endif

	xfree(vpthis);
}


/*--------------------------------------------------
	MaticeEcho
  --------------------------------------------------*/

void MaticeEcho(void* vpthis)
{
	/* vpthis for polymorphic function compatibility */

	Matice* this = (Matice*)vpthis;

	if (this) {

		printf("\nMatice record:\n");
		printf("   name: %s\n", this->name);

		if ( UNDEF!=this->mid) 		printf("    mid: %d\n", this->mid );
		else                   		printf("    mid: UNDEF\n");

		if( UNDEF!=(float)this->B)  printf("      B: %lf\n",this->B   );
		else                      	printf("      B: UNDEF\n");

		if( UNDEF!=(float)this->n)printf(  "      n: %g\n",this->n   );
		else                      	printf("      n: UNDEF\n");
	
	}
	else {

		printf("   null pointer input for Matice object; no echo\n");
	}
}


/*--------------------------------------------------
	MaticeSizeOfInDoubles
  --------------------------------------------------*/

int MaticeSizeOfInDoubles(void* vpthis)
{
	/* vpthis for polymorphic function compatibility */

	Matice* this = (Matice*)vpthis;
	int struct_doubles=0;

	struct_doubles = 
		(int)ceil((float)sizeof(*this)/(float)sizeof(double));

	#ifdef _DEBUG_
	_printf_("\n   MaticeSizeOfInDoubles diagnostics:\n");
	_printf_("      returning %d doubles\n", struct_doubles);
	#endif

	return struct_doubles;
}


/*--------------------------------------------------
	MaticeShrinkWrap
  --------------------------------------------------*/

void MaticeShrinkWrap(void* to, void* vpthis)
{
	/* no "extended" data means a simple copy will do: */

	memcpy( to, vpthis, sizeof(double)*MaticeSizeOfInDoubles(vpthis));
}


/*--------------------------------------------------
	MaticeUnwrap
  --------------------------------------------------*/

int  MaticeUnwrap(void* vpthis)
{
	/* no extended data means no internal pointers to reestablish  */

	return 1;
}


/*--------------------------------------------------
	MaticeAddToDataSet
  --------------------------------------------------*/

int MaticeAddToDataSet( DataSet* dataset, Matice* this)
{
	/*

	Provided as an alternate interface to DataSetAddObject.

	In the c++ world, the dataset could just add an object to its list
	using a dataset member function and a pointer to the object's base
	class.

	In the c world, however, only the object itself is capable of telling
	the dataset its type, hence the structure here; a call to a generic
	DataSetAddObject function via an object "member function."

	*/

	DataSetAddObject( dataset, MaticeEnum(), (void*)this,
		&DeleteMatice, &MaticeEcho, &MaticeSizeOfInDoubles,
		&MaticeShrinkWrap, &MaticeUnwrap,
		&MaticeMarshall,
		&MaticeGetID, &MaticeSetVirtualFunctions);
	
	return 1;
}


/*--------------------------------------------------
	MaticeGetID
  --------------------------------------------------*/

int MaticeGetID( void* vpthis )
{
	/* vpthis for polymorphic function compatibility */

	Matice* this = (Matice*)vpthis;

	return this->mid;
}


/*--------------------------------------------------
    MaticeGetName
  --------------------------------------------------*/

char* MaticeGetName( void* vpthis)
{
    Matice* this = (Matice*)vpthis;

    #ifdef _DEBUG_
    _printf_("virtual function: MaticeGetName\n");
    #endif

    return this->name;
}


/*--------------------------------------------------
    MaticeSetVirtualFunctions
  --------------------------------------------------*/

int MaticeSetVirtualFunctions( void* vp)
{

    /* VirtualFunctions implemented as a "friend" class: */

    /* void* for polymorphic function compatibility */
    VirtualFunctions* vf = (VirtualFunctions*)vp;
    
    if ( VirtualFunctionsInit(vf) ) {

        /* set Matice-specific virtual function pointers: */

        /* general: */

        vf->GetName          = &MaticeGetName;

        return 1;
    }
    else
        return 0;

}


/*--------------------------------------------------
	MaticeMarshall
  --------------------------------------------------*/

void MaticeMarshall( char* *pbuffer, void* vpthis, int* size)
{
	/* vpthis for polymorphic function compatibility */
	Matice* this = (Matice*)vpthis;

	char* buffer=*pbuffer;

	int size_string1; 
	
	size_string1=strlen(this->name)+1;

	cellmar(&size_string1,&buffer,INTEGER_FLAG);
	cellmar(&this->name,&buffer,STRING_FLAG);
	cellmar(&this->mid,&buffer,INTEGER_FLAG);
	cellmar(&this->B,&buffer,DOUBLE_FLAG);
	cellmar(&this->n,&buffer,DOUBLE_FLAG);

	/* return values: */

	*size=size_string1*sizeof(char)+ 2*sizeof(int)+2*sizeof(double);
	*pbuffer=buffer; 

	return;
}


/*--------------------------------------------------
	MaticeDemarshall
  --------------------------------------------------*/

void* MaticeDemarshall( DataSet* ds, char* *pbuffer, int machine_flag)
{ 
	int size_string1;

	char* buffer=*pbuffer; 
	Matice* this=NewMatice();

	celldem(&size_string1,&buffer,INTEGER_FLAG,machine_flag);
	celldem(&this->name,&buffer,STRING_FLAG,machine_flag,size_string1);
	celldem(&this->mid,&buffer,INTEGER_FLAG,machine_flag);
	celldem(&this->B,&buffer,DOUBLE_FLAG,machine_flag);
	celldem(&this->n,&buffer,DOUBLE_FLAG,machine_flag);

	if (ds) MaticeAddToDataSet( ds, this);
	*pbuffer=buffer;  

	return (void*)this;
}
	
/*--------------------------------------------------
	MaticeGetViscosity2d
  --------------------------------------------------*/
int MaticeGetViscosity2d(double* pmu, void* vpthis, double* epsilon){

/*From a string tensor and a material object, return viscosity, using Glen's flow law.
	                                  2*B
	  mu= -------------------------------------------------------------------
	      2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]
	 
	      where mu is the viscotiy, B the flow law parameter , (u,v) the velocity 
	      vector, and n the flow law exponent.
	 
	  If epsilon is NULL, it means this is the first time Emg is being run, and we 
	  return 10^14, initial viscosity.
*/

	int noerr=1;
	
	/*output: */
	double mu;

	/*input from the material card: */
	Matice* this=NULL; /* vpthis for polymorphic function compatibility */
	double B,n;

	/*input strain rate: */
	double exx,eyy,exy;

	/*Intermediary value A and exponent e: */
	double A,e;

	/*Retrieve  material card info: */
	this=(Matice*)vpthis;


	if (this->n==1){
		/*Viscous behaviour! mu=B: */
		mu=this->B;
	}
	else{
		if((epsilon[0]==0) && (epsilon[1]==0) && (epsilon[2]==0)){
			mu=pow(10,14);
		}
		else{
			B=this->B;
			n=this->n;

			/*Retrive strain rate components: */
			exx=*(epsilon+0);
			eyy=*(epsilon+1);
			exy=*(epsilon+2);

			/*Build viscosity: mu=2*B/(2*A^e) */
			A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+exx*eyy;
			if(A==0){
				/*Maximum viscosity for 0 shear areas: */
				mu=4.5*pow(10,17);
			}
			else{
				e=(n-1)/2/n;
			
				mu=2*B/(2*pow(A,e));
			}
		}
	}
	#ifdef _DEBUG_
	_printf_("Viscosity %lf\n",mu);
	#endif

	/*Return: */
	*pmu=mu;
	return noerr;
}

/*--------------------------------------------------
	MaticeGetViscosity3d
  --------------------------------------------------*/
int MaticeGetViscosity3d(double* pmu, void* vpthis, double* epsilon){

	/*Return viscosity accounting for steady state power law creep [Thomas and MacAyeal, 1982]: 
	 *
	 *                                 2*B
	 * mu= -------------------------------------------------------------------
	 *     2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]
	 *
	 *     where mu is the viscotiy, B the flow law parameter , (u,v) the velocity 
	 *     vector, and n the flow law exponent.
	 *
	 * If epsilon is NULL, it means this is the first time Emg is being run, and we 
	 * return g, initial viscosity.
	 */
	
	int noerr=1;
	
	/*output: */
	double mu;

	/*input from the material card: */
	Matice* this=NULL; /* vpthis for polymorphic function compatibility */
	double B,n;

	/*input strain rate: */
	double exx,eyy,exy,exz,eyz;

	/*Intermediary value A and exponent e: */
	double A,e;

	/*Retrieve  material card info: */
	this=(Matice*)vpthis;

	if (this->n==1){
		/*Viscous behaviour! mu=B: */
		mu=this->B;
	}
	else{
		if((epsilon[0]==0) && (epsilon[1]==0) && (epsilon[2]==0) && 
				(epsilon[3]==0) && (epsilon[4]==0)){
			mu=pow(10,14);
		}
		else{

			B=this->B;
			n=this->n;

			/*Retrive strain rate components: */
			exx=*(epsilon+0);
			eyy=*(epsilon+1);
			exy=*(epsilon+2);
			exz=*(epsilon+3);
			eyz=*(epsilon+4);

			/*Build viscosity: mu=2*B/(2*A^e) */
			A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+pow(exz,2)+pow(eyz,2)+exx*eyy;
			if(A==0){
				/*Maximum viscosity for 0 shear areas: */
				mu=4.5*pow(10,17);
			}
			else{
				e=(n-1)/2/n;
			
				mu=2*B/(2*pow(A,e));
			}
		}
	}
	#ifdef _DEBUG_
	_printf_("Viscosity %lf\n",mu);
	#endif

	/*Return: */
	*pmu=mu;
	return noerr;
}

/*--------------------------------------------------
	MaticeGetViscosity2
  --------------------------------------------------*/
int MaticeGetViscosity2(double* pmu2, void* vpthis, double* epsilon){

	/*Return viscosity accounting for steady state power law creep [Thomas and MacAyeal, 1982]: 
	 *
	 *                                  2* (1-n)/2n
	 * mu2= -------------------------------------------------------------------
	 *     2[ (du/dx)^2+(dv/dy)^2+1/4*(du/dy+dv/dx)^2+du/dx*dv/dy ]^[(3n-1)/2n]
	 *
	 *     where mu2 is the second viscosity, (u,v) the velocity 
	 *     vector, and n the flow law exponent.
	 *
	 * If epsilon is NULL, it means this is the first time Gradjb is being run, and we 
	 * return mu20, initial viscosity.
	 */
	
	int noerr=1;
	
	/*output: */
	double mu2;

	/*input from the material card: */
	Matice* this=NULL; /* vpthis for polymorphic function compatibility */
	double n;

	/*input strain rate: */
	double exx,eyy,exy;

	/*Intermediary value A and exponent e: */
	double A,e;

	/*Retrieve  material card info: */
	this=(Matice*)vpthis;

	if(epsilon){
		n=this->n;

		/*Retrive strain rate components: */
		exx=*(epsilon+0);
		eyy=*(epsilon+1);
		exy=*(epsilon+2);

		/*Build viscosity: mu2=B/(2*A^e) */
		A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+exx*eyy;
		if(A==0){
			/*Maximum viscosity2 for 0 shear areas: */
			mu2=4.5*pow(10,17);
		}
		else{
			e=(3*n-1)/2/n;
		
			mu2=2*(1-n)/2/n/(2*pow(A,e));
		}
	}
	else{
		mu2=4.5*pow(10,17);
	}
		
	#ifdef _DEBUG_
	_printf_("Viscosity2 %lf\n",mu2);
	#endif

	/*Return: */
	*pmu2=mu2;
	return noerr;
}

/*--------------------------------------------------
	MaticeSetFlowLawParameter
  --------------------------------------------------*/
int MaticeSetFlowLawParameter(void* vpthis, double B_value){

	Matice* this=NULL; /* vpthis for polymorphic function compatibility */
	this=(Matice*)vpthis;

	this->B=B_value;
	return 1;
}
