
/*
	PengridLoad.c: 
*/

#include <stdarg.h>
#include <stdio.h>
#include <string.h>	
#include <math.h>

/* environment: */
#include "../../include/cielo_types.h"
#include "../../include/cielo_macros.h"
#include "../../include/cielo_externals.h"

/* hosting environment: */
#include "../../include/matlab_includes.h"

/* object handlers: */
#include "../../DataSet/DataSet.h"
#include "../../EnumDefinitions/EnumDefinitions.h"

/* shared utilities (includes alloc.h memory management): */
#include "../../shared/shared.h"

/* "virtual function" access: */
#include "./../VirtualFunctions.h"

/* object itself: */
#include "./PengridLoad.h"

/*Objects: */
#include "../objects.h"

/*--------------------------------------------------
	NewPengridLoad
  --------------------------------------------------*/

PengridLoad* NewPengridLoad( void* vppengrid)
{

	Pengrid* 		pengrid = (Pengrid*)vppengrid;
	PengridLoad*	this   = NULL;
	int	i,j;

	this=(PengridLoad*)xcalloc(1,sizeof(PengridLoad));	/* guaranteed NULL, 0 initialisation */

	/* include the "base class" in the "derived" object: */
	memcpy( &this->pengrid, pengrid, sizeof(Pengrid));

	/* since zero could be a valid index, set all DataSet index
	"pointers" to UNDEF: */
	this->internalgrid_index=UNDEF;
	this->internalgrid_indexb=UNDEF;
	this->pg=NULL;

	this->matpar = NULL;
	this->matpar_index=0;
	this->matpar_enum=0;

	return this;
}


/*--------------------------------------------------
	DeletePengridLoad
  --------------------------------------------------*/

void DeletePengridLoad(void* *vpthis)
{
	/* vpthis for polymorphic function compatibility */

	PengridLoad* *pthis = (PengridLoad**)vpthis;
	PengridLoad* this = *pthis;

	xfree(vpthis);
}


/*--------------------------------------------------
	PengridLoadEcho
  --------------------------------------------------*/

void PengridLoadEcho(void* vpthis){
	int i;
	
	/* vpthis for polymorphic function compatibility */
	PengridLoad* this = (PengridLoad*)vpthis;

	if (this) {

		_printf_("\nPengridLoad echo:\n");
		printf("-------------------\n");
		printf("   base Pengrid object:\n");
		PengridEcho( &this->pengrid);
		printf("\n");
		_printf_("    penalty grid points to following grids (pointer-based ops):\n");
		_printf_("    local grid number   corresponding grid id\n");
		_printf_("    -----------------   ---------------------\n");
		if ( this->pg)
			_printf_("        %8d              %8d\n", i, InternalGridGetID( this->pg));
		if ( this->matpar) printf("   matpar entry of enum type %d\n", this->matpar_enum);
		else			   printf("   matpar reference is NULL\n");

	}
	else{
		_printf_("   null pointer input for PengridLoad object; no echo\n");
	}
}

/*--------------------------------------------------
	PengridLoadSizeOfInDoubles
  --------------------------------------------------*/

int PengridLoadSizeOfInDoubles(void* vpthis)
{
	/* vpthis for polymorphic function compatibility */

	PengridLoad* this = (PengridLoad*)vpthis;
	int struct_doubles=0;

	struct_doubles = 
		(int)ceil((float)sizeof(*this)/(float)sizeof(double));

	#ifdef _DEBUG_
	_printf_("\n   PengridLoadSizeOfInDoubles diagnostics:\n");
	_printf_("      returning %d doubles\n", struct_doubles);
	#endif

	return struct_doubles;
}


/*--------------------------------------------------
	PengridLoadShrinkWrap
  --------------------------------------------------*/

void PengridLoadShrinkWrap(void* to, void* vpthis)
{
	/* no "extended" data means a simple copy will do: */

	memcpy( to, vpthis, sizeof(double)*PengridLoadSizeOfInDoubles(vpthis));
}


/*--------------------------------------------------
	PengridLoadUnwrap
  --------------------------------------------------*/

int  PengridLoadUnwrap(void* vpthis)
{
	PengridLoad* this = (PengridLoad*)vpthis;
	
	this->matpar = NULL;
	
	return 1;
}

/*--------------------------------------------------
	PengridLoadMarshall
  --------------------------------------------------*/

void PengridLoadMarshall( char* *pbuffer, void* vpthis, int* size)
{
	/* vpthis for polymorphic function compatibility */
	PengridLoad* this = (PengridLoad*)vpthis;

	char* buffer=*pbuffer;

	int i, j, size_pengrid=0; 
	int ig;

	int num_ints=0, num_doubles=0;


	/* first marshall the embedded Pengrid object: ... */

	PengridMarshall( &buffer, &this->pengrid, &size_pengrid);


	/* ... then marshall the PengridLoad member data: */

	cellmar( &this->internalgrid_index, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->internalgrid_indexb, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->pg, &buffer, INTEGER_FLAG); num_ints++;

	cellmar( &this->matpar_index, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->matpar_enum, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->matpar, &buffer, INTEGER_FLAG); num_ints++;

	*size = size_pengrid + num_ints*sizeof(int) + num_doubles*sizeof(double);
	*pbuffer = buffer;

	return;
}


/*--------------------------------------------------
	PengridLoadDemarshall
  --------------------------------------------------*/

void* PengridLoadDemarshall( DataSet* ds, char* *pbuffer, int machine_flag)
{ 
	int i, j, ig;
	Pengrid* ppengrid=NULL;	/* for embedded object */

	char* buffer=*pbuffer; 
	PengridLoad* this=NULL;


	/* first demarshall the embedded Pengrid object: ... */

	ppengrid = PengridDemarshall( NULL, &buffer, machine_flag);
	/* create the derived object: */
	this = NewPengridLoad( ppengrid);
	/* and free pengrid now that it's embedded in the derived object: */
	DeletePengrid( (void**)&ppengrid);


	/* ... then demarshall the PengridLoad member data: */

	celldem( &this->internalgrid_index, &buffer, INTEGER_FLAG, machine_flag);
	celldem( &this->internalgrid_indexb, &buffer, INTEGER_FLAG, machine_flag);
	celldem( &this->pg, &buffer, INTEGER_FLAG, machine_flag);

	celldem( &this->matpar_index, &buffer, INTEGER_FLAG, machine_flag);
	celldem( &this->matpar_enum, &buffer, INTEGER_FLAG, machine_flag);
	celldem( &this->matpar, &buffer, INTEGER_FLAG, machine_flag);

	/*Set pointers to NULL :*/
	this->pg=NULL;
	this->matpar=NULL;
	
	if (ds) PengridLoadAddToDataSet( ds, this);
	*pbuffer=buffer;  

	return (void*)this;
}	

/*--------------------------------------------------
	PengridLoadGetID
  --------------------------------------------------*/

int PengridLoadGetID( void* vpthis )
{
	/* vpthis for polymorphic function compatibility */
	PengridLoad* this = (PengridLoad*)vpthis;

	return PengridGetID( &this->pengrid);
}

/*--------------------------------------------------
    PengridLoadSetVirtualFunctions
  --------------------------------------------------*/

int PengridLoadSetVirtualFunctions( void* vp)
{

    /* VirtualFunctions implemented as a "friend" class: */

    /* void* for polymorphic function compatibility */
    VirtualFunctions* vf = (VirtualFunctions*)vp;
    
    if ( VirtualFunctionsInit(vf) ) {

        /* set PengridLoad-specific virtual function pointers: */

        /* general: */

		/* configuration: */
        vf->Configure          = &PengridLoadConfigure;

		/* penalty stiffnesses and loads: */
		vf->CreateKMatrix        = NULL;
		vf->PenaltyCreateKMatrix = &PengridLoadPenaltyCreateKMatrix;
		vf->CreatePVector        = NULL;
		vf->PenaltyCreatePVector = &PengridLoadPenaltyCreatePVector;
		vf->PenaltyConstrain     = &PengridLoadPenaltyConstrain;

        return 1;
    }
    else
        return 0;

}

/*--------------------------------------------------
	PengridLoadAddToDataSet
  --------------------------------------------------*/

int PengridLoadAddToDataSet( DataSet* dataset, PengridLoad* this)
{
	/*

	Provided as an alternate interface to DataSetAddObject.

	In the c++ world, the dataset could just add an object to its list
	using a dataset member function and a pointer to the object's base
	class.

	In the c world, however, only the object itself is capable of telling
	the dataset its type, hence the structure here; a call to a generic
	DataSetAddObject function via an object "member function."

	*/

	DataSetAddObject( dataset, PengridLoadEnum(), (void*)this,
		&DeletePengridLoad, &PengridLoadEcho, &PengridLoadSizeOfInDoubles,
		&PengridLoadShrinkWrap, &PengridLoadUnwrap,
		&PengridLoadMarshall,
		&PengridLoadGetID, &PengridLoadSetVirtualFunctions);
	
	return 1;
}


/*--------------------------------------------------
	PengridLoadConfigure
  --------------------------------------------------*/

int PengridLoadConfigure( void* vpthis, int num_datasets, ...) {

	/* set up data connections to all necessary, related objects: */

	PengridLoad* this = (PengridLoad*)vpthis;

	int	i,j, found, foundgrid, noerr=1, first=1;

	/* expected DataSet input: */

	DataSet*	pgpdt=NULL;
	DataSet*	pgpdtb=NULL;
	DataSet*	pmpt=NULL;

	/* variable argument list handling: */

	va_list args;
	DataSet* *arglist=NULL;

	#ifdef _DEBUG_
	_printf_("virtual function: PengridLoadConfigure\n");
	#endif
	if (this) {

		if (!num_datasets) {

			_printf_("   %s error:\n", __func__);
			_printf_("   null input for number of datasets;\n");
			_printf_("   could not establish grid connections.\n");
			return 0;
		}
		else {

			/* read, save dataset pointers for data connection operations: */
			arglist = xmalloc( num_datasets*sizeof(DataSet*));
			va_start( args, num_datasets);
			for ( i=0; i<num_datasets; i++) {
				*(arglist+i) = va_arg(args,DataSet*);
			}
			va_end(args);
		}

		#ifdef _DEBUG_
		_printf_("   variable argument list statistics:\n");
		_printf_("   input number of DataSets = %d\n", num_datasets);
		_printf_("   enum types of input DataSets:");
		for ( i=0; i<num_datasets; i++)  _printf_("   %d", DataSetGetType(*(arglist+i)));
		_printf_("\n");
		#endif


		/*
			establish grid connections for this PengridLoad object:
		*/

		/*Reinitialize this->pg*/
		this->pg=NULL;

		/* grid connections based on DataSet of type "gpdt": */

		#ifndef _PARALLEL_
		found=0;
		for ( i=0; i<num_datasets; i++) {
			if (*(arglist+i)==NULL)continue;
			if ( GpdtEnum() == DataSetGetType(*(arglist+i)) ) {
				pgpdt = *(arglist+i);
				found = 1;
				break;
			}
		}

		if (found) {

			int	current_grid_id;
			int* pgrid_id_list = PengridGetGridIDPtr(&this->pengrid);

			if ( (current_grid_id=pgrid_id_list[0]) ) {	/* assign and test presence */

				/* if grid pointer information has been established once already
				(e.g., by a previous module or operation), grid index information
				may already be present and valid.  check to see if corresponding
				pointer information can be reestablished without an expensive
				seek operation: */

				if ( UNDEF != (int)this->internalgrid_index ) {

					/* does index still point to a valid grid? (i.e., the one with
					the same id): */

					if ( InternalGridEnum() == DataSetGetEnum( pgpdt, this->internalgrid_index)
						 &&
						 current_grid_id == DataSetGetObjectID( pgpdt, this->internalgrid_index) ) {

						/* great, still valid!  since we have the index position in
						gpdt, just use it to get, save, the grid pointer: */

						this->pg = DataSetGetObjectPtr( pgpdt, this->internalgrid_index);
					}
				}

				/* if we haven't yet "recovered" the InternalGrid object pointer,
				go seek it out: */

				if ( !this->pg ) {

					noerr *= DataSetSeek( (void**)(&this->pg),	/* cast because some compilers recognise as */
																/* ptr to ptr to struct otherwise           */
								&this->internalgrid_index ,
								pgpdt, current_grid_id, 1, InternalGridEnum());

					if (!noerr) {
						if (first) {
							_printf_("   %s error:\n", __func__);
							first = 0;
						}
						_printf_("   could not find grid with id of %d for load %d of type \"%s\"\n",
							current_grid_id, PengridLoadGetID(this), PengridLoadGetName(this));
					}
				}

			}
		}
		else {

			if (first) {
				_printf_("   %s error:\n", __func__);
				first=0;
			}
			_printf_("   could not find DataSet of type \"gpdt\", grid object\n");
			_printf_("   pointers not established for pengrid load id = %d\n",
				PengridLoadGetID(this));
			noerr = 0;
		}
		#else //ifdef _PARALLEL_

		/*Configure the load when we are running on a parallel cluster: in this case, 
		 we  have partitioned the grids and elements across the cluster. Every node has 
		 its own lst, its own bgpdt (which owns the internal grids of the partition) and 
		 a common bgpdtb (b for boundaries of the partition). The boundary grids have to be 
		 dealt with. What happens is that bgpdt might not own all the grids referenced by this 
		 load, bgpdtb might own some of these grids. So we have to do a double configuration
		 sort of: 
		 */

		/*First, we need to recover pgpdt and pgpdtb from the arglist: */
		found=0;
		for ( i=0; i<num_datasets; i++) {
			if (*(arglist+i)==NULL)continue;
			if ( GpdtEnum() == DataSetGetType(*(arglist+i)) ) {
				if (found==0){
					/*This is the first GpdtEnum type dataset encountered, 
					 *ie  the internal grids: */
					pgpdt = *(arglist+i);
					found = 1;
					continue;
				}
				if (found==1){
					/*This is the second GpdtEnum type dataset encountered, 
					 *ie  the boundary grids: */
					pgpdtb = *(arglist+i);
					found = 2;
					continue;
				}
			}
		}
		
		if (found==2) {

			int	current_grid_id;
			int* pgrid_id_list = PengridGetGridIDPtr(&this->pengrid);

			if ( (current_grid_id=pgrid_id_list[0]) ) {	/* assign and test presence */

				/* if grid pointer information has been established once already
				(e.g., by a previous module or operation), grid index information
				may already be present and valid.  check to see if corresponding
				pointer information can be reestablished without an expensive
				seek operation: */

				if ( UNDEF != (int)this->internalgrid_index ) {

					/* does index still point to a valid grid? (i.e., the one with
					the same id): */

					if ( InternalGridEnum() == DataSetGetEnum( pgpdt, this->internalgrid_index)
						 &&
						 current_grid_id == DataSetGetObjectID( pgpdt, this->internalgrid_index) ) {

						/* great, still valid!  since we have the index position in
						gpdt, just use it to get, save, the grid pointer: */

						this->pg = DataSetGetObjectPtr( pgpdt, this->internalgrid_index);
					}
				}
				if ( UNDEF != (int)this->internalgrid_indexb ) {

					/* does index still point to a valid grid? (i.e., the one with
					the same id): */

					if ( InternalGridEnum() == DataSetGetEnum( pgpdtb, this->internalgrid_indexb)
						 &&
						 current_grid_id == DataSetGetObjectID( pgpdtb, this->internalgrid_indexb) ) {

						/* great, still valid!  since we have the index position in
						gpdtb, just use it to get, save, the grid pointer: */

						this->pg = DataSetGetObjectPtr( pgpdtb, this->internalgrid_indexb);
					}
				}

				/* if we haven't yet "recovered" the InternalGrid object pointer,
				go seek it out: */

				if ( !this->pg ) {

					foundgrid = DataSetSeek( (void**)(&this->pg),	/* cast because some compilers recognise as */
																/* ptr to ptr to struct otherwise           */
								&this->internalgrid_index ,
								pgpdt, current_grid_id, 1, InternalGridEnum());
					if(!foundgrid){
						/*We did not find this grid in pgpdt, go find it in pgpdtb:*/
						foundgrid = DataSetSeek( (void**)(&this->pg),	/* cast because some compilers recognise as */
																/* ptr to ptr to struct otherwise           */
								&this->internalgrid_indexb ,
								pgpdtb, current_grid_id, 1, InternalGridEnum());
					}

					if(!foundgrid){
						/*We could not find our grid in pgpdt and pgpdtb, this is not good!:*/
						noerr=0;
					}
					if (!noerr) {
						if (first) {
							_printf_("   %s error:\n", __func__);
							first = 0;
						}
						_printf_("   could not find grid with id of %d for element %d of type \"%s\"\n",
							current_grid_id, PengridLoadGetID(this), PengridLoadGetName(this));
					}
				}
			}
		}
		else {

			if (first) {
				_printf_("   %s error:\n", __func__);
				first=0;
			}
			_printf_("   could not find DataSets of type \"gpdt\", grid object\n");
			_printf_("   pointers not established for pengrid element id = %d\n",
				PengridLoadGetID(this));
			noerr = 0;
		}

		#endif

		
				
		
		
		/* matice property connections based on DataSet of type "mpt": */
		found=0;
		for ( i=0; i<num_datasets; i++) {
			if (*(arglist+i)==NULL)continue;
			if ( MptEnum() == DataSetGetType(*(arglist+i)) ) {
				pmpt  = *(arglist+i);
				found = 1;
				break;
			}
		}
		if (found) {
		

			int	current_matpar_id = PengridGetMaterialID(&this->pengrid);

			/* as for grids, check to see if a nonzero matpar index still
			points to the matpar entry of interest in mpt, otherwise do
			a seek for it: */

			if ( this->matpar_enum != UNDEF ){
			

				if ( this->matpar_enum == DataSetGetEnum( pmpt, this->matpar_index)
					 &&
					 current_matpar_id == DataSetGetObjectID( pmpt, this->matpar_index) ) {

					/* index is still valid; just get matpar entry pointer directly: */

					this->matpar = DataSetGetObjectPtr( pmpt, this->matpar_index);
				}
			}

			/* if we haven't yet "recovered" the matpar object pointer, go seek it out: */

			if ( NULL == this->matpar) {

				noerr *= DataSetSeek( &this->matpar, &this->matpar_index,
							pmpt, current_matpar_id, 1, MatparEnum() );	/* 0 to generalise later to Pcomp, etc. ... */

				if (!noerr) {
					if (first) {
						_printf_("   %s error:\n", __func__);
						first = 0;
					}
					_printf_("   could not find matpar entry with id of %d for element %d of type \"%s\"\n",
						current_matpar_id, PengridGetID(&this->pengrid), PengridGetName(&this->pengrid));
				}
				else {

					/* though the answer for now is just MaticeEnum(), for generality go ahead and
					ask the question, "what type of matpar has been referenced?" */

					this->matpar_enum = DataSetGetEnum( *(arglist+i), this->matpar_index);
				}
			}
		}
		else {

			if (first) {
				_printf_("   %s error:\n", __func__);
				first=0;
			}
			_printf_("   could not find DataSet of type \"mpt\"; matpar relationships\n");
			_printf_("   not established for penta element id = %d\n",
				PentaElementGetID(this));
			noerr = 0;
		}

		#ifdef _DEBUG_
		_printf_("   load data after all lookups:\n");
		PengridLoadEcho( this);
		#endif

		xfree((void**)&arglist);

	}
	else {

		/* quiet return */
		;
	}
	cleanup_and_return:

	return noerr;

}
/*--------------------------------------------------
	PengridLoadPenaltyCreateKMatrix
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PengridLoadPenaltyCreateKMatrix"
int PengridLoadPenaltyCreateKMatrix( ElemMatrix* *pKe_gg, void* vpthis, ParameterInputs* inputs, int K_flag, double kmax, int analysis_type){

	int noerr=1;
	if  ((analysis_type==ThermalSteadyAnalysisEnum()) || (analysis_type==ThermalTransientAnalysisEnum())){

		PengridLoadPenaltyCreateKMatrixThermal( pKe_gg, vpthis, inputs, K_flag, kmax, analysis_type);
	}
	else if  (analysis_type==DiagnosticStokesAnalysisEnum()){
		_printf_("%s%s\n",__FUNCT__," error message: analysis_type not supported yet!\n");
		return 0;
	}
	else if  (analysis_type==MeltingAnalysisEnum()){
		
		PengridLoadPenaltyCreateKMatrixMelting( pKe_gg, vpthis, inputs, K_flag, kmax, analysis_type);
	}
	else{
		_printf_("%s%s\n",__FUNCT__," error message: analysis_type not supported yet!\n");
		return 0;
	}
	return noerr;

}
/*--------------------------------------------------
	PengridLoadPenaltyCreateKMatrixThermal
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PengridLoadPenaltyCreateKMatrixThermal"
int PengridLoadPenaltyCreateKMatrixThermal( ElemMatrix* *pKe_gg, void* vpthis, ParameterInputs* inputs, int K_flag, double kmax, int analysis_type){

	int             i,counter;
	int             noerr=1;

	PengridLoad* this   = NULL;

	/* output: */
	ElemMatrix*     Ke_gg        = NULL;

	int*            structural_dof_list   =  NULL;
	int             dof;

	double*         dt_param;
	double          dt;


	/*Some pointer intialization: */
	this = (PengridLoad*)vpthis;

	/*Is penalty active? :*/
	if (this->pengrid.active){

		/*Recover dt input: */
		dt_param=ParameterInputsRecover(inputs,"dt");
		if (!dt_param){
			_printf_("%s%s\n",__FUNCT__," error message: dt needed to apply penalty!");
			*pKe_gg=NULL;
			return 0;
		}
		else{
			dt=*dt_param;
		}

		/*initialize stiffness matrix: */
		Ke_gg=NewElemMatrix(1); 

		/*Build dof list for stiffness matrix: */
		structural_dof_list= InternalGridGetStructuralDoflistPtr(this->pg);
		dof=structural_dof_list[this->pengrid.dof-1];
		Ke_gg->row_indices[0]=dof;
		
		if (this->pengrid.thermal_steadystate){
			*(Ke_gg->terms+Ke_gg->nrows*0+0)=this->pengrid.penalty_melting;
		}
		else{
			*(Ke_gg->terms+Ke_gg->nrows*0+0)=dt*this->pengrid.penalty_melting;
		}
				
		#ifdef _DEBUG_
			ElemMatrixEcho(Ke_gg);
		#endif
	}
	else{
		Ke_gg=NULL;
	}
		
	/*Assign output pointer: */
	*pKe_gg=Ke_gg;
	return noerr;
}
/*--------------------------------------------------
	PengridLoadPenaltyCreateKMatrixMelting
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PengridLoadPenaltyCreateKMatrixMelting"
int PengridLoadPenaltyCreateKMatrixMelting( ElemMatrix* *pKe_gg, void* vpthis, ParameterInputs* inputs, int K_flag, double kmax, int analysis_type){

	int             i,counter;
	int             noerr=1;

	PengridLoad* this   = NULL;
	Matpar*      matpar = NULL;

	/* output: */
	ElemMatrix*     Ke_gg        = NULL;

	int*            structural_dof_list   =  NULL;
	int             dof;

	double*         temperature_param;
	double          temperature;

	double*         pressure_param;
	double          pressure;
	double          meltingpoint,beta,t_pmp;

	/*Some pointer intialization: */
	this = (PengridLoad*)vpthis;
	matpar=this->matpar;


	//Recover parameters: */
	meltingpoint=matpar->meltingpoint;
	beta=matpar->beta;
	
	//First recover pressure and temperature vectors from inputs
	temperature_param=ParameterInputsRecover(inputs,"temperature");
	pressure_param=ParameterInputsRecover(inputs,"pressure");

	if((!pressure_param) || (!temperature_param)){
		_printf_("%s%s\n",__FUNCT__," error message: missing input temperature or pressure!");
		return 0;
	}

	/*Recover pressure and temperatureat the grid:*/
	structural_dof_list= InternalGridGetStructuralDoflistPtr(this->pg);
	dof=structural_dof_list[0];
	temperature=temperature_param[dof];
	pressure=pressure_param[dof];
	
	/*Compute pressure melting point: */
	t_pmp=meltingpoint-beta*pressure;


	/*Add penalty load: */
	if (temperature>t_pmp){
		Ke_gg=NULL;
	}
	else{
		//If T<Tpmp, there must be no melting. Therefore, melting should be  constrained to 0 when T<Tpmp
		//instead of using spcs, one uses penalties
		Ke_gg=NewElemMatrix(1); 
		Ke_gg->row_indices[0]=dof;
		*(Ke_gg->terms+Ke_gg->nrows*0+0)=kmax*pow(10,this->pengrid.penalty_offset);
	}

	#ifdef _DEBUG_
		ElemMatrixEcho(Ke_gg);
	#endif
		
	/*Assign output pointer: */
	*pKe_gg=Ke_gg;

	return noerr;
}

/*--------------------------------------------------
	PengridLoadPenaltyCreatePVector
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PengridLoadPenaltyCreatePVector"

int PengridLoadPenaltyCreatePVector( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs, double kmax, int analysis_type){

	int noerr=1;
	if  ((analysis_type==ThermalSteadyAnalysisEnum()) || (analysis_type==ThermalTransientAnalysisEnum())){

		PengridLoadPenaltyCreatePVectorThermal(ppe_g, vpthis, inputs, kmax, analysis_type);
	}
	else if  (analysis_type==DiagnosticStokesAnalysisEnum()){
		_printf_("%s%s\n",__FUNCT__," error message: analysis_type not supported yet!\n");
		return 0;
	}
	else if  (analysis_type==MeltingAnalysisEnum()){
		
		PengridLoadPenaltyCreatePVectorMelting(ppe_g, vpthis, inputs, kmax, analysis_type);
	}
	else{
		_printf_("%s%s\n",__FUNCT__," error message: analysis_type not supported yet!\n");
		return 0;
	}
	return noerr;
}


/*--------------------------------------------------
	PengridLoadPenaltyCreatePVectorThermal
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PengridLoadPenaltyCreatePVectorThermal"

int PengridLoadPenaltyCreatePVectorThermal( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs, double kmax, int analysis_type){

	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	PengridLoad* this   = NULL;
	Matpar*      matpar = NULL;

	/* output: */
	ElemVector*     pe_g        = NULL;

	/*input parameters: */
	double* dt_param=NULL;
	double  dt;
	double* pressure_param=NULL;
	double  pressure;
	
	/*material parameters: */
	double  beta;
	double  meltingpoint;
	double  t_pmp;

	int*            structural_dof_list   =  NULL;
	int             dof;

	/*Some pointer intialization: */
	this = (PengridLoad*)vpthis;
	matpar=this->matpar;
		
	/*Is penalty load active? :*/
	if (!this->pengrid.active){
		/*do nothing: */
		pe_g=NULL;
	}
	else{

		/* Allocate new load vector: */
		pe_g=NewElemVector(1);

		/* recover input parameters: */
		dt_param=ParameterInputsRecover(inputs,"dt");
		pressure_param=ParameterInputsRecover(inputs,"pressure");

		if( (!pressure_param)){
			_printf_("%s%s\n",__FUNCT__," error message: missing pressure in input parameters!");
			noerr=0;goto cleanup_and_return;
		}
		if(this->pengrid.thermal_steadystate==0){
			if( (!dt_param)){
				_printf_("%s%s\n",__FUNCT__," error message: missing dt in input parameters!");
				noerr=0;goto cleanup_and_return;
			}
		}

		/*Recover pressure at the grid: */
		structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg);
		dof=structural_dof_list[0]; //thermal dof #1
		*(pe_g->row_indices+0)=dof;
		pressure=pressure_param[dof];

		/*Recover material parameters: */
		beta=matpar->beta;
		meltingpoint=matpar->meltingpoint;

		//compute  pressure melting point
		t_pmp=meltingpoint-beta*pressure;

		//add penalty load:
		if(this->pengrid.thermal_steadystate){
			*(pe_g->terms+0)= this->pengrid.penalty_melting*t_pmp;
		}
		else{
			dt=*dt_param;
			*(pe_g->terms+0)= dt*this->pengrid.penalty_melting*t_pmp;
		}

		
	}

	cleanup_and_return: 

    if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build load vector.");
		return 0;
	}
	else{
		#ifdef _DEBUG_
		ElemVectorEcho(pe_g);
		#endif

		/*Assign output pointer: */
		*ppe_g=pe_g;
		return 1;
	}

}

/*--------------------------------------------------
	PengridLoadPenaltyCreatePVectorMelting
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PengridLoadPenaltyCreatePVectorMelting"

int PengridLoadPenaltyCreatePVectorMelting( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs, double kmax, int analysis_type){

	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	PengridLoad* this   = NULL;
	Matpar*      matpar = NULL;

	/* output: */
	ElemVector*     pe_g        = NULL;

	/*input parameters: */
	double* pressure_param=NULL;
	double  pressure;
	double* temperature_param=NULL;
	double  temperature;
	
	/*material parameters: */
	double  beta;
	double  meltingpoint;
	double  heatcapacity;
	double  latentheat;
	double  t_pmp;

	int*            structural_dof_list   =  NULL;
	int             dof;

	/*Some pointer intialization: */
	this = (PengridLoad*)vpthis;
	matpar=this->matpar;
		
	/* Allocate new load vector: */
	pe_g=NewElemVector(1);

	/* recover input parameters: */
	pressure_param=ParameterInputsRecover(inputs,"pressure");
	temperature_param=ParameterInputsRecover(inputs,"temperature");

	if( (!pressure_param) || (!temperature_param)){
		_printf_("%s%s\n",__FUNCT__," error message: missing pressure or temperature in input parameters!");
		noerr=0;goto cleanup_and_return;
	}

	/*Recover pressure and temperature at the grid: */
	structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg);
	dof=structural_dof_list[0]; //thermal dof #1
	*(pe_g->row_indices+0)=dof;
	pressure=pressure_param[dof];
	temperature=temperature_param[dof];

	/*Recover material parameters: */
	beta=matpar->beta;
	meltingpoint=matpar->meltingpoint;
	heatcapacity=matpar->heatcapacity;
	latentheat=matpar->latentheat;

	//compute  pressure melting point
	t_pmp=meltingpoint-beta*pressure;

	//add penalty load:
	if (temperature>t_pmp){
		*(pe_g->terms+0)= this->pengrid.penalty_melting*(temperature-t_pmp);
	}
	else{
		*(pe_g->terms+0)= 0;
	}

	cleanup_and_return: 

    if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build load vector.");
		return 0;
	}
	else{
		#ifdef _DEBUG_
		ElemVectorEcho(pe_g);
		#endif

		/*Assign output pointer: */
		*ppe_g=pe_g;
		return 1;
	}

}
/*--------------------------------------------------
	PengridLoadPenaltyConstrain
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PengridLoadPenaltyConstrain"

int PengridLoadPenaltyConstrain(int* punstable, void* vpthis,ParameterInputs* inputs, int analysis_type){

	/* vpthis for polymorphic function compatibility */
	PengridLoad* this   = NULL;
	Matpar*      matpar = NULL;

	int noerr=1;

	/*parameters: */
	double* pressure_param=NULL;
	double  pressure;
	double* temperature_param=NULL;
	double  temperature;
	
	/*Material parameters: */
	double  beta;
	double  meltingpoint;
	double  t_pmp;

	int*            structural_dof_list   =  NULL;
	int             dof;
	int             unstable;
	int             active;

	/*Some pointer intialization: */
	this = (PengridLoad*)vpthis;
	matpar=this->matpar;

	/*First recover parameter inputs: */
	pressure_param=ParameterInputsRecover(inputs,"pressure");
	temperature_param=ParameterInputsRecover(inputs,"temperature");

	if(!pressure_param || !temperature_param ){
		_printf_("%s%s\n",__FUNCT__," error message: missing input parameters!");
		noerr=0 ;goto cleanup_and_return;
	}

	/*Recover pressure and temperature at grid: */
	structural_dof_list= InternalGridGetStructuralDoflistPtr(this->pg);
	dof=structural_dof_list[0]; //that's where temperatures and pressure were solved for.
	temperature=temperature_param[dof];
	pressure=pressure_param[dof];

	//Compute pressure melting point
	meltingpoint=matpar->meltingpoint;
	beta=matpar->beta;
	t_pmp=meltingpoint-beta*pressure;

	//Figure out if temperature is over meltingpoint, in which case, this penalty needs to be activated.
	if (temperature>t_pmp){
		active=1;
	}
	else{
		active=0;
	}

	//Optional!
	/*if(this->pengrid.active){ //once activated, a constraint will remain on
		active=1;
	}*/

	//Figure out stability of this penalty
	if(this->pengrid.active==active){
		unstable=0;
	}
	else{
		unstable=1;
	}

	//Set penalty flag
	this->pengrid.active=active;

	cleanup_and_return: 
	/*assign output pointer: */
	*punstable=unstable;

	return noerr;
}


/*--------------------------------------------------
	PengridLoadGetName
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PengridLoadGetName"

char*     PengridLoadGetName( void* vpthis){

	/* vpthis for polymorphic function compatibility */
	PengridLoad* this = (PengridLoad*)vpthis;

	return PengridGetName(&this->pengrid);

}
