/*
	TriaElement.c
*/

#include "../../include/cielo.h"

#define NDOF1 1
#define NDOF2 2
#define numgrids 3

/*For debugging purposes: */
#define ELID 141 //element for which to print debug statements
#define RANK 2 //rank of cpu for which to print debug statements.
//#define _DEBUGELEMENTS_
//#define _DEBUGGAUSS_

/*--------------------------------------------------
	NewTriaElement
  --------------------------------------------------*/

TriaElement* NewTriaElement( void* vptria)
{

	Tria* 		tria = (Tria*)vptria;
	TriaElement*	this   = NULL;
	int	i,j;

	this=(TriaElement*)xcalloc(1,sizeof(TriaElement));	/* guaranteed NULL, 0 initialisation */

	/* include the "base class" in the "derived" object: */
	memcpy( &this->tria, tria, sizeof(Tria));

	/* since zero could be a valid index, set all DataSet index
	"pointers" to UNDEF: */

	for ( i=0; i<3; i++){
		this->internalgrid_indices[i] = UNDEF;
		this->internalgrid_indicesb[i] = UNDEF;
		this->pg[i] = NULL;
	}
	
	this->property_index=UNDEF;
	this->property_enum=UNDEF;
	this->property=NULL;


	this->matice_index=UNDEF;
	this->matice_enum=UNDEF;
	this->matice=NULL;

	this->matpar_index=UNDEF;
	this->matpar_enum=UNDEF;
	this->matpar=NULL;


	return this;
}


/*--------------------------------------------------
	DeleteTriaElement
  --------------------------------------------------*/

void DeleteTriaElement(void* *vpthis)
{
	/* vpthis for polymorphic function compatibility */

	TriaElement* *pthis = (TriaElement**)vpthis;
	TriaElement*   this = *pthis;

	#ifdef _DEBUG_
	printf("   DeleteTriaElement message:\n");
	printf("   deleting TriaElement with eid = %d\n", this->tria.eid);
	#endif
	xfree(vpthis);
}


/*--------------------------------------------------
	TriaElementEcho
  --------------------------------------------------*/

void TriaElementEcho(void* vpthis)
{
	/* vpthis for polymorphic function compatibility */
	TriaElement* this = (TriaElement*)vpthis;

	int	i,j;

	printf("\nTriaElement echo:\n");
	printf("-------------------\n");
	printf("   base Tria object:\n");
	TriaEcho( &this->tria);
	printf("\n");

	printf("   element points to following grids (pointer-based ops):\n");

	printf("   local grid number   corresponding grid id\n");
	printf("   -----------------   ---------------------\n");

	for ( i=0; i<3; i++) {
	if ( this->pg[i])
	printf("        %8d              %8d\n", i, InternalGridGetID( this->pg[i]));
	}

	printf("   element references the following:\n");

	if ( this->property) printf("   property entry of enum type %d\n", this->property_enum);
	else                  printf("   property reference is NULL\n");
	if ( this->matice) printf("   matice entry of enum type %d\n", this->matice_enum);
	else                  printf("   matice reference is NULL\n");
	if ( this->matpar) printf("   matpar entry of enum type %d\n", this->matpar_enum);
	else                  printf("   matpar reference is NULL\n");
}


/*--------------------------------------------------
	TriaElementSizeOfInDoubles
  --------------------------------------------------*/

int TriaElementSizeOfInDoubles(void* vpthis)
{
	/* vpthis for polymorphic function compatibility */

	TriaElement* this = (TriaElement*)vpthis;
	int struct_doubles=0;

	struct_doubles = 
		(int)ceil((float)sizeof(*this)/(float)sizeof(double));

	#ifdef _DEBUG_
	printf("\n   TriaElementSizeOfInDoubles diagnostics:\n");
	printf("      returning %d doubles\n", struct_doubles);
	#endif

	return struct_doubles;
}

/*--------------------------------------------------
	TriaElementShrinkWrap
  --------------------------------------------------*/

void TriaElementShrinkWrap(void* to, void* vpthis)
{
	/* no "extended" data means a simple copy will do: */

	memcpy( to, vpthis, TriaElementSizeOfInDoubles(vpthis)*sizeof(double));
}


/*--------------------------------------------------
	TriaElementUnwrap
  --------------------------------------------------*/

int  TriaElementUnwrap(void* vpthis)
{
	/* though no extended data means no internal pointers to
	reestablish, re-null pointers whose values must properly come
	from a call to Configure(): */

	TriaElement* this = (TriaElement*)vpthis;
	int i;

	for ( i=0; i<3; i++) this->pg[i] = NULL;
	this->matice = NULL;
	this->matpar = NULL;
	this->property = NULL;
	return 1;
}

/*--------------------------------------------------
	TriaElementMarshall
  --------------------------------------------------*/

void TriaElementMarshall( char* *pbuffer, void* vpthis, int* size)
{
	/* vpthis for polymorphic function compatibility */
	TriaElement* this = (TriaElement*)vpthis;

	char* buffer=*pbuffer;

	int i, j, size_tria=0; 
	int ig;

	int num_ints=0, num_doubles=0;


	/* first marshall the embedded Tria object: ... */

	TriaMarshall( &buffer, &this->tria, &size_tria);


	/* ... then marshall the TriaElement member data: */

	for ( i=0; i<3; i++) {	
		cellmar( &this->internalgrid_indices[i], &buffer, INTEGER_FLAG);
		cellmar( &this->internalgrid_indicesb[i], &buffer, INTEGER_FLAG);
		num_ints++;
	}
	/* marshall InternalGrid pointers even though pointer information won't be
	valid after demarshalling (assumes pointer can fit into integer space): */
	for ( i=0; i<3; i++) {
		cellmar( &this->pg[i], &buffer, INTEGER_FLAG);
		num_ints++;
	}

	cellmar( &this->property_index, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->property_enum, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->property, &buffer, INTEGER_FLAG); num_ints++;
	
	cellmar( &this->matice_index, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->matice_enum, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->matice, &buffer, INTEGER_FLAG); num_ints++;

	cellmar( &this->matpar_index, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->matpar_enum, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->matpar, &buffer, INTEGER_FLAG); num_ints++;

	*size = size_tria + num_ints*sizeof(int) + num_doubles*sizeof(double);
	*pbuffer = buffer;

	return;
}


/*--------------------------------------------------
	TriaElementDemarshall
  --------------------------------------------------*/

void* TriaElementDemarshall( DataSet* ds, char* *pbuffer, int machine_flag)
{ 
	int i, j, ig;
	Tria* ptria=NULL;	/* for embedded object */

	char* buffer=*pbuffer; 
	TriaElement* this=NULL;


	/* first demarshall the embedded Tria object: ... */

	ptria = TriaDemarshall( NULL, &buffer, machine_flag);
	/* create the derived object: */
	this = NewTriaElement( ptria);
	/* and free tria now that it's embedded in the derived object: */
	DeleteTria( (void**)&ptria);


	/* ... then demarshall the TriaElement member data: */

	for ( i=0; i<3; i++) {
		celldem( &this->internalgrid_indices[i], &buffer, INTEGER_FLAG, machine_flag);
		celldem( &this->internalgrid_indicesb[i], &buffer, INTEGER_FLAG, machine_flag);
	}

	/* though they're meaningless, demarshall InternalGrid pointers: */
	for ( i=0; i<3; i++) {	/* order+1 due to collapsed node formulation */
		celldem( &this->pg[i], &buffer, INTEGER_FLAG, machine_flag);
	}
	
	celldem( &this->property_index, &buffer, INTEGER_FLAG, machine_flag);
	celldem( &this->property_enum, &buffer, INTEGER_FLAG, machine_flag);
	celldem( &this->property, &buffer, INTEGER_FLAG, machine_flag);
	
	
	celldem( &this->property_index, &buffer, INTEGER_FLAG, machine_flag);
	celldem( &this->property_enum, &buffer, INTEGER_FLAG, machine_flag);
	celldem( &this->property, &buffer, INTEGER_FLAG, machine_flag);
	
	/*Set pointers to NULL :*/
	for (i=0;i<3;i++){
		this->pg[i]=NULL;
	}
	this->property=NULL;
	this->matice=NULL;
	this->matpar=NULL;
	
	if (ds) TriaElementAddToDataSet( ds, this);
	*pbuffer=buffer;  

	return (void*)this;
}

/*--------------------------------------------------
	TriaElementGetID
  --------------------------------------------------*/

int TriaElementGetID( void* vpthis )
{
	/* vpthis for polymorphic function compatibility */
	TriaElement* this = (TriaElement*)vpthis;

	return TriaGetID( &this->tria);
}



/*--------------------------------------------------
	TriaElementSetVirtualFunctions
  --------------------------------------------------*/

int TriaElementSetVirtualFunctions( void* vp)
{

	/* VirtualFunctions implemented as a "friend" class: */

	/* void* for polymorphic function compatibility */
	VirtualFunctions* vf = (VirtualFunctions*)vp;
	
	if ( VirtualFunctionsInit(vf) ) {

		/* set TriaElement-specific virtual function pointers: */

		/* general: */

		vf->Configure        = &TriaElementConfigure;

		/* grids, connectivity, basis conversions for element types: */

		/* element matrix generation: */
		vf->CreateKMatrix    = &TriaElementCreateKMatrix;

		/* element load generation: */
		vf->CreatePVector    = &TriaElementCreatePVector;
	
		/* velocity difference generation: */
		vf->CreateDuVector    = &TriaElementCreateDuVector;
		vf->CreateGradjVectors    = &TriaElementCreateGradjVectors;

		/* velocity misfit: */
		vf->Misfit=&TriaElementMisfit;

	
	
		return 1;
	}
	else
		return 0;

}

/*--------------------------------------------------
	TriaElementConfigure
  --------------------------------------------------*/

int TriaElementConfigure( void* vpthis, int num_datasets, ...)
{

	/* set up data connections to all necessary, related objects: */

	TriaElement* this = (TriaElement*)vpthis;

	int	i,j, found, foundgrid, noerr=1, first=1;

	/* expected DataSet input: */

	DataSet*	pgpdt=NULL;
	DataSet*	pgpdtb=NULL;
	DataSet*	pmpt=NULL;
	DataSet*	pest=NULL;

	/* variable argument list handling: */

	va_list args;
	DataSet* *arglist=NULL;

	#ifdef _DEBUG_
	printf("virtual function: TriaElementConfigure\n");
	#endif


	if (this) {

		if (!num_datasets) {

			printf("   %s error:\n", __func__);
			printf("   null input for number of datasets;\n");
			printf("   could not establish grid, property  and material data connections.\n");
			return 0;
		}
		else {

			/* read, save dataset pointers for data connection operations: */
			arglist = xmalloc( num_datasets*sizeof(DataSet*));
			va_start( args, num_datasets);
			for ( i=0; i<num_datasets; i++) {
				*(arglist+i) = va_arg(args,DataSet*);
			}
			va_end(args);
		}

		#ifdef _DEBUG_
		printf("   variable argument list statistics:\n");
		printf("   input number of DataSets = %d\n", num_datasets);
		printf("   enum types of input DataSets:");
		for ( i=0; i<num_datasets; i++)  printf("   %d", DataSetGetType(*(arglist+i)));
		printf("\n");
		#endif


		/*
			establish grid, property and material object data connections
			for this TriaElement object:
		*/

		/*Reinitialize this->pg[i]*/
		for (i=0;i<3;i++){
			this->pg[i]=NULL;
		}

		/* grid connections based on DataSet of type "gpdt": */

		#ifndef _PARALLEL_
		found=0;
		for ( i=0; i<num_datasets; i++) {
			if (*(arglist+i)==NULL)continue;
			if ( GpdtEnum() == DataSetGetType(*(arglist+i)) ) {
				pgpdt = *(arglist+i);
				found = 1;
				break;
			}
		}
			
		if (found) {

			int	current_grid_id;
			int* pgrid_id_list = TriaGetGridIDPtr(&this->tria);

			for ( i=0; i<3; i++) {

				if ( (current_grid_id=pgrid_id_list[i]) ) {	/* assign and test presence */

					/* if grid pointer information has been established once already
					(e.g., by a previous module or operation), grid index information
					may already be present and valid.  check to see if corresponding
					pointer information can be reestablished without an expensive
					seek operation: */

					if ( UNDEF != (int)this->internalgrid_indices[i] ) {

						/* does index still point to a valid grid? (i.e., the one with
						the same id): */

						if ( InternalGridEnum() == DataSetGetEnum( pgpdt, this->internalgrid_indices[i])
							 &&
						     current_grid_id == DataSetGetObjectID( pgpdt, this->internalgrid_indices[i]) ) {

							/* great, still valid!  since we have the index position in
							gpdt, just use it to get, save, the grid pointer: */

							this->pg[i] = DataSetGetObjectPtr( pgpdt, this->internalgrid_indices[i]);
						}
					}

					/* if we haven't yet "recovered" the InternalGrid object pointer,
					go seek it out: */

					if ( !this->pg[i] ) {

						noerr *= DataSetSeek( (void**)(this->pg+i),	/* cast because some compilers recognise as */
																	/* ptr to ptr to struct otherwise           */
									(this->internalgrid_indices+i) ,
									pgpdt, current_grid_id, 1, InternalGridEnum());

						if (!noerr) {
							if (first) {
								printf("   %s error:\n", __func__);
								first = 0;
							}
							printf("   could not find grid with id of %d for element %d of type \"%s\"\n",
								current_grid_id, TriaGetID(&this->tria), TriaGetName(&this->tria));
						}
					}
				}
			}

		}
		else {

			if (first) {
				printf("   %s error:\n", __func__);
				first=0;
			}
			printf("   could not find DataSet of type \"gpdt\", grid object\n");
			printf("   pointers not established for tria element id = %d\n",
				TriaElementGetID(this));
			noerr = 0;
		}
		#else //ifdef _PARALLEL_

		/*Configure the element when we are running on a parallel cluster: in this case, 
		 we  have partitioned the elements and the grids correspondingly. Every node has 
		 its own est, its own bgpdt (which owns the internal grids of the partition) and 
		 a common bgpdtb (b for boundaries of the partition). The boundary grids have to be 
		 dealt with. What happens is that bgpdt might not own all the grids referenced by this 
		 element, bgpdtb might own some of these grids. So we have to do a double configuration
		 sort of: 
		 */

		/*First, we need to recover pgpdt and pgpdtb from the arglist: */
		found=0;
		for ( i=0; i<num_datasets; i++) {
			if (*(arglist+i)==NULL)continue;
			if ( GpdtEnum() == DataSetGetType(*(arglist+i)) ) {
				if (found==0){
					/*This is the first GpdtEnum type dataset encountered, 
					 *ie  the internal grids: */
					pgpdt = *(arglist+i);
					found = 1;
					continue;
				}
				if (found==1){
					/*This is the second GpdtEnum type dataset encountered, 
					 *ie  the boundary grids: */
					pgpdtb = *(arglist+i);
					found = 2;
					continue;
				}
			}
		}
		
		if (found==2) {

			int	current_grid_id;
			int* pgrid_id_list = TriaGetGridIDPtr(&this->tria);

			for ( i=0; i<3; i++) {

				if ( (current_grid_id=pgrid_id_list[i]) ) {	/* assign and test presence */

					/* if grid pointer information has been established once already
					(e.g., by a previous module or operation), grid index information
					may already be present and valid.  check to see if corresponding
					pointer information can be reestablished without an expensive
					seek operation: */

					if ( UNDEF != (int)this->internalgrid_indices[i] ) {

						/* does index still point to a valid grid? (i.e., the one with
						the same id): */

						if ( InternalGridEnum() == DataSetGetEnum( pgpdt, this->internalgrid_indices[i])
							 &&
						     current_grid_id == DataSetGetObjectID( pgpdt, this->internalgrid_indices[i]) ) {

							/* great, still valid!  since we have the index position in
							gpdt, just use it to get, save, the grid pointer: */

							this->pg[i] = DataSetGetObjectPtr( pgpdt, this->internalgrid_indices[i]);
						}
					}
					if ( UNDEF != (int)this->internalgrid_indicesb[i] ) {

						/* does index still point to a valid grid? (i.e., the one with
						the same id): */

						if ( InternalGridEnum() == DataSetGetEnum( pgpdtb, this->internalgrid_indices[i])
							 &&
						     current_grid_id == DataSetGetObjectID( pgpdtb, this->internalgrid_indices[i]) ) {

							/* great, still valid!  since we have the index position in
							gpdtb, just use it to get, save, the grid pointer: */

							this->pg[i] = DataSetGetObjectPtr( pgpdtb, this->internalgrid_indices[i]);
						}
					}

					/* if we haven't yet "recovered" the InternalGrid object pointer,
					go seek it out: */

					if ( !this->pg[i] ) {

						foundgrid = DataSetSeek( (void**)(this->pg+i),	/* cast because some compilers recognise as */
																	/* ptr to ptr to struct otherwise           */
									(this->internalgrid_indices+i) ,
									pgpdt, current_grid_id, 1, InternalGridEnum());
						if(!foundgrid){
							/*We did not find this grid in pgpdt, go find it in pgpdtb:*/
							foundgrid = DataSetSeek( (void**)(this->pg+i),	/* cast because some compilers recognise as */
																	/* ptr to ptr to struct otherwise           */
									(this->internalgrid_indicesb+i) ,
									pgpdtb, current_grid_id, 1, InternalGridEnum());
						}

						if(!foundgrid){
							/*We could not find our grid in pgpdt and pgpdtb, this is not good!:*/
							noerr=0;
						}
						if (!noerr) {
							if (first) {
								printf("   %s error:\n", __func__);
								first = 0;
							}
							printf("   could not find grid with id of %d for element %d of type \"%s\"\n",
								current_grid_id, TriaGetID(&this->tria), TriaGetName(&this->tria));
						}
					}
				}
			}
		}
		else {

			if (first) {
				printf("   %s error:\n", __func__);
				first=0;
			}
			printf("   could not find DataSets of type \"gpdt\", grid object\n");
			printf("   pointers not established for tria element id = %d\n",
				TriaElementGetID(this));
			noerr = 0;
		}

		#endif

		/* material property connections based on DataSet of type "mpt": */

		found=0;
		for ( i=0; i<num_datasets; i++) {
			if (*(arglist+i)==NULL)continue;
			if ( MptEnum() == DataSetGetType(*(arglist+i)) ) {
				pmpt  = *(arglist+i);
				found = 1;
				break;
			}
		}
		if (found) {

			int	current_matice_id = TriaGetMaterialID(&this->tria);

			/* as for grids, check to see if a nonzero matice index still
			points to the matice entry of interest in mpt, otherwise do
			a seek for it: */

			if ( this->matice_index != UNDEF) {
			

				if ( this->matice_enum == DataSetGetEnum( pmpt, this->matice_index)
					 &&
					 current_matice_id == DataSetGetObjectID( pmpt, this->matice_index) ) {

					/* index is still valid; just get matice entry pointer directly: */

					this->matice = DataSetGetObjectPtr( pmpt, this->matice_index);
				}
			}

			/* if we haven't yet "recovered" the matice object pointer, go seek it out: */

			if ( NULL == this->matice) {

				noerr *= DataSetSeek( &this->matice, &this->matice_index,
							pmpt, current_matice_id, 1, MaticeEnum() );	/* 0 to generalise later to Pcomp, etc. ... */

				if (!noerr) {
					if (first) {
						printf("   %s error:\n", __func__);
						first = 0;
					}
					printf("   could not find matice entry with id of %d for element %d of type \"%s\"\n",
						current_matice_id, TriaGetID(&this->tria), TriaGetName(&this->tria));
				}
				else {

					/* though the answer for now is just MaticeEnum(), for generality go ahead and
					ask the question, "what type of matice has been referenced?" */

					this->matice_enum = DataSetGetEnum( *(arglist+i), this->matice_index);
				}
			}
			
			int	current_matpar_id = TriaGetMaterialParID(&this->tria);

			/* as for grids, check to see if a nonzero matpar index still
			points to the matpar entry of interest in mpt, otherwise do
			a seek for it: */

			if ( this->matpar_index != UNDEF) {
			

				if ( this->matpar_enum == DataSetGetEnum( pmpt, this->matpar_index)
					 &&
					 current_matpar_id == DataSetGetObjectID( pmpt, this->matpar_index) ) {

					/* index is still valid; just get matpar entry pointer directly: */

					this->matpar = DataSetGetObjectPtr( pmpt, this->matpar_index);
				}
			}

			/* if we haven't yet "recovered" the matpar object pointer, go seek it out: */

			if ( NULL == this->matpar) {

				noerr *= DataSetSeek( &this->matpar, &this->matpar_index,
							pmpt, current_matpar_id, 1, MatparEnum() );	/* 0 to generalise later to Pcomp, etc. ... */

				if (!noerr) {
					if (first) {
						printf("   %s error:\n", __func__);
						first = 0;
					}
					printf("   could not find matpar entry with id of %d for element %d of type \"%s\"\n",
						current_matpar_id, TriaGetID(&this->tria), TriaGetName(&this->tria));
				}
				else {

					/* though the answer for now is just MatparEnum(), for generality go ahead and
					ask the question, "what type of matpar has been referenced?" */

					this->matpar_enum = DataSetGetEnum( *(arglist+i), this->matpar_index);
				}
			}
		}
		else {

			if (first) {
				printf("   %s error:\n", __func__);
				first=0;
			}
			printf("   could not find DataSet of type \"mpt\"; matice and matpar relationships\n");
			printf("   not established for tria element id = %d\n",
				TriaElementGetID(this));
			noerr = 0;
		}


		#ifdef _DEBUG_
		printf("   element data after all lookups:\n");
		TriaElementEcho( this);
		#endif


		xfree((void**)&arglist);
	}
	else {

		/* quiet return */
		;
	}
	cleanup_and_return:

	return noerr;

}

/*--------------------------------------------------
	TriaElementAddToDataSet
  --------------------------------------------------*/

int TriaElementAddToDataSet( DataSet* dataset, TriaElement* this)
{
	/*

	Provided as an alternate interface to DataSetAddObject.

	In the c++ world, the dataset could just add an object to its list
	using a dataset member function and a pointer to the object's base
	class.

	In the c world, however, only the object itself is capable of telling
	the dataset its type, hence the structure here; a call to a generic
	DataSetAddObject function via an object "member function."

	*/

	DataSetAddObject( dataset, TriaElementEnum(), (void*)this,
		&DeleteTriaElement, &TriaElementEcho, &TriaElementSizeOfInDoubles,
		&TriaElementShrinkWrap, &TriaElementUnwrap,
		&TriaElementMarshall,
		&TriaElementGetID, &TriaElementSetVirtualFunctions);
	
	return 1;
}

/*--------------------------------------------------
	TriaElementCreateKMatrix
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementCreateKMatrix"
int TriaElementCreateKMatrix( ElemMatrix* *pKe_gg, ElemMatrix** pKTe_gg, void* vpthis, ParameterInputs* inputs, int K_flag, int KT_flag, int analysis_type){


	int noerr=1;
	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if ((analysis_type==DiagnosticHorizAnalysisEnum()) || (analysis_type==ControlAnalysisEnum())){

		noerr=TriaElementCreateKMatrixDiagnosticHoriz( pKe_gg, vpthis, inputs);

	}
	else{
		printf("%s%s%i%s\n",__FUNCT__," error message: analysis ",analysis_type," not supported yet");
		noerr=0;
	}
	return noerr;
}

/*--------------------------------------------------
	TriaElementCreatePVector
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementCreatePVector"

int TriaElementCreatePVector( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs, int analysis_type){

	int noerr=1;
	
	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	if ((analysis_type==DiagnosticHorizAnalysisEnum()) || (analysis_type==ControlAnalysisEnum())){
		noerr=TriaElementCreatePVectorDiagnosticHoriz( ppe_g,vpthis,inputs);
	}
	else{
		printf("%s%s%i%s\n",__FUNCT__," error message: analysis ",analysis_type," not supported yet");
		noerr=0;
	}
	return noerr;
}

/*--------------------------------------------------
	TriaElementCreateKMatrixDiagnosticHoriz
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementCreateKMatrixDiagnosticHoriz"
int TriaElementCreateKMatrixDiagnosticHoriz( ElemMatrix* *pKe_gg, void* vpthis, ParameterInputs* inputs){


	int             i,j,il,counter;
	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	TriaElement* this   = NULL;
	Matice        * matice = NULL;
	Matpar        * matpar = NULL;

	/* local declarations */

	/* output: */
	ElemMatrix*     Ke_gg        = NULL;
	
	
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;


	/* grid data: */
	int    cid_list[numgrids];
	double xyz_list[numgrids][3];
	double T_bg_list[numgrids][9];
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	
	/* specific gaussian point: */
	double   gauss_weight;
	double  gauss_l1l2l3[3];

	/*drag: */
	double  pcoeff,qcoeff;
	double  rcoeff,scoeff;
	double  velocity_x,velocity_y,velocity_mag;

	/* material data: */
	double viscosity; //viscosity
	double old_viscosity;
	double viscosity_change;
	double B_param;
	double gravity=9.8;
	double rho_ice,rho_water;
	
	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/
	double old_epsilon[3];

	/* matrices: */
	double B[3][NDOF2*numgrids];
	double Bprime[3][NDOF2*numgrids];
	double L[2][NDOF2*numgrids];
	double D[3][3]={{ 0,0,0 },{0,0,0},{0,0,0}};              // material matrix, simple scalar matrix.
	double D_scalar;

	double DL[2][2]={{ 0,0 },{0,0}}; //for basal drag
	double DL_scalar;

	double Ke_gg_gaussian[NDOF2*numgrids][NDOF2*numgrids]; //stiffness matrix evaluated at the gaussian point.
	double Ke_gg_drag_gaussian[NDOF2*numgrids][NDOF2*numgrids]; //stiffness matrix contribution from drag
	double Jdet;
	
	/*slope: */
	double  slope[NDOF2]={0.0,0.0};
	double  slope_magnitude;

	/*input parameters for structural analysis (diagnostic): */
	double* thickness_param=NULL;
	double thickness_list[numgrids];
	double thickness;
	double* surface_param=NULL;
	double surface_list[numgrids];
	double* bed_param=NULL;
	double bed_list[numgrids];
	double* flow_law_param=NULL;
	double B_list[numgrids];
	double* basal_drag=NULL;
	double K_list[numgrids];
	double* velocity=NULL;
	double vxvy_list[numgrids][2];
	double* temperature_average_param=NULL;
	double  temperature_average_list[numgrids];
	double  temperature_average;

	/*friction: */
	double alpha2_list[numgrids]={0.0,0.0,0.0};
	double alpha2;

	double MAXSLOPE=.06; // 6 %
	double MOUNTAINKEXPONENT=10;

	/* The number of g-set dof for this element is 2*3 (vx and vy, horizontal velocities for ice, and 3 grids 
	 * per element: */
	numdof=numgrids*NDOF2;

	/*Some pointer intialization: */
	this = (TriaElement*)vpthis;
	matice=this->matice;
	matpar=this->matpar;

	/*First check that acceleration is not on: */
	if (this->tria.acceleration){
		*pKe_gg=NULL;
		return;
	}

	/* Allocate new stiffness element  matrix: */
	Ke_gg=NewElemMatrix(numdof);

	/*Recover material: */
	gravity=matpar->g;
	rho_ice=matpar->rho_ice;
	rho_water=matpar->rho_water;

	/*recover extra inputs from users, at current convergence iteration: */
	basal_drag=ParameterInputsRecover(inputs,"drag");
	thickness_param=ParameterInputsRecover(inputs,"thickness");
	surface_param=ParameterInputsRecover(inputs,"surface");
	bed_param=ParameterInputsRecover(inputs,"bed");
	temperature_average_param=ParameterInputsRecover(inputs,"temperature_average");
	flow_law_param=ParameterInputsRecover(inputs,"B");
	velocity=ParameterInputsRecover(inputs,"velocity"); 

	/* Get all element grid data */
	noerr = GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], cid_list, this->pg, numgrids, 1 );
	if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not get element grid data!");
		goto cleanup_and_return;
	}
	
	/* Build the row index vector (same as column index vector). 
	 * At the same time, get element exterior data for each grid: */
	for (i=0;i<numgrids;i++){
		/* We use the structural dof list, from which we keep only the first two dof (x for vx, and y for vy).
		 * The other degrees of freedom will be spc'd for now.*/
		structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
		for (j=0;j<NDOF2;j++){
			dof=structural_dof_list[j];
			*(Ke_gg->row_indices+i*NDOF2+j)=dof;
			if(velocity){
				vxvy_list[i][j]=*(velocity+dof);
			}
			else{
				vxvy_list[i][j]=0;
			}
		}
		dof=structural_dof_list[0];
		if(flow_law_param) 
			B_list[i] = *(flow_law_param+dof);
		else 
			B_list[i] =0;

		if(basal_drag){
			K_list[i]=*(basal_drag+dof);
		}
		else{
			K_list[i]=this->tria.k[i];
		}
		if(thickness_param){
			thickness_list[i]=*(thickness_param+dof);
		}
		else{
			thickness_list[i]=this->tria.h[i];
		}
		if(surface_param){
			surface_list[i]= *(surface_param+dof);
		}
		else{
			surface_list[i]= this->tria.s[i];
		}
		if(bed_param){
			bed_list[i]=*(bed_param+dof);
		}
		else{
			bed_list[i]= this->tria.b[i];
		}

		if(temperature_average_param) temperature_average_list[i]=*(temperature_average_param+dof);
	}

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && TriaElementGetID(this)==ELID){ 
		printf("El id %i Rank %i TriaElemnet input list before gaussian loop: \n",ELID,RANK); 
		printf("   rho_ice: %g \n",rho_ice);
		printf("   gravity: %g \n",gravity);
		printf("   rho_water: %g \n",rho_water);
		printf("   Velocity: \n");
		for (i=0;i<numgrids;i++){
			printf("      grid %i  [%g,%g,%g]\n",i,vxvy_list[i][0],vxvy_list[i][1],vxvy_list[i][2]);
		}
		printf("   B [%g %g %g ]\n",B_list[0],B_list[1],B_list[2]);
		printf("   K [%g %g %g ]\n",K_list[0],K_list[1],K_list[2]);
		printf("   thickness [%g %g %g]\n",thickness_list[0],thickness_list[1],thickness_list[2]);
		printf("   surface [%g %g %g ]\n",surface_list[0],surface_list[1],surface_list[2]);
		printf("   bed [%g %g %g]\n",bed_list[0],bed_list[1],bed_list[2]);
		if(temperature_average_param)printf("   temperature_average [%g %g %g]\n",temperature_average_list[0],temperature_average_list[1],temperature_average_list[2]);
	}
	#endif



	/*Build alpha2_list used by drag stiffness matrix*/
	if (!this->tria.shelf && (this->tria.friction_type==2)){
		
		/*Allocate friction object: */
		Friction* friction=NewFriction();
		
		/*Initialize all fields: */
		friction->element_type=xmalloc((strlen("2d")+1)*sizeof(char));
		strcpy(friction->element_type,"2d");
		
		friction->gravity=gravity;
		friction->rho_ice=rho_ice;
		friction->rho_water=rho_water;
		friction->K=&K_list[0];
		friction->bed=&bed_list[0];
		friction->thickness=&thickness_list[0];
		friction->velocities=&vxvy_list[0][0];
		friction->p=this->tria.p;
		friction->q=this->tria.q;

		/*Compute alpha2_list: */
		FrictionGetAlpha2(&alpha2_list[0],friction);

		/*Erase friction object: */
		DeleteFriction(&friction);
	}

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && TriaElementGetID(this)==ELID){ 
		printf("   alpha2_list [%g %g %g ]\n",alpha2_list[0],alpha2_list[1],alpha2_list[2]);
	}
	#endif

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && TriaElementGetID(this)==ELID){ 
		printf("   gaussian points: \n");
		for(i=0;i<num_gauss;i++){
			printf("    %g %g %g : %g\n",first_gauss_area_coord[i],second_gauss_area_coord[i],third_gauss_area_coord[i],gauss_weights[i]);
		}
	}
	#endif

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);


		/*Compute thickness at gaussian point: */
		TriaElementGetParameterValue(&thickness, &thickness_list[0],gauss_l1l2l3);

		// If we have a slope > 6% for this element,  it means  we are on a mountain. In this particular case, 
		//velocity should be = 0. To achieve this result, we set alpha2_list to a very high value: */
		if(!this->tria.shelf){
			TriaElementGetParameterDerivativeValue(&slope[0], &surface_list[0],&xyz_list[0][0], gauss_l1l2l3);
			slope_magnitude=sqrt(pow(slope[0],2)+pow(slope[1],2));

			if (slope_magnitude>MAXSLOPE){
				alpha2_list[0]=pow(10,MOUNTAINKEXPONENT);
				alpha2_list[1]=pow(10,MOUNTAINKEXPONENT);
				alpha2_list[2]=pow(10,MOUNTAINKEXPONENT);
			}
		}

		/*Get strain rate from velocity: */
		TriaElementGetStrainRate(&epsilon[0],&vxvy_list[0][0],&xyz_list[0][0],gauss_l1l2l3);
		
		//Update material if temperature is provided.
		if(temperature_average_param){
			TriaElementGetParameterValue(&temperature_average, &temperature_average_list[0],gauss_l1l2l3);
			B_param=Paterson(temperature_average);
			MaticeSetFlowLawParameter(this->matice,B_param);
		}
		//Update material if B is provided.
		if(flow_law_param){
			TriaElementGetParameterValue(&B_param, &B_list[0],gauss_l1l2l3);
			MaticeSetFlowLawParameter(this->matice,B_param);
		}

		/*Get viscosity: */
		MaticeGetViscosity2d(&viscosity, this->matice, &epsilon[0]);
		
		
		/* Get Jacobian determinant: */
		TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/* Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
		   onto this scalar matrix, so that we win some computational time: */
		D_scalar=viscosity*thickness*gauss_weight*Jdet;
		for (i=0;i<3;i++){
			D[i][i]=D_scalar;
		}
		
		#ifdef _DEBUGELEMENTS_
		if(my_rank==RANK && TriaElementGetID(this)==ELID){ 
			printf("   gaussian loop %i\n",ig);
			printf("      thickness %g\n",thickness);
			printf("      slope [%g,%g]\n",slope[0],slope[1]);
			printf("      alpha2_list [%g,%g,%g]\n",alpha2_list[0],alpha2_list[1],alpha2_list[2]);
			printf("      epsilon [%g,%g,%g]\n",epsilon[0],epsilon[1],epsilon[2]);
			printf("      Matice: \n");
			MaticeEcho(this->matice);
			printf("\n      viscosity: %g \n",viscosity);
			printf("      jacobian: %g \n",Jdet);
			printf("      gauss_weight: %g \n",gauss_weight);
		}
		#endif

		/*Get B and Bprime matrices: */
		TriaElementGetB(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);
		TriaElementGetBPrime(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);

		/*Get L matrix if viscous basal drag present: */
		if((this->tria.friction_type==2) && (!this->tria.shelf)){
			TriaElementGetL(&L[0][0], &xyz_list[0][0], gauss_l1l2l3,NDOF2);
		}	
		/*  Do the triple product tB*D*Bprime: */
		TripleMultiply( &B[0][0],3,numdof,1,
					  &D[0][0],3,3,0,
					  &Bprime[0][0],3,numdof,0,
					  &Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<Ke_gg->nrows; i++){
			for (j=0;j<Ke_gg->nrows;j++){
				*(Ke_gg->terms+Ke_gg->nrows*i+j)+=Ke_gg_gaussian[i][j];
			}
		}	
			
		/*Now, take care of the basal friction if there is any: */
		if((!this->tria.shelf) && (this->tria.friction_type==2)){

			TriaElementGetParameterValue(&alpha2, &alpha2_list[0],gauss_l1l2l3);

			if (velocity){
				DL_scalar=alpha2*gauss_weight*Jdet;
			}
			else{
				DL_scalar=0;
			}
				
			for (i=0;i<2;i++){
				DL[i][i]=DL_scalar;
			}
			
			/*  Do the triple producte tL*D*L: */
			TripleMultiply( &L[0][0],2,numdof,1,
						&DL[0][0],2,2,0,
						&L[0][0],2,numdof,0,
						&Ke_gg_drag_gaussian[0][0],0);

			for( i=0; i<Ke_gg->nrows; i++){
				for (j=0;j<Ke_gg->nrows;j++){
					*(Ke_gg->terms+Ke_gg->nrows*i+j)+=Ke_gg_drag_gaussian[i][j];
				}
			}
		}


		#ifdef _DEBUGELEMENTS_
		if(my_rank==RANK && TriaElementGetID(this)==ELID){ 
			printf("      alpha2 %g\n",alpha2);
			printf("      B:\n");
			for(i=0;i<3;i++){
				for(j=0;j<NDOF2*numgrids;j++){
					printf("%g ",B[i][j]);
				}
				printf("\n");
			}
			printf("      Bprime:\n");
			for(i=0;i<3;i++){
				for(j=0;j<NDOF2*numgrids;j++){
					printf("%g ",Bprime[i][j]);
				}
				printf("\n");
			}
			printf("      L:\n");
			for(i=0;i<2;i++){
				for(j=0;j<NDOF2*numgrids;j++){
					printf("%g ",L[i][j]);
				}
				printf("\n");
			}

		}
		#endif
	} // for (ig=0; ig<num_gauss; ig++)


	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && TriaElementGetID(this)==ELID){ 
		printf("      Ke_gg->terms:\n");
		for( i=0; i<Ke_gg->nrows; i++){
			for (j=0;j<Ke_gg->nrows;j++){
				printf("%g ",*(Ke_gg->terms+Ke_gg->nrows*i+j));
			}
			printf("\n");
		}
		printf("      Ke_gg->row_indices:\n");
		for( i=0; i<Ke_gg->nrows; i++){
			printf("%i ",Ke_gg->row_indices[i]);
		}

	}
	#endif

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

    if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not build stiffness matrix.");
		return 0;
	}
	else{
		/*Assign output pointer: */
		*pKe_gg=Ke_gg;
		
		return 1;
	}

}

#undef __FUNCT__ 
#define __FUNCT__ "TriaElementCreatePVectorDiagnosticHoriz"

int TriaElementCreatePVectorDiagnosticHoriz( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs){

	int             i,j;
	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	TriaElement* this   = NULL;
	Matice        * matice = NULL;
	Matpar        * matpar = NULL;

	/* output: */
	ElemVector*     pe_g        = NULL;
	
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	
	/* parameters: */
	double  rho_ice;
	double  gravity=9.8;
	double  plastic_stress; 
	double  slope[NDOF2];
	double  driving_stress_baseline;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* grid data: */
	int    cid_list[numgrids];
	double xyz_list[numgrids][3];
	double T_bg_list[numgrids][9];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*element vector at the gaussian points: */
	double  pe_g_gaussian[numgrids*NDOF2];

	/*input parameters for structural analysis (diagnostic): */
	double* thickness_param=NULL;
	double  thickness_list[numgrids];
	double  thickness;
	double* surface_param=NULL;
	double  surface_list[numgrids];
	double* bed_param=NULL;
	double  bed_list[numgrids];
	double* basal_drag_param=NULL;
	double  K_list[numgrids];


	/*Some pointer intialization: */
	this = (TriaElement*)vpthis;
	matice=this->matice;
	matpar=this->matpar;

	/*First check that acceleration is not on: */
	if (this->tria.acceleration){
		*ppe_g=NULL;
		return;
	}

	/* The number of g-set dof for this element is 2*3 (vx and vy, horizontal velocities for ice, and 3 grids 
	 * per element: */
	numdof=numgrids*NDOF2;

	/* Allocate new load element  vector: */
	pe_g=NewElemVector(numdof);

	/* recover material parameters: */
	gravity=matpar->g;
	rho_ice=matpar->rho_ice;

	/* recover input parameters: */
	thickness_param=ParameterInputsRecover(inputs,"thickness");
	surface_param=ParameterInputsRecover(inputs,"surface");
	bed_param=ParameterInputsRecover(inputs,"bed");
	basal_drag_param=ParameterInputsRecover(inputs,"drag");

	/* Get all element grid data */
	noerr *= GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], &cid_list[0], this->pg, numgrids, 1 );

	/* Build the row index vector : */
	for (i=0;i<numgrids;i++){
		/* We use the structural dof list, from which we keep only the first two dof (x for vx, and y for vy).
		 * The other degrees of freedom will be spc'd for now.*/
		structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
		for (j=0;j<NDOF2;j++){
			*(pe_g->row_indices+i*NDOF2+j)=structural_dof_list[j];
		}
		if(thickness_param){
			dof=structural_dof_list[0];
			thickness_list[i]=*(thickness_param+dof);
		}
		else{
			thickness_list[i]=this->tria.h[i];
		}

		if(surface_param){
			dof=structural_dof_list[0];
			surface_list[i]=*(surface_param+dof);
		}
		else{
			surface_list[i]=this->tria.s[i];
		}
		if(bed_param){
			dof=structural_dof_list[0];
			bed_list[i]=*(bed_param+dof);
		}
		else{
			bed_list[i]=this->tria.b[i];
		}
		if(basal_drag_param){
			dof=structural_dof_list[0];
			K_list[i]=*(basal_drag_param+dof);
		}
		else{
			K_list[i]=this->tria.k[i];
		}
	}

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && TriaElementGetID(this)==ELID){ 
		printf("gravity %g\n",gravity);
		printf("rho_ice %g\n",rho_ice);
		printf("thickness_list [%g,%g,%g]\n",thickness_list[0],thickness_list[1],thickness_list[2]);
		printf("surface_list [%g,%g,%g]\n",surface_list[0],surface_list[1],surface_list[2]);
		printf("bed_list [%g,%g,%g]\n",bed_list[0],bed_list[1],bed_list[2]);
		printf("K_list [%g,%g,%g]\n",K_list[0],K_list[1],K_list[2]);
	}
	#endif


	/* Get gaussian points and weights: */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2); /*We need higher order because our load is order 2*/

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && TriaElementGetID(this)==ELID){ 
		printf("   gaussian points: \n");
		for(i=0;i<num_gauss;i++){
			printf("    %g %g %g : %g\n",first_gauss_area_coord[i],second_gauss_area_coord[i],third_gauss_area_coord[i],gauss_weights[i]);
		}
	}
	#endif



	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Compute thickness at gaussian point: */
		TriaElementGetParameterValue(&thickness, &thickness_list[0],gauss_l1l2l3);
	
		TriaElementGetParameterDerivativeValue(&slope[0], &surface_list[0],&xyz_list[0][0], gauss_l1l2l3);
		
		/*In case we have plastic basal drag, compute plastic stress at gaussian point from k1, k2 and k3 fields in the 
		 * element itself: */
		if(this->tria.friction_type==1){
			TriaElementGetParameterValue(&plastic_stress, &K_list[0],gauss_l1l2l3);
		}

		/* Get Jacobian determinant: */
		TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		 /*Get nodal functions: */
		TriaElementGetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Compute driving stress: */
		driving_stress_baseline=rho_ice*gravity*thickness;


	 	#ifdef _DEBUGELEMENTS_
		if(my_rank==RANK && TriaElementGetID(this)==ELID){ 
			printf("      gaussian %i\n",ig);
			printf("      thickness %g\n",thickness);
			printf("      slope(%g,%g)\n",slope[0],slope[1]);
			printf("      Jdet %g\n",Jdet);
			printf("      gaussweigth %g\n",gauss_weight);
			printf("      l1l2l3 (%g,%g,%g)\n",l1l2l3[0],l1l2l3[1],l1l2l3[2]);
			if(this->tria.friction_type==1)printf("      plastic_stress(%g)\n",plastic_stress);
		}
		#endif

		/*Build pe_g_gaussian vector: */
		if(this->tria.friction_type==1){
			for (i=0;i<numgrids;i++){
				for (j=0;j<NDOF2;j++){
					pe_g_gaussian[i*NDOF2+j]=(-driving_stress_baseline*slope[j]-plastic_stress)*Jdet*gauss_weight*l1l2l3[i]; 
				}
			}
		}
		else {
			for (i=0;i<numgrids;i++){
				for (j=0;j<NDOF2;j++){
					pe_g_gaussian[i*NDOF2+j]=-driving_stress_baseline*slope[j]*Jdet*gauss_weight*l1l2l3[i];
				}
			}
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<pe_g->nrows; i++){
			*(pe_g->terms+i)+=*(pe_g_gaussian+i);
		}
	} //for (ig=0; ig<num_gauss; ig++){

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && TriaElementGetID(this)==ELID){ 
		printf("      pe_g->terms\n",ig);
		for( i=0; i<pe_g->nrows; i++){
			printf("%g ",*(pe_g->terms+i));
		}
		printf("\n");
		printf("      pe_g->row_indices\n",ig);
		for( i=0; i<pe_g->nrows; i++){
			printf("%i ",*(pe_g->row_indices+i));
		}
	}
	#endif


	cleanup_and_return: 

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

    if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not build load vector.");
		return 0;
	}
	else{
		#ifdef _DEBUG_
		ElemVectorEcho(pe_g);
		#endif

		/*Assign output pointer: */
		*ppe_g=pe_g;
		return 1;
	}
}



/*--------------------------------------------------
	TriaElementGetParameterValue
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementGetParameterValue"
int TriaElementGetParameterValue(double* pp, double* plist, double* gauss_l1l2l3){
	
	/*From grid values of parameter p (plist[0],plist[1],plist[2]), return parameter value at gaussian 
	 * point specifie by gauss_l1l2l3: */
	
	int noerr=1;

	/*nodal functions: */
	double l1l2l3[3];

	/*output: */
	double p;

	noerr = TriaElementGetNodalFunctions(l1l2l3, gauss_l1l2l3);
	if(!noerr)goto cleanup_and_return;

	p=l1l2l3[0]*plist[0]+l1l2l3[1]*plist[1]+l1l2l3[2]*plist[2];


	cleanup_and_return:
	if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not return parameter value.");
		return 0;
	}
	else{
		*pp=p;
		return 1;
	}
}


/*--------------------------------------------------
	TriaElementGetParameterDerivativeValue
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementGetParameterDerivativeValue"
int TriaElementGetParameterDerivativeValue(double* p, double* plist,double* xyz_list, double* gauss_l1l2l3){
	 
	/*From grid values of parameter p (plist[0],plist[1],plist[2]), return parameter derivative value at gaussian 
	 * point specified by gauss_l1l2l3:
	 *   dp/dx=plist[0]*dh1/dx+plist[1]*dh2/dx+plist[2]*dh3/dx
	 *   dp/dx=plist[0]*dh1/dx+plist[1]*dh2/dx+plist[2]*dh3/dx
	 *
	 * p is a vector of size 2x1 already allocated.
	 */
	
	int noerr=1;

	double dh1dh2dh3_basic[NDOF2][numgrids]; //nodal derivative functions in basic coordinate system.

	/*Get dh1dh2dh3 in basic coordinate system: */
	noerr=TriaElementGetNodalFunctionsDerivativesBasic(&dh1dh2dh3_basic[0][0],xyz_list, gauss_l1l2l3);
	if(!noerr)goto cleanup_and_return;

	*(p+0)=plist[0]*dh1dh2dh3_basic[0][0]+plist[1]*dh1dh2dh3_basic[0][1]+plist[2]*dh1dh2dh3_basic[0][2];
	*(p+1)=plist[0]*dh1dh2dh3_basic[1][0]+plist[1]*dh1dh2dh3_basic[1][1]+plist[2]*dh1dh2dh3_basic[1][2];


	cleanup_and_return:
	if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not return parameter derivative vector.");
		return 0;
	}
	else{
		return 1;
	}
}

/*--------------------------------------------------
	TriaElementGetStrainRate
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementGetStrainRate"
int TriaElementGetStrainRate(double* epsilon, double* velocity, double* xyz_list, double* gauss_l1l2l3){

	int noerr=1;
	int i;

	double B[3][NDOF2*numgrids];

	/*Get B matrix: */
	noerr=TriaElementGetB(&B[0][0], xyz_list, gauss_l1l2l3);
	if(!noerr)goto cleanup_and_return;

	#ifdef _DEBUG_
	printf("B for grid1 : [ %lf   %lf  \n",B[0][0],B[0][1]);
	printf("              [ %lf   %lf  \n",B[1][0],B[1][1]);
	printf("              [ %lf   %lf  ]\n",B[2][0],B[2][1]);

	printf("B for grid2 : [ %lf   %lf  \n",B[0][2],B[0][3]);
	printf("              [ %lf   %lf  \n",B[1][2],B[1][3]);
	printf("              [ %lf   %lf  ]\n",B[2][2],B[2][3]);

	printf("B for grid3 : [ %lf   %lf  \n",B[0][4],B[0][5]);
	printf("              [ %lf   %lf  \n",B[1][4],B[1][5]);
	printf("              [ %lf   %lf  ]\n",B[2][4],B[2][5]);
		
	for (i=0;i<numgrids;i++){
		printf("Velocity for grid %i %lf %lf\n",i,*(vxvy_list+2*i+0),*(vxvy_list+2*i+1));
	}
	#endif

	/*Multiply B by velocity, to get strain rate: */
	noerr=MatrixMultiply( &B[0][0],3,NDOF2*numgrids,0,
			              velocity,NDOF2*numgrids,1,0,
						  epsilon,0);
	if(!noerr)goto cleanup_and_return;


	cleanup_and_return:
	if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not compute strain rate.");
		return 0;
	}
	else{
		return 1;
	}

}


/*--------------------------------------------------
	TriaElementGetB
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementGetB"

int TriaElementGetB(double* B, double* xyz_list, double* gauss_l1l2l3){

	/*Compute B  matrix. B=[B1 B2 B3] where Bi is of size 3*NDOF2. 
	 * For grid i, Bi can be expressed in the basic coordinate system
	 * by: 
	 *       Bi_basic=[ dh/dx    0    ]
	 *                [   0    dh/dy  ]
	 *                [ 1/2*dh/dy  1/2*dh/dx  ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(NDOF2*numgrids)
	 */
	
	int noerr=1;
	int i;

	double dh1dh2dh3_basic[NDOF2][numgrids];


	/*Get dh1dh2dh3 in basic coordinate system: */
	noerr=TriaElementGetNodalFunctionsDerivativesBasic(&dh1dh2dh3_basic[0][0],xyz_list, gauss_l1l2l3);
	if(!noerr)goto cleanup_and_return;

	#ifdef _DEBUG_ 
	for (i=0;i<3;i++){
		printf("Node %i  dh/dx=%lf dh/dy=%lf \n",i,dh1dh2dh3_basic[0][i],dh1dh2dh3_basic[1][i]);
	}
	#endif

	/*Build B: */
	for (i=0;i<numgrids;i++){
		*(B+NDOF2*numgrids*0+NDOF2*i)=dh1dh2dh3_basic[0][i]; //B[0][NDOF2*i]=dh1dh2dh3_basic[0][i];
		*(B+NDOF2*numgrids*0+NDOF2*i+1)=0;
		*(B+NDOF2*numgrids*1+NDOF2*i)=0;
		*(B+NDOF2*numgrids*1+NDOF2*i+1)=dh1dh2dh3_basic[1][i];
		*(B+NDOF2*numgrids*2+NDOF2*i)=(float).5*dh1dh2dh3_basic[1][i]; 
		*(B+NDOF2*numgrids*2+NDOF2*i+1)=(float).5*dh1dh2dh3_basic[0][i]; 
	}

	
	cleanup_and_return:

	if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not compute B matrix.");
				return 0;
	}
	else{
		return 1;
	}
}

/*--------------------------------------------------
	TriaElementGetB_prog
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementGetB_prog"

int TriaElementGetB_prog(double* B, double* xyz_list, double* gauss_l1l2l3){

//	Compute B  matrix. B=[B1 B2 B3 ] where Bi is of size numgrids*NDOF
//	For grid i, Bi can be expressed in the basic coordinate system
//	by: 
//		   Bi_basic=[dh/dx]
//                   [dh/dy]
//	where h is the interpolation function for grid i.
//
	
	int noerr=1;
	int i;

	double dh1dh2dh3_basic[NDOF2][numgrids];


	/*Get dh1dh2dh3 in basic coordinate system: */
	noerr=TriaElementGetNodalFunctionsDerivativesBasic(&dh1dh2dh3_basic[0][0],xyz_list, gauss_l1l2l3);
	if(!noerr)goto cleanup_and_return;

	#ifdef _DEBUG_ 
	for (i=0;i<3;i++){
		printf("Node %i  dh/dx=%lf dh/dy=%lf \n",i,dh1dh2dh3_basic[0][i],dh1dh2dh3_basic[1][i]);
	}
	#endif

	/*Build B: */
	for (i=0;i<numgrids;i++){
		*(B+numgrids*0+i)=dh1dh2dh3_basic[0][i];
		*(B+numgrids*1+i)=dh1dh2dh3_basic[1][i];
	}
	
	cleanup_and_return:

	if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not compute B matrix.");
				return 0;
	}
	else{
		return 1;
	}
}
/*--------------------------------------------------
	TriaElementGetBPrime
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementGetBPrime"

int TriaElementGetBPrime(double* Bprime, double* xyz_list, double* gauss_l1l2l3){

	/*Compute B'  matrix. B'=[B1' B2' B3'] where Bi' is of size 3*NDOF2. 
	 * For grid i, Bi' can be expressed in the basic coordinate system
	 * by: 
	 *       Bi_prime__basic=[ 2*dh/dx dh/dy ]
	 *                       [ dh/dx  2*dh/dy]
	 *                       [dh/dy dh/dx]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B' has been allocated already, of size: 3x(NDOF2*numgrids)
	 */
	
	int noerr=1;
	int i;

	/*Same thing in the basic coordinate system: */
	double dh1dh2dh3_basic[NDOF2][numgrids];


	/*Get dh1dh2dh3 in basic coordinates system : */
	noerr=TriaElementGetNodalFunctionsDerivativesBasic(&dh1dh2dh3_basic[0][0],xyz_list,gauss_l1l2l3);
	if(!noerr)goto cleanup_and_return;

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(Bprime+NDOF2*numgrids*0+NDOF2*i)=2*dh1dh2dh3_basic[0][i]; 
		*(Bprime+NDOF2*numgrids*0+NDOF2*i+1)=dh1dh2dh3_basic[1][i]; 
		*(Bprime+NDOF2*numgrids*1+NDOF2*i)=dh1dh2dh3_basic[0][i]; 
		*(Bprime+NDOF2*numgrids*1+NDOF2*i+1)=2*dh1dh2dh3_basic[1][i]; 
		*(Bprime+NDOF2*numgrids*2+NDOF2*i)=dh1dh2dh3_basic[1][i]; 
		*(Bprime+NDOF2*numgrids*2+NDOF2*i+1)=dh1dh2dh3_basic[0][i]; 
	}


	cleanup_and_return:

	if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not compute B' matrix.");
				return 0;
	}
	else{
		return 1;
	}
}

/*--------------------------------------------------
	TriaElementGetL
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementGetL"

int TriaElementGetL(double* L, double* xyz_list, double* gauss_l1l2l3,int numdof){

	/*Compute L  matrix. L=[L1 L2 L3] where Li is square and of size numdof. 
	 * For grid i, Li can be expressed in the basic coordinate system
	 * by: 
	 *       numdof=1: 
	 *       Li_basic=h;
	 *       numdof=2:
	 *       Li_basic=[ h    0    ]
	 *                [   0   h  ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume L has been allocated already, of size: numgrids (numdof=1), or numdofx(numdof*numgrids) (numdof=2)
	 */
	
	int noerr=1;
	int i;

	double l1l2l3[3];


	/*Get l1l2l3 in basic coordinate system: */
	noerr = TriaElementGetNodalFunctions(l1l2l3, gauss_l1l2l3);
	if(!noerr)goto cleanup_and_return;

	#ifdef _DELUG_ 
	for (i=0;i<3;i++){
		printf("Node %i  h=%lf \n",i,l1l2l3[i]);
	}
	#endif

	/*Build L: */
	if(numdof==1){
		for (i=0;i<numgrids;i++){
			L[i]=l1l2l3[i]; 
		}
	}
	else{
		for (i=0;i<numgrids;i++){
			*(L+numdof*numgrids*0+numdof*i)=l1l2l3[i]; //L[0][NDOF2*i]=dh1dh2dh3_basic[0][i];
			*(L+numdof*numgrids*0+numdof*i+1)=0;
			*(L+numdof*numgrids*1+numdof*i)=0;
			*(L+numdof*numgrids*1+numdof*i+1)=l1l2l3[i];
		}
	}

	
	cleanup_and_return:

	if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not compute L matrix.");
				return 0;
	}
	else{
		return 1;
	}
}

/*--------------------------------------------------
	TriaElementGetJacobianDeterminant
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementGetJacobianDeterminant" 
int TriaElementGetJacobianDeterminant(double*  Jdet, double* xyz_list,double* gauss_l1l2l3){

	/*The Jacobian determinant is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	double x1,x2,x3,y1,y2,y3;
	
	x1=*(xyz_list+3*0+0);
	y1=*(xyz_list+3*0+1);
	x2=*(xyz_list+3*1+0);
	y2=*(xyz_list+3*1+1);
	x3=*(xyz_list+3*2+0);
	y3=*(xyz_list+3*2+1);


	*Jdet=sqrt(3.0)/6.0*((x2-x1)*(y3-y1)-(x3-x1)*(y2-y1));

	if(Jdet<0){
		printf("%s%s\n",__FUNCT__," error message: negative jacobian determinant!");
	}
	
	return 1;
}

/*--------------------------------------------------
	TriaElementGetNodalFunctions
  --------------------------------------------------*/
int TriaElementGetNodalFunctions(double* l1l2l3, double* gauss_l1l2l3){
	
	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	/*First nodal function: */
	l1l2l3[0]=gauss_l1l2l3[0];

	/*Second nodal function: */
	l1l2l3[1]=gauss_l1l2l3[1];

	/*Third nodal function: */
	l1l2l3[2]=gauss_l1l2l3[2];

	return 1;
}

/*--------------------------------------------------
	TriaElementGetNodalFunctionsDerivativesBasic
  --------------------------------------------------*/
int TriaElementGetNodalFunctionsDerivativesBasic(double* dh1dh2dh3_basic,double* xyz_list, double* gauss_l1l2l3){
	
	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * basic coordinate system: */

	int noerr=1;
	int i;

	double dh1dh2dh3_param[NDOF2][numgrids];
	double Jinv[NDOF2][NDOF2];


	/*Get derivative values with respect to parametric coordinate system: */
	noerr=TriaElementGetNodalFunctionsDerivativesParams(&dh1dh2dh3_param[0][0], gauss_l1l2l3); 
	if(!noerr)goto cleanup_and_return;

	/*Get Jacobian invert: */
	noerr=TriaElementGetJacobianInvert(&Jinv[0][0], xyz_list, gauss_l1l2l3);
	if(!noerr)goto cleanup_and_return;

	/*Build dh1dh2dh3_basic: 
	 *
	 * [dhi/dx]= Jinv*[dhi/dr]
	 * [dhi/dy]       [dhi/ds]
	 */

	for (i=0;i<numgrids;i++){
		*(dh1dh2dh3_basic+numgrids*0+i)=Jinv[0][0]*dh1dh2dh3_param[0][i]+Jinv[0][1]*dh1dh2dh3_param[1][i];
		*(dh1dh2dh3_basic+numgrids*1+i)=Jinv[1][0]*dh1dh2dh3_param[0][i]+Jinv[1][1]*dh1dh2dh3_param[1][i];
	}

	cleanup_and_return:
	if(!noerr){
		return 0;
	}
	else{
		return 1;
	}
}

/*--------------------------------------------------
	TriaElementGetNodalFunctionsDerivativesParams
  --------------------------------------------------*/
int TriaElementGetNodalFunctionsDerivativesParams(double* dl1dl2dl3,double* gauss_l1l2l3){
	
	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * natural coordinate system) at the gaussian point. */

	double sqrt3=sqrt(3.0);

	/*First nodal function: */
	*(dl1dl2dl3+numgrids*0+0)=-1.0/2.0; 
	*(dl1dl2dl3+numgrids*1+0)=-1.0/(2.0*sqrt3);

	/*Second nodal function: */
	*(dl1dl2dl3+numgrids*0+1)=1.0/2.0;
	*(dl1dl2dl3+numgrids*1+1)=-1.0/(2.0*sqrt3);

	/*Third nodal function: */
	*(dl1dl2dl3+numgrids*0+2)=0;
	*(dl1dl2dl3+numgrids*1+2)=1.0/sqrt3;


	return 1;
}

/*--------------------------------------------------
	TriaElementGetJacobianInvert
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementGetJacobianInvert"
int TriaElementGetJacobianInvert(double*  Jinv, double* xyz_list,double* gauss_l1l2l3){

	int noerr=1;
	double Jdet;
	
	/*Call Jacobian routine to get the jacobian:*/
	noerr=TriaElementGetJacobian(Jinv, xyz_list, gauss_l1l2l3);
	if(!noerr)goto cleanup_and_return;

	/*Invert Jacobian matrix: */
	noerr=MatrixInverse(Jinv,NDOF2,NDOF2,NULL,0,&Jdet);
		
	cleanup_and_return:
	if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not compute Jacobian matrix.");
		return 0;
	}
	else{
		return 1;
	}
}

/*--------------------------------------------------
	TriaElementGetJacobian
  --------------------------------------------------*/
int TriaElementGetJacobian(double* J, double* xyz_list,double* gauss_l1l2l3){

	/*The Jacobian is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	double x1,y1,x2,y2,x3,y3;
	double sqrt3=sqrt(3.0);
	
	x1=*(xyz_list+numgrids*0+0);
	y1=*(xyz_list+numgrids*0+1);
	x2=*(xyz_list+numgrids*1+0);
	y2=*(xyz_list+numgrids*1+1);
	x3=*(xyz_list+numgrids*2+0);
	y3=*(xyz_list+numgrids*2+1);


	*(J+NDOF2*0+0)=1.0/2.0*(x2-x1);
	*(J+NDOF2*1+0)=sqrt3/6.0*(2*x3-x1-x2);
	*(J+NDOF2*0+1)=1.0/2.0*(y2-y1);
	*(J+NDOF2*1+1)=sqrt3/6.0*(2*y3-y1-y2);
		
	return 1;
}



/*--------------------------------------------------
	TriaElementCreateDuVector
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementCreateDuVector"
int TriaElementCreateDuVector( ElemVector* *pdue_g, void* vpthis, double* velocity, double* obs_velocity, ParameterInputs* inputs, int analysis_type){
					
	int             i,j;
	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	TriaElement* this   = NULL;

	/* local declarations */

	/* output: */
	ElemVector*     due_g        = NULL;
	
	int             num_dof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	/* grid data: */
	int    cid_list[numgrids];
	double xyz_list[numgrids][3];
	double T_bg_list[numgrids][9];
	double vx_list[numgrids];
	double vy_list[numgrids];
	double obs_vx_list[numgrids];
	double obs_vy_list[numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	
	/* specific gaussian point: */
	double   gauss_weight;
	double  gauss_l1l2l3[3];

	/* parameters: */
	double  velocity_x,velocity_y,velocity_mag;
	double  obs_velocity_x,obs_velocity_y,obs_velocity_mag;

	/*element vector at the gaussian points: */
	double  due_g_gaussian[numgrids*NDOF2];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*relative and algorithmic fitting: */
	double meanvel=0;
	double epsvel=0;
	double scalex=1;
	double scaley=1;
	double scale=0;
	double* fit_param=NULL;
	double  fit=-1;


	/*Some pointer intialization: */
	this = (TriaElement*)vpthis;

	/* The number of g-set dof for this element is 2*3 (vx and vy, horizontal velocities for ice, and 3 grids 
	 * per element: */
	num_dof=numgrids*NDOF2;

	/* Allocate new stiffness element  matrix: */
	due_g=NewElemVector(num_dof);
	
	/* Recover input data: */
	fit_param=ParameterInputsRecover(inputs,"fit");
	if(fit_param){
		fit=*fit_param;
	}
	else{
		fit=this->tria.fit;
	}
	
	/* Get all element grid data */
	noerr = GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], cid_list, this->pg, numgrids, 1 );

	/* Build the row index vector : */
	for (i=0;i<numgrids;i++){
		/* We use the structural dof list, from which we keep only the first two dof (x for vx, and y for vy).
		 * The other degrees of freedom will be spc'd for now.*/
		structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
		for (j=0;j<NDOF2;j++){
			dof=structural_dof_list[j];
			*(due_g->row_indices+i*NDOF2+j)=dof;
		}
		dof=structural_dof_list[0];
		vx_list[i]=*(velocity+dof);
		obs_vx_list[i]=*(obs_velocity+dof);
		dof=structural_dof_list[1];
		vy_list[i]=*(velocity+dof);
		obs_vy_list[i]=*(obs_velocity+dof);
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);
	
	#ifdef _DEBUG_ 
	for (i=0;i<num_gauss;i++){
		printf("Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(gauss_weights+i));
	}
	#endif

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		
		/*Compute velocities at gaussian point: */
		noerr = TriaElementGetParameterValue(&velocity_x, &vx_list[0],gauss_l1l2l3);
		if(!noerr)goto cleanup_and_return;
		noerr = TriaElementGetParameterValue(&velocity_y, &vy_list[0],gauss_l1l2l3);
		if(!noerr)goto cleanup_and_return;
		#ifdef _DEBUG_ 
			printf("Velocity: %lf %lf\n", velocity_x,velocity_y);
		#endif
	
		/*Compute obs_velocities at gaussian point: */
		noerr = TriaElementGetParameterValue(&obs_velocity_x, &obs_vx_list[0],gauss_l1l2l3);
		if(!noerr)goto cleanup_and_return;
		noerr = TriaElementGetParameterValue(&obs_velocity_y, &obs_vy_list[0],gauss_l1l2l3);
		if(!noerr)goto cleanup_and_return;

		#ifdef _DEBUG_ 
			printf("Observed velocity: %g %g\n", obs_velocity_x,obs_velocity_y);
		#endif

		/* Get Jacobian determinant: */
		TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		#ifdef _DEBUG_ 
		printf("Element id %i Jacobian determinant: %lf\n",TriaElementGetID(this),Jdet);
		#endif
		
		/* Get nodal functions value at gaussian point:*/
		TriaElementGetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Build due_g_gaussian vector: we have three cases here, according to which type of misfit we are using. */
		if(fit==0){
			/*We are using an absolute misfit: */
			for (i=0;i<numgrids;i++){
				due_g_gaussian[i*NDOF2+0]=(velocity_x-obs_velocity_x)*Jdet*gauss_weight*l1l2l3[i]; 
				due_g_gaussian[i*NDOF2+1]=(velocity_y-obs_velocity_y)*Jdet*gauss_weight*l1l2l3[i]; 
			}
		}
		else if(fit==1){
			/*We are using a relative misfit: */
			meanvel=this->tria.meanvel;
			epsvel=this->tria.epsvel;
			scalex=pow(meanvel/(obs_velocity_x+epsvel),2);
			scaley=pow(meanvel/(obs_velocity_y+epsvel),2);
			if(obs_velocity_x==0)scalex=0;
			if(obs_velocity_y==0)scaley=0;
			for (i=0;i<numgrids;i++){
				due_g_gaussian[i*NDOF2+0]=scalex*(velocity_x-obs_velocity_x)*Jdet*gauss_weight*l1l2l3[i]; 
				due_g_gaussian[i*NDOF2+1]=scaley*(velocity_y-obs_velocity_y)*Jdet*gauss_weight*l1l2l3[i]; 
			}
		}
		else if(fit==2){

			/*We are using a logarithmic misfit: */
			meanvel=this->tria.meanvel;
			epsvel=this->tria.epsvel;
			velocity_mag=sqrt(pow(velocity_x,2)+pow(velocity_y,2))+epsvel; //epsvel to avoid velocity being nil.
			obs_velocity_mag=sqrt(pow(obs_velocity_x,2)+pow(obs_velocity_y,2))+epsvel; //epsvel to avoid observed velocity being nil.
			scale=8*pow(meanvel,2)/pow(velocity_mag,2)*log(velocity_mag/obs_velocity_mag);
			for (i=0;i<numgrids;i++){
				due_g_gaussian[i*NDOF2+0]=scale*velocity_x*Jdet*gauss_weight*l1l2l3[i]; 
				due_g_gaussian[i*NDOF2+1]=scale*velocity_y*Jdet*gauss_weight*l1l2l3[i]; 
			}
		}
		else{
			/*Not supported yet! : */
			printf("%s%s\n",__FUNCT__," error message: unsupported type of fitting.");
			noerr=0;goto cleanup_and_return;
		}
		
		
		/*Add due_g_gaussian vector to due_g: */
		for( i=0; i<due_g->nrows; i++){
			*(due_g->terms+i)+=*(due_g_gaussian+i);
		}
	}
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

    if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not build velocity difference vector.");
		return 0;
	}
	else{
		#ifdef _DEBUG_
		ElemVectorEcho(due_g);
		#endif

		/*Assign output pointer: */
		*pdue_g=due_g;

		return 1;
	}
}


/*--------------------------------------------------
	TriaElementCreateGradjVectors
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementCreateGradVectors"
int TriaElementCreateGradjVectors( ElemVector* *pgradje_g, void* vpthis, double* velocity, double* adjoint, ParameterInputs* inputs,int analysis_type,char* control_type){

	int noerr=1;
	
	if strcasecmp_eq(control_type,"drag"){
		noerr=TriaElementCreateGradjVectorsDrag( pgradje_g, vpthis, velocity, adjoint, inputs,analysis_type);
	}
	else if strcasecmp_eq(control_type,"B"){
		noerr=TriaElementCreateGradjVectorsB( pgradje_g, vpthis, velocity, adjoint, inputs,analysis_type);
	}
	else{
		printf("%s%s\n",__FUNCT__," error message: control type not yet supported!");
		noerr=0;
	}
	return noerr;
}

/*--------------------------------------------------
	TriaElementCreateGradjVectorsDrag
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementCreateGradVectorsDrag"
int TriaElementCreateGradjVectorsDrag( ElemVector* *pgradje_g, void* vpthis, double* velocity, double* adjoint, ParameterInputs* inputs,int analysis_type){

	int             i,j;
	int             noerr;

	/* vpthis for polymorphic function compatibility */
	TriaElement* this   = NULL;
	Matice        * matice = NULL;
	Matpar        * matpar = NULL;

	/* local declarations */

	/* output: */
	ElemVector*     gradje_g        = NULL;
	
	int             num_dof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	/* grid data: */
	int    cid_list[numgrids];
	double xyz_list[numgrids][3];
	double T_bg_list[numgrids][9];
	double vx_list[numgrids];
	double vy_list[numgrids];
	double vxvy_list[numgrids][2];
	double adjx_list[numgrids];
	double adjy_list[numgrids];

	double* basal_drag_param=NULL;
	double K_list[numgrids];
	double k;

	double* bed_param=NULL;
	double bed_list[numgrids];
	double* thickness_param=NULL;
	double thickness_list[numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	
	/* specific gaussian point: */
	double   gauss_weight;
	double  gauss_l1l2l3[3];

	/* parameters: */
	double  viscosity2;
	double  dvx[NDOF2];
	double  dvy[NDOF2]; 
	double  dadjx[NDOF2];
	double  dadjy[NDOF2];
	double  vx,vy;
	double  lambda,mu;
	double  bed,thickness,Neff;
	
	/*drag: */
	double  pcoeff,qcoeff;
	double gravity=9.8;
	double rho_ice,rho_water;
	double alpha_complement_list[numgrids];
	double alpha_complement;


	/*element vector at the gaussian points: */
	double  gradje_g_gaussian[numgrids];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/


	/*Some pointer intialization: */
	this = (TriaElement*)vpthis;
	matice=this->matice;
	matpar=this->matpar;

	/*recover material parameters: */
	rho_ice=matpar->rho_ice;
	rho_water=matpar->rho_water;
			
	/* The number of g-set dof for this element is 2*3 (vx and vy, horizontal velocities for ice, and 3 grids 
	 * per element: */
	num_dof=numgrids*NDOF2;

	/* Allocate element vector: */
	gradje_g=NewElemVector(numgrids);

	/* Get all element grid data */
	noerr *= GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], cid_list, this->pg, numgrids, 1 );
	
	/* recover input parameters: */
	basal_drag_param=ParameterInputsRecover(inputs,"drag");
	bed_param=ParameterInputsRecover(inputs,"bed");
	thickness_param=ParameterInputsRecover(inputs,"thickness");

	/* Build the row index vector : */
	for (i=0;i<numgrids;i++){
		/* We use the structural dof list, from which we keep only the first two dof (x for vx, and y for vy).
		 * The other degrees of freedom will be spc'd for now.*/
		structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
		for (j=0;j<NDOF2;j++){
			dof=structural_dof_list[j];
			vxvy_list[i][j]=*(velocity+dof);
		}

		dof=structural_dof_list[0];
		vx_list[i]=*(velocity+dof);
		adjx_list[i]=*(adjoint+dof);

		dof=structural_dof_list[1];
		vy_list[i]=*(velocity+dof);
		adjy_list[i]=*(adjoint+dof);


		dof=structural_dof_list[0];
		*(gradje_g->row_indices+i)=dof;
		
		if(basal_drag_param){
			K_list[i]=*(basal_drag_param+dof);
		}
		else{
			K_list[i]=this->tria.k[i];
		}
		if(bed_param){
			bed_list[i]=*(bed_param+dof);
		}
		else{
			bed_list[i]=this->tria.b[i];
		}
		if(thickness_param){
			thickness_list[i]=*(thickness_param+dof);
		}
		else{
			thickness_list[i]=this->tria.h[i];
		}

	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 4);
	#ifdef _DEBUG_ 
	for (i=0;i<num_gauss;i++){
		printf("Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(gauss_weights+i));
	}
	#endif

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Build alpha_complement_list: */
		if (!this->tria.shelf && (this->tria.friction_type==2)){
		
			/*Allocate friction object: */
			Friction* friction=NewFriction();
			
			/*Initialize all fields: */
			friction->element_type=xmalloc((strlen("2d")+1)*sizeof(char));
			strcpy(friction->element_type,"2d");
			
			friction->gravity=gravity;
			friction->rho_ice=rho_ice;
			friction->rho_water=rho_water;
			friction->K=&K_list[0];
			friction->bed=&bed_list[0];
			friction->thickness=&thickness_list[0];
			friction->velocities=&vxvy_list[0][0];
			friction->p=this->tria.p;
			friction->q=this->tria.q;

			if(friction->p!=1){
				printf("Non-linear friction not supported yet in control methods!\n");
				noerr=0;goto cleanup_and_return;
			}

			/*Compute alpha complement: */
			FrictionGetAlphaComplement(&alpha_complement_list[0],friction);

			/*Erase friction object: */
			DeleteFriction(&friction);
		}
		else{
			alpha_complement_list[0]=0;
			alpha_complement_list[1]=0;
			alpha_complement_list[2]=0;
		}
		
		/*Recover alpha_complement and k: */
		noerr = TriaElementGetParameterValue(&alpha_complement, &alpha_complement_list[0],gauss_l1l2l3);
		noerr = TriaElementGetParameterValue(&k, &K_list[0],gauss_l1l2l3);
		#ifdef _DEBUG_ 
			printf("Drag complement: %20.20lf Drag: %20.20lf\n",alpha_complement,k);
		#endif

		/*recover lambda and mu: */
		noerr = TriaElementGetParameterValue(&lambda, &adjx_list[0],gauss_l1l2l3);
		if(!noerr)goto cleanup_and_return;
		
		noerr = TriaElementGetParameterValue(&mu, &adjy_list[0],gauss_l1l2l3);
		if(!noerr)goto cleanup_and_return;

		#ifdef _DEBUG_ 
			printf("Adjoint vector %20.20lf %20.20lf\n",lambda,mu);
		#endif
			
		/*recover vx and vy: */
		noerr = TriaElementGetParameterValue(&vx, &vx_list[0],gauss_l1l2l3);
		if(!noerr)goto cleanup_and_return;
		
		noerr = TriaElementGetParameterValue(&vy, &vy_list[0],gauss_l1l2l3);
		if(!noerr)goto cleanup_and_return;

		#ifdef _DEBUG_ 
			printf("Velocity vector %20.20lf %20.20lf\n",vx,vy);
		#endif

		/* Get Jacobian determinant: */
		noerr=TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		if(!noerr)goto cleanup_and_return;
		#ifdef _DEBUG_ 
		printf("Element id %i Jacobian determinant: %lf\n",TriaElementGetID(this),Jdet);
		#endif
		
		/* Get nodal functions value at gaussian point:*/
		noerr = TriaElementGetNodalFunctions(l1l2l3, gauss_l1l2l3);
		if(!noerr)goto cleanup_and_return;

		
		/*Build gradje_g_gaussian vector: */
		for (i=0;i<numgrids;i++){
			gradje_g_gaussian[i]=-2*k*alpha_complement*( (lambda*vx+mu*vy))*Jdet*gauss_weight*l1l2l3[i]; 
		}
		
		/*Add gradje_g_gaussian vector to gradje_g: */
		for( i=0; i<gradje_g->nrows; i++){
			*(gradje_g->terms+i)+=*(gradje_g_gaussian+i);
		}
	}
	
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

    if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not build velocity difference vector.");
		return 0;
	}
	else{
		#ifdef _DEBUG_
			ElemVectorEcho(gradje_g);
		#endif
		

		/*Assign output pointer: */
		*pgradje_g=gradje_g;

		return 1;
	}
}


/*--------------------------------------------------
	TriaElementCreateGradjVectorsB
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementCreateGradjVectorsB"
int TriaElementCreateGradjVectorsB( ElemVector* *pgradje_g, void* vpthis, double* velocity, double* adjoint, ParameterInputs* inputs,int analysis_type){

	int             i,j;
	int             noerr;

	/* vpthis for polymorphic function compatibility */
	TriaElement* this   = NULL;

	/* local declarations */

	/* output: */
	ElemVector*     gradje_g        = NULL;
	
	int             num_dof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	/* grid data: */
	int    cid_list[numgrids];
	double xyz_list[numgrids][3];
	double T_bg_list[numgrids][9];
	double vx_list[numgrids];
	double vy_list[numgrids];
	double vxvy_list[numgrids][2];
	double adjx_list[numgrids];
	double adjy_list[numgrids];

	double* thickness_param=NULL;
	double thickness_list[numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	
	/* specific gaussian point: */
	double   gauss_weight;
	double  gauss_l1l2l3[3];

	/* parameters: */
	double  viscosity2;
	double  dvx[NDOF2];
	double  dvy[NDOF2]; 
	double  dadjx[NDOF2];
	double  dadjy[NDOF2];
	double  vx,vy;
	double  lambda,mu;
	double  thickness;
	

	/*element vector at the gaussian points: */
	double  gradje_g_gaussian[numgrids];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/

	/*Some pointer intialization: */
	this = (TriaElement*)vpthis;

			
	/* The number of g-set dof for this element is 2*3 (vx and vy, horizontal velocities for ice, and 3 grids 
	 * per element: */
	num_dof=numgrids*NDOF2;

	/* Allocate element vector: */
	gradje_g=NewElemVector(numgrids);

	/* Get all element grid data */
	noerr = GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], cid_list, this->pg, numgrids, 1 ); if(!noerr)goto cleanup_and_return;
	
	/* recover input parameters: */
	thickness_param=ParameterInputsRecover(inputs,"drag");

	/* Build the row index vector : */
	for (i=0;i<numgrids;i++){
		/* We use the structural dof list, from which we keep only the first two dof (x for vx, and y for vy).
		 * The other degrees of freedom will be spc'd for now.*/
		structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
		for (j=0;j<NDOF2;j++){
			dof=structural_dof_list[j];
			vxvy_list[i][j]=*(velocity+dof);
		}

		dof=structural_dof_list[0];
		vx_list[i]=*(velocity+dof);
		adjx_list[i]=*(adjoint+dof);

		dof=structural_dof_list[1];
		vy_list[i]=*(velocity+dof);
		adjy_list[i]=*(adjoint+dof);

		dof=structural_dof_list[0];
		*(gradje_g->row_indices+i)=dof;
		
		if(thickness_param){
			thickness_list[i]=*(thickness_param+dof);
		}
		else{
			thickness_list[i]=this->tria.h[i];
		}

	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 4);
	#ifdef _DEBUG_ 
	for (i=0;i<num_gauss;i++){
		printf("Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(gauss_weights+i));
	}
	#endif

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Get thickness: */
		TriaElementGetParameterValue(&thickness, &thickness_list[0],gauss_l1l2l3);
	
		/*On an icesheet, gradjB can be adversely affected by viscosity2 and thickness, to the point where the gradient does not make any sense. Just stick to the iceshelf for now:*/
		/*if(!this->ctrice3.shelf){
			thickness=0; //this will ensure gradjbe_g_gaussian is nil;
		}*/
		
		/*Get strain rate, if velocity has been supplied: */
		noerr=TriaElementGetStrainRate(&epsilon[0],&vxvy_list[0][0],&xyz_list[0][0],gauss_l1l2l3); if(!noerr)goto cleanup_and_return;
	
		/*Get viscosity2: */
		noerr=MaticeGetViscosity2(&viscosity2, this->matice, &epsilon[0]); if(!noerr)goto cleanup_and_return;
		
		/*Get dvx: */
		noerr=TriaElementGetParameterDerivativeValue(&dvx[0], &vx_list[0],&xyz_list[0][0], gauss_l1l2l3); if(!noerr)goto cleanup_and_return;

		/*Get dvy: */
		noerr=TriaElementGetParameterDerivativeValue(&dvy[0], &vy_list[0],&xyz_list[0][0], gauss_l1l2l3); if(!noerr)goto cleanup_and_return;

		/*Get dadjx: */
		noerr=TriaElementGetParameterDerivativeValue(&dadjx[0], &adjx_list[0],&xyz_list[0][0], gauss_l1l2l3); if(!noerr)goto cleanup_and_return;

		/*Get dadjy: */
		noerr=TriaElementGetParameterDerivativeValue(&dadjy[0], &adjy_list[0],&xyz_list[0][0], gauss_l1l2l3); if(!noerr)goto cleanup_and_return;


		/* Get Jacobian determinant: */
		noerr=TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3); if(!noerr)goto cleanup_and_return;
		
		/* Get nodal functions value at gaussian point:*/
		noerr = TriaElementGetNodalFunctions(l1l2l3, gauss_l1l2l3); if(!noerr)goto cleanup_and_return;

		#ifdef _DEBUG_
			_printf_("viscosity2 %g thickness %g dvx [%g %g] dvy [%g %g]  dadjx [%g %g] dadjy[%g %g]\n",viscosity2,thickness,dvx[0],dvx[1],dvy[0],dvy[1],dadjx[0],dadjx[1],dadjy[0],dadjy[1]);
		#endif

		/*Build gradje_g_gaussian vector: */
		for (i=0;i<numgrids;i++){
			gradje_g_gaussian[i]=viscosity2*thickness*( (2*dvx[0]+dvy[1])*2*dadjx[0]+(dvx[1]+dvy[0])*(dadjx[1]+dadjy[0])+(2*dvy[1]+dvx[0])*2*dadjy[1])*Jdet*gauss_weight*l1l2l3[i]; 
		}

		/*Add gradje_g_gaussian vector to gradje_g: */
		for( i=0; i<gradje_g->nrows; i++){
			*(gradje_g->terms+i)+=*(gradje_g_gaussian+i);
		}
	}
	
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

    if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not build gradjB vector.");
		return 0;
	}
	else{
		#ifdef _DEBUG_
			ElemVectorEcho(gradje_g);
		#endif
		/*Assign output pointer: */
		*pgradje_g=gradje_g;

		return 1;
	}
}




/*--------------------------------------------------
	TriaElementMisfit
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "TriaElementMisfit"
int TriaElementMisfit( double* pJelem,void* vpthis, double* velocity, double* obs_velocity, ParameterInputs* inputs, int analysis_type){

	int             i,j;
	int             noerr;

	/* vpthis for polymorphic function compatibility */
	TriaElement* this   = NULL;

	/* local declarations */

	/* output: */
	double Jelem=0;
	
	int             num_dof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	/* grid data: */
	int    cid_list[numgrids];
	double xyz_list[numgrids][3];
	double T_bg_list[numgrids][9];
	double vx_list[numgrids];
	double vy_list[numgrids];
	double obs_vx_list[numgrids];
	double obs_vy_list[numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	
	/* specific gaussian point: */
	double   gauss_weight;
	double  gauss_l1l2l3[3];

	/* parameters: */
	double  velocity_x,velocity_y,velocity_mag;
	double  obs_velocity_x,obs_velocity_y,obs_velocity_mag;

	/* Jacobian: */
	double Jdet;
	
	/*relative and logarithmic control method :*/
	double meanvel=0;
	double epsvel=0;
	double scalex=1;
	double scaley=1;
	double* fit_param=NULL;
	double  fit=-1;


	/*Some pointer intialization: */
	this = (TriaElement*)vpthis;

	/* The number of g-set dof for this element is 2*3 (vx and vy, horizontal velocities for ice, and 3 grids 
	 * per element: */
	num_dof=numgrids*NDOF2;

	/* Get all element grid data */
	noerr *= GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], cid_list, this->pg, numgrids, 1 );

	/* Recover input data: */
	fit_param=ParameterInputsRecover(inputs,"fit");
	if(fit_param){
		fit=*fit_param;
	}
	else{
		fit=this->tria.fit;
	}
	
	/* Build the row index vector : */
	for (i=0;i<numgrids;i++){
		/* We use the structural dof list, from which we keep only the first two dof (x for vx, and y for vy).
		 * The other degrees of freedom will be spc'd for now.*/
		structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
		dof=structural_dof_list[0];
		vx_list[i]=*(velocity+dof);
		obs_vx_list[i]=*(obs_velocity+dof);
		dof=structural_dof_list[1];
		vy_list[i]=*(velocity+dof);
		obs_vy_list[i]=*(obs_velocity+dof);
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);
	
	#ifdef _DEBUG_ 
	for (i=0;i<num_gauss;i++){
		printf("Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(gauss_weights+i));
	}
	#endif

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		
		/*Compute velocities at gaussian point: */
		TriaElementGetParameterValue(&velocity_x, &vx_list[0],gauss_l1l2l3);
		TriaElementGetParameterValue(&velocity_y, &vy_list[0],gauss_l1l2l3);
		#ifdef _DEBUG_ 
			printf("Velocity: %lf %lf\n", velocity_x,velocity_y);
		#endif
	
		/*Compute obs_velocities at gaussian point: */
		TriaElementGetParameterValue(&obs_velocity_x, &obs_vx_list[0],gauss_l1l2l3);
		TriaElementGetParameterValue(&obs_velocity_y, &obs_vy_list[0],gauss_l1l2l3);
		#ifdef _DEBUG_ 
			printf("Observed velocity: %lf %lf\n", obs_velocity_x,obs_velocity_y);
		#endif

		/* Get Jacobian determinant: */
		noerr=TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		if(!noerr)goto cleanup_and_return;
		#ifdef _DEBUG_ 
		printf("Element id %i Jacobian determinant: %lf\n",TriaElementGetID(this),Jdet);
		#endif

		/*Differents misfits are allowed: */
		if(fit==0){
			/*Absolute misfit: */
			Jelem+=.5*(pow((velocity_x-obs_velocity_x),2)+pow((velocity_y-obs_velocity_y),2))*Jdet*gauss_weight;
		}
		else if(fit==1){
			/*Relative misfit: */
			meanvel=this->tria.meanvel;
			epsvel=this->tria.epsvel;
			scalex=pow(meanvel/(obs_velocity_x+epsvel),2);
			scaley=pow(meanvel/(obs_velocity_y+epsvel),2);
			if(obs_velocity_x==0)scalex=0;
			if(obs_velocity_y==0)scaley=0;
			Jelem+=.5*(scalex*pow((velocity_x-obs_velocity_x),2)+scaley*pow((velocity_y-obs_velocity_y),2))*Jdet*gauss_weight;
		}	
		else if(fit==2){
			/*Logarithmic misfit: */
			meanvel=this->tria.meanvel;
			epsvel=this->tria.epsvel;
			velocity_mag=sqrt(pow(velocity_x,2)+pow(velocity_y,2))+epsvel; //epsvel to avoid velocity being nil.
			obs_velocity_mag=sqrt(pow(obs_velocity_x,2)+pow(obs_velocity_y,2))+epsvel; //epsvel to avoid observed velocity being nil.
			Jelem+=4*pow(meanvel,2)*pow(log(velocity_mag/obs_velocity_mag),2);
		}
		else{
			printf("%s%s\n",__FUNCT__," error message: unsupported type of fitting.");
			noerr=0;goto cleanup_and_return;
		}

	}
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

    if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not compute velocity misfit.");
		return 0;
	}
	else{
		/*Assign output pointer: */
		*pJelem=Jelem;

		return 1;
	}
}
#undef NDOF1 
#undef NDOF2 
#undef numgrids



