
/*
	IcefrontLoad.c: 
*/

/*  IcefrontLoad: this object deals with boundary conditions at the icefront, 
 *  for each type of runs: dynamic water pressure for diagnostic runs, free or 
 *  set radiation for prognostic runs. */

/*For debugging purposes: */
#define ELID 41 //element for which to print debug statements
#define RANK 2 //rank of cpu for which to print debug statements.
//#define _DEBUGELEMENTS_
//#define _DEBUGGAUSS_


#include <stdarg.h>
#include <stdio.h>
#include <string.h>	
#include <math.h>

/* environment: */
#include "../../include/cielo_types.h"
#include "../../include/cielo_macros.h"
#include "../../include/cielo_externals.h"

/* hosting environment: */
#include "../../include/matlab_includes.h"

/* object handlers: */
#include "../../DataSet/DataSet.h"
#include "../../EnumDefinitions/EnumDefinitions.h"

/* shared utilities (includes alloc.h memory management): */
#include "../../shared/shared.h"

/* "virtual function" access: */
#include "./../VirtualFunctions.h"

/* object itself: */
#include "./IcefrontLoad.h"

/*Objects: */
#include "../objects.h"

#include "../ElementObjects/GetElementGridData.h"
#include "../ElementObjects/GaussPoints.h"

#define NDOF1 1
#define NDOF2 2


/*--------------------------------------------------
	NewIcefrontLoad
  --------------------------------------------------*/

IcefrontLoad* NewIcefrontLoad( void* vpicefront)
{

	Icefront* 		icefront = (Icefront*)vpicefront;
	IcefrontLoad*	this   = NULL;
	
	int	i;

	this=(IcefrontLoad*)xcalloc(1,sizeof(IcefrontLoad));	/* guaranteed NULL, 0 initialisation */

	/* include the "base class" in the "derived" object: */
	memcpy( &this->icefront, icefront, sizeof(Icefront));

	strcpy(this->name,"IcefrontLoad");
	for(i=0;i<MAX_ICEFRONT_GRIDS;i++){
		this->internalgrid_indices[i]=UNDEF;
		this->internalgrid_indicesb[i]=UNDEF;
		this->pg[i]=NULL;
		this->element_index=UNDEF;
		this->element_enum=UNDEF;
		this->element=NULL;
		this->matpar_index=UNDEF;
		this->matpar_enum=UNDEF;
		this->matpar=NULL;
}

	return this;
}

/*--------------------------------------------------
	DeleteIcefrontLoad
  --------------------------------------------------*/

void DeleteIcefrontLoad(void* *vpthis)
{
	/* vpthis for polymorphic function compatibility */

	IcefrontLoad* *pthis = (IcefrontLoad**)vpthis;
	IcefrontLoad* this = *pthis;

	xfree(vpthis);
}


/*--------------------------------------------------
	IcefrontLoadEcho
  --------------------------------------------------*/

void IcefrontLoadEcho(void* vpthis)
{
	/* vpthis for polymorphic function compatibility */
	IcefrontLoad* this = (IcefrontLoad*)vpthis;

	int	i,j;

	printf("\nIcefrontLoad echo:\n");
	printf("-------------------\n");
	printf("   base Icefront object:\n");
	IcefrontEcho( &this->icefront);
	printf("\n");

	printf("   name: %s\n" ,this->name);
	printf("   load point to following grids (pointer-based ops):\n");
	printf("    local grid number   corresponding grid id\n");
	printf("    -----------------   ---------------------\n");
	for ( i=0; i<MAX_ICEFRONT_GRIDS; i++) {
		if ( this->pg[i])
			printf("        %8d              %8d\n", i, InternalGridGetID( this->pg[i]));
	}
	if(this->element)printf("   load element: %p enum %i\n",this->element,this->element_enum);
	if(this->matpar)printf("   load matpar: %p enum %i\n",this->matpar,this->matpar_enum);

	return;
}

/*--------------------------------------------------
	IcefrontLoadSizeOfInDoubles
  --------------------------------------------------*/

int IcefrontLoadSizeOfInDoubles(void* vpthis)
{
	/* vpthis for polymorphic function compatibility */

	IcefrontLoad* this = (IcefrontLoad*)vpthis;
	int struct_doubles=0;

	struct_doubles = 
		(int)ceil((float)sizeof(*this)/(float)sizeof(double));

	#ifdef _DEBUG_
	printf("\n   IcefrontLoadSizeOfInDoubles diagnostics:\n");
	printf("      returning %d doubles\n", struct_doubles);
	#endif

	return struct_doubles;
}


/*--------------------------------------------------
	IcefrontLoadShrinkWrap
  --------------------------------------------------*/

void IcefrontLoadShrinkWrap(void* to, void* vpthis)
{
	/* no "extended" data means a simple copy will do: */

	memcpy( to, vpthis, sizeof(double)*IcefrontLoadSizeOfInDoubles(vpthis));
}


/*--------------------------------------------------
	IcefrontLoadUnwrap
  --------------------------------------------------*/

int  IcefrontLoadUnwrap(void* vpthis)
{
	/* no extended data means no internal pointers to reestablish  */

	return 1;
}

/*--------------------------------------------------
	IcefrontLoadMarshall
  --------------------------------------------------*/

void IcefrontLoadMarshall( char* *pbuffer, void* vpthis, int* size)
{
	/* vpthis for polymorphic function compatibility */
	IcefrontLoad* this = (IcefrontLoad*)vpthis;

	char* buffer=*pbuffer;

	int i, j, size_icefront=0; 
	int size_string1;
	int ig;

	int num_int=0, num_double=0;

	/* first marshall the embedded Icefront object: ... */
	IcefrontMarshall( &buffer, &this->icefront, &size_icefront);

	/* ... then marshall the IcefrontLoad member data: */

	size_string1=strlen(this->name)+1;
	cellmar(&size_string1,&buffer,INTEGER_FLAG);num_int++;
	cellmar(&this->name,&buffer,STRING_FLAG);
	for(i=0;i<MAX_ICEFRONT_GRIDS;i++){
		cellmar(&this->internalgrid_indices[i],&buffer,INTEGER_FLAG);num_int++;
		cellmar(&this->internalgrid_indicesb[i],&buffer,INTEGER_FLAG);num_int++;
		cellmar(&this->pg[i],&buffer,INTEGER_FLAG);num_int++;
	}
	cellmar(&this->element_index,&buffer,INTEGER_FLAG);num_int++;
	cellmar(&this->element_enum,&buffer,INTEGER_FLAG);num_int++;
	cellmar(&this->element,&buffer,INTEGER_FLAG);num_int++;

	cellmar(&this->matpar_index,&buffer,INTEGER_FLAG);num_int++;
	cellmar(&this->matpar_enum,&buffer,INTEGER_FLAG);num_int++;
	cellmar(&this->matpar,&buffer,INTEGER_FLAG);num_int++;
	
	/* return values: */
	*size=size_string1*sizeof(char)+ num_int*sizeof(int)+num_double*sizeof(double);
	*pbuffer=buffer; 

	return;
}

/*--------------------------------------------------
	IcefrontLoadDemarshall
  --------------------------------------------------*/

void* IcefrontLoadDemarshall( DataSet* ds, char* *pbuffer, int machine_flag)
{ 
	int i, j, ig;
	int size_string1;
	Icefront* picefront=NULL;	/* for embedded object */

	char* buffer=*pbuffer; 
	IcefrontLoad* this=NULL;

	/* first demarshall the embedded Icefront object: ... */

	picefront = IcefrontDemarshall( NULL, &buffer, machine_flag);
	/* create the derived object: */
	this = NewIcefrontLoad( picefront);
	/* and free icefront now that it's embedded in the derived object: */
	DeleteIcefront( (void**)&picefront);

	/* ... then demarshall the IcefrontLoad member data: */


	celldem(&size_string1,&buffer,INTEGER_FLAG,machine_flag);
	celldem(&this->name,&buffer,STRING_FLAG,machine_flag,size_string1);
	for(i=0;i<MAX_ICEFRONT_GRIDS;i++){
		celldem(&this->internalgrid_indices[i],&buffer,INTEGER_FLAG,machine_flag);
		celldem(&this->internalgrid_indicesb[i],&buffer,INTEGER_FLAG,machine_flag);
		celldem(&this->pg[i],&buffer,INTEGER_FLAG,machine_flag);
		this->pg[i]=NULL;
	}
	celldem(&this->element_index,&buffer,INTEGER_FLAG,machine_flag);
	celldem(&this->element_enum,&buffer,INTEGER_FLAG,machine_flag);
	celldem(&this->element,&buffer,INTEGER_FLAG,machine_flag);

	celldem(&this->matpar_index,&buffer,INTEGER_FLAG,machine_flag);
	celldem(&this->matpar_enum,&buffer,INTEGER_FLAG,machine_flag);
	celldem(&this->matpar,&buffer,INTEGER_FLAG,machine_flag);

	if (ds) IcefrontLoadAddToDataSet( ds, this);
	*pbuffer=buffer;  

	return (void*)this;
}

/*--------------------------------------------------
	IcefrontLoadGetID
  --------------------------------------------------*/

int IcefrontLoadGetID( void* vpthis )
{
	/* vpthis for polymorphic function compatibility */
	IcefrontLoad* this = (IcefrontLoad*)vpthis;

	return IcefrontGetID(&this->icefront);
}

/*--------------------------------------------------
	IcefrontLoadGetElementID
  --------------------------------------------------*/

int IcefrontLoadGetElementID( void* vpthis )
{
	/* vpthis for polymorphic function compatibility */
	IcefrontLoad* this = (IcefrontLoad*)vpthis;

	return IcefrontGetElementID(&this->icefront);
}


/*--------------------------------------------------
    IcefrontLoadSetVirtualFunctions
  --------------------------------------------------*/

int IcefrontLoadSetVirtualFunctions( void* vp)
{

    /* VirtualFunctions implemented as a "friend" class: */

    /* void* for polymorphic function compatibility */
    VirtualFunctions* vf = (VirtualFunctions*)vp;
    
    if ( VirtualFunctionsInit(vf) ) {

        /* set IcefrontLoad-specific virtual function pointers: */

        /* general: */

        vf->GetName          = &IcefrontLoadGetName;

		/* configuration: */
        vf->Configure          = &IcefrontLoadConfigure;

		/* loads generation: */
		vf->CreatePVector = &IcefrontLoadCreatePVector;

        return 1;
    }
    else
        return 0;

}

/*--------------------------------------------------
    IcefrontLoadGetName
  --------------------------------------------------*/

char* IcefrontLoadGetName( void* vpthis)
{
    IcefrontLoad* this = (IcefrontLoad*)vpthis;

    #ifdef _DEBUG_
    printf("virtual function: IcefrontLoadGetName\n");
    #endif

    return this->name;
}


/*--------------------------------------------------
	IcefrontLoadAddToDataSet
  --------------------------------------------------*/

int IcefrontLoadAddToDataSet( DataSet* dataset, IcefrontLoad* this)
{
	/*

	Provided as an alternate interface to DataSetAddObject.

	In the c++ world, the dataset could just add an object to its list
	using a dataset member function and a pointer to the object's base
	class.

	In the c world, however, only the object itself is capable of telling
	the dataset its type, hence the structure here; a call to a generic
	DataSetAddObject function via an object "member function."

	*/

	DataSetAddObject( dataset, IcefrontLoadEnum(), (void*)this,
		&DeleteIcefrontLoad, &IcefrontLoadEcho, &IcefrontLoadSizeOfInDoubles,
		&IcefrontLoadShrinkWrap, &IcefrontLoadUnwrap,
		&IcefrontLoadMarshall,
		&IcefrontLoadGetID, &IcefrontLoadSetVirtualFunctions);
	
	return 1;
}


/*--------------------------------------------------
	IcefrontLoadConfigure
  --------------------------------------------------*/

int IcefrontLoadConfigure( void* vpthis, int num_datasets, ...) {

	/* set up data connections to all necessary, related objects: */

	IcefrontLoad* this = (IcefrontLoad*)vpthis;

	int	i,j, found, foundgrid, noerr=1, first=1;

	/* expected DataSet input: */

	DataSet*	pgpdt=NULL;
	DataSet*	pgpdtb=NULL;
	DataSet*    pest=NULL;
	DataSet*    pmpt=NULL;

	/* variable argument list handling: */

	va_list args;
	DataSet* *arglist=NULL;

	#ifdef _DEBUG_
	printf("virtual function: IcefrontLoadConfigure\n");
	#endif


	if (this) {

		if (!num_datasets) {

			printf("   %s error:\n", __func__);
			printf("   null input for number of datasets;\n");
			printf("   could not establish grid connections.\n");
			return 0;
		}
		else {

			/* read, save dataset pointers for data connection operations: */
			arglist = xmalloc( num_datasets*sizeof(DataSet*));
			va_start( args, num_datasets);
			for ( i=0; i<num_datasets; i++) {
				*(arglist+i) = va_arg(args,DataSet*);
			}
			va_end(args);
		}

		#ifdef _DEBUG_
		printf("   variable argument list statistics:\n");
		printf("   input number of DataSets = %d\n", num_datasets);
		printf("   enum types of input DataSets:");
		for ( i=0; i<num_datasets; i++)  printf("   %d", DataSetGetType(*(arglist+i)));
		printf("\n");
		#endif


		/*
			establish grid connections for this IcefrontLoad object:
		*/

		/*Reinitialize this->pg[i]*/
		for (i=0;i<MAX_ICEFRONT_GRIDS;i++){
			this->pg[i]=NULL;
		}

		/* grid connections based on DataSet of type "gpdt": */

		#ifndef _PARALLEL_
		found=0;
		for ( i=0; i<num_datasets; i++) {
			if (*(arglist+i)==NULL)continue;
			if ( GpdtEnum() == DataSetGetType(*(arglist+i)) ) {
				pgpdt = *(arglist+i);
				found = 1;
				break;
			}
		}
		if (found) {

			int	current_grid_id;
			int* pgrid_id_list = this->icefront.g;

			for ( i=0; i<IcefrontGetNumberOfGrids(&this->icefront); i++) {

				if ( (current_grid_id=pgrid_id_list[i]) ) {	/* assign and test presence */

					/* if grid pointer information has been established once already
					(e.g., by a previous module or operation), grid index information
					may already be present and valid.  check to see if corresponding
					pointer information can be reestablished without an expensive
					seek operation: */

					if ( UNDEF != (int)this->internalgrid_indices[i] ) {

						/* does index still point to a valid grid? (i.e., the one with
						the same id): */

						if ( InternalGridEnum() == DataSetGetEnum( pgpdt, this->internalgrid_indices[i])
							 &&
						     current_grid_id == DataSetGetObjectID( pgpdt, this->internalgrid_indices[i]) ) {

							/* great, still valid!  since we have the index position in
							gpdt, just use it to get, save, the grid pointer: */

							this->pg[i] = DataSetGetObjectPtr( pgpdt, this->internalgrid_indices[i]);
						}
					}

					/* if we haven't yet "recovered" the InternalGrid object pointer,
					go seek it out: */

					if ( !this->pg[i] ) {

						noerr *= DataSetSeek( (void**)(this->pg+i),	/* cast because some compilers recognise as */
																	/* ptr to ptr to struct otherwise           */
									(this->internalgrid_indices+i) ,
									pgpdt, current_grid_id, 1, InternalGridEnum());

						if (!noerr) {
							if (first) {
								printf("   %s error:\n", __func__);
								first = 0;
							}
							printf("   could not find grid with id of %d for load %d of type \"%s\"\n",
								current_grid_id, IcefrontLoadGetID(this), IcefrontLoadGetName(this));
						}
					}
				}
			}
		}
		else {

			if (first) {
				printf("   %s error:\n", __func__);
				first=0;
			}
			printf("   could not find DataSet of type \"gpdt\", grid object\n");
			printf("   pointers not established for penpair load id = %d\n",
				IcefrontLoadGetID(this));
			noerr = 0;
		}
		#else //ifdef _PARALLEL_

		/*Configure the load when we are running on a parallel cluster: in this case, 
		 we  have partitioned the grids and elements across the cluster. Every node has 
		 its own lst, its own bgpdt (which owns the internal grids of the partition) and 
		 a common bgpdtb (b for boundaries of the partition). The boundary grids have to be 
		 dealt with. What happens is that bgpdt might not own all the grids referenced by this 
		 load, bgpdtb might own some of these grids. So we have to do a double configuration
		 sort of: 
		 */

		/*First, we need to recover pgpdt and pgpdtb from the arglist: */
		found=0;
		for ( i=0; i<num_datasets; i++) {
			if (*(arglist+i)==NULL)continue;
			if ( GpdtEnum() == DataSetGetType(*(arglist+i)) ) {
				if (found==0){
					/*This is the first GpdtEnum type dataset encountered, 
					 *ie  the internal grids: */
					pgpdt = *(arglist+i);
					found = 1;
					continue;
				}
				if (found==1){
					/*This is the second GpdtEnum type dataset encountered, 
					 *ie  the boundary grids: */
					pgpdtb = *(arglist+i);
					found = 2;
					continue;
				}
			}
		}
		
		if (found==2) {

			int	current_grid_id;
			int* pgrid_id_list = this->icefront.g;

			for ( i=0; i<IcefrontGetNumberOfGrids(&this->icefront); i++) {



				if ( (current_grid_id=pgrid_id_list[i]) ) {	/* assign and test presence */

					/* if grid pointer information has been established once already
					(e.g., by a previous module or operation), grid index information
					may already be present and valid.  check to see if corresponding
					pointer information can be reestablished without an expensive
					seek operation: */

					if ( UNDEF != (int)this->internalgrid_indices[i] ) {

						/* does index still point to a valid grid? (i.e., the one with
						the same id): */

						if ( InternalGridEnum() == DataSetGetEnum( pgpdt, this->internalgrid_indices[i])
							 &&
						     current_grid_id == DataSetGetObjectID( pgpdt, this->internalgrid_indices[i]) ) {

							/* great, still valid!  since we have the index position in
							gpdt, just use it to get, save, the grid pointer: */

							this->pg[i] = DataSetGetObjectPtr( pgpdt, this->internalgrid_indices[i]);
						}
					}
					if ( UNDEF != (int)this->internalgrid_indicesb[i] ) {

						/* does index still point to a valid grid? (i.e., the one with
						the same id): */

						if ( InternalGridEnum() == DataSetGetEnum( pgpdtb, this->internalgrid_indices[i])
							 &&
						     current_grid_id == DataSetGetObjectID( pgpdtb, this->internalgrid_indices[i]) ) {

							/* great, still valid!  since we have the index position in
							gpdtb, just use it to get, save, the grid pointer: */

							this->pg[i] = DataSetGetObjectPtr( pgpdtb, this->internalgrid_indices[i]);
						}
					}

					/* if we haven't yet "recovered" the InternalGrid object pointer,
					go seek it out: */

					if ( !this->pg[i] ) {

						foundgrid = DataSetSeek( (void**)(this->pg+i),	/* cast because some compilers recognise as */
																	/* ptr to ptr to struct otherwise           */
									(this->internalgrid_indices+i) ,
									pgpdt, current_grid_id, 1, InternalGridEnum());
						if(!foundgrid){
							/*We did not find this grid in pgpdt, go find it in pgpdtb:*/
							foundgrid = DataSetSeek( (void**)(this->pg+i),	/* cast because some compilers recognise as */
																	/* ptr to ptr to struct otherwise           */
									(this->internalgrid_indicesb+i) ,
									pgpdtb, current_grid_id, 1, InternalGridEnum());
						}

						if(!foundgrid){
							/*We could not find our grid in pgpdt and pgpdtb, this is not good!:*/
							noerr=0;
						}
						if (!noerr) {
							if (first) {
								printf("   %s error:\n", __func__);
								first = 0;
							}
							printf("   could not find grid with id of %d for element %d of type \"%s\"\n",
								current_grid_id, IcefrontLoadGetElementID(this), IcefrontLoadGetName(this));
							if(current_grid_id==306)IcefrontLoadEcho(this);
						}
					}
				}
			}
		}
		else {

			if (first) {
				printf("   %s error:\n", __func__);
				first=0;
			}
			printf("   could not find DataSets of type \"gpdt\", grid object\n");
			printf("   pointers not established for penpair element id = %d\n",
				IcefrontLoadGetID(this));
			noerr = 0;
		}

		#endif

		
		/*Now look for the element pointer corresponding to this load: */
		found=0;
		for ( j=0; j<num_datasets; j++) {
			if (*(arglist+j)==NULL)continue;
			if ( EstEnum() == DataSetGetType(*(arglist+j)) ) {
				pest  = *(arglist+j);
				found = 1;
				break;
			}
		}
		if(!found){
			printf("   could not find est in list of datasets, to configure penalty load\n");
			noerr=0;goto cleanup_and_return;
		}

		if(UNDEF!=(int)this->element_index) {
			this->element = DataSetGetObjectPtr( pest, this->element_index);
			this->element_enum = DataSetGetEnum( pest, this->element_index);
		}
						
		if ( !this->element) {

			noerr *= DataSetSeek( (void**)(&this->element),	//cast because some compilers recognise as ptr to ptr to struct otherwise 
					&this->element_index , pest, IcefrontLoadGetElementID(this), 0);
			this->element_enum = DataSetGetEnum( pest, this->element_index);
		}
		if (!noerr) {
			printf("   Icefront load %i could not find element with id %d \n", IcefrontLoadGetID(this),IcefrontLoadGetElementID(this));
			goto cleanup_and_return;
		}

		/* matpar property connections based on DataSet of type "mpt": */
		found=0;
		for ( i=0; i<num_datasets; i++) {
			if (*(arglist+i)==NULL)continue;
			if ( MptEnum() == DataSetGetType(*(arglist+i)) ) {
				pmpt  = *(arglist+i);
				found = 1;
				break;
			}
		}
		if (found) {

			int	current_matpar_id = IcefrontGetMaterialID(&this->icefront);
			
			if ( this->matpar_index!= UNDEF){
			

				if ( this->matpar_enum == DataSetGetEnum( pmpt, this->matpar_index)
					 &&
					 current_matpar_id == DataSetGetObjectID( pmpt, this->matpar_index) ) {

					/* index is still valid; just get matpar entry pointer directly: */

					this->matpar = DataSetGetObjectPtr( pmpt, this->matpar_index);
				}
			}

			/* if we haven't yet "recovered" the matpar object pointer, go seek it out: */

			if ( NULL == this->matpar) {

				noerr *= DataSetSeek( &this->matpar, &this->matpar_index,
							pmpt, current_matpar_id, 1, MatparEnum() );	/* 0 to generalise later to Pcomp, etc. ... */

				if (!noerr) {
					if (first) {
						_printf_("   %s error:\n", __func__);
						first = 0;
					}
					_printf_("   could not find matpar entry with id of %d for element %d of type \"%s\"\n",
						current_matpar_id, IcefrontGetID(&this->icefront), IcefrontGetName(&this->icefront));
				}
				else {

					/* though the answer for now is just MatparEnum(), for generality go ahead and
					ask the question, "what type of matpar has been referenced?" */

					this->matpar_enum = DataSetGetEnum( *(arglist+i), this->matpar_index);
				}
			}
			
		}
		else {

			if (first) {
				_printf_("   %s error:\n", __func__);
				first=0;
			}
			_printf_("   could not find DataSet of type \"mpt\"; matice  and matpar relationships\n");
			_printf_("   not established for penta element id = %d\n", IcefrontGetElementID(this));
			noerr = 0;
		}

		#ifdef _DEBUG_
		printf("   load data after all lookups:\n");
		IcefrontLoadEcho( this);
		#endif

		xfree((void**)&arglist);

	}
	else {

		/* quiet return */
		;
	}
	cleanup_and_return:

	return noerr;

}

/*--------------------------------------------------
	IcefrontLoadCreatePVector
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "IcefrontLoadCreatePVector"

int IcefrontLoadCreatePVector( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs, int analysis_type){
	int noerr=1;

	IcefrontLoad* load=(IcefrontLoad*)vpthis;
	//printf("Eid for icefront load: %i rank %i\n",load->icefront.eid,my_rank);
	
	/*Just branch to the correct element icefront vector generator, according to the type of analysis we are carrying out: */
	if ((analysis_type==DiagnosticHorizAnalysisEnum()) || (analysis_type==ControlAnalysisEnum())){
		noerr=IcefrontLoadCreatePVectorDiagnosticHoriz( ppe_g, vpthis, inputs);
	}
	else{
		printf("%s%s%i%s\n",__FUNCT__," error message: analysis ",analysis_type," not supported yet");
		noerr=0;
	}
	return noerr;

}

/*--------------------------------------------------
	IcefrontLoadCreatePVectorDiagnosticHoriz
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "IcefrontLoadCreatePVectorDiagnosticHoriz"

int IcefrontLoadCreatePVectorDiagnosticHoriz( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs){

	
	int noerr=1;
	IcefrontLoad* this=NULL;
	
	/*First recover pointer to icefront: */
	this = (IcefrontLoad*)vpthis;

	/*Branck on the type of icefront: */
	if strcasecmp_eq(this->icefront.type,"segment"){
		noerr=IcefrontLoadCreatePVectorDiagnosticHorizSegment(ppe_g,vpthis, inputs);
	}
	else{
		noerr=IcefrontLoadCreatePVectorDiagnosticHorizQuad(ppe_g,vpthis, inputs);
	}
	return noerr;
}

/*--------------------------------------------------
	IcefrontLoadCreatePVectorDiagnosticHorizSegment
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "IcefrontLoadCreatePVectorDiagnosticHorizSegment"

int IcefrontLoadCreatePVectorDiagnosticHorizSegment( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs){

	int noerr=1;
	int i,j;
	
	int numgrids=2;
	int numdof=numgrids*NDOF2;

	int*            structural_dof_list   =  NULL;
	int             dof;

	/* output: */
	ElemVector*   pe_g        = NULL;
		
	IcefrontLoad* this=NULL;
	Matpar*       matpar=NULL;
	TriaElement*  triaelement=NULL;
	PentaElement* pentaelement=NULL;

	/* material parameters: */
	double        rho_ice;
	double        rho_water;
	double        gravity;

	double x1,y1,x2,y2;

	/* input parameters: */
	double* thickness_param=NULL;
	double thickness_list_element[3];
	double thickness_list[2];
	double* bed_param=NULL;
	double bed_list_element[3];
	double bed_list[2];
	int    grid1,grid2;

	/* grid data: */
	int    cid_list[numgrids];
	double xyz_list[numgrids][3];
	double T_bg_list[numgrids][9];

	double normal[2];
	double length;

	int    fill;

	/*First recover pointer to icefront: */
	this = (IcefrontLoad*)vpthis;

	/*Recover material and fill parameters, according to which element the icefront belongs to: */
	if (this->icefront.element_type==TriaEnum()){
		triaelement=this->element;
		matpar=triaelement->matpar;
		fill=triaelement->tria.shelf;
	}
	else{
		pentaelement=this->element;
		matpar=pentaelement->matpar;
		fill=pentaelement->penta.shelf;
	}
		
	//check that the element is onbed (collapsed formulation) otherwise:pe=0
	if(this->icefront.element_type==PentaEnum()){
		if  (!pentaelement->penta.onbed){
			*ppe_g=NULL;
			return noerr;
		}
	}

	rho_ice=matpar->rho_ice;
	rho_water=matpar->rho_water;
	gravity=matpar->g;
	
	/* Allocate new load element  vector: */
	pe_g=NewElemVector(numdof);

	/*recover extra inputs: */
	thickness_param=ParameterInputsRecover(inputs,"thickness");
	bed_param=ParameterInputsRecover(inputs,"bed");

	/* Get all element grid data */
	GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], cid_list, this->pg, numgrids, 1 );

	/*Identify which grids are comprised in the segment. Watch out, different logic if we are in a penta, or a tria: */
	if (this->icefront.element_type==TriaEnum()){
		/*go through all grids and identify 1st and 2nd grid: */
		for(i=0;i<3;i++){
			if (triaelement->pg[i]==this->pg[0])grid1=i;
			if (triaelement->pg[i]==this->pg[1])grid2=i;
		}
	}
	else{
		/*go through first 3 grids, which are at the base of the penta, and identify 1st and 2nd grid: */
		for(i=0;i<3;i++){
			if (pentaelement->pg[i]==this->pg[0])grid1=i;
			if (pentaelement->pg[i]==this->pg[1])grid2=i;
		}
	}

	/* Build the row index vector (same as column index vector). 
	 * At the same time, get element exterior data for each grid: */
	for (i=0;i<numgrids;i++){
		structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
		for (j=0;j<NDOF2;j++){
			dof=structural_dof_list[j];
			*(pe_g->row_indices+i*NDOF2+j)=dof;
		}
	}
	/*Now build thickness_list_element and bed_list_element: */
	for (i=0;i<3;i++){

		if (this->icefront.element_type==TriaEnum()) structural_dof_list= InternalGridGetStructuralDoflistPtr( triaelement->pg[i]);
		else structural_dof_list= InternalGridGetStructuralDoflistPtr( pentaelement->pg[i]);

		dof=structural_dof_list[0];
		if(thickness_param){
			thickness_list_element[i]=*(thickness_param+dof);
		}
		else{
			if (this->icefront.element_type==TriaEnum()) 	thickness_list_element[i]=triaelement->tria.h[i];
			else thickness_list_element[i]=pentaelement->penta.h[i];
		}
		if(bed_param){
			bed_list_element[i]=*(bed_param+dof);
		}
		else{
			if (this->icefront.element_type==TriaEnum()) 	bed_list_element[i]= triaelement->tria.b[i];
			else bed_list_element[i]= pentaelement->penta.b[i];
		}
	}
	/*Build thickness_list and bed_list: */
	thickness_list[0]=thickness_list_element[grid1];
	thickness_list[1]=thickness_list_element[grid2];
	bed_list[0]=bed_list_element[grid1];
	bed_list[1]=bed_list_element[grid2];

	//Recover grid coordinates
	x1=xyz_list[0][0];
	y1=xyz_list[0][1];
	x2=xyz_list[1][0];
	y2=xyz_list[1][1];

	//Compute length and normal of segment
	normal[0]=cos(atan2(x1-x2,y2-y1));
	normal[1]=sin(atan2(x1-x2,y2-y1));
	length=sqrt(pow(x2-x1,2)+pow(y2-y1,2));

	//Compute load contribution for this segment:
	noerr=SegmentPressureLoad(pe_g,rho_water,rho_ice,gravity,thickness_list,bed_list,normal,length,fill);
		
	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && this->icefront.eid==ELID){ 
		printf("\nicefront load\n");
		printf("grids %i %i\n",grid1,grid2);
		printf("rho_water %g\n",rho_water);
		printf("rho_ice %g\n",rho_ice);
		printf("gravity %g\n",gravity);
		printf("thickness_list (%g,%g)\n",thickness_list[0],thickness_list[1]);
		printf("bed_list (%g,%g)\n",bed_list[0],bed_list[1]);
		printf("normal (%g,%g)\n",normal[0],normal[1]);
		printf("length %g\n",length);
		printf("fill %i\n",fill);
		printf("pe_g->terms\n");
		for(i=0;i<numgrids*NDOF2;i++){
			printf("%g ",*(pe_g->terms+i));
		}
	}
	#endif

	cleanup_and_return: 

    if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not build icefront load vector.");
		return 0;
	}
	else{
		/*Assign output pointer: */
		*ppe_g=pe_g;
	}
	return noerr;
}

/*--------------------------------------------------
	SegmentPressureLoad
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "SegmentPressureLoad"

int SegmentPressureLoad(ElemVector* pe_g,double rho_water,double rho_ice,double gravity, double* thickness_list, double* bed_list, double* normal,double length,int fill){
	int      noerr=1;

	double   nx,ny;
	double   h1,h2,b1,b2;
	int      num_gauss;
	double*  segment_gauss_coord=NULL;
	double*  gauss_weights=NULL;
	int      ig;
	double   Jdet;
	double   thickness,bed;
	double   ice_pressure,water_pressure,air_pressure;
	double   surface_under_water,base_under_water;
	double   pressure;

	double   pe_g_gaussian[4];

	nx=normal[0];
	ny=normal[1];

	//Get gaussian points and weights. order 2 since we have a product of 2 nodal functions
	num_gauss=3;
	GaussSegment(&segment_gauss_coord, &gauss_weights, num_gauss);

	//recover thickness and bed at two grids
	h1=thickness_list[0];
	h2=thickness_list[1];
	b1=bed_list[0];
	b2=bed_list[1];

	//compute Jacobian of segment
	Jdet=1./2.*length;

	for(ig=0;ig<num_gauss;ig++){

		thickness=h1*(1+segment_gauss_coord[ig])/2+h2*(1-segment_gauss_coord[ig])/2;
		bed=b1*(1+segment_gauss_coord[ig])/2+b2*(1-segment_gauss_coord[ig])/2;

		if (fill==1){
			//icefront ends in water: 
			ice_pressure=1.0/2.0*gravity*rho_ice*pow(thickness,2);
			air_pressure=0;

			//Now deal with water pressure
			surface_under_water=min(0,thickness+bed); // 0 if the top of the glacier is above water level
			base_under_water=min(0,bed);              // 0 if the bottom of the glacier is above water level
			water_pressure=1.0/2.0*gravity*rho_water*(pow(surface_under_water,2) - pow(base_under_water,2));
		}
		else if (fill==0){
			ice_pressure=1.0/2.0*gravity*rho_ice*pow(thickness,2);
			air_pressure=0;
			water_pressure=0;
		}
		else{
			printf("%s%s\n",__FUNCT__,"error message: fill type not supported yet");
			noerr=0; return noerr;
		}

		pressure = ice_pressure + water_pressure + air_pressure;

		*(pe_g->terms+2*0+0)+= pressure*Jdet*gauss_weights[ig]*(1+segment_gauss_coord[ig])/2*nx;
		*(pe_g->terms+2*0+1)+= pressure*Jdet*gauss_weights[ig]*(1+segment_gauss_coord[ig])/2*ny;
		*(pe_g->terms+2*1+0)+= pressure*Jdet*gauss_weights[ig]*(1-segment_gauss_coord[ig])/2*nx;
		*(pe_g->terms+2*1+1)+= pressure*Jdet*gauss_weights[ig]*(1-segment_gauss_coord[ig])/2*ny;

	} //for(ig=0;ig<num_gauss;ig++)

	xfree((void**)&segment_gauss_coord);
	xfree((void**)&gauss_weights);

	return noerr;
}


/*--------------------------------------------------
	IcefrontLoadCreatePVectorDiagnosticHorizQuad
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "IcefrontLoadCreatePVectorDiagnosticHorizQuad"

int IcefrontLoadCreatePVectorDiagnosticHorizQuad( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs){

	int noerr=1;
	int i,j;
	
	int numgridselement=6;
	int numgridsload=4;

	int*            structural_dof_list   =  NULL;
	int             dof;

	/* output: */
	ElemVector*   pe_g        = NULL;
		
	IcefrontLoad* this=NULL;
	Matpar*       matpar=NULL;
	PentaElement* pentaelement=NULL;

	/* material parameters: */
	double        rho_ice;
	double        rho_water;
	double        gravity;

	/* input parameters: */
	double* thickness_param=NULL;
	double thickness_list[numgridselement];
	double thickness_list_quad[numgridsload];
	double* bed_param=NULL;
	double bed_list[numgridselement];
	double bed_list_quad[numgridsload];

	/* grid data: */
	int    cid_list[numgridsload];
	double xyz_list_quad[5][3];
	double xyz_list[6][3];
	double T_bg_list[numgridsload][9];

	int    fill;

	/*quad grids: */
	int grid1,grid2,grid3,grid4;

	/*normals: */
	double normal1[3];
	double normal2[3];
	double normal3[3];
	double normal4[3];
	double normal_norm;
	double v15[3];
	double v25[3];
	double v35[3];
	double v45[3];
	
	/*First recover pointer to icefront: */
	this = (IcefrontLoad*)vpthis;

	/*Check quad icefront is associated to a pentaelem: */
	if (this->icefront.element_type==TriaEnum()){
		_printf_("%s%s\n",__FUNCT__," error message: quad icefront is associated to a TriaElem!");
		return 0;
	}

	/*Recover material and fill parameters: */
	pentaelement=this->element;
	matpar=pentaelement->matpar;
	fill=pentaelement->penta.shelf;

	/*Recover material parameters: */
	rho_ice=matpar->rho_ice;
	rho_water=matpar->rho_water;
	gravity=matpar->g;
	
	/* Allocate new load element  vector: */
	pe_g=NewElemVector(numgridsload*NDOF2);

	/*recover extra inputs: */
	thickness_param=ParameterInputsRecover(inputs,"thickness");
	bed_param=ParameterInputsRecover(inputs,"bed");

	/* Get all element grid data */
	GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], cid_list, pentaelement->pg, numgridselement, 0 );

	/* Build the row index vector (same as column index vector). 
	 * At the same time, get element exterior data for each grid: */
	for (i=0;i<numgridsload;i++){
		structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
		for (j=0;j<NDOF2;j++){
			dof=structural_dof_list[j];
			*(pe_g->row_indices+i*NDOF2+j)=dof;
		}
	}
	
	for (i=0;i<numgridselement;i++){
		structural_dof_list= InternalGridGetStructuralDoflistPtr( pentaelement->pg[i]);
		dof=structural_dof_list[0];
		if(thickness_param){
			thickness_list[i]=*(thickness_param+dof);
		}
		else{
			thickness_list[i]=pentaelement->penta.h[i];
		}
		if(bed_param){
			bed_list[i]=*(bed_param+dof);
		}
		else{
			bed_list[i]= pentaelement->penta.b[i];
		}
	}

	//Identify which grids are comprised in the quad: 
	grid1=-1; grid2=-1; grid3=-1; grid4=-1;
	for(i=0;i<numgridselement;i++){
		if (pentaelement->pg[i]==this->pg[0])grid1=i;
		if (pentaelement->pg[i]==this->pg[1])grid2=i;
		if (pentaelement->pg[i]==this->pg[2])grid3=i;
		if (pentaelement->pg[i]==this->pg[3])grid4=i;
	}
	if((grid1==-1) || (grid2==-1)|| (grid3==-1)||(grid4==-1)){
		printf("%s%s\n",__FUNCT__," error message: could not find element grids corresponding to quad icefront!");
		return 0;
	}

	/*Build new xyz, bed and thickness lists for grids 1 to 4: */
	xyz_list_quad[0][0]=xyz_list[grid1][0];
	xyz_list_quad[0][1]=xyz_list[grid1][1];
	xyz_list_quad[0][2]=xyz_list[grid1][2];
	xyz_list_quad[1][0]=xyz_list[grid2][0];
	xyz_list_quad[1][1]=xyz_list[grid2][1];
	xyz_list_quad[1][2]=xyz_list[grid2][2];
	xyz_list_quad[2][0]=xyz_list[grid3][0];
	xyz_list_quad[2][1]=xyz_list[grid3][1];
	xyz_list_quad[2][2]=xyz_list[grid3][2];
	xyz_list_quad[3][0]=xyz_list[grid4][0];
	xyz_list_quad[3][1]=xyz_list[grid4][1];
	xyz_list_quad[3][2]=xyz_list[grid4][2];

	thickness_list_quad[0]=thickness_list[grid1];
	thickness_list_quad[1]=thickness_list[grid2];
	thickness_list_quad[2]=thickness_list[grid3];
	thickness_list_quad[3]=thickness_list[grid4];

	bed_list_quad[0]=bed_list[grid1];
	bed_list_quad[1]=bed_list[grid2];
	bed_list_quad[2]=bed_list[grid3];
	bed_list_quad[3]=bed_list[grid4];

	//Create a new grid in the midle of the quad and add it at the end of the list
	xyz_list_quad[4][0] = (xyz_list_quad[0][0]+xyz_list_quad[1][0]+xyz_list_quad[2][0]+xyz_list_quad[3][0])/4.0;
	xyz_list_quad[4][1] = (xyz_list_quad[0][1]+xyz_list_quad[1][1]+xyz_list_quad[2][1]+xyz_list_quad[3][1])/4.0;
	xyz_list_quad[4][2] = (xyz_list_quad[0][2]+xyz_list_quad[1][2]+xyz_list_quad[2][2]+xyz_list_quad[3][2])/4.0;

	//Compute four normals (the quad is divided into four triangles)
	v15[0]=xyz_list_quad[0][0]-xyz_list_quad[4][0];
	v15[1]=xyz_list_quad[0][1]-xyz_list_quad[4][1];
	v15[2]=xyz_list_quad[0][2]-xyz_list_quad[4][2];
	
	v25[0]=xyz_list_quad[1][0]-xyz_list_quad[4][0];
	v25[1]=xyz_list_quad[1][1]-xyz_list_quad[4][1];
	v25[2]=xyz_list_quad[1][2]-xyz_list_quad[4][2];
	
	v35[0]=xyz_list_quad[2][0]-xyz_list_quad[4][0];
	v35[1]=xyz_list_quad[2][1]-xyz_list_quad[4][1];
	v35[2]=xyz_list_quad[2][2]-xyz_list_quad[4][2];
	
	v45[0]=xyz_list_quad[3][0]-xyz_list_quad[4][0];
	v45[1]=xyz_list_quad[3][1]-xyz_list_quad[4][1];
	v45[2]=xyz_list_quad[3][2]-xyz_list_quad[4][2];

	cross(normal1,v15,v25); 
	cross(normal2,v25,v35);
	cross(normal3,v35,v45);
	cross(normal4,v45,v15);

	normal_norm=norm(normal1);normal1[0]=normal1[0]/normal_norm;normal1[1]=normal1[1]/normal_norm;normal1[2]=normal1[2]/normal_norm;
	normal_norm=norm(normal2);normal2[0]=normal2[0]/normal_norm;normal2[1]=normal2[1]/normal_norm;normal2[2]=normal2[2]/normal_norm;
	normal_norm=norm(normal3);normal3[0]=normal3[0]/normal_norm;normal3[1]=normal3[1]/normal_norm;normal3[2]=normal3[2]/normal_norm;
	normal_norm=norm(normal4);normal4[0]=normal4[0]/normal_norm;normal4[1]=normal4[1]/normal_norm;normal4[2]=normal4[2]/normal_norm;


	//Compute load contribution for this quad:
	noerr=QuadPressureLoad(pe_g,rho_water,rho_ice,gravity,thickness_list_quad,bed_list_quad,normal1,normal2,normal3,normal4,&xyz_list_quad[0][0],fill,this->icefront.eid);


	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && this->icefront.eid==ELID){ 
		printf("\nicefront load\n");
		printf("grids %i %i %i %i\n",grid1,grid2,grid3,grid4);
		printf("rho_water %g\n",rho_water);
		printf("rho_ice %g\n",rho_ice);
		printf("gravity %g\n",gravity);
		printf("thickness_list (%g,%g,%g,%g)\n",thickness_list[0],thickness_list[1],thickness_list[2],thickness_list[3]);
		printf("bed_list (%g,%g,%g,%g)\n",bed_list[0],bed_list[1],bed_list[2],bed_list[3]);
		printf("normal1 (%g,%g,%g)\n",normal1[0],normal1[1],normal1[2]);
		printf("normal2 (%g,%g,%g)\n",normal2[0],normal2[1],normal2[2]);
		printf("normal3 (%g,%g,%g)\n",normal3[0],normal3[1],normal3[2]);
		printf("normal4 (%g,%g,%g)\n",normal4[0],normal4[1],normal4[2]);
		printf("fill %i\n",fill);
		printf("pe_g->terms\n");
		for(i=0;i<numgridsload*NDOF2;i++){
			printf("%g ",*(pe_g->terms+i));
		}
	}
	#endif

	cleanup_and_return: 

    if(!noerr){
		printf("%s%s\n",__FUNCT__," error message: could not build icefront load vector.");
		return 0;
	}
	else{
		#ifdef _DEBUG_
		ElemVectorEcho(pe_g);
		#endif
		/*Assign output pointer: */
		*ppe_g=pe_g;
	}
	return noerr;
}



/*--------------------------------------------------
	QuadPressureLoad
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "QuadPressureLoad"

int QuadPressureLoad(ElemVector* pe_g,double rho_water,double rho_ice,double gravity, double* thickness_list, double* bed_list, 
		                              double* normal1,double* normal2,double* normal3,double* normal4,double* xyz_list, int fill,int eid){
	
	
	//The quad is divided into four triangles tria1 tria2 tria3 and tria4 as follows
	//
	//   grid 4 +-----------------+ grid 3
	//          |\2            1 /|
	//          |1\    tria3    /2|
	//          |  \           /  |
	//          |   \         /   |
	//          |    \       /    |
	//          |     \     /     |
	//          |      \ 3 /      |
	//          |tria4  \ / 3     |
	//          |      3 \grid5   |
	//          |       / \       | 
	//          |      / 3 \ tria2|
	//          |     /     \     |
	//          |    /       \    |
	//          |   /         \   |
	//          |  /   tria1   \  |
	//          |2/1           2\1|
	//   grid1  +-----------------+ grid 2
	//
	//

	int      noerr=1;
	int      i,j;

	double nx[4];
	double ny[4];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_coord[3];

	double  surface_list[5];
	double  x[5];
	double  y[5];
	double  z[5];
	double l12,l14,l15,l23,l25,l34,l35,l45;
	double cos_theta_triangle1,cos_theta_triangle2,cos_theta_triangle3,cos_theta_triangle4;

	double xyz_tria[12][2];
	double l1l2l3_tria[3];
	double r_tria,s_tria;
	double r_quad[4];
	double s_quad[4];
	double l1l4_tria[4][4];

	double J[4];
	double z_g[4];
	double ice_pressure_tria[4];
	double air_pressure_tria;
	double water_level_above_g_tria;
	double water_pressure_tria;
	double pressure_tria[4];



	//Build the four normal vectors 
	nx[0]=normal1[0]; nx[1]=normal2[0]; nx[2]=normal3[0]; nx[3]=normal4[0];
	ny[0]=normal1[1]; ny[1]=normal2[1]; ny[2]=normal3[1]; ny[3]=normal4[1];

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);
	
	//Recover the surface of the four nodes
	for(i=0;i<4;i++){
		surface_list[i]=thickness_list[i]+bed_list[i];
	}
	//Add surface sor the fifth point (average of the surfaces)
	surface_list[4]=(surface_list[0]+surface_list[1]+surface_list[2]+surface_list[3])/4.0;

	//Recover grid coordinates
	for(i=0;i<5;i++){
		x[i]=*(xyz_list+3*i+0);
		y[i]=*(xyz_list+3*i+1);
		z[i]=*(xyz_list+3*i+2);
	}
	
	//Build triangles in a 2D plan before using reference elements
	l12=pow((x[1]-x[0]),2)+pow((y[1]-y[0]),2)+pow((z[1]-z[0]),2);
	l14=pow((x[3]-x[0]),2)+pow((y[3]-y[0]),2)+pow((z[3]-z[0]),2);
	l15=pow((x[4]-x[0]),2)+pow((y[4]-y[0]),2)+pow((z[4]-z[0]),2);
	l23=pow((x[2]-x[1]),2)+pow((y[2]-y[1]),2)+pow((z[2]-z[1]),2);
	l25=pow((x[4]-x[1]),2)+pow((y[4]-y[1]),2)+pow((z[4]-z[1]),2);
	l34=pow((x[3]-x[2]),2)+pow((y[3]-y[2]),2)+pow((z[3]-z[2]),2);
	l35=pow((x[4]-x[2]),2)+pow((y[4]-y[2]),2)+pow((z[4]-z[2]),2);
	l45=pow((x[4]-x[3]),2)+pow((y[4]-y[3]),2)+pow((z[4]-z[3]),2);

	cos_theta_triangle1=(l15+l12-l25)/(2*sqrt(l12*l15));
	cos_theta_triangle2=(l25+l23-l35)/(2*sqrt(l23*l25));
	cos_theta_triangle3=(l35+l34-l45)/(2*sqrt(l34*l35));
	cos_theta_triangle4=(l45+l14-l15)/(2*sqrt(l14*l45));

	//First triangle : nodes 1, 2 and 5
	xyz_tria[0][0]=0;
	xyz_tria[0][1]=0;
	xyz_tria[1][0]=sqrt(l12);
	xyz_tria[1][1]=0;
	xyz_tria[2][0]=cos_theta_triangle1*sqrt(l15);
	xyz_tria[2][1]=sqrt(1-pow(cos_theta_triangle1,2))*sqrt(l15);

	//Second triangle : nodes 2, 3 and 5
	xyz_tria[3][0]=0;
	xyz_tria[3][1]=0;
	xyz_tria[4][0]=sqrt(l23);
	xyz_tria[4][1]=0;
	xyz_tria[5][0]=cos_theta_triangle2*sqrt(l25);
	xyz_tria[5][1]=sqrt(1-pow(cos_theta_triangle2,2))*sqrt(l25);
	
	//Third triangle : nodes 3, 4 and 5
	xyz_tria[6][0]=0;
	xyz_tria[6][1]=0;
	xyz_tria[7][0]=sqrt(l34);
	xyz_tria[7][1]=0;
	xyz_tria[8][0]=cos_theta_triangle3*sqrt(l35);
	xyz_tria[8][1]=sqrt(1-pow(cos_theta_triangle3,2))*sqrt(l35);

	//Fourth triangle : nodes 4, 1 and 5
	xyz_tria[9][0]=0;
	xyz_tria[9][1]=0;
	xyz_tria[10][0]=sqrt(l14);
	xyz_tria[10][1]=0;
	xyz_tria[11][0]=cos_theta_triangle4*sqrt(l45);
	xyz_tria[11][1]=sqrt(1-pow(cos_theta_triangle4,2))*sqrt(l45);

	

	//Start  looping on the triangle's gaussian points: 
	for(ig=0;ig<num_gauss;ig++){

		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_coord[0]=*(first_gauss_area_coord+ig);
		gauss_coord[1]=*(second_gauss_area_coord+ig);
		gauss_coord[2]=*(third_gauss_area_coord+ig);
        
		TriaElementGetNodalFunctions(l1l2l3_tria,gauss_coord);

		//Get the coordinates of gauss point for each triangle in penta/quad
		r_tria=gauss_coord[1]-gauss_coord[0];
		s_tria=-3/sqrt(3)*(gauss_coord[0]+gauss_coord[1]-2/3);

		//Coordinates of gauss points in the reference triangle
		r_quad[0]=r_tria;
		s_quad[0]=1/sqrt(3)*s_tria-2/3;
		r_quad[1]=-1/sqrt(3)*s_tria+2/3;
		s_quad[1]=r_tria;
		r_quad[2]=-r_tria;
		s_quad[2]=-1/sqrt(3)*s_tria+2/3;
		r_quad[3]=1/sqrt(3)*s_tria-2/3;
		s_quad[3]=-r_tria;

		//Get the nodal function of the quad for the gauss points of each triangle
		for(i=0;i<4;i++){
			l1l4_tria[i][0]=1.0/4.0*(
					(r_quad[i]-1.0)*(s_quad[i]-1.0) 
					); 
			
			l1l4_tria[i][1]=1.0/4.0*(
					 -(r_quad[i]+1.0)*(s_quad[i]-1.0)
					); 

			l1l4_tria[i][2]=1.0/4.0*(
					(r_quad[i]+1.0)*(s_quad[i]+1.0) 
					);
			
			l1l4_tria[i][3]=1.0/4.0*(
					 -(r_quad[i]-1.0)*(s_quad[i]+1.0)
					);

		} //for(i=0;i<4;i++)
		
		
		//Compute jacobian of each triangle
		for (i=0;i<4;i++){
			double complete_list[3][3]; //a third coordinate is needed for the jacobian determinant calculation, here it is zero
			for(j=0;j<3;j++){
				complete_list[j][0]=xyz_tria[3*i+j][0];
				complete_list[j][1]=xyz_tria[3*i+j][1];
				complete_list[j][2]=0;
			}
			TriaElementGetJacobianDeterminant(&J[i],&complete_list[0][0],l1l2l3_tria);
		}

		//Calculation of the z coordinate for the gaussian point ig for each triangle
		z_g[0]=z[0]*l1l2l3_tria[0]+z[1]*l1l2l3_tria[1]+z[4]*l1l2l3_tria[2];
		z_g[1]=z[1]*l1l2l3_tria[0]+z[2]*l1l2l3_tria[1]+z[4]*l1l2l3_tria[2];
		z_g[2]=z[2]*l1l2l3_tria[0]+z[3]*l1l2l3_tria[1]+z[4]*l1l2l3_tria[2];
		z_g[3]=z[3]*l1l2l3_tria[0]+z[0]*l1l2l3_tria[1]+z[4]*l1l2l3_tria[2];
		
		//Loop on the triangles
		for(i=0;i<4;i++){

			//Loop on the grids of the quad
			//Calculate the ice pressure
			for(j=0;j<4;j++){
				ice_pressure_tria[j]=gravity*rho_ice*(surface_list[j]-z_g[i]);
			}
			air_pressure_tria=0;

			//Now deal with water pressure: 
			if(fill==1){ //icefront ends in water
				water_level_above_g_tria=min(0,z_g[i]);//0 if the gaussian point is above water level
				water_pressure_tria=rho_water*gravity*water_level_above_g_tria;
			}
			else if(fill==0){
				water_pressure_tria=0;
			}
			else{
				_printf_("%s%s\n",__FUNCT__," error message: QuadPressureLoad error message: unknow fill type for icefront boundary condition");
				return 0;
			}

			//Add pressure from triangle i
			//Loop on the grids of the quad
			for(j=0;j<4;j++){
				pressure_tria[j] = ice_pressure_tria[j] + water_pressure_tria + air_pressure_tria;
			}


			pe_g->terms[0]+= J[i]*gauss_weight* pressure_tria[0]*l1l4_tria[i][0]*nx[i];
			pe_g->terms[1]+= J[i]*gauss_weight* pressure_tria[0]*l1l4_tria[i][0]*ny[i];
			pe_g->terms[2]+= J[i]*gauss_weight* pressure_tria[1]*l1l4_tria[i][1]*nx[i];
			pe_g->terms[3]+= J[i]*gauss_weight* pressure_tria[1]*l1l4_tria[i][1]*ny[i];
			pe_g->terms[4]+= J[i]*gauss_weight* pressure_tria[2]*l1l4_tria[i][2]*nx[i];
			pe_g->terms[5]+= J[i]*gauss_weight* pressure_tria[2]*l1l4_tria[i][2]*ny[i];
			pe_g->terms[6]+= J[i]*gauss_weight* pressure_tria[3]*l1l4_tria[i][3]*nx[i];
			pe_g->terms[7]+= J[i]*gauss_weight* pressure_tria[3]*l1l4_tria[i][3]*ny[i];

			

		} //for(i=0;i<4;i++)
	} //for(ig=0;ig<num_gauss;ig++)

	return noerr;
}

#undef NDOF1 
#undef NDOF2 
