%GIA class definition for Caron model (Caron et al, Geophysical Journal International, 2017)
%
%   Usage:
%      giacaron=giacaron();
classdef giacaron
	properties (SetAccess=public) 
		numlayers   = NaN;
		radius      = NaN;
		lame_mu     = NaN;
		lame_lambda = NaN;
		issolid     = NaN;
		density     = NaN; 
		viscosity   = NaN; 
		isburger    = NaN; 
		transient_viscosity    = NaN; 
		transient_mu           = NaN; 
	end
	methods
		function self = extrude(self,md) % {{{
		end % }}}
		function self = giacaron(varargin) % {{{
			switch nargin
				case 0
				otherwise
					options=pairoptions(varargin{:});
					body=getfieldvalue(options,'body');
					if strcmpi(body,'earth'), 
						self.numlayers=getfieldvalue(options,'numlayers',5);
						%[self.radius,self.lame_mu, self.lame_lambda, self.issolid, self.density, ...
						%self.viscosity, self.isburger, self.transient_viscosity, self.transient_mu]=...
						%modelinit(self.numlayers);
					elseif strcmpi(body,'europa'), 
						error('giacaron constructor error message: ''europa'' body not implemented yet!');
					else 
						error('giacaron constructor error message: body not implemented yet!');
					end
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{
		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			if ~ismember('GiaAnalysis',analyses), return; end
			md = checkfield(md,'fieldname','gia.numlayers','NaN',1,'Inf',1,'>',0,'numel',1);
			md = checkfield(md,'fieldname','gia.radius','NaN',1,'Inf',1,'size',[md.gia.numlayers 1],'>',0);
			md = checkfield(md,'fieldname','gia.lame_mu','NaN',1,'Inf',1,'size',[md.gia.numlayers 1],'>',0);
			md = checkfield(md,'fieldname','gia.lame_lambda','NaN',1,'Inf',1,'size',[md.gia.numlayers 1],'>',0);
			md = checkfield(md,'fieldname','gia.issolid','NaN',1,'Inf',1,'size',[md.gia.numlayers 1],'>',0,'<',2);
			md = checkfield(md,'fieldname','gia.density','NaN',1,'Inf',1,'size',[md.gia.numlayers 1],'>',0);
			md = checkfield(md,'fieldname','gia.viscosity','NaN',1,'Inf',1,'size',[md.gia.numlayers 1],'>',0);
			md = checkfield(md,'fieldname','gia.isburger','NaN',1,'Inf',1,'size',[md.gia.numlayers 1],'>',0,'<',2);
			md = checkfield(md,'fieldname','gia.transient_viscosity','NaN',1,'Inf',1,'size',[md.gia.numlayers 1],'>',0);
			md = checkfield(md,'fieldname','gia.transient_mu','NaN',1,'Inf',1,'size',[md.gia.numlayers 1],'>',0);
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   giacaron parameters:'));
			fielddisplay(self,'numlayers','number of layers (default 5)');
			fielddisplay(self,'radius','array describing the radius for each interface (numlayers+1) [m]');
			fielddisplay(self,'lame_mu','array describing the shear modulus for each layers (numlayers) [Pa]');
			fielddisplay(self,'lame_lambda','array describing the lame lambda parameter (numlayers) [Pa]');
			fielddisplay(self,'issolid','array describing whether the layer is solid or liquid (default 1) (numlayers)');
			fielddisplay(self,'density','array describing each layer''s density (numlayers) [kg/m^3]');
			fielddisplay(self,'viscosity','array describing each layer''s viscosity (numlayers) [Pa.s]');
			fielddisplay(self,'isburger','array describing whether we adopt a MaxWell (0) or Burgers (1) rheology (default 0)');
			fielddisplay(self,'transient_viscosity','array describing each layer''s transient viscosity, only for Burgers rheologies  (numlayers) [Pa.s]');
			fielddisplay(self,'transient_mu','array describing each layer''s transient shear modulus, only for Burgers rheologies  (numlayers) [Pa]');

		end % }}}
		function marshall(self,prefix,md,fid) % {{{
			WriteData(fid,prefix,'object',self,'fieldname','mantle_viscosity','format','DoubleMat','mattype',1);
			WriteData(fid,prefix,'object',self,'fieldname','lithosphere_thickness','format','DoubleMat','mattype',1,'scale',10^3); %from km to m
			WriteData(fid,prefix,'object',self,'fieldname','cross_section_shape','format','Integer');
		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
		
			writejsdouble(fid,[modelname '.gia.mantle_viscosity'],self.mantle_viscosity);
			writejsdouble(fid,[modelname '.gia.lithosphere_thickness'],self.lithosphere_thickness);
			writejsdouble(fid,[modelname '.gia.cross_section_shape'],self.cross_section_shape);

		end % }}}
	end
end
