/*!\file AdaptiveMeshrefinement.cpp
 * \brief: implementation of the adaptive mesh refinement tool based on NeoPZ library: github.com/labmec/neopz
 */

 #ifdef HAVE_CONFIG_H
    #include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./AdaptiveMeshRefinement.h"

/*Constructor, copy, clean up and destructor*/
AdaptiveMeshRefinement::AdaptiveMeshRefinement(){/*{{{*/

    /*Set pointers to NULL*/
    this->fathermesh    = NULL;
    this->previousmesh  = NULL;
    this->hmax          = -1;
    this->elementswidth = -1;

}
/*}}}*/
AdaptiveMeshRefinement::AdaptiveMeshRefinement(const AdaptiveMeshRefinement &cp){/*{{{*/
    this->operator =(cp);
}
/*}}}*/
AdaptiveMeshRefinement & AdaptiveMeshRefinement::operator =(const AdaptiveMeshRefinement &cp){/*{{{*/

    /*Clean all attributes*/
    this->CleanUp();

    /*Copy all data*/
    this->fathermesh    = cp.fathermesh;
    this->previousmesh  = cp.previousmesh;
    this->hmax          = cp.hmax;
    this->elementswidth = cp.elementswidth;

    return *this;

}
/*}}}*/
AdaptiveMeshRefinement::~AdaptiveMeshRefinement(){/*{{{*/
    this->CleanUp();
}
/*}}}*/
void AdaptiveMeshRefinement::CleanUp(){/*{{{*/

    /*Verify and delete all data*/
    if(this->fathermesh)    delete this->fathermesh;
    if(this->previousmesh)  delete this->previousmesh;
	 this->hmax=-1;
	 this->elementswidth=-1;
}
/*}}}*/
int AdaptiveMeshRefinement::ClassId() const{/*{{{*/
    return 13829430; //Antartic area with ice shelves (km^2)
}
/*}}}*/
void AdaptiveMeshRefinement::Read(TPZStream &buf, void *context){/*{{{*/

    try
    {
        /* Read the id context*/
        TPZSaveable::Read(buf,context);

        /* Read class id*/
        int classid;
        buf.Read(&classid,1);
        
        /* Verify the class id*/
        if (classid != this->ClassId() )
        {
            std::cout << "Error in restoring AdaptiveMeshRefinement!\n";
            std::cout.flush();
            DebugStop();
        }
        
        /* Read simple attributes */
        buf.Read(&this->hmax,1);
        buf.Read(&this->elementswidth,1);
        
        /* Read geometric mesh*/
        TPZSaveable *sv1 = TPZSaveable::Restore(buf,0);
        this->fathermesh = dynamic_cast<TPZGeoMesh*>(sv1);
        
        TPZSaveable *sv2 = TPZSaveable::Restore(buf,0);
        this->previousmesh = dynamic_cast<TPZGeoMesh*>(sv2);
    }
    catch(const std::exception& e)
    {
        std::cout << "Exception catched! " << e.what() << std::endl;
        std::cout.flush();
        DebugStop();
    }
}
/*}}}*/
template class TPZRestoreClass<AdaptiveMeshRefinement,13829430>;/*{{{*/
/*}}}*/
void AdaptiveMeshRefinement::Write(TPZStream &buf, int withclassid){/*{{{*/
    
    try
    {
        /* Write context (this class) class ID*/
        TPZSaveable::Write(buf,withclassid);

        /* Write this class id*/
        int classid = ClassId();
        buf.Write(&classid,1);

        /* Write simple attributes */
        buf.Write(&this->hmax,1);
        buf.Write(&this->elementswidth,1);
        
        /* Write the geometric mesh*/
        this->fathermesh->Write(buf, this->ClassId());
        this->previousmesh->Write(buf, this->ClassId());
    }
    catch(const std::exception& e)
    {
        std::cout << "Exception catched! " << e.what() << std::endl;
        std::cout.flush();
        DebugStop();
    }
}
/*}}}*/

/*Mesh refinement methods*/
#include "TPZVTKGeoMesh.h" //itapopo
void AdaptiveMeshRefinement::ExecuteRefinement(int &type_process,double *vx, double *vy, double *masklevelset, int &nvertices, int &nelements, int &nsegments, double** x, double** y, double** z, int*** elements, int*** segments){/*{{{*/

    //itapopo _assert_(this->fathermesh);
    //itapopo _assert_(this->previousmesh);
    
    /*Calculate the position of the grounding line using previous mesh*/
    std::vector<TPZVec<REAL> > GLvec;
    this->CalcGroundingLinePosition(masklevelset, GLvec);
    
    std::ofstream file1("/ronne_1/home/santos/mesh0.vtk");
    TPZVTKGeoMesh::PrintGMeshVTK(this->previousmesh,file1 );
    
    /*run refinement or unrefinement process*/
    TPZGeoMesh *newmesh;
    switch (type_process) {
        case 0: newmesh = this->previousmesh; break;                    // refine previous mesh
        case 1: newmesh = new TPZGeoMesh(*this->fathermesh); break;     // refine mesh 0 (unrefine process)
        default: DebugStop(); break;//itapopo verificar se irá usar _assert_
    }
    
    this->RefinementProcess(newmesh,GLvec);
	
    std::ofstream file2("/ronne_1/home/santos/mesh1.vtk");
    TPZVTKGeoMesh::PrintGMeshVTK(this->previousmesh,file2 );
    
    /*Set new mesh pointer. Previous mesh just have uniform elements*/
    if(type_process==1){
        if(this->previousmesh) delete this->previousmesh;
        this->previousmesh = newmesh;
    }
    
    /*Refine elements to avoid hanging nodes*/
    TPZGeoMesh *nohangingnodesmesh = new TPZGeoMesh(*newmesh);
    
    std::ofstream file3("/ronne_1/home/santos/mesh2.vtk");
    TPZVTKGeoMesh::PrintGMeshVTK(newmesh,file3);
    
    this->RefineMeshToAvoidHangingNodes(nohangingnodesmesh);
    
    std::ofstream file4("/ronne_1/home/santos/mesh3.vtk");
    TPZVTKGeoMesh::PrintGMeshVTK(nohangingnodesmesh,file4);
    
    /*Get new geometric mesh in ISSM data structure*/
    this->GetMesh(nohangingnodesmesh,nvertices,nelements,nsegments,x,y,z,elements,segments);
	 
    /*Verify the new geometry*/
    this->CheckMesh(nvertices,nelements,nsegments,this->elementswidth,x,y,z,elements,segments);
     
    delete nohangingnodesmesh;
}
/*}}}*/
void AdaptiveMeshRefinement::RefinementProcess(TPZGeoMesh *gmesh,std::vector<TPZVec<REAL> > &GLvec){/*{{{*/
    
    /*Refine mesh hmax times*/
    for(int hlevel=1;hlevel<=this->hmax;hlevel++){
        
        /*Set elements to be refined using some criteria*/
        std::vector<int> ElemVec; //elements without children
        this->SetElementsToRefine(gmesh,GLvec,hlevel,ElemVec);
        
        /*Refine the mesh*/
        this->RefineMesh(gmesh, ElemVec);
    }
    
}
/*}}}*/
void AdaptiveMeshRefinement::RefineMesh(TPZGeoMesh *gmesh, std::vector<int> &ElemVec){/*{{{*/

	/*Refine elements in ElemVec: uniform pattern refinement*/
	for(int i = 0; i < ElemVec.size(); i++){
		
        /*Get geometric element and verify if it has already been refined*/
        int index = ElemVec[i];
        TPZGeoEl * geoel = gmesh->Element(index);
        if(geoel->HasSubElement()) DebugStop();                              //itapopo _assert_(!geoel->HasSubElement());
        if(geoel->MaterialId() != this->GetElemMaterialID()) DebugStop();   //itapopo verificar se usará _assert_
        
        /*Divide geoel*/
        TPZVec<TPZGeoEl *> Sons;
		  geoel->Divide(Sons);
        
        /*If a 1D segment is neighbor, it must be divided too*/
        if(this->elementswidth != 3) DebugStop(); //itapopo verificar o segment para malha 3D
        
        std::vector<int> sides(3);
        sides[0] = 3; sides[1] = 4; sides[2] = 5;
        for(int j = 0; j < sides.size(); j++ ){
            
            TPZGeoElSide Neighbour = geoel->Neighbour(sides[j]);
            
            if( Neighbour.Element()->MaterialId() == this->GetBoundaryMaterialID() && !Neighbour.Element()->HasSubElement() ){
                TPZVec<TPZGeoEl *> pv2;
                Neighbour.Element()->Divide(pv2);
            }
        }
	}
    
    gmesh->BuildConnectivity();

}
/*}}}*/
void AdaptiveMeshRefinement::RefineMeshToAvoidHangingNodes(TPZGeoMesh *gmesh){/*{{{*/
    
    /*Refine elements to avoid hanging nodes: non-uniform refinement*/
	 const int NElem = gmesh->NElements();
    for(int i = 0; i < NElem; i++){
        
        /*Get geometric element and verify if it has already been refined. Geoel may not have been previously refined*/
        TPZGeoEl * geoel = gmesh->Element(i);
        if(!geoel) continue;
        if(geoel->HasSubElement()) continue;
        if(geoel->MaterialId() != this->GetElemMaterialID()) continue;
        
        /*Get the refinement pattern for this element and refine it*/
        TPZAutoPointer<TPZRefPattern> refp = TPZRefPatternTools::PerfectMatchRefPattern(geoel);
        if(refp){
            TPZVec<TPZGeoEl *> Sons;
            geoel->SetRefPattern(refp);
            geoel->Divide(Sons);
        }
        
    }
    
    gmesh->BuildConnectivity();
    
}
/*}}}*/
void AdaptiveMeshRefinement::GetMesh(TPZGeoMesh *gmesh, int &nvertices, int &nelements, int &nsegments, double** meshX, double** meshY, double** meshZ, int*** elements, int*** segments){/*{{{*/

	/* vertices */
    int ntotalvertices = gmesh->NNodes();//total
    
    /* mesh coords */
    double *newmeshX = new double[ntotalvertices];
    double *newmeshY = new double[ntotalvertices];
    double *newmeshZ = new double[ntotalvertices];
   	
   /* getting mesh coords */
    for(int i = 0; i < ntotalvertices; i++ ){
        TPZVec<REAL> coords(3,0.);
        gmesh->NodeVec()[i].GetCoordinates(coords);
        newmeshX[i] = coords[0];
        newmeshY[i] = coords[1];
        newmeshZ[i] = coords[2];
    }
    
	/* elements */
    std::vector<TPZGeoEl*> GeoVec; GeoVec.clear();
    for(int i = 0; i < gmesh->NElements(); i++){ 
        if( gmesh->ElementVec()[i]->HasSubElement() ) continue;
        if( gmesh->ElementVec()[i]->MaterialId() != this->GetElemMaterialID() ) continue;
        GeoVec.push_back( gmesh->ElementVec()[i]);
    }
    
    int ntotalelements = (int)GeoVec.size();
    int ** newelements = new int*[ntotalelements];

    if ( !(this->elementswidth == 3) && !(this->elementswidth == 4) && !(this->elementswidth == 6) ) DebugStop();

    for(int i = 0; i < GeoVec.size(); i++){
        newelements[i] = new int[this->elementswidth];
        for(int j = 0; j < this->elementswidth; j++) newelements[i][j] = (int)GeoVec[i]->NodeIndex(j);
    }
    
    /* segments */
    std::vector<TPZGeoEl*> SegVec; SegVec.clear();
    for(int i = 0; i < gmesh->NElements(); i++){
        if( gmesh->ElementVec()[i]->HasSubElement() ) continue;
        if( gmesh->ElementVec()[i]->MaterialId() != this->GetBoundaryMaterialID() ) continue;
        SegVec.push_back( gmesh->ElementVec()[i]);
    }
    
    int ntotalsegments = (int)SegVec.size();
    int ** newsegments = new int*[ntotalsegments];
    
    for(int i = 0; i < SegVec.size(); i++){
        
        newsegments[i] = new int[3];
        for(int j = 0; j < 2; j++) newsegments[i][j] = SegVec[i]->NodeIndex(j);
        
        int neighborindex = SegVec[i]->Neighbour(2).Element()->Index();
        int neighbourid = -1;
        
        for(int j = 0; j < GeoVec.size(); j++){
            if( GeoVec[j]->Index() == neighborindex || GeoVec[j]->FatherIndex() == neighborindex){
                neighbourid = j;
                break;
            }
        }
        
        if(neighbourid==-1) DebugStop(); //itapopo talvez passar para _assert_
        newsegments[i][2] = neighbourid;
    }
    
    //setting outputs
    nvertices   = ntotalvertices;
    nelements   = ntotalelements;
    nsegments   = ntotalsegments;
    *meshX      = newmeshX;
    *meshY      = newmeshY;
    *meshZ      = newmeshZ;
    *elements   = newelements;
    *segments   = newsegments;
    
}
/*}}}*/
void AdaptiveMeshRefinement::CalcGroundingLinePosition(double *masklevelset,std::vector<TPZVec<REAL> > &GLvec){/*{{{*/
    
    /* Find grounding line using elments center point */
    GLvec.clear();
    for(int i=0;i<this->previousmesh->NElements();i++){
        
        if(this->previousmesh->Element(i)->MaterialId()!=this->GetElemMaterialID()) continue;
        if(this->previousmesh->Element(i)->HasSubElement()) continue;
        
        //itapopo apenas malha 2D triangular!
        int vertex0 = this->previousmesh->Element(i)->NodeIndex(0);
        int vertex1 = this->previousmesh->Element(i)->NodeIndex(1);
        int vertex2 = this->previousmesh->Element(i)->NodeIndex(2);
        
        double mls0 = masklevelset[vertex0];
        double mls1 = masklevelset[vertex1];
        double mls2 = masklevelset[vertex2];
        
        if( mls0*mls1 < 0. || mls1*mls2 < 0. ){
            const int side = 6;
            TPZVec<double> qsi(2,0.);
            TPZVec<double> X(3,0.);
            this->previousmesh->Element(i)->CenterPoint(side, qsi);
            this->previousmesh->Element(i)->X(qsi, X);
            GLvec.push_back(X);
        }
    }
    
//    itapopo apenas para debugar
//    std::ofstream fileGL("/Users/santos/Desktop/gl.nb");
//    fileGL << "ListPlot[{";
//    for(int i = 0; i < GLvec.size(); i++){
//        fileGL << "{" << GLvec[i][0] << "," << GLvec[i][1] << /*"," << 0. << */"}";
//        if(i != GLvec.size()-1) fileGL << ",";
//    }
//    fileGL << "}]";
//    fileGL.flush();
//    fileGL.close();
    
}
/*}}}*/
void AdaptiveMeshRefinement::SetElementsToRefine(TPZGeoMesh *gmesh,std::vector<TPZVec<REAL> > &GLvec,int &hlevel,std::vector<int> &ElemVec){/*{{{*/

    if(!gmesh) DebugStop(); //itapopo verificar se usará _assert_

    ElemVec.clear();
    
    // itapopo inserir modo de encontrar criterio
    if(false) this->TagAllElements(gmesh,ElemVec); //uniform, refine all elements!

    /* Adaptive refinement. This refines some elements following some criteria*/
    this->TagElementsNearGroundingLine(gmesh, GLvec, hlevel, ElemVec);

}
/*}}}*/
void AdaptiveMeshRefinement::TagAllElements(TPZGeoMesh *gmesh,std::vector<int> &ElemVec){/*{{{*/
    
    /* Uniform refinement. This refines the entire mesh */
    int nelements = gmesh->NElements();
    for(int i=0;i<nelements;i++){
        if(gmesh->Element(i)->MaterialId()!=this->GetElemMaterialID()) continue;
        if(gmesh->Element(i)->HasSubElement()) continue;
        ElemVec.push_back(i);
    }
}
/*}}}*/
void AdaptiveMeshRefinement::TagElementsNearGroundingLine(TPZGeoMesh *gmesh,std::vector<TPZVec<REAL> > &GLvec,int &hlevel,std::vector<int> &ElemVec){/*{{{*/
    
    /* Tag elements near grounding line */
    double MaxRegion = 20000.; //itapopo
    double alpha = 1.0;         //itapopo
    double MaxDistance = MaxRegion / std::exp(alpha*(hlevel-1));
    
    for(int i=0;i<gmesh->NElements();i++){
        
        if(gmesh->Element(i)->MaterialId()!=this->GetElemMaterialID()) continue;
        if(gmesh->Element(i)->HasSubElement()) continue;
        if(gmesh->Element(i)->Level()>=hlevel) continue;
        
        const int side2D = 6;
        TPZVec<REAL> qsi(2,0.);
        TPZVec<REAL> centerPoint(3,0.);
        gmesh->Element(i)->CenterPoint(side2D, qsi);
        gmesh->Element(i)->X(qsi, centerPoint);
        
        REAL distance = MaxDistance;
        
        for (int j = 0; j < GLvec.size(); j++) {
            
            REAL value = ( GLvec[j][0] - centerPoint[0] ) * ( GLvec[j][0] - centerPoint[0] ); // (x2-x1)^2
            value += ( GLvec[j][1] - centerPoint[1] ) * ( GLvec[j][1] - centerPoint[1] );// (y2-y1)^2
            value = std::sqrt(value); ///Radius
            
            //finding the min distance to the grounding line
            if(value < distance) distance = value;
            
        }
        
        if(distance < MaxDistance) ElemVec.push_back(i);
    }
    
}
/*}}}*/
void AdaptiveMeshRefinement::CreateInitialMesh(int &nvertices, int &nelements, int &nsegments, int &width, double* x, double* y, double* z, int** elements, int** segments){/*{{{*/

	// itapopo _assert_(nvertices>0);
    // itapoo _assert_(nelements>0);
   this->SetElementWidth(width);

    /*Verify and creating initial mesh*/
    //itapopo if(this->fathermesh) _error_("Initial mesh already exists!");
    
   this->fathermesh = new TPZGeoMesh();
	this->fathermesh->NodeVec().Resize( nvertices );

    /*Set the vertices (geometric nodes in NeoPZ context)*/
	for(int i=0;i<nvertices;i++){
        
        /*x,y,z coords*/
        TPZManVector<REAL,3> coord(3,0.);
        coord[0]= x[i];
        coord[1]= y[i];
        coord[2]= z[i];
		
        /*Insert in the mesh*/
        this->fathermesh->NodeVec()[i].SetCoord(coord);
		  this->fathermesh->NodeVec()[i].SetNodeId(i);
	}
	
	/*Generate the elements*/
    long index;
    const int mat = this->GetElemMaterialID();
    TPZManVector<long> elem(this->elementswidth,0);
    
	for(int iel=0;iel<nelements;iel++){

		for(int jel=0;jel<this->elementswidth;jel++) elem[jel]=elements[iel][jel];

        /*reftype = 0: uniform, fast / reftype = 1: uniform and non-uniform (avoid hanging nodes), it is not too fast */
        const int reftype = 1;
        switch(this->elementswidth){
			case 3: this->fathermesh->CreateGeoElement(ETriangle, elem, mat, index, reftype);			break;
            case 4:	this->fathermesh->CreateGeoElement(ETetraedro, elem, mat, index, reftype); DebugStop(); break;// itapopo _error_("tetra elements must be verified!")	break;
			case 6:	this->fathermesh->CreateGeoElement(EPrisma, elem, mat, index, reftype);		break;
            default:	DebugStop();//itapopo _error_("mesh not supported yet");
		}
        
        /*Define the element ID*/        
        this->fathermesh->ElementVec()[index]->SetId(iel);
        
	}
    
    /*Generate the 1D segments elements (boundary)*/
    const int matboundary = this->GetBoundaryMaterialID();
    TPZManVector<long> boundary(2,0.);
    
    for(int iel=nelements;iel<nelements+nsegments;iel++){
        
        boundary[0] = segments[iel-nelements][0];
        boundary[1] = segments[iel-nelements][1];
        
        /*reftype = 0: uniform, fast / reftype = 1: uniform and non-uniform (avoid hanging nodes), it is not too fast */
        const int reftype = 0;
        this->fathermesh->CreateGeoElement(EOned, boundary, matboundary, index, reftype);//cria elemento unidimensional
        this->fathermesh->ElementVec()[index]->SetId(iel);
        
    }
    
    /*Build element and node connectivities*/
    this->fathermesh->BuildConnectivity();
    /*Create previous mesh as a copy of father mesh*/
    this->previousmesh = new TPZGeoMesh(*this->fathermesh);
    
}
/*}}}*/
void AdaptiveMeshRefinement::SetHMax(int &h){/*{{{*/
    this->hmax = h;
}
/*}}}*/
void AdaptiveMeshRefinement::SetElementWidth(int &width){/*{{{*/
    this->elementswidth = width;
}
/*}}}*/
void AdaptiveMeshRefinement::CheckMesh(int &nvertices, int &nelements, int &nsegments,int &width, double** x, double** y, double** z, int*** elements, int*** segments){/*{{{*/

    /*Basic verification*/
    if( !(nvertices > 0) || !(nelements > 0) ) DebugStop(); //itapopo verificar se irá usar o _assert_
    
    if ( !(width == 3) && !(width == 4) && !(width == 6) ) DebugStop(); // itapopo verifcar se irá usar o _assert_
    
    if( !x || !y || !z || !elements ) DebugStop(); // itapopo verifcar se irá usar o _assert_
    
    /*Verify if there are orphan nodes*/
    std::set<int> elemvertices;
    elemvertices.clear(); 
    for(int i = 0; i < nelements; i++){
        for(int j = 0; j < width; j++) {
            elemvertices.insert((*elements)[i][j]);
		  }
	 }
    
    if( elemvertices.size() != nvertices ) DebugStop();//itapopo verificar se irá usar o _assert_
	
    //Verify if there are inf or NaN in coords
    for(int i = 0; i < nvertices; i++) if(isnan((*x)[i]) || isinf((*x)[i])) DebugStop();
    for(int i = 0; i < nvertices; i++) if(isnan((*y)[i]) || isinf((*y)[i])) DebugStop();
    for(int i = 0; i < nvertices; i++) if(isnan((*z)[i]) || isinf((*z)[i])) DebugStop();
   
	 for(int i = 0; i < nelements; i++){
        for(int j = 0; j < width; j++){
            if( isnan((*elements)[i][j]) || isinf((*elements)[i][j]) ) DebugStop();
        }
    }
    
}
/*}}}*/
