function varargout=updatearchive(varargin)
%RUNME - test deck for ISSM nightly runs
%
%   Usage:
%      varargout=updatearchive(varargin);
%
%   Examples:
%      updatearchive;
%      updatearchive({'ice'});
%      updatearchive({'ice','cielo_serial'},{'diagnostic'});
%      md=updatearchive({'cielo_parallel'});
%
%   See Also: UPDATEARCHIVE

% This file can be run to check that the current version of issm is giving 
% coherent results. This test deals with an icesheet without icefront for a 2d model. The geometry 
% is square. Just run this file in Matlab, with a properly setup ISSM code. 
% The results of this test will indicate if there is a difference between current computations 
% and archived results.

% Errors  between archived results and the current version will get flagged if they are not within 
% a certain tolerance. The current tolerance is 10^-12. If you have good reasons to believe this 
% tolerance should be lowered (for example, if you are running single precision compilers?), feel 
% free to tweak the tolerance variable.

%check arguments
if (nargin>2 | nargout>1)
	help updatearchive
	error('updatearchive error message: bad usage');
end

%warning if a updatearchive already exists:
if exist('./updatearchive.m')
	disp('this is the nighlty run updatearchive, located in src/m/utils/Nightly/updatearchive.m')
	disp('Another local updatearchive exist but is not executed')
end

%recover options
options=recover_options(varargin{:});

%read configuration
configuration;

%packages and analysis_types to be tested
if nargin==2,
	solutions=varargin{2};
end

if nargin,
	packages=varargin{1};
end

%go through the sequences requested.
for i=1:size(sequences,1),
	sequence=sequences{i};

	%recover what is being requested.
	analysis_type=sequence{1};
	sub_analysis_type=sequence{2};
	qmu=sequence{3};
	control=sequence{4};
	control_fit=sequence{5};
	parallel=sequence{6};

	%CHECK analysis_type
	analysis_type_arg=find_option(options,'analysis_type');
	if ~isempty(analysis_type_arg),
		if ~ismember(analysis_type,analysis_type_arg)
			continue
		end
	end

	%CHECK sub_analysis_type
	sub_analysis_type_arg=find_option(options,'sub_analysis_type');
	if ~isempty(sub_analysis_type_arg),
		if ~ismember(sub_analysis_type,sub_analysis_type_arg)
			continue
		end
	end

	%CHECK parallel
	parallel_arg=find_option(options,'parallel');
	if ~isempty(parallel_arg),
		if parallel~=parallel_arg,
			continue
		end
	end

	%initialize model, using presolve if need be, or just the piece of generic code below.
	md=model;
	if ~rifts,
		md=mesh(md,domainname,resolution);
	else
		md=mesh(md,domainname,riftoutline,resolution);
		md=meshprocessrifts(md);
	end
	md=geography(md,iceshelfname,icesheetname);
	md=parameterize(md,parametername);
	md=extrude(md,numlayers,extrusionexponent);
	eval(['md=setelementstype(md,' elementstype ');']);

	if parallel, 
		%this run will be done in parallel
		md.cluster=oshostname;
		%check and modify number of cpus if necessary
		np=feature('numCores');
		if np<md.np;
			disp(['warning: changing number of cpus to ' num2str(np) ]);
			md.np=np;
		end
		runtype='parallel';
	else
		runtype='serial';
	end

	%some solution checks;
	if (  (strcmpi(analysis_type,'thermal') & strcmpi(sub_analysis_type,'steady')) | ...
			(strcmpi(analysis_type,'steadystate'))...
		)
		md.dt=0;
	end

	if control,
		md.control_analysis=1;
		if strcmpi(control_fit,'absolute'),
			md.fit(:)=0;
		elseif strcmpi(control_fit,'relative'),
			md.fit(:)=1;
		elseif strcmpi(control_fit,'logarithmic'),
			md.fit(:)=2;
		else
			error(['updatearchive error message: for control solution, ' control_fit ' fit is not supported!']);
		end
	end
	
	%presolve
	if exist('./testpresolve.m','file')==2,
		testpresolve;
	end

	%solve
	md=solve(md,'analysis_type',analysis_type,'sub_analysis_type',sub_analysis_type);

	%postsolve
	if exist('./testpostsolve.m','file')==2,
		testpostsolve;
	end
	md=solve(md,'analysis_type',analysis_type,'sub_analysis_type',sub_analysis_type);

	%compute fields to be checked
	fields=testsgetfields2(md,sequence);

	%save new archive
	if parallel,
		runtype='parallel';
	else
		runtype='serial';
	end
	for k=1:length(fields),
		field=fields{k};
		eval(['Archive_'  analysis_type '_' sub_analysis_type '_' runtype '_field' num2str(k) ' = md.results. ' field  ';']);
	end
	eval(['save Archive_'  analysis_type '_' sub_analysis_type '_' runtype ' Archive_'  analysis_type '_' sub_analysis_type '_' runtype '_field*']);
	disp(sprintf(['File Archive_'  analysis_type '_' sub_analysis_type '_' runtype ' saved\n']));

end

%output model md if requested
if nargout==1, 
	varargout{1}=md; 
end
